/*!
 * SAP UI development toolkit for HTML5 (SAPUI5)
 *  (C) 2016 SAP SE or an SAP affiliate company. All rights reserved.
 */

var sap = sap || {};
sap.ve = sap.ve || {};
sap.ve.matai = sap.ve.matai || {};
sap.ve.matai.createRuntime = function(options) {
	return new Promise(function(resolve, reject) {
		function createModule(options) {
			// Emscripten module definition begins here. It ends in module_end.js.

			var mergeOption = function(optionName, defaultOptionValue) {
				if (!options) {
					return defaultOptionValue;
				}
				return typeof options[optionName] === "undefined" ? defaultOptionValue : options[optionName];
			};

			// Merge any input options with defaults.
			var mergedOptions = {
				totalMemory:          mergeOption("totalMemory",          128 * 1024 * 1024), // 128 MB
				logElementId:         mergeOption("logElementId",         null),
				statusElementId:      mergeOption("statusElementId",      null),
				prefixURL: mergeOption("prefixURL", null)
			};

			//Create initial Module object with defaults
			var Module = {
				TOTAL_MEMORY: mergedOptions.totalMemory,
				prefixURL: mergedOptions.prefixURL,
				preInit: [],
				postRun: [],
				totalDependencies: 0,
				monitorRunDependencies: null,
				noInitialRun: true,
				noFSInit: true, // NB: Do not initialize standard streams (a workaround for cases when a file is opened before standard streams are initialized).
				print: function() {
					var args = Array.prototype.slice.call(arguments);
					var text = args.join(" ");
					if (mergedOptions.logElementId) {
						var element = document.getElementById(mergedOptions.logElementId);
						element.value += text + "\n";
						element.scrollTop = 99999;
					} else if (self.console) {
						self.console.log("print: " + text);
					}
				},
				printf: function() {
					var args = Array.prototype.slice.call(arguments);
					args.push("(printf)");
					Module.print(args);
				},
				printErr: function() {
					var args = Array.prototype.slice.call(arguments);
					args.push("(printErr)");
					Module.print(args);
				},
				// setStatus is called from Emscripten libraries.
				setStatus: function(text) {
					if (mergedOptions.statusElementId) {
						if (Module.setStatus.interval) {
							clearInterval(Module.setStatus.interval);
						}
						var statusElement = document.getElementById(mergedOptions.statusElementId);
						statusElement.innerHTML = text;
					} else if (Module.logElementId) {
						var args = Array.prototype.slice.call(arguments);
						args.push("(setStatus)");
						Module.print(args);
					}
				},
				locateFile: function(path, scriptDirectory) {
					return this.prefixURL + path;
				}
			};

			// NB: Emscripten generated code will be inserted after this line.

// The Module object: Our interface to the outside world. We import
// and export values on it. There are various ways Module can be used:
// 1. Not defined. We create it here
// 2. A function parameter, function(Module) { ..generated code.. }
// 3. pre-run appended it, var Module = {}; ..generated code..
// 4. External script tag defines var Module.
// We need to check if Module already exists (e.g. case 3 above).
// Substitution will be replaced with actual code on later stage of the build,
// this way Closure Compiler will not mangle it (e.g. case 4. above).
// Note that if you want to run closure, and also to use Module
// after the generated code, you will need to define   var Module = {};
// before the code. Then that object will be used in the code, and you
// can continue to use Module afterwards as well.
var Module = typeof Module !== 'undefined' ? Module : {};

// --pre-jses are emitted after the Module integration code, so that they can
// refer to Module (if they choose; they can also define Module)
// {{PRE_JSES}}

// Sometimes an existing Module object exists with properties
// meant to overwrite the default module functionality. Here
// we collect those properties and reapply _after_ we configure
// the current environment's defaults to avoid having to be so
// defensive during initialization.
var moduleOverrides = {};
var key;
for (key in Module) {
  if (Module.hasOwnProperty(key)) {
    moduleOverrides[key] = Module[key];
  }
}

Module['arguments'] = [];
Module['thisProgram'] = './this.program';
Module['quit'] = function(status, toThrow) {
  throw toThrow;
};
Module['preRun'] = [];
Module['postRun'] = [];

// Determine the runtime environment we are in. You can customize this by
// setting the ENVIRONMENT setting at compile time (see settings.js).

var ENVIRONMENT_IS_WEB = false;
var ENVIRONMENT_IS_WORKER = false;
var ENVIRONMENT_IS_NODE = false;
var ENVIRONMENT_IS_SHELL = false;
ENVIRONMENT_IS_WEB = typeof window === 'object';
ENVIRONMENT_IS_WORKER = typeof importScripts === 'function';
ENVIRONMENT_IS_NODE = typeof process === 'object' && typeof require === 'function' && !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_WORKER;
ENVIRONMENT_IS_SHELL = !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_NODE && !ENVIRONMENT_IS_WORKER;


// Three configurations we can be running in:
// 1) We could be the application main() thread running in the main JS UI thread. (ENVIRONMENT_IS_WORKER == false and ENVIRONMENT_IS_PTHREAD == false)
// 2) We could be the application main() thread proxied to worker. (with Emscripten -s PROXY_TO_WORKER=1) (ENVIRONMENT_IS_WORKER == true, ENVIRONMENT_IS_PTHREAD == false)
// 3) We could be an application pthread running in a worker. (ENVIRONMENT_IS_WORKER == true and ENVIRONMENT_IS_PTHREAD == true)


// `/` should be present at the end if `scriptDirectory` is not empty
var scriptDirectory = '';
function locateFile(path) {
  if (Module['locateFile']) {
    return Module['locateFile'](path, scriptDirectory);
  } else {
    return scriptDirectory + path;
  }
}

if (ENVIRONMENT_IS_NODE) {
  scriptDirectory = __dirname + '/';

  // Expose functionality in the same simple way that the shells work
  // Note that we pollute the global namespace here, otherwise we break in node
  var nodeFS;
  var nodePath;

  Module['read'] = function shell_read(filename, binary) {
    var ret;
    ret = tryParseAsDataURI(filename);
    if (!ret) {
      if (!nodeFS) nodeFS = require('fs');
      if (!nodePath) nodePath = require('path');
      filename = nodePath['normalize'](filename);
      ret = nodeFS['readFileSync'](filename);
    }
    return binary ? ret : ret.toString();
  };

  Module['readBinary'] = function readBinary(filename) {
    var ret = Module['read'](filename, true);
    if (!ret.buffer) {
      ret = new Uint8Array(ret);
    }
    assert(ret.buffer);
    return ret;
  };

  if (process['argv'].length > 1) {
    Module['thisProgram'] = process['argv'][1].replace(/\\/g, '/');
  }

  Module['arguments'] = process['argv'].slice(2);

  if (typeof module !== 'undefined') {
    module['exports'] = Module;
  }

  process['on']('uncaughtException', function(ex) {
    // suppress ExitStatus exceptions from showing an error
    if (!(ex instanceof ExitStatus)) {
      throw ex;
    }
  });
  // Currently node will swallow unhandled rejections, but this behavior is
  // deprecated, and in the future it will exit with error status.
  process['on']('unhandledRejection', function(reason, p) {
    process['exit'](1);
  });

  Module['quit'] = function(status) {
    process['exit'](status);
  };

  Module['inspect'] = function () { return '[Emscripten Module object]'; };
} else
if (ENVIRONMENT_IS_SHELL) {


  if (typeof read != 'undefined') {
    Module['read'] = function shell_read(f) {
      var data = tryParseAsDataURI(f);
      if (data) {
        return intArrayToString(data);
      }
      return read(f);
    };
  }

  Module['readBinary'] = function readBinary(f) {
    var data;
    data = tryParseAsDataURI(f);
    if (data) {
      return data;
    }
    if (typeof readbuffer === 'function') {
      return new Uint8Array(readbuffer(f));
    }
    data = read(f, 'binary');
    assert(typeof data === 'object');
    return data;
  };

  if (typeof scriptArgs != 'undefined') {
    Module['arguments'] = scriptArgs;
  } else if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }

  if (typeof quit === 'function') {
    Module['quit'] = function(status) {
      quit(status);
    }
  }
} else
if (ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) {
  if (ENVIRONMENT_IS_WEB) {
    if (document.currentScript) {
      scriptDirectory = document.currentScript.src;
    }
  } else { // worker
    scriptDirectory = self.location.href;
  }
  // blob urls look like blob:http://site.com/etc/etc and we cannot infer anything from them.
  // otherwise, slice off the final part of the url to find the script directory.
  // if scriptDirectory does not contain a slash, lastIndexOf will return -1,
  // and scriptDirectory will correctly be replaced with an empty string.
  if (scriptDirectory.indexOf('blob:') !== 0) {
    scriptDirectory = scriptDirectory.substr(0, scriptDirectory.lastIndexOf('/')+1);
  } else {
    scriptDirectory = '';
  }


  Module['read'] = function shell_read(url) {
    try {
      var xhr = new XMLHttpRequest();
      xhr.open('GET', url, false);
      xhr.send(null);
      return xhr.responseText;
    } catch (err) {
      var data = tryParseAsDataURI(url);
      if (data) {
        return intArrayToString(data);
      }
      throw err;
    }
  };

  if (ENVIRONMENT_IS_WORKER) {
    Module['readBinary'] = function readBinary(url) {
      try {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', url, false);
        xhr.responseType = 'arraybuffer';
        xhr.send(null);
        return new Uint8Array(xhr.response);
      } catch (err) {
        var data = tryParseAsDataURI(url);
        if (data) {
          return data;
        }
        throw err;
      }
    };
  }

  Module['readAsync'] = function readAsync(url, onload, onerror) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, true);
    xhr.responseType = 'arraybuffer';
    xhr.onload = function xhr_onload() {
      if (xhr.status == 200 || (xhr.status == 0 && xhr.response)) { // file URLs can return 0
        onload(xhr.response);
        return;
      }
      var data = tryParseAsDataURI(url);
      if (data) {
        onload(data.buffer);
        return;
      }
      onerror();
    };
    xhr.onerror = onerror;
    xhr.send(null);
  };

  Module['setWindowTitle'] = function(title) { document.title = title };
} else
{
}

// Set up the out() and err() hooks, which are how we can print to stdout or
// stderr, respectively.
// If the user provided Module.print or printErr, use that. Otherwise,
// console.log is checked first, as 'print' on the web will open a print dialogue
// printErr is preferable to console.warn (works better in shells)
// bind(console) is necessary to fix IE/Edge closed dev tools panel behavior.
var out = Module['print'] || (typeof console !== 'undefined' ? console.log.bind(console) : (typeof print !== 'undefined' ? print : null));
var err = Module['printErr'] || (typeof printErr !== 'undefined' ? printErr : ((typeof console !== 'undefined' && console.warn.bind(console)) || out));

// Merge back in the overrides
for (key in moduleOverrides) {
  if (moduleOverrides.hasOwnProperty(key)) {
    Module[key] = moduleOverrides[key];
  }
}
// Free the object hierarchy contained in the overrides, this lets the GC
// reclaim data used e.g. in memoryInitializerRequest, which is a large typed array.
moduleOverrides = undefined;



// {{PREAMBLE_ADDITIONS}}

var STACK_ALIGN = 16;


function staticAlloc(size) {
  var ret = STATICTOP;
  STATICTOP = (STATICTOP + size + 15) & -16;
  return ret;
}

function dynamicAlloc(size) {
  var ret = HEAP32[DYNAMICTOP_PTR>>2];
  var end = (ret + size + 15) & -16;
  HEAP32[DYNAMICTOP_PTR>>2] = end;
  if (end >= TOTAL_MEMORY) {
    var success = enlargeMemory();
    if (!success) {
      HEAP32[DYNAMICTOP_PTR>>2] = ret;
      return 0;
    }
  }
  return ret;
}

function alignMemory(size, factor) {
  if (!factor) factor = STACK_ALIGN; // stack alignment (16-byte) by default
  var ret = size = Math.ceil(size / factor) * factor;
  return ret;
}

function getNativeTypeSize(type) {
  switch (type) {
    case 'i1': case 'i8': return 1;
    case 'i16': return 2;
    case 'i32': return 4;
    case 'i64': return 8;
    case 'float': return 4;
    case 'double': return 8;
    default: {
      if (type[type.length-1] === '*') {
        return 4; // A pointer
      } else if (type[0] === 'i') {
        var bits = parseInt(type.substr(1));
        assert(bits % 8 === 0);
        return bits / 8;
      } else {
        return 0;
      }
    }
  }
}

function warnOnce(text) {
  if (!warnOnce.shown) warnOnce.shown = {};
  if (!warnOnce.shown[text]) {
    warnOnce.shown[text] = 1;
    err(text);
  }
}

var asm2wasmImports = { // special asm2wasm imports
    "f64-rem": function(x, y) {
        return x % y;
    },
    "debugger": function() {
        debugger;
    }
};



var jsCallStartIndex = 1;
var functionPointers = new Array(0);

// 'sig' parameter is only used on LLVM wasm backend
function addFunction(func, sig) {
  var base = 0;
  for (var i = base; i < base + 0; i++) {
    if (!functionPointers[i]) {
      functionPointers[i] = func;
      return jsCallStartIndex + i;
    }
  }
  throw 'Finished up all reserved function pointers. Use a higher value for RESERVED_FUNCTION_POINTERS.';
}

function removeFunction(index) {
  functionPointers[index-jsCallStartIndex] = null;
}

var funcWrappers = {};

function getFuncWrapper(func, sig) {
  if (!func) return; // on null pointer, return undefined
  assert(sig);
  if (!funcWrappers[sig]) {
    funcWrappers[sig] = {};
  }
  var sigCache = funcWrappers[sig];
  if (!sigCache[func]) {
    // optimize away arguments usage in common cases
    if (sig.length === 1) {
      sigCache[func] = function dynCall_wrapper() {
        return dynCall(sig, func);
      };
    } else if (sig.length === 2) {
      sigCache[func] = function dynCall_wrapper(arg) {
        return dynCall(sig, func, [arg]);
      };
    } else {
      // general case
      sigCache[func] = function dynCall_wrapper() {
        return dynCall(sig, func, Array.prototype.slice.call(arguments));
      };
    }
  }
  return sigCache[func];
}


function makeBigInt(low, high, unsigned) {
  return unsigned ? ((+((low>>>0)))+((+((high>>>0)))*4294967296.0)) : ((+((low>>>0)))+((+((high|0)))*4294967296.0));
}

function dynCall(sig, ptr, args) {
  if (args && args.length) {
    return Module['dynCall_' + sig].apply(null, [ptr].concat(args));
  } else {
    return Module['dynCall_' + sig].call(null, ptr);
  }
}



var Runtime = {
  // FIXME backwards compatibility layer for ports. Support some Runtime.*
  //       for now, fix it there, then remove it from here. That way we
  //       can minimize any period of breakage.
  dynCall: dynCall, // for SDL2 port
};

// The address globals begin at. Very low in memory, for code size and optimization opportunities.
// Above 0 is static memory, starting with globals.
// Then the stack.
// Then 'dynamic' memory for sbrk.
var GLOBAL_BASE = 8;


// === Preamble library stuff ===

// Documentation for the public APIs defined in this file must be updated in:
//    site/source/docs/api_reference/preamble.js.rst
// A prebuilt local version of the documentation is available at:
//    site/build/text/docs/api_reference/preamble.js.txt
// You can also build docs locally as HTML or other formats in site/
// An online HTML version (which may be of a different version of Emscripten)
//    is up at http://kripken.github.io/emscripten-site/docs/api_reference/preamble.js.html



//========================================
// Runtime essentials
//========================================

// whether we are quitting the application. no code should run after this.
// set in exit() and abort()
var ABORT = false;

// set by exit() and abort().  Passed to 'onExit' handler.
// NOTE: This is also used as the process return code code in shell environments
// but only when noExitRuntime is false.
var EXITSTATUS = 0;

/** @type {function(*, string=)} */
function assert(condition, text) {
  if (!condition) {
    abort('Assertion failed: ' + text);
  }
}

var globalScope = this;

// Returns the C function with a specified identifier (for C++, you need to do manual name mangling)
function getCFunc(ident) {
  var func = Module['_' + ident]; // closure exported function
  assert(func, 'Cannot call unknown function ' + ident + ', make sure it is exported');
  return func;
}

var JSfuncs = {
  // Helpers for cwrap -- it can't refer to Runtime directly because it might
  // be renamed by closure, instead it calls JSfuncs['stackSave'].body to find
  // out what the minified function name is.
  'stackSave': function() {
    stackSave()
  },
  'stackRestore': function() {
    stackRestore()
  },
  // type conversion from js to c
  'arrayToC' : function(arr) {
    var ret = stackAlloc(arr.length);
    writeArrayToMemory(arr, ret);
    return ret;
  },
  'stringToC' : function(str) {
    var ret = 0;
    if (str !== null && str !== undefined && str !== 0) { // null string
      // at most 4 bytes per UTF-8 code point, +1 for the trailing '\0'
      var len = (str.length << 2) + 1;
      ret = stackAlloc(len);
      stringToUTF8(str, ret, len);
    }
    return ret;
  }
};

// For fast lookup of conversion functions
var toC = {
  'string': JSfuncs['stringToC'], 'array': JSfuncs['arrayToC']
};


// C calling interface.
function ccall(ident, returnType, argTypes, args, opts) {
  function convertReturnValue(ret) {
    if (returnType === 'string') return Pointer_stringify(ret);
    if (returnType === 'boolean') return Boolean(ret);
    return ret;
  }

  var func = getCFunc(ident);
  var cArgs = [];
  var stack = 0;
  if (args) {
    for (var i = 0; i < args.length; i++) {
      var converter = toC[argTypes[i]];
      if (converter) {
        if (stack === 0) stack = stackSave();
        cArgs[i] = converter(args[i]);
      } else {
        cArgs[i] = args[i];
      }
    }
  }
  var ret = func.apply(null, cArgs);
  ret = convertReturnValue(ret);
  if (stack !== 0) stackRestore(stack);
  return ret;
}

function cwrap(ident, returnType, argTypes, opts) {
  argTypes = argTypes || [];
  // When the function takes numbers and returns a number, we can just return
  // the original function
  var numericArgs = argTypes.every(function(type){ return type === 'number'});
  var numericRet = returnType !== 'string';
  if (numericRet && numericArgs && !opts) {
    return getCFunc(ident);
  }
  return function() {
    return ccall(ident, returnType, argTypes, arguments, opts);
  }
}

/** @type {function(number, number, string, boolean=)} */
function setValue(ptr, value, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': HEAP8[((ptr)>>0)]=value; break;
      case 'i8': HEAP8[((ptr)>>0)]=value; break;
      case 'i16': HEAP16[((ptr)>>1)]=value; break;
      case 'i32': HEAP32[((ptr)>>2)]=value; break;
      case 'i64': (tempI64 = [value>>>0,(tempDouble=value,(+(Math_abs(tempDouble))) >= 1.0 ? (tempDouble > 0.0 ? ((Math_min((+(Math_floor((tempDouble)/4294967296.0))), 4294967295.0))|0)>>>0 : (~~((+(Math_ceil((tempDouble - +(((~~(tempDouble)))>>>0))/4294967296.0)))))>>>0) : 0)],HEAP32[((ptr)>>2)]=tempI64[0],HEAP32[(((ptr)+(4))>>2)]=tempI64[1]); break;
      case 'float': HEAPF32[((ptr)>>2)]=value; break;
      case 'double': HEAPF64[((ptr)>>3)]=value; break;
      default: abort('invalid type for setValue: ' + type);
    }
}

/** @type {function(number, string, boolean=)} */
function getValue(ptr, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': return HEAP8[((ptr)>>0)];
      case 'i8': return HEAP8[((ptr)>>0)];
      case 'i16': return HEAP16[((ptr)>>1)];
      case 'i32': return HEAP32[((ptr)>>2)];
      case 'i64': return HEAP32[((ptr)>>2)];
      case 'float': return HEAPF32[((ptr)>>2)];
      case 'double': return HEAPF64[((ptr)>>3)];
      default: abort('invalid type for getValue: ' + type);
    }
  return null;
}

var ALLOC_NORMAL = 0; // Tries to use _malloc()
var ALLOC_STACK = 1; // Lives for the duration of the current function call
var ALLOC_STATIC = 2; // Cannot be freed
var ALLOC_DYNAMIC = 3; // Cannot be freed except through sbrk
var ALLOC_NONE = 4; // Do not allocate

// allocate(): This is for internal use. You can use it yourself as well, but the interface
//             is a little tricky (see docs right below). The reason is that it is optimized
//             for multiple syntaxes to save space in generated code. So you should
//             normally not use allocate(), and instead allocate memory using _malloc(),
//             initialize it with setValue(), and so forth.
// @slab: An array of data, or a number. If a number, then the size of the block to allocate,
//        in *bytes* (note that this is sometimes confusing: the next parameter does not
//        affect this!)
// @types: Either an array of types, one for each byte (or 0 if no type at that position),
//         or a single type which is used for the entire block. This only matters if there
//         is initial data - if @slab is a number, then this does not matter at all and is
//         ignored.
// @allocator: How to allocate memory, see ALLOC_*
/** @type {function((TypedArray|Array<number>|number), string, number, number=)} */
function allocate(slab, types, allocator, ptr) {
  var zeroinit, size;
  if (typeof slab === 'number') {
    zeroinit = true;
    size = slab;
  } else {
    zeroinit = false;
    size = slab.length;
  }

  var singleType = typeof types === 'string' ? types : null;

  var ret;
  if (allocator == ALLOC_NONE) {
    ret = ptr;
  } else {
    ret = [typeof _malloc === 'function' ? _malloc : staticAlloc, stackAlloc, staticAlloc, dynamicAlloc][allocator === undefined ? ALLOC_STATIC : allocator](Math.max(size, singleType ? 1 : types.length));
  }

  if (zeroinit) {
    var stop;
    ptr = ret;
    assert((ret & 3) == 0);
    stop = ret + (size & ~3);
    for (; ptr < stop; ptr += 4) {
      HEAP32[((ptr)>>2)]=0;
    }
    stop = ret + size;
    while (ptr < stop) {
      HEAP8[((ptr++)>>0)]=0;
    }
    return ret;
  }

  if (singleType === 'i8') {
    if (slab.subarray || slab.slice) {
      HEAPU8.set(/** @type {!Uint8Array} */ (slab), ret);
    } else {
      HEAPU8.set(new Uint8Array(slab), ret);
    }
    return ret;
  }

  var i = 0, type, typeSize, previousType;
  while (i < size) {
    var curr = slab[i];

    type = singleType || types[i];
    if (type === 0) {
      i++;
      continue;
    }

    if (type == 'i64') type = 'i32'; // special case: we have one i32 here, and one i32 later

    setValue(ret+i, curr, type);

    // no need to look up size unless type changes, so cache it
    if (previousType !== type) {
      typeSize = getNativeTypeSize(type);
      previousType = type;
    }
    i += typeSize;
  }

  return ret;
}

// Allocate memory during any stage of startup - static memory early on, dynamic memory later, malloc when ready
function getMemory(size) {
  if (!staticSealed) return staticAlloc(size);
  if (!runtimeInitialized) return dynamicAlloc(size);
  return _malloc(size);
}

/** @type {function(number, number=)} */
function Pointer_stringify(ptr, length) {
  if (length === 0 || !ptr) return '';
  // Find the length, and check for UTF while doing so
  var hasUtf = 0;
  var t;
  var i = 0;
  while (1) {
    t = HEAPU8[(((ptr)+(i))>>0)];
    hasUtf |= t;
    if (t == 0 && !length) break;
    i++;
    if (length && i == length) break;
  }
  if (!length) length = i;

  var ret = '';

  if (hasUtf < 128) {
    var MAX_CHUNK = 1024; // split up into chunks, because .apply on a huge string can overflow the stack
    var curr;
    while (length > 0) {
      curr = String.fromCharCode.apply(String, HEAPU8.subarray(ptr, ptr + Math.min(length, MAX_CHUNK)));
      ret = ret ? ret + curr : curr;
      ptr += MAX_CHUNK;
      length -= MAX_CHUNK;
    }
    return ret;
  }
  return UTF8ToString(ptr);
}

// Given a pointer 'ptr' to a null-terminated ASCII-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function AsciiToString(ptr) {
  var str = '';
  while (1) {
    var ch = HEAP8[((ptr++)>>0)];
    if (!ch) return str;
    str += String.fromCharCode(ch);
  }
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in ASCII form. The copy will require at most str.length+1 bytes of space in the HEAP.

function stringToAscii(str, outPtr) {
  return writeAsciiToMemory(str, outPtr, false);
}

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the given array that contains uint8 values, returns
// a copy of that string as a Javascript String object.

var UTF8Decoder = typeof TextDecoder !== 'undefined' ? new TextDecoder('utf8') : undefined;
function UTF8ArrayToString(u8Array, idx) {
  var endPtr = idx;
  // TextDecoder needs to know the byte length in advance, it doesn't stop on null terminator by itself.
  // Also, use the length info to avoid running tiny strings through TextDecoder, since .subarray() allocates garbage.
  while (u8Array[endPtr]) ++endPtr;

  if (endPtr - idx > 16 && u8Array.subarray && UTF8Decoder) {
    return UTF8Decoder.decode(u8Array.subarray(idx, endPtr));
  } else {
    var u0, u1, u2, u3, u4, u5;

    var str = '';
    while (1) {
      // For UTF8 byte structure, see:
      // http://en.wikipedia.org/wiki/UTF-8#Description
      // https://www.ietf.org/rfc/rfc2279.txt
      // https://tools.ietf.org/html/rfc3629
      u0 = u8Array[idx++];
      if (!u0) return str;
      if (!(u0 & 0x80)) { str += String.fromCharCode(u0); continue; }
      u1 = u8Array[idx++] & 63;
      if ((u0 & 0xE0) == 0xC0) { str += String.fromCharCode(((u0 & 31) << 6) | u1); continue; }
      u2 = u8Array[idx++] & 63;
      if ((u0 & 0xF0) == 0xE0) {
        u0 = ((u0 & 15) << 12) | (u1 << 6) | u2;
      } else {
        u3 = u8Array[idx++] & 63;
        if ((u0 & 0xF8) == 0xF0) {
          u0 = ((u0 & 7) << 18) | (u1 << 12) | (u2 << 6) | u3;
        } else {
          u4 = u8Array[idx++] & 63;
          if ((u0 & 0xFC) == 0xF8) {
            u0 = ((u0 & 3) << 24) | (u1 << 18) | (u2 << 12) | (u3 << 6) | u4;
          } else {
            u5 = u8Array[idx++] & 63;
            u0 = ((u0 & 1) << 30) | (u1 << 24) | (u2 << 18) | (u3 << 12) | (u4 << 6) | u5;
          }
        }
      }
      if (u0 < 0x10000) {
        str += String.fromCharCode(u0);
      } else {
        var ch = u0 - 0x10000;
        str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
      }
    }
  }
}

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function UTF8ToString(ptr) {
  return UTF8ArrayToString(HEAPU8,ptr);
}

// Copies the given Javascript String object 'str' to the given byte array at address 'outIdx',
// encoded in UTF8 form and null-terminated. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8 to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outU8Array: the array to copy to. Each index in this array is assumed to be one 8-byte element.
//   outIdx: The starting offset in the array to begin the copying.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array.
//                    This count should include the null terminator,
//                    i.e. if maxBytesToWrite=1, only the null terminator will be written and nothing else.
//                    maxBytesToWrite=0 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8Array(str, outU8Array, outIdx, maxBytesToWrite) {
  if (!(maxBytesToWrite > 0)) // Parameter maxBytesToWrite is not optional. Negative values, 0, null, undefined and false each don't write out any bytes.
    return 0;

  var startIdx = outIdx;
  var endIdx = outIdx + maxBytesToWrite - 1; // -1 for string null terminator.
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    // For UTF8 byte structure, see http://en.wikipedia.org/wiki/UTF-8#Description and https://www.ietf.org/rfc/rfc2279.txt and https://tools.ietf.org/html/rfc3629
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) {
      var u1 = str.charCodeAt(++i);
      u = 0x10000 + ((u & 0x3FF) << 10) | (u1 & 0x3FF);
    }
    if (u <= 0x7F) {
      if (outIdx >= endIdx) break;
      outU8Array[outIdx++] = u;
    } else if (u <= 0x7FF) {
      if (outIdx + 1 >= endIdx) break;
      outU8Array[outIdx++] = 0xC0 | (u >> 6);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0xFFFF) {
      if (outIdx + 2 >= endIdx) break;
      outU8Array[outIdx++] = 0xE0 | (u >> 12);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0x1FFFFF) {
      if (outIdx + 3 >= endIdx) break;
      outU8Array[outIdx++] = 0xF0 | (u >> 18);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0x3FFFFFF) {
      if (outIdx + 4 >= endIdx) break;
      outU8Array[outIdx++] = 0xF8 | (u >> 24);
      outU8Array[outIdx++] = 0x80 | ((u >> 18) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else {
      if (outIdx + 5 >= endIdx) break;
      outU8Array[outIdx++] = 0xFC | (u >> 30);
      outU8Array[outIdx++] = 0x80 | ((u >> 24) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 18) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    }
  }
  // Null-terminate the pointer to the buffer.
  outU8Array[outIdx] = 0;
  return outIdx - startIdx;
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF8 form. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8 to compute the exact number of bytes (excluding null terminator) that this function will write.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8(str, outPtr, maxBytesToWrite) {
  return stringToUTF8Array(str, HEAPU8,outPtr, maxBytesToWrite);
}

// Returns the number of bytes the given Javascript string takes if encoded as a UTF8 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF8(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) u = 0x10000 + ((u & 0x3FF) << 10) | (str.charCodeAt(++i) & 0x3FF);
    if (u <= 0x7F) {
      ++len;
    } else if (u <= 0x7FF) {
      len += 2;
    } else if (u <= 0xFFFF) {
      len += 3;
    } else if (u <= 0x1FFFFF) {
      len += 4;
    } else if (u <= 0x3FFFFFF) {
      len += 5;
    } else {
      len += 6;
    }
  }
  return len;
}

// Given a pointer 'ptr' to a null-terminated UTF16LE-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

var UTF16Decoder = typeof TextDecoder !== 'undefined' ? new TextDecoder('utf-16le') : undefined;
function UTF16ToString(ptr) {
  var endPtr = ptr;
  // TextDecoder needs to know the byte length in advance, it doesn't stop on null terminator by itself.
  // Also, use the length info to avoid running tiny strings through TextDecoder, since .subarray() allocates garbage.
  var idx = endPtr >> 1;
  while (HEAP16[idx]) ++idx;
  endPtr = idx << 1;

  if (endPtr - ptr > 32 && UTF16Decoder) {
    return UTF16Decoder.decode(HEAPU8.subarray(ptr, endPtr));
  } else {
    var i = 0;

    var str = '';
    while (1) {
      var codeUnit = HEAP16[(((ptr)+(i*2))>>1)];
      if (codeUnit == 0) return str;
      ++i;
      // fromCharCode constructs a character from a UTF-16 code unit, so we can pass the UTF16 string right through.
      str += String.fromCharCode(codeUnit);
    }
  }
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF16 form. The copy will require at most str.length*4+2 bytes of space in the HEAP.
// Use the function lengthBytesUTF16() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=2, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<2 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF16(str, outPtr, maxBytesToWrite) {
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 2) return 0;
  maxBytesToWrite -= 2; // Null terminator.
  var startPtr = outPtr;
  var numCharsToWrite = (maxBytesToWrite < str.length*2) ? (maxBytesToWrite / 2) : str.length;
  for (var i = 0; i < numCharsToWrite; ++i) {
    // charCodeAt returns a UTF-16 encoded code unit, so it can be directly written to the HEAP.
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    HEAP16[((outPtr)>>1)]=codeUnit;
    outPtr += 2;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP16[((outPtr)>>1)]=0;
  return outPtr - startPtr;
}

// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF16(str) {
  return str.length*2;
}

function UTF32ToString(ptr) {
  var i = 0;

  var str = '';
  while (1) {
    var utf32 = HEAP32[(((ptr)+(i*4))>>2)];
    if (utf32 == 0)
      return str;
    ++i;
    // Gotcha: fromCharCode constructs a character from a UTF-16 encoded code (pair), not from a Unicode code point! So encode the code point to UTF-16 for constructing.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    if (utf32 >= 0x10000) {
      var ch = utf32 - 0x10000;
      str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
    } else {
      str += String.fromCharCode(utf32);
    }
  }
}

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF32 form. The copy will require at most str.length*4+4 bytes of space in the HEAP.
// Use the function lengthBytesUTF32() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null
//                    terminator, i.e. if maxBytesToWrite=4, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<4 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF32(str, outPtr, maxBytesToWrite) {
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 4) return 0;
  var startPtr = outPtr;
  var endPtr = startPtr + maxBytesToWrite - 4;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) {
      var trailSurrogate = str.charCodeAt(++i);
      codeUnit = 0x10000 + ((codeUnit & 0x3FF) << 10) | (trailSurrogate & 0x3FF);
    }
    HEAP32[((outPtr)>>2)]=codeUnit;
    outPtr += 4;
    if (outPtr + 4 > endPtr) break;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP32[((outPtr)>>2)]=0;
  return outPtr - startPtr;
}

// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF32(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i);
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) ++i; // possibly a lead surrogate, so skip over the tail surrogate.
    len += 4;
  }

  return len;
}

// Allocate heap space for a JS string, and write it there.
// It is the responsibility of the caller to free() that memory.
function allocateUTF8(str) {
  var size = lengthBytesUTF8(str) + 1;
  var ret = _malloc(size);
  if (ret) stringToUTF8Array(str, HEAP8, ret, size);
  return ret;
}

// Allocate stack space for a JS string, and write it there.
function allocateUTF8OnStack(str) {
  var size = lengthBytesUTF8(str) + 1;
  var ret = stackAlloc(size);
  stringToUTF8Array(str, HEAP8, ret, size);
  return ret;
}

function demangle(func) {
  return func;
}

function demangleAll(text) {
  var regex =
    /__Z[\w\d_]+/g;
  return text.replace(regex,
    function(x) {
      var y = demangle(x);
      return x === y ? x : (x + ' [' + y + ']');
    });
}

function jsStackTrace() {
  var err = new Error();
  if (!err.stack) {
    // IE10+ special cases: It does have callstack info, but it is only populated if an Error object is thrown,
    // so try that as a special-case.
    try {
      throw new Error(0);
    } catch(e) {
      err = e;
    }
    if (!err.stack) {
      return '(no stack trace available)';
    }
  }
  return err.stack.toString();
}

function stackTrace() {
  var js = jsStackTrace();
  if (Module['extraStackTrace']) js += '\n' + Module['extraStackTrace']();
  return demangleAll(js);
}

// Memory management

var PAGE_SIZE = 16384;
var WASM_PAGE_SIZE = 65536;
var ASMJS_PAGE_SIZE = 16777216;
var MIN_TOTAL_MEMORY = 16777216;

function alignUp(x, multiple) {
  if (x % multiple > 0) {
    x += multiple - (x % multiple);
  }
  return x;
}

var HEAP,
/** @type {ArrayBuffer} */
  buffer,
/** @type {Int8Array} */
  HEAP8,
/** @type {Uint8Array} */
  HEAPU8,
/** @type {Int16Array} */
  HEAP16,
/** @type {Uint16Array} */
  HEAPU16,
/** @type {Int32Array} */
  HEAP32,
/** @type {Uint32Array} */
  HEAPU32,
/** @type {Float32Array} */
  HEAPF32,
/** @type {Float64Array} */
  HEAPF64;

function updateGlobalBuffer(buf) {
  Module['buffer'] = buffer = buf;
}

function updateGlobalBufferViews() {
  Module['HEAP8'] = HEAP8 = new Int8Array(buffer);
  Module['HEAP16'] = HEAP16 = new Int16Array(buffer);
  Module['HEAP32'] = HEAP32 = new Int32Array(buffer);
  Module['HEAPU8'] = HEAPU8 = new Uint8Array(buffer);
  Module['HEAPU16'] = HEAPU16 = new Uint16Array(buffer);
  Module['HEAPU32'] = HEAPU32 = new Uint32Array(buffer);
  Module['HEAPF32'] = HEAPF32 = new Float32Array(buffer);
  Module['HEAPF64'] = HEAPF64 = new Float64Array(buffer);
}

var STATIC_BASE, STATICTOP, staticSealed; // static area
var STACK_BASE, STACKTOP, STACK_MAX; // stack area
var DYNAMIC_BASE, DYNAMICTOP_PTR; // dynamic area handled by sbrk

  STATIC_BASE = STATICTOP = STACK_BASE = STACKTOP = STACK_MAX = DYNAMIC_BASE = DYNAMICTOP_PTR = 0;
  staticSealed = false;




function abortOnCannotGrowMemory() {
  abort('Cannot enlarge memory arrays. Either (1) compile with  -s TOTAL_MEMORY=X  with X higher than the current value ' + TOTAL_MEMORY + ', (2) compile with  -s ALLOW_MEMORY_GROWTH=1  which allows increasing the size at runtime but prevents some optimizations, (3) set Module.TOTAL_MEMORY to a higher value before the program runs, or (4) if you want malloc to return NULL (0) instead of this abort, compile with  -s ABORTING_MALLOC=0 ');
}

if (!Module['reallocBuffer']) Module['reallocBuffer'] = function(size) {
  var ret;
  try {
    var oldHEAP8 = HEAP8;
    ret = new ArrayBuffer(size);
    var temp = new Int8Array(ret);
    temp.set(oldHEAP8);
  } catch(e) {
    return false;
  }
  var success = _emscripten_replace_memory(ret);
  if (!success) return false;
  return ret;
};

function enlargeMemory() {
  // TOTAL_MEMORY is the current size of the actual array, and DYNAMICTOP is the new top.


  var PAGE_MULTIPLE = Module["usingWasm"] ? WASM_PAGE_SIZE : ASMJS_PAGE_SIZE; // In wasm, heap size must be a multiple of 64KB. In asm.js, they need to be multiples of 16MB.
  var LIMIT = 2147483648 - PAGE_MULTIPLE; // We can do one page short of 2GB as theoretical maximum.

  if (HEAP32[DYNAMICTOP_PTR>>2] > LIMIT) {
    return false;
  }

  var OLD_TOTAL_MEMORY = TOTAL_MEMORY;
  TOTAL_MEMORY = Math.max(TOTAL_MEMORY, MIN_TOTAL_MEMORY); // So the loop below will not be infinite, and minimum asm.js memory size is 16MB.

  while (TOTAL_MEMORY < HEAP32[DYNAMICTOP_PTR>>2]) { // Keep incrementing the heap size as long as it's less than what is requested.
    if (TOTAL_MEMORY <= 536870912) {
      TOTAL_MEMORY = alignUp(2 * TOTAL_MEMORY, PAGE_MULTIPLE); // Simple heuristic: double until 1GB...
    } else {
      // ..., but after that, add smaller increments towards 2GB, which we cannot reach
      TOTAL_MEMORY = Math.min(alignUp((3 * TOTAL_MEMORY + 2147483648) / 4, PAGE_MULTIPLE), LIMIT);
    }
  }



  var replacement = Module['reallocBuffer'](TOTAL_MEMORY);
  if (!replacement || replacement.byteLength != TOTAL_MEMORY) {
    // restore the state to before this call, we failed
    TOTAL_MEMORY = OLD_TOTAL_MEMORY;
    return false;
  }

  // everything worked

  updateGlobalBuffer(replacement);
  updateGlobalBufferViews();



  return true;
}

var byteLength;
try {
  byteLength = Function.prototype.call.bind(Object.getOwnPropertyDescriptor(ArrayBuffer.prototype, 'byteLength').get);
  byteLength(new ArrayBuffer(4)); // can fail on older ie
} catch(e) { // can fail on older node/v8
  byteLength = function(buffer) { return buffer.byteLength; };
}

var TOTAL_STACK = Module['TOTAL_STACK'] || 5242880;
var TOTAL_MEMORY = Module['TOTAL_MEMORY'] || 16777216;
if (TOTAL_MEMORY < TOTAL_STACK) err('TOTAL_MEMORY should be larger than TOTAL_STACK, was ' + TOTAL_MEMORY + '! (TOTAL_STACK=' + TOTAL_STACK + ')');

// Initialize the runtime's memory



// Use a provided buffer, if there is one, or else allocate a new one
if (Module['buffer']) {
  buffer = Module['buffer'];
} else {
  // Use a WebAssembly memory where available
  {
    buffer = new ArrayBuffer(TOTAL_MEMORY);
  }
  Module['buffer'] = buffer;
}
updateGlobalBufferViews();


function getTotalMemory() {
  return TOTAL_MEMORY;
}

// Endianness check (note: assumes compiler arch was little-endian)

function callRuntimeCallbacks(callbacks) {
  while(callbacks.length > 0) {
    var callback = callbacks.shift();
    if (typeof callback == 'function') {
      callback();
      continue;
    }
    var func = callback.func;
    if (typeof func === 'number') {
      if (callback.arg === undefined) {
        Module['dynCall_v'](func);
      } else {
        Module['dynCall_vi'](func, callback.arg);
      }
    } else {
      func(callback.arg === undefined ? null : callback.arg);
    }
  }
}

var __ATPRERUN__  = []; // functions called before the runtime is initialized
var __ATINIT__    = []; // functions called during startup
var __ATMAIN__    = []; // functions called when main() is to be run
var __ATEXIT__    = []; // functions called during shutdown
var __ATPOSTRUN__ = []; // functions called after the main() is called

var runtimeInitialized = false;
var runtimeExited = false;


function preRun() {
  // compatibility - merge in anything from Module['preRun'] at this time
  if (Module['preRun']) {
    if (typeof Module['preRun'] == 'function') Module['preRun'] = [Module['preRun']];
    while (Module['preRun'].length) {
      addOnPreRun(Module['preRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPRERUN__);
}

function ensureInitRuntime() {
  if (runtimeInitialized) return;
  runtimeInitialized = true;
  callRuntimeCallbacks(__ATINIT__);
}

function preMain() {
  callRuntimeCallbacks(__ATMAIN__);
}

function exitRuntime() {
  callRuntimeCallbacks(__ATEXIT__);
  runtimeExited = true;
}

function postRun() {
  // compatibility - merge in anything from Module['postRun'] at this time
  if (Module['postRun']) {
    if (typeof Module['postRun'] == 'function') Module['postRun'] = [Module['postRun']];
    while (Module['postRun'].length) {
      addOnPostRun(Module['postRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPOSTRUN__);
}

function addOnPreRun(cb) {
  __ATPRERUN__.unshift(cb);
}

function addOnInit(cb) {
  __ATINIT__.unshift(cb);
}

function addOnPreMain(cb) {
  __ATMAIN__.unshift(cb);
}

function addOnExit(cb) {
  __ATEXIT__.unshift(cb);
}

function addOnPostRun(cb) {
  __ATPOSTRUN__.unshift(cb);
}

// Deprecated: This function should not be called because it is unsafe and does not provide
// a maximum length limit of how many bytes it is allowed to write. Prefer calling the
// function stringToUTF8Array() instead, which takes in a maximum length that can be used
// to be secure from out of bounds writes.
/** @deprecated */
function writeStringToMemory(string, buffer, dontAddNull) {
  warnOnce('writeStringToMemory is deprecated and should not be called! Use stringToUTF8() instead!');

  var /** @type {number} */ lastChar, /** @type {number} */ end;
  if (dontAddNull) {
    // stringToUTF8Array always appends null. If we don't want to do that, remember the
    // character that existed at the location where the null will be placed, and restore
    // that after the write (below).
    end = buffer + lengthBytesUTF8(string);
    lastChar = HEAP8[end];
  }
  stringToUTF8(string, buffer, Infinity);
  if (dontAddNull) HEAP8[end] = lastChar; // Restore the value under the null character.
}

function writeArrayToMemory(array, buffer) {
  HEAP8.set(array, buffer);
}

function writeAsciiToMemory(str, buffer, dontAddNull) {
  for (var i = 0; i < str.length; ++i) {
    HEAP8[((buffer++)>>0)]=str.charCodeAt(i);
  }
  // Null-terminate the pointer to the HEAP.
  if (!dontAddNull) HEAP8[((buffer)>>0)]=0;
}

function unSign(value, bits, ignore) {
  if (value >= 0) {
    return value;
  }
  return bits <= 32 ? 2*Math.abs(1 << (bits-1)) + value // Need some trickery, since if bits == 32, we are right at the limit of the bits JS uses in bitshifts
                    : Math.pow(2, bits)         + value;
}
function reSign(value, bits, ignore) {
  if (value <= 0) {
    return value;
  }
  var half = bits <= 32 ? Math.abs(1 << (bits-1)) // abs is needed if bits == 32
                        : Math.pow(2, bits-1);
  if (value >= half && (bits <= 32 || value > half)) { // for huge values, we can hit the precision limit and always get true here. so don't do that
                                                       // but, in general there is no perfect solution here. With 64-bit ints, we get rounding and errors
                                                       // TODO: In i64 mode 1, resign the two parts separately and safely
    value = -2*half + value; // Cannot bitshift half, as it may be at the limit of the bits JS uses in bitshifts
  }
  return value;
}

// check for imul support, and also for correctness ( https://bugs.webkit.org/show_bug.cgi?id=126345 )
if (!Math['imul'] || Math['imul'](0xffffffff, 5) !== -5) Math['imul'] = function imul(a, b) {
  var ah  = a >>> 16;
  var al = a & 0xffff;
  var bh  = b >>> 16;
  var bl = b & 0xffff;
  return (al*bl + ((ah*bl + al*bh) << 16))|0;
};
Math.imul = Math['imul'];


if (!Math['clz32']) Math['clz32'] = function(x) {
  var n = 32;
  var y = x >> 16; if (y) { n -= 16; x = y; }
  y = x >> 8; if (y) { n -= 8; x = y; }
  y = x >> 4; if (y) { n -= 4; x = y; }
  y = x >> 2; if (y) { n -= 2; x = y; }
  y = x >> 1; if (y) return n - 2;
  return n - x;
};
Math.clz32 = Math['clz32']

if (!Math['trunc']) Math['trunc'] = function(x) {
  return x < 0 ? Math.ceil(x) : Math.floor(x);
};
Math.trunc = Math['trunc'];

var Math_abs = Math.abs;
var Math_cos = Math.cos;
var Math_sin = Math.sin;
var Math_tan = Math.tan;
var Math_acos = Math.acos;
var Math_asin = Math.asin;
var Math_atan = Math.atan;
var Math_atan2 = Math.atan2;
var Math_exp = Math.exp;
var Math_log = Math.log;
var Math_sqrt = Math.sqrt;
var Math_ceil = Math.ceil;
var Math_floor = Math.floor;
var Math_pow = Math.pow;
var Math_imul = Math.imul;
var Math_fround = Math.fround;
var Math_round = Math.round;
var Math_min = Math.min;
var Math_max = Math.max;
var Math_clz32 = Math.clz32;
var Math_trunc = Math.trunc;

// A counter of dependencies for calling run(). If we need to
// do asynchronous work before running, increment this and
// decrement it. Incrementing must happen in a place like
// Module.preRun (used by emcc to add file preloading).
// Note that you can add dependencies in preRun, even though
// it happens right before run - run will be postponed until
// the dependencies are met.
var runDependencies = 0;
var runDependencyWatcher = null;
var dependenciesFulfilled = null; // overridden to take different actions when all run dependencies are fulfilled

function getUniqueRunDependency(id) {
  return id;
}

function addRunDependency(id) {
  runDependencies++;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
}

function removeRunDependency(id) {
  runDependencies--;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (runDependencies == 0) {
    if (runDependencyWatcher !== null) {
      clearInterval(runDependencyWatcher);
      runDependencyWatcher = null;
    }
    if (dependenciesFulfilled) {
      var callback = dependenciesFulfilled;
      dependenciesFulfilled = null;
      callback(); // can add another dependenciesFulfilled
    }
  }
}

Module["preloadedImages"] = {}; // maps url to image data
Module["preloadedAudios"] = {}; // maps url to audio data



var memoryInitializer = null;






// Prefix of data URIs emitted by SINGLE_FILE and related options.
var dataURIPrefix = 'data:application/octet-stream;base64,';

// Indicates whether filename is a base64 data URI.
function isDataURI(filename) {
  return String.prototype.startsWith ?
      filename.startsWith(dataURIPrefix) :
      filename.indexOf(dataURIPrefix) === 0;
}





// === Body ===

var ASM_CONSTS = [function($0, $1) { console.log('dtParametricPolygon', $0, $1) },
 function($0) { console.log('InsertParametricObject: unsupported parametric object of type', $0) },
 function($0) { console.log('InsertParametricPathSegment: unsupported path segment of type', $0) },
 function($0) { console.log('InsertParametricTextContent: unsupported text content of type', $0) },
 function($0) { console.log('RealiseContent: unsupported object of type', $0) },
 function($0, $1, $2) { console.log('geometry: failed ' + $0 + '-' + $1 + '-' + $2) },
 function($0, $1, $2) { console.log('image: failed ' + $0 + '-' + $1 + '-' + $2) },
 function($0, $1, $2) { console.log('metadata: failed ' + $0 + '-' + $1 + '-' + $2) },
 function($0, $1, $2, $3) { console.log('metadata for ' + $0 + '-' + $1 + '-' + $2 + ': ' + Pointer_stringify($3)) }];

function _emscripten_asm_const_ii(code, a0) {
  return ASM_CONSTS[code](a0);
}

function _emscripten_asm_const_idi(code, a0, a1) {
  return ASM_CONSTS[code](a0, a1);
}

function _emscripten_asm_const_iiii(code, a0, a1, a2) {
  return ASM_CONSTS[code](a0, a1, a2);
}

function _emscripten_asm_const_iiiii(code, a0, a1, a2, a3) {
  return ASM_CONSTS[code](a0, a1, a2, a3);
}




STATIC_BASE = GLOBAL_BASE;

STATICTOP = STATIC_BASE + 155648;
/* global initializers */  __ATINIT__.push({ func: function() { __GLOBAL__sub_I_ChangeManager_cpp() } }, { func: function() { __GLOBAL__sub_I_GraphSelectorResolver_cpp() } }, { func: function() { __GLOBAL__sub_I_Math_cpp() } }, { func: function() { __GLOBAL__sub_I_MathConstants_cpp() } }, { func: function() { __GLOBAL__sub_I_DataModelSQLiteReader_cpp() } }, { func: function() { __GLOBAL__sub_I_ZIPVFSCompression_cpp() } }, { func: function() { ___emscripten_environ_constructor() } });


memoryInitializer = "data:application/octet-stream;base64,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";





/* no memory initializer */
var tempDoublePtr = STATICTOP; STATICTOP += 16;

function copyTempFloat(ptr) { // functions, because inlining this code increases code size too much

  HEAP8[tempDoublePtr] = HEAP8[ptr];

  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];

  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];

  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];

}

function copyTempDouble(ptr) {

  HEAP8[tempDoublePtr] = HEAP8[ptr];

  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];

  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];

  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];

  HEAP8[tempDoublePtr+4] = HEAP8[ptr+4];

  HEAP8[tempDoublePtr+5] = HEAP8[ptr+5];

  HEAP8[tempDoublePtr+6] = HEAP8[ptr+6];

  HEAP8[tempDoublePtr+7] = HEAP8[ptr+7];

}

// {{PRE_LIBRARY}}


  
  var SB={init:function () {
  			var nextSceneBuilderId = 1;
  			var idToSceneBuilder = new Map();
  			var sceneBuilderToId = new Map();
  			var shapes = [];
  			var pathSegments = [];
  			var textContent;
  			var textContents = [];
  			var lineStyle;
  			var fillStyle;
  
  			SB.add = function(sceneBuilder) {
  				idToSceneBuilder.set(nextSceneBuilderId, sceneBuilder);
  				sceneBuilderToId.set(sceneBuilder, nextSceneBuilderId);
  				return nextSceneBuilderId++;
  			};
  
  			SB.delete = function(sceneBuilder) {
  				idToSceneBuilder.delete(sceneBuilderToId.get(sceneBuilder));
  				sceneBuilderToId.delete(sceneBuilder);
  			};
  
  			SB.getById = function(id) {
  				return idToSceneBuilder.get(id);
  			};
  
  			SB.addShape = function(shape) {
  				if (lineStyle) {
  					shape.lineStyle = lineStyle;
  				}
  				if (fillStyle) {
  					shape.fillStyle = fillStyle;
  				}
  				shapes.push(shape);
  			};
  
  			SB.flushShapes = function() {
  				var result = shapes;
  				shapes = [];
  				return result;
  			};
  
  			SB.addPathSegment = function(shape) {
  				pathSegments.push(shape);
  			};
  
  			SB.flushPathSegments = function() {
  				var result = pathSegments;
  				pathSegments = [];
  				return result;
  			};
  
  			SB.setLineStyle = function(style) {
  				lineStyle = style;
  			};
  
  			SB.getLineStyle = function(style) {
  				return lineStyle;
  			};
  
  			SB.setFillStyle = function(style) {
  				fillStyle = style;
  			};
  
  			SB.getFillStyle = function(style) {
  				return fillStyle;
  			};
  
  			SB.beginParametricTextContent = function(newContext, type) {
  				if (textContents.length > 0) {
  					textContents[ textContents.length - 1 ].content.push(newContext);
  				}
  				textContents.push(newContext);
  			};
  
  			SB.insertParametricTextData = function(textData) {
  				textContents[ textContents.length - 1 ].content.push(textData);
  			};
  
  			SB.endParametricTextContent = function() {
  				textContent = textContents.pop();
  				if (pathSegments.length > 0) {
  					textContent.pathSegments = SB.flushPathSegments();
  				}
  			};
  
  			SB.flushTextContent = function() {
  				textContents = [];
  				return textContent;
  			};
  		}};function _SceneBuilderAdapter_beginParametricTextContent(
  		contentSubtype,
  		size,
  		horizontalAlign,
  		dominantBaseline,
  		linePitch,
  		widthScale,
  		fontFace,
  		isSHXFont,
  		italic,
  		bold,
  		underline,
  		hasOverline,
  		hasStrikethrough,
  		expand,
  		textHeightAs,
  		direction,
  		upsideDown,
  		backward,
  		obliquingAngle,
  		textLength,
  		lengthUnit,
  		lengthAdjust) {
  		var style = {
  			size: size,
  			horizontalAlign: horizontalAlign,
  			dominantBaseline: dominantBaseline,
  			linePitch: linePitch,
  			widthScale: widthScale,
  			fontFace: UTF8ToString(fontFace),
  			isSHXFont: !!isSHXFont,
  			italic: !!italic,
  			bold: !!bold,
  			underline: !!underline,
  			hasOverline: !!hasOverline,
  			hasStrikethrough: !!hasStrikethrough,
  			expand: expand,
  			textHeightAs: textHeightAs,
  			direction: direction,
  			upsideDown: !!upsideDown,
  			backward: !!backward,
  			obliquingAngle: obliquingAngle,
  			textLength: getArrayF32(textLength, 2),
  			lengthUnit: lengthUnit,
  			lengthAdjust: lengthAdjust
  		};
  
  		var lineStyle = SB.getLineStyle();
  		if (lineStyle) {
  			style.lineStyle = lineStyle;
  		}
  
  		var fillStyle = SB.getFillStyle();
  		if (fillStyle) {
  			style.fillStyle = fillStyle;
  		}
  
  		SB.beginParametricTextContent({
  			type: contentSubtype,
  			style: style,
  			content: []
  		});
  	}

  function _SceneBuilderAdapter_clearFillStyle() {
  		SB.setFillStyle();
  	}

  function _SceneBuilderAdapter_clearLineStyle() {
  		SB.setLineStyle();
  	}

  function _SceneBuilderAdapter_createCamera(
  		sceneBuilderId,
  		matrix,
  		isOrtho,
  		aspect,
  		autoEvaluateClipPlanes,
  		nearClip,
  		farClip,
  		fov,
  		fovBinding,
  		orthoZoomFactor,
  		origin,
  		up,
  		target,
  		cameraRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createCamera({
  			matrix: matrix ? getArrayF64(matrix, 16) : null,
  			projection: isOrtho ? "orthographic" : "perspective",
  			aspect: aspect,
  			autoEvaluateClipPlanes: !!autoEvaluateClipPlanes,
  			nearClip: nearClip,
  			farClip: farClip,
  			fov: fov,
  			fovBinding: fovBinding,
  			orthoZoomFactor: orthoZoomFactor,
  			origin: getArrayF32(origin, 3),
  			up: getArrayF32(up, 3),
  			target: getArrayF32(target, 3),
  			cameraRef: cameraRef
  		});
  	}

  function _SceneBuilderAdapter_createDefaultMaterial(
  		sceneBuilderId
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createMaterial({
  			name: "",
  			ambient: new Float32Array([0.12, 0.12, 0.12]),
  			diffuse: new Float32Array([0.75, 0.75, 0.75]),
  			specular: new Float32Array([0.75, 0.75, 0.75]),
  			emissive: new Float32Array([0, 0, 0]),
  			opacity: 1,
  			glossiness: 0.2,
  			specularLevel: 1,
  			materialRef: 0
  		});
  	}

  function _SceneBuilderAdapter_createDetailView(
  		sceneBuilderId,
  		nodeRef,
  		detailViewRef,
  		name,
  		description,
  		cameraRef,
  		type,
  		shape,
  		borderWidth,
  		backgroundColor,
  		borderColor,
  		renderOrder,
  		origin,
  		size,
  		attachmentPoint,
  		metadata,
  		veids,
  		visibleNodeCount,
  		visibleNodeRefs,
  		targetNodeCount,
  		targetNodeRefs
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createDetailView({
  			nodeRef: nodeRef,
  			detailViewRef: detailViewRef,
  			name: UTF8ToString(name),
  			description: UTF8ToString(description),
  			cameraRef: cameraRef,
  			type: type,
  			shape: shape,
  			borderWidth: borderWidth,
  			backgroundColor: backgroundColor,
  			borderColor: borderColor,
  			renderOrder: renderOrder,
  			origin: getArrayF32(origin, 2),
  			size: getArrayF32(size, 2),
  			attachmentPoint: attachmentPoint ? getArrayF32(attachmentPoint, 3) : null,
  			metadata: metadata ? JSON.parse(UTF8ToString(metadata)) : {},
  			veids: veids ? JSON.parse(UTF8ToString(veids)) : {},
  			visibleNodes: getArrayU32(visibleNodeRefs, visibleNodeCount),
  			targetNodes: getArrayU32(targetNodeRefs, targetNodeCount)
  		});
  	}

  function _SceneBuilderAdapter_createImage(
  		sceneBuilderId,
  		imageRef,
  		path,
  		width,
  		height,
  		colorDepth,
  		format,
  		data,
  		dataSize
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createImage({
  			imageRef: imageRef,
  			path: UTF8ToString(path),
  			width: width,
  			height: height,
  			colorDepth: colorDepth,
  			format: imageFormat[format],	// TODO: move imageFormat into this library $SB?
  			data: new Uint8Array(Module.buffer, data, dataSize).slice()
  		});
  	}

  function _SceneBuilderAdapter_createImageNote(
  		sceneBuilderId,
  		nodeRef,
  		position,
  		width,
  		height,
  		materialRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var info = {
  			nodeRef: nodeRef,
  			position: getArrayF32(position, 3),
  			width: width,
  			height: height,
  			materialRef: materialRef
  		};
  
  		builder.createImageNote(info);
  	}

  function _SceneBuilderAdapter_createMaterial(
  		sceneBuilderId,
  		name,
  		ambient,
  		diffuse,
  		specular,
  		emissive,
  		opacity,
  		glossiness,
  		specularLevel,
  		colourMapEnabled,
  		width,
  		haloWidth,
  		color,
  		endCapStyle,
  		dashPatternScale,
  		dashPattern,
  		dashPatternCount,
  		widthCoordinateSpace,
  		textureCount,
  		texturesBuffer,
  		materialRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var info = {
  			name: UTF8ToString(name),
  			ambient: getArrayF32(ambient, 4),
  			diffuse: getArrayF32(diffuse, 4),
  			specular: getArrayF32(specular, 4),
  			emissive: getArrayF32(emissive, 4),
  			opacity: opacity,
  			glossiness: glossiness,
  			specularLevel: specularLevel,
  			colourMapEnabled: colourMapEnabled,
  			linestyle: {
  				width: width,
  				haloWidth: haloWidth,
  				color: getArrayF32(color, 4),
  				endCapStyle: endCapStyle,
  				dashPatternScale: dashPatternScale,
  				dashPattern: getArrayF32(dashPattern, dashPatternCount),
  				widthCoordinateSpace: widthCoordinateSpace,
  			},
  			materialRef: materialRef
  		};
  
  		if (textureCount > 0) {
  			info.textures = [];
  			var p = texturesBuffer;
  			for (var i = 0; i < textureCount; i++, p += 56) {
  				var flags = getValue(p + 54, "i16");
  				info.textures.push({
  					imageRef: getValue(p, "i32"),
  					matrix: getArrayF32(p + 4, 6),
  					scaleX: getValue(p + 28, "float"),
  					scaleY: getValue(p + 32, "float"),
  					offsetX: getValue(p + 36, "float"),
  					offsetY: getValue(p + 40, "float"),
  					angle: getValue(p + 44, "float"),
  					amount: getValue(p + 48, "float"),
  					type: textureType[ getValue(p + 52, "i8") ],	// TODO: move textureType to this library.
  					filterMode: getValue(p + 53, "i8"),
  					repeatX: (flags & 1) !== 0,
  					repeatY: (flags & 2) !== 0,
  					clampToBorder: (flags & 3) !== 0,
  					invert: (flags & 4) !== 0,
  					modulate: (flags & 5) !== 0
  				});
  			}
  		}
  		builder.createMaterial(info);
  	}

  function _SceneBuilderAdapter_createMesh(
  		sceneBuilderId,
  		materialRef,
  		boundingBox,
  		matrix,
  		meshRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var info = {
  			materialRef: materialRef,
  			boundingBox: getArrayF32(boundingBox, 6),
  			matrix: matrix ? getArrayF64(matrix, 16) : null,
  			meshRef: meshRef
  		};
  
  		builder.createMesh(info);
  	}

  function _SceneBuilderAdapter_createNode(
  		sceneBuilderId,
  		parentRef,
  		name,
  		matrix,
  		opacity,
  		visible,
  		closed,
  		selectable,
  		visualisable,
  		transformType,
  		renderOrder,
  		renderMethod,
  		renderStage,
  		metadata,
  		veids,
  		nodeRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createNode({
  			parentRef: parentRef,
  			name: UTF8ToString(name),
  			matrix: getArrayF64(matrix, 16),
  			opacity: opacity,
  			visible: !!visible,
  			closed: !!closed,
  			selectable: !!selectable,
  			visualisable: !!visualisable,
  			transformType: transformType,
  			renderOrder: renderOrder,
  			renderMethod: renderMethod,
  			renderStage: renderStage,
  			metadata: JSON.parse(UTF8ToString(metadata)),
  			veids: JSON.parse(UTF8ToString(veids)),
  			nodeRef: nodeRef
  		});
  	}

  function _SceneBuilderAdapter_createTextAnnotation(
  		sceneBuilderId,
  		nodeRef,
  		position,
  		encoding,
  		text,
  		font,
  		fontSize,
  		fontWeight,
  		fontItalic,
  		width,
  		height,
  		style,
  		textColor,
  		backgroundColor,
  		backgroundOpacity,
  		borderWidth,
  		borderLineStyle,
  		borderColor,
  		borderOpacity,
  		horizontalAlignment,
  		verticalAlignment,
  		link
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var info = {
  			nodeRef: nodeRef,
  			position: getArrayF32(position, 3),
  			encoding: encoding,
  			text: UTF8ToString(text),
  			font: UTF8ToString(font),
  			fontSize: fontSize,
  			fontWeight: fontWeight,
  			fontItalic: !!fontItalic,
  			width: width,
  			height: height,
  			style: style,
  			textColor: textColor,
  			backgroundColor: backgroundColor,
  			backgroundOpacity: backgroundOpacity,
  			borderWidth: borderWidth,
  			borderLineStyle: borderLineStyle,
  			borderColor: borderColor,
  			borderOpacity: borderOpacity,
  			horizontalAlignment: horizontalAlignment,
  			verticalAlignment: verticalAlignment,
  			link: UTF8ToString(link)
  		};
  
  		builder.createTextAnnotation(info);
  	}

  function _SceneBuilderAdapter_createTextNote(
  		sceneBuilderId,
  		nodeRef,
  		targetNodeRef,
  		position,
  		alwaysOnTop,
  		encoding,
  		text,
  		font,
  		fontSize,
  		fontWeight,
  		fontItalic,
  		width,
  		height,
  		style,
  		textColor,
  		backgroundColor,
  		backgroundOpacity,
  		borderWidth,
  		borderLineStyle,
  		borderColor,
  		borderOpacity,
  		horizontalAlignment,
  		verticalAlignment,
  		link
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var info = {
  			nodeRef: nodeRef,
  			targetNodeRef: targetNodeRef,
  			position: getArrayF32(position, 3),
  			alwaysOnTop: !!alwaysOnTop,
  			encoding: encoding,
  			text: UTF8ToString(text),
  			font: UTF8ToString(font),
  			fontSize: fontSize,
  			fontWeight: fontWeight,
  			fontItalic: !!fontItalic,
  			width: width,
  			height: height,
  			style: style,
  			textColor: textColor,
  			backgroundColor: backgroundColor,
  			backgroundOpacity: backgroundOpacity,
  			borderWidth: borderWidth,
  			borderLineStyle: borderLineStyle,
  			borderColor: borderColor,
  			borderOpacity: borderOpacity,
  			horizontalAlignment: horizontalAlignment,
  			verticalAlignment: verticalAlignment,
  			link: UTF8ToString(link)
  		};
  
  		builder.createTextNote(info);
  	}

  function _SceneBuilderAdapter_createThumbnail(
  		sceneBuilderId,
  		imageRef,
  		path,
  		width,
  		height,
  		format,
  		data,
  		dataSize
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createThumbnail({
  			imageRef: imageRef,
  			path: UTF8ToString(path),
  			width: width,
  			height: height,
  			format: imageFormat[format],
  			data: new Uint8Array(Module.buffer, data, dataSize).slice()
  		});
  	}

  function _SceneBuilderAdapter_createViewportGroup(
  		sceneBuilderId,
  		viewportGroupRef,
  		name,
  		description,
  		type,
  		metadata,
  		veids
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.createViewportGroup({
  			viewportGroupRef: viewportGroupRef,
  			name: UTF8ToString(name),
  			description: UTF8ToString(description),
  			type: type,
  			metadata: metadata ? JSON.parse(UTF8ToString(metadata)) : {},
  			veids: veids ? JSON.parse(UTF8ToString(veids)) : {}
  		});
  	}

  function _SceneBuilderAdapter_endParametricTextContent() {
  		SB.endParametricTextContent();
  	}

  function _SceneBuilderAdapter_finalizeAnimation(
  		sceneBuilderId,
  		animationRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  
  		builder.finalizeAnimation(animationRef);
  	}

  function _SceneBuilderAdapter_finalizePlaybacks(
  		sceneBuilderId,
  		viewportGroupRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.finalizePlaybacks({
  			viewportGroupRef: viewportGroupRef
  		});
  	}

  function _SceneBuilderAdapter_insertAnimation(
  		sceneBuilderId,
  		animationGroupRef,
  		animationRef,
  		animationType
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertAnimation({
  			animationGroupRef: animationGroupRef,
  			animationRef: animationRef,
  			animationType: animationType
  		});
  	}

  function _SceneBuilderAdapter_insertAnimationGroup(
  		sceneBuilderId,
  		modelViewRef,
  		animationGroupRef,
  		startTime,
  		endTime,
  		frameRate,
  		active,
  		cyclic,
  		currentTime,
  		name,
  		playbackSpeed,
  		playbackPreDelay,
  		playbackPostDelay,
  		playbackRepeat,
  		playbackReversed
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertAnimationGroup({
  			modelViewRef: modelViewRef,
  			animationGroupRef: animationGroupRef,
  			startTime: startTime,
  			endTime: endTime,
  			frameRate: frameRate,
  			active: active,
  			cyclic: cyclic,
  			currentTime: currentTime,
  			name: UTF8ToString(name),
  			playbackSpeed: playbackSpeed,
  			playbackPreDelay: playbackPreDelay,
  			playbackPostDelay: playbackPostDelay,
  			playbackRepeat: playbackRepeat,
  			playbackReversed: playbackReversed
  		});
  	}

  function _SceneBuilderAdapter_insertAnimationTarget(
  		sceneBuilderId,
  		animationRef,
  		targetRef,
  		targetPivotX,
  		targetPivotY,
  		targetPivotZ
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertAnimationTarget({
  			animationRef: animationRef,
  			targetRef: targetRef,
  			targetPivotX: targetPivotX,
  			targetPivotY: targetPivotY,
  			targetPivotZ: targetPivotZ
  		});
  	}

  function _SceneBuilderAdapter_insertAnimationTrack(
  		sceneBuilderId,
  		animationRef,
  		animationTrackRef,
  		type,
  		dataType,
  		interpolationType,
  		cyclicStart,
  		cyclicEnd,
  		keyCount,
  		valueCount,
  		times,
  		tensions,
  		continuities,
  		biases,
  		easeIns,
  		easeOuts,
  		values
  	) {
  		var builder = SB.getById(sceneBuilderId);
  
  		builder.insertAnimationTrack({
  			animationRef: animationRef,
  			animationTrackRef: animationTrackRef,
  			type: type,
  			dataType: dataType,
  			interpolationType: interpolationType,
  			cyclicStart: cyclicStart,
  			cyclicEnd: cyclicEnd,
  			keyCount: keyCount,
  			valueCount: valueCount,
  			times: getArrayF32(times, keyCount),
  			tensions: getArrayF32(tensions, keyCount),
  			continuities: getArrayF32(continuities, keyCount),
  			biases: getArrayF32(biases, keyCount),
  			easeIns: getArrayF32(easeIns, keyCount),
  			easeOuts: getArrayF32(easeOuts, keyCount),
  			values: getArrayF32(values, valueCount)
  		});
  	}

  function _SceneBuilderAdapter_insertArcPathSegment(
  		major,
  		minor,
  		angle,
  		followLargeArc,
  		clockwise,
  		numCoords,
  		coords,
  		dimension,
  		relative
  	) {
  		SB.addPathSegment({
  			type: "arc",
  			major: major,
  			minor: minor,
  			angle: angle,
  			followLargeArc: followLargeArc,
  			clockwise: clockwise,
  			points: coords ? getArrayF32(coords, numCoords) : null,
  			dimension: dimension,
  			relative: relative
  		});
  	}

  function _SceneBuilderAdapter_insertBezierPathSegment(
  		degree,
  		numCoords,
  		coords,
  		dimension,
  		relative
  	) {
  		SB.addPathSegment({
  			type: "bezier",
  			degree: degree,
  			points: coords ? getArrayF32(coords, numCoords) : null,
  			dimension: dimension,
  			relative: relative
  		});
  	}

  function _SceneBuilderAdapter_insertCamera(
  		sceneBuilderId,
  		nodeRef,
  		cameraRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertCamera(nodeRef, cameraRef);
  	}

  function _SceneBuilderAdapter_insertClosePathSegment() {
  		SB.addPathSegment({
  			type: "close"
  		});
  	}

  function _SceneBuilderAdapter_insertLeaderLine(
  		sceneBuilderId,
  		nodeRef,
  		targetNodeRef,
  		materialRef,
  		pointCount,
  		points,
  		startPointStyle,
  		endPointStyle,
  		styleConstant,
  		extensionLength
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertLeaderLine({
  			nodeRef: nodeRef,
  			targetNodeRef: targetNodeRef,
  			materialRef: materialRef,
  			points: points ? getArrayF32(points, pointCount * 3) : null,
  			startPointStyle: startPointStyle,
  			endPointStyle: endPointStyle,
  			styleConstant: getArrayF32(styleConstant, 2),
  			extensionLength: extensionLength
  		});
  	}

  function _SceneBuilderAdapter_insertLinePathSegment(
  		numCoords,
  		coords,
  		dimension,
  		relative
  	) {
  		SB.addPathSegment({
  			type: "line",
  			points: coords ? getArrayF32(coords, numCoords) : null,
  			dimension: dimension,
  			relative: relative
  		});
  	}

  function _SceneBuilderAdapter_insertMesh(
  		sceneBuilderId,
  		nodeRef,
  		meshRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertMesh(nodeRef, meshRef);
  	}

  function _SceneBuilderAdapter_insertModelView(
  		sceneBuilderId,
  		viewportGroupRef,
  		modelViewRef,
  		name,
  		description,
  		thumbnail,
  		cameraRef,
  		type,
  		flyToTime,
  		preDelay,
  		postDelay,
  		navigationMode,
  		topColour,
  		bottomColour,
  		renderMethod,
  		dimension,
  		query,
  		metadata,
  		veids
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertModelView({
  			viewportGroupRef: viewportGroupRef,
  			modelViewRef: modelViewRef,
  			name: UTF8ToString(name),
  			description: UTF8ToString(description),
  			thumbnail: thumbnail,
  			cameraRef: cameraRef,
  			type: type,
  			flyToTime: flyToTime,
  			preDelay: preDelay,
  			postDelay: postDelay,
  			navigationMode: navigationMode,
  			topColour: topColour,
  			bottomColour: bottomColour,
  			renderMethod: renderMethod,
  			dimension: dimension,
  			query: UTF8ToString(query),
  			metadata: metadata ? JSON.parse(UTF8ToString(metadata)) : {},
  			veids: veids ? JSON.parse(UTF8ToString(veids)) : {}
  		});
  	}

  function _SceneBuilderAdapter_insertModelViewHighlight(
  		sceneBuilderId,
  		modelViewRef,
  		highlightNodeCount,
  		highlightNodeRefs,
  		color1,
  		color2,
  		opacity1,
  		opacity2,
  		duration,
  		cycles
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertModelViewHighlight({
  			modelViewRef: modelViewRef,
  			highlightNodes: getArrayU32(highlightNodeRefs, highlightNodeCount),
  			color1: color1,
  			color2: color2,
  			opacity1: opacity1,
  			opacity2: opacity2,
  			duration: duration,
  			cycles: cycles
  		});
  	}

  function _SceneBuilderAdapter_insertMovePathSegment(
  		numCoords,
  		coords,
  		dimension,
  		relative
  	) {
  		SB.addPathSegment({
  			type: "move",
  			points: coords ? getArrayF32(coords, numCoords) : null,
  			dimension: dimension,
  			relative: relative
  		});
  	}

  function _SceneBuilderAdapter_insertParametricArc(
  		pos,
  		quat,
  		scale,
  		materialRef,
  		rx,
  		ry,
  		startAngle,
  		endAngle,
  		fillStrokeVisibility,
  		closureType
  	) {
  		SB.addShape({
  			t: pos ? getArrayF32(pos, 3) : null,
  			r: quat ? getArrayF32(quat, 4) : null,
  			s: scale ? getArrayF32(scale, 3) : null,
  			materialID: materialRef,
  			type: "arc",
  			major: rx,
  			minor: ry,
  			start: startAngle,
  			end: endAngle,
  			fillStrokeVisibility: fillStrokeVisibility,
  			closureType: closureType
  		});
  	}

  function _SceneBuilderAdapter_insertParametricEllipse(
  		pos,
  		quat,
  		scale,
  		materialRef,
  		rx,
  		ry,
  		fillStrokeVisibility
  	) {
  		SB.addShape({
  			t: pos ? getArrayF32(pos, 3) : null,
  			r: quat ? getArrayF32(quat, 4) : null,
  			s: scale ? getArrayF32(scale, 3) : null,
  			materialID: materialRef,
  			type: "ellipse",
  			major: rx,
  			minor: ry,
  			fillStrokeVisibility: fillStrokeVisibility
  		});
  	}

  function _SceneBuilderAdapter_insertParametricPath(
  		pos,
  		quat,
  		scale,
  		materialRef,
  		numSegments
  	) {
  		var segments = SB.flushPathSegments();
  		SB.addShape({
  			t: pos ? getArrayF32(pos, 3) : null,
  			r: quat ? getArrayF32(quat, 4) : null,
  			s: scale ? getArrayF32(scale, 3) : null,
  			materialID: materialRef,
  			type: "path",
  			numSegments: numSegments,
  			segments: segments
  		});
  	}

  function _SceneBuilderAdapter_insertParametricPolyline(
  		pos,
  		quat,
  		scale,
  		materialRef,
  		numCoords,
  		coords,
  		dimension,
  		closed
  	) {
  		coords = getArrayF32(coords, numCoords);
  		if (numCoords === 2 * dimension) {
  			SB.addShape({
  				t: pos ? getArrayF32(pos, 3) : null,
  				r: quat ? getArrayF32(quat, 4) : null,
  				s: scale ? getArrayF32(scale, 3) : null,
  				materialID: materialRef,
  				type: "line",
  				x1: coords[ 0 ],
  				y1: coords[ 1 ],
  				x2: coords[ dimension ],
  				y2: coords[ dimension + 1 ]
  			});
  		} else {
  			SB.addShape({
  				t: pos ? getArrayF32(pos, 3) : null,
  				r: quat ? getArrayF32(quat, 4) : null,
  				s: scale ? getArrayF32(scale, 3) : null,
  				materialID: materialRef,
  				type: "polyline",
  				points: coords,
  				dimension: dimension,
  				closed: closed
  			});
  		}
  	}

  function _SceneBuilderAdapter_insertParametricRectangle(
  		pos,
  		quat,
  		scale,
  		materialRef,
  		width,
  		height
  	) {
  		SB.addShape({
  			t: pos ? getArrayF32(pos, 3) : null,
  			r: quat ? getArrayF32(quat, 4) : null,
  			s: scale ? getArrayF32(scale, 3) : null,
  			materialID: materialRef,
  			type: "rectangle",
  			width: width,
  			height: height
  		});
  	}

  function _SceneBuilderAdapter_insertParametricText(
  		pos,
  		quat,
  		scale,
  		materialRef,
  		textTranslate,
  		translateDelta) {
  		var content = SB.flushTextContent();
  		SB.addShape({
  			t: pos ? getArrayF32(pos, 3) : null,
  			r: quat ? getArrayF32(quat, 4) : null,
  			s: scale ? getArrayF32(scale, 3) : null,
  			materialID: materialRef,
  			type: "text",
  			textTranslate: textTranslate ? getArrayF32(textTranslate, 3) : null,
  			translateDelta: translateDelta ? getArrayF32(translateDelta, 3) : null,
  			style: content.style,
  			content: content.content
  		});
  	}

  function _SceneBuilderAdapter_insertParametricTextData(textData, contentSubtype) {
  		SB.insertParametricTextData({
  			type: contentSubtype,
  			text: UTF8ToString(textData)
  		});
  	}

  function _SceneBuilderAdapter_insertThrustline(
  		sceneBuilderId,
  		thrustlineRef,
  		principleAxis,
      	materialRef,
  		itemCount,
      	targets,
      	itemMajorAxisesIndices,
  		itemBasisAxisesCounts,
  		itemBasisAxisesCoordinatesDataSize,
  		itemBasisAxisesCoordinates,
  		itemDimensionsCoordinates,
  		itemCentersCoordinates,
  		itemBoundPointsCounts,
  		itemBoundPointsCoordinatesDataSize,
  		itemBoundPointsCoordinates,
  		segmentCount,
  		segmentsStartItemIndices,
  		segmentsEndItemIndices,
  		segmentsStartBoundIndices,
  		segmentsEndBoundIndices,
  		segmentRatioCounts,
  		segmentRatiosCoordinatesDataSize,
  		segmentRatiosCoordinates
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.insertThrustline({
  			thrustlineRef: thrustlineRef,
  			principleAxis: getArrayF32(principleAxis, 3),
  			material: materialRef,
  			itemCount: itemCount,
  			targets: getArrayU32(targets, itemCount),
  			itemMajorAxisesIndices: getArrayU32(itemMajorAxisesIndices, itemCount),
  			itemBasisAxisesCounts: getArrayU32(itemBasisAxisesCounts, itemCount),
  			itemBasisAxisesCoordinates: getArrayF32(itemBasisAxisesCoordinates,itemBasisAxisesCoordinatesDataSize),
  			itemDimensionsCoordinates: getArrayF32(itemDimensionsCoordinates, itemCount * 3),
  			itemCentersCoordinates: getArrayF32(itemCentersCoordinates, itemCount * 3),
  			itemBoundPointsCounts: getArrayU32(itemBoundPointsCounts, itemCount),
  			itemBoundPointsCoordinates: getArrayF32(itemBoundPointsCoordinates, itemBoundPointsCoordinatesDataSize),
  			segmentCount: segmentCount,
  			segmentsStartItemIndices: getArrayU32(segmentsStartItemIndices, segmentCount),
  			segmentsEndItemIndices: getArrayU32(segmentsEndItemIndices, segmentCount),
  			segmentsStartBoundIndices: getArrayU32(segmentsStartBoundIndices, segmentCount),
  			segmentsEndBoundIndices: getArrayU32(segmentsEndBoundIndices, segmentCount),
  			segmentRatioCounts: getArrayU32(segmentRatioCounts, segmentCount),
  			segmentRatiosCoordinates: getArrayF32(segmentRatiosCoordinates, segmentRatiosCoordinatesDataSize)
  		});
  	}

  function _SceneBuilderAdapter_progress(
  		sceneBuilderId,
  		progress
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.progress(progress);
  	}

  function _SceneBuilderAdapter_setFillStyle(
  		type,
  		start,
  		end,
  		radii,
  		numStops,
  		stopsData,
  		coordinateSpace,
  		gradientTransform,
  		spreadMethod) {
  
  		stopsData = getArrayF32(stopsData, numStops * 5);
  		var stops = [];
  		for (var i = 0, j = 0; i < numStops; i++, j += 5) {
  			stops.push({
  				offsetFraction: stopsData[ j ],
  				color: new Float32Array([ stopsData[ j + 1 ], stopsData[ j + 2 ], stopsData[ j + 3 ], stopsData[ j + 3 ] ])
  			});
  		}
  
  		SB.setFillStyle({
  			type: type,
  			start: getArrayF32(start, 3),
  			end: getArrayF32(end, 3),
  			radii: getArrayF32(radii, 2),
  			stops: stops,
  			coordinateSpace: coordinateSpace,
  			gradientTransform: gradientTransform ? getArrayF64(gradientTransform, 16) : null,
  			spreadMethod: spreadMethod
  		});
  	}

  function _SceneBuilderAdapter_setLineStyle(
  		color,
  		width,
  		units,
  		endCapStyle,
  		joinStyle,
  		miterLimit,
  		dashPatternOffset,
  		dashArrayLength,
  		dashArray) {
  		SB.setLineStyle({
  			color: getArrayF32(color, 4),
  			width: width,
  			units: units,
  			endCapStyle: endCapStyle,
  			joinStyle: joinStyle,
  			miterLimit: miterLimit,
  			dashPatternOffset: dashPatternOffset,
  			dashArray: getArrayF32(dashArray, dashArrayLength)
  		});
  	}

  function _SceneBuilderAdapter_setMeshGeometry(
  		sceneBuilderId,
  		flags,
  		pointCount,
  		triangleCount,
  		splatCount,
  		data,
  		dataSize,
  		vertexDataSize,
  		indexDataSize,
  		vertexSize,
  		meshRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var info = {
  			flags: flags,
  			pointCount: pointCount,
  			triangleCount: triangleCount,
  			splatCount: splatCount,
  			meshRef: meshRef
  		};
  		if (data && dataSize) {
  			var indexData = new Uint16Array(Module.buffer, data + vertexDataSize, indexDataSize >> 1);
  			var vertexData = new Float32Array(Module.buffer, data, vertexDataSize >> 2);
  			var vertexCount = pointCount + splatCount;
  			var vertexSize = vertexSize;
  			var flags = flags;
  			var offset = 0;
  			var i;
  			var si;
  			var di;
  
  			var data = {};
  			data.index = new Uint16Array(indexData); // clone index data, because we can't use dynamic memory data
  
  			var positionData = new Float32Array(vertexCount * 3);
  			for (i = 0, di = 0, si = offset; i < vertexCount; i++, si += vertexSize) {
  				positionData[di++] = vertexData[si + 0];
  				positionData[di++] = vertexData[si + 1];
  				positionData[di++] = vertexData[si + 2];
  			}
  			data.position = positionData;
  			offset += 3;
  
  			if (flags & 2) {
  				var normalData = new Float32Array(vertexCount * 3);
  				for (i = 0, di = 0, si = offset; i < vertexCount; i++, si += vertexSize) {
  					normalData[di++] = vertexData[si + 0];
  					normalData[di++] = vertexData[si + 1];
  					normalData[di++] = vertexData[si + 2];
  				}
  				data.normal = normalData;
  				offset += 3;
  			}
  
  			if (flags & 4) {
  				var uvData = new Float32Array(vertexCount * 2);
  				for (i = 0, di = 0, si = offset; i < vertexCount; i++, si += vertexSize) {
  					uvData[di++] = vertexData[si + 0];
  					uvData[di++] = vertexData[si + 1];
  				}
  				data.uv = uvData;
  				offset += 2;
  			}
  
  			info.data = data;
  		}
  
  		builder.setMeshGeometry(info);
  	}

  function _SceneBuilderAdapter_setModelViewVisibility(
  		sceneBuilderId,
  		modelViewRef,
  		visibleNodeCount,
  		visibleNodeRefs
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.setModelViewVisibilitySet({
  			modelViewRef: modelViewRef,
  			visibleNodes: getArrayU32(visibleNodeRefs, visibleNodeCount)
  		});
  	}

  function _SceneBuilderAdapter_setParametricContent(
  		sceneBuilderId,
  		nodeRef
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		var shapes = SB.flushShapes();
  		if (shapes.length > 0) {
  			builder.setParametricContent(nodeRef, shapes.length > 1 ? { shapes: shapes } : shapes[ 0 ]);
  		}
  	}

  function _SceneBuilderAdapter_setScene(
  		sceneBuilderId,
  		result,
  		cameraRef,
  		backgroundTopColor,
  		backgroundBottomColor,
  		renderMethod,
  		upAxis,
  		meshCount,
  		imageCount,
  		modelViewCount
  	) {
  		var builder = SB.getById(sceneBuilderId);
  		builder.setScene({
  			result: result,
  			cameraRef: cameraRef,
  			backgroundTopColor: backgroundTopColor,
  			backgroundBottomColor: backgroundBottomColor,
  			renderMethod: renderMethod,
  			upAxis: upAxis,
  			meshCount: meshCount,
  			imageCount: imageCount,
  			modelViewCount: modelViewCount
  		});
  	}

  
  var ENV={};function ___buildEnvironment(environ) {
      // WARNING: Arbitrary limit!
      var MAX_ENV_VALUES = 64;
      var TOTAL_ENV_SIZE = 1024;
  
      // Statically allocate memory for the environment.
      var poolPtr;
      var envPtr;
      if (!___buildEnvironment.called) {
        ___buildEnvironment.called = true;
        // Set default values. Use string keys for Closure Compiler compatibility.
        ENV['USER'] = ENV['LOGNAME'] = 'web_user';
        ENV['PATH'] = '/';
        ENV['PWD'] = '/';
        ENV['HOME'] = '/home/web_user';
        ENV['LANG'] = 'C.UTF-8';
        ENV['_'] = Module['thisProgram'];
        // Allocate memory.
        poolPtr = getMemory(TOTAL_ENV_SIZE);
        envPtr = getMemory(MAX_ENV_VALUES * 4);
        HEAP32[((envPtr)>>2)]=poolPtr;
        HEAP32[((environ)>>2)]=envPtr;
      } else {
        envPtr = HEAP32[((environ)>>2)];
        poolPtr = HEAP32[((envPtr)>>2)];
      }
  
      // Collect key=value lines.
      var strings = [];
      var totalSize = 0;
      for (var key in ENV) {
        if (typeof ENV[key] === 'string') {
          var line = key + '=' + ENV[key];
          strings.push(line);
          totalSize += line.length;
        }
      }
      if (totalSize > TOTAL_ENV_SIZE) {
        throw new Error('Environment size exceeded TOTAL_ENV_SIZE!');
      }
  
      // Make new.
      var ptrSize = 4;
      for (var i = 0; i < strings.length; i++) {
        var line = strings[i];
        writeAsciiToMemory(line, poolPtr);
        HEAP32[(((envPtr)+(i * ptrSize))>>2)]=poolPtr;
        poolPtr += line.length + 1;
      }
      HEAP32[(((envPtr)+(strings.length * ptrSize))>>2)]=0;
    }

  
  
  function _emscripten_get_now() { abort() }
  
  function _emscripten_get_now_is_monotonic() {
      // return whether emscripten_get_now is guaranteed monotonic; the Date.now
      // implementation is not :(
      return ENVIRONMENT_IS_NODE || (typeof dateNow !== 'undefined') ||
          ((ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) && self['performance'] && self['performance']['now']);
    }
  
  var ERRNO_CODES={EPERM:1,ENOENT:2,ESRCH:3,EINTR:4,EIO:5,ENXIO:6,E2BIG:7,ENOEXEC:8,EBADF:9,ECHILD:10,EAGAIN:11,EWOULDBLOCK:11,ENOMEM:12,EACCES:13,EFAULT:14,ENOTBLK:15,EBUSY:16,EEXIST:17,EXDEV:18,ENODEV:19,ENOTDIR:20,EISDIR:21,EINVAL:22,ENFILE:23,EMFILE:24,ENOTTY:25,ETXTBSY:26,EFBIG:27,ENOSPC:28,ESPIPE:29,EROFS:30,EMLINK:31,EPIPE:32,EDOM:33,ERANGE:34,ENOMSG:42,EIDRM:43,ECHRNG:44,EL2NSYNC:45,EL3HLT:46,EL3RST:47,ELNRNG:48,EUNATCH:49,ENOCSI:50,EL2HLT:51,EDEADLK:35,ENOLCK:37,EBADE:52,EBADR:53,EXFULL:54,ENOANO:55,EBADRQC:56,EBADSLT:57,EDEADLOCK:35,EBFONT:59,ENOSTR:60,ENODATA:61,ETIME:62,ENOSR:63,ENONET:64,ENOPKG:65,EREMOTE:66,ENOLINK:67,EADV:68,ESRMNT:69,ECOMM:70,EPROTO:71,EMULTIHOP:72,EDOTDOT:73,EBADMSG:74,ENOTUNIQ:76,EBADFD:77,EREMCHG:78,ELIBACC:79,ELIBBAD:80,ELIBSCN:81,ELIBMAX:82,ELIBEXEC:83,ENOSYS:38,ENOTEMPTY:39,ENAMETOOLONG:36,ELOOP:40,EOPNOTSUPP:95,EPFNOSUPPORT:96,ECONNRESET:104,ENOBUFS:105,EAFNOSUPPORT:97,EPROTOTYPE:91,ENOTSOCK:88,ENOPROTOOPT:92,ESHUTDOWN:108,ECONNREFUSED:111,EADDRINUSE:98,ECONNABORTED:103,ENETUNREACH:101,ENETDOWN:100,ETIMEDOUT:110,EHOSTDOWN:112,EHOSTUNREACH:113,EINPROGRESS:115,EALREADY:114,EDESTADDRREQ:89,EMSGSIZE:90,EPROTONOSUPPORT:93,ESOCKTNOSUPPORT:94,EADDRNOTAVAIL:99,ENETRESET:102,EISCONN:106,ENOTCONN:107,ETOOMANYREFS:109,EUSERS:87,EDQUOT:122,ESTALE:116,ENOTSUP:95,ENOMEDIUM:123,EILSEQ:84,EOVERFLOW:75,ECANCELED:125,ENOTRECOVERABLE:131,EOWNERDEAD:130,ESTRPIPE:86};
  
  function ___setErrNo(value) {
      if (Module['___errno_location']) HEAP32[((Module['___errno_location']())>>2)]=value;
      return value;
    }function _clock_gettime(clk_id, tp) {
      // int clock_gettime(clockid_t clk_id, struct timespec *tp);
      var now;
      if (clk_id === 0) {
        now = Date.now();
      } else if (clk_id === 1 && _emscripten_get_now_is_monotonic()) {
        now = _emscripten_get_now();
      } else {
        ___setErrNo(ERRNO_CODES.EINVAL);
        return -1;
      }
      HEAP32[((tp)>>2)]=(now/1000)|0; // seconds
      HEAP32[(((tp)+(4))>>2)]=((now % 1000)*1000*1000)|0; // nanoseconds
      return 0;
    }function ___clock_gettime() {
  return _clock_gettime.apply(null, arguments)
  }

  function ___cxa_allocate_exception(size) {
      return _malloc(size);
    }

  
  var EXCEPTIONS={last:0,caught:[],infos:{},deAdjust:function (adjusted) {
        if (!adjusted || EXCEPTIONS.infos[adjusted]) return adjusted;
        for (var key in EXCEPTIONS.infos) {
          var ptr = +key; // the iteration key is a string, and if we throw this, it must be an integer as that is what we look for
          var info = EXCEPTIONS.infos[ptr];
          if (info.adjusted === adjusted) {
            return ptr;
          }
        }
        return adjusted;
      },addRef:function (ptr) {
        if (!ptr) return;
        var info = EXCEPTIONS.infos[ptr];
        info.refcount++;
      },decRef:function (ptr) {
        if (!ptr) return;
        var info = EXCEPTIONS.infos[ptr];
        assert(info.refcount > 0);
        info.refcount--;
        // A rethrown exception can reach refcount 0; it must not be discarded
        // Its next handler will clear the rethrown flag and addRef it, prior to
        // final decRef and destruction here
        if (info.refcount === 0 && !info.rethrown) {
          if (info.destructor) {
            Module['dynCall_vi'](info.destructor, ptr);
          }
          delete EXCEPTIONS.infos[ptr];
          ___cxa_free_exception(ptr);
        }
      },clearRef:function (ptr) {
        if (!ptr) return;
        var info = EXCEPTIONS.infos[ptr];
        info.refcount = 0;
      }};function ___cxa_begin_catch(ptr) {
      var info = EXCEPTIONS.infos[ptr];
      if (info && !info.caught) {
        info.caught = true;
        __ZSt18uncaught_exceptionv.uncaught_exception--;
      }
      if (info) info.rethrown = false;
      EXCEPTIONS.caught.push(ptr);
      EXCEPTIONS.addRef(EXCEPTIONS.deAdjust(ptr));
      return ptr;
    }

  function ___cxa_pure_virtual() {
      ABORT = true;
      throw 'Pure virtual function called!';
    }

  
  
  function ___cxa_free_exception(ptr) {
      try {
        return _free(ptr);
      } catch(e) { // XXX FIXME
      }
    }function ___cxa_end_catch() {
      // Clear state flag.
      Module['setThrew'](0);
      // Call destructor if one is registered then clear it.
      var ptr = EXCEPTIONS.caught.pop();
      if (ptr) {
        EXCEPTIONS.decRef(EXCEPTIONS.deAdjust(ptr));
        EXCEPTIONS.last = 0; // XXX in decRef?
      }
    }function ___cxa_rethrow() {
      var ptr = EXCEPTIONS.caught.pop();
      ptr = EXCEPTIONS.deAdjust(ptr);
      if (!EXCEPTIONS.infos[ptr].rethrown) {
        // Only pop if the corresponding push was through rethrow_primary_exception
        EXCEPTIONS.caught.push(ptr)
        EXCEPTIONS.infos[ptr].rethrown = true;
      }
      EXCEPTIONS.last = ptr;
      throw ptr + " - Exception catching is disabled, this exception cannot be caught. Compile with -s DISABLE_EXCEPTION_CATCHING=0 or DISABLE_EXCEPTION_CATCHING=2 to catch.";
    }

  
  
  function ___resumeException(ptr) {
      if (!EXCEPTIONS.last) { EXCEPTIONS.last = ptr; }
      throw ptr + " - Exception catching is disabled, this exception cannot be caught. Compile with -s DISABLE_EXCEPTION_CATCHING=0 or DISABLE_EXCEPTION_CATCHING=2 to catch.";
    }function ___cxa_find_matching_catch() {
      var thrown = EXCEPTIONS.last;
      if (!thrown) {
        // just pass through the null ptr
        return ((setTempRet0(0),0)|0);
      }
      var info = EXCEPTIONS.infos[thrown];
      var throwntype = info.type;
      if (!throwntype) {
        // just pass through the thrown ptr
        return ((setTempRet0(0),thrown)|0);
      }
      var typeArray = Array.prototype.slice.call(arguments);
  
      var pointer = Module['___cxa_is_pointer_type'](throwntype);
      // can_catch receives a **, add indirection
      if (!___cxa_find_matching_catch.buffer) ___cxa_find_matching_catch.buffer = _malloc(4);
      HEAP32[((___cxa_find_matching_catch.buffer)>>2)]=thrown;
      thrown = ___cxa_find_matching_catch.buffer;
      // The different catch blocks are denoted by different types.
      // Due to inheritance, those types may not precisely match the
      // type of the thrown object. Find one which matches, and
      // return the type of the catch block which should be called.
      for (var i = 0; i < typeArray.length; i++) {
        if (typeArray[i] && Module['___cxa_can_catch'](typeArray[i], throwntype, thrown)) {
          thrown = HEAP32[((thrown)>>2)]; // undo indirection
          info.adjusted = thrown;
          return ((setTempRet0(typeArray[i]),thrown)|0);
        }
      }
      // Shouldn't happen unless we have bogus data in typeArray
      // or encounter a type for which emscripten doesn't have suitable
      // typeinfo defined. Best-efforts match just in case.
      thrown = HEAP32[((thrown)>>2)]; // undo indirection
      return ((setTempRet0(throwntype),thrown)|0);
    }function ___cxa_throw(ptr, type, destructor) {
      EXCEPTIONS.infos[ptr] = {
        ptr: ptr,
        adjusted: ptr,
        type: type,
        destructor: destructor,
        refcount: 0,
        caught: false,
        rethrown: false
      };
      EXCEPTIONS.last = ptr;
      if (!("uncaught_exception" in __ZSt18uncaught_exceptionv)) {
        __ZSt18uncaught_exceptionv.uncaught_exception = 1;
      } else {
        __ZSt18uncaught_exceptionv.uncaught_exception++;
      }
      throw ptr + " - Exception catching is disabled, this exception cannot be caught. Compile with -s DISABLE_EXCEPTION_CATCHING=0 or DISABLE_EXCEPTION_CATCHING=2 to catch.";
    }

  function ___cxa_uncaught_exception() {
      return !!__ZSt18uncaught_exceptionv.uncaught_exception;
    }

  
  
   
  
   
  
     

  function ___gxx_personality_v0() {
    }

  function ___lock() {}

  function ___map_file(pathname, size) {
      ___setErrNo(ERRNO_CODES.EPERM);
      return -1;
    }

  
    

   

  
  
  
  var ERRNO_MESSAGES={0:"Success",1:"Not super-user",2:"No such file or directory",3:"No such process",4:"Interrupted system call",5:"I/O error",6:"No such device or address",7:"Arg list too long",8:"Exec format error",9:"Bad file number",10:"No children",11:"No more processes",12:"Not enough core",13:"Permission denied",14:"Bad address",15:"Block device required",16:"Mount device busy",17:"File exists",18:"Cross-device link",19:"No such device",20:"Not a directory",21:"Is a directory",22:"Invalid argument",23:"Too many open files in system",24:"Too many open files",25:"Not a typewriter",26:"Text file busy",27:"File too large",28:"No space left on device",29:"Illegal seek",30:"Read only file system",31:"Too many links",32:"Broken pipe",33:"Math arg out of domain of func",34:"Math result not representable",35:"File locking deadlock error",36:"File or path name too long",37:"No record locks available",38:"Function not implemented",39:"Directory not empty",40:"Too many symbolic links",42:"No message of desired type",43:"Identifier removed",44:"Channel number out of range",45:"Level 2 not synchronized",46:"Level 3 halted",47:"Level 3 reset",48:"Link number out of range",49:"Protocol driver not attached",50:"No CSI structure available",51:"Level 2 halted",52:"Invalid exchange",53:"Invalid request descriptor",54:"Exchange full",55:"No anode",56:"Invalid request code",57:"Invalid slot",59:"Bad font file fmt",60:"Device not a stream",61:"No data (for no delay io)",62:"Timer expired",63:"Out of streams resources",64:"Machine is not on the network",65:"Package not installed",66:"The object is remote",67:"The link has been severed",68:"Advertise error",69:"Srmount error",70:"Communication error on send",71:"Protocol error",72:"Multihop attempted",73:"Cross mount point (not really error)",74:"Trying to read unreadable message",75:"Value too large for defined data type",76:"Given log. name not unique",77:"f.d. invalid for this operation",78:"Remote address changed",79:"Can   access a needed shared lib",80:"Accessing a corrupted shared lib",81:".lib section in a.out corrupted",82:"Attempting to link in too many libs",83:"Attempting to exec a shared library",84:"Illegal byte sequence",86:"Streams pipe error",87:"Too many users",88:"Socket operation on non-socket",89:"Destination address required",90:"Message too long",91:"Protocol wrong type for socket",92:"Protocol not available",93:"Unknown protocol",94:"Socket type not supported",95:"Not supported",96:"Protocol family not supported",97:"Address family not supported by protocol family",98:"Address already in use",99:"Address not available",100:"Network interface is not configured",101:"Network is unreachable",102:"Connection reset by network",103:"Connection aborted",104:"Connection reset by peer",105:"No buffer space available",106:"Socket is already connected",107:"Socket is not connected",108:"Can't send after socket shutdown",109:"Too many references",110:"Connection timed out",111:"Connection refused",112:"Host is down",113:"Host is unreachable",114:"Socket already connected",115:"Connection already in progress",116:"Stale file handle",122:"Quota exceeded",123:"No medium (in tape drive)",125:"Operation canceled",130:"Previous owner died",131:"State not recoverable"};
  
  var PATH={splitPath:function (filename) {
        var splitPathRe = /^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/;
        return splitPathRe.exec(filename).slice(1);
      },normalizeArray:function (parts, allowAboveRoot) {
        // if the path tries to go above the root, `up` ends up > 0
        var up = 0;
        for (var i = parts.length - 1; i >= 0; i--) {
          var last = parts[i];
          if (last === '.') {
            parts.splice(i, 1);
          } else if (last === '..') {
            parts.splice(i, 1);
            up++;
          } else if (up) {
            parts.splice(i, 1);
            up--;
          }
        }
        // if the path is allowed to go above the root, restore leading ..s
        if (allowAboveRoot) {
          for (; up; up--) {
            parts.unshift('..');
          }
        }
        return parts;
      },normalize:function (path) {
        var isAbsolute = path.charAt(0) === '/',
            trailingSlash = path.substr(-1) === '/';
        // Normalize the path
        path = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), !isAbsolute).join('/');
        if (!path && !isAbsolute) {
          path = '.';
        }
        if (path && trailingSlash) {
          path += '/';
        }
        return (isAbsolute ? '/' : '') + path;
      },dirname:function (path) {
        var result = PATH.splitPath(path),
            root = result[0],
            dir = result[1];
        if (!root && !dir) {
          // No dirname whatsoever
          return '.';
        }
        if (dir) {
          // It has a dirname, strip trailing slash
          dir = dir.substr(0, dir.length - 1);
        }
        return root + dir;
      },basename:function (path) {
        // EMSCRIPTEN return '/'' for '/', not an empty string
        if (path === '/') return '/';
        var lastSlash = path.lastIndexOf('/');
        if (lastSlash === -1) return path;
        return path.substr(lastSlash+1);
      },extname:function (path) {
        return PATH.splitPath(path)[3];
      },join:function () {
        var paths = Array.prototype.slice.call(arguments, 0);
        return PATH.normalize(paths.join('/'));
      },join2:function (l, r) {
        return PATH.normalize(l + '/' + r);
      },resolve:function () {
        var resolvedPath = '',
          resolvedAbsolute = false;
        for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
          var path = (i >= 0) ? arguments[i] : FS.cwd();
          // Skip empty and invalid entries
          if (typeof path !== 'string') {
            throw new TypeError('Arguments to path.resolve must be strings');
          } else if (!path) {
            return ''; // an invalid portion invalidates the whole thing
          }
          resolvedPath = path + '/' + resolvedPath;
          resolvedAbsolute = path.charAt(0) === '/';
        }
        // At this point the path should be resolved to a full absolute path, but
        // handle relative paths to be safe (might happen when process.cwd() fails)
        resolvedPath = PATH.normalizeArray(resolvedPath.split('/').filter(function(p) {
          return !!p;
        }), !resolvedAbsolute).join('/');
        return ((resolvedAbsolute ? '/' : '') + resolvedPath) || '.';
      },relative:function (from, to) {
        from = PATH.resolve(from).substr(1);
        to = PATH.resolve(to).substr(1);
        function trim(arr) {
          var start = 0;
          for (; start < arr.length; start++) {
            if (arr[start] !== '') break;
          }
          var end = arr.length - 1;
          for (; end >= 0; end--) {
            if (arr[end] !== '') break;
          }
          if (start > end) return [];
          return arr.slice(start, end - start + 1);
        }
        var fromParts = trim(from.split('/'));
        var toParts = trim(to.split('/'));
        var length = Math.min(fromParts.length, toParts.length);
        var samePartsLength = length;
        for (var i = 0; i < length; i++) {
          if (fromParts[i] !== toParts[i]) {
            samePartsLength = i;
            break;
          }
        }
        var outputParts = [];
        for (var i = samePartsLength; i < fromParts.length; i++) {
          outputParts.push('..');
        }
        outputParts = outputParts.concat(toParts.slice(samePartsLength));
        return outputParts.join('/');
      }};
  
  var TTY={ttys:[],init:function () {
        // https://github.com/kripken/emscripten/pull/1555
        // if (ENVIRONMENT_IS_NODE) {
        //   // currently, FS.init does not distinguish if process.stdin is a file or TTY
        //   // device, it always assumes it's a TTY device. because of this, we're forcing
        //   // process.stdin to UTF8 encoding to at least make stdin reading compatible
        //   // with text files until FS.init can be refactored.
        //   process['stdin']['setEncoding']('utf8');
        // }
      },shutdown:function () {
        // https://github.com/kripken/emscripten/pull/1555
        // if (ENVIRONMENT_IS_NODE) {
        //   // inolen: any idea as to why node -e 'process.stdin.read()' wouldn't exit immediately (with process.stdin being a tty)?
        //   // isaacs: because now it's reading from the stream, you've expressed interest in it, so that read() kicks off a _read() which creates a ReadReq operation
        //   // inolen: I thought read() in that case was a synchronous operation that just grabbed some amount of buffered data if it exists?
        //   // isaacs: it is. but it also triggers a _read() call, which calls readStart() on the handle
        //   // isaacs: do process.stdin.pause() and i'd think it'd probably close the pending call
        //   process['stdin']['pause']();
        // }
      },register:function (dev, ops) {
        TTY.ttys[dev] = { input: [], output: [], ops: ops };
        FS.registerDevice(dev, TTY.stream_ops);
      },stream_ops:{open:function (stream) {
          var tty = TTY.ttys[stream.node.rdev];
          if (!tty) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          stream.tty = tty;
          stream.seekable = false;
        },close:function (stream) {
          // flush any pending line data
          stream.tty.ops.flush(stream.tty);
        },flush:function (stream) {
          stream.tty.ops.flush(stream.tty);
        },read:function (stream, buffer, offset, length, pos /* ignored */) {
          if (!stream.tty || !stream.tty.ops.get_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          var bytesRead = 0;
          for (var i = 0; i < length; i++) {
            var result;
            try {
              result = stream.tty.ops.get_char(stream.tty);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            if (result === undefined && bytesRead === 0) {
              throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
            }
            if (result === null || result === undefined) break;
            bytesRead++;
            buffer[offset+i] = result;
          }
          if (bytesRead) {
            stream.node.timestamp = Date.now();
          }
          return bytesRead;
        },write:function (stream, buffer, offset, length, pos) {
          if (!stream.tty || !stream.tty.ops.put_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          for (var i = 0; i < length; i++) {
            try {
              stream.tty.ops.put_char(stream.tty, buffer[offset+i]);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
          }
          if (length) {
            stream.node.timestamp = Date.now();
          }
          return i;
        }},default_tty_ops:{get_char:function (tty) {
          if (!tty.input.length) {
            var result = null;
            if (ENVIRONMENT_IS_NODE) {
              // we will read data by chunks of BUFSIZE
              var BUFSIZE = 256;
              var buf = new Buffer(BUFSIZE);
              var bytesRead = 0;
  
              var isPosixPlatform = (process.platform != 'win32'); // Node doesn't offer a direct check, so test by exclusion
  
              var fd = process.stdin.fd;
              if (isPosixPlatform) {
                // Linux and Mac cannot use process.stdin.fd (which isn't set up as sync)
                var usingDevice = false;
                try {
                  fd = fs.openSync('/dev/stdin', 'r');
                  usingDevice = true;
                } catch (e) {}
              }
  
              try {
                bytesRead = fs.readSync(fd, buf, 0, BUFSIZE, null);
              } catch(e) {
                // Cross-platform differences: on Windows, reading EOF throws an exception, but on other OSes,
                // reading EOF returns 0. Uniformize behavior by treating the EOF exception to return 0.
                if (e.toString().indexOf('EOF') != -1) bytesRead = 0;
                else throw e;
              }
  
              if (usingDevice) { fs.closeSync(fd); }
              if (bytesRead > 0) {
                result = buf.slice(0, bytesRead).toString('utf-8');
              } else {
                result = null;
              }
  
            } else if (typeof window != 'undefined' &&
              typeof window.prompt == 'function') {
              // Browser.
              result = window.prompt('Input: ');  // returns null on cancel
              if (result !== null) {
                result += '\n';
              }
            } else if (typeof readline == 'function') {
              // Command line.
              result = readline();
              if (result !== null) {
                result += '\n';
              }
            }
            if (!result) {
              return null;
            }
            tty.input = intArrayFromString(result, true);
          }
          return tty.input.shift();
        },put_char:function (tty, val) {
          if (val === null || val === 10) {
            out(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          } else {
            if (val != 0) tty.output.push(val); // val == 0 would cut text output off in the middle.
          }
        },flush:function (tty) {
          if (tty.output && tty.output.length > 0) {
            out(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          }
        }},default_tty1_ops:{put_char:function (tty, val) {
          if (val === null || val === 10) {
            err(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          } else {
            if (val != 0) tty.output.push(val);
          }
        },flush:function (tty) {
          if (tty.output && tty.output.length > 0) {
            err(UTF8ArrayToString(tty.output, 0));
            tty.output = [];
          }
        }}};
  
  var MEMFS={ops_table:null,mount:function (mount) {
        return MEMFS.createNode(null, '/', 16384 | 511 /* 0777 */, 0);
      },createNode:function (parent, name, mode, dev) {
        if (FS.isBlkdev(mode) || FS.isFIFO(mode)) {
          // no supported
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (!MEMFS.ops_table) {
          MEMFS.ops_table = {
            dir: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr,
                lookup: MEMFS.node_ops.lookup,
                mknod: MEMFS.node_ops.mknod,
                rename: MEMFS.node_ops.rename,
                unlink: MEMFS.node_ops.unlink,
                rmdir: MEMFS.node_ops.rmdir,
                readdir: MEMFS.node_ops.readdir,
                symlink: MEMFS.node_ops.symlink
              },
              stream: {
                llseek: MEMFS.stream_ops.llseek
              }
            },
            file: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr
              },
              stream: {
                llseek: MEMFS.stream_ops.llseek,
                read: MEMFS.stream_ops.read,
                write: MEMFS.stream_ops.write,
                allocate: MEMFS.stream_ops.allocate,
                mmap: MEMFS.stream_ops.mmap,
                msync: MEMFS.stream_ops.msync
              }
            },
            link: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr,
                readlink: MEMFS.node_ops.readlink
              },
              stream: {}
            },
            chrdev: {
              node: {
                getattr: MEMFS.node_ops.getattr,
                setattr: MEMFS.node_ops.setattr
              },
              stream: FS.chrdev_stream_ops
            }
          };
        }
        var node = FS.createNode(parent, name, mode, dev);
        if (FS.isDir(node.mode)) {
          node.node_ops = MEMFS.ops_table.dir.node;
          node.stream_ops = MEMFS.ops_table.dir.stream;
          node.contents = {};
        } else if (FS.isFile(node.mode)) {
          node.node_ops = MEMFS.ops_table.file.node;
          node.stream_ops = MEMFS.ops_table.file.stream;
          node.usedBytes = 0; // The actual number of bytes used in the typed array, as opposed to contents.length which gives the whole capacity.
          // When the byte data of the file is populated, this will point to either a typed array, or a normal JS array. Typed arrays are preferred
          // for performance, and used by default. However, typed arrays are not resizable like normal JS arrays are, so there is a small disk size
          // penalty involved for appending file writes that continuously grow a file similar to std::vector capacity vs used -scheme.
          node.contents = null; 
        } else if (FS.isLink(node.mode)) {
          node.node_ops = MEMFS.ops_table.link.node;
          node.stream_ops = MEMFS.ops_table.link.stream;
        } else if (FS.isChrdev(node.mode)) {
          node.node_ops = MEMFS.ops_table.chrdev.node;
          node.stream_ops = MEMFS.ops_table.chrdev.stream;
        }
        node.timestamp = Date.now();
        // add the new node to the parent
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },getFileDataAsRegularArray:function (node) {
        if (node.contents && node.contents.subarray) {
          var arr = [];
          for (var i = 0; i < node.usedBytes; ++i) arr.push(node.contents[i]);
          return arr; // Returns a copy of the original data.
        }
        return node.contents; // No-op, the file contents are already in a JS array. Return as-is.
      },getFileDataAsTypedArray:function (node) {
        if (!node.contents) return new Uint8Array;
        if (node.contents.subarray) return node.contents.subarray(0, node.usedBytes); // Make sure to not return excess unused bytes.
        return new Uint8Array(node.contents);
      },expandFileStorage:function (node, newCapacity) {
        // If we are asked to expand the size of a file that already exists, revert to using a standard JS array to store the file
        // instead of a typed array. This makes resizing the array more flexible because we can just .push() elements at the back to
        // increase the size.
        if (node.contents && node.contents.subarray && newCapacity > node.contents.length) {
          node.contents = MEMFS.getFileDataAsRegularArray(node);
          node.usedBytes = node.contents.length; // We might be writing to a lazy-loaded file which had overridden this property, so force-reset it.
        }
  
        if (!node.contents || node.contents.subarray) { // Keep using a typed array if creating a new storage, or if old one was a typed array as well.
          var prevCapacity = node.contents ? node.contents.length : 0;
          if (prevCapacity >= newCapacity) return; // No need to expand, the storage was already large enough.
          // Don't expand strictly to the given requested limit if it's only a very small increase, but instead geometrically grow capacity.
          // For small filesizes (<1MB), perform size*2 geometric increase, but for large sizes, do a much more conservative size*1.125 increase to
          // avoid overshooting the allocation cap by a very large margin.
          var CAPACITY_DOUBLING_MAX = 1024 * 1024;
          newCapacity = Math.max(newCapacity, (prevCapacity * (prevCapacity < CAPACITY_DOUBLING_MAX ? 2.0 : 1.125)) | 0);
          if (prevCapacity != 0) newCapacity = Math.max(newCapacity, 256); // At minimum allocate 256b for each file when expanding.
          var oldContents = node.contents;
          node.contents = new Uint8Array(newCapacity); // Allocate new storage.
          if (node.usedBytes > 0) node.contents.set(oldContents.subarray(0, node.usedBytes), 0); // Copy old data over to the new storage.
          return;
        }
        // Not using a typed array to back the file storage. Use a standard JS array instead.
        if (!node.contents && newCapacity > 0) node.contents = [];
        while (node.contents.length < newCapacity) node.contents.push(0);
      },resizeFileStorage:function (node, newSize) {
        if (node.usedBytes == newSize) return;
        if (newSize == 0) {
          node.contents = null; // Fully decommit when requesting a resize to zero.
          node.usedBytes = 0;
          return;
        }
        if (!node.contents || node.contents.subarray) { // Resize a typed array if that is being used as the backing store.
          var oldContents = node.contents;
          node.contents = new Uint8Array(new ArrayBuffer(newSize)); // Allocate new storage.
          if (oldContents) {
            node.contents.set(oldContents.subarray(0, Math.min(newSize, node.usedBytes))); // Copy old data over to the new storage.
          }
          node.usedBytes = newSize;
          return;
        }
        // Backing with a JS array.
        if (!node.contents) node.contents = [];
        if (node.contents.length > newSize) node.contents.length = newSize;
        else while (node.contents.length < newSize) node.contents.push(0);
        node.usedBytes = newSize;
      },node_ops:{getattr:function (node) {
          var attr = {};
          // device numbers reuse inode numbers.
          attr.dev = FS.isChrdev(node.mode) ? node.id : 1;
          attr.ino = node.id;
          attr.mode = node.mode;
          attr.nlink = 1;
          attr.uid = 0;
          attr.gid = 0;
          attr.rdev = node.rdev;
          if (FS.isDir(node.mode)) {
            attr.size = 4096;
          } else if (FS.isFile(node.mode)) {
            attr.size = node.usedBytes;
          } else if (FS.isLink(node.mode)) {
            attr.size = node.link.length;
          } else {
            attr.size = 0;
          }
          attr.atime = new Date(node.timestamp);
          attr.mtime = new Date(node.timestamp);
          attr.ctime = new Date(node.timestamp);
          // NOTE: In our implementation, st_blocks = Math.ceil(st_size/st_blksize),
          //       but this is not required by the standard.
          attr.blksize = 4096;
          attr.blocks = Math.ceil(attr.size / attr.blksize);
          return attr;
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
          if (attr.size !== undefined) {
            MEMFS.resizeFileStorage(node, attr.size);
          }
        },lookup:function (parent, name) {
          throw FS.genericErrors[ERRNO_CODES.ENOENT];
        },mknod:function (parent, name, mode, dev) {
          return MEMFS.createNode(parent, name, mode, dev);
        },rename:function (old_node, new_dir, new_name) {
          // if we're overwriting a directory at new_name, make sure it's empty.
          if (FS.isDir(old_node.mode)) {
            var new_node;
            try {
              new_node = FS.lookupNode(new_dir, new_name);
            } catch (e) {
            }
            if (new_node) {
              for (var i in new_node.contents) {
                throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
              }
            }
          }
          // do the internal rewiring
          delete old_node.parent.contents[old_node.name];
          old_node.name = new_name;
          new_dir.contents[new_name] = old_node;
          old_node.parent = new_dir;
        },unlink:function (parent, name) {
          delete parent.contents[name];
        },rmdir:function (parent, name) {
          var node = FS.lookupNode(parent, name);
          for (var i in node.contents) {
            throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
          }
          delete parent.contents[name];
        },readdir:function (node) {
          var entries = ['.', '..']
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newname, oldpath) {
          var node = MEMFS.createNode(parent, newname, 511 /* 0777 */ | 40960, 0);
          node.link = oldpath;
          return node;
        },readlink:function (node) {
          if (!FS.isLink(node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return node.link;
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          var contents = stream.node.contents;
          if (position >= stream.node.usedBytes) return 0;
          var size = Math.min(stream.node.usedBytes - position, length);
          assert(size >= 0);
          if (size > 8 && contents.subarray) { // non-trivial, and typed array
            buffer.set(contents.subarray(position, position + size), offset);
          } else {
            for (var i = 0; i < size; i++) buffer[offset + i] = contents[position + i];
          }
          return size;
        },write:function (stream, buffer, offset, length, position, canOwn) {
          if (!length) return 0;
          var node = stream.node;
          node.timestamp = Date.now();
  
          if (buffer.subarray && (!node.contents || node.contents.subarray)) { // This write is from a typed array to a typed array?
            if (canOwn) {
              node.contents = buffer.subarray(offset, offset + length);
              node.usedBytes = length;
              return length;
            } else if (node.usedBytes === 0 && position === 0) { // If this is a simple first write to an empty file, do a fast set since we don't need to care about old data.
              node.contents = new Uint8Array(buffer.subarray(offset, offset + length));
              node.usedBytes = length;
              return length;
            } else if (position + length <= node.usedBytes) { // Writing to an already allocated and used subrange of the file?
              node.contents.set(buffer.subarray(offset, offset + length), position);
              return length;
            }
          }
  
          // Appending to an existing file and we need to reallocate, or source data did not come as a typed array.
          MEMFS.expandFileStorage(node, position+length);
          if (node.contents.subarray && buffer.subarray) node.contents.set(buffer.subarray(offset, offset + length), position); // Use typed array write if available.
          else {
            for (var i = 0; i < length; i++) {
             node.contents[position + i] = buffer[offset + i]; // Or fall back to manual write if not.
            }
          }
          node.usedBytes = Math.max(node.usedBytes, position+length);
          return length;
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.usedBytes;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return position;
        },allocate:function (stream, offset, length) {
          MEMFS.expandFileStorage(stream.node, offset + length);
          stream.node.usedBytes = Math.max(stream.node.usedBytes, offset + length);
        },mmap:function (stream, buffer, offset, length, position, prot, flags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          var ptr;
          var allocated;
          var contents = stream.node.contents;
          // Only make a new copy when MAP_PRIVATE is specified.
          if ( !(flags & 2) &&
                (contents.buffer === buffer || contents.buffer === buffer.buffer) ) {
            // We can't emulate MAP_SHARED when the file is not backed by the buffer
            // we're mapping to (e.g. the HEAP buffer).
            allocated = false;
            ptr = contents.byteOffset;
          } else {
            // Try to avoid unnecessary slices.
            if (position > 0 || position + length < stream.node.usedBytes) {
              if (contents.subarray) {
                contents = contents.subarray(position, position + length);
              } else {
                contents = Array.prototype.slice.call(contents, position, position + length);
              }
            }
            allocated = true;
            ptr = _malloc(length);
            if (!ptr) {
              throw new FS.ErrnoError(ERRNO_CODES.ENOMEM);
            }
            buffer.set(contents, ptr);
          }
          return { ptr: ptr, allocated: allocated };
        },msync:function (stream, buffer, offset, length, mmapFlags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          if (mmapFlags & 2) {
            // MAP_PRIVATE calls need not to be synced back to underlying fs
            return 0;
          }
  
          var bytesWritten = MEMFS.stream_ops.write(stream, buffer, 0, length, offset, false);
          // should we check if bytesWritten and length are the same?
          return 0;
        }}};
  
  var IDBFS={dbs:{},indexedDB:function () {
        if (typeof indexedDB !== 'undefined') return indexedDB;
        var ret = null;
        if (typeof window === 'object') ret = window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;
        assert(ret, 'IDBFS used, but indexedDB not supported');
        return ret;
      },DB_VERSION:21,DB_STORE_NAME:"FILE_DATA",mount:function (mount) {
        // reuse all of the core MEMFS functionality
        return MEMFS.mount.apply(null, arguments);
      },syncfs:function (mount, populate, callback) {
        IDBFS.getLocalSet(mount, function(err, local) {
          if (err) return callback(err);
  
          IDBFS.getRemoteSet(mount, function(err, remote) {
            if (err) return callback(err);
  
            var src = populate ? remote : local;
            var dst = populate ? local : remote;
  
            IDBFS.reconcile(src, dst, callback);
          });
        });
      },getDB:function (name, callback) {
        // check the cache first
        var db = IDBFS.dbs[name];
        if (db) {
          return callback(null, db);
        }
  
        var req;
        try {
          req = IDBFS.indexedDB().open(name, IDBFS.DB_VERSION);
        } catch (e) {
          return callback(e);
        }
        if (!req) {
          return callback("Unable to connect to IndexedDB");
        }
        req.onupgradeneeded = function(e) {
          var db = e.target.result;
          var transaction = e.target.transaction;
  
          var fileStore;
  
          if (db.objectStoreNames.contains(IDBFS.DB_STORE_NAME)) {
            fileStore = transaction.objectStore(IDBFS.DB_STORE_NAME);
          } else {
            fileStore = db.createObjectStore(IDBFS.DB_STORE_NAME);
          }
  
          if (!fileStore.indexNames.contains('timestamp')) {
            fileStore.createIndex('timestamp', 'timestamp', { unique: false });
          }
        };
        req.onsuccess = function() {
          db = req.result;
  
          // add to the cache
          IDBFS.dbs[name] = db;
          callback(null, db);
        };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },getLocalSet:function (mount, callback) {
        var entries = {};
  
        function isRealDir(p) {
          return p !== '.' && p !== '..';
        };
        function toAbsolute(root) {
          return function(p) {
            return PATH.join2(root, p);
          }
        };
  
        var check = FS.readdir(mount.mountpoint).filter(isRealDir).map(toAbsolute(mount.mountpoint));
  
        while (check.length) {
          var path = check.pop();
          var stat;
  
          try {
            stat = FS.stat(path);
          } catch (e) {
            return callback(e);
          }
  
          if (FS.isDir(stat.mode)) {
            check.push.apply(check, FS.readdir(path).filter(isRealDir).map(toAbsolute(path)));
          }
  
          entries[path] = { timestamp: stat.mtime };
        }
  
        return callback(null, { type: 'local', entries: entries });
      },getRemoteSet:function (mount, callback) {
        var entries = {};
  
        IDBFS.getDB(mount.mountpoint, function(err, db) {
          if (err) return callback(err);
  
          try {
            var transaction = db.transaction([IDBFS.DB_STORE_NAME], 'readonly');
            transaction.onerror = function(e) {
              callback(this.error);
              e.preventDefault();
            };
  
            var store = transaction.objectStore(IDBFS.DB_STORE_NAME);
            var index = store.index('timestamp');
  
            index.openKeyCursor().onsuccess = function(event) {
              var cursor = event.target.result;
  
              if (!cursor) {
                return callback(null, { type: 'remote', db: db, entries: entries });
              }
  
              entries[cursor.primaryKey] = { timestamp: cursor.key };
  
              cursor.continue();
            };
          } catch (e) {
            return callback(e);
          }
        });
      },loadLocalEntry:function (path, callback) {
        var stat, node;
  
        try {
          var lookup = FS.lookupPath(path);
          node = lookup.node;
          stat = FS.stat(path);
        } catch (e) {
          return callback(e);
        }
  
        if (FS.isDir(stat.mode)) {
          return callback(null, { timestamp: stat.mtime, mode: stat.mode });
        } else if (FS.isFile(stat.mode)) {
          // Performance consideration: storing a normal JavaScript array to a IndexedDB is much slower than storing a typed array.
          // Therefore always convert the file contents to a typed array first before writing the data to IndexedDB.
          node.contents = MEMFS.getFileDataAsTypedArray(node);
          return callback(null, { timestamp: stat.mtime, mode: stat.mode, contents: node.contents });
        } else {
          return callback(new Error('node type not supported'));
        }
      },storeLocalEntry:function (path, entry, callback) {
        try {
          if (FS.isDir(entry.mode)) {
            FS.mkdir(path, entry.mode);
          } else if (FS.isFile(entry.mode)) {
            FS.writeFile(path, entry.contents, { canOwn: true });
          } else {
            return callback(new Error('node type not supported'));
          }
  
          FS.chmod(path, entry.mode);
          FS.utime(path, entry.timestamp, entry.timestamp);
        } catch (e) {
          return callback(e);
        }
  
        callback(null);
      },removeLocalEntry:function (path, callback) {
        try {
          var lookup = FS.lookupPath(path);
          var stat = FS.stat(path);
  
          if (FS.isDir(stat.mode)) {
            FS.rmdir(path);
          } else if (FS.isFile(stat.mode)) {
            FS.unlink(path);
          }
        } catch (e) {
          return callback(e);
        }
  
        callback(null);
      },loadRemoteEntry:function (store, path, callback) {
        var req = store.get(path);
        req.onsuccess = function(event) { callback(null, event.target.result); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },storeRemoteEntry:function (store, path, entry, callback) {
        var req = store.put(entry, path);
        req.onsuccess = function() { callback(null); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },removeRemoteEntry:function (store, path, callback) {
        var req = store.delete(path);
        req.onsuccess = function() { callback(null); };
        req.onerror = function(e) {
          callback(this.error);
          e.preventDefault();
        };
      },reconcile:function (src, dst, callback) {
        var total = 0;
  
        var create = [];
        Object.keys(src.entries).forEach(function (key) {
          var e = src.entries[key];
          var e2 = dst.entries[key];
          if (!e2 || e.timestamp > e2.timestamp) {
            create.push(key);
            total++;
          }
        });
  
        var remove = [];
        Object.keys(dst.entries).forEach(function (key) {
          var e = dst.entries[key];
          var e2 = src.entries[key];
          if (!e2) {
            remove.push(key);
            total++;
          }
        });
  
        if (!total) {
          return callback(null);
        }
  
        var errored = false;
        var completed = 0;
        var db = src.type === 'remote' ? src.db : dst.db;
        var transaction = db.transaction([IDBFS.DB_STORE_NAME], 'readwrite');
        var store = transaction.objectStore(IDBFS.DB_STORE_NAME);
  
        function done(err) {
          if (err) {
            if (!done.errored) {
              done.errored = true;
              return callback(err);
            }
            return;
          }
          if (++completed >= total) {
            return callback(null);
          }
        };
  
        transaction.onerror = function(e) {
          done(this.error);
          e.preventDefault();
        };
  
        // sort paths in ascending order so directory entries are created
        // before the files inside them
        create.sort().forEach(function (path) {
          if (dst.type === 'local') {
            IDBFS.loadRemoteEntry(store, path, function (err, entry) {
              if (err) return done(err);
              IDBFS.storeLocalEntry(path, entry, done);
            });
          } else {
            IDBFS.loadLocalEntry(path, function (err, entry) {
              if (err) return done(err);
              IDBFS.storeRemoteEntry(store, path, entry, done);
            });
          }
        });
  
        // sort paths in descending order so files are deleted before their
        // parent directories
        remove.sort().reverse().forEach(function(path) {
          if (dst.type === 'local') {
            IDBFS.removeLocalEntry(path, done);
          } else {
            IDBFS.removeRemoteEntry(store, path, done);
          }
        });
      }};
  
  var NODEFS={isWindows:false,staticInit:function () {
        NODEFS.isWindows = !!process.platform.match(/^win/);
        var flags = process["binding"]("constants");
        // Node.js 4 compatibility: it has no namespaces for constants
        if (flags["fs"]) {
          flags = flags["fs"];
        }
        NODEFS.flagsForNodeMap = {
          "1024": flags["O_APPEND"],
          "64": flags["O_CREAT"],
          "128": flags["O_EXCL"],
          "0": flags["O_RDONLY"],
          "2": flags["O_RDWR"],
          "4096": flags["O_SYNC"],
          "512": flags["O_TRUNC"],
          "1": flags["O_WRONLY"]
        };
      },bufferFrom:function (arrayBuffer) {
        // Node.js < 4.5 compatibility: Buffer.from does not support ArrayBuffer
        // Buffer.from before 4.5 was just a method inherited from Uint8Array
        // Buffer.alloc has been added with Buffer.from together, so check it instead
        return Buffer.alloc ? Buffer.from(arrayBuffer) : new Buffer(arrayBuffer);
      },mount:function (mount) {
        assert(ENVIRONMENT_IS_NODE);
        return NODEFS.createNode(null, '/', NODEFS.getMode(mount.opts.root), 0);
      },createNode:function (parent, name, mode, dev) {
        if (!FS.isDir(mode) && !FS.isFile(mode) && !FS.isLink(mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var node = FS.createNode(parent, name, mode);
        node.node_ops = NODEFS.node_ops;
        node.stream_ops = NODEFS.stream_ops;
        return node;
      },getMode:function (path) {
        var stat;
        try {
          stat = fs.lstatSync(path);
          if (NODEFS.isWindows) {
            // Node.js on Windows never represents permission bit 'x', so
            // propagate read bits to execute bits
            stat.mode = stat.mode | ((stat.mode & 292) >> 2);
          }
        } catch (e) {
          if (!e.code) throw e;
          throw new FS.ErrnoError(ERRNO_CODES[e.code]);
        }
        return stat.mode;
      },realPath:function (node) {
        var parts = [];
        while (node.parent !== node) {
          parts.push(node.name);
          node = node.parent;
        }
        parts.push(node.mount.opts.root);
        parts.reverse();
        return PATH.join.apply(null, parts);
      },flagsForNode:function (flags) {
        flags &= ~0x200000 /*O_PATH*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x800 /*O_NONBLOCK*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x8000 /*O_LARGEFILE*/; // Ignore this flag from musl, otherwise node.js fails to open the file.
        flags &= ~0x80000 /*O_CLOEXEC*/; // Some applications may pass it; it makes no sense for a single process.
        var newFlags = 0;
        for (var k in NODEFS.flagsForNodeMap) {
          if (flags & k) {
            newFlags |= NODEFS.flagsForNodeMap[k];
            flags ^= k;
          }
        }
  
        if (!flags) {
          return newFlags;
        } else {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
      },node_ops:{getattr:function (node) {
          var path = NODEFS.realPath(node);
          var stat;
          try {
            stat = fs.lstatSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          // node.js v0.10.20 doesn't report blksize and blocks on Windows. Fake them with default blksize of 4096.
          // See http://support.microsoft.com/kb/140365
          if (NODEFS.isWindows && !stat.blksize) {
            stat.blksize = 4096;
          }
          if (NODEFS.isWindows && !stat.blocks) {
            stat.blocks = (stat.size+stat.blksize-1)/stat.blksize|0;
          }
          return {
            dev: stat.dev,
            ino: stat.ino,
            mode: stat.mode,
            nlink: stat.nlink,
            uid: stat.uid,
            gid: stat.gid,
            rdev: stat.rdev,
            size: stat.size,
            atime: stat.atime,
            mtime: stat.mtime,
            ctime: stat.ctime,
            blksize: stat.blksize,
            blocks: stat.blocks
          };
        },setattr:function (node, attr) {
          var path = NODEFS.realPath(node);
          try {
            if (attr.mode !== undefined) {
              fs.chmodSync(path, attr.mode);
              // update the common node structure mode as well
              node.mode = attr.mode;
            }
            if (attr.timestamp !== undefined) {
              var date = new Date(attr.timestamp);
              fs.utimesSync(path, date, date);
            }
            if (attr.size !== undefined) {
              fs.truncateSync(path, attr.size);
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },lookup:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          var mode = NODEFS.getMode(path);
          return NODEFS.createNode(parent, name, mode);
        },mknod:function (parent, name, mode, dev) {
          var node = NODEFS.createNode(parent, name, mode, dev);
          // create the backing node for this in the fs root as well
          var path = NODEFS.realPath(node);
          try {
            if (FS.isDir(node.mode)) {
              fs.mkdirSync(path, node.mode);
            } else {
              fs.writeFileSync(path, '', { mode: node.mode });
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
          return node;
        },rename:function (oldNode, newDir, newName) {
          var oldPath = NODEFS.realPath(oldNode);
          var newPath = PATH.join2(NODEFS.realPath(newDir), newName);
          try {
            fs.renameSync(oldPath, newPath);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },unlink:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          try {
            fs.unlinkSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },rmdir:function (parent, name) {
          var path = PATH.join2(NODEFS.realPath(parent), name);
          try {
            fs.rmdirSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },readdir:function (node) {
          var path = NODEFS.realPath(node);
          try {
            return fs.readdirSync(path);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },symlink:function (parent, newName, oldPath) {
          var newPath = PATH.join2(NODEFS.realPath(parent), newName);
          try {
            fs.symlinkSync(oldPath, newPath);
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },readlink:function (node) {
          var path = NODEFS.realPath(node);
          try {
            path = fs.readlinkSync(path);
            path = NODEJS_PATH.relative(NODEJS_PATH.resolve(node.mount.opts.root), path);
            return path;
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        }},stream_ops:{open:function (stream) {
          var path = NODEFS.realPath(stream.node);
          try {
            if (FS.isFile(stream.node.mode)) {
              stream.nfd = fs.openSync(path, NODEFS.flagsForNode(stream.flags));
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },close:function (stream) {
          try {
            if (FS.isFile(stream.node.mode) && stream.nfd) {
              fs.closeSync(stream.nfd);
            }
          } catch (e) {
            if (!e.code) throw e;
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },read:function (stream, buffer, offset, length, position) {
          // Node.js < 6 compatibility: node errors on 0 length reads
          if (length === 0) return 0;
          try {
            return fs.readSync(stream.nfd, NODEFS.bufferFrom(buffer.buffer), offset, length, position);
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },write:function (stream, buffer, offset, length, position) {
          try {
            return fs.writeSync(stream.nfd, NODEFS.bufferFrom(buffer.buffer), offset, length, position);
          } catch (e) {
            throw new FS.ErrnoError(ERRNO_CODES[e.code]);
          }
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              try {
                var stat = fs.fstatSync(stream.nfd);
                position += stat.size;
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES[e.code]);
              }
            }
          }
  
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
  
          return position;
        }}};
  
  var WORKERFS={DIR_MODE:16895,FILE_MODE:33279,reader:null,mount:function (mount) {
        assert(ENVIRONMENT_IS_WORKER);
        if (!WORKERFS.reader) WORKERFS.reader = new FileReaderSync();
        var root = WORKERFS.createNode(null, '/', WORKERFS.DIR_MODE, 0);
        var createdParents = {};
        function ensureParent(path) {
          // return the parent node, creating subdirs as necessary
          var parts = path.split('/');
          var parent = root;
          for (var i = 0; i < parts.length-1; i++) {
            var curr = parts.slice(0, i+1).join('/');
            // Issue 4254: Using curr as a node name will prevent the node
            // from being found in FS.nameTable when FS.open is called on
            // a path which holds a child of this node,
            // given that all FS functions assume node names
            // are just their corresponding parts within their given path,
            // rather than incremental aggregates which include their parent's
            // directories.
            if (!createdParents[curr]) {
              createdParents[curr] = WORKERFS.createNode(parent, parts[i], WORKERFS.DIR_MODE, 0);
            }
            parent = createdParents[curr];
          }
          return parent;
        }
        function base(path) {
          var parts = path.split('/');
          return parts[parts.length-1];
        }
        // We also accept FileList here, by using Array.prototype
        Array.prototype.forEach.call(mount.opts["files"] || [], function(file) {
          WORKERFS.createNode(ensureParent(file.name), base(file.name), WORKERFS.FILE_MODE, 0, file, file.lastModifiedDate);
        });
        (mount.opts["blobs"] || []).forEach(function(obj) {
          WORKERFS.createNode(ensureParent(obj["name"]), base(obj["name"]), WORKERFS.FILE_MODE, 0, obj["data"]);
        });
        (mount.opts["packages"] || []).forEach(function(pack) {
          pack['metadata'].files.forEach(function(file) {
            var name = file.filename.substr(1); // remove initial slash
            WORKERFS.createNode(ensureParent(name), base(name), WORKERFS.FILE_MODE, 0, pack['blob'].slice(file.start, file.end));
          });
        });
        return root;
      },createNode:function (parent, name, mode, dev, contents, mtime) {
        var node = FS.createNode(parent, name, mode);
        node.mode = mode;
        node.node_ops = WORKERFS.node_ops;
        node.stream_ops = WORKERFS.stream_ops;
        node.timestamp = (mtime || new Date).getTime();
        assert(WORKERFS.FILE_MODE !== WORKERFS.DIR_MODE);
        if (mode === WORKERFS.FILE_MODE) {
          node.size = contents.size;
          node.contents = contents;
        } else {
          node.size = 4096;
          node.contents = {};
        }
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },node_ops:{getattr:function (node) {
          return {
            dev: 1,
            ino: undefined,
            mode: node.mode,
            nlink: 1,
            uid: 0,
            gid: 0,
            rdev: undefined,
            size: node.size,
            atime: new Date(node.timestamp),
            mtime: new Date(node.timestamp),
            ctime: new Date(node.timestamp),
            blksize: 4096,
            blocks: Math.ceil(node.size / 4096),
          };
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
        },lookup:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        },mknod:function (parent, name, mode, dev) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },rename:function (oldNode, newDir, newName) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },unlink:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },rmdir:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },readdir:function (node) {
          var entries = ['.', '..'];
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newName, oldPath) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        },readlink:function (node) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          if (position >= stream.node.size) return 0;
          var chunk = stream.node.contents.slice(position, position + length);
          var ab = WORKERFS.reader.readAsArrayBuffer(chunk);
          buffer.set(new Uint8Array(ab), offset);
          return chunk.size;
        },write:function (stream, buffer, offset, length, position) {
          throw new FS.ErrnoError(ERRNO_CODES.EIO);
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.size;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return position;
        }}};
  
  var _stdin=STATICTOP; STATICTOP += 16;;
  
  var _stdout=STATICTOP; STATICTOP += 16;;
  
  var _stderr=STATICTOP; STATICTOP += 16;;var FS={root:null,mounts:[],devices:{},streams:[],nextInode:1,nameTable:null,currentPath:"/",initialized:false,ignorePermissions:true,trackingDelegate:{},tracking:{openFlags:{READ:1,WRITE:2}},ErrnoError:null,genericErrors:{},filesystems:null,syncFSRequests:0,handleFSError:function (e) {
        if (!(e instanceof FS.ErrnoError)) throw e + ' : ' + stackTrace();
        return ___setErrNo(e.errno);
      },lookupPath:function (path, opts) {
        path = PATH.resolve(FS.cwd(), path);
        opts = opts || {};
  
        if (!path) return { path: '', node: null };
  
        var defaults = {
          follow_mount: true,
          recurse_count: 0
        };
        for (var key in defaults) {
          if (opts[key] === undefined) {
            opts[key] = defaults[key];
          }
        }
  
        if (opts.recurse_count > 8) {  // max recursive lookup of 8
          throw new FS.ErrnoError(ERRNO_CODES.ELOOP);
        }
  
        // split the path
        var parts = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), false);
  
        // start at the root
        var current = FS.root;
        var current_path = '/';
  
        for (var i = 0; i < parts.length; i++) {
          var islast = (i === parts.length-1);
          if (islast && opts.parent) {
            // stop resolving
            break;
          }
  
          current = FS.lookupNode(current, parts[i]);
          current_path = PATH.join2(current_path, parts[i]);
  
          // jump to the mount's root node if this is a mountpoint
          if (FS.isMountpoint(current)) {
            if (!islast || (islast && opts.follow_mount)) {
              current = current.mounted.root;
            }
          }
  
          // by default, lookupPath will not follow a symlink if it is the final path component.
          // setting opts.follow = true will override this behavior.
          if (!islast || opts.follow) {
            var count = 0;
            while (FS.isLink(current.mode)) {
              var link = FS.readlink(current_path);
              current_path = PATH.resolve(PATH.dirname(current_path), link);
  
              var lookup = FS.lookupPath(current_path, { recurse_count: opts.recurse_count });
              current = lookup.node;
  
              if (count++ > 40) {  // limit max consecutive symlinks to 40 (SYMLOOP_MAX).
                throw new FS.ErrnoError(ERRNO_CODES.ELOOP);
              }
            }
          }
        }
  
        return { path: current_path, node: current };
      },getPath:function (node) {
        var path;
        while (true) {
          if (FS.isRoot(node)) {
            var mount = node.mount.mountpoint;
            if (!path) return mount;
            return mount[mount.length-1] !== '/' ? mount + '/' + path : mount + path;
          }
          path = path ? node.name + '/' + path : node.name;
          node = node.parent;
        }
      },hashName:function (parentid, name) {
        var hash = 0;
  
  
        for (var i = 0; i < name.length; i++) {
          hash = ((hash << 5) - hash + name.charCodeAt(i)) | 0;
        }
        return ((parentid + hash) >>> 0) % FS.nameTable.length;
      },hashAddNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        node.name_next = FS.nameTable[hash];
        FS.nameTable[hash] = node;
      },hashRemoveNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        if (FS.nameTable[hash] === node) {
          FS.nameTable[hash] = node.name_next;
        } else {
          var current = FS.nameTable[hash];
          while (current) {
            if (current.name_next === node) {
              current.name_next = node.name_next;
              break;
            }
            current = current.name_next;
          }
        }
      },lookupNode:function (parent, name) {
        var err = FS.mayLookup(parent);
        if (err) {
          throw new FS.ErrnoError(err, parent);
        }
        var hash = FS.hashName(parent.id, name);
        for (var node = FS.nameTable[hash]; node; node = node.name_next) {
          var nodeName = node.name;
          if (node.parent.id === parent.id && nodeName === name) {
            return node;
          }
        }
        // if we failed to find it in the cache, call into the VFS
        return FS.lookup(parent, name);
      },createNode:function (parent, name, mode, rdev) {
        if (!FS.FSNode) {
          FS.FSNode = function(parent, name, mode, rdev) {
            if (!parent) {
              parent = this;  // root node sets parent to itself
            }
            this.parent = parent;
            this.mount = parent.mount;
            this.mounted = null;
            this.id = FS.nextInode++;
            this.name = name;
            this.mode = mode;
            this.node_ops = {};
            this.stream_ops = {};
            this.rdev = rdev;
          };
  
          FS.FSNode.prototype = {};
  
          // compatibility
          var readMode = 292 | 73;
          var writeMode = 146;
  
          // NOTE we must use Object.defineProperties instead of individual calls to
          // Object.defineProperty in order to make closure compiler happy
          Object.defineProperties(FS.FSNode.prototype, {
            read: {
              get: function() { return (this.mode & readMode) === readMode; },
              set: function(val) { val ? this.mode |= readMode : this.mode &= ~readMode; }
            },
            write: {
              get: function() { return (this.mode & writeMode) === writeMode; },
              set: function(val) { val ? this.mode |= writeMode : this.mode &= ~writeMode; }
            },
            isFolder: {
              get: function() { return FS.isDir(this.mode); }
            },
            isDevice: {
              get: function() { return FS.isChrdev(this.mode); }
            }
          });
        }
  
        var node = new FS.FSNode(parent, name, mode, rdev);
  
        FS.hashAddNode(node);
  
        return node;
      },destroyNode:function (node) {
        FS.hashRemoveNode(node);
      },isRoot:function (node) {
        return node === node.parent;
      },isMountpoint:function (node) {
        return !!node.mounted;
      },isFile:function (mode) {
        return (mode & 61440) === 32768;
      },isDir:function (mode) {
        return (mode & 61440) === 16384;
      },isLink:function (mode) {
        return (mode & 61440) === 40960;
      },isChrdev:function (mode) {
        return (mode & 61440) === 8192;
      },isBlkdev:function (mode) {
        return (mode & 61440) === 24576;
      },isFIFO:function (mode) {
        return (mode & 61440) === 4096;
      },isSocket:function (mode) {
        return (mode & 49152) === 49152;
      },flagModes:{"r":0,"rs":1052672,"r+":2,"w":577,"wx":705,"xw":705,"w+":578,"wx+":706,"xw+":706,"a":1089,"ax":1217,"xa":1217,"a+":1090,"ax+":1218,"xa+":1218},modeStringToFlags:function (str) {
        var flags = FS.flagModes[str];
        if (typeof flags === 'undefined') {
          throw new Error('Unknown file open mode: ' + str);
        }
        return flags;
      },flagsToPermissionString:function (flag) {
        var perms = ['r', 'w', 'rw'][flag & 3];
        if ((flag & 512)) {
          perms += 'w';
        }
        return perms;
      },nodePermissions:function (node, perms) {
        if (FS.ignorePermissions) {
          return 0;
        }
        // return 0 if any user, group or owner bits are set.
        if (perms.indexOf('r') !== -1 && !(node.mode & 292)) {
          return ERRNO_CODES.EACCES;
        } else if (perms.indexOf('w') !== -1 && !(node.mode & 146)) {
          return ERRNO_CODES.EACCES;
        } else if (perms.indexOf('x') !== -1 && !(node.mode & 73)) {
          return ERRNO_CODES.EACCES;
        }
        return 0;
      },mayLookup:function (dir) {
        var err = FS.nodePermissions(dir, 'x');
        if (err) return err;
        if (!dir.node_ops.lookup) return ERRNO_CODES.EACCES;
        return 0;
      },mayCreate:function (dir, name) {
        try {
          var node = FS.lookupNode(dir, name);
          return ERRNO_CODES.EEXIST;
        } catch (e) {
        }
        return FS.nodePermissions(dir, 'wx');
      },mayDelete:function (dir, name, isdir) {
        var node;
        try {
          node = FS.lookupNode(dir, name);
        } catch (e) {
          return e.errno;
        }
        var err = FS.nodePermissions(dir, 'wx');
        if (err) {
          return err;
        }
        if (isdir) {
          if (!FS.isDir(node.mode)) {
            return ERRNO_CODES.ENOTDIR;
          }
          if (FS.isRoot(node) || FS.getPath(node) === FS.cwd()) {
            return ERRNO_CODES.EBUSY;
          }
        } else {
          if (FS.isDir(node.mode)) {
            return ERRNO_CODES.EISDIR;
          }
        }
        return 0;
      },mayOpen:function (node, flags) {
        if (!node) {
          return ERRNO_CODES.ENOENT;
        }
        if (FS.isLink(node.mode)) {
          return ERRNO_CODES.ELOOP;
        } else if (FS.isDir(node.mode)) {
          if (FS.flagsToPermissionString(flags) !== 'r' || // opening for write
              (flags & 512)) { // TODO: check for O_SEARCH? (== search for dir only)
            return ERRNO_CODES.EISDIR;
          }
        }
        return FS.nodePermissions(node, FS.flagsToPermissionString(flags));
      },MAX_OPEN_FDS:4096,nextfd:function (fd_start, fd_end) {
        fd_start = fd_start || 0;
        fd_end = fd_end || FS.MAX_OPEN_FDS;
        for (var fd = fd_start; fd <= fd_end; fd++) {
          if (!FS.streams[fd]) {
            return fd;
          }
        }
        throw new FS.ErrnoError(ERRNO_CODES.EMFILE);
      },getStream:function (fd) {
        return FS.streams[fd];
      },createStream:function (stream, fd_start, fd_end) {
        if (!FS.FSStream) {
          FS.FSStream = function(){};
          FS.FSStream.prototype = {};
          // compatibility
          Object.defineProperties(FS.FSStream.prototype, {
            object: {
              get: function() { return this.node; },
              set: function(val) { this.node = val; }
            },
            isRead: {
              get: function() { return (this.flags & 2097155) !== 1; }
            },
            isWrite: {
              get: function() { return (this.flags & 2097155) !== 0; }
            },
            isAppend: {
              get: function() { return (this.flags & 1024); }
            }
          });
        }
        // clone it, so we can return an instance of FSStream
        var newStream = new FS.FSStream();
        for (var p in stream) {
          newStream[p] = stream[p];
        }
        stream = newStream;
        var fd = FS.nextfd(fd_start, fd_end);
        stream.fd = fd;
        FS.streams[fd] = stream;
        return stream;
      },closeStream:function (fd) {
        FS.streams[fd] = null;
      },chrdev_stream_ops:{open:function (stream) {
          var device = FS.getDevice(stream.node.rdev);
          // override node's stream ops with the device's
          stream.stream_ops = device.stream_ops;
          // forward the open call
          if (stream.stream_ops.open) {
            stream.stream_ops.open(stream);
          }
        },llseek:function () {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }},major:function (dev) {
        return ((dev) >> 8);
      },minor:function (dev) {
        return ((dev) & 0xff);
      },makedev:function (ma, mi) {
        return ((ma) << 8 | (mi));
      },registerDevice:function (dev, ops) {
        FS.devices[dev] = { stream_ops: ops };
      },getDevice:function (dev) {
        return FS.devices[dev];
      },getMounts:function (mount) {
        var mounts = [];
        var check = [mount];
  
        while (check.length) {
          var m = check.pop();
  
          mounts.push(m);
  
          check.push.apply(check, m.mounts);
        }
  
        return mounts;
      },syncfs:function (populate, callback) {
        if (typeof(populate) === 'function') {
          callback = populate;
          populate = false;
        }
  
        FS.syncFSRequests++;
  
        if (FS.syncFSRequests > 1) {
          console.log('warning: ' + FS.syncFSRequests + ' FS.syncfs operations in flight at once, probably just doing extra work');
        }
  
        var mounts = FS.getMounts(FS.root.mount);
        var completed = 0;
  
        function doCallback(err) {
          assert(FS.syncFSRequests > 0);
          FS.syncFSRequests--;
          return callback(err);
        }
  
        function done(err) {
          if (err) {
            if (!done.errored) {
              done.errored = true;
              return doCallback(err);
            }
            return;
          }
          if (++completed >= mounts.length) {
            doCallback(null);
          }
        };
  
        // sync all mounts
        mounts.forEach(function (mount) {
          if (!mount.type.syncfs) {
            return done(null);
          }
          mount.type.syncfs(mount, populate, done);
        });
      },mount:function (type, opts, mountpoint) {
        var root = mountpoint === '/';
        var pseudo = !mountpoint;
        var node;
  
        if (root && FS.root) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        } else if (!root && !pseudo) {
          var lookup = FS.lookupPath(mountpoint, { follow_mount: false });
  
          mountpoint = lookup.path;  // use the absolute path
          node = lookup.node;
  
          if (FS.isMountpoint(node)) {
            throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
          }
  
          if (!FS.isDir(node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
          }
        }
  
        var mount = {
          type: type,
          opts: opts,
          mountpoint: mountpoint,
          mounts: []
        };
  
        // create a root node for the fs
        var mountRoot = type.mount(mount);
        mountRoot.mount = mount;
        mount.root = mountRoot;
  
        if (root) {
          FS.root = mountRoot;
        } else if (node) {
          // set as a mountpoint
          node.mounted = mount;
  
          // add the new mount to the current mount's children
          if (node.mount) {
            node.mount.mounts.push(mount);
          }
        }
  
        return mountRoot;
      },unmount:function (mountpoint) {
        var lookup = FS.lookupPath(mountpoint, { follow_mount: false });
  
        if (!FS.isMountpoint(lookup.node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
  
        // destroy the nodes for this mount, and all its child mounts
        var node = lookup.node;
        var mount = node.mounted;
        var mounts = FS.getMounts(mount);
  
        Object.keys(FS.nameTable).forEach(function (hash) {
          var current = FS.nameTable[hash];
  
          while (current) {
            var next = current.name_next;
  
            if (mounts.indexOf(current.mount) !== -1) {
              FS.destroyNode(current);
            }
  
            current = next;
          }
        });
  
        // no longer a mountpoint
        node.mounted = null;
  
        // remove this mount from the child mounts
        var idx = node.mount.mounts.indexOf(mount);
        assert(idx !== -1);
        node.mount.mounts.splice(idx, 1);
      },lookup:function (parent, name) {
        return parent.node_ops.lookup(parent, name);
      },mknod:function (path, mode, dev) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        if (!name || name === '.' || name === '..') {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var err = FS.mayCreate(parent, name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.mknod) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return parent.node_ops.mknod(parent, name, mode, dev);
      },create:function (path, mode) {
        mode = mode !== undefined ? mode : 438 /* 0666 */;
        mode &= 4095;
        mode |= 32768;
        return FS.mknod(path, mode, 0);
      },mkdir:function (path, mode) {
        mode = mode !== undefined ? mode : 511 /* 0777 */;
        mode &= 511 | 512;
        mode |= 16384;
        return FS.mknod(path, mode, 0);
      },mkdirTree:function (path, mode) {
        var dirs = path.split('/');
        var d = '';
        for (var i = 0; i < dirs.length; ++i) {
          if (!dirs[i]) continue;
          d += '/' + dirs[i];
          try {
            FS.mkdir(d, mode);
          } catch(e) {
            if (e.errno != ERRNO_CODES.EEXIST) throw e;
          }
        }
      },mkdev:function (path, mode, dev) {
        if (typeof(dev) === 'undefined') {
          dev = mode;
          mode = 438 /* 0666 */;
        }
        mode |= 8192;
        return FS.mknod(path, mode, dev);
      },symlink:function (oldpath, newpath) {
        if (!PATH.resolve(oldpath)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        var lookup = FS.lookupPath(newpath, { parent: true });
        var parent = lookup.node;
        if (!parent) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        var newname = PATH.basename(newpath);
        var err = FS.mayCreate(parent, newname);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.symlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return parent.node_ops.symlink(parent, newname, oldpath);
      },rename:function (old_path, new_path) {
        var old_dirname = PATH.dirname(old_path);
        var new_dirname = PATH.dirname(new_path);
        var old_name = PATH.basename(old_path);
        var new_name = PATH.basename(new_path);
        // parents must exist
        var lookup, old_dir, new_dir;
        try {
          lookup = FS.lookupPath(old_path, { parent: true });
          old_dir = lookup.node;
          lookup = FS.lookupPath(new_path, { parent: true });
          new_dir = lookup.node;
        } catch (e) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        if (!old_dir || !new_dir) throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        // need to be part of the same mount
        if (old_dir.mount !== new_dir.mount) {
          throw new FS.ErrnoError(ERRNO_CODES.EXDEV);
        }
        // source must exist
        var old_node = FS.lookupNode(old_dir, old_name);
        // old path should not be an ancestor of the new path
        var relative = PATH.relative(old_path, new_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        // new path should not be an ancestor of the old path
        relative = PATH.relative(new_path, old_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
        }
        // see if the new path already exists
        var new_node;
        try {
          new_node = FS.lookupNode(new_dir, new_name);
        } catch (e) {
          // not fatal
        }
        // early out if nothing needs to change
        if (old_node === new_node) {
          return;
        }
        // we'll need to delete the old entry
        var isdir = FS.isDir(old_node.mode);
        var err = FS.mayDelete(old_dir, old_name, isdir);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        // need delete permissions if we'll be overwriting.
        // need create permissions if new doesn't already exist.
        err = new_node ?
          FS.mayDelete(new_dir, new_name, isdir) :
          FS.mayCreate(new_dir, new_name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!old_dir.node_ops.rename) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(old_node) || (new_node && FS.isMountpoint(new_node))) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        // if we are going to change the parent, check write permissions
        if (new_dir !== old_dir) {
          err = FS.nodePermissions(old_dir, 'w');
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        try {
          if (FS.trackingDelegate['willMovePath']) {
            FS.trackingDelegate['willMovePath'](old_path, new_path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willMovePath']('"+old_path+"', '"+new_path+"') threw an exception: " + e.message);
        }
        // remove the node from the lookup hash
        FS.hashRemoveNode(old_node);
        // do the underlying fs rename
        try {
          old_dir.node_ops.rename(old_node, new_dir, new_name);
        } catch (e) {
          throw e;
        } finally {
          // add the node back to the hash (in case node_ops.rename
          // changed its name)
          FS.hashAddNode(old_node);
        }
        try {
          if (FS.trackingDelegate['onMovePath']) FS.trackingDelegate['onMovePath'](old_path, new_path);
        } catch(e) {
          console.log("FS.trackingDelegate['onMovePath']('"+old_path+"', '"+new_path+"') threw an exception: " + e.message);
        }
      },rmdir:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, true);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.rmdir) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        try {
          if (FS.trackingDelegate['willDeletePath']) {
            FS.trackingDelegate['willDeletePath'](path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: " + e.message);
        }
        parent.node_ops.rmdir(parent, name);
        FS.destroyNode(node);
        try {
          if (FS.trackingDelegate['onDeletePath']) FS.trackingDelegate['onDeletePath'](path);
        } catch(e) {
          console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: " + e.message);
        }
      },readdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        if (!node.node_ops.readdir) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        return node.node_ops.readdir(node);
      },unlink:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, false);
        if (err) {
          // According to POSIX, we should map EISDIR to EPERM, but
          // we instead do what Linux does (and we must, as we use
          // the musl linux libc).
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.unlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        try {
          if (FS.trackingDelegate['willDeletePath']) {
            FS.trackingDelegate['willDeletePath'](path);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['willDeletePath']('"+path+"') threw an exception: " + e.message);
        }
        parent.node_ops.unlink(parent, name);
        FS.destroyNode(node);
        try {
          if (FS.trackingDelegate['onDeletePath']) FS.trackingDelegate['onDeletePath'](path);
        } catch(e) {
          console.log("FS.trackingDelegate['onDeletePath']('"+path+"') threw an exception: " + e.message);
        }
      },readlink:function (path) {
        var lookup = FS.lookupPath(path);
        var link = lookup.node;
        if (!link) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        if (!link.node_ops.readlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        return PATH.resolve(FS.getPath(link.parent), link.node_ops.readlink(link));
      },stat:function (path, dontFollow) {
        var lookup = FS.lookupPath(path, { follow: !dontFollow });
        var node = lookup.node;
        if (!node) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        if (!node.node_ops.getattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return node.node_ops.getattr(node);
      },lstat:function (path) {
        return FS.stat(path, true);
      },chmod:function (path, mode, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        node.node_ops.setattr(node, {
          mode: (mode & 4095) | (node.mode & ~4095),
          timestamp: Date.now()
        });
      },lchmod:function (path, mode) {
        FS.chmod(path, mode, true);
      },fchmod:function (fd, mode) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        FS.chmod(stream.node, mode);
      },chown:function (path, uid, gid, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        node.node_ops.setattr(node, {
          timestamp: Date.now()
          // we ignore the uid / gid for now
        });
      },lchown:function (path, uid, gid) {
        FS.chown(path, uid, gid, true);
      },fchown:function (fd, uid, gid) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        FS.chown(stream.node, uid, gid);
      },truncate:function (path, len) {
        if (len < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: true });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isDir(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!FS.isFile(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var err = FS.nodePermissions(node, 'w');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        node.node_ops.setattr(node, {
          size: len,
          timestamp: Date.now()
        });
      },ftruncate:function (fd, len) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        FS.truncate(stream.node, len);
      },utime:function (path, atime, mtime) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        node.node_ops.setattr(node, {
          timestamp: Math.max(atime, mtime)
        });
      },open:function (path, flags, mode, fd_start, fd_end) {
        if (path === "") {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        flags = typeof flags === 'string' ? FS.modeStringToFlags(flags) : flags;
        mode = typeof mode === 'undefined' ? 438 /* 0666 */ : mode;
        if ((flags & 64)) {
          mode = (mode & 4095) | 32768;
        } else {
          mode = 0;
        }
        var node;
        if (typeof path === 'object') {
          node = path;
        } else {
          path = PATH.normalize(path);
          try {
            var lookup = FS.lookupPath(path, {
              follow: !(flags & 131072)
            });
            node = lookup.node;
          } catch (e) {
            // ignore
          }
        }
        // perhaps we need to create the node
        var created = false;
        if ((flags & 64)) {
          if (node) {
            // if O_CREAT and O_EXCL are set, error out if the node already exists
            if ((flags & 128)) {
              throw new FS.ErrnoError(ERRNO_CODES.EEXIST);
            }
          } else {
            // node doesn't exist, try to create it
            node = FS.mknod(path, mode, 0);
            created = true;
          }
        }
        if (!node) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        // can't truncate a device
        if (FS.isChrdev(node.mode)) {
          flags &= ~512;
        }
        // if asked only for a directory, then this must be one
        if ((flags & 65536) && !FS.isDir(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        // check permissions, if this is not a file we just created now (it is ok to
        // create and write to a file with read-only permissions; it is read-only
        // for later use)
        if (!created) {
          var err = FS.mayOpen(node, flags);
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        // do truncation if necessary
        if ((flags & 512)) {
          FS.truncate(node, 0);
        }
        // we've already handled these, don't pass down to the underlying vfs
        flags &= ~(128 | 512);
  
        // register the stream with the filesystem
        var stream = FS.createStream({
          node: node,
          path: FS.getPath(node),  // we want the absolute path to the node
          flags: flags,
          seekable: true,
          position: 0,
          stream_ops: node.stream_ops,
          // used by the file family libc calls (fopen, fwrite, ferror, etc.)
          ungotten: [],
          error: false
        }, fd_start, fd_end);
        // call the new stream's open function
        if (stream.stream_ops.open) {
          stream.stream_ops.open(stream);
        }
        if (Module['logReadFiles'] && !(flags & 1)) {
          if (!FS.readFiles) FS.readFiles = {};
          if (!(path in FS.readFiles)) {
            FS.readFiles[path] = 1;
            err('read file: ' + path);
          }
        }
        try {
          if (FS.trackingDelegate['onOpenFile']) {
            var trackingFlags = 0;
            if ((flags & 2097155) !== 1) {
              trackingFlags |= FS.tracking.openFlags.READ;
            }
            if ((flags & 2097155) !== 0) {
              trackingFlags |= FS.tracking.openFlags.WRITE;
            }
            FS.trackingDelegate['onOpenFile'](path, trackingFlags);
          }
        } catch(e) {
          console.log("FS.trackingDelegate['onOpenFile']('"+path+"', flags) threw an exception: " + e.message);
        }
        return stream;
      },close:function (stream) {
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (stream.getdents) stream.getdents = null; // free readdir state
        try {
          if (stream.stream_ops.close) {
            stream.stream_ops.close(stream);
          }
        } catch (e) {
          throw e;
        } finally {
          FS.closeStream(stream.fd);
        }
        stream.fd = null;
      },isClosed:function (stream) {
        return stream.fd === null;
      },llseek:function (stream, offset, whence) {
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (!stream.seekable || !stream.stream_ops.llseek) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        stream.position = stream.stream_ops.llseek(stream, offset, whence);
        stream.ungotten = [];
        return stream.position;
      },read:function (stream, buffer, offset, length, position) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if ((stream.flags & 2097155) === 1) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!stream.stream_ops.read) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var seeking = typeof position !== 'undefined';
        if (!seeking) {
          position = stream.position;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        var bytesRead = stream.stream_ops.read(stream, buffer, offset, length, position);
        if (!seeking) stream.position += bytesRead;
        return bytesRead;
      },write:function (stream, buffer, offset, length, position, canOwn) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!stream.stream_ops.write) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if (stream.flags & 1024) {
          // seek to the end before writing in append mode
          FS.llseek(stream, 0, 2);
        }
        var seeking = typeof position !== 'undefined';
        if (!seeking) {
          position = stream.position;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        var bytesWritten = stream.stream_ops.write(stream, buffer, offset, length, position, canOwn);
        if (!seeking) stream.position += bytesWritten;
        try {
          if (stream.path && FS.trackingDelegate['onWriteToFile']) FS.trackingDelegate['onWriteToFile'](stream.path);
        } catch(e) {
          console.log("FS.trackingDelegate['onWriteToFile']('"+path+"') threw an exception: " + e.message);
        }
        return bytesWritten;
      },allocate:function (stream, offset, length) {
        if (FS.isClosed(stream)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (offset < 0 || length <= 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 2097155) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (!FS.isFile(stream.node.mode) && !FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
        }
        if (!stream.stream_ops.allocate) {
          throw new FS.ErrnoError(ERRNO_CODES.EOPNOTSUPP);
        }
        stream.stream_ops.allocate(stream, offset, length);
      },mmap:function (stream, buffer, offset, length, position, prot, flags) {
        // TODO if PROT is PROT_WRITE, make sure we have write access
        if ((stream.flags & 2097155) === 1) {
          throw new FS.ErrnoError(ERRNO_CODES.EACCES);
        }
        if (!stream.stream_ops.mmap) {
          throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
        }
        return stream.stream_ops.mmap(stream, buffer, offset, length, position, prot, flags);
      },msync:function (stream, buffer, offset, length, mmapFlags) {
        if (!stream || !stream.stream_ops.msync) {
          return 0;
        }
        return stream.stream_ops.msync(stream, buffer, offset, length, mmapFlags);
      },munmap:function (stream) {
        return 0;
      },ioctl:function (stream, cmd, arg) {
        if (!stream.stream_ops.ioctl) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTTY);
        }
        return stream.stream_ops.ioctl(stream, cmd, arg);
      },readFile:function (path, opts) {
        opts = opts || {};
        opts.flags = opts.flags || 'r';
        opts.encoding = opts.encoding || 'binary';
        if (opts.encoding !== 'utf8' && opts.encoding !== 'binary') {
          throw new Error('Invalid encoding type "' + opts.encoding + '"');
        }
        var ret;
        var stream = FS.open(path, opts.flags);
        var stat = FS.stat(path);
        var length = stat.size;
        var buf = new Uint8Array(length);
        FS.read(stream, buf, 0, length, 0);
        if (opts.encoding === 'utf8') {
          ret = UTF8ArrayToString(buf, 0);
        } else if (opts.encoding === 'binary') {
          ret = buf;
        }
        FS.close(stream);
        return ret;
      },writeFile:function (path, data, opts) {
        opts = opts || {};
        opts.flags = opts.flags || 'w';
        var stream = FS.open(path, opts.flags, opts.mode);
        if (typeof data === 'string') {
          var buf = new Uint8Array(lengthBytesUTF8(data)+1);
          var actualNumBytes = stringToUTF8Array(data, buf, 0, buf.length);
          FS.write(stream, buf, 0, actualNumBytes, undefined, opts.canOwn);
        } else if (ArrayBuffer.isView(data)) {
          FS.write(stream, data, 0, data.byteLength, undefined, opts.canOwn);
        } else {
          throw new Error('Unsupported data type');
        }
        FS.close(stream);
      },cwd:function () {
        return FS.currentPath;
      },chdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        if (lookup.node === null) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        if (!FS.isDir(lookup.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        var err = FS.nodePermissions(lookup.node, 'x');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        FS.currentPath = lookup.path;
      },createDefaultDirectories:function () {
        FS.mkdir('/tmp');
        FS.mkdir('/home');
        FS.mkdir('/home/web_user');
      },createDefaultDevices:function () {
        // create /dev
        FS.mkdir('/dev');
        // setup /dev/null
        FS.registerDevice(FS.makedev(1, 3), {
          read: function() { return 0; },
          write: function(stream, buffer, offset, length, pos) { return length; }
        });
        FS.mkdev('/dev/null', FS.makedev(1, 3));
        // setup /dev/tty and /dev/tty1
        // stderr needs to print output using Module['printErr']
        // so we register a second tty just for it.
        TTY.register(FS.makedev(5, 0), TTY.default_tty_ops);
        TTY.register(FS.makedev(6, 0), TTY.default_tty1_ops);
        FS.mkdev('/dev/tty', FS.makedev(5, 0));
        FS.mkdev('/dev/tty1', FS.makedev(6, 0));
        // setup /dev/[u]random
        var random_device;
        if (typeof crypto !== 'undefined') {
          // for modern web browsers
          var randomBuffer = new Uint8Array(1);
          random_device = function() { crypto.getRandomValues(randomBuffer); return randomBuffer[0]; };
        } else if (ENVIRONMENT_IS_NODE) {
          // for nodejs
          random_device = function() { return require('crypto')['randomBytes'](1)[0]; };
        } else {
          // default for ES5 platforms
          random_device = function() { return (Math.random()*256)|0; };
        }
        FS.createDevice('/dev', 'random', random_device);
        FS.createDevice('/dev', 'urandom', random_device);
        // we're not going to emulate the actual shm device,
        // just create the tmp dirs that reside in it commonly
        FS.mkdir('/dev/shm');
        FS.mkdir('/dev/shm/tmp');
      },createSpecialDirectories:function () {
        // create /proc/self/fd which allows /proc/self/fd/6 => readlink gives the name of the stream for fd 6 (see test_unistd_ttyname)
        FS.mkdir('/proc');
        FS.mkdir('/proc/self');
        FS.mkdir('/proc/self/fd');
        FS.mount({
          mount: function() {
            var node = FS.createNode('/proc/self', 'fd', 16384 | 511 /* 0777 */, 73);
            node.node_ops = {
              lookup: function(parent, name) {
                var fd = +name;
                var stream = FS.getStream(fd);
                if (!stream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
                var ret = {
                  parent: null,
                  mount: { mountpoint: 'fake' },
                  node_ops: { readlink: function() { return stream.path } }
                };
                ret.parent = ret; // make it look like a simple root node
                return ret;
              }
            };
            return node;
          }
        }, {}, '/proc/self/fd');
      },createStandardStreams:function () {
        // TODO deprecate the old functionality of a single
        // input / output callback and that utilizes FS.createDevice
        // and instead require a unique set of stream ops
  
        // by default, we symlink the standard streams to the
        // default tty devices. however, if the standard streams
        // have been overwritten we create a unique device for
        // them instead.
        if (Module['stdin']) {
          FS.createDevice('/dev', 'stdin', Module['stdin']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdin');
        }
        if (Module['stdout']) {
          FS.createDevice('/dev', 'stdout', null, Module['stdout']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdout');
        }
        if (Module['stderr']) {
          FS.createDevice('/dev', 'stderr', null, Module['stderr']);
        } else {
          FS.symlink('/dev/tty1', '/dev/stderr');
        }
  
        // open default streams for the stdin, stdout and stderr devices
        var stdin = FS.open('/dev/stdin', 'r');
        assert(stdin.fd === 0, 'invalid handle for stdin (' + stdin.fd + ')');
  
        var stdout = FS.open('/dev/stdout', 'w');
        assert(stdout.fd === 1, 'invalid handle for stdout (' + stdout.fd + ')');
  
        var stderr = FS.open('/dev/stderr', 'w');
        assert(stderr.fd === 2, 'invalid handle for stderr (' + stderr.fd + ')');
      },ensureErrnoError:function () {
        if (FS.ErrnoError) return;
        FS.ErrnoError = function ErrnoError(errno, node) {
          //err(stackTrace()); // useful for debugging
          this.node = node;
          this.setErrno = function(errno) {
            this.errno = errno;
            for (var key in ERRNO_CODES) {
              if (ERRNO_CODES[key] === errno) {
                this.code = key;
                break;
              }
            }
          };
          this.setErrno(errno);
          this.message = ERRNO_MESSAGES[errno];
          // Node.js compatibility: assigning on this.stack fails on Node 4 (but fixed on Node 8)
          if (this.stack) Object.defineProperty(this, "stack", { value: (new Error).stack, writable: true });
        };
        FS.ErrnoError.prototype = new Error();
        FS.ErrnoError.prototype.constructor = FS.ErrnoError;
        // Some errors may happen quite a bit, to avoid overhead we reuse them (and suffer a lack of stack info)
        [ERRNO_CODES.ENOENT].forEach(function(code) {
          FS.genericErrors[code] = new FS.ErrnoError(code);
          FS.genericErrors[code].stack = '<generic error, no stack>';
        });
      },staticInit:function () {
        FS.ensureErrnoError();
  
        FS.nameTable = new Array(4096);
  
        FS.mount(MEMFS, {}, '/');
  
        FS.createDefaultDirectories();
        FS.createDefaultDevices();
        FS.createSpecialDirectories();
  
        FS.filesystems = {
          'MEMFS': MEMFS,
          'IDBFS': IDBFS,
          'NODEFS': NODEFS,
          'WORKERFS': WORKERFS,
        };
      },init:function (input, output, error) {
        assert(!FS.init.initialized, 'FS.init was previously called. If you want to initialize later with custom parameters, remove any earlier calls (note that one is automatically added to the generated code)');
        FS.init.initialized = true;
  
        FS.ensureErrnoError();
  
        // Allow Module.stdin etc. to provide defaults, if none explicitly passed to us here
        Module['stdin'] = input || Module['stdin'];
        Module['stdout'] = output || Module['stdout'];
        Module['stderr'] = error || Module['stderr'];
  
        FS.createStandardStreams();
      },quit:function () {
        FS.init.initialized = false;
        // force-flush all streams, so we get musl std streams printed out
        var fflush = Module['_fflush'];
        if (fflush) fflush(0);
        // close all of our streams
        for (var i = 0; i < FS.streams.length; i++) {
          var stream = FS.streams[i];
          if (!stream) {
            continue;
          }
          FS.close(stream);
        }
      },getMode:function (canRead, canWrite) {
        var mode = 0;
        if (canRead) mode |= 292 | 73;
        if (canWrite) mode |= 146;
        return mode;
      },joinPath:function (parts, forceRelative) {
        var path = PATH.join.apply(null, parts);
        if (forceRelative && path[0] == '/') path = path.substr(1);
        return path;
      },absolutePath:function (relative, base) {
        return PATH.resolve(base, relative);
      },standardizePath:function (path) {
        return PATH.normalize(path);
      },findObject:function (path, dontResolveLastLink) {
        var ret = FS.analyzePath(path, dontResolveLastLink);
        if (ret.exists) {
          return ret.object;
        } else {
          ___setErrNo(ret.error);
          return null;
        }
      },analyzePath:function (path, dontResolveLastLink) {
        // operate from within the context of the symlink's target
        try {
          var lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          path = lookup.path;
        } catch (e) {
        }
        var ret = {
          isRoot: false, exists: false, error: 0, name: null, path: null, object: null,
          parentExists: false, parentPath: null, parentObject: null
        };
        try {
          var lookup = FS.lookupPath(path, { parent: true });
          ret.parentExists = true;
          ret.parentPath = lookup.path;
          ret.parentObject = lookup.node;
          ret.name = PATH.basename(path);
          lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          ret.exists = true;
          ret.path = lookup.path;
          ret.object = lookup.node;
          ret.name = lookup.node.name;
          ret.isRoot = lookup.path === '/';
        } catch (e) {
          ret.error = e.errno;
        };
        return ret;
      },createFolder:function (parent, name, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.mkdir(path, mode);
      },createPath:function (parent, path, canRead, canWrite) {
        parent = typeof parent === 'string' ? parent : FS.getPath(parent);
        var parts = path.split('/').reverse();
        while (parts.length) {
          var part = parts.pop();
          if (!part) continue;
          var current = PATH.join2(parent, part);
          try {
            FS.mkdir(current);
          } catch (e) {
            // ignore EEXIST
          }
          parent = current;
        }
        return current;
      },createFile:function (parent, name, properties, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.create(path, mode);
      },createDataFile:function (parent, name, data, canRead, canWrite, canOwn) {
        var path = name ? PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name) : parent;
        var mode = FS.getMode(canRead, canWrite);
        var node = FS.create(path, mode);
        if (data) {
          if (typeof data === 'string') {
            var arr = new Array(data.length);
            for (var i = 0, len = data.length; i < len; ++i) arr[i] = data.charCodeAt(i);
            data = arr;
          }
          // make sure we can write to the file
          FS.chmod(node, mode | 146);
          var stream = FS.open(node, 'w');
          FS.write(stream, data, 0, data.length, 0, canOwn);
          FS.close(stream);
          FS.chmod(node, mode);
        }
        return node;
      },createDevice:function (parent, name, input, output) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(!!input, !!output);
        if (!FS.createDevice.major) FS.createDevice.major = 64;
        var dev = FS.makedev(FS.createDevice.major++, 0);
        // Create a fake device that a set of stream ops to emulate
        // the old behavior.
        FS.registerDevice(dev, {
          open: function(stream) {
            stream.seekable = false;
          },
          close: function(stream) {
            // flush any pending line data
            if (output && output.buffer && output.buffer.length) {
              output(10);
            }
          },
          read: function(stream, buffer, offset, length, pos /* ignored */) {
            var bytesRead = 0;
            for (var i = 0; i < length; i++) {
              var result;
              try {
                result = input();
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES.EIO);
              }
              if (result === undefined && bytesRead === 0) {
                throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
              }
              if (result === null || result === undefined) break;
              bytesRead++;
              buffer[offset+i] = result;
            }
            if (bytesRead) {
              stream.node.timestamp = Date.now();
            }
            return bytesRead;
          },
          write: function(stream, buffer, offset, length, pos) {
            for (var i = 0; i < length; i++) {
              try {
                output(buffer[offset+i]);
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES.EIO);
              }
            }
            if (length) {
              stream.node.timestamp = Date.now();
            }
            return i;
          }
        });
        return FS.mkdev(path, mode, dev);
      },createLink:function (parent, name, target, canRead, canWrite) {
        var path = PATH.join2(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        return FS.symlink(target, path);
      },forceLoadFile:function (obj) {
        if (obj.isDevice || obj.isFolder || obj.link || obj.contents) return true;
        var success = true;
        if (typeof XMLHttpRequest !== 'undefined') {
          throw new Error("Lazy loading should have been performed (contents set) in createLazyFile, but it was not. Lazy loading only works in web workers. Use --embed-file or --preload-file in emcc on the main thread.");
        } else if (Module['read']) {
          // Command-line.
          try {
            // WARNING: Can't read binary files in V8's d8 or tracemonkey's js, as
            //          read() will try to parse UTF8.
            obj.contents = intArrayFromString(Module['read'](obj.url), true);
            obj.usedBytes = obj.contents.length;
          } catch (e) {
            success = false;
          }
        } else {
          throw new Error('Cannot load without read() or XMLHttpRequest.');
        }
        if (!success) ___setErrNo(ERRNO_CODES.EIO);
        return success;
      },createLazyFile:function (parent, name, url, canRead, canWrite) {
        // Lazy chunked Uint8Array (implements get and length from Uint8Array). Actual getting is abstracted away for eventual reuse.
        function LazyUint8Array() {
          this.lengthKnown = false;
          this.chunks = []; // Loaded chunks. Index is the chunk number
        }
        LazyUint8Array.prototype.get = function LazyUint8Array_get(idx) {
          if (idx > this.length-1 || idx < 0) {
            return undefined;
          }
          var chunkOffset = idx % this.chunkSize;
          var chunkNum = (idx / this.chunkSize)|0;
          return this.getter(chunkNum)[chunkOffset];
        }
        LazyUint8Array.prototype.setDataGetter = function LazyUint8Array_setDataGetter(getter) {
          this.getter = getter;
        }
        LazyUint8Array.prototype.cacheLength = function LazyUint8Array_cacheLength() {
          // Find length
          var xhr = new XMLHttpRequest();
          xhr.open('HEAD', url, false);
          xhr.send(null);
          if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
          var datalength = Number(xhr.getResponseHeader("Content-length"));
          var header;
          var hasByteServing = (header = xhr.getResponseHeader("Accept-Ranges")) && header === "bytes";
          var usesGzip = (header = xhr.getResponseHeader("Content-Encoding")) && header === "gzip";
  
          var chunkSize = 1024*1024; // Chunk size in bytes
  
          if (!hasByteServing) chunkSize = datalength;
  
          // Function to get a range from the remote URL.
          var doXHR = (function(from, to) {
            if (from > to) throw new Error("invalid range (" + from + ", " + to + ") or no bytes requested!");
            if (to > datalength-1) throw new Error("only " + datalength + " bytes available! programmer error!");
  
            // TODO: Use mozResponseArrayBuffer, responseStream, etc. if available.
            var xhr = new XMLHttpRequest();
            xhr.open('GET', url, false);
            if (datalength !== chunkSize) xhr.setRequestHeader("Range", "bytes=" + from + "-" + to);
  
            // Some hints to the browser that we want binary data.
            if (typeof Uint8Array != 'undefined') xhr.responseType = 'arraybuffer';
            if (xhr.overrideMimeType) {
              xhr.overrideMimeType('text/plain; charset=x-user-defined');
            }
  
            xhr.send(null);
            if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
            if (xhr.response !== undefined) {
              return new Uint8Array(xhr.response || []);
            } else {
              return intArrayFromString(xhr.responseText || '', true);
            }
          });
          var lazyArray = this;
          lazyArray.setDataGetter(function(chunkNum) {
            var start = chunkNum * chunkSize;
            var end = (chunkNum+1) * chunkSize - 1; // including this byte
            end = Math.min(end, datalength-1); // if datalength-1 is selected, this is the last block
            if (typeof(lazyArray.chunks[chunkNum]) === "undefined") {
              lazyArray.chunks[chunkNum] = doXHR(start, end);
            }
            if (typeof(lazyArray.chunks[chunkNum]) === "undefined") throw new Error("doXHR failed!");
            return lazyArray.chunks[chunkNum];
          });
  
          if (usesGzip || !datalength) {
            // if the server uses gzip or doesn't supply the length, we have to download the whole file to get the (uncompressed) length
            chunkSize = datalength = 1; // this will force getter(0)/doXHR do download the whole file
            datalength = this.getter(0).length;
            chunkSize = datalength;
            console.log("LazyFiles on gzip forces download of the whole file when length is accessed");
          }
  
          this._length = datalength;
          this._chunkSize = chunkSize;
          this.lengthKnown = true;
        }
        if (typeof XMLHttpRequest !== 'undefined') {
          if (!ENVIRONMENT_IS_WORKER) throw 'Cannot do synchronous binary XHRs outside webworkers in modern browsers. Use --embed-file or --preload-file in emcc';
          var lazyArray = new LazyUint8Array();
          Object.defineProperties(lazyArray, {
            length: {
              get: function() {
                if(!this.lengthKnown) {
                  this.cacheLength();
                }
                return this._length;
              }
            },
            chunkSize: {
              get: function() {
                if(!this.lengthKnown) {
                  this.cacheLength();
                }
                return this._chunkSize;
              }
            }
          });
  
          var properties = { isDevice: false, contents: lazyArray };
        } else {
          var properties = { isDevice: false, url: url };
        }
  
        var node = FS.createFile(parent, name, properties, canRead, canWrite);
        // This is a total hack, but I want to get this lazy file code out of the
        // core of MEMFS. If we want to keep this lazy file concept I feel it should
        // be its own thin LAZYFS proxying calls to MEMFS.
        if (properties.contents) {
          node.contents = properties.contents;
        } else if (properties.url) {
          node.contents = null;
          node.url = properties.url;
        }
        // Add a function that defers querying the file size until it is asked the first time.
        Object.defineProperties(node, {
          usedBytes: {
            get: function() { return this.contents.length; }
          }
        });
        // override each stream op with one that tries to force load the lazy file first
        var stream_ops = {};
        var keys = Object.keys(node.stream_ops);
        keys.forEach(function(key) {
          var fn = node.stream_ops[key];
          stream_ops[key] = function forceLoadLazyFile() {
            if (!FS.forceLoadFile(node)) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            return fn.apply(null, arguments);
          };
        });
        // use a custom read function
        stream_ops.read = function stream_ops_read(stream, buffer, offset, length, position) {
          if (!FS.forceLoadFile(node)) {
            throw new FS.ErrnoError(ERRNO_CODES.EIO);
          }
          var contents = stream.node.contents;
          if (position >= contents.length)
            return 0;
          var size = Math.min(contents.length - position, length);
          assert(size >= 0);
          if (contents.slice) { // normal array
            for (var i = 0; i < size; i++) {
              buffer[offset + i] = contents[position + i];
            }
          } else {
            for (var i = 0; i < size; i++) { // LazyUint8Array from sync binary XHR
              buffer[offset + i] = contents.get(position + i);
            }
          }
          return size;
        };
        node.stream_ops = stream_ops;
        return node;
      },createPreloadedFile:function (parent, name, url, canRead, canWrite, onload, onerror, dontCreateFile, canOwn, preFinish) {
        Browser.init(); // XXX perhaps this method should move onto Browser?
        // TODO we should allow people to just pass in a complete filename instead
        // of parent and name being that we just join them anyways
        var fullname = name ? PATH.resolve(PATH.join2(parent, name)) : parent;
        var dep = getUniqueRunDependency('cp ' + fullname); // might have several active requests for the same fullname
        function processData(byteArray) {
          function finish(byteArray) {
            if (preFinish) preFinish();
            if (!dontCreateFile) {
              FS.createDataFile(parent, name, byteArray, canRead, canWrite, canOwn);
            }
            if (onload) onload();
            removeRunDependency(dep);
          }
          var handled = false;
          Module['preloadPlugins'].forEach(function(plugin) {
            if (handled) return;
            if (plugin['canHandle'](fullname)) {
              plugin['handle'](byteArray, fullname, finish, function() {
                if (onerror) onerror();
                removeRunDependency(dep);
              });
              handled = true;
            }
          });
          if (!handled) finish(byteArray);
        }
        addRunDependency(dep);
        if (typeof url == 'string') {
          Browser.asyncLoad(url, function(byteArray) {
            processData(byteArray);
          }, onerror);
        } else {
          processData(url);
        }
      },indexedDB:function () {
        return window.indexedDB || window.mozIndexedDB || window.webkitIndexedDB || window.msIndexedDB;
      },DB_NAME:function () {
        return 'EM_FS_' + window.location.pathname;
      },DB_VERSION:20,DB_STORE_NAME:"FILE_DATA",saveFilesToDB:function (paths, onload, onerror) {
        onload = onload || function(){};
        onerror = onerror || function(){};
        var indexedDB = FS.indexedDB();
        try {
          var openRequest = indexedDB.open(FS.DB_NAME(), FS.DB_VERSION);
        } catch (e) {
          return onerror(e);
        }
        openRequest.onupgradeneeded = function openRequest_onupgradeneeded() {
          console.log('creating db');
          var db = openRequest.result;
          db.createObjectStore(FS.DB_STORE_NAME);
        };
        openRequest.onsuccess = function openRequest_onsuccess() {
          var db = openRequest.result;
          var transaction = db.transaction([FS.DB_STORE_NAME], 'readwrite');
          var files = transaction.objectStore(FS.DB_STORE_NAME);
          var ok = 0, fail = 0, total = paths.length;
          function finish() {
            if (fail == 0) onload(); else onerror();
          }
          paths.forEach(function(path) {
            var putRequest = files.put(FS.analyzePath(path).object.contents, path);
            putRequest.onsuccess = function putRequest_onsuccess() { ok++; if (ok + fail == total) finish() };
            putRequest.onerror = function putRequest_onerror() { fail++; if (ok + fail == total) finish() };
          });
          transaction.onerror = onerror;
        };
        openRequest.onerror = onerror;
      },loadFilesFromDB:function (paths, onload, onerror) {
        onload = onload || function(){};
        onerror = onerror || function(){};
        var indexedDB = FS.indexedDB();
        try {
          var openRequest = indexedDB.open(FS.DB_NAME(), FS.DB_VERSION);
        } catch (e) {
          return onerror(e);
        }
        openRequest.onupgradeneeded = onerror; // no database to load from
        openRequest.onsuccess = function openRequest_onsuccess() {
          var db = openRequest.result;
          try {
            var transaction = db.transaction([FS.DB_STORE_NAME], 'readonly');
          } catch(e) {
            onerror(e);
            return;
          }
          var files = transaction.objectStore(FS.DB_STORE_NAME);
          var ok = 0, fail = 0, total = paths.length;
          function finish() {
            if (fail == 0) onload(); else onerror();
          }
          paths.forEach(function(path) {
            var getRequest = files.get(path);
            getRequest.onsuccess = function getRequest_onsuccess() {
              if (FS.analyzePath(path).exists) {
                FS.unlink(path);
              }
              FS.createDataFile(PATH.dirname(path), PATH.basename(path), getRequest.result, true, true, true);
              ok++;
              if (ok + fail == total) finish();
            };
            getRequest.onerror = function getRequest_onerror() { fail++; if (ok + fail == total) finish() };
          });
          transaction.onerror = onerror;
        };
        openRequest.onerror = onerror;
      }};var SYSCALLS={DEFAULT_POLLMASK:5,mappings:{},umask:511,calculateAt:function (dirfd, path) {
        if (path[0] !== '/') {
          // relative path
          var dir;
          if (dirfd === -100) {
            dir = FS.cwd();
          } else {
            var dirstream = FS.getStream(dirfd);
            if (!dirstream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
            dir = dirstream.path;
          }
          path = PATH.join2(dir, path);
        }
        return path;
      },doStat:function (func, path, buf) {
        try {
          var stat = func(path);
        } catch (e) {
          if (e && e.node && PATH.normalize(path) !== PATH.normalize(FS.getPath(e.node))) {
            // an error occurred while trying to look up the path; we should just report ENOTDIR
            return -ERRNO_CODES.ENOTDIR;
          }
          throw e;
        }
        HEAP32[((buf)>>2)]=stat.dev;
        HEAP32[(((buf)+(4))>>2)]=0;
        HEAP32[(((buf)+(8))>>2)]=stat.ino;
        HEAP32[(((buf)+(12))>>2)]=stat.mode;
        HEAP32[(((buf)+(16))>>2)]=stat.nlink;
        HEAP32[(((buf)+(20))>>2)]=stat.uid;
        HEAP32[(((buf)+(24))>>2)]=stat.gid;
        HEAP32[(((buf)+(28))>>2)]=stat.rdev;
        HEAP32[(((buf)+(32))>>2)]=0;
        HEAP32[(((buf)+(36))>>2)]=stat.size;
        HEAP32[(((buf)+(40))>>2)]=4096;
        HEAP32[(((buf)+(44))>>2)]=stat.blocks;
        HEAP32[(((buf)+(48))>>2)]=(stat.atime.getTime() / 1000)|0;
        HEAP32[(((buf)+(52))>>2)]=0;
        HEAP32[(((buf)+(56))>>2)]=(stat.mtime.getTime() / 1000)|0;
        HEAP32[(((buf)+(60))>>2)]=0;
        HEAP32[(((buf)+(64))>>2)]=(stat.ctime.getTime() / 1000)|0;
        HEAP32[(((buf)+(68))>>2)]=0;
        HEAP32[(((buf)+(72))>>2)]=stat.ino;
        return 0;
      },doMsync:function (addr, stream, len, flags) {
        var buffer = new Uint8Array(HEAPU8.subarray(addr, addr + len));
        FS.msync(stream, buffer, 0, len, flags);
      },doMkdir:function (path, mode) {
        // remove a trailing slash, if one - /a/b/ has basename of '', but
        // we want to create b in the context of this function
        path = PATH.normalize(path);
        if (path[path.length-1] === '/') path = path.substr(0, path.length-1);
        FS.mkdir(path, mode, 0);
        return 0;
      },doMknod:function (path, mode, dev) {
        // we don't want this in the JS API as it uses mknod to create all nodes.
        switch (mode & 61440) {
          case 32768:
          case 8192:
          case 24576:
          case 4096:
          case 49152:
            break;
          default: return -ERRNO_CODES.EINVAL;
        }
        FS.mknod(path, mode, dev);
        return 0;
      },doReadlink:function (path, buf, bufsize) {
        if (bufsize <= 0) return -ERRNO_CODES.EINVAL;
        var ret = FS.readlink(path);
  
        var len = Math.min(bufsize, lengthBytesUTF8(ret));
        var endChar = HEAP8[buf+len];
        stringToUTF8(ret, buf, bufsize+1);
        // readlink is one of the rare functions that write out a C string, but does never append a null to the output buffer(!)
        // stringToUTF8() always appends a null byte, so restore the character under the null byte after the write.
        HEAP8[buf+len] = endChar;
  
        return len;
      },doAccess:function (path, amode) {
        if (amode & ~7) {
          // need a valid mode
          return -ERRNO_CODES.EINVAL;
        }
        var node;
        var lookup = FS.lookupPath(path, { follow: true });
        node = lookup.node;
        var perms = '';
        if (amode & 4) perms += 'r';
        if (amode & 2) perms += 'w';
        if (amode & 1) perms += 'x';
        if (perms /* otherwise, they've just passed F_OK */ && FS.nodePermissions(node, perms)) {
          return -ERRNO_CODES.EACCES;
        }
        return 0;
      },doDup:function (path, flags, suggestFD) {
        var suggest = FS.getStream(suggestFD);
        if (suggest) FS.close(suggest);
        return FS.open(path, flags, 0, suggestFD, suggestFD).fd;
      },doReadv:function (stream, iov, iovcnt, offset) {
        var ret = 0;
        for (var i = 0; i < iovcnt; i++) {
          var ptr = HEAP32[(((iov)+(i*8))>>2)];
          var len = HEAP32[(((iov)+(i*8 + 4))>>2)];
          var curr = FS.read(stream, HEAP8,ptr, len, offset);
          if (curr < 0) return -1;
          ret += curr;
          if (curr < len) break; // nothing more to read
        }
        return ret;
      },doWritev:function (stream, iov, iovcnt, offset) {
        var ret = 0;
        for (var i = 0; i < iovcnt; i++) {
          var ptr = HEAP32[(((iov)+(i*8))>>2)];
          var len = HEAP32[(((iov)+(i*8 + 4))>>2)];
          var curr = FS.write(stream, HEAP8,ptr, len, offset);
          if (curr < 0) return -1;
          ret += curr;
        }
        return ret;
      },varargs:0,get:function (varargs) {
        SYSCALLS.varargs += 4;
        var ret = HEAP32[(((SYSCALLS.varargs)-(4))>>2)];
        return ret;
      },getStr:function () {
        var ret = Pointer_stringify(SYSCALLS.get());
        return ret;
      },getStreamFromFD:function () {
        var stream = FS.getStream(SYSCALLS.get());
        if (!stream) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        return stream;
      },getSocketFromFD:function () {
        var socket = SOCKFS.getSocket(SYSCALLS.get());
        if (!socket) throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        return socket;
      },getSocketAddress:function (allowNull) {
        var addrp = SYSCALLS.get(), addrlen = SYSCALLS.get();
        if (allowNull && addrp === 0) return null;
        var info = __read_sockaddr(addrp, addrlen);
        if (info.errno) throw new FS.ErrnoError(info.errno);
        info.addr = DNS.lookup_addr(info.addr) || info.addr;
        return info;
      },get64:function () {
        var low = SYSCALLS.get(), high = SYSCALLS.get();
        if (low >= 0) assert(high === 0);
        else assert(high === -1);
        return low;
      },getZero:function () {
        assert(SYSCALLS.get() === 0);
      }};function ___syscall10(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // unlink
      var path = SYSCALLS.getStr();
      FS.unlink(path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall118(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fsync
      var stream = SYSCALLS.getStreamFromFD();
      return 0; // we can't do anything synchronously; the in-memory FS is already synced to
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall140(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // llseek
      var stream = SYSCALLS.getStreamFromFD(), offset_high = SYSCALLS.get(), offset_low = SYSCALLS.get(), result = SYSCALLS.get(), whence = SYSCALLS.get();
      // NOTE: offset_high is unused - Emscripten's off_t is 32-bit
      var offset = offset_low;
      FS.llseek(stream, offset, whence);
      HEAP32[((result)>>2)]=stream.position;
      if (stream.getdents && offset === 0 && whence === 0) stream.getdents = null; // reset readdir state
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall145(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // readv
      var stream = SYSCALLS.getStreamFromFD(), iov = SYSCALLS.get(), iovcnt = SYSCALLS.get();
      return SYSCALLS.doReadv(stream, iov, iovcnt);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall146(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // writev
      var stream = SYSCALLS.getStreamFromFD(), iov = SYSCALLS.get(), iovcnt = SYSCALLS.get();
      return SYSCALLS.doWritev(stream, iov, iovcnt);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall15(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // chmod
      var path = SYSCALLS.getStr(), mode = SYSCALLS.get();
      FS.chmod(path, mode);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall183(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // getcwd
      var buf = SYSCALLS.get(), size = SYSCALLS.get();
      if (size === 0) return -ERRNO_CODES.EINVAL;
      var cwd = FS.cwd();
      var cwdLengthInBytes = lengthBytesUTF8(cwd);
      if (size < cwdLengthInBytes + 1) return -ERRNO_CODES.ERANGE;
      stringToUTF8(cwd, buf, size);
      return buf;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall192(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // mmap2
      var addr = SYSCALLS.get(), len = SYSCALLS.get(), prot = SYSCALLS.get(), flags = SYSCALLS.get(), fd = SYSCALLS.get(), off = SYSCALLS.get()
      off <<= 12; // undo pgoffset
      var ptr;
      var allocated = false;
      if (fd === -1) {
        ptr = _memalign(PAGE_SIZE, len);
        if (!ptr) return -ERRNO_CODES.ENOMEM;
        _memset(ptr, 0, len);
        allocated = true;
      } else {
        var info = FS.getStream(fd);
        if (!info) return -ERRNO_CODES.EBADF;
        var res = FS.mmap(info, HEAPU8, addr, len, off, prot, flags);
        ptr = res.ptr;
        allocated = res.allocated;
      }
      SYSCALLS.mappings[ptr] = { malloc: ptr, len: len, allocated: allocated, fd: fd, flags: flags };
      return ptr;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall194(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // ftruncate64
      var fd = SYSCALLS.get(), zero = SYSCALLS.getZero(), length = SYSCALLS.get64();
      FS.ftruncate(fd, length);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall195(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_stat64
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.stat, path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall196(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_lstat64
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.lstat, path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall197(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // SYS_fstat64
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get();
      return SYSCALLS.doStat(FS.stat, stream.path, buf);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  
  var PROCINFO={ppid:1,pid:42,sid:42,pgid:42};function ___syscall20(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // getpid
      return PROCINFO.pid;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  
  function ___syscall202(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // getgid32
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }function ___syscall201() {
  return ___syscall202.apply(null, arguments)
  }

  function ___syscall207(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fchown32
      var fd = SYSCALLS.get(), owner = SYSCALLS.get(), group = SYSCALLS.get();
      FS.fchown(fd, owner, group);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall212(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // chown32
      var path = SYSCALLS.getStr(), owner = SYSCALLS.get(), group = SYSCALLS.get();
      FS.chown(path, owner, group);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall221(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fcntl64
      var stream = SYSCALLS.getStreamFromFD(), cmd = SYSCALLS.get();
      switch (cmd) {
        case 0: {
          var arg = SYSCALLS.get();
          if (arg < 0) {
            return -ERRNO_CODES.EINVAL;
          }
          var newStream;
          newStream = FS.open(stream.path, stream.flags, 0, arg);
          return newStream.fd;
        }
        case 1:
        case 2:
          return 0;  // FD_CLOEXEC makes no sense for a single process.
        case 3:
          return stream.flags;
        case 4: {
          var arg = SYSCALLS.get();
          stream.flags |= arg;
          return 0;
        }
        case 12:
        case 12: {
          var arg = SYSCALLS.get();
          var offset = 0;
          // We're always unlocked.
          HEAP16[(((arg)+(offset))>>1)]=2;
          return 0;
        }
        case 13:
        case 14:
        case 13:
        case 14:
          return 0; // Pretend that the locking is successful.
        case 16:
        case 8:
          return -ERRNO_CODES.EINVAL; // These are for sockets. We don't have them fully implemented yet.
        case 9:
          // musl trusts getown return values, due to a bug where they must be, as they overlap with errors. just return -1 here, so fnctl() returns that, and we set errno ourselves.
          ___setErrNo(ERRNO_CODES.EINVAL);
          return -1;
        default: {
          return -ERRNO_CODES.EINVAL;
        }
      }
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall3(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // read
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get(), count = SYSCALLS.get();
      return FS.read(stream, HEAP8,buf, count);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall33(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // access
      var path = SYSCALLS.getStr(), amode = SYSCALLS.get();
      return SYSCALLS.doAccess(path, amode);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall39(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // mkdir
      var path = SYSCALLS.getStr(), mode = SYSCALLS.get();
      return SYSCALLS.doMkdir(path, mode);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall4(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // write
      var stream = SYSCALLS.getStreamFromFD(), buf = SYSCALLS.get(), count = SYSCALLS.get();
      return FS.write(stream, HEAP8,buf, count);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall40(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // rmdir
      var path = SYSCALLS.getStr();
      FS.rmdir(path);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall5(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // open
      var pathname = SYSCALLS.getStr(), flags = SYSCALLS.get(), mode = SYSCALLS.get() // optional TODO
      var stream = FS.open(pathname, flags, mode);
      return stream.fd;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall54(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // ioctl
      var stream = SYSCALLS.getStreamFromFD(), op = SYSCALLS.get();
      switch (op) {
        case 21509:
        case 21505: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0;
        }
        case 21510:
        case 21511:
        case 21512:
        case 21506:
        case 21507:
        case 21508: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0; // no-op, not actually adjusting terminal settings
        }
        case 21519: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          var argp = SYSCALLS.get();
          HEAP32[((argp)>>2)]=0;
          return 0;
        }
        case 21520: {
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return -ERRNO_CODES.EINVAL; // not supported
        }
        case 21531: {
          var argp = SYSCALLS.get();
          return FS.ioctl(stream, op, argp);
        }
        case 21523: {
          // TODO: in theory we should write to the winsize struct that gets
          // passed in, but for now musl doesn't read anything on it
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0;
        }
        case 21524: {
          // TODO: technically, this ioctl call should change the window size.
          // but, since emscripten doesn't have any concept of a terminal window
          // yet, we'll just silently throw it away as we do TIOCGWINSZ
          if (!stream.tty) return -ERRNO_CODES.ENOTTY;
          return 0;
        }
        default: abort('bad ioctl syscall ' + op);
      }
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall6(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // close
      var stream = SYSCALLS.getStreamFromFD();
      FS.close(stream);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall85(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // readlink
      var path = SYSCALLS.getStr(), buf = SYSCALLS.get(), bufsize = SYSCALLS.get();
      return SYSCALLS.doReadlink(path, buf, bufsize);
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall91(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // munmap
      var addr = SYSCALLS.get(), len = SYSCALLS.get();
      // TODO: support unmmap'ing parts of allocations
      var info = SYSCALLS.mappings[addr];
      if (!info) return 0;
      if (len === info.len) {
        var stream = FS.getStream(info.fd);
        SYSCALLS.doMsync(addr, stream, len, info.flags)
        FS.munmap(stream);
        SYSCALLS.mappings[addr] = null;
        if (info.allocated) {
          _free(info.malloc);
        }
      }
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

  function ___syscall94(which, varargs) {SYSCALLS.varargs = varargs;
  try {
   // fchmod
      var fd = SYSCALLS.get(), mode = SYSCALLS.get();
      FS.fchmod(fd, mode);
      return 0;
    } catch (e) {
    if (typeof FS === 'undefined' || !(e instanceof FS.ErrnoError)) abort(e);
    return -e.errno;
  }
  }

   

  function ___unlock() {}

   

  function _abort() {
      Module['abort']();
    }

   

   

   

  function _clock() {
      if (_clock.start === undefined) _clock.start = Date.now();
      return ((Date.now() - _clock.start) * (1000000 / 1000))|0;
    }


  function _decrypt(bufIn, sizeIn, bufOut, sizeOut) {
  		var handler = null;
  		if (handler && Module._mataiDerivedKey) {
  			var iv = Module.HEAPU8.subarray(bufIn, bufIn + 16);
  			var input = Module.HEAPU8.subarray(bufIn + 16, bufIn + sizeIn);
  			var decrypted = handler.decrypt(Module._mataiDerivedKey, iv, input);
  			if (decrypted && decrypted.length !== 0 && decrypted.length <= sizeOut) {
  				Module.HEAPU8.set(decrypted, bufOut);
  				return decrypted.length;
  			}
  		}
  		return 0;
  	}

  function _derive_key(salt, password) {
  		var handler = null;
  		if (handler) {
  			// salt is 16 bytes, see SQLCRYPTOSALTLENGTH in ZIPVFSCompression.h.
  			var saltArray = Module.HEAPU8.subarray(salt, salt + 16);
  			var passwordArray = Module.HEAPU8.subarray(password, password + Module._strlen(password));
  			Module._mataiDerivedKey = handler.deriveKey(saltArray, passwordArray);
  			return !!Module._mataiDerivedKey;
  		}
  		return false;
  	}

  
  var DLFCN={error:null,errorMsg:null,loadedLibs:{},loadedLibNames:{}};function _dlclose(handle) {
      // int dlclose(void *handle);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/dlclose.html
      if (!DLFCN.loadedLibs[handle]) {
        DLFCN.errorMsg = 'Tried to dlclose() unopened handle: ' + handle;
        return 1;
      } else {
        var lib_record = DLFCN.loadedLibs[handle];
        if (--lib_record.refcount == 0) {
          if (lib_record.module.cleanups) {
            lib_record.module.cleanups.forEach(function(cleanup) { cleanup() });
          }
          delete DLFCN.loadedLibNames[lib_record.name];
          delete DLFCN.loadedLibs[handle];
        }
        return 0;
      }
    }

  function _dlerror() {
      // char *dlerror(void);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/dlerror.html
      if (DLFCN.errorMsg === null) {
        return 0;
      } else {
        if (DLFCN.error) _free(DLFCN.error);
        var msgArr = intArrayFromString(DLFCN.errorMsg);
        DLFCN.error = allocate(msgArr, 'i8', ALLOC_NORMAL);
        DLFCN.errorMsg = null;
        return DLFCN.error;
      }
    }

  function _dlopen(filenameAddr, flag) {
      abort("To use dlopen, you need to use Emscripten's linking support, see https://github.com/kripken/emscripten/wiki/Linking");
      // void *dlopen(const char *file, int mode);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/dlopen.html
      var searchpaths = [];
      var filename;
      if (filenameAddr === 0) {
        filename = '__self__';
      } else {
        filename = Pointer_stringify(filenameAddr);
  
        var isValidFile = function (filename) {
          var target = FS.findObject(filename);
          return target && !target.isFolder && !target.isDevice;
        };
  
        if (!isValidFile(filename)) {
          if (ENV['LD_LIBRARY_PATH']) {
            searchpaths = ENV['LD_LIBRARY_PATH'].split(':');
          }
  
          for (var ident in searchpaths) {
            var searchfile = PATH.join2(searchpaths[ident], filename);
            if (isValidFile(searchfile)) {
              filename = searchfile;
              break;
            }
          }
        }
      }
  
      if (DLFCN.loadedLibNames[filename]) {
        // Already loaded; increment ref count and return.
        var handle = DLFCN.loadedLibNames[filename];
        DLFCN.loadedLibs[handle].refcount++;
        return handle;
      }
  
      var lib_module;
      if (filename === '__self__') {
        var handle = -1;
        lib_module = Module;
      } else {
        if (Module['preloadedWasm'] !== undefined &&
            Module['preloadedWasm'][filename] !== undefined) {
          lib_module = Module['preloadedWasm'][filename];
        } else {
          var target = FS.findObject(filename);
          if (!target || target.isFolder || target.isDevice) {
            DLFCN.errorMsg = 'Could not find dynamic lib: ' + filename;
            return 0;
          }
          FS.forceLoadFile(target);
  
          try {
            // the shared library is a JS file, which we eval
            var lib_data = FS.readFile(filename, { encoding: 'utf8' });
            lib_module = eval(lib_data)(
              alignFunctionTables(),
              Module
            );
          } catch (e) {
            DLFCN.errorMsg = 'Could not evaluate dynamic lib: ' + filename + '\n' + e;
            return 0;
          }
        }
  
        // Not all browsers support Object.keys().
        var handle = 1;
        for (var key in DLFCN.loadedLibs) {
          if (DLFCN.loadedLibs.hasOwnProperty(key)) handle++;
        }
  
        // We don't care about RTLD_NOW and RTLD_LAZY.
        if (flag & 256) { // RTLD_GLOBAL
          for (var ident in lib_module) {
            if (lib_module.hasOwnProperty(ident)) {
              // When RTLD_GLOBAL is enable, the symbols defined by this shared object will be made
              // available for symbol resolution of subsequently loaded shared objects.
              //
              // We should copy the symbols (which include methods and variables) from SIDE_MODULE to MAIN_MODULE.
              //
              // Module of SIDE_MODULE has not only the symbols (which should be copied)
              // but also others (print*, asmGlobal*, FUNCTION_TABLE_**, NAMED_GLOBALS, and so on).
              //
              // When the symbol (which should be copied) is method, Module._* 's type becomes function.
              // When the symbol (which should be copied) is variable, Module._* 's type becomes number.
              //
              // Except for the symbol prefix (_), there is no difference in the symbols (which should be copied) and others.
              // So this just copies over compiled symbols (which start with _).
              if (ident[0] == '_') {
                Module[ident] = lib_module[ident];
              }
            }
          }
        }
      }
      DLFCN.loadedLibs[handle] = {
        refcount: 1,
        name: filename,
        module: lib_module
      };
      DLFCN.loadedLibNames[filename] = handle;
  
      return handle;
    }

  function _dlsym(handle, symbol) {
      // void *dlsym(void *restrict handle, const char *restrict name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/dlsym.html
      symbol = Pointer_stringify(symbol);
  
      if (!DLFCN.loadedLibs[handle]) {
        DLFCN.errorMsg = 'Tried to dlsym() from an unopened handle: ' + handle;
        return 0;
      } else {
        var lib = DLFCN.loadedLibs[handle];
        symbol = '_' + symbol;
        if (!lib.module.hasOwnProperty(symbol)) {
          DLFCN.errorMsg = ('Tried to lookup unknown symbol "' + symbol +
                                 '" in dynamic lib: ' + lib.name);
          return 0;
        } else {
          var result = lib.module[symbol];
          if (typeof result === 'function') {
            // convert the exported function into a function pointer using our generic
            // JS mechanism.
            return addFunction(result);
          }
          return result;
        }
      }
    }

  var _emscripten_asm_const_int=true;

  function _getenv(name) {
      // char *getenv(const char *name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/getenv.html
      if (name === 0) return 0;
      name = Pointer_stringify(name);
      if (!ENV.hasOwnProperty(name)) return 0;
  
      if (_getenv.ret) _free(_getenv.ret);
      _getenv.ret = allocateUTF8(ENV[name]);
      return _getenv.ret;
    }

  function _gettimeofday(ptr) {
      var now = Date.now();
      HEAP32[((ptr)>>2)]=(now/1000)|0; // seconds
      HEAP32[(((ptr)+(4))>>2)]=((now % 1000)*1000)|0; // microseconds
      return 0;
    }



  function _is_decryption_handler_installed() {
  		return false;
  	}

  function _jsStringify(str) {
  		var resStr = JSON.stringify(Pointer_stringify(str));
  		var len = lengthBytesUTF8(resStr) + 1;
  		var buf = Module._malloc(len);
  		var len2 = stringToUTF8(resStr, buf, len);
  		return buf;
  	}

   

  var _llvm_ceil_f32=Math_ceil;

  var _llvm_cos_f32=Math_cos;

  var _llvm_ctlz_i32=true;

  var _llvm_fabs_f32=Math_abs;

  var _llvm_fabs_f64=Math_abs;

  
  function _llvm_log10_f32(x) {
      return Math.log(x) / Math.LN10; // TODO: Math.log10, when browser support is there
    }function _llvm_log10_f64() {
  return _llvm_log10_f32.apply(null, arguments)
  }

   

  var _llvm_sin_f32=Math_sin;

  function _llvm_stackrestore(p) {
      var self = _llvm_stacksave;
      var ret = self.LLVM_SAVEDSTACKS[p];
      self.LLVM_SAVEDSTACKS.splice(p, 1);
      stackRestore(ret);
    }

  function _llvm_stacksave() {
      var self = _llvm_stacksave;
      if (!self.LLVM_SAVEDSTACKS) {
        self.LLVM_SAVEDSTACKS = [];
      }
      self.LLVM_SAVEDSTACKS.push(stackSave());
      return self.LLVM_SAVEDSTACKS.length-1;
    }

  function _llvm_trap() {
      abort('trap!');
    }

  
  var ___tm_current=STATICTOP; STATICTOP += 48;;
  
  
  var ___tm_timezone=allocate(intArrayFromString("GMT"), "i8", ALLOC_STATIC);
  
  function _tzset() {
      // TODO: Use (malleable) environment variables instead of system settings.
      if (_tzset.called) return;
      _tzset.called = true;
  
      // timezone is specified as seconds west of UTC ("The external variable
      // `timezone` shall be set to the difference, in seconds, between
      // Coordinated Universal Time (UTC) and local standard time."), the same
      // as returned by getTimezoneOffset().
      // See http://pubs.opengroup.org/onlinepubs/009695399/functions/tzset.html
      HEAP32[((__get_timezone())>>2)]=(new Date()).getTimezoneOffset() * 60;
  
      var winter = new Date(2000, 0, 1);
      var summer = new Date(2000, 6, 1);
      HEAP32[((__get_daylight())>>2)]=Number(winter.getTimezoneOffset() != summer.getTimezoneOffset());
  
      function extractZone(date) {
        var match = date.toTimeString().match(/\(([A-Za-z ]+)\)$/);
        return match ? match[1] : "GMT";
      };
      var winterName = extractZone(winter);
      var summerName = extractZone(summer);
      var winterNamePtr = allocate(intArrayFromString(winterName), 'i8', ALLOC_NORMAL);
      var summerNamePtr = allocate(intArrayFromString(summerName), 'i8', ALLOC_NORMAL);
      if (summer.getTimezoneOffset() < winter.getTimezoneOffset()) {
        // Northern hemisphere
        HEAP32[((__get_tzname())>>2)]=winterNamePtr;
        HEAP32[(((__get_tzname())+(4))>>2)]=summerNamePtr;
      } else {
        HEAP32[((__get_tzname())>>2)]=summerNamePtr;
        HEAP32[(((__get_tzname())+(4))>>2)]=winterNamePtr;
      }
    }function _localtime_r(time, tmPtr) {
      _tzset();
      var date = new Date(HEAP32[((time)>>2)]*1000);
      HEAP32[((tmPtr)>>2)]=date.getSeconds();
      HEAP32[(((tmPtr)+(4))>>2)]=date.getMinutes();
      HEAP32[(((tmPtr)+(8))>>2)]=date.getHours();
      HEAP32[(((tmPtr)+(12))>>2)]=date.getDate();
      HEAP32[(((tmPtr)+(16))>>2)]=date.getMonth();
      HEAP32[(((tmPtr)+(20))>>2)]=date.getFullYear()-1900;
      HEAP32[(((tmPtr)+(24))>>2)]=date.getDay();
  
      var start = new Date(date.getFullYear(), 0, 1);
      var yday = ((date.getTime() - start.getTime()) / (1000 * 60 * 60 * 24))|0;
      HEAP32[(((tmPtr)+(28))>>2)]=yday;
      HEAP32[(((tmPtr)+(36))>>2)]=-(date.getTimezoneOffset() * 60);
  
      // Attention: DST is in December in South, and some regions don't have DST at all.
      var summerOffset = new Date(2000, 6, 1).getTimezoneOffset();
      var winterOffset = start.getTimezoneOffset();
      var dst = (summerOffset != winterOffset && date.getTimezoneOffset() == Math.min(winterOffset, summerOffset))|0;
      HEAP32[(((tmPtr)+(32))>>2)]=dst;
  
      var zonePtr = HEAP32[(((__get_tzname())+(dst ? 4 : 0))>>2)];
      HEAP32[(((tmPtr)+(40))>>2)]=zonePtr;
  
      return tmPtr;
    }function _localtime(time) {
      return _localtime_r(time, ___tm_current);
    }


  
  function _emscripten_memcpy_big(dest, src, num) {
      HEAPU8.set(HEAPU8.subarray(src, src+num), dest);
      return dest;
    } 

   

   

  
  function _usleep(useconds) {
      // int usleep(useconds_t useconds);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/usleep.html
      // We're single-threaded, so use a busy loop. Super-ugly.
      var msec = useconds / 1000;
      if ((ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) && self['performance'] && self['performance']['now']) {
        var start = self['performance']['now']();
        while (self['performance']['now']() - start < msec) {
          // Do nothing.
        }
      } else {
        var start = Date.now();
        while (Date.now() - start < msec) {
          // Do nothing.
        }
      }
      return 0;
    }
  Module["_usleep"] = _usleep;function _nanosleep(rqtp, rmtp) {
      // int nanosleep(const struct timespec  *rqtp, struct timespec *rmtp);
      var seconds = HEAP32[((rqtp)>>2)];
      var nanoseconds = HEAP32[(((rqtp)+(4))>>2)];
      if (rmtp !== 0) {
        HEAP32[((rmtp)>>2)]=0;
        HEAP32[(((rmtp)+(4))>>2)]=0;
      }
      return _usleep((seconds * 1e6) + (nanoseconds / 1000));
    }

   

  function _pthread_cond_wait() { return 0; }

  function _pthread_create() {
      return 11;
    }

  
  var PTHREAD_SPECIFIC={};function _pthread_getspecific(key) {
      return PTHREAD_SPECIFIC[key] || 0;
    }

  function _pthread_join() {}

  
  var PTHREAD_SPECIFIC_NEXT_KEY=1;function _pthread_key_create(key, destructor) {
      if (key == 0) {
        return ERRNO_CODES.EINVAL;
      }
      HEAP32[((key)>>2)]=PTHREAD_SPECIFIC_NEXT_KEY;
      // values start at 0
      PTHREAD_SPECIFIC[PTHREAD_SPECIFIC_NEXT_KEY] = 0;
      PTHREAD_SPECIFIC_NEXT_KEY++;
      return 0;
    }

  function _pthread_mutex_destroy() {}

  function _pthread_mutex_init() {}

   

   

   

  function _pthread_mutexattr_destroy() {}

  function _pthread_mutexattr_init() {}

  function _pthread_mutexattr_settype() {}

  function _pthread_once(ptr, func) {
      if (!_pthread_once.seen) _pthread_once.seen = {};
      if (ptr in _pthread_once.seen) return;
      Module['dynCall_v'](func);
      _pthread_once.seen[ptr] = 1;
    }

  function _pthread_setspecific(key, value) {
      if (!(key in PTHREAD_SPECIFIC)) {
        return ERRNO_CODES.EINVAL;
      }
      PTHREAD_SPECIFIC[key] = value;
      return 0;
    }

   

  
  function __isLeapYear(year) {
        return year%4 === 0 && (year%100 !== 0 || year%400 === 0);
    }
  
  function __arraySum(array, index) {
      var sum = 0;
      for (var i = 0; i <= index; sum += array[i++]);
      return sum;
    }
  
  
  var __MONTH_DAYS_LEAP=[31,29,31,30,31,30,31,31,30,31,30,31];
  
  var __MONTH_DAYS_REGULAR=[31,28,31,30,31,30,31,31,30,31,30,31];function __addDays(date, days) {
      var newDate = new Date(date.getTime());
      while(days > 0) {
        var leap = __isLeapYear(newDate.getFullYear());
        var currentMonth = newDate.getMonth();
        var daysInCurrentMonth = (leap ? __MONTH_DAYS_LEAP : __MONTH_DAYS_REGULAR)[currentMonth];
  
        if (days > daysInCurrentMonth-newDate.getDate()) {
          // we spill over to next month
          days -= (daysInCurrentMonth-newDate.getDate()+1);
          newDate.setDate(1);
          if (currentMonth < 11) {
            newDate.setMonth(currentMonth+1)
          } else {
            newDate.setMonth(0);
            newDate.setFullYear(newDate.getFullYear()+1);
          }
        } else {
          // we stay in current month
          newDate.setDate(newDate.getDate()+days);
          return newDate;
        }
      }
  
      return newDate;
    }function _strftime(s, maxsize, format, tm) {
      // size_t strftime(char *restrict s, size_t maxsize, const char *restrict format, const struct tm *restrict timeptr);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/strftime.html
  
      var tm_zone = HEAP32[(((tm)+(40))>>2)];
  
      var date = {
        tm_sec: HEAP32[((tm)>>2)],
        tm_min: HEAP32[(((tm)+(4))>>2)],
        tm_hour: HEAP32[(((tm)+(8))>>2)],
        tm_mday: HEAP32[(((tm)+(12))>>2)],
        tm_mon: HEAP32[(((tm)+(16))>>2)],
        tm_year: HEAP32[(((tm)+(20))>>2)],
        tm_wday: HEAP32[(((tm)+(24))>>2)],
        tm_yday: HEAP32[(((tm)+(28))>>2)],
        tm_isdst: HEAP32[(((tm)+(32))>>2)],
        tm_gmtoff: HEAP32[(((tm)+(36))>>2)],
        tm_zone: tm_zone ? Pointer_stringify(tm_zone) : ''
      };
  
      var pattern = Pointer_stringify(format);
  
      // expand format
      var EXPANSION_RULES_1 = {
        '%c': '%a %b %d %H:%M:%S %Y',     // Replaced by the locale's appropriate date and time representation - e.g., Mon Aug  3 14:02:01 2013
        '%D': '%m/%d/%y',                 // Equivalent to %m / %d / %y
        '%F': '%Y-%m-%d',                 // Equivalent to %Y - %m - %d
        '%h': '%b',                       // Equivalent to %b
        '%r': '%I:%M:%S %p',              // Replaced by the time in a.m. and p.m. notation
        '%R': '%H:%M',                    // Replaced by the time in 24-hour notation
        '%T': '%H:%M:%S',                 // Replaced by the time
        '%x': '%m/%d/%y',                 // Replaced by the locale's appropriate date representation
        '%X': '%H:%M:%S'                  // Replaced by the locale's appropriate date representation
      };
      for (var rule in EXPANSION_RULES_1) {
        pattern = pattern.replace(new RegExp(rule, 'g'), EXPANSION_RULES_1[rule]);
      }
  
      var WEEKDAYS = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
      var MONTHS = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
  
      function leadingSomething(value, digits, character) {
        var str = typeof value === 'number' ? value.toString() : (value || '');
        while (str.length < digits) {
          str = character[0]+str;
        }
        return str;
      };
  
      function leadingNulls(value, digits) {
        return leadingSomething(value, digits, '0');
      };
  
      function compareByDay(date1, date2) {
        function sgn(value) {
          return value < 0 ? -1 : (value > 0 ? 1 : 0);
        };
  
        var compare;
        if ((compare = sgn(date1.getFullYear()-date2.getFullYear())) === 0) {
          if ((compare = sgn(date1.getMonth()-date2.getMonth())) === 0) {
            compare = sgn(date1.getDate()-date2.getDate());
          }
        }
        return compare;
      };
  
      function getFirstWeekStartDate(janFourth) {
          switch (janFourth.getDay()) {
            case 0: // Sunday
              return new Date(janFourth.getFullYear()-1, 11, 29);
            case 1: // Monday
              return janFourth;
            case 2: // Tuesday
              return new Date(janFourth.getFullYear(), 0, 3);
            case 3: // Wednesday
              return new Date(janFourth.getFullYear(), 0, 2);
            case 4: // Thursday
              return new Date(janFourth.getFullYear(), 0, 1);
            case 5: // Friday
              return new Date(janFourth.getFullYear()-1, 11, 31);
            case 6: // Saturday
              return new Date(janFourth.getFullYear()-1, 11, 30);
          }
      };
  
      function getWeekBasedYear(date) {
          var thisDate = __addDays(new Date(date.tm_year+1900, 0, 1), date.tm_yday);
  
          var janFourthThisYear = new Date(thisDate.getFullYear(), 0, 4);
          var janFourthNextYear = new Date(thisDate.getFullYear()+1, 0, 4);
  
          var firstWeekStartThisYear = getFirstWeekStartDate(janFourthThisYear);
          var firstWeekStartNextYear = getFirstWeekStartDate(janFourthNextYear);
  
          if (compareByDay(firstWeekStartThisYear, thisDate) <= 0) {
            // this date is after the start of the first week of this year
            if (compareByDay(firstWeekStartNextYear, thisDate) <= 0) {
              return thisDate.getFullYear()+1;
            } else {
              return thisDate.getFullYear();
            }
          } else {
            return thisDate.getFullYear()-1;
          }
      };
  
      var EXPANSION_RULES_2 = {
        '%a': function(date) {
          return WEEKDAYS[date.tm_wday].substring(0,3);
        },
        '%A': function(date) {
          return WEEKDAYS[date.tm_wday];
        },
        '%b': function(date) {
          return MONTHS[date.tm_mon].substring(0,3);
        },
        '%B': function(date) {
          return MONTHS[date.tm_mon];
        },
        '%C': function(date) {
          var year = date.tm_year+1900;
          return leadingNulls((year/100)|0,2);
        },
        '%d': function(date) {
          return leadingNulls(date.tm_mday, 2);
        },
        '%e': function(date) {
          return leadingSomething(date.tm_mday, 2, ' ');
        },
        '%g': function(date) {
          // %g, %G, and %V give values according to the ISO 8601:2000 standard week-based year.
          // In this system, weeks begin on a Monday and week 1 of the year is the week that includes
          // January 4th, which is also the week that includes the first Thursday of the year, and
          // is also the first week that contains at least four days in the year.
          // If the first Monday of January is the 2nd, 3rd, or 4th, the preceding days are part of
          // the last week of the preceding year; thus, for Saturday 2nd January 1999,
          // %G is replaced by 1998 and %V is replaced by 53. If December 29th, 30th,
          // or 31st is a Monday, it and any following days are part of week 1 of the following year.
          // Thus, for Tuesday 30th December 1997, %G is replaced by 1998 and %V is replaced by 01.
  
          return getWeekBasedYear(date).toString().substring(2);
        },
        '%G': function(date) {
          return getWeekBasedYear(date);
        },
        '%H': function(date) {
          return leadingNulls(date.tm_hour, 2);
        },
        '%I': function(date) {
          var twelveHour = date.tm_hour;
          if (twelveHour == 0) twelveHour = 12;
          else if (twelveHour > 12) twelveHour -= 12;
          return leadingNulls(twelveHour, 2);
        },
        '%j': function(date) {
          // Day of the year (001-366)
          return leadingNulls(date.tm_mday+__arraySum(__isLeapYear(date.tm_year+1900) ? __MONTH_DAYS_LEAP : __MONTH_DAYS_REGULAR, date.tm_mon-1), 3);
        },
        '%m': function(date) {
          return leadingNulls(date.tm_mon+1, 2);
        },
        '%M': function(date) {
          return leadingNulls(date.tm_min, 2);
        },
        '%n': function() {
          return '\n';
        },
        '%p': function(date) {
          if (date.tm_hour >= 0 && date.tm_hour < 12) {
            return 'AM';
          } else {
            return 'PM';
          }
        },
        '%S': function(date) {
          return leadingNulls(date.tm_sec, 2);
        },
        '%t': function() {
          return '\t';
        },
        '%u': function(date) {
          var day = new Date(date.tm_year+1900, date.tm_mon+1, date.tm_mday, 0, 0, 0, 0);
          return day.getDay() || 7;
        },
        '%U': function(date) {
          // Replaced by the week number of the year as a decimal number [00,53].
          // The first Sunday of January is the first day of week 1;
          // days in the new year before this are in week 0. [ tm_year, tm_wday, tm_yday]
          var janFirst = new Date(date.tm_year+1900, 0, 1);
          var firstSunday = janFirst.getDay() === 0 ? janFirst : __addDays(janFirst, 7-janFirst.getDay());
          var endDate = new Date(date.tm_year+1900, date.tm_mon, date.tm_mday);
  
          // is target date after the first Sunday?
          if (compareByDay(firstSunday, endDate) < 0) {
            // calculate difference in days between first Sunday and endDate
            var februaryFirstUntilEndMonth = __arraySum(__isLeapYear(endDate.getFullYear()) ? __MONTH_DAYS_LEAP : __MONTH_DAYS_REGULAR, endDate.getMonth()-1)-31;
            var firstSundayUntilEndJanuary = 31-firstSunday.getDate();
            var days = firstSundayUntilEndJanuary+februaryFirstUntilEndMonth+endDate.getDate();
            return leadingNulls(Math.ceil(days/7), 2);
          }
  
          return compareByDay(firstSunday, janFirst) === 0 ? '01': '00';
        },
        '%V': function(date) {
          // Replaced by the week number of the year (Monday as the first day of the week)
          // as a decimal number [01,53]. If the week containing 1 January has four
          // or more days in the new year, then it is considered week 1.
          // Otherwise, it is the last week of the previous year, and the next week is week 1.
          // Both January 4th and the first Thursday of January are always in week 1. [ tm_year, tm_wday, tm_yday]
          var janFourthThisYear = new Date(date.tm_year+1900, 0, 4);
          var janFourthNextYear = new Date(date.tm_year+1901, 0, 4);
  
          var firstWeekStartThisYear = getFirstWeekStartDate(janFourthThisYear);
          var firstWeekStartNextYear = getFirstWeekStartDate(janFourthNextYear);
  
          var endDate = __addDays(new Date(date.tm_year+1900, 0, 1), date.tm_yday);
  
          if (compareByDay(endDate, firstWeekStartThisYear) < 0) {
            // if given date is before this years first week, then it belongs to the 53rd week of last year
            return '53';
          }
  
          if (compareByDay(firstWeekStartNextYear, endDate) <= 0) {
            // if given date is after next years first week, then it belongs to the 01th week of next year
            return '01';
          }
  
          // given date is in between CW 01..53 of this calendar year
          var daysDifference;
          if (firstWeekStartThisYear.getFullYear() < date.tm_year+1900) {
            // first CW of this year starts last year
            daysDifference = date.tm_yday+32-firstWeekStartThisYear.getDate()
          } else {
            // first CW of this year starts this year
            daysDifference = date.tm_yday+1-firstWeekStartThisYear.getDate();
          }
          return leadingNulls(Math.ceil(daysDifference/7), 2);
        },
        '%w': function(date) {
          var day = new Date(date.tm_year+1900, date.tm_mon+1, date.tm_mday, 0, 0, 0, 0);
          return day.getDay();
        },
        '%W': function(date) {
          // Replaced by the week number of the year as a decimal number [00,53].
          // The first Monday of January is the first day of week 1;
          // days in the new year before this are in week 0. [ tm_year, tm_wday, tm_yday]
          var janFirst = new Date(date.tm_year, 0, 1);
          var firstMonday = janFirst.getDay() === 1 ? janFirst : __addDays(janFirst, janFirst.getDay() === 0 ? 1 : 7-janFirst.getDay()+1);
          var endDate = new Date(date.tm_year+1900, date.tm_mon, date.tm_mday);
  
          // is target date after the first Monday?
          if (compareByDay(firstMonday, endDate) < 0) {
            var februaryFirstUntilEndMonth = __arraySum(__isLeapYear(endDate.getFullYear()) ? __MONTH_DAYS_LEAP : __MONTH_DAYS_REGULAR, endDate.getMonth()-1)-31;
            var firstMondayUntilEndJanuary = 31-firstMonday.getDate();
            var days = firstMondayUntilEndJanuary+februaryFirstUntilEndMonth+endDate.getDate();
            return leadingNulls(Math.ceil(days/7), 2);
          }
          return compareByDay(firstMonday, janFirst) === 0 ? '01': '00';
        },
        '%y': function(date) {
          // Replaced by the last two digits of the year as a decimal number [00,99]. [ tm_year]
          return (date.tm_year+1900).toString().substring(2);
        },
        '%Y': function(date) {
          // Replaced by the year as a decimal number (for example, 1997). [ tm_year]
          return date.tm_year+1900;
        },
        '%z': function(date) {
          // Replaced by the offset from UTC in the ISO 8601:2000 standard format ( +hhmm or -hhmm ).
          // For example, "-0430" means 4 hours 30 minutes behind UTC (west of Greenwich).
          var off = date.tm_gmtoff;
          var ahead = off >= 0;
          off = Math.abs(off) / 60;
          // convert from minutes into hhmm format (which means 60 minutes = 100 units)
          off = (off / 60)*100 + (off % 60);
          return (ahead ? '+' : '-') + String("0000" + off).slice(-4);
        },
        '%Z': function(date) {
          return date.tm_zone;
        },
        '%%': function() {
          return '%';
        }
      };
      for (var rule in EXPANSION_RULES_2) {
        if (pattern.indexOf(rule) >= 0) {
          pattern = pattern.replace(new RegExp(rule, 'g'), EXPANSION_RULES_2[rule](date));
        }
      }
  
      var bytes = intArrayFromString(pattern, false);
      if (bytes.length > maxsize) {
        return 0;
      }
  
      writeArrayToMemory(bytes, s);
      return bytes.length-1;
    }

  function _strftime_l(s, maxsize, format, tm) {
      return _strftime(s, maxsize, format, tm); // no locale support yet
    }

  function _sysconf(name) {
      // long sysconf(int name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/sysconf.html
      switch(name) {
        case 30: return PAGE_SIZE;
        case 85:
          var maxHeapSize = 2*1024*1024*1024 - 16777216;
          return maxHeapSize / PAGE_SIZE;
        case 132:
        case 133:
        case 12:
        case 137:
        case 138:
        case 15:
        case 235:
        case 16:
        case 17:
        case 18:
        case 19:
        case 20:
        case 149:
        case 13:
        case 10:
        case 236:
        case 153:
        case 9:
        case 21:
        case 22:
        case 159:
        case 154:
        case 14:
        case 77:
        case 78:
        case 139:
        case 80:
        case 81:
        case 82:
        case 68:
        case 67:
        case 164:
        case 11:
        case 29:
        case 47:
        case 48:
        case 95:
        case 52:
        case 51:
        case 46:
          return 200809;
        case 79:
          return 0;
        case 27:
        case 246:
        case 127:
        case 128:
        case 23:
        case 24:
        case 160:
        case 161:
        case 181:
        case 182:
        case 242:
        case 183:
        case 184:
        case 243:
        case 244:
        case 245:
        case 165:
        case 178:
        case 179:
        case 49:
        case 50:
        case 168:
        case 169:
        case 175:
        case 170:
        case 171:
        case 172:
        case 97:
        case 76:
        case 32:
        case 173:
        case 35:
          return -1;
        case 176:
        case 177:
        case 7:
        case 155:
        case 8:
        case 157:
        case 125:
        case 126:
        case 92:
        case 93:
        case 129:
        case 130:
        case 131:
        case 94:
        case 91:
          return 1;
        case 74:
        case 60:
        case 69:
        case 70:
        case 4:
          return 1024;
        case 31:
        case 42:
        case 72:
          return 32;
        case 87:
        case 26:
        case 33:
          return 2147483647;
        case 34:
        case 1:
          return 47839;
        case 38:
        case 36:
          return 99;
        case 43:
        case 37:
          return 2048;
        case 0: return 2097152;
        case 3: return 65536;
        case 28: return 32768;
        case 44: return 32767;
        case 75: return 16384;
        case 39: return 1000;
        case 89: return 700;
        case 71: return 256;
        case 40: return 255;
        case 2: return 100;
        case 180: return 64;
        case 25: return 20;
        case 5: return 16;
        case 6: return 6;
        case 73: return 4;
        case 84: {
          if (typeof navigator === 'object') return navigator['hardwareConcurrency'] || 1;
          return 1;
        }
      }
      ___setErrNo(ERRNO_CODES.EINVAL);
      return -1;
    }

  function _time(ptr) {
      var ret = (Date.now()/1000)|0;
      if (ptr) {
        HEAP32[((ptr)>>2)]=ret;
      }
      return ret;
    }

  function _utimes(path, times) {
      var time;
      if (times) {
        var offset = 8 + 0;
        time = HEAP32[(((times)+(offset))>>2)] * 1000;
        offset = 8 + 4;
        time += HEAP32[(((times)+(offset))>>2)] / 1000;
      } else {
        time = Date.now();
      }
      path = Pointer_stringify(path);
      try {
        FS.utime(path, time, time);
        return 0;
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }
SB.init();;
if (ENVIRONMENT_IS_NODE) {
    _emscripten_get_now = function _emscripten_get_now_actual() {
      var t = process['hrtime']();
      return t[0] * 1e3 + t[1] / 1e6;
    };
  } else if (typeof dateNow !== 'undefined') {
    _emscripten_get_now = dateNow;
  } else if (typeof self === 'object' && self['performance'] && typeof self['performance']['now'] === 'function') {
    _emscripten_get_now = function() { return self['performance']['now'](); };
  } else if (typeof performance === 'object' && typeof performance['now'] === 'function') {
    _emscripten_get_now = function() { return performance['now'](); };
  } else {
    _emscripten_get_now = Date.now;
  };
FS.staticInit();__ATINIT__.unshift(function() { if (!Module["noFSInit"] && !FS.init.initialized) FS.init() });__ATMAIN__.push(function() { FS.ignorePermissions = false });__ATEXIT__.push(function() { FS.quit() });;
__ATINIT__.unshift(function() { TTY.init() });__ATEXIT__.push(function() { TTY.shutdown() });;
if (ENVIRONMENT_IS_NODE) { var fs = require("fs"); var NODEJS_PATH = require("path"); NODEFS.staticInit(); };
DYNAMICTOP_PTR = staticAlloc(4);

STACK_BASE = STACKTOP = alignMemory(STATICTOP);

STACK_MAX = STACK_BASE + TOTAL_STACK;

DYNAMIC_BASE = alignMemory(STACK_MAX);

HEAP32[DYNAMICTOP_PTR>>2] = DYNAMIC_BASE;

staticSealed = true; // seal the static portion of memory

var ASSERTIONS = false;

/** @type {function(string, boolean=, number=)} */
function intArrayFromString(stringy, dontAddNull, length) {
  var len = length > 0 ? length : lengthBytesUTF8(stringy)+1;
  var u8array = new Array(len);
  var numBytesWritten = stringToUTF8Array(stringy, u8array, 0, u8array.length);
  if (dontAddNull) u8array.length = numBytesWritten;
  return u8array;
}

function intArrayToString(array) {
  var ret = [];
  for (var i = 0; i < array.length; i++) {
    var chr = array[i];
    if (chr > 0xFF) {
      if (ASSERTIONS) {
        assert(false, 'Character code ' + chr + ' (' + String.fromCharCode(chr) + ')  at offset ' + i + ' not in 0x00-0xFF.');
      }
      chr &= 0xFF;
    }
    ret.push(String.fromCharCode(chr));
  }
  return ret.join('');
}


// Copied from https://github.com/strophe/strophejs/blob/e06d027/src/polyfills.js#L149

// This code was written by Tyler Akins and has been placed in the
// public domain.  It would be nice if you left this header intact.
// Base64 code from Tyler Akins -- http://rumkin.com

/**
 * Decodes a base64 string.
 * @param {String} input The string to decode.
 */
var decodeBase64 = typeof atob === 'function' ? atob : function (input) {
  var keyStr = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';

  var output = '';
  var chr1, chr2, chr3;
  var enc1, enc2, enc3, enc4;
  var i = 0;
  // remove all characters that are not A-Z, a-z, 0-9, +, /, or =
  input = input.replace(/[^A-Za-z0-9\+\/\=]/g, '');
  do {
    enc1 = keyStr.indexOf(input.charAt(i++));
    enc2 = keyStr.indexOf(input.charAt(i++));
    enc3 = keyStr.indexOf(input.charAt(i++));
    enc4 = keyStr.indexOf(input.charAt(i++));

    chr1 = (enc1 << 2) | (enc2 >> 4);
    chr2 = ((enc2 & 15) << 4) | (enc3 >> 2);
    chr3 = ((enc3 & 3) << 6) | enc4;

    output = output + String.fromCharCode(chr1);

    if (enc3 !== 64) {
      output = output + String.fromCharCode(chr2);
    }
    if (enc4 !== 64) {
      output = output + String.fromCharCode(chr3);
    }
  } while (i < input.length);
  return output;
};

// Converts a string of base64 into a byte array.
// Throws error on invalid input.
function intArrayFromBase64(s) {
  if (typeof ENVIRONMENT_IS_NODE === 'boolean' && ENVIRONMENT_IS_NODE) {
    var buf;
    try {
      buf = Buffer.from(s, 'base64');
    } catch (_) {
      buf = new Buffer(s, 'base64');
    }
    return new Uint8Array(buf.buffer, buf.byteOffset, buf.byteLength);
  }

  try {
    var decoded = decodeBase64(s);
    var bytes = new Uint8Array(decoded.length);
    for (var i = 0 ; i < decoded.length ; ++i) {
      bytes[i] = decoded.charCodeAt(i);
    }
    return bytes;
  } catch (_) {
    throw new Error('Converting base64 string to bytes failed.');
  }
}

// If filename is a base64 data URI, parses and returns data (Buffer on node,
// Uint8Array otherwise). If filename is not a base64 data URI, returns undefined.
function tryParseAsDataURI(filename) {
  if (!isDataURI(filename)) {
    return;
  }

  return intArrayFromBase64(filename.slice(dataURIPrefix.length));
}



function invoke_di(index,a1) {
  var sp = stackSave();
  try {
    return Module["dynCall_di"](index,a1);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_dii(index,a1,a2) {
  var sp = stackSave();
  try {
    return Module["dynCall_dii"](index,a1,a2);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_i(index) {
  var sp = stackSave();
  try {
    return Module["dynCall_i"](index);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_ii(index,a1) {
  var sp = stackSave();
  try {
    return Module["dynCall_ii"](index,a1);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iii(index,a1,a2) {
  var sp = stackSave();
  try {
    return Module["dynCall_iii"](index,a1,a2);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiid(index,a1,a2,a3) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiid"](index,a1,a2,a3);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiii(index,a1,a2,a3) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiii"](index,a1,a2,a3);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiidii(index,a1,a2,a3,a4,a5,a6) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiidii"](index,a1,a2,a3,a4,a5,a6);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiii(index,a1,a2,a3,a4) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiii"](index,a1,a2,a3,a4);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiid(index,a1,a2,a3,a4,a5) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiid"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiii(index,a1,a2,a3,a4,a5) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiii"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiiid(index,a1,a2,a3,a4,a5,a6) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiiid"](index,a1,a2,a3,a4,a5,a6);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiiii(index,a1,a2,a3,a4,a5,a6) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiiii"](index,a1,a2,a3,a4,a5,a6);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiiiii(index,a1,a2,a3,a4,a5,a6,a7) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiiiii"](index,a1,a2,a3,a4,a5,a6,a7);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiiiiii(index,a1,a2,a3,a4,a5,a6,a7,a8) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiiiiii"](index,a1,a2,a3,a4,a5,a6,a7,a8);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiiiiiii(index,a1,a2,a3,a4,a5,a6,a7,a8,a9) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiiiiiii"](index,a1,a2,a3,a4,a5,a6,a7,a8,a9);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_iiiiiiiiiii(index,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10) {
  var sp = stackSave();
  try {
    return Module["dynCall_iiiiiiiiiii"](index,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_v(index) {
  var sp = stackSave();
  try {
    Module["dynCall_v"](index);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_vi(index,a1) {
  var sp = stackSave();
  try {
    Module["dynCall_vi"](index,a1);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_vid(index,a1,a2) {
  var sp = stackSave();
  try {
    Module["dynCall_vid"](index,a1,a2);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_vii(index,a1,a2) {
  var sp = stackSave();
  try {
    Module["dynCall_vii"](index,a1,a2);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viii(index,a1,a2,a3) {
  var sp = stackSave();
  try {
    Module["dynCall_viii"](index,a1,a2,a3);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiidi(index,a1,a2,a3,a4,a5) {
  var sp = stackSave();
  try {
    Module["dynCall_viiidi"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiii(index,a1,a2,a3,a4) {
  var sp = stackSave();
  try {
    Module["dynCall_viiii"](index,a1,a2,a3,a4);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiiid(index,a1,a2,a3,a4,a5) {
  var sp = stackSave();
  try {
    Module["dynCall_viiiid"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiiii(index,a1,a2,a3,a4,a5) {
  var sp = stackSave();
  try {
    Module["dynCall_viiiii"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

function invoke_viiiiii(index,a1,a2,a3,a4,a5,a6) {
  var sp = stackSave();
  try {
    Module["dynCall_viiiiii"](index,a1,a2,a3,a4,a5,a6);
  } catch(e) {
    stackRestore(sp);
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    Module["setThrew"](1, 0);
  }
}

Module.asmGlobalArg = { "Math": Math, "Int8Array": Int8Array, "Int16Array": Int16Array, "Int32Array": Int32Array, "Uint8Array": Uint8Array, "Uint16Array": Uint16Array, "Uint32Array": Uint32Array, "Float32Array": Float32Array, "Float64Array": Float64Array, "NaN": NaN, "Infinity": Infinity, "byteLength": byteLength };

Module.asmLibraryArg = { "abort": abort, "assert": assert, "enlargeMemory": enlargeMemory, "getTotalMemory": getTotalMemory, "abortOnCannotGrowMemory": abortOnCannotGrowMemory, "invoke_di": invoke_di, "invoke_dii": invoke_dii, "invoke_i": invoke_i, "invoke_ii": invoke_ii, "invoke_iii": invoke_iii, "invoke_iiid": invoke_iiid, "invoke_iiii": invoke_iiii, "invoke_iiiidii": invoke_iiiidii, "invoke_iiiii": invoke_iiiii, "invoke_iiiiid": invoke_iiiiid, "invoke_iiiiii": invoke_iiiiii, "invoke_iiiiiid": invoke_iiiiiid, "invoke_iiiiiii": invoke_iiiiiii, "invoke_iiiiiiii": invoke_iiiiiiii, "invoke_iiiiiiiii": invoke_iiiiiiiii, "invoke_iiiiiiiiii": invoke_iiiiiiiiii, "invoke_iiiiiiiiiii": invoke_iiiiiiiiiii, "invoke_v": invoke_v, "invoke_vi": invoke_vi, "invoke_vid": invoke_vid, "invoke_vii": invoke_vii, "invoke_viii": invoke_viii, "invoke_viiidi": invoke_viiidi, "invoke_viiii": invoke_viiii, "invoke_viiiid": invoke_viiiid, "invoke_viiiii": invoke_viiiii, "invoke_viiiiii": invoke_viiiiii, "_SceneBuilderAdapter_beginParametricTextContent": _SceneBuilderAdapter_beginParametricTextContent, "_SceneBuilderAdapter_clearFillStyle": _SceneBuilderAdapter_clearFillStyle, "_SceneBuilderAdapter_clearLineStyle": _SceneBuilderAdapter_clearLineStyle, "_SceneBuilderAdapter_createCamera": _SceneBuilderAdapter_createCamera, "_SceneBuilderAdapter_createDefaultMaterial": _SceneBuilderAdapter_createDefaultMaterial, "_SceneBuilderAdapter_createDetailView": _SceneBuilderAdapter_createDetailView, "_SceneBuilderAdapter_createImage": _SceneBuilderAdapter_createImage, "_SceneBuilderAdapter_createImageNote": _SceneBuilderAdapter_createImageNote, "_SceneBuilderAdapter_createMaterial": _SceneBuilderAdapter_createMaterial, "_SceneBuilderAdapter_createMesh": _SceneBuilderAdapter_createMesh, "_SceneBuilderAdapter_createNode": _SceneBuilderAdapter_createNode, "_SceneBuilderAdapter_createTextAnnotation": _SceneBuilderAdapter_createTextAnnotation, "_SceneBuilderAdapter_createTextNote": _SceneBuilderAdapter_createTextNote, "_SceneBuilderAdapter_createThumbnail": _SceneBuilderAdapter_createThumbnail, "_SceneBuilderAdapter_createViewportGroup": _SceneBuilderAdapter_createViewportGroup, "_SceneBuilderAdapter_endParametricTextContent": _SceneBuilderAdapter_endParametricTextContent, "_SceneBuilderAdapter_finalizeAnimation": _SceneBuilderAdapter_finalizeAnimation, "_SceneBuilderAdapter_finalizePlaybacks": _SceneBuilderAdapter_finalizePlaybacks, "_SceneBuilderAdapter_insertAnimation": _SceneBuilderAdapter_insertAnimation, "_SceneBuilderAdapter_insertAnimationGroup": _SceneBuilderAdapter_insertAnimationGroup, "_SceneBuilderAdapter_insertAnimationTarget": _SceneBuilderAdapter_insertAnimationTarget, "_SceneBuilderAdapter_insertAnimationTrack": _SceneBuilderAdapter_insertAnimationTrack, "_SceneBuilderAdapter_insertArcPathSegment": _SceneBuilderAdapter_insertArcPathSegment, "_SceneBuilderAdapter_insertBezierPathSegment": _SceneBuilderAdapter_insertBezierPathSegment, "_SceneBuilderAdapter_insertCamera": _SceneBuilderAdapter_insertCamera, "_SceneBuilderAdapter_insertClosePathSegment": _SceneBuilderAdapter_insertClosePathSegment, "_SceneBuilderAdapter_insertLeaderLine": _SceneBuilderAdapter_insertLeaderLine, "_SceneBuilderAdapter_insertLinePathSegment": _SceneBuilderAdapter_insertLinePathSegment, "_SceneBuilderAdapter_insertMesh": _SceneBuilderAdapter_insertMesh, "_SceneBuilderAdapter_insertModelView": _SceneBuilderAdapter_insertModelView, "_SceneBuilderAdapter_insertModelViewHighlight": _SceneBuilderAdapter_insertModelViewHighlight, "_SceneBuilderAdapter_insertMovePathSegment": _SceneBuilderAdapter_insertMovePathSegment, "_SceneBuilderAdapter_insertParametricArc": _SceneBuilderAdapter_insertParametricArc, "_SceneBuilderAdapter_insertParametricEllipse": _SceneBuilderAdapter_insertParametricEllipse, "_SceneBuilderAdapter_insertParametricPath": _SceneBuilderAdapter_insertParametricPath, "_SceneBuilderAdapter_insertParametricPolyline": _SceneBuilderAdapter_insertParametricPolyline, "_SceneBuilderAdapter_insertParametricRectangle": _SceneBuilderAdapter_insertParametricRectangle, "_SceneBuilderAdapter_insertParametricText": _SceneBuilderAdapter_insertParametricText, "_SceneBuilderAdapter_insertParametricTextData": _SceneBuilderAdapter_insertParametricTextData, "_SceneBuilderAdapter_insertThrustline": _SceneBuilderAdapter_insertThrustline, "_SceneBuilderAdapter_progress": _SceneBuilderAdapter_progress, "_SceneBuilderAdapter_setFillStyle": _SceneBuilderAdapter_setFillStyle, "_SceneBuilderAdapter_setLineStyle": _SceneBuilderAdapter_setLineStyle, "_SceneBuilderAdapter_setMeshGeometry": _SceneBuilderAdapter_setMeshGeometry, "_SceneBuilderAdapter_setModelViewVisibility": _SceneBuilderAdapter_setModelViewVisibility, "_SceneBuilderAdapter_setParametricContent": _SceneBuilderAdapter_setParametricContent, "_SceneBuilderAdapter_setScene": _SceneBuilderAdapter_setScene, "___buildEnvironment": ___buildEnvironment, "___clock_gettime": ___clock_gettime, "___cxa_allocate_exception": ___cxa_allocate_exception, "___cxa_begin_catch": ___cxa_begin_catch, "___cxa_end_catch": ___cxa_end_catch, "___cxa_find_matching_catch": ___cxa_find_matching_catch, "___cxa_free_exception": ___cxa_free_exception, "___cxa_pure_virtual": ___cxa_pure_virtual, "___cxa_rethrow": ___cxa_rethrow, "___cxa_throw": ___cxa_throw, "___cxa_uncaught_exception": ___cxa_uncaught_exception, "___gxx_personality_v0": ___gxx_personality_v0, "___lock": ___lock, "___map_file": ___map_file, "___resumeException": ___resumeException, "___setErrNo": ___setErrNo, "___syscall10": ___syscall10, "___syscall118": ___syscall118, "___syscall140": ___syscall140, "___syscall145": ___syscall145, "___syscall146": ___syscall146, "___syscall15": ___syscall15, "___syscall183": ___syscall183, "___syscall192": ___syscall192, "___syscall194": ___syscall194, "___syscall195": ___syscall195, "___syscall196": ___syscall196, "___syscall197": ___syscall197, "___syscall20": ___syscall20, "___syscall201": ___syscall201, "___syscall202": ___syscall202, "___syscall207": ___syscall207, "___syscall212": ___syscall212, "___syscall221": ___syscall221, "___syscall3": ___syscall3, "___syscall33": ___syscall33, "___syscall39": ___syscall39, "___syscall4": ___syscall4, "___syscall40": ___syscall40, "___syscall5": ___syscall5, "___syscall54": ___syscall54, "___syscall6": ___syscall6, "___syscall85": ___syscall85, "___syscall91": ___syscall91, "___syscall94": ___syscall94, "___unlock": ___unlock, "__addDays": __addDays, "__arraySum": __arraySum, "__isLeapYear": __isLeapYear, "_abort": _abort, "_clock": _clock, "_clock_gettime": _clock_gettime, "_decrypt": _decrypt, "_derive_key": _derive_key, "_dlclose": _dlclose, "_dlerror": _dlerror, "_dlopen": _dlopen, "_dlsym": _dlsym, "_emscripten_asm_const_idi": _emscripten_asm_const_idi, "_emscripten_asm_const_ii": _emscripten_asm_const_ii, "_emscripten_asm_const_iiii": _emscripten_asm_const_iiii, "_emscripten_asm_const_iiiii": _emscripten_asm_const_iiiii, "_emscripten_get_now": _emscripten_get_now, "_emscripten_get_now_is_monotonic": _emscripten_get_now_is_monotonic, "_emscripten_memcpy_big": _emscripten_memcpy_big, "_getenv": _getenv, "_gettimeofday": _gettimeofday, "_is_decryption_handler_installed": _is_decryption_handler_installed, "_jsStringify": _jsStringify, "_llvm_ceil_f32": _llvm_ceil_f32, "_llvm_cos_f32": _llvm_cos_f32, "_llvm_fabs_f32": _llvm_fabs_f32, "_llvm_fabs_f64": _llvm_fabs_f64, "_llvm_log10_f32": _llvm_log10_f32, "_llvm_log10_f64": _llvm_log10_f64, "_llvm_sin_f32": _llvm_sin_f32, "_llvm_stackrestore": _llvm_stackrestore, "_llvm_stacksave": _llvm_stacksave, "_llvm_trap": _llvm_trap, "_localtime": _localtime, "_localtime_r": _localtime_r, "_nanosleep": _nanosleep, "_pthread_cond_wait": _pthread_cond_wait, "_pthread_create": _pthread_create, "_pthread_getspecific": _pthread_getspecific, "_pthread_join": _pthread_join, "_pthread_key_create": _pthread_key_create, "_pthread_mutex_destroy": _pthread_mutex_destroy, "_pthread_mutex_init": _pthread_mutex_init, "_pthread_mutexattr_destroy": _pthread_mutexattr_destroy, "_pthread_mutexattr_init": _pthread_mutexattr_init, "_pthread_mutexattr_settype": _pthread_mutexattr_settype, "_pthread_once": _pthread_once, "_pthread_setspecific": _pthread_setspecific, "_strftime": _strftime, "_strftime_l": _strftime_l, "_sysconf": _sysconf, "_time": _time, "_tzset": _tzset, "_usleep": _usleep, "_utimes": _utimes, "DYNAMICTOP_PTR": DYNAMICTOP_PTR, "tempDoublePtr": tempDoublePtr, "STACKTOP": STACKTOP, "STACK_MAX": STACK_MAX };
// EMSCRIPTEN_START_ASM
var asm = (/** @suppress {uselessCode} */ function(global, env, buffer) {
'almost asm';


  var Int8View = global.Int8Array;
  var HEAP8 = new Int8View(buffer);
  var Int16View = global.Int16Array;
  var HEAP16 = new Int16View(buffer);
  var Int32View = global.Int32Array;
  var HEAP32 = new Int32View(buffer);
  var Uint8View = global.Uint8Array;
  var HEAPU8 = new Uint8View(buffer);
  var Uint16View = global.Uint16Array;
  var HEAPU16 = new Uint16View(buffer);
  var Uint32View = global.Uint32Array;
  var HEAPU32 = new Uint32View(buffer);
  var Float32View = global.Float32Array;
  var HEAPF32 = new Float32View(buffer);
  var Float64View = global.Float64Array;
  var HEAPF64 = new Float64View(buffer);
  var byteLength = global.byteLength;

  var DYNAMICTOP_PTR=env.DYNAMICTOP_PTR|0;
  var tempDoublePtr=env.tempDoublePtr|0;
  var STACKTOP=env.STACKTOP|0;
  var STACK_MAX=env.STACK_MAX|0;

  var __THREW__ = 0;
  var threwValue = 0;
  var setjmpId = 0;
  var undef = 0;
  var nan = global.NaN, inf = global.Infinity;
  var tempInt = 0, tempBigInt = 0, tempBigIntS = 0, tempValue = 0, tempDouble = 0.0;
  var tempRet0 = 0;

  var Math_floor=global.Math.floor;
  var Math_abs=global.Math.abs;
  var Math_sqrt=global.Math.sqrt;
  var Math_pow=global.Math.pow;
  var Math_cos=global.Math.cos;
  var Math_sin=global.Math.sin;
  var Math_tan=global.Math.tan;
  var Math_acos=global.Math.acos;
  var Math_asin=global.Math.asin;
  var Math_atan=global.Math.atan;
  var Math_atan2=global.Math.atan2;
  var Math_exp=global.Math.exp;
  var Math_log=global.Math.log;
  var Math_ceil=global.Math.ceil;
  var Math_imul=global.Math.imul;
  var Math_min=global.Math.min;
  var Math_max=global.Math.max;
  var Math_clz32=global.Math.clz32;
  var abort=env.abort;
  var assert=env.assert;
  var enlargeMemory=env.enlargeMemory;
  var getTotalMemory=env.getTotalMemory;
  var abortOnCannotGrowMemory=env.abortOnCannotGrowMemory;
  var invoke_di=env.invoke_di;
  var invoke_dii=env.invoke_dii;
  var invoke_i=env.invoke_i;
  var invoke_ii=env.invoke_ii;
  var invoke_iii=env.invoke_iii;
  var invoke_iiid=env.invoke_iiid;
  var invoke_iiii=env.invoke_iiii;
  var invoke_iiiidii=env.invoke_iiiidii;
  var invoke_iiiii=env.invoke_iiiii;
  var invoke_iiiiid=env.invoke_iiiiid;
  var invoke_iiiiii=env.invoke_iiiiii;
  var invoke_iiiiiid=env.invoke_iiiiiid;
  var invoke_iiiiiii=env.invoke_iiiiiii;
  var invoke_iiiiiiii=env.invoke_iiiiiiii;
  var invoke_iiiiiiiii=env.invoke_iiiiiiiii;
  var invoke_iiiiiiiiii=env.invoke_iiiiiiiiii;
  var invoke_iiiiiiiiiii=env.invoke_iiiiiiiiiii;
  var invoke_v=env.invoke_v;
  var invoke_vi=env.invoke_vi;
  var invoke_vid=env.invoke_vid;
  var invoke_vii=env.invoke_vii;
  var invoke_viii=env.invoke_viii;
  var invoke_viiidi=env.invoke_viiidi;
  var invoke_viiii=env.invoke_viiii;
  var invoke_viiiid=env.invoke_viiiid;
  var invoke_viiiii=env.invoke_viiiii;
  var invoke_viiiiii=env.invoke_viiiiii;
  var _SceneBuilderAdapter_beginParametricTextContent=env._SceneBuilderAdapter_beginParametricTextContent;
  var _SceneBuilderAdapter_clearFillStyle=env._SceneBuilderAdapter_clearFillStyle;
  var _SceneBuilderAdapter_clearLineStyle=env._SceneBuilderAdapter_clearLineStyle;
  var _SceneBuilderAdapter_createCamera=env._SceneBuilderAdapter_createCamera;
  var _SceneBuilderAdapter_createDefaultMaterial=env._SceneBuilderAdapter_createDefaultMaterial;
  var _SceneBuilderAdapter_createDetailView=env._SceneBuilderAdapter_createDetailView;
  var _SceneBuilderAdapter_createImage=env._SceneBuilderAdapter_createImage;
  var _SceneBuilderAdapter_createImageNote=env._SceneBuilderAdapter_createImageNote;
  var _SceneBuilderAdapter_createMaterial=env._SceneBuilderAdapter_createMaterial;
  var _SceneBuilderAdapter_createMesh=env._SceneBuilderAdapter_createMesh;
  var _SceneBuilderAdapter_createNode=env._SceneBuilderAdapter_createNode;
  var _SceneBuilderAdapter_createTextAnnotation=env._SceneBuilderAdapter_createTextAnnotation;
  var _SceneBuilderAdapter_createTextNote=env._SceneBuilderAdapter_createTextNote;
  var _SceneBuilderAdapter_createThumbnail=env._SceneBuilderAdapter_createThumbnail;
  var _SceneBuilderAdapter_createViewportGroup=env._SceneBuilderAdapter_createViewportGroup;
  var _SceneBuilderAdapter_endParametricTextContent=env._SceneBuilderAdapter_endParametricTextContent;
  var _SceneBuilderAdapter_finalizeAnimation=env._SceneBuilderAdapter_finalizeAnimation;
  var _SceneBuilderAdapter_finalizePlaybacks=env._SceneBuilderAdapter_finalizePlaybacks;
  var _SceneBuilderAdapter_insertAnimation=env._SceneBuilderAdapter_insertAnimation;
  var _SceneBuilderAdapter_insertAnimationGroup=env._SceneBuilderAdapter_insertAnimationGroup;
  var _SceneBuilderAdapter_insertAnimationTarget=env._SceneBuilderAdapter_insertAnimationTarget;
  var _SceneBuilderAdapter_insertAnimationTrack=env._SceneBuilderAdapter_insertAnimationTrack;
  var _SceneBuilderAdapter_insertArcPathSegment=env._SceneBuilderAdapter_insertArcPathSegment;
  var _SceneBuilderAdapter_insertBezierPathSegment=env._SceneBuilderAdapter_insertBezierPathSegment;
  var _SceneBuilderAdapter_insertCamera=env._SceneBuilderAdapter_insertCamera;
  var _SceneBuilderAdapter_insertClosePathSegment=env._SceneBuilderAdapter_insertClosePathSegment;
  var _SceneBuilderAdapter_insertLeaderLine=env._SceneBuilderAdapter_insertLeaderLine;
  var _SceneBuilderAdapter_insertLinePathSegment=env._SceneBuilderAdapter_insertLinePathSegment;
  var _SceneBuilderAdapter_insertMesh=env._SceneBuilderAdapter_insertMesh;
  var _SceneBuilderAdapter_insertModelView=env._SceneBuilderAdapter_insertModelView;
  var _SceneBuilderAdapter_insertModelViewHighlight=env._SceneBuilderAdapter_insertModelViewHighlight;
  var _SceneBuilderAdapter_insertMovePathSegment=env._SceneBuilderAdapter_insertMovePathSegment;
  var _SceneBuilderAdapter_insertParametricArc=env._SceneBuilderAdapter_insertParametricArc;
  var _SceneBuilderAdapter_insertParametricEllipse=env._SceneBuilderAdapter_insertParametricEllipse;
  var _SceneBuilderAdapter_insertParametricPath=env._SceneBuilderAdapter_insertParametricPath;
  var _SceneBuilderAdapter_insertParametricPolyline=env._SceneBuilderAdapter_insertParametricPolyline;
  var _SceneBuilderAdapter_insertParametricRectangle=env._SceneBuilderAdapter_insertParametricRectangle;
  var _SceneBuilderAdapter_insertParametricText=env._SceneBuilderAdapter_insertParametricText;
  var _SceneBuilderAdapter_insertParametricTextData=env._SceneBuilderAdapter_insertParametricTextData;
  var _SceneBuilderAdapter_insertThrustline=env._SceneBuilderAdapter_insertThrustline;
  var _SceneBuilderAdapter_progress=env._SceneBuilderAdapter_progress;
  var _SceneBuilderAdapter_setFillStyle=env._SceneBuilderAdapter_setFillStyle;
  var _SceneBuilderAdapter_setLineStyle=env._SceneBuilderAdapter_setLineStyle;
  var _SceneBuilderAdapter_setMeshGeometry=env._SceneBuilderAdapter_setMeshGeometry;
  var _SceneBuilderAdapter_setModelViewVisibility=env._SceneBuilderAdapter_setModelViewVisibility;
  var _SceneBuilderAdapter_setParametricContent=env._SceneBuilderAdapter_setParametricContent;
  var _SceneBuilderAdapter_setScene=env._SceneBuilderAdapter_setScene;
  var ___buildEnvironment=env.___buildEnvironment;
  var ___clock_gettime=env.___clock_gettime;
  var ___cxa_allocate_exception=env.___cxa_allocate_exception;
  var ___cxa_begin_catch=env.___cxa_begin_catch;
  var ___cxa_end_catch=env.___cxa_end_catch;
  var ___cxa_find_matching_catch=env.___cxa_find_matching_catch;
  var ___cxa_free_exception=env.___cxa_free_exception;
  var ___cxa_pure_virtual=env.___cxa_pure_virtual;
  var ___cxa_rethrow=env.___cxa_rethrow;
  var ___cxa_throw=env.___cxa_throw;
  var ___cxa_uncaught_exception=env.___cxa_uncaught_exception;
  var ___gxx_personality_v0=env.___gxx_personality_v0;
  var ___lock=env.___lock;
  var ___map_file=env.___map_file;
  var ___resumeException=env.___resumeException;
  var ___setErrNo=env.___setErrNo;
  var ___syscall10=env.___syscall10;
  var ___syscall118=env.___syscall118;
  var ___syscall140=env.___syscall140;
  var ___syscall145=env.___syscall145;
  var ___syscall146=env.___syscall146;
  var ___syscall15=env.___syscall15;
  var ___syscall183=env.___syscall183;
  var ___syscall192=env.___syscall192;
  var ___syscall194=env.___syscall194;
  var ___syscall195=env.___syscall195;
  var ___syscall196=env.___syscall196;
  var ___syscall197=env.___syscall197;
  var ___syscall20=env.___syscall20;
  var ___syscall201=env.___syscall201;
  var ___syscall202=env.___syscall202;
  var ___syscall207=env.___syscall207;
  var ___syscall212=env.___syscall212;
  var ___syscall221=env.___syscall221;
  var ___syscall3=env.___syscall3;
  var ___syscall33=env.___syscall33;
  var ___syscall39=env.___syscall39;
  var ___syscall4=env.___syscall4;
  var ___syscall40=env.___syscall40;
  var ___syscall5=env.___syscall5;
  var ___syscall54=env.___syscall54;
  var ___syscall6=env.___syscall6;
  var ___syscall85=env.___syscall85;
  var ___syscall91=env.___syscall91;
  var ___syscall94=env.___syscall94;
  var ___unlock=env.___unlock;
  var __addDays=env.__addDays;
  var __arraySum=env.__arraySum;
  var __isLeapYear=env.__isLeapYear;
  var _abort=env._abort;
  var _clock=env._clock;
  var _clock_gettime=env._clock_gettime;
  var _decrypt=env._decrypt;
  var _derive_key=env._derive_key;
  var _dlclose=env._dlclose;
  var _dlerror=env._dlerror;
  var _dlopen=env._dlopen;
  var _dlsym=env._dlsym;
  var _emscripten_asm_const_idi=env._emscripten_asm_const_idi;
  var _emscripten_asm_const_ii=env._emscripten_asm_const_ii;
  var _emscripten_asm_const_iiii=env._emscripten_asm_const_iiii;
  var _emscripten_asm_const_iiiii=env._emscripten_asm_const_iiiii;
  var _emscripten_get_now=env._emscripten_get_now;
  var _emscripten_get_now_is_monotonic=env._emscripten_get_now_is_monotonic;
  var _emscripten_memcpy_big=env._emscripten_memcpy_big;
  var _getenv=env._getenv;
  var _gettimeofday=env._gettimeofday;
  var _is_decryption_handler_installed=env._is_decryption_handler_installed;
  var _jsStringify=env._jsStringify;
  var _llvm_ceil_f32=env._llvm_ceil_f32;
  var _llvm_cos_f32=env._llvm_cos_f32;
  var _llvm_fabs_f32=env._llvm_fabs_f32;
  var _llvm_fabs_f64=env._llvm_fabs_f64;
  var _llvm_log10_f32=env._llvm_log10_f32;
  var _llvm_log10_f64=env._llvm_log10_f64;
  var _llvm_sin_f32=env._llvm_sin_f32;
  var _llvm_stackrestore=env._llvm_stackrestore;
  var _llvm_stacksave=env._llvm_stacksave;
  var _llvm_trap=env._llvm_trap;
  var _localtime=env._localtime;
  var _localtime_r=env._localtime_r;
  var _nanosleep=env._nanosleep;
  var _pthread_cond_wait=env._pthread_cond_wait;
  var _pthread_create=env._pthread_create;
  var _pthread_getspecific=env._pthread_getspecific;
  var _pthread_join=env._pthread_join;
  var _pthread_key_create=env._pthread_key_create;
  var _pthread_mutex_destroy=env._pthread_mutex_destroy;
  var _pthread_mutex_init=env._pthread_mutex_init;
  var _pthread_mutexattr_destroy=env._pthread_mutexattr_destroy;
  var _pthread_mutexattr_init=env._pthread_mutexattr_init;
  var _pthread_mutexattr_settype=env._pthread_mutexattr_settype;
  var _pthread_once=env._pthread_once;
  var _pthread_setspecific=env._pthread_setspecific;
  var _strftime=env._strftime;
  var _strftime_l=env._strftime_l;
  var _sysconf=env._sysconf;
  var _time=env._time;
  var _tzset=env._tzset;
  var _usleep=env._usleep;
  var _utimes=env._utimes;
  var tempFloat = 0.0;

function _emscripten_replace_memory(newBuffer) {
  if ((byteLength(newBuffer) & 0xffffff || byteLength(newBuffer) <= 0xffffff) || byteLength(newBuffer) > 0x80000000) return false;
  HEAP8 = new Int8View(newBuffer);
  HEAP16 = new Int16View(newBuffer);
  HEAP32 = new Int32View(newBuffer);
  HEAPU8 = new Uint8View(newBuffer);
  HEAPU16 = new Uint16View(newBuffer);
  HEAPU32 = new Uint32View(newBuffer);
  HEAPF32 = new Float32View(newBuffer);
  HEAPF64 = new Float64View(newBuffer);
  buffer = newBuffer;
  return true;
}

// EMSCRIPTEN_START_FUNCS

function stackAlloc(size) {
  size = size|0;
  var ret = 0;
  ret = STACKTOP;
  STACKTOP = (STACKTOP + size)|0;
  STACKTOP = (STACKTOP + 15)&-16;

  return ret|0;
}
function stackSave() {
  return STACKTOP|0;
}
function stackRestore(top) {
  top = top|0;
  STACKTOP = top;
}
function establishStackSpace(stackBase, stackMax) {
  stackBase = stackBase|0;
  stackMax = stackMax|0;
  STACKTOP = stackBase;
  STACK_MAX = stackMax;
}

function setThrew(threw, value) {
  threw = threw|0;
  value = value|0;
  if ((__THREW__|0) == 0) {
    __THREW__ = threw;
    threwValue = value;
  }
}

function setTempRet0(value) {
  value = value|0;
  tempRet0 = value;
}
function getTempRet0() {
  return tempRet0|0;
}

function __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $2 = sp + 16|0;
 $3 = sp + 12|0;
 $4 = sp + 8|0;
 $5 = sp + 4|0;
 $6 = sp;
 HEAP32[$4>>2] = $0;
 HEAP32[$5>>2] = $1;
 $7 = HEAP32[$5>>2]|0;
 $8 = (_jsStringify(($7|0))|0);
 HEAP32[$6>>2] = $8;
 $9 = HEAP32[$6>>2]|0;
 $10 = HEAP32[$4>>2]|0;
 HEAP32[$2>>2] = $10;
 HEAP32[$3>>2] = $9;
 $11 = HEAP32[$2>>2]|0;
 $12 = HEAP32[$3>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($11,$12)|0);
 $13 = HEAP32[$6>>2]|0;
 _free($13);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter19GetOrCreateMaterialEPKN3sap2ve4core8MaterialE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0.0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0.0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0.0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0.0, $125 = 0, $126 = 0, $127 = 0, $128 = 0.0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0.0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0.0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0.0, $144 = 0, $145 = 0, $146 = 0, $147 = 0.0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0.0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0.0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0.0, $163 = 0, $164 = 0, $165 = 0, $166 = 0.0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0.0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0.0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0.0, $182 = 0, $183 = 0, $184 = 0, $185 = 0.0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0.0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0.0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $2 = 0, $20 = 0, $200 = 0.0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0;
 var $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0;
 var $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0;
 var $244 = 0, $245 = 0, $246 = 0, $247 = 0.0, $248 = 0.0, $249 = 0.0, $25 = 0, $250 = 0, $251 = 0.0, $252 = 0.0, $253 = 0.0, $254 = 0, $255 = 0.0, $256 = 0.0, $257 = 0.0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0.0;
 var $262 = 0.0, $263 = 0.0, $264 = 0.0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0.0, $27 = 0, $270 = 0, $271 = 0, $272 = 0.0, $273 = 0.0, $274 = 0, $275 = 0.0, $276 = 0, $277 = 0, $278 = 0, $279 = 0.0, $28 = 0;
 var $280 = 0.0, $281 = 0.0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0.0, $287 = 0.0, $288 = 0.0, $289 = 0.0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0.0, $295 = 0.0, $296 = 0.0, $297 = 0.0, $298 = 0;
 var $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0.0, $303 = 0, $304 = 0, $305 = 0, $306 = 0.0, $307 = 0.0, $308 = 0.0, $309 = 0.0, $31 = 0, $310 = 0, $311 = 0.0, $312 = 0, $313 = 0, $314 = 0, $315 = 0.0;
 var $316 = 0.0, $317 = 0.0, $318 = 0.0, $319 = 0.0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0.0, $325 = 0, $326 = 0, $327 = 0, $328 = 0.0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0.0;
 var $334 = 0, $335 = 0, $336 = 0, $337 = 0.0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0.0, $343 = 0, $344 = 0, $345 = 0, $346 = 0.0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0;
 var $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0;
 var $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0;
 var $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0;
 var $406 = 0, $407 = 0.0, $408 = 0, $409 = 0.0, $41 = 0, $410 = 0, $411 = 0.0, $412 = 0, $413 = 0, $414 = 0, $415 = 0.0, $416 = 0, $417 = 0.0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0.0, $422 = 0, $423 = 0;
 var $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0;
 var $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0;
 var $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0;
 var $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0;
 var $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 1072|0;
 $2 = sp + 1064|0;
 $3 = sp + 1060|0;
 $4 = sp + 1056|0;
 $5 = sp + 1052|0;
 $6 = sp + 1048|0;
 $7 = sp + 776|0;
 $8 = sp + 1044|0;
 $9 = sp + 1040|0;
 $10 = sp + 1036|0;
 $11 = sp + 1032|0;
 $12 = sp + 1028|0;
 $13 = sp + 1024|0;
 $14 = sp + 1020|0;
 $15 = sp + 1016|0;
 $16 = sp + 1008|0;
 $17 = sp + 1000|0;
 $18 = sp + 996|0;
 $19 = sp + 992|0;
 $20 = sp + 988|0;
 $21 = sp + 984|0;
 $22 = sp + 768|0;
 $23 = sp + 980|0;
 $24 = sp + 976|0;
 $25 = sp + 972|0;
 $26 = sp + 968|0;
 $27 = sp + 964|0;
 $28 = sp + 960|0;
 $29 = sp + 956|0;
 $30 = sp + 952|0;
 $31 = sp + 948|0;
 $32 = sp + 944|0;
 $33 = sp + 936|0;
 $34 = sp + 928|0;
 $35 = sp + 760|0;
 $36 = sp + 924|0;
 $37 = sp + 920|0;
 $38 = sp + 916|0;
 $39 = sp + 912|0;
 $40 = sp + 908|0;
 $41 = sp + 904|0;
 $42 = sp + 900|0;
 $43 = sp + 896|0;
 $44 = sp + 892|0;
 $45 = sp + 888|0;
 $46 = sp + 884|0;
 $47 = sp + 880|0;
 $48 = sp + 876|0;
 $49 = sp + 752|0;
 $50 = sp + 872|0;
 $51 = sp + 868|0;
 $52 = sp + 864|0;
 $53 = sp + 860|0;
 $54 = sp + 856|0;
 $55 = sp + 852|0;
 $56 = sp + 848|0;
 $57 = sp + 844|0;
 $58 = sp + 736|0;
 $59 = sp + 720|0;
 $60 = sp + 704|0;
 $61 = sp + 688|0;
 $62 = sp + 672|0;
 $63 = sp + 840|0;
 $64 = sp;
 $65 = sp + 836|0;
 $66 = sp + 832|0;
 $67 = sp + 828|0;
 $68 = sp + 824|0;
 $69 = sp + 820|0;
 $70 = sp + 816|0;
 $71 = sp + 808|0;
 $72 = sp + 804|0;
 $73 = sp + 800|0;
 $74 = sp + 792|0;
 $75 = sp + 784|0;
 HEAP32[$54>>2] = $0;
 HEAP32[$55>>2] = $1;
 $76 = HEAP32[$54>>2]|0;
 $77 = ((($76)) + 92|0);
 HEAP32[$51>>2] = $77;
 HEAP32[$52>>2] = $55;
 $78 = HEAP32[$51>>2]|0;
 $79 = HEAP32[$52>>2]|0;
 $80 = (__ZNSt3__26__treeIPKN3sap2ve4core8MaterialENS_4lessIS6_EENS_9allocatorIS6_EEE4findIS6_EENS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEERKT_($78,$79)|0);
 HEAP32[$53>>2] = $80;
 ;HEAP8[$49>>0]=HEAP8[$53>>0]|0;HEAP8[$49+1>>0]=HEAP8[$53+1>>0]|0;HEAP8[$49+2>>0]=HEAP8[$53+2>>0]|0;HEAP8[$49+3>>0]=HEAP8[$53+3>>0]|0;
 HEAP32[$48>>2] = $50;
 $81 = HEAP32[$48>>2]|0;
 $82 = HEAP32[$49>>2]|0;
 HEAP32[$81>>2] = $82;
 $83 = HEAP32[$50>>2]|0;
 HEAP32[$56>>2] = $83;
 $84 = ((($76)) + 92|0);
 HEAP32[$46>>2] = $84;
 $85 = HEAP32[$46>>2]|0;
 HEAP32[$44>>2] = $85;
 $86 = HEAP32[$44>>2]|0;
 HEAP32[$42>>2] = $86;
 $87 = HEAP32[$42>>2]|0;
 $88 = ((($87)) + 4|0);
 HEAP32[$41>>2] = $88;
 $89 = HEAP32[$41>>2]|0;
 HEAP32[$40>>2] = $89;
 $90 = HEAP32[$40>>2]|0;
 HEAP32[$39>>2] = $90;
 $91 = HEAP32[$39>>2]|0;
 HEAP32[$38>>2] = $91;
 $92 = HEAP32[$38>>2]|0;
 HEAP32[$36>>2] = $43;
 HEAP32[$37>>2] = $92;
 $93 = HEAP32[$36>>2]|0;
 $94 = HEAP32[$37>>2]|0;
 HEAP32[$93>>2] = $94;
 $95 = HEAP32[$43>>2]|0;
 HEAP32[$47>>2] = $95;
 ;HEAP8[$35>>0]=HEAP8[$47>>0]|0;HEAP8[$35+1>>0]=HEAP8[$47+1>>0]|0;HEAP8[$35+2>>0]=HEAP8[$47+2>>0]|0;HEAP8[$35+3>>0]=HEAP8[$47+3>>0]|0;
 HEAP32[$34>>2] = $45;
 $96 = HEAP32[$34>>2]|0;
 $97 = HEAP32[$35>>2]|0;
 HEAP32[$96>>2] = $97;
 $98 = HEAP32[$45>>2]|0;
 HEAP32[$57>>2] = $98;
 HEAP32[$19>>2] = $56;
 HEAP32[$20>>2] = $57;
 $99 = HEAP32[$19>>2]|0;
 $100 = HEAP32[$99>>2]|0;
 $101 = HEAP32[$20>>2]|0;
 $102 = HEAP32[$101>>2]|0;
 $103 = ($100|0)==($102|0);
 if (!($103)) {
  $452 = HEAP32[$55>>2]|0;
  STACKTOP = sp;return ($452|0);
 }
 $104 = HEAP32[$55>>2]|0;
 $105 = ($104|0)!=(0|0);
 if ($105) {
  $106 = HEAP32[$55>>2]|0;
  $107 = (__ZNK3sap2ve4core8Material10GetAmbientEv($106)|0);
  $108 = (__ZNK3sap2ve4core6Colour3RedEv($107)|0);
  $109 = +HEAPF32[$108>>2];
  HEAPF32[$58>>2] = $109;
  $110 = ((($58)) + 4|0);
  $111 = HEAP32[$55>>2]|0;
  $112 = (__ZNK3sap2ve4core8Material10GetAmbientEv($111)|0);
  $113 = (__ZNK3sap2ve4core6Colour5GreenEv($112)|0);
  $114 = +HEAPF32[$113>>2];
  HEAPF32[$110>>2] = $114;
  $115 = ((($110)) + 4|0);
  $116 = HEAP32[$55>>2]|0;
  $117 = (__ZNK3sap2ve4core8Material10GetAmbientEv($116)|0);
  $118 = (__ZNK3sap2ve4core6Colour4BlueEv($117)|0);
  $119 = +HEAPF32[$118>>2];
  HEAPF32[$115>>2] = $119;
  $120 = ((($115)) + 4|0);
  $121 = HEAP32[$55>>2]|0;
  $122 = (__ZNK3sap2ve4core8Material10GetAmbientEv($121)|0);
  $123 = (__ZNK3sap2ve4core6Colour5AlphaEv($122)|0);
  $124 = +HEAPF32[$123>>2];
  HEAPF32[$120>>2] = $124;
  $125 = HEAP32[$55>>2]|0;
  $126 = (__ZNK3sap2ve4core8Material10GetDiffuseEv($125)|0);
  $127 = (__ZNK3sap2ve4core6Colour3RedEv($126)|0);
  $128 = +HEAPF32[$127>>2];
  HEAPF32[$59>>2] = $128;
  $129 = ((($59)) + 4|0);
  $130 = HEAP32[$55>>2]|0;
  $131 = (__ZNK3sap2ve4core8Material10GetDiffuseEv($130)|0);
  $132 = (__ZNK3sap2ve4core6Colour5GreenEv($131)|0);
  $133 = +HEAPF32[$132>>2];
  HEAPF32[$129>>2] = $133;
  $134 = ((($129)) + 4|0);
  $135 = HEAP32[$55>>2]|0;
  $136 = (__ZNK3sap2ve4core8Material10GetDiffuseEv($135)|0);
  $137 = (__ZNK3sap2ve4core6Colour4BlueEv($136)|0);
  $138 = +HEAPF32[$137>>2];
  HEAPF32[$134>>2] = $138;
  $139 = ((($134)) + 4|0);
  $140 = HEAP32[$55>>2]|0;
  $141 = (__ZNK3sap2ve4core8Material10GetAmbientEv($140)|0);
  $142 = (__ZNK3sap2ve4core6Colour5AlphaEv($141)|0);
  $143 = +HEAPF32[$142>>2];
  HEAPF32[$139>>2] = $143;
  $144 = HEAP32[$55>>2]|0;
  $145 = (__ZNK3sap2ve4core8Material11GetSpecularEv($144)|0);
  $146 = (__ZNK3sap2ve4core6Colour3RedEv($145)|0);
  $147 = +HEAPF32[$146>>2];
  HEAPF32[$60>>2] = $147;
  $148 = ((($60)) + 4|0);
  $149 = HEAP32[$55>>2]|0;
  $150 = (__ZNK3sap2ve4core8Material11GetSpecularEv($149)|0);
  $151 = (__ZNK3sap2ve4core6Colour5GreenEv($150)|0);
  $152 = +HEAPF32[$151>>2];
  HEAPF32[$148>>2] = $152;
  $153 = ((($148)) + 4|0);
  $154 = HEAP32[$55>>2]|0;
  $155 = (__ZNK3sap2ve4core8Material11GetSpecularEv($154)|0);
  $156 = (__ZNK3sap2ve4core6Colour4BlueEv($155)|0);
  $157 = +HEAPF32[$156>>2];
  HEAPF32[$153>>2] = $157;
  $158 = ((($153)) + 4|0);
  $159 = HEAP32[$55>>2]|0;
  $160 = (__ZNK3sap2ve4core8Material10GetAmbientEv($159)|0);
  $161 = (__ZNK3sap2ve4core6Colour5AlphaEv($160)|0);
  $162 = +HEAPF32[$161>>2];
  HEAPF32[$158>>2] = $162;
  $163 = HEAP32[$55>>2]|0;
  $164 = (__ZNK3sap2ve4core8Material11GetEmissiveEv($163)|0);
  $165 = (__ZNK3sap2ve4core6Colour3RedEv($164)|0);
  $166 = +HEAPF32[$165>>2];
  HEAPF32[$61>>2] = $166;
  $167 = ((($61)) + 4|0);
  $168 = HEAP32[$55>>2]|0;
  $169 = (__ZNK3sap2ve4core8Material11GetEmissiveEv($168)|0);
  $170 = (__ZNK3sap2ve4core6Colour5GreenEv($169)|0);
  $171 = +HEAPF32[$170>>2];
  HEAPF32[$167>>2] = $171;
  $172 = ((($167)) + 4|0);
  $173 = HEAP32[$55>>2]|0;
  $174 = (__ZNK3sap2ve4core8Material11GetEmissiveEv($173)|0);
  $175 = (__ZNK3sap2ve4core6Colour4BlueEv($174)|0);
  $176 = +HEAPF32[$175>>2];
  HEAPF32[$172>>2] = $176;
  $177 = ((($172)) + 4|0);
  $178 = HEAP32[$55>>2]|0;
  $179 = (__ZNK3sap2ve4core8Material10GetAmbientEv($178)|0);
  $180 = (__ZNK3sap2ve4core6Colour5AlphaEv($179)|0);
  $181 = +HEAPF32[$180>>2];
  HEAPF32[$177>>2] = $181;
  $182 = HEAP32[$55>>2]|0;
  $183 = (__ZNK3sap2ve4core8Material13GetLineColourEv($182)|0);
  $184 = (__ZNK3sap2ve4core6Colour3RedEv($183)|0);
  $185 = +HEAPF32[$184>>2];
  HEAPF32[$62>>2] = $185;
  $186 = ((($62)) + 4|0);
  $187 = HEAP32[$55>>2]|0;
  $188 = (__ZNK3sap2ve4core8Material13GetLineColourEv($187)|0);
  $189 = (__ZNK3sap2ve4core6Colour5GreenEv($188)|0);
  $190 = +HEAPF32[$189>>2];
  HEAPF32[$186>>2] = $190;
  $191 = ((($186)) + 4|0);
  $192 = HEAP32[$55>>2]|0;
  $193 = (__ZNK3sap2ve4core8Material13GetLineColourEv($192)|0);
  $194 = (__ZNK3sap2ve4core6Colour4BlueEv($193)|0);
  $195 = +HEAPF32[$194>>2];
  HEAPF32[$191>>2] = $195;
  $196 = ((($191)) + 4|0);
  $197 = HEAP32[$55>>2]|0;
  $198 = (__ZNK3sap2ve4core8Material10GetAmbientEv($197)|0);
  $199 = (__ZNK3sap2ve4core6Colour5AlphaEv($198)|0);
  $200 = +HEAPF32[$199>>2];
  HEAPF32[$196>>2] = $200;
  HEAP32[$63>>2] = 0;
  HEAP32[$65>>2] = 0;
  while(1) {
   $201 = HEAP32[$65>>2]|0;
   $202 = ($201>>>0)<(12);
   if (!($202)) {
    break;
   }
   $203 = HEAP32[$65>>2]|0;
   HEAP32[$66>>2] = $203;
   $204 = HEAP32[$55>>2]|0;
   $205 = HEAP32[$66>>2]|0;
   $206 = (__ZNK3sap2ve4core8Material15GetTextureCountENS1_7Texture4TypeE($204,$205)|0);
   $207 = ($206<<24>>24)!=(0);
   if ($207) {
    $208 = HEAP32[$55>>2]|0;
    $209 = HEAP32[$66>>2]|0;
    $210 = (__ZNK3sap2ve4core8Material10GetTextureENS1_7Texture4TypeEh($208,$209,0)|0);
    HEAP32[$67>>2] = $210;
    $211 = HEAP32[$67>>2]|0;
    $212 = ($211|0)!=(0|0);
    if ($212) {
     $213 = HEAP32[$67>>2]|0;
     $214 = (__ZNK3sap2ve4core7Texture8GetImageEv($213)|0);
     HEAP32[$68>>2] = $214;
     $215 = HEAP32[$68>>2]|0;
     $216 = ($215|0)!=(0|0);
     if ($216) {
      $217 = HEAP32[$68>>2]|0;
      $218 = (__ZNK3sap2ve4core5Image12HasImageDataEv($217)|0);
      if ($218) {
       $219 = HEAP32[$63>>2]|0;
       $220 = (($219) + 1)|0;
       HEAP32[$63>>2] = $220;
       $221 = (($64) + (($219*56)|0)|0);
       HEAP32[$69>>2] = $221;
       $222 = ((($76)) + 80|0);
       $223 = HEAP32[$68>>2]|0;
       HEAP32[$70>>2] = $223;
       HEAP32[$14>>2] = $222;
       HEAP32[$15>>2] = $70;
       $224 = HEAP32[$14>>2]|0;
       $225 = HEAP32[$15>>2]|0;
       HEAP32[$13>>2] = $225;
       $226 = HEAP32[$13>>2]|0;
       HEAP32[$4>>2] = $224;
       HEAP32[$5>>2] = $226;
       $227 = HEAP32[$4>>2]|0;
       $228 = HEAP32[$5>>2]|0;
       HEAP32[$3>>2] = $228;
       $229 = HEAP32[$3>>2]|0;
       $230 = HEAP32[$5>>2]|0;
       HEAP32[$2>>2] = $230;
       $231 = HEAP32[$2>>2]|0;
       __ZNSt3__26__treeIPKN3sap2ve4core5ImageENS_4lessIS6_EENS_9allocatorIS6_EEE25__emplace_unique_key_argsIS6_JS6_EEENS_4pairINS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEEbEERKT_DpOT0_($16,$227,$229,$231);
       HEAP32[$10>>2] = $71;
       HEAP32[$11>>2] = $16;
       $232 = HEAP32[$10>>2]|0;
       $233 = HEAP32[$11>>2]|0;
       HEAP32[$9>>2] = $233;
       $234 = HEAP32[$9>>2]|0;
       ;HEAP32[$12>>2]=HEAP32[$234>>2]|0;
       ;HEAP8[$7>>0]=HEAP8[$12>>0]|0;HEAP8[$7+1>>0]=HEAP8[$12+1>>0]|0;HEAP8[$7+2>>0]=HEAP8[$12+2>>0]|0;HEAP8[$7+3>>0]=HEAP8[$12+3>>0]|0;
       HEAP32[$6>>2] = $232;
       $235 = HEAP32[$6>>2]|0;
       $236 = HEAP32[$7>>2]|0;
       HEAP32[$235>>2] = $236;
       $237 = ((($232)) + 4|0);
       $238 = HEAP32[$11>>2]|0;
       $239 = ((($238)) + 4|0);
       HEAP32[$8>>2] = $239;
       $240 = HEAP32[$8>>2]|0;
       $241 = HEAP8[$240>>0]|0;
       $242 = $241&1;
       $243 = $242&1;
       HEAP8[$237>>0] = $243;
       $244 = HEAP32[$68>>2]|0;
       $245 = HEAP32[$69>>2]|0;
       HEAP32[$245>>2] = $244;
       $246 = HEAP32[$67>>2]|0;
       $247 = (+__ZNK3sap2ve4core7Texture18GetUvRotationAngleEv($246));
       HEAPF32[$17>>2] = $247;
       $248 = +HEAPF32[$17>>2];
       $249 = (+Math_sin((+$248)));
       HEAPF32[$72>>2] = $249;
       $250 = HEAP32[$67>>2]|0;
       $251 = (+__ZNK3sap2ve4core7Texture18GetUvRotationAngleEv($250));
       HEAPF32[$18>>2] = $251;
       $252 = +HEAPF32[$18>>2];
       $253 = (+Math_cos((+$252)));
       HEAPF32[$73>>2] = $253;
       $254 = HEAP32[$67>>2]|0;
       $255 = (+__ZNK3sap2ve4core7Texture20GetUvHorizontalScaleEv($254));
       $256 = +HEAPF32[$73>>2];
       $257 = $255 * $256;
       $258 = HEAP32[$69>>2]|0;
       $259 = ((($258)) + 4|0);
       HEAPF32[$259>>2] = $257;
       $260 = HEAP32[$67>>2]|0;
       $261 = (+__ZNK3sap2ve4core7Texture20GetUvHorizontalScaleEv($260));
       $262 = - $261;
       $263 = +HEAPF32[$72>>2];
       $264 = $262 * $263;
       $265 = HEAP32[$69>>2]|0;
       $266 = ((($265)) + 4|0);
       $267 = ((($266)) + 4|0);
       HEAPF32[$267>>2] = $264;
       $268 = HEAP32[$67>>2]|0;
       $269 = (+__ZNK3sap2ve4core7Texture21GetUvHorizontalOffsetEv($268));
       $270 = HEAP32[$69>>2]|0;
       $271 = ((($270)) + 4|0);
       $272 = +HEAPF32[$271>>2];
       $273 = $269 * $272;
       $274 = HEAP32[$67>>2]|0;
       $275 = (+__ZNK3sap2ve4core7Texture19GetUvVerticalOffsetEv($274));
       $276 = HEAP32[$69>>2]|0;
       $277 = ((($276)) + 4|0);
       $278 = ((($277)) + 4|0);
       $279 = +HEAPF32[$278>>2];
       $280 = $275 * $279;
       $281 = $273 + $280;
       $282 = HEAP32[$69>>2]|0;
       $283 = ((($282)) + 4|0);
       $284 = ((($283)) + 8|0);
       HEAPF32[$284>>2] = $281;
       $285 = HEAP32[$67>>2]|0;
       $286 = (+__ZNK3sap2ve4core7Texture18GetUvVerticalScaleEv($285));
       $287 = - $286;
       $288 = +HEAPF32[$72>>2];
       $289 = $287 * $288;
       $290 = HEAP32[$69>>2]|0;
       $291 = ((($290)) + 4|0);
       $292 = ((($291)) + 12|0);
       HEAPF32[$292>>2] = $289;
       $293 = HEAP32[$67>>2]|0;
       $294 = (+__ZNK3sap2ve4core7Texture18GetUvVerticalScaleEv($293));
       $295 = - $294;
       $296 = +HEAPF32[$73>>2];
       $297 = $295 * $296;
       $298 = HEAP32[$69>>2]|0;
       $299 = ((($298)) + 4|0);
       $300 = ((($299)) + 16|0);
       HEAPF32[$300>>2] = $297;
       $301 = HEAP32[$67>>2]|0;
       $302 = (+__ZNK3sap2ve4core7Texture21GetUvHorizontalOffsetEv($301));
       $303 = HEAP32[$69>>2]|0;
       $304 = ((($303)) + 4|0);
       $305 = ((($304)) + 12|0);
       $306 = +HEAPF32[$305>>2];
       $307 = $302 * $306;
       $308 = $307;
       $309 = 1.0 + $308;
       $310 = HEAP32[$67>>2]|0;
       $311 = (+__ZNK3sap2ve4core7Texture19GetUvVerticalOffsetEv($310));
       $312 = HEAP32[$69>>2]|0;
       $313 = ((($312)) + 4|0);
       $314 = ((($313)) + 16|0);
       $315 = +HEAPF32[$314>>2];
       $316 = $311 * $315;
       $317 = $316;
       $318 = $309 + $317;
       $319 = $318;
       $320 = HEAP32[$69>>2]|0;
       $321 = ((($320)) + 4|0);
       $322 = ((($321)) + 20|0);
       HEAPF32[$322>>2] = $319;
       $323 = HEAP32[$67>>2]|0;
       $324 = (+__ZNK3sap2ve4core7Texture20GetUvHorizontalScaleEv($323));
       $325 = HEAP32[$69>>2]|0;
       $326 = ((($325)) + 28|0);
       HEAPF32[$326>>2] = $324;
       $327 = HEAP32[$67>>2]|0;
       $328 = (+__ZNK3sap2ve4core7Texture18GetUvVerticalScaleEv($327));
       $329 = HEAP32[$69>>2]|0;
       $330 = ((($329)) + 28|0);
       $331 = ((($330)) + 4|0);
       HEAPF32[$331>>2] = $328;
       $332 = HEAP32[$67>>2]|0;
       $333 = (+__ZNK3sap2ve4core7Texture21GetUvHorizontalOffsetEv($332));
       $334 = HEAP32[$69>>2]|0;
       $335 = ((($334)) + 36|0);
       HEAPF32[$335>>2] = $333;
       $336 = HEAP32[$67>>2]|0;
       $337 = (+__ZNK3sap2ve4core7Texture19GetUvVerticalOffsetEv($336));
       $338 = HEAP32[$69>>2]|0;
       $339 = ((($338)) + 36|0);
       $340 = ((($339)) + 4|0);
       HEAPF32[$340>>2] = $337;
       $341 = HEAP32[$67>>2]|0;
       $342 = (+__ZNK3sap2ve4core7Texture18GetUvRotationAngleEv($341));
       $343 = HEAP32[$69>>2]|0;
       $344 = ((($343)) + 44|0);
       HEAPF32[$344>>2] = $342;
       $345 = HEAP32[$67>>2]|0;
       $346 = (+__ZNK3sap2ve4core7Texture12GetInfluenceEv($345));
       $347 = HEAP32[$69>>2]|0;
       $348 = ((($347)) + 48|0);
       HEAPF32[$348>>2] = $346;
       $349 = HEAP32[$66>>2]|0;
       $350 = $349&255;
       $351 = HEAP32[$69>>2]|0;
       $352 = ((($351)) + 52|0);
       HEAP8[$352>>0] = $350;
       $353 = HEAP32[$67>>2]|0;
       $354 = (__ZNK3sap2ve4core7Texture13GetFilterModeEv($353)|0);
       $355 = $354&255;
       $356 = HEAP32[$69>>2]|0;
       $357 = ((($356)) + 53|0);
       HEAP8[$357>>0] = $355;
       $358 = HEAP32[$69>>2]|0;
       $359 = ((($358)) + 54|0);
       HEAP16[$359>>1] = 0;
       $360 = HEAP32[$67>>2]|0;
       $361 = (__ZNK3sap2ve4core7Texture27IsUvHorizontalTilingEnabledEv($360)|0);
       if ($361) {
        $362 = HEAP32[$69>>2]|0;
        $363 = ((($362)) + 54|0);
        $364 = HEAP16[$363>>1]|0;
        $365 = $364&65535;
        $366 = $365 | 1;
        $367 = $366&65535;
        HEAP16[$363>>1] = $367;
       }
       $368 = HEAP32[$67>>2]|0;
       $369 = (__ZNK3sap2ve4core7Texture25IsUvVerticalTilingEnabledEv($368)|0);
       if ($369) {
        $370 = HEAP32[$69>>2]|0;
        $371 = ((($370)) + 54|0);
        $372 = HEAP16[$371>>1]|0;
        $373 = $372&65535;
        $374 = $373 | 2;
        $375 = $374&65535;
        HEAP16[$371>>1] = $375;
       }
       $376 = HEAP32[$67>>2]|0;
       $377 = (__ZNK3sap2ve4core7Texture25IsUvClampToBordersEnabledEv($376)|0);
       if ($377) {
        $378 = HEAP32[$69>>2]|0;
        $379 = ((($378)) + 54|0);
        $380 = HEAP16[$379>>1]|0;
        $381 = $380&65535;
        $382 = $381 | 4;
        $383 = $382&65535;
        HEAP16[$379>>1] = $383;
       }
       $384 = HEAP32[$67>>2]|0;
       $385 = (__ZNK3sap2ve4core7Texture10IsInvertedEv($384)|0);
       if ($385) {
        $386 = HEAP32[$69>>2]|0;
        $387 = ((($386)) + 54|0);
        $388 = HEAP16[$387>>1]|0;
        $389 = $388&65535;
        $390 = $389 | 8;
        $391 = $390&65535;
        HEAP16[$387>>1] = $391;
       }
       $392 = HEAP32[$67>>2]|0;
       $393 = (__ZNK3sap2ve4core7Texture10IsModulateEv($392)|0);
       if ($393) {
        $394 = HEAP32[$69>>2]|0;
        $395 = ((($394)) + 54|0);
        $396 = HEAP16[$395>>1]|0;
        $397 = $396&65535;
        $398 = $397 | 16;
        $399 = $398&65535;
        HEAP16[$395>>1] = $399;
       }
      }
     }
    }
   }
   $400 = HEAP32[$65>>2]|0;
   $401 = (($400) + 1)|0;
   HEAP32[$65>>2] = $401;
  }
  $402 = ((($76)) + 24|0);
  $403 = HEAP32[$402>>2]|0;
  $404 = HEAP32[$55>>2]|0;
  __ZNK3sap2ve4core8Material7GetNameEv($74,$404);
  $405 = (__ZNK3sap2ve6String4Utf8Ev($74)|0);
  $406 = HEAP32[$55>>2]|0;
  $407 = (+__ZNK3sap2ve4core8Material10GetOpacityEv($406));
  $408 = HEAP32[$55>>2]|0;
  $409 = (+__ZNK3sap2ve4core8Material13GetGlossinessEv($408));
  $410 = HEAP32[$55>>2]|0;
  $411 = (+__ZNK3sap2ve4core8Material16GetSpecularLevelEv($410));
  $412 = HEAP32[$55>>2]|0;
  $413 = (__ZNK3sap2ve4core8Material18IsColourMapEnabledEv($412)|0);
  $414 = HEAP32[$55>>2]|0;
  $415 = (+__ZNK3sap2ve4core8Material12GetLineWidthEv($414));
  $416 = HEAP32[$55>>2]|0;
  $417 = (+__ZNK3sap2ve4core8Material16GetLineHaloWidthEv($416));
  $418 = HEAP32[$55>>2]|0;
  $419 = (__ZNK3sap2ve4core8Material18GetLineEndCapStyleEv($418)|0);
  $420 = HEAP32[$55>>2]|0;
  $421 = (+__ZNK3sap2ve4core8Material23GetLineDashPatternScaleEv($420));
  $422 = HEAP32[$55>>2]|0;
  $423 = (__ZNK3sap2ve4core8Material26GetLineDashPatternSegmentsEv($422)|0);
  $424 = HEAP32[$55>>2]|0;
  $425 = (__ZNK3sap2ve4core8Material30GetLineDashPatternSegmentCountEv($424)|0);
  $426 = $425&255;
  $427 = HEAP32[$55>>2]|0;
  $428 = (__ZNK3sap2ve4core8Material27GetLineWidthCoordinateSpaceEv($427)|0);
  $429 = HEAP32[$63>>2]|0;
  $430 = HEAP32[$55>>2]|0;
  _SceneBuilderAdapter_createMaterial(($403|0),($405|0),($58|0),($59|0),($60|0),($61|0),(+$407),(+$409),(+$411),($413|0),(+$415),(+$417),($62|0),($419|0),(+$421),($423|0),($426|0),($428|0),($429|0),($64|0),($430|0));
  __ZN3sap2ve6StringD2Ev($74);
 } else {
  $431 = ((($76)) + 24|0);
  $432 = HEAP32[$431>>2]|0;
  _SceneBuilderAdapter_createDefaultMaterial(($432|0));
 }
 $433 = ((($76)) + 92|0);
 HEAP32[$31>>2] = $433;
 HEAP32[$32>>2] = $55;
 $434 = HEAP32[$31>>2]|0;
 $435 = HEAP32[$32>>2]|0;
 HEAP32[$29>>2] = $434;
 HEAP32[$30>>2] = $435;
 $436 = HEAP32[$29>>2]|0;
 $437 = HEAP32[$30>>2]|0;
 HEAP32[$28>>2] = $437;
 $438 = HEAP32[$28>>2]|0;
 $439 = HEAP32[$30>>2]|0;
 __ZNSt3__26__treeIPKN3sap2ve4core8MaterialENS_4lessIS6_EENS_9allocatorIS6_EEE25__emplace_unique_key_argsIS6_JRKS6_EEENS_4pairINS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEEbEERKT_DpOT0_($33,$436,$438,$439);
 HEAP32[$25>>2] = $75;
 HEAP32[$26>>2] = $33;
 $440 = HEAP32[$25>>2]|0;
 $441 = HEAP32[$26>>2]|0;
 HEAP32[$24>>2] = $441;
 $442 = HEAP32[$24>>2]|0;
 ;HEAP32[$27>>2]=HEAP32[$442>>2]|0;
 ;HEAP8[$22>>0]=HEAP8[$27>>0]|0;HEAP8[$22+1>>0]=HEAP8[$27+1>>0]|0;HEAP8[$22+2>>0]=HEAP8[$27+2>>0]|0;HEAP8[$22+3>>0]=HEAP8[$27+3>>0]|0;
 HEAP32[$21>>2] = $440;
 $443 = HEAP32[$21>>2]|0;
 $444 = HEAP32[$22>>2]|0;
 HEAP32[$443>>2] = $444;
 $445 = ((($440)) + 4|0);
 $446 = HEAP32[$26>>2]|0;
 $447 = ((($446)) + 4|0);
 HEAP32[$23>>2] = $447;
 $448 = HEAP32[$23>>2]|0;
 $449 = HEAP8[$448>>0]|0;
 $450 = $449&1;
 $451 = $450&1;
 HEAP8[$445>>0] = $451;
 $452 = HEAP32[$55>>2]|0;
 STACKTOP = sp;return ($452|0);
}
function __Z17CombineAppearanceRNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEEEPKNS3_10AppearanceE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0;
 var $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0;
 var $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0;
 var $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0;
 var $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0;
 var $280 = 0, $281 = 0, $282 = 0, $283 = 0.0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0;
 var $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0;
 var $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0;
 var $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0;
 var $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0;
 var $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0;
 var $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0;
 var $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0;
 var $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0;
 var $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0;
 var $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0;
 var $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0;
 var $497 = 0, $498 = 0, $499 = 0, $5 = 0, $50 = 0, $500 = 0, $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0;
 var $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0;
 var $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0;
 var $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 1008|0;
 $2 = sp + 996|0;
 $3 = sp + 992|0;
 $4 = sp + 988|0;
 $5 = sp + 984|0;
 $6 = sp + 980|0;
 $7 = sp + 976|0;
 $8 = sp + 972|0;
 $9 = sp + 968|0;
 $10 = sp + 964|0;
 $11 = sp + 960|0;
 $12 = sp + 956|0;
 $13 = sp + 952|0;
 $14 = sp + 948|0;
 $15 = sp + 944|0;
 $16 = sp + 64|0;
 $17 = sp + 940|0;
 $18 = sp + 936|0;
 $19 = sp + 932|0;
 $20 = sp + 928|0;
 $21 = sp + 924|0;
 $22 = sp + 920|0;
 $23 = sp + 916|0;
 $24 = sp + 912|0;
 $25 = sp + 56|0;
 $26 = sp + 908|0;
 $27 = sp + 904|0;
 $28 = sp + 900|0;
 $29 = sp + 896|0;
 $30 = sp + 892|0;
 $31 = sp + 888|0;
 $32 = sp + 884|0;
 $33 = sp + 880|0;
 $34 = sp + 876|0;
 $35 = sp + 872|0;
 $36 = sp + 868|0;
 $37 = sp + 864|0;
 $38 = sp + 860|0;
 $39 = sp + 856|0;
 $40 = sp + 852|0;
 $41 = sp + 848|0;
 $42 = sp + 844|0;
 $43 = sp + 840|0;
 $44 = sp + 836|0;
 $45 = sp + 832|0;
 $46 = sp + 828|0;
 $47 = sp + 824|0;
 $48 = sp + 820|0;
 $49 = sp + 816|0;
 $50 = sp + 812|0;
 $51 = sp + 808|0;
 $52 = sp + 804|0;
 $53 = sp + 800|0;
 $54 = sp + 796|0;
 $55 = sp + 792|0;
 $56 = sp + 788|0;
 $57 = sp + 48|0;
 $58 = sp + 784|0;
 $59 = sp + 780|0;
 $60 = sp + 1001|0;
 $61 = sp + 776|0;
 $62 = sp + 772|0;
 $63 = sp + 768|0;
 $64 = sp + 764|0;
 $65 = sp + 760|0;
 $66 = sp + 756|0;
 $67 = sp + 752|0;
 $68 = sp + 748|0;
 $69 = sp + 744|0;
 $70 = sp + 740|0;
 $71 = sp + 736|0;
 $72 = sp + 732|0;
 $73 = sp + 728|0;
 $74 = sp + 716|0;
 $75 = sp + 712|0;
 $76 = sp + 708|0;
 $77 = sp + 704|0;
 $78 = sp + 40|0;
 $79 = sp + 700|0;
 $80 = sp + 696|0;
 $81 = sp + 692|0;
 $82 = sp + 1000|0;
 $83 = sp + 688|0;
 $84 = sp + 32|0;
 $85 = sp + 684|0;
 $86 = sp + 680|0;
 $87 = sp + 676|0;
 $88 = sp + 672|0;
 $89 = sp + 668|0;
 $90 = sp + 664|0;
 $91 = sp + 660|0;
 $92 = sp + 656|0;
 $93 = sp + 648|0;
 $94 = sp + 640|0;
 $95 = sp + 24|0;
 $96 = sp + 636|0;
 $97 = sp + 632|0;
 $98 = sp + 628|0;
 $99 = sp + 624|0;
 $100 = sp + 620|0;
 $101 = sp + 16|0;
 $102 = sp + 616|0;
 $103 = sp + 612|0;
 $104 = sp + 608|0;
 $105 = sp + 604|0;
 $106 = sp + 600|0;
 $107 = sp + 596|0;
 $108 = sp + 592|0;
 $109 = sp + 588|0;
 $110 = sp + 584|0;
 $111 = sp + 580|0;
 $112 = sp + 576|0;
 $113 = sp + 572|0;
 $114 = sp + 568|0;
 $115 = sp + 564|0;
 $116 = sp + 560|0;
 $117 = sp + 556|0;
 $118 = sp + 552|0;
 $119 = sp + 548|0;
 $120 = sp + 544|0;
 $121 = sp + 540|0;
 $122 = sp + 536|0;
 $123 = sp + 532|0;
 $124 = sp + 528|0;
 $125 = sp + 524|0;
 $126 = sp + 520|0;
 $127 = sp + 516|0;
 $128 = sp + 512|0;
 $129 = sp + 508|0;
 $130 = sp + 504|0;
 $131 = sp + 500|0;
 $132 = sp + 496|0;
 $133 = sp + 492|0;
 $134 = sp + 488|0;
 $135 = sp + 484|0;
 $136 = sp + 8|0;
 $137 = sp + 480|0;
 $138 = sp + 476|0;
 $139 = sp + 472|0;
 $140 = sp + 468|0;
 $141 = sp + 464|0;
 $142 = sp + 460|0;
 $143 = sp + 456|0;
 $144 = sp + 452|0;
 $145 = sp + 448|0;
 $146 = sp + 444|0;
 $147 = sp;
 $148 = sp + 440|0;
 $149 = sp + 436|0;
 $150 = sp + 432|0;
 $151 = sp + 428|0;
 $152 = sp + 424|0;
 $153 = sp + 420|0;
 $154 = sp + 416|0;
 $155 = sp + 412|0;
 $156 = sp + 408|0;
 $157 = sp + 404|0;
 $158 = sp + 400|0;
 $159 = sp + 396|0;
 $160 = sp + 392|0;
 $161 = sp + 388|0;
 $162 = sp + 384|0;
 $163 = sp + 380|0;
 $164 = sp + 376|0;
 $165 = sp + 372|0;
 $166 = sp + 368|0;
 $167 = sp + 364|0;
 $168 = sp + 360|0;
 $169 = sp + 356|0;
 $170 = sp + 352|0;
 $171 = sp + 348|0;
 $172 = sp + 344|0;
 $173 = sp + 340|0;
 $174 = sp + 336|0;
 $175 = sp + 332|0;
 $176 = sp + 328|0;
 $177 = sp + 324|0;
 $178 = sp + 320|0;
 $179 = sp + 316|0;
 $180 = sp + 312|0;
 $181 = sp + 308|0;
 $182 = sp + 304|0;
 $183 = sp + 300|0;
 $184 = sp + 296|0;
 $185 = sp + 292|0;
 $186 = sp + 288|0;
 $187 = sp + 284|0;
 $188 = sp + 280|0;
 $189 = sp + 276|0;
 $190 = sp + 272|0;
 $191 = sp + 268|0;
 $192 = sp + 264|0;
 $193 = sp + 260|0;
 $194 = sp + 256|0;
 $195 = sp + 252|0;
 $196 = sp + 248|0;
 $197 = sp + 244|0;
 $198 = sp + 240|0;
 $199 = sp + 236|0;
 $200 = sp + 232|0;
 $201 = sp + 228|0;
 $202 = sp + 224|0;
 $203 = sp + 220|0;
 $204 = sp + 216|0;
 $205 = sp + 212|0;
 $206 = sp + 208|0;
 $207 = sp + 204|0;
 $208 = sp + 200|0;
 $209 = sp + 196|0;
 $210 = sp + 192|0;
 $211 = sp + 188|0;
 $212 = sp + 184|0;
 $213 = sp + 180|0;
 $214 = sp + 176|0;
 $215 = sp + 172|0;
 $216 = sp + 168|0;
 $217 = sp + 148|0;
 $218 = sp + 144|0;
 $219 = sp + 140|0;
 $220 = sp + 136|0;
 $221 = sp + 128|0;
 $222 = sp + 124|0;
 $223 = sp + 120|0;
 $224 = sp + 112|0;
 $225 = sp + 104|0;
 $226 = sp + 100|0;
 $227 = sp + 96|0;
 $228 = sp + 92|0;
 $229 = sp + 88|0;
 $230 = sp + 84|0;
 $231 = sp + 80|0;
 $232 = sp + 76|0;
 $233 = sp + 72|0;
 $234 = sp + 68|0;
 HEAP32[$214>>2] = $0;
 HEAP32[$215>>2] = $1;
 $235 = HEAP32[$215>>2]|0;
 $236 = ($235|0)!=(0|0);
 if (!($236)) {
  STACKTOP = sp;return;
 }
 $237 = HEAP32[$215>>2]|0;
 $238 = (__ZNK3sap2ve4core10Appearance19GetOverrideMaterialEv($237)|0);
 HEAP32[$216>>2] = $238;
 $239 = HEAP32[$216>>2]|0;
 $240 = ($239|0)!=(0|0);
 if ($240) {
  $462 = HEAP32[$214>>2]|0;
  HEAP32[$231>>2] = $462;
  $463 = HEAP32[$231>>2]|0;
  HEAP32[$144>>2] = $463;
  $464 = HEAP32[$144>>2]|0;
  HEAP32[$142>>2] = $464;
  $465 = HEAP32[$142>>2]|0;
  $466 = ((($465)) + 8|0);
  HEAP32[$140>>2] = $466;
  $467 = HEAP32[$140>>2]|0;
  HEAP32[$139>>2] = $467;
  $468 = HEAP32[$139>>2]|0;
  $469 = HEAP32[$468>>2]|0;
  HEAP32[$137>>2] = $141;
  HEAP32[$138>>2] = $469;
  $470 = HEAP32[$137>>2]|0;
  $471 = HEAP32[$138>>2]|0;
  HEAP32[$470>>2] = $471;
  $472 = HEAP32[$141>>2]|0;
  HEAP32[$145>>2] = $472;
  ;HEAP8[$136>>0]=HEAP8[$145>>0]|0;HEAP8[$136+1>>0]=HEAP8[$145+1>>0]|0;HEAP8[$136+2>>0]=HEAP8[$145+2>>0]|0;HEAP8[$136+3>>0]=HEAP8[$145+3>>0]|0;
  HEAP32[$135>>2] = $143;
  $473 = HEAP32[$135>>2]|0;
  ;HEAP32[$473>>2]=HEAP32[$136>>2]|0;
  $474 = HEAP32[$143>>2]|0;
  HEAP32[$232>>2] = $474;
  $475 = HEAP32[$231>>2]|0;
  HEAP32[$153>>2] = $475;
  $476 = HEAP32[$153>>2]|0;
  HEAP32[$151>>2] = $476;
  HEAP32[$148>>2] = $150;
  HEAP32[$149>>2] = 0;
  $477 = HEAP32[$148>>2]|0;
  $478 = HEAP32[$149>>2]|0;
  HEAP32[$477>>2] = $478;
  $479 = HEAP32[$150>>2]|0;
  HEAP32[$154>>2] = $479;
  ;HEAP8[$147>>0]=HEAP8[$154>>0]|0;HEAP8[$147+1>>0]=HEAP8[$154+1>>0]|0;HEAP8[$147+2>>0]=HEAP8[$154+2>>0]|0;HEAP8[$147+3>>0]=HEAP8[$154+3>>0]|0;
  HEAP32[$146>>2] = $152;
  $480 = HEAP32[$146>>2]|0;
  ;HEAP32[$480>>2]=HEAP32[$147>>2]|0;
  $481 = HEAP32[$152>>2]|0;
  HEAP32[$233>>2] = $481;
  while(1) {
   HEAP32[$159>>2] = $232;
   HEAP32[$160>>2] = $233;
   $482 = HEAP32[$159>>2]|0;
   $483 = HEAP32[$160>>2]|0;
   HEAP32[$157>>2] = $482;
   HEAP32[$158>>2] = $483;
   $484 = HEAP32[$157>>2]|0;
   $485 = HEAP32[$158>>2]|0;
   HEAP32[$155>>2] = $484;
   HEAP32[$156>>2] = $485;
   $486 = HEAP32[$155>>2]|0;
   $487 = HEAP32[$486>>2]|0;
   $488 = HEAP32[$156>>2]|0;
   $489 = HEAP32[$488>>2]|0;
   $490 = ($487|0)==($489|0);
   $491 = $490 ^ 1;
   if (!($491)) {
    break;
   }
   HEAP32[$167>>2] = $232;
   $492 = HEAP32[$167>>2]|0;
   HEAP32[$166>>2] = $492;
   $493 = HEAP32[$166>>2]|0;
   $494 = HEAP32[$493>>2]|0;
   HEAP32[$165>>2] = $494;
   $495 = HEAP32[$165>>2]|0;
   HEAP32[$164>>2] = $495;
   $496 = HEAP32[$164>>2]|0;
   HEAP32[$163>>2] = $496;
   $497 = HEAP32[$163>>2]|0;
   $498 = ((($497)) + 8|0);
   HEAP32[$162>>2] = $498;
   $499 = HEAP32[$162>>2]|0;
   HEAP32[$161>>2] = $499;
   $500 = HEAP32[$161>>2]|0;
   HEAP32[$234>>2] = $500;
   $501 = HEAP32[$216>>2]|0;
   $502 = HEAP32[$234>>2]|0;
   $503 = ((($502)) + 4|0);
   HEAP32[$503>>2] = $501;
   HEAP32[$169>>2] = $232;
   $504 = HEAP32[$169>>2]|0;
   HEAP32[$168>>2] = $504;
   $505 = HEAP32[$168>>2]|0;
   $506 = HEAP32[$505>>2]|0;
   $507 = HEAP32[$506>>2]|0;
   HEAP32[$505>>2] = $507;
  }
  STACKTOP = sp;return;
 }
 HEAP32[$213>>2] = $217;
 $241 = HEAP32[$213>>2]|0;
 HEAP32[$210>>2] = $241;
 $242 = HEAP32[$210>>2]|0;
 HEAP32[$208>>2] = $242;
 $243 = HEAP32[$208>>2]|0;
 HEAP32[$209>>2] = 0;
 HEAP32[$206>>2] = $243;
 HEAP32[$207>>2] = $209;
 $244 = HEAP32[$206>>2]|0;
 $245 = HEAP32[$207>>2]|0;
 HEAP32[$205>>2] = $245;
 $246 = HEAP32[$205>>2]|0;
 HEAP32[$192>>2] = $244;
 HEAP32[$193>>2] = $246;
 $247 = HEAP32[$192>>2]|0;
 $248 = HEAP32[$193>>2]|0;
 HEAP32[$191>>2] = $248;
 $249 = HEAP32[$191>>2]|0;
 $250 = HEAP32[$249>>2]|0;
 HEAP32[$247>>2] = $250;
 $251 = ((($244)) + 4|0);
 HEAP32[$204>>2] = $251;
 $252 = HEAP32[$204>>2]|0;
 HEAP32[$202>>2] = $252;
 $253 = HEAP32[$202>>2]|0;
 HEAP32[$203>>2] = 0;
 HEAP32[$200>>2] = $253;
 HEAP32[$201>>2] = $203;
 $254 = HEAP32[$200>>2]|0;
 $255 = HEAP32[$201>>2]|0;
 HEAP32[$199>>2] = $255;
 $256 = HEAP32[$199>>2]|0;
 HEAP32[$195>>2] = $254;
 HEAP32[$196>>2] = $256;
 $257 = HEAP32[$195>>2]|0;
 $258 = HEAP32[$196>>2]|0;
 HEAP32[$194>>2] = $258;
 $259 = HEAP32[$194>>2]|0;
 $260 = HEAP32[$259>>2]|0;
 HEAP32[$257>>2] = $260;
 HEAP32[$198>>2] = $254;
 $261 = HEAP32[$198>>2]|0;
 HEAP32[$197>>2] = $261;
 $262 = ((($242)) + 8|0);
 HEAP32[$190>>2] = $262;
 $263 = HEAP32[$190>>2]|0;
 HEAP32[$189>>2] = $263;
 $264 = HEAP32[$189>>2]|0;
 HEAP32[$188>>2] = $264;
 $265 = HEAP32[$188>>2]|0;
 HEAP32[$265>>2] = 0;
 HEAP32[$187>>2] = $263;
 $266 = HEAP32[$187>>2]|0;
 HEAP32[$186>>2] = $266;
 $267 = ((($242)) + 12|0);
 HEAP32[$211>>2] = 0;
 HEAP32[$184>>2] = $267;
 HEAP32[$185>>2] = $211;
 $268 = HEAP32[$184>>2]|0;
 $269 = HEAP32[$185>>2]|0;
 HEAP32[$183>>2] = $269;
 $270 = HEAP32[$183>>2]|0;
 HEAP32[$179>>2] = $268;
 HEAP32[$180>>2] = $270;
 $271 = HEAP32[$179>>2]|0;
 $272 = HEAP32[$180>>2]|0;
 HEAP32[$178>>2] = $272;
 $273 = HEAP32[$178>>2]|0;
 $274 = HEAP32[$273>>2]|0;
 HEAP32[$271>>2] = $274;
 HEAP32[$182>>2] = $268;
 $275 = HEAP32[$182>>2]|0;
 HEAP32[$181>>2] = $275;
 $276 = ((($242)) + 16|0);
 HEAPF32[$212>>2] = 1.0;
 HEAP32[$176>>2] = $276;
 HEAP32[$177>>2] = $212;
 $277 = HEAP32[$176>>2]|0;
 $278 = HEAP32[$177>>2]|0;
 HEAP32[$175>>2] = $278;
 $279 = HEAP32[$175>>2]|0;
 HEAP32[$171>>2] = $277;
 HEAP32[$172>>2] = $279;
 $280 = HEAP32[$171>>2]|0;
 $281 = HEAP32[$172>>2]|0;
 HEAP32[$170>>2] = $281;
 $282 = HEAP32[$170>>2]|0;
 $283 = +HEAPF32[$282>>2];
 HEAPF32[$280>>2] = $283;
 HEAP32[$174>>2] = $277;
 $284 = HEAP32[$174>>2]|0;
 HEAP32[$173>>2] = $284;
 $285 = HEAP32[$215>>2]|0;
 $286 = (__ZNK3sap2ve4core10Appearance15GetMappingCountEv($285)|0);
 HEAP32[$218>>2] = $286;
 HEAP32[$219>>2] = 0;
 while(1) {
  $287 = HEAP32[$219>>2]|0;
  $288 = HEAP32[$218>>2]|0;
  $289 = ($287>>>0)<($288>>>0);
  if (!($289)) {
   break;
  }
  $290 = HEAP32[$215>>2]|0;
  $291 = HEAP32[$219>>2]|0;
  $292 = (__ZNK3sap2ve4core10Appearance10GetMappingEj($290,$291)|0);
  HEAP32[$220>>2] = $292;
  $293 = HEAP32[$220>>2]|0;
  $294 = (__ZNK3sap2ve4core15MaterialMapping17GetSourceMaterialEv($293)|0);
  HEAP32[$222>>2] = $294;
  $295 = HEAP32[$220>>2]|0;
  $296 = (__ZNK3sap2ve4core15MaterialMapping17GetTargetMaterialEv($295)|0);
  HEAP32[$223>>2] = $296;
  HEAP32[$133>>2] = $222;
  HEAP32[$134>>2] = $223;
  $297 = HEAP32[$133>>2]|0;
  HEAP32[$132>>2] = $297;
  $298 = HEAP32[$132>>2]|0;
  $299 = HEAP32[$134>>2]|0;
  HEAP32[$126>>2] = $299;
  $300 = HEAP32[$126>>2]|0;
  HEAP32[$129>>2] = $221;
  HEAP32[$130>>2] = $298;
  HEAP32[$131>>2] = $300;
  $301 = HEAP32[$129>>2]|0;
  $302 = HEAP32[$130>>2]|0;
  HEAP32[$128>>2] = $302;
  $303 = HEAP32[$128>>2]|0;
  $304 = HEAP32[$303>>2]|0;
  HEAP32[$301>>2] = $304;
  $305 = ((($301)) + 4|0);
  $306 = HEAP32[$131>>2]|0;
  HEAP32[$127>>2] = $306;
  $307 = HEAP32[$127>>2]|0;
  $308 = HEAP32[$307>>2]|0;
  HEAP32[$305>>2] = $308;
  HEAP32[$91>>2] = $217;
  HEAP32[$92>>2] = $221;
  $309 = HEAP32[$91>>2]|0;
  $310 = HEAP32[$92>>2]|0;
  HEAP32[$90>>2] = $310;
  $311 = HEAP32[$90>>2]|0;
  HEAP32[$80>>2] = $309;
  HEAP32[$81>>2] = $311;
  $312 = HEAP32[$80>>2]|0;
  $313 = HEAP32[$81>>2]|0;
  HEAP32[$79>>2] = $313;
  $314 = HEAP32[$79>>2]|0;
  ;HEAP8[$78>>0]=HEAP8[$82>>0]|0;
  HEAP32[$76>>2] = $312;
  HEAP32[$77>>2] = $314;
  $315 = HEAP32[$76>>2]|0;
  $316 = HEAP32[$77>>2]|0;
  HEAP32[$75>>2] = $316;
  $317 = HEAP32[$75>>2]|0;
  HEAP32[$72>>2] = $315;
  HEAP32[$73>>2] = $317;
  $318 = HEAP32[$72>>2]|0;
  $319 = HEAP32[$73>>2]|0;
  HEAP32[$71>>2] = $319;
  $320 = HEAP32[$71>>2]|0;
  __ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEE16__construct_nodeIJNS_4pairIPS5_SM_EEEEENS_10unique_ptrINS_11__hash_nodeIS8_PvEENS_22__hash_node_destructorINSH_ISR_EEEEEEDpOT_($74,$318,$320);
  HEAP32[$43>>2] = $74;
  $321 = HEAP32[$43>>2]|0;
  HEAP32[$42>>2] = $321;
  $322 = HEAP32[$42>>2]|0;
  HEAP32[$41>>2] = $322;
  $323 = HEAP32[$41>>2]|0;
  $324 = HEAP32[$323>>2]|0;
  __ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEE20__node_insert_uniqueEPNS_11__hash_nodeIS8_PvEE($93,$318,$324);
  $325 = ((($93)) + 4|0);
  $326 = HEAP8[$325>>0]|0;
  $327 = $326&1;
  if ($327) {
   HEAP32[$39>>2] = $74;
   $328 = HEAP32[$39>>2]|0;
   HEAP32[$38>>2] = $328;
   $329 = HEAP32[$38>>2]|0;
   HEAP32[$37>>2] = $329;
   $330 = HEAP32[$37>>2]|0;
   $331 = HEAP32[$330>>2]|0;
   HEAP32[$40>>2] = $331;
   HEAP32[$36>>2] = $328;
   $332 = HEAP32[$36>>2]|0;
   HEAP32[$35>>2] = $332;
   $333 = HEAP32[$35>>2]|0;
   HEAP32[$333>>2] = 0;
  }
  HEAP32[$70>>2] = $74;
  $334 = HEAP32[$70>>2]|0;
  HEAP32[$67>>2] = $334;
  HEAP32[$68>>2] = 0;
  $335 = HEAP32[$67>>2]|0;
  HEAP32[$66>>2] = $335;
  $336 = HEAP32[$66>>2]|0;
  HEAP32[$65>>2] = $336;
  $337 = HEAP32[$65>>2]|0;
  $338 = HEAP32[$337>>2]|0;
  HEAP32[$69>>2] = $338;
  $339 = HEAP32[$68>>2]|0;
  HEAP32[$47>>2] = $335;
  $340 = HEAP32[$47>>2]|0;
  HEAP32[$46>>2] = $340;
  $341 = HEAP32[$46>>2]|0;
  HEAP32[$341>>2] = $339;
  $342 = HEAP32[$69>>2]|0;
  $343 = ($342|0)!=(0|0);
  if ($343) {
   HEAP32[$45>>2] = $335;
   $344 = HEAP32[$45>>2]|0;
   $345 = ((($344)) + 4|0);
   HEAP32[$44>>2] = $345;
   $346 = HEAP32[$44>>2]|0;
   $347 = HEAP32[$69>>2]|0;
   HEAP32[$63>>2] = $346;
   HEAP32[$64>>2] = $347;
   $348 = HEAP32[$63>>2]|0;
   $349 = ((($348)) + 4|0);
   $350 = HEAP8[$349>>0]|0;
   $351 = $350&1;
   if ($351) {
    $352 = HEAP32[$348>>2]|0;
    $353 = HEAP32[$64>>2]|0;
    $354 = ((($353)) + 8|0);
    HEAP32[$62>>2] = $354;
    $355 = HEAP32[$62>>2]|0;
    HEAP32[$61>>2] = $355;
    $356 = HEAP32[$61>>2]|0;
    HEAP32[$58>>2] = $352;
    HEAP32[$59>>2] = $356;
    $357 = HEAP32[$58>>2]|0;
    $358 = HEAP32[$59>>2]|0;
    ;HEAP8[$57>>0]=HEAP8[$60>>0]|0;
    HEAP32[$55>>2] = $357;
    HEAP32[$56>>2] = $358;
   }
   $359 = HEAP32[$64>>2]|0;
   $360 = ($359|0)!=(0|0);
   if ($360) {
    $361 = HEAP32[$348>>2]|0;
    $362 = HEAP32[$64>>2]|0;
    HEAP32[$52>>2] = $361;
    HEAP32[$53>>2] = $362;
    HEAP32[$54>>2] = 1;
    $363 = HEAP32[$52>>2]|0;
    $364 = HEAP32[$53>>2]|0;
    $365 = HEAP32[$54>>2]|0;
    HEAP32[$49>>2] = $363;
    HEAP32[$50>>2] = $364;
    HEAP32[$51>>2] = $365;
    $366 = HEAP32[$50>>2]|0;
    HEAP32[$48>>2] = $366;
    $367 = HEAP32[$48>>2]|0;
    __ZdlPv($367);
   }
  }
  HEAP32[$87>>2] = $224;
  HEAP32[$88>>2] = $93;
  $368 = HEAP32[$87>>2]|0;
  $369 = HEAP32[$88>>2]|0;
  HEAP32[$86>>2] = $369;
  $370 = HEAP32[$86>>2]|0;
  ;HEAP32[$89>>2]=HEAP32[$370>>2]|0;
  ;HEAP8[$84>>0]=HEAP8[$89>>0]|0;HEAP8[$84+1>>0]=HEAP8[$89+1>>0]|0;HEAP8[$84+2>>0]=HEAP8[$89+2>>0]|0;HEAP8[$84+3>>0]=HEAP8[$89+3>>0]|0;
  HEAP32[$83>>2] = $368;
  $371 = HEAP32[$83>>2]|0;
  ;HEAP32[$371>>2]=HEAP32[$84>>2]|0;
  $372 = ((($368)) + 4|0);
  $373 = HEAP32[$88>>2]|0;
  $374 = ((($373)) + 4|0);
  HEAP32[$85>>2] = $374;
  $375 = HEAP32[$85>>2]|0;
  $376 = HEAP8[$375>>0]|0;
  $377 = $376&1;
  $378 = $377&1;
  HEAP8[$372>>0] = $378;
  $379 = HEAP32[$219>>2]|0;
  $380 = (($379) + 1)|0;
  HEAP32[$219>>2] = $380;
 }
 $381 = HEAP32[$214>>2]|0;
 HEAP32[$225>>2] = $381;
 $382 = HEAP32[$225>>2]|0;
 HEAP32[$33>>2] = $382;
 $383 = HEAP32[$33>>2]|0;
 HEAP32[$31>>2] = $383;
 $384 = HEAP32[$31>>2]|0;
 $385 = ((($384)) + 8|0);
 HEAP32[$29>>2] = $385;
 $386 = HEAP32[$29>>2]|0;
 HEAP32[$28>>2] = $386;
 $387 = HEAP32[$28>>2]|0;
 $388 = HEAP32[$387>>2]|0;
 HEAP32[$26>>2] = $30;
 HEAP32[$27>>2] = $388;
 $389 = HEAP32[$26>>2]|0;
 $390 = HEAP32[$27>>2]|0;
 HEAP32[$389>>2] = $390;
 $391 = HEAP32[$30>>2]|0;
 HEAP32[$34>>2] = $391;
 ;HEAP8[$25>>0]=HEAP8[$34>>0]|0;HEAP8[$25+1>>0]=HEAP8[$34+1>>0]|0;HEAP8[$25+2>>0]=HEAP8[$34+2>>0]|0;HEAP8[$25+3>>0]=HEAP8[$34+3>>0]|0;
 HEAP32[$24>>2] = $32;
 $392 = HEAP32[$24>>2]|0;
 ;HEAP32[$392>>2]=HEAP32[$25>>2]|0;
 $393 = HEAP32[$32>>2]|0;
 HEAP32[$226>>2] = $393;
 $394 = HEAP32[$225>>2]|0;
 HEAP32[$22>>2] = $394;
 $395 = HEAP32[$22>>2]|0;
 HEAP32[$20>>2] = $395;
 HEAP32[$17>>2] = $19;
 HEAP32[$18>>2] = 0;
 $396 = HEAP32[$17>>2]|0;
 $397 = HEAP32[$18>>2]|0;
 HEAP32[$396>>2] = $397;
 $398 = HEAP32[$19>>2]|0;
 HEAP32[$23>>2] = $398;
 ;HEAP8[$16>>0]=HEAP8[$23>>0]|0;HEAP8[$16+1>>0]=HEAP8[$23+1>>0]|0;HEAP8[$16+2>>0]=HEAP8[$23+2>>0]|0;HEAP8[$16+3>>0]=HEAP8[$23+3>>0]|0;
 HEAP32[$15>>2] = $21;
 $399 = HEAP32[$15>>2]|0;
 ;HEAP32[$399>>2]=HEAP32[$16>>2]|0;
 $400 = HEAP32[$21>>2]|0;
 HEAP32[$227>>2] = $400;
 while(1) {
  HEAP32[$13>>2] = $226;
  HEAP32[$14>>2] = $227;
  $401 = HEAP32[$13>>2]|0;
  $402 = HEAP32[$14>>2]|0;
  HEAP32[$11>>2] = $401;
  HEAP32[$12>>2] = $402;
  $403 = HEAP32[$11>>2]|0;
  $404 = HEAP32[$12>>2]|0;
  HEAP32[$9>>2] = $403;
  HEAP32[$10>>2] = $404;
  $405 = HEAP32[$9>>2]|0;
  $406 = HEAP32[$405>>2]|0;
  $407 = HEAP32[$10>>2]|0;
  $408 = HEAP32[$407>>2]|0;
  $409 = ($406|0)==($408|0);
  $410 = $409 ^ 1;
  if (!($410)) {
   break;
  }
  HEAP32[$8>>2] = $226;
  $411 = HEAP32[$8>>2]|0;
  HEAP32[$7>>2] = $411;
  $412 = HEAP32[$7>>2]|0;
  $413 = HEAP32[$412>>2]|0;
  HEAP32[$6>>2] = $413;
  $414 = HEAP32[$6>>2]|0;
  HEAP32[$5>>2] = $414;
  $415 = HEAP32[$5>>2]|0;
  HEAP32[$4>>2] = $415;
  $416 = HEAP32[$4>>2]|0;
  $417 = ((($416)) + 8|0);
  HEAP32[$3>>2] = $417;
  $418 = HEAP32[$3>>2]|0;
  HEAP32[$2>>2] = $418;
  $419 = HEAP32[$2>>2]|0;
  HEAP32[$228>>2] = $419;
  $420 = HEAP32[$228>>2]|0;
  $421 = ((($420)) + 4|0);
  HEAP32[$97>>2] = $217;
  HEAP32[$98>>2] = $421;
  $422 = HEAP32[$97>>2]|0;
  $423 = HEAP32[$98>>2]|0;
  $424 = (__ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEE4findIS7_EENS_15__hash_iteratorIPNS_11__hash_nodeIS8_PvEEEERKT_($422,$423)|0);
  HEAP32[$99>>2] = $424;
  ;HEAP8[$95>>0]=HEAP8[$99>>0]|0;HEAP8[$95+1>>0]=HEAP8[$99+1>>0]|0;HEAP8[$95+2>>0]=HEAP8[$99+2>>0]|0;HEAP8[$95+3>>0]=HEAP8[$99+3>>0]|0;
  HEAP32[$94>>2] = $96;
  $425 = HEAP32[$94>>2]|0;
  ;HEAP32[$425>>2]=HEAP32[$95>>2]|0;
  $426 = HEAP32[$96>>2]|0;
  HEAP32[$229>>2] = $426;
  HEAP32[$107>>2] = $217;
  $427 = HEAP32[$107>>2]|0;
  HEAP32[$105>>2] = $427;
  HEAP32[$102>>2] = $104;
  HEAP32[$103>>2] = 0;
  $428 = HEAP32[$102>>2]|0;
  $429 = HEAP32[$103>>2]|0;
  HEAP32[$428>>2] = $429;
  $430 = HEAP32[$104>>2]|0;
  HEAP32[$108>>2] = $430;
  ;HEAP8[$101>>0]=HEAP8[$108>>0]|0;HEAP8[$101+1>>0]=HEAP8[$108+1>>0]|0;HEAP8[$101+2>>0]=HEAP8[$108+2>>0]|0;HEAP8[$101+3>>0]=HEAP8[$108+3>>0]|0;
  HEAP32[$100>>2] = $106;
  $431 = HEAP32[$100>>2]|0;
  ;HEAP32[$431>>2]=HEAP32[$101>>2]|0;
  $432 = HEAP32[$106>>2]|0;
  HEAP32[$230>>2] = $432;
  HEAP32[$113>>2] = $229;
  HEAP32[$114>>2] = $230;
  $433 = HEAP32[$113>>2]|0;
  $434 = HEAP32[$114>>2]|0;
  HEAP32[$111>>2] = $433;
  HEAP32[$112>>2] = $434;
  $435 = HEAP32[$111>>2]|0;
  $436 = HEAP32[$112>>2]|0;
  HEAP32[$109>>2] = $435;
  HEAP32[$110>>2] = $436;
  $437 = HEAP32[$109>>2]|0;
  $438 = HEAP32[$437>>2]|0;
  $439 = HEAP32[$110>>2]|0;
  $440 = HEAP32[$439>>2]|0;
  $441 = ($438|0)==($440|0);
  $442 = $441 ^ 1;
  if ($442) {
   HEAP32[$123>>2] = $229;
   $443 = HEAP32[$123>>2]|0;
   HEAP32[$122>>2] = $443;
   $444 = HEAP32[$122>>2]|0;
   $445 = HEAP32[$444>>2]|0;
   HEAP32[$121>>2] = $445;
   $446 = HEAP32[$121>>2]|0;
   HEAP32[$120>>2] = $446;
   $447 = HEAP32[$120>>2]|0;
   HEAP32[$119>>2] = $447;
   $448 = HEAP32[$119>>2]|0;
   $449 = ((($448)) + 8|0);
   HEAP32[$118>>2] = $449;
   $450 = HEAP32[$118>>2]|0;
   HEAP32[$117>>2] = $450;
   $451 = HEAP32[$117>>2]|0;
   HEAP32[$116>>2] = $451;
   $452 = HEAP32[$116>>2]|0;
   HEAP32[$115>>2] = $452;
   $453 = HEAP32[$115>>2]|0;
   $454 = ((($453)) + 4|0);
   $455 = HEAP32[$454>>2]|0;
   $456 = HEAP32[$228>>2]|0;
   $457 = ((($456)) + 4|0);
   HEAP32[$457>>2] = $455;
  }
  HEAP32[$125>>2] = $226;
  $458 = HEAP32[$125>>2]|0;
  HEAP32[$124>>2] = $458;
  $459 = HEAP32[$124>>2]|0;
  $460 = HEAP32[$459>>2]|0;
  $461 = HEAP32[$460>>2]|0;
  HEAP32[$459>>2] = $461;
 }
 __ZNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEED2Ev($217);
 STACKTOP = sp;return;
}
function __ZNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 __ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEED2Ev($2);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter10InsertMeshEPKN3sap2ve4core4MeshEPKNS2_5INodeEb($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $10 = 0, $100 = 0, $1000 = 0, $1001 = 0, $1002 = 0, $1003 = 0, $1004 = 0, $1005 = 0, $1006 = 0, $1007 = 0, $1008 = 0, $1009 = 0, $101 = 0, $1010 = 0, $1011 = 0, $1012 = 0, $1013 = 0, $1014 = 0, $1015 = 0, $1016 = 0;
 var $1017 = 0, $1018 = 0, $1019 = 0, $102 = 0, $1020 = 0, $1021 = 0, $1022 = 0, $1023 = 0, $1024 = 0, $1025 = 0, $1026 = 0, $1027 = 0, $1028 = 0, $1029 = 0, $103 = 0, $1030 = 0, $1031 = 0, $1032 = 0, $1033 = 0, $1034 = 0;
 var $1035 = 0, $1036 = 0, $1037 = 0, $1038 = 0, $1039 = 0, $104 = 0, $1040 = 0, $1041 = 0, $1042 = 0, $1043 = 0, $1044 = 0, $1045 = 0, $1046 = 0, $1047 = 0, $1048 = 0, $1049 = 0, $105 = 0, $1050 = 0, $1051 = 0, $1052 = 0;
 var $1053 = 0, $1054 = 0, $1055 = 0, $1056 = 0, $1057 = 0, $1058 = 0, $1059 = 0, $106 = 0, $1060 = 0, $1061 = 0, $1062 = 0, $1063 = 0, $1064 = 0, $1065 = 0, $1066 = 0, $1067 = 0, $1068 = 0, $1069 = 0, $107 = 0, $1070 = 0;
 var $1071 = 0, $1072 = 0, $1073 = 0, $1074 = 0, $1075 = 0, $1076 = 0, $1077 = 0, $1078 = 0, $1079 = 0, $108 = 0, $1080 = 0, $1081 = 0, $1082 = 0, $1083 = 0, $1084 = 0, $1085 = 0, $1086 = 0, $1087 = 0, $1088 = 0, $1089 = 0;
 var $109 = 0, $1090 = 0, $1091 = 0, $1092 = 0, $1093 = 0, $1094 = 0, $1095 = 0, $1096 = 0, $1097 = 0, $1098 = 0, $1099 = 0, $11 = 0, $110 = 0, $1100 = 0, $1101 = 0, $1102 = 0, $1103 = 0, $1104 = 0, $1105 = 0, $1106 = 0;
 var $1107 = 0, $1108 = 0, $1109 = 0, $111 = 0, $1110 = 0, $1111 = 0, $1112 = 0, $1113 = 0, $1114 = 0, $1115 = 0, $1116 = 0, $1117 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0;
 var $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0, $136 = 0, $137 = 0;
 var $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0, $154 = 0, $155 = 0;
 var $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0, $172 = 0, $173 = 0;
 var $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0, $190 = 0, $191 = 0;
 var $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0, $209 = 0, $21 = 0;
 var $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0, $227 = 0, $228 = 0;
 var $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0;
 var $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0;
 var $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0;
 var $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $30 = 0, $300 = 0;
 var $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0;
 var $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0;
 var $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0;
 var $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0;
 var $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0;
 var $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0;
 var $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0;
 var $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0;
 var $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0;
 var $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0;
 var $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0, $5 = 0;
 var $50 = 0, $500 = 0, $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $509 = 0, $51 = 0, $510 = 0, $511 = 0, $512 = 0, $513 = 0, $514 = 0, $515 = 0, $516 = 0, $517 = 0;
 var $518 = 0, $519 = 0, $52 = 0, $520 = 0, $521 = 0, $522 = 0, $523 = 0, $524 = 0, $525 = 0, $526 = 0, $527 = 0, $528 = 0, $529 = 0, $53 = 0, $530 = 0, $531 = 0, $532 = 0, $533 = 0, $534 = 0, $535 = 0;
 var $536 = 0, $537 = 0, $538 = 0, $539 = 0, $54 = 0, $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $545 = 0, $546 = 0, $547 = 0, $548 = 0, $549 = 0, $55 = 0, $550 = 0, $551 = 0, $552 = 0, $553 = 0;
 var $554 = 0, $555 = 0, $556 = 0, $557 = 0, $558 = 0, $559 = 0, $56 = 0, $560 = 0, $561 = 0, $562 = 0, $563 = 0, $564 = 0, $565 = 0, $566 = 0, $567 = 0, $568 = 0, $569 = 0, $57 = 0, $570 = 0, $571 = 0;
 var $572 = 0, $573 = 0, $574 = 0, $575 = 0, $576 = 0, $577 = 0, $578 = 0, $579 = 0, $58 = 0, $580 = 0, $581 = 0, $582 = 0, $583 = 0, $584 = 0, $585 = 0, $586 = 0, $587 = 0, $588 = 0, $589 = 0, $59 = 0;
 var $590 = 0, $591 = 0, $592 = 0, $593 = 0, $594 = 0, $595 = 0, $596 = 0, $597 = 0, $598 = 0, $599 = 0, $6 = 0, $60 = 0, $600 = 0, $601 = 0, $602 = 0, $603 = 0, $604 = 0, $605 = 0, $606 = 0, $607 = 0;
 var $608 = 0, $609 = 0.0, $61 = 0, $610 = 0, $611 = 0, $612 = 0, $613 = 0, $614 = 0, $615 = 0, $616 = 0, $617 = 0, $618 = 0, $619 = 0, $62 = 0, $620 = 0, $621 = 0, $622 = 0, $623 = 0, $624 = 0, $625 = 0;
 var $626 = 0, $627 = 0, $628 = 0, $629 = 0, $63 = 0, $630 = 0, $631 = 0, $632 = 0, $633 = 0, $634 = 0, $635 = 0, $636 = 0, $637 = 0, $638 = 0, $639 = 0, $64 = 0, $640 = 0, $641 = 0, $642 = 0, $643 = 0;
 var $644 = 0, $645 = 0, $646 = 0, $647 = 0, $648 = 0, $649 = 0, $65 = 0, $650 = 0, $651 = 0, $652 = 0, $653 = 0, $654 = 0, $655 = 0, $656 = 0, $657 = 0, $658 = 0, $659 = 0, $66 = 0, $660 = 0, $661 = 0;
 var $662 = 0, $663 = 0, $664 = 0, $665 = 0, $666 = 0, $667 = 0, $668 = 0, $669 = 0, $67 = 0, $670 = 0, $671 = 0, $672 = 0, $673 = 0, $674 = 0, $675 = 0, $676 = 0, $677 = 0, $678 = 0, $679 = 0, $68 = 0;
 var $680 = 0, $681 = 0, $682 = 0, $683 = 0, $684 = 0, $685 = 0, $686 = 0, $687 = 0, $688 = 0, $689 = 0, $69 = 0, $690 = 0, $691 = 0, $692 = 0, $693 = 0, $694 = 0, $695 = 0, $696 = 0, $697 = 0, $698 = 0;
 var $699 = 0, $7 = 0, $70 = 0, $700 = 0, $701 = 0, $702 = 0, $703 = 0, $704 = 0, $705 = 0, $706 = 0, $707 = 0, $708 = 0, $709 = 0, $71 = 0, $710 = 0, $711 = 0, $712 = 0, $713 = 0, $714 = 0, $715 = 0;
 var $716 = 0, $717 = 0, $718 = 0, $719 = 0, $72 = 0, $720 = 0, $721 = 0, $722 = 0, $723 = 0, $724 = 0, $725 = 0, $726 = 0, $727 = 0, $728 = 0, $729 = 0, $73 = 0, $730 = 0, $731 = 0, $732 = 0, $733 = 0;
 var $734 = 0, $735 = 0, $736 = 0, $737 = 0, $738 = 0, $739 = 0, $74 = 0, $740 = 0, $741 = 0, $742 = 0, $743 = 0, $744 = 0, $745 = 0, $746 = 0, $747 = 0, $748 = 0, $749 = 0, $75 = 0, $750 = 0, $751 = 0;
 var $752 = 0, $753 = 0, $754 = 0, $755 = 0, $756 = 0, $757 = 0, $758 = 0, $759 = 0, $76 = 0, $760 = 0, $761 = 0, $762 = 0, $763 = 0, $764 = 0, $765 = 0, $766 = 0, $767 = 0, $768 = 0, $769 = 0, $77 = 0;
 var $770 = 0, $771 = 0, $772 = 0, $773 = 0, $774 = 0, $775 = 0, $776 = 0, $777 = 0, $778 = 0, $779 = 0, $78 = 0, $780 = 0, $781 = 0, $782 = 0, $783 = 0, $784 = 0, $785 = 0, $786 = 0, $787 = 0, $788 = 0;
 var $789 = 0.0, $79 = 0, $790 = 0, $791 = 0, $792 = 0, $793 = 0.0, $794 = 0, $795 = 0, $796 = 0, $797 = 0.0, $798 = 0, $799 = 0, $8 = 0, $80 = 0, $800 = 0.0, $801 = 0, $802 = 0, $803 = 0, $804 = 0.0, $805 = 0;
 var $806 = 0, $807 = 0, $808 = 0.0, $809 = 0, $81 = 0, $810 = 0, $811 = 0, $812 = 0, $813 = 0, $814 = 0, $815 = 0, $816 = 0, $817 = 0, $818 = 0, $819 = 0, $82 = 0, $820 = 0, $821 = 0, $822 = 0, $823 = 0;
 var $824 = 0, $825 = 0, $826 = 0, $827 = 0, $828 = 0, $829 = 0, $83 = 0, $830 = 0, $831 = 0, $832 = 0, $833 = 0, $834 = 0, $835 = 0, $836 = 0, $837 = 0, $838 = 0, $839 = 0, $84 = 0, $840 = 0, $841 = 0;
 var $842 = 0, $843 = 0, $844 = 0, $845 = 0, $846 = 0, $847 = 0, $848 = 0, $849 = 0, $85 = 0, $850 = 0, $851 = 0, $852 = 0, $853 = 0, $854 = 0, $855 = 0, $856 = 0, $857 = 0, $858 = 0, $859 = 0, $86 = 0;
 var $860 = 0, $861 = 0, $862 = 0, $863 = 0, $864 = 0, $865 = 0, $866 = 0, $867 = 0, $868 = 0, $869 = 0, $87 = 0, $870 = 0, $871 = 0, $872 = 0, $873 = 0, $874 = 0, $875 = 0, $876 = 0, $877 = 0, $878 = 0;
 var $879 = 0, $88 = 0, $880 = 0, $881 = 0, $882 = 0, $883 = 0, $884 = 0, $885 = 0, $886 = 0, $887 = 0, $888 = 0, $889 = 0, $89 = 0, $890 = 0, $891 = 0, $892 = 0, $893 = 0, $894 = 0, $895 = 0, $896 = 0;
 var $897 = 0, $898 = 0, $899 = 0, $9 = 0, $90 = 0, $900 = 0, $901 = 0, $902 = 0, $903 = 0, $904 = 0, $905 = 0, $906 = 0, $907 = 0, $908 = 0, $909 = 0, $91 = 0, $910 = 0, $911 = 0, $912 = 0, $913 = 0;
 var $914 = 0, $915 = 0, $916 = 0, $917 = 0, $918 = 0, $919 = 0, $92 = 0, $920 = 0, $921 = 0, $922 = 0, $923 = 0, $924 = 0, $925 = 0, $926 = 0, $927 = 0, $928 = 0, $929 = 0, $93 = 0, $930 = 0, $931 = 0;
 var $932 = 0, $933 = 0, $934 = 0, $935 = 0, $936 = 0, $937 = 0, $938 = 0, $939 = 0, $94 = 0, $940 = 0, $941 = 0, $942 = 0, $943 = 0, $944 = 0, $945 = 0, $946 = 0, $947 = 0, $948 = 0, $949 = 0, $95 = 0;
 var $950 = 0, $951 = 0, $952 = 0, $953 = 0, $954 = 0, $955 = 0, $956 = 0, $957 = 0, $958 = 0, $959 = 0, $96 = 0, $960 = 0, $961 = 0, $962 = 0, $963 = 0, $964 = 0, $965 = 0, $966 = 0, $967 = 0, $968 = 0;
 var $969 = 0, $97 = 0, $970 = 0, $971 = 0, $972 = 0, $973 = 0, $974 = 0, $975 = 0, $976 = 0, $977 = 0, $978 = 0, $979 = 0, $98 = 0, $980 = 0, $981 = 0, $982 = 0, $983 = 0, $984 = 0, $985 = 0, $986 = 0;
 var $987 = 0, $988 = 0, $989 = 0, $99 = 0, $990 = 0, $991 = 0, $992 = 0, $993 = 0, $994 = 0, $995 = 0, $996 = 0, $997 = 0, $998 = 0, $999 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 1936|0;
 $4 = sp + 1916|0;
 $5 = sp + 1912|0;
 $6 = sp + 1908|0;
 $7 = sp + 1904|0;
 $8 = sp + 1900|0;
 $9 = sp + 1896|0;
 $10 = sp + 1892|0;
 $11 = sp + 1888|0;
 $12 = sp + 1884|0;
 $13 = sp + 1880|0;
 $14 = sp + 1876|0;
 $15 = sp + 1872|0;
 $16 = sp + 1868|0;
 $17 = sp + 1864|0;
 $18 = sp + 1860|0;
 $19 = sp + 1856|0;
 $20 = sp + 1852|0;
 $21 = sp + 1848|0;
 $22 = sp + 1844|0;
 $23 = sp + 1840|0;
 $24 = sp + 1836|0;
 $25 = sp + 1832|0;
 $26 = sp + 1828|0;
 $27 = sp + 1824|0;
 $28 = sp + 1820|0;
 $29 = sp + 1816|0;
 $30 = sp + 1812|0;
 $31 = sp + 1808|0;
 $32 = sp + 1804|0;
 $33 = sp + 1800|0;
 $34 = sp + 1796|0;
 $35 = sp + 1792|0;
 $36 = sp + 1788|0;
 $37 = sp + 1784|0;
 $38 = sp + 1780|0;
 $39 = sp + 1776|0;
 $40 = sp + 1772|0;
 $41 = sp + 1768|0;
 $42 = sp + 1764|0;
 $43 = sp + 1760|0;
 $44 = sp + 1756|0;
 $45 = sp + 1752|0;
 $46 = sp + 1748|0;
 $47 = sp + 1744|0;
 $48 = sp + 1740|0;
 $49 = sp + 1736|0;
 $50 = sp + 1732|0;
 $51 = sp + 1728|0;
 $52 = sp + 1724|0;
 $53 = sp + 1720|0;
 $54 = sp + 1716|0;
 $55 = sp + 1712|0;
 $56 = sp + 1708|0;
 $57 = sp + 1704|0;
 $58 = sp + 1700|0;
 $59 = sp + 1696|0;
 $60 = sp + 1692|0;
 $61 = sp + 1688|0;
 $62 = sp + 1684|0;
 $63 = sp + 1680|0;
 $64 = sp + 1676|0;
 $65 = sp + 1672|0;
 $66 = sp + 1668|0;
 $67 = sp + 1664|0;
 $68 = sp + 1660|0;
 $69 = sp + 1656|0;
 $70 = sp + 1652|0;
 $71 = sp + 1648|0;
 $72 = sp + 1644|0;
 $73 = sp + 1640|0;
 $74 = sp + 1636|0;
 $75 = sp + 1632|0;
 $76 = sp + 1628|0;
 $77 = sp + 1624|0;
 $78 = sp + 1620|0;
 $79 = sp + 1616|0;
 $80 = sp + 128|0;
 $81 = sp + 1612|0;
 $82 = sp + 1608|0;
 $83 = sp + 1604|0;
 $84 = sp + 1600|0;
 $85 = sp + 1596|0;
 $86 = sp + 1592|0;
 $87 = sp + 1588|0;
 $88 = sp + 1584|0;
 $89 = sp + 1576|0;
 $90 = sp + 1572|0;
 $91 = sp + 120|0;
 $92 = sp + 1568|0;
 $93 = sp + 1564|0;
 $94 = sp + 1560|0;
 $95 = sp + 1556|0;
 $96 = sp + 1552|0;
 $97 = sp + 1548|0;
 $98 = sp + 1544|0;
 $99 = sp + 1540|0;
 $100 = sp + 1536|0;
 $101 = sp + 1532|0;
 $102 = sp + 1528|0;
 $103 = sp + 1524|0;
 $104 = sp + 1520|0;
 $105 = sp + 112|0;
 $106 = sp + 1516|0;
 $107 = sp + 1512|0;
 $108 = sp + 1508|0;
 $109 = sp + 1504|0;
 $110 = sp + 1500|0;
 $111 = sp + 1496|0;
 $112 = sp + 1492|0;
 $113 = sp + 1488|0;
 $114 = sp + 1484|0;
 $115 = sp + 1480|0;
 $116 = sp + 1476|0;
 $117 = sp + 1472|0;
 $118 = sp + 1468|0;
 $119 = sp + 1464|0;
 $120 = sp + 1460|0;
 $121 = sp + 1456|0;
 $122 = sp + 1452|0;
 $123 = sp + 104|0;
 $124 = sp + 1448|0;
 $125 = sp + 1444|0;
 $126 = sp + 1440|0;
 $127 = sp + 1436|0;
 $128 = sp + 1935|0;
 $129 = sp + 1432|0;
 $130 = sp + 1428|0;
 $131 = sp + 1424|0;
 $132 = sp + 1420|0;
 $133 = sp + 1416|0;
 $134 = sp + 1412|0;
 $135 = sp + 1408|0;
 $136 = sp + 1404|0;
 $137 = sp + 1400|0;
 $138 = sp + 1396|0;
 $139 = sp + 1934|0;
 $140 = sp + 1392|0;
 $141 = sp + 1388|0;
 $142 = sp + 1384|0;
 $143 = sp + 1380|0;
 $144 = sp + 1376|0;
 $145 = sp + 1372|0;
 $146 = sp + 1368|0;
 $147 = sp + 1364|0;
 $148 = sp + 1360|0;
 $149 = sp + 1356|0;
 $150 = sp + 1352|0;
 $151 = sp + 1348|0;
 $152 = sp + 1344|0;
 $153 = sp + 1340|0;
 $154 = sp + 1336|0;
 $155 = sp + 1332|0;
 $156 = sp + 96|0;
 $157 = sp + 1328|0;
 $158 = sp + 1324|0;
 $159 = sp + 1320|0;
 $160 = sp + 1316|0;
 $161 = sp + 1312|0;
 $162 = sp + 1308|0;
 $163 = sp + 1304|0;
 $164 = sp + 1300|0;
 $165 = sp + 88|0;
 $166 = sp + 1296|0;
 $167 = sp + 1292|0;
 $168 = sp + 1288|0;
 $169 = sp + 1284|0;
 $170 = sp + 1280|0;
 $171 = sp + 1276|0;
 $172 = sp + 1272|0;
 $173 = sp + 1268|0;
 $174 = sp + 1264|0;
 $175 = sp + 1260|0;
 $176 = sp + 1256|0;
 $177 = sp + 1252|0;
 $178 = sp + 1248|0;
 $179 = sp + 1244|0;
 $180 = sp + 1240|0;
 $181 = sp + 1236|0;
 $182 = sp + 1232|0;
 $183 = sp + 1228|0;
 $184 = sp + 1224|0;
 $185 = sp + 1220|0;
 $186 = sp + 1216|0;
 $187 = sp + 1212|0;
 $188 = sp + 1208|0;
 $189 = sp + 1204|0;
 $190 = sp + 1200|0;
 $191 = sp + 1196|0;
 $192 = sp + 80|0;
 $193 = sp + 1192|0;
 $194 = sp + 1188|0;
 $195 = sp + 1933|0;
 $196 = sp + 1184|0;
 $197 = sp + 1180|0;
 $198 = sp + 1176|0;
 $199 = sp + 1172|0;
 $200 = sp + 1168|0;
 $201 = sp + 1164|0;
 $202 = sp + 1160|0;
 $203 = sp + 1156|0;
 $204 = sp + 1152|0;
 $205 = sp + 1148|0;
 $206 = sp + 1144|0;
 $207 = sp + 1140|0;
 $208 = sp + 1136|0;
 $209 = sp + 1124|0;
 $210 = sp + 1120|0;
 $211 = sp + 1116|0;
 $212 = sp + 1112|0;
 $213 = sp + 72|0;
 $214 = sp + 1108|0;
 $215 = sp + 1104|0;
 $216 = sp + 1100|0;
 $217 = sp + 1932|0;
 $218 = sp + 1096|0;
 $219 = sp + 64|0;
 $220 = sp + 1092|0;
 $221 = sp + 1088|0;
 $222 = sp + 1084|0;
 $223 = sp + 1080|0;
 $224 = sp + 1076|0;
 $225 = sp + 1072|0;
 $226 = sp + 1068|0;
 $227 = sp + 1064|0;
 $228 = sp + 1056|0;
 $229 = sp + 1048|0;
 $230 = sp + 1044|0;
 $231 = sp + 1040|0;
 $232 = sp + 1036|0;
 $233 = sp + 1032|0;
 $234 = sp + 1028|0;
 $235 = sp + 1024|0;
 $236 = sp + 1020|0;
 $237 = sp + 1016|0;
 $238 = sp + 1012|0;
 $239 = sp + 1008|0;
 $240 = sp + 1004|0;
 $241 = sp + 1000|0;
 $242 = sp + 996|0;
 $243 = sp + 992|0;
 $244 = sp + 988|0;
 $245 = sp + 984|0;
 $246 = sp + 980|0;
 $247 = sp + 976|0;
 $248 = sp + 972|0;
 $249 = sp + 968|0;
 $250 = sp + 964|0;
 $251 = sp + 960|0;
 $252 = sp + 956|0;
 $253 = sp + 952|0;
 $254 = sp + 948|0;
 $255 = sp + 944|0;
 $256 = sp + 940|0;
 $257 = sp + 936|0;
 $258 = sp + 932|0;
 $259 = sp + 928|0;
 $260 = sp + 924|0;
 $261 = sp + 920|0;
 $262 = sp + 916|0;
 $263 = sp + 912|0;
 $264 = sp + 908|0;
 $265 = sp + 904|0;
 $266 = sp + 900|0;
 $267 = sp + 896|0;
 $268 = sp + 892|0;
 $269 = sp + 888|0;
 $270 = sp + 884|0;
 $271 = sp + 880|0;
 $272 = sp + 876|0;
 $273 = sp + 872|0;
 $274 = sp + 868|0;
 $275 = sp + 864|0;
 $276 = sp + 860|0;
 $277 = sp + 856|0;
 $278 = sp + 852|0;
 $279 = sp + 848|0;
 $280 = sp + 844|0;
 $281 = sp + 840|0;
 $282 = sp + 836|0;
 $283 = sp + 832|0;
 $284 = sp + 828|0;
 $285 = sp + 824|0;
 $286 = sp + 820|0;
 $287 = sp + 816|0;
 $288 = sp + 812|0;
 $289 = sp + 808|0;
 $290 = sp + 804|0;
 $291 = sp + 800|0;
 $292 = sp + 796|0;
 $293 = sp + 792|0;
 $294 = sp + 788|0;
 $295 = sp + 784|0;
 $296 = sp + 780|0;
 $297 = sp + 776|0;
 $298 = sp + 772|0;
 $299 = sp + 768|0;
 $300 = sp + 764|0;
 $301 = sp + 760|0;
 $302 = sp + 756|0;
 $303 = sp + 752|0;
 $304 = sp + 748|0;
 $305 = sp + 744|0;
 $306 = sp + 740|0;
 $307 = sp + 736|0;
 $308 = sp + 732|0;
 $309 = sp + 728|0;
 $310 = sp + 724|0;
 $311 = sp + 720|0;
 $312 = sp + 716|0;
 $313 = sp + 712|0;
 $314 = sp + 708|0;
 $315 = sp + 704|0;
 $316 = sp + 700|0;
 $317 = sp + 696|0;
 $318 = sp + 692|0;
 $319 = sp + 688|0;
 $320 = sp + 56|0;
 $321 = sp + 684|0;
 $322 = sp + 680|0;
 $323 = sp + 676|0;
 $324 = sp + 1931|0;
 $325 = sp + 672|0;
 $326 = sp + 48|0;
 $327 = sp + 668|0;
 $328 = sp + 664|0;
 $329 = sp + 660|0;
 $330 = sp + 656|0;
 $331 = sp + 652|0;
 $332 = sp + 648|0;
 $333 = sp + 644|0;
 $334 = sp + 640|0;
 $335 = sp + 632|0;
 $336 = sp + 628|0;
 $337 = sp + 624|0;
 $338 = sp + 620|0;
 $339 = sp + 616|0;
 $340 = sp + 612|0;
 $341 = sp + 608|0;
 $342 = sp + 604|0;
 $343 = sp + 600|0;
 $344 = sp + 596|0;
 $345 = sp + 592|0;
 $346 = sp + 588|0;
 $347 = sp + 584|0;
 $348 = sp + 580|0;
 $349 = sp + 576|0;
 $350 = sp + 572|0;
 $351 = sp + 568|0;
 $352 = sp + 564|0;
 $353 = sp + 560|0;
 $354 = sp + 556|0;
 $355 = sp + 552|0;
 $356 = sp + 548|0;
 $357 = sp + 544|0;
 $358 = sp + 540|0;
 $359 = sp + 536|0;
 $360 = sp + 532|0;
 $361 = sp + 528|0;
 $362 = sp + 524|0;
 $363 = sp + 520|0;
 $364 = sp + 516|0;
 $365 = sp + 512|0;
 $366 = sp + 508|0;
 $367 = sp + 504|0;
 $368 = sp + 500|0;
 $369 = sp + 496|0;
 $370 = sp + 40|0;
 $371 = sp + 492|0;
 $372 = sp + 488|0;
 $373 = sp + 484|0;
 $374 = sp + 480|0;
 $375 = sp + 476|0;
 $376 = sp + 472|0;
 $377 = sp + 468|0;
 $378 = sp + 464|0;
 $379 = sp + 460|0;
 $380 = sp + 456|0;
 $381 = sp + 452|0;
 $382 = sp + 448|0;
 $383 = sp + 444|0;
 $384 = sp + 32|0;
 $385 = sp + 440|0;
 $386 = sp + 436|0;
 $387 = sp + 432|0;
 $388 = sp + 428|0;
 $389 = sp + 424|0;
 $390 = sp + 24|0;
 $391 = sp + 420|0;
 $392 = sp + 416|0;
 $393 = sp + 412|0;
 $394 = sp + 408|0;
 $395 = sp + 404|0;
 $396 = sp + 400|0;
 $397 = sp + 396|0;
 $398 = sp + 392|0;
 $399 = sp + 388|0;
 $400 = sp + 384|0;
 $401 = sp + 380|0;
 $402 = sp + 376|0;
 $403 = sp + 372|0;
 $404 = sp + 368|0;
 $405 = sp + 364|0;
 $406 = sp + 1930|0;
 $407 = sp + 360|0;
 $408 = sp + 356|0;
 $409 = sp + 352|0;
 $410 = sp + 348|0;
 $411 = sp + 344|0;
 $412 = sp + 340|0;
 $413 = sp + 328|0;
 $414 = sp + 324|0;
 $415 = sp + 320|0;
 $416 = sp + 300|0;
 $417 = sp + 296|0;
 $418 = sp + 292|0;
 $419 = sp + 288|0;
 $420 = sp + 280|0;
 $421 = sp + 272|0;
 $422 = sp + 268|0;
 $423 = sp + 264|0;
 $424 = sp + 260|0;
 $425 = sp + 256|0;
 $426 = sp + 252|0;
 $427 = sp + 248|0;
 $428 = sp + 244|0;
 $429 = sp + 240|0;
 $430 = sp + 236|0;
 $431 = sp + 232|0;
 $432 = sp;
 $433 = sp + 228|0;
 $434 = sp + 1928|0;
 $435 = sp + 224|0;
 $436 = sp + 220|0;
 $437 = sp + 216|0;
 $438 = sp + 212|0;
 $439 = sp + 208|0;
 $440 = sp + 204|0;
 $441 = sp + 200|0;
 $442 = sp + 192|0;
 $443 = sp + 188|0;
 $444 = sp + 184|0;
 $445 = sp + 180|0;
 $446 = sp + 176|0;
 $447 = sp + 172|0;
 $448 = sp + 168|0;
 $449 = sp + 1926|0;
 $450 = sp + 1924|0;
 $451 = sp + 1922|0;
 $452 = sp + 1920|0;
 $453 = sp + 160|0;
 $454 = sp + 144|0;
 $455 = sp + 136|0;
 HEAP32[$403>>2] = $0;
 HEAP32[$404>>2] = $1;
 HEAP32[$405>>2] = $2;
 $456 = $3&1;
 HEAP8[$406>>0] = $456;
 $457 = HEAP32[$403>>2]|0;
 $458 = HEAP8[$406>>0]|0;
 $459 = $458&1;
 $460 = ((($457)) + 32|0);
 if ($459) {
  HEAP32[$401>>2] = $460;
  $461 = HEAP32[$401>>2]|0;
  HEAP32[$399>>2] = $461;
  $462 = HEAP32[$399>>2]|0;
  HEAP32[$397>>2] = $462;
  $463 = HEAP32[$397>>2]|0;
  $464 = ((($463)) + 4|0);
  HEAP32[$396>>2] = $464;
  $465 = HEAP32[$396>>2]|0;
  HEAP32[$395>>2] = $465;
  $466 = HEAP32[$395>>2]|0;
  HEAP32[$394>>2] = $466;
  $467 = HEAP32[$394>>2]|0;
  HEAP32[$393>>2] = $467;
  $468 = HEAP32[$393>>2]|0;
  HEAP32[$391>>2] = $398;
  HEAP32[$392>>2] = $468;
  $469 = HEAP32[$391>>2]|0;
  $470 = HEAP32[$392>>2]|0;
  HEAP32[$469>>2] = $470;
  $471 = HEAP32[$398>>2]|0;
  HEAP32[$402>>2] = $471;
  ;HEAP8[$390>>0]=HEAP8[$402>>0]|0;HEAP8[$390+1>>0]=HEAP8[$402+1>>0]|0;HEAP8[$390+2>>0]=HEAP8[$402+2>>0]|0;HEAP8[$390+3>>0]=HEAP8[$402+3>>0]|0;
  HEAP32[$389>>2] = $400;
  $472 = HEAP32[$389>>2]|0;
  ;HEAP32[$472>>2]=HEAP32[$390>>2]|0;
  $473 = HEAP32[$400>>2]|0;
  HEAP32[$407>>2] = $473;
 } else {
  HEAP32[$386>>2] = $460;
  HEAP32[$387>>2] = $404;
  $474 = HEAP32[$386>>2]|0;
  $475 = HEAP32[$387>>2]|0;
  $476 = (__ZNSt3__26__treeINS_12__value_typeIPKN3sap2ve4core4MeshENS_6vectorIPKNS4_7SubMeshENS_9allocatorISB_EEEEEENS_19__map_value_compareIS7_SF_NS_4lessIS7_EELb1EEENSC_ISF_EEE4findIS7_EENS_15__tree_iteratorISF_PNS_11__tree_nodeISF_PvEElEERKT_($474,$475)|0);
  HEAP32[$388>>2] = $476;
  ;HEAP8[$384>>0]=HEAP8[$388>>0]|0;HEAP8[$384+1>>0]=HEAP8[$388+1>>0]|0;HEAP8[$384+2>>0]=HEAP8[$388+2>>0]|0;HEAP8[$384+3>>0]=HEAP8[$388+3>>0]|0;
  HEAP32[$383>>2] = $385;
  $477 = HEAP32[$383>>2]|0;
  ;HEAP32[$477>>2]=HEAP32[$384>>2]|0;
  $478 = HEAP32[$385>>2]|0;
  HEAP32[$407>>2] = $478;
 }
 $479 = ((($457)) + 32|0);
 HEAP32[$381>>2] = $479;
 $480 = HEAP32[$381>>2]|0;
 HEAP32[$379>>2] = $480;
 $481 = HEAP32[$379>>2]|0;
 HEAP32[$377>>2] = $481;
 $482 = HEAP32[$377>>2]|0;
 $483 = ((($482)) + 4|0);
 HEAP32[$376>>2] = $483;
 $484 = HEAP32[$376>>2]|0;
 HEAP32[$375>>2] = $484;
 $485 = HEAP32[$375>>2]|0;
 HEAP32[$374>>2] = $485;
 $486 = HEAP32[$374>>2]|0;
 HEAP32[$373>>2] = $486;
 $487 = HEAP32[$373>>2]|0;
 HEAP32[$371>>2] = $378;
 HEAP32[$372>>2] = $487;
 $488 = HEAP32[$371>>2]|0;
 $489 = HEAP32[$372>>2]|0;
 HEAP32[$488>>2] = $489;
 $490 = HEAP32[$378>>2]|0;
 HEAP32[$382>>2] = $490;
 ;HEAP8[$370>>0]=HEAP8[$382>>0]|0;HEAP8[$370+1>>0]=HEAP8[$382+1>>0]|0;HEAP8[$370+2>>0]=HEAP8[$382+2>>0]|0;HEAP8[$370+3>>0]=HEAP8[$382+3>>0]|0;
 HEAP32[$369>>2] = $380;
 $491 = HEAP32[$369>>2]|0;
 ;HEAP32[$491>>2]=HEAP32[$370>>2]|0;
 $492 = HEAP32[$380>>2]|0;
 HEAP32[$408>>2] = $492;
 HEAP32[$367>>2] = $407;
 HEAP32[$368>>2] = $408;
 $493 = HEAP32[$367>>2]|0;
 $494 = HEAP32[$368>>2]|0;
 HEAP32[$365>>2] = $493;
 HEAP32[$366>>2] = $494;
 $495 = HEAP32[$365>>2]|0;
 $496 = HEAP32[$366>>2]|0;
 HEAP32[$363>>2] = $495;
 HEAP32[$364>>2] = $496;
 $497 = HEAP32[$363>>2]|0;
 $498 = HEAP32[$497>>2]|0;
 $499 = HEAP32[$364>>2]|0;
 $500 = HEAP32[$499>>2]|0;
 $501 = ($498|0)==($500|0);
 $502 = $501 ^ 1;
 if ($502) {
  HEAP32[$362>>2] = $407;
  $503 = HEAP32[$362>>2]|0;
  HEAP32[$361>>2] = $503;
  $504 = HEAP32[$361>>2]|0;
  HEAP32[$360>>2] = $504;
  $505 = HEAP32[$360>>2]|0;
  $506 = HEAP32[$505>>2]|0;
  $507 = ((($506)) + 16|0);
  HEAP32[$359>>2] = $507;
  $508 = HEAP32[$359>>2]|0;
  HEAP32[$358>>2] = $508;
  $509 = HEAP32[$358>>2]|0;
  HEAP32[$357>>2] = $509;
  $510 = HEAP32[$357>>2]|0;
  HEAP32[$356>>2] = $510;
  $511 = HEAP32[$356>>2]|0;
  $512 = ((($511)) + 4|0);
  HEAP32[$409>>2] = $512;
  $513 = HEAP32[$409>>2]|0;
  HEAP32[$355>>2] = $513;
  $514 = HEAP32[$355>>2]|0;
  $515 = HEAP32[$514>>2]|0;
  HEAP32[$352>>2] = $514;
  HEAP32[$353>>2] = $515;
  $516 = HEAP32[$353>>2]|0;
  HEAP32[$349>>2] = $351;
  HEAP32[$350>>2] = $516;
  $517 = HEAP32[$349>>2]|0;
  $518 = HEAP32[$350>>2]|0;
  HEAP32[$517>>2] = $518;
  $519 = HEAP32[$351>>2]|0;
  HEAP32[$354>>2] = $519;
  $520 = HEAP32[$354>>2]|0;
  HEAP32[$410>>2] = $520;
  $521 = HEAP32[$409>>2]|0;
  HEAP32[$348>>2] = $521;
  $522 = HEAP32[$348>>2]|0;
  $523 = ((($522)) + 4|0);
  $524 = HEAP32[$523>>2]|0;
  HEAP32[$345>>2] = $522;
  HEAP32[$346>>2] = $524;
  $525 = HEAP32[$346>>2]|0;
  HEAP32[$342>>2] = $344;
  HEAP32[$343>>2] = $525;
  $526 = HEAP32[$342>>2]|0;
  $527 = HEAP32[$343>>2]|0;
  HEAP32[$526>>2] = $527;
  $528 = HEAP32[$344>>2]|0;
  HEAP32[$347>>2] = $528;
  $529 = HEAP32[$347>>2]|0;
  HEAP32[$411>>2] = $529;
  while(1) {
   HEAP32[$340>>2] = $410;
   HEAP32[$341>>2] = $411;
   $530 = HEAP32[$340>>2]|0;
   $531 = HEAP32[$341>>2]|0;
   HEAP32[$338>>2] = $530;
   HEAP32[$339>>2] = $531;
   $532 = HEAP32[$338>>2]|0;
   HEAP32[$337>>2] = $532;
   $533 = HEAP32[$337>>2]|0;
   $534 = HEAP32[$533>>2]|0;
   $535 = HEAP32[$339>>2]|0;
   HEAP32[$336>>2] = $535;
   $536 = HEAP32[$336>>2]|0;
   $537 = HEAP32[$536>>2]|0;
   $538 = ($534|0)==($537|0);
   $539 = $538 ^ 1;
   if (!($539)) {
    break;
   }
   HEAP32[$316>>2] = $410;
   $540 = HEAP32[$316>>2]|0;
   $541 = HEAP32[$540>>2]|0;
   $542 = HEAP32[$541>>2]|0;
   HEAP32[$412>>2] = $542;
   $543 = ((($457)) + 24|0);
   $544 = HEAP32[$543>>2]|0;
   $545 = HEAP32[$405>>2]|0;
   $546 = HEAP32[$412>>2]|0;
   _SceneBuilderAdapter_insertMesh(($544|0),($545|0),($546|0));
   HEAP32[$306>>2] = $410;
   $547 = HEAP32[$306>>2]|0;
   $548 = HEAP32[$547>>2]|0;
   $549 = ((($548)) + 4|0);
   HEAP32[$547>>2] = $549;
  }
  STACKTOP = sp;return;
 }
 HEAP32[$305>>2] = $413;
 $550 = HEAP32[$305>>2]|0;
 HEAP32[$303>>2] = $550;
 $551 = HEAP32[$303>>2]|0;
 HEAP32[$302>>2] = $551;
 HEAP32[$551>>2] = 0;
 $552 = ((($551)) + 4|0);
 HEAP32[$552>>2] = 0;
 $553 = ((($551)) + 8|0);
 HEAP32[$304>>2] = 0;
 HEAP32[$300>>2] = $553;
 HEAP32[$301>>2] = $304;
 $554 = HEAP32[$300>>2]|0;
 $555 = HEAP32[$301>>2]|0;
 HEAP32[$299>>2] = $555;
 $556 = HEAP32[$299>>2]|0;
 HEAP32[$295>>2] = $554;
 HEAP32[$296>>2] = $556;
 $557 = HEAP32[$295>>2]|0;
 $558 = HEAP32[$296>>2]|0;
 HEAP32[$294>>2] = $558;
 HEAP32[$557>>2] = 0;
 HEAP32[$298>>2] = $554;
 $559 = HEAP32[$298>>2]|0;
 HEAP32[$297>>2] = $559;
 $560 = HEAP32[$404>>2]|0;
 $561 = HEAP32[$560>>2]|0;
 $562 = ((($561)) + 84|0);
 $563 = HEAP32[$562>>2]|0;
 $564 = (FUNCTION_TABLE_ii[$563 & 2047]($560)|0);
 HEAP32[$414>>2] = $564;
 $565 = HEAP32[$404>>2]|0;
 $566 = (__ZNK3sap2ve4core4Mesh13GetSubMeshPtrEv($565)|0);
 HEAP32[$415>>2] = $566;
 HEAP32[$293>>2] = $416;
 $567 = HEAP32[$293>>2]|0;
 HEAP32[$290>>2] = $567;
 $568 = HEAP32[$290>>2]|0;
 HEAP32[$288>>2] = $568;
 $569 = HEAP32[$288>>2]|0;
 HEAP32[$289>>2] = 0;
 HEAP32[$286>>2] = $569;
 HEAP32[$287>>2] = $289;
 $570 = HEAP32[$286>>2]|0;
 $571 = HEAP32[$287>>2]|0;
 HEAP32[$285>>2] = $571;
 $572 = HEAP32[$285>>2]|0;
 HEAP32[$272>>2] = $570;
 HEAP32[$273>>2] = $572;
 $573 = HEAP32[$272>>2]|0;
 $574 = HEAP32[$273>>2]|0;
 HEAP32[$271>>2] = $574;
 $575 = HEAP32[$271>>2]|0;
 $576 = HEAP32[$575>>2]|0;
 HEAP32[$573>>2] = $576;
 $577 = ((($570)) + 4|0);
 HEAP32[$284>>2] = $577;
 $578 = HEAP32[$284>>2]|0;
 HEAP32[$282>>2] = $578;
 $579 = HEAP32[$282>>2]|0;
 HEAP32[$283>>2] = 0;
 HEAP32[$280>>2] = $579;
 HEAP32[$281>>2] = $283;
 $580 = HEAP32[$280>>2]|0;
 $581 = HEAP32[$281>>2]|0;
 HEAP32[$279>>2] = $581;
 $582 = HEAP32[$279>>2]|0;
 HEAP32[$275>>2] = $580;
 HEAP32[$276>>2] = $582;
 $583 = HEAP32[$275>>2]|0;
 $584 = HEAP32[$276>>2]|0;
 HEAP32[$274>>2] = $584;
 $585 = HEAP32[$274>>2]|0;
 $586 = HEAP32[$585>>2]|0;
 HEAP32[$583>>2] = $586;
 HEAP32[$278>>2] = $580;
 $587 = HEAP32[$278>>2]|0;
 HEAP32[$277>>2] = $587;
 $588 = ((($568)) + 8|0);
 HEAP32[$270>>2] = $588;
 $589 = HEAP32[$270>>2]|0;
 HEAP32[$269>>2] = $589;
 $590 = HEAP32[$269>>2]|0;
 HEAP32[$268>>2] = $590;
 $591 = HEAP32[$268>>2]|0;
 HEAP32[$591>>2] = 0;
 HEAP32[$267>>2] = $589;
 $592 = HEAP32[$267>>2]|0;
 HEAP32[$266>>2] = $592;
 $593 = ((($568)) + 12|0);
 HEAP32[$291>>2] = 0;
 HEAP32[$264>>2] = $593;
 HEAP32[$265>>2] = $291;
 $594 = HEAP32[$264>>2]|0;
 $595 = HEAP32[$265>>2]|0;
 HEAP32[$263>>2] = $595;
 $596 = HEAP32[$263>>2]|0;
 HEAP32[$259>>2] = $594;
 HEAP32[$260>>2] = $596;
 $597 = HEAP32[$259>>2]|0;
 $598 = HEAP32[$260>>2]|0;
 HEAP32[$258>>2] = $598;
 $599 = HEAP32[$258>>2]|0;
 $600 = HEAP32[$599>>2]|0;
 HEAP32[$597>>2] = $600;
 HEAP32[$262>>2] = $594;
 $601 = HEAP32[$262>>2]|0;
 HEAP32[$261>>2] = $601;
 $602 = ((($568)) + 16|0);
 HEAPF32[$292>>2] = 1.0;
 HEAP32[$256>>2] = $602;
 HEAP32[$257>>2] = $292;
 $603 = HEAP32[$256>>2]|0;
 $604 = HEAP32[$257>>2]|0;
 HEAP32[$255>>2] = $604;
 $605 = HEAP32[$255>>2]|0;
 HEAP32[$251>>2] = $603;
 HEAP32[$252>>2] = $605;
 $606 = HEAP32[$251>>2]|0;
 $607 = HEAP32[$252>>2]|0;
 HEAP32[$250>>2] = $607;
 $608 = HEAP32[$250>>2]|0;
 $609 = +HEAPF32[$608>>2];
 HEAPF32[$606>>2] = $609;
 HEAP32[$254>>2] = $603;
 $610 = HEAP32[$254>>2]|0;
 HEAP32[$253>>2] = $610;
 $611 = HEAP32[$405>>2]|0;
 $612 = HEAP32[$611>>2]|0;
 $613 = ((($612)) + 52|0);
 $614 = HEAP32[$613>>2]|0;
 $615 = (FUNCTION_TABLE_iii[$614 & 1023]($611,0)|0);
 HEAP32[$417>>2] = $615;
 $616 = HEAP32[$417>>2]|0;
 $617 = ($616|0)!=(0|0);
 if ($617) {
  HEAP32[$418>>2] = 0;
  while(1) {
   $618 = HEAP32[$418>>2]|0;
   $619 = HEAP32[$414>>2]|0;
   $620 = ($618>>>0)<($619>>>0);
   if (!($620)) {
    break;
   }
   $621 = HEAP32[$415>>2]|0;
   $622 = HEAP32[$418>>2]|0;
   $623 = (($621) + ($622<<2)|0);
   $624 = (__ZNK3sap2ve4core7SubMesh11GetMaterialEv($623)|0);
   HEAP32[$419>>2] = $624;
   $625 = HEAP32[$419>>2]|0;
   $626 = ($625|0)!=(0|0);
   if ($626) {
    HEAP32[$236>>2] = $419;
    HEAP32[$237>>2] = $419;
    $627 = HEAP32[$236>>2]|0;
    HEAP32[$235>>2] = $627;
    $628 = HEAP32[$235>>2]|0;
    $629 = HEAP32[$237>>2]|0;
    HEAP32[$229>>2] = $629;
    $630 = HEAP32[$229>>2]|0;
    HEAP32[$232>>2] = $420;
    HEAP32[$233>>2] = $628;
    HEAP32[$234>>2] = $630;
    $631 = HEAP32[$232>>2]|0;
    $632 = HEAP32[$233>>2]|0;
    HEAP32[$231>>2] = $632;
    $633 = HEAP32[$231>>2]|0;
    $634 = HEAP32[$633>>2]|0;
    HEAP32[$631>>2] = $634;
    $635 = ((($631)) + 4|0);
    $636 = HEAP32[$234>>2]|0;
    HEAP32[$230>>2] = $636;
    $637 = HEAP32[$230>>2]|0;
    $638 = HEAP32[$637>>2]|0;
    HEAP32[$635>>2] = $638;
    HEAP32[$226>>2] = $416;
    HEAP32[$227>>2] = $420;
    $639 = HEAP32[$226>>2]|0;
    $640 = HEAP32[$227>>2]|0;
    HEAP32[$225>>2] = $640;
    $641 = HEAP32[$225>>2]|0;
    HEAP32[$215>>2] = $639;
    HEAP32[$216>>2] = $641;
    $642 = HEAP32[$215>>2]|0;
    $643 = HEAP32[$216>>2]|0;
    HEAP32[$214>>2] = $643;
    $644 = HEAP32[$214>>2]|0;
    ;HEAP8[$213>>0]=HEAP8[$217>>0]|0;
    HEAP32[$211>>2] = $642;
    HEAP32[$212>>2] = $644;
    $645 = HEAP32[$211>>2]|0;
    $646 = HEAP32[$212>>2]|0;
    HEAP32[$210>>2] = $646;
    $647 = HEAP32[$210>>2]|0;
    HEAP32[$207>>2] = $645;
    HEAP32[$208>>2] = $647;
    $648 = HEAP32[$207>>2]|0;
    $649 = HEAP32[$208>>2]|0;
    HEAP32[$206>>2] = $649;
    $650 = HEAP32[$206>>2]|0;
    __ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEE16__construct_nodeIJNS_4pairIPS5_SM_EEEEENS_10unique_ptrINS_11__hash_nodeIS8_PvEENS_22__hash_node_destructorINSH_ISR_EEEEEEDpOT_($209,$648,$650);
    HEAP32[$178>>2] = $209;
    $651 = HEAP32[$178>>2]|0;
    HEAP32[$177>>2] = $651;
    $652 = HEAP32[$177>>2]|0;
    HEAP32[$176>>2] = $652;
    $653 = HEAP32[$176>>2]|0;
    $654 = HEAP32[$653>>2]|0;
    __ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEE20__node_insert_uniqueEPNS_11__hash_nodeIS8_PvEE($228,$648,$654);
    $655 = ((($228)) + 4|0);
    $656 = HEAP8[$655>>0]|0;
    $657 = $656&1;
    if ($657) {
     HEAP32[$174>>2] = $209;
     $658 = HEAP32[$174>>2]|0;
     HEAP32[$173>>2] = $658;
     $659 = HEAP32[$173>>2]|0;
     HEAP32[$172>>2] = $659;
     $660 = HEAP32[$172>>2]|0;
     $661 = HEAP32[$660>>2]|0;
     HEAP32[$175>>2] = $661;
     HEAP32[$171>>2] = $658;
     $662 = HEAP32[$171>>2]|0;
     HEAP32[$170>>2] = $662;
     $663 = HEAP32[$170>>2]|0;
     HEAP32[$663>>2] = 0;
    }
    HEAP32[$205>>2] = $209;
    $664 = HEAP32[$205>>2]|0;
    HEAP32[$202>>2] = $664;
    HEAP32[$203>>2] = 0;
    $665 = HEAP32[$202>>2]|0;
    HEAP32[$201>>2] = $665;
    $666 = HEAP32[$201>>2]|0;
    HEAP32[$200>>2] = $666;
    $667 = HEAP32[$200>>2]|0;
    $668 = HEAP32[$667>>2]|0;
    HEAP32[$204>>2] = $668;
    $669 = HEAP32[$203>>2]|0;
    HEAP32[$182>>2] = $665;
    $670 = HEAP32[$182>>2]|0;
    HEAP32[$181>>2] = $670;
    $671 = HEAP32[$181>>2]|0;
    HEAP32[$671>>2] = $669;
    $672 = HEAP32[$204>>2]|0;
    $673 = ($672|0)!=(0|0);
    if ($673) {
     HEAP32[$180>>2] = $665;
     $674 = HEAP32[$180>>2]|0;
     $675 = ((($674)) + 4|0);
     HEAP32[$179>>2] = $675;
     $676 = HEAP32[$179>>2]|0;
     $677 = HEAP32[$204>>2]|0;
     HEAP32[$198>>2] = $676;
     HEAP32[$199>>2] = $677;
     $678 = HEAP32[$198>>2]|0;
     $679 = ((($678)) + 4|0);
     $680 = HEAP8[$679>>0]|0;
     $681 = $680&1;
     if ($681) {
      $682 = HEAP32[$678>>2]|0;
      $683 = HEAP32[$199>>2]|0;
      $684 = ((($683)) + 8|0);
      HEAP32[$197>>2] = $684;
      $685 = HEAP32[$197>>2]|0;
      HEAP32[$196>>2] = $685;
      $686 = HEAP32[$196>>2]|0;
      HEAP32[$193>>2] = $682;
      HEAP32[$194>>2] = $686;
      $687 = HEAP32[$193>>2]|0;
      $688 = HEAP32[$194>>2]|0;
      ;HEAP8[$192>>0]=HEAP8[$195>>0]|0;
      HEAP32[$190>>2] = $687;
      HEAP32[$191>>2] = $688;
     }
     $689 = HEAP32[$199>>2]|0;
     $690 = ($689|0)!=(0|0);
     if ($690) {
      $691 = HEAP32[$678>>2]|0;
      $692 = HEAP32[$199>>2]|0;
      HEAP32[$187>>2] = $691;
      HEAP32[$188>>2] = $692;
      HEAP32[$189>>2] = 1;
      $693 = HEAP32[$187>>2]|0;
      $694 = HEAP32[$188>>2]|0;
      $695 = HEAP32[$189>>2]|0;
      HEAP32[$184>>2] = $693;
      HEAP32[$185>>2] = $694;
      HEAP32[$186>>2] = $695;
      $696 = HEAP32[$185>>2]|0;
      HEAP32[$183>>2] = $696;
      $697 = HEAP32[$183>>2]|0;
      __ZdlPv($697);
     }
    }
    HEAP32[$222>>2] = $421;
    HEAP32[$223>>2] = $228;
    $698 = HEAP32[$222>>2]|0;
    $699 = HEAP32[$223>>2]|0;
    HEAP32[$221>>2] = $699;
    $700 = HEAP32[$221>>2]|0;
    ;HEAP32[$224>>2]=HEAP32[$700>>2]|0;
    ;HEAP8[$219>>0]=HEAP8[$224>>0]|0;HEAP8[$219+1>>0]=HEAP8[$224+1>>0]|0;HEAP8[$219+2>>0]=HEAP8[$224+2>>0]|0;HEAP8[$219+3>>0]=HEAP8[$224+3>>0]|0;
    HEAP32[$218>>2] = $698;
    $701 = HEAP32[$218>>2]|0;
    ;HEAP32[$701>>2]=HEAP32[$219>>2]|0;
    $702 = ((($698)) + 4|0);
    $703 = HEAP32[$223>>2]|0;
    $704 = ((($703)) + 4|0);
    HEAP32[$220>>2] = $704;
    $705 = HEAP32[$220>>2]|0;
    $706 = HEAP8[$705>>0]|0;
    $707 = $706&1;
    $708 = $707&1;
    HEAP8[$702>>0] = $708;
   }
   $709 = HEAP32[$418>>2]|0;
   $710 = (($709) + 1)|0;
   HEAP32[$418>>2] = $710;
  }
  $711 = HEAP32[$405>>2]|0;
  $712 = HEAP32[$711>>2]|0;
  $713 = ((($712)) + 52|0);
  $714 = HEAP32[$713>>2]|0;
  $715 = (FUNCTION_TABLE_iii[$714 & 1023]($711,3)|0);
  __Z17CombineAppearanceRNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEEEPKNS3_10AppearanceE($416,$715);
  $716 = HEAP32[$405>>2]|0;
  $717 = HEAP32[$716>>2]|0;
  $718 = ((($717)) + 52|0);
  $719 = HEAP32[$718>>2]|0;
  $720 = (FUNCTION_TABLE_iii[$719 & 1023]($716,16)|0);
  __Z17CombineAppearanceRNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEEEPKNS3_10AppearanceE($416,$720);
  $721 = HEAP32[$405>>2]|0;
  $722 = HEAP32[$721>>2]|0;
  $723 = ((($722)) + 52|0);
  $724 = HEAP32[$723>>2]|0;
  $725 = (FUNCTION_TABLE_iii[$724 & 1023]($721,2)|0);
  __Z17CombineAppearanceRNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEEEPKNS3_10AppearanceE($416,$725);
  $726 = HEAP32[$405>>2]|0;
  $727 = HEAP32[$726>>2]|0;
  $728 = ((($727)) + 52|0);
  $729 = HEAP32[$728>>2]|0;
  $730 = (FUNCTION_TABLE_iii[$729 & 1023]($726,15)|0);
  __Z17CombineAppearanceRNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEEEPKNS3_10AppearanceE($416,$730);
 }
 $731 = HEAP32[$405>>2]|0;
 $732 = HEAP32[$731>>2]|0;
 $733 = ((($732)) + 100|0);
 $734 = HEAP32[$733>>2]|0;
 $735 = (FUNCTION_TABLE_ii[$734 & 2047]($731)|0);
 HEAP32[$422>>2] = $735;
 HEAP32[$423>>2] = 0;
 while(1) {
  $736 = HEAP32[$423>>2]|0;
  $737 = HEAP32[$414>>2]|0;
  $738 = ($736>>>0)<($737>>>0);
  if (!($738)) {
   break;
  }
  $739 = HEAP32[$415>>2]|0;
  $740 = HEAP32[$423>>2]|0;
  $741 = (($739) + ($740<<2)|0);
  HEAP32[$424>>2] = $741;
  $742 = HEAP32[$424>>2]|0;
  $743 = (__ZNK3sap2ve4core7SubMesh11GetMaterialEv($742)|0);
  HEAP32[$425>>2] = $743;
  $744 = HEAP32[$417>>2]|0;
  $745 = ($744|0)!=(0|0);
  if ($745) {
   HEAP32[$167>>2] = $416;
   HEAP32[$168>>2] = $425;
   $746 = HEAP32[$167>>2]|0;
   $747 = HEAP32[$168>>2]|0;
   $748 = (__ZNSt3__212__hash_tableINS_17__hash_value_typeIPKN3sap2ve4core8MaterialES7_EENS_22__unordered_map_hasherIS7_S8_NS_4hashIS7_EELb1EEENS_21__unordered_map_equalIS7_S8_NS_8equal_toIS7_EELb1EEENS_9allocatorIS8_EEE4findIS7_EENS_15__hash_iteratorIPNS_11__hash_nodeIS8_PvEEEERKT_($746,$747)|0);
   HEAP32[$169>>2] = $748;
   ;HEAP8[$165>>0]=HEAP8[$169>>0]|0;HEAP8[$165+1>>0]=HEAP8[$169+1>>0]|0;HEAP8[$165+2>>0]=HEAP8[$169+2>>0]|0;HEAP8[$165+3>>0]=HEAP8[$169+3>>0]|0;
   HEAP32[$164>>2] = $166;
   $749 = HEAP32[$164>>2]|0;
   ;HEAP32[$749>>2]=HEAP32[$165>>2]|0;
   $750 = HEAP32[$166>>2]|0;
   HEAP32[$426>>2] = $750;
   HEAP32[$162>>2] = $416;
   $751 = HEAP32[$162>>2]|0;
   HEAP32[$160>>2] = $751;
   HEAP32[$157>>2] = $159;
   HEAP32[$158>>2] = 0;
   $752 = HEAP32[$157>>2]|0;
   $753 = HEAP32[$158>>2]|0;
   HEAP32[$752>>2] = $753;
   $754 = HEAP32[$159>>2]|0;
   HEAP32[$163>>2] = $754;
   ;HEAP8[$156>>0]=HEAP8[$163>>0]|0;HEAP8[$156+1>>0]=HEAP8[$163+1>>0]|0;HEAP8[$156+2>>0]=HEAP8[$163+2>>0]|0;HEAP8[$156+3>>0]=HEAP8[$163+3>>0]|0;
   HEAP32[$155>>2] = $161;
   $755 = HEAP32[$155>>2]|0;
   ;HEAP32[$755>>2]=HEAP32[$156>>2]|0;
   $756 = HEAP32[$161>>2]|0;
   HEAP32[$427>>2] = $756;
   HEAP32[$153>>2] = $426;
   HEAP32[$154>>2] = $427;
   $757 = HEAP32[$153>>2]|0;
   $758 = HEAP32[$154>>2]|0;
   HEAP32[$151>>2] = $757;
   HEAP32[$152>>2] = $758;
   $759 = HEAP32[$151>>2]|0;
   $760 = HEAP32[$152>>2]|0;
   HEAP32[$149>>2] = $759;
   HEAP32[$150>>2] = $760;
   $761 = HEAP32[$149>>2]|0;
   $762 = HEAP32[$761>>2]|0;
   $763 = HEAP32[$150>>2]|0;
   $764 = HEAP32[$763>>2]|0;
   $765 = ($762|0)==($764|0);
   $766 = $765 ^ 1;
   if ($766) {
    HEAP32[$148>>2] = $426;
    $767 = HEAP32[$148>>2]|0;
    HEAP32[$147>>2] = $767;
    $768 = HEAP32[$147>>2]|0;
    $769 = HEAP32[$768>>2]|0;
    HEAP32[$146>>2] = $769;
    $770 = HEAP32[$146>>2]|0;
    HEAP32[$145>>2] = $770;
    $771 = HEAP32[$145>>2]|0;
    HEAP32[$144>>2] = $771;
    $772 = HEAP32[$144>>2]|0;
    $773 = ((($772)) + 8|0);
    HEAP32[$143>>2] = $773;
    $774 = HEAP32[$143>>2]|0;
    HEAP32[$142>>2] = $774;
    $775 = HEAP32[$142>>2]|0;
    HEAP32[$141>>2] = $775;
    $776 = HEAP32[$141>>2]|0;
    HEAP32[$140>>2] = $776;
    $777 = HEAP32[$140>>2]|0;
    $778 = ((($777)) + 4|0);
    $779 = HEAP32[$778>>2]|0;
    HEAP32[$425>>2] = $779;
   }
  }
  $780 = HEAP32[$425>>2]|0;
  $781 = (__ZN17DataModelImporter19GetOrCreateMaterialEPKN3sap2ve4core8MaterialE($457,$780)|0);
  HEAP32[$428>>2] = $781;
  $782 = HEAP32[$424>>2]|0;
  $783 = (__ZNK3sap2ve4core7SubMesh14GetBoundingBoxEv($782)|0);
  HEAP32[$429>>2] = $783;
  $784 = HEAP32[$429>>2]|0;
  $785 = (__ZNK3sap2ve4core11BoundingBox6GetMinEv($784)|0);
  HEAP32[$430>>2] = $785;
  $786 = HEAP32[$429>>2]|0;
  $787 = (__ZNK3sap2ve4core11BoundingBox6GetMaxEv($786)|0);
  HEAP32[$431>>2] = $787;
  $788 = HEAP32[$430>>2]|0;
  $789 = +HEAPF32[$788>>2];
  HEAPF32[$432>>2] = $789;
  $790 = ((($432)) + 4|0);
  $791 = HEAP32[$430>>2]|0;
  $792 = ((($791)) + 4|0);
  $793 = +HEAPF32[$792>>2];
  HEAPF32[$790>>2] = $793;
  $794 = ((($790)) + 4|0);
  $795 = HEAP32[$430>>2]|0;
  $796 = ((($795)) + 8|0);
  $797 = +HEAPF32[$796>>2];
  HEAPF32[$794>>2] = $797;
  $798 = ((($794)) + 4|0);
  $799 = HEAP32[$431>>2]|0;
  $800 = +HEAPF32[$799>>2];
  HEAPF32[$798>>2] = $800;
  $801 = ((($798)) + 4|0);
  $802 = HEAP32[$431>>2]|0;
  $803 = ((($802)) + 4|0);
  $804 = +HEAPF32[$803>>2];
  HEAPF32[$801>>2] = $804;
  $805 = ((($801)) + 4|0);
  $806 = HEAP32[$431>>2]|0;
  $807 = ((($806)) + 8|0);
  $808 = +HEAPF32[$807>>2];
  HEAPF32[$805>>2] = $808;
  $809 = ((($457)) + 24|0);
  $810 = HEAP32[$809>>2]|0;
  $811 = HEAP32[$428>>2]|0;
  $812 = HEAP32[$422>>2]|0;
  $813 = (__ZNK3sap2ve6Matrix4DataEv($812)|0);
  $814 = HEAP32[$424>>2]|0;
  _SceneBuilderAdapter_createMesh(($810|0),($811|0),($432|0),($813|0),($814|0));
  $815 = ((($457)) + 24|0);
  $816 = HEAP32[$815>>2]|0;
  $817 = HEAP32[$405>>2]|0;
  $818 = HEAP32[$424>>2]|0;
  _SceneBuilderAdapter_insertMesh(($816|0),($817|0),($818|0));
  $819 = HEAP8[$406>>0]|0;
  $820 = $819&1;
  $821 = HEAP32[$424>>2]|0;
  do {
   if ($820) {
    $822 = (__ZNK3sap2ve4core7SubMesh30GetUncompressedGeometryStorageEv($821)|0);
    HEAP32[$433>>2] = $822;
    $823 = HEAP32[$433>>2]|0;
    $824 = ($823|0)!=(0|0);
    if ($824) {
     $825 = HEAP32[$424>>2]|0;
     $826 = (__ZNK3sap2ve4core7SubMesh8GetFlagsEv($825)|0);
     HEAP16[$434>>1] = $826;
     HEAP32[$435>>2] = 12;
     $827 = HEAP32[$424>>2]|0;
     $828 = (__ZNK3sap2ve4core7SubMesh14TextureUVCountEv($827)|0);
     $829 = ($828|0)!=(0);
     if ($829) {
      $830 = HEAP32[$435>>2]|0;
      $831 = (($830) + 8)|0;
      HEAP32[$435>>2] = $831;
     }
     $832 = HEAP32[$424>>2]|0;
     $833 = (__ZNK3sap2ve4core7SubMesh10HasNormalsEv($832)|0);
     if ($833) {
      $834 = HEAP32[$435>>2]|0;
      $835 = (($834) + 12)|0;
      HEAP32[$435>>2] = $835;
     }
     $836 = HEAP32[$424>>2]|0;
     $837 = (__ZNK3sap2ve4core7SubMesh13GetSubMeshLODEj($836,0)|0);
     HEAP32[$436>>2] = $837;
     $838 = ((($457)) + 24|0);
     $839 = HEAP32[$838>>2]|0;
     $840 = HEAP16[$434>>1]|0;
     $841 = $840&65535;
     $842 = HEAP32[$436>>2]|0;
     $843 = (__ZNK3sap2ve4core11sSubMeshLOD13GetPointCountEv($842)|0);
     $844 = $843&65535;
     $845 = HEAP32[$436>>2]|0;
     $846 = (__ZNK3sap2ve4core11sSubMeshLOD16GetTriangleCountEv($845)|0);
     $847 = $846&65535;
     $848 = HEAP32[$436>>2]|0;
     $849 = (__ZNK3sap2ve4core11sSubMeshLOD13GetSplatCountEv($848)|0);
     $850 = $849&65535;
     $851 = HEAP32[$433>>2]|0;
     $852 = HEAP32[$851>>2]|0;
     $853 = ((($852)) + 8|0);
     $854 = HEAP32[$853>>2]|0;
     $855 = (FUNCTION_TABLE_ii[$854 & 2047]($851)|0);
     $856 = HEAP32[$433>>2]|0;
     $857 = HEAP32[$856>>2]|0;
     $858 = ((($857)) + 12|0);
     $859 = HEAP32[$858>>2]|0;
     $860 = (FUNCTION_TABLE_ii[$859 & 2047]($856)|0);
     $861 = HEAP32[$436>>2]|0;
     $862 = (__ZNK3sap2ve4core11sSubMeshLOD13GetPointCountEv($861)|0);
     $863 = $862&65535;
     $864 = HEAP32[$436>>2]|0;
     $865 = (__ZNK3sap2ve4core11sSubMeshLOD13GetSplatCountEv($864)|0);
     $866 = $865&65535;
     $867 = (($863) + ($866))|0;
     $868 = HEAP32[$435>>2]|0;
     $869 = Math_imul($867, $868)|0;
     $870 = HEAP32[$436>>2]|0;
     $871 = (__ZNK3sap2ve4core11sSubMeshLOD16GetTriangleCountEv($870)|0);
     $872 = $871&65535;
     $873 = $872<<1;
     $874 = HEAP32[$424>>2]|0;
     $875 = (__ZNK3sap2ve4core7SubMesh10IsPolylineEv($874)|0);
     $876 = $875 ? 2 : 3;
     $877 = Math_imul($873, $876)|0;
     $878 = HEAP32[$435>>2]|0;
     $879 = $878 >> 2;
     $880 = HEAP32[$424>>2]|0;
     _SceneBuilderAdapter_setMeshGeometry(($839|0),($841|0),($844|0),($847|0),($850|0),($855|0),($860|0),($869|0),($877|0),($879|0),($880|0));
    }
   } else {
    HEAP32[$437>>2] = $821;
    HEAP32[$137>>2] = $413;
    HEAP32[$138>>2] = $437;
    $881 = HEAP32[$137>>2]|0;
    $882 = ((($881)) + 4|0);
    $883 = HEAP32[$882>>2]|0;
    HEAP32[$136>>2] = $881;
    $884 = HEAP32[$136>>2]|0;
    $885 = ((($884)) + 8|0);
    HEAP32[$135>>2] = $885;
    $886 = HEAP32[$135>>2]|0;
    HEAP32[$134>>2] = $886;
    $887 = HEAP32[$134>>2]|0;
    $888 = HEAP32[$887>>2]|0;
    $889 = ($883>>>0)<($888>>>0);
    if ($889) {
     HEAP32[$131>>2] = $139;
     HEAP32[$132>>2] = $881;
     HEAP32[$133>>2] = 1;
     HEAP32[$112>>2] = $881;
     $890 = HEAP32[$112>>2]|0;
     $891 = ((($890)) + 8|0);
     HEAP32[$111>>2] = $891;
     $892 = HEAP32[$111>>2]|0;
     HEAP32[$110>>2] = $892;
     $893 = HEAP32[$110>>2]|0;
     $894 = ((($881)) + 4|0);
     $895 = HEAP32[$894>>2]|0;
     HEAP32[$113>>2] = $895;
     $896 = HEAP32[$113>>2]|0;
     $897 = HEAP32[$138>>2]|0;
     HEAP32[$114>>2] = $897;
     $898 = HEAP32[$114>>2]|0;
     HEAP32[$125>>2] = $893;
     HEAP32[$126>>2] = $896;
     HEAP32[$127>>2] = $898;
     $899 = HEAP32[$125>>2]|0;
     $900 = HEAP32[$126>>2]|0;
     $901 = HEAP32[$127>>2]|0;
     HEAP32[$124>>2] = $901;
     $902 = HEAP32[$124>>2]|0;
     ;HEAP8[$123>>0]=HEAP8[$128>>0]|0;
     HEAP32[$120>>2] = $899;
     HEAP32[$121>>2] = $900;
     HEAP32[$122>>2] = $902;
     $903 = HEAP32[$120>>2]|0;
     $904 = HEAP32[$121>>2]|0;
     $905 = HEAP32[$122>>2]|0;
     HEAP32[$119>>2] = $905;
     $906 = HEAP32[$119>>2]|0;
     HEAP32[$116>>2] = $903;
     HEAP32[$117>>2] = $904;
     HEAP32[$118>>2] = $906;
     $907 = HEAP32[$117>>2]|0;
     $908 = HEAP32[$118>>2]|0;
     HEAP32[$115>>2] = $908;
     $909 = HEAP32[$115>>2]|0;
     $910 = HEAP32[$909>>2]|0;
     HEAP32[$907>>2] = $910;
     HEAP32[$129>>2] = $139;
     $911 = ((($881)) + 4|0);
     $912 = HEAP32[$911>>2]|0;
     $913 = ((($912)) + 4|0);
     HEAP32[$911>>2] = $913;
    } else {
     $914 = HEAP32[$138>>2]|0;
     HEAP32[$130>>2] = $914;
     $915 = HEAP32[$130>>2]|0;
     __ZNSt3__26vectorIPKN3sap2ve4core7SubMeshENS_9allocatorIS6_EEE21__push_back_slow_pathIS6_EEvOT_($881,$915);
    }
    $916 = ((($457)) + 44|0);
    $917 = HEAP32[$424>>2]|0;
    HEAP32[$439>>2] = $917;
    HEAP32[$107>>2] = $916;
    HEAP32[$108>>2] = $439;
    $918 = HEAP32[$107>>2]|0;
    $919 = HEAP32[$108>>2]|0;
    $920 = (__ZNSt3__26__treeINS_12__value_typeIPKN3sap2ve4core7SubMeshENS_10unique_ptrI13sOnDemandMeshNS_14default_deleteIS9_EEEEEENS_19__map_value_compareIS7_SD_NS_4lessIS7_EELb1EEENS_9allocatorISD_EEE4findIS7_EENS_15__tree_iteratorISD_PNS_11__tree_nodeISD_PvEElEERKT_($918,$919)|0);
    HEAP32[$109>>2] = $920;
    ;HEAP8[$105>>0]=HEAP8[$109>>0]|0;HEAP8[$105+1>>0]=HEAP8[$109+1>>0]|0;HEAP8[$105+2>>0]=HEAP8[$109+2>>0]|0;HEAP8[$105+3>>0]=HEAP8[$109+3>>0]|0;
    HEAP32[$104>>2] = $106;
    $921 = HEAP32[$104>>2]|0;
    ;HEAP32[$921>>2]=HEAP32[$105>>2]|0;
    $922 = HEAP32[$106>>2]|0;
    HEAP32[$438>>2] = $922;
    $923 = ((($457)) + 44|0);
    HEAP32[$102>>2] = $923;
    $924 = HEAP32[$102>>2]|0;
    HEAP32[$100>>2] = $924;
    $925 = HEAP32[$100>>2]|0;
    HEAP32[$98>>2] = $925;
    $926 = HEAP32[$98>>2]|0;
    $927 = ((($926)) + 4|0);
    HEAP32[$97>>2] = $927;
    $928 = HEAP32[$97>>2]|0;
    HEAP32[$96>>2] = $928;
    $929 = HEAP32[$96>>2]|0;
    HEAP32[$95>>2] = $929;
    $930 = HEAP32[$95>>2]|0;
    HEAP32[$94>>2] = $930;
    $931 = HEAP32[$94>>2]|0;
    HEAP32[$92>>2] = $99;
    HEAP32[$93>>2] = $931;
    $932 = HEAP32[$92>>2]|0;
    $933 = HEAP32[$93>>2]|0;
    HEAP32[$932>>2] = $933;
    $934 = HEAP32[$99>>2]|0;
    HEAP32[$103>>2] = $934;
    ;HEAP8[$91>>0]=HEAP8[$103>>0]|0;HEAP8[$91+1>>0]=HEAP8[$103+1>>0]|0;HEAP8[$91+2>>0]=HEAP8[$103+2>>0]|0;HEAP8[$91+3>>0]=HEAP8[$103+3>>0]|0;
    HEAP32[$90>>2] = $101;
    $935 = HEAP32[$90>>2]|0;
    ;HEAP32[$935>>2]=HEAP32[$91>>2]|0;
    $936 = HEAP32[$101>>2]|0;
    HEAP32[$440>>2] = $936;
    HEAP32[$73>>2] = $438;
    HEAP32[$74>>2] = $440;
    $937 = HEAP32[$73>>2]|0;
    $938 = HEAP32[$74>>2]|0;
    HEAP32[$71>>2] = $937;
    HEAP32[$72>>2] = $938;
    $939 = HEAP32[$71>>2]|0;
    $940 = HEAP32[$72>>2]|0;
    HEAP32[$69>>2] = $939;
    HEAP32[$70>>2] = $940;
    $941 = HEAP32[$69>>2]|0;
    $942 = HEAP32[$941>>2]|0;
    $943 = HEAP32[$70>>2]|0;
    $944 = HEAP32[$943>>2]|0;
    $945 = ($942|0)==($944|0);
    $946 = $945 ^ 1;
    if ($946) {
     HEAP32[$13>>2] = $438;
     $947 = HEAP32[$13>>2]|0;
     HEAP32[$12>>2] = $947;
     $948 = HEAP32[$12>>2]|0;
     HEAP32[$11>>2] = $948;
     $949 = HEAP32[$11>>2]|0;
     $950 = HEAP32[$949>>2]|0;
     $951 = ((($950)) + 16|0);
     HEAP32[$10>>2] = $951;
     $952 = HEAP32[$10>>2]|0;
     HEAP32[$9>>2] = $952;
     $953 = HEAP32[$9>>2]|0;
     HEAP32[$8>>2] = $953;
     $954 = HEAP32[$8>>2]|0;
     HEAP32[$7>>2] = $954;
     $955 = HEAP32[$7>>2]|0;
     $956 = ((($955)) + 4|0);
     HEAP32[$6>>2] = $956;
     $957 = HEAP32[$6>>2]|0;
     HEAP32[$5>>2] = $957;
     $958 = HEAP32[$5>>2]|0;
     HEAP32[$4>>2] = $958;
     $959 = HEAP32[$4>>2]|0;
     $960 = HEAP32[$959>>2]|0;
     $961 = ((($960)) + 24|0);
     $962 = HEAP32[$961>>2]|0;
     $963 = (($962) + 1)|0;
     HEAP32[$961>>2] = $963;
     break;
    }
    $964 = HEAP32[$424>>2]|0;
    $965 = (__ZNK3sap2ve4core7SubMesh13GetSubMeshLODEj($964,0)|0);
    HEAP32[$441>>2] = $965;
    $966 = ((($457)) + 44|0);
    $967 = HEAP32[$424>>2]|0;
    HEAP32[$443>>2] = $967;
    $968 = HEAP32[$424>>2]|0;
    HEAP32[$445>>2] = $968;
    HEAP32[$446>>2] = 0;
    $969 = HEAP32[$441>>2]|0;
    $970 = (__ZNK3sap2ve4core11sSubMeshLOD16GetGeometryTableEv($969)|0);
    HEAP32[$447>>2] = $970;
    $971 = HEAP32[$441>>2]|0;
    $972 = (__ZNK3sap2ve4core11sSubMeshLOD16GetGeometryIndexEv($971)|0);
    HEAP32[$448>>2] = $972;
    $973 = HEAP32[$441>>2]|0;
    $974 = (__ZNK3sap2ve4core11sSubMeshLOD13GetPointCountEv($973)|0);
    HEAP16[$449>>1] = $974;
    $975 = HEAP32[$441>>2]|0;
    $976 = (__ZNK3sap2ve4core11sSubMeshLOD16GetTriangleCountEv($975)|0);
    HEAP16[$450>>1] = $976;
    $977 = HEAP32[$441>>2]|0;
    $978 = (__ZNK3sap2ve4core11sSubMeshLOD13GetSplatCountEv($977)|0);
    HEAP16[$451>>1] = $978;
    $979 = HEAP32[$441>>2]|0;
    $980 = (__ZNK3sap2ve4core11sSubMeshLOD8GetFlagsEv($979)|0);
    HEAP16[$452>>1] = $980;
    HEAP32[$31>>2] = $445;
    HEAP32[$32>>2] = $446;
    HEAP32[$33>>2] = $447;
    HEAP32[$34>>2] = $448;
    HEAP32[$35>>2] = $449;
    HEAP32[$36>>2] = $450;
    HEAP32[$37>>2] = $451;
    HEAP32[$38>>2] = $452;
    HEAP32[$39>>2] = $432;
    $981 = (__Znwm(56)|0);
    $982 = HEAP32[$31>>2]|0;
    HEAP32[$30>>2] = $982;
    $983 = HEAP32[$30>>2]|0;
    $984 = HEAP32[$983>>2]|0;
    $985 = HEAP32[$32>>2]|0;
    HEAP32[$21>>2] = $985;
    $986 = HEAP32[$21>>2]|0;
    $987 = HEAP32[$986>>2]|0;
    $988 = HEAP32[$33>>2]|0;
    HEAP32[$14>>2] = $988;
    $989 = HEAP32[$14>>2]|0;
    $990 = HEAP32[$989>>2]|0;
    $991 = HEAP32[$34>>2]|0;
    HEAP32[$15>>2] = $991;
    $992 = HEAP32[$15>>2]|0;
    $993 = HEAP32[$992>>2]|0;
    $994 = HEAP32[$35>>2]|0;
    HEAP32[$16>>2] = $994;
    $995 = HEAP32[$16>>2]|0;
    $996 = HEAP16[$995>>1]|0;
    $997 = HEAP32[$36>>2]|0;
    HEAP32[$17>>2] = $997;
    $998 = HEAP32[$17>>2]|0;
    $999 = HEAP16[$998>>1]|0;
    $1000 = HEAP32[$37>>2]|0;
    HEAP32[$18>>2] = $1000;
    $1001 = HEAP32[$18>>2]|0;
    $1002 = HEAP16[$1001>>1]|0;
    $1003 = HEAP32[$38>>2]|0;
    HEAP32[$19>>2] = $1003;
    $1004 = HEAP32[$19>>2]|0;
    $1005 = HEAP16[$1004>>1]|0;
    $1006 = HEAP32[$39>>2]|0;
    HEAP32[$20>>2] = $1006;
    $1007 = HEAP32[$20>>2]|0;
    __ZN13sOnDemandMeshC2EPKvjjjttttPKf($981,$984,$987,$990,$993,$996,$999,$1002,$1005,$1007);
    HEAP32[$28>>2] = $444;
    HEAP32[$29>>2] = $981;
    $1008 = HEAP32[$28>>2]|0;
    HEAP32[$26>>2] = $1008;
    HEAP32[$27>>2] = $29;
    $1009 = HEAP32[$26>>2]|0;
    $1010 = HEAP32[$27>>2]|0;
    HEAP32[$25>>2] = $1010;
    $1011 = HEAP32[$25>>2]|0;
    HEAP32[$23>>2] = $1009;
    HEAP32[$24>>2] = $1011;
    $1012 = HEAP32[$23>>2]|0;
    $1013 = HEAP32[$24>>2]|0;
    HEAP32[$22>>2] = $1013;
    $1014 = HEAP32[$22>>2]|0;
    $1015 = HEAP32[$1014>>2]|0;
    HEAP32[$1012>>2] = $1015;
    HEAP32[$66>>2] = $442;
    HEAP32[$67>>2] = $443;
    HEAP32[$68>>2] = $444;
    $1016 = HEAP32[$66>>2]|0;
    $1017 = HEAP32[$67>>2]|0;
    HEAP32[$65>>2] = $1017;
    $1018 = HEAP32[$65>>2]|0;
    $1019 = HEAP32[$1018>>2]|0;
    HEAP32[$1016>>2] = $1019;
    $1020 = ((($1016)) + 4|0);
    $1021 = HEAP32[$68>>2]|0;
    HEAP32[$40>>2] = $1021;
    $1022 = HEAP32[$40>>2]|0;
    HEAP32[$62>>2] = $1020;
    HEAP32[$63>>2] = $1022;
    $1023 = HEAP32[$62>>2]|0;
    $1024 = HEAP32[$63>>2]|0;
    HEAP32[$60>>2] = $1024;
    $1025 = HEAP32[$60>>2]|0;
    HEAP32[$59>>2] = $1025;
    $1026 = HEAP32[$59>>2]|0;
    HEAP32[$58>>2] = $1026;
    $1027 = HEAP32[$58>>2]|0;
    $1028 = HEAP32[$1027>>2]|0;
    HEAP32[$61>>2] = $1028;
    HEAP32[$57>>2] = $1025;
    $1029 = HEAP32[$57>>2]|0;
    HEAP32[$56>>2] = $1029;
    $1030 = HEAP32[$56>>2]|0;
    HEAP32[$1030>>2] = 0;
    $1031 = HEAP32[$61>>2]|0;
    HEAP32[$64>>2] = $1031;
    $1032 = HEAP32[$63>>2]|0;
    HEAP32[$55>>2] = $1032;
    $1033 = HEAP32[$55>>2]|0;
    HEAP32[$54>>2] = $1033;
    $1034 = HEAP32[$54>>2]|0;
    HEAP32[$53>>2] = $1034;
    $1035 = HEAP32[$53>>2]|0;
    HEAP32[$41>>2] = $1035;
    $1036 = HEAP32[$41>>2]|0;
    HEAP32[$50>>2] = $1023;
    HEAP32[$51>>2] = $64;
    HEAP32[$52>>2] = $1036;
    $1037 = HEAP32[$50>>2]|0;
    $1038 = HEAP32[$51>>2]|0;
    HEAP32[$49>>2] = $1038;
    $1039 = HEAP32[$49>>2]|0;
    HEAP32[$43>>2] = $1037;
    HEAP32[$44>>2] = $1039;
    $1040 = HEAP32[$43>>2]|0;
    $1041 = HEAP32[$44>>2]|0;
    HEAP32[$42>>2] = $1041;
    $1042 = HEAP32[$42>>2]|0;
    $1043 = HEAP32[$1042>>2]|0;
    HEAP32[$1040>>2] = $1043;
    $1044 = HEAP32[$52>>2]|0;
    HEAP32[$45>>2] = $1044;
    $1045 = HEAP32[$45>>2]|0;
    HEAP32[$47>>2] = $1037;
    HEAP32[$48>>2] = $1045;
    $1046 = HEAP32[$48>>2]|0;
    HEAP32[$46>>2] = $1046;
    HEAP32[$87>>2] = $966;
    HEAP32[$88>>2] = $442;
    $1047 = HEAP32[$87>>2]|0;
    $1048 = HEAP32[$88>>2]|0;
    HEAP32[$86>>2] = $1048;
    $1049 = HEAP32[$86>>2]|0;
    HEAP32[$77>>2] = $1047;
    HEAP32[$78>>2] = $1049;
    $1050 = HEAP32[$77>>2]|0;
    $1051 = HEAP32[$78>>2]|0;
    HEAP32[$76>>2] = $1051;
    $1052 = HEAP32[$76>>2]|0;
    $1053 = HEAP32[$78>>2]|0;
    HEAP32[$75>>2] = $1053;
    $1054 = HEAP32[$75>>2]|0;
    __ZNSt3__26__treeINS_12__value_typeIPKN3sap2ve4core7SubMeshENS_10unique_ptrI13sOnDemandMeshNS_14default_deleteIS9_EEEEEENS_19__map_value_compareIS7_SD_NS_4lessIS7_EELb1EEENS_9allocatorISD_EEE25__emplace_unique_key_argsIS7_JNS_4pairIKS7_SC_EEEEENSM_INS_15__tree_iteratorISD_PNS_11__tree_nodeISD_PvEElEEbEERKT_DpOT0_($89,$1050,$1052,$1054);
    HEAP32[$83>>2] = $453;
    HEAP32[$84>>2] = $89;
    $1055 = HEAP32[$83>>2]|0;
    $1056 = HEAP32[$84>>2]|0;
    HEAP32[$82>>2] = $1056;
    $1057 = HEAP32[$82>>2]|0;
    ;HEAP32[$85>>2]=HEAP32[$1057>>2]|0;
    ;HEAP8[$80>>0]=HEAP8[$85>>0]|0;HEAP8[$80+1>>0]=HEAP8[$85+1>>0]|0;HEAP8[$80+2>>0]=HEAP8[$85+2>>0]|0;HEAP8[$80+3>>0]=HEAP8[$85+3>>0]|0;
    HEAP32[$79>>2] = $1055;
    $1058 = HEAP32[$79>>2]|0;
    ;HEAP32[$1058>>2]=HEAP32[$80>>2]|0;
    $1059 = ((($1055)) + 4|0);
    $1060 = HEAP32[$84>>2]|0;
    $1061 = ((($1060)) + 4|0);
    HEAP32[$81>>2] = $1061;
    $1062 = HEAP32[$81>>2]|0;
    $1063 = HEAP8[$1062>>0]|0;
    $1064 = $1063&1;
    $1065 = $1064&1;
    HEAP8[$1059>>0] = $1065;
    __ZNSt3__24pairIKPKN3sap2ve4core7SubMeshENS_10unique_ptrI13sOnDemandMeshNS_14default_deleteIS9_EEEEED2Ev($442);
    HEAP32[$249>>2] = $444;
    $1066 = HEAP32[$249>>2]|0;
    HEAP32[$246>>2] = $1066;
    HEAP32[$247>>2] = 0;
    $1067 = HEAP32[$246>>2]|0;
    HEAP32[$245>>2] = $1067;
    $1068 = HEAP32[$245>>2]|0;
    HEAP32[$244>>2] = $1068;
    $1069 = HEAP32[$244>>2]|0;
    $1070 = HEAP32[$1069>>2]|0;
    HEAP32[$248>>2] = $1070;
    $1071 = HEAP32[$247>>2]|0;
    HEAP32[$241>>2] = $1067;
    $1072 = HEAP32[$241>>2]|0;
    HEAP32[$240>>2] = $1072;
    $1073 = HEAP32[$240>>2]|0;
    HEAP32[$1073>>2] = $1071;
    $1074 = HEAP32[$248>>2]|0;
    $1075 = ($1074|0)!=(0|0);
    if ($1075) {
     HEAP32[$239>>2] = $1067;
     $1076 = HEAP32[$239>>2]|0;
     HEAP32[$238>>2] = $1076;
     $1077 = HEAP32[$238>>2]|0;
     $1078 = HEAP32[$248>>2]|0;
     HEAP32[$242>>2] = $1077;
     HEAP32[$243>>2] = $1078;
     $1079 = HEAP32[$243>>2]|0;
     $1080 = ($1079|0)==(0|0);
     if (!($1080)) {
      __ZdlPv($1079);
     }
    }
   }
  } while(0);
  $1081 = HEAP32[$423>>2]|0;
  $1082 = (($1081) + 1)|0;
  HEAP32[$423>>2] = $1082;
 }
 $1083 = HEAP8[$406>>0]|0;
 $1084 = $1083&1;
 if (!($1084)) {
  $1085 = ((($457)) + 32|0);
  HEAP32[$314>>2] = $404;
  HEAP32[$315>>2] = $413;
  $1086 = HEAP32[$314>>2]|0;
  HEAP32[$313>>2] = $1086;
  $1087 = HEAP32[$313>>2]|0;
  $1088 = HEAP32[$315>>2]|0;
  HEAP32[$307>>2] = $1088;
  $1089 = HEAP32[$307>>2]|0;
  HEAP32[$310>>2] = $454;
  HEAP32[$311>>2] = $1087;
  HEAP32[$312>>2] = $1089;
  $1090 = HEAP32[$310>>2]|0;
  $1091 = HEAP32[$311>>2]|0;
  HEAP32[$309>>2] = $1091;
  $1092 = HEAP32[$309>>2]|0;
  $1093 = HEAP32[$1092>>2]|0;
  HEAP32[$1090>>2] = $1093;
  $1094 = ((($1090)) + 4|0);
  $1095 = HEAP32[$312>>2]|0;
  HEAP32[$308>>2] = $1095;
  $1096 = HEAP32[$308>>2]|0;
  __ZNSt3__26vectorIPKN3sap2ve4core7SubMeshENS_9allocatorIS6_EEEC2ERKS9_($1094,$1096);
  HEAP32[$333>>2] = $1085;
  HEAP32[$334>>2] = $454;
  $1097 = HEAP32[$333>>2]|0;
  $1098 = HEAP32[$334>>2]|0;
  HEAP32[$332>>2] = $1098;
  $1099 = HEAP32[$332>>2]|0;
  HEAP32[$322>>2] = $1097;
  HEAP32[$323>>2] = $1099;
  $1100 = HEAP32[$322>>2]|0;
  $1101 = HEAP32[$323>>2]|0;
  HEAP32[$321>>2] = $1101;
  $1102 = HEAP32[$321>>2]|0;
  ;HEAP8[$320>>0]=HEAP8[$324>>0]|0;
  HEAP32[$318>>2] = $1100;
  HEAP32[$319>>2] = $1102;
  $1103 = HEAP32[$318>>2]|0;
  $1104 = HEAP32[$319>>2]|0;
  $1105 = HEAP32[$319>>2]|0;
  HEAP32[$317>>2] = $1105;
  $1106 = HEAP32[$317>>2]|0;
  __ZNSt3__26__treeINS_12__value_typeIPKN3sap2ve4core4MeshENS_6vectorIPKNS4_7SubMeshENS_9allocatorISB_EEEEEENS_19__map_value_compareIS7_SF_NS_4lessIS7_EELb1EEENSC_ISF_EEE25__emplace_unique_key_argsIS7_JNS_4pairIS7_SE_EEEEENSN_INS_15__tree_iteratorISF_PNS_11__tree_nodeISF_PvEElEEbEERKT_DpOT0_($335,$1103,$1104,$1106);
  HEAP32[$329>>2] = $455;
  HEAP32[$330>>2] = $335;
  $1107 = HEAP32[$329>>2]|0;
  $1108 = HEAP32[$330>>2]|0;
  HEAP32[$328>>2] = $1108;
  $1109 = HEAP32[$328>>2]|0;
  ;HEAP32[$331>>2]=HEAP32[$1109>>2]|0;
  ;HEAP8[$326>>0]=HEAP8[$331>>0]|0;HEAP8[$326+1>>0]=HEAP8[$331+1>>0]|0;HEAP8[$326+2>>0]=HEAP8[$331+2>>0]|0;HEAP8[$326+3>>0]=HEAP8[$331+3>>0]|0;
  HEAP32[$325>>2] = $1107;
  $1110 = HEAP32[$325>>2]|0;
  ;HEAP32[$1110>>2]=HEAP32[$326>>2]|0;
  $1111 = ((($1107)) + 4|0);
  $1112 = HEAP32[$330>>2]|0;
  $1113 = ((($1112)) + 4|0);
  HEAP32[$327>>2] = $1113;
  $1114 = HEAP32[$327>>2]|0;
  $1115 = HEAP8[$1114>>0]|0;
  $1116 = $1115&1;
  $1117 = $1116&1;
  HEAP8[$1111>>0] = $1117;
  __ZNSt3__24pairIPKN3sap2ve4core4MeshENS_6vectorIPKNS3_7SubMeshENS_9allocatorISA_EEEEED2Ev($454);
 }
 __ZNSt3__213unordered_mapIPKN3sap2ve4core8MaterialES6_NS_4hashIS6_EENS_8equal_toIS6_EENS_9allocatorINS_4pairIKS6_S6_EEEEED2Ev($416);
 __ZNSt3__26vectorIPKN3sap2ve4core7SubMeshENS_9allocatorIS6_EEED2Ev($413);
 STACKTOP = sp;return;
}
function __ZNK3sap2ve6Matrix4DataEv($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 STACKTOP = sp;return ($2|0);
}
function __ZNSt3__24pairIKPKN3sap2ve4core7SubMeshENS_10unique_ptrI13sOnDemandMeshNS_14default_deleteIS9_EEEEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0;
 var $28 = 0, $29 = 0, $3 = 0, $30 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 64|0;
 $1 = sp + 48|0;
 $2 = sp + 44|0;
 $3 = sp + 40|0;
 $4 = sp + 36|0;
 $5 = sp + 32|0;
 $6 = sp + 28|0;
 $7 = sp + 24|0;
 $8 = sp + 20|0;
 $9 = sp + 16|0;
 $10 = sp + 12|0;
 $11 = sp + 8|0;
 $12 = sp + 4|0;
 $13 = sp;
 HEAP32[$13>>2] = $0;
 $14 = HEAP32[$13>>2]|0;
 $15 = ((($14)) + 4|0);
 HEAP32[$12>>2] = $15;
 $16 = HEAP32[$12>>2]|0;
 HEAP32[$9>>2] = $16;
 HEAP32[$10>>2] = 0;
 $17 = HEAP32[$9>>2]|0;
 HEAP32[$8>>2] = $17;
 $18 = HEAP32[$8>>2]|0;
 HEAP32[$7>>2] = $18;
 $19 = HEAP32[$7>>2]|0;
 $20 = HEAP32[$19>>2]|0;
 HEAP32[$11>>2] = $20;
 $21 = HEAP32[$10>>2]|0;
 HEAP32[$4>>2] = $17;
 $22 = HEAP32[$4>>2]|0;
 HEAP32[$3>>2] = $22;
 $23 = HEAP32[$3>>2]|0;
 HEAP32[$23>>2] = $21;
 $24 = HEAP32[$11>>2]|0;
 $25 = ($24|0)!=(0|0);
 if (!($25)) {
  STACKTOP = sp;return;
 }
 HEAP32[$2>>2] = $17;
 $26 = HEAP32[$2>>2]|0;
 HEAP32[$1>>2] = $26;
 $27 = HEAP32[$1>>2]|0;
 $28 = HEAP32[$11>>2]|0;
 HEAP32[$5>>2] = $27;
 HEAP32[$6>>2] = $28;
 $29 = HEAP32[$6>>2]|0;
 $30 = ($29|0)==(0|0);
 if ($30) {
  STACKTOP = sp;return;
 }
 __ZdlPv($29);
 STACKTOP = sp;return;
}
function __ZNSt3__24pairIPKN3sap2ve4core4MeshENS_6vectorIPKNS3_7SubMeshENS_9allocatorISA_EEEEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 $3 = ((($2)) + 4|0);
 __ZNSt3__26vectorIPKN3sap2ve4core7SubMeshENS_9allocatorIS6_EEED2Ev($3);
 STACKTOP = sp;return;
}
function __ZNSt3__26vectorIPKN3sap2ve4core7SubMeshENS_9allocatorIS6_EEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 __ZNSt3__213__vector_baseIPKN3sap2ve4core7SubMeshENS_9allocatorIS6_EEED2Ev($2);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter20InsertTextAnnotationEPKN3sap2ve4core14TextAnnotationEPKNS2_5INodeE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $101 = 0.0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0.0, $107 = 0, $108 = 0, $109 = 0.0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0.0, $122 = 0, $123 = 0, $124 = 0.0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0.0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0.0, $169 = 0, $17 = 0, $18 = 0, $19 = 0;
 var $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0;
 var $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0.0, $55 = 0, $56 = 0;
 var $57 = 0.0, $58 = 0, $59 = 0, $6 = 0, $60 = 0.0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0;
 var $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0;
 var $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0.0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 240|0;
 $3 = sp + 228|0;
 $4 = sp + 224|0;
 $5 = sp + 220|0;
 $6 = sp + 216|0;
 $7 = sp + 212|0;
 $8 = sp + 208|0;
 $9 = sp + 204|0;
 $10 = sp + 200|0;
 $11 = sp + 196|0;
 $12 = sp + 192|0;
 $13 = sp + 188|0;
 $14 = sp + 184|0;
 $15 = sp + 180|0;
 $16 = sp + 176|0;
 $17 = sp + 172|0;
 $18 = sp + 168|0;
 $19 = sp + 164|0;
 $20 = sp + 160|0;
 $21 = sp + 156|0;
 $22 = sp + 152|0;
 $23 = sp + 148|0;
 $24 = sp + 144|0;
 $25 = sp + 140|0;
 $26 = sp + 136|0;
 $27 = sp + 132|0;
 $28 = sp + 128|0;
 $29 = sp + 124|0;
 $30 = sp + 120|0;
 $31 = sp + 116|0;
 $32 = sp + 112|0;
 $33 = sp + 108|0;
 $34 = sp + 104|0;
 $35 = sp + 100|0;
 $36 = sp + 96|0;
 $37 = sp + 92|0;
 $38 = sp + 88|0;
 $39 = sp + 76|0;
 $40 = sp + 64|0;
 $41 = sp + 48|0;
 $42 = sp + 32|0;
 $43 = sp + 16|0;
 $44 = sp;
 HEAP32[$33>>2] = $0;
 HEAP32[$34>>2] = $1;
 HEAP32[$35>>2] = $2;
 $45 = HEAP32[$33>>2]|0;
 $46 = HEAP32[$34>>2]|0;
 $47 = (__ZNK3sap2ve4core18TextAnnotationBase17GetTextDefinitionEv($46)|0);
 HEAP32[$36>>2] = $47;
 $48 = HEAP32[$34>>2]|0;
 $49 = (__ZNK3sap2ve4core18TextAnnotationBase18GetLabelDefinitionEv($48)|0);
 HEAP32[$37>>2] = $49;
 $50 = HEAP32[$34>>2]|0;
 $51 = (__ZNK3sap2ve4core20AnnotationWithAction9GetActionEv($50)|0);
 HEAP32[$38>>2] = $51;
 $52 = HEAP32[$34>>2]|0;
 $53 = (__ZNK3sap2ve4core18TextAnnotationBase11GetPositionEv($52)|0);
 __ZNK3sap2ve4core15AnnotationPoint8GetPointEv($39,$53);
 $54 = +HEAPF32[$39>>2];
 HEAPF32[$40>>2] = $54;
 $55 = ((($40)) + 4|0);
 $56 = ((($39)) + 4|0);
 $57 = +HEAPF32[$56>>2];
 HEAPF32[$55>>2] = $57;
 $58 = ((($55)) + 4|0);
 $59 = ((($39)) + 8|0);
 $60 = +HEAPF32[$59>>2];
 HEAPF32[$58>>2] = $60;
 $61 = ((($45)) + 24|0);
 $62 = HEAP32[$61>>2]|0;
 $63 = HEAP32[$35>>2]|0;
 $64 = HEAP32[$36>>2]|0;
 $65 = (__ZNK3sap2ve4core14TextDefinition11GetEncodingEv($64)|0);
 $66 = HEAP32[$36>>2]|0;
 $67 = ((($45)) + 12|0);
 HEAP32[$32>>2] = $67;
 $68 = HEAP32[$32>>2]|0;
 HEAP32[$31>>2] = $68;
 $69 = HEAP32[$31>>2]|0;
 HEAP32[$30>>2] = $69;
 $70 = HEAP32[$30>>2]|0;
 HEAP32[$29>>2] = $70;
 $71 = HEAP32[$29>>2]|0;
 HEAP32[$28>>2] = $71;
 $72 = HEAP32[$28>>2]|0;
 HEAP32[$27>>2] = $72;
 $73 = HEAP32[$27>>2]|0;
 $74 = ((($73)) + 11|0);
 $75 = HEAP8[$74>>0]|0;
 $76 = $75&255;
 $77 = $76 & 128;
 $78 = ($77|0)!=(0);
 if ($78) {
  HEAP32[$21>>2] = $70;
  $79 = HEAP32[$21>>2]|0;
  HEAP32[$20>>2] = $79;
  $80 = HEAP32[$20>>2]|0;
  HEAP32[$19>>2] = $80;
  $81 = HEAP32[$19>>2]|0;
  $82 = HEAP32[$81>>2]|0;
  $88 = $82;
 } else {
  HEAP32[$26>>2] = $70;
  $83 = HEAP32[$26>>2]|0;
  HEAP32[$25>>2] = $83;
  $84 = HEAP32[$25>>2]|0;
  HEAP32[$24>>2] = $84;
  $85 = HEAP32[$24>>2]|0;
  HEAP32[$23>>2] = $85;
  $86 = HEAP32[$23>>2]|0;
  HEAP32[$22>>2] = $86;
  $87 = HEAP32[$22>>2]|0;
  $88 = $87;
 }
 HEAP32[$18>>2] = $88;
 $89 = HEAP32[$18>>2]|0;
 $90 = (__ZNK3sap2ve4core14TextDefinition7GetTextEPKc($66,$89)|0);
 $91 = (__ZNK3sap2ve6String4Utf8Ev($90)|0);
 $92 = HEAP32[$36>>2]|0;
 $93 = (__ZNK3sap2ve4core14TextDefinition11GetFontFaceEv($92)|0);
 $94 = (__ZNK3sap2ve6String4Utf8Ev($93)|0);
 $95 = HEAP32[$36>>2]|0;
 $96 = (__ZNK3sap2ve4core14TextDefinition11GetFontSizeEv($95)|0);
 $97 = (Math_abs(($96|0))|0);
 $98 = (+($97|0));
 $99 = HEAP32[$36>>2]|0;
 $100 = (__ZNK3sap2ve4core14TextDefinition13GetFontWeightEv($99)|0);
 $101 = (+($100|0));
 $102 = HEAP32[$36>>2]|0;
 $103 = (__ZNK3sap2ve4core14TextDefinition13GetFontItalicEv($102)|0);
 $104 = HEAP32[$37>>2]|0;
 $105 = (__ZNK3sap2ve4core15LabelDefinition8GetWidthEv($104)|0);
 $106 = +HEAPF32[$105>>2];
 $107 = HEAP32[$37>>2]|0;
 $108 = (__ZNK3sap2ve4core15LabelDefinition9GetHeightEv($107)|0);
 $109 = +HEAPF32[$108>>2];
 $110 = HEAP32[$37>>2]|0;
 $111 = (__ZNK3sap2ve4core15LabelDefinition8GetStyleEv($110)|0);
 $112 = HEAP32[$36>>2]|0;
 $113 = (__ZNK3sap2ve4core14TextDefinition9GetColourEv($112)|0);
 $114 = (__ZL8GetColorRKN3sap2ve4core6ColourE($113)|0);
 $115 = HEAP32[$37>>2]|0;
 __ZNK3sap2ve4core15LabelDefinition19GetBackgroundColourEv($41,$115);
 $116 = (__ZL8GetColorRKN3sap2ve4core6ColourE($41)|0);
 $117 = HEAP32[$37>>2]|0;
 $118 = (__ZNK3sap2ve4core15LabelDefinition23GetBackgroundVisibilityEv($117)|0);
 if ($118) {
  $119 = HEAP32[$37>>2]|0;
  __ZNK3sap2ve4core15LabelDefinition19GetBackgroundColourEv($42,$119);
  $120 = (__ZNK3sap2ve4core6Colour5AlphaEv($42)|0);
  $121 = +HEAPF32[$120>>2];
  $168 = $121;
 } else {
  $168 = 0.0;
 }
 $122 = HEAP32[$37>>2]|0;
 $123 = (__ZNK3sap2ve4core15LabelDefinition14GetBorderWidthEv($122)|0);
 $124 = (+($123|0));
 $125 = HEAP32[$37>>2]|0;
 $126 = (__ZNK3sap2ve4core15LabelDefinition19GetBorderVisibilityEv($125)|0);
 if ($126) {
  $127 = HEAP32[$37>>2]|0;
  $128 = (__ZNK3sap2ve4core15LabelDefinition18GetBorderLineStyleEv($127)|0);
  $169 = $128;
 } else {
  $169 = 0;
 }
 $129 = HEAP32[$37>>2]|0;
 __ZNK3sap2ve4core15LabelDefinition15GetBorderColourEv($43,$129);
 $130 = (__ZL8GetColorRKN3sap2ve4core6ColourE($43)|0);
 $131 = HEAP32[$37>>2]|0;
 __ZNK3sap2ve4core15LabelDefinition15GetBorderColourEv($44,$131);
 $132 = (__ZNK3sap2ve4core6Colour5AlphaEv($44)|0);
 $133 = +HEAPF32[$132>>2];
 $134 = HEAP32[$36>>2]|0;
 $135 = (__ZNK3sap2ve4core14TextDefinition22GetHorizontalAlignmentEv($134)|0);
 $136 = HEAP32[$36>>2]|0;
 $137 = (__ZNK3sap2ve4core14TextDefinition20GetVerticalAlignmentEv($136)|0);
 $138 = HEAP32[$38>>2]|0;
 $139 = (__ZNK3sap2ve4core16AnnotationAction14HasDisplayTextEv($138)|0);
 $140 = HEAP32[$38>>2]|0;
 if (!($139)) {
  $165 = (__ZNK3sap2ve4core16AnnotationAction6GetUrlEv($140)|0);
  $166 = $165;
  $167 = (__ZNK3sap2ve6String4Utf8Ev($166)|0);
  _SceneBuilderAdapter_createTextAnnotation(($62|0),($63|0),($40|0),($65|0),($91|0),($94|0),(+$98),(+$101),($103|0),(+$106),(+$109),($111|0),($114|0),($116|0),(+$168),(+$124),($169|0),($130|0),(+$133),($135|0),($137|0),($167|0));
  STACKTOP = sp;return;
 }
 $141 = ((($45)) + 12|0);
 HEAP32[$17>>2] = $141;
 $142 = HEAP32[$17>>2]|0;
 HEAP32[$16>>2] = $142;
 $143 = HEAP32[$16>>2]|0;
 HEAP32[$15>>2] = $143;
 $144 = HEAP32[$15>>2]|0;
 HEAP32[$14>>2] = $144;
 $145 = HEAP32[$14>>2]|0;
 HEAP32[$13>>2] = $145;
 $146 = HEAP32[$13>>2]|0;
 HEAP32[$12>>2] = $146;
 $147 = HEAP32[$12>>2]|0;
 $148 = ((($147)) + 11|0);
 $149 = HEAP8[$148>>0]|0;
 $150 = $149&255;
 $151 = $150 & 128;
 $152 = ($151|0)!=(0);
 if ($152) {
  HEAP32[$6>>2] = $144;
  $153 = HEAP32[$6>>2]|0;
  HEAP32[$5>>2] = $153;
  $154 = HEAP32[$5>>2]|0;
  HEAP32[$4>>2] = $154;
  $155 = HEAP32[$4>>2]|0;
  $156 = HEAP32[$155>>2]|0;
  $162 = $156;
 } else {
  HEAP32[$11>>2] = $144;
  $157 = HEAP32[$11>>2]|0;
  HEAP32[$10>>2] = $157;
  $158 = HEAP32[$10>>2]|0;
  HEAP32[$9>>2] = $158;
  $159 = HEAP32[$9>>2]|0;
  HEAP32[$8>>2] = $159;
  $160 = HEAP32[$8>>2]|0;
  HEAP32[$7>>2] = $160;
  $161 = HEAP32[$7>>2]|0;
  $162 = $161;
 }
 HEAP32[$3>>2] = $162;
 $163 = HEAP32[$3>>2]|0;
 $164 = (__ZNK3sap2ve4core16AnnotationAction14GetDisplayTextEPKc($140,$163)|0);
 $166 = $164;
 $167 = (__ZNK3sap2ve6String4Utf8Ev($166)|0);
 _SceneBuilderAdapter_createTextAnnotation(($62|0),($63|0),($40|0),($65|0),($91|0),($94|0),(+$98),(+$101),($103|0),(+$106),(+$109),($111|0),($114|0),($116|0),(+$168),(+$124),($169|0),($130|0),(+$133),($135|0),($137|0),($167|0));
 STACKTOP = sp;return;
}
function __ZL8GetColorRKN3sap2ve4core6ColourE($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0.0, $11 = 0.0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0.0, $18 = 0.0, $19 = 0, $2 = 0, $20 = 0, $3 = 0, $4 = 0.0, $5 = 0.0, $6 = 0, $7 = 0, $8 = 0, $9 = 0;
 var label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 $3 = (__ZNK3sap2ve4core6Colour3RedEv($2)|0);
 $4 = +HEAPF32[$3>>2];
 $5 = $4 * 255.0;
 $6 = (~~(($5))>>>0);
 $7 = $6 << 16;
 $8 = HEAP32[$1>>2]|0;
 $9 = (__ZNK3sap2ve4core6Colour5GreenEv($8)|0);
 $10 = +HEAPF32[$9>>2];
 $11 = $10 * 255.0;
 $12 = (~~(($11))>>>0);
 $13 = $12 << 8;
 $14 = $7 | $13;
 $15 = HEAP32[$1>>2]|0;
 $16 = (__ZNK3sap2ve4core6Colour4BlueEv($15)|0);
 $17 = +HEAPF32[$16>>2];
 $18 = $17 * 255.0;
 $19 = (~~(($18))>>>0);
 $20 = $14 | $19;
 STACKTOP = sp;return ($20|0);
}
function __ZNK17DataModelImporter21GetAnnotationPositionERKN3sap2ve4core15AnnotationPointEPKNS2_5INodeE($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $4 = sp + 24|0;
 $5 = sp + 20|0;
 $6 = sp + 16|0;
 $7 = sp + 12|0;
 $8 = sp;
 HEAP32[$4>>2] = $1;
 HEAP32[$5>>2] = $2;
 HEAP32[$6>>2] = $3;
 $9 = HEAP32[$5>>2]|0;
 __ZNK3sap2ve4core15AnnotationPoint8GetPointEv($0,$9);
 $10 = HEAP32[$6>>2]|0;
 $11 = ($10|0)!=(0|0);
 if (!($11)) {
  STACKTOP = sp;return;
 }
 $12 = HEAP32[$6>>2]|0;
 $13 = HEAP32[$12>>2]|0;
 $14 = ((($13)) + 100|0);
 $15 = HEAP32[$14>>2]|0;
 $16 = (FUNCTION_TABLE_ii[$15 & 2047]($12)|0);
 HEAP32[$7>>2] = $16;
 $17 = HEAP32[$7>>2]|0;
 $18 = ($17|0)!=(0|0);
 if (!($18)) {
  STACKTOP = sp;return;
 }
 $19 = HEAP32[$7>>2]|0;
 __ZN3sap2vemlERKNS0_6MatrixERKNS0_6VectorE($8,$19,$0);
 ;HEAP32[$0>>2]=HEAP32[$8>>2]|0;HEAP32[$0+4>>2]=HEAP32[$8+4>>2]|0;HEAP32[$0+8>>2]=HEAP32[$8+8>>2]|0;
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter14InsertTextNoteEPKN3sap2ve4core8TextNoteEPKNS2_5INodeE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0.0, $14 = 0, $140 = 0, $141 = 0, $142 = 0.0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0.0, $148 = 0, $149 = 0, $15 = 0, $150 = 0.0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0.0, $163 = 0, $164 = 0, $165 = 0.0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0.0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0;
 var $209 = 0.0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0;
 var $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0.0;
 var $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0.0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0.0, $261 = 0, $262 = 0;
 var $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0;
 var $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0.0, $292 = 0.0, $293 = 0, $294 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0;
 var $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0;
 var $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0;
 var $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0.0, $85 = 0, $86 = 0, $87 = 0.0, $88 = 0;
 var $89 = 0, $9 = 0, $90 = 0.0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 336|0;
 $3 = sp + 320|0;
 $4 = sp + 316|0;
 $5 = sp + 312|0;
 $6 = sp + 308|0;
 $7 = sp + 304|0;
 $8 = sp + 300|0;
 $9 = sp + 296|0;
 $10 = sp + 292|0;
 $11 = sp + 288|0;
 $12 = sp + 284|0;
 $13 = sp + 280|0;
 $14 = sp + 276|0;
 $15 = sp + 272|0;
 $16 = sp + 268|0;
 $17 = sp + 264|0;
 $18 = sp + 260|0;
 $19 = sp + 256|0;
 $20 = sp + 252|0;
 $21 = sp + 248|0;
 $22 = sp + 244|0;
 $23 = sp + 240|0;
 $24 = sp + 236|0;
 $25 = sp + 232|0;
 $26 = sp + 228|0;
 $27 = sp + 224|0;
 $28 = sp + 220|0;
 $29 = sp + 216|0;
 $30 = sp + 212|0;
 $31 = sp + 208|0;
 $32 = sp + 204|0;
 $33 = sp + 200|0;
 $34 = sp + 196|0;
 $35 = sp + 192|0;
 $36 = sp + 188|0;
 $37 = sp + 184|0;
 $38 = sp + 180|0;
 $39 = sp + 176|0;
 $40 = sp + 172|0;
 $41 = sp + 168|0;
 $42 = sp + 164|0;
 $43 = sp + 160|0;
 $44 = sp + 156|0;
 $45 = sp + 152|0;
 $46 = sp + 148|0;
 $47 = sp + 144|0;
 $48 = sp + 132|0;
 $49 = sp + 120|0;
 $50 = sp + 324|0;
 $51 = sp + 104|0;
 $52 = sp + 88|0;
 $53 = sp + 72|0;
 $54 = sp + 56|0;
 $55 = sp + 52|0;
 $56 = sp + 48|0;
 $57 = sp + 44|0;
 $58 = sp + 40|0;
 $59 = sp + 36|0;
 $60 = sp + 24|0;
 $61 = sp + 20|0;
 $62 = sp + 16|0;
 $63 = sp + 4|0;
 $64 = sp;
 HEAP32[$41>>2] = $0;
 HEAP32[$42>>2] = $1;
 HEAP32[$43>>2] = $2;
 $65 = HEAP32[$41>>2]|0;
 $66 = HEAP32[$42>>2]|0;
 $67 = (__ZNK3sap2ve4core18TextAnnotationBase17GetTextDefinitionEv($66)|0);
 HEAP32[$44>>2] = $67;
 $68 = HEAP32[$42>>2]|0;
 $69 = (__ZNK3sap2ve4core18TextAnnotationBase18GetLabelDefinitionEv($68)|0);
 HEAP32[$45>>2] = $69;
 $70 = HEAP32[$42>>2]|0;
 $71 = (__ZNK3sap2ve4core20AnnotationWithAction9GetActionEv($70)|0);
 HEAP32[$46>>2] = $71;
 $72 = ((($65)) + 4|0);
 $73 = (__ZNK3sap2ve4core4IPtrINS1_6ISceneEEptEv($72)|0);
 $74 = HEAP32[$73>>2]|0;
 $75 = ((($74)) + 44|0);
 $76 = HEAP32[$75>>2]|0;
 $77 = HEAP32[$42>>2]|0;
 $78 = (__ZNK3sap2ve4core18TextAnnotationBase11GetPositionEv($77)|0);
 $79 = (__ZNK3sap2ve4core15AnnotationPoint9GetTargetEv($78)|0);
 $80 = (FUNCTION_TABLE_iii[$76 & 1023]($73,$79)|0);
 HEAP32[$47>>2] = $80;
 $81 = HEAP32[$42>>2]|0;
 $82 = (__ZNK3sap2ve4core18TextAnnotationBase11GetPositionEv($81)|0);
 $83 = HEAP32[$47>>2]|0;
 __ZNK17DataModelImporter21GetAnnotationPositionERKN3sap2ve4core15AnnotationPointEPKNS2_5INodeE($48,$65,$82,$83);
 $84 = +HEAPF32[$48>>2];
 HEAPF32[$49>>2] = $84;
 $85 = ((($49)) + 4|0);
 $86 = ((($48)) + 4|0);
 $87 = +HEAPF32[$86>>2];
 HEAPF32[$85>>2] = $87;
 $88 = ((($85)) + 4|0);
 $89 = ((($48)) + 8|0);
 $90 = +HEAPF32[$89>>2];
 HEAPF32[$88>>2] = $90;
 $91 = HEAP32[$42>>2]|0;
 $92 = (__ZNK3sap2ve4core10Annotation8GetFlagsEv($91)|0);
 $93 = tempRet0;
 $94 = $92 & 1;
 $95 = ($94|0)!=(0);
 $96 = (0)!=(0);
 $97 = $95 | $96;
 $98 = $97&1;
 HEAP8[$50>>0] = $98;
 $99 = ((($65)) + 24|0);
 $100 = HEAP32[$99>>2]|0;
 $101 = HEAP32[$43>>2]|0;
 $102 = HEAP32[$47>>2]|0;
 $103 = HEAP8[$50>>0]|0;
 $104 = $103&1;
 $105 = HEAP32[$44>>2]|0;
 $106 = (__ZNK3sap2ve4core14TextDefinition11GetEncodingEv($105)|0);
 $107 = HEAP32[$44>>2]|0;
 $108 = ((($65)) + 12|0);
 HEAP32[$40>>2] = $108;
 $109 = HEAP32[$40>>2]|0;
 HEAP32[$39>>2] = $109;
 $110 = HEAP32[$39>>2]|0;
 HEAP32[$38>>2] = $110;
 $111 = HEAP32[$38>>2]|0;
 HEAP32[$37>>2] = $111;
 $112 = HEAP32[$37>>2]|0;
 HEAP32[$36>>2] = $112;
 $113 = HEAP32[$36>>2]|0;
 HEAP32[$35>>2] = $113;
 $114 = HEAP32[$35>>2]|0;
 $115 = ((($114)) + 11|0);
 $116 = HEAP8[$115>>0]|0;
 $117 = $116&255;
 $118 = $117 & 128;
 $119 = ($118|0)!=(0);
 if ($119) {
  HEAP32[$29>>2] = $111;
  $120 = HEAP32[$29>>2]|0;
  HEAP32[$28>>2] = $120;
  $121 = HEAP32[$28>>2]|0;
  HEAP32[$27>>2] = $121;
  $122 = HEAP32[$27>>2]|0;
  $123 = HEAP32[$122>>2]|0;
  $129 = $123;
 } else {
  HEAP32[$34>>2] = $111;
  $124 = HEAP32[$34>>2]|0;
  HEAP32[$33>>2] = $124;
  $125 = HEAP32[$33>>2]|0;
  HEAP32[$32>>2] = $125;
  $126 = HEAP32[$32>>2]|0;
  HEAP32[$31>>2] = $126;
  $127 = HEAP32[$31>>2]|0;
  HEAP32[$30>>2] = $127;
  $128 = HEAP32[$30>>2]|0;
  $129 = $128;
 }
 HEAP32[$26>>2] = $129;
 $130 = HEAP32[$26>>2]|0;
 $131 = (__ZNK3sap2ve4core14TextDefinition7GetTextEPKc($107,$130)|0);
 $132 = (__ZNK3sap2ve6String4Utf8Ev($131)|0);
 $133 = HEAP32[$44>>2]|0;
 $134 = (__ZNK3sap2ve4core14TextDefinition11GetFontFaceEv($133)|0);
 $135 = (__ZNK3sap2ve6String4Utf8Ev($134)|0);
 $136 = HEAP32[$44>>2]|0;
 $137 = (__ZNK3sap2ve4core14TextDefinition11GetFontSizeEv($136)|0);
 $138 = (Math_abs(($137|0))|0);
 $139 = (+($138|0));
 $140 = HEAP32[$44>>2]|0;
 $141 = (__ZNK3sap2ve4core14TextDefinition13GetFontWeightEv($140)|0);
 $142 = (+($141|0));
 $143 = HEAP32[$44>>2]|0;
 $144 = (__ZNK3sap2ve4core14TextDefinition13GetFontItalicEv($143)|0);
 $145 = HEAP32[$45>>2]|0;
 $146 = (__ZNK3sap2ve4core15LabelDefinition8GetWidthEv($145)|0);
 $147 = +HEAPF32[$146>>2];
 $148 = HEAP32[$45>>2]|0;
 $149 = (__ZNK3sap2ve4core15LabelDefinition9GetHeightEv($148)|0);
 $150 = +HEAPF32[$149>>2];
 $151 = HEAP32[$45>>2]|0;
 $152 = (__ZNK3sap2ve4core15LabelDefinition8GetStyleEv($151)|0);
 $153 = HEAP32[$44>>2]|0;
 $154 = (__ZNK3sap2ve4core14TextDefinition9GetColourEv($153)|0);
 $155 = (__ZL8GetColorRKN3sap2ve4core6ColourE($154)|0);
 $156 = HEAP32[$45>>2]|0;
 __ZNK3sap2ve4core15LabelDefinition19GetBackgroundColourEv($51,$156);
 $157 = (__ZL8GetColorRKN3sap2ve4core6ColourE($51)|0);
 $158 = HEAP32[$45>>2]|0;
 $159 = (__ZNK3sap2ve4core15LabelDefinition23GetBackgroundVisibilityEv($158)|0);
 if ($159) {
  $160 = HEAP32[$45>>2]|0;
  __ZNK3sap2ve4core15LabelDefinition19GetBackgroundColourEv($52,$160);
  $161 = (__ZNK3sap2ve4core6Colour5AlphaEv($52)|0);
  $162 = +HEAPF32[$161>>2];
  $209 = $162;
 } else {
  $209 = 0.0;
 }
 $163 = HEAP32[$45>>2]|0;
 $164 = (__ZNK3sap2ve4core15LabelDefinition14GetBorderWidthEv($163)|0);
 $165 = (+($164|0));
 $166 = HEAP32[$45>>2]|0;
 $167 = (__ZNK3sap2ve4core15LabelDefinition19GetBorderVisibilityEv($166)|0);
 if ($167) {
  $168 = HEAP32[$45>>2]|0;
  $169 = (__ZNK3sap2ve4core15LabelDefinition18GetBorderLineStyleEv($168)|0);
  $210 = $169;
 } else {
  $210 = 0;
 }
 $170 = HEAP32[$45>>2]|0;
 __ZNK3sap2ve4core15LabelDefinition15GetBorderColourEv($53,$170);
 $171 = (__ZL8GetColorRKN3sap2ve4core6ColourE($53)|0);
 $172 = HEAP32[$45>>2]|0;
 __ZNK3sap2ve4core15LabelDefinition15GetBorderColourEv($54,$172);
 $173 = (__ZNK3sap2ve4core6Colour5AlphaEv($54)|0);
 $174 = +HEAPF32[$173>>2];
 $175 = HEAP32[$44>>2]|0;
 $176 = (__ZNK3sap2ve4core14TextDefinition22GetHorizontalAlignmentEv($175)|0);
 $177 = HEAP32[$44>>2]|0;
 $178 = (__ZNK3sap2ve4core14TextDefinition20GetVerticalAlignmentEv($177)|0);
 $179 = HEAP32[$46>>2]|0;
 $180 = (__ZNK3sap2ve4core16AnnotationAction14HasDisplayTextEv($179)|0);
 $181 = HEAP32[$46>>2]|0;
 if ($180) {
  $182 = ((($65)) + 12|0);
  HEAP32[$25>>2] = $182;
  $183 = HEAP32[$25>>2]|0;
  HEAP32[$24>>2] = $183;
  $184 = HEAP32[$24>>2]|0;
  HEAP32[$23>>2] = $184;
  $185 = HEAP32[$23>>2]|0;
  HEAP32[$22>>2] = $185;
  $186 = HEAP32[$22>>2]|0;
  HEAP32[$21>>2] = $186;
  $187 = HEAP32[$21>>2]|0;
  HEAP32[$20>>2] = $187;
  $188 = HEAP32[$20>>2]|0;
  $189 = ((($188)) + 11|0);
  $190 = HEAP8[$189>>0]|0;
  $191 = $190&255;
  $192 = $191 & 128;
  $193 = ($192|0)!=(0);
  if ($193) {
   HEAP32[$14>>2] = $185;
   $194 = HEAP32[$14>>2]|0;
   HEAP32[$13>>2] = $194;
   $195 = HEAP32[$13>>2]|0;
   HEAP32[$12>>2] = $195;
   $196 = HEAP32[$12>>2]|0;
   $197 = HEAP32[$196>>2]|0;
   $203 = $197;
  } else {
   HEAP32[$19>>2] = $185;
   $198 = HEAP32[$19>>2]|0;
   HEAP32[$18>>2] = $198;
   $199 = HEAP32[$18>>2]|0;
   HEAP32[$17>>2] = $199;
   $200 = HEAP32[$17>>2]|0;
   HEAP32[$16>>2] = $200;
   $201 = HEAP32[$16>>2]|0;
   HEAP32[$15>>2] = $201;
   $202 = HEAP32[$15>>2]|0;
   $203 = $202;
  }
  HEAP32[$11>>2] = $203;
  $204 = HEAP32[$11>>2]|0;
  $205 = (__ZNK3sap2ve4core16AnnotationAction14GetDisplayTextEPKc($181,$204)|0);
  $207 = $205;
 } else {
  $206 = (__ZNK3sap2ve4core16AnnotationAction6GetUrlEv($181)|0);
  $207 = $206;
 }
 $208 = (__ZNK3sap2ve6String4Utf8Ev($207)|0);
 _SceneBuilderAdapter_createTextNote(($100|0),($101|0),($102|0),($49|0),($104|0),($106|0),($132|0),($135|0),(+$139),(+$142),($144|0),(+$147),(+$150),($152|0),($155|0),($157|0),(+$209),(+$165),($210|0),($171|0),(+$174),($176|0),($178|0),($208|0));
 HEAP32[$55>>2] = 0;
 while(1) {
  $211 = HEAP32[$55>>2]|0;
  $212 = HEAP32[$42>>2]|0;
  $213 = (__ZNK3sap2ve4core29TextAnnotationWithLeaderLines18GetLeaderLineCountEv($212)|0);
  $214 = ($211>>>0)<($213>>>0);
  if (!($214)) {
   break;
  }
  $215 = HEAP32[$42>>2]|0;
  $216 = HEAP32[$55>>2]|0;
  $217 = (__ZNK3sap2ve4core29TextAnnotationWithLeaderLines13GetLeaderLineEj($215,$216)|0);
  HEAP32[$56>>2] = $217;
  $218 = HEAP32[$56>>2]|0;
  $219 = (__ZNK3sap2ve4core10LeaderLine11GetMaterialEv($218)|0);
  $220 = (__ZN17DataModelImporter19GetOrCreateMaterialEPKN3sap2ve4core8MaterialE($65,$219)|0);
  HEAP32[$57>>2] = $220;
  $221 = HEAP32[$56>>2]|0;
  $222 = (__ZNK3sap2ve4core10LeaderLine23GetAnnotationPointCountEv($221)|0);
  HEAP32[$58>>2] = $222;
  $223 = HEAP32[$58>>2]|0;
  $224 = ($223>>>0)>(0);
  if ($224) {
   $225 = ((($65)) + 4|0);
   $226 = (__ZNK3sap2ve4core4IPtrINS1_6ISceneEEptEv($225)|0);
   $227 = HEAP32[$226>>2]|0;
   $228 = ((($227)) + 44|0);
   $229 = HEAP32[$228>>2]|0;
   $230 = HEAP32[$56>>2]|0;
   $231 = (__ZNK3sap2ve4core10LeaderLine18GetAnnotationPointEj($230,0)|0);
   $232 = (__ZNK3sap2ve4core15AnnotationPoint9GetTargetEv($231)|0);
   $233 = (FUNCTION_TABLE_iii[$229 & 1023]($226,$232)|0);
   $234 = $233;
  } else {
   $234 = 0;
  }
  HEAP32[$59>>2] = $234;
  $235 = HEAP32[$58>>2]|0;
  $236 = ($235*3)|0;
  __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($60,$236);
  HEAP32[$61>>2] = 0;
  HEAP32[$62>>2] = 0;
  while(1) {
   $237 = HEAP32[$61>>2]|0;
   $238 = HEAP32[$58>>2]|0;
   $239 = ($237>>>0)<($238>>>0);
   $240 = HEAP32[$56>>2]|0;
   if (!($239)) {
    break;
   }
   $241 = HEAP32[$61>>2]|0;
   $242 = (__ZNK3sap2ve4core10LeaderLine18GetAnnotationPointEj($240,$241)|0);
   $243 = HEAP32[$59>>2]|0;
   __ZNK17DataModelImporter21GetAnnotationPositionERKN3sap2ve4core15AnnotationPointEPKNS2_5INodeE($63,$65,$242,$243);
   $244 = +HEAPF32[$63>>2];
   $245 = HEAP32[$62>>2]|0;
   $246 = (($245) + 1)|0;
   HEAP32[$62>>2] = $246;
   HEAP32[$7>>2] = $60;
   HEAP32[$8>>2] = $245;
   $247 = HEAP32[$7>>2]|0;
   $248 = HEAP32[$247>>2]|0;
   $249 = HEAP32[$8>>2]|0;
   $250 = (($248) + ($249<<2)|0);
   HEAPF32[$250>>2] = $244;
   $251 = ((($63)) + 4|0);
   $252 = +HEAPF32[$251>>2];
   $253 = HEAP32[$62>>2]|0;
   $254 = (($253) + 1)|0;
   HEAP32[$62>>2] = $254;
   HEAP32[$3>>2] = $60;
   HEAP32[$4>>2] = $253;
   $255 = HEAP32[$3>>2]|0;
   $256 = HEAP32[$255>>2]|0;
   $257 = HEAP32[$4>>2]|0;
   $258 = (($256) + ($257<<2)|0);
   HEAPF32[$258>>2] = $252;
   $259 = ((($63)) + 8|0);
   $260 = +HEAPF32[$259>>2];
   $261 = HEAP32[$62>>2]|0;
   $262 = (($261) + 1)|0;
   HEAP32[$62>>2] = $262;
   HEAP32[$5>>2] = $60;
   HEAP32[$6>>2] = $261;
   $263 = HEAP32[$5>>2]|0;
   $264 = HEAP32[$263>>2]|0;
   $265 = HEAP32[$6>>2]|0;
   $266 = (($264) + ($265<<2)|0);
   HEAPF32[$266>>2] = $260;
   $267 = HEAP32[$61>>2]|0;
   $268 = (($267) + 1)|0;
   HEAP32[$61>>2] = $268;
  }
  $269 = (__ZNK3sap2ve4core10LeaderLine22GetLeaderLineExtensionEv($240)|0);
  HEAP32[$64>>2] = $269;
  $270 = ((($65)) + 24|0);
  $271 = HEAP32[$270>>2]|0;
  $272 = HEAP32[$43>>2]|0;
  $273 = HEAP32[$59>>2]|0;
  $274 = HEAP32[$57>>2]|0;
  $275 = HEAP32[$58>>2]|0;
  HEAP32[$10>>2] = $60;
  $276 = HEAP32[$10>>2]|0;
  $277 = HEAP32[$276>>2]|0;
  HEAP32[$9>>2] = $277;
  $278 = HEAP32[$9>>2]|0;
  $279 = HEAP32[$56>>2]|0;
  $280 = (__ZNK3sap2ve4core10LeaderLine18GetStartPointStyleEv($279)|0);
  $281 = HEAP32[$56>>2]|0;
  $282 = (__ZNK3sap2ve4core10LeaderLine16GetEndPointStyleEv($281)|0);
  $283 = HEAP32[$56>>2]|0;
  $284 = (__ZNK3sap2ve4core10LeaderLine16GetStyleConstantEv($283)|0);
  $285 = HEAP32[$64>>2]|0;
  $286 = ($285|0)!=(0|0);
  if ($286) {
   $287 = HEAP32[$64>>2]|0;
   $288 = (__ZNK3sap2ve4core19LeaderLineExtension8GetStyleEv($287)|0);
   $289 = ($288|0)==(1);
   if ($289) {
    $290 = HEAP32[$64>>2]|0;
    $291 = (+__ZNK3sap2ve4core19LeaderLineExtension9GetLengthEv($290));
    $292 = $291;
   } else {
    $292 = 0.0;
   }
  } else {
   $292 = 0.0;
  }
  _SceneBuilderAdapter_insertLeaderLine(($271|0),($272|0),($273|0),($274|0),($275|0),($278|0),($280|0),($282|0),($284|0),(+$292));
  __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($60);
  $293 = HEAP32[$55>>2]|0;
  $294 = (($293) + 1)|0;
  HEAP32[$55>>2] = $294;
 }
 STACKTOP = sp;return;
}
function __ZNK3sap2ve4core4IPtrINS1_6ISceneEEptEv($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 $3 = HEAP32[$2>>2]|0;
 STACKTOP = sp;return ($3|0);
}
function __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 64|0;
 $2 = sp + 48|0;
 $3 = sp + 44|0;
 $4 = sp + 40|0;
 $5 = sp + 36|0;
 $6 = sp + 32|0;
 $7 = sp + 28|0;
 $8 = sp + 24|0;
 $9 = sp + 20|0;
 $10 = sp + 16|0;
 $11 = sp + 12|0;
 $12 = sp + 8|0;
 $13 = sp + 4|0;
 $14 = sp;
 HEAP32[$13>>2] = $0;
 HEAP32[$14>>2] = $1;
 $15 = HEAP32[$13>>2]|0;
 HEAP32[$11>>2] = $15;
 $16 = HEAP32[$11>>2]|0;
 HEAP32[$10>>2] = $16;
 HEAP32[$16>>2] = 0;
 $17 = ((($16)) + 4|0);
 HEAP32[$17>>2] = 0;
 $18 = ((($16)) + 8|0);
 HEAP32[$12>>2] = 0;
 HEAP32[$8>>2] = $18;
 HEAP32[$9>>2] = $12;
 $19 = HEAP32[$8>>2]|0;
 $20 = HEAP32[$9>>2]|0;
 HEAP32[$7>>2] = $20;
 $21 = HEAP32[$7>>2]|0;
 HEAP32[$3>>2] = $19;
 HEAP32[$4>>2] = $21;
 $22 = HEAP32[$3>>2]|0;
 $23 = HEAP32[$4>>2]|0;
 HEAP32[$2>>2] = $23;
 HEAP32[$22>>2] = 0;
 HEAP32[$6>>2] = $19;
 $24 = HEAP32[$6>>2]|0;
 HEAP32[$5>>2] = $24;
 $25 = HEAP32[$14>>2]|0;
 $26 = ($25>>>0)>(0);
 if (!($26)) {
  STACKTOP = sp;return;
 }
 $27 = HEAP32[$14>>2]|0;
 __ZNSt3__26vectorIfNS_9allocatorIfEEE8allocateEm($15,$27);
 $28 = HEAP32[$14>>2]|0;
 __ZNSt3__26vectorIfNS_9allocatorIfEEE18__construct_at_endEm($15,$28);
 STACKTOP = sp;return;
}
function __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 __ZNSt3__213__vector_baseIfNS_9allocatorIfEEED2Ev($2);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter15InsertImageNoteEPKN3sap2ve4core9ImageNoteEPKNS2_5INodeE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0.0, $19 = 0.0, $20 = 0, $21 = 0, $22 = 0.0, $23 = 0.0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0.0, $42 = 0.0, $43 = 0, $44 = 0.0, $45 = 0.0, $46 = 0.0, $47 = 0.0;
 var $48 = 0, $49 = 0, $5 = 0, $50 = 0.0, $51 = 0.0, $52 = 0, $53 = 0, $54 = 0, $55 = 0.0, $56 = 0.0, $57 = 0.0, $58 = 0.0, $59 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 80|0;
 $3 = sp + 68|0;
 $4 = sp + 64|0;
 $5 = sp + 60|0;
 $6 = sp + 56|0;
 $7 = sp + 44|0;
 $8 = sp + 40|0;
 $9 = sp + 28|0;
 $10 = sp + 16|0;
 $11 = sp + 4|0;
 $12 = sp;
 HEAP32[$3>>2] = $0;
 HEAP32[$4>>2] = $1;
 HEAP32[$5>>2] = $2;
 $13 = HEAP32[$3>>2]|0;
 $14 = HEAP32[$4>>2]|0;
 $15 = (__ZNK3sap2ve4core9ImageNote18GetLabelDefinitionEv($14)|0);
 HEAP32[$6>>2] = $15;
 $16 = HEAP32[$6>>2]|0;
 $17 = (__ZNK3sap2ve4core15LabelDefinition8GetWidthEv($16)|0);
 $18 = +HEAPF32[$17>>2];
 $19 = $18 * 0.5;
 $20 = HEAP32[$6>>2]|0;
 $21 = (__ZNK3sap2ve4core15LabelDefinition9GetHeightEv($20)|0);
 $22 = +HEAPF32[$21>>2];
 $23 = $22 * 0.5;
 __ZN3sap2ve6VectorC2Efff($7,$19,$23,0.0);
 $24 = HEAP32[$5>>2]|0;
 $25 = HEAP32[$24>>2]|0;
 $26 = ((($25)) + 96|0);
 $27 = HEAP32[$26>>2]|0;
 $28 = (FUNCTION_TABLE_ii[$27 & 2047]($24)|0);
 HEAP32[$8>>2] = $28;
 $29 = HEAP32[$8>>2]|0;
 $30 = HEAP32[$4>>2]|0;
 $31 = (__ZNK3sap2ve4core9ImageNote11GetPositionEv($30)|0);
 __ZNK3sap2ve4core15AnnotationPoint8GetPointEv($11,$31);
 __ZN3sap2veplERKNS0_6VectorES3_($10,$11,$7);
 __ZN3sap2vemlERKNS0_6MatrixERKNS0_6VectorE($9,$29,$10);
 $32 = HEAP32[$6>>2]|0;
 $33 = (__ZNK3sap2ve4core15LabelDefinition11GetMaterialEv($32)|0);
 $34 = (__ZN17DataModelImporter19GetOrCreateMaterialEPKN3sap2ve4core8MaterialE($13,$33)|0);
 HEAP32[$12>>2] = $34;
 $35 = ((($13)) + 24|0);
 $36 = HEAP32[$35>>2]|0;
 $37 = HEAP32[$5>>2]|0;
 $38 = (__ZNK3sap2ve6Vector4DataEv($9)|0);
 $39 = HEAP32[$6>>2]|0;
 $40 = (__ZNK3sap2ve4core15LabelDefinition8GetWidthEv($39)|0);
 $41 = +HEAPF32[$40>>2];
 $42 = $41;
 $43 = HEAP32[$8>>2]|0;
 $44 = +HEAPF64[$43>>3];
 $45 = $42 * $44;
 $46 = $45 * 0.5;
 $47 = $46;
 $48 = HEAP32[$6>>2]|0;
 $49 = (__ZNK3sap2ve4core15LabelDefinition9GetHeightEv($48)|0);
 $50 = +HEAPF32[$49>>2];
 $51 = $50;
 $52 = HEAP32[$8>>2]|0;
 $53 = ((($52)) + 32|0);
 $54 = ((($53)) + 8|0);
 $55 = +HEAPF64[$54>>3];
 $56 = $51 * $55;
 $57 = $56 * 0.5;
 $58 = $57;
 $59 = HEAP32[$12>>2]|0;
 _SceneBuilderAdapter_createImageNote(($36|0),($37|0),($38|0),(+$47),(+$58),($59|0));
 STACKTOP = sp;return;
}
function __ZN3sap2ve6VectorC2Efff($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = +$1;
 $2 = +$2;
 $3 = +$3;
 var $10 = 0, $11 = 0.0, $12 = 0, $13 = 0.0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0.0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $4 = sp + 12|0;
 $5 = sp + 8|0;
 $6 = sp + 4|0;
 $7 = sp;
 HEAP32[$4>>2] = $0;
 HEAPF32[$5>>2] = $1;
 HEAPF32[$6>>2] = $2;
 HEAPF32[$7>>2] = $3;
 $8 = HEAP32[$4>>2]|0;
 $9 = +HEAPF32[$5>>2];
 HEAPF32[$8>>2] = $9;
 $10 = ((($8)) + 4|0);
 $11 = +HEAPF32[$6>>2];
 HEAPF32[$10>>2] = $11;
 $12 = ((($8)) + 8|0);
 $13 = +HEAPF32[$7>>2];
 HEAPF32[$12>>2] = $13;
 STACKTOP = sp;return;
}
function __ZN3sap2veplERKNS0_6VectorES3_($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 32|0;
 $3 = sp + 16|0;
 $4 = sp + 12|0;
 $5 = sp;
 HEAP32[$3>>2] = $1;
 HEAP32[$4>>2] = $2;
 $6 = HEAP32[$4>>2]|0;
 $7 = HEAP32[$3>>2]|0;
 ;HEAP32[$5>>2]=HEAP32[$7>>2]|0;HEAP32[$5+4>>2]=HEAP32[$7+4>>2]|0;HEAP32[$5+8>>2]=HEAP32[$7+8>>2]|0;
 $8 = (__ZN3sap2ve6VectorpLERKS1_($5,$6)|0);
 ;HEAP32[$0>>2]=HEAP32[$8>>2]|0;HEAP32[$0+4>>2]=HEAP32[$8+4>>2]|0;HEAP32[$0+8>>2]=HEAP32[$8+8>>2]|0;
 STACKTOP = sp;return;
}
function __ZNK3sap2ve6Vector4DataEv($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 STACKTOP = sp;return ($2|0);
}
function __ZN17DataModelImporter22InsertParametricObjectEPKN3sap2ve4core16ParametricObjectE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0.0, $106 = 0.0, $107 = 0, $108 = 0.0, $109 = 0.0, $11 = 0, $110 = 0, $111 = 0.0, $112 = 0.0, $113 = 0, $114 = 0.0, $115 = 0.0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0.0, $124 = 0.0, $125 = 0, $126 = 0.0, $127 = 0.0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0.0, $134 = 0.0, $135 = 0;
 var $136 = 0.0, $137 = 0.0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0.0, $143 = 0, $144 = 0, $145 = 0.0, $146 = 0, $147 = 0.0, $148 = 0, $149 = 0, $15 = 0, $150 = 0.0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0.0, $165 = 0.0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0.0, $184 = 0, $185 = 0.0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0;
 var $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0.0, $43 = 0, $44 = 0, $45 = 0, $46 = 0.0, $47 = 0;
 var $48 = 0, $49 = 0, $5 = 0, $50 = 0.0, $51 = 0, $52 = 0.0, $53 = 0, $54 = 0, $55 = 0, $56 = 0.0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0.0, $61 = 0, $62 = 0, $63 = 0, $64 = 0.0, $65 = 0;
 var $66 = 0.0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0.0, $71 = 0, $72 = 0, $73 = 0, $74 = 0.0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0;
 var $84 = 0, $85 = 0, $86 = 0, $87 = 0.0, $88 = 0.0, $89 = 0, $9 = 0, $90 = 0.0, $91 = 0.0, $92 = 0, $93 = 0.0, $94 = 0.0, $95 = 0, $96 = 0.0, $97 = 0.0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 304|0;
 $2 = sp + 288|0;
 $3 = sp + 284|0;
 $4 = sp + 280|0;
 $5 = sp + 268|0;
 $6 = sp + 256|0;
 $7 = sp + 244|0;
 $8 = sp + 232|0;
 $9 = sp + 16|0;
 $10 = sp + 216|0;
 $11 = sp + 200|0;
 $12 = sp + 184|0;
 $13 = sp + 168|0;
 $14 = sp + 152|0;
 $15 = sp + 140|0;
 $16 = sp + 128|0;
 $17 = sp + 116|0;
 $18 = sp + 112|0;
 $19 = sp + 108|0;
 $20 = sp + 104|0;
 $21 = sp + 100|0;
 $22 = sp + 96|0;
 $23 = sp + 88|0;
 $24 = sp + 80|0;
 $25 = sp;
 $26 = sp + 72|0;
 $27 = sp + 68|0;
 $28 = sp + 64|0;
 $29 = sp + 60|0;
 $30 = sp + 56|0;
 $31 = sp + 52|0;
 $32 = sp + 48|0;
 $33 = sp + 44|0;
 $34 = sp + 40|0;
 $35 = sp + 36|0;
 $36 = sp + 32|0;
 HEAP32[$2>>2] = $0;
 HEAP32[$3>>2] = $1;
 $37 = HEAP32[$2>>2]|0;
 $38 = HEAP32[$3>>2]|0;
 $39 = (__ZNK3sap2ve4core16ParametricObject11GetMaterialEv($38)|0);
 $40 = (__ZN17DataModelImporter19GetOrCreateMaterialEPKN3sap2ve4core8MaterialE($37,$39)|0);
 HEAP32[$4>>2] = $40;
 $41 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject12GetTranslateEv($6,$41);
 $42 = +HEAPF32[$6>>2];
 HEAPF32[$5>>2] = $42;
 $43 = ((($5)) + 4|0);
 $44 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject12GetTranslateEv($7,$44);
 $45 = ((($7)) + 4|0);
 $46 = +HEAPF32[$45>>2];
 HEAPF32[$43>>2] = $46;
 $47 = ((($43)) + 4|0);
 $48 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject12GetTranslateEv($8,$48);
 $49 = ((($8)) + 8|0);
 $50 = +HEAPF32[$49>>2];
 HEAPF32[$47>>2] = $50;
 $51 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject9GetRotateEv($10,$51);
 $52 = +HEAPF32[$10>>2];
 HEAPF32[$9>>2] = $52;
 $53 = ((($9)) + 4|0);
 $54 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject9GetRotateEv($11,$54);
 $55 = ((($11)) + 4|0);
 $56 = +HEAPF32[$55>>2];
 HEAPF32[$53>>2] = $56;
 $57 = ((($53)) + 4|0);
 $58 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject9GetRotateEv($12,$58);
 $59 = ((($12)) + 8|0);
 $60 = +HEAPF32[$59>>2];
 HEAPF32[$57>>2] = $60;
 $61 = ((($57)) + 4|0);
 $62 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject9GetRotateEv($13,$62);
 $63 = ((($13)) + 12|0);
 $64 = +HEAPF32[$63>>2];
 HEAPF32[$61>>2] = $64;
 $65 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject8GetScaleEv($15,$65);
 $66 = +HEAPF32[$15>>2];
 HEAPF32[$14>>2] = $66;
 $67 = ((($14)) + 4|0);
 $68 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject8GetScaleEv($16,$68);
 $69 = ((($16)) + 4|0);
 $70 = +HEAPF32[$69>>2];
 HEAPF32[$67>>2] = $70;
 $71 = ((($67)) + 4|0);
 $72 = HEAP32[$3>>2]|0;
 __ZNK3sap2ve4core16ParametricObject8GetScaleEv($17,$72);
 $73 = ((($17)) + 8|0);
 $74 = +HEAPF32[$73>>2];
 HEAPF32[$71>>2] = $74;
 $75 = HEAP32[$3>>2]|0;
 $76 = (__ZNK3sap2ve4core16ParametricObject12GetLineStyleEv($75)|0);
 $77 = HEAP32[$3>>2]|0;
 $78 = (__ZNK3sap2ve4core16ParametricObject12GetFillStyleEv($77)|0);
 __ZN17DataModelImporter18SetParametricStyleEPKN3sap2ve4core9LineStyleEPKNS2_9FillStyleE($37,$76,$78);
 $79 = HEAP32[$3>>2]|0;
 $80 = HEAP32[$79>>2]|0;
 $81 = ((($80)) + 12|0);
 $82 = HEAP32[$81>>2]|0;
 $83 = (FUNCTION_TABLE_ii[$82 & 2047]($79)|0);
 do {
  switch ($83|0) {
  case 57:  {
   $84 = HEAP32[$3>>2]|0;
   HEAP32[$18>>2] = $84;
   $85 = HEAP32[$4>>2]|0;
   $86 = HEAP32[$18>>2]|0;
   $87 = (+__ZNK3sap2ve4core13ParametricArc9GetRadiusEv($86));
   $88 = $87;
   $89 = HEAP32[$18>>2]|0;
   $90 = (+__ZNK3sap2ve4core13ParametricArc9GetRadiusEv($89));
   $91 = $90;
   $92 = HEAP32[$18>>2]|0;
   $93 = (+__ZNK3sap2ve4core13ParametricArc13GetStartAngleEv($92));
   $94 = $93;
   $95 = HEAP32[$18>>2]|0;
   $96 = (+__ZNK3sap2ve4core13ParametricArc11GetEndAngleEv($95));
   $97 = $96;
   $98 = HEAP32[$18>>2]|0;
   $99 = (__ZNK3sap2ve4core13ParametricArc23GetFillStrokeVisibilityEv($98)|0);
   $100 = HEAP32[$18>>2]|0;
   $101 = (__ZNK3sap2ve4core13ParametricArc14GetClosureTypeEv($100)|0);
   _SceneBuilderAdapter_insertParametricArc(($5|0),($9|0),($14|0),($85|0),(+$88),(+$91),(+$94),(+$97),($99|0),($101|0));
   STACKTOP = sp;return;
   break;
  }
  case 103:  {
   $102 = HEAP32[$3>>2]|0;
   HEAP32[$19>>2] = $102;
   $103 = HEAP32[$4>>2]|0;
   $104 = HEAP32[$19>>2]|0;
   $105 = (+__ZNK3sap2ve4core23ParametricEllipticalArc14GetMajorRadiusEv($104));
   $106 = $105;
   $107 = HEAP32[$19>>2]|0;
   $108 = (+__ZNK3sap2ve4core23ParametricEllipticalArc14GetMinorRadiusEv($107));
   $109 = $108;
   $110 = HEAP32[$19>>2]|0;
   $111 = (+__ZNK3sap2ve4core23ParametricEllipticalArc13GetStartAngleEv($110));
   $112 = $111;
   $113 = HEAP32[$19>>2]|0;
   $114 = (+__ZNK3sap2ve4core23ParametricEllipticalArc11GetEndAngleEv($113));
   $115 = $114;
   $116 = HEAP32[$19>>2]|0;
   $117 = (__ZNK3sap2ve4core23ParametricEllipticalArc23GetFillStrokeVisibilityEv($116)|0);
   $118 = HEAP32[$19>>2]|0;
   $119 = (__ZNK3sap2ve4core23ParametricEllipticalArc14GetClosureTypeEv($118)|0);
   _SceneBuilderAdapter_insertParametricArc(($5|0),($9|0),($14|0),($103|0),(+$106),(+$109),(+$112),(+$115),($117|0),($119|0));
   STACKTOP = sp;return;
   break;
  }
  case 55:  {
   $120 = HEAP32[$3>>2]|0;
   HEAP32[$20>>2] = $120;
   $121 = HEAP32[$4>>2]|0;
   $122 = HEAP32[$20>>2]|0;
   $123 = (+__ZNK3sap2ve4core16ParametricCircle15GetCircleRadiusEv($122));
   $124 = $123;
   $125 = HEAP32[$20>>2]|0;
   $126 = (+__ZNK3sap2ve4core16ParametricCircle15GetCircleRadiusEv($125));
   $127 = $126;
   $128 = HEAP32[$20>>2]|0;
   $129 = (__ZNK3sap2ve4core16ParametricCircle23GetFillStrokeVisibilityEv($128)|0);
   _SceneBuilderAdapter_insertParametricEllipse(($5|0),($9|0),($14|0),($121|0),(+$124),(+$127),($129|0));
   STACKTOP = sp;return;
   break;
  }
  case 102:  {
   $130 = HEAP32[$3>>2]|0;
   HEAP32[$21>>2] = $130;
   $131 = HEAP32[$4>>2]|0;
   $132 = HEAP32[$21>>2]|0;
   $133 = (+__ZNK3sap2ve4core17ParametricEllipse14GetMajorRadiusEv($132));
   $134 = $133;
   $135 = HEAP32[$21>>2]|0;
   $136 = (+__ZNK3sap2ve4core17ParametricEllipse14GetMinorRadiusEv($135));
   $137 = $136;
   $138 = HEAP32[$21>>2]|0;
   $139 = (__ZNK3sap2ve4core17ParametricEllipse23GetFillStrokeVisibilityEv($138)|0);
   _SceneBuilderAdapter_insertParametricEllipse(($5|0),($9|0),($14|0),($131|0),(+$134),(+$137),($139|0));
   STACKTOP = sp;return;
   break;
  }
  case 104:  {
   $140 = HEAP32[$3>>2]|0;
   HEAP32[$22>>2] = $140;
   __ZN3sap2ve8Vector2fC2Ev($23);
   __ZN3sap2ve8Vector2fC2Ev($24);
   $141 = HEAP32[$22>>2]|0;
   __ZNK3sap2ve4core14ParametricLine9GetPointsERNS0_8Vector2fES4_($141,$23,$24);
   $142 = +HEAPF32[$23>>2];
   HEAPF32[$25>>2] = $142;
   $143 = ((($25)) + 4|0);
   $144 = ((($23)) + 4|0);
   $145 = +HEAPF32[$144>>2];
   HEAPF32[$143>>2] = $145;
   $146 = ((($143)) + 4|0);
   $147 = +HEAPF32[$24>>2];
   HEAPF32[$146>>2] = $147;
   $148 = ((($146)) + 4|0);
   $149 = ((($24)) + 4|0);
   $150 = +HEAPF32[$149>>2];
   HEAPF32[$148>>2] = $150;
   $151 = HEAP32[$4>>2]|0;
   _SceneBuilderAdapter_insertParametricPolyline(($5|0),($9|0),($14|0),($151|0),4,($25|0),2,0);
   STACKTOP = sp;return;
   break;
  }
  case 105:  {
   $152 = HEAP32[$3>>2]|0;
   HEAP32[$26>>2] = $152;
   $153 = HEAP32[$26>>2]|0;
   __ZNK3sap2ve4core18ParametricPolyline9GetPointsERPKfRj($153,$28,$27);
   $154 = HEAP32[$4>>2]|0;
   $155 = HEAP32[$27>>2]|0;
   $156 = HEAP32[$28>>2]|0;
   $157 = HEAP32[$26>>2]|0;
   $158 = (__ZNK3sap2ve4core18ParametricPolyline12GetDimensionEv($157)|0);
   $159 = $158&255;
   $160 = HEAP32[$26>>2]|0;
   $161 = (__ZNK3sap2ve4core18ParametricPolyline9GetClosedEv($160)|0);
   _SceneBuilderAdapter_insertParametricPolyline(($5|0),($9|0),($14|0),($154|0),($155|0),($156|0),($159|0),($161|0));
   STACKTOP = sp;return;
   break;
  }
  case 56:  {
   $162 = HEAP32[$3>>2]|0;
   HEAP32[$29>>2] = $162;
   $163 = HEAP32[$29>>2]|0;
   $164 = (+__ZNK3sap2ve4core17ParametricPolygon9GetRadiusEv($163));
   $165 = $164;
   $166 = HEAP32[$29>>2]|0;
   $167 = (__ZNK3sap2ve4core17ParametricPolygon8GetSidesEv($166)|0);
   $168 = _emscripten_asm_const_idi(0,(+$165),($167|0))|0;
   STACKTOP = sp;return;
   break;
  }
  case 106:  {
   $169 = HEAP32[$3>>2]|0;
   HEAP32[$30>>2] = $169;
   HEAP32[$31>>2] = 0;
   while(1) {
    $170 = HEAP32[$31>>2]|0;
    $171 = HEAP32[$30>>2]|0;
    $172 = (__ZNK3sap2ve4core14ParametricPath15GetSegmentCountEv($171)|0);
    $173 = ($170>>>0)<($172>>>0);
    if (!($173)) {
     break;
    }
    $174 = HEAP32[$30>>2]|0;
    $175 = HEAP32[$31>>2]|0;
    $176 = (__ZNK3sap2ve4core14ParametricPath10GetSegmentEj($174,$175)|0);
    __ZN17DataModelImporter27InsertParametricPathSegmentEPKN3sap2ve4core24ParametricAnyPathSegmentE($37,$176);
    $177 = HEAP32[$31>>2]|0;
    $178 = (($177) + 1)|0;
    HEAP32[$31>>2] = $178;
   }
   $179 = HEAP32[$4>>2]|0;
   _SceneBuilderAdapter_insertParametricPath(($5|0),($9|0),($14|0),($179|0));
   STACKTOP = sp;return;
   break;
  }
  case 65:  {
   $180 = HEAP32[$3>>2]|0;
   HEAP32[$32>>2] = $180;
   $181 = HEAP32[$4>>2]|0;
   $182 = HEAP32[$32>>2]|0;
   $183 = (+__ZNK3sap2ve4core19ParametricRectangle8GetWidthEv($182));
   $184 = HEAP32[$32>>2]|0;
   $185 = (+__ZNK3sap2ve4core19ParametricRectangle9GetLengthEv($184));
   _SceneBuilderAdapter_insertParametricRectangle(($5|0),($9|0),($14|0),($181|0),(+$183),(+$185));
   STACKTOP = sp;return;
   break;
  }
  case 107:  {
   $186 = HEAP32[$3>>2]|0;
   HEAP32[$33>>2] = $186;
   $187 = HEAP32[$33>>2]|0;
   $188 = (__ZNK3sap2ve4core14ParametricText8GetStyleEv($187)|0);
   $189 = HEAP32[$33>>2]|0;
   $190 = HEAP32[$189>>2]|0;
   $191 = ((($190)) + 72|0);
   $192 = HEAP32[$191>>2]|0;
   $193 = (FUNCTION_TABLE_ii[$192 & 2047]($189)|0);
   __ZN17DataModelImporter26BeginParametricTextContentEPKN3sap2ve4core9TextStyleEj($37,$188,$193);
   HEAP32[$34>>2] = 0;
   while(1) {
    $194 = HEAP32[$34>>2]|0;
    $195 = HEAP32[$33>>2]|0;
    $196 = (__ZNK3sap2ve4core14ParametricText15GetContentCountEv($195)|0);
    $197 = ($194>>>0)<($196>>>0);
    if (!($197)) {
     break;
    }
    $198 = HEAP32[$33>>2]|0;
    $199 = HEAP32[$34>>2]|0;
    $200 = (__ZNK3sap2ve4core14ParametricText10GetContentEm($198,$199)|0);
    __ZN17DataModelImporter27InsertParametricTextContentEPKN3sap2ve4core21ParametricTextContentE($37,$200);
    $201 = HEAP32[$34>>2]|0;
    $202 = (($201) + 1)|0;
    HEAP32[$34>>2] = $202;
   }
   __ZN17DataModelImporter24EndParametricTextContentEv($37);
   $203 = HEAP32[$33>>2]|0;
   $204 = (__ZNK3sap2ve4core14ParametricText16GetTextTranslateEv($203)|0);
   HEAP32[$35>>2] = $204;
   $205 = HEAP32[$33>>2]|0;
   $206 = (__ZNK3sap2ve4core14ParametricText17GetTranslateDeltaEv($205)|0);
   HEAP32[$36>>2] = $206;
   $207 = HEAP32[$4>>2]|0;
   $208 = HEAP32[$35>>2]|0;
   $209 = HEAP32[$36>>2]|0;
   $210 = ($209|0)!=(0|0);
   $211 = HEAP32[$36>>2]|0;
   $212 = $210 ? $211 : 0;
   _SceneBuilderAdapter_insertParametricText(($5|0),($9|0),($14|0),($207|0),($208|0),($212|0));
   STACKTOP = sp;return;
   break;
  }
  default: {
   $213 = HEAP32[$3>>2]|0;
   $214 = HEAP32[$213>>2]|0;
   $215 = ((($214)) + 12|0);
   $216 = HEAP32[$215>>2]|0;
   $217 = (FUNCTION_TABLE_ii[$216 & 2047]($213)|0);
   $218 = _emscripten_asm_const_ii(1,($217|0))|0;
   STACKTOP = sp;return;
  }
  }
 } while(0);
}
function __ZN17DataModelImporter18SetParametricStyleEPKN3sap2ve4core9LineStyleEPKNS2_9FillStyleE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0.0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0.0, $183 = 0, $184 = 0, $185 = 0, $186 = 0.0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0.0, $191 = 0, $192 = 0.0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0.0, $204 = 0, $205 = 0.0, $206 = 0, $207 = 0, $208 = 0;
 var $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0;
 var $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0;
 var $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0;
 var $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0.0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0;
 var $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0;
 var $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0.0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0;
 var $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0;
 var $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0.0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0;
 var $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0;
 var $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0.0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0;
 var $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0;
 var $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0.0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0;
 var $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0;
 var $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0;
 var $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0;
 var $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 784|0;
 $3 = sp + 756|0;
 $4 = sp + 752|0;
 $5 = sp + 748|0;
 $6 = sp + 744|0;
 $7 = sp + 740|0;
 $8 = sp + 736|0;
 $9 = sp + 732|0;
 $10 = sp + 728|0;
 $11 = sp + 724|0;
 $12 = sp + 720|0;
 $13 = sp + 716|0;
 $14 = sp + 712|0;
 $15 = sp + 708|0;
 $16 = sp + 704|0;
 $17 = sp + 48|0;
 $18 = sp + 700|0;
 $19 = sp + 696|0;
 $20 = sp + 692|0;
 $21 = sp + 688|0;
 $22 = sp + 771|0;
 $23 = sp + 684|0;
 $24 = sp + 680|0;
 $25 = sp + 676|0;
 $26 = sp + 672|0;
 $27 = sp + 668|0;
 $28 = sp + 664|0;
 $29 = sp + 660|0;
 $30 = sp + 656|0;
 $31 = sp + 652|0;
 $32 = sp + 770|0;
 $33 = sp + 648|0;
 $34 = sp + 644|0;
 $35 = sp + 640|0;
 $36 = sp + 636|0;
 $37 = sp + 632|0;
 $38 = sp + 628|0;
 $39 = sp + 624|0;
 $40 = sp + 620|0;
 $41 = sp + 616|0;
 $42 = sp + 612|0;
 $43 = sp + 608|0;
 $44 = sp + 604|0;
 $45 = sp + 40|0;
 $46 = sp + 600|0;
 $47 = sp + 596|0;
 $48 = sp + 592|0;
 $49 = sp + 588|0;
 $50 = sp + 769|0;
 $51 = sp + 584|0;
 $52 = sp + 580|0;
 $53 = sp + 576|0;
 $54 = sp + 572|0;
 $55 = sp + 568|0;
 $56 = sp + 564|0;
 $57 = sp + 560|0;
 $58 = sp + 556|0;
 $59 = sp + 552|0;
 $60 = sp + 768|0;
 $61 = sp + 548|0;
 $62 = sp + 544|0;
 $63 = sp + 540|0;
 $64 = sp + 536|0;
 $65 = sp + 532|0;
 $66 = sp + 528|0;
 $67 = sp + 524|0;
 $68 = sp + 520|0;
 $69 = sp + 516|0;
 $70 = sp + 512|0;
 $71 = sp + 508|0;
 $72 = sp + 504|0;
 $73 = sp + 32|0;
 $74 = sp + 500|0;
 $75 = sp + 496|0;
 $76 = sp + 492|0;
 $77 = sp + 488|0;
 $78 = sp + 767|0;
 $79 = sp + 484|0;
 $80 = sp + 480|0;
 $81 = sp + 476|0;
 $82 = sp + 472|0;
 $83 = sp + 468|0;
 $84 = sp + 464|0;
 $85 = sp + 460|0;
 $86 = sp + 456|0;
 $87 = sp + 452|0;
 $88 = sp + 766|0;
 $89 = sp + 448|0;
 $90 = sp + 444|0;
 $91 = sp + 440|0;
 $92 = sp + 436|0;
 $93 = sp + 432|0;
 $94 = sp + 428|0;
 $95 = sp + 424|0;
 $96 = sp + 420|0;
 $97 = sp + 416|0;
 $98 = sp + 412|0;
 $99 = sp + 408|0;
 $100 = sp + 404|0;
 $101 = sp + 24|0;
 $102 = sp + 400|0;
 $103 = sp + 396|0;
 $104 = sp + 392|0;
 $105 = sp + 388|0;
 $106 = sp + 765|0;
 $107 = sp + 384|0;
 $108 = sp + 380|0;
 $109 = sp + 376|0;
 $110 = sp + 372|0;
 $111 = sp + 368|0;
 $112 = sp + 364|0;
 $113 = sp + 360|0;
 $114 = sp + 356|0;
 $115 = sp + 352|0;
 $116 = sp + 764|0;
 $117 = sp + 348|0;
 $118 = sp + 344|0;
 $119 = sp + 340|0;
 $120 = sp + 336|0;
 $121 = sp + 332|0;
 $122 = sp + 328|0;
 $123 = sp + 324|0;
 $124 = sp + 320|0;
 $125 = sp + 316|0;
 $126 = sp + 312|0;
 $127 = sp + 308|0;
 $128 = sp + 304|0;
 $129 = sp + 16|0;
 $130 = sp + 300|0;
 $131 = sp + 296|0;
 $132 = sp + 292|0;
 $133 = sp + 288|0;
 $134 = sp + 763|0;
 $135 = sp + 284|0;
 $136 = sp + 280|0;
 $137 = sp + 276|0;
 $138 = sp + 272|0;
 $139 = sp + 268|0;
 $140 = sp + 264|0;
 $141 = sp + 260|0;
 $142 = sp + 256|0;
 $143 = sp + 252|0;
 $144 = sp + 762|0;
 $145 = sp + 248|0;
 $146 = sp + 244|0;
 $147 = sp + 240|0;
 $148 = sp + 236|0;
 $149 = sp + 232|0;
 $150 = sp + 228|0;
 $151 = sp + 224|0;
 $152 = sp + 220|0;
 $153 = sp + 216|0;
 $154 = sp + 212|0;
 $155 = sp + 208|0;
 $156 = sp + 204|0;
 $157 = sp + 200|0;
 $158 = sp + 196|0;
 $159 = sp + 192|0;
 $160 = sp;
 $161 = sp + 176|0;
 $162 = sp + 160|0;
 $163 = sp + 144|0;
 $164 = sp + 128|0;
 $165 = sp + 116|0;
 $166 = sp + 104|0;
 $167 = sp + 96|0;
 $168 = sp + 88|0;
 $169 = sp + 76|0;
 $170 = sp + 761|0;
 $171 = sp + 760|0;
 $172 = sp + 72|0;
 $173 = sp + 56|0;
 HEAP32[$157>>2] = $0;
 HEAP32[$158>>2] = $1;
 HEAP32[$159>>2] = $2;
 $174 = HEAP32[$158>>2]|0;
 $175 = ($174|0)!=(0|0);
 if ($175) {
  $176 = HEAP32[$158>>2]|0;
  __ZNK3sap2ve4core9LineStyle9GetColourEv($161,$176);
  $177 = (__ZNK3sap2ve4core6Colour3RedEv($161)|0);
  $178 = +HEAPF32[$177>>2];
  HEAPF32[$160>>2] = $178;
  $179 = ((($160)) + 4|0);
  $180 = HEAP32[$158>>2]|0;
  __ZNK3sap2ve4core9LineStyle9GetColourEv($162,$180);
  $181 = (__ZNK3sap2ve4core6Colour5GreenEv($162)|0);
  $182 = +HEAPF32[$181>>2];
  HEAPF32[$179>>2] = $182;
  $183 = ((($179)) + 4|0);
  $184 = HEAP32[$158>>2]|0;
  __ZNK3sap2ve4core9LineStyle9GetColourEv($163,$184);
  $185 = (__ZNK3sap2ve4core6Colour4BlueEv($163)|0);
  $186 = +HEAPF32[$185>>2];
  HEAPF32[$183>>2] = $186;
  $187 = ((($183)) + 4|0);
  $188 = HEAP32[$158>>2]|0;
  __ZNK3sap2ve4core9LineStyle9GetColourEv($164,$188);
  $189 = (__ZNK3sap2ve4core6Colour5AlphaEv($164)|0);
  $190 = +HEAPF32[$189>>2];
  HEAPF32[$187>>2] = $190;
  $191 = HEAP32[$158>>2]|0;
  $192 = (+__ZNK3sap2ve4core9LineStyle8GetWidthEv($191));
  $193 = HEAP32[$158>>2]|0;
  $194 = HEAP32[$193>>2]|0;
  $195 = ((($194)) + 60|0);
  $196 = HEAP32[$195>>2]|0;
  $197 = (FUNCTION_TABLE_ii[$196 & 2047]($193)|0);
  $198 = HEAP32[$158>>2]|0;
  $199 = (__ZNK3sap2ve4core9LineStyle14GetEndCapStyleEv($198)|0);
  $200 = HEAP32[$158>>2]|0;
  $201 = (__ZNK3sap2ve4core9LineStyle16GetLineJoinStyleEv($200)|0);
  $202 = HEAP32[$158>>2]|0;
  $203 = (+__ZNK3sap2ve4core9LineStyle13GetMiterLimitEv($202));
  $204 = HEAP32[$158>>2]|0;
  $205 = (+__ZNK3sap2ve4core9LineStyle20GetDashPatternOffsetEv($204));
  $206 = HEAP32[$158>>2]|0;
  $207 = (__ZNK3sap2ve4core9LineStyle18GetDashArrayLengthEv($206)|0);
  $208 = $207&255;
  $209 = HEAP32[$158>>2]|0;
  $210 = (__ZNK3sap2ve4core9LineStyle12GetDashArrayEv($209)|0);
  _SceneBuilderAdapter_setLineStyle(($160|0),(+$192),($197|0),($199|0),($201|0),(+$203),(+$205),($208|0),($210|0));
 } else {
  _SceneBuilderAdapter_clearLineStyle();
 }
 $211 = HEAP32[$159>>2]|0;
 $212 = ($211|0)!=(0|0);
 if (!($212)) {
  _SceneBuilderAdapter_clearFillStyle();
  STACKTOP = sp;return;
 }
 ;HEAP32[$165>>2]=0|0;HEAP32[$165+4>>2]=0|0;HEAP32[$165+8>>2]=0|0;
 ;HEAP32[$166>>2]=0|0;HEAP32[$166+4>>2]=0|0;HEAP32[$166+8>>2]=0|0;
 $213 = HEAP32[$159>>2]|0;
 $214 = ((($165)) + 4|0);
 $215 = ((($165)) + 8|0);
 __ZNK3sap2ve4core9FillStyle8GetStartERfS3_S3_($213,$165,$214,$215);
 $216 = HEAP32[$159>>2]|0;
 $217 = ((($166)) + 4|0);
 $218 = ((($166)) + 8|0);
 __ZNK3sap2ve4core9FillStyle6GetEndERfS3_S3_($216,$166,$217,$218);
 ;HEAP32[$167>>2]=0|0;HEAP32[$167+4>>2]=0|0;
 $219 = HEAP32[$159>>2]|0;
 $220 = ((($167)) + 4|0);
 __ZNK3sap2ve4core9FillStyle8GetRadiiERfS3_($219,$167,$220);
 $221 = HEAP32[$159>>2]|0;
 $222 = (__ZNK3sap2ve4core9FillStyle10GetPatternEv($221)|0);
 HEAP32[$168>>2] = $222;
 HEAP32[$156>>2] = $169;
 $223 = HEAP32[$156>>2]|0;
 HEAP32[$154>>2] = $223;
 $224 = HEAP32[$154>>2]|0;
 HEAP32[$153>>2] = $224;
 HEAP32[$224>>2] = 0;
 $225 = ((($224)) + 4|0);
 HEAP32[$225>>2] = 0;
 $226 = ((($224)) + 8|0);
 HEAP32[$155>>2] = 0;
 HEAP32[$151>>2] = $226;
 HEAP32[$152>>2] = $155;
 $227 = HEAP32[$151>>2]|0;
 $228 = HEAP32[$152>>2]|0;
 HEAP32[$150>>2] = $228;
 $229 = HEAP32[$150>>2]|0;
 HEAP32[$146>>2] = $227;
 HEAP32[$147>>2] = $229;
 $230 = HEAP32[$146>>2]|0;
 $231 = HEAP32[$147>>2]|0;
 HEAP32[$145>>2] = $231;
 HEAP32[$230>>2] = 0;
 HEAP32[$149>>2] = $227;
 $232 = HEAP32[$149>>2]|0;
 HEAP32[$148>>2] = $232;
 $233 = HEAP32[$159>>2]|0;
 $234 = (__ZNK3sap2ve4core9FillStyle11GetNumStopsEv($233)|0);
 $235 = $234&255;
 $236 = ($235*5)|0;
 __ZNSt3__26vectorIfNS_9allocatorIfEEE7reserveEm($169,$236);
 HEAP8[$170>>0] = 0;
 $237 = HEAP32[$159>>2]|0;
 $238 = (__ZNK3sap2ve4core9FillStyle11GetNumStopsEv($237)|0);
 HEAP8[$171>>0] = $238;
 while(1) {
  $239 = HEAP8[$170>>0]|0;
  $240 = $239&255;
  $241 = HEAP8[$171>>0]|0;
  $242 = $241&255;
  $243 = ($240|0)<($242|0);
  if (!($243)) {
   break;
  }
  __ZN3sap2ve4core6ColourC2Ev($173);
  $244 = HEAP32[$159>>2]|0;
  $245 = HEAP8[$170>>0]|0;
  (__ZNK3sap2ve4core9FillStyle14GetStopByIndexEhRfRNS1_6ColourE($244,$245,$172,$173)|0);
  HEAP32[$142>>2] = $169;
  HEAP32[$143>>2] = $172;
  $246 = HEAP32[$142>>2]|0;
  $247 = ((($246)) + 4|0);
  $248 = HEAP32[$247>>2]|0;
  HEAP32[$141>>2] = $246;
  $249 = HEAP32[$141>>2]|0;
  $250 = ((($249)) + 8|0);
  HEAP32[$140>>2] = $250;
  $251 = HEAP32[$140>>2]|0;
  HEAP32[$139>>2] = $251;
  $252 = HEAP32[$139>>2]|0;
  $253 = HEAP32[$252>>2]|0;
  $254 = ($248|0)!=($253|0);
  if ($254) {
   HEAP32[$136>>2] = $144;
   HEAP32[$137>>2] = $246;
   HEAP32[$138>>2] = 1;
   HEAP32[$119>>2] = $246;
   $255 = HEAP32[$119>>2]|0;
   $256 = ((($255)) + 8|0);
   HEAP32[$118>>2] = $256;
   $257 = HEAP32[$118>>2]|0;
   HEAP32[$117>>2] = $257;
   $258 = HEAP32[$117>>2]|0;
   $259 = ((($246)) + 4|0);
   $260 = HEAP32[$259>>2]|0;
   HEAP32[$120>>2] = $260;
   $261 = HEAP32[$120>>2]|0;
   $262 = HEAP32[$143>>2]|0;
   HEAP32[$131>>2] = $258;
   HEAP32[$132>>2] = $261;
   HEAP32[$133>>2] = $262;
   $263 = HEAP32[$131>>2]|0;
   $264 = HEAP32[$132>>2]|0;
   $265 = HEAP32[$133>>2]|0;
   HEAP32[$130>>2] = $265;
   $266 = HEAP32[$130>>2]|0;
   ;HEAP8[$129>>0]=HEAP8[$134>>0]|0;
   HEAP32[$126>>2] = $263;
   HEAP32[$127>>2] = $264;
   HEAP32[$128>>2] = $266;
   $267 = HEAP32[$126>>2]|0;
   $268 = HEAP32[$127>>2]|0;
   $269 = HEAP32[$128>>2]|0;
   HEAP32[$125>>2] = $269;
   $270 = HEAP32[$125>>2]|0;
   HEAP32[$122>>2] = $267;
   HEAP32[$123>>2] = $268;
   HEAP32[$124>>2] = $270;
   $271 = HEAP32[$123>>2]|0;
   $272 = HEAP32[$124>>2]|0;
   HEAP32[$121>>2] = $272;
   $273 = HEAP32[$121>>2]|0;
   $274 = +HEAPF32[$273>>2];
   HEAPF32[$271>>2] = $274;
   HEAP32[$135>>2] = $144;
   $275 = ((($246)) + 4|0);
   $276 = HEAP32[$275>>2]|0;
   $277 = ((($276)) + 4|0);
   HEAP32[$275>>2] = $277;
  } else {
   $278 = HEAP32[$143>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE21__push_back_slow_pathIRKfEEvOT_($246,$278);
  }
  $279 = (__ZNK3sap2ve4core6Colour3RedEv($173)|0);
  HEAP32[$114>>2] = $169;
  HEAP32[$115>>2] = $279;
  $280 = HEAP32[$114>>2]|0;
  $281 = ((($280)) + 4|0);
  $282 = HEAP32[$281>>2]|0;
  HEAP32[$113>>2] = $280;
  $283 = HEAP32[$113>>2]|0;
  $284 = ((($283)) + 8|0);
  HEAP32[$112>>2] = $284;
  $285 = HEAP32[$112>>2]|0;
  HEAP32[$111>>2] = $285;
  $286 = HEAP32[$111>>2]|0;
  $287 = HEAP32[$286>>2]|0;
  $288 = ($282|0)!=($287|0);
  if ($288) {
   HEAP32[$108>>2] = $116;
   HEAP32[$109>>2] = $280;
   HEAP32[$110>>2] = 1;
   HEAP32[$91>>2] = $280;
   $289 = HEAP32[$91>>2]|0;
   $290 = ((($289)) + 8|0);
   HEAP32[$90>>2] = $290;
   $291 = HEAP32[$90>>2]|0;
   HEAP32[$89>>2] = $291;
   $292 = HEAP32[$89>>2]|0;
   $293 = ((($280)) + 4|0);
   $294 = HEAP32[$293>>2]|0;
   HEAP32[$92>>2] = $294;
   $295 = HEAP32[$92>>2]|0;
   $296 = HEAP32[$115>>2]|0;
   HEAP32[$103>>2] = $292;
   HEAP32[$104>>2] = $295;
   HEAP32[$105>>2] = $296;
   $297 = HEAP32[$103>>2]|0;
   $298 = HEAP32[$104>>2]|0;
   $299 = HEAP32[$105>>2]|0;
   HEAP32[$102>>2] = $299;
   $300 = HEAP32[$102>>2]|0;
   ;HEAP8[$101>>0]=HEAP8[$106>>0]|0;
   HEAP32[$98>>2] = $297;
   HEAP32[$99>>2] = $298;
   HEAP32[$100>>2] = $300;
   $301 = HEAP32[$98>>2]|0;
   $302 = HEAP32[$99>>2]|0;
   $303 = HEAP32[$100>>2]|0;
   HEAP32[$97>>2] = $303;
   $304 = HEAP32[$97>>2]|0;
   HEAP32[$94>>2] = $301;
   HEAP32[$95>>2] = $302;
   HEAP32[$96>>2] = $304;
   $305 = HEAP32[$95>>2]|0;
   $306 = HEAP32[$96>>2]|0;
   HEAP32[$93>>2] = $306;
   $307 = HEAP32[$93>>2]|0;
   $308 = +HEAPF32[$307>>2];
   HEAPF32[$305>>2] = $308;
   HEAP32[$107>>2] = $116;
   $309 = ((($280)) + 4|0);
   $310 = HEAP32[$309>>2]|0;
   $311 = ((($310)) + 4|0);
   HEAP32[$309>>2] = $311;
  } else {
   $312 = HEAP32[$115>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE21__push_back_slow_pathIRKfEEvOT_($280,$312);
  }
  $313 = (__ZNK3sap2ve4core6Colour5GreenEv($173)|0);
  HEAP32[$86>>2] = $169;
  HEAP32[$87>>2] = $313;
  $314 = HEAP32[$86>>2]|0;
  $315 = ((($314)) + 4|0);
  $316 = HEAP32[$315>>2]|0;
  HEAP32[$85>>2] = $314;
  $317 = HEAP32[$85>>2]|0;
  $318 = ((($317)) + 8|0);
  HEAP32[$84>>2] = $318;
  $319 = HEAP32[$84>>2]|0;
  HEAP32[$83>>2] = $319;
  $320 = HEAP32[$83>>2]|0;
  $321 = HEAP32[$320>>2]|0;
  $322 = ($316|0)!=($321|0);
  if ($322) {
   HEAP32[$80>>2] = $88;
   HEAP32[$81>>2] = $314;
   HEAP32[$82>>2] = 1;
   HEAP32[$63>>2] = $314;
   $323 = HEAP32[$63>>2]|0;
   $324 = ((($323)) + 8|0);
   HEAP32[$62>>2] = $324;
   $325 = HEAP32[$62>>2]|0;
   HEAP32[$61>>2] = $325;
   $326 = HEAP32[$61>>2]|0;
   $327 = ((($314)) + 4|0);
   $328 = HEAP32[$327>>2]|0;
   HEAP32[$64>>2] = $328;
   $329 = HEAP32[$64>>2]|0;
   $330 = HEAP32[$87>>2]|0;
   HEAP32[$75>>2] = $326;
   HEAP32[$76>>2] = $329;
   HEAP32[$77>>2] = $330;
   $331 = HEAP32[$75>>2]|0;
   $332 = HEAP32[$76>>2]|0;
   $333 = HEAP32[$77>>2]|0;
   HEAP32[$74>>2] = $333;
   $334 = HEAP32[$74>>2]|0;
   ;HEAP8[$73>>0]=HEAP8[$78>>0]|0;
   HEAP32[$70>>2] = $331;
   HEAP32[$71>>2] = $332;
   HEAP32[$72>>2] = $334;
   $335 = HEAP32[$70>>2]|0;
   $336 = HEAP32[$71>>2]|0;
   $337 = HEAP32[$72>>2]|0;
   HEAP32[$69>>2] = $337;
   $338 = HEAP32[$69>>2]|0;
   HEAP32[$66>>2] = $335;
   HEAP32[$67>>2] = $336;
   HEAP32[$68>>2] = $338;
   $339 = HEAP32[$67>>2]|0;
   $340 = HEAP32[$68>>2]|0;
   HEAP32[$65>>2] = $340;
   $341 = HEAP32[$65>>2]|0;
   $342 = +HEAPF32[$341>>2];
   HEAPF32[$339>>2] = $342;
   HEAP32[$79>>2] = $88;
   $343 = ((($314)) + 4|0);
   $344 = HEAP32[$343>>2]|0;
   $345 = ((($344)) + 4|0);
   HEAP32[$343>>2] = $345;
  } else {
   $346 = HEAP32[$87>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE21__push_back_slow_pathIRKfEEvOT_($314,$346);
  }
  $347 = (__ZNK3sap2ve4core6Colour4BlueEv($173)|0);
  HEAP32[$58>>2] = $169;
  HEAP32[$59>>2] = $347;
  $348 = HEAP32[$58>>2]|0;
  $349 = ((($348)) + 4|0);
  $350 = HEAP32[$349>>2]|0;
  HEAP32[$57>>2] = $348;
  $351 = HEAP32[$57>>2]|0;
  $352 = ((($351)) + 8|0);
  HEAP32[$56>>2] = $352;
  $353 = HEAP32[$56>>2]|0;
  HEAP32[$55>>2] = $353;
  $354 = HEAP32[$55>>2]|0;
  $355 = HEAP32[$354>>2]|0;
  $356 = ($350|0)!=($355|0);
  if ($356) {
   HEAP32[$52>>2] = $60;
   HEAP32[$53>>2] = $348;
   HEAP32[$54>>2] = 1;
   HEAP32[$35>>2] = $348;
   $357 = HEAP32[$35>>2]|0;
   $358 = ((($357)) + 8|0);
   HEAP32[$34>>2] = $358;
   $359 = HEAP32[$34>>2]|0;
   HEAP32[$33>>2] = $359;
   $360 = HEAP32[$33>>2]|0;
   $361 = ((($348)) + 4|0);
   $362 = HEAP32[$361>>2]|0;
   HEAP32[$36>>2] = $362;
   $363 = HEAP32[$36>>2]|0;
   $364 = HEAP32[$59>>2]|0;
   HEAP32[$47>>2] = $360;
   HEAP32[$48>>2] = $363;
   HEAP32[$49>>2] = $364;
   $365 = HEAP32[$47>>2]|0;
   $366 = HEAP32[$48>>2]|0;
   $367 = HEAP32[$49>>2]|0;
   HEAP32[$46>>2] = $367;
   $368 = HEAP32[$46>>2]|0;
   ;HEAP8[$45>>0]=HEAP8[$50>>0]|0;
   HEAP32[$42>>2] = $365;
   HEAP32[$43>>2] = $366;
   HEAP32[$44>>2] = $368;
   $369 = HEAP32[$42>>2]|0;
   $370 = HEAP32[$43>>2]|0;
   $371 = HEAP32[$44>>2]|0;
   HEAP32[$41>>2] = $371;
   $372 = HEAP32[$41>>2]|0;
   HEAP32[$38>>2] = $369;
   HEAP32[$39>>2] = $370;
   HEAP32[$40>>2] = $372;
   $373 = HEAP32[$39>>2]|0;
   $374 = HEAP32[$40>>2]|0;
   HEAP32[$37>>2] = $374;
   $375 = HEAP32[$37>>2]|0;
   $376 = +HEAPF32[$375>>2];
   HEAPF32[$373>>2] = $376;
   HEAP32[$51>>2] = $60;
   $377 = ((($348)) + 4|0);
   $378 = HEAP32[$377>>2]|0;
   $379 = ((($378)) + 4|0);
   HEAP32[$377>>2] = $379;
  } else {
   $380 = HEAP32[$59>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE21__push_back_slow_pathIRKfEEvOT_($348,$380);
  }
  $381 = (__ZNK3sap2ve4core6Colour5AlphaEv($173)|0);
  HEAP32[$30>>2] = $169;
  HEAP32[$31>>2] = $381;
  $382 = HEAP32[$30>>2]|0;
  $383 = ((($382)) + 4|0);
  $384 = HEAP32[$383>>2]|0;
  HEAP32[$29>>2] = $382;
  $385 = HEAP32[$29>>2]|0;
  $386 = ((($385)) + 8|0);
  HEAP32[$28>>2] = $386;
  $387 = HEAP32[$28>>2]|0;
  HEAP32[$27>>2] = $387;
  $388 = HEAP32[$27>>2]|0;
  $389 = HEAP32[$388>>2]|0;
  $390 = ($384|0)!=($389|0);
  if ($390) {
   HEAP32[$24>>2] = $32;
   HEAP32[$25>>2] = $382;
   HEAP32[$26>>2] = 1;
   HEAP32[$7>>2] = $382;
   $391 = HEAP32[$7>>2]|0;
   $392 = ((($391)) + 8|0);
   HEAP32[$6>>2] = $392;
   $393 = HEAP32[$6>>2]|0;
   HEAP32[$5>>2] = $393;
   $394 = HEAP32[$5>>2]|0;
   $395 = ((($382)) + 4|0);
   $396 = HEAP32[$395>>2]|0;
   HEAP32[$8>>2] = $396;
   $397 = HEAP32[$8>>2]|0;
   $398 = HEAP32[$31>>2]|0;
   HEAP32[$19>>2] = $394;
   HEAP32[$20>>2] = $397;
   HEAP32[$21>>2] = $398;
   $399 = HEAP32[$19>>2]|0;
   $400 = HEAP32[$20>>2]|0;
   $401 = HEAP32[$21>>2]|0;
   HEAP32[$18>>2] = $401;
   $402 = HEAP32[$18>>2]|0;
   ;HEAP8[$17>>0]=HEAP8[$22>>0]|0;
   HEAP32[$14>>2] = $399;
   HEAP32[$15>>2] = $400;
   HEAP32[$16>>2] = $402;
   $403 = HEAP32[$14>>2]|0;
   $404 = HEAP32[$15>>2]|0;
   $405 = HEAP32[$16>>2]|0;
   HEAP32[$13>>2] = $405;
   $406 = HEAP32[$13>>2]|0;
   HEAP32[$10>>2] = $403;
   HEAP32[$11>>2] = $404;
   HEAP32[$12>>2] = $406;
   $407 = HEAP32[$11>>2]|0;
   $408 = HEAP32[$12>>2]|0;
   HEAP32[$9>>2] = $408;
   $409 = HEAP32[$9>>2]|0;
   $410 = +HEAPF32[$409>>2];
   HEAPF32[$407>>2] = $410;
   HEAP32[$23>>2] = $32;
   $411 = ((($382)) + 4|0);
   $412 = HEAP32[$411>>2]|0;
   $413 = ((($412)) + 4|0);
   HEAP32[$411>>2] = $413;
  } else {
   $414 = HEAP32[$31>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE21__push_back_slow_pathIRKfEEvOT_($382,$414);
  }
  $415 = HEAP8[$170>>0]|0;
  $416 = (($415) + 1)<<24>>24;
  HEAP8[$170>>0] = $416;
 }
 $417 = HEAP32[$159>>2]|0;
 $418 = (__ZNK3sap2ve4core9FillStyle7GetTypeEv($417)|0);
 $419 = HEAP32[$159>>2]|0;
 $420 = (__ZNK3sap2ve4core9FillStyle11GetNumStopsEv($419)|0);
 $421 = $420&255;
 HEAP32[$4>>2] = $169;
 $422 = HEAP32[$4>>2]|0;
 $423 = HEAP32[$422>>2]|0;
 HEAP32[$3>>2] = $423;
 $424 = HEAP32[$3>>2]|0;
 $425 = HEAP32[$159>>2]|0;
 $426 = (__ZNK3sap2ve4core9FillStyle18GetCoordinateSpaceEv($425)|0);
 $427 = HEAP32[$159>>2]|0;
 $428 = (__ZNK3sap2ve4core9FillStyle20GetGradientTransformEv($427)|0);
 $429 = (__ZNK3sap2ve6Matrix4DataEv($428)|0);
 $430 = HEAP32[$159>>2]|0;
 $431 = (__ZNK3sap2ve4core9FillStyle15GetSpreadMethodEv($430)|0);
 _SceneBuilderAdapter_setFillStyle(($418|0),($165|0),($166|0),($167|0),($421|0),($424|0),($426|0),($429|0),($431|0));
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($169);
 STACKTOP = sp;return;
}
function __ZN3sap2ve8Vector2fC2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 HEAPF32[$2>>2] = 0.0;
 $3 = ((($2)) + 4|0);
 HEAPF32[$3>>2] = 0.0;
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter27InsertParametricPathSegmentEPKN3sap2ve4core24ParametricAnyPathSegmentE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0.0, $38 = 0, $39 = 0.0, $4 = 0, $40 = 0, $41 = 0.0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0;
 var $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0;
 var $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 48|0;
 $2 = sp + 32|0;
 $3 = sp + 28|0;
 $4 = sp + 24|0;
 $5 = sp + 20|0;
 $6 = sp + 16|0;
 $7 = sp + 12|0;
 $8 = sp + 8|0;
 $9 = sp + 4|0;
 $10 = sp;
 HEAP32[$2>>2] = $0;
 HEAP32[$3>>2] = $1;
 HEAP32[$4>>2] = 0;
 HEAP32[$5>>2] = 0;
 $11 = HEAP32[$3>>2]|0;
 $12 = HEAP32[$11>>2]|0;
 $13 = ((($12)) + 16|0);
 $14 = HEAP32[$13>>2]|0;
 $15 = (FUNCTION_TABLE_ii[$14 & 2047]($11)|0);
 switch ($15|0) {
 case 0:  {
  $16 = HEAP32[$3>>2]|0;
  HEAP32[$6>>2] = $16;
  $17 = HEAP32[$6>>2]|0;
  __ZNK3sap2ve4core21ParametricPathSegment9GetPointsERPKfRj($17,$5,$4);
  $18 = HEAP32[$4>>2]|0;
  $19 = HEAP32[$5>>2]|0;
  $20 = HEAP32[$6>>2]|0;
  $21 = (__ZNK3sap2ve4core21ParametricPathSegment12GetDimensionEv($20)|0);
  $22 = $21&255;
  $23 = HEAP32[$6>>2]|0;
  $24 = (__ZNK3sap2ve4core21ParametricPathSegment13GetIsRelativeEv($23)|0);
  _SceneBuilderAdapter_insertMovePathSegment(($18|0),($19|0),($22|0),($24|0));
  STACKTOP = sp;return;
  break;
 }
 case 1:  {
  $25 = HEAP32[$3>>2]|0;
  HEAP32[$7>>2] = $25;
  $26 = HEAP32[$7>>2]|0;
  __ZNK3sap2ve4core21ParametricPathSegment9GetPointsERPKfRj($26,$5,$4);
  $27 = HEAP32[$4>>2]|0;
  $28 = HEAP32[$5>>2]|0;
  $29 = HEAP32[$7>>2]|0;
  $30 = (__ZNK3sap2ve4core21ParametricPathSegment12GetDimensionEv($29)|0);
  $31 = $30&255;
  $32 = HEAP32[$7>>2]|0;
  $33 = (__ZNK3sap2ve4core21ParametricPathSegment13GetIsRelativeEv($32)|0);
  _SceneBuilderAdapter_insertLinePathSegment(($27|0),($28|0),($31|0),($33|0));
  STACKTOP = sp;return;
  break;
 }
 case 2:  {
  $34 = HEAP32[$3>>2]|0;
  HEAP32[$8>>2] = $34;
  $35 = HEAP32[$8>>2]|0;
  __ZNK3sap2ve4core21ParametricPathSegment9GetPointsERPKfRj($35,$5,$4);
  $36 = HEAP32[$8>>2]|0;
  $37 = (+__ZNK3sap2ve4core14ArcPathSegment14GetMajorRadiusEv($36));
  $38 = HEAP32[$8>>2]|0;
  $39 = (+__ZNK3sap2ve4core14ArcPathSegment14GetMinorRadiusEv($38));
  $40 = HEAP32[$8>>2]|0;
  $41 = (+__ZNK3sap2ve4core14ArcPathSegment8GetAngleEv($40));
  $42 = HEAP32[$8>>2]|0;
  $43 = (__ZNK3sap2ve4core14ArcPathSegment17GetFollowLargeArcEv($42)|0);
  $44 = HEAP32[$8>>2]|0;
  $45 = (__ZNK3sap2ve4core14ArcPathSegment12GetClockwiseEv($44)|0);
  $46 = HEAP32[$4>>2]|0;
  $47 = HEAP32[$5>>2]|0;
  $48 = HEAP32[$8>>2]|0;
  $49 = (__ZNK3sap2ve4core21ParametricPathSegment12GetDimensionEv($48)|0);
  $50 = $49&255;
  $51 = HEAP32[$8>>2]|0;
  $52 = (__ZNK3sap2ve4core21ParametricPathSegment13GetIsRelativeEv($51)|0);
  _SceneBuilderAdapter_insertArcPathSegment((+$37),(+$39),(+$41),($43|0),($45|0),($46|0),($47|0),($50|0),($52|0));
  STACKTOP = sp;return;
  break;
 }
 case 3:  {
  $53 = HEAP32[$3>>2]|0;
  HEAP32[$9>>2] = $53;
  $54 = HEAP32[$9>>2]|0;
  __ZNK3sap2ve4core21ParametricPathSegment9GetPointsERPKfRj($54,$5,$4);
  $55 = HEAP32[$9>>2]|0;
  $56 = (__ZNK3sap2ve4core17BezierPathSegment9GetDegreeEv($55)|0);
  $57 = $56&255;
  $58 = HEAP32[$4>>2]|0;
  $59 = HEAP32[$5>>2]|0;
  $60 = HEAP32[$9>>2]|0;
  $61 = (__ZNK3sap2ve4core21ParametricPathSegment12GetDimensionEv($60)|0);
  $62 = $61&255;
  $63 = HEAP32[$9>>2]|0;
  $64 = (__ZNK3sap2ve4core21ParametricPathSegment13GetIsRelativeEv($63)|0);
  _SceneBuilderAdapter_insertBezierPathSegment(($57|0),($58|0),($59|0),($62|0),($64|0));
  STACKTOP = sp;return;
  break;
 }
 case 4:  {
  $65 = HEAP32[$3>>2]|0;
  HEAP32[$10>>2] = $65;
  _SceneBuilderAdapter_insertClosePathSegment();
  STACKTOP = sp;return;
  break;
 }
 default: {
  $66 = HEAP32[$3>>2]|0;
  $67 = HEAP32[$66>>2]|0;
  $68 = ((($67)) + 16|0);
  $69 = HEAP32[$68>>2]|0;
  $70 = (FUNCTION_TABLE_ii[$69 & 2047]($66)|0);
  $71 = _emscripten_asm_const_ii(2,($70|0))|0;
  STACKTOP = sp;return;
 }
 }
}
function __ZN17DataModelImporter26BeginParametricTextContentEPKN3sap2ve4core9TextStyleEj($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0.0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0.0, $20 = 0, $21 = 0.0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0.0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0;
 var $48 = 0.0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp + 8|0;
 $4 = sp + 4|0;
 $5 = sp;
 HEAP32[$3>>2] = $0;
 HEAP32[$4>>2] = $1;
 HEAP32[$5>>2] = $2;
 $6 = HEAP32[$3>>2]|0;
 $7 = HEAP32[$4>>2]|0;
 $8 = (__ZNK3sap2ve4core9TextStyle12GetLineStyleEv($7)|0);
 $9 = HEAP32[$4>>2]|0;
 $10 = (__ZNK3sap2ve4core9TextStyle12GetFillStyleEv($9)|0);
 __ZN17DataModelImporter18SetParametricStyleEPKN3sap2ve4core9LineStyleEPKNS2_9FillStyleE($6,$8,$10);
 $11 = HEAP32[$5>>2]|0;
 $12 = HEAP32[$4>>2]|0;
 $13 = (+__ZNK3sap2ve4core9TextStyle7GetSizeEv($12));
 $14 = HEAP32[$4>>2]|0;
 $15 = (__ZNK3sap2ve4core9TextStyle18GetHorizontalAlignEv($14)|0);
 $16 = HEAP32[$4>>2]|0;
 $17 = (__ZNK3sap2ve4core9TextStyle19GetDominantBaselineEv($16)|0);
 $18 = HEAP32[$4>>2]|0;
 $19 = (+__ZNK3sap2ve4core9TextStyle12GetLinePitchEv($18));
 $20 = HEAP32[$4>>2]|0;
 $21 = (+__ZNK3sap2ve4core9TextStyle13GetWidthScaleEv($20));
 $22 = HEAP32[$4>>2]|0;
 $23 = (__ZNK3sap2ve4core9TextStyle11GetFontFaceEv($22)|0);
 $24 = (__ZNK3sap2ve6String4Utf8Ev($23)|0);
 $25 = HEAP32[$4>>2]|0;
 $26 = (__ZNK3sap2ve4core9TextStyle9IsSHXFontEv($25)|0);
 $27 = HEAP32[$4>>2]|0;
 $28 = (__ZNK3sap2ve4core9TextStyle8IsItalicEv($27)|0);
 $29 = HEAP32[$4>>2]|0;
 $30 = (__ZNK3sap2ve4core9TextStyle6IsBoldEv($29)|0);
 $31 = HEAP32[$4>>2]|0;
 $32 = (__ZNK3sap2ve4core9TextStyle11IsUnderlineEv($31)|0);
 $33 = HEAP32[$4>>2]|0;
 $34 = (__ZNK3sap2ve4core9TextStyle11HasOverlineEv($33)|0);
 $35 = HEAP32[$4>>2]|0;
 $36 = (__ZNK3sap2ve4core9TextStyle16HasStrikethroughEv($35)|0);
 $37 = HEAP32[$4>>2]|0;
 $38 = (+__ZNK3sap2ve4core9TextStyle9GetExpandEv($37));
 $39 = HEAP32[$4>>2]|0;
 $40 = (__ZNK3sap2ve4core9TextStyle15GetTextHeightAsEv($39)|0);
 $41 = HEAP32[$4>>2]|0;
 $42 = (__ZNK3sap2ve4core9TextStyle12GetDirectionEv($41)|0);
 $43 = HEAP32[$4>>2]|0;
 $44 = (__ZNK3sap2ve4core9TextStyle13GetUpsideDownEv($43)|0);
 $45 = HEAP32[$4>>2]|0;
 $46 = (__ZNK3sap2ve4core9TextStyle11GetBackwardEv($45)|0);
 $47 = HEAP32[$4>>2]|0;
 $48 = (+__ZNK3sap2ve4core9TextStyle17GetObliquingAngleEv($47));
 $49 = HEAP32[$4>>2]|0;
 $50 = (__ZNK3sap2ve4core9TextStyle13GetTextLengthEv($49)|0);
 $51 = HEAP32[$4>>2]|0;
 $52 = (__ZNK3sap2ve4core9TextStyle13GetLengthUnitEv($51)|0);
 $53 = HEAP32[$4>>2]|0;
 $54 = (__ZNK3sap2ve4core9TextStyle15GetLengthAdjustEv($53)|0);
 _SceneBuilderAdapter_beginParametricTextContent(($11|0),(+$13),($15|0),($17|0),(+$19),(+$21),($24|0),($26|0),($28|0),($30|0),($32|0),($34|0),($36|0),(+$38),($40|0),($42|0),($44|0),($46|0),(+$48),($50|0),($52|0),($54|0));
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter27InsertParametricTextContentEPKN3sap2ve4core21ParametricTextContentE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $12 = 0, $13 = 0, $14 = 0;
 var $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0;
 var $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0;
 var $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0;
 var $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0;
 var $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 96|0;
 $2 = sp + 92|0;
 $3 = sp + 88|0;
 $4 = sp + 84|0;
 $5 = sp + 80|0;
 $6 = sp + 76|0;
 $7 = sp + 72|0;
 $8 = sp + 68|0;
 $9 = sp + 64|0;
 $10 = sp + 60|0;
 $11 = sp + 56|0;
 $12 = sp + 52|0;
 $13 = sp + 48|0;
 $14 = sp + 44|0;
 $15 = sp + 40|0;
 $16 = sp + 36|0;
 $17 = sp + 32|0;
 $18 = sp + 28|0;
 $19 = sp + 24|0;
 $20 = sp + 20|0;
 $21 = sp + 16|0;
 $22 = sp + 12|0;
 $23 = sp + 8|0;
 $24 = sp + 4|0;
 $25 = sp;
 HEAP32[$17>>2] = $0;
 HEAP32[$18>>2] = $1;
 $26 = HEAP32[$17>>2]|0;
 $27 = HEAP32[$18>>2]|0;
 $28 = HEAP32[$27>>2]|0;
 $29 = ((($28)) + 16|0);
 $30 = HEAP32[$29>>2]|0;
 $31 = (FUNCTION_TABLE_ii[$30 & 2047]($27)|0);
 switch ($31|0) {
 case 10:  {
  $32 = HEAP32[$18>>2]|0;
  HEAP32[$19>>2] = $32;
  $33 = HEAP32[$19>>2]|0;
  $34 = ((($26)) + 12|0);
  HEAP32[$16>>2] = $34;
  $35 = HEAP32[$16>>2]|0;
  HEAP32[$15>>2] = $35;
  $36 = HEAP32[$15>>2]|0;
  HEAP32[$14>>2] = $36;
  $37 = HEAP32[$14>>2]|0;
  HEAP32[$13>>2] = $37;
  $38 = HEAP32[$13>>2]|0;
  HEAP32[$12>>2] = $38;
  $39 = HEAP32[$12>>2]|0;
  HEAP32[$11>>2] = $39;
  $40 = HEAP32[$11>>2]|0;
  $41 = ((($40)) + 11|0);
  $42 = HEAP8[$41>>0]|0;
  $43 = $42&255;
  $44 = $43 & 128;
  $45 = ($44|0)!=(0);
  if ($45) {
   HEAP32[$5>>2] = $37;
   $46 = HEAP32[$5>>2]|0;
   HEAP32[$4>>2] = $46;
   $47 = HEAP32[$4>>2]|0;
   HEAP32[$3>>2] = $47;
   $48 = HEAP32[$3>>2]|0;
   $49 = HEAP32[$48>>2]|0;
   $55 = $49;
  } else {
   HEAP32[$10>>2] = $37;
   $50 = HEAP32[$10>>2]|0;
   HEAP32[$9>>2] = $50;
   $51 = HEAP32[$9>>2]|0;
   HEAP32[$8>>2] = $51;
   $52 = HEAP32[$8>>2]|0;
   HEAP32[$7>>2] = $52;
   $53 = HEAP32[$7>>2]|0;
   HEAP32[$6>>2] = $53;
   $54 = HEAP32[$6>>2]|0;
   $55 = $54;
  }
  HEAP32[$2>>2] = $55;
  $56 = HEAP32[$2>>2]|0;
  $57 = (__ZNK3sap2ve4core18ParametricTextData7GetTextEPKc($33,$56)|0);
  $58 = (__ZNK3sap2ve6String4Utf8Ev($57)|0);
  $59 = HEAP32[$19>>2]|0;
  $60 = HEAP32[$59>>2]|0;
  $61 = ((($60)) + 16|0);
  $62 = HEAP32[$61>>2]|0;
  $63 = (FUNCTION_TABLE_ii[$62 & 2047]($59)|0);
  _SceneBuilderAdapter_insertParametricTextData(($58|0),($63|0));
  STACKTOP = sp;return;
  break;
 }
 case 11:  {
  $64 = HEAP32[$18>>2]|0;
  HEAP32[$20>>2] = $64;
  $65 = HEAP32[$20>>2]|0;
  $66 = (__ZNK3sap2ve4core18ParametricTextSpan8GetStyleEv($65)|0);
  $67 = HEAP32[$18>>2]|0;
  $68 = HEAP32[$67>>2]|0;
  $69 = ((($68)) + 16|0);
  $70 = HEAP32[$69>>2]|0;
  $71 = (FUNCTION_TABLE_ii[$70 & 2047]($67)|0);
  __ZN17DataModelImporter26BeginParametricTextContentEPKN3sap2ve4core9TextStyleEj($26,$66,$71);
  HEAP32[$21>>2] = 0;
  while(1) {
   $72 = HEAP32[$21>>2]|0;
   $73 = HEAP32[$20>>2]|0;
   $74 = (__ZNK3sap2ve4core18ParametricTextSpan15GetContentCountEv($73)|0);
   $75 = ($72>>>0)<($74>>>0);
   if (!($75)) {
    break;
   }
   $76 = HEAP32[$20>>2]|0;
   $77 = HEAP32[$21>>2]|0;
   $78 = (__ZNK3sap2ve4core18ParametricTextSpan10GetContentEm($76,$77)|0);
   __ZN17DataModelImporter27InsertParametricTextContentEPKN3sap2ve4core21ParametricTextContentE($26,$78);
   $79 = HEAP32[$21>>2]|0;
   $80 = (($79) + 1)|0;
   HEAP32[$21>>2] = $80;
  }
  __ZN17DataModelImporter24EndParametricTextContentEv($26);
  STACKTOP = sp;return;
  break;
 }
 case 12:  {
  $81 = HEAP32[$18>>2]|0;
  HEAP32[$22>>2] = $81;
  $82 = HEAP32[$22>>2]|0;
  $83 = (__ZNK3sap2ve4core18ParametricTextPath8GetStyleEv($82)|0);
  $84 = HEAP32[$18>>2]|0;
  $85 = HEAP32[$84>>2]|0;
  $86 = ((($85)) + 16|0);
  $87 = HEAP32[$86>>2]|0;
  $88 = (FUNCTION_TABLE_ii[$87 & 2047]($84)|0);
  __ZN17DataModelImporter26BeginParametricTextContentEPKN3sap2ve4core9TextStyleEj($26,$83,$88);
  $89 = HEAP32[$22>>2]|0;
  $90 = (__ZNK3sap2ve4core18ParametricTextPath7GetPathEv($89)|0);
  HEAP32[$23>>2] = $90;
  HEAP32[$24>>2] = 0;
  while(1) {
   $91 = HEAP32[$24>>2]|0;
   $92 = HEAP32[$23>>2]|0;
   $93 = (__ZNK3sap2ve4core14ParametricPath15GetSegmentCountEv($92)|0);
   $94 = ($91>>>0)<($93>>>0);
   if (!($94)) {
    break;
   }
   $95 = HEAP32[$23>>2]|0;
   $96 = HEAP32[$24>>2]|0;
   $97 = (__ZNK3sap2ve4core14ParametricPath10GetSegmentEj($95,$96)|0);
   __ZN17DataModelImporter27InsertParametricPathSegmentEPKN3sap2ve4core24ParametricAnyPathSegmentE($26,$97);
   $98 = HEAP32[$24>>2]|0;
   $99 = (($98) + 1)|0;
   HEAP32[$24>>2] = $99;
  }
  HEAP32[$25>>2] = 0;
  while(1) {
   $100 = HEAP32[$25>>2]|0;
   $101 = HEAP32[$22>>2]|0;
   $102 = (__ZNK3sap2ve4core18ParametricTextPath15GetContentCountEv($101)|0);
   $103 = ($100>>>0)<($102>>>0);
   if (!($103)) {
    break;
   }
   $104 = HEAP32[$22>>2]|0;
   $105 = HEAP32[$25>>2]|0;
   $106 = (__ZNK3sap2ve4core18ParametricTextPath10GetContentEm($104,$105)|0);
   __ZN17DataModelImporter27InsertParametricTextContentEPKN3sap2ve4core21ParametricTextContentE($26,$106);
   $107 = HEAP32[$25>>2]|0;
   $108 = (($107) + 1)|0;
   HEAP32[$25>>2] = $108;
  }
  __ZN17DataModelImporter24EndParametricTextContentEv($26);
  STACKTOP = sp;return;
  break;
 }
 default: {
  $109 = HEAP32[$18>>2]|0;
  $110 = HEAP32[$109>>2]|0;
  $111 = ((($110)) + 16|0);
  $112 = HEAP32[$111>>2]|0;
  $113 = (FUNCTION_TABLE_ii[$112 & 2047]($109)|0);
  $114 = _emscripten_asm_const_ii(3,($113|0))|0;
  STACKTOP = sp;return;
 }
 }
}
function __ZN17DataModelImporter24EndParametricTextContentEv($0) {
 $0 = $0|0;
 var $1 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 _SceneBuilderAdapter_endParametricTextContent();
 STACKTOP = sp;return;
}
function __ZNSt3__26vectorIfNS_9allocatorIfEEE7reserveEm($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $5 = 0, $6 = 0, $7 = 0;
 var $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 80|0;
 $2 = sp + 64|0;
 $3 = sp + 60|0;
 $4 = sp + 56|0;
 $5 = sp + 52|0;
 $6 = sp + 48|0;
 $7 = sp + 44|0;
 $8 = sp + 40|0;
 $9 = sp + 36|0;
 $10 = sp + 32|0;
 $11 = sp + 28|0;
 $12 = sp + 24|0;
 $13 = sp + 20|0;
 $14 = sp;
 HEAP32[$11>>2] = $0;
 HEAP32[$12>>2] = $1;
 $15 = HEAP32[$11>>2]|0;
 $16 = HEAP32[$12>>2]|0;
 HEAP32[$10>>2] = $15;
 $17 = HEAP32[$10>>2]|0;
 HEAP32[$9>>2] = $17;
 $18 = HEAP32[$9>>2]|0;
 HEAP32[$8>>2] = $18;
 $19 = HEAP32[$8>>2]|0;
 $20 = ((($19)) + 8|0);
 HEAP32[$7>>2] = $20;
 $21 = HEAP32[$7>>2]|0;
 HEAP32[$6>>2] = $21;
 $22 = HEAP32[$6>>2]|0;
 $23 = HEAP32[$22>>2]|0;
 $24 = HEAP32[$18>>2]|0;
 $25 = $23;
 $26 = $24;
 $27 = (($25) - ($26))|0;
 $28 = (($27|0) / 4)&-1;
 $29 = ($16>>>0)>($28>>>0);
 if (!($29)) {
  STACKTOP = sp;return;
 }
 HEAP32[$5>>2] = $15;
 $30 = HEAP32[$5>>2]|0;
 $31 = ((($30)) + 8|0);
 HEAP32[$4>>2] = $31;
 $32 = HEAP32[$4>>2]|0;
 HEAP32[$3>>2] = $32;
 $33 = HEAP32[$3>>2]|0;
 HEAP32[$13>>2] = $33;
 $34 = HEAP32[$12>>2]|0;
 HEAP32[$2>>2] = $15;
 $35 = HEAP32[$2>>2]|0;
 $36 = ((($35)) + 4|0);
 $37 = HEAP32[$36>>2]|0;
 $38 = HEAP32[$35>>2]|0;
 $39 = $37;
 $40 = $38;
 $41 = (($39) - ($40))|0;
 $42 = (($41|0) / 4)&-1;
 $43 = HEAP32[$13>>2]|0;
 __ZNSt3__214__split_bufferIfRNS_9allocatorIfEEEC2EmmS3_($14,$34,$42,$43);
 __ZNSt3__26vectorIfNS_9allocatorIfEEE26__swap_out_circular_bufferERNS_14__split_bufferIfRS2_EE($15,$14);
 __ZNSt3__214__split_bufferIfRNS_9allocatorIfEEED2Ev($14);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter17GetOrCreateCameraEPKN3sap2ve4core6CameraEPKNS1_6MatrixE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0;
 var $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0;
 var $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0.0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0.0;
 var $84 = 0, $85 = 0.0, $86 = 0, $87 = 0.0, $88 = 0.0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0.0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 192|0;
 $3 = sp + 184|0;
 $4 = sp + 180|0;
 $5 = sp + 176|0;
 $6 = sp + 16|0;
 $7 = sp + 172|0;
 $8 = sp + 168|0;
 $9 = sp + 164|0;
 $10 = sp + 160|0;
 $11 = sp + 156|0;
 $12 = sp + 152|0;
 $13 = sp + 148|0;
 $14 = sp + 144|0;
 $15 = sp + 140|0;
 $16 = sp + 136|0;
 $17 = sp + 128|0;
 $18 = sp + 120|0;
 $19 = sp + 8|0;
 $20 = sp + 116|0;
 $21 = sp + 112|0;
 $22 = sp + 108|0;
 $23 = sp + 104|0;
 $24 = sp + 100|0;
 $25 = sp + 96|0;
 $26 = sp + 92|0;
 $27 = sp + 88|0;
 $28 = sp + 84|0;
 $29 = sp + 80|0;
 $30 = sp + 76|0;
 $31 = sp + 72|0;
 $32 = sp + 68|0;
 $33 = sp;
 $34 = sp + 64|0;
 $35 = sp + 60|0;
 $36 = sp + 56|0;
 $37 = sp + 52|0;
 $38 = sp + 48|0;
 $39 = sp + 44|0;
 $40 = sp + 40|0;
 $41 = sp + 36|0;
 $42 = sp + 32|0;
 $43 = sp + 24|0;
 HEAP32[$38>>2] = $0;
 HEAP32[$39>>2] = $1;
 HEAP32[$40>>2] = $2;
 $44 = HEAP32[$38>>2]|0;
 $45 = ((($44)) + 68|0);
 HEAP32[$35>>2] = $45;
 HEAP32[$36>>2] = $39;
 $46 = HEAP32[$35>>2]|0;
 $47 = HEAP32[$36>>2]|0;
 $48 = (__ZNSt3__26__treeIPKN3sap2ve4core6CameraENS_4lessIS6_EENS_9allocatorIS6_EEE4findIS6_EENS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEERKT_($46,$47)|0);
 HEAP32[$37>>2] = $48;
 ;HEAP8[$33>>0]=HEAP8[$37>>0]|0;HEAP8[$33+1>>0]=HEAP8[$37+1>>0]|0;HEAP8[$33+2>>0]=HEAP8[$37+2>>0]|0;HEAP8[$33+3>>0]=HEAP8[$37+3>>0]|0;
 HEAP32[$32>>2] = $34;
 $49 = HEAP32[$32>>2]|0;
 $50 = HEAP32[$33>>2]|0;
 HEAP32[$49>>2] = $50;
 $51 = HEAP32[$34>>2]|0;
 HEAP32[$41>>2] = $51;
 $52 = ((($44)) + 68|0);
 HEAP32[$30>>2] = $52;
 $53 = HEAP32[$30>>2]|0;
 HEAP32[$28>>2] = $53;
 $54 = HEAP32[$28>>2]|0;
 HEAP32[$26>>2] = $54;
 $55 = HEAP32[$26>>2]|0;
 $56 = ((($55)) + 4|0);
 HEAP32[$25>>2] = $56;
 $57 = HEAP32[$25>>2]|0;
 HEAP32[$24>>2] = $57;
 $58 = HEAP32[$24>>2]|0;
 HEAP32[$23>>2] = $58;
 $59 = HEAP32[$23>>2]|0;
 HEAP32[$22>>2] = $59;
 $60 = HEAP32[$22>>2]|0;
 HEAP32[$20>>2] = $27;
 HEAP32[$21>>2] = $60;
 $61 = HEAP32[$20>>2]|0;
 $62 = HEAP32[$21>>2]|0;
 HEAP32[$61>>2] = $62;
 $63 = HEAP32[$27>>2]|0;
 HEAP32[$31>>2] = $63;
 ;HEAP8[$19>>0]=HEAP8[$31>>0]|0;HEAP8[$19+1>>0]=HEAP8[$31+1>>0]|0;HEAP8[$19+2>>0]=HEAP8[$31+2>>0]|0;HEAP8[$19+3>>0]=HEAP8[$31+3>>0]|0;
 HEAP32[$18>>2] = $29;
 $64 = HEAP32[$18>>2]|0;
 $65 = HEAP32[$19>>2]|0;
 HEAP32[$64>>2] = $65;
 $66 = HEAP32[$29>>2]|0;
 HEAP32[$42>>2] = $66;
 HEAP32[$3>>2] = $41;
 HEAP32[$4>>2] = $42;
 $67 = HEAP32[$3>>2]|0;
 $68 = HEAP32[$67>>2]|0;
 $69 = HEAP32[$4>>2]|0;
 $70 = HEAP32[$69>>2]|0;
 $71 = ($68|0)==($70|0);
 if (!($71)) {
  $119 = HEAP32[$39>>2]|0;
  STACKTOP = sp;return ($119|0);
 }
 $72 = ((($44)) + 24|0);
 $73 = HEAP32[$72>>2]|0;
 $74 = HEAP32[$40>>2]|0;
 $75 = (__ZNK3sap2ve6Matrix4DataEv($74)|0);
 $76 = HEAP32[$39>>2]|0;
 $77 = (__ZNK3sap2ve4core6Camera7IsOrthoEv($76)|0);
 $78 = HEAP32[$39>>2]|0;
 $79 = (+__ZNK3sap2ve4core6Camera9GetAspectEv($78));
 $80 = HEAP32[$39>>2]|0;
 $81 = (__ZNK3sap2ve4core6Camera28ShouldAutoEvaluateClipPlanesEv($80)|0);
 $82 = HEAP32[$39>>2]|0;
 $83 = (+__ZNK3sap2ve4core6Camera11GetNearClipEv($82));
 $84 = HEAP32[$39>>2]|0;
 $85 = (+__ZNK3sap2ve4core6Camera10GetFarClipEv($84));
 $86 = HEAP32[$39>>2]|0;
 $87 = (+__ZNK3sap2ve4core6Camera6GetFOVEv($86));
 $88 = $87 * 57.295780181884766;
 $89 = HEAP32[$39>>2]|0;
 $90 = (__ZNK3sap2ve4core6Camera13GetFOVBindingEv($89)|0);
 $91 = HEAP32[$39>>2]|0;
 $92 = (+__ZNK3sap2ve4core6Camera18GetOrthoZoomFactorEv($91));
 $93 = HEAP32[$39>>2]|0;
 $94 = (__ZNK3sap2ve4core6Camera9GetOriginEv($93)|0);
 $95 = HEAP32[$39>>2]|0;
 $96 = (__ZNK3sap2ve4core6Camera5GetUpEv($95)|0);
 $97 = HEAP32[$39>>2]|0;
 $98 = (__ZNK3sap2ve4core6Camera9GetTargetEv($97)|0);
 $99 = HEAP32[$39>>2]|0;
 _SceneBuilderAdapter_createCamera(($73|0),($75|0),($77|0),(+$79),($81|0),(+$83),(+$85),(+$88),($90|0),(+$92),($94|0),($96|0),($98|0),($99|0));
 $100 = ((($44)) + 68|0);
 HEAP32[$15>>2] = $100;
 HEAP32[$16>>2] = $39;
 $101 = HEAP32[$15>>2]|0;
 $102 = HEAP32[$16>>2]|0;
 HEAP32[$13>>2] = $101;
 HEAP32[$14>>2] = $102;
 $103 = HEAP32[$13>>2]|0;
 $104 = HEAP32[$14>>2]|0;
 HEAP32[$12>>2] = $104;
 $105 = HEAP32[$12>>2]|0;
 $106 = HEAP32[$14>>2]|0;
 __ZNSt3__26__treeIPKN3sap2ve4core6CameraENS_4lessIS6_EENS_9allocatorIS6_EEE25__emplace_unique_key_argsIS6_JRKS6_EEENS_4pairINS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEEbEERKT_DpOT0_($17,$103,$105,$106);
 HEAP32[$9>>2] = $43;
 HEAP32[$10>>2] = $17;
 $107 = HEAP32[$9>>2]|0;
 $108 = HEAP32[$10>>2]|0;
 HEAP32[$8>>2] = $108;
 $109 = HEAP32[$8>>2]|0;
 ;HEAP32[$11>>2]=HEAP32[$109>>2]|0;
 ;HEAP8[$6>>0]=HEAP8[$11>>0]|0;HEAP8[$6+1>>0]=HEAP8[$11+1>>0]|0;HEAP8[$6+2>>0]=HEAP8[$11+2>>0]|0;HEAP8[$6+3>>0]=HEAP8[$11+3>>0]|0;
 HEAP32[$5>>2] = $107;
 $110 = HEAP32[$5>>2]|0;
 $111 = HEAP32[$6>>2]|0;
 HEAP32[$110>>2] = $111;
 $112 = ((($107)) + 4|0);
 $113 = HEAP32[$10>>2]|0;
 $114 = ((($113)) + 4|0);
 HEAP32[$7>>2] = $114;
 $115 = HEAP32[$7>>2]|0;
 $116 = HEAP8[$115>>0]|0;
 $117 = $116&1;
 $118 = $117&1;
 HEAP8[$112>>0] = $118;
 $119 = HEAP32[$39>>2]|0;
 STACKTOP = sp;return ($119|0);
}
function __ZN17DataModelImporter12InsertCameraEPKN3sap2ve4core6CameraEPKNS2_5INodeE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $3 = sp + 8|0;
 $4 = sp + 4|0;
 $5 = sp;
 HEAP32[$3>>2] = $0;
 HEAP32[$4>>2] = $1;
 HEAP32[$5>>2] = $2;
 $6 = HEAP32[$3>>2]|0;
 $7 = HEAP32[$4>>2]|0;
 $8 = HEAP32[$5>>2]|0;
 $9 = HEAP32[$8>>2]|0;
 $10 = ((($9)) + 100|0);
 $11 = HEAP32[$10>>2]|0;
 $12 = (FUNCTION_TABLE_ii[$11 & 2047]($8)|0);
 $13 = (__ZN17DataModelImporter17GetOrCreateCameraEPKN3sap2ve4core6CameraEPKNS1_6MatrixE($6,$7,$12)|0);
 HEAP32[$4>>2] = $13;
 $14 = ((($6)) + 24|0);
 $15 = HEAP32[$14>>2]|0;
 $16 = HEAP32[$5>>2]|0;
 $17 = HEAP32[$4>>2]|0;
 _SceneBuilderAdapter_insertCamera(($15|0),($16|0),($17|0));
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter19InsertViewportGroupEPKN3sap2ve4core13ViewportGroupE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $2 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0;
 var $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0;
 var $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0;
 var $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0;
 var $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0;
 var $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0;
 var $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0;
 var $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0;
 var $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0;
 var $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 432|0;
 $2 = sp + 420|0;
 $3 = sp + 416|0;
 $4 = sp + 412|0;
 $5 = sp + 408|0;
 $6 = sp + 404|0;
 $7 = sp + 400|0;
 $8 = sp + 396|0;
 $9 = sp + 392|0;
 $10 = sp + 388|0;
 $11 = sp + 384|0;
 $12 = sp + 380|0;
 $13 = sp + 376|0;
 $14 = sp + 372|0;
 $15 = sp + 368|0;
 $16 = sp + 364|0;
 $17 = sp + 360|0;
 $18 = sp + 356|0;
 $19 = sp + 352|0;
 $20 = sp + 348|0;
 $21 = sp + 344|0;
 $22 = sp + 340|0;
 $23 = sp + 336|0;
 $24 = sp + 332|0;
 $25 = sp + 328|0;
 $26 = sp + 324|0;
 $27 = sp + 320|0;
 $28 = sp + 316|0;
 $29 = sp + 312|0;
 $30 = sp + 308|0;
 $31 = sp + 304|0;
 $32 = sp + 300|0;
 $33 = sp + 296|0;
 $34 = sp + 292|0;
 $35 = sp + 288|0;
 $36 = sp + 284|0;
 $37 = sp + 280|0;
 $38 = sp + 276|0;
 $39 = sp + 272|0;
 $40 = sp + 268|0;
 $41 = sp + 264|0;
 $42 = sp + 260|0;
 $43 = sp + 256|0;
 $44 = sp + 252|0;
 $45 = sp + 248|0;
 $46 = sp + 244|0;
 $47 = sp + 240|0;
 $48 = sp + 236|0;
 $49 = sp + 232|0;
 $50 = sp + 228|0;
 $51 = sp + 224|0;
 $52 = sp + 220|0;
 $53 = sp + 216|0;
 $54 = sp + 212|0;
 $55 = sp + 208|0;
 $56 = sp + 204|0;
 $57 = sp + 200|0;
 $58 = sp + 196|0;
 $59 = sp + 192|0;
 $60 = sp + 188|0;
 $61 = sp + 184|0;
 $62 = sp + 180|0;
 $63 = sp + 176|0;
 $64 = sp + 172|0;
 $65 = sp + 168|0;
 $66 = sp + 164|0;
 $67 = sp + 160|0;
 $68 = sp + 156|0;
 $69 = sp + 152|0;
 $70 = sp + 148|0;
 $71 = sp + 144|0;
 $72 = sp + 140|0;
 $73 = sp + 136|0;
 $74 = sp + 132|0;
 $75 = sp + 128|0;
 $76 = sp + 124|0;
 $77 = sp + 120|0;
 $78 = sp + 116|0;
 $79 = sp + 112|0;
 $80 = sp + 108|0;
 $81 = sp + 104|0;
 $82 = sp + 100|0;
 $83 = sp + 96|0;
 $84 = sp + 92|0;
 $85 = sp + 88|0;
 $86 = sp + 84|0;
 $87 = sp + 80|0;
 $88 = sp + 76|0;
 $89 = sp + 72|0;
 $90 = sp + 68|0;
 $91 = sp + 64|0;
 $92 = sp + 60|0;
 $93 = sp + 56|0;
 $94 = sp + 52|0;
 $95 = sp + 48|0;
 $96 = sp + 44|0;
 $97 = sp + 40|0;
 $98 = sp + 36|0;
 $99 = sp + 24|0;
 $100 = sp + 12|0;
 $101 = sp + 8|0;
 $102 = sp + 4|0;
 $103 = sp;
 HEAP32[$97>>2] = $0;
 HEAP32[$98>>2] = $1;
 $104 = HEAP32[$97>>2]|0;
 HEAP32[$96>>2] = $99;
 $105 = HEAP32[$96>>2]|0;
 HEAP32[$95>>2] = $105;
 $106 = HEAP32[$95>>2]|0;
 HEAP32[$94>>2] = $106;
 $107 = HEAP32[$94>>2]|0;
 ;HEAP32[$107>>2]=0|0;HEAP32[$107+4>>2]=0|0;HEAP32[$107+8>>2]=0|0;
 HEAP32[$93>>2] = $106;
 $108 = HEAP32[$93>>2]|0;
 HEAP32[$92>>2] = $108;
 HEAP32[$89>>2] = $105;
 $109 = HEAP32[$89>>2]|0;
 HEAP32[$88>>2] = $109;
 $110 = HEAP32[$88>>2]|0;
 HEAP32[$87>>2] = $110;
 $111 = HEAP32[$87>>2]|0;
 HEAP32[$90>>2] = $111;
 HEAP32[$91>>2] = 0;
 while(1) {
  $112 = HEAP32[$91>>2]|0;
  $113 = ($112>>>0)<(3);
  if (!($113)) {
   break;
  }
  $114 = HEAP32[$90>>2]|0;
  $115 = HEAP32[$91>>2]|0;
  $116 = (($114) + ($115<<2)|0);
  HEAP32[$116>>2] = 0;
  $117 = HEAP32[$91>>2]|0;
  $118 = (($117) + 1)|0;
  HEAP32[$91>>2] = $118;
 }
 HEAP32[$71>>2] = $100;
 $119 = HEAP32[$71>>2]|0;
 HEAP32[$70>>2] = $119;
 $120 = HEAP32[$70>>2]|0;
 HEAP32[$69>>2] = $120;
 $121 = HEAP32[$69>>2]|0;
 ;HEAP32[$121>>2]=0|0;HEAP32[$121+4>>2]=0|0;HEAP32[$121+8>>2]=0|0;
 HEAP32[$68>>2] = $120;
 $122 = HEAP32[$68>>2]|0;
 HEAP32[$67>>2] = $122;
 HEAP32[$64>>2] = $119;
 $123 = HEAP32[$64>>2]|0;
 HEAP32[$63>>2] = $123;
 $124 = HEAP32[$63>>2]|0;
 HEAP32[$62>>2] = $124;
 $125 = HEAP32[$62>>2]|0;
 HEAP32[$65>>2] = $125;
 HEAP32[$66>>2] = 0;
 while(1) {
  $126 = HEAP32[$66>>2]|0;
  $127 = ($126>>>0)<(3);
  if (!($127)) {
   break;
  }
  $128 = HEAP32[$65>>2]|0;
  $129 = HEAP32[$66>>2]|0;
  $130 = (($128) + ($129<<2)|0);
  HEAP32[$130>>2] = 0;
  $131 = HEAP32[$66>>2]|0;
  $132 = (($131) + 1)|0;
  HEAP32[$66>>2] = $132;
 }
 $133 = HEAP32[$98>>2]|0;
 __ZN17DataModelImporter19GetViewportMetadataEPKN3sap2ve4core18ViewportDefinitionERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEESD_($133,$99,$100);
 $134 = ((($104)) + 24|0);
 $135 = HEAP32[$134>>2]|0;
 $136 = HEAP32[$98>>2]|0;
 $137 = HEAP32[$98>>2]|0;
 $138 = ((($104)) + 12|0);
 HEAP32[$16>>2] = $138;
 $139 = HEAP32[$16>>2]|0;
 HEAP32[$15>>2] = $139;
 $140 = HEAP32[$15>>2]|0;
 HEAP32[$14>>2] = $140;
 $141 = HEAP32[$14>>2]|0;
 HEAP32[$13>>2] = $141;
 $142 = HEAP32[$13>>2]|0;
 HEAP32[$12>>2] = $142;
 $143 = HEAP32[$12>>2]|0;
 HEAP32[$11>>2] = $143;
 $144 = HEAP32[$11>>2]|0;
 $145 = ((($144)) + 11|0);
 $146 = HEAP8[$145>>0]|0;
 $147 = $146&255;
 $148 = $147 & 128;
 $149 = ($148|0)!=(0);
 if ($149) {
  HEAP32[$5>>2] = $141;
  $150 = HEAP32[$5>>2]|0;
  HEAP32[$4>>2] = $150;
  $151 = HEAP32[$4>>2]|0;
  HEAP32[$3>>2] = $151;
  $152 = HEAP32[$3>>2]|0;
  $153 = HEAP32[$152>>2]|0;
  $159 = $153;
 } else {
  HEAP32[$10>>2] = $141;
  $154 = HEAP32[$10>>2]|0;
  HEAP32[$9>>2] = $154;
  $155 = HEAP32[$9>>2]|0;
  HEAP32[$8>>2] = $155;
  $156 = HEAP32[$8>>2]|0;
  HEAP32[$7>>2] = $156;
  $157 = HEAP32[$7>>2]|0;
  HEAP32[$6>>2] = $157;
  $158 = HEAP32[$6>>2]|0;
  $159 = $158;
 }
 HEAP32[$2>>2] = $159;
 $160 = HEAP32[$2>>2]|0;
 $161 = (__ZNK3sap2ve4core18ViewportDefinition7GetNameEPKc($137,$160)|0);
 $162 = (__ZNK3sap2ve6String4Utf8Ev($161)|0);
 $163 = HEAP32[$98>>2]|0;
 $164 = ((($104)) + 12|0);
 HEAP32[$31>>2] = $164;
 $165 = HEAP32[$31>>2]|0;
 HEAP32[$30>>2] = $165;
 $166 = HEAP32[$30>>2]|0;
 HEAP32[$29>>2] = $166;
 $167 = HEAP32[$29>>2]|0;
 HEAP32[$28>>2] = $167;
 $168 = HEAP32[$28>>2]|0;
 HEAP32[$27>>2] = $168;
 $169 = HEAP32[$27>>2]|0;
 HEAP32[$26>>2] = $169;
 $170 = HEAP32[$26>>2]|0;
 $171 = ((($170)) + 11|0);
 $172 = HEAP8[$171>>0]|0;
 $173 = $172&255;
 $174 = $173 & 128;
 $175 = ($174|0)!=(0);
 if ($175) {
  HEAP32[$20>>2] = $167;
  $176 = HEAP32[$20>>2]|0;
  HEAP32[$19>>2] = $176;
  $177 = HEAP32[$19>>2]|0;
  HEAP32[$18>>2] = $177;
  $178 = HEAP32[$18>>2]|0;
  $179 = HEAP32[$178>>2]|0;
  $185 = $179;
 } else {
  HEAP32[$25>>2] = $167;
  $180 = HEAP32[$25>>2]|0;
  HEAP32[$24>>2] = $180;
  $181 = HEAP32[$24>>2]|0;
  HEAP32[$23>>2] = $181;
  $182 = HEAP32[$23>>2]|0;
  HEAP32[$22>>2] = $182;
  $183 = HEAP32[$22>>2]|0;
  HEAP32[$21>>2] = $183;
  $184 = HEAP32[$21>>2]|0;
  $185 = $184;
 }
 HEAP32[$17>>2] = $185;
 $186 = HEAP32[$17>>2]|0;
 $187 = (__ZNK3sap2ve4core18ViewportDefinition7GetInfoEPKc($163,$186)|0);
 $188 = (__ZNK3sap2ve6String4Utf8Ev($187)|0);
 $189 = HEAP32[$98>>2]|0;
 $190 = (__ZNK3sap2ve4core13ViewportGroup7GetTypeEv($189)|0);
 HEAP32[$46>>2] = $99;
 $191 = HEAP32[$46>>2]|0;
 HEAP32[$45>>2] = $191;
 $192 = HEAP32[$45>>2]|0;
 HEAP32[$44>>2] = $192;
 $193 = HEAP32[$44>>2]|0;
 HEAP32[$43>>2] = $193;
 $194 = HEAP32[$43>>2]|0;
 HEAP32[$42>>2] = $194;
 $195 = HEAP32[$42>>2]|0;
 HEAP32[$41>>2] = $195;
 $196 = HEAP32[$41>>2]|0;
 $197 = ((($196)) + 11|0);
 $198 = HEAP8[$197>>0]|0;
 $199 = $198&255;
 $200 = $199 & 128;
 $201 = ($200|0)!=(0);
 if ($201) {
  HEAP32[$35>>2] = $193;
  $202 = HEAP32[$35>>2]|0;
  HEAP32[$34>>2] = $202;
  $203 = HEAP32[$34>>2]|0;
  HEAP32[$33>>2] = $203;
  $204 = HEAP32[$33>>2]|0;
  $205 = HEAP32[$204>>2]|0;
  $211 = $205;
 } else {
  HEAP32[$40>>2] = $193;
  $206 = HEAP32[$40>>2]|0;
  HEAP32[$39>>2] = $206;
  $207 = HEAP32[$39>>2]|0;
  HEAP32[$38>>2] = $207;
  $208 = HEAP32[$38>>2]|0;
  HEAP32[$37>>2] = $208;
  $209 = HEAP32[$37>>2]|0;
  HEAP32[$36>>2] = $209;
  $210 = HEAP32[$36>>2]|0;
  $211 = $210;
 }
 HEAP32[$32>>2] = $211;
 $212 = HEAP32[$32>>2]|0;
 HEAP32[$61>>2] = $100;
 $213 = HEAP32[$61>>2]|0;
 HEAP32[$60>>2] = $213;
 $214 = HEAP32[$60>>2]|0;
 HEAP32[$59>>2] = $214;
 $215 = HEAP32[$59>>2]|0;
 HEAP32[$58>>2] = $215;
 $216 = HEAP32[$58>>2]|0;
 HEAP32[$57>>2] = $216;
 $217 = HEAP32[$57>>2]|0;
 HEAP32[$56>>2] = $217;
 $218 = HEAP32[$56>>2]|0;
 $219 = ((($218)) + 11|0);
 $220 = HEAP8[$219>>0]|0;
 $221 = $220&255;
 $222 = $221 & 128;
 $223 = ($222|0)!=(0);
 if ($223) {
  HEAP32[$50>>2] = $215;
  $224 = HEAP32[$50>>2]|0;
  HEAP32[$49>>2] = $224;
  $225 = HEAP32[$49>>2]|0;
  HEAP32[$48>>2] = $225;
  $226 = HEAP32[$48>>2]|0;
  $227 = HEAP32[$226>>2]|0;
  $233 = $227;
 } else {
  HEAP32[$55>>2] = $215;
  $228 = HEAP32[$55>>2]|0;
  HEAP32[$54>>2] = $228;
  $229 = HEAP32[$54>>2]|0;
  HEAP32[$53>>2] = $229;
  $230 = HEAP32[$53>>2]|0;
  HEAP32[$52>>2] = $230;
  $231 = HEAP32[$52>>2]|0;
  HEAP32[$51>>2] = $231;
  $232 = HEAP32[$51>>2]|0;
  $233 = $232;
 }
 HEAP32[$47>>2] = $233;
 $234 = HEAP32[$47>>2]|0;
 _SceneBuilderAdapter_createViewportGroup(($135|0),($136|0),($162|0),($188|0),($190|0),($212|0),($234|0));
 HEAP32[$101>>2] = 0;
 $235 = HEAP32[$98>>2]|0;
 $236 = (__ZNK3sap2ve4core13ViewportGroup17GetModelViewCountEv($235)|0);
 HEAP32[$102>>2] = $236;
 while(1) {
  $237 = HEAP32[$101>>2]|0;
  $238 = HEAP32[$102>>2]|0;
  $239 = ($237>>>0)<($238>>>0);
  if (!($239)) {
   break;
  }
  $240 = HEAP32[$98>>2]|0;
  $241 = HEAP32[$101>>2]|0;
  $242 = (__ZNK3sap2ve4core13ViewportGroup12GetModelViewEj($240,$241)|0);
  HEAP32[$103>>2] = $242;
  $243 = HEAP32[$103>>2]|0;
  $244 = HEAP32[$98>>2]|0;
  __ZN17DataModelImporter15InsertModelViewEPKN3sap2ve4core9ModelViewEPKNS2_13ViewportGroupE($104,$243,$244);
  $245 = ((($104)) + 28|0);
  $246 = HEAP32[$245>>2]|0;
  $247 = ($246|0)!=(0|0);
  if (!($247)) {
   $248 = HEAP32[$98>>2]|0;
   $249 = (__ZNK3sap2ve4core13ViewportGroup7GetTypeEv($248)|0);
   $250 = ($249|0)==(0);
   if ($250) {
    $251 = HEAP32[$103>>2]|0;
    $252 = ((($104)) + 12|0);
    HEAP32[$86>>2] = $252;
    $253 = HEAP32[$86>>2]|0;
    HEAP32[$85>>2] = $253;
    $254 = HEAP32[$85>>2]|0;
    HEAP32[$84>>2] = $254;
    $255 = HEAP32[$84>>2]|0;
    HEAP32[$83>>2] = $255;
    $256 = HEAP32[$83>>2]|0;
    HEAP32[$82>>2] = $256;
    $257 = HEAP32[$82>>2]|0;
    HEAP32[$81>>2] = $257;
    $258 = HEAP32[$81>>2]|0;
    $259 = ((($258)) + 11|0);
    $260 = HEAP8[$259>>0]|0;
    $261 = $260&255;
    $262 = $261 & 128;
    $263 = ($262|0)!=(0);
    if ($263) {
     HEAP32[$75>>2] = $255;
     $264 = HEAP32[$75>>2]|0;
     HEAP32[$74>>2] = $264;
     $265 = HEAP32[$74>>2]|0;
     HEAP32[$73>>2] = $265;
     $266 = HEAP32[$73>>2]|0;
     $267 = HEAP32[$266>>2]|0;
     $273 = $267;
    } else {
     HEAP32[$80>>2] = $255;
     $268 = HEAP32[$80>>2]|0;
     HEAP32[$79>>2] = $268;
     $269 = HEAP32[$79>>2]|0;
     HEAP32[$78>>2] = $269;
     $270 = HEAP32[$78>>2]|0;
     HEAP32[$77>>2] = $270;
     $271 = HEAP32[$77>>2]|0;
     HEAP32[$76>>2] = $271;
     $272 = HEAP32[$76>>2]|0;
     $273 = $272;
    }
    HEAP32[$72>>2] = $273;
    $274 = HEAP32[$72>>2]|0;
    $275 = (__ZNK3sap2ve4core18ViewportDefinition7GetNameEPKc($251,$274)|0);
    $276 = (__ZNK3sap2ve6String4Utf8Ev($275)|0);
    $277 = (_strcmp($276,82372)|0);
    $278 = ($277|0)!=(0);
    if (!($278)) {
     $279 = HEAP32[$103>>2]|0;
     $280 = ((($104)) + 28|0);
     HEAP32[$280>>2] = $279;
    }
   }
  }
  $281 = HEAP32[$101>>2]|0;
  $282 = (($281) + 1)|0;
  HEAP32[$101>>2] = $282;
 }
 $283 = ((($104)) + 24|0);
 $284 = HEAP32[$283>>2]|0;
 $285 = HEAP32[$98>>2]|0;
 _SceneBuilderAdapter_finalizePlaybacks(($284|0),($285|0));
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($100);
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($99);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter19GetViewportMetadataEPKN3sap2ve4core18ViewportDefinitionERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEESD_($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0;
 var $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0;
 var $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 64|0;
 $3 = sp + 52|0;
 $4 = sp + 48|0;
 $5 = sp + 44|0;
 $6 = sp + 40|0;
 $7 = sp + 36|0;
 $8 = sp + 32|0;
 $9 = sp + 28|0;
 $10 = sp + 24|0;
 $11 = sp + 20|0;
 $12 = sp + 16|0;
 $13 = sp + 12|0;
 $14 = sp + 8|0;
 $15 = sp + 4|0;
 $16 = sp;
 HEAP32[$11>>2] = $0;
 HEAP32[$12>>2] = $1;
 HEAP32[$13>>2] = $2;
 $17 = HEAP32[$12>>2]|0;
 HEAP32[$9>>2] = $17;
 HEAP32[$10>>2] = 82455;
 $18 = HEAP32[$9>>2]|0;
 $19 = HEAP32[$10>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6assignEPKc($18,$19)|0);
 $20 = HEAP32[$11>>2]|0;
 $21 = (__ZNK3sap2ve4core18ViewportDefinition11GetMetadataEv($20)|0);
 $22 = HEAP32[$12>>2]|0;
 __ZN17DataModelImporter15RealiseMetadataEPKN3sap2ve4core8MetadataERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($21,$22);
 $23 = HEAP32[$12>>2]|0;
 HEAP32[$7>>2] = $23;
 HEAP32[$8>>2] = 82453;
 $24 = HEAP32[$7>>2]|0;
 $25 = HEAP32[$8>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($24,$25)|0);
 $26 = HEAP32[$13>>2]|0;
 HEAP32[$5>>2] = $26;
 HEAP32[$6>>2] = 82455;
 $27 = HEAP32[$5>>2]|0;
 $28 = HEAP32[$6>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6assignEPKc($27,$28)|0);
 HEAP32[$14>>2] = 0;
 $29 = HEAP32[$11>>2]|0;
 $30 = HEAP32[$29>>2]|0;
 $31 = ((($30)) + 88|0);
 $32 = HEAP32[$31>>2]|0;
 $33 = (FUNCTION_TABLE_ii[$32 & 2047]($29)|0);
 HEAP32[$15>>2] = $33;
 while(1) {
  $34 = HEAP32[$14>>2]|0;
  $35 = HEAP32[$15>>2]|0;
  $36 = ($34>>>0)<($35>>>0);
  $37 = HEAP32[$11>>2]|0;
  if (!($36)) {
   break;
  }
  $38 = HEAP32[$37>>2]|0;
  $39 = ((($38)) + 92|0);
  $40 = HEAP32[$39>>2]|0;
  $41 = HEAP32[$14>>2]|0;
  $42 = (FUNCTION_TABLE_iii[$40 & 1023]($37,$41)|0);
  $43 = HEAP32[$13>>2]|0;
  __ZN17DataModelImporter11RealiseVEIDERKN3sap2ve4core2IDERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($42,$43);
  $44 = HEAP32[$14>>2]|0;
  $45 = (($44) + 1)|0;
  HEAP32[$14>>2] = $45;
 }
 $46 = HEAP32[$37>>2]|0;
 $47 = ((($46)) + 28|0);
 $48 = HEAP32[$47>>2]|0;
 FUNCTION_TABLE_vii[$48 & 1023]($16,$37);
 $49 = HEAP32[$13>>2]|0;
 __ZN17DataModelImporter11RealiseVEIDERKN3sap2ve4core2IDERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($16,$49);
 __ZN3sap2ve4core2IDD2Ev($16);
 $50 = HEAP32[$13>>2]|0;
 HEAP32[$3>>2] = $50;
 HEAP32[$4>>2] = 82453;
 $51 = HEAP32[$3>>2]|0;
 $52 = HEAP32[$4>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($51,$52)|0);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter15InsertModelViewEPKN3sap2ve4core9ModelViewEPKNS2_13ViewportGroupE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0;
 var $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0, $226 = 0;
 var $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0;
 var $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0;
 var $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0;
 var $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0;
 var $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0, $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0;
 var $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0;
 var $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0;
 var $353 = 0.0, $354 = 0, $355 = 0.0, $356 = 0, $357 = 0.0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0;
 var $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0;
 var $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0;
 var $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0;
 var $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0;
 var $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0;
 var $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0.0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0.0;
 var $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0.0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0.0, $494 = 0.0, $495 = 0.0, $496 = 0, $497 = 0;
 var $498 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0;
 var $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0;
 var $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 752|0;
 $3 = sp + 744|0;
 $4 = sp + 740|0;
 $5 = sp + 736|0;
 $6 = sp + 732|0;
 $7 = sp + 728|0;
 $8 = sp + 724|0;
 $9 = sp + 720|0;
 $10 = sp + 716|0;
 $11 = sp + 712|0;
 $12 = sp + 708|0;
 $13 = sp + 704|0;
 $14 = sp + 700|0;
 $15 = sp + 696|0;
 $16 = sp + 692|0;
 $17 = sp + 688|0;
 $18 = sp + 684|0;
 $19 = sp + 680|0;
 $20 = sp + 676|0;
 $21 = sp + 672|0;
 $22 = sp + 668|0;
 $23 = sp + 664|0;
 $24 = sp + 660|0;
 $25 = sp + 656|0;
 $26 = sp + 652|0;
 $27 = sp + 648|0;
 $28 = sp + 644|0;
 $29 = sp + 640|0;
 $30 = sp + 636|0;
 $31 = sp + 632|0;
 $32 = sp + 628|0;
 $33 = sp + 624|0;
 $34 = sp + 620|0;
 $35 = sp + 616|0;
 $36 = sp + 612|0;
 $37 = sp + 608|0;
 $38 = sp + 604|0;
 $39 = sp + 600|0;
 $40 = sp + 596|0;
 $41 = sp + 592|0;
 $42 = sp + 588|0;
 $43 = sp + 584|0;
 $44 = sp + 580|0;
 $45 = sp + 576|0;
 $46 = sp + 572|0;
 $47 = sp + 568|0;
 $48 = sp + 564|0;
 $49 = sp + 560|0;
 $50 = sp + 556|0;
 $51 = sp + 552|0;
 $52 = sp + 548|0;
 $53 = sp + 544|0;
 $54 = sp + 540|0;
 $55 = sp + 536|0;
 $56 = sp + 532|0;
 $57 = sp + 528|0;
 $58 = sp + 524|0;
 $59 = sp + 520|0;
 $60 = sp + 516|0;
 $61 = sp + 512|0;
 $62 = sp + 508|0;
 $63 = sp + 504|0;
 $64 = sp + 500|0;
 $65 = sp + 496|0;
 $66 = sp + 492|0;
 $67 = sp + 488|0;
 $68 = sp + 484|0;
 $69 = sp + 480|0;
 $70 = sp + 476|0;
 $71 = sp + 472|0;
 $72 = sp + 468|0;
 $73 = sp + 464|0;
 $74 = sp + 460|0;
 $75 = sp + 456|0;
 $76 = sp + 452|0;
 $77 = sp + 448|0;
 $78 = sp + 444|0;
 $79 = sp + 440|0;
 $80 = sp + 436|0;
 $81 = sp + 432|0;
 $82 = sp + 428|0;
 $83 = sp + 424|0;
 $84 = sp + 420|0;
 $85 = sp + 416|0;
 $86 = sp + 412|0;
 $87 = sp + 408|0;
 $88 = sp + 404|0;
 $89 = sp + 400|0;
 $90 = sp + 396|0;
 $91 = sp + 392|0;
 $92 = sp + 388|0;
 $93 = sp + 384|0;
 $94 = sp + 380|0;
 $95 = sp + 376|0;
 $96 = sp + 372|0;
 $97 = sp + 368|0;
 $98 = sp + 24|0;
 $99 = sp + 364|0;
 $100 = sp + 360|0;
 $101 = sp + 356|0;
 $102 = sp + 352|0;
 $103 = sp + 749|0;
 $104 = sp + 348|0;
 $105 = sp + 344|0;
 $106 = sp + 340|0;
 $107 = sp + 336|0;
 $108 = sp + 332|0;
 $109 = sp + 328|0;
 $110 = sp + 324|0;
 $111 = sp + 320|0;
 $112 = sp + 316|0;
 $113 = sp + 748|0;
 $114 = sp + 312|0;
 $115 = sp + 16|0;
 $116 = sp + 308|0;
 $117 = sp + 304|0;
 $118 = sp + 300|0;
 $119 = sp + 296|0;
 $120 = sp + 292|0;
 $121 = sp + 288|0;
 $122 = sp + 284|0;
 $123 = sp + 280|0;
 $124 = sp + 276|0;
 $125 = sp + 272|0;
 $126 = sp + 264|0;
 $127 = sp + 256|0;
 $128 = sp + 252|0;
 $129 = sp + 248|0;
 $130 = sp + 244|0;
 $131 = sp + 240|0;
 $132 = sp + 8|0;
 $133 = sp + 236|0;
 $134 = sp + 232|0;
 $135 = sp + 228|0;
 $136 = sp + 224|0;
 $137 = sp + 220|0;
 $138 = sp + 216|0;
 $139 = sp + 212|0;
 $140 = sp + 208|0;
 $141 = sp + 204|0;
 $142 = sp + 200|0;
 $143 = sp + 196|0;
 $144 = sp + 192|0;
 $145 = sp + 188|0;
 $146 = sp;
 $147 = sp + 184|0;
 $148 = sp + 180|0;
 $149 = sp + 176|0;
 $150 = sp + 172|0;
 $151 = sp + 168|0;
 $152 = sp + 164|0;
 $153 = sp + 160|0;
 $154 = sp + 156|0;
 $155 = sp + 152|0;
 $156 = sp + 144|0;
 $157 = sp + 140|0;
 $158 = sp + 136|0;
 $159 = sp + 124|0;
 $160 = sp + 112|0;
 $161 = sp + 96|0;
 $162 = sp + 80|0;
 $163 = sp + 72|0;
 $164 = sp + 64|0;
 $165 = sp + 60|0;
 $166 = sp + 56|0;
 $167 = sp + 52|0;
 $168 = sp + 48|0;
 $169 = sp + 44|0;
 $170 = sp + 40|0;
 $171 = sp + 28|0;
 HEAP32[$151>>2] = $0;
 HEAP32[$152>>2] = $1;
 HEAP32[$153>>2] = $2;
 $172 = HEAP32[$151>>2]|0;
 $173 = ((($172)) + 104|0);
 HEAP32[$148>>2] = $173;
 HEAP32[$149>>2] = $152;
 $174 = HEAP32[$148>>2]|0;
 $175 = HEAP32[$149>>2]|0;
 $176 = (__ZNSt3__26__treeIPKN3sap2ve4core9ModelViewENS_4lessIS6_EENS_9allocatorIS6_EEE4findIS6_EENS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEERKT_($174,$175)|0);
 HEAP32[$150>>2] = $176;
 ;HEAP8[$146>>0]=HEAP8[$150>>0]|0;HEAP8[$146+1>>0]=HEAP8[$150+1>>0]|0;HEAP8[$146+2>>0]=HEAP8[$150+2>>0]|0;HEAP8[$146+3>>0]=HEAP8[$150+3>>0]|0;
 HEAP32[$145>>2] = $147;
 $177 = HEAP32[$145>>2]|0;
 $178 = HEAP32[$146>>2]|0;
 HEAP32[$177>>2] = $178;
 $179 = HEAP32[$147>>2]|0;
 HEAP32[$154>>2] = $179;
 $180 = ((($172)) + 104|0);
 HEAP32[$143>>2] = $180;
 $181 = HEAP32[$143>>2]|0;
 HEAP32[$141>>2] = $181;
 $182 = HEAP32[$141>>2]|0;
 HEAP32[$139>>2] = $182;
 $183 = HEAP32[$139>>2]|0;
 $184 = ((($183)) + 4|0);
 HEAP32[$138>>2] = $184;
 $185 = HEAP32[$138>>2]|0;
 HEAP32[$137>>2] = $185;
 $186 = HEAP32[$137>>2]|0;
 HEAP32[$136>>2] = $186;
 $187 = HEAP32[$136>>2]|0;
 HEAP32[$135>>2] = $187;
 $188 = HEAP32[$135>>2]|0;
 HEAP32[$133>>2] = $140;
 HEAP32[$134>>2] = $188;
 $189 = HEAP32[$133>>2]|0;
 $190 = HEAP32[$134>>2]|0;
 HEAP32[$189>>2] = $190;
 $191 = HEAP32[$140>>2]|0;
 HEAP32[$144>>2] = $191;
 ;HEAP8[$132>>0]=HEAP8[$144>>0]|0;HEAP8[$132+1>>0]=HEAP8[$144+1>>0]|0;HEAP8[$132+2>>0]=HEAP8[$144+2>>0]|0;HEAP8[$132+3>>0]=HEAP8[$144+3>>0]|0;
 HEAP32[$131>>2] = $142;
 $192 = HEAP32[$131>>2]|0;
 $193 = HEAP32[$132>>2]|0;
 HEAP32[$192>>2] = $193;
 $194 = HEAP32[$142>>2]|0;
 HEAP32[$155>>2] = $194;
 HEAP32[$129>>2] = $154;
 HEAP32[$130>>2] = $155;
 $195 = HEAP32[$129>>2]|0;
 $196 = HEAP32[$130>>2]|0;
 HEAP32[$127>>2] = $195;
 HEAP32[$128>>2] = $196;
 $197 = HEAP32[$127>>2]|0;
 $198 = HEAP32[$197>>2]|0;
 $199 = HEAP32[$128>>2]|0;
 $200 = HEAP32[$199>>2]|0;
 $201 = ($198|0)==($200|0);
 $202 = $201 ^ 1;
 if ($202) {
  STACKTOP = sp;return;
 }
 $203 = ((($172)) + 104|0);
 HEAP32[$124>>2] = $203;
 HEAP32[$125>>2] = $152;
 $204 = HEAP32[$124>>2]|0;
 $205 = HEAP32[$125>>2]|0;
 HEAP32[$122>>2] = $204;
 HEAP32[$123>>2] = $205;
 $206 = HEAP32[$122>>2]|0;
 $207 = HEAP32[$123>>2]|0;
 HEAP32[$121>>2] = $207;
 $208 = HEAP32[$121>>2]|0;
 $209 = HEAP32[$123>>2]|0;
 __ZNSt3__26__treeIPKN3sap2ve4core9ModelViewENS_4lessIS6_EENS_9allocatorIS6_EEE25__emplace_unique_key_argsIS6_JRKS6_EEENS_4pairINS_15__tree_iteratorIS6_PNS_11__tree_nodeIS6_PvEElEEbEERKT_DpOT0_($126,$206,$208,$209);
 HEAP32[$118>>2] = $156;
 HEAP32[$119>>2] = $126;
 $210 = HEAP32[$118>>2]|0;
 $211 = HEAP32[$119>>2]|0;
 HEAP32[$117>>2] = $211;
 $212 = HEAP32[$117>>2]|0;
 ;HEAP32[$120>>2]=HEAP32[$212>>2]|0;
 ;HEAP8[$115>>0]=HEAP8[$120>>0]|0;HEAP8[$115+1>>0]=HEAP8[$120+1>>0]|0;HEAP8[$115+2>>0]=HEAP8[$120+2>>0]|0;HEAP8[$115+3>>0]=HEAP8[$120+3>>0]|0;
 HEAP32[$114>>2] = $210;
 $213 = HEAP32[$114>>2]|0;
 $214 = HEAP32[$115>>2]|0;
 HEAP32[$213>>2] = $214;
 $215 = ((($210)) + 4|0);
 $216 = HEAP32[$119>>2]|0;
 $217 = ((($216)) + 4|0);
 HEAP32[$116>>2] = $217;
 $218 = HEAP32[$116>>2]|0;
 $219 = HEAP8[$218>>0]|0;
 $220 = $219&1;
 $221 = $220&1;
 HEAP8[$215>>0] = $221;
 $222 = ((($172)) + 116|0);
 HEAP32[$111>>2] = $222;
 HEAP32[$112>>2] = $152;
 $223 = HEAP32[$111>>2]|0;
 $224 = ((($223)) + 4|0);
 $225 = HEAP32[$224>>2]|0;
 HEAP32[$110>>2] = $223;
 $226 = HEAP32[$110>>2]|0;
 $227 = ((($226)) + 8|0);
 HEAP32[$109>>2] = $227;
 $228 = HEAP32[$109>>2]|0;
 HEAP32[$108>>2] = $228;
 $229 = HEAP32[$108>>2]|0;
 $230 = HEAP32[$229>>2]|0;
 $231 = ($225|0)!=($230|0);
 if ($231) {
  HEAP32[$105>>2] = $113;
  HEAP32[$106>>2] = $223;
  HEAP32[$107>>2] = 1;
  HEAP32[$88>>2] = $223;
  $232 = HEAP32[$88>>2]|0;
  $233 = ((($232)) + 8|0);
  HEAP32[$87>>2] = $233;
  $234 = HEAP32[$87>>2]|0;
  HEAP32[$86>>2] = $234;
  $235 = HEAP32[$86>>2]|0;
  $236 = ((($223)) + 4|0);
  $237 = HEAP32[$236>>2]|0;
  HEAP32[$89>>2] = $237;
  $238 = HEAP32[$89>>2]|0;
  $239 = HEAP32[$112>>2]|0;
  HEAP32[$100>>2] = $235;
  HEAP32[$101>>2] = $238;
  HEAP32[$102>>2] = $239;
  $240 = HEAP32[$100>>2]|0;
  $241 = HEAP32[$101>>2]|0;
  $242 = HEAP32[$102>>2]|0;
  HEAP32[$99>>2] = $242;
  $243 = HEAP32[$99>>2]|0;
  ;HEAP8[$98>>0]=HEAP8[$103>>0]|0;
  HEAP32[$95>>2] = $240;
  HEAP32[$96>>2] = $241;
  HEAP32[$97>>2] = $243;
  $244 = HEAP32[$95>>2]|0;
  $245 = HEAP32[$96>>2]|0;
  $246 = HEAP32[$97>>2]|0;
  HEAP32[$94>>2] = $246;
  $247 = HEAP32[$94>>2]|0;
  HEAP32[$91>>2] = $244;
  HEAP32[$92>>2] = $245;
  HEAP32[$93>>2] = $247;
  $248 = HEAP32[$92>>2]|0;
  $249 = HEAP32[$93>>2]|0;
  HEAP32[$90>>2] = $249;
  $250 = HEAP32[$90>>2]|0;
  $251 = HEAP32[$250>>2]|0;
  HEAP32[$248>>2] = $251;
  HEAP32[$104>>2] = $113;
  $252 = ((($223)) + 4|0);
  $253 = HEAP32[$252>>2]|0;
  $254 = ((($253)) + 4|0);
  HEAP32[$252>>2] = $254;
 } else {
  $255 = HEAP32[$112>>2]|0;
  __ZNSt3__26vectorIPKN3sap2ve4core9ModelViewENS_9allocatorIS6_EEE21__push_back_slow_pathIRKS6_EEvOT_($223,$255);
 }
 $256 = HEAP32[$152>>2]|0;
 $257 = (__ZNK3sap2ve4core9ModelView18GetDisplaySettingsEv($256)|0);
 HEAP32[$157>>2] = $257;
 $258 = HEAP32[$152>>2]|0;
 $259 = (__ZNK3sap2ve4core9ModelView9GetCameraEv($258)|0);
 HEAP32[$158>>2] = $259;
 $260 = HEAP32[$158>>2]|0;
 (__ZN17DataModelImporter17GetOrCreateCameraEPKN3sap2ve4core6CameraEPKNS1_6MatrixE($172,$260,0)|0);
 HEAP32[$85>>2] = $159;
 $261 = HEAP32[$85>>2]|0;
 HEAP32[$84>>2] = $261;
 $262 = HEAP32[$84>>2]|0;
 HEAP32[$83>>2] = $262;
 $263 = HEAP32[$83>>2]|0;
 ;HEAP32[$263>>2]=0|0;HEAP32[$263+4>>2]=0|0;HEAP32[$263+8>>2]=0|0;
 HEAP32[$82>>2] = $262;
 $264 = HEAP32[$82>>2]|0;
 HEAP32[$81>>2] = $264;
 HEAP32[$78>>2] = $261;
 $265 = HEAP32[$78>>2]|0;
 HEAP32[$77>>2] = $265;
 $266 = HEAP32[$77>>2]|0;
 HEAP32[$76>>2] = $266;
 $267 = HEAP32[$76>>2]|0;
 HEAP32[$79>>2] = $267;
 HEAP32[$80>>2] = 0;
 while(1) {
  $268 = HEAP32[$80>>2]|0;
  $269 = ($268>>>0)<(3);
  if (!($269)) {
   break;
  }
  $270 = HEAP32[$79>>2]|0;
  $271 = HEAP32[$80>>2]|0;
  $272 = (($270) + ($271<<2)|0);
  HEAP32[$272>>2] = 0;
  $273 = HEAP32[$80>>2]|0;
  $274 = (($273) + 1)|0;
  HEAP32[$80>>2] = $274;
 }
 HEAP32[$75>>2] = $160;
 $275 = HEAP32[$75>>2]|0;
 HEAP32[$74>>2] = $275;
 $276 = HEAP32[$74>>2]|0;
 HEAP32[$73>>2] = $276;
 $277 = HEAP32[$73>>2]|0;
 ;HEAP32[$277>>2]=0|0;HEAP32[$277+4>>2]=0|0;HEAP32[$277+8>>2]=0|0;
 HEAP32[$72>>2] = $276;
 $278 = HEAP32[$72>>2]|0;
 HEAP32[$71>>2] = $278;
 HEAP32[$68>>2] = $275;
 $279 = HEAP32[$68>>2]|0;
 HEAP32[$67>>2] = $279;
 $280 = HEAP32[$67>>2]|0;
 HEAP32[$66>>2] = $280;
 $281 = HEAP32[$66>>2]|0;
 HEAP32[$69>>2] = $281;
 HEAP32[$70>>2] = 0;
 while(1) {
  $282 = HEAP32[$70>>2]|0;
  $283 = ($282>>>0)<(3);
  if (!($283)) {
   break;
  }
  $284 = HEAP32[$69>>2]|0;
  $285 = HEAP32[$70>>2]|0;
  $286 = (($284) + ($285<<2)|0);
  HEAP32[$286>>2] = 0;
  $287 = HEAP32[$70>>2]|0;
  $288 = (($287) + 1)|0;
  HEAP32[$70>>2] = $288;
 }
 $289 = HEAP32[$152>>2]|0;
 __ZN17DataModelImporter19GetViewportMetadataEPKN3sap2ve4core18ViewportDefinitionERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEESD_($289,$159,$160);
 $290 = ((($172)) + 24|0);
 $291 = HEAP32[$290>>2]|0;
 $292 = HEAP32[$153>>2]|0;
 $293 = HEAP32[$152>>2]|0;
 $294 = HEAP32[$152>>2]|0;
 $295 = ((($172)) + 12|0);
 HEAP32[$65>>2] = $295;
 $296 = HEAP32[$65>>2]|0;
 HEAP32[$64>>2] = $296;
 $297 = HEAP32[$64>>2]|0;
 HEAP32[$63>>2] = $297;
 $298 = HEAP32[$63>>2]|0;
 HEAP32[$62>>2] = $298;
 $299 = HEAP32[$62>>2]|0;
 HEAP32[$61>>2] = $299;
 $300 = HEAP32[$61>>2]|0;
 HEAP32[$60>>2] = $300;
 $301 = HEAP32[$60>>2]|0;
 $302 = ((($301)) + 11|0);
 $303 = HEAP8[$302>>0]|0;
 $304 = $303&255;
 $305 = $304 & 128;
 $306 = ($305|0)!=(0);
 if ($306) {
  HEAP32[$54>>2] = $298;
  $307 = HEAP32[$54>>2]|0;
  HEAP32[$53>>2] = $307;
  $308 = HEAP32[$53>>2]|0;
  HEAP32[$52>>2] = $308;
  $309 = HEAP32[$52>>2]|0;
  $310 = HEAP32[$309>>2]|0;
  $316 = $310;
 } else {
  HEAP32[$59>>2] = $298;
  $311 = HEAP32[$59>>2]|0;
  HEAP32[$58>>2] = $311;
  $312 = HEAP32[$58>>2]|0;
  HEAP32[$57>>2] = $312;
  $313 = HEAP32[$57>>2]|0;
  HEAP32[$56>>2] = $313;
  $314 = HEAP32[$56>>2]|0;
  HEAP32[$55>>2] = $314;
  $315 = HEAP32[$55>>2]|0;
  $316 = $315;
 }
 HEAP32[$51>>2] = $316;
 $317 = HEAP32[$51>>2]|0;
 $318 = (__ZNK3sap2ve4core18ViewportDefinition7GetNameEPKc($294,$317)|0);
 $319 = (__ZNK3sap2ve6String4Utf8Ev($318)|0);
 $320 = HEAP32[$152>>2]|0;
 $321 = ((($172)) + 12|0);
 HEAP32[$47>>2] = $321;
 $322 = HEAP32[$47>>2]|0;
 HEAP32[$46>>2] = $322;
 $323 = HEAP32[$46>>2]|0;
 HEAP32[$45>>2] = $323;
 $324 = HEAP32[$45>>2]|0;
 HEAP32[$44>>2] = $324;
 $325 = HEAP32[$44>>2]|0;
 HEAP32[$43>>2] = $325;
 $326 = HEAP32[$43>>2]|0;
 HEAP32[$42>>2] = $326;
 $327 = HEAP32[$42>>2]|0;
 $328 = ((($327)) + 11|0);
 $329 = HEAP8[$328>>0]|0;
 $330 = $329&255;
 $331 = $330 & 128;
 $332 = ($331|0)!=(0);
 if ($332) {
  HEAP32[$36>>2] = $324;
  $333 = HEAP32[$36>>2]|0;
  HEAP32[$35>>2] = $333;
  $334 = HEAP32[$35>>2]|0;
  HEAP32[$34>>2] = $334;
  $335 = HEAP32[$34>>2]|0;
  $336 = HEAP32[$335>>2]|0;
  $342 = $336;
 } else {
  HEAP32[$41>>2] = $324;
  $337 = HEAP32[$41>>2]|0;
  HEAP32[$40>>2] = $337;
  $338 = HEAP32[$40>>2]|0;
  HEAP32[$39>>2] = $338;
  $339 = HEAP32[$39>>2]|0;
  HEAP32[$38>>2] = $339;
  $340 = HEAP32[$38>>2]|0;
  HEAP32[$37>>2] = $340;
  $341 = HEAP32[$37>>2]|0;
  $342 = $341;
 }
 HEAP32[$33>>2] = $342;
 $343 = HEAP32[$33>>2]|0;
 $344 = (__ZNK3sap2ve4core18ViewportDefinition7GetInfoEPKc($320,$343)|0);
 $345 = (__ZNK3sap2ve6String4Utf8Ev($344)|0);
 $346 = HEAP32[$152>>2]|0;
 $347 = (__ZNK3sap2ve4core9ModelView12GetThumbnailEv($346)|0);
 $348 = HEAP32[$158>>2]|0;
 $349 = HEAP32[$152>>2]|0;
 $350 = (__ZNK3sap2ve4core9ModelView7GetTypeEv($349)|0);
 $351 = HEAP32[$157>>2]|0;
 $352 = (__ZNK3sap2ve4core15ViewportDisplay12GetFlyToTimeEv($351)|0);
 $353 = (+($352>>>0));
 $354 = HEAP32[$157>>2]|0;
 $355 = (+__ZNK3sap2ve4core15ViewportDisplay11GetPreDelayEv($354));
 $356 = HEAP32[$157>>2]|0;
 $357 = (+__ZNK3sap2ve4core15ViewportDisplay12GetPostDelayEv($356));
 $358 = HEAP32[$157>>2]|0;
 $359 = (__ZNK3sap2ve4core15ViewportDisplay17GetNavigationModeEv($358)|0);
 $360 = HEAP32[$157>>2]|0;
 __ZNK3sap2ve4core15ViewportDisplay12GetTopColourEv($161,$360);
 $361 = (__ZL8GetColorRKN3sap2ve4core6ColourE($161)|0);
 $362 = HEAP32[$157>>2]|0;
 __ZNK3sap2ve4core15ViewportDisplay15GetBottomColourEv($162,$362);
 $363 = (__ZL8GetColorRKN3sap2ve4core6ColourE($162)|0);
 $364 = HEAP32[$157>>2]|0;
 $365 = (__ZNK3sap2ve4core15ViewportDisplay15GetRenderMethodEv($364)|0);
 $366 = HEAP32[$152>>2]|0;
 $367 = (__ZNK3sap2ve4core9ModelView9DimensionEv($366)|0);
 $368 = (__ZNK3sap2ve4core16SpatialDimension3GetEv($367)|0);
 $369 = HEAP32[$152>>2]|0;
 __ZNK3sap2ve4core9ModelView8GetQueryEv($163,$369);
 $370 = (__ZNK3sap2ve6String4Utf8Ev($163)|0);
 HEAP32[$32>>2] = $159;
 $371 = HEAP32[$32>>2]|0;
 HEAP32[$31>>2] = $371;
 $372 = HEAP32[$31>>2]|0;
 HEAP32[$30>>2] = $372;
 $373 = HEAP32[$30>>2]|0;
 HEAP32[$29>>2] = $373;
 $374 = HEAP32[$29>>2]|0;
 HEAP32[$28>>2] = $374;
 $375 = HEAP32[$28>>2]|0;
 HEAP32[$27>>2] = $375;
 $376 = HEAP32[$27>>2]|0;
 $377 = ((($376)) + 11|0);
 $378 = HEAP8[$377>>0]|0;
 $379 = $378&255;
 $380 = $379 & 128;
 $381 = ($380|0)!=(0);
 if ($381) {
  HEAP32[$21>>2] = $373;
  $382 = HEAP32[$21>>2]|0;
  HEAP32[$20>>2] = $382;
  $383 = HEAP32[$20>>2]|0;
  HEAP32[$19>>2] = $383;
  $384 = HEAP32[$19>>2]|0;
  $385 = HEAP32[$384>>2]|0;
  $391 = $385;
 } else {
  HEAP32[$26>>2] = $373;
  $386 = HEAP32[$26>>2]|0;
  HEAP32[$25>>2] = $386;
  $387 = HEAP32[$25>>2]|0;
  HEAP32[$24>>2] = $387;
  $388 = HEAP32[$24>>2]|0;
  HEAP32[$23>>2] = $388;
  $389 = HEAP32[$23>>2]|0;
  HEAP32[$22>>2] = $389;
  $390 = HEAP32[$22>>2]|0;
  $391 = $390;
 }
 HEAP32[$18>>2] = $391;
 $392 = HEAP32[$18>>2]|0;
 HEAP32[$17>>2] = $160;
 $393 = HEAP32[$17>>2]|0;
 HEAP32[$16>>2] = $393;
 $394 = HEAP32[$16>>2]|0;
 HEAP32[$15>>2] = $394;
 $395 = HEAP32[$15>>2]|0;
 HEAP32[$14>>2] = $395;
 $396 = HEAP32[$14>>2]|0;
 HEAP32[$13>>2] = $396;
 $397 = HEAP32[$13>>2]|0;
 HEAP32[$12>>2] = $397;
 $398 = HEAP32[$12>>2]|0;
 $399 = ((($398)) + 11|0);
 $400 = HEAP8[$399>>0]|0;
 $401 = $400&255;
 $402 = $401 & 128;
 $403 = ($402|0)!=(0);
 if ($403) {
  HEAP32[$6>>2] = $395;
  $404 = HEAP32[$6>>2]|0;
  HEAP32[$5>>2] = $404;
  $405 = HEAP32[$5>>2]|0;
  HEAP32[$4>>2] = $405;
  $406 = HEAP32[$4>>2]|0;
  $407 = HEAP32[$406>>2]|0;
  $413 = $407;
 } else {
  HEAP32[$11>>2] = $395;
  $408 = HEAP32[$11>>2]|0;
  HEAP32[$10>>2] = $408;
  $409 = HEAP32[$10>>2]|0;
  HEAP32[$9>>2] = $409;
  $410 = HEAP32[$9>>2]|0;
  HEAP32[$8>>2] = $410;
  $411 = HEAP32[$8>>2]|0;
  HEAP32[$7>>2] = $411;
  $412 = HEAP32[$7>>2]|0;
  $413 = $412;
 }
 HEAP32[$3>>2] = $413;
 $414 = HEAP32[$3>>2]|0;
 _SceneBuilderAdapter_insertModelView(($291|0),($292|0),($293|0),($319|0),($345|0),($347|0),($348|0),($350|0),(+$353),(+$355),(+$357),($359|0),($361|0),($363|0),($365|0),($368|0),($370|0),($392|0),($414|0));
 __ZN3sap2ve6StringD2Ev($163);
 HEAP32[$164>>2] = 0;
 while(1) {
  $415 = HEAP32[$164>>2]|0;
  $416 = HEAP32[$152>>2]|0;
  $417 = (__ZNK3sap2ve4core9ModelView25GetAnimationPlaybackCountEv($416)|0);
  $418 = ($415>>>0)<($417>>>0);
  $419 = HEAP32[$152>>2]|0;
  if (!($418)) {
   break;
  }
  $420 = HEAP32[$164>>2]|0;
  $421 = (__ZNK3sap2ve4core9ModelView20GetAnimationPlaybackEj($419,$420)|0);
  HEAP32[$165>>2] = $421;
  $422 = HEAP32[$165>>2]|0;
  $423 = (__ZNK3sap2ve4core17AnimationPlayback17GetAnimationGroupEv($422)|0);
  HEAP32[$166>>2] = $423;
  $424 = HEAP32[$166>>2]|0;
  $425 = HEAP32[$152>>2]|0;
  $426 = HEAP32[$165>>2]|0;
  __ZN17DataModelImporter20InsertAnimationGroupEPKN3sap2ve4core14AnimationGroupEPKNS2_9ModelViewEPNS2_17AnimationPlaybackE($172,$424,$425,$426);
  $427 = HEAP32[$164>>2]|0;
  $428 = (($427) + 1)|0;
  HEAP32[$164>>2] = $428;
 }
 $429 = (__ZNK3sap2ve4core9ModelView17GetHighlightCountEv($419)|0);
 HEAP32[$167>>2] = $429;
 $430 = HEAP32[$167>>2]|0;
 $431 = ($430|0)!=(0);
 L36: do {
  if ($431) {
   HEAP32[$168>>2] = 0;
   while(1) {
    $432 = HEAP32[$168>>2]|0;
    $433 = HEAP32[$167>>2]|0;
    $434 = ($432>>>0)<($433>>>0);
    if (!($434)) {
     break L36;
    }
    $435 = HEAP32[$152>>2]|0;
    $436 = HEAP32[$168>>2]|0;
    $437 = (__ZNK3sap2ve4core9ModelView12GetHighlightEj($435,$436)|0);
    HEAP32[$169>>2] = $437;
    $438 = HEAP32[$169>>2]|0;
    $439 = (__ZNK3sap2ve4core9Highlight8GetStyleEv($438)|0);
    HEAP32[$170>>2] = $439;
    $440 = HEAP32[$170>>2]|0;
    $441 = ($440|0)!=(0|0);
    if ($441) {
     $442 = HEAP32[$169>>2]|0;
     $443 = (__ZNK3sap2ve4core9Highlight15GetSelectionSetEv($442)|0);
     __ZNK17DataModelImporter35GetNodesArrayFromSelectorCollectionERKN3sap2ve4core18SelectorCollectionE($171,$172,$443);
     $444 = ((($172)) + 24|0);
     $445 = HEAP32[$444>>2]|0;
     $446 = HEAP32[$152>>2]|0;
     HEAP32[$48>>2] = $171;
     $447 = HEAP32[$48>>2]|0;
     $448 = ((($447)) + 4|0);
     $449 = HEAP32[$448>>2]|0;
     $450 = HEAP32[$447>>2]|0;
     $451 = $449;
     $452 = $450;
     $453 = (($451) - ($452))|0;
     $454 = (($453|0) / 4)&-1;
     HEAP32[$50>>2] = $171;
     $455 = HEAP32[$50>>2]|0;
     $456 = HEAP32[$455>>2]|0;
     HEAP32[$49>>2] = $456;
     $457 = HEAP32[$49>>2]|0;
     $458 = HEAP32[$170>>2]|0;
     $459 = (__ZNK3sap2ve4core14HighlightStyle14GetColourCountEv($458)|0);
     $460 = ($459>>>0)>(0);
     if ($460) {
      $461 = HEAP32[$170>>2]|0;
      $462 = (__ZNK3sap2ve4core14HighlightStyle9GetColourEj($461,0)|0);
      $463 = (__ZL14GetColorARGB32RKN3sap2ve4core6ColourE($462)|0);
      $491 = $463;
     } else {
      $491 = 0;
     }
     $464 = HEAP32[$170>>2]|0;
     $465 = (__ZNK3sap2ve4core14HighlightStyle14GetColourCountEv($464)|0);
     $466 = ($465>>>0)>(1);
     if ($466) {
      $467 = HEAP32[$170>>2]|0;
      $468 = (__ZNK3sap2ve4core14HighlightStyle9GetColourEj($467,1)|0);
      $469 = (__ZL14GetColorARGB32RKN3sap2ve4core6ColourE($468)|0);
      $492 = $469;
     } else {
      $492 = 0;
     }
     $470 = HEAP32[$170>>2]|0;
     $471 = (__ZNK3sap2ve4core14HighlightStyle15GetOpacityCountEv($470)|0);
     $472 = ($471>>>0)>(0);
     if ($472) {
      $473 = HEAP32[$170>>2]|0;
      $474 = (+__ZNK3sap2ve4core14HighlightStyle10GetOpacityEj($473,0));
      $493 = $474;
     } else {
      $493 = 0.0;
     }
     $475 = HEAP32[$170>>2]|0;
     $476 = (__ZNK3sap2ve4core14HighlightStyle15GetOpacityCountEv($475)|0);
     $477 = ($476>>>0)>(1);
     if ($477) {
      $478 = HEAP32[$170>>2]|0;
      $479 = (+__ZNK3sap2ve4core14HighlightStyle10GetOpacityEj($478,1));
      $494 = $479;
     } else {
      $494 = 1.0;
     }
     $480 = HEAP32[$170>>2]|0;
     $481 = (__ZNK3sap2ve4core14HighlightStyle7GetTypeEv($480)|0);
     $482 = ($481|0)==(0);
     if ($482) {
      $495 = 0.0;
     } else {
      $483 = HEAP32[$170>>2]|0;
      $484 = (+__ZNK3sap2ve4core14HighlightStyle16GetCycleDurationEv($483));
      $495 = $484;
     }
     $485 = HEAP32[$170>>2]|0;
     $486 = (__ZNK3sap2ve4core14HighlightStyle7GetTypeEv($485)|0);
     $487 = ($486|0)==(2);
     if ($487) {
      $496 = 0;
     } else {
      $488 = HEAP32[$170>>2]|0;
      $489 = (__ZNK3sap2ve4core14HighlightStyle9GetCyclesEv($488)|0);
      $490 = $489&255;
      $496 = $490;
     }
     _SceneBuilderAdapter_insertModelViewHighlight(($445|0),($446|0),($454|0),($457|0),($491|0),($492|0),(+$493),(+$494),(+$495),($496|0));
     __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($171);
    }
    $497 = HEAP32[$168>>2]|0;
    $498 = (($497) + 1)|0;
    HEAP32[$168>>2] = $498;
   }
  }
 } while(0);
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($160);
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($159);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter20InsertAnimationGroupEPKN3sap2ve4core14AnimationGroupEPKNS2_9ModelViewEPNS2_17AnimationPlaybackE($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $10 = 0, $100 = 0, $1000 = 0, $1001 = 0, $1002 = 0, $1003 = 0, $1004 = 0, $1005 = 0.0, $1006 = 0, $1007 = 0, $1008 = 0, $1009 = 0, $101 = 0, $1010 = 0, $1011 = 0, $1012 = 0, $1013 = 0, $1014 = 0, $1015 = 0, $1016 = 0;
 var $1017 = 0, $1018 = 0, $1019 = 0, $102 = 0, $1020 = 0, $1021 = 0, $1022 = 0, $1023 = 0, $1024 = 0, $1025 = 0, $1026 = 0, $1027 = 0, $1028 = 0, $1029 = 0.0, $103 = 0, $1030 = 0, $1031 = 0, $1032 = 0, $1033 = 0, $1034 = 0;
 var $1035 = 0, $1036 = 0, $1037 = 0.0, $1038 = 0, $1039 = 0, $104 = 0, $1040 = 0, $1041 = 0, $1042 = 0, $1043 = 0, $1044 = 0, $1045 = 0.0, $1046 = 0, $1047 = 0, $1048 = 0, $1049 = 0, $105 = 0, $1050 = 0, $1051 = 0, $1052 = 0;
 var $1053 = 0.0, $1054 = 0, $1055 = 0, $1056 = 0, $1057 = 0, $1058 = 0, $1059 = 0, $106 = 0, $1060 = 0, $1061 = 0.0, $1062 = 0, $1063 = 0, $1064 = 0, $1065 = 0, $1066 = 0, $1067 = 0, $1068 = 0, $1069 = 0.0, $107 = 0, $1070 = 0;
 var $1071 = 0, $1072 = 0, $1073 = 0, $1074 = 0, $1075 = 0, $1076 = 0, $1077 = 0.0, $1078 = 0, $1079 = 0, $108 = 0, $1080 = 0, $1081 = 0, $1082 = 0, $1083 = 0, $1084 = 0, $1085 = 0, $1086 = 0, $1087 = 0.0, $1088 = 0, $1089 = 0;
 var $109 = 0, $1090 = 0, $1091 = 0, $1092 = 0, $1093 = 0, $1094 = 0, $1095 = 0, $1096 = 0, $1097 = 0, $1098 = 0.0, $1099 = 0, $11 = 0, $110 = 0, $1100 = 0, $1101 = 0, $1102 = 0, $1103 = 0, $1104 = 0, $1105 = 0, $1106 = 0;
 var $1107 = 0, $1108 = 0, $1109 = 0, $111 = 0, $1110 = 0, $1111 = 0, $1112 = 0, $1113 = 0, $1114 = 0, $1115 = 0, $1116 = 0, $1117 = 0, $1118 = 0, $1119 = 0, $112 = 0, $1120 = 0, $1121 = 0, $1122 = 0, $1123 = 0, $1124 = 0;
 var $1125 = 0, $1126 = 0, $1127 = 0, $1128 = 0, $1129 = 0, $113 = 0, $1130 = 0, $1131 = 0, $1132 = 0, $1133 = 0, $1134 = 0, $1135 = 0, $1136 = 0, $1137 = 0, $1138 = 0, $1139 = 0, $114 = 0, $1140 = 0, $1141 = 0, $1142 = 0;
 var $1143 = 0, $1144 = 0, $1145 = 0, $1146 = 0, $1147 = 0, $1148 = 0, $1149 = 0, $115 = 0, $1150 = 0, $1151 = 0, $1152 = 0, $1153 = 0, $1154 = 0, $1155 = 0, $1156 = 0, $1157 = 0, $1158 = 0, $1159 = 0, $116 = 0, $1160 = 0;
 var $1161 = 0, $1162 = 0, $1163 = 0.0, $1164 = 0, $1165 = 0, $1166 = 0, $1167 = 0, $1168 = 0, $1169 = 0, $117 = 0, $1170 = 0, $1171 = 0.0, $1172 = 0, $1173 = 0, $1174 = 0, $1175 = 0, $1176 = 0, $1177 = 0, $1178 = 0, $1179 = 0.0;
 var $118 = 0, $1180 = 0, $1181 = 0, $1182 = 0, $1183 = 0, $1184 = 0, $1185 = 0, $1186 = 0, $1187 = 0.0, $1188 = 0, $1189 = 0, $119 = 0, $1190 = 0, $1191 = 0, $1192 = 0, $1193 = 0, $1194 = 0, $1195 = 0.0, $1196 = 0, $1197 = 0;
 var $1198 = 0, $1199 = 0, $12 = 0, $120 = 0, $1200 = 0, $1201 = 0, $1202 = 0, $1203 = 0.0, $1204 = 0, $1205 = 0, $1206 = 0, $1207 = 0, $1208 = 0, $1209 = 0, $121 = 0, $1210 = 0, $1211 = 0.0, $1212 = 0, $1213 = 0, $1214 = 0;
 var $1215 = 0, $1216 = 0, $1217 = 0, $1218 = 0, $1219 = 0, $122 = 0, $1220 = 0, $1221 = 0, $1222 = 0, $1223 = 0, $1224 = 0, $1225 = 0, $1226 = 0, $1227 = 0, $1228 = 0, $1229 = 0, $123 = 0, $1230 = 0, $1231 = 0, $1232 = 0;
 var $1233 = 0, $1234 = 0, $1235 = 0, $1236 = 0, $1237 = 0, $1238 = 0, $1239 = 0, $124 = 0, $1240 = 0, $1241 = 0, $1242 = 0, $1243 = 0, $1244 = 0, $1245 = 0, $1246 = 0, $1247 = 0, $1248 = 0, $1249 = 0, $125 = 0, $1250 = 0;
 var $1251 = 0, $1252 = 0, $1253 = 0, $1254 = 0, $1255 = 0, $1256 = 0, $1257 = 0, $1258 = 0, $1259 = 0, $126 = 0, $1260 = 0, $1261 = 0, $1262 = 0, $1263 = 0, $1264 = 0, $1265 = 0, $1266 = 0, $1267 = 0, $1268 = 0, $1269 = 0;
 var $127 = 0, $1270 = 0, $1271 = 0, $1272 = 0, $1273 = 0, $1274 = 0, $1275 = 0, $1276 = 0, $1277 = 0, $1278 = 0, $1279 = 0, $128 = 0, $1280 = 0, $1281 = 0, $1282 = 0, $1283 = 0, $1284 = 0, $1285 = 0, $1286 = 0, $1287 = 0.0;
 var $1288 = 0, $1289 = 0, $129 = 0, $1290 = 0, $1291 = 0, $1292 = 0, $1293 = 0, $1294 = 0, $1295 = 0.0, $1296 = 0, $1297 = 0, $1298 = 0, $1299 = 0, $13 = 0, $130 = 0, $1300 = 0, $1301 = 0, $1302 = 0, $1303 = 0.0, $1304 = 0;
 var $1305 = 0, $1306 = 0, $1307 = 0, $1308 = 0, $1309 = 0, $131 = 0, $1310 = 0, $1311 = 0.0, $1312 = 0, $1313 = 0, $1314 = 0, $1315 = 0, $1316 = 0, $1317 = 0, $1318 = 0, $1319 = 0.0, $132 = 0, $1320 = 0, $1321 = 0, $1322 = 0;
 var $1323 = 0, $1324 = 0, $1325 = 0, $1326 = 0, $1327 = 0.0, $1328 = 0, $1329 = 0, $133 = 0, $1330 = 0, $1331 = 0, $1332 = 0, $1333 = 0, $1334 = 0, $1335 = 0.0, $1336 = 0, $1337 = 0, $1338 = 0, $1339 = 0, $134 = 0, $1340 = 0;
 var $1341 = 0, $1342 = 0, $1343 = 0, $1344 = 0, $1345 = 0, $1346 = 0, $1347 = 0, $1348 = 0, $1349 = 0, $135 = 0, $1350 = 0, $1351 = 0, $1352 = 0, $1353 = 0, $1354 = 0, $1355 = 0, $1356 = 0, $1357 = 0, $1358 = 0, $1359 = 0;
 var $136 = 0, $1360 = 0, $1361 = 0, $1362 = 0, $1363 = 0, $1364 = 0, $1365 = 0, $1366 = 0, $1367 = 0, $1368 = 0, $1369 = 0, $137 = 0, $1370 = 0, $1371 = 0, $1372 = 0, $1373 = 0, $1374 = 0, $1375 = 0, $1376 = 0, $1377 = 0;
 var $1378 = 0, $1379 = 0, $138 = 0, $1380 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0;
 var $153 = 0, $154 = 0, $155 = 0, $156 = 0, $157 = 0, $158 = 0, $159 = 0, $16 = 0, $160 = 0, $161 = 0, $162 = 0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0;
 var $171 = 0, $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0;
 var $19 = 0, $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0;
 var $208 = 0, $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0;
 var $226 = 0, $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0;
 var $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0;
 var $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0;
 var $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0;
 var $299 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0.0, $311 = 0, $312 = 0.0, $313 = 0, $314 = 0.0, $315 = 0, $316 = 0;
 var $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0.0, $328 = 0, $329 = 0, $33 = 0, $330 = 0, $331 = 0.0, $332 = 0, $333 = 0, $334 = 0;
 var $335 = 0.0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0.0, $342 = 0, $343 = 0, $344 = 0, $345 = 0.0, $346 = 0.0, $347 = 0.0, $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0;
 var $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0, $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0;
 var $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0, $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0;
 var $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0.0, $397 = 0, $398 = 0, $399 = 0.0, $4 = 0, $40 = 0, $400 = 0, $401 = 0, $402 = 0.0, $403 = 0, $404 = 0, $405 = 0, $406 = 0;
 var $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0, $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0;
 var $425 = 0, $426 = 0, $427 = 0.0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0.0, $436 = 0, $437 = 0, $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0;
 var $443 = 0.0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0.0, $452 = 0, $453 = 0, $454 = 0, $455 = 0, $456 = 0, $457 = 0, $458 = 0, $459 = 0.0, $46 = 0, $460 = 0;
 var $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0.0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0, $474 = 0, $475 = 0.0, $476 = 0, $477 = 0, $478 = 0, $479 = 0;
 var $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0.0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0, $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0.0, $497 = 0;
 var $498 = 0, $499 = 0, $5 = 0, $50 = 0, $500 = 0, $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $509 = 0, $51 = 0, $510 = 0, $511 = 0, $512 = 0, $513 = 0, $514 = 0;
 var $515 = 0, $516 = 0, $517 = 0, $518 = 0, $519 = 0, $52 = 0, $520 = 0, $521 = 0, $522 = 0, $523 = 0, $524 = 0, $525 = 0, $526 = 0, $527 = 0, $528 = 0, $529 = 0, $53 = 0, $530 = 0, $531 = 0, $532 = 0;
 var $533 = 0, $534 = 0, $535 = 0, $536 = 0, $537 = 0, $538 = 0, $539 = 0, $54 = 0, $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $545 = 0, $546 = 0, $547 = 0, $548 = 0, $549 = 0, $55 = 0, $550 = 0;
 var $551 = 0, $552 = 0, $553 = 0, $554 = 0, $555 = 0, $556 = 0, $557 = 0, $558 = 0, $559 = 0, $56 = 0, $560 = 0, $561 = 0, $562 = 0, $563 = 0, $564 = 0, $565 = 0, $566 = 0, $567 = 0, $568 = 0, $569 = 0.0;
 var $57 = 0, $570 = 0, $571 = 0, $572 = 0, $573 = 0, $574 = 0, $575 = 0, $576 = 0, $577 = 0.0, $578 = 0, $579 = 0, $58 = 0, $580 = 0, $581 = 0, $582 = 0, $583 = 0, $584 = 0, $585 = 0.0, $586 = 0, $587 = 0;
 var $588 = 0, $589 = 0, $59 = 0, $590 = 0, $591 = 0, $592 = 0, $593 = 0.0, $594 = 0, $595 = 0, $596 = 0, $597 = 0, $598 = 0, $599 = 0, $6 = 0, $60 = 0, $600 = 0, $601 = 0.0, $602 = 0, $603 = 0, $604 = 0;
 var $605 = 0, $606 = 0, $607 = 0, $608 = 0, $609 = 0.0, $61 = 0, $610 = 0, $611 = 0, $612 = 0, $613 = 0, $614 = 0, $615 = 0, $616 = 0, $617 = 0, $618 = 0.0, $619 = 0, $62 = 0, $620 = 0, $621 = 0, $622 = 0;
 var $623 = 0, $624 = 0, $625 = 0, $626 = 0, $627 = 0, $628 = 0, $629 = 0.0, $63 = 0, $630 = 0, $631 = 0, $632 = 0, $633 = 0, $634 = 0, $635 = 0, $636 = 0, $637 = 0, $638 = 0, $639 = 0, $64 = 0, $640 = 0;
 var $641 = 0.0, $642 = 0, $643 = 0, $644 = 0, $645 = 0, $646 = 0, $647 = 0, $648 = 0, $649 = 0, $65 = 0, $650 = 0, $651 = 0.0, $652 = 0, $653 = 0, $654 = 0, $655 = 0, $656 = 0, $657 = 0, $658 = 0, $659 = 0;
 var $66 = 0, $660 = 0, $661 = 0, $662 = 0, $663 = 0, $664 = 0, $665 = 0, $666 = 0, $667 = 0, $668 = 0, $669 = 0, $67 = 0, $670 = 0, $671 = 0, $672 = 0, $673 = 0, $674 = 0, $675 = 0.0, $676 = 0, $677 = 0;
 var $678 = 0, $679 = 0, $68 = 0, $680 = 0, $681 = 0, $682 = 0, $683 = 0.0, $684 = 0, $685 = 0, $686 = 0, $687 = 0, $688 = 0, $689 = 0, $69 = 0, $690 = 0, $691 = 0.0, $692 = 0, $693 = 0, $694 = 0, $695 = 0;
 var $696 = 0, $697 = 0, $698 = 0, $699 = 0.0, $7 = 0, $70 = 0, $700 = 0, $701 = 0, $702 = 0, $703 = 0, $704 = 0, $705 = 0, $706 = 0, $707 = 0.0, $708 = 0, $709 = 0, $71 = 0, $710 = 0, $711 = 0, $712 = 0;
 var $713 = 0, $714 = 0, $715 = 0.0, $716 = 0, $717 = 0, $718 = 0, $719 = 0, $72 = 0, $720 = 0, $721 = 0, $722 = 0, $723 = 0.0, $724 = 0, $725 = 0, $726 = 0, $727 = 0, $728 = 0, $729 = 0, $73 = 0, $730 = 0;
 var $731 = 0, $732 = 0, $733 = 0.0, $734 = 0, $735 = 0, $736 = 0, $737 = 0, $738 = 0, $739 = 0, $74 = 0, $740 = 0, $741 = 0, $742 = 0, $743 = 0, $744 = 0.0, $745 = 0, $746 = 0, $747 = 0, $748 = 0, $749 = 0;
 var $75 = 0, $750 = 0, $751 = 0, $752 = 0, $753 = 0, $754 = 0, $755 = 0.0, $756 = 0, $757 = 0, $758 = 0, $759 = 0, $76 = 0, $760 = 0, $761 = 0, $762 = 0, $763 = 0, $764 = 0, $765 = 0, $766 = 0, $767 = 0;
 var $768 = 0, $769 = 0, $77 = 0, $770 = 0, $771 = 0, $772 = 0, $773 = 0, $774 = 0, $775 = 0, $776 = 0, $777 = 0, $778 = 0, $779 = 0.0, $78 = 0, $780 = 0, $781 = 0, $782 = 0, $783 = 0, $784 = 0, $785 = 0;
 var $786 = 0, $787 = 0.0, $788 = 0, $789 = 0, $79 = 0, $790 = 0, $791 = 0, $792 = 0, $793 = 0, $794 = 0, $795 = 0.0, $796 = 0, $797 = 0, $798 = 0, $799 = 0, $8 = 0, $80 = 0, $800 = 0, $801 = 0, $802 = 0;
 var $803 = 0.0, $804 = 0, $805 = 0, $806 = 0, $807 = 0, $808 = 0, $809 = 0, $81 = 0, $810 = 0, $811 = 0.0, $812 = 0, $813 = 0, $814 = 0, $815 = 0, $816 = 0, $817 = 0, $818 = 0, $819 = 0.0, $82 = 0, $820 = 0;
 var $821 = 0, $822 = 0, $823 = 0, $824 = 0, $825 = 0, $826 = 0, $827 = 0.0, $828 = 0, $829 = 0, $83 = 0, $830 = 0, $831 = 0, $832 = 0, $833 = 0, $834 = 0, $835 = 0, $836 = 0, $837 = 0.0, $838 = 0, $839 = 0;
 var $84 = 0, $840 = 0, $841 = 0, $842 = 0, $843 = 0, $844 = 0, $845 = 0, $846 = 0, $847 = 0, $848 = 0.0, $849 = 0, $85 = 0, $850 = 0, $851 = 0, $852 = 0, $853 = 0, $854 = 0, $855 = 0, $856 = 0, $857 = 0;
 var $858 = 0, $859 = 0, $86 = 0, $860 = 0.0, $861 = 0, $862 = 0, $863 = 0, $864 = 0, $865 = 0, $866 = 0, $867 = 0, $868 = 0, $869 = 0, $87 = 0, $870 = 0, $871 = 0, $872 = 0, $873 = 0, $874 = 0, $875 = 0;
 var $876 = 0, $877 = 0, $878 = 0, $879 = 0, $88 = 0, $880 = 0, $881 = 0, $882 = 0, $883 = 0, $884 = 0, $885 = 0, $886 = 0, $887 = 0, $888 = 0, $889 = 0, $89 = 0, $890 = 0, $891 = 0, $892 = 0, $893 = 0;
 var $894 = 0, $895 = 0, $896 = 0, $897 = 0, $898 = 0, $899 = 0, $9 = 0, $90 = 0, $900 = 0, $901 = 0, $902 = 0, $903 = 0, $904 = 0, $905 = 0, $906 = 0, $907 = 0, $908 = 0, $909 = 0, $91 = 0, $910 = 0;
 var $911 = 0, $912 = 0, $913 = 0, $914 = 0, $915 = 0, $916 = 0, $917 = 0, $918 = 0, $919 = 0, $92 = 0, $920 = 0, $921 = 0, $922 = 0, $923 = 0, $924 = 0, $925 = 0, $926 = 0, $927 = 0, $928 = 0, $929 = 0;
 var $93 = 0, $930 = 0, $931 = 0, $932 = 0, $933 = 0, $934 = 0, $935 = 0, $936 = 0, $937 = 0, $938 = 0.0, $939 = 0, $94 = 0, $940 = 0, $941 = 0, $942 = 0, $943 = 0, $944 = 0, $945 = 0, $946 = 0.0, $947 = 0;
 var $948 = 0, $949 = 0, $95 = 0, $950 = 0, $951 = 0, $952 = 0, $953 = 0, $954 = 0.0, $955 = 0, $956 = 0, $957 = 0, $958 = 0, $959 = 0, $96 = 0, $960 = 0, $961 = 0, $962 = 0.0, $963 = 0, $964 = 0, $965 = 0;
 var $966 = 0, $967 = 0, $968 = 0, $969 = 0, $97 = 0, $970 = 0.0, $971 = 0, $972 = 0, $973 = 0, $974 = 0, $975 = 0, $976 = 0, $977 = 0, $978 = 0.0, $979 = 0, $98 = 0, $980 = 0, $981 = 0, $982 = 0, $983 = 0;
 var $984 = 0, $985 = 0, $986 = 0.0, $987 = 0, $988 = 0, $989 = 0, $99 = 0, $990 = 0, $991 = 0, $992 = 0, $993 = 0, $994 = 0, $995 = 0.0, $996 = 0, $997 = 0, $998 = 0, $999 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 1488|0;
 $4 = sp + 1480|0;
 $5 = sp + 1476|0;
 $6 = sp + 1472|0;
 $7 = sp + 1468|0;
 $8 = sp + 1464|0;
 $9 = sp + 1460|0;
 $10 = sp + 1456|0;
 $11 = sp + 1452|0;
 $12 = sp + 1448|0;
 $13 = sp + 1444|0;
 $14 = sp + 1440|0;
 $15 = sp + 1436|0;
 $16 = sp + 1432|0;
 $17 = sp + 1428|0;
 $18 = sp + 1424|0;
 $19 = sp + 1420|0;
 $20 = sp + 1416|0;
 $21 = sp + 1412|0;
 $22 = sp + 1408|0;
 $23 = sp + 1404|0;
 $24 = sp + 1400|0;
 $25 = sp + 1396|0;
 $26 = sp + 1392|0;
 $27 = sp + 1388|0;
 $28 = sp + 1384|0;
 $29 = sp + 1380|0;
 $30 = sp + 1376|0;
 $31 = sp + 1372|0;
 $32 = sp + 1368|0;
 $33 = sp + 1364|0;
 $34 = sp + 1360|0;
 $35 = sp + 1356|0;
 $36 = sp + 1352|0;
 $37 = sp + 1348|0;
 $38 = sp + 1344|0;
 $39 = sp + 1340|0;
 $40 = sp + 1336|0;
 $41 = sp + 1332|0;
 $42 = sp + 1328|0;
 $43 = sp + 1324|0;
 $44 = sp + 1320|0;
 $45 = sp + 1316|0;
 $46 = sp + 1312|0;
 $47 = sp + 1308|0;
 $48 = sp + 1304|0;
 $49 = sp + 1300|0;
 $50 = sp + 1296|0;
 $51 = sp + 1292|0;
 $52 = sp + 1288|0;
 $53 = sp + 1284|0;
 $54 = sp + 1280|0;
 $55 = sp + 1276|0;
 $56 = sp + 1272|0;
 $57 = sp + 1268|0;
 $58 = sp + 1264|0;
 $59 = sp + 1260|0;
 $60 = sp + 1256|0;
 $61 = sp + 1252|0;
 $62 = sp + 1248|0;
 $63 = sp + 1244|0;
 $64 = sp + 1240|0;
 $65 = sp + 1236|0;
 $66 = sp + 1232|0;
 $67 = sp + 1228|0;
 $68 = sp + 1224|0;
 $69 = sp + 1220|0;
 $70 = sp + 1216|0;
 $71 = sp + 1212|0;
 $72 = sp + 1208|0;
 $73 = sp + 1204|0;
 $74 = sp + 1200|0;
 $75 = sp + 1196|0;
 $76 = sp + 1192|0;
 $77 = sp + 1188|0;
 $78 = sp + 1184|0;
 $79 = sp + 1180|0;
 $80 = sp + 1176|0;
 $81 = sp + 1172|0;
 $82 = sp + 1168|0;
 $83 = sp + 1164|0;
 $84 = sp + 1160|0;
 $85 = sp + 1156|0;
 $86 = sp + 1152|0;
 $87 = sp + 1148|0;
 $88 = sp + 1144|0;
 $89 = sp + 1140|0;
 $90 = sp + 1136|0;
 $91 = sp + 1132|0;
 $92 = sp + 1128|0;
 $93 = sp + 1124|0;
 $94 = sp + 1120|0;
 $95 = sp + 1116|0;
 $96 = sp + 1112|0;
 $97 = sp + 1108|0;
 $98 = sp + 1104|0;
 $99 = sp + 1100|0;
 $100 = sp + 1096|0;
 $101 = sp + 1092|0;
 $102 = sp + 1088|0;
 $103 = sp + 1084|0;
 $104 = sp + 1080|0;
 $105 = sp + 1076|0;
 $106 = sp + 1072|0;
 $107 = sp + 1068|0;
 $108 = sp + 1064|0;
 $109 = sp + 1060|0;
 $110 = sp + 1056|0;
 $111 = sp + 1052|0;
 $112 = sp + 1048|0;
 $113 = sp + 1044|0;
 $114 = sp + 1040|0;
 $115 = sp + 1036|0;
 $116 = sp + 1032|0;
 $117 = sp + 1028|0;
 $118 = sp + 1024|0;
 $119 = sp + 1020|0;
 $120 = sp + 1016|0;
 $121 = sp + 1012|0;
 $122 = sp + 1008|0;
 $123 = sp + 1004|0;
 $124 = sp + 1000|0;
 $125 = sp + 996|0;
 $126 = sp + 992|0;
 $127 = sp + 988|0;
 $128 = sp + 984|0;
 $129 = sp + 980|0;
 $130 = sp + 976|0;
 $131 = sp + 972|0;
 $132 = sp + 968|0;
 $133 = sp + 964|0;
 $134 = sp + 960|0;
 $135 = sp + 956|0;
 $136 = sp + 952|0;
 $137 = sp + 948|0;
 $138 = sp + 944|0;
 $139 = sp + 940|0;
 $140 = sp + 936|0;
 $141 = sp + 932|0;
 $142 = sp + 928|0;
 $143 = sp + 924|0;
 $144 = sp + 920|0;
 $145 = sp + 916|0;
 $146 = sp + 912|0;
 $147 = sp + 908|0;
 $148 = sp + 904|0;
 $149 = sp + 900|0;
 $150 = sp + 896|0;
 $151 = sp + 892|0;
 $152 = sp + 888|0;
 $153 = sp + 884|0;
 $154 = sp + 880|0;
 $155 = sp + 876|0;
 $156 = sp + 872|0;
 $157 = sp + 868|0;
 $158 = sp + 864|0;
 $159 = sp + 860|0;
 $160 = sp + 856|0;
 $161 = sp + 852|0;
 $162 = sp + 848|0;
 $163 = sp + 844|0;
 $164 = sp + 840|0;
 $165 = sp + 836|0;
 $166 = sp + 832|0;
 $167 = sp + 828|0;
 $168 = sp + 824|0;
 $169 = sp + 820|0;
 $170 = sp + 816|0;
 $171 = sp + 812|0;
 $172 = sp + 808|0;
 $173 = sp + 804|0;
 $174 = sp + 800|0;
 $175 = sp + 796|0;
 $176 = sp + 792|0;
 $177 = sp + 788|0;
 $178 = sp + 784|0;
 $179 = sp + 780|0;
 $180 = sp + 776|0;
 $181 = sp + 772|0;
 $182 = sp + 768|0;
 $183 = sp + 764|0;
 $184 = sp + 760|0;
 $185 = sp + 756|0;
 $186 = sp + 752|0;
 $187 = sp + 748|0;
 $188 = sp + 744|0;
 $189 = sp + 740|0;
 $190 = sp + 736|0;
 $191 = sp + 732|0;
 $192 = sp + 728|0;
 $193 = sp + 724|0;
 $194 = sp + 720|0;
 $195 = sp + 716|0;
 $196 = sp + 712|0;
 $197 = sp + 708|0;
 $198 = sp + 704|0;
 $199 = sp + 700|0;
 $200 = sp + 696|0;
 $201 = sp + 692|0;
 $202 = sp + 688|0;
 $203 = sp + 684|0;
 $204 = sp + 680|0;
 $205 = sp + 676|0;
 $206 = sp + 672|0;
 $207 = sp + 668|0;
 $208 = sp + 664|0;
 $209 = sp + 660|0;
 $210 = sp + 656|0;
 $211 = sp + 652|0;
 $212 = sp + 648|0;
 $213 = sp + 644|0;
 $214 = sp + 640|0;
 $215 = sp + 636|0;
 $216 = sp + 632|0;
 $217 = sp + 628|0;
 $218 = sp + 624|0;
 $219 = sp + 620|0;
 $220 = sp + 616|0;
 $221 = sp + 612|0;
 $222 = sp + 608|0;
 $223 = sp + 604|0;
 $224 = sp + 1484|0;
 $225 = sp + 600|0;
 $226 = sp + 596|0;
 $227 = sp + 592|0;
 $228 = sp + 588|0;
 $229 = sp + 584|0;
 $230 = sp + 580|0;
 $231 = sp + 576|0;
 $232 = sp + 572|0;
 $233 = sp + 568|0;
 $234 = sp + 564|0;
 $235 = sp + 552|0;
 $236 = sp + 540|0;
 $237 = sp + 528|0;
 $238 = sp + 516|0;
 $239 = sp + 504|0;
 $240 = sp + 492|0;
 $241 = sp + 480|0;
 $242 = sp + 476|0;
 $243 = sp + 472|0;
 $244 = sp + 468|0;
 $245 = sp + 464|0;
 $246 = sp + 460|0;
 $247 = sp + 448|0;
 $248 = sp + 436|0;
 $249 = sp + 424|0;
 $250 = sp + 412|0;
 $251 = sp + 400|0;
 $252 = sp + 388|0;
 $253 = sp + 376|0;
 $254 = sp + 372|0;
 $255 = sp + 368|0;
 $256 = sp + 364|0;
 $257 = sp + 360|0;
 $258 = sp + 356|0;
 $259 = sp + 352|0;
 $260 = sp + 348|0;
 $261 = sp + 344|0;
 $262 = sp + 340|0;
 $263 = sp + 336|0;
 $264 = sp + 332|0;
 $265 = sp + 328|0;
 $266 = sp + 316|0;
 $267 = sp + 304|0;
 $268 = sp + 292|0;
 $269 = sp + 280|0;
 $270 = sp + 268|0;
 $271 = sp + 256|0;
 $272 = sp + 244|0;
 $273 = sp + 240|0;
 $274 = sp + 236|0;
 $275 = sp + 232|0;
 $276 = sp + 228|0;
 $277 = sp + 224|0;
 $278 = sp + 220|0;
 $279 = sp + 216|0;
 $280 = sp + 212|0;
 $281 = sp + 208|0;
 $282 = sp + 196|0;
 $283 = sp + 184|0;
 $284 = sp + 172|0;
 $285 = sp + 160|0;
 $286 = sp + 148|0;
 $287 = sp + 136|0;
 $288 = sp + 124|0;
 $289 = sp + 120|0;
 $290 = sp + 116|0;
 $291 = sp + 112|0;
 $292 = sp + 108|0;
 $293 = sp + 104|0;
 $294 = sp + 100|0;
 $295 = sp + 96|0;
 $296 = sp + 92|0;
 $297 = sp + 80|0;
 $298 = sp + 68|0;
 $299 = sp + 56|0;
 $300 = sp + 44|0;
 $301 = sp + 32|0;
 $302 = sp + 20|0;
 $303 = sp + 8|0;
 $304 = sp + 4|0;
 $305 = sp;
 HEAP32[$216>>2] = $0;
 HEAP32[$217>>2] = $1;
 HEAP32[$218>>2] = $2;
 HEAP32[$219>>2] = $3;
 $306 = HEAP32[$216>>2]|0;
 HEAPF32[$220>>2] = 0.0;
 HEAPF32[$221>>2] = 0.0;
 HEAPF32[$222>>2] = 0.0;
 HEAP32[$223>>2] = 0;
 HEAP8[$224>>0] = 0;
 $307 = HEAP32[$219>>2]|0;
 $308 = ($307|0)!=(0|0);
 if ($308) {
  $309 = HEAP32[$219>>2]|0;
  $310 = (+__ZNK3sap2ve4core17AnimationPlayback8GetSpeedEv($309));
  HEAPF32[$220>>2] = $310;
  $311 = HEAP32[$219>>2]|0;
  $312 = (+__ZNK3sap2ve4core17AnimationPlayback11GetPreDelayEv($311));
  HEAPF32[$221>>2] = $312;
  $313 = HEAP32[$219>>2]|0;
  $314 = (+__ZNK3sap2ve4core17AnimationPlayback12GetPostDelayEv($313));
  HEAPF32[$222>>2] = $314;
  $315 = HEAP32[$219>>2]|0;
  $316 = (__ZNK3sap2ve4core17AnimationPlayback9GetRepeatEv($315)|0);
  $317 = $316&65535;
  HEAP32[$223>>2] = $317;
  $318 = HEAP32[$219>>2]|0;
  $319 = (__ZNK3sap2ve4core17AnimationPlayback10IsReversedEv($318)|0);
  $320 = $319&1;
  HEAP8[$224>>0] = $320;
 }
 $321 = ((($306)) + 24|0);
 $322 = HEAP32[$321>>2]|0;
 $323 = HEAP32[$218>>2]|0;
 $324 = HEAP32[$217>>2]|0;
 $325 = HEAP32[$217>>2]|0;
 $326 = (__ZNK3sap2ve4core14AnimationGroup21GetAnimationStartTimeEv($325)|0);
 $327 = +HEAPF32[$326>>2];
 $328 = (~~(($327))>>>0);
 $329 = HEAP32[$217>>2]|0;
 $330 = (__ZNK3sap2ve4core14AnimationGroup19GetAnimationEndTimeEv($329)|0);
 $331 = +HEAPF32[$330>>2];
 $332 = (~~(($331))>>>0);
 $333 = HEAP32[$217>>2]|0;
 $334 = (__ZNK3sap2ve4core14AnimationGroup21GetAnimationFrameRateEv($333)|0);
 $335 = +HEAPF32[$334>>2];
 $336 = HEAP32[$217>>2]|0;
 $337 = (__ZNK3sap2ve4core14AnimationGroup8IsCyclicEv($336)|0);
 $338 = HEAP32[$217>>2]|0;
 $339 = (__ZNK3sap2ve4core14AnimationGroup8IsActiveEv($338)|0);
 $340 = HEAP32[$217>>2]|0;
 $341 = (+__ZNK3sap2ve4core14AnimationGroup18GetCurrentPositionEv($340));
 $342 = HEAP32[$217>>2]|0;
 $343 = (__ZNK3sap2ve4core14AnimationGroup7GetNameEv($342)|0);
 $344 = (__ZNK3sap2ve6String4Utf8Ev($343)|0);
 $345 = +HEAPF32[$220>>2];
 $346 = +HEAPF32[$221>>2];
 $347 = +HEAPF32[$222>>2];
 $348 = HEAP32[$223>>2]|0;
 $349 = HEAP8[$224>>0]|0;
 $350 = $349&1;
 _SceneBuilderAdapter_insertAnimationGroup(($322|0),($323|0),($324|0),($328|0),($332|0),(+$335),($337|0),($339|0),(+$341),($344|0),(+$345),(+$346),(+$347),($348|0),($350|0));
 HEAP32[$225>>2] = 0;
 while(1) {
  $351 = HEAP32[$225>>2]|0;
  $352 = HEAP32[$217>>2]|0;
  $353 = (__ZNK3sap2ve4core14AnimationGroup17GetAnimationCountEv($352)|0);
  $354 = ($351>>>0)<($353>>>0);
  if (!($354)) {
   break;
  }
  $355 = HEAP32[$217>>2]|0;
  $356 = HEAP32[$225>>2]|0;
  $357 = (__ZNK3sap2ve4core14AnimationGroup12GetAnimationEj($355,$356)|0);
  HEAP32[$226>>2] = $357;
  $358 = ((($306)) + 24|0);
  $359 = HEAP32[$358>>2]|0;
  $360 = HEAP32[$217>>2]|0;
  $361 = HEAP32[$226>>2]|0;
  $362 = HEAP32[$226>>2]|0;
  $363 = HEAP32[$362>>2]|0;
  $364 = ((($363)) + 16|0);
  $365 = HEAP32[$364>>2]|0;
  $366 = (FUNCTION_TABLE_ii[$365 & 2047]($362)|0);
  _SceneBuilderAdapter_insertAnimation(($359|0),($360|0),($361|0),($366|0));
  $367 = HEAP32[$226>>2]|0;
  $368 = HEAP32[$367>>2]|0;
  $369 = ((($368)) + 16|0);
  $370 = HEAP32[$369>>2]|0;
  $371 = (FUNCTION_TABLE_ii[$370 & 2047]($367)|0);
  switch ($371|0) {
  case 0:  {
   $372 = HEAP32[$226>>2]|0;
   HEAP32[$227>>2] = $372;
   HEAP32[$228>>2] = 0;
   while(1) {
    $373 = HEAP32[$228>>2]|0;
    $374 = HEAP32[$227>>2]|0;
    $375 = (__ZNK3sap2ve4core17InstanceAnimation14GetTargetCountEv($374)|0);
    $376 = ($373>>>0)<($375>>>0);
    $377 = HEAP32[$227>>2]|0;
    if (!($376)) {
     break;
    }
    $378 = HEAP32[$228>>2]|0;
    $379 = (__ZNK3sap2ve4core17InstanceAnimation13GetPivotPointEj($377,$378)|0);
    HEAP32[$229>>2] = $379;
    $380 = HEAP32[$227>>2]|0;
    $381 = HEAP32[$228>>2]|0;
    $382 = (__ZN3sap2ve4core17InstanceAnimation9GetTargetEj($380,$381)|0);
    $383 = (__ZNK3sap2ve4core23InstanceAnimationTarget11GetSelectorEv($382)|0);
    HEAP32[$230>>2] = $383;
    $384 = ((($306)) + 4|0);
    $385 = (__ZNK3sap2ve4core4IPtrINS1_6ISceneEEptEv($384)|0);
    $386 = HEAP32[$385>>2]|0;
    $387 = ((($386)) + 44|0);
    $388 = HEAP32[$387>>2]|0;
    $389 = HEAP32[$230>>2]|0;
    $390 = (FUNCTION_TABLE_iii[$388 & 1023]($385,$389)|0);
    HEAP32[$231>>2] = $390;
    $391 = ((($306)) + 24|0);
    $392 = HEAP32[$391>>2]|0;
    $393 = HEAP32[$226>>2]|0;
    $394 = HEAP32[$231>>2]|0;
    $395 = HEAP32[$229>>2]|0;
    $396 = +HEAPF32[$395>>2];
    $397 = HEAP32[$229>>2]|0;
    $398 = ((($397)) + 4|0);
    $399 = +HEAPF32[$398>>2];
    $400 = HEAP32[$229>>2]|0;
    $401 = ((($400)) + 8|0);
    $402 = +HEAPF32[$401>>2];
    _SceneBuilderAdapter_insertAnimationTarget(($392|0),($393|0),($394|0),(+$396),(+$399),(+$402));
    $403 = HEAP32[$228>>2]|0;
    $404 = (($403) + 1)|0;
    HEAP32[$228>>2] = $404;
   }
   $405 = (__ZNK3sap2ve4core17InstanceAnimation20GetPositionAnimationEv($377)|0);
   HEAP32[$232>>2] = $405;
   $406 = HEAP32[$232>>2]|0;
   $407 = ($406|0)!=(0|0);
   if ($407) {
    $408 = HEAP32[$232>>2]|0;
    $409 = (__ZNK3sap2ve4core14AnimationTrack11GetKeyCountEv($408)|0);
    HEAP32[$233>>2] = $409;
    $410 = HEAP32[$233>>2]|0;
    $411 = ($410*3)|0;
    HEAP32[$234>>2] = $411;
    $412 = HEAP32[$233>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($235,$412);
    $413 = HEAP32[$233>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($236,$413);
    $414 = HEAP32[$233>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($237,$414);
    $415 = HEAP32[$233>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($238,$415);
    $416 = HEAP32[$233>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($239,$416);
    $417 = HEAP32[$233>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($240,$417);
    $418 = HEAP32[$234>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($241,$418);
    HEAP32[$242>>2] = 0;
    while(1) {
     $419 = HEAP32[$242>>2]|0;
     $420 = HEAP32[$233>>2]|0;
     $421 = ($419|0)<($420|0);
     if (!($421)) {
      break;
     }
     $422 = HEAP32[$232>>2]|0;
     $423 = HEAP32[$242>>2]|0;
     $424 = (__ZN3sap2ve4core20VectorAnimationTrack6GetKeyEj($422,$423)|0);
     HEAP32[$243>>2] = $424;
     $425 = HEAP32[$243>>2]|0;
     $426 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($425)|0);
     $427 = +HEAPF32[$426>>2];
     $428 = HEAP32[$242>>2]|0;
     HEAP32[$214>>2] = $235;
     HEAP32[$215>>2] = $428;
     $429 = HEAP32[$214>>2]|0;
     $430 = HEAP32[$429>>2]|0;
     $431 = HEAP32[$215>>2]|0;
     $432 = (($430) + ($431<<2)|0);
     HEAPF32[$432>>2] = $427;
     $433 = HEAP32[$243>>2]|0;
     $434 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($433)|0);
     $435 = +HEAPF32[$434>>2];
     $436 = HEAP32[$242>>2]|0;
     HEAP32[$212>>2] = $236;
     HEAP32[$213>>2] = $436;
     $437 = HEAP32[$212>>2]|0;
     $438 = HEAP32[$437>>2]|0;
     $439 = HEAP32[$213>>2]|0;
     $440 = (($438) + ($439<<2)|0);
     HEAPF32[$440>>2] = $435;
     $441 = HEAP32[$243>>2]|0;
     $442 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($441)|0);
     $443 = +HEAPF32[$442>>2];
     $444 = HEAP32[$242>>2]|0;
     HEAP32[$210>>2] = $237;
     HEAP32[$211>>2] = $444;
     $445 = HEAP32[$210>>2]|0;
     $446 = HEAP32[$445>>2]|0;
     $447 = HEAP32[$211>>2]|0;
     $448 = (($446) + ($447<<2)|0);
     HEAPF32[$448>>2] = $443;
     $449 = HEAP32[$243>>2]|0;
     $450 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($449)|0);
     $451 = +HEAPF32[$450>>2];
     $452 = HEAP32[$242>>2]|0;
     HEAP32[$208>>2] = $238;
     HEAP32[$209>>2] = $452;
     $453 = HEAP32[$208>>2]|0;
     $454 = HEAP32[$453>>2]|0;
     $455 = HEAP32[$209>>2]|0;
     $456 = (($454) + ($455<<2)|0);
     HEAPF32[$456>>2] = $451;
     $457 = HEAP32[$243>>2]|0;
     $458 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($457)|0);
     $459 = +HEAPF32[$458>>2];
     $460 = HEAP32[$242>>2]|0;
     HEAP32[$206>>2] = $239;
     HEAP32[$207>>2] = $460;
     $461 = HEAP32[$206>>2]|0;
     $462 = HEAP32[$461>>2]|0;
     $463 = HEAP32[$207>>2]|0;
     $464 = (($462) + ($463<<2)|0);
     HEAPF32[$464>>2] = $459;
     $465 = HEAP32[$243>>2]|0;
     $466 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($465)|0);
     $467 = +HEAPF32[$466>>2];
     $468 = HEAP32[$242>>2]|0;
     HEAP32[$204>>2] = $240;
     HEAP32[$205>>2] = $468;
     $469 = HEAP32[$204>>2]|0;
     $470 = HEAP32[$469>>2]|0;
     $471 = HEAP32[$205>>2]|0;
     $472 = (($470) + ($471<<2)|0);
     HEAPF32[$472>>2] = $467;
     $473 = HEAP32[$243>>2]|0;
     $474 = (__ZNK3sap2ve4core18VectorAnimationKey8GetValueEv($473)|0);
     $475 = +HEAPF32[$474>>2];
     $476 = HEAP32[$242>>2]|0;
     $477 = ($476*3)|0;
     HEAP32[$202>>2] = $241;
     HEAP32[$203>>2] = $477;
     $478 = HEAP32[$202>>2]|0;
     $479 = HEAP32[$478>>2]|0;
     $480 = HEAP32[$203>>2]|0;
     $481 = (($479) + ($480<<2)|0);
     HEAPF32[$481>>2] = $475;
     $482 = HEAP32[$243>>2]|0;
     $483 = (__ZNK3sap2ve4core18VectorAnimationKey8GetValueEv($482)|0);
     $484 = ((($483)) + 4|0);
     $485 = +HEAPF32[$484>>2];
     $486 = HEAP32[$242>>2]|0;
     $487 = ($486*3)|0;
     $488 = (($487) + 1)|0;
     HEAP32[$200>>2] = $241;
     HEAP32[$201>>2] = $488;
     $489 = HEAP32[$200>>2]|0;
     $490 = HEAP32[$489>>2]|0;
     $491 = HEAP32[$201>>2]|0;
     $492 = (($490) + ($491<<2)|0);
     HEAPF32[$492>>2] = $485;
     $493 = HEAP32[$243>>2]|0;
     $494 = (__ZNK3sap2ve4core18VectorAnimationKey8GetValueEv($493)|0);
     $495 = ((($494)) + 8|0);
     $496 = +HEAPF32[$495>>2];
     $497 = HEAP32[$242>>2]|0;
     $498 = ($497*3)|0;
     $499 = (($498) + 2)|0;
     HEAP32[$198>>2] = $241;
     HEAP32[$199>>2] = $499;
     $500 = HEAP32[$198>>2]|0;
     $501 = HEAP32[$500>>2]|0;
     $502 = HEAP32[$199>>2]|0;
     $503 = (($501) + ($502<<2)|0);
     HEAPF32[$503>>2] = $496;
     $504 = HEAP32[$242>>2]|0;
     $505 = (($504) + 1)|0;
     HEAP32[$242>>2] = $505;
    }
    $506 = ((($306)) + 24|0);
    $507 = HEAP32[$506>>2]|0;
    $508 = HEAP32[$226>>2]|0;
    $509 = HEAP32[$232>>2]|0;
    $510 = HEAP32[$232>>2]|0;
    $511 = (__ZNK3sap2ve4core14AnimationTrack20GetInterpolationTypeEv($510)|0);
    $512 = HEAP32[$232>>2]|0;
    $513 = (__ZNK3sap2ve4core14AnimationTrack14GetCyclicStartEv($512)|0);
    $514 = HEAP32[$232>>2]|0;
    $515 = (__ZNK3sap2ve4core14AnimationTrack12GetCyclicEndEv($514)|0);
    $516 = HEAP32[$233>>2]|0;
    $517 = HEAP32[$234>>2]|0;
    HEAP32[$197>>2] = $235;
    $518 = HEAP32[$197>>2]|0;
    $519 = HEAP32[$518>>2]|0;
    HEAP32[$196>>2] = $519;
    $520 = HEAP32[$196>>2]|0;
    HEAP32[$195>>2] = $236;
    $521 = HEAP32[$195>>2]|0;
    $522 = HEAP32[$521>>2]|0;
    HEAP32[$194>>2] = $522;
    $523 = HEAP32[$194>>2]|0;
    HEAP32[$193>>2] = $237;
    $524 = HEAP32[$193>>2]|0;
    $525 = HEAP32[$524>>2]|0;
    HEAP32[$192>>2] = $525;
    $526 = HEAP32[$192>>2]|0;
    HEAP32[$191>>2] = $238;
    $527 = HEAP32[$191>>2]|0;
    $528 = HEAP32[$527>>2]|0;
    HEAP32[$190>>2] = $528;
    $529 = HEAP32[$190>>2]|0;
    HEAP32[$189>>2] = $239;
    $530 = HEAP32[$189>>2]|0;
    $531 = HEAP32[$530>>2]|0;
    HEAP32[$188>>2] = $531;
    $532 = HEAP32[$188>>2]|0;
    HEAP32[$173>>2] = $240;
    $533 = HEAP32[$173>>2]|0;
    $534 = HEAP32[$533>>2]|0;
    HEAP32[$172>>2] = $534;
    $535 = HEAP32[$172>>2]|0;
    HEAP32[$157>>2] = $241;
    $536 = HEAP32[$157>>2]|0;
    $537 = HEAP32[$536>>2]|0;
    HEAP32[$156>>2] = $537;
    $538 = HEAP32[$156>>2]|0;
    _SceneBuilderAdapter_insertAnimationTrack(($507|0),($508|0),($509|0),2,3,($511|0),($513|0),($515|0),($516|0),($517|0),($520|0),($523|0),($526|0),($529|0),($532|0),($535|0),($538|0));
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($241);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($240);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($239);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($238);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($237);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($236);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($235);
   }
   $539 = HEAP32[$227>>2]|0;
   $540 = (__ZNK3sap2ve4core17InstanceAnimation20GetRotationAnimationEv($539)|0);
   HEAP32[$244>>2] = $540;
   $541 = HEAP32[$244>>2]|0;
   $542 = ($541|0)!=(0|0);
   if ($542) {
    $543 = HEAP32[$244>>2]|0;
    $544 = (__ZNK3sap2ve4core14AnimationTrack11GetKeyCountEv($543)|0);
    HEAP32[$245>>2] = $544;
    $545 = HEAP32[$245>>2]|0;
    $546 = $545<<2;
    HEAP32[$246>>2] = $546;
    $547 = HEAP32[$245>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($247,$547);
    $548 = HEAP32[$245>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($248,$548);
    $549 = HEAP32[$245>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($249,$549);
    $550 = HEAP32[$245>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($250,$550);
    $551 = HEAP32[$245>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($251,$551);
    $552 = HEAP32[$245>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($252,$552);
    $553 = HEAP32[$246>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($253,$553);
    $554 = HEAP32[$244>>2]|0;
    $555 = HEAP32[$554>>2]|0;
    $556 = ((($555)) + 16|0);
    $557 = HEAP32[$556>>2]|0;
    $558 = (FUNCTION_TABLE_ii[$557 & 2047]($554)|0);
    $559 = ($558|0)==(4);
    $560 = HEAP32[$244>>2]|0;
    L22: do {
     if ($559) {
      HEAP32[$254>>2] = $560;
      HEAP32[$255>>2] = 0;
      while(1) {
       $561 = HEAP32[$255>>2]|0;
       $562 = HEAP32[$245>>2]|0;
       $563 = ($561|0)<($562|0);
       if (!($563)) {
        break L22;
       }
       $564 = HEAP32[$254>>2]|0;
       $565 = HEAP32[$255>>2]|0;
       $566 = (__ZN3sap2ve4core22AngleAxisRotationTrack6GetKeyEj($564,$565)|0);
       HEAP32[$256>>2] = $566;
       $567 = HEAP32[$256>>2]|0;
       $568 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($567)|0);
       $569 = +HEAPF32[$568>>2];
       $570 = HEAP32[$255>>2]|0;
       HEAP32[$154>>2] = $247;
       HEAP32[$155>>2] = $570;
       $571 = HEAP32[$154>>2]|0;
       $572 = HEAP32[$571>>2]|0;
       $573 = HEAP32[$155>>2]|0;
       $574 = (($572) + ($573<<2)|0);
       HEAPF32[$574>>2] = $569;
       $575 = HEAP32[$256>>2]|0;
       $576 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($575)|0);
       $577 = +HEAPF32[$576>>2];
       $578 = HEAP32[$255>>2]|0;
       HEAP32[$152>>2] = $248;
       HEAP32[$153>>2] = $578;
       $579 = HEAP32[$152>>2]|0;
       $580 = HEAP32[$579>>2]|0;
       $581 = HEAP32[$153>>2]|0;
       $582 = (($580) + ($581<<2)|0);
       HEAPF32[$582>>2] = $577;
       $583 = HEAP32[$256>>2]|0;
       $584 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($583)|0);
       $585 = +HEAPF32[$584>>2];
       $586 = HEAP32[$255>>2]|0;
       HEAP32[$150>>2] = $249;
       HEAP32[$151>>2] = $586;
       $587 = HEAP32[$150>>2]|0;
       $588 = HEAP32[$587>>2]|0;
       $589 = HEAP32[$151>>2]|0;
       $590 = (($588) + ($589<<2)|0);
       HEAPF32[$590>>2] = $585;
       $591 = HEAP32[$256>>2]|0;
       $592 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($591)|0);
       $593 = +HEAPF32[$592>>2];
       $594 = HEAP32[$255>>2]|0;
       HEAP32[$148>>2] = $250;
       HEAP32[$149>>2] = $594;
       $595 = HEAP32[$148>>2]|0;
       $596 = HEAP32[$595>>2]|0;
       $597 = HEAP32[$149>>2]|0;
       $598 = (($596) + ($597<<2)|0);
       HEAPF32[$598>>2] = $593;
       $599 = HEAP32[$256>>2]|0;
       $600 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($599)|0);
       $601 = +HEAPF32[$600>>2];
       $602 = HEAP32[$255>>2]|0;
       HEAP32[$146>>2] = $251;
       HEAP32[$147>>2] = $602;
       $603 = HEAP32[$146>>2]|0;
       $604 = HEAP32[$603>>2]|0;
       $605 = HEAP32[$147>>2]|0;
       $606 = (($604) + ($605<<2)|0);
       HEAPF32[$606>>2] = $601;
       $607 = HEAP32[$256>>2]|0;
       $608 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($607)|0);
       $609 = +HEAPF32[$608>>2];
       $610 = HEAP32[$255>>2]|0;
       HEAP32[$144>>2] = $252;
       HEAP32[$145>>2] = $610;
       $611 = HEAP32[$144>>2]|0;
       $612 = HEAP32[$611>>2]|0;
       $613 = HEAP32[$145>>2]|0;
       $614 = (($612) + ($613<<2)|0);
       HEAPF32[$614>>2] = $609;
       $615 = HEAP32[$256>>2]|0;
       $616 = (__ZNK3sap2ve4core21AngleAxisAnimationKey8GetValueEv($615)|0);
       $617 = (__ZNK3sap2ve9AngleAxis7GetAxisEv($616)|0);
       $618 = +HEAPF32[$617>>2];
       $619 = HEAP32[$255>>2]|0;
       $620 = $619<<2;
       HEAP32[$142>>2] = $253;
       HEAP32[$143>>2] = $620;
       $621 = HEAP32[$142>>2]|0;
       $622 = HEAP32[$621>>2]|0;
       $623 = HEAP32[$143>>2]|0;
       $624 = (($622) + ($623<<2)|0);
       HEAPF32[$624>>2] = $618;
       $625 = HEAP32[$256>>2]|0;
       $626 = (__ZNK3sap2ve4core21AngleAxisAnimationKey8GetValueEv($625)|0);
       $627 = (__ZNK3sap2ve9AngleAxis7GetAxisEv($626)|0);
       $628 = ((($627)) + 4|0);
       $629 = +HEAPF32[$628>>2];
       $630 = HEAP32[$255>>2]|0;
       $631 = $630<<2;
       $632 = (($631) + 1)|0;
       HEAP32[$140>>2] = $253;
       HEAP32[$141>>2] = $632;
       $633 = HEAP32[$140>>2]|0;
       $634 = HEAP32[$633>>2]|0;
       $635 = HEAP32[$141>>2]|0;
       $636 = (($634) + ($635<<2)|0);
       HEAPF32[$636>>2] = $629;
       $637 = HEAP32[$256>>2]|0;
       $638 = (__ZNK3sap2ve4core21AngleAxisAnimationKey8GetValueEv($637)|0);
       $639 = (__ZNK3sap2ve9AngleAxis7GetAxisEv($638)|0);
       $640 = ((($639)) + 8|0);
       $641 = +HEAPF32[$640>>2];
       $642 = HEAP32[$255>>2]|0;
       $643 = $642<<2;
       $644 = (($643) + 2)|0;
       HEAP32[$138>>2] = $253;
       HEAP32[$139>>2] = $644;
       $645 = HEAP32[$138>>2]|0;
       $646 = HEAP32[$645>>2]|0;
       $647 = HEAP32[$139>>2]|0;
       $648 = (($646) + ($647<<2)|0);
       HEAPF32[$648>>2] = $641;
       $649 = HEAP32[$256>>2]|0;
       $650 = (__ZNK3sap2ve4core21AngleAxisAnimationKey8GetValueEv($649)|0);
       $651 = (+__ZNK3sap2ve9AngleAxis8GetAngleEv($650));
       $652 = HEAP32[$255>>2]|0;
       $653 = $652<<2;
       $654 = (($653) + 3)|0;
       HEAP32[$136>>2] = $253;
       HEAP32[$137>>2] = $654;
       $655 = HEAP32[$136>>2]|0;
       $656 = HEAP32[$655>>2]|0;
       $657 = HEAP32[$137>>2]|0;
       $658 = (($656) + ($657<<2)|0);
       HEAPF32[$658>>2] = $651;
       $659 = HEAP32[$255>>2]|0;
       $660 = (($659) + 1)|0;
       HEAP32[$255>>2] = $660;
      }
     } else {
      $661 = HEAP32[$560>>2]|0;
      $662 = ((($661)) + 16|0);
      $663 = HEAP32[$662>>2]|0;
      $664 = (FUNCTION_TABLE_ii[$663 & 2047]($560)|0);
      $665 = ($664|0)==(5);
      $666 = HEAP32[$244>>2]|0;
      if ($665) {
       HEAP32[$257>>2] = $666;
       HEAP32[$258>>2] = 0;
       while(1) {
        $667 = HEAP32[$258>>2]|0;
        $668 = HEAP32[$245>>2]|0;
        $669 = ($667|0)<($668|0);
        if (!($669)) {
         break L22;
        }
        $670 = HEAP32[$257>>2]|0;
        $671 = HEAP32[$258>>2]|0;
        $672 = (__ZN3sap2ve4core23QuaternionRotationTrack6GetKeyEj($670,$671)|0);
        HEAP32[$259>>2] = $672;
        $673 = HEAP32[$259>>2]|0;
        $674 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($673)|0);
        $675 = +HEAPF32[$674>>2];
        $676 = HEAP32[$258>>2]|0;
        HEAP32[$134>>2] = $247;
        HEAP32[$135>>2] = $676;
        $677 = HEAP32[$134>>2]|0;
        $678 = HEAP32[$677>>2]|0;
        $679 = HEAP32[$135>>2]|0;
        $680 = (($678) + ($679<<2)|0);
        HEAPF32[$680>>2] = $675;
        $681 = HEAP32[$259>>2]|0;
        $682 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($681)|0);
        $683 = +HEAPF32[$682>>2];
        $684 = HEAP32[$258>>2]|0;
        HEAP32[$132>>2] = $248;
        HEAP32[$133>>2] = $684;
        $685 = HEAP32[$132>>2]|0;
        $686 = HEAP32[$685>>2]|0;
        $687 = HEAP32[$133>>2]|0;
        $688 = (($686) + ($687<<2)|0);
        HEAPF32[$688>>2] = $683;
        $689 = HEAP32[$259>>2]|0;
        $690 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($689)|0);
        $691 = +HEAPF32[$690>>2];
        $692 = HEAP32[$258>>2]|0;
        HEAP32[$130>>2] = $249;
        HEAP32[$131>>2] = $692;
        $693 = HEAP32[$130>>2]|0;
        $694 = HEAP32[$693>>2]|0;
        $695 = HEAP32[$131>>2]|0;
        $696 = (($694) + ($695<<2)|0);
        HEAPF32[$696>>2] = $691;
        $697 = HEAP32[$259>>2]|0;
        $698 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($697)|0);
        $699 = +HEAPF32[$698>>2];
        $700 = HEAP32[$258>>2]|0;
        HEAP32[$128>>2] = $250;
        HEAP32[$129>>2] = $700;
        $701 = HEAP32[$128>>2]|0;
        $702 = HEAP32[$701>>2]|0;
        $703 = HEAP32[$129>>2]|0;
        $704 = (($702) + ($703<<2)|0);
        HEAPF32[$704>>2] = $699;
        $705 = HEAP32[$259>>2]|0;
        $706 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($705)|0);
        $707 = +HEAPF32[$706>>2];
        $708 = HEAP32[$258>>2]|0;
        HEAP32[$126>>2] = $251;
        HEAP32[$127>>2] = $708;
        $709 = HEAP32[$126>>2]|0;
        $710 = HEAP32[$709>>2]|0;
        $711 = HEAP32[$127>>2]|0;
        $712 = (($710) + ($711<<2)|0);
        HEAPF32[$712>>2] = $707;
        $713 = HEAP32[$259>>2]|0;
        $714 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($713)|0);
        $715 = +HEAPF32[$714>>2];
        $716 = HEAP32[$258>>2]|0;
        HEAP32[$124>>2] = $252;
        HEAP32[$125>>2] = $716;
        $717 = HEAP32[$124>>2]|0;
        $718 = HEAP32[$717>>2]|0;
        $719 = HEAP32[$125>>2]|0;
        $720 = (($718) + ($719<<2)|0);
        HEAPF32[$720>>2] = $715;
        $721 = HEAP32[$259>>2]|0;
        $722 = (__ZNK3sap2ve4core22QuaternionAnimationKey8GetValueEv($721)|0);
        $723 = +HEAPF32[$722>>2];
        $724 = HEAP32[$258>>2]|0;
        $725 = $724<<2;
        HEAP32[$122>>2] = $253;
        HEAP32[$123>>2] = $725;
        $726 = HEAP32[$122>>2]|0;
        $727 = HEAP32[$726>>2]|0;
        $728 = HEAP32[$123>>2]|0;
        $729 = (($727) + ($728<<2)|0);
        HEAPF32[$729>>2] = $723;
        $730 = HEAP32[$259>>2]|0;
        $731 = (__ZNK3sap2ve4core22QuaternionAnimationKey8GetValueEv($730)|0);
        $732 = ((($731)) + 4|0);
        $733 = +HEAPF32[$732>>2];
        $734 = HEAP32[$258>>2]|0;
        $735 = $734<<2;
        $736 = (($735) + 1)|0;
        HEAP32[$120>>2] = $253;
        HEAP32[$121>>2] = $736;
        $737 = HEAP32[$120>>2]|0;
        $738 = HEAP32[$737>>2]|0;
        $739 = HEAP32[$121>>2]|0;
        $740 = (($738) + ($739<<2)|0);
        HEAPF32[$740>>2] = $733;
        $741 = HEAP32[$259>>2]|0;
        $742 = (__ZNK3sap2ve4core22QuaternionAnimationKey8GetValueEv($741)|0);
        $743 = ((($742)) + 8|0);
        $744 = +HEAPF32[$743>>2];
        $745 = HEAP32[$258>>2]|0;
        $746 = $745<<2;
        $747 = (($746) + 2)|0;
        HEAP32[$118>>2] = $253;
        HEAP32[$119>>2] = $747;
        $748 = HEAP32[$118>>2]|0;
        $749 = HEAP32[$748>>2]|0;
        $750 = HEAP32[$119>>2]|0;
        $751 = (($749) + ($750<<2)|0);
        HEAPF32[$751>>2] = $744;
        $752 = HEAP32[$259>>2]|0;
        $753 = (__ZNK3sap2ve4core22QuaternionAnimationKey8GetValueEv($752)|0);
        $754 = ((($753)) + 12|0);
        $755 = +HEAPF32[$754>>2];
        $756 = HEAP32[$258>>2]|0;
        $757 = $756<<2;
        $758 = (($757) + 3)|0;
        HEAP32[$102>>2] = $253;
        HEAP32[$103>>2] = $758;
        $759 = HEAP32[$102>>2]|0;
        $760 = HEAP32[$759>>2]|0;
        $761 = HEAP32[$103>>2]|0;
        $762 = (($760) + ($761<<2)|0);
        HEAPF32[$762>>2] = $755;
        $763 = HEAP32[$258>>2]|0;
        $764 = (($763) + 1)|0;
        HEAP32[$258>>2] = $764;
       }
      }
      $765 = HEAP32[$666>>2]|0;
      $766 = ((($765)) + 16|0);
      $767 = HEAP32[$766>>2]|0;
      $768 = (FUNCTION_TABLE_ii[$767 & 2047]($666)|0);
      $769 = ($768|0)==(6);
      if ($769) {
       $770 = HEAP32[$244>>2]|0;
       HEAP32[$260>>2] = $770;
       HEAP32[$261>>2] = 0;
       while(1) {
        $771 = HEAP32[$261>>2]|0;
        $772 = HEAP32[$245>>2]|0;
        $773 = ($771|0)<($772|0);
        if (!($773)) {
         break L22;
        }
        $774 = HEAP32[$260>>2]|0;
        $775 = HEAP32[$261>>2]|0;
        $776 = (__ZN3sap2ve4core23EulerAngleRotationTrack6GetKeyEj($774,$775)|0);
        HEAP32[$262>>2] = $776;
        $777 = HEAP32[$262>>2]|0;
        $778 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($777)|0);
        $779 = +HEAPF32[$778>>2];
        $780 = HEAP32[$261>>2]|0;
        HEAP32[$86>>2] = $247;
        HEAP32[$87>>2] = $780;
        $781 = HEAP32[$86>>2]|0;
        $782 = HEAP32[$781>>2]|0;
        $783 = HEAP32[$87>>2]|0;
        $784 = (($782) + ($783<<2)|0);
        HEAPF32[$784>>2] = $779;
        $785 = HEAP32[$262>>2]|0;
        $786 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($785)|0);
        $787 = +HEAPF32[$786>>2];
        $788 = HEAP32[$261>>2]|0;
        HEAP32[$84>>2] = $248;
        HEAP32[$85>>2] = $788;
        $789 = HEAP32[$84>>2]|0;
        $790 = HEAP32[$789>>2]|0;
        $791 = HEAP32[$85>>2]|0;
        $792 = (($790) + ($791<<2)|0);
        HEAPF32[$792>>2] = $787;
        $793 = HEAP32[$262>>2]|0;
        $794 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($793)|0);
        $795 = +HEAPF32[$794>>2];
        $796 = HEAP32[$261>>2]|0;
        HEAP32[$82>>2] = $249;
        HEAP32[$83>>2] = $796;
        $797 = HEAP32[$82>>2]|0;
        $798 = HEAP32[$797>>2]|0;
        $799 = HEAP32[$83>>2]|0;
        $800 = (($798) + ($799<<2)|0);
        HEAPF32[$800>>2] = $795;
        $801 = HEAP32[$262>>2]|0;
        $802 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($801)|0);
        $803 = +HEAPF32[$802>>2];
        $804 = HEAP32[$261>>2]|0;
        HEAP32[$80>>2] = $250;
        HEAP32[$81>>2] = $804;
        $805 = HEAP32[$80>>2]|0;
        $806 = HEAP32[$805>>2]|0;
        $807 = HEAP32[$81>>2]|0;
        $808 = (($806) + ($807<<2)|0);
        HEAPF32[$808>>2] = $803;
        $809 = HEAP32[$262>>2]|0;
        $810 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($809)|0);
        $811 = +HEAPF32[$810>>2];
        $812 = HEAP32[$261>>2]|0;
        HEAP32[$78>>2] = $251;
        HEAP32[$79>>2] = $812;
        $813 = HEAP32[$78>>2]|0;
        $814 = HEAP32[$813>>2]|0;
        $815 = HEAP32[$79>>2]|0;
        $816 = (($814) + ($815<<2)|0);
        HEAPF32[$816>>2] = $811;
        $817 = HEAP32[$262>>2]|0;
        $818 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($817)|0);
        $819 = +HEAPF32[$818>>2];
        $820 = HEAP32[$261>>2]|0;
        HEAP32[$76>>2] = $252;
        HEAP32[$77>>2] = $820;
        $821 = HEAP32[$76>>2]|0;
        $822 = HEAP32[$821>>2]|0;
        $823 = HEAP32[$77>>2]|0;
        $824 = (($822) + ($823<<2)|0);
        HEAPF32[$824>>2] = $819;
        $825 = HEAP32[$262>>2]|0;
        $826 = (__ZNK3sap2ve4core22EulerAngleAnimationKey8GetValueEv($825)|0);
        $827 = +HEAPF32[$826>>2];
        $828 = HEAP32[$261>>2]|0;
        $829 = $828<<2;
        HEAP32[$74>>2] = $253;
        HEAP32[$75>>2] = $829;
        $830 = HEAP32[$74>>2]|0;
        $831 = HEAP32[$830>>2]|0;
        $832 = HEAP32[$75>>2]|0;
        $833 = (($831) + ($832<<2)|0);
        HEAPF32[$833>>2] = $827;
        $834 = HEAP32[$262>>2]|0;
        $835 = (__ZNK3sap2ve4core22EulerAngleAnimationKey8GetValueEv($834)|0);
        $836 = ((($835)) + 4|0);
        $837 = +HEAPF32[$836>>2];
        $838 = HEAP32[$261>>2]|0;
        $839 = $838<<2;
        $840 = (($839) + 1)|0;
        HEAP32[$72>>2] = $253;
        HEAP32[$73>>2] = $840;
        $841 = HEAP32[$72>>2]|0;
        $842 = HEAP32[$841>>2]|0;
        $843 = HEAP32[$73>>2]|0;
        $844 = (($842) + ($843<<2)|0);
        HEAPF32[$844>>2] = $837;
        $845 = HEAP32[$262>>2]|0;
        $846 = (__ZNK3sap2ve4core22EulerAngleAnimationKey8GetValueEv($845)|0);
        $847 = ((($846)) + 8|0);
        $848 = +HEAPF32[$847>>2];
        $849 = HEAP32[$261>>2]|0;
        $850 = $849<<2;
        $851 = (($850) + 2)|0;
        HEAP32[$70>>2] = $253;
        HEAP32[$71>>2] = $851;
        $852 = HEAP32[$70>>2]|0;
        $853 = HEAP32[$852>>2]|0;
        $854 = HEAP32[$71>>2]|0;
        $855 = (($853) + ($854<<2)|0);
        HEAPF32[$855>>2] = $848;
        $856 = HEAP32[$262>>2]|0;
        $857 = (__ZNK3sap2ve4core22EulerAngleAnimationKey8GetValueEv($856)|0);
        $858 = ((($857)) + 12|0);
        $859 = HEAP8[$858>>0]|0;
        $860 = (+($859&255));
        $861 = HEAP32[$261>>2]|0;
        $862 = $861<<2;
        $863 = (($862) + 3)|0;
        HEAP32[$68>>2] = $253;
        HEAP32[$69>>2] = $863;
        $864 = HEAP32[$68>>2]|0;
        $865 = HEAP32[$864>>2]|0;
        $866 = HEAP32[$69>>2]|0;
        $867 = (($865) + ($866<<2)|0);
        HEAPF32[$867>>2] = $860;
        $868 = HEAP32[$261>>2]|0;
        $869 = (($868) + 1)|0;
        HEAP32[$261>>2] = $869;
       }
      }
     }
    } while(0);
    $870 = ((($306)) + 24|0);
    $871 = HEAP32[$870>>2]|0;
    $872 = HEAP32[$226>>2]|0;
    $873 = HEAP32[$244>>2]|0;
    $874 = HEAP32[$244>>2]|0;
    $875 = HEAP32[$874>>2]|0;
    $876 = ((($875)) + 16|0);
    $877 = HEAP32[$876>>2]|0;
    $878 = (FUNCTION_TABLE_ii[$877 & 2047]($874)|0);
    $879 = HEAP32[$244>>2]|0;
    $880 = (__ZNK3sap2ve4core14AnimationTrack20GetInterpolationTypeEv($879)|0);
    $881 = HEAP32[$244>>2]|0;
    $882 = (__ZNK3sap2ve4core14AnimationTrack14GetCyclicStartEv($881)|0);
    $883 = HEAP32[$244>>2]|0;
    $884 = (__ZNK3sap2ve4core14AnimationTrack12GetCyclicEndEv($883)|0);
    $885 = HEAP32[$245>>2]|0;
    $886 = HEAP32[$246>>2]|0;
    HEAP32[$67>>2] = $247;
    $887 = HEAP32[$67>>2]|0;
    $888 = HEAP32[$887>>2]|0;
    HEAP32[$66>>2] = $888;
    $889 = HEAP32[$66>>2]|0;
    HEAP32[$65>>2] = $248;
    $890 = HEAP32[$65>>2]|0;
    $891 = HEAP32[$890>>2]|0;
    HEAP32[$64>>2] = $891;
    $892 = HEAP32[$64>>2]|0;
    HEAP32[$63>>2] = $249;
    $893 = HEAP32[$63>>2]|0;
    $894 = HEAP32[$893>>2]|0;
    HEAP32[$62>>2] = $894;
    $895 = HEAP32[$62>>2]|0;
    HEAP32[$61>>2] = $250;
    $896 = HEAP32[$61>>2]|0;
    $897 = HEAP32[$896>>2]|0;
    HEAP32[$60>>2] = $897;
    $898 = HEAP32[$60>>2]|0;
    HEAP32[$59>>2] = $251;
    $899 = HEAP32[$59>>2]|0;
    $900 = HEAP32[$899>>2]|0;
    HEAP32[$58>>2] = $900;
    $901 = HEAP32[$58>>2]|0;
    HEAP32[$57>>2] = $252;
    $902 = HEAP32[$57>>2]|0;
    $903 = HEAP32[$902>>2]|0;
    HEAP32[$56>>2] = $903;
    $904 = HEAP32[$56>>2]|0;
    HEAP32[$55>>2] = $253;
    $905 = HEAP32[$55>>2]|0;
    $906 = HEAP32[$905>>2]|0;
    HEAP32[$54>>2] = $906;
    $907 = HEAP32[$54>>2]|0;
    _SceneBuilderAdapter_insertAnimationTrack(($871|0),($872|0),($873|0),4,($878|0),($880|0),($882|0),($884|0),($885|0),($886|0),($889|0),($892|0),($895|0),($898|0),($901|0),($904|0),($907|0));
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($253);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($252);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($251);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($250);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($249);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($248);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($247);
   }
   $908 = HEAP32[$227>>2]|0;
   $909 = (__ZNK3sap2ve4core17InstanceAnimation17GetScaleAnimationEv($908)|0);
   HEAP32[$263>>2] = $909;
   $910 = HEAP32[$263>>2]|0;
   $911 = ($910|0)!=(0|0);
   if ($911) {
    $912 = HEAP32[$263>>2]|0;
    $913 = (__ZNK3sap2ve4core14AnimationTrack11GetKeyCountEv($912)|0);
    HEAP32[$264>>2] = $913;
    $914 = HEAP32[$264>>2]|0;
    $915 = ($914*3)|0;
    HEAP32[$265>>2] = $915;
    $916 = HEAP32[$264>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($266,$916);
    $917 = HEAP32[$264>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($267,$917);
    $918 = HEAP32[$264>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($268,$918);
    $919 = HEAP32[$264>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($269,$919);
    $920 = HEAP32[$264>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($270,$920);
    $921 = HEAP32[$264>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($271,$921);
    $922 = HEAP32[$265>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($272,$922);
    $923 = HEAP32[$263>>2]|0;
    $924 = HEAP32[$923>>2]|0;
    $925 = ((($924)) + 16|0);
    $926 = HEAP32[$925>>2]|0;
    $927 = (FUNCTION_TABLE_ii[$926 & 2047]($923)|0);
    $928 = ($927|0)==(0);
    $929 = HEAP32[$263>>2]|0;
    L42: do {
     if ($928) {
      HEAP32[$273>>2] = $929;
      HEAP32[$274>>2] = 0;
      while(1) {
       $930 = HEAP32[$274>>2]|0;
       $931 = HEAP32[$264>>2]|0;
       $932 = ($930|0)<($931|0);
       if (!($932)) {
        break L42;
       }
       $933 = HEAP32[$273>>2]|0;
       $934 = HEAP32[$274>>2]|0;
       $935 = (__ZN3sap2ve4core20ScalarAnimationTrack6GetKeyEj($933,$934)|0);
       HEAP32[$275>>2] = $935;
       $936 = HEAP32[$275>>2]|0;
       $937 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($936)|0);
       $938 = +HEAPF32[$937>>2];
       $939 = HEAP32[$274>>2]|0;
       HEAP32[$52>>2] = $266;
       HEAP32[$53>>2] = $939;
       $940 = HEAP32[$52>>2]|0;
       $941 = HEAP32[$940>>2]|0;
       $942 = HEAP32[$53>>2]|0;
       $943 = (($941) + ($942<<2)|0);
       HEAPF32[$943>>2] = $938;
       $944 = HEAP32[$275>>2]|0;
       $945 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($944)|0);
       $946 = +HEAPF32[$945>>2];
       $947 = HEAP32[$274>>2]|0;
       HEAP32[$50>>2] = $267;
       HEAP32[$51>>2] = $947;
       $948 = HEAP32[$50>>2]|0;
       $949 = HEAP32[$948>>2]|0;
       $950 = HEAP32[$51>>2]|0;
       $951 = (($949) + ($950<<2)|0);
       HEAPF32[$951>>2] = $946;
       $952 = HEAP32[$275>>2]|0;
       $953 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($952)|0);
       $954 = +HEAPF32[$953>>2];
       $955 = HEAP32[$274>>2]|0;
       HEAP32[$48>>2] = $268;
       HEAP32[$49>>2] = $955;
       $956 = HEAP32[$48>>2]|0;
       $957 = HEAP32[$956>>2]|0;
       $958 = HEAP32[$49>>2]|0;
       $959 = (($957) + ($958<<2)|0);
       HEAPF32[$959>>2] = $954;
       $960 = HEAP32[$275>>2]|0;
       $961 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($960)|0);
       $962 = +HEAPF32[$961>>2];
       $963 = HEAP32[$274>>2]|0;
       HEAP32[$14>>2] = $269;
       HEAP32[$15>>2] = $963;
       $964 = HEAP32[$14>>2]|0;
       $965 = HEAP32[$964>>2]|0;
       $966 = HEAP32[$15>>2]|0;
       $967 = (($965) + ($966<<2)|0);
       HEAPF32[$967>>2] = $962;
       $968 = HEAP32[$275>>2]|0;
       $969 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($968)|0);
       $970 = +HEAPF32[$969>>2];
       $971 = HEAP32[$274>>2]|0;
       HEAP32[$12>>2] = $270;
       HEAP32[$13>>2] = $971;
       $972 = HEAP32[$12>>2]|0;
       $973 = HEAP32[$972>>2]|0;
       $974 = HEAP32[$13>>2]|0;
       $975 = (($973) + ($974<<2)|0);
       HEAPF32[$975>>2] = $970;
       $976 = HEAP32[$275>>2]|0;
       $977 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($976)|0);
       $978 = +HEAPF32[$977>>2];
       $979 = HEAP32[$274>>2]|0;
       HEAP32[$4>>2] = $271;
       HEAP32[$5>>2] = $979;
       $980 = HEAP32[$4>>2]|0;
       $981 = HEAP32[$980>>2]|0;
       $982 = HEAP32[$5>>2]|0;
       $983 = (($981) + ($982<<2)|0);
       HEAPF32[$983>>2] = $978;
       $984 = HEAP32[$275>>2]|0;
       $985 = (__ZNK3sap2ve4core18ScalarAnimationKey8GetValueEv($984)|0);
       $986 = +HEAPF32[$985>>2];
       $987 = HEAP32[$274>>2]|0;
       $988 = ($987*3)|0;
       HEAP32[$6>>2] = $272;
       HEAP32[$7>>2] = $988;
       $989 = HEAP32[$6>>2]|0;
       $990 = HEAP32[$989>>2]|0;
       $991 = HEAP32[$7>>2]|0;
       $992 = (($990) + ($991<<2)|0);
       HEAPF32[$992>>2] = $986;
       $993 = HEAP32[$275>>2]|0;
       $994 = (__ZNK3sap2ve4core18ScalarAnimationKey8GetValueEv($993)|0);
       $995 = +HEAPF32[$994>>2];
       $996 = HEAP32[$274>>2]|0;
       $997 = ($996*3)|0;
       $998 = (($997) + 1)|0;
       HEAP32[$8>>2] = $272;
       HEAP32[$9>>2] = $998;
       $999 = HEAP32[$8>>2]|0;
       $1000 = HEAP32[$999>>2]|0;
       $1001 = HEAP32[$9>>2]|0;
       $1002 = (($1000) + ($1001<<2)|0);
       HEAPF32[$1002>>2] = $995;
       $1003 = HEAP32[$275>>2]|0;
       $1004 = (__ZNK3sap2ve4core18ScalarAnimationKey8GetValueEv($1003)|0);
       $1005 = +HEAPF32[$1004>>2];
       $1006 = HEAP32[$274>>2]|0;
       $1007 = ($1006*3)|0;
       $1008 = (($1007) + 2)|0;
       HEAP32[$10>>2] = $272;
       HEAP32[$11>>2] = $1008;
       $1009 = HEAP32[$10>>2]|0;
       $1010 = HEAP32[$1009>>2]|0;
       $1011 = HEAP32[$11>>2]|0;
       $1012 = (($1010) + ($1011<<2)|0);
       HEAPF32[$1012>>2] = $1005;
       $1013 = HEAP32[$274>>2]|0;
       $1014 = (($1013) + 1)|0;
       HEAP32[$274>>2] = $1014;
      }
     } else {
      $1015 = HEAP32[$929>>2]|0;
      $1016 = ((($1015)) + 16|0);
      $1017 = HEAP32[$1016>>2]|0;
      $1018 = (FUNCTION_TABLE_ii[$1017 & 2047]($929)|0);
      $1019 = ($1018|0)==(3);
      if ($1019) {
       $1020 = HEAP32[$263>>2]|0;
       HEAP32[$276>>2] = $1020;
       HEAP32[$277>>2] = 0;
       while(1) {
        $1021 = HEAP32[$277>>2]|0;
        $1022 = HEAP32[$264>>2]|0;
        $1023 = ($1021|0)<($1022|0);
        if (!($1023)) {
         break L42;
        }
        $1024 = HEAP32[$276>>2]|0;
        $1025 = HEAP32[$277>>2]|0;
        $1026 = (__ZN3sap2ve4core20VectorAnimationTrack6GetKeyEj($1024,$1025)|0);
        HEAP32[$278>>2] = $1026;
        $1027 = HEAP32[$278>>2]|0;
        $1028 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($1027)|0);
        $1029 = +HEAPF32[$1028>>2];
        $1030 = HEAP32[$277>>2]|0;
        HEAP32[$16>>2] = $266;
        HEAP32[$17>>2] = $1030;
        $1031 = HEAP32[$16>>2]|0;
        $1032 = HEAP32[$1031>>2]|0;
        $1033 = HEAP32[$17>>2]|0;
        $1034 = (($1032) + ($1033<<2)|0);
        HEAPF32[$1034>>2] = $1029;
        $1035 = HEAP32[$278>>2]|0;
        $1036 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($1035)|0);
        $1037 = +HEAPF32[$1036>>2];
        $1038 = HEAP32[$277>>2]|0;
        HEAP32[$18>>2] = $267;
        HEAP32[$19>>2] = $1038;
        $1039 = HEAP32[$18>>2]|0;
        $1040 = HEAP32[$1039>>2]|0;
        $1041 = HEAP32[$19>>2]|0;
        $1042 = (($1040) + ($1041<<2)|0);
        HEAPF32[$1042>>2] = $1037;
        $1043 = HEAP32[$278>>2]|0;
        $1044 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($1043)|0);
        $1045 = +HEAPF32[$1044>>2];
        $1046 = HEAP32[$277>>2]|0;
        HEAP32[$20>>2] = $268;
        HEAP32[$21>>2] = $1046;
        $1047 = HEAP32[$20>>2]|0;
        $1048 = HEAP32[$1047>>2]|0;
        $1049 = HEAP32[$21>>2]|0;
        $1050 = (($1048) + ($1049<<2)|0);
        HEAPF32[$1050>>2] = $1045;
        $1051 = HEAP32[$278>>2]|0;
        $1052 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($1051)|0);
        $1053 = +HEAPF32[$1052>>2];
        $1054 = HEAP32[$277>>2]|0;
        HEAP32[$22>>2] = $269;
        HEAP32[$23>>2] = $1054;
        $1055 = HEAP32[$22>>2]|0;
        $1056 = HEAP32[$1055>>2]|0;
        $1057 = HEAP32[$23>>2]|0;
        $1058 = (($1056) + ($1057<<2)|0);
        HEAPF32[$1058>>2] = $1053;
        $1059 = HEAP32[$278>>2]|0;
        $1060 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($1059)|0);
        $1061 = +HEAPF32[$1060>>2];
        $1062 = HEAP32[$277>>2]|0;
        HEAP32[$24>>2] = $270;
        HEAP32[$25>>2] = $1062;
        $1063 = HEAP32[$24>>2]|0;
        $1064 = HEAP32[$1063>>2]|0;
        $1065 = HEAP32[$25>>2]|0;
        $1066 = (($1064) + ($1065<<2)|0);
        HEAPF32[$1066>>2] = $1061;
        $1067 = HEAP32[$278>>2]|0;
        $1068 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($1067)|0);
        $1069 = +HEAPF32[$1068>>2];
        $1070 = HEAP32[$277>>2]|0;
        HEAP32[$26>>2] = $271;
        HEAP32[$27>>2] = $1070;
        $1071 = HEAP32[$26>>2]|0;
        $1072 = HEAP32[$1071>>2]|0;
        $1073 = HEAP32[$27>>2]|0;
        $1074 = (($1072) + ($1073<<2)|0);
        HEAPF32[$1074>>2] = $1069;
        $1075 = HEAP32[$278>>2]|0;
        $1076 = (__ZNK3sap2ve4core18VectorAnimationKey8GetValueEv($1075)|0);
        $1077 = +HEAPF32[$1076>>2];
        $1078 = HEAP32[$277>>2]|0;
        $1079 = ($1078*3)|0;
        HEAP32[$28>>2] = $272;
        HEAP32[$29>>2] = $1079;
        $1080 = HEAP32[$28>>2]|0;
        $1081 = HEAP32[$1080>>2]|0;
        $1082 = HEAP32[$29>>2]|0;
        $1083 = (($1081) + ($1082<<2)|0);
        HEAPF32[$1083>>2] = $1077;
        $1084 = HEAP32[$278>>2]|0;
        $1085 = (__ZNK3sap2ve4core18VectorAnimationKey8GetValueEv($1084)|0);
        $1086 = ((($1085)) + 4|0);
        $1087 = +HEAPF32[$1086>>2];
        $1088 = HEAP32[$277>>2]|0;
        $1089 = ($1088*3)|0;
        $1090 = (($1089) + 1)|0;
        HEAP32[$30>>2] = $272;
        HEAP32[$31>>2] = $1090;
        $1091 = HEAP32[$30>>2]|0;
        $1092 = HEAP32[$1091>>2]|0;
        $1093 = HEAP32[$31>>2]|0;
        $1094 = (($1092) + ($1093<<2)|0);
        HEAPF32[$1094>>2] = $1087;
        $1095 = HEAP32[$278>>2]|0;
        $1096 = (__ZNK3sap2ve4core18VectorAnimationKey8GetValueEv($1095)|0);
        $1097 = ((($1096)) + 8|0);
        $1098 = +HEAPF32[$1097>>2];
        $1099 = HEAP32[$277>>2]|0;
        $1100 = ($1099*3)|0;
        $1101 = (($1100) + 2)|0;
        HEAP32[$32>>2] = $272;
        HEAP32[$33>>2] = $1101;
        $1102 = HEAP32[$32>>2]|0;
        $1103 = HEAP32[$1102>>2]|0;
        $1104 = HEAP32[$33>>2]|0;
        $1105 = (($1103) + ($1104<<2)|0);
        HEAPF32[$1105>>2] = $1098;
        $1106 = HEAP32[$277>>2]|0;
        $1107 = (($1106) + 1)|0;
        HEAP32[$277>>2] = $1107;
       }
      }
     }
    } while(0);
    $1108 = ((($306)) + 24|0);
    $1109 = HEAP32[$1108>>2]|0;
    $1110 = HEAP32[$226>>2]|0;
    $1111 = HEAP32[$263>>2]|0;
    $1112 = HEAP32[$263>>2]|0;
    $1113 = (__ZNK3sap2ve4core14AnimationTrack20GetInterpolationTypeEv($1112)|0);
    $1114 = HEAP32[$263>>2]|0;
    $1115 = (__ZNK3sap2ve4core14AnimationTrack14GetCyclicStartEv($1114)|0);
    $1116 = HEAP32[$263>>2]|0;
    $1117 = (__ZNK3sap2ve4core14AnimationTrack12GetCyclicEndEv($1116)|0);
    $1118 = HEAP32[$264>>2]|0;
    $1119 = HEAP32[$265>>2]|0;
    HEAP32[$35>>2] = $266;
    $1120 = HEAP32[$35>>2]|0;
    $1121 = HEAP32[$1120>>2]|0;
    HEAP32[$34>>2] = $1121;
    $1122 = HEAP32[$34>>2]|0;
    HEAP32[$37>>2] = $267;
    $1123 = HEAP32[$37>>2]|0;
    $1124 = HEAP32[$1123>>2]|0;
    HEAP32[$36>>2] = $1124;
    $1125 = HEAP32[$36>>2]|0;
    HEAP32[$39>>2] = $268;
    $1126 = HEAP32[$39>>2]|0;
    $1127 = HEAP32[$1126>>2]|0;
    HEAP32[$38>>2] = $1127;
    $1128 = HEAP32[$38>>2]|0;
    HEAP32[$41>>2] = $269;
    $1129 = HEAP32[$41>>2]|0;
    $1130 = HEAP32[$1129>>2]|0;
    HEAP32[$40>>2] = $1130;
    $1131 = HEAP32[$40>>2]|0;
    HEAP32[$43>>2] = $270;
    $1132 = HEAP32[$43>>2]|0;
    $1133 = HEAP32[$1132>>2]|0;
    HEAP32[$42>>2] = $1133;
    $1134 = HEAP32[$42>>2]|0;
    HEAP32[$45>>2] = $271;
    $1135 = HEAP32[$45>>2]|0;
    $1136 = HEAP32[$1135>>2]|0;
    HEAP32[$44>>2] = $1136;
    $1137 = HEAP32[$44>>2]|0;
    HEAP32[$47>>2] = $272;
    $1138 = HEAP32[$47>>2]|0;
    $1139 = HEAP32[$1138>>2]|0;
    HEAP32[$46>>2] = $1139;
    $1140 = HEAP32[$46>>2]|0;
    _SceneBuilderAdapter_insertAnimationTrack(($1109|0),($1110|0),($1111|0),3,3,($1113|0),($1115|0),($1117|0),($1118|0),($1119|0),($1122|0),($1125|0),($1128|0),($1131|0),($1134|0),($1137|0),($1140|0));
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($272);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($271);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($270);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($269);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($268);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($267);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($266);
   }
   $1141 = HEAP32[$227>>2]|0;
   $1142 = (__ZNK3sap2ve4core17InstanceAnimation19GetOpacityAnimationEv($1141)|0);
   HEAP32[$279>>2] = $1142;
   $1143 = HEAP32[$279>>2]|0;
   $1144 = ($1143|0)!=(0|0);
   if ($1144) {
    $1145 = HEAP32[$279>>2]|0;
    $1146 = (__ZNK3sap2ve4core14AnimationTrack11GetKeyCountEv($1145)|0);
    HEAP32[$280>>2] = $1146;
    $1147 = HEAP32[$280>>2]|0;
    HEAP32[$281>>2] = $1147;
    $1148 = HEAP32[$280>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($282,$1148);
    $1149 = HEAP32[$280>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($283,$1149);
    $1150 = HEAP32[$280>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($284,$1150);
    $1151 = HEAP32[$280>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($285,$1151);
    $1152 = HEAP32[$280>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($286,$1152);
    $1153 = HEAP32[$280>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($287,$1153);
    $1154 = HEAP32[$281>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($288,$1154);
    HEAP32[$289>>2] = 0;
    while(1) {
     $1155 = HEAP32[$289>>2]|0;
     $1156 = HEAP32[$280>>2]|0;
     $1157 = ($1155|0)<($1156|0);
     if (!($1157)) {
      break;
     }
     $1158 = HEAP32[$279>>2]|0;
     $1159 = HEAP32[$289>>2]|0;
     $1160 = (__ZN3sap2ve4core20ScalarAnimationTrack6GetKeyEj($1158,$1159)|0);
     HEAP32[$290>>2] = $1160;
     $1161 = HEAP32[$290>>2]|0;
     $1162 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($1161)|0);
     $1163 = +HEAPF32[$1162>>2];
     $1164 = HEAP32[$289>>2]|0;
     HEAP32[$88>>2] = $282;
     HEAP32[$89>>2] = $1164;
     $1165 = HEAP32[$88>>2]|0;
     $1166 = HEAP32[$1165>>2]|0;
     $1167 = HEAP32[$89>>2]|0;
     $1168 = (($1166) + ($1167<<2)|0);
     HEAPF32[$1168>>2] = $1163;
     $1169 = HEAP32[$290>>2]|0;
     $1170 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($1169)|0);
     $1171 = +HEAPF32[$1170>>2];
     $1172 = HEAP32[$289>>2]|0;
     HEAP32[$90>>2] = $283;
     HEAP32[$91>>2] = $1172;
     $1173 = HEAP32[$90>>2]|0;
     $1174 = HEAP32[$1173>>2]|0;
     $1175 = HEAP32[$91>>2]|0;
     $1176 = (($1174) + ($1175<<2)|0);
     HEAPF32[$1176>>2] = $1171;
     $1177 = HEAP32[$290>>2]|0;
     $1178 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($1177)|0);
     $1179 = +HEAPF32[$1178>>2];
     $1180 = HEAP32[$289>>2]|0;
     HEAP32[$92>>2] = $284;
     HEAP32[$93>>2] = $1180;
     $1181 = HEAP32[$92>>2]|0;
     $1182 = HEAP32[$1181>>2]|0;
     $1183 = HEAP32[$93>>2]|0;
     $1184 = (($1182) + ($1183<<2)|0);
     HEAPF32[$1184>>2] = $1179;
     $1185 = HEAP32[$290>>2]|0;
     $1186 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($1185)|0);
     $1187 = +HEAPF32[$1186>>2];
     $1188 = HEAP32[$289>>2]|0;
     HEAP32[$94>>2] = $285;
     HEAP32[$95>>2] = $1188;
     $1189 = HEAP32[$94>>2]|0;
     $1190 = HEAP32[$1189>>2]|0;
     $1191 = HEAP32[$95>>2]|0;
     $1192 = (($1190) + ($1191<<2)|0);
     HEAPF32[$1192>>2] = $1187;
     $1193 = HEAP32[$290>>2]|0;
     $1194 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($1193)|0);
     $1195 = +HEAPF32[$1194>>2];
     $1196 = HEAP32[$289>>2]|0;
     HEAP32[$96>>2] = $286;
     HEAP32[$97>>2] = $1196;
     $1197 = HEAP32[$96>>2]|0;
     $1198 = HEAP32[$1197>>2]|0;
     $1199 = HEAP32[$97>>2]|0;
     $1200 = (($1198) + ($1199<<2)|0);
     HEAPF32[$1200>>2] = $1195;
     $1201 = HEAP32[$290>>2]|0;
     $1202 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($1201)|0);
     $1203 = +HEAPF32[$1202>>2];
     $1204 = HEAP32[$289>>2]|0;
     HEAP32[$98>>2] = $287;
     HEAP32[$99>>2] = $1204;
     $1205 = HEAP32[$98>>2]|0;
     $1206 = HEAP32[$1205>>2]|0;
     $1207 = HEAP32[$99>>2]|0;
     $1208 = (($1206) + ($1207<<2)|0);
     HEAPF32[$1208>>2] = $1203;
     $1209 = HEAP32[$290>>2]|0;
     $1210 = (__ZNK3sap2ve4core18ScalarAnimationKey8GetValueEv($1209)|0);
     $1211 = +HEAPF32[$1210>>2];
     $1212 = HEAP32[$289>>2]|0;
     HEAP32[$100>>2] = $288;
     HEAP32[$101>>2] = $1212;
     $1213 = HEAP32[$100>>2]|0;
     $1214 = HEAP32[$1213>>2]|0;
     $1215 = HEAP32[$101>>2]|0;
     $1216 = (($1214) + ($1215<<2)|0);
     HEAPF32[$1216>>2] = $1211;
     $1217 = HEAP32[$289>>2]|0;
     $1218 = (($1217) + 1)|0;
     HEAP32[$289>>2] = $1218;
    }
    $1219 = ((($306)) + 24|0);
    $1220 = HEAP32[$1219>>2]|0;
    $1221 = HEAP32[$226>>2]|0;
    $1222 = HEAP32[$279>>2]|0;
    $1223 = HEAP32[$279>>2]|0;
    $1224 = (__ZNK3sap2ve4core14AnimationTrack20GetInterpolationTypeEv($1223)|0);
    $1225 = HEAP32[$279>>2]|0;
    $1226 = (__ZNK3sap2ve4core14AnimationTrack14GetCyclicStartEv($1225)|0);
    $1227 = HEAP32[$279>>2]|0;
    $1228 = (__ZNK3sap2ve4core14AnimationTrack12GetCyclicEndEv($1227)|0);
    $1229 = HEAP32[$280>>2]|0;
    $1230 = HEAP32[$281>>2]|0;
    HEAP32[$105>>2] = $282;
    $1231 = HEAP32[$105>>2]|0;
    $1232 = HEAP32[$1231>>2]|0;
    HEAP32[$104>>2] = $1232;
    $1233 = HEAP32[$104>>2]|0;
    HEAP32[$107>>2] = $283;
    $1234 = HEAP32[$107>>2]|0;
    $1235 = HEAP32[$1234>>2]|0;
    HEAP32[$106>>2] = $1235;
    $1236 = HEAP32[$106>>2]|0;
    HEAP32[$109>>2] = $284;
    $1237 = HEAP32[$109>>2]|0;
    $1238 = HEAP32[$1237>>2]|0;
    HEAP32[$108>>2] = $1238;
    $1239 = HEAP32[$108>>2]|0;
    HEAP32[$111>>2] = $285;
    $1240 = HEAP32[$111>>2]|0;
    $1241 = HEAP32[$1240>>2]|0;
    HEAP32[$110>>2] = $1241;
    $1242 = HEAP32[$110>>2]|0;
    HEAP32[$113>>2] = $286;
    $1243 = HEAP32[$113>>2]|0;
    $1244 = HEAP32[$1243>>2]|0;
    HEAP32[$112>>2] = $1244;
    $1245 = HEAP32[$112>>2]|0;
    HEAP32[$115>>2] = $287;
    $1246 = HEAP32[$115>>2]|0;
    $1247 = HEAP32[$1246>>2]|0;
    HEAP32[$114>>2] = $1247;
    $1248 = HEAP32[$114>>2]|0;
    HEAP32[$117>>2] = $288;
    $1249 = HEAP32[$117>>2]|0;
    $1250 = HEAP32[$1249>>2]|0;
    HEAP32[$116>>2] = $1250;
    $1251 = HEAP32[$116>>2]|0;
    _SceneBuilderAdapter_insertAnimationTrack(($1220|0),($1221|0),($1222|0),0,0,($1224|0),($1226|0),($1228|0),($1229|0),($1230|0),($1233|0),($1236|0),($1239|0),($1242|0),($1245|0),($1248|0),($1251|0));
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($288);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($287);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($286);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($285);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($284);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($283);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($282);
   }
   break;
  }
  case 1:  {
   $1252 = HEAP32[$226>>2]|0;
   HEAP32[$291>>2] = $1252;
   HEAP32[$292>>2] = 0;
   while(1) {
    $1253 = HEAP32[$292>>2]|0;
    $1254 = HEAP32[$291>>2]|0;
    $1255 = (__ZNK3sap2ve4core17MaterialAnimation14GetTargetCountEv($1254)|0);
    $1256 = ($1253>>>0)<($1255>>>0);
    $1257 = HEAP32[$291>>2]|0;
    if (!($1256)) {
     break;
    }
    $1258 = HEAP32[$292>>2]|0;
    $1259 = (__ZNK3sap2ve4core17MaterialAnimation9GetTargetEj($1257,$1258)|0);
    HEAP32[$293>>2] = $1259;
    $1260 = ((($306)) + 24|0);
    $1261 = HEAP32[$1260>>2]|0;
    $1262 = HEAP32[$226>>2]|0;
    $1263 = HEAP32[$293>>2]|0;
    _SceneBuilderAdapter_insertAnimationTarget(($1261|0),($1262|0),($1263|0),0.0,0.0,0.0);
    $1264 = HEAP32[$292>>2]|0;
    $1265 = (($1264) + 1)|0;
    HEAP32[$292>>2] = $1265;
   }
   $1266 = (__ZNK3sap2ve4core17MaterialAnimation19GetOpacityAnimationEv($1257)|0);
   HEAP32[$294>>2] = $1266;
   $1267 = HEAP32[$294>>2]|0;
   $1268 = ($1267|0)!=(0|0);
   if ($1268) {
    $1269 = HEAP32[$294>>2]|0;
    $1270 = (__ZNK3sap2ve4core14AnimationTrack11GetKeyCountEv($1269)|0);
    HEAP32[$295>>2] = $1270;
    $1271 = HEAP32[$295>>2]|0;
    HEAP32[$296>>2] = $1271;
    $1272 = HEAP32[$295>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($297,$1272);
    $1273 = HEAP32[$295>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($298,$1273);
    $1274 = HEAP32[$295>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($299,$1274);
    $1275 = HEAP32[$295>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($300,$1275);
    $1276 = HEAP32[$295>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($301,$1276);
    $1277 = HEAP32[$295>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($302,$1277);
    $1278 = HEAP32[$296>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($303,$1278);
    HEAP32[$304>>2] = 0;
    while(1) {
     $1279 = HEAP32[$304>>2]|0;
     $1280 = HEAP32[$295>>2]|0;
     $1281 = ($1279|0)<($1280|0);
     if (!($1281)) {
      break;
     }
     $1282 = HEAP32[$294>>2]|0;
     $1283 = HEAP32[$304>>2]|0;
     $1284 = (__ZN3sap2ve4core20ScalarAnimationTrack6GetKeyEj($1282,$1283)|0);
     HEAP32[$305>>2] = $1284;
     $1285 = HEAP32[$305>>2]|0;
     $1286 = (__ZNK3sap2ve4core12AnimationKey7GetTimeEv($1285)|0);
     $1287 = +HEAPF32[$1286>>2];
     $1288 = HEAP32[$304>>2]|0;
     HEAP32[$158>>2] = $297;
     HEAP32[$159>>2] = $1288;
     $1289 = HEAP32[$158>>2]|0;
     $1290 = HEAP32[$1289>>2]|0;
     $1291 = HEAP32[$159>>2]|0;
     $1292 = (($1290) + ($1291<<2)|0);
     HEAPF32[$1292>>2] = $1287;
     $1293 = HEAP32[$305>>2]|0;
     $1294 = (__ZNK3sap2ve4core12AnimationKey10GetTensionEv($1293)|0);
     $1295 = +HEAPF32[$1294>>2];
     $1296 = HEAP32[$304>>2]|0;
     HEAP32[$160>>2] = $298;
     HEAP32[$161>>2] = $1296;
     $1297 = HEAP32[$160>>2]|0;
     $1298 = HEAP32[$1297>>2]|0;
     $1299 = HEAP32[$161>>2]|0;
     $1300 = (($1298) + ($1299<<2)|0);
     HEAPF32[$1300>>2] = $1295;
     $1301 = HEAP32[$305>>2]|0;
     $1302 = (__ZNK3sap2ve4core12AnimationKey13GetContinuityEv($1301)|0);
     $1303 = +HEAPF32[$1302>>2];
     $1304 = HEAP32[$304>>2]|0;
     HEAP32[$162>>2] = $299;
     HEAP32[$163>>2] = $1304;
     $1305 = HEAP32[$162>>2]|0;
     $1306 = HEAP32[$1305>>2]|0;
     $1307 = HEAP32[$163>>2]|0;
     $1308 = (($1306) + ($1307<<2)|0);
     HEAPF32[$1308>>2] = $1303;
     $1309 = HEAP32[$305>>2]|0;
     $1310 = (__ZNK3sap2ve4core12AnimationKey7GetBiasEv($1309)|0);
     $1311 = +HEAPF32[$1310>>2];
     $1312 = HEAP32[$304>>2]|0;
     HEAP32[$164>>2] = $300;
     HEAP32[$165>>2] = $1312;
     $1313 = HEAP32[$164>>2]|0;
     $1314 = HEAP32[$1313>>2]|0;
     $1315 = HEAP32[$165>>2]|0;
     $1316 = (($1314) + ($1315<<2)|0);
     HEAPF32[$1316>>2] = $1311;
     $1317 = HEAP32[$305>>2]|0;
     $1318 = (__ZNK3sap2ve4core12AnimationKey9GetEaseInEv($1317)|0);
     $1319 = +HEAPF32[$1318>>2];
     $1320 = HEAP32[$304>>2]|0;
     HEAP32[$166>>2] = $301;
     HEAP32[$167>>2] = $1320;
     $1321 = HEAP32[$166>>2]|0;
     $1322 = HEAP32[$1321>>2]|0;
     $1323 = HEAP32[$167>>2]|0;
     $1324 = (($1322) + ($1323<<2)|0);
     HEAPF32[$1324>>2] = $1319;
     $1325 = HEAP32[$305>>2]|0;
     $1326 = (__ZNK3sap2ve4core12AnimationKey10GetEaseOutEv($1325)|0);
     $1327 = +HEAPF32[$1326>>2];
     $1328 = HEAP32[$304>>2]|0;
     HEAP32[$168>>2] = $302;
     HEAP32[$169>>2] = $1328;
     $1329 = HEAP32[$168>>2]|0;
     $1330 = HEAP32[$1329>>2]|0;
     $1331 = HEAP32[$169>>2]|0;
     $1332 = (($1330) + ($1331<<2)|0);
     HEAPF32[$1332>>2] = $1327;
     $1333 = HEAP32[$305>>2]|0;
     $1334 = (__ZNK3sap2ve4core18ScalarAnimationKey8GetValueEv($1333)|0);
     $1335 = +HEAPF32[$1334>>2];
     $1336 = HEAP32[$304>>2]|0;
     HEAP32[$170>>2] = $303;
     HEAP32[$171>>2] = $1336;
     $1337 = HEAP32[$170>>2]|0;
     $1338 = HEAP32[$1337>>2]|0;
     $1339 = HEAP32[$171>>2]|0;
     $1340 = (($1338) + ($1339<<2)|0);
     HEAPF32[$1340>>2] = $1335;
     $1341 = HEAP32[$304>>2]|0;
     $1342 = (($1341) + 1)|0;
     HEAP32[$304>>2] = $1342;
    }
    $1343 = ((($306)) + 24|0);
    $1344 = HEAP32[$1343>>2]|0;
    $1345 = HEAP32[$226>>2]|0;
    $1346 = HEAP32[$294>>2]|0;
    $1347 = HEAP32[$294>>2]|0;
    $1348 = (__ZNK3sap2ve4core14AnimationTrack20GetInterpolationTypeEv($1347)|0);
    $1349 = HEAP32[$294>>2]|0;
    $1350 = (__ZNK3sap2ve4core14AnimationTrack14GetCyclicStartEv($1349)|0);
    $1351 = HEAP32[$294>>2]|0;
    $1352 = (__ZNK3sap2ve4core14AnimationTrack12GetCyclicEndEv($1351)|0);
    $1353 = HEAP32[$295>>2]|0;
    $1354 = HEAP32[$296>>2]|0;
    HEAP32[$175>>2] = $297;
    $1355 = HEAP32[$175>>2]|0;
    $1356 = HEAP32[$1355>>2]|0;
    HEAP32[$174>>2] = $1356;
    $1357 = HEAP32[$174>>2]|0;
    HEAP32[$177>>2] = $298;
    $1358 = HEAP32[$177>>2]|0;
    $1359 = HEAP32[$1358>>2]|0;
    HEAP32[$176>>2] = $1359;
    $1360 = HEAP32[$176>>2]|0;
    HEAP32[$179>>2] = $299;
    $1361 = HEAP32[$179>>2]|0;
    $1362 = HEAP32[$1361>>2]|0;
    HEAP32[$178>>2] = $1362;
    $1363 = HEAP32[$178>>2]|0;
    HEAP32[$181>>2] = $300;
    $1364 = HEAP32[$181>>2]|0;
    $1365 = HEAP32[$1364>>2]|0;
    HEAP32[$180>>2] = $1365;
    $1366 = HEAP32[$180>>2]|0;
    HEAP32[$183>>2] = $301;
    $1367 = HEAP32[$183>>2]|0;
    $1368 = HEAP32[$1367>>2]|0;
    HEAP32[$182>>2] = $1368;
    $1369 = HEAP32[$182>>2]|0;
    HEAP32[$185>>2] = $302;
    $1370 = HEAP32[$185>>2]|0;
    $1371 = HEAP32[$1370>>2]|0;
    HEAP32[$184>>2] = $1371;
    $1372 = HEAP32[$184>>2]|0;
    HEAP32[$187>>2] = $303;
    $1373 = HEAP32[$187>>2]|0;
    $1374 = HEAP32[$1373>>2]|0;
    HEAP32[$186>>2] = $1374;
    $1375 = HEAP32[$186>>2]|0;
    _SceneBuilderAdapter_insertAnimationTrack(($1344|0),($1345|0),($1346|0),0,0,($1348|0),($1350|0),($1352|0),($1353|0),($1354|0),($1357|0),($1360|0),($1363|0),($1366|0),($1369|0),($1372|0),($1375|0));
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($303);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($302);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($301);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($300);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($299);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($298);
    __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($297);
   }
   break;
  }
  default: {
  }
  }
  $1376 = ((($306)) + 24|0);
  $1377 = HEAP32[$1376>>2]|0;
  $1378 = HEAP32[$226>>2]|0;
  _SceneBuilderAdapter_finalizeAnimation(($1377|0),($1378|0));
  $1379 = HEAP32[$225>>2]|0;
  $1380 = (($1379) + 1)|0;
  HEAP32[$225>>2] = $1380;
 }
 STACKTOP = sp;return;
}
function __ZNK17DataModelImporter35GetNodesArrayFromSelectorCollectionERKN3sap2ve4core18SelectorCollectionE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$expand_i1_val = 0, $$expand_i1_val2 = 0, $$pre_trunc = 0, $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0;
 var $115 = 0, $116 = 0, $117 = 0, $118 = 0, $119 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0;
 var $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0;
 var $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0;
 var $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0;
 var label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 192|0;
 $3 = sp + 176|0;
 $4 = sp + 172|0;
 $5 = sp + 168|0;
 $6 = sp + 164|0;
 $7 = sp + 160|0;
 $8 = sp + 156|0;
 $9 = sp + 152|0;
 $10 = sp + 148|0;
 $11 = sp + 144|0;
 $12 = sp + 140|0;
 $13 = sp + 136|0;
 $14 = sp + 132|0;
 $15 = sp + 128|0;
 $16 = sp;
 $17 = sp + 124|0;
 $18 = sp + 120|0;
 $19 = sp + 116|0;
 $20 = sp + 112|0;
 $21 = sp + 182|0;
 $22 = sp + 108|0;
 $23 = sp + 104|0;
 $24 = sp + 100|0;
 $25 = sp + 96|0;
 $26 = sp + 92|0;
 $27 = sp + 88|0;
 $28 = sp + 84|0;
 $29 = sp + 80|0;
 $30 = sp + 76|0;
 $31 = sp + 72|0;
 $32 = sp + 181|0;
 $33 = sp + 68|0;
 $34 = sp + 64|0;
 $35 = sp + 60|0;
 $36 = sp + 56|0;
 $37 = sp + 52|0;
 $38 = sp + 48|0;
 $39 = sp + 44|0;
 $40 = sp + 40|0;
 $41 = sp + 36|0;
 $42 = sp + 32|0;
 $43 = sp + 28|0;
 $44 = sp + 24|0;
 $45 = sp + 20|0;
 $46 = sp + 16|0;
 $47 = sp + 180|0;
 $48 = sp + 12|0;
 $49 = sp + 8|0;
 $50 = sp + 4|0;
 HEAP32[$45>>2] = $1;
 HEAP32[$46>>2] = $2;
 $51 = HEAP32[$45>>2]|0;
 $$expand_i1_val = 0;
 HEAP8[$47>>0] = $$expand_i1_val;
 HEAP32[$44>>2] = $0;
 $52 = HEAP32[$44>>2]|0;
 HEAP32[$42>>2] = $52;
 $53 = HEAP32[$42>>2]|0;
 HEAP32[$41>>2] = $53;
 HEAP32[$53>>2] = 0;
 $54 = ((($53)) + 4|0);
 HEAP32[$54>>2] = 0;
 $55 = ((($53)) + 8|0);
 HEAP32[$43>>2] = 0;
 HEAP32[$39>>2] = $55;
 HEAP32[$40>>2] = $43;
 $56 = HEAP32[$39>>2]|0;
 $57 = HEAP32[$40>>2]|0;
 HEAP32[$38>>2] = $57;
 $58 = HEAP32[$38>>2]|0;
 HEAP32[$34>>2] = $56;
 HEAP32[$35>>2] = $58;
 $59 = HEAP32[$34>>2]|0;
 $60 = HEAP32[$35>>2]|0;
 HEAP32[$33>>2] = $60;
 HEAP32[$59>>2] = 0;
 HEAP32[$37>>2] = $56;
 $61 = HEAP32[$37>>2]|0;
 HEAP32[$36>>2] = $61;
 $62 = HEAP32[$46>>2]|0;
 $63 = (__ZNK3sap2ve4core18SelectorCollection5CountEv($62)|0);
 HEAP32[$48>>2] = $63;
 $64 = HEAP32[$48>>2]|0;
 $65 = ($64|0)!=(0);
 L1: do {
  if ($65) {
   $66 = HEAP32[$48>>2]|0;
   __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEE7reserveEm($0,$66);
   HEAP32[$49>>2] = 0;
   while(1) {
    $67 = HEAP32[$49>>2]|0;
    $68 = HEAP32[$48>>2]|0;
    $69 = ($67>>>0)<($68>>>0);
    if (!($69)) {
     break L1;
    }
    $70 = ((($51)) + 4|0);
    $71 = (__ZNK3sap2ve4core4IPtrINS1_6ISceneEEptEv($70)|0);
    $72 = HEAP32[$71>>2]|0;
    $73 = ((($72)) + 44|0);
    $74 = HEAP32[$73>>2]|0;
    $75 = HEAP32[$46>>2]|0;
    $76 = HEAP32[$49>>2]|0;
    $77 = (__ZNK3sap2ve4core18SelectorCollection3GetEj($75,$76)|0);
    $78 = (__ZNK3sap2ve4core22SelectorCollectionItem11GetSelectorEv($77)|0);
    $79 = (FUNCTION_TABLE_iii[$74 & 1023]($71,$78)|0);
    HEAP32[$50>>2] = $79;
    $80 = HEAP32[$50>>2]|0;
    $81 = ($80|0)!=(0|0);
    do {
     if ($81) {
      HEAP32[$30>>2] = $0;
      HEAP32[$31>>2] = $50;
      $82 = HEAP32[$30>>2]|0;
      $83 = ((($82)) + 4|0);
      $84 = HEAP32[$83>>2]|0;
      HEAP32[$29>>2] = $82;
      $85 = HEAP32[$29>>2]|0;
      $86 = ((($85)) + 8|0);
      HEAP32[$28>>2] = $86;
      $87 = HEAP32[$28>>2]|0;
      HEAP32[$27>>2] = $87;
      $88 = HEAP32[$27>>2]|0;
      $89 = HEAP32[$88>>2]|0;
      $90 = ($84>>>0)<($89>>>0);
      if ($90) {
       HEAP32[$24>>2] = $32;
       HEAP32[$25>>2] = $82;
       HEAP32[$26>>2] = 1;
       HEAP32[$5>>2] = $82;
       $91 = HEAP32[$5>>2]|0;
       $92 = ((($91)) + 8|0);
       HEAP32[$4>>2] = $92;
       $93 = HEAP32[$4>>2]|0;
       HEAP32[$3>>2] = $93;
       $94 = HEAP32[$3>>2]|0;
       $95 = ((($82)) + 4|0);
       $96 = HEAP32[$95>>2]|0;
       HEAP32[$6>>2] = $96;
       $97 = HEAP32[$6>>2]|0;
       $98 = HEAP32[$31>>2]|0;
       HEAP32[$7>>2] = $98;
       $99 = HEAP32[$7>>2]|0;
       HEAP32[$18>>2] = $94;
       HEAP32[$19>>2] = $97;
       HEAP32[$20>>2] = $99;
       $100 = HEAP32[$18>>2]|0;
       $101 = HEAP32[$19>>2]|0;
       $102 = HEAP32[$20>>2]|0;
       HEAP32[$17>>2] = $102;
       $103 = HEAP32[$17>>2]|0;
       ;HEAP8[$16>>0]=HEAP8[$21>>0]|0;
       HEAP32[$13>>2] = $100;
       HEAP32[$14>>2] = $101;
       HEAP32[$15>>2] = $103;
       $104 = HEAP32[$13>>2]|0;
       $105 = HEAP32[$14>>2]|0;
       $106 = HEAP32[$15>>2]|0;
       HEAP32[$12>>2] = $106;
       $107 = HEAP32[$12>>2]|0;
       HEAP32[$9>>2] = $104;
       HEAP32[$10>>2] = $105;
       HEAP32[$11>>2] = $107;
       $108 = HEAP32[$10>>2]|0;
       $109 = HEAP32[$11>>2]|0;
       HEAP32[$8>>2] = $109;
       $110 = HEAP32[$8>>2]|0;
       $111 = HEAP32[$110>>2]|0;
       HEAP32[$108>>2] = $111;
       HEAP32[$22>>2] = $32;
       $112 = ((($82)) + 4|0);
       $113 = HEAP32[$112>>2]|0;
       $114 = ((($113)) + 4|0);
       HEAP32[$112>>2] = $114;
       break;
      } else {
       $115 = HEAP32[$31>>2]|0;
       HEAP32[$23>>2] = $115;
       $116 = HEAP32[$23>>2]|0;
       __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEE24__emplace_back_slow_pathIJRPKN3sap2ve4core5INodeEEEEvDpOT_($82,$116);
       break;
      }
     }
    } while(0);
    $117 = HEAP32[$49>>2]|0;
    $118 = (($117) + 1)|0;
    HEAP32[$49>>2] = $118;
   }
  }
 } while(0);
 $$expand_i1_val2 = 1;
 HEAP8[$47>>0] = $$expand_i1_val2;
 $$pre_trunc = HEAP8[$47>>0]|0;
 $119 = $$pre_trunc&1;
 if ($119) {
  STACKTOP = sp;return;
 }
 __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($0);
 STACKTOP = sp;return;
}
function __ZL14GetColorARGB32RKN3sap2ve4core6ColourE($0) {
 $0 = $0|0;
 var $1 = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0.0, $9 = 0.0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 $3 = (__ZL8GetColorRKN3sap2ve4core6ColourE($2)|0);
 $4 = HEAP32[$1>>2]|0;
 $5 = (__ZNK3sap2ve4core6Colour7IsValidEv($4)|0);
 if (!($5)) {
  $13 = 0;
  $12 = $3 | $13;
  STACKTOP = sp;return ($12|0);
 }
 $6 = HEAP32[$1>>2]|0;
 $7 = (__ZNK3sap2ve4core6Colour5AlphaEv($6)|0);
 $8 = +HEAPF32[$7>>2];
 $9 = $8 * 255.0;
 $10 = (~~(($9))>>>0);
 $11 = $10 << 24;
 $13 = $11;
 $12 = $3 | $13;
 STACKTOP = sp;return ($12|0);
}
function __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 __ZNSt3__213__vector_baseIPKvNS_9allocatorIS2_EEED2Ev($2);
 STACKTOP = sp;return;
}
function __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEE7reserveEm($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $5 = 0, $6 = 0, $7 = 0;
 var $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 80|0;
 $2 = sp + 64|0;
 $3 = sp + 60|0;
 $4 = sp + 56|0;
 $5 = sp + 52|0;
 $6 = sp + 48|0;
 $7 = sp + 44|0;
 $8 = sp + 40|0;
 $9 = sp + 36|0;
 $10 = sp + 32|0;
 $11 = sp + 28|0;
 $12 = sp + 24|0;
 $13 = sp + 20|0;
 $14 = sp;
 HEAP32[$11>>2] = $0;
 HEAP32[$12>>2] = $1;
 $15 = HEAP32[$11>>2]|0;
 $16 = HEAP32[$12>>2]|0;
 HEAP32[$10>>2] = $15;
 $17 = HEAP32[$10>>2]|0;
 HEAP32[$9>>2] = $17;
 $18 = HEAP32[$9>>2]|0;
 HEAP32[$8>>2] = $18;
 $19 = HEAP32[$8>>2]|0;
 $20 = ((($19)) + 8|0);
 HEAP32[$7>>2] = $20;
 $21 = HEAP32[$7>>2]|0;
 HEAP32[$6>>2] = $21;
 $22 = HEAP32[$6>>2]|0;
 $23 = HEAP32[$22>>2]|0;
 $24 = HEAP32[$18>>2]|0;
 $25 = $23;
 $26 = $24;
 $27 = (($25) - ($26))|0;
 $28 = (($27|0) / 4)&-1;
 $29 = ($16>>>0)>($28>>>0);
 if (!($29)) {
  STACKTOP = sp;return;
 }
 HEAP32[$5>>2] = $15;
 $30 = HEAP32[$5>>2]|0;
 $31 = ((($30)) + 8|0);
 HEAP32[$4>>2] = $31;
 $32 = HEAP32[$4>>2]|0;
 HEAP32[$3>>2] = $32;
 $33 = HEAP32[$3>>2]|0;
 HEAP32[$13>>2] = $33;
 $34 = HEAP32[$12>>2]|0;
 HEAP32[$2>>2] = $15;
 $35 = HEAP32[$2>>2]|0;
 $36 = ((($35)) + 4|0);
 $37 = HEAP32[$36>>2]|0;
 $38 = HEAP32[$35>>2]|0;
 $39 = $37;
 $40 = $38;
 $41 = (($39) - ($40))|0;
 $42 = (($41|0) / 4)&-1;
 $43 = HEAP32[$13>>2]|0;
 __ZNSt3__214__split_bufferIPKvRNS_9allocatorIS2_EEEC2EmmS5_($14,$34,$42,$43);
 __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEE26__swap_out_circular_bufferERNS_14__split_bufferIS2_RS4_EE($15,$14);
 __ZNSt3__214__split_bufferIPKvRNS_9allocatorIS2_EEED2Ev($14);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter16InsertDetailViewEPKN3sap2ve4core10DetailViewEPKNS2_5INodeE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0.0, $15 = 0, $150 = 0.0, $151 = 0.0, $152 = 0.0, $153 = 0.0;
 var $154 = 0, $155 = 0, $156 = 0, $157 = 0.0, $158 = 0.0, $159 = 0.0, $16 = 0, $160 = 0, $161 = 0.0, $162 = 0.0, $163 = 0, $164 = 0, $165 = 0, $166 = 0, $167 = 0, $168 = 0, $169 = 0, $17 = 0, $170 = 0, $171 = 0;
 var $172 = 0, $173 = 0, $174 = 0, $175 = 0, $176 = 0, $177 = 0, $178 = 0, $179 = 0, $18 = 0, $180 = 0, $181 = 0, $182 = 0, $183 = 0, $184 = 0, $185 = 0, $186 = 0, $187 = 0, $188 = 0, $189 = 0, $19 = 0;
 var $190 = 0, $191 = 0, $192 = 0, $193 = 0, $194 = 0, $195 = 0, $196 = 0, $197 = 0, $198 = 0, $199 = 0, $20 = 0, $200 = 0, $201 = 0, $202 = 0, $203 = 0, $204 = 0, $205 = 0, $206 = 0, $207 = 0, $208 = 0;
 var $209 = 0, $21 = 0, $210 = 0, $211 = 0, $212 = 0, $213 = 0, $214 = 0, $215 = 0, $216 = 0, $217 = 0, $218 = 0, $219 = 0, $22 = 0, $220 = 0, $221 = 0, $222 = 0, $223 = 0, $224 = 0, $225 = 0.0, $226 = 0;
 var $227 = 0, $228 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0, $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0;
 var $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0, $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0;
 var $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0, $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0;
 var $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0, $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0;
 var $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0;
 var $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0;
 var $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0;
 var $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0;
 var $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 496|0;
 $3 = sp + 484|0;
 $4 = sp + 480|0;
 $5 = sp + 476|0;
 $6 = sp + 472|0;
 $7 = sp + 468|0;
 $8 = sp + 464|0;
 $9 = sp + 460|0;
 $10 = sp + 456|0;
 $11 = sp + 452|0;
 $12 = sp + 448|0;
 $13 = sp + 444|0;
 $14 = sp + 440|0;
 $15 = sp + 436|0;
 $16 = sp + 432|0;
 $17 = sp + 428|0;
 $18 = sp + 424|0;
 $19 = sp + 420|0;
 $20 = sp + 416|0;
 $21 = sp + 412|0;
 $22 = sp + 408|0;
 $23 = sp + 404|0;
 $24 = sp + 400|0;
 $25 = sp + 396|0;
 $26 = sp + 392|0;
 $27 = sp + 388|0;
 $28 = sp + 384|0;
 $29 = sp + 380|0;
 $30 = sp + 376|0;
 $31 = sp + 372|0;
 $32 = sp + 368|0;
 $33 = sp + 364|0;
 $34 = sp + 360|0;
 $35 = sp + 356|0;
 $36 = sp + 352|0;
 $37 = sp + 348|0;
 $38 = sp + 344|0;
 $39 = sp + 340|0;
 $40 = sp + 336|0;
 $41 = sp + 332|0;
 $42 = sp + 328|0;
 $43 = sp + 324|0;
 $44 = sp + 320|0;
 $45 = sp + 316|0;
 $46 = sp + 312|0;
 $47 = sp + 308|0;
 $48 = sp + 304|0;
 $49 = sp + 300|0;
 $50 = sp + 296|0;
 $51 = sp + 292|0;
 $52 = sp + 288|0;
 $53 = sp + 284|0;
 $54 = sp + 280|0;
 $55 = sp + 276|0;
 $56 = sp + 272|0;
 $57 = sp + 268|0;
 $58 = sp + 264|0;
 $59 = sp + 260|0;
 $60 = sp + 256|0;
 $61 = sp + 252|0;
 $62 = sp + 248|0;
 $63 = sp + 244|0;
 $64 = sp + 240|0;
 $65 = sp + 236|0;
 $66 = sp + 232|0;
 $67 = sp + 228|0;
 $68 = sp + 224|0;
 $69 = sp + 220|0;
 $70 = sp + 216|0;
 $71 = sp + 212|0;
 $72 = sp + 208|0;
 $73 = sp + 204|0;
 $74 = sp + 200|0;
 $75 = sp + 196|0;
 $76 = sp + 192|0;
 $77 = sp + 188|0;
 $78 = sp + 184|0;
 $79 = sp + 180|0;
 $80 = sp + 176|0;
 $81 = sp + 172|0;
 $82 = sp + 168|0;
 $83 = sp + 164|0;
 $84 = sp + 160|0;
 $85 = sp + 156|0;
 $86 = sp + 152|0;
 $87 = sp + 148|0;
 $88 = sp + 144|0;
 $89 = sp + 140|0;
 $90 = sp + 136|0;
 $91 = sp + 132|0;
 $92 = sp + 128|0;
 $93 = sp + 116|0;
 $94 = sp + 104|0;
 $95 = sp + 92|0;
 $96 = sp + 80|0;
 $97 = sp + 72|0;
 $98 = sp + 64|0;
 $99 = sp + 56|0;
 $100 = sp + 48|0;
 $101 = sp + 32|0;
 $102 = sp + 16|0;
 $103 = sp;
 HEAP32[$89>>2] = $0;
 HEAP32[$90>>2] = $1;
 HEAP32[$91>>2] = $2;
 $104 = HEAP32[$89>>2]|0;
 $105 = HEAP32[$91>>2]|0;
 $106 = ((($104)) + 128|0);
 $107 = (__ZNSt3__23mapIPKN3sap2ve4core10DetailViewEPKNS3_5INodeENS_4lessIS6_EENS_9allocatorINS_4pairIKS6_S9_EEEEEixERSE_($106,$90)|0);
 HEAP32[$107>>2] = $105;
 $108 = HEAP32[$90>>2]|0;
 $109 = (__ZNK3sap2ve4core10DetailView9GetCameraEv($108)|0);
 HEAP32[$92>>2] = $109;
 $110 = HEAP32[$92>>2]|0;
 (__ZN17DataModelImporter17GetOrCreateCameraEPKN3sap2ve4core6CameraEPKNS1_6MatrixE($104,$110,0)|0);
 HEAP32[$88>>2] = $93;
 $111 = HEAP32[$88>>2]|0;
 HEAP32[$87>>2] = $111;
 $112 = HEAP32[$87>>2]|0;
 HEAP32[$86>>2] = $112;
 $113 = HEAP32[$86>>2]|0;
 ;HEAP32[$113>>2]=0|0;HEAP32[$113+4>>2]=0|0;HEAP32[$113+8>>2]=0|0;
 HEAP32[$85>>2] = $112;
 $114 = HEAP32[$85>>2]|0;
 HEAP32[$84>>2] = $114;
 HEAP32[$81>>2] = $111;
 $115 = HEAP32[$81>>2]|0;
 HEAP32[$80>>2] = $115;
 $116 = HEAP32[$80>>2]|0;
 HEAP32[$79>>2] = $116;
 $117 = HEAP32[$79>>2]|0;
 HEAP32[$82>>2] = $117;
 HEAP32[$83>>2] = 0;
 while(1) {
  $118 = HEAP32[$83>>2]|0;
  $119 = ($118>>>0)<(3);
  if (!($119)) {
   break;
  }
  $120 = HEAP32[$82>>2]|0;
  $121 = HEAP32[$83>>2]|0;
  $122 = (($120) + ($121<<2)|0);
  HEAP32[$122>>2] = 0;
  $123 = HEAP32[$83>>2]|0;
  $124 = (($123) + 1)|0;
  HEAP32[$83>>2] = $124;
 }
 HEAP32[$78>>2] = $94;
 $125 = HEAP32[$78>>2]|0;
 HEAP32[$77>>2] = $125;
 $126 = HEAP32[$77>>2]|0;
 HEAP32[$76>>2] = $126;
 $127 = HEAP32[$76>>2]|0;
 ;HEAP32[$127>>2]=0|0;HEAP32[$127+4>>2]=0|0;HEAP32[$127+8>>2]=0|0;
 HEAP32[$75>>2] = $126;
 $128 = HEAP32[$75>>2]|0;
 HEAP32[$74>>2] = $128;
 HEAP32[$71>>2] = $125;
 $129 = HEAP32[$71>>2]|0;
 HEAP32[$70>>2] = $129;
 $130 = HEAP32[$70>>2]|0;
 HEAP32[$69>>2] = $130;
 $131 = HEAP32[$69>>2]|0;
 HEAP32[$72>>2] = $131;
 HEAP32[$73>>2] = 0;
 while(1) {
  $132 = HEAP32[$73>>2]|0;
  $133 = ($132>>>0)<(3);
  if (!($133)) {
   break;
  }
  $134 = HEAP32[$72>>2]|0;
  $135 = HEAP32[$73>>2]|0;
  $136 = (($134) + ($135<<2)|0);
  HEAP32[$136>>2] = 0;
  $137 = HEAP32[$73>>2]|0;
  $138 = (($137) + 1)|0;
  HEAP32[$73>>2] = $138;
 }
 $139 = HEAP32[$90>>2]|0;
 __ZN17DataModelImporter19GetViewportMetadataEPKN3sap2ve4core18ViewportDefinitionERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEESD_($139,$93,$94);
 $140 = HEAP32[$90>>2]|0;
 $141 = HEAP32[$140>>2]|0;
 $142 = ((($141)) + 124|0);
 $143 = HEAP32[$142>>2]|0;
 $144 = (FUNCTION_TABLE_ii[$143 & 2047]($140)|0);
 __ZNK17DataModelImporter35GetNodesArrayFromSelectorCollectionERKN3sap2ve4core18SelectorCollectionE($95,$104,$144);
 $145 = HEAP32[$90>>2]|0;
 $146 = (__ZNK3sap2ve4core10DetailView12GetTargetSetEv($145)|0);
 __ZNK17DataModelImporter35GetNodesArrayFromSelectorCollectionERKN3sap2ve4core18SelectorCollectionE($96,$104,$146);
 $147 = HEAP32[$90>>2]|0;
 __ZNK3sap2ve4core10DetailView7GetSizeEv($97,$147);
 $148 = HEAP32[$90>>2]|0;
 __ZNK3sap2ve4core10DetailView9GetOriginEv($99,$148);
 $149 = +HEAPF32[$99>>2];
 $150 = $149 * 2.0;
 $151 = $150 - 1.0;
 $152 = +HEAPF32[$97>>2];
 $153 = $151 + $152;
 HEAPF32[$98>>2] = $153;
 $154 = ((($98)) + 4|0);
 $155 = HEAP32[$90>>2]|0;
 __ZNK3sap2ve4core10DetailView9GetOriginEv($100,$155);
 $156 = ((($100)) + 4|0);
 $157 = +HEAPF32[$156>>2];
 $158 = $157 * -2.0;
 $159 = $158 + 1.0;
 $160 = ((($97)) + 4|0);
 $161 = +HEAPF32[$160>>2];
 $162 = $159 - $161;
 HEAPF32[$154>>2] = $162;
 $163 = ((($104)) + 24|0);
 $164 = HEAP32[$163>>2]|0;
 $165 = HEAP32[$91>>2]|0;
 $166 = HEAP32[$90>>2]|0;
 $167 = HEAP32[$90>>2]|0;
 $168 = ((($104)) + 12|0);
 HEAP32[$68>>2] = $168;
 $169 = HEAP32[$68>>2]|0;
 HEAP32[$67>>2] = $169;
 $170 = HEAP32[$67>>2]|0;
 HEAP32[$66>>2] = $170;
 $171 = HEAP32[$66>>2]|0;
 HEAP32[$65>>2] = $171;
 $172 = HEAP32[$65>>2]|0;
 HEAP32[$64>>2] = $172;
 $173 = HEAP32[$64>>2]|0;
 HEAP32[$63>>2] = $173;
 $174 = HEAP32[$63>>2]|0;
 $175 = ((($174)) + 11|0);
 $176 = HEAP8[$175>>0]|0;
 $177 = $176&255;
 $178 = $177 & 128;
 $179 = ($178|0)!=(0);
 if ($179) {
  HEAP32[$57>>2] = $171;
  $180 = HEAP32[$57>>2]|0;
  HEAP32[$56>>2] = $180;
  $181 = HEAP32[$56>>2]|0;
  HEAP32[$55>>2] = $181;
  $182 = HEAP32[$55>>2]|0;
  $183 = HEAP32[$182>>2]|0;
  $189 = $183;
 } else {
  HEAP32[$62>>2] = $171;
  $184 = HEAP32[$62>>2]|0;
  HEAP32[$61>>2] = $184;
  $185 = HEAP32[$61>>2]|0;
  HEAP32[$60>>2] = $185;
  $186 = HEAP32[$60>>2]|0;
  HEAP32[$59>>2] = $186;
  $187 = HEAP32[$59>>2]|0;
  HEAP32[$58>>2] = $187;
  $188 = HEAP32[$58>>2]|0;
  $189 = $188;
 }
 HEAP32[$54>>2] = $189;
 $190 = HEAP32[$54>>2]|0;
 $191 = (__ZNK3sap2ve4core18ViewportDefinition7GetNameEPKc($167,$190)|0);
 $192 = (__ZNK3sap2ve6String4Utf8Ev($191)|0);
 $193 = HEAP32[$90>>2]|0;
 $194 = ((($104)) + 12|0);
 HEAP32[$53>>2] = $194;
 $195 = HEAP32[$53>>2]|0;
 HEAP32[$52>>2] = $195;
 $196 = HEAP32[$52>>2]|0;
 HEAP32[$51>>2] = $196;
 $197 = HEAP32[$51>>2]|0;
 HEAP32[$50>>2] = $197;
 $198 = HEAP32[$50>>2]|0;
 HEAP32[$49>>2] = $198;
 $199 = HEAP32[$49>>2]|0;
 HEAP32[$48>>2] = $199;
 $200 = HEAP32[$48>>2]|0;
 $201 = ((($200)) + 11|0);
 $202 = HEAP8[$201>>0]|0;
 $203 = $202&255;
 $204 = $203 & 128;
 $205 = ($204|0)!=(0);
 if ($205) {
  HEAP32[$42>>2] = $197;
  $206 = HEAP32[$42>>2]|0;
  HEAP32[$41>>2] = $206;
  $207 = HEAP32[$41>>2]|0;
  HEAP32[$40>>2] = $207;
  $208 = HEAP32[$40>>2]|0;
  $209 = HEAP32[$208>>2]|0;
  $215 = $209;
 } else {
  HEAP32[$47>>2] = $197;
  $210 = HEAP32[$47>>2]|0;
  HEAP32[$46>>2] = $210;
  $211 = HEAP32[$46>>2]|0;
  HEAP32[$45>>2] = $211;
  $212 = HEAP32[$45>>2]|0;
  HEAP32[$44>>2] = $212;
  $213 = HEAP32[$44>>2]|0;
  HEAP32[$43>>2] = $213;
  $214 = HEAP32[$43>>2]|0;
  $215 = $214;
 }
 HEAP32[$39>>2] = $215;
 $216 = HEAP32[$39>>2]|0;
 $217 = (__ZNK3sap2ve4core18ViewportDefinition7GetInfoEPKc($193,$216)|0);
 $218 = (__ZNK3sap2ve6String4Utf8Ev($217)|0);
 $219 = HEAP32[$92>>2]|0;
 $220 = HEAP32[$90>>2]|0;
 $221 = (__ZNK3sap2ve4core10DetailView7GetTypeEv($220)|0);
 $222 = HEAP32[$90>>2]|0;
 $223 = (__ZNK3sap2ve4core10DetailView8GetShapeEv($222)|0);
 $224 = HEAP32[$90>>2]|0;
 $225 = (+__ZNK3sap2ve4core10DetailView14GetBorderWidthEv($224));
 $226 = HEAP32[$90>>2]|0;
 __ZNK3sap2ve4core10DetailView19GetBackgroundColourEv($101,$226);
 $227 = (__ZL8GetColorRKN3sap2ve4core6ColourE($101)|0);
 $228 = HEAP32[$90>>2]|0;
 __ZNK3sap2ve4core10DetailView15GetBorderColourEv($102,$228);
 $229 = (__ZL8GetColorRKN3sap2ve4core6ColourE($102)|0);
 $230 = HEAP32[$90>>2]|0;
 $231 = (__ZNK3sap2ve4core10DetailView14GetRenderOrderEv($230)|0);
 $232 = (__ZNK3sap2ve8Vector2f4DataEv($97)|0);
 $233 = HEAP32[$90>>2]|0;
 $234 = (__ZNK3sap2ve4core10DetailView20IsAttachmentPointSetEv($233)|0);
 if ($234) {
  $235 = HEAP32[$90>>2]|0;
  __ZNK3sap2ve4core10DetailView18GetAttachmentPointEv($103,$235);
  $236 = (__ZN3sap2ve6Vector4DataEv($103)|0);
  $303 = $236;
 } else {
  $303 = 0;
 }
 HEAP32[$38>>2] = $93;
 $237 = HEAP32[$38>>2]|0;
 HEAP32[$37>>2] = $237;
 $238 = HEAP32[$37>>2]|0;
 HEAP32[$36>>2] = $238;
 $239 = HEAP32[$36>>2]|0;
 HEAP32[$35>>2] = $239;
 $240 = HEAP32[$35>>2]|0;
 HEAP32[$34>>2] = $240;
 $241 = HEAP32[$34>>2]|0;
 HEAP32[$33>>2] = $241;
 $242 = HEAP32[$33>>2]|0;
 $243 = ((($242)) + 11|0);
 $244 = HEAP8[$243>>0]|0;
 $245 = $244&255;
 $246 = $245 & 128;
 $247 = ($246|0)!=(0);
 if ($247) {
  HEAP32[$27>>2] = $239;
  $248 = HEAP32[$27>>2]|0;
  HEAP32[$26>>2] = $248;
  $249 = HEAP32[$26>>2]|0;
  HEAP32[$25>>2] = $249;
  $250 = HEAP32[$25>>2]|0;
  $251 = HEAP32[$250>>2]|0;
  $257 = $251;
 } else {
  HEAP32[$32>>2] = $239;
  $252 = HEAP32[$32>>2]|0;
  HEAP32[$31>>2] = $252;
  $253 = HEAP32[$31>>2]|0;
  HEAP32[$30>>2] = $253;
  $254 = HEAP32[$30>>2]|0;
  HEAP32[$29>>2] = $254;
  $255 = HEAP32[$29>>2]|0;
  HEAP32[$28>>2] = $255;
  $256 = HEAP32[$28>>2]|0;
  $257 = $256;
 }
 HEAP32[$24>>2] = $257;
 $258 = HEAP32[$24>>2]|0;
 HEAP32[$17>>2] = $94;
 $259 = HEAP32[$17>>2]|0;
 HEAP32[$16>>2] = $259;
 $260 = HEAP32[$16>>2]|0;
 HEAP32[$15>>2] = $260;
 $261 = HEAP32[$15>>2]|0;
 HEAP32[$14>>2] = $261;
 $262 = HEAP32[$14>>2]|0;
 HEAP32[$13>>2] = $262;
 $263 = HEAP32[$13>>2]|0;
 HEAP32[$12>>2] = $263;
 $264 = HEAP32[$12>>2]|0;
 $265 = ((($264)) + 11|0);
 $266 = HEAP8[$265>>0]|0;
 $267 = $266&255;
 $268 = $267 & 128;
 $269 = ($268|0)!=(0);
 if ($269) {
  HEAP32[$6>>2] = $261;
  $270 = HEAP32[$6>>2]|0;
  HEAP32[$5>>2] = $270;
  $271 = HEAP32[$5>>2]|0;
  HEAP32[$4>>2] = $271;
  $272 = HEAP32[$4>>2]|0;
  $273 = HEAP32[$272>>2]|0;
  $279 = $273;
  HEAP32[$3>>2] = $279;
  $280 = HEAP32[$3>>2]|0;
  HEAP32[$18>>2] = $95;
  $281 = HEAP32[$18>>2]|0;
  $282 = ((($281)) + 4|0);
  $283 = HEAP32[$282>>2]|0;
  $284 = HEAP32[$281>>2]|0;
  $285 = $283;
  $286 = $284;
  $287 = (($285) - ($286))|0;
  $288 = (($287|0) / 4)&-1;
  HEAP32[$20>>2] = $95;
  $289 = HEAP32[$20>>2]|0;
  $290 = HEAP32[$289>>2]|0;
  HEAP32[$19>>2] = $290;
  $291 = HEAP32[$19>>2]|0;
  HEAP32[$21>>2] = $96;
  $292 = HEAP32[$21>>2]|0;
  $293 = ((($292)) + 4|0);
  $294 = HEAP32[$293>>2]|0;
  $295 = HEAP32[$292>>2]|0;
  $296 = $294;
  $297 = $295;
  $298 = (($296) - ($297))|0;
  $299 = (($298|0) / 4)&-1;
  HEAP32[$23>>2] = $96;
  $300 = HEAP32[$23>>2]|0;
  $301 = HEAP32[$300>>2]|0;
  HEAP32[$22>>2] = $301;
  $302 = HEAP32[$22>>2]|0;
  _SceneBuilderAdapter_createDetailView(($164|0),($165|0),($166|0),($192|0),($218|0),($219|0),($221|0),($223|0),(+$225),($227|0),($229|0),($231|0),($98|0),($232|0),($303|0),($258|0),($280|0),($288|0),($291|0),($299|0),($302|0));
  __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($96);
  __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($95);
  __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($94);
  __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($93);
  STACKTOP = sp;return;
 } else {
  HEAP32[$11>>2] = $261;
  $274 = HEAP32[$11>>2]|0;
  HEAP32[$10>>2] = $274;
  $275 = HEAP32[$10>>2]|0;
  HEAP32[$9>>2] = $275;
  $276 = HEAP32[$9>>2]|0;
  HEAP32[$8>>2] = $276;
  $277 = HEAP32[$8>>2]|0;
  HEAP32[$7>>2] = $277;
  $278 = HEAP32[$7>>2]|0;
  $279 = $278;
  HEAP32[$3>>2] = $279;
  $280 = HEAP32[$3>>2]|0;
  HEAP32[$18>>2] = $95;
  $281 = HEAP32[$18>>2]|0;
  $282 = ((($281)) + 4|0);
  $283 = HEAP32[$282>>2]|0;
  $284 = HEAP32[$281>>2]|0;
  $285 = $283;
  $286 = $284;
  $287 = (($285) - ($286))|0;
  $288 = (($287|0) / 4)&-1;
  HEAP32[$20>>2] = $95;
  $289 = HEAP32[$20>>2]|0;
  $290 = HEAP32[$289>>2]|0;
  HEAP32[$19>>2] = $290;
  $291 = HEAP32[$19>>2]|0;
  HEAP32[$21>>2] = $96;
  $292 = HEAP32[$21>>2]|0;
  $293 = ((($292)) + 4|0);
  $294 = HEAP32[$293>>2]|0;
  $295 = HEAP32[$292>>2]|0;
  $296 = $294;
  $297 = $295;
  $298 = (($296) - ($297))|0;
  $299 = (($298|0) / 4)&-1;
  HEAP32[$23>>2] = $96;
  $300 = HEAP32[$23>>2]|0;
  $301 = HEAP32[$300>>2]|0;
  HEAP32[$22>>2] = $301;
  $302 = HEAP32[$22>>2]|0;
  _SceneBuilderAdapter_createDetailView(($164|0),($165|0),($166|0),($192|0),($218|0),($219|0),($221|0),($223|0),(+$225),($227|0),($229|0),($231|0),($98|0),($232|0),($303|0),($258|0),($280|0),($288|0),($291|0),($299|0),($302|0));
  __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($96);
  __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($95);
  __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($94);
  __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($93);
  STACKTOP = sp;return;
 }
}
function __ZNSt3__23mapIPKN3sap2ve4core10DetailViewEPKNS3_5INodeENS_4lessIS6_EENS_9allocatorINS_4pairIKS6_S9_EEEEEixERSE_($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0;
 var $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 128|0;
 $2 = sp + 108|0;
 $3 = sp + 117|0;
 $4 = sp + 104|0;
 $5 = sp + 100|0;
 $6 = sp + 96|0;
 $7 = sp + 92|0;
 $8 = sp + 88|0;
 $9 = sp + 84|0;
 $10 = sp + 80|0;
 $11 = sp + 76|0;
 $12 = sp + 72|0;
 $13 = sp + 68|0;
 $14 = sp + 24|0;
 $15 = sp + 16|0;
 $16 = sp + 8|0;
 $17 = sp;
 $18 = sp + 64|0;
 $19 = sp + 60|0;
 $20 = sp + 116|0;
 $21 = sp + 115|0;
 $22 = sp + 114|0;
 $23 = sp + 113|0;
 $24 = sp + 56|0;
 $25 = sp + 52|0;
 $26 = sp + 48|0;
 $27 = sp + 44|0;
 $28 = sp + 40|0;
 $29 = sp + 32|0;
 $30 = sp + 28|0;
 $31 = sp + 112|0;
 HEAP32[$27>>2] = $0;
 HEAP32[$28>>2] = $1;
 $32 = HEAP32[$27>>2]|0;
 $33 = HEAP32[$28>>2]|0;
 $34 = HEAP32[$28>>2]|0;
 HEAP32[$26>>2] = $34;
 $35 = HEAP32[$26>>2]|0;
 HEAP32[$24>>2] = $35;
 $36 = HEAP32[$24>>2]|0;
 HEAP32[$18>>2] = $25;
 HEAP32[$19>>2] = $36;
 $37 = HEAP32[$18>>2]|0;
 $38 = HEAP32[$19>>2]|0;
 ;HEAP8[$14>>0]=HEAP8[$23>>0]|0;
 ;HEAP8[$15>>0]=HEAP8[$22>>0]|0;
 ;HEAP8[$16>>0]=HEAP8[$21>>0]|0;
 ;HEAP8[$17>>0]=HEAP8[$20>>0]|0;
 HEAP32[$12>>2] = $37;
 HEAP32[$13>>2] = $38;
 $39 = HEAP32[$12>>2]|0;
 $40 = HEAP32[$13>>2]|0;
 HEAP32[$11>>2] = $40;
 $41 = HEAP32[$11>>2]|0;
 HEAP32[$9>>2] = $39;
 HEAP32[$10>>2] = $41;
 $42 = HEAP32[$9>>2]|0;
 $43 = HEAP32[$10>>2]|0;
 HEAP32[$8>>2] = $43;
 $44 = HEAP32[$8>>2]|0;
 HEAP32[$42>>2] = $44;
 $45 = HEAP32[$25>>2]|0;
 HEAP32[$30>>2] = $45;
 HEAP32[$2>>2] = $3;
 __ZNSt3__26__treeINS_12__value_typeIPKN3sap2ve4core10DetailViewEPKNS4_5INodeEEENS_19__map_value_compareIS7_SB_NS_4lessIS7_EELb1EEENS_9allocatorISB_EEE25__emplace_unique_key_argsIS7_JRKNS_21piecewise_construct_tENS_5tupleIJRKS7_EEENSN_IJEEEEEENS_4pairINS_15__tree_iteratorISB_PNS_11__tree_nodeISB_PvEElEEbEERKT_DpOT0_($29,$32,$33,155088,$30,$31);
 HEAP32[$7>>2] = $29;
 $46 = HEAP32[$7>>2]|0;
 HEAP32[$6>>2] = $46;
 $47 = HEAP32[$6>>2]|0;
 $48 = HEAP32[$47>>2]|0;
 $49 = ((($48)) + 16|0);
 HEAP32[$5>>2] = $49;
 $50 = HEAP32[$5>>2]|0;
 HEAP32[$4>>2] = $50;
 $51 = HEAP32[$4>>2]|0;
 $52 = ((($51)) + 4|0);
 STACKTOP = sp;return ($52|0);
}
function __ZNK3sap2ve8Vector2f4DataEv($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 STACKTOP = sp;return ($2|0);
}
function __ZN3sap2ve6Vector4DataEv($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 STACKTOP = sp;return ($2|0);
}
function __ZN17DataModelImporter14RealiseContentEPKN3sap2ve4core5INodeE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0;
 var $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0;
 var $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0;
 var $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0;
 var $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 176|0;
 $2 = sp + 156|0;
 $3 = sp + 152|0;
 $4 = sp + 148|0;
 $5 = sp + 144|0;
 $6 = sp + 140|0;
 $7 = sp + 136|0;
 $8 = sp + 132|0;
 $9 = sp + 128|0;
 $10 = sp + 124|0;
 $11 = sp + 120|0;
 $12 = sp + 116|0;
 $13 = sp + 112|0;
 $14 = sp + 108|0;
 $15 = sp + 104|0;
 $16 = sp;
 $17 = sp + 100|0;
 $18 = sp + 96|0;
 $19 = sp + 92|0;
 $20 = sp + 88|0;
 $21 = sp + 161|0;
 $22 = sp + 84|0;
 $23 = sp + 80|0;
 $24 = sp + 76|0;
 $25 = sp + 72|0;
 $26 = sp + 68|0;
 $27 = sp + 64|0;
 $28 = sp + 60|0;
 $29 = sp + 56|0;
 $30 = sp + 52|0;
 $31 = sp + 48|0;
 $32 = sp + 160|0;
 $33 = sp + 44|0;
 $34 = sp + 40|0;
 $35 = sp + 32|0;
 $36 = sp + 24|0;
 $37 = sp + 20|0;
 $38 = sp + 16|0;
 $39 = sp + 12|0;
 $40 = sp + 8|0;
 $41 = sp + 4|0;
 HEAP32[$33>>2] = $0;
 HEAP32[$34>>2] = $1;
 $42 = HEAP32[$33>>2]|0;
 $43 = HEAP32[$34>>2]|0;
 $44 = ((($43)) + 4|0);
 $45 = HEAP32[$44>>2]|0;
 $46 = ((($45)) + 16|0);
 $47 = HEAP32[$46>>2]|0;
 $48 = (FUNCTION_TABLE_ii[$47 & 2047]($44)|0);
 HEAP32[$35>>2] = $48;
 $49 = ((($35)) + 4|0);
 $50 = HEAP32[$34>>2]|0;
 $51 = ((($50)) + 4|0);
 $52 = HEAP32[$51>>2]|0;
 $53 = ((($52)) + 8|0);
 $54 = HEAP32[$53>>2]|0;
 $55 = (FUNCTION_TABLE_ii[$54 & 2047]($51)|0);
 HEAP32[$49>>2] = $55;
 HEAP32[$36>>2] = 0;
 L1: while(1) {
  $56 = HEAP32[$36>>2]|0;
  $57 = ($56|0)<(2);
  if (!($57)) {
   label = 30;
   break;
  }
  $58 = HEAP32[$36>>2]|0;
  $59 = (($35) + ($58<<2)|0);
  $60 = HEAP32[$59>>2]|0;
  HEAP32[$37>>2] = $60;
  $61 = HEAP32[$37>>2]|0;
  $62 = ($61|0)!=(0|0);
  L4: do {
   if ($62) {
    $63 = HEAP32[$37>>2]|0;
    $64 = HEAP32[$63>>2]|0;
    $65 = ((($64)) + 12|0);
    $66 = HEAP32[$65>>2]|0;
    $67 = (FUNCTION_TABLE_ii[$66 & 2047]($63)|0);
    do {
     switch ($67|0) {
     case 107: case 65: case 56: case 106: case 105: case 104: case 102: case 55: case 103: case 57:  {
      break L1;
      break;
     }
     case 12:  {
      $68 = HEAP32[$37>>2]|0;
      $69 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter10InsertMeshEPKN3sap2ve4core4MeshEPKNS2_5INodeEb($42,$68,$69,0);
      break L4;
      break;
     }
     case 19:  {
      $88 = HEAP32[$37>>2]|0;
      __ZN3sap2ve4core21ParametricMeshFactory10CreateMeshEPNS1_16ParametricObjectE($41,$88);
      $89 = (__ZNK3sap2ve4core4IPtrINS1_4MeshEEcvPS3_Ev($41)|0);
      $90 = ($89|0)!=(0|0);
      if ($90) {
       $91 = ((($42)) + 56|0);
       HEAP32[$30>>2] = $91;
       HEAP32[$31>>2] = $41;
       $92 = HEAP32[$30>>2]|0;
       $93 = ((($92)) + 4|0);
       $94 = HEAP32[$93>>2]|0;
       HEAP32[$29>>2] = $92;
       $95 = HEAP32[$29>>2]|0;
       $96 = ((($95)) + 8|0);
       HEAP32[$28>>2] = $96;
       $97 = HEAP32[$28>>2]|0;
       HEAP32[$27>>2] = $97;
       $98 = HEAP32[$27>>2]|0;
       $99 = HEAP32[$98>>2]|0;
       $100 = ($94>>>0)<($99>>>0);
       if ($100) {
        HEAP32[$24>>2] = $32;
        HEAP32[$25>>2] = $92;
        HEAP32[$26>>2] = 1;
        HEAP32[$7>>2] = $92;
        $101 = HEAP32[$7>>2]|0;
        $102 = ((($101)) + 8|0);
        HEAP32[$6>>2] = $102;
        $103 = HEAP32[$6>>2]|0;
        HEAP32[$5>>2] = $103;
        $104 = HEAP32[$5>>2]|0;
        $105 = ((($92)) + 4|0);
        $106 = HEAP32[$105>>2]|0;
        HEAP32[$3>>2] = $106;
        $107 = HEAP32[$3>>2]|0;
        $108 = HEAP32[$31>>2]|0;
        HEAP32[$4>>2] = $108;
        $109 = HEAP32[$4>>2]|0;
        HEAP32[$18>>2] = $104;
        HEAP32[$19>>2] = $107;
        HEAP32[$20>>2] = $109;
        $110 = HEAP32[$18>>2]|0;
        $111 = HEAP32[$19>>2]|0;
        $112 = HEAP32[$20>>2]|0;
        HEAP32[$17>>2] = $112;
        $113 = HEAP32[$17>>2]|0;
        ;HEAP8[$16>>0]=HEAP8[$21>>0]|0;
        HEAP32[$13>>2] = $110;
        HEAP32[$14>>2] = $111;
        HEAP32[$15>>2] = $113;
        $114 = HEAP32[$13>>2]|0;
        $115 = HEAP32[$14>>2]|0;
        $116 = HEAP32[$15>>2]|0;
        HEAP32[$12>>2] = $116;
        $117 = HEAP32[$12>>2]|0;
        HEAP32[$9>>2] = $114;
        HEAP32[$10>>2] = $115;
        HEAP32[$11>>2] = $117;
        $118 = HEAP32[$10>>2]|0;
        $119 = HEAP32[$11>>2]|0;
        HEAP32[$8>>2] = $119;
        $120 = HEAP32[$8>>2]|0;
        __ZN3sap2ve4core4IPtrINS1_4MeshEEC2ERKS4_($118,$120);
        HEAP32[$22>>2] = $32;
        $121 = ((($92)) + 4|0);
        $122 = HEAP32[$121>>2]|0;
        $123 = ((($122)) + 4|0);
        HEAP32[$121>>2] = $123;
       } else {
        $124 = HEAP32[$31>>2]|0;
        HEAP32[$23>>2] = $124;
        $125 = HEAP32[$23>>2]|0;
        __ZNSt3__26vectorIN3sap2ve4core4IPtrINS3_4MeshEEENS_9allocatorIS6_EEE24__emplace_back_slow_pathIJRS6_EEEvDpOT_($92,$125);
       }
       $126 = ((($42)) + 56|0);
       HEAP32[$2>>2] = $126;
       $127 = HEAP32[$2>>2]|0;
       $128 = ((($127)) + 4|0);
       $129 = HEAP32[$128>>2]|0;
       $130 = ((($129)) + -4|0);
       $131 = (__ZNK3sap2ve4core4IPtrINS1_4MeshEEcvPS3_Ev($130)|0);
       $132 = HEAP32[$34>>2]|0;
       __ZN17DataModelImporter10InsertMeshEPKN3sap2ve4core4MeshEPKNS2_5INodeEb($42,$131,$132,1);
      }
      __ZN3sap2ve4core4IPtrINS1_4MeshEED2Ev($41);
      break L4;
      break;
     }
     case 24:  {
      $133 = HEAP32[$37>>2]|0;
      $134 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter20InsertTextAnnotationEPKN3sap2ve4core14TextAnnotationEPKNS2_5INodeE($42,$133,$134);
      break L4;
      break;
     }
     case 25:  {
      $135 = HEAP32[$37>>2]|0;
      $136 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter14InsertTextNoteEPKN3sap2ve4core8TextNoteEPKNS2_5INodeE($42,$135,$136);
      break L4;
      break;
     }
     case 27:  {
      $137 = HEAP32[$37>>2]|0;
      $138 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter15InsertImageNoteEPKN3sap2ve4core9ImageNoteEPKNS2_5INodeE($42,$137,$138);
      break L4;
      break;
     }
     case 23:  {
      $139 = HEAP32[$37>>2]|0;
      $140 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter12InsertCameraEPKN3sap2ve4core6CameraEPKNS2_5INodeE($42,$139,$140);
      break L4;
      break;
     }
     case 122:  {
      $141 = HEAP32[$37>>2]|0;
      __ZN17DataModelImporter19InsertViewportGroupEPKN3sap2ve4core13ViewportGroupE($42,$141);
      break L4;
      break;
     }
     case 124:  {
      $142 = HEAP32[$37>>2]|0;
      $143 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter16InsertDetailViewEPKN3sap2ve4core10DetailViewEPKNS2_5INodeE($42,$142,$143);
      break L4;
      break;
     }
     case 120:  {
      $144 = HEAP32[$37>>2]|0;
      __ZN17DataModelImporter20InsertAnimationGroupEPKN3sap2ve4core14AnimationGroupEPKNS2_9ModelViewEPNS2_17AnimationPlaybackE($42,$144,0,0);
      break L4;
      break;
     }
     case 31:  {
      $145 = HEAP32[$37>>2]|0;
      $146 = HEAP32[$34>>2]|0;
      __ZN17DataModelImporter16InsertThrustlineEPKN3sap2ve4core10ThrustlineEPKNS2_5INodeE($42,$145,$146);
      break L4;
      break;
     }
     default: {
      $147 = HEAP32[$37>>2]|0;
      $148 = HEAP32[$147>>2]|0;
      $149 = ((($148)) + 12|0);
      $150 = HEAP32[$149>>2]|0;
      $151 = (FUNCTION_TABLE_ii[$150 & 2047]($147)|0);
      $152 = _emscripten_asm_const_ii(4,($151|0))|0;
      break L4;
     }
     }
    } while(0);
   }
  } while(0);
  $153 = HEAP32[$36>>2]|0;
  $154 = (($153) + 1)|0;
  HEAP32[$36>>2] = $154;
 }
 if ((label|0) == 30) {
  STACKTOP = sp;return;
 }
 $70 = HEAP32[$37>>2]|0;
 HEAP32[$38>>2] = $70;
 $71 = HEAP32[$38>>2]|0;
 $72 = (__ZNK3sap2ve4core16ParametricObject18GetCollectionCountEv($71)|0);
 $73 = ($72|0)!=(0);
 L29: do {
  if ($73) {
   HEAP32[$39>>2] = 0;
   while(1) {
    $74 = HEAP32[$39>>2]|0;
    $75 = HEAP32[$38>>2]|0;
    $76 = (__ZNK3sap2ve4core16ParametricObject18GetCollectionCountEv($75)|0);
    $77 = ($74>>>0)<($76>>>0);
    if (!($77)) {
     break L29;
    }
    $78 = HEAP32[$37>>2]|0;
    $79 = HEAP32[$39>>2]|0;
    __ZN3sap2ve4core18ParametricIteratorC2EPNS1_16ParametricObjectExb($40,$78,$79,0,1);
    while(1) {
     $80 = (__ZN3sap2ve4core18ParametricIterator7CurrentEv($40)|0);
     __ZN17DataModelImporter22InsertParametricObjectEPKN3sap2ve4core16ParametricObjectE($42,$80);
     $81 = (__ZN3sap2ve4core18ParametricIterator4NextEv($40)|0);
     if (!($81)) {
      break;
     }
    }
    __ZN3sap2ve4core18ParametricIteratorD2Ev($40);
    $82 = HEAP32[$39>>2]|0;
    $83 = (($82) + 1)|0;
    HEAP32[$39>>2] = $83;
   }
  } else {
   $84 = HEAP32[$38>>2]|0;
   __ZN17DataModelImporter22InsertParametricObjectEPKN3sap2ve4core16ParametricObjectE($42,$84);
  }
 } while(0);
 $85 = ((($42)) + 24|0);
 $86 = HEAP32[$85>>2]|0;
 $87 = HEAP32[$34>>2]|0;
 _SceneBuilderAdapter_setParametricContent(($86|0),($87|0));
 STACKTOP = sp;return;
}
function __ZNK3sap2ve4core4IPtrINS1_4MeshEEcvPS3_Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 $3 = HEAP32[$2>>2]|0;
 STACKTOP = sp;return ($3|0);
}
function __ZN3sap2ve4core4IPtrINS1_4MeshEED2Ev($0) {
 $0 = $0|0;
 var $1 = 0, $2 = 0, $3 = 0, $4 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 16|0;
 $1 = sp;
 HEAP32[$1>>2] = $0;
 $2 = HEAP32[$1>>2]|0;
 $3 = HEAP32[$2>>2]|0;
 $4 = ($3|0)!=(0|0);
 if (!($4)) {
  STACKTOP = sp;return;
 }
 $5 = HEAP32[$2>>2]|0;
 $6 = HEAP32[$5>>2]|0;
 $7 = ((($6)) + 4|0);
 $8 = HEAP32[$7>>2]|0;
 FUNCTION_TABLE_vi[$8 & 1023]($5);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter16InsertThrustlineEPKN3sap2ve4core10ThrustlineEPKNS2_5INodeE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $10 = 0, $100 = 0, $1000 = 0, $1001 = 0, $1002 = 0, $1003 = 0, $1004 = 0, $1005 = 0, $1006 = 0, $1007 = 0, $1008 = 0, $1009 = 0, $101 = 0, $1010 = 0, $1011 = 0, $1012 = 0, $1013 = 0, $1014 = 0, $1015 = 0, $1016 = 0;
 var $1017 = 0, $1018 = 0, $1019 = 0, $102 = 0, $1020 = 0, $1021 = 0, $1022 = 0, $1023 = 0, $1024 = 0, $1025 = 0, $1026 = 0, $1027 = 0, $1028 = 0, $1029 = 0, $103 = 0, $1030 = 0, $1031 = 0, $1032 = 0, $1033 = 0, $1034 = 0;
 var $1035 = 0, $1036 = 0, $1037 = 0, $1038 = 0, $1039 = 0, $104 = 0, $1040 = 0, $1041 = 0, $1042 = 0, $1043 = 0, $1044 = 0, $1045 = 0, $1046 = 0, $1047 = 0, $1048 = 0, $1049 = 0, $105 = 0, $1050 = 0, $1051 = 0, $1052 = 0;
 var $1053 = 0, $1054 = 0, $1055 = 0, $1056 = 0, $1057 = 0.0, $1058 = 0, $1059 = 0, $106 = 0, $1060 = 0, $1061 = 0, $1062 = 0, $1063 = 0.0, $1064 = 0, $1065 = 0, $1066 = 0, $1067 = 0, $1068 = 0, $1069 = 0.0, $107 = 0, $1070 = 0;
 var $1071 = 0, $1072 = 0, $1073 = 0, $1074 = 0, $1075 = 0, $1076 = 0, $1077 = 0, $1078 = 0, $1079 = 0, $108 = 0, $1080 = 0, $1081 = 0, $1082 = 0, $1083 = 0, $1084 = 0, $1085 = 0, $1086 = 0, $1087 = 0, $1088 = 0, $1089 = 0;
 var $109 = 0, $1090 = 0, $1091 = 0, $1092 = 0, $1093 = 0, $1094 = 0, $1095 = 0, $1096 = 0, $1097 = 0, $1098 = 0, $1099 = 0, $11 = 0, $110 = 0, $1100 = 0, $1101 = 0, $1102 = 0, $1103 = 0, $1104 = 0, $1105 = 0, $1106 = 0;
 var $1107 = 0, $1108 = 0, $1109 = 0, $111 = 0, $1110 = 0, $1111 = 0, $1112 = 0, $1113 = 0, $1114 = 0, $1115 = 0, $1116 = 0, $1117 = 0, $1118 = 0, $1119 = 0, $112 = 0, $1120 = 0, $1121 = 0, $1122 = 0, $1123 = 0, $1124 = 0;
 var $1125 = 0, $1126 = 0, $1127 = 0, $1128 = 0, $1129 = 0, $113 = 0, $1130 = 0, $1131 = 0, $1132 = 0, $1133 = 0, $1134 = 0, $1135 = 0, $1136 = 0, $1137 = 0, $1138 = 0, $1139 = 0, $114 = 0, $1140 = 0, $1141 = 0, $1142 = 0;
 var $1143 = 0, $1144 = 0, $1145 = 0, $1146 = 0, $1147 = 0, $1148 = 0, $1149 = 0, $115 = 0, $1150 = 0, $1151 = 0, $1152 = 0, $1153 = 0, $1154 = 0, $1155 = 0, $1156 = 0, $1157 = 0, $1158 = 0, $1159 = 0, $116 = 0, $1160 = 0;
 var $1161 = 0, $1162 = 0, $1163 = 0, $1164 = 0, $1165 = 0, $1166 = 0, $1167 = 0, $1168 = 0, $1169 = 0, $117 = 0, $1170 = 0, $1171 = 0, $1172 = 0, $1173 = 0, $1174 = 0, $1175 = 0, $1176 = 0, $1177 = 0, $1178 = 0, $1179 = 0;
 var $118 = 0, $1180 = 0, $1181 = 0, $1182 = 0, $1183 = 0, $1184 = 0, $1185 = 0, $1186 = 0, $1187 = 0, $1188 = 0, $1189 = 0, $119 = 0, $1190 = 0, $1191 = 0, $1192 = 0, $1193 = 0, $1194 = 0, $1195 = 0, $1196 = 0, $1197 = 0;
 var $1198 = 0, $1199 = 0, $12 = 0, $120 = 0, $1200 = 0, $1201 = 0, $1202 = 0, $1203 = 0, $1204 = 0, $1205 = 0, $1206 = 0, $1207 = 0, $1208 = 0, $1209 = 0, $121 = 0, $1210 = 0, $1211 = 0, $1212 = 0, $1213 = 0, $1214 = 0;
 var $1215 = 0, $1216 = 0, $1217 = 0, $1218 = 0, $1219 = 0, $122 = 0, $1220 = 0, $1221 = 0, $1222 = 0, $1223 = 0, $1224 = 0, $1225 = 0, $1226 = 0, $1227 = 0, $1228 = 0, $1229 = 0, $123 = 0, $1230 = 0, $1231 = 0, $1232 = 0;
 var $1233 = 0, $1234 = 0, $1235 = 0, $1236 = 0, $1237 = 0, $1238 = 0, $1239 = 0, $124 = 0, $1240 = 0, $1241 = 0, $1242 = 0, $1243 = 0, $1244 = 0, $1245 = 0, $1246 = 0, $1247 = 0, $1248 = 0, $1249 = 0, $125 = 0, $1250 = 0;
 var $1251 = 0, $1252 = 0, $1253 = 0, $1254 = 0, $1255 = 0, $1256 = 0, $1257 = 0, $1258 = 0, $1259 = 0, $126 = 0, $1260 = 0, $1261 = 0, $1262 = 0, $1263 = 0, $1264 = 0, $1265 = 0, $1266 = 0, $1267 = 0, $1268 = 0, $1269 = 0;
 var $127 = 0, $1270 = 0, $1271 = 0, $1272 = 0, $1273 = 0, $1274 = 0, $1275 = 0, $1276 = 0, $1277 = 0, $1278 = 0, $1279 = 0, $128 = 0, $1280 = 0, $1281 = 0, $1282 = 0, $1283 = 0, $1284 = 0, $1285 = 0, $1286 = 0, $1287 = 0;
 var $1288 = 0, $1289 = 0, $129 = 0, $1290 = 0, $1291 = 0, $1292 = 0, $1293 = 0, $1294 = 0, $1295 = 0, $1296 = 0, $1297 = 0, $1298 = 0, $1299 = 0, $13 = 0, $130 = 0, $1300 = 0, $1301 = 0, $1302 = 0, $1303 = 0, $1304 = 0;
 var $1305 = 0, $1306 = 0, $1307 = 0, $1308 = 0, $1309 = 0, $131 = 0, $1310 = 0, $1311 = 0, $1312 = 0, $1313 = 0, $1314 = 0, $1315 = 0, $1316 = 0, $1317 = 0, $1318 = 0.0, $1319 = 0, $132 = 0, $1320 = 0, $1321 = 0, $1322 = 0;
 var $1323 = 0, $1324 = 0, $1325 = 0, $1326 = 0, $1327 = 0, $1328 = 0, $1329 = 0, $133 = 0, $1330 = 0, $1331 = 0, $1332 = 0, $1333 = 0, $1334 = 0, $1335 = 0, $1336 = 0, $1337 = 0, $1338 = 0, $1339 = 0, $134 = 0, $1340 = 0;
 var $1341 = 0, $1342 = 0, $1343 = 0, $1344 = 0, $1345 = 0, $1346 = 0, $1347 = 0, $1348 = 0, $1349 = 0, $135 = 0, $1350 = 0, $1351 = 0, $1352 = 0, $1353 = 0, $1354 = 0.0, $1355 = 0, $1356 = 0, $1357 = 0, $1358 = 0, $1359 = 0;
 var $136 = 0, $1360 = 0, $1361 = 0, $1362 = 0, $1363 = 0, $1364 = 0, $1365 = 0, $1366 = 0, $1367 = 0, $1368 = 0, $1369 = 0, $137 = 0, $1370 = 0, $1371 = 0, $1372 = 0, $1373 = 0, $1374 = 0, $1375 = 0, $1376 = 0, $1377 = 0;
 var $1378 = 0, $1379 = 0, $138 = 0, $1380 = 0, $1381 = 0, $1382 = 0, $1383 = 0, $1384 = 0, $1385 = 0, $1386 = 0, $1387 = 0, $1388 = 0, $1389 = 0, $139 = 0, $1390 = 0.0, $1391 = 0, $1392 = 0, $1393 = 0, $1394 = 0, $1395 = 0;
 var $1396 = 0, $1397 = 0, $1398 = 0, $1399 = 0, $14 = 0, $140 = 0, $1400 = 0, $1401 = 0, $1402 = 0, $1403 = 0, $1404 = 0, $1405 = 0, $1406 = 0, $1407 = 0, $1408 = 0, $1409 = 0, $141 = 0, $1410 = 0, $1411 = 0, $1412 = 0;
 var $1413 = 0, $1414 = 0, $1415 = 0, $1416 = 0, $1417 = 0, $1418 = 0, $1419 = 0, $142 = 0, $1420 = 0, $1421 = 0, $1422 = 0, $1423 = 0, $1424 = 0, $1425 = 0, $1426 = 0, $1427 = 0.0, $1428 = 0, $1429 = 0, $143 = 0, $1430 = 0;
 var $1431 = 0, $1432 = 0, $1433 = 0, $1434 = 0, $1435 = 0, $1436 = 0, $1437 = 0, $1438 = 0, $1439 = 0, $144 = 0, $1440 = 0, $1441 = 0, $1442 = 0, $1443 = 0, $1444 = 0, $1445 = 0, $1446 = 0, $1447 = 0, $1448 = 0, $1449 = 0;
 var $145 = 0, $1450 = 0, $1451 = 0, $1452 = 0, $1453 = 0, $1454 = 0, $1455 = 0, $1456 = 0, $1457 = 0, $1458 = 0, $1459 = 0, $146 = 0, $1460 = 0, $1461 = 0, $1462 = 0, $1463 = 0.0, $1464 = 0, $1465 = 0, $1466 = 0, $1467 = 0;
 var $1468 = 0, $1469 = 0, $147 = 0, $1470 = 0, $1471 = 0, $1472 = 0, $1473 = 0, $1474 = 0, $1475 = 0, $1476 = 0, $1477 = 0, $1478 = 0, $1479 = 0, $148 = 0, $1480 = 0, $1481 = 0, $1482 = 0, $1483 = 0, $1484 = 0, $1485 = 0;
 var $1486 = 0, $1487 = 0, $1488 = 0, $1489 = 0, $149 = 0, $1490 = 0, $1491 = 0, $1492 = 0, $1493 = 0, $1494 = 0, $1495 = 0, $1496 = 0, $1497 = 0, $1498 = 0, $1499 = 0.0, $15 = 0, $150 = 0, $1500 = 0, $1501 = 0, $1502 = 0;
 var $1503 = 0, $1504 = 0, $1505 = 0, $1506 = 0, $1507 = 0, $1508 = 0, $1509 = 0, $151 = 0, $1510 = 0, $1511 = 0, $1512 = 0, $1513 = 0, $1514 = 0, $1515 = 0, $1516 = 0, $1517 = 0, $1518 = 0, $1519 = 0, $152 = 0, $1520 = 0;
 var $1521 = 0, $1522 = 0, $1523 = 0, $1524 = 0, $1525 = 0, $1526 = 0, $1527 = 0, $1528 = 0, $1529 = 0, $153 = 0, $1530 = 0, $1531 = 0, $1532 = 0, $1533 = 0, $1534 = 0, $1535 = 0.0, $1536 = 0, $1537 = 0, $1538 = 0, $1539 = 0;
 var $154 = 0, $1540 = 0, $1541 = 0, $1542 = 0, $1543 = 0, $1544 = 0, $1545 = 0, $1546 = 0, $1547 = 0, $1548 = 0, $1549 = 0, $155 = 0, $1550 = 0, $1551 = 0, $1552 = 0, $1553 = 0, $1554 = 0, $1555 = 0, $1556 = 0, $1557 = 0;
 var $1558 = 0, $1559 = 0, $156 = 0, $1560 = 0, $1561 = 0, $1562 = 0, $1563 = 0, $1564 = 0, $1565 = 0, $1566 = 0, $1567 = 0, $1568 = 0, $1569 = 0, $157 = 0, $1570 = 0, $1571 = 0.0, $1572 = 0, $1573 = 0, $1574 = 0, $1575 = 0;
 var $1576 = 0, $1577 = 0, $1578 = 0, $1579 = 0, $158 = 0, $1580 = 0, $1581 = 0, $1582 = 0, $1583 = 0, $1584 = 0, $1585 = 0, $1586 = 0, $1587 = 0, $1588 = 0, $1589 = 0, $159 = 0, $1590 = 0, $1591 = 0, $1592 = 0, $1593 = 0;
 var $1594 = 0, $1595 = 0, $1596 = 0, $1597 = 0, $1598 = 0, $1599 = 0, $16 = 0, $160 = 0, $1600 = 0, $1601 = 0, $1602 = 0, $1603 = 0, $1604 = 0, $1605 = 0, $1606 = 0, $1607 = 0.0, $1608 = 0, $1609 = 0, $161 = 0, $1610 = 0;
 var $1611 = 0, $1612 = 0, $1613 = 0, $1614 = 0, $1615 = 0, $1616 = 0, $1617 = 0, $1618 = 0, $1619 = 0, $162 = 0, $1620 = 0, $1621 = 0, $1622 = 0, $1623 = 0, $1624 = 0, $1625 = 0, $1626 = 0, $1627 = 0, $1628 = 0, $1629 = 0;
 var $163 = 0, $1630 = 0, $1631 = 0, $1632 = 0, $1633 = 0, $1634 = 0, $1635 = 0, $1636 = 0, $1637 = 0, $1638 = 0, $1639 = 0, $164 = 0, $1640 = 0, $1641 = 0, $1642 = 0, $1643 = 0, $1644 = 0, $1645 = 0, $1646 = 0, $1647 = 0;
 var $1648 = 0, $1649 = 0, $165 = 0, $1650 = 0, $1651 = 0, $1652 = 0, $1653 = 0, $1654 = 0, $1655 = 0, $1656 = 0, $1657 = 0, $1658 = 0, $1659 = 0, $166 = 0, $1660 = 0, $1661 = 0, $1662 = 0, $1663 = 0, $1664 = 0, $1665 = 0;
 var $1666 = 0, $1667 = 0, $1668 = 0, $1669 = 0, $167 = 0, $1670 = 0, $1671 = 0, $1672 = 0, $1673 = 0, $1674 = 0, $1675 = 0, $1676 = 0, $1677 = 0, $1678 = 0, $1679 = 0, $168 = 0, $1680 = 0, $1681 = 0, $1682 = 0, $1683 = 0;
 var $1684 = 0.0, $1685 = 0, $1686 = 0, $1687 = 0, $1688 = 0, $1689 = 0, $169 = 0, $1690 = 0, $1691 = 0, $1692 = 0, $1693 = 0, $1694 = 0, $1695 = 0, $1696 = 0, $1697 = 0, $1698 = 0, $1699 = 0, $17 = 0, $170 = 0, $1700 = 0;
 var $1701 = 0, $1702 = 0, $1703 = 0, $1704 = 0, $1705 = 0, $1706 = 0, $1707 = 0, $1708 = 0, $1709 = 0, $171 = 0, $1710 = 0, $1711 = 0, $1712 = 0, $1713 = 0, $1714 = 0, $1715 = 0, $1716 = 0, $1717 = 0, $1718 = 0, $1719 = 0;
 var $172 = 0, $1720 = 0.0, $1721 = 0, $1722 = 0, $1723 = 0, $1724 = 0, $1725 = 0, $1726 = 0, $1727 = 0, $1728 = 0, $1729 = 0, $173 = 0, $1730 = 0, $1731 = 0, $1732 = 0, $1733 = 0, $1734 = 0, $1735 = 0, $1736 = 0, $1737 = 0;
 var $1738 = 0, $1739 = 0, $174 = 0, $1740 = 0, $1741 = 0, $1742 = 0, $1743 = 0, $1744 = 0, $1745 = 0, $1746 = 0, $1747 = 0, $1748 = 0, $1749 = 0, $175 = 0, $1750 = 0, $1751 = 0, $1752 = 0, $1753 = 0, $1754 = 0, $1755 = 0;
 var $1756 = 0.0, $1757 = 0, $1758 = 0, $1759 = 0, $176 = 0, $1760 = 0, $1761 = 0, $1762 = 0, $1763 = 0, $1764 = 0, $1765 = 0, $1766 = 0, $1767 = 0, $1768 = 0, $1769 = 0, $177 = 0, $1770 = 0, $1771 = 0, $1772 = 0, $1773 = 0;
 var $1774 = 0, $1775 = 0, $1776 = 0, $1777 = 0, $1778 = 0, $1779 = 0, $178 = 0, $1780 = 0, $1781 = 0, $1782 = 0, $1783 = 0, $1784 = 0, $1785 = 0, $1786 = 0, $1787 = 0, $1788 = 0, $1789 = 0, $179 = 0, $1790 = 0, $1791 = 0;
 var $1792 = 0, $1793 = 0, $1794 = 0, $1795 = 0, $1796 = 0, $1797 = 0, $1798 = 0, $1799 = 0, $18 = 0, $180 = 0, $1800 = 0, $1801 = 0, $1802 = 0, $1803 = 0, $1804 = 0, $1805 = 0, $1806 = 0, $1807 = 0, $1808 = 0, $1809 = 0;
 var $181 = 0, $1810 = 0, $1811 = 0, $1812 = 0, $1813 = 0, $1814 = 0, $1815 = 0, $1816 = 0, $1817 = 0, $1818 = 0, $1819 = 0, $182 = 0, $1820 = 0, $1821 = 0, $1822 = 0, $1823 = 0, $1824 = 0, $1825 = 0, $1826 = 0, $1827 = 0;
 var $1828 = 0, $1829 = 0, $183 = 0, $1830 = 0, $1831 = 0, $1832 = 0, $1833 = 0, $1834 = 0, $1835 = 0, $1836 = 0, $1837 = 0, $1838 = 0, $1839 = 0, $184 = 0, $1840 = 0, $1841 = 0, $1842 = 0, $1843 = 0, $1844 = 0, $1845 = 0;
 var $1846 = 0, $1847 = 0, $1848 = 0, $1849 = 0, $185 = 0, $1850 = 0, $1851 = 0, $1852 = 0, $1853 = 0, $1854 = 0, $1855 = 0, $1856 = 0, $1857 = 0, $1858 = 0, $1859 = 0, $186 = 0, $1860 = 0, $1861 = 0, $1862 = 0, $1863 = 0;
 var $1864 = 0, $1865 = 0, $1866 = 0, $1867 = 0, $1868 = 0, $1869 = 0, $187 = 0, $1870 = 0, $1871 = 0, $1872 = 0, $1873 = 0, $1874 = 0, $1875 = 0, $1876 = 0, $1877 = 0, $1878 = 0, $1879 = 0, $188 = 0, $1880 = 0, $1881 = 0;
 var $1882 = 0, $1883 = 0, $1884 = 0, $1885 = 0, $1886 = 0, $1887 = 0, $1888 = 0, $1889 = 0, $189 = 0, $1890 = 0, $1891 = 0, $1892 = 0, $1893 = 0, $1894 = 0, $1895 = 0, $1896 = 0, $1897 = 0, $1898 = 0, $1899 = 0, $19 = 0;
 var $190 = 0, $1900 = 0, $1901 = 0, $1902 = 0, $1903 = 0, $1904 = 0, $1905 = 0, $1906 = 0, $1907 = 0, $1908 = 0, $1909 = 0, $191 = 0, $1910 = 0, $1911 = 0, $1912 = 0, $1913 = 0, $1914 = 0, $1915 = 0, $1916 = 0, $1917 = 0;
 var $1918 = 0, $1919 = 0, $192 = 0, $1920 = 0, $1921 = 0, $1922 = 0, $1923 = 0, $1924 = 0, $1925 = 0, $1926 = 0, $1927 = 0, $1928 = 0, $1929 = 0, $193 = 0, $1930 = 0, $1931 = 0, $1932 = 0, $1933 = 0, $1934 = 0, $1935 = 0;
 var $1936 = 0, $1937 = 0, $1938 = 0, $1939 = 0, $194 = 0, $1940 = 0, $1941 = 0, $1942 = 0, $1943 = 0, $1944 = 0, $1945 = 0, $1946 = 0, $1947 = 0, $1948 = 0, $1949 = 0, $195 = 0, $1950 = 0, $1951 = 0, $1952 = 0, $1953 = 0;
 var $1954 = 0, $1955 = 0, $1956 = 0, $1957 = 0, $1958 = 0, $1959 = 0, $196 = 0, $1960 = 0, $1961 = 0, $1962 = 0, $1963 = 0, $1964 = 0, $1965 = 0, $1966 = 0, $1967 = 0, $1968 = 0, $1969 = 0, $197 = 0, $1970 = 0, $1971 = 0;
 var $1972 = 0, $1973 = 0, $1974 = 0, $1975 = 0, $1976 = 0, $1977 = 0, $1978 = 0, $1979 = 0, $198 = 0, $1980 = 0, $1981 = 0, $1982 = 0, $1983 = 0, $1984 = 0, $1985 = 0, $1986 = 0, $1987 = 0, $1988 = 0, $1989 = 0, $199 = 0;
 var $1990 = 0, $1991 = 0, $1992 = 0, $1993 = 0, $1994 = 0, $1995 = 0, $1996 = 0, $1997 = 0, $1998 = 0, $1999 = 0, $20 = 0, $200 = 0, $2000 = 0, $2001 = 0, $2002 = 0, $2003 = 0, $2004 = 0, $2005 = 0, $2006 = 0, $2007 = 0;
 var $2008 = 0, $2009 = 0, $201 = 0, $2010 = 0, $2011 = 0, $2012 = 0, $2013 = 0, $2014 = 0, $2015 = 0, $2016 = 0, $2017 = 0, $2018 = 0, $2019 = 0, $202 = 0, $2020 = 0, $2021 = 0, $2022 = 0, $2023 = 0, $2024 = 0, $2025 = 0;
 var $2026 = 0, $2027 = 0, $2028 = 0, $2029 = 0, $203 = 0, $2030 = 0, $2031 = 0, $2032 = 0, $2033 = 0, $2034 = 0, $2035 = 0, $2036 = 0, $2037 = 0, $2038 = 0, $2039 = 0, $204 = 0, $2040 = 0, $2041 = 0, $2042 = 0, $2043 = 0;
 var $2044 = 0, $2045 = 0, $2046 = 0, $2047 = 0, $2048 = 0, $2049 = 0, $205 = 0, $2050 = 0, $2051 = 0, $2052 = 0, $2053 = 0, $2054 = 0, $2055 = 0, $2056 = 0, $2057 = 0, $2058 = 0, $2059 = 0, $206 = 0, $2060 = 0, $2061 = 0;
 var $2062 = 0, $2063 = 0, $2064 = 0, $2065 = 0, $2066 = 0, $2067 = 0, $2068 = 0, $2069 = 0, $207 = 0, $2070 = 0, $2071 = 0, $2072 = 0, $2073 = 0, $2074 = 0, $2075 = 0, $2076 = 0, $2077 = 0, $2078 = 0, $2079 = 0, $208 = 0;
 var $2080 = 0, $2081 = 0, $2082 = 0, $2083 = 0, $2084 = 0, $2085 = 0, $2086 = 0, $2087 = 0, $2088 = 0, $2089 = 0, $209 = 0, $2090 = 0, $2091 = 0, $2092 = 0, $2093 = 0, $2094 = 0, $2095 = 0, $2096 = 0, $2097 = 0, $2098 = 0;
 var $2099 = 0, $21 = 0, $210 = 0, $2100 = 0, $2101 = 0, $2102 = 0, $2103 = 0, $2104 = 0, $2105 = 0, $2106 = 0, $2107 = 0, $2108 = 0, $2109 = 0, $211 = 0, $2110 = 0, $2111 = 0, $2112 = 0, $2113 = 0, $2114 = 0, $2115 = 0;
 var $2116 = 0, $2117 = 0, $2118 = 0, $2119 = 0, $212 = 0, $2120 = 0, $2121 = 0, $2122 = 0, $2123 = 0, $2124 = 0, $2125 = 0, $2126 = 0, $2127 = 0, $2128 = 0, $2129 = 0, $213 = 0, $2130 = 0, $2131 = 0, $2132 = 0, $2133 = 0;
 var $2134 = 0, $2135 = 0, $2136 = 0, $2137 = 0, $2138 = 0, $2139 = 0, $214 = 0, $2140 = 0, $2141 = 0, $2142 = 0, $2143 = 0, $2144 = 0, $2145 = 0, $2146 = 0, $2147 = 0, $2148 = 0, $2149 = 0, $215 = 0, $2150 = 0, $2151 = 0;
 var $2152 = 0, $2153 = 0, $2154 = 0, $2155 = 0, $2156 = 0, $2157 = 0, $2158 = 0, $2159 = 0, $216 = 0, $2160 = 0, $2161 = 0, $2162 = 0, $2163 = 0, $2164 = 0, $2165 = 0.0, $2166 = 0, $2167 = 0, $2168 = 0, $2169 = 0, $217 = 0;
 var $2170 = 0, $2171 = 0, $2172 = 0, $2173 = 0, $2174 = 0, $2175 = 0, $2176 = 0, $2177 = 0, $2178 = 0, $2179 = 0, $218 = 0, $2180 = 0, $2181 = 0, $2182 = 0, $2183 = 0, $2184 = 0, $2185 = 0, $2186 = 0, $2187 = 0, $2188 = 0;
 var $2189 = 0, $219 = 0, $2190 = 0, $2191 = 0, $2192 = 0, $2193 = 0, $2194 = 0, $2195 = 0, $2196 = 0, $2197 = 0, $2198 = 0, $2199 = 0, $22 = 0, $220 = 0, $2200 = 0, $2201 = 0.0, $2202 = 0, $2203 = 0, $2204 = 0, $2205 = 0;
 var $2206 = 0, $2207 = 0, $2208 = 0, $2209 = 0, $221 = 0, $2210 = 0, $2211 = 0, $2212 = 0, $2213 = 0, $2214 = 0, $2215 = 0, $2216 = 0, $2217 = 0, $2218 = 0, $2219 = 0, $222 = 0, $2220 = 0, $2221 = 0, $2222 = 0, $2223 = 0;
 var $2224 = 0, $2225 = 0, $2226 = 0, $2227 = 0, $2228 = 0, $2229 = 0, $223 = 0, $2230 = 0, $2231 = 0, $2232 = 0, $2233 = 0, $2234 = 0, $2235 = 0, $2236 = 0, $2237 = 0, $2238 = 0, $2239 = 0, $224 = 0, $2240 = 0, $2241 = 0;
 var $2242 = 0, $2243 = 0, $2244 = 0, $2245 = 0, $2246 = 0, $2247 = 0, $2248 = 0, $2249 = 0, $225 = 0, $2250 = 0, $2251 = 0, $2252 = 0, $2253 = 0, $2254 = 0, $2255 = 0, $2256 = 0, $2257 = 0, $2258 = 0, $2259 = 0, $226 = 0;
 var $2260 = 0, $2261 = 0, $2262 = 0, $2263 = 0, $2264 = 0, $2265 = 0, $2266 = 0, $2267 = 0, $2268 = 0, $2269 = 0, $227 = 0, $2270 = 0, $2271 = 0, $2272 = 0, $2273 = 0, $2274 = 0, $2275 = 0, $2276 = 0, $2277 = 0, $2278 = 0;
 var $2279 = 0, $228 = 0, $2280 = 0, $2281 = 0, $2282 = 0, $2283 = 0, $2284 = 0, $2285 = 0, $229 = 0, $23 = 0, $230 = 0, $231 = 0, $232 = 0, $233 = 0, $234 = 0, $235 = 0, $236 = 0, $237 = 0, $238 = 0, $239 = 0;
 var $24 = 0, $240 = 0, $241 = 0, $242 = 0, $243 = 0, $244 = 0, $245 = 0, $246 = 0, $247 = 0, $248 = 0, $249 = 0, $25 = 0, $250 = 0, $251 = 0, $252 = 0, $253 = 0, $254 = 0, $255 = 0, $256 = 0, $257 = 0;
 var $258 = 0, $259 = 0, $26 = 0, $260 = 0, $261 = 0, $262 = 0, $263 = 0, $264 = 0, $265 = 0, $266 = 0, $267 = 0, $268 = 0, $269 = 0, $27 = 0, $270 = 0, $271 = 0, $272 = 0, $273 = 0, $274 = 0, $275 = 0;
 var $276 = 0, $277 = 0, $278 = 0, $279 = 0, $28 = 0, $280 = 0, $281 = 0, $282 = 0, $283 = 0, $284 = 0, $285 = 0, $286 = 0, $287 = 0, $288 = 0, $289 = 0, $29 = 0, $290 = 0, $291 = 0, $292 = 0, $293 = 0;
 var $294 = 0, $295 = 0, $296 = 0, $297 = 0, $298 = 0, $299 = 0, $3 = 0, $30 = 0, $300 = 0, $301 = 0, $302 = 0, $303 = 0, $304 = 0, $305 = 0, $306 = 0, $307 = 0, $308 = 0, $309 = 0, $31 = 0, $310 = 0;
 var $311 = 0, $312 = 0, $313 = 0, $314 = 0, $315 = 0, $316 = 0, $317 = 0, $318 = 0, $319 = 0, $32 = 0, $320 = 0, $321 = 0, $322 = 0, $323 = 0, $324 = 0, $325 = 0, $326 = 0, $327 = 0, $328 = 0, $329 = 0;
 var $33 = 0, $330 = 0, $331 = 0, $332 = 0, $333 = 0, $334 = 0, $335 = 0, $336 = 0, $337 = 0, $338 = 0, $339 = 0, $34 = 0, $340 = 0, $341 = 0, $342 = 0, $343 = 0, $344 = 0, $345 = 0, $346 = 0, $347 = 0;
 var $348 = 0, $349 = 0, $35 = 0, $350 = 0, $351 = 0, $352 = 0, $353 = 0, $354 = 0, $355 = 0, $356 = 0, $357 = 0, $358 = 0, $359 = 0, $36 = 0, $360 = 0, $361 = 0, $362 = 0, $363 = 0, $364 = 0, $365 = 0;
 var $366 = 0, $367 = 0, $368 = 0, $369 = 0, $37 = 0, $370 = 0, $371 = 0, $372 = 0, $373 = 0, $374 = 0, $375 = 0, $376 = 0, $377 = 0, $378 = 0, $379 = 0, $38 = 0, $380 = 0, $381 = 0, $382 = 0, $383 = 0;
 var $384 = 0, $385 = 0, $386 = 0, $387 = 0, $388 = 0, $389 = 0, $39 = 0, $390 = 0, $391 = 0, $392 = 0, $393 = 0, $394 = 0, $395 = 0, $396 = 0, $397 = 0, $398 = 0, $399 = 0, $4 = 0, $40 = 0, $400 = 0;
 var $401 = 0, $402 = 0, $403 = 0, $404 = 0, $405 = 0, $406 = 0, $407 = 0, $408 = 0, $409 = 0, $41 = 0, $410 = 0, $411 = 0, $412 = 0, $413 = 0, $414 = 0, $415 = 0, $416 = 0, $417 = 0, $418 = 0, $419 = 0;
 var $42 = 0, $420 = 0, $421 = 0, $422 = 0, $423 = 0, $424 = 0, $425 = 0, $426 = 0, $427 = 0, $428 = 0, $429 = 0, $43 = 0, $430 = 0, $431 = 0, $432 = 0, $433 = 0, $434 = 0, $435 = 0, $436 = 0, $437 = 0;
 var $438 = 0, $439 = 0, $44 = 0, $440 = 0, $441 = 0, $442 = 0, $443 = 0, $444 = 0, $445 = 0, $446 = 0, $447 = 0, $448 = 0, $449 = 0, $45 = 0, $450 = 0, $451 = 0, $452 = 0, $453 = 0, $454 = 0, $455 = 0;
 var $456 = 0, $457 = 0, $458 = 0, $459 = 0, $46 = 0, $460 = 0, $461 = 0, $462 = 0, $463 = 0, $464 = 0, $465 = 0, $466 = 0, $467 = 0, $468 = 0, $469 = 0, $47 = 0, $470 = 0, $471 = 0, $472 = 0, $473 = 0;
 var $474 = 0, $475 = 0, $476 = 0, $477 = 0, $478 = 0, $479 = 0, $48 = 0, $480 = 0, $481 = 0, $482 = 0, $483 = 0, $484 = 0, $485 = 0, $486 = 0, $487 = 0, $488 = 0, $489 = 0, $49 = 0, $490 = 0, $491 = 0;
 var $492 = 0, $493 = 0, $494 = 0, $495 = 0, $496 = 0, $497 = 0, $498 = 0, $499 = 0, $5 = 0, $50 = 0, $500 = 0, $501 = 0, $502 = 0, $503 = 0, $504 = 0, $505 = 0, $506 = 0, $507 = 0, $508 = 0, $509 = 0;
 var $51 = 0, $510 = 0, $511 = 0, $512 = 0, $513 = 0, $514 = 0, $515 = 0, $516 = 0, $517 = 0, $518 = 0, $519 = 0, $52 = 0, $520 = 0, $521 = 0, $522 = 0, $523 = 0, $524 = 0, $525 = 0, $526 = 0, $527 = 0;
 var $528 = 0, $529 = 0, $53 = 0, $530 = 0, $531 = 0, $532 = 0, $533 = 0, $534 = 0, $535 = 0, $536 = 0, $537 = 0, $538 = 0, $539 = 0, $54 = 0, $540 = 0, $541 = 0, $542 = 0, $543 = 0, $544 = 0, $545 = 0;
 var $546 = 0, $547 = 0, $548 = 0, $549 = 0, $55 = 0, $550 = 0, $551 = 0, $552 = 0, $553 = 0, $554 = 0, $555 = 0, $556 = 0, $557 = 0, $558 = 0, $559 = 0, $56 = 0, $560 = 0, $561 = 0, $562 = 0, $563 = 0;
 var $564 = 0, $565 = 0, $566 = 0, $567 = 0, $568 = 0, $569 = 0, $57 = 0, $570 = 0, $571 = 0, $572 = 0, $573 = 0, $574 = 0, $575 = 0, $576 = 0, $577 = 0, $578 = 0, $579 = 0, $58 = 0, $580 = 0, $581 = 0;
 var $582 = 0, $583 = 0, $584 = 0, $585 = 0, $586 = 0, $587 = 0, $588 = 0, $589 = 0, $59 = 0, $590 = 0, $591 = 0, $592 = 0, $593 = 0, $594 = 0, $595 = 0, $596 = 0, $597 = 0, $598 = 0, $599 = 0, $6 = 0;
 var $60 = 0, $600 = 0, $601 = 0, $602 = 0, $603 = 0, $604 = 0, $605 = 0, $606 = 0, $607 = 0, $608 = 0, $609 = 0, $61 = 0, $610 = 0, $611 = 0, $612 = 0, $613 = 0, $614 = 0, $615 = 0, $616 = 0, $617 = 0;
 var $618 = 0, $619 = 0, $62 = 0, $620 = 0, $621 = 0, $622 = 0, $623 = 0, $624 = 0, $625 = 0, $626 = 0, $627 = 0, $628 = 0, $629 = 0, $63 = 0, $630 = 0, $631 = 0, $632 = 0, $633 = 0, $634 = 0, $635 = 0;
 var $636 = 0, $637 = 0, $638 = 0, $639 = 0, $64 = 0, $640 = 0, $641 = 0, $642 = 0, $643 = 0, $644 = 0, $645 = 0, $646 = 0, $647 = 0, $648 = 0, $649 = 0, $65 = 0, $650 = 0, $651 = 0, $652 = 0, $653 = 0;
 var $654 = 0, $655 = 0, $656 = 0, $657 = 0, $658 = 0, $659 = 0, $66 = 0, $660 = 0, $661 = 0, $662 = 0, $663 = 0, $664 = 0, $665 = 0, $666 = 0, $667 = 0, $668 = 0, $669 = 0, $67 = 0, $670 = 0, $671 = 0;
 var $672 = 0, $673 = 0, $674 = 0, $675 = 0, $676 = 0, $677 = 0, $678 = 0, $679 = 0, $68 = 0, $680 = 0, $681 = 0, $682 = 0, $683 = 0, $684 = 0, $685 = 0, $686 = 0, $687 = 0, $688 = 0, $689 = 0, $69 = 0;
 var $690 = 0, $691 = 0, $692 = 0, $693 = 0, $694 = 0, $695 = 0, $696 = 0, $697 = 0, $698 = 0, $699 = 0, $7 = 0, $70 = 0, $700 = 0, $701 = 0, $702 = 0, $703 = 0, $704 = 0, $705 = 0, $706 = 0, $707 = 0;
 var $708 = 0, $709 = 0, $71 = 0, $710 = 0, $711 = 0, $712 = 0, $713 = 0, $714 = 0, $715 = 0, $716 = 0, $717 = 0, $718 = 0, $719 = 0, $72 = 0, $720 = 0, $721 = 0, $722 = 0, $723 = 0, $724 = 0, $725 = 0;
 var $726 = 0, $727 = 0, $728 = 0, $729 = 0, $73 = 0, $730 = 0, $731 = 0, $732 = 0, $733 = 0, $734 = 0, $735 = 0, $736 = 0, $737 = 0, $738 = 0, $739 = 0, $74 = 0, $740 = 0, $741 = 0, $742 = 0, $743 = 0;
 var $744 = 0, $745 = 0, $746 = 0, $747 = 0, $748 = 0, $749 = 0, $75 = 0, $750 = 0, $751 = 0, $752 = 0, $753 = 0, $754 = 0, $755 = 0, $756 = 0, $757 = 0, $758 = 0, $759 = 0, $76 = 0, $760 = 0, $761 = 0;
 var $762 = 0, $763 = 0, $764 = 0, $765 = 0, $766 = 0, $767 = 0, $768 = 0, $769 = 0, $77 = 0, $770 = 0, $771 = 0, $772 = 0, $773 = 0, $774 = 0, $775 = 0, $776 = 0, $777 = 0, $778 = 0, $779 = 0, $78 = 0;
 var $780 = 0, $781 = 0, $782 = 0, $783 = 0, $784 = 0, $785 = 0, $786 = 0, $787 = 0, $788 = 0, $789 = 0, $79 = 0, $790 = 0, $791 = 0, $792 = 0, $793 = 0, $794 = 0, $795 = 0, $796 = 0, $797 = 0, $798 = 0;
 var $799 = 0, $8 = 0, $80 = 0, $800 = 0, $801 = 0, $802 = 0, $803 = 0, $804 = 0, $805 = 0, $806 = 0, $807 = 0, $808 = 0, $809 = 0, $81 = 0, $810 = 0, $811 = 0, $812 = 0, $813 = 0, $814 = 0, $815 = 0;
 var $816 = 0, $817 = 0, $818 = 0, $819 = 0, $82 = 0, $820 = 0, $821 = 0, $822 = 0, $823 = 0, $824 = 0, $825 = 0, $826 = 0, $827 = 0, $828 = 0, $829 = 0, $83 = 0, $830 = 0, $831 = 0, $832 = 0, $833 = 0;
 var $834 = 0, $835 = 0, $836 = 0, $837 = 0, $838 = 0, $839 = 0, $84 = 0, $840 = 0, $841 = 0, $842 = 0, $843 = 0, $844 = 0, $845 = 0, $846 = 0, $847 = 0, $848 = 0, $849 = 0, $85 = 0, $850 = 0, $851 = 0;
 var $852 = 0, $853 = 0, $854 = 0, $855 = 0, $856 = 0, $857 = 0, $858 = 0, $859 = 0, $86 = 0, $860 = 0, $861 = 0, $862 = 0, $863 = 0, $864 = 0, $865 = 0, $866 = 0, $867 = 0, $868 = 0, $869 = 0, $87 = 0;
 var $870 = 0, $871 = 0, $872 = 0, $873 = 0, $874 = 0, $875 = 0, $876 = 0, $877 = 0, $878 = 0, $879 = 0, $88 = 0, $880 = 0, $881 = 0, $882 = 0, $883 = 0, $884 = 0, $885 = 0, $886 = 0, $887 = 0, $888 = 0;
 var $889 = 0, $89 = 0, $890 = 0, $891 = 0, $892 = 0, $893 = 0, $894 = 0, $895 = 0, $896 = 0, $897 = 0, $898 = 0, $899 = 0, $9 = 0, $90 = 0, $900 = 0, $901 = 0, $902 = 0, $903 = 0, $904 = 0, $905 = 0;
 var $906 = 0, $907 = 0, $908 = 0, $909 = 0, $91 = 0, $910 = 0, $911 = 0, $912 = 0, $913 = 0, $914 = 0, $915 = 0, $916 = 0, $917 = 0, $918 = 0, $919 = 0, $92 = 0, $920 = 0, $921 = 0, $922 = 0, $923 = 0;
 var $924 = 0, $925 = 0, $926 = 0, $927 = 0, $928 = 0, $929 = 0, $93 = 0, $930 = 0, $931 = 0, $932 = 0, $933 = 0, $934 = 0, $935 = 0, $936 = 0, $937 = 0, $938 = 0, $939 = 0, $94 = 0, $940 = 0, $941 = 0;
 var $942 = 0, $943 = 0, $944 = 0, $945 = 0, $946 = 0, $947 = 0, $948 = 0, $949 = 0, $95 = 0, $950 = 0, $951 = 0, $952 = 0, $953 = 0, $954 = 0, $955 = 0, $956 = 0, $957 = 0, $958 = 0, $959 = 0, $96 = 0;
 var $960 = 0, $961 = 0, $962 = 0, $963 = 0, $964 = 0, $965 = 0, $966 = 0, $967 = 0, $968 = 0, $969 = 0, $97 = 0, $970 = 0, $971 = 0, $972 = 0, $973 = 0, $974 = 0, $975 = 0, $976 = 0, $977 = 0, $978 = 0;
 var $979 = 0, $98 = 0, $980 = 0, $981 = 0, $982 = 0, $983 = 0, $984 = 0, $985 = 0, $986 = 0, $987 = 0, $988 = 0, $989 = 0, $99 = 0, $990 = 0, $991 = 0, $992 = 0, $993 = 0, $994 = 0, $995 = 0, $996 = 0;
 var $997 = 0, $998 = 0, $999 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 4368|0;
 $3 = sp + 4308|0;
 $4 = sp + 4304|0;
 $5 = sp + 4300|0;
 $6 = sp + 4296|0;
 $7 = sp + 4292|0;
 $8 = sp + 4288|0;
 $9 = sp + 4284|0;
 $10 = sp + 4280|0;
 $11 = sp + 4276|0;
 $12 = sp + 4272|0;
 $13 = sp + 4268|0;
 $14 = sp + 4264|0;
 $15 = sp + 4260|0;
 $16 = sp + 4256|0;
 $17 = sp + 4252|0;
 $18 = sp + 4248|0;
 $19 = sp + 4244|0;
 $20 = sp + 4240|0;
 $21 = sp + 4236|0;
 $22 = sp + 4232|0;
 $23 = sp + 4228|0;
 $24 = sp + 4224|0;
 $25 = sp + 4220|0;
 $26 = sp + 4216|0;
 $27 = sp + 4212|0;
 $28 = sp + 4208|0;
 $29 = sp + 4204|0;
 $30 = sp + 4200|0;
 $31 = sp + 4196|0;
 $32 = sp + 4192|0;
 $33 = sp + 4188|0;
 $34 = sp + 4184|0;
 $35 = sp + 4180|0;
 $36 = sp + 4176|0;
 $37 = sp + 4172|0;
 $38 = sp + 4168|0;
 $39 = sp + 4164|0;
 $40 = sp + 4160|0;
 $41 = sp + 4156|0;
 $42 = sp + 4152|0;
 $43 = sp + 4148|0;
 $44 = sp + 4144|0;
 $45 = sp + 4140|0;
 $46 = sp + 4136|0;
 $47 = sp + 4132|0;
 $48 = sp + 4128|0;
 $49 = sp + 232|0;
 $50 = sp + 4124|0;
 $51 = sp + 4120|0;
 $52 = sp + 4116|0;
 $53 = sp + 4112|0;
 $54 = sp + 4360|0;
 $55 = sp + 4108|0;
 $56 = sp + 4104|0;
 $57 = sp + 4100|0;
 $58 = sp + 4096|0;
 $59 = sp + 4092|0;
 $60 = sp + 4088|0;
 $61 = sp + 4084|0;
 $62 = sp + 4080|0;
 $63 = sp + 4076|0;
 $64 = sp + 4072|0;
 $65 = sp + 4359|0;
 $66 = sp + 4068|0;
 $67 = sp + 4064|0;
 $68 = sp + 4060|0;
 $69 = sp + 4056|0;
 $70 = sp + 4052|0;
 $71 = sp + 4048|0;
 $72 = sp + 4044|0;
 $73 = sp + 4040|0;
 $74 = sp + 4036|0;
 $75 = sp + 4032|0;
 $76 = sp + 4028|0;
 $77 = sp + 4024|0;
 $78 = sp + 4020|0;
 $79 = sp + 224|0;
 $80 = sp + 4016|0;
 $81 = sp + 4012|0;
 $82 = sp + 4008|0;
 $83 = sp + 4004|0;
 $84 = sp + 4358|0;
 $85 = sp + 4000|0;
 $86 = sp + 3996|0;
 $87 = sp + 3992|0;
 $88 = sp + 3988|0;
 $89 = sp + 3984|0;
 $90 = sp + 3980|0;
 $91 = sp + 3976|0;
 $92 = sp + 3972|0;
 $93 = sp + 3968|0;
 $94 = sp + 3964|0;
 $95 = sp + 4357|0;
 $96 = sp + 3960|0;
 $97 = sp + 3956|0;
 $98 = sp + 3952|0;
 $99 = sp + 3948|0;
 $100 = sp + 3944|0;
 $101 = sp + 3940|0;
 $102 = sp + 3936|0;
 $103 = sp + 3932|0;
 $104 = sp + 3928|0;
 $105 = sp + 3924|0;
 $106 = sp + 3920|0;
 $107 = sp + 3916|0;
 $108 = sp + 3912|0;
 $109 = sp + 216|0;
 $110 = sp + 3908|0;
 $111 = sp + 3904|0;
 $112 = sp + 3900|0;
 $113 = sp + 3896|0;
 $114 = sp + 4356|0;
 $115 = sp + 3892|0;
 $116 = sp + 3888|0;
 $117 = sp + 3884|0;
 $118 = sp + 3880|0;
 $119 = sp + 3876|0;
 $120 = sp + 3872|0;
 $121 = sp + 3868|0;
 $122 = sp + 3864|0;
 $123 = sp + 3860|0;
 $124 = sp + 3856|0;
 $125 = sp + 4355|0;
 $126 = sp + 3852|0;
 $127 = sp + 3848|0;
 $128 = sp + 3844|0;
 $129 = sp + 3840|0;
 $130 = sp + 3836|0;
 $131 = sp + 3832|0;
 $132 = sp + 3828|0;
 $133 = sp + 3824|0;
 $134 = sp + 3820|0;
 $135 = sp + 3816|0;
 $136 = sp + 3812|0;
 $137 = sp + 3808|0;
 $138 = sp + 3804|0;
 $139 = sp + 208|0;
 $140 = sp + 3800|0;
 $141 = sp + 3796|0;
 $142 = sp + 3792|0;
 $143 = sp + 3788|0;
 $144 = sp + 4354|0;
 $145 = sp + 3784|0;
 $146 = sp + 3780|0;
 $147 = sp + 3776|0;
 $148 = sp + 3772|0;
 $149 = sp + 3768|0;
 $150 = sp + 3764|0;
 $151 = sp + 3760|0;
 $152 = sp + 3756|0;
 $153 = sp + 3752|0;
 $154 = sp + 3748|0;
 $155 = sp + 4353|0;
 $156 = sp + 3744|0;
 $157 = sp + 3740|0;
 $158 = sp + 3736|0;
 $159 = sp + 3732|0;
 $160 = sp + 3728|0;
 $161 = sp + 3724|0;
 $162 = sp + 3720|0;
 $163 = sp + 3716|0;
 $164 = sp + 3712|0;
 $165 = sp + 3708|0;
 $166 = sp + 3704|0;
 $167 = sp + 3700|0;
 $168 = sp + 3696|0;
 $169 = sp + 200|0;
 $170 = sp + 3692|0;
 $171 = sp + 3688|0;
 $172 = sp + 3684|0;
 $173 = sp + 3680|0;
 $174 = sp + 4352|0;
 $175 = sp + 3676|0;
 $176 = sp + 3672|0;
 $177 = sp + 3668|0;
 $178 = sp + 3664|0;
 $179 = sp + 3660|0;
 $180 = sp + 3656|0;
 $181 = sp + 3652|0;
 $182 = sp + 3648|0;
 $183 = sp + 3644|0;
 $184 = sp + 3640|0;
 $185 = sp + 4351|0;
 $186 = sp + 3636|0;
 $187 = sp + 3632|0;
 $188 = sp + 3628|0;
 $189 = sp + 3624|0;
 $190 = sp + 3620|0;
 $191 = sp + 3616|0;
 $192 = sp + 3612|0;
 $193 = sp + 3608|0;
 $194 = sp + 3604|0;
 $195 = sp + 3600|0;
 $196 = sp + 3596|0;
 $197 = sp + 3592|0;
 $198 = sp + 3588|0;
 $199 = sp + 192|0;
 $200 = sp + 3584|0;
 $201 = sp + 3580|0;
 $202 = sp + 3576|0;
 $203 = sp + 3572|0;
 $204 = sp + 4350|0;
 $205 = sp + 3568|0;
 $206 = sp + 3564|0;
 $207 = sp + 3560|0;
 $208 = sp + 3556|0;
 $209 = sp + 3552|0;
 $210 = sp + 3548|0;
 $211 = sp + 3544|0;
 $212 = sp + 3540|0;
 $213 = sp + 3536|0;
 $214 = sp + 3532|0;
 $215 = sp + 4349|0;
 $216 = sp + 3528|0;
 $217 = sp + 3524|0;
 $218 = sp + 3520|0;
 $219 = sp + 3516|0;
 $220 = sp + 3512|0;
 $221 = sp + 3508|0;
 $222 = sp + 3504|0;
 $223 = sp + 3500|0;
 $224 = sp + 3496|0;
 $225 = sp + 3492|0;
 $226 = sp + 3488|0;
 $227 = sp + 3484|0;
 $228 = sp + 3480|0;
 $229 = sp + 3476|0;
 $230 = sp + 3472|0;
 $231 = sp + 3468|0;
 $232 = sp + 3464|0;
 $233 = sp + 3460|0;
 $234 = sp + 3456|0;
 $235 = sp + 3452|0;
 $236 = sp + 184|0;
 $237 = sp + 3448|0;
 $238 = sp + 3444|0;
 $239 = sp + 3440|0;
 $240 = sp + 3436|0;
 $241 = sp + 4348|0;
 $242 = sp + 3432|0;
 $243 = sp + 3428|0;
 $244 = sp + 3424|0;
 $245 = sp + 3420|0;
 $246 = sp + 3416|0;
 $247 = sp + 3412|0;
 $248 = sp + 3408|0;
 $249 = sp + 3404|0;
 $250 = sp + 3400|0;
 $251 = sp + 3396|0;
 $252 = sp + 4347|0;
 $253 = sp + 3392|0;
 $254 = sp + 3388|0;
 $255 = sp + 3384|0;
 $256 = sp + 3380|0;
 $257 = sp + 3376|0;
 $258 = sp + 3372|0;
 $259 = sp + 3368|0;
 $260 = sp + 3364|0;
 $261 = sp + 3360|0;
 $262 = sp + 3356|0;
 $263 = sp + 3352|0;
 $264 = sp + 3348|0;
 $265 = sp + 3344|0;
 $266 = sp + 3340|0;
 $267 = sp + 176|0;
 $268 = sp + 3336|0;
 $269 = sp + 3332|0;
 $270 = sp + 3328|0;
 $271 = sp + 3324|0;
 $272 = sp + 3320|0;
 $273 = sp + 3316|0;
 $274 = sp + 3312|0;
 $275 = sp + 3308|0;
 $276 = sp + 3304|0;
 $277 = sp + 3300|0;
 $278 = sp + 3296|0;
 $279 = sp + 3292|0;
 $280 = sp + 3288|0;
 $281 = sp + 3284|0;
 $282 = sp + 3280|0;
 $283 = sp + 3276|0;
 $284 = sp + 3272|0;
 $285 = sp + 3268|0;
 $286 = sp + 3264|0;
 $287 = sp + 168|0;
 $288 = sp + 3260|0;
 $289 = sp + 3256|0;
 $290 = sp + 3252|0;
 $291 = sp + 3248|0;
 $292 = sp + 3244|0;
 $293 = sp + 3240|0;
 $294 = sp + 3236|0;
 $295 = sp + 3232|0;
 $296 = sp + 3228|0;
 $297 = sp + 3224|0;
 $298 = sp + 3220|0;
 $299 = sp + 3216|0;
 $300 = sp + 3212|0;
 $301 = sp + 160|0;
 $302 = sp + 3208|0;
 $303 = sp + 3204|0;
 $304 = sp + 3200|0;
 $305 = sp + 3196|0;
 $306 = sp + 3192|0;
 $307 = sp + 152|0;
 $308 = sp + 3188|0;
 $309 = sp + 3184|0;
 $310 = sp + 3180|0;
 $311 = sp + 3176|0;
 $312 = sp + 3172|0;
 $313 = sp + 3168|0;
 $314 = sp + 3164|0;
 $315 = sp + 3160|0;
 $316 = sp + 3156|0;
 $317 = sp + 3152|0;
 $318 = sp + 3148|0;
 $319 = sp + 3144|0;
 $320 = sp + 3140|0;
 $321 = sp + 3136|0;
 $322 = sp + 3132|0;
 $323 = sp + 3128|0;
 $324 = sp + 3124|0;
 $325 = sp + 3120|0;
 $326 = sp + 3116|0;
 $327 = sp + 3112|0;
 $328 = sp + 3108|0;
 $329 = sp + 3104|0;
 $330 = sp + 3100|0;
 $331 = sp + 3096|0;
 $332 = sp + 3092|0;
 $333 = sp + 3088|0;
 $334 = sp + 3084|0;
 $335 = sp + 3080|0;
 $336 = sp + 3076|0;
 $337 = sp + 3072|0;
 $338 = sp + 3068|0;
 $339 = sp + 3064|0;
 $340 = sp + 3060|0;
 $341 = sp + 3056|0;
 $342 = sp + 3052|0;
 $343 = sp + 3048|0;
 $344 = sp + 3044|0;
 $345 = sp + 3040|0;
 $346 = sp + 3036|0;
 $347 = sp + 3032|0;
 $348 = sp + 3028|0;
 $349 = sp + 3024|0;
 $350 = sp + 3020|0;
 $351 = sp + 3016|0;
 $352 = sp + 3012|0;
 $353 = sp + 3008|0;
 $354 = sp + 3004|0;
 $355 = sp + 3000|0;
 $356 = sp + 2996|0;
 $357 = sp + 2992|0;
 $358 = sp + 2988|0;
 $359 = sp + 2984|0;
 $360 = sp + 2980|0;
 $361 = sp + 2976|0;
 $362 = sp + 2972|0;
 $363 = sp + 2968|0;
 $364 = sp + 2964|0;
 $365 = sp + 2960|0;
 $366 = sp + 2956|0;
 $367 = sp + 2952|0;
 $368 = sp + 2948|0;
 $369 = sp + 2944|0;
 $370 = sp + 2940|0;
 $371 = sp + 2936|0;
 $372 = sp + 2932|0;
 $373 = sp + 2928|0;
 $374 = sp + 2924|0;
 $375 = sp + 2920|0;
 $376 = sp + 2916|0;
 $377 = sp + 2912|0;
 $378 = sp + 2908|0;
 $379 = sp + 2904|0;
 $380 = sp + 2900|0;
 $381 = sp + 2896|0;
 $382 = sp + 2892|0;
 $383 = sp + 2888|0;
 $384 = sp + 2884|0;
 $385 = sp + 2880|0;
 $386 = sp + 2876|0;
 $387 = sp + 144|0;
 $388 = sp + 2872|0;
 $389 = sp + 2868|0;
 $390 = sp + 2864|0;
 $391 = sp + 4346|0;
 $392 = sp + 2860|0;
 $393 = sp + 2856|0;
 $394 = sp + 2852|0;
 $395 = sp + 2848|0;
 $396 = sp + 136|0;
 $397 = sp + 2844|0;
 $398 = sp + 2840|0;
 $399 = sp + 2836|0;
 $400 = sp + 2832|0;
 $401 = sp + 2828|0;
 $402 = sp + 2824|0;
 $403 = sp + 2820|0;
 $404 = sp + 2816|0;
 $405 = sp + 2808|0;
 $406 = sp + 2804|0;
 $407 = sp + 2800|0;
 $408 = sp + 2796|0;
 $409 = sp + 2792|0;
 $410 = sp + 2788|0;
 $411 = sp + 2784|0;
 $412 = sp + 2780|0;
 $413 = sp + 2776|0;
 $414 = sp + 2772|0;
 $415 = sp + 2768|0;
 $416 = sp + 2764|0;
 $417 = sp + 2760|0;
 $418 = sp + 2756|0;
 $419 = sp + 2752|0;
 $420 = sp + 2748|0;
 $421 = sp + 2744|0;
 $422 = sp + 2740|0;
 $423 = sp + 2736|0;
 $424 = sp + 2732|0;
 $425 = sp + 2728|0;
 $426 = sp + 2724|0;
 $427 = sp + 2720|0;
 $428 = sp + 128|0;
 $429 = sp + 2716|0;
 $430 = sp + 2712|0;
 $431 = sp + 2708|0;
 $432 = sp + 2704|0;
 $433 = sp + 4345|0;
 $434 = sp + 2700|0;
 $435 = sp + 2696|0;
 $436 = sp + 2692|0;
 $437 = sp + 2688|0;
 $438 = sp + 2684|0;
 $439 = sp + 2680|0;
 $440 = sp + 2676|0;
 $441 = sp + 2672|0;
 $442 = sp + 2668|0;
 $443 = sp + 2664|0;
 $444 = sp + 4344|0;
 $445 = sp + 2660|0;
 $446 = sp + 2656|0;
 $447 = sp + 2652|0;
 $448 = sp + 2648|0;
 $449 = sp + 2644|0;
 $450 = sp + 2640|0;
 $451 = sp + 2636|0;
 $452 = sp + 2632|0;
 $453 = sp + 2628|0;
 $454 = sp + 2624|0;
 $455 = sp + 2620|0;
 $456 = sp + 2616|0;
 $457 = sp + 2612|0;
 $458 = sp + 120|0;
 $459 = sp + 2608|0;
 $460 = sp + 2604|0;
 $461 = sp + 2600|0;
 $462 = sp + 2596|0;
 $463 = sp + 4343|0;
 $464 = sp + 2592|0;
 $465 = sp + 2588|0;
 $466 = sp + 2584|0;
 $467 = sp + 2580|0;
 $468 = sp + 2576|0;
 $469 = sp + 2572|0;
 $470 = sp + 2568|0;
 $471 = sp + 2564|0;
 $472 = sp + 2560|0;
 $473 = sp + 2556|0;
 $474 = sp + 4342|0;
 $475 = sp + 2552|0;
 $476 = sp + 2548|0;
 $477 = sp + 2544|0;
 $478 = sp + 2540|0;
 $479 = sp + 2536|0;
 $480 = sp + 2532|0;
 $481 = sp + 2528|0;
 $482 = sp + 2524|0;
 $483 = sp + 2520|0;
 $484 = sp + 2516|0;
 $485 = sp + 2512|0;
 $486 = sp + 2508|0;
 $487 = sp + 2504|0;
 $488 = sp + 112|0;
 $489 = sp + 2500|0;
 $490 = sp + 2496|0;
 $491 = sp + 2492|0;
 $492 = sp + 2488|0;
 $493 = sp + 4341|0;
 $494 = sp + 2484|0;
 $495 = sp + 2480|0;
 $496 = sp + 2476|0;
 $497 = sp + 2472|0;
 $498 = sp + 2468|0;
 $499 = sp + 2464|0;
 $500 = sp + 2460|0;
 $501 = sp + 2456|0;
 $502 = sp + 2452|0;
 $503 = sp + 2448|0;
 $504 = sp + 4340|0;
 $505 = sp + 2444|0;
 $506 = sp + 2440|0;
 $507 = sp + 2436|0;
 $508 = sp + 2432|0;
 $509 = sp + 2428|0;
 $510 = sp + 2424|0;
 $511 = sp + 2420|0;
 $512 = sp + 2416|0;
 $513 = sp + 2412|0;
 $514 = sp + 2408|0;
 $515 = sp + 2404|0;
 $516 = sp + 2400|0;
 $517 = sp + 2396|0;
 $518 = sp + 104|0;
 $519 = sp + 2392|0;
 $520 = sp + 2388|0;
 $521 = sp + 2384|0;
 $522 = sp + 2380|0;
 $523 = sp + 4339|0;
 $524 = sp + 2376|0;
 $525 = sp + 2372|0;
 $526 = sp + 2368|0;
 $527 = sp + 2364|0;
 $528 = sp + 2360|0;
 $529 = sp + 2356|0;
 $530 = sp + 2352|0;
 $531 = sp + 2348|0;
 $532 = sp + 2344|0;
 $533 = sp + 2340|0;
 $534 = sp + 4338|0;
 $535 = sp + 2336|0;
 $536 = sp + 2332|0;
 $537 = sp + 2328|0;
 $538 = sp + 2324|0;
 $539 = sp + 2320|0;
 $540 = sp + 2316|0;
 $541 = sp + 2312|0;
 $542 = sp + 2308|0;
 $543 = sp + 2304|0;
 $544 = sp + 2300|0;
 $545 = sp + 2296|0;
 $546 = sp + 2292|0;
 $547 = sp + 2288|0;
 $548 = sp + 96|0;
 $549 = sp + 2284|0;
 $550 = sp + 2280|0;
 $551 = sp + 2276|0;
 $552 = sp + 2272|0;
 $553 = sp + 4337|0;
 $554 = sp + 2268|0;
 $555 = sp + 2264|0;
 $556 = sp + 2260|0;
 $557 = sp + 2256|0;
 $558 = sp + 2252|0;
 $559 = sp + 2248|0;
 $560 = sp + 2244|0;
 $561 = sp + 2240|0;
 $562 = sp + 2236|0;
 $563 = sp + 2232|0;
 $564 = sp + 4336|0;
 $565 = sp + 2228|0;
 $566 = sp + 2224|0;
 $567 = sp + 2220|0;
 $568 = sp + 2216|0;
 $569 = sp + 2212|0;
 $570 = sp + 2208|0;
 $571 = sp + 2204|0;
 $572 = sp + 2200|0;
 $573 = sp + 2196|0;
 $574 = sp + 2192|0;
 $575 = sp + 2188|0;
 $576 = sp + 2184|0;
 $577 = sp + 2180|0;
 $578 = sp + 88|0;
 $579 = sp + 2176|0;
 $580 = sp + 2172|0;
 $581 = sp + 2168|0;
 $582 = sp + 2164|0;
 $583 = sp + 4335|0;
 $584 = sp + 2160|0;
 $585 = sp + 2156|0;
 $586 = sp + 2152|0;
 $587 = sp + 2148|0;
 $588 = sp + 2144|0;
 $589 = sp + 2140|0;
 $590 = sp + 2136|0;
 $591 = sp + 2132|0;
 $592 = sp + 2128|0;
 $593 = sp + 2124|0;
 $594 = sp + 4334|0;
 $595 = sp + 2120|0;
 $596 = sp + 2116|0;
 $597 = sp + 2112|0;
 $598 = sp + 2108|0;
 $599 = sp + 2104|0;
 $600 = sp + 2100|0;
 $601 = sp + 2096|0;
 $602 = sp + 2092|0;
 $603 = sp + 2088|0;
 $604 = sp + 2084|0;
 $605 = sp + 2080|0;
 $606 = sp + 2076|0;
 $607 = sp + 2072|0;
 $608 = sp + 80|0;
 $609 = sp + 2068|0;
 $610 = sp + 2064|0;
 $611 = sp + 2060|0;
 $612 = sp + 2056|0;
 $613 = sp + 4333|0;
 $614 = sp + 2052|0;
 $615 = sp + 2048|0;
 $616 = sp + 2044|0;
 $617 = sp + 2040|0;
 $618 = sp + 2036|0;
 $619 = sp + 2032|0;
 $620 = sp + 2028|0;
 $621 = sp + 2024|0;
 $622 = sp + 2020|0;
 $623 = sp + 2016|0;
 $624 = sp + 4332|0;
 $625 = sp + 2012|0;
 $626 = sp + 2008|0;
 $627 = sp + 2004|0;
 $628 = sp + 2000|0;
 $629 = sp + 1996|0;
 $630 = sp + 1992|0;
 $631 = sp + 1988|0;
 $632 = sp + 1984|0;
 $633 = sp + 1980|0;
 $634 = sp + 1976|0;
 $635 = sp + 1972|0;
 $636 = sp + 1968|0;
 $637 = sp + 1964|0;
 $638 = sp + 72|0;
 $639 = sp + 1960|0;
 $640 = sp + 1956|0;
 $641 = sp + 1952|0;
 $642 = sp + 1948|0;
 $643 = sp + 4331|0;
 $644 = sp + 1944|0;
 $645 = sp + 1940|0;
 $646 = sp + 1936|0;
 $647 = sp + 1932|0;
 $648 = sp + 1928|0;
 $649 = sp + 1924|0;
 $650 = sp + 1920|0;
 $651 = sp + 1916|0;
 $652 = sp + 1912|0;
 $653 = sp + 1908|0;
 $654 = sp + 4330|0;
 $655 = sp + 1904|0;
 $656 = sp + 1900|0;
 $657 = sp + 1896|0;
 $658 = sp + 1892|0;
 $659 = sp + 1888|0;
 $660 = sp + 1884|0;
 $661 = sp + 1880|0;
 $662 = sp + 1876|0;
 $663 = sp + 1872|0;
 $664 = sp + 1868|0;
 $665 = sp + 1864|0;
 $666 = sp + 1860|0;
 $667 = sp + 1856|0;
 $668 = sp + 64|0;
 $669 = sp + 1852|0;
 $670 = sp + 1848|0;
 $671 = sp + 1844|0;
 $672 = sp + 1840|0;
 $673 = sp + 4329|0;
 $674 = sp + 1836|0;
 $675 = sp + 1832|0;
 $676 = sp + 1828|0;
 $677 = sp + 1824|0;
 $678 = sp + 1820|0;
 $679 = sp + 1816|0;
 $680 = sp + 1812|0;
 $681 = sp + 1808|0;
 $682 = sp + 1804|0;
 $683 = sp + 1800|0;
 $684 = sp + 4328|0;
 $685 = sp + 1796|0;
 $686 = sp + 1792|0;
 $687 = sp + 1788|0;
 $688 = sp + 1784|0;
 $689 = sp + 1780|0;
 $690 = sp + 1776|0;
 $691 = sp + 1772|0;
 $692 = sp + 1768|0;
 $693 = sp + 1764|0;
 $694 = sp + 1760|0;
 $695 = sp + 1756|0;
 $696 = sp + 1752|0;
 $697 = sp + 1748|0;
 $698 = sp + 56|0;
 $699 = sp + 1744|0;
 $700 = sp + 1740|0;
 $701 = sp + 1736|0;
 $702 = sp + 1732|0;
 $703 = sp + 4327|0;
 $704 = sp + 1728|0;
 $705 = sp + 1724|0;
 $706 = sp + 1720|0;
 $707 = sp + 1716|0;
 $708 = sp + 1712|0;
 $709 = sp + 1708|0;
 $710 = sp + 1704|0;
 $711 = sp + 1700|0;
 $712 = sp + 1696|0;
 $713 = sp + 1692|0;
 $714 = sp + 4326|0;
 $715 = sp + 1688|0;
 $716 = sp + 1684|0;
 $717 = sp + 1680|0;
 $718 = sp + 1676|0;
 $719 = sp + 1672|0;
 $720 = sp + 1668|0;
 $721 = sp + 1664|0;
 $722 = sp + 1660|0;
 $723 = sp + 1656|0;
 $724 = sp + 1652|0;
 $725 = sp + 1648|0;
 $726 = sp + 1644|0;
 $727 = sp + 1640|0;
 $728 = sp + 48|0;
 $729 = sp + 1636|0;
 $730 = sp + 1632|0;
 $731 = sp + 1628|0;
 $732 = sp + 1624|0;
 $733 = sp + 4325|0;
 $734 = sp + 1620|0;
 $735 = sp + 1616|0;
 $736 = sp + 1612|0;
 $737 = sp + 1608|0;
 $738 = sp + 1604|0;
 $739 = sp + 1600|0;
 $740 = sp + 1596|0;
 $741 = sp + 1592|0;
 $742 = sp + 1588|0;
 $743 = sp + 1584|0;
 $744 = sp + 4324|0;
 $745 = sp + 1580|0;
 $746 = sp + 1576|0;
 $747 = sp + 1572|0;
 $748 = sp + 1568|0;
 $749 = sp + 1564|0;
 $750 = sp + 1560|0;
 $751 = sp + 1556|0;
 $752 = sp + 1552|0;
 $753 = sp + 1548|0;
 $754 = sp + 1544|0;
 $755 = sp + 1540|0;
 $756 = sp + 1536|0;
 $757 = sp + 1532|0;
 $758 = sp + 40|0;
 $759 = sp + 1528|0;
 $760 = sp + 1524|0;
 $761 = sp + 1520|0;
 $762 = sp + 1516|0;
 $763 = sp + 4323|0;
 $764 = sp + 1512|0;
 $765 = sp + 1508|0;
 $766 = sp + 1504|0;
 $767 = sp + 1500|0;
 $768 = sp + 1496|0;
 $769 = sp + 1492|0;
 $770 = sp + 1488|0;
 $771 = sp + 1484|0;
 $772 = sp + 1480|0;
 $773 = sp + 1476|0;
 $774 = sp + 4322|0;
 $775 = sp + 1472|0;
 $776 = sp + 1468|0;
 $777 = sp + 1464|0;
 $778 = sp + 1460|0;
 $779 = sp + 1456|0;
 $780 = sp + 1452|0;
 $781 = sp + 1448|0;
 $782 = sp + 1444|0;
 $783 = sp + 1440|0;
 $784 = sp + 1436|0;
 $785 = sp + 1432|0;
 $786 = sp + 1428|0;
 $787 = sp + 1424|0;
 $788 = sp + 32|0;
 $789 = sp + 1420|0;
 $790 = sp + 1416|0;
 $791 = sp + 1412|0;
 $792 = sp + 1408|0;
 $793 = sp + 4321|0;
 $794 = sp + 1404|0;
 $795 = sp + 1400|0;
 $796 = sp + 1396|0;
 $797 = sp + 1392|0;
 $798 = sp + 1388|0;
 $799 = sp + 1384|0;
 $800 = sp + 1380|0;
 $801 = sp + 1376|0;
 $802 = sp + 1372|0;
 $803 = sp + 1368|0;
 $804 = sp + 4320|0;
 $805 = sp + 1364|0;
 $806 = sp + 1360|0;
 $807 = sp + 1356|0;
 $808 = sp + 1352|0;
 $809 = sp + 1348|0;
 $810 = sp + 1344|0;
 $811 = sp + 1340|0;
 $812 = sp + 1336|0;
 $813 = sp + 1332|0;
 $814 = sp + 1328|0;
 $815 = sp + 1324|0;
 $816 = sp + 1320|0;
 $817 = sp + 1316|0;
 $818 = sp + 24|0;
 $819 = sp + 1312|0;
 $820 = sp + 1308|0;
 $821 = sp + 1304|0;
 $822 = sp + 1300|0;
 $823 = sp + 4319|0;
 $824 = sp + 1296|0;
 $825 = sp + 1292|0;
 $826 = sp + 1288|0;
 $827 = sp + 1284|0;
 $828 = sp + 1280|0;
 $829 = sp + 1276|0;
 $830 = sp + 1272|0;
 $831 = sp + 1268|0;
 $832 = sp + 1264|0;
 $833 = sp + 1260|0;
 $834 = sp + 4318|0;
 $835 = sp + 1256|0;
 $836 = sp + 1252|0;
 $837 = sp + 1248|0;
 $838 = sp + 1244|0;
 $839 = sp + 1240|0;
 $840 = sp + 1236|0;
 $841 = sp + 1232|0;
 $842 = sp + 1228|0;
 $843 = sp + 1224|0;
 $844 = sp + 1220|0;
 $845 = sp + 1216|0;
 $846 = sp + 1212|0;
 $847 = sp + 1208|0;
 $848 = sp + 16|0;
 $849 = sp + 1204|0;
 $850 = sp + 1200|0;
 $851 = sp + 1196|0;
 $852 = sp + 1192|0;
 $853 = sp + 4317|0;
 $854 = sp + 1188|0;
 $855 = sp + 1184|0;
 $856 = sp + 1180|0;
 $857 = sp + 1176|0;
 $858 = sp + 1172|0;
 $859 = sp + 1168|0;
 $860 = sp + 1164|0;
 $861 = sp + 1160|0;
 $862 = sp + 1156|0;
 $863 = sp + 1152|0;
 $864 = sp + 4316|0;
 $865 = sp + 1148|0;
 $866 = sp + 1144|0;
 $867 = sp + 1140|0;
 $868 = sp + 1136|0;
 $869 = sp + 1132|0;
 $870 = sp + 1128|0;
 $871 = sp + 1124|0;
 $872 = sp + 1120|0;
 $873 = sp + 1116|0;
 $874 = sp + 1112|0;
 $875 = sp + 1108|0;
 $876 = sp + 1104|0;
 $877 = sp + 1100|0;
 $878 = sp + 8|0;
 $879 = sp + 1096|0;
 $880 = sp + 1092|0;
 $881 = sp + 1088|0;
 $882 = sp + 1084|0;
 $883 = sp + 4315|0;
 $884 = sp + 1080|0;
 $885 = sp + 1076|0;
 $886 = sp + 1072|0;
 $887 = sp + 1068|0;
 $888 = sp + 1064|0;
 $889 = sp + 1060|0;
 $890 = sp + 1056|0;
 $891 = sp + 1052|0;
 $892 = sp + 1048|0;
 $893 = sp + 1044|0;
 $894 = sp + 4314|0;
 $895 = sp + 1040|0;
 $896 = sp;
 $897 = sp + 1036|0;
 $898 = sp + 4313|0;
 $899 = sp + 4312|0;
 $900 = sp + 1032|0;
 $901 = sp + 1028|0;
 $902 = sp + 1024|0;
 $903 = sp + 1020|0;
 $904 = sp + 1016|0;
 $905 = sp + 1012|0;
 $906 = sp + 1008|0;
 $907 = sp + 1004|0;
 $908 = sp + 1000|0;
 $909 = sp + 996|0;
 $910 = sp + 992|0;
 $911 = sp + 988|0;
 $912 = sp + 984|0;
 $913 = sp + 980|0;
 $914 = sp + 976|0;
 $915 = sp + 972|0;
 $916 = sp + 968|0;
 $917 = sp + 964|0;
 $918 = sp + 960|0;
 $919 = sp + 956|0;
 $920 = sp + 952|0;
 $921 = sp + 948|0;
 $922 = sp + 944|0;
 $923 = sp + 940|0;
 $924 = sp + 936|0;
 $925 = sp + 932|0;
 $926 = sp + 928|0;
 $927 = sp + 924|0;
 $928 = sp + 920|0;
 $929 = sp + 916|0;
 $930 = sp + 912|0;
 $931 = sp + 908|0;
 $932 = sp + 904|0;
 $933 = sp + 900|0;
 $934 = sp + 896|0;
 $935 = sp + 892|0;
 $936 = sp + 888|0;
 $937 = sp + 884|0;
 $938 = sp + 880|0;
 $939 = sp + 876|0;
 $940 = sp + 872|0;
 $941 = sp + 868|0;
 $942 = sp + 864|0;
 $943 = sp + 860|0;
 $944 = sp + 856|0;
 $945 = sp + 852|0;
 $946 = sp + 848|0;
 $947 = sp + 844|0;
 $948 = sp + 840|0;
 $949 = sp + 836|0;
 $950 = sp + 832|0;
 $951 = sp + 828|0;
 $952 = sp + 824|0;
 $953 = sp + 820|0;
 $954 = sp + 816|0;
 $955 = sp + 812|0;
 $956 = sp + 808|0;
 $957 = sp + 804|0;
 $958 = sp + 800|0;
 $959 = sp + 796|0;
 $960 = sp + 792|0;
 $961 = sp + 788|0;
 $962 = sp + 784|0;
 $963 = sp + 780|0;
 $964 = sp + 776|0;
 $965 = sp + 772|0;
 $966 = sp + 768|0;
 $967 = sp + 764|0;
 $968 = sp + 760|0;
 $969 = sp + 756|0;
 $970 = sp + 752|0;
 $971 = sp + 748|0;
 $972 = sp + 744|0;
 $973 = sp + 740|0;
 $974 = sp + 736|0;
 $975 = sp + 732|0;
 $976 = sp + 728|0;
 $977 = sp + 724|0;
 $978 = sp + 720|0;
 $979 = sp + 716|0;
 $980 = sp + 712|0;
 $981 = sp + 708|0;
 $982 = sp + 704|0;
 $983 = sp + 700|0;
 $984 = sp + 696|0;
 $985 = sp + 692|0;
 $986 = sp + 688|0;
 $987 = sp + 684|0;
 $988 = sp + 680|0;
 $989 = sp + 676|0;
 $990 = sp + 672|0;
 $991 = sp + 668|0;
 $992 = sp + 664|0;
 $993 = sp + 660|0;
 $994 = sp + 656|0;
 $995 = sp + 652|0;
 $996 = sp + 648|0;
 $997 = sp + 644|0;
 $998 = sp + 640|0;
 $999 = sp + 636|0;
 $1000 = sp + 632|0;
 $1001 = sp + 628|0;
 $1002 = sp + 624|0;
 $1003 = sp + 620|0;
 $1004 = sp + 616|0;
 $1005 = sp + 604|0;
 $1006 = sp + 592|0;
 $1007 = sp + 588|0;
 $1008 = sp + 584|0;
 $1009 = sp + 572|0;
 $1010 = sp + 560|0;
 $1011 = sp + 548|0;
 $1012 = sp + 536|0;
 $1013 = sp + 524|0;
 $1014 = sp + 512|0;
 $1015 = sp + 500|0;
 $1016 = sp + 488|0;
 $1017 = sp + 476|0;
 $1018 = sp + 472|0;
 $1019 = sp + 468|0;
 $1020 = sp + 464|0;
 $1021 = sp + 460|0;
 $1022 = sp + 456|0;
 $1023 = sp + 452|0;
 $1024 = sp + 448|0;
 $1025 = sp + 436|0;
 $1026 = sp + 424|0;
 $1027 = sp + 412|0;
 $1028 = sp + 408|0;
 $1029 = sp + 404|0;
 $1030 = sp + 392|0;
 $1031 = sp + 384|0;
 $1032 = sp + 376|0;
 $1033 = sp + 360|0;
 $1034 = sp + 348|0;
 $1035 = sp + 336|0;
 $1036 = sp + 324|0;
 $1037 = sp + 312|0;
 $1038 = sp + 300|0;
 $1039 = sp + 296|0;
 $1040 = sp + 292|0;
 $1041 = sp + 288|0;
 $1042 = sp + 284|0;
 $1043 = sp + 280|0;
 $1044 = sp + 276|0;
 $1045 = sp + 272|0;
 $1046 = sp + 268|0;
 $1047 = sp + 264|0;
 $1048 = sp + 260|0;
 $1049 = sp + 256|0;
 $1050 = sp + 252|0;
 $1051 = sp + 248|0;
 $1052 = sp + 240|0;
 HEAP32[$1002>>2] = $0;
 HEAP32[$1003>>2] = $1;
 HEAP32[$1004>>2] = $2;
 $1053 = HEAP32[$1002>>2]|0;
 $1054 = HEAP32[$1003>>2]|0;
 $1055 = (__ZNK3sap2ve4core10Thrustline7IsValidEv($1054)|0);
 if (!($1055)) {
  STACKTOP = sp;return;
 }
 $1056 = HEAP32[$1003>>2]|0;
 __ZNK3sap2ve4core10Thrustline13PrincipleAxisEv($1005,$1056);
 __ZNSt3__26vectorIfNS_9allocatorIfEEEC2Em($1006,3);
 $1057 = +HEAPF32[$1005>>2];
 HEAP32[$1000>>2] = $1006;
 HEAP32[$1001>>2] = 0;
 $1058 = HEAP32[$1000>>2]|0;
 $1059 = HEAP32[$1058>>2]|0;
 $1060 = HEAP32[$1001>>2]|0;
 $1061 = (($1059) + ($1060<<2)|0);
 HEAPF32[$1061>>2] = $1057;
 $1062 = ((($1005)) + 4|0);
 $1063 = +HEAPF32[$1062>>2];
 HEAP32[$998>>2] = $1006;
 HEAP32[$999>>2] = 1;
 $1064 = HEAP32[$998>>2]|0;
 $1065 = HEAP32[$1064>>2]|0;
 $1066 = HEAP32[$999>>2]|0;
 $1067 = (($1065) + ($1066<<2)|0);
 HEAPF32[$1067>>2] = $1063;
 $1068 = ((($1005)) + 8|0);
 $1069 = +HEAPF32[$1068>>2];
 HEAP32[$996>>2] = $1006;
 HEAP32[$997>>2] = 2;
 $1070 = HEAP32[$996>>2]|0;
 $1071 = HEAP32[$1070>>2]|0;
 $1072 = HEAP32[$997>>2]|0;
 $1073 = (($1071) + ($1072<<2)|0);
 HEAPF32[$1073>>2] = $1069;
 $1074 = HEAP32[$1003>>2]|0;
 $1075 = (__ZNK3sap2ve4core10Thrustline11GetMaterialEv($1074)|0);
 $1076 = (__ZN17DataModelImporter19GetOrCreateMaterialEPKN3sap2ve4core8MaterialE($1053,$1075)|0);
 HEAP32[$1007>>2] = $1076;
 $1077 = HEAP32[$1003>>2]|0;
 $1078 = (__ZNK3sap2ve4core10Thrustline9ItemCountEv($1077)|0);
 HEAP32[$1008>>2] = $1078;
 HEAP32[$995>>2] = $1009;
 $1079 = HEAP32[$995>>2]|0;
 HEAP32[$993>>2] = $1079;
 $1080 = HEAP32[$993>>2]|0;
 HEAP32[$992>>2] = $1080;
 HEAP32[$1080>>2] = 0;
 $1081 = ((($1080)) + 4|0);
 HEAP32[$1081>>2] = 0;
 $1082 = ((($1080)) + 8|0);
 HEAP32[$994>>2] = 0;
 HEAP32[$990>>2] = $1082;
 HEAP32[$991>>2] = $994;
 $1083 = HEAP32[$990>>2]|0;
 $1084 = HEAP32[$991>>2]|0;
 HEAP32[$989>>2] = $1084;
 $1085 = HEAP32[$989>>2]|0;
 HEAP32[$985>>2] = $1083;
 HEAP32[$986>>2] = $1085;
 $1086 = HEAP32[$985>>2]|0;
 $1087 = HEAP32[$986>>2]|0;
 HEAP32[$984>>2] = $1087;
 HEAP32[$1086>>2] = 0;
 HEAP32[$988>>2] = $1083;
 $1088 = HEAP32[$988>>2]|0;
 HEAP32[$987>>2] = $1088;
 HEAP32[$983>>2] = $1010;
 $1089 = HEAP32[$983>>2]|0;
 HEAP32[$981>>2] = $1089;
 $1090 = HEAP32[$981>>2]|0;
 HEAP32[$980>>2] = $1090;
 HEAP32[$1090>>2] = 0;
 $1091 = ((($1090)) + 4|0);
 HEAP32[$1091>>2] = 0;
 $1092 = ((($1090)) + 8|0);
 HEAP32[$982>>2] = 0;
 HEAP32[$978>>2] = $1092;
 HEAP32[$979>>2] = $982;
 $1093 = HEAP32[$978>>2]|0;
 $1094 = HEAP32[$979>>2]|0;
 HEAP32[$977>>2] = $1094;
 $1095 = HEAP32[$977>>2]|0;
 HEAP32[$973>>2] = $1093;
 HEAP32[$974>>2] = $1095;
 $1096 = HEAP32[$973>>2]|0;
 $1097 = HEAP32[$974>>2]|0;
 HEAP32[$972>>2] = $1097;
 HEAP32[$1096>>2] = 0;
 HEAP32[$976>>2] = $1093;
 $1098 = HEAP32[$976>>2]|0;
 HEAP32[$975>>2] = $1098;
 HEAP32[$971>>2] = $1011;
 $1099 = HEAP32[$971>>2]|0;
 HEAP32[$969>>2] = $1099;
 $1100 = HEAP32[$969>>2]|0;
 HEAP32[$968>>2] = $1100;
 HEAP32[$1100>>2] = 0;
 $1101 = ((($1100)) + 4|0);
 HEAP32[$1101>>2] = 0;
 $1102 = ((($1100)) + 8|0);
 HEAP32[$970>>2] = 0;
 HEAP32[$966>>2] = $1102;
 HEAP32[$967>>2] = $970;
 $1103 = HEAP32[$966>>2]|0;
 $1104 = HEAP32[$967>>2]|0;
 HEAP32[$965>>2] = $1104;
 $1105 = HEAP32[$965>>2]|0;
 HEAP32[$961>>2] = $1103;
 HEAP32[$962>>2] = $1105;
 $1106 = HEAP32[$961>>2]|0;
 $1107 = HEAP32[$962>>2]|0;
 HEAP32[$960>>2] = $1107;
 HEAP32[$1106>>2] = 0;
 HEAP32[$964>>2] = $1103;
 $1108 = HEAP32[$964>>2]|0;
 HEAP32[$963>>2] = $1108;
 HEAP32[$959>>2] = $1012;
 $1109 = HEAP32[$959>>2]|0;
 HEAP32[$957>>2] = $1109;
 $1110 = HEAP32[$957>>2]|0;
 HEAP32[$956>>2] = $1110;
 HEAP32[$1110>>2] = 0;
 $1111 = ((($1110)) + 4|0);
 HEAP32[$1111>>2] = 0;
 $1112 = ((($1110)) + 8|0);
 HEAP32[$958>>2] = 0;
 HEAP32[$954>>2] = $1112;
 HEAP32[$955>>2] = $958;
 $1113 = HEAP32[$954>>2]|0;
 $1114 = HEAP32[$955>>2]|0;
 HEAP32[$953>>2] = $1114;
 $1115 = HEAP32[$953>>2]|0;
 HEAP32[$949>>2] = $1113;
 HEAP32[$950>>2] = $1115;
 $1116 = HEAP32[$949>>2]|0;
 $1117 = HEAP32[$950>>2]|0;
 HEAP32[$948>>2] = $1117;
 HEAP32[$1116>>2] = 0;
 HEAP32[$952>>2] = $1113;
 $1118 = HEAP32[$952>>2]|0;
 HEAP32[$951>>2] = $1118;
 HEAP32[$947>>2] = $1013;
 $1119 = HEAP32[$947>>2]|0;
 HEAP32[$945>>2] = $1119;
 $1120 = HEAP32[$945>>2]|0;
 HEAP32[$944>>2] = $1120;
 HEAP32[$1120>>2] = 0;
 $1121 = ((($1120)) + 4|0);
 HEAP32[$1121>>2] = 0;
 $1122 = ((($1120)) + 8|0);
 HEAP32[$946>>2] = 0;
 HEAP32[$942>>2] = $1122;
 HEAP32[$943>>2] = $946;
 $1123 = HEAP32[$942>>2]|0;
 $1124 = HEAP32[$943>>2]|0;
 HEAP32[$941>>2] = $1124;
 $1125 = HEAP32[$941>>2]|0;
 HEAP32[$937>>2] = $1123;
 HEAP32[$938>>2] = $1125;
 $1126 = HEAP32[$937>>2]|0;
 $1127 = HEAP32[$938>>2]|0;
 HEAP32[$936>>2] = $1127;
 HEAP32[$1126>>2] = 0;
 HEAP32[$940>>2] = $1123;
 $1128 = HEAP32[$940>>2]|0;
 HEAP32[$939>>2] = $1128;
 HEAP32[$935>>2] = $1014;
 $1129 = HEAP32[$935>>2]|0;
 HEAP32[$933>>2] = $1129;
 $1130 = HEAP32[$933>>2]|0;
 HEAP32[$932>>2] = $1130;
 HEAP32[$1130>>2] = 0;
 $1131 = ((($1130)) + 4|0);
 HEAP32[$1131>>2] = 0;
 $1132 = ((($1130)) + 8|0);
 HEAP32[$934>>2] = 0;
 HEAP32[$930>>2] = $1132;
 HEAP32[$931>>2] = $934;
 $1133 = HEAP32[$930>>2]|0;
 $1134 = HEAP32[$931>>2]|0;
 HEAP32[$929>>2] = $1134;
 $1135 = HEAP32[$929>>2]|0;
 HEAP32[$925>>2] = $1133;
 HEAP32[$926>>2] = $1135;
 $1136 = HEAP32[$925>>2]|0;
 $1137 = HEAP32[$926>>2]|0;
 HEAP32[$924>>2] = $1137;
 HEAP32[$1136>>2] = 0;
 HEAP32[$928>>2] = $1133;
 $1138 = HEAP32[$928>>2]|0;
 HEAP32[$927>>2] = $1138;
 HEAP32[$923>>2] = $1015;
 $1139 = HEAP32[$923>>2]|0;
 HEAP32[$921>>2] = $1139;
 $1140 = HEAP32[$921>>2]|0;
 HEAP32[$920>>2] = $1140;
 HEAP32[$1140>>2] = 0;
 $1141 = ((($1140)) + 4|0);
 HEAP32[$1141>>2] = 0;
 $1142 = ((($1140)) + 8|0);
 HEAP32[$922>>2] = 0;
 HEAP32[$918>>2] = $1142;
 HEAP32[$919>>2] = $922;
 $1143 = HEAP32[$918>>2]|0;
 $1144 = HEAP32[$919>>2]|0;
 HEAP32[$917>>2] = $1144;
 $1145 = HEAP32[$917>>2]|0;
 HEAP32[$913>>2] = $1143;
 HEAP32[$914>>2] = $1145;
 $1146 = HEAP32[$913>>2]|0;
 $1147 = HEAP32[$914>>2]|0;
 HEAP32[$912>>2] = $1147;
 HEAP32[$1146>>2] = 0;
 HEAP32[$916>>2] = $1143;
 $1148 = HEAP32[$916>>2]|0;
 HEAP32[$915>>2] = $1148;
 HEAP32[$911>>2] = $1016;
 $1149 = HEAP32[$911>>2]|0;
 HEAP32[$909>>2] = $1149;
 $1150 = HEAP32[$909>>2]|0;
 HEAP32[$908>>2] = $1150;
 HEAP32[$1150>>2] = 0;
 $1151 = ((($1150)) + 4|0);
 HEAP32[$1151>>2] = 0;
 $1152 = ((($1150)) + 8|0);
 HEAP32[$910>>2] = 0;
 HEAP32[$906>>2] = $1152;
 HEAP32[$907>>2] = $910;
 $1153 = HEAP32[$906>>2]|0;
 $1154 = HEAP32[$907>>2]|0;
 HEAP32[$905>>2] = $1154;
 $1155 = HEAP32[$905>>2]|0;
 HEAP32[$901>>2] = $1153;
 HEAP32[$902>>2] = $1155;
 $1156 = HEAP32[$901>>2]|0;
 $1157 = HEAP32[$902>>2]|0;
 HEAP32[$900>>2] = $1157;
 HEAP32[$1156>>2] = 0;
 HEAP32[$904>>2] = $1153;
 $1158 = HEAP32[$904>>2]|0;
 HEAP32[$903>>2] = $1158;
 HEAP32[$897>>2] = $1017;
 $1159 = HEAP32[$897>>2]|0;
 ;HEAP8[$896>>0]=HEAP8[$899>>0]|0;
 HEAP32[$895>>2] = $898;
 __ZNSt3__26__treeINS_12__value_typeIPN3sap2ve4core14ThrustlineItemEiEENS_19__map_value_compareIS6_S7_NS_4lessIS6_EELb1EEENS_9allocatorIS7_EEEC2ERKSB_($1159,$898);
 HEAP32[$1018>>2] = 0;
 while(1) {
  $1160 = HEAP32[$1018>>2]|0;
  $1161 = HEAP32[$1008>>2]|0;
  $1162 = ($1160>>>0)<($1161>>>0);
  if (!($1162)) {
   break;
  }
  $1163 = HEAP32[$1003>>2]|0;
  $1164 = HEAP32[$1018>>2]|0;
  $1165 = (__ZNK3sap2ve4core10Thrustline4ItemEm($1163,$1164)|0);
  HEAP32[$1019>>2] = $1165;
  $1166 = HEAP32[$1019>>2]|0;
  $1167 = (__ZNK3sap2ve4core14ThrustlineItem6TargetEv($1166)|0);
  HEAP32[$1020>>2] = $1167;
  $1168 = ((($1053)) + 4|0);
  $1169 = (__ZNK3sap2ve4core4IPtrINS1_6ISceneEEptEv($1168)|0);
  $1170 = HEAP32[$1169>>2]|0;
  $1171 = ((($1170)) + 44|0);
  $1172 = HEAP32[$1171>>2]|0;
  $1173 = HEAP32[$1020>>2]|0;
  $1174 = (FUNCTION_TABLE_iii[$1172 & 1023]($1169,$1173)|0);
  HEAP32[$1021>>2] = $1174;
  HEAP32[$892>>2] = $1009;
  HEAP32[$893>>2] = $1021;
  $1175 = HEAP32[$892>>2]|0;
  $1176 = ((($1175)) + 4|0);
  $1177 = HEAP32[$1176>>2]|0;
  HEAP32[$891>>2] = $1175;
  $1178 = HEAP32[$891>>2]|0;
  $1179 = ((($1178)) + 8|0);
  HEAP32[$890>>2] = $1179;
  $1180 = HEAP32[$890>>2]|0;
  HEAP32[$889>>2] = $1180;
  $1181 = HEAP32[$889>>2]|0;
  $1182 = HEAP32[$1181>>2]|0;
  $1183 = ($1177>>>0)<($1182>>>0);
  if ($1183) {
   HEAP32[$886>>2] = $894;
   HEAP32[$887>>2] = $1175;
   HEAP32[$888>>2] = 1;
   HEAP32[$867>>2] = $1175;
   $1184 = HEAP32[$867>>2]|0;
   $1185 = ((($1184)) + 8|0);
   HEAP32[$866>>2] = $1185;
   $1186 = HEAP32[$866>>2]|0;
   HEAP32[$865>>2] = $1186;
   $1187 = HEAP32[$865>>2]|0;
   $1188 = ((($1175)) + 4|0);
   $1189 = HEAP32[$1188>>2]|0;
   HEAP32[$868>>2] = $1189;
   $1190 = HEAP32[$868>>2]|0;
   $1191 = HEAP32[$893>>2]|0;
   HEAP32[$869>>2] = $1191;
   $1192 = HEAP32[$869>>2]|0;
   HEAP32[$880>>2] = $1187;
   HEAP32[$881>>2] = $1190;
   HEAP32[$882>>2] = $1192;
   $1193 = HEAP32[$880>>2]|0;
   $1194 = HEAP32[$881>>2]|0;
   $1195 = HEAP32[$882>>2]|0;
   HEAP32[$879>>2] = $1195;
   $1196 = HEAP32[$879>>2]|0;
   ;HEAP8[$878>>0]=HEAP8[$883>>0]|0;
   HEAP32[$875>>2] = $1193;
   HEAP32[$876>>2] = $1194;
   HEAP32[$877>>2] = $1196;
   $1197 = HEAP32[$875>>2]|0;
   $1198 = HEAP32[$876>>2]|0;
   $1199 = HEAP32[$877>>2]|0;
   HEAP32[$874>>2] = $1199;
   $1200 = HEAP32[$874>>2]|0;
   HEAP32[$871>>2] = $1197;
   HEAP32[$872>>2] = $1198;
   HEAP32[$873>>2] = $1200;
   $1201 = HEAP32[$872>>2]|0;
   $1202 = HEAP32[$873>>2]|0;
   HEAP32[$870>>2] = $1202;
   $1203 = HEAP32[$870>>2]|0;
   $1204 = HEAP32[$1203>>2]|0;
   HEAP32[$1201>>2] = $1204;
   HEAP32[$884>>2] = $894;
   $1205 = ((($1175)) + 4|0);
   $1206 = HEAP32[$1205>>2]|0;
   $1207 = ((($1206)) + 4|0);
   HEAP32[$1205>>2] = $1207;
  } else {
   $1208 = HEAP32[$893>>2]|0;
   HEAP32[$885>>2] = $1208;
   $1209 = HEAP32[$885>>2]|0;
   __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEE24__emplace_back_slow_pathIJRPKN3sap2ve4core5INodeEEEEvDpOT_($1175,$1209);
  }
  $1210 = HEAP32[$1019>>2]|0;
  $1211 = (__ZNK3sap2ve4core14ThrustlineItem14MajorAxisIndexEv($1210)|0);
  HEAP32[$1022>>2] = $1211;
  HEAP32[$862>>2] = $1010;
  HEAP32[$863>>2] = $1022;
  $1212 = HEAP32[$862>>2]|0;
  $1213 = ((($1212)) + 4|0);
  $1214 = HEAP32[$1213>>2]|0;
  HEAP32[$861>>2] = $1212;
  $1215 = HEAP32[$861>>2]|0;
  $1216 = ((($1215)) + 8|0);
  HEAP32[$860>>2] = $1216;
  $1217 = HEAP32[$860>>2]|0;
  HEAP32[$859>>2] = $1217;
  $1218 = HEAP32[$859>>2]|0;
  $1219 = HEAP32[$1218>>2]|0;
  $1220 = ($1214>>>0)<($1219>>>0);
  if ($1220) {
   HEAP32[$856>>2] = $864;
   HEAP32[$857>>2] = $1212;
   HEAP32[$858>>2] = 1;
   HEAP32[$837>>2] = $1212;
   $1221 = HEAP32[$837>>2]|0;
   $1222 = ((($1221)) + 8|0);
   HEAP32[$836>>2] = $1222;
   $1223 = HEAP32[$836>>2]|0;
   HEAP32[$835>>2] = $1223;
   $1224 = HEAP32[$835>>2]|0;
   $1225 = ((($1212)) + 4|0);
   $1226 = HEAP32[$1225>>2]|0;
   HEAP32[$838>>2] = $1226;
   $1227 = HEAP32[$838>>2]|0;
   $1228 = HEAP32[$863>>2]|0;
   HEAP32[$839>>2] = $1228;
   $1229 = HEAP32[$839>>2]|0;
   HEAP32[$850>>2] = $1224;
   HEAP32[$851>>2] = $1227;
   HEAP32[$852>>2] = $1229;
   $1230 = HEAP32[$850>>2]|0;
   $1231 = HEAP32[$851>>2]|0;
   $1232 = HEAP32[$852>>2]|0;
   HEAP32[$849>>2] = $1232;
   $1233 = HEAP32[$849>>2]|0;
   ;HEAP8[$848>>0]=HEAP8[$853>>0]|0;
   HEAP32[$845>>2] = $1230;
   HEAP32[$846>>2] = $1231;
   HEAP32[$847>>2] = $1233;
   $1234 = HEAP32[$845>>2]|0;
   $1235 = HEAP32[$846>>2]|0;
   $1236 = HEAP32[$847>>2]|0;
   HEAP32[$844>>2] = $1236;
   $1237 = HEAP32[$844>>2]|0;
   HEAP32[$841>>2] = $1234;
   HEAP32[$842>>2] = $1235;
   HEAP32[$843>>2] = $1237;
   $1238 = HEAP32[$842>>2]|0;
   $1239 = HEAP32[$843>>2]|0;
   HEAP32[$840>>2] = $1239;
   $1240 = HEAP32[$840>>2]|0;
   $1241 = HEAP32[$1240>>2]|0;
   HEAP32[$1238>>2] = $1241;
   HEAP32[$854>>2] = $864;
   $1242 = ((($1212)) + 4|0);
   $1243 = HEAP32[$1242>>2]|0;
   $1244 = ((($1243)) + 4|0);
   HEAP32[$1242>>2] = $1244;
  } else {
   $1245 = HEAP32[$863>>2]|0;
   HEAP32[$855>>2] = $1245;
   $1246 = HEAP32[$855>>2]|0;
   __ZNSt3__26vectorIiNS_9allocatorIiEEE24__emplace_back_slow_pathIJiEEEvDpOT_($1212,$1246);
  }
  $1247 = HEAP32[$1019>>2]|0;
  $1248 = (__ZNK3sap2ve4core14ThrustlineItem14BasisAxisCountEv($1247)|0);
  HEAP32[$1023>>2] = $1248;
  HEAP32[$832>>2] = $1011;
  HEAP32[$833>>2] = $1023;
  $1249 = HEAP32[$832>>2]|0;
  $1250 = ((($1249)) + 4|0);
  $1251 = HEAP32[$1250>>2]|0;
  HEAP32[$831>>2] = $1249;
  $1252 = HEAP32[$831>>2]|0;
  $1253 = ((($1252)) + 8|0);
  HEAP32[$830>>2] = $1253;
  $1254 = HEAP32[$830>>2]|0;
  HEAP32[$829>>2] = $1254;
  $1255 = HEAP32[$829>>2]|0;
  $1256 = HEAP32[$1255>>2]|0;
  $1257 = ($1251>>>0)<($1256>>>0);
  if ($1257) {
   HEAP32[$826>>2] = $834;
   HEAP32[$827>>2] = $1249;
   HEAP32[$828>>2] = 1;
   HEAP32[$807>>2] = $1249;
   $1258 = HEAP32[$807>>2]|0;
   $1259 = ((($1258)) + 8|0);
   HEAP32[$806>>2] = $1259;
   $1260 = HEAP32[$806>>2]|0;
   HEAP32[$805>>2] = $1260;
   $1261 = HEAP32[$805>>2]|0;
   $1262 = ((($1249)) + 4|0);
   $1263 = HEAP32[$1262>>2]|0;
   HEAP32[$808>>2] = $1263;
   $1264 = HEAP32[$808>>2]|0;
   $1265 = HEAP32[$833>>2]|0;
   HEAP32[$809>>2] = $1265;
   $1266 = HEAP32[$809>>2]|0;
   HEAP32[$820>>2] = $1261;
   HEAP32[$821>>2] = $1264;
   HEAP32[$822>>2] = $1266;
   $1267 = HEAP32[$820>>2]|0;
   $1268 = HEAP32[$821>>2]|0;
   $1269 = HEAP32[$822>>2]|0;
   HEAP32[$819>>2] = $1269;
   $1270 = HEAP32[$819>>2]|0;
   ;HEAP8[$818>>0]=HEAP8[$823>>0]|0;
   HEAP32[$815>>2] = $1267;
   HEAP32[$816>>2] = $1268;
   HEAP32[$817>>2] = $1270;
   $1271 = HEAP32[$815>>2]|0;
   $1272 = HEAP32[$816>>2]|0;
   $1273 = HEAP32[$817>>2]|0;
   HEAP32[$814>>2] = $1273;
   $1274 = HEAP32[$814>>2]|0;
   HEAP32[$811>>2] = $1271;
   HEAP32[$812>>2] = $1272;
   HEAP32[$813>>2] = $1274;
   $1275 = HEAP32[$812>>2]|0;
   $1276 = HEAP32[$813>>2]|0;
   HEAP32[$810>>2] = $1276;
   $1277 = HEAP32[$810>>2]|0;
   $1278 = HEAP32[$1277>>2]|0;
   HEAP32[$1275>>2] = $1278;
   HEAP32[$824>>2] = $834;
   $1279 = ((($1249)) + 4|0);
   $1280 = HEAP32[$1279>>2]|0;
   $1281 = ((($1280)) + 4|0);
   HEAP32[$1279>>2] = $1281;
  } else {
   $1282 = HEAP32[$833>>2]|0;
   HEAP32[$825>>2] = $1282;
   $1283 = HEAP32[$825>>2]|0;
   __ZNSt3__26vectorIiNS_9allocatorIiEEE24__emplace_back_slow_pathIJRjEEEvDpOT_($1249,$1283);
  }
  HEAP32[$1024>>2] = 0;
  while(1) {
   $1284 = HEAP32[$1024>>2]|0;
   $1285 = HEAP32[$1023>>2]|0;
   $1286 = ($1284>>>0)<($1285>>>0);
   $1287 = HEAP32[$1019>>2]|0;
   if (!($1286)) {
    break;
   }
   $1288 = HEAP32[$1024>>2]|0;
   __ZNK3sap2ve4core14ThrustlineItem9BasisAxisEm($1025,$1287,$1288);
   HEAP32[$802>>2] = $1012;
   HEAP32[$803>>2] = $1025;
   $1289 = HEAP32[$802>>2]|0;
   $1290 = ((($1289)) + 4|0);
   $1291 = HEAP32[$1290>>2]|0;
   HEAP32[$801>>2] = $1289;
   $1292 = HEAP32[$801>>2]|0;
   $1293 = ((($1292)) + 8|0);
   HEAP32[$800>>2] = $1293;
   $1294 = HEAP32[$800>>2]|0;
   HEAP32[$799>>2] = $1294;
   $1295 = HEAP32[$799>>2]|0;
   $1296 = HEAP32[$1295>>2]|0;
   $1297 = ($1291>>>0)<($1296>>>0);
   if ($1297) {
    HEAP32[$796>>2] = $804;
    HEAP32[$797>>2] = $1289;
    HEAP32[$798>>2] = 1;
    HEAP32[$777>>2] = $1289;
    $1298 = HEAP32[$777>>2]|0;
    $1299 = ((($1298)) + 8|0);
    HEAP32[$776>>2] = $1299;
    $1300 = HEAP32[$776>>2]|0;
    HEAP32[$775>>2] = $1300;
    $1301 = HEAP32[$775>>2]|0;
    $1302 = ((($1289)) + 4|0);
    $1303 = HEAP32[$1302>>2]|0;
    HEAP32[$778>>2] = $1303;
    $1304 = HEAP32[$778>>2]|0;
    $1305 = HEAP32[$803>>2]|0;
    HEAP32[$779>>2] = $1305;
    $1306 = HEAP32[$779>>2]|0;
    HEAP32[$790>>2] = $1301;
    HEAP32[$791>>2] = $1304;
    HEAP32[$792>>2] = $1306;
    $1307 = HEAP32[$790>>2]|0;
    $1308 = HEAP32[$791>>2]|0;
    $1309 = HEAP32[$792>>2]|0;
    HEAP32[$789>>2] = $1309;
    $1310 = HEAP32[$789>>2]|0;
    ;HEAP8[$788>>0]=HEAP8[$793>>0]|0;
    HEAP32[$785>>2] = $1307;
    HEAP32[$786>>2] = $1308;
    HEAP32[$787>>2] = $1310;
    $1311 = HEAP32[$785>>2]|0;
    $1312 = HEAP32[$786>>2]|0;
    $1313 = HEAP32[$787>>2]|0;
    HEAP32[$784>>2] = $1313;
    $1314 = HEAP32[$784>>2]|0;
    HEAP32[$781>>2] = $1311;
    HEAP32[$782>>2] = $1312;
    HEAP32[$783>>2] = $1314;
    $1315 = HEAP32[$782>>2]|0;
    $1316 = HEAP32[$783>>2]|0;
    HEAP32[$780>>2] = $1316;
    $1317 = HEAP32[$780>>2]|0;
    $1318 = +HEAPF32[$1317>>2];
    HEAPF32[$1315>>2] = $1318;
    HEAP32[$794>>2] = $804;
    $1319 = ((($1289)) + 4|0);
    $1320 = HEAP32[$1319>>2]|0;
    $1321 = ((($1320)) + 4|0);
    HEAP32[$1319>>2] = $1321;
   } else {
    $1322 = HEAP32[$803>>2]|0;
    HEAP32[$795>>2] = $1322;
    $1323 = HEAP32[$795>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1289,$1323);
   }
   $1324 = ((($1025)) + 4|0);
   HEAP32[$772>>2] = $1012;
   HEAP32[$773>>2] = $1324;
   $1325 = HEAP32[$772>>2]|0;
   $1326 = ((($1325)) + 4|0);
   $1327 = HEAP32[$1326>>2]|0;
   HEAP32[$771>>2] = $1325;
   $1328 = HEAP32[$771>>2]|0;
   $1329 = ((($1328)) + 8|0);
   HEAP32[$770>>2] = $1329;
   $1330 = HEAP32[$770>>2]|0;
   HEAP32[$769>>2] = $1330;
   $1331 = HEAP32[$769>>2]|0;
   $1332 = HEAP32[$1331>>2]|0;
   $1333 = ($1327>>>0)<($1332>>>0);
   if ($1333) {
    HEAP32[$766>>2] = $774;
    HEAP32[$767>>2] = $1325;
    HEAP32[$768>>2] = 1;
    HEAP32[$747>>2] = $1325;
    $1334 = HEAP32[$747>>2]|0;
    $1335 = ((($1334)) + 8|0);
    HEAP32[$746>>2] = $1335;
    $1336 = HEAP32[$746>>2]|0;
    HEAP32[$745>>2] = $1336;
    $1337 = HEAP32[$745>>2]|0;
    $1338 = ((($1325)) + 4|0);
    $1339 = HEAP32[$1338>>2]|0;
    HEAP32[$748>>2] = $1339;
    $1340 = HEAP32[$748>>2]|0;
    $1341 = HEAP32[$773>>2]|0;
    HEAP32[$749>>2] = $1341;
    $1342 = HEAP32[$749>>2]|0;
    HEAP32[$760>>2] = $1337;
    HEAP32[$761>>2] = $1340;
    HEAP32[$762>>2] = $1342;
    $1343 = HEAP32[$760>>2]|0;
    $1344 = HEAP32[$761>>2]|0;
    $1345 = HEAP32[$762>>2]|0;
    HEAP32[$759>>2] = $1345;
    $1346 = HEAP32[$759>>2]|0;
    ;HEAP8[$758>>0]=HEAP8[$763>>0]|0;
    HEAP32[$755>>2] = $1343;
    HEAP32[$756>>2] = $1344;
    HEAP32[$757>>2] = $1346;
    $1347 = HEAP32[$755>>2]|0;
    $1348 = HEAP32[$756>>2]|0;
    $1349 = HEAP32[$757>>2]|0;
    HEAP32[$754>>2] = $1349;
    $1350 = HEAP32[$754>>2]|0;
    HEAP32[$751>>2] = $1347;
    HEAP32[$752>>2] = $1348;
    HEAP32[$753>>2] = $1350;
    $1351 = HEAP32[$752>>2]|0;
    $1352 = HEAP32[$753>>2]|0;
    HEAP32[$750>>2] = $1352;
    $1353 = HEAP32[$750>>2]|0;
    $1354 = +HEAPF32[$1353>>2];
    HEAPF32[$1351>>2] = $1354;
    HEAP32[$764>>2] = $774;
    $1355 = ((($1325)) + 4|0);
    $1356 = HEAP32[$1355>>2]|0;
    $1357 = ((($1356)) + 4|0);
    HEAP32[$1355>>2] = $1357;
   } else {
    $1358 = HEAP32[$773>>2]|0;
    HEAP32[$765>>2] = $1358;
    $1359 = HEAP32[$765>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1325,$1359);
   }
   $1360 = ((($1025)) + 8|0);
   HEAP32[$742>>2] = $1012;
   HEAP32[$743>>2] = $1360;
   $1361 = HEAP32[$742>>2]|0;
   $1362 = ((($1361)) + 4|0);
   $1363 = HEAP32[$1362>>2]|0;
   HEAP32[$741>>2] = $1361;
   $1364 = HEAP32[$741>>2]|0;
   $1365 = ((($1364)) + 8|0);
   HEAP32[$740>>2] = $1365;
   $1366 = HEAP32[$740>>2]|0;
   HEAP32[$739>>2] = $1366;
   $1367 = HEAP32[$739>>2]|0;
   $1368 = HEAP32[$1367>>2]|0;
   $1369 = ($1363>>>0)<($1368>>>0);
   if ($1369) {
    HEAP32[$736>>2] = $744;
    HEAP32[$737>>2] = $1361;
    HEAP32[$738>>2] = 1;
    HEAP32[$717>>2] = $1361;
    $1370 = HEAP32[$717>>2]|0;
    $1371 = ((($1370)) + 8|0);
    HEAP32[$716>>2] = $1371;
    $1372 = HEAP32[$716>>2]|0;
    HEAP32[$715>>2] = $1372;
    $1373 = HEAP32[$715>>2]|0;
    $1374 = ((($1361)) + 4|0);
    $1375 = HEAP32[$1374>>2]|0;
    HEAP32[$718>>2] = $1375;
    $1376 = HEAP32[$718>>2]|0;
    $1377 = HEAP32[$743>>2]|0;
    HEAP32[$719>>2] = $1377;
    $1378 = HEAP32[$719>>2]|0;
    HEAP32[$730>>2] = $1373;
    HEAP32[$731>>2] = $1376;
    HEAP32[$732>>2] = $1378;
    $1379 = HEAP32[$730>>2]|0;
    $1380 = HEAP32[$731>>2]|0;
    $1381 = HEAP32[$732>>2]|0;
    HEAP32[$729>>2] = $1381;
    $1382 = HEAP32[$729>>2]|0;
    ;HEAP8[$728>>0]=HEAP8[$733>>0]|0;
    HEAP32[$725>>2] = $1379;
    HEAP32[$726>>2] = $1380;
    HEAP32[$727>>2] = $1382;
    $1383 = HEAP32[$725>>2]|0;
    $1384 = HEAP32[$726>>2]|0;
    $1385 = HEAP32[$727>>2]|0;
    HEAP32[$724>>2] = $1385;
    $1386 = HEAP32[$724>>2]|0;
    HEAP32[$721>>2] = $1383;
    HEAP32[$722>>2] = $1384;
    HEAP32[$723>>2] = $1386;
    $1387 = HEAP32[$722>>2]|0;
    $1388 = HEAP32[$723>>2]|0;
    HEAP32[$720>>2] = $1388;
    $1389 = HEAP32[$720>>2]|0;
    $1390 = +HEAPF32[$1389>>2];
    HEAPF32[$1387>>2] = $1390;
    HEAP32[$734>>2] = $744;
    $1391 = ((($1361)) + 4|0);
    $1392 = HEAP32[$1391>>2]|0;
    $1393 = ((($1392)) + 4|0);
    HEAP32[$1391>>2] = $1393;
   } else {
    $1394 = HEAP32[$743>>2]|0;
    HEAP32[$735>>2] = $1394;
    $1395 = HEAP32[$735>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1361,$1395);
   }
   $1396 = HEAP32[$1024>>2]|0;
   $1397 = (($1396) + 1)|0;
   HEAP32[$1024>>2] = $1397;
  }
  __ZNK3sap2ve4core14ThrustlineItem9DimensionEv($1026,$1287);
  HEAP32[$712>>2] = $1013;
  HEAP32[$713>>2] = $1026;
  $1398 = HEAP32[$712>>2]|0;
  $1399 = ((($1398)) + 4|0);
  $1400 = HEAP32[$1399>>2]|0;
  HEAP32[$711>>2] = $1398;
  $1401 = HEAP32[$711>>2]|0;
  $1402 = ((($1401)) + 8|0);
  HEAP32[$710>>2] = $1402;
  $1403 = HEAP32[$710>>2]|0;
  HEAP32[$709>>2] = $1403;
  $1404 = HEAP32[$709>>2]|0;
  $1405 = HEAP32[$1404>>2]|0;
  $1406 = ($1400>>>0)<($1405>>>0);
  if ($1406) {
   HEAP32[$706>>2] = $714;
   HEAP32[$707>>2] = $1398;
   HEAP32[$708>>2] = 1;
   HEAP32[$687>>2] = $1398;
   $1407 = HEAP32[$687>>2]|0;
   $1408 = ((($1407)) + 8|0);
   HEAP32[$686>>2] = $1408;
   $1409 = HEAP32[$686>>2]|0;
   HEAP32[$685>>2] = $1409;
   $1410 = HEAP32[$685>>2]|0;
   $1411 = ((($1398)) + 4|0);
   $1412 = HEAP32[$1411>>2]|0;
   HEAP32[$688>>2] = $1412;
   $1413 = HEAP32[$688>>2]|0;
   $1414 = HEAP32[$713>>2]|0;
   HEAP32[$689>>2] = $1414;
   $1415 = HEAP32[$689>>2]|0;
   HEAP32[$700>>2] = $1410;
   HEAP32[$701>>2] = $1413;
   HEAP32[$702>>2] = $1415;
   $1416 = HEAP32[$700>>2]|0;
   $1417 = HEAP32[$701>>2]|0;
   $1418 = HEAP32[$702>>2]|0;
   HEAP32[$699>>2] = $1418;
   $1419 = HEAP32[$699>>2]|0;
   ;HEAP8[$698>>0]=HEAP8[$703>>0]|0;
   HEAP32[$695>>2] = $1416;
   HEAP32[$696>>2] = $1417;
   HEAP32[$697>>2] = $1419;
   $1420 = HEAP32[$695>>2]|0;
   $1421 = HEAP32[$696>>2]|0;
   $1422 = HEAP32[$697>>2]|0;
   HEAP32[$694>>2] = $1422;
   $1423 = HEAP32[$694>>2]|0;
   HEAP32[$691>>2] = $1420;
   HEAP32[$692>>2] = $1421;
   HEAP32[$693>>2] = $1423;
   $1424 = HEAP32[$692>>2]|0;
   $1425 = HEAP32[$693>>2]|0;
   HEAP32[$690>>2] = $1425;
   $1426 = HEAP32[$690>>2]|0;
   $1427 = +HEAPF32[$1426>>2];
   HEAPF32[$1424>>2] = $1427;
   HEAP32[$704>>2] = $714;
   $1428 = ((($1398)) + 4|0);
   $1429 = HEAP32[$1428>>2]|0;
   $1430 = ((($1429)) + 4|0);
   HEAP32[$1428>>2] = $1430;
  } else {
   $1431 = HEAP32[$713>>2]|0;
   HEAP32[$705>>2] = $1431;
   $1432 = HEAP32[$705>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1398,$1432);
  }
  $1433 = ((($1026)) + 4|0);
  HEAP32[$682>>2] = $1013;
  HEAP32[$683>>2] = $1433;
  $1434 = HEAP32[$682>>2]|0;
  $1435 = ((($1434)) + 4|0);
  $1436 = HEAP32[$1435>>2]|0;
  HEAP32[$681>>2] = $1434;
  $1437 = HEAP32[$681>>2]|0;
  $1438 = ((($1437)) + 8|0);
  HEAP32[$680>>2] = $1438;
  $1439 = HEAP32[$680>>2]|0;
  HEAP32[$679>>2] = $1439;
  $1440 = HEAP32[$679>>2]|0;
  $1441 = HEAP32[$1440>>2]|0;
  $1442 = ($1436>>>0)<($1441>>>0);
  if ($1442) {
   HEAP32[$676>>2] = $684;
   HEAP32[$677>>2] = $1434;
   HEAP32[$678>>2] = 1;
   HEAP32[$657>>2] = $1434;
   $1443 = HEAP32[$657>>2]|0;
   $1444 = ((($1443)) + 8|0);
   HEAP32[$656>>2] = $1444;
   $1445 = HEAP32[$656>>2]|0;
   HEAP32[$655>>2] = $1445;
   $1446 = HEAP32[$655>>2]|0;
   $1447 = ((($1434)) + 4|0);
   $1448 = HEAP32[$1447>>2]|0;
   HEAP32[$658>>2] = $1448;
   $1449 = HEAP32[$658>>2]|0;
   $1450 = HEAP32[$683>>2]|0;
   HEAP32[$659>>2] = $1450;
   $1451 = HEAP32[$659>>2]|0;
   HEAP32[$670>>2] = $1446;
   HEAP32[$671>>2] = $1449;
   HEAP32[$672>>2] = $1451;
   $1452 = HEAP32[$670>>2]|0;
   $1453 = HEAP32[$671>>2]|0;
   $1454 = HEAP32[$672>>2]|0;
   HEAP32[$669>>2] = $1454;
   $1455 = HEAP32[$669>>2]|0;
   ;HEAP8[$668>>0]=HEAP8[$673>>0]|0;
   HEAP32[$665>>2] = $1452;
   HEAP32[$666>>2] = $1453;
   HEAP32[$667>>2] = $1455;
   $1456 = HEAP32[$665>>2]|0;
   $1457 = HEAP32[$666>>2]|0;
   $1458 = HEAP32[$667>>2]|0;
   HEAP32[$664>>2] = $1458;
   $1459 = HEAP32[$664>>2]|0;
   HEAP32[$661>>2] = $1456;
   HEAP32[$662>>2] = $1457;
   HEAP32[$663>>2] = $1459;
   $1460 = HEAP32[$662>>2]|0;
   $1461 = HEAP32[$663>>2]|0;
   HEAP32[$660>>2] = $1461;
   $1462 = HEAP32[$660>>2]|0;
   $1463 = +HEAPF32[$1462>>2];
   HEAPF32[$1460>>2] = $1463;
   HEAP32[$674>>2] = $684;
   $1464 = ((($1434)) + 4|0);
   $1465 = HEAP32[$1464>>2]|0;
   $1466 = ((($1465)) + 4|0);
   HEAP32[$1464>>2] = $1466;
  } else {
   $1467 = HEAP32[$683>>2]|0;
   HEAP32[$675>>2] = $1467;
   $1468 = HEAP32[$675>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1434,$1468);
  }
  $1469 = ((($1026)) + 8|0);
  HEAP32[$652>>2] = $1013;
  HEAP32[$653>>2] = $1469;
  $1470 = HEAP32[$652>>2]|0;
  $1471 = ((($1470)) + 4|0);
  $1472 = HEAP32[$1471>>2]|0;
  HEAP32[$651>>2] = $1470;
  $1473 = HEAP32[$651>>2]|0;
  $1474 = ((($1473)) + 8|0);
  HEAP32[$650>>2] = $1474;
  $1475 = HEAP32[$650>>2]|0;
  HEAP32[$649>>2] = $1475;
  $1476 = HEAP32[$649>>2]|0;
  $1477 = HEAP32[$1476>>2]|0;
  $1478 = ($1472>>>0)<($1477>>>0);
  if ($1478) {
   HEAP32[$646>>2] = $654;
   HEAP32[$647>>2] = $1470;
   HEAP32[$648>>2] = 1;
   HEAP32[$627>>2] = $1470;
   $1479 = HEAP32[$627>>2]|0;
   $1480 = ((($1479)) + 8|0);
   HEAP32[$626>>2] = $1480;
   $1481 = HEAP32[$626>>2]|0;
   HEAP32[$625>>2] = $1481;
   $1482 = HEAP32[$625>>2]|0;
   $1483 = ((($1470)) + 4|0);
   $1484 = HEAP32[$1483>>2]|0;
   HEAP32[$628>>2] = $1484;
   $1485 = HEAP32[$628>>2]|0;
   $1486 = HEAP32[$653>>2]|0;
   HEAP32[$629>>2] = $1486;
   $1487 = HEAP32[$629>>2]|0;
   HEAP32[$640>>2] = $1482;
   HEAP32[$641>>2] = $1485;
   HEAP32[$642>>2] = $1487;
   $1488 = HEAP32[$640>>2]|0;
   $1489 = HEAP32[$641>>2]|0;
   $1490 = HEAP32[$642>>2]|0;
   HEAP32[$639>>2] = $1490;
   $1491 = HEAP32[$639>>2]|0;
   ;HEAP8[$638>>0]=HEAP8[$643>>0]|0;
   HEAP32[$635>>2] = $1488;
   HEAP32[$636>>2] = $1489;
   HEAP32[$637>>2] = $1491;
   $1492 = HEAP32[$635>>2]|0;
   $1493 = HEAP32[$636>>2]|0;
   $1494 = HEAP32[$637>>2]|0;
   HEAP32[$634>>2] = $1494;
   $1495 = HEAP32[$634>>2]|0;
   HEAP32[$631>>2] = $1492;
   HEAP32[$632>>2] = $1493;
   HEAP32[$633>>2] = $1495;
   $1496 = HEAP32[$632>>2]|0;
   $1497 = HEAP32[$633>>2]|0;
   HEAP32[$630>>2] = $1497;
   $1498 = HEAP32[$630>>2]|0;
   $1499 = +HEAPF32[$1498>>2];
   HEAPF32[$1496>>2] = $1499;
   HEAP32[$644>>2] = $654;
   $1500 = ((($1470)) + 4|0);
   $1501 = HEAP32[$1500>>2]|0;
   $1502 = ((($1501)) + 4|0);
   HEAP32[$1500>>2] = $1502;
  } else {
   $1503 = HEAP32[$653>>2]|0;
   HEAP32[$645>>2] = $1503;
   $1504 = HEAP32[$645>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1470,$1504);
  }
  $1505 = HEAP32[$1019>>2]|0;
  __ZNK3sap2ve4core14ThrustlineItem6CenterEv($1027,$1505);
  HEAP32[$622>>2] = $1014;
  HEAP32[$623>>2] = $1027;
  $1506 = HEAP32[$622>>2]|0;
  $1507 = ((($1506)) + 4|0);
  $1508 = HEAP32[$1507>>2]|0;
  HEAP32[$621>>2] = $1506;
  $1509 = HEAP32[$621>>2]|0;
  $1510 = ((($1509)) + 8|0);
  HEAP32[$620>>2] = $1510;
  $1511 = HEAP32[$620>>2]|0;
  HEAP32[$619>>2] = $1511;
  $1512 = HEAP32[$619>>2]|0;
  $1513 = HEAP32[$1512>>2]|0;
  $1514 = ($1508>>>0)<($1513>>>0);
  if ($1514) {
   HEAP32[$616>>2] = $624;
   HEAP32[$617>>2] = $1506;
   HEAP32[$618>>2] = 1;
   HEAP32[$597>>2] = $1506;
   $1515 = HEAP32[$597>>2]|0;
   $1516 = ((($1515)) + 8|0);
   HEAP32[$596>>2] = $1516;
   $1517 = HEAP32[$596>>2]|0;
   HEAP32[$595>>2] = $1517;
   $1518 = HEAP32[$595>>2]|0;
   $1519 = ((($1506)) + 4|0);
   $1520 = HEAP32[$1519>>2]|0;
   HEAP32[$598>>2] = $1520;
   $1521 = HEAP32[$598>>2]|0;
   $1522 = HEAP32[$623>>2]|0;
   HEAP32[$599>>2] = $1522;
   $1523 = HEAP32[$599>>2]|0;
   HEAP32[$610>>2] = $1518;
   HEAP32[$611>>2] = $1521;
   HEAP32[$612>>2] = $1523;
   $1524 = HEAP32[$610>>2]|0;
   $1525 = HEAP32[$611>>2]|0;
   $1526 = HEAP32[$612>>2]|0;
   HEAP32[$609>>2] = $1526;
   $1527 = HEAP32[$609>>2]|0;
   ;HEAP8[$608>>0]=HEAP8[$613>>0]|0;
   HEAP32[$605>>2] = $1524;
   HEAP32[$606>>2] = $1525;
   HEAP32[$607>>2] = $1527;
   $1528 = HEAP32[$605>>2]|0;
   $1529 = HEAP32[$606>>2]|0;
   $1530 = HEAP32[$607>>2]|0;
   HEAP32[$604>>2] = $1530;
   $1531 = HEAP32[$604>>2]|0;
   HEAP32[$601>>2] = $1528;
   HEAP32[$602>>2] = $1529;
   HEAP32[$603>>2] = $1531;
   $1532 = HEAP32[$602>>2]|0;
   $1533 = HEAP32[$603>>2]|0;
   HEAP32[$600>>2] = $1533;
   $1534 = HEAP32[$600>>2]|0;
   $1535 = +HEAPF32[$1534>>2];
   HEAPF32[$1532>>2] = $1535;
   HEAP32[$614>>2] = $624;
   $1536 = ((($1506)) + 4|0);
   $1537 = HEAP32[$1536>>2]|0;
   $1538 = ((($1537)) + 4|0);
   HEAP32[$1536>>2] = $1538;
  } else {
   $1539 = HEAP32[$623>>2]|0;
   HEAP32[$615>>2] = $1539;
   $1540 = HEAP32[$615>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1506,$1540);
  }
  $1541 = ((($1027)) + 4|0);
  HEAP32[$592>>2] = $1014;
  HEAP32[$593>>2] = $1541;
  $1542 = HEAP32[$592>>2]|0;
  $1543 = ((($1542)) + 4|0);
  $1544 = HEAP32[$1543>>2]|0;
  HEAP32[$591>>2] = $1542;
  $1545 = HEAP32[$591>>2]|0;
  $1546 = ((($1545)) + 8|0);
  HEAP32[$590>>2] = $1546;
  $1547 = HEAP32[$590>>2]|0;
  HEAP32[$589>>2] = $1547;
  $1548 = HEAP32[$589>>2]|0;
  $1549 = HEAP32[$1548>>2]|0;
  $1550 = ($1544>>>0)<($1549>>>0);
  if ($1550) {
   HEAP32[$586>>2] = $594;
   HEAP32[$587>>2] = $1542;
   HEAP32[$588>>2] = 1;
   HEAP32[$567>>2] = $1542;
   $1551 = HEAP32[$567>>2]|0;
   $1552 = ((($1551)) + 8|0);
   HEAP32[$566>>2] = $1552;
   $1553 = HEAP32[$566>>2]|0;
   HEAP32[$565>>2] = $1553;
   $1554 = HEAP32[$565>>2]|0;
   $1555 = ((($1542)) + 4|0);
   $1556 = HEAP32[$1555>>2]|0;
   HEAP32[$568>>2] = $1556;
   $1557 = HEAP32[$568>>2]|0;
   $1558 = HEAP32[$593>>2]|0;
   HEAP32[$569>>2] = $1558;
   $1559 = HEAP32[$569>>2]|0;
   HEAP32[$580>>2] = $1554;
   HEAP32[$581>>2] = $1557;
   HEAP32[$582>>2] = $1559;
   $1560 = HEAP32[$580>>2]|0;
   $1561 = HEAP32[$581>>2]|0;
   $1562 = HEAP32[$582>>2]|0;
   HEAP32[$579>>2] = $1562;
   $1563 = HEAP32[$579>>2]|0;
   ;HEAP8[$578>>0]=HEAP8[$583>>0]|0;
   HEAP32[$575>>2] = $1560;
   HEAP32[$576>>2] = $1561;
   HEAP32[$577>>2] = $1563;
   $1564 = HEAP32[$575>>2]|0;
   $1565 = HEAP32[$576>>2]|0;
   $1566 = HEAP32[$577>>2]|0;
   HEAP32[$574>>2] = $1566;
   $1567 = HEAP32[$574>>2]|0;
   HEAP32[$571>>2] = $1564;
   HEAP32[$572>>2] = $1565;
   HEAP32[$573>>2] = $1567;
   $1568 = HEAP32[$572>>2]|0;
   $1569 = HEAP32[$573>>2]|0;
   HEAP32[$570>>2] = $1569;
   $1570 = HEAP32[$570>>2]|0;
   $1571 = +HEAPF32[$1570>>2];
   HEAPF32[$1568>>2] = $1571;
   HEAP32[$584>>2] = $594;
   $1572 = ((($1542)) + 4|0);
   $1573 = HEAP32[$1572>>2]|0;
   $1574 = ((($1573)) + 4|0);
   HEAP32[$1572>>2] = $1574;
  } else {
   $1575 = HEAP32[$593>>2]|0;
   HEAP32[$585>>2] = $1575;
   $1576 = HEAP32[$585>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1542,$1576);
  }
  $1577 = ((($1027)) + 8|0);
  HEAP32[$562>>2] = $1014;
  HEAP32[$563>>2] = $1577;
  $1578 = HEAP32[$562>>2]|0;
  $1579 = ((($1578)) + 4|0);
  $1580 = HEAP32[$1579>>2]|0;
  HEAP32[$561>>2] = $1578;
  $1581 = HEAP32[$561>>2]|0;
  $1582 = ((($1581)) + 8|0);
  HEAP32[$560>>2] = $1582;
  $1583 = HEAP32[$560>>2]|0;
  HEAP32[$559>>2] = $1583;
  $1584 = HEAP32[$559>>2]|0;
  $1585 = HEAP32[$1584>>2]|0;
  $1586 = ($1580>>>0)<($1585>>>0);
  if ($1586) {
   HEAP32[$556>>2] = $564;
   HEAP32[$557>>2] = $1578;
   HEAP32[$558>>2] = 1;
   HEAP32[$537>>2] = $1578;
   $1587 = HEAP32[$537>>2]|0;
   $1588 = ((($1587)) + 8|0);
   HEAP32[$536>>2] = $1588;
   $1589 = HEAP32[$536>>2]|0;
   HEAP32[$535>>2] = $1589;
   $1590 = HEAP32[$535>>2]|0;
   $1591 = ((($1578)) + 4|0);
   $1592 = HEAP32[$1591>>2]|0;
   HEAP32[$538>>2] = $1592;
   $1593 = HEAP32[$538>>2]|0;
   $1594 = HEAP32[$563>>2]|0;
   HEAP32[$539>>2] = $1594;
   $1595 = HEAP32[$539>>2]|0;
   HEAP32[$550>>2] = $1590;
   HEAP32[$551>>2] = $1593;
   HEAP32[$552>>2] = $1595;
   $1596 = HEAP32[$550>>2]|0;
   $1597 = HEAP32[$551>>2]|0;
   $1598 = HEAP32[$552>>2]|0;
   HEAP32[$549>>2] = $1598;
   $1599 = HEAP32[$549>>2]|0;
   ;HEAP8[$548>>0]=HEAP8[$553>>0]|0;
   HEAP32[$545>>2] = $1596;
   HEAP32[$546>>2] = $1597;
   HEAP32[$547>>2] = $1599;
   $1600 = HEAP32[$545>>2]|0;
   $1601 = HEAP32[$546>>2]|0;
   $1602 = HEAP32[$547>>2]|0;
   HEAP32[$544>>2] = $1602;
   $1603 = HEAP32[$544>>2]|0;
   HEAP32[$541>>2] = $1600;
   HEAP32[$542>>2] = $1601;
   HEAP32[$543>>2] = $1603;
   $1604 = HEAP32[$542>>2]|0;
   $1605 = HEAP32[$543>>2]|0;
   HEAP32[$540>>2] = $1605;
   $1606 = HEAP32[$540>>2]|0;
   $1607 = +HEAPF32[$1606>>2];
   HEAPF32[$1604>>2] = $1607;
   HEAP32[$554>>2] = $564;
   $1608 = ((($1578)) + 4|0);
   $1609 = HEAP32[$1608>>2]|0;
   $1610 = ((($1609)) + 4|0);
   HEAP32[$1608>>2] = $1610;
  } else {
   $1611 = HEAP32[$563>>2]|0;
   HEAP32[$555>>2] = $1611;
   $1612 = HEAP32[$555>>2]|0;
   __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1578,$1612);
  }
  $1613 = HEAP32[$1019>>2]|0;
  $1614 = (__ZNK3sap2ve4core14ThrustlineItem16BoundPointsCountEv($1613)|0);
  HEAP32[$1028>>2] = $1614;
  HEAP32[$532>>2] = $1015;
  HEAP32[$533>>2] = $1028;
  $1615 = HEAP32[$532>>2]|0;
  $1616 = ((($1615)) + 4|0);
  $1617 = HEAP32[$1616>>2]|0;
  HEAP32[$531>>2] = $1615;
  $1618 = HEAP32[$531>>2]|0;
  $1619 = ((($1618)) + 8|0);
  HEAP32[$530>>2] = $1619;
  $1620 = HEAP32[$530>>2]|0;
  HEAP32[$529>>2] = $1620;
  $1621 = HEAP32[$529>>2]|0;
  $1622 = HEAP32[$1621>>2]|0;
  $1623 = ($1617>>>0)<($1622>>>0);
  if ($1623) {
   HEAP32[$526>>2] = $534;
   HEAP32[$527>>2] = $1615;
   HEAP32[$528>>2] = 1;
   HEAP32[$507>>2] = $1615;
   $1624 = HEAP32[$507>>2]|0;
   $1625 = ((($1624)) + 8|0);
   HEAP32[$506>>2] = $1625;
   $1626 = HEAP32[$506>>2]|0;
   HEAP32[$505>>2] = $1626;
   $1627 = HEAP32[$505>>2]|0;
   $1628 = ((($1615)) + 4|0);
   $1629 = HEAP32[$1628>>2]|0;
   HEAP32[$508>>2] = $1629;
   $1630 = HEAP32[$508>>2]|0;
   $1631 = HEAP32[$533>>2]|0;
   HEAP32[$509>>2] = $1631;
   $1632 = HEAP32[$509>>2]|0;
   HEAP32[$520>>2] = $1627;
   HEAP32[$521>>2] = $1630;
   HEAP32[$522>>2] = $1632;
   $1633 = HEAP32[$520>>2]|0;
   $1634 = HEAP32[$521>>2]|0;
   $1635 = HEAP32[$522>>2]|0;
   HEAP32[$519>>2] = $1635;
   $1636 = HEAP32[$519>>2]|0;
   ;HEAP8[$518>>0]=HEAP8[$523>>0]|0;
   HEAP32[$515>>2] = $1633;
   HEAP32[$516>>2] = $1634;
   HEAP32[$517>>2] = $1636;
   $1637 = HEAP32[$515>>2]|0;
   $1638 = HEAP32[$516>>2]|0;
   $1639 = HEAP32[$517>>2]|0;
   HEAP32[$514>>2] = $1639;
   $1640 = HEAP32[$514>>2]|0;
   HEAP32[$511>>2] = $1637;
   HEAP32[$512>>2] = $1638;
   HEAP32[$513>>2] = $1640;
   $1641 = HEAP32[$512>>2]|0;
   $1642 = HEAP32[$513>>2]|0;
   HEAP32[$510>>2] = $1642;
   $1643 = HEAP32[$510>>2]|0;
   $1644 = HEAP32[$1643>>2]|0;
   HEAP32[$1641>>2] = $1644;
   HEAP32[$524>>2] = $534;
   $1645 = ((($1615)) + 4|0);
   $1646 = HEAP32[$1645>>2]|0;
   $1647 = ((($1646)) + 4|0);
   HEAP32[$1645>>2] = $1647;
  } else {
   $1648 = HEAP32[$533>>2]|0;
   HEAP32[$525>>2] = $1648;
   $1649 = HEAP32[$525>>2]|0;
   __ZNSt3__26vectorIiNS_9allocatorIiEEE24__emplace_back_slow_pathIJRjEEEvDpOT_($1615,$1649);
  }
  HEAP32[$1029>>2] = 0;
  while(1) {
   $1650 = HEAP32[$1029>>2]|0;
   $1651 = HEAP32[$1028>>2]|0;
   $1652 = ($1650>>>0)<($1651>>>0);
   if (!($1652)) {
    break;
   }
   $1653 = HEAP32[$1019>>2]|0;
   $1654 = HEAP32[$1029>>2]|0;
   __ZNK3sap2ve4core14ThrustlineItem10BoundPointEm($1030,$1653,$1654);
   HEAP32[$502>>2] = $1016;
   HEAP32[$503>>2] = $1030;
   $1655 = HEAP32[$502>>2]|0;
   $1656 = ((($1655)) + 4|0);
   $1657 = HEAP32[$1656>>2]|0;
   HEAP32[$501>>2] = $1655;
   $1658 = HEAP32[$501>>2]|0;
   $1659 = ((($1658)) + 8|0);
   HEAP32[$500>>2] = $1659;
   $1660 = HEAP32[$500>>2]|0;
   HEAP32[$499>>2] = $1660;
   $1661 = HEAP32[$499>>2]|0;
   $1662 = HEAP32[$1661>>2]|0;
   $1663 = ($1657>>>0)<($1662>>>0);
   if ($1663) {
    HEAP32[$496>>2] = $504;
    HEAP32[$497>>2] = $1655;
    HEAP32[$498>>2] = 1;
    HEAP32[$477>>2] = $1655;
    $1664 = HEAP32[$477>>2]|0;
    $1665 = ((($1664)) + 8|0);
    HEAP32[$476>>2] = $1665;
    $1666 = HEAP32[$476>>2]|0;
    HEAP32[$475>>2] = $1666;
    $1667 = HEAP32[$475>>2]|0;
    $1668 = ((($1655)) + 4|0);
    $1669 = HEAP32[$1668>>2]|0;
    HEAP32[$478>>2] = $1669;
    $1670 = HEAP32[$478>>2]|0;
    $1671 = HEAP32[$503>>2]|0;
    HEAP32[$479>>2] = $1671;
    $1672 = HEAP32[$479>>2]|0;
    HEAP32[$490>>2] = $1667;
    HEAP32[$491>>2] = $1670;
    HEAP32[$492>>2] = $1672;
    $1673 = HEAP32[$490>>2]|0;
    $1674 = HEAP32[$491>>2]|0;
    $1675 = HEAP32[$492>>2]|0;
    HEAP32[$489>>2] = $1675;
    $1676 = HEAP32[$489>>2]|0;
    ;HEAP8[$488>>0]=HEAP8[$493>>0]|0;
    HEAP32[$485>>2] = $1673;
    HEAP32[$486>>2] = $1674;
    HEAP32[$487>>2] = $1676;
    $1677 = HEAP32[$485>>2]|0;
    $1678 = HEAP32[$486>>2]|0;
    $1679 = HEAP32[$487>>2]|0;
    HEAP32[$484>>2] = $1679;
    $1680 = HEAP32[$484>>2]|0;
    HEAP32[$481>>2] = $1677;
    HEAP32[$482>>2] = $1678;
    HEAP32[$483>>2] = $1680;
    $1681 = HEAP32[$482>>2]|0;
    $1682 = HEAP32[$483>>2]|0;
    HEAP32[$480>>2] = $1682;
    $1683 = HEAP32[$480>>2]|0;
    $1684 = +HEAPF32[$1683>>2];
    HEAPF32[$1681>>2] = $1684;
    HEAP32[$494>>2] = $504;
    $1685 = ((($1655)) + 4|0);
    $1686 = HEAP32[$1685>>2]|0;
    $1687 = ((($1686)) + 4|0);
    HEAP32[$1685>>2] = $1687;
   } else {
    $1688 = HEAP32[$503>>2]|0;
    HEAP32[$495>>2] = $1688;
    $1689 = HEAP32[$495>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1655,$1689);
   }
   $1690 = ((($1030)) + 4|0);
   HEAP32[$472>>2] = $1016;
   HEAP32[$473>>2] = $1690;
   $1691 = HEAP32[$472>>2]|0;
   $1692 = ((($1691)) + 4|0);
   $1693 = HEAP32[$1692>>2]|0;
   HEAP32[$471>>2] = $1691;
   $1694 = HEAP32[$471>>2]|0;
   $1695 = ((($1694)) + 8|0);
   HEAP32[$470>>2] = $1695;
   $1696 = HEAP32[$470>>2]|0;
   HEAP32[$469>>2] = $1696;
   $1697 = HEAP32[$469>>2]|0;
   $1698 = HEAP32[$1697>>2]|0;
   $1699 = ($1693>>>0)<($1698>>>0);
   if ($1699) {
    HEAP32[$466>>2] = $474;
    HEAP32[$467>>2] = $1691;
    HEAP32[$468>>2] = 1;
    HEAP32[$447>>2] = $1691;
    $1700 = HEAP32[$447>>2]|0;
    $1701 = ((($1700)) + 8|0);
    HEAP32[$446>>2] = $1701;
    $1702 = HEAP32[$446>>2]|0;
    HEAP32[$445>>2] = $1702;
    $1703 = HEAP32[$445>>2]|0;
    $1704 = ((($1691)) + 4|0);
    $1705 = HEAP32[$1704>>2]|0;
    HEAP32[$448>>2] = $1705;
    $1706 = HEAP32[$448>>2]|0;
    $1707 = HEAP32[$473>>2]|0;
    HEAP32[$449>>2] = $1707;
    $1708 = HEAP32[$449>>2]|0;
    HEAP32[$460>>2] = $1703;
    HEAP32[$461>>2] = $1706;
    HEAP32[$462>>2] = $1708;
    $1709 = HEAP32[$460>>2]|0;
    $1710 = HEAP32[$461>>2]|0;
    $1711 = HEAP32[$462>>2]|0;
    HEAP32[$459>>2] = $1711;
    $1712 = HEAP32[$459>>2]|0;
    ;HEAP8[$458>>0]=HEAP8[$463>>0]|0;
    HEAP32[$455>>2] = $1709;
    HEAP32[$456>>2] = $1710;
    HEAP32[$457>>2] = $1712;
    $1713 = HEAP32[$455>>2]|0;
    $1714 = HEAP32[$456>>2]|0;
    $1715 = HEAP32[$457>>2]|0;
    HEAP32[$454>>2] = $1715;
    $1716 = HEAP32[$454>>2]|0;
    HEAP32[$451>>2] = $1713;
    HEAP32[$452>>2] = $1714;
    HEAP32[$453>>2] = $1716;
    $1717 = HEAP32[$452>>2]|0;
    $1718 = HEAP32[$453>>2]|0;
    HEAP32[$450>>2] = $1718;
    $1719 = HEAP32[$450>>2]|0;
    $1720 = +HEAPF32[$1719>>2];
    HEAPF32[$1717>>2] = $1720;
    HEAP32[$464>>2] = $474;
    $1721 = ((($1691)) + 4|0);
    $1722 = HEAP32[$1721>>2]|0;
    $1723 = ((($1722)) + 4|0);
    HEAP32[$1721>>2] = $1723;
   } else {
    $1724 = HEAP32[$473>>2]|0;
    HEAP32[$465>>2] = $1724;
    $1725 = HEAP32[$465>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1691,$1725);
   }
   $1726 = ((($1030)) + 8|0);
   HEAP32[$442>>2] = $1016;
   HEAP32[$443>>2] = $1726;
   $1727 = HEAP32[$442>>2]|0;
   $1728 = ((($1727)) + 4|0);
   $1729 = HEAP32[$1728>>2]|0;
   HEAP32[$441>>2] = $1727;
   $1730 = HEAP32[$441>>2]|0;
   $1731 = ((($1730)) + 8|0);
   HEAP32[$440>>2] = $1731;
   $1732 = HEAP32[$440>>2]|0;
   HEAP32[$439>>2] = $1732;
   $1733 = HEAP32[$439>>2]|0;
   $1734 = HEAP32[$1733>>2]|0;
   $1735 = ($1729>>>0)<($1734>>>0);
   if ($1735) {
    HEAP32[$436>>2] = $444;
    HEAP32[$437>>2] = $1727;
    HEAP32[$438>>2] = 1;
    HEAP32[$417>>2] = $1727;
    $1736 = HEAP32[$417>>2]|0;
    $1737 = ((($1736)) + 8|0);
    HEAP32[$416>>2] = $1737;
    $1738 = HEAP32[$416>>2]|0;
    HEAP32[$415>>2] = $1738;
    $1739 = HEAP32[$415>>2]|0;
    $1740 = ((($1727)) + 4|0);
    $1741 = HEAP32[$1740>>2]|0;
    HEAP32[$418>>2] = $1741;
    $1742 = HEAP32[$418>>2]|0;
    $1743 = HEAP32[$443>>2]|0;
    HEAP32[$419>>2] = $1743;
    $1744 = HEAP32[$419>>2]|0;
    HEAP32[$430>>2] = $1739;
    HEAP32[$431>>2] = $1742;
    HEAP32[$432>>2] = $1744;
    $1745 = HEAP32[$430>>2]|0;
    $1746 = HEAP32[$431>>2]|0;
    $1747 = HEAP32[$432>>2]|0;
    HEAP32[$429>>2] = $1747;
    $1748 = HEAP32[$429>>2]|0;
    ;HEAP8[$428>>0]=HEAP8[$433>>0]|0;
    HEAP32[$425>>2] = $1745;
    HEAP32[$426>>2] = $1746;
    HEAP32[$427>>2] = $1748;
    $1749 = HEAP32[$425>>2]|0;
    $1750 = HEAP32[$426>>2]|0;
    $1751 = HEAP32[$427>>2]|0;
    HEAP32[$424>>2] = $1751;
    $1752 = HEAP32[$424>>2]|0;
    HEAP32[$421>>2] = $1749;
    HEAP32[$422>>2] = $1750;
    HEAP32[$423>>2] = $1752;
    $1753 = HEAP32[$422>>2]|0;
    $1754 = HEAP32[$423>>2]|0;
    HEAP32[$420>>2] = $1754;
    $1755 = HEAP32[$420>>2]|0;
    $1756 = +HEAPF32[$1755>>2];
    HEAPF32[$1753>>2] = $1756;
    HEAP32[$434>>2] = $444;
    $1757 = ((($1727)) + 4|0);
    $1758 = HEAP32[$1757>>2]|0;
    $1759 = ((($1758)) + 4|0);
    HEAP32[$1757>>2] = $1759;
   } else {
    $1760 = HEAP32[$443>>2]|0;
    HEAP32[$435>>2] = $1760;
    $1761 = HEAP32[$435>>2]|0;
    __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($1727,$1761);
   }
   $1762 = HEAP32[$1029>>2]|0;
   $1763 = (($1762) + 1)|0;
   HEAP32[$1029>>2] = $1763;
  }
  HEAP32[$413>>2] = $1019;
  HEAP32[$414>>2] = $1018;
  $1764 = HEAP32[$413>>2]|0;
  HEAP32[$412>>2] = $1764;
  $1765 = HEAP32[$412>>2]|0;
  $1766 = HEAP32[$414>>2]|0;
  HEAP32[$406>>2] = $1766;
  $1767 = HEAP32[$406>>2]|0;
  HEAP32[$409>>2] = $1031;
  HEAP32[$410>>2] = $1765;
  HEAP32[$411>>2] = $1767;
  $1768 = HEAP32[$409>>2]|0;
  $1769 = HEAP32[$410>>2]|0;
  HEAP32[$408>>2] = $1769;
  $1770 = HEAP32[$408>>2]|0;
  $1771 = HEAP32[$1770>>2]|0;
  HEAP32[$1768>>2] = $1771;
  $1772 = ((($1768)) + 4|0);
  $1773 = HEAP32[$411>>2]|0;
  HEAP32[$407>>2] = $1773;
  $1774 = HEAP32[$407>>2]|0;
  $1775 = HEAP32[$1774>>2]|0;
  HEAP32[$1772>>2] = $1775;
  HEAP32[$403>>2] = $1017;
  HEAP32[$404>>2] = $1031;
  $1776 = HEAP32[$403>>2]|0;
  $1777 = HEAP32[$404>>2]|0;
  HEAP32[$402>>2] = $1777;
  $1778 = HEAP32[$402>>2]|0;
  HEAP32[$393>>2] = $1776;
  HEAP32[$394>>2] = $1778;
  $1779 = HEAP32[$393>>2]|0;
  $1780 = HEAP32[$394>>2]|0;
  HEAP32[$392>>2] = $1780;
  $1781 = HEAP32[$392>>2]|0;
  HEAP32[$389>>2] = $1779;
  HEAP32[$390>>2] = $1781;
  $1782 = HEAP32[$389>>2]|0;
  $1783 = HEAP32[$390>>2]|0;
  HEAP32[$388>>2] = $1783;
  $1784 = HEAP32[$388>>2]|0;
  ;HEAP8[$387>>0]=HEAP8[$391>>0]|0;
  HEAP32[$385>>2] = $1782;
  HEAP32[$386>>2] = $1784;
  $1785 = HEAP32[$385>>2]|0;
  $1786 = HEAP32[$386>>2]|0;
  $1787 = HEAP32[$386>>2]|0;
  HEAP32[$384>>2] = $1787;
  $1788 = HEAP32[$384>>2]|0;
  __ZNSt3__26__treeINS_12__value_typeIPN3sap2ve4core14ThrustlineItemEiEENS_19__map_value_compareIS6_S7_NS_4lessIS6_EELb1EEENS_9allocatorIS7_EEE25__emplace_unique_key_argsIS6_JNS_4pairIS6_jEEEEENSG_INS_15__tree_iteratorIS7_PNS_11__tree_nodeIS7_PvEElEEbEERKT_DpOT0_($405,$1785,$1786,$1788);
  HEAP32[$399>>2] = $1032;
  HEAP32[$400>>2] = $405;
  $1789 = HEAP32[$399>>2]|0;
  $1790 = HEAP32[$400>>2]|0;
  HEAP32[$398>>2] = $1790;
  $1791 = HEAP32[$398>>2]|0;
  ;HEAP32[$401>>2]=HEAP32[$1791>>2]|0;
  ;HEAP8[$396>>0]=HEAP8[$401>>0]|0;HEAP8[$396+1>>0]=HEAP8[$401+1>>0]|0;HEAP8[$396+2>>0]=HEAP8[$401+2>>0]|0;HEAP8[$396+3>>0]=HEAP8[$401+3>>0]|0;
  HEAP32[$395>>2] = $1789;
  $1792 = HEAP32[$395>>2]|0;
  ;HEAP32[$1792>>2]=HEAP32[$396>>2]|0;
  $1793 = ((($1789)) + 4|0);
  $1794 = HEAP32[$400>>2]|0;
  $1795 = ((($1794)) + 4|0);
  HEAP32[$397>>2] = $1795;
  $1796 = HEAP32[$397>>2]|0;
  $1797 = HEAP8[$1796>>0]|0;
  $1798 = $1797&1;
  $1799 = $1798&1;
  HEAP8[$1793>>0] = $1799;
  $1800 = HEAP32[$1018>>2]|0;
  $1801 = (($1800) + 1)|0;
  HEAP32[$1018>>2] = $1801;
 }
 HEAP32[$383>>2] = $1033;
 $1802 = HEAP32[$383>>2]|0;
 HEAP32[$381>>2] = $1802;
 $1803 = HEAP32[$381>>2]|0;
 HEAP32[$380>>2] = $1803;
 HEAP32[$1803>>2] = 0;
 $1804 = ((($1803)) + 4|0);
 HEAP32[$1804>>2] = 0;
 $1805 = ((($1803)) + 8|0);
 HEAP32[$382>>2] = 0;
 HEAP32[$378>>2] = $1805;
 HEAP32[$379>>2] = $382;
 $1806 = HEAP32[$378>>2]|0;
 $1807 = HEAP32[$379>>2]|0;
 HEAP32[$377>>2] = $1807;
 $1808 = HEAP32[$377>>2]|0;
 HEAP32[$373>>2] = $1806;
 HEAP32[$374>>2] = $1808;
 $1809 = HEAP32[$373>>2]|0;
 $1810 = HEAP32[$374>>2]|0;
 HEAP32[$372>>2] = $1810;
 HEAP32[$1809>>2] = 0;
 HEAP32[$376>>2] = $1806;
 $1811 = HEAP32[$376>>2]|0;
 HEAP32[$375>>2] = $1811;
 HEAP32[$371>>2] = $1034;
 $1812 = HEAP32[$371>>2]|0;
 HEAP32[$369>>2] = $1812;
 $1813 = HEAP32[$369>>2]|0;
 HEAP32[$368>>2] = $1813;
 HEAP32[$1813>>2] = 0;
 $1814 = ((($1813)) + 4|0);
 HEAP32[$1814>>2] = 0;
 $1815 = ((($1813)) + 8|0);
 HEAP32[$370>>2] = 0;
 HEAP32[$366>>2] = $1815;
 HEAP32[$367>>2] = $370;
 $1816 = HEAP32[$366>>2]|0;
 $1817 = HEAP32[$367>>2]|0;
 HEAP32[$365>>2] = $1817;
 $1818 = HEAP32[$365>>2]|0;
 HEAP32[$361>>2] = $1816;
 HEAP32[$362>>2] = $1818;
 $1819 = HEAP32[$361>>2]|0;
 $1820 = HEAP32[$362>>2]|0;
 HEAP32[$360>>2] = $1820;
 HEAP32[$1819>>2] = 0;
 HEAP32[$364>>2] = $1816;
 $1821 = HEAP32[$364>>2]|0;
 HEAP32[$363>>2] = $1821;
 HEAP32[$359>>2] = $1035;
 $1822 = HEAP32[$359>>2]|0;
 HEAP32[$357>>2] = $1822;
 $1823 = HEAP32[$357>>2]|0;
 HEAP32[$356>>2] = $1823;
 HEAP32[$1823>>2] = 0;
 $1824 = ((($1823)) + 4|0);
 HEAP32[$1824>>2] = 0;
 $1825 = ((($1823)) + 8|0);
 HEAP32[$358>>2] = 0;
 HEAP32[$354>>2] = $1825;
 HEAP32[$355>>2] = $358;
 $1826 = HEAP32[$354>>2]|0;
 $1827 = HEAP32[$355>>2]|0;
 HEAP32[$353>>2] = $1827;
 $1828 = HEAP32[$353>>2]|0;
 HEAP32[$349>>2] = $1826;
 HEAP32[$350>>2] = $1828;
 $1829 = HEAP32[$349>>2]|0;
 $1830 = HEAP32[$350>>2]|0;
 HEAP32[$348>>2] = $1830;
 HEAP32[$1829>>2] = 0;
 HEAP32[$352>>2] = $1826;
 $1831 = HEAP32[$352>>2]|0;
 HEAP32[$351>>2] = $1831;
 HEAP32[$347>>2] = $1036;
 $1832 = HEAP32[$347>>2]|0;
 HEAP32[$345>>2] = $1832;
 $1833 = HEAP32[$345>>2]|0;
 HEAP32[$344>>2] = $1833;
 HEAP32[$1833>>2] = 0;
 $1834 = ((($1833)) + 4|0);
 HEAP32[$1834>>2] = 0;
 $1835 = ((($1833)) + 8|0);
 HEAP32[$346>>2] = 0;
 HEAP32[$342>>2] = $1835;
 HEAP32[$343>>2] = $346;
 $1836 = HEAP32[$342>>2]|0;
 $1837 = HEAP32[$343>>2]|0;
 HEAP32[$341>>2] = $1837;
 $1838 = HEAP32[$341>>2]|0;
 HEAP32[$337>>2] = $1836;
 HEAP32[$338>>2] = $1838;
 $1839 = HEAP32[$337>>2]|0;
 $1840 = HEAP32[$338>>2]|0;
 HEAP32[$336>>2] = $1840;
 HEAP32[$1839>>2] = 0;
 HEAP32[$340>>2] = $1836;
 $1841 = HEAP32[$340>>2]|0;
 HEAP32[$339>>2] = $1841;
 HEAP32[$335>>2] = $1037;
 $1842 = HEAP32[$335>>2]|0;
 HEAP32[$333>>2] = $1842;
 $1843 = HEAP32[$333>>2]|0;
 HEAP32[$332>>2] = $1843;
 HEAP32[$1843>>2] = 0;
 $1844 = ((($1843)) + 4|0);
 HEAP32[$1844>>2] = 0;
 $1845 = ((($1843)) + 8|0);
 HEAP32[$334>>2] = 0;
 HEAP32[$330>>2] = $1845;
 HEAP32[$331>>2] = $334;
 $1846 = HEAP32[$330>>2]|0;
 $1847 = HEAP32[$331>>2]|0;
 HEAP32[$329>>2] = $1847;
 $1848 = HEAP32[$329>>2]|0;
 HEAP32[$325>>2] = $1846;
 HEAP32[$326>>2] = $1848;
 $1849 = HEAP32[$325>>2]|0;
 $1850 = HEAP32[$326>>2]|0;
 HEAP32[$324>>2] = $1850;
 HEAP32[$1849>>2] = 0;
 HEAP32[$328>>2] = $1846;
 $1851 = HEAP32[$328>>2]|0;
 HEAP32[$327>>2] = $1851;
 HEAP32[$323>>2] = $1038;
 $1852 = HEAP32[$323>>2]|0;
 HEAP32[$321>>2] = $1852;
 $1853 = HEAP32[$321>>2]|0;
 HEAP32[$320>>2] = $1853;
 HEAP32[$1853>>2] = 0;
 $1854 = ((($1853)) + 4|0);
 HEAP32[$1854>>2] = 0;
 $1855 = ((($1853)) + 8|0);
 HEAP32[$322>>2] = 0;
 HEAP32[$318>>2] = $1855;
 HEAP32[$319>>2] = $322;
 $1856 = HEAP32[$318>>2]|0;
 $1857 = HEAP32[$319>>2]|0;
 HEAP32[$317>>2] = $1857;
 $1858 = HEAP32[$317>>2]|0;
 HEAP32[$313>>2] = $1856;
 HEAP32[$314>>2] = $1858;
 $1859 = HEAP32[$313>>2]|0;
 $1860 = HEAP32[$314>>2]|0;
 HEAP32[$312>>2] = $1860;
 HEAP32[$1859>>2] = 0;
 HEAP32[$316>>2] = $1856;
 $1861 = HEAP32[$316>>2]|0;
 HEAP32[$315>>2] = $1861;
 $1862 = HEAP32[$1003>>2]|0;
 $1863 = (__ZNK3sap2ve4core10Thrustline12SegmentCountEv($1862)|0);
 HEAP32[$1039>>2] = $1863;
 HEAP32[$1040>>2] = 0;
 while(1) {
  $1864 = HEAP32[$1040>>2]|0;
  $1865 = HEAP32[$1039>>2]|0;
  $1866 = ($1864>>>0)<($1865>>>0);
  if (!($1866)) {
   break;
  }
  $1867 = HEAP32[$1003>>2]|0;
  $1868 = HEAP32[$1040>>2]|0;
  $1869 = (__ZNK3sap2ve4core10Thrustline7SegmentEm($1867,$1868)|0);
  HEAP32[$1041>>2] = $1869;
  $1870 = HEAP32[$1041>>2]|0;
  $1871 = (__ZNK3sap2ve4core17ThrustlineSegment9ItemStartEv($1870)|0);
  HEAP32[$1043>>2] = $1871;
  HEAP32[$309>>2] = $1017;
  HEAP32[$310>>2] = $1043;
  $1872 = HEAP32[$309>>2]|0;
  $1873 = HEAP32[$310>>2]|0;
  $1874 = (__ZNSt3__26__treeINS_12__value_typeIPN3sap2ve4core14ThrustlineItemEiEENS_19__map_value_compareIS6_S7_NS_4lessIS6_EELb1EEENS_9allocatorIS7_EEE4findIS6_EENS_15__tree_iteratorIS7_PNS_11__tree_nodeIS7_PvEElEERKT_($1872,$1873)|0);
  HEAP32[$311>>2] = $1874;
  ;HEAP8[$307>>0]=HEAP8[$311>>0]|0;HEAP8[$307+1>>0]=HEAP8[$311+1>>0]|0;HEAP8[$307+2>>0]=HEAP8[$311+2>>0]|0;HEAP8[$307+3>>0]=HEAP8[$311+3>>0]|0;
  HEAP32[$306>>2] = $308;
  $1875 = HEAP32[$306>>2]|0;
  ;HEAP32[$1875>>2]=HEAP32[$307>>2]|0;
  $1876 = HEAP32[$308>>2]|0;
  HEAP32[$1042>>2] = $1876;
  $1877 = HEAP32[$1041>>2]|0;
  $1878 = (__ZNK3sap2ve4core17ThrustlineSegment7ItemEndEv($1877)|0);
  HEAP32[$1045>>2] = $1878;
  HEAP32[$303>>2] = $1017;
  HEAP32[$304>>2] = $1045;
  $1879 = HEAP32[$303>>2]|0;
  $1880 = HEAP32[$304>>2]|0;
  $1881 = (__ZNSt3__26__treeINS_12__value_typeIPN3sap2ve4core14ThrustlineItemEiEENS_19__map_value_compareIS6_S7_NS_4lessIS6_EELb1EEENS_9allocatorIS7_EEE4findIS6_EENS_15__tree_iteratorIS7_PNS_11__tree_nodeIS7_PvEElEERKT_($1879,$1880)|0);
  HEAP32[$305>>2] = $1881;
  ;HEAP8[$301>>0]=HEAP8[$305>>0]|0;HEAP8[$301+1>>0]=HEAP8[$305+1>>0]|0;HEAP8[$301+2>>0]=HEAP8[$305+2>>0]|0;HEAP8[$301+3>>0]=HEAP8[$305+3>>0]|0;
  HEAP32[$300>>2] = $302;
  $1882 = HEAP32[$300>>2]|0;
  ;HEAP32[$1882>>2]=HEAP32[$301>>2]|0;
  $1883 = HEAP32[$302>>2]|0;
  HEAP32[$1044>>2] = $1883;
  HEAP32[$298>>2] = $1017;
  $1884 = HEAP32[$298>>2]|0;
  HEAP32[$296>>2] = $1884;
  $1885 = HEAP32[$296>>2]|0;
  HEAP32[$294>>2] = $1885;
  $1886 = HEAP32[$294>>2]|0;
  $1887 = ((($1886)) + 4|0);
  HEAP32[$293>>2] = $1887;
  $1888 = HEAP32[$293>>2]|0;
  HEAP32[$292>>2] = $1888;
  $1889 = HEAP32[$292>>2]|0;
  HEAP32[$291>>2] = $1889;
  $1890 = HEAP32[$291>>2]|0;
  HEAP32[$290>>2] = $1890;
  $1891 = HEAP32[$290>>2]|0;
  HEAP32[$288>>2] = $295;
  HEAP32[$289>>2] = $1891;
  $1892 = HEAP32[$288>>2]|0;
  $1893 = HEAP32[$289>>2]|0;
  HEAP32[$1892>>2] = $1893;
  $1894 = HEAP32[$295>>2]|0;
  HEAP32[$299>>2] = $1894;
  ;HEAP8[$287>>0]=HEAP8[$299>>0]|0;HEAP8[$287+1>>0]=HEAP8[$299+1>>0]|0;HEAP8[$287+2>>0]=HEAP8[$299+2>>0]|0;HEAP8[$287+3>>0]=HEAP8[$299+3>>0]|0;
  HEAP32[$286>>2] = $297;
  $1895 = HEAP32[$286>>2]|0;
  ;HEAP32[$1895>>2]=HEAP32[$287>>2]|0;
  $1896 = HEAP32[$297>>2]|0;
  HEAP32[$1046>>2] = $1896;
  HEAP32[$284>>2] = $1042;
  HEAP32[$285>>2] = $1046;
  $1897 = HEAP32[$284>>2]|0;
  $1898 = HEAP32[$285>>2]|0;
  HEAP32[$282>>2] = $1897;
  HEAP32[$283>>2] = $1898;
  $1899 = HEAP32[$282>>2]|0;
  $1900 = HEAP32[$283>>2]|0;
  HEAP32[$280>>2] = $1899;
  HEAP32[$281>>2] = $1900;
  $1901 = HEAP32[$280>>2]|0;
  $1902 = HEAP32[$1901>>2]|0;
  $1903 = HEAP32[$281>>2]|0;
  $1904 = HEAP32[$1903>>2]|0;
  $1905 = ($1902|0)==($1904|0);
  $1906 = $1905 ^ 1;
  L83: do {
   if ($1906) {
    HEAP32[$278>>2] = $1017;
    $1907 = HEAP32[$278>>2]|0;
    HEAP32[$276>>2] = $1907;
    $1908 = HEAP32[$276>>2]|0;
    HEAP32[$274>>2] = $1908;
    $1909 = HEAP32[$274>>2]|0;
    $1910 = ((($1909)) + 4|0);
    HEAP32[$273>>2] = $1910;
    $1911 = HEAP32[$273>>2]|0;
    HEAP32[$272>>2] = $1911;
    $1912 = HEAP32[$272>>2]|0;
    HEAP32[$271>>2] = $1912;
    $1913 = HEAP32[$271>>2]|0;
    HEAP32[$270>>2] = $1913;
    $1914 = HEAP32[$270>>2]|0;
    HEAP32[$268>>2] = $275;
    HEAP32[$269>>2] = $1914;
    $1915 = HEAP32[$268>>2]|0;
    $1916 = HEAP32[$269>>2]|0;
    HEAP32[$1915>>2] = $1916;
    $1917 = HEAP32[$275>>2]|0;
    HEAP32[$279>>2] = $1917;
    ;HEAP8[$267>>0]=HEAP8[$279>>0]|0;HEAP8[$267+1>>0]=HEAP8[$279+1>>0]|0;HEAP8[$267+2>>0]=HEAP8[$279+2>>0]|0;HEAP8[$267+3>>0]=HEAP8[$279+3>>0]|0;
    HEAP32[$266>>2] = $277;
    $1918 = HEAP32[$266>>2]|0;
    ;HEAP32[$1918>>2]=HEAP32[$267>>2]|0;
    $1919 = HEAP32[$277>>2]|0;
    HEAP32[$1047>>2] = $1919;
    HEAP32[$264>>2] = $1044;
    HEAP32[$265>>2] = $1047;
    $1920 = HEAP32[$264>>2]|0;
    $1921 = HEAP32[$265>>2]|0;
    HEAP32[$262>>2] = $1920;
    HEAP32[$263>>2] = $1921;
    $1922 = HEAP32[$262>>2]|0;
    $1923 = HEAP32[$263>>2]|0;
    HEAP32[$260>>2] = $1922;
    HEAP32[$261>>2] = $1923;
    $1924 = HEAP32[$260>>2]|0;
    $1925 = HEAP32[$1924>>2]|0;
    $1926 = HEAP32[$261>>2]|0;
    $1927 = HEAP32[$1926>>2]|0;
    $1928 = ($1925|0)==($1927|0);
    $1929 = $1928 ^ 1;
    if ($1929) {
     HEAP32[$259>>2] = $1042;
     $1930 = HEAP32[$259>>2]|0;
     HEAP32[$258>>2] = $1930;
     $1931 = HEAP32[$258>>2]|0;
     HEAP32[$257>>2] = $1931;
     $1932 = HEAP32[$257>>2]|0;
     $1933 = HEAP32[$1932>>2]|0;
     $1934 = ((($1933)) + 16|0);
     HEAP32[$256>>2] = $1934;
     $1935 = HEAP32[$256>>2]|0;
     HEAP32[$255>>2] = $1935;
     $1936 = HEAP32[$255>>2]|0;
     HEAP32[$254>>2] = $1936;
     $1937 = HEAP32[$254>>2]|0;
     HEAP32[$253>>2] = $1937;
     $1938 = HEAP32[$253>>2]|0;
     $1939 = ((($1938)) + 4|0);
     HEAP32[$250>>2] = $1033;
     HEAP32[$251>>2] = $1939;
     $1940 = HEAP32[$250>>2]|0;
     $1941 = ((($1940)) + 4|0);
     $1942 = HEAP32[$1941>>2]|0;
     HEAP32[$249>>2] = $1940;
     $1943 = HEAP32[$249>>2]|0;
     $1944 = ((($1943)) + 8|0);
     HEAP32[$248>>2] = $1944;
     $1945 = HEAP32[$248>>2]|0;
     HEAP32[$247>>2] = $1945;
     $1946 = HEAP32[$247>>2]|0;
     $1947 = HEAP32[$1946>>2]|0;
     $1948 = ($1942>>>0)<($1947>>>0);
     if ($1948) {
      HEAP32[$244>>2] = $252;
      HEAP32[$245>>2] = $1940;
      HEAP32[$246>>2] = 1;
      HEAP32[$225>>2] = $1940;
      $1949 = HEAP32[$225>>2]|0;
      $1950 = ((($1949)) + 8|0);
      HEAP32[$224>>2] = $1950;
      $1951 = HEAP32[$224>>2]|0;
      HEAP32[$223>>2] = $1951;
      $1952 = HEAP32[$223>>2]|0;
      $1953 = ((($1940)) + 4|0);
      $1954 = HEAP32[$1953>>2]|0;
      HEAP32[$226>>2] = $1954;
      $1955 = HEAP32[$226>>2]|0;
      $1956 = HEAP32[$251>>2]|0;
      HEAP32[$227>>2] = $1956;
      $1957 = HEAP32[$227>>2]|0;
      HEAP32[$238>>2] = $1952;
      HEAP32[$239>>2] = $1955;
      HEAP32[$240>>2] = $1957;
      $1958 = HEAP32[$238>>2]|0;
      $1959 = HEAP32[$239>>2]|0;
      $1960 = HEAP32[$240>>2]|0;
      HEAP32[$237>>2] = $1960;
      $1961 = HEAP32[$237>>2]|0;
      ;HEAP8[$236>>0]=HEAP8[$241>>0]|0;
      HEAP32[$233>>2] = $1958;
      HEAP32[$234>>2] = $1959;
      HEAP32[$235>>2] = $1961;
      $1962 = HEAP32[$233>>2]|0;
      $1963 = HEAP32[$234>>2]|0;
      $1964 = HEAP32[$235>>2]|0;
      HEAP32[$232>>2] = $1964;
      $1965 = HEAP32[$232>>2]|0;
      HEAP32[$229>>2] = $1962;
      HEAP32[$230>>2] = $1963;
      HEAP32[$231>>2] = $1965;
      $1966 = HEAP32[$230>>2]|0;
      $1967 = HEAP32[$231>>2]|0;
      HEAP32[$228>>2] = $1967;
      $1968 = HEAP32[$228>>2]|0;
      $1969 = HEAP32[$1968>>2]|0;
      HEAP32[$1966>>2] = $1969;
      HEAP32[$242>>2] = $252;
      $1970 = ((($1940)) + 4|0);
      $1971 = HEAP32[$1970>>2]|0;
      $1972 = ((($1971)) + 4|0);
      HEAP32[$1970>>2] = $1972;
     } else {
      $1973 = HEAP32[$251>>2]|0;
      HEAP32[$243>>2] = $1973;
      $1974 = HEAP32[$243>>2]|0;
      __ZNSt3__26vectorIjNS_9allocatorIjEEE24__emplace_back_slow_pathIJRiEEEvDpOT_($1940,$1974);
     }
     HEAP32[$222>>2] = $1044;
     $1975 = HEAP32[$222>>2]|0;
     HEAP32[$221>>2] = $1975;
     $1976 = HEAP32[$221>>2]|0;
     HEAP32[$220>>2] = $1976;
     $1977 = HEAP32[$220>>2]|0;
     $1978 = HEAP32[$1977>>2]|0;
     $1979 = ((($1978)) + 16|0);
     HEAP32[$219>>2] = $1979;
     $1980 = HEAP32[$219>>2]|0;
     HEAP32[$218>>2] = $1980;
     $1981 = HEAP32[$218>>2]|0;
     HEAP32[$217>>2] = $1981;
     $1982 = HEAP32[$217>>2]|0;
     HEAP32[$216>>2] = $1982;
     $1983 = HEAP32[$216>>2]|0;
     $1984 = ((($1983)) + 4|0);
     HEAP32[$213>>2] = $1034;
     HEAP32[$214>>2] = $1984;
     $1985 = HEAP32[$213>>2]|0;
     $1986 = ((($1985)) + 4|0);
     $1987 = HEAP32[$1986>>2]|0;
     HEAP32[$212>>2] = $1985;
     $1988 = HEAP32[$212>>2]|0;
     $1989 = ((($1988)) + 8|0);
     HEAP32[$211>>2] = $1989;
     $1990 = HEAP32[$211>>2]|0;
     HEAP32[$210>>2] = $1990;
     $1991 = HEAP32[$210>>2]|0;
     $1992 = HEAP32[$1991>>2]|0;
     $1993 = ($1987>>>0)<($1992>>>0);
     if ($1993) {
      HEAP32[$207>>2] = $215;
      HEAP32[$208>>2] = $1985;
      HEAP32[$209>>2] = 1;
      HEAP32[$188>>2] = $1985;
      $1994 = HEAP32[$188>>2]|0;
      $1995 = ((($1994)) + 8|0);
      HEAP32[$187>>2] = $1995;
      $1996 = HEAP32[$187>>2]|0;
      HEAP32[$186>>2] = $1996;
      $1997 = HEAP32[$186>>2]|0;
      $1998 = ((($1985)) + 4|0);
      $1999 = HEAP32[$1998>>2]|0;
      HEAP32[$189>>2] = $1999;
      $2000 = HEAP32[$189>>2]|0;
      $2001 = HEAP32[$214>>2]|0;
      HEAP32[$190>>2] = $2001;
      $2002 = HEAP32[$190>>2]|0;
      HEAP32[$201>>2] = $1997;
      HEAP32[$202>>2] = $2000;
      HEAP32[$203>>2] = $2002;
      $2003 = HEAP32[$201>>2]|0;
      $2004 = HEAP32[$202>>2]|0;
      $2005 = HEAP32[$203>>2]|0;
      HEAP32[$200>>2] = $2005;
      $2006 = HEAP32[$200>>2]|0;
      ;HEAP8[$199>>0]=HEAP8[$204>>0]|0;
      HEAP32[$196>>2] = $2003;
      HEAP32[$197>>2] = $2004;
      HEAP32[$198>>2] = $2006;
      $2007 = HEAP32[$196>>2]|0;
      $2008 = HEAP32[$197>>2]|0;
      $2009 = HEAP32[$198>>2]|0;
      HEAP32[$195>>2] = $2009;
      $2010 = HEAP32[$195>>2]|0;
      HEAP32[$192>>2] = $2007;
      HEAP32[$193>>2] = $2008;
      HEAP32[$194>>2] = $2010;
      $2011 = HEAP32[$193>>2]|0;
      $2012 = HEAP32[$194>>2]|0;
      HEAP32[$191>>2] = $2012;
      $2013 = HEAP32[$191>>2]|0;
      $2014 = HEAP32[$2013>>2]|0;
      HEAP32[$2011>>2] = $2014;
      HEAP32[$205>>2] = $215;
      $2015 = ((($1985)) + 4|0);
      $2016 = HEAP32[$2015>>2]|0;
      $2017 = ((($2016)) + 4|0);
      HEAP32[$2015>>2] = $2017;
     } else {
      $2018 = HEAP32[$214>>2]|0;
      HEAP32[$206>>2] = $2018;
      $2019 = HEAP32[$206>>2]|0;
      __ZNSt3__26vectorIjNS_9allocatorIjEEE24__emplace_back_slow_pathIJRiEEEvDpOT_($1985,$2019);
     }
     $2020 = HEAP32[$1041>>2]|0;
     $2021 = (__ZNK3sap2ve4core17ThrustlineSegment15StartBoundIndexEv($2020)|0);
     HEAP32[$1048>>2] = $2021;
     HEAP32[$183>>2] = $1035;
     HEAP32[$184>>2] = $1048;
     $2022 = HEAP32[$183>>2]|0;
     $2023 = ((($2022)) + 4|0);
     $2024 = HEAP32[$2023>>2]|0;
     HEAP32[$182>>2] = $2022;
     $2025 = HEAP32[$182>>2]|0;
     $2026 = ((($2025)) + 8|0);
     HEAP32[$181>>2] = $2026;
     $2027 = HEAP32[$181>>2]|0;
     HEAP32[$180>>2] = $2027;
     $2028 = HEAP32[$180>>2]|0;
     $2029 = HEAP32[$2028>>2]|0;
     $2030 = ($2024>>>0)<($2029>>>0);
     if ($2030) {
      HEAP32[$177>>2] = $185;
      HEAP32[$178>>2] = $2022;
      HEAP32[$179>>2] = 1;
      HEAP32[$158>>2] = $2022;
      $2031 = HEAP32[$158>>2]|0;
      $2032 = ((($2031)) + 8|0);
      HEAP32[$157>>2] = $2032;
      $2033 = HEAP32[$157>>2]|0;
      HEAP32[$156>>2] = $2033;
      $2034 = HEAP32[$156>>2]|0;
      $2035 = ((($2022)) + 4|0);
      $2036 = HEAP32[$2035>>2]|0;
      HEAP32[$159>>2] = $2036;
      $2037 = HEAP32[$159>>2]|0;
      $2038 = HEAP32[$184>>2]|0;
      HEAP32[$160>>2] = $2038;
      $2039 = HEAP32[$160>>2]|0;
      HEAP32[$171>>2] = $2034;
      HEAP32[$172>>2] = $2037;
      HEAP32[$173>>2] = $2039;
      $2040 = HEAP32[$171>>2]|0;
      $2041 = HEAP32[$172>>2]|0;
      $2042 = HEAP32[$173>>2]|0;
      HEAP32[$170>>2] = $2042;
      $2043 = HEAP32[$170>>2]|0;
      ;HEAP8[$169>>0]=HEAP8[$174>>0]|0;
      HEAP32[$166>>2] = $2040;
      HEAP32[$167>>2] = $2041;
      HEAP32[$168>>2] = $2043;
      $2044 = HEAP32[$166>>2]|0;
      $2045 = HEAP32[$167>>2]|0;
      $2046 = HEAP32[$168>>2]|0;
      HEAP32[$165>>2] = $2046;
      $2047 = HEAP32[$165>>2]|0;
      HEAP32[$162>>2] = $2044;
      HEAP32[$163>>2] = $2045;
      HEAP32[$164>>2] = $2047;
      $2048 = HEAP32[$163>>2]|0;
      $2049 = HEAP32[$164>>2]|0;
      HEAP32[$161>>2] = $2049;
      $2050 = HEAP32[$161>>2]|0;
      $2051 = HEAP32[$2050>>2]|0;
      HEAP32[$2048>>2] = $2051;
      HEAP32[$175>>2] = $185;
      $2052 = ((($2022)) + 4|0);
      $2053 = HEAP32[$2052>>2]|0;
      $2054 = ((($2053)) + 4|0);
      HEAP32[$2052>>2] = $2054;
     } else {
      $2055 = HEAP32[$184>>2]|0;
      HEAP32[$176>>2] = $2055;
      $2056 = HEAP32[$176>>2]|0;
      __ZNSt3__26vectorIjNS_9allocatorIjEEE24__emplace_back_slow_pathIJmEEEvDpOT_($2022,$2056);
     }
     $2057 = HEAP32[$1041>>2]|0;
     $2058 = (__ZNK3sap2ve4core17ThrustlineSegment13EndBoundIndexEv($2057)|0);
     HEAP32[$1049>>2] = $2058;
     HEAP32[$153>>2] = $1036;
     HEAP32[$154>>2] = $1049;
     $2059 = HEAP32[$153>>2]|0;
     $2060 = ((($2059)) + 4|0);
     $2061 = HEAP32[$2060>>2]|0;
     HEAP32[$152>>2] = $2059;
     $2062 = HEAP32[$152>>2]|0;
     $2063 = ((($2062)) + 8|0);
     HEAP32[$151>>2] = $2063;
     $2064 = HEAP32[$151>>2]|0;
     HEAP32[$150>>2] = $2064;
     $2065 = HEAP32[$150>>2]|0;
     $2066 = HEAP32[$2065>>2]|0;
     $2067 = ($2061>>>0)<($2066>>>0);
     if ($2067) {
      HEAP32[$147>>2] = $155;
      HEAP32[$148>>2] = $2059;
      HEAP32[$149>>2] = 1;
      HEAP32[$128>>2] = $2059;
      $2068 = HEAP32[$128>>2]|0;
      $2069 = ((($2068)) + 8|0);
      HEAP32[$127>>2] = $2069;
      $2070 = HEAP32[$127>>2]|0;
      HEAP32[$126>>2] = $2070;
      $2071 = HEAP32[$126>>2]|0;
      $2072 = ((($2059)) + 4|0);
      $2073 = HEAP32[$2072>>2]|0;
      HEAP32[$129>>2] = $2073;
      $2074 = HEAP32[$129>>2]|0;
      $2075 = HEAP32[$154>>2]|0;
      HEAP32[$130>>2] = $2075;
      $2076 = HEAP32[$130>>2]|0;
      HEAP32[$141>>2] = $2071;
      HEAP32[$142>>2] = $2074;
      HEAP32[$143>>2] = $2076;
      $2077 = HEAP32[$141>>2]|0;
      $2078 = HEAP32[$142>>2]|0;
      $2079 = HEAP32[$143>>2]|0;
      HEAP32[$140>>2] = $2079;
      $2080 = HEAP32[$140>>2]|0;
      ;HEAP8[$139>>0]=HEAP8[$144>>0]|0;
      HEAP32[$136>>2] = $2077;
      HEAP32[$137>>2] = $2078;
      HEAP32[$138>>2] = $2080;
      $2081 = HEAP32[$136>>2]|0;
      $2082 = HEAP32[$137>>2]|0;
      $2083 = HEAP32[$138>>2]|0;
      HEAP32[$135>>2] = $2083;
      $2084 = HEAP32[$135>>2]|0;
      HEAP32[$132>>2] = $2081;
      HEAP32[$133>>2] = $2082;
      HEAP32[$134>>2] = $2084;
      $2085 = HEAP32[$133>>2]|0;
      $2086 = HEAP32[$134>>2]|0;
      HEAP32[$131>>2] = $2086;
      $2087 = HEAP32[$131>>2]|0;
      $2088 = HEAP32[$2087>>2]|0;
      HEAP32[$2085>>2] = $2088;
      HEAP32[$145>>2] = $155;
      $2089 = ((($2059)) + 4|0);
      $2090 = HEAP32[$2089>>2]|0;
      $2091 = ((($2090)) + 4|0);
      HEAP32[$2089>>2] = $2091;
     } else {
      $2092 = HEAP32[$154>>2]|0;
      HEAP32[$146>>2] = $2092;
      $2093 = HEAP32[$146>>2]|0;
      __ZNSt3__26vectorIjNS_9allocatorIjEEE24__emplace_back_slow_pathIJmEEEvDpOT_($2059,$2093);
     }
     $2094 = HEAP32[$1041>>2]|0;
     $2095 = (__ZNK3sap2ve4core17ThrustlineSegment17SegmentRatioCountEv($2094)|0);
     HEAP32[$1050>>2] = $2095;
     HEAP32[$123>>2] = $1037;
     HEAP32[$124>>2] = $1050;
     $2096 = HEAP32[$123>>2]|0;
     $2097 = ((($2096)) + 4|0);
     $2098 = HEAP32[$2097>>2]|0;
     HEAP32[$122>>2] = $2096;
     $2099 = HEAP32[$122>>2]|0;
     $2100 = ((($2099)) + 8|0);
     HEAP32[$121>>2] = $2100;
     $2101 = HEAP32[$121>>2]|0;
     HEAP32[$120>>2] = $2101;
     $2102 = HEAP32[$120>>2]|0;
     $2103 = HEAP32[$2102>>2]|0;
     $2104 = ($2098>>>0)<($2103>>>0);
     if ($2104) {
      HEAP32[$117>>2] = $125;
      HEAP32[$118>>2] = $2096;
      HEAP32[$119>>2] = 1;
      HEAP32[$98>>2] = $2096;
      $2105 = HEAP32[$98>>2]|0;
      $2106 = ((($2105)) + 8|0);
      HEAP32[$97>>2] = $2106;
      $2107 = HEAP32[$97>>2]|0;
      HEAP32[$96>>2] = $2107;
      $2108 = HEAP32[$96>>2]|0;
      $2109 = ((($2096)) + 4|0);
      $2110 = HEAP32[$2109>>2]|0;
      HEAP32[$99>>2] = $2110;
      $2111 = HEAP32[$99>>2]|0;
      $2112 = HEAP32[$124>>2]|0;
      HEAP32[$100>>2] = $2112;
      $2113 = HEAP32[$100>>2]|0;
      HEAP32[$111>>2] = $2108;
      HEAP32[$112>>2] = $2111;
      HEAP32[$113>>2] = $2113;
      $2114 = HEAP32[$111>>2]|0;
      $2115 = HEAP32[$112>>2]|0;
      $2116 = HEAP32[$113>>2]|0;
      HEAP32[$110>>2] = $2116;
      $2117 = HEAP32[$110>>2]|0;
      ;HEAP8[$109>>0]=HEAP8[$114>>0]|0;
      HEAP32[$106>>2] = $2114;
      HEAP32[$107>>2] = $2115;
      HEAP32[$108>>2] = $2117;
      $2118 = HEAP32[$106>>2]|0;
      $2119 = HEAP32[$107>>2]|0;
      $2120 = HEAP32[$108>>2]|0;
      HEAP32[$105>>2] = $2120;
      $2121 = HEAP32[$105>>2]|0;
      HEAP32[$102>>2] = $2118;
      HEAP32[$103>>2] = $2119;
      HEAP32[$104>>2] = $2121;
      $2122 = HEAP32[$103>>2]|0;
      $2123 = HEAP32[$104>>2]|0;
      HEAP32[$101>>2] = $2123;
      $2124 = HEAP32[$101>>2]|0;
      $2125 = HEAP32[$2124>>2]|0;
      HEAP32[$2122>>2] = $2125;
      HEAP32[$115>>2] = $125;
      $2126 = ((($2096)) + 4|0);
      $2127 = HEAP32[$2126>>2]|0;
      $2128 = ((($2127)) + 4|0);
      HEAP32[$2126>>2] = $2128;
     } else {
      $2129 = HEAP32[$124>>2]|0;
      HEAP32[$116>>2] = $2129;
      $2130 = HEAP32[$116>>2]|0;
      __ZNSt3__26vectorIjNS_9allocatorIjEEE24__emplace_back_slow_pathIJRjEEEvDpOT_($2096,$2130);
     }
     HEAP32[$1051>>2] = 0;
     while(1) {
      $2131 = HEAP32[$1051>>2]|0;
      $2132 = HEAP32[$1050>>2]|0;
      $2133 = ($2131>>>0)<($2132>>>0);
      if (!($2133)) {
       break L83;
      }
      $2134 = HEAP32[$1041>>2]|0;
      $2135 = HEAP32[$1051>>2]|0;
      __ZNK3sap2ve4core17ThrustlineSegment12SegmentRatioEm($1052,$2134,$2135);
      HEAP32[$93>>2] = $1038;
      HEAP32[$94>>2] = $1052;
      $2136 = HEAP32[$93>>2]|0;
      $2137 = ((($2136)) + 4|0);
      $2138 = HEAP32[$2137>>2]|0;
      HEAP32[$92>>2] = $2136;
      $2139 = HEAP32[$92>>2]|0;
      $2140 = ((($2139)) + 8|0);
      HEAP32[$91>>2] = $2140;
      $2141 = HEAP32[$91>>2]|0;
      HEAP32[$90>>2] = $2141;
      $2142 = HEAP32[$90>>2]|0;
      $2143 = HEAP32[$2142>>2]|0;
      $2144 = ($2138>>>0)<($2143>>>0);
      if ($2144) {
       HEAP32[$87>>2] = $95;
       HEAP32[$88>>2] = $2136;
       HEAP32[$89>>2] = 1;
       HEAP32[$68>>2] = $2136;
       $2145 = HEAP32[$68>>2]|0;
       $2146 = ((($2145)) + 8|0);
       HEAP32[$67>>2] = $2146;
       $2147 = HEAP32[$67>>2]|0;
       HEAP32[$66>>2] = $2147;
       $2148 = HEAP32[$66>>2]|0;
       $2149 = ((($2136)) + 4|0);
       $2150 = HEAP32[$2149>>2]|0;
       HEAP32[$69>>2] = $2150;
       $2151 = HEAP32[$69>>2]|0;
       $2152 = HEAP32[$94>>2]|0;
       HEAP32[$70>>2] = $2152;
       $2153 = HEAP32[$70>>2]|0;
       HEAP32[$81>>2] = $2148;
       HEAP32[$82>>2] = $2151;
       HEAP32[$83>>2] = $2153;
       $2154 = HEAP32[$81>>2]|0;
       $2155 = HEAP32[$82>>2]|0;
       $2156 = HEAP32[$83>>2]|0;
       HEAP32[$80>>2] = $2156;
       $2157 = HEAP32[$80>>2]|0;
       ;HEAP8[$79>>0]=HEAP8[$84>>0]|0;
       HEAP32[$76>>2] = $2154;
       HEAP32[$77>>2] = $2155;
       HEAP32[$78>>2] = $2157;
       $2158 = HEAP32[$76>>2]|0;
       $2159 = HEAP32[$77>>2]|0;
       $2160 = HEAP32[$78>>2]|0;
       HEAP32[$75>>2] = $2160;
       $2161 = HEAP32[$75>>2]|0;
       HEAP32[$72>>2] = $2158;
       HEAP32[$73>>2] = $2159;
       HEAP32[$74>>2] = $2161;
       $2162 = HEAP32[$73>>2]|0;
       $2163 = HEAP32[$74>>2]|0;
       HEAP32[$71>>2] = $2163;
       $2164 = HEAP32[$71>>2]|0;
       $2165 = +HEAPF32[$2164>>2];
       HEAPF32[$2162>>2] = $2165;
       HEAP32[$85>>2] = $95;
       $2166 = ((($2136)) + 4|0);
       $2167 = HEAP32[$2166>>2]|0;
       $2168 = ((($2167)) + 4|0);
       HEAP32[$2166>>2] = $2168;
      } else {
       $2169 = HEAP32[$94>>2]|0;
       HEAP32[$86>>2] = $2169;
       $2170 = HEAP32[$86>>2]|0;
       __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($2136,$2170);
      }
      $2171 = ((($1052)) + 4|0);
      HEAP32[$63>>2] = $1038;
      HEAP32[$64>>2] = $2171;
      $2172 = HEAP32[$63>>2]|0;
      $2173 = ((($2172)) + 4|0);
      $2174 = HEAP32[$2173>>2]|0;
      HEAP32[$62>>2] = $2172;
      $2175 = HEAP32[$62>>2]|0;
      $2176 = ((($2175)) + 8|0);
      HEAP32[$61>>2] = $2176;
      $2177 = HEAP32[$61>>2]|0;
      HEAP32[$60>>2] = $2177;
      $2178 = HEAP32[$60>>2]|0;
      $2179 = HEAP32[$2178>>2]|0;
      $2180 = ($2174>>>0)<($2179>>>0);
      if ($2180) {
       HEAP32[$57>>2] = $65;
       HEAP32[$58>>2] = $2172;
       HEAP32[$59>>2] = 1;
       HEAP32[$38>>2] = $2172;
       $2181 = HEAP32[$38>>2]|0;
       $2182 = ((($2181)) + 8|0);
       HEAP32[$37>>2] = $2182;
       $2183 = HEAP32[$37>>2]|0;
       HEAP32[$36>>2] = $2183;
       $2184 = HEAP32[$36>>2]|0;
       $2185 = ((($2172)) + 4|0);
       $2186 = HEAP32[$2185>>2]|0;
       HEAP32[$39>>2] = $2186;
       $2187 = HEAP32[$39>>2]|0;
       $2188 = HEAP32[$64>>2]|0;
       HEAP32[$40>>2] = $2188;
       $2189 = HEAP32[$40>>2]|0;
       HEAP32[$51>>2] = $2184;
       HEAP32[$52>>2] = $2187;
       HEAP32[$53>>2] = $2189;
       $2190 = HEAP32[$51>>2]|0;
       $2191 = HEAP32[$52>>2]|0;
       $2192 = HEAP32[$53>>2]|0;
       HEAP32[$50>>2] = $2192;
       $2193 = HEAP32[$50>>2]|0;
       ;HEAP8[$49>>0]=HEAP8[$54>>0]|0;
       HEAP32[$46>>2] = $2190;
       HEAP32[$47>>2] = $2191;
       HEAP32[$48>>2] = $2193;
       $2194 = HEAP32[$46>>2]|0;
       $2195 = HEAP32[$47>>2]|0;
       $2196 = HEAP32[$48>>2]|0;
       HEAP32[$45>>2] = $2196;
       $2197 = HEAP32[$45>>2]|0;
       HEAP32[$42>>2] = $2194;
       HEAP32[$43>>2] = $2195;
       HEAP32[$44>>2] = $2197;
       $2198 = HEAP32[$43>>2]|0;
       $2199 = HEAP32[$44>>2]|0;
       HEAP32[$41>>2] = $2199;
       $2200 = HEAP32[$41>>2]|0;
       $2201 = +HEAPF32[$2200>>2];
       HEAPF32[$2198>>2] = $2201;
       HEAP32[$55>>2] = $65;
       $2202 = ((($2172)) + 4|0);
       $2203 = HEAP32[$2202>>2]|0;
       $2204 = ((($2203)) + 4|0);
       HEAP32[$2202>>2] = $2204;
      } else {
       $2205 = HEAP32[$64>>2]|0;
       HEAP32[$56>>2] = $2205;
       $2206 = HEAP32[$56>>2]|0;
       __ZNSt3__26vectorIfNS_9allocatorIfEEE24__emplace_back_slow_pathIJRfEEEvDpOT_($2172,$2206);
      }
      $2207 = HEAP32[$1051>>2]|0;
      $2208 = (($2207) + 1)|0;
      HEAP32[$1051>>2] = $2208;
     }
    }
   }
  } while(0);
  $2209 = HEAP32[$1040>>2]|0;
  $2210 = (($2209) + 1)|0;
  HEAP32[$1040>>2] = $2210;
 }
 $2211 = ((($1053)) + 24|0);
 $2212 = HEAP32[$2211>>2]|0;
 $2213 = HEAP32[$1004>>2]|0;
 HEAP32[$35>>2] = $1006;
 $2214 = HEAP32[$35>>2]|0;
 $2215 = HEAP32[$2214>>2]|0;
 HEAP32[$34>>2] = $2215;
 $2216 = HEAP32[$34>>2]|0;
 $2217 = HEAP32[$1007>>2]|0;
 $2218 = HEAP32[$1008>>2]|0;
 HEAP32[$33>>2] = $1009;
 $2219 = HEAP32[$33>>2]|0;
 $2220 = HEAP32[$2219>>2]|0;
 HEAP32[$32>>2] = $2220;
 $2221 = HEAP32[$32>>2]|0;
 HEAP32[$31>>2] = $1010;
 $2222 = HEAP32[$31>>2]|0;
 $2223 = HEAP32[$2222>>2]|0;
 HEAP32[$30>>2] = $2223;
 $2224 = HEAP32[$30>>2]|0;
 HEAP32[$4>>2] = $1011;
 $2225 = HEAP32[$4>>2]|0;
 $2226 = HEAP32[$2225>>2]|0;
 HEAP32[$3>>2] = $2226;
 $2227 = HEAP32[$3>>2]|0;
 HEAP32[$5>>2] = $1012;
 $2228 = HEAP32[$5>>2]|0;
 $2229 = ((($2228)) + 4|0);
 $2230 = HEAP32[$2229>>2]|0;
 $2231 = HEAP32[$2228>>2]|0;
 $2232 = $2230;
 $2233 = $2231;
 $2234 = (($2232) - ($2233))|0;
 $2235 = (($2234|0) / 4)&-1;
 HEAP32[$7>>2] = $1012;
 $2236 = HEAP32[$7>>2]|0;
 $2237 = HEAP32[$2236>>2]|0;
 HEAP32[$6>>2] = $2237;
 $2238 = HEAP32[$6>>2]|0;
 HEAP32[$9>>2] = $1013;
 $2239 = HEAP32[$9>>2]|0;
 $2240 = HEAP32[$2239>>2]|0;
 HEAP32[$8>>2] = $2240;
 $2241 = HEAP32[$8>>2]|0;
 HEAP32[$11>>2] = $1014;
 $2242 = HEAP32[$11>>2]|0;
 $2243 = HEAP32[$2242>>2]|0;
 HEAP32[$10>>2] = $2243;
 $2244 = HEAP32[$10>>2]|0;
 HEAP32[$13>>2] = $1015;
 $2245 = HEAP32[$13>>2]|0;
 $2246 = HEAP32[$2245>>2]|0;
 HEAP32[$12>>2] = $2246;
 $2247 = HEAP32[$12>>2]|0;
 HEAP32[$14>>2] = $1016;
 $2248 = HEAP32[$14>>2]|0;
 $2249 = ((($2248)) + 4|0);
 $2250 = HEAP32[$2249>>2]|0;
 $2251 = HEAP32[$2248>>2]|0;
 $2252 = $2250;
 $2253 = $2251;
 $2254 = (($2252) - ($2253))|0;
 $2255 = (($2254|0) / 4)&-1;
 HEAP32[$16>>2] = $1016;
 $2256 = HEAP32[$16>>2]|0;
 $2257 = HEAP32[$2256>>2]|0;
 HEAP32[$15>>2] = $2257;
 $2258 = HEAP32[$15>>2]|0;
 $2259 = HEAP32[$1039>>2]|0;
 HEAP32[$18>>2] = $1033;
 $2260 = HEAP32[$18>>2]|0;
 $2261 = HEAP32[$2260>>2]|0;
 HEAP32[$17>>2] = $2261;
 $2262 = HEAP32[$17>>2]|0;
 HEAP32[$20>>2] = $1034;
 $2263 = HEAP32[$20>>2]|0;
 $2264 = HEAP32[$2263>>2]|0;
 HEAP32[$19>>2] = $2264;
 $2265 = HEAP32[$19>>2]|0;
 HEAP32[$22>>2] = $1035;
 $2266 = HEAP32[$22>>2]|0;
 $2267 = HEAP32[$2266>>2]|0;
 HEAP32[$21>>2] = $2267;
 $2268 = HEAP32[$21>>2]|0;
 HEAP32[$24>>2] = $1036;
 $2269 = HEAP32[$24>>2]|0;
 $2270 = HEAP32[$2269>>2]|0;
 HEAP32[$23>>2] = $2270;
 $2271 = HEAP32[$23>>2]|0;
 HEAP32[$26>>2] = $1037;
 $2272 = HEAP32[$26>>2]|0;
 $2273 = HEAP32[$2272>>2]|0;
 HEAP32[$25>>2] = $2273;
 $2274 = HEAP32[$25>>2]|0;
 HEAP32[$27>>2] = $1038;
 $2275 = HEAP32[$27>>2]|0;
 $2276 = ((($2275)) + 4|0);
 $2277 = HEAP32[$2276>>2]|0;
 $2278 = HEAP32[$2275>>2]|0;
 $2279 = $2277;
 $2280 = $2278;
 $2281 = (($2279) - ($2280))|0;
 $2282 = (($2281|0) / 4)&-1;
 HEAP32[$29>>2] = $1038;
 $2283 = HEAP32[$29>>2]|0;
 $2284 = HEAP32[$2283>>2]|0;
 HEAP32[$28>>2] = $2284;
 $2285 = HEAP32[$28>>2]|0;
 _SceneBuilderAdapter_insertThrustline(($2212|0),($2213|0),($2216|0),($2217|0),($2218|0),($2221|0),($2224|0),($2227|0),($2235|0),($2238|0),($2241|0),($2244|0),($2247|0),($2255|0),($2258|0),($2259|0),($2262|0),($2265|0),($2268|0),($2271|0),($2274|0),($2282|0),($2285|0));
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($1038);
 __ZNSt3__26vectorIjNS_9allocatorIjEEED2Ev($1037);
 __ZNSt3__26vectorIjNS_9allocatorIjEEED2Ev($1036);
 __ZNSt3__26vectorIjNS_9allocatorIjEEED2Ev($1035);
 __ZNSt3__26vectorIjNS_9allocatorIjEEED2Ev($1034);
 __ZNSt3__26vectorIjNS_9allocatorIjEEED2Ev($1033);
 __ZNSt3__23mapIPN3sap2ve4core14ThrustlineItemEiNS_4lessIS5_EENS_9allocatorINS_4pairIKS5_iEEEEED2Ev($1017);
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($1016);
 __ZNSt3__26vectorIiNS_9allocatorIiEEED2Ev($1015);
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($1014);
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($1013);
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($1012);
 __ZNSt3__26vectorIiNS_9allocatorIiEEED2Ev($1011);
 __ZNSt3__26vectorIiNS_9allocatorIiEEED2Ev($1010);
 __ZNSt3__26vectorIPKvNS_9allocatorIS2_EEED2Ev($1009);
 __ZNSt3__26vectorIfNS_9allocatorIfEEED2Ev($1006);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter15RealiseMetadataEPKN3sap2ve4core8MetadataERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0;
 var $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0;
 var $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0;
 var $65 = 0, $66 = 0, $67 = 0, $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0;
 var $83 = 0, $84 = 0, $85 = 0, $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 144|0;
 $2 = sp + 132|0;
 $3 = sp + 128|0;
 $4 = sp + 124|0;
 $5 = sp + 120|0;
 $6 = sp + 116|0;
 $7 = sp + 112|0;
 $8 = sp + 108|0;
 $9 = sp + 104|0;
 $10 = sp + 100|0;
 $11 = sp + 96|0;
 $12 = sp + 92|0;
 $13 = sp + 88|0;
 $14 = sp + 84|0;
 $15 = sp + 80|0;
 $16 = sp + 76|0;
 $17 = sp + 72|0;
 $18 = sp + 68|0;
 $19 = sp + 64|0;
 $20 = sp + 60|0;
 $21 = sp + 56|0;
 $22 = sp + 52|0;
 $23 = sp + 48|0;
 $24 = sp + 44|0;
 $25 = sp + 40|0;
 $26 = sp + 36|0;
 $27 = sp + 32|0;
 $28 = sp + 24|0;
 $29 = sp + 136|0;
 $30 = sp + 16|0;
 $31 = sp + 8|0;
 $32 = sp;
 HEAP32[$23>>2] = $0;
 HEAP32[$24>>2] = $1;
 HEAP32[$25>>2] = 0;
 $33 = HEAP32[$23>>2]|0;
 $34 = (__ZNK3sap2ve4core8Metadata16GetCategoryCountEv($33)|0);
 HEAP32[$26>>2] = $34;
 while(1) {
  $35 = HEAP32[$25>>2]|0;
  $36 = HEAP32[$26>>2]|0;
  $37 = ($35>>>0)<($36>>>0);
  if (!($37)) {
   break;
  }
  $38 = HEAP32[$23>>2]|0;
  $39 = HEAP32[$25>>2]|0;
  $40 = (__ZNK3sap2ve4core8Metadata19GetExistingCategoryEm($38,$39)|0);
  HEAP32[$27>>2] = $40;
  $41 = HEAP32[$27>>2]|0;
  $42 = ($41|0)!=(0|0);
  if ($42) {
   $43 = HEAP32[$24>>2]|0;
   HEAP32[$22>>2] = $43;
   $44 = HEAP32[$22>>2]|0;
   HEAP32[$21>>2] = $44;
   $45 = HEAP32[$21>>2]|0;
   HEAP32[$20>>2] = $45;
   $46 = HEAP32[$20>>2]|0;
   HEAP32[$19>>2] = $46;
   $47 = HEAP32[$19>>2]|0;
   HEAP32[$18>>2] = $47;
   $48 = HEAP32[$18>>2]|0;
   $49 = ((($48)) + 11|0);
   $50 = HEAP8[$49>>0]|0;
   $51 = $50&255;
   $52 = $51 & 128;
   $53 = ($52|0)!=(0);
   if ($53) {
    HEAP32[$14>>2] = $45;
    $54 = HEAP32[$14>>2]|0;
    HEAP32[$13>>2] = $54;
    $55 = HEAP32[$13>>2]|0;
    HEAP32[$12>>2] = $55;
    $56 = HEAP32[$12>>2]|0;
    $57 = ((($56)) + 4|0);
    $58 = HEAP32[$57>>2]|0;
    $65 = $58;
   } else {
    HEAP32[$17>>2] = $45;
    $59 = HEAP32[$17>>2]|0;
    HEAP32[$16>>2] = $59;
    $60 = HEAP32[$16>>2]|0;
    HEAP32[$15>>2] = $60;
    $61 = HEAP32[$15>>2]|0;
    $62 = ((($61)) + 11|0);
    $63 = HEAP8[$62>>0]|0;
    $64 = $63&255;
    $65 = $64;
   }
   $66 = ($65>>>0)>(1);
   if ($66) {
    $67 = HEAP32[$24>>2]|0;
    HEAP32[$8>>2] = $67;
    HEAP32[$9>>2] = 119826;
    $68 = HEAP32[$8>>2]|0;
    $69 = HEAP32[$9>>2]|0;
    (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($68,$69)|0);
   }
   $70 = HEAP32[$24>>2]|0;
   $71 = HEAP32[$27>>2]|0;
   __ZNK3sap2ve4core16MetadataCategory7GetNameEv($28,$71);
   $72 = (__ZNK3sap2ve6String4Utf8Ev($28)|0);
   __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($70,$72);
   __ZN3sap2ve6StringD2Ev($28);
   $73 = HEAP32[$24>>2]|0;
   HEAP32[$6>>2] = $73;
   HEAP32[$7>>2] = 82450;
   $74 = HEAP32[$6>>2]|0;
   $75 = HEAP32[$7>>2]|0;
   (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($74,$75)|0);
   HEAP8[$29>>0] = 1;
   $76 = HEAP32[$27>>2]|0;
   __ZNK3sap2ve4core16MetadataCategory11GetIteratorEv($30,$76);
   while(1) {
    $77 = (__ZN3sap2ve4core16MetadataIterator4NextEv($30)|0);
    if (!($77)) {
     break;
    }
    $81 = HEAP8[$29>>0]|0;
    $82 = $81&1;
    if ($82) {
     HEAP8[$29>>0] = 0;
    } else {
     $83 = HEAP32[$24>>2]|0;
     HEAP32[$4>>2] = $83;
     HEAP32[$5>>2] = 119826;
     $84 = HEAP32[$4>>2]|0;
     $85 = HEAP32[$5>>2]|0;
     (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($84,$85)|0);
    }
    $86 = HEAP32[$24>>2]|0;
    $87 = (__ZNK3sap2ve4core16MetadataIterator7CurrentEv($30)|0);
    __ZNK3sap2ve4core9Metadatum7GetNameEv($31,$87);
    $88 = (__ZNK3sap2ve6String4Utf8Ev($31)|0);
    __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($86,$88);
    __ZN3sap2ve6StringD2Ev($31);
    $89 = HEAP32[$24>>2]|0;
    HEAP32[$2>>2] = $89;
    HEAP32[$3>>2] = 98372;
    $90 = HEAP32[$2>>2]|0;
    $91 = HEAP32[$3>>2]|0;
    (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($90,$91)|0);
    $92 = HEAP32[$24>>2]|0;
    $93 = (__ZNK3sap2ve4core16MetadataIterator7CurrentEv($30)|0);
    __ZNK3sap2ve4core9Metadatum8GetValueEv($32,$93);
    $94 = (__ZNK3sap2ve6String4Utf8Ev($32)|0);
    __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($92,$94);
    __ZN3sap2ve6StringD2Ev($32);
   }
   __ZN3sap2ve4core16MetadataIteratorD2Ev($30);
   $78 = HEAP32[$24>>2]|0;
   HEAP32[$10>>2] = $78;
   HEAP32[$11>>2] = 82453;
   $79 = HEAP32[$10>>2]|0;
   $80 = HEAP32[$11>>2]|0;
   (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($79,$80)|0);
  }
  $95 = HEAP32[$25>>2]|0;
  $96 = (($95) + 1)|0;
  HEAP32[$25>>2] = $96;
 }
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter11RealiseVEIDERKN3sap2ve4core2IDERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($0,$1) {
 $0 = $0|0;
 $1 = $1|0;
 var $10 = 0, $100 = 0, $101 = 0, $102 = 0, $103 = 0, $104 = 0, $105 = 0, $106 = 0, $107 = 0, $108 = 0, $109 = 0, $11 = 0, $110 = 0, $111 = 0, $112 = 0, $113 = 0, $114 = 0, $115 = 0, $116 = 0, $117 = 0;
 var $118 = 0, $119 = 0, $12 = 0, $120 = 0, $121 = 0, $122 = 0, $123 = 0, $124 = 0, $125 = 0, $126 = 0, $127 = 0, $128 = 0, $129 = 0, $13 = 0, $130 = 0, $131 = 0, $132 = 0, $133 = 0, $134 = 0, $135 = 0;
 var $136 = 0, $137 = 0, $138 = 0, $139 = 0, $14 = 0, $140 = 0, $141 = 0, $142 = 0, $143 = 0, $144 = 0, $145 = 0, $146 = 0, $147 = 0, $148 = 0, $149 = 0, $15 = 0, $150 = 0, $151 = 0, $152 = 0, $153 = 0;
 var $154 = 0, $155 = 0, $156 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $2 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0, $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0;
 var $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0;
 var $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $56 = 0, $57 = 0, $58 = 0, $59 = 0, $6 = 0, $60 = 0, $61 = 0, $62 = 0, $63 = 0, $64 = 0, $65 = 0, $66 = 0, $67 = 0;
 var $68 = 0, $69 = 0, $7 = 0, $70 = 0, $71 = 0, $72 = 0, $73 = 0, $74 = 0, $75 = 0, $76 = 0, $77 = 0, $78 = 0, $79 = 0, $8 = 0, $80 = 0, $81 = 0, $82 = 0, $83 = 0, $84 = 0, $85 = 0;
 var $86 = 0, $87 = 0, $88 = 0, $89 = 0, $9 = 0, $90 = 0, $91 = 0, $92 = 0, $93 = 0, $94 = 0, $95 = 0, $96 = 0, $97 = 0, $98 = 0, $99 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 272|0;
 $2 = sp + 264|0;
 $3 = sp + 260|0;
 $4 = sp + 256|0;
 $5 = sp + 252|0;
 $6 = sp + 248|0;
 $7 = sp + 244|0;
 $8 = sp + 240|0;
 $9 = sp + 236|0;
 $10 = sp + 232|0;
 $11 = sp + 228|0;
 $12 = sp + 224|0;
 $13 = sp + 220|0;
 $14 = sp + 216|0;
 $15 = sp + 212|0;
 $16 = sp + 208|0;
 $17 = sp + 204|0;
 $18 = sp + 200|0;
 $19 = sp + 196|0;
 $20 = sp + 192|0;
 $21 = sp + 188|0;
 $22 = sp + 184|0;
 $23 = sp + 180|0;
 $24 = sp + 176|0;
 $25 = sp + 172|0;
 $26 = sp + 168|0;
 $27 = sp + 164|0;
 $28 = sp + 160|0;
 $29 = sp + 156|0;
 $30 = sp + 152|0;
 $31 = sp + 148|0;
 $32 = sp + 144|0;
 $33 = sp + 140|0;
 $34 = sp + 136|0;
 $35 = sp + 132|0;
 $36 = sp + 128|0;
 $37 = sp + 124|0;
 $38 = sp + 120|0;
 $39 = sp + 116|0;
 $40 = sp + 112|0;
 $41 = sp + 108|0;
 $42 = sp + 104|0;
 $43 = sp + 100|0;
 $44 = sp + 96|0;
 $45 = sp + 92|0;
 $46 = sp + 88|0;
 $47 = sp + 84|0;
 $48 = sp + 80|0;
 $49 = sp + 76|0;
 $50 = sp + 72|0;
 $51 = sp + 68|0;
 $52 = sp + 64|0;
 $53 = sp + 60|0;
 $54 = sp + 56|0;
 $55 = sp + 52|0;
 $56 = sp + 40|0;
 $57 = sp + 32|0;
 $58 = sp + 24|0;
 $59 = sp + 268|0;
 $60 = sp + 16|0;
 $61 = sp + 8|0;
 $62 = sp;
 HEAP32[$54>>2] = $0;
 HEAP32[$55>>2] = $1;
 $63 = HEAP32[$55>>2]|0;
 HEAP32[$53>>2] = $63;
 $64 = HEAP32[$53>>2]|0;
 HEAP32[$52>>2] = $64;
 $65 = HEAP32[$52>>2]|0;
 HEAP32[$51>>2] = $65;
 $66 = HEAP32[$51>>2]|0;
 HEAP32[$50>>2] = $66;
 $67 = HEAP32[$50>>2]|0;
 HEAP32[$49>>2] = $67;
 $68 = HEAP32[$49>>2]|0;
 $69 = ((($68)) + 11|0);
 $70 = HEAP8[$69>>0]|0;
 $71 = $70&255;
 $72 = $71 & 128;
 $73 = ($72|0)!=(0);
 if ($73) {
  HEAP32[$45>>2] = $65;
  $74 = HEAP32[$45>>2]|0;
  HEAP32[$44>>2] = $74;
  $75 = HEAP32[$44>>2]|0;
  HEAP32[$43>>2] = $75;
  $76 = HEAP32[$43>>2]|0;
  $77 = ((($76)) + 4|0);
  $78 = HEAP32[$77>>2]|0;
  $85 = $78;
 } else {
  HEAP32[$48>>2] = $65;
  $79 = HEAP32[$48>>2]|0;
  HEAP32[$47>>2] = $79;
  $80 = HEAP32[$47>>2]|0;
  HEAP32[$46>>2] = $80;
  $81 = HEAP32[$46>>2]|0;
  $82 = ((($81)) + 11|0);
  $83 = HEAP8[$82>>0]|0;
  $84 = $83&255;
  $85 = $84;
 }
 $86 = ($85>>>0)>(1);
 if ($86) {
  $87 = HEAP32[$55>>2]|0;
  HEAP32[$39>>2] = $87;
  HEAP32[$40>>2] = 119826;
  $88 = HEAP32[$39>>2]|0;
  $89 = HEAP32[$40>>2]|0;
  (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($88,$89)|0);
 }
 HEAP32[$38>>2] = $56;
 $90 = HEAP32[$38>>2]|0;
 HEAP32[$37>>2] = $90;
 $91 = HEAP32[$37>>2]|0;
 HEAP32[$36>>2] = $91;
 $92 = HEAP32[$36>>2]|0;
 ;HEAP32[$92>>2]=0|0;HEAP32[$92+4>>2]=0|0;HEAP32[$92+8>>2]=0|0;
 HEAP32[$35>>2] = $91;
 $93 = HEAP32[$35>>2]|0;
 HEAP32[$34>>2] = $93;
 HEAP32[$31>>2] = $90;
 $94 = HEAP32[$31>>2]|0;
 HEAP32[$30>>2] = $94;
 $95 = HEAP32[$30>>2]|0;
 HEAP32[$29>>2] = $95;
 $96 = HEAP32[$29>>2]|0;
 HEAP32[$32>>2] = $96;
 HEAP32[$33>>2] = 0;
 while(1) {
  $97 = HEAP32[$33>>2]|0;
  $98 = ($97>>>0)<(3);
  if (!($98)) {
   break;
  }
  $99 = HEAP32[$32>>2]|0;
  $100 = HEAP32[$33>>2]|0;
  $101 = (($99) + ($100<<2)|0);
  HEAP32[$101>>2] = 0;
  $102 = HEAP32[$33>>2]|0;
  $103 = (($102) + 1)|0;
  HEAP32[$33>>2] = $103;
 }
 $104 = HEAP32[$54>>2]|0;
 __ZNK3sap2ve4core2ID9GetSourceEv($57,$104);
 $105 = (__ZNK3sap2ve6String4Utf8Ev($57)|0);
 HEAP32[$27>>2] = $56;
 HEAP32[$28>>2] = $105;
 $106 = HEAP32[$27>>2]|0;
 $107 = HEAP32[$28>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6assignEPKc($106,$107)|0);
 __ZN3sap2ve6StringD2Ev($57);
 HEAP32[$25>>2] = $56;
 HEAP32[$26>>2] = 102464;
 $108 = HEAP32[$25>>2]|0;
 $109 = HEAP32[$26>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($108,$109)|0);
 $110 = HEAP32[$54>>2]|0;
 __ZNK3sap2ve4core2ID7GetTypeEv($58,$110);
 $111 = (__ZNK3sap2ve6String4Utf8Ev($58)|0);
 HEAP32[$23>>2] = $56;
 HEAP32[$24>>2] = $111;
 $112 = HEAP32[$23>>2]|0;
 $113 = HEAP32[$24>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($112,$113)|0);
 __ZN3sap2ve6StringD2Ev($58);
 $114 = HEAP32[$55>>2]|0;
 HEAP32[$22>>2] = $56;
 $115 = HEAP32[$22>>2]|0;
 HEAP32[$21>>2] = $115;
 $116 = HEAP32[$21>>2]|0;
 HEAP32[$20>>2] = $116;
 $117 = HEAP32[$20>>2]|0;
 HEAP32[$19>>2] = $117;
 $118 = HEAP32[$19>>2]|0;
 HEAP32[$18>>2] = $118;
 $119 = HEAP32[$18>>2]|0;
 HEAP32[$17>>2] = $119;
 $120 = HEAP32[$17>>2]|0;
 $121 = ((($120)) + 11|0);
 $122 = HEAP8[$121>>0]|0;
 $123 = $122&255;
 $124 = $123 & 128;
 $125 = ($124|0)!=(0);
 if ($125) {
  HEAP32[$11>>2] = $117;
  $126 = HEAP32[$11>>2]|0;
  HEAP32[$10>>2] = $126;
  $127 = HEAP32[$10>>2]|0;
  HEAP32[$9>>2] = $127;
  $128 = HEAP32[$9>>2]|0;
  $129 = HEAP32[$128>>2]|0;
  $135 = $129;
 } else {
  HEAP32[$16>>2] = $117;
  $130 = HEAP32[$16>>2]|0;
  HEAP32[$15>>2] = $130;
  $131 = HEAP32[$15>>2]|0;
  HEAP32[$14>>2] = $131;
  $132 = HEAP32[$14>>2]|0;
  HEAP32[$13>>2] = $132;
  $133 = HEAP32[$13>>2]|0;
  HEAP32[$12>>2] = $133;
  $134 = HEAP32[$12>>2]|0;
  $135 = $134;
 }
 HEAP32[$8>>2] = $135;
 $136 = HEAP32[$8>>2]|0;
 __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($114,$136);
 $137 = HEAP32[$55>>2]|0;
 HEAP32[$4>>2] = $137;
 HEAP32[$5>>2] = 82450;
 $138 = HEAP32[$4>>2]|0;
 $139 = HEAP32[$5>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($138,$139)|0);
 HEAP8[$59>>0] = 1;
 $140 = HEAP32[$54>>2]|0;
 __ZNK3sap2ve4core2ID16GetFieldIteratorEv($60,$140);
 while(1) {
  $141 = (__ZN3sap2ve4core13FieldIterator4NextEv($60)|0);
  if (!($141)) {
   break;
  }
  $142 = HEAP8[$59>>0]|0;
  $143 = $142&1;
  if ($143) {
   HEAP8[$59>>0] = 0;
  } else {
   $144 = HEAP32[$55>>2]|0;
   HEAP32[$2>>2] = $144;
   HEAP32[$3>>2] = 119826;
   $145 = HEAP32[$2>>2]|0;
   $146 = HEAP32[$3>>2]|0;
   (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($145,$146)|0);
  }
  $147 = HEAP32[$55>>2]|0;
  __ZNK3sap2ve4core13FieldIterator7GetNameEv($61,$60);
  $148 = (__ZNK3sap2ve6String4Utf8Ev($61)|0);
  __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($147,$148);
  __ZN3sap2ve6StringD2Ev($61);
  $149 = HEAP32[$55>>2]|0;
  HEAP32[$6>>2] = $149;
  HEAP32[$7>>2] = 98372;
  $150 = HEAP32[$6>>2]|0;
  $151 = HEAP32[$7>>2]|0;
  (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($150,$151)|0);
  $152 = HEAP32[$55>>2]|0;
  __ZNK3sap2ve4core13FieldIterator8GetValueEv($62,$60);
  $153 = (__ZNK3sap2ve6String4Utf8Ev($62)|0);
  __Z16appendJsonStringRNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEEEPKc($152,$153);
  __ZN3sap2ve6StringD2Ev($62);
 }
 $154 = HEAP32[$55>>2]|0;
 HEAP32[$41>>2] = $154;
 HEAP32[$42>>2] = 82453;
 $155 = HEAP32[$41>>2]|0;
 $156 = HEAP32[$42>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($155,$156)|0);
 __ZN3sap2ve4core13FieldIteratorD2Ev($60);
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($56);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter24CreateMetadataJSONStringEPKN3sap2ve4core17IContentContainerE($0,$1,$2) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 var $$expand_i1_val = 0, $$expand_i1_val2 = 0, $$pre_trunc = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $3 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0;
 var $45 = 0, $46 = 0, $47 = 0, $48 = 0, $49 = 0, $5 = 0, $50 = 0, $51 = 0, $52 = 0, $53 = 0, $54 = 0, $55 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 64|0;
 $3 = sp + 52|0;
 $4 = sp + 48|0;
 $5 = sp + 44|0;
 $6 = sp + 40|0;
 $7 = sp + 36|0;
 $8 = sp + 32|0;
 $9 = sp + 28|0;
 $10 = sp + 24|0;
 $11 = sp + 20|0;
 $12 = sp + 16|0;
 $13 = sp + 56|0;
 $14 = sp + 12|0;
 $15 = sp + 8|0;
 $16 = sp + 4|0;
 $17 = sp;
 HEAP32[$11>>2] = $1;
 HEAP32[$12>>2] = $2;
 $$expand_i1_val = 0;
 HEAP8[$13>>0] = $$expand_i1_val;
 HEAP32[$9>>2] = $0;
 HEAP32[$10>>2] = 82455;
 $18 = HEAP32[$9>>2]|0;
 HEAP32[$8>>2] = $18;
 $19 = HEAP32[$8>>2]|0;
 HEAP32[$7>>2] = $19;
 $20 = HEAP32[$7>>2]|0;
 ;HEAP32[$20>>2]=0|0;HEAP32[$20+4>>2]=0|0;HEAP32[$20+8>>2]=0|0;
 HEAP32[$6>>2] = $19;
 $21 = HEAP32[$6>>2]|0;
 HEAP32[$5>>2] = $21;
 $22 = HEAP32[$10>>2]|0;
 $23 = HEAP32[$10>>2]|0;
 $24 = (__ZNSt3__211char_traitsIcE6lengthEPKc($23)|0);
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6__initEPKcm($18,$22,$24);
 $25 = HEAP32[$12>>2]|0;
 $26 = HEAP32[$25>>2]|0;
 $27 = HEAP32[$26>>2]|0;
 $28 = (FUNCTION_TABLE_iii[$27 & 1023]($25,4)|0);
 HEAP32[$14>>2] = $28;
 $29 = HEAP32[$14>>2]|0;
 $30 = ($29|0)!=(0|0);
 if ($30) {
  $31 = HEAP32[$14>>2]|0;
  __ZN17DataModelImporter15RealiseMetadataEPKN3sap2ve4core8MetadataERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($31,$0);
 }
 $32 = HEAP32[$12>>2]|0;
 $33 = HEAP32[$32>>2]|0;
 $34 = HEAP32[$33>>2]|0;
 $35 = (FUNCTION_TABLE_iii[$34 & 1023]($32,2)|0);
 HEAP32[$15>>2] = $35;
 $36 = HEAP32[$15>>2]|0;
 $37 = ($36|0)!=(0|0);
 if ($37) {
  $38 = HEAP32[$15>>2]|0;
  __ZN17DataModelImporter15RealiseMetadataEPKN3sap2ve4core8MetadataERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($38,$0);
 }
 $39 = HEAP32[$12>>2]|0;
 $40 = HEAP32[$39>>2]|0;
 $41 = HEAP32[$40>>2]|0;
 $42 = (FUNCTION_TABLE_iii[$41 & 1023]($39,1)|0);
 HEAP32[$16>>2] = $42;
 $43 = HEAP32[$16>>2]|0;
 $44 = ($43|0)!=(0|0);
 if ($44) {
  $45 = HEAP32[$16>>2]|0;
  __ZN17DataModelImporter15RealiseMetadataEPKN3sap2ve4core8MetadataERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($45,$0);
 }
 $46 = HEAP32[$12>>2]|0;
 $47 = HEAP32[$46>>2]|0;
 $48 = HEAP32[$47>>2]|0;
 $49 = (FUNCTION_TABLE_iii[$48 & 1023]($46,3)|0);
 HEAP32[$17>>2] = $49;
 $50 = HEAP32[$17>>2]|0;
 $51 = ($50|0)!=(0|0);
 if ($51) {
  $52 = HEAP32[$17>>2]|0;
  __ZN17DataModelImporter15RealiseMetadataEPKN3sap2ve4core8MetadataERNSt3__212basic_stringIcNS6_11char_traitsIcEENS6_9allocatorIcEEEE($52,$0);
 }
 HEAP32[$3>>2] = $0;
 HEAP32[$4>>2] = 82453;
 $53 = HEAP32[$3>>2]|0;
 $54 = HEAP32[$4>>2]|0;
 (__ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEE6appendEPKc($53,$54)|0);
 $$expand_i1_val2 = 1;
 HEAP8[$13>>0] = $$expand_i1_val2;
 $$pre_trunc = HEAP8[$13>>0]|0;
 $55 = $$pre_trunc&1;
 if ($55) {
  STACKTOP = sp;return;
 }
 __ZNSt3__212basic_stringIcNS_11char_traitsIcEENS_9allocatorIcEEED2Ev($0);
 STACKTOP = sp;return;
}
function __ZN17DataModelImporter21CreateVEIDsJSONStringEy($0,$1,$2,$3) {
 $0 = $0|0;
 $1 = $1|0;
 $2 = $2|0;
 $3 = $3|0;
 var $$expand_i1_val = 0, $$expand_i1_val2 = 0, $$pre_trunc = 0, $10 = 0, $11 = 0, $12 = 0, $13 = 0, $14 = 0, $15 = 0, $16 = 0, $17 = 0, $18 = 0, $19 = 0, $20 = 0, $21 = 0, $22 = 0, $23 = 0, $24 = 0, $25 = 0, $26 = 0;
 var $27 = 0, $28 = 0, $29 = 0, $30 = 0, $31 = 0, $32 = 0, $33 = 0, $34 = 0, $35 = 0, $36 = 0, $37 = 0, $38 = 0, $39 = 0, $4 = 0, $40 = 0, $41 = 0, $42 = 0, $43 = 0, $44 = 0, $45 = 0;
 var $46 = 0, $5 = 0, $6 = 0, $7 = 0, $8 = 0, $9 = 0, label = 0, sp = 0;
 sp = STACKTOP;
 STACKTOP = STACKTOP + 64|0;
 $4 = sp + 56|0;
 $5 = sp + 52|0;
 $6 = sp + 48|0;
 $7 = sp + 44|0;
 $8 = sp + 40|0;
 $9 = sp + 36|0;
 $10 = sp + 32|0;
 $11 = sp + 28|0;
 $12 = sp + 24|0;
 $13 = sp;
 $14 = sp + 60|0;
 $15 = sp + 16|0;
 $16 = sp + 8|0;
 HEAP32[$12>>2] = $1;
 $17 = $13;
 $18 = $17;
 HEAP32[$18>>2] = $2;
 $19 = (($17) + 4)|0;
 $20 = $19;
 HEAP32[$20>>2] = $3;
 $21 = HEAP32[$12>>2]|0;
 $$exp