#!/bin/sh
#  $Id$
#  *********************************************************************
# ***********************************************************************
# ****                                                               ****
# ***           Auxiliary script for migrations with SYSTEM MOVE      ***
# ***                                                                 ***
# ***           Copyright (c) 2019 SAP-SE                             ***
# ****                                                               ****
# ***********************************************************************
#  *********************************************************************
#
#  Hint: Use "ssh-copy-id" first to enable password less access to the target machine.
#
# ------------------------------------------------------------------------------------------------
set -e

USAGE="dmosystemmove.sh -a <full|conti|stop> -n <hostname> [-t <target_sum_dir>] [-l <logfilename>]"

gpv_sleep_time_in_sec=30
gpv_action=stop
gpv_control_file="syncfiles.lst"

gpv_user=""
gpv_sid=""
gpv_host_target=""

gpv_sum_path_source=""
gpv_sum_path_target=""

gpv_pidfile_path=""

# ------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------

setUser()
{
    gpv_user="$USER"

    if [ -z "$USER" ] ; then
        gpv_user=`whoami`
    fi

    if [ -z "$gpv_user" ] ; then
        echo "ERROR: Could not determine user name!"
        exit 10
    fi
}

# ------------------------------------------------------------------------------------------------

setSID()
{
    gpv_sid="$SAPSYSTEMNAME"

    if [ -z "$gpv_sid" ] ; then
        gpv_sid=`expr "$gpv_user" : '\(...\)adm' | tr '[a-z]' '[A-Z]'`
    fi

    if [ -z "$gpv_sid" ] ; then
        echo "ERROR: Could not determine SID!"
        exit 10
    fi
}

# ------------------------------------------------------------------------------------------------

setPaths()
{
    _scriptPath=`realpath $0`
    _binPath=`dirname $_scriptPath`

    cd "$_binPath/../../"
    gpv_sum_path_source=`/bin/pwd`

    if [ ! -d "$gpv_sum_path_source" ] ; then
        echo "ERROR: Could not determine SUM directory!"
        exit 10
    fi

    gpv_sum_path_target="/usr/sap/$gpv_sid/SUM"
    if [ -n "$gpv_target_dir" ] ; then
        gpv_sum_path_target="$gpv_target_dir"
    fi

    gpv_pidfile_path="$gpv_sum_path_source/abap/srv/daemon_CONTI_SYNC.pid"
}

# ------------------------------------------------------------------------------------------------

createPIDFile()
{
    echo $$ > "$gpv_pidfile_path"
}

deletePIDFile()
{
    if [ -f "$gpv_pidfile_path" ] ; then
        rm "$gpv_pidfile_path"
    fi
}

setSTOPRequest()
{
    echo "STOP-EXECUTION" > "$gpv_pidfile_path"
}

# ------------------------------------------------------------------------------------------------

initialTransfer()
{
    _tmstp=`date`
    echo "=========================================================================="
    echo "[$_tmstp] START initial transfer: $gpv_sum_path_source -> $gpv_user@$gpv_host_target:$gpv_sum_path_target ..."

    if ssh "$gpv_user@$gpv_host_target" "test -r $gpv_sum_path_target/abap/log/SAPup_NEW.log" ; then
        echo "ERROR: SAPup already started on target host, must not sync fully anymore!"
        exit 10
    fi

    _targetBase=`dirname $gpv_sum_path_target`
    ssh "$gpv_user@$gpv_host_target" "test -d $_targetBase || mkdir $_targetBase"

    #  Specify "--del" to have the source and target exactly matching!
    rsync --del -av "$gpv_sum_path_source/" "$gpv_user@$gpv_host_target:$gpv_sum_path_target/"

    _tmstp=`date`
    echo "[$_tmstp] END initial transfer"
    echo ""
}

# ------------------------------------------------------------------------------------------------

doSync()
{
    _lastRun=$1

    _loadPathSource="$gpv_sum_path_source/abap/load"
    _loadPathTarget="$gpv_sum_path_target/abap/load"

    _tmstp=`date`
    echo "=========================================================================="
    echo "[$_tmstp] START parallel transfer: $_loadPathSource/ -> $gpv_user@$gpv_host_target:$_loadPathTarget/ ..."

    # Cleanup first
    if [ -f tmp/$gpv_control_file ] ; then
        rm tmp/$gpv_control_file*
    fi

    # Create and split worklist
    rsync -avm --out-format="FILE=%n" --dry-run "$_loadPathSource/" "$gpv_user@$gpv_host_target:$_loadPathTarget/" | \
    grep "^FILE=" | grep -v '^FILE=.*/$' | sed s/^FILE=//g > tmp/$gpv_control_file

    _totalLines=`wc -l < tmp/$gpv_control_file`

    if [ "$_totalLines" -gt "0" ] ; then
        if [ "$_totalLines" -gt "$DMO_SYSTEMMOVE_NPROCS" ] ; then
            _splitLines=`expr $_totalLines / $DMO_SYSTEMMOVE_NPROCS`
            split -l $_splitLines tmp/$gpv_control_file tmp/${gpv_control_file}_
        else
            cp tmp/$gpv_control_file tmp/${gpv_control_file}_
        fi

        # Sync worklist
        _pids=""
        for i in `ls tmp/${gpv_control_file}_*`
        do
            rsync -avm --files-from=$i "$_loadPathSource/" "$gpv_user@$gpv_host_target:$_loadPathTarget/" &
            _pids="$_pids $!"
        done

        if [ $_lastRun = true ] ; then
            wait $_pids
        else
            wait $_pids || true
        fi

        rm tmp/$gpv_control_file*
    fi

    _tmstp=`date`
    echo "[$_tmstp] END parallel transfer"
    echo ""
}

# ------------------------------------------------------------------------------------------------

continuousTransfer()
{
    createPIDFile

    cd "$gpv_sum_path_source/abap"

    while :
    do
        #  avoid race condition: after receiving stop request, must transfer one more time!
        _dostop=false
        if [ `grep STOP-EXECUTION $gpv_pidfile_path | wc -l` -gt 0 ] ; then
            echo "*** Received request to stop continuous transfer!"
            _dostop=true
        fi

        doSync $_dostop

        if [ $_dostop = true ] ; then
            break
        fi

        sleep $gpv_sleep_time_in_sec
    done

    deletePIDFile
}

# ------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------

# Parse command parameters
while getopts "ha:n:t:l:" option
do
    case "$option" in
        h)
            echo $USAGE
            exit 0
            ;;
        a)
            gpv_action=${OPTARG}
            ;;
        n)
            gpv_host_target=$OPTARG
            ;;
        t)
            gpv_target_dir=$OPTARG
            ;;
        l)
            gpv_log_file=$OPTARG
            ;;
        \?)
            echo $USAGE
            exit 10
            ;;
    esac
done

if [ -z "$DMO_SYSTEMMOVE_NPROCS" ] ; then
    DMO_SYSTEMMOVE_NPROCS=4
fi

setUser
setSID
setPaths

if [ -n "$gpv_log_file" ] ; then
    _logFile="$gpv_sum_path_source/abap/log/$gpv_log_file"
    echo "Writing to log file $_logFile ..."
    exec > $_logFile
fi

echo "=========================================================================="
echo "  User:              $gpv_user"
echo "  SID:               $gpv_sid"
echo "  Target host:       $gpv_host_target"
echo ""
echo "  SUM path (source): $gpv_sum_path_source"
echo "  SUM path (target): $gpv_sum_path_target"
echo ""
echo "  Action:            $gpv_action"
echo "  Waiting time:      $gpv_sleep_time_in_sec sec"
echo "  PID file path:     $gpv_pidfile_path"
echo "  #Processes:        $DMO_SYSTEMMOVE_NPROCS"
echo "=========================================================================="
echo ""

# ------------------------------------------------------------------------------------------------

if [ $gpv_action = stop ] ; then
    setSTOPRequest
elif [ $gpv_action = full ] ; then
    initialTransfer
else
    continuousTransfer
fi
