sap.ui.define([
	"sap/m/MessageToast",
	"SUMControls/FragmentControl",
	"SUMControls/LogViewer"
], function(MessageToast, FragmentControl, LogViewer) {
	"use strict";

	var CRRBucketTable = FragmentControl.extend("SUMControls.CRRBucketTable", {
		
		metadata: {
			properties: {
				daemon: { type: "String", defaultValue: null }
			}
		},
		
		init : function() {
			/* call super initialization */
			this.initFragmentControls();
			
			this._model = new sap.ui.model.json.JSONModel();
			
			this._table = this.byId("idTableInfo");
			this._table.setModel(this._model);
			
			this._isRunning = false;
			
			this._oToggleFinished = this.byId("oToggleFinished");
			this._oToggleRescheduled = this.byId("oToggleRescheduled");
			this._oToggleRunning = this.byId("oToggleRunning");
			this._oToggleStopping = this.byId("oToggleStopping");
			this._oToggleFailed = this.byId("oToggleFailed");
			this._oRestartButton = this.byId("oRestartButton");
			
			this._mayUpdate = true;
			
			this._clearModel();
			this._applyToggleFilter();
		},
		
		/* ****************************************************** */
		/* PUBLIC HELPER FUNCTIONS								  */
		/* ****************************************************** */
		
		doUpdate : function() {
			if(this._mayUpdate === true) {
				this._updateTable();
			}
		},
		
		getFragmentName : function() {
			return "SUMFragments/CRRBucketTable";
		},
		
		setDaemon : function(daemon) {
			this.setProperty("daemon", daemon);
		},
		
		setIsRunning : function(isRunning) {
			this._isRunning = isRunning;
		},
		
		/* ****************************************************** */
		/* PRIVATE HELPER FUNCTIONS								  */
		/* ****************************************************** */
		
		_clearModel : function() {
			var modelData = {};
			modelData.data = [];
			this._model.setData(modelData);
			
			this._nTotal = 0;
			this._nPlanned = 0;
			this._nFinished = 0;
			this._nRunning = 0;
			this._nRescheduled = 0;
			this._nFailed = 0;
			
			this._setFooterText();
		},
		
		/* ****************************************************** */

		_getToggleFilters : function(isFinished, isRunning, isStopping, isRescheduled, isFailed) {
			
			var oFilters = [];
			
			if(isFinished) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "finished"));
			}
			
			if(isRunning) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "running"));
			}
			
			if(isStopping) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "stopping"));
			}
			
			if(isRescheduled) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "rescheduled"));
			}
			
			if(isFailed) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "aborted"));
			}
			
			return new sap.ui.model.Filter({ filters: oFilters, and : false });
		},
		
		_applyToggleFilter : function() {
			var isFinished    = this._oToggleFinished.getPressed();
			var isRunning     = this._oToggleRunning.getPressed();
			var isStopping    = this._oToggleStopping.getPressed();
			var isRescheduled = this._oToggleRescheduled.getPressed();
			var isFailed      = this._oToggleFailed.getPressed();
			
			var oFilter = this._getToggleFilters(isFinished, isRunning, isStopping, isRescheduled, isFailed);
			this._table.getBinding("rows").filter(oFilter);
		},
		
		/* ****************************************************** */

		_getStatus : function(sExp, sImp) {
			if(sExp === 'stopping' || sImp === 'stopping') {
				return "stopping";
			}
			
			if(sExp === 'finished' && (sImp === 'finished' || sImp === null)) {
				return "finished";
			}
			
			if(sExp === 'running' && (sImp === 'running' || sImp === null)) {
				return "running";
			}
			
			if(sExp === 'failed' || sExp === 'aborted' || sExp === 'killed' || sExp === 'error') {
				return "aborted";
			}
			
			if(sImp !== null && (sImp === 'failed' || sImp === 'aborted' || sImp === 'killed' || sImp === 'error')) {
				return "aborted";
			}
			
			return "rescheduled";
		},
		
		_getStatusState : function(status) {
			return (status === "running" || status === "finished") ? "Success" :
				   (status === "aborted") ? "Error"   : "Warning";
		},
		
		/* ****************************************************** */
		
		_getSelectedRowModel : function(oEvent) {
			var path = oEvent.getSource().getBindingContext().getPath();
			return this._model.getProperty(path);
		},
		
		/* ****************************************************** */
		
		_openLogViewer : function(logFile) {
			this._logFile = logFile;
			this._oLogViewer = new LogViewer({ 
										logFile: this._logFile,
										closed : function() {
											this._oLogViewer.destroy();
										}.bind(this)
								});
			this._oLogViewer.open();
		},
		
		/* ****************************************************** */
		
		_getStatusUrl : function() {
			var daemon = this.getProperty("daemon") ? "/daemon/" + this.getProperty("daemon") : "";
			return ".." + daemon + "/status?details=1&xsl=0";
		},
		
		_getProcessUrl : function() {
			var daemon = this.getProperty("daemon") ? "/daemon/" + this.getProperty("daemon") : "";
			return ".." + daemon + "/parallel_processes?xsl=0";
		},
		
		/* ****************************************************** */
		
		_fillTable : function(cloning, buckets, isRunning) {
            this._nTotal       = parseInt(UIGetAttribute(cloning, "totalprocs", 0));
            this._nFinished    = parseInt(UIGetAttribute(cloning, "finishedprocs", 0));
            this._nRunning     = parseInt(UIGetAttribute(cloning, "exportprocs", 0)) + parseInt(UIGetAttribute(cloning, "impprocs", 0));
            this._nRescheduled = parseInt(UIGetAttribute(cloning, "restartedprocs", 0));
            this._nFailed      = parseInt(UIGetAttribute(cloning, "failedprocs", 0));
            
            var modelData = {};
			modelData.data = [];
			
			for(var i = 0 ; i < buckets.length ; i++) {
				var bucketNode = buckets[i];
				
				var bucket = {};
				bucket.refid       = UIGetAttribute(bucketNode, "refid", "");
				bucket.expduration = parseInt(UIGetAttribute(bucketNode, "exportduration", 0));
				bucket.expwaiting  = parseInt(UIGetAttribute(bucketNode, "exportwaiting", 0));
				bucket.impduration = parseInt(UIGetAttribute(bucketNode, "importduration", 0));
				bucket.impwaiting  = parseInt(UIGetAttribute(bucketNode, "importwaiting", 0));
				bucket.export      = UIGetAttribute(bucketNode, "export", null);
				bucket.import      = UIGetAttribute(bucketNode, "import", null);
				bucket.exportLog   = UIGetAttribute(bucketNode, "exportlog", null);
				bucket.importLog   = UIGetAttribute(bucketNode, "importlog", null);
				
				bucket.duration    = (bucket.expduration - bucket.expwaiting) + " / " + (bucket.impduration - bucket.impwaiting);
				
				bucket.status      = this._getStatus(bucket.export, bucket.import);
				bucket.statusState = this._getStatusState(bucket.status);
				
				if(isRunning === 0) {
					bucket.status = "Old status: " + bucket.status;
					bucket.statusState = "Warning";
				}
				
				modelData.data.push(bucket);
			}
			
			this._model.setData(modelData);
            this._setFooterText();
            
            this._mayUpdate = true;
		},
		
		/* ****************************************************** */
		
		_loadOldTable : function() {
			$.ajax({
				url: "../doc/analysis/MIGRATE_OPTDMO_REPLAY_DAEMONSTAT.XML",
				type: "GET",
				dataType: "text",
				cache: false,
				context: this,

				error: function() {
					this._clearModel();
		            this._mayUpdate = true;
				},
				
				success: function(data) {
					var xmlData = $.parseXML(data);
					
					var cloning = $("Cloning", xmlData);
					var buckets = $(cloning).find("bucket");
					
					if(buckets == null || cloning == null || buckets.length == 0 || cloning.length == 0) {
						this._clearModel();
			            this._mayUpdate = true;
			            return;
					}
					
					this._fillTable(cloning, buckets, 0);
				}
			});
		},
		
		/* ****************************************************** */
		
		_updateTable : function() {
			this._mayUpdate = false;
			
			$.ajax({
				url: this._getStatusUrl(),
				type: "GET",
				dataType: "text",
				cache: false,
				context: this,

				error: function() {
					this._clearModel();
		            this._mayUpdate = true;
				},
				
				success: function(data) {
					var xmlData = $.parseXML(data);
					
					var cloning = $("Cloning", xmlData);
					var buckets = $(cloning).find("bucket");
					
					if(buckets == null || cloning == null || buckets.length == 0 || cloning.length == 0) {
						this._loadOldTable();
			            return;
					}
					
					this._fillTable(cloning, buckets, 1);
				}
			});
		},
		
		/* ****************************************************** */
		
		_restartFailedBuckets : function(bucketsToRestart) {
			if(bucketsToRestart.length == 0) {
				MessageToast.show("No failed buckets selected.");
	            return;
			}
			
			this._mayUpdate = false;
			this._cntRestarted = bucketsToRestart.length;
			
			var postString = "<ParallelProcesses>";
			for(var i = 0 ; i < bucketsToRestart.length ; i++) {
				var bucket = bucketsToRestart[i];
				postString += '<FailedBucket enabled="true">' + bucket.refid + '</FailedBucket>';
			}
			postString += "</ParallelProcesses>";		
			
        	doPost(
	        	function(data) {
					MessageToast.show("" + this._cntRestarted + " bucket" + (this._cntRestarted === 1 ? "" : "s") + " rescheduled.");
					this._updateTable();
				}, 
	        	function() {
					MessageToast.show("Error: Rescheduling buckets was not successfull!");
					this._mayUpdate = true;
				}, 
	        	this._getProcessUrl(), 
	        	postString, 
	        	this
        	);

/*			$.ajax({
				url: this._getProcessUrl(),
				type: "POST",
				data: postString,
				dataType: "text",
				cache: false,
				context: this,

				error: function() {
					MessageToast.show("Error: Rescheduling buckets was not successfull!");
					this._mayUpdate = true;
				},
				
				success: function(data) {
					MessageToast.show("" + this._cntRestarted + " bucket" + (this._cntRestarted === 1 ? "" : "s") + " rescheduled.");
					this._updateTable();
				}
			});
*/
		},
		
		/* ****************************************************** */
		
		_setFooterText : function() {
			var text = "Process Statistics - Planned: " + this._nPlanned + " / Rescheduled: " + this._nRescheduled + " / Running: " + this._nRunning  + " / Failed: " + this._nFailed + " / Finished: " + this._nFinished;
			this._table.setFooter(text);
		},
		
		/* ****************************************************** */
		
		_getSelectedBuckets : function() {
			var buckets = [];
			
			if(this._isRunning !== true) {
				return buckets;
			}
			
			var selectedIds = this._table.getSelectedIndices();
			for(var i = 0 ; i < selectedIds.length ; i++) {
				var cxt = this._table.getContextByIndex(selectedIds[i]);
				var bucket = cxt.getObject();
				if(bucket.status == "aborted") {
					buckets.push(bucket);
				}
			}
			
			return buckets;
		},
		
		/* ****************************************************** */
		/* PUBLIC HANDLER FUNCTIONS								  */
		/* ****************************************************** */
		
		handleFilterToggle : function(oEvent) {
			this._applyToggleFilter();
		},
		
		onExportLog : function(oEvent) {
			var bucket = this._getSelectedRowModel(oEvent);
			this._openLogViewer("load/" + bucket.exportLog);
		},
		
		onImportLog : function(oEvent) {
			var bucket = this._getSelectedRowModel(oEvent);
			this._openLogViewer("load/" + bucket.importLog);
		},
		
		onRestartButton : function(oEvent) {
			var buckets = this._getSelectedBuckets();
			this._restartFailedBuckets(buckets);
		},
		
		/* ****************************************************** */
		/* ****************************************************** */
		
		renderer : function(oRenderManager, oControl) {
			oControl.doRendering(oRenderManager, oControl);
		}

	});

	return CRRBucketTable;
});