sap.ui.define([
    "sap/m/MessageToast",
    "SUMControls/FragmentControl",
    "SUMControls/LogViewer"
], function(MessageToast, FragmentControl, LogViewer) {
    "use strict";

    var CRRDaemonStatus = FragmentControl.extend("SUMControls.CRRDaemonStatus", {
        
        metadata: {
            properties: {
                title:          { type: "String", defaultValue: "" },
                daemon:         { type: "String", defaultValue: "" },
                labelNumProcs:  { type: "String", defaultValue: "" },
                showBuckets:    { type: "Boolean", defaultValue: false },
                showProcesses:  { type: "Boolean", defaultValue: false }
            }
        },
        
        init : function() {
            /* call super initialization */
            this.initFragmentControls();
            
            this._totalCrrRate = -1;
            this._mayUpdate = true;
            this._logFile = null;
            
            this._sumReplayed = 0;
            this._sumRecorded = 0;
            this._sumPending  = 0;
            
            this._isRegistered = false;
            this._isRunning = false;
            
            this._nPlanned = 0;
            this._nIdle = 0;
            this._nRunning = 0;
            this._nFailed = 0;
            this._nInactive = 0;
            
            this._model = new sap.ui.model.json.JSONModel();
            
            this._table = this.byId("idTableInfo");
            this._table.setModel(this._model);
            
            this._oPanel = this.byId("oPanel");
            this._oTitle = this.byId("oTitle");
            this._oLabelNumProcs = this.byId("oLabelNumProcs");
            this._oStatus = this.byId("oStatus");
            this._oIsBusy = this.byId("oIsBusy");
            
            this._oNProcsRun = this.byId("oNProcsRun");
            this._oNProcsReq = this.byId("oNProcsReq");
            
            this._oStartButton = this.byId("oStart");
            this._oStopButton = this.byId("oStop");
            this._oAcceptButton = this.byId("oAccept");
            this._oLogFileButton = this.byId("oLogFile");
            
            this._oTogglePlanned = this.byId("oTogglePlanned");
            this._oToggleIdle = this.byId("oToggleIdle");
            this._oToggleRunning = this.byId("oToggleRunning");
            this._oToggleFailed = this.byId("oToggleFailed");
            
            this._oBuckets = this.byId("oBuckets");
            this._oProcesses = this.byId("oProcesses");
            
            this._iniProcess();
            this._setFooterText();
        },
        
        /* ****************************************************** */
        /* PUBLIC HELPER FUNCTIONS                                  */
        /* ****************************************************** */
        
        doUpdate : function() {
            if(this._mayUpdate) {
                this._updateMonitor();
                
                if(this._oBuckets.getVisible()) {
                    this._oBuckets.doUpdate(); /* R3load CRR */
                }
                
                if(this._oProcesses.getVisible()) {
                    this._oProcesses.doUpdate(); /* ABAP CRR */
                }
            }
        },
        
        getFragmentName : function() {
            return "SUMFragments/CRRDaemonStatus";
        },
        
        getRegistered : function() {
            return this._isRegistered;
        },
        
        getTotalCrrRate : function() {
            return this._isRunning === true ? this._totalCrrRate : -1;
        },
        
        getReadyForDowntime : function() {
            if(this._isRegistered === false) {
                return true;
            }
            
            if(this._isRunning === false || this._nFailed > 0 || this._nRunning < 1 || this._nInactive > 0) {
                return false;
            }
            
            return true;
        },
        
        getSumCrrPending : function() {
            return this._sumPending;
        },
        
        getSumCrrReplayed : function() {
            return this._sumReplayed;
        },
        
        getSumCrrRecorded : function() {
            return this._sumRecorded;
        },
        
        setDaemon : function(daemon) {
            this.setProperty("daemon", daemon);
            this._oBuckets.setDaemon(daemon);
            this._oProcesses.setDaemon(daemon);
        },
        
        setLabelNumProcs : function(label) {
            this._oLabelNumProcs.setText(label);
        },
        
        setTitle : function(stitle) {
            this._oTitle.setText(stitle);
        },
        
        setShowBuckets : function(value) {
            this._oBuckets.setVisible(value == "true" ? true : false);
        },
        
        setShowProcesses : function(value) {
            this._oProcesses.setVisible(value == "true" ? true : false);
        },
        
        /* ****************************************************** */
        /* PRIVATE HELPER FUNCTIONS                                  */
        /* ****************************************************** */

        _clearModel : function() {
            this._model.setData({ data : [] });
            
            this._nPlanned = 0;
            this._nIdle = 0;
            this._nRunning = 0;
            this._nFailed = 0;
            this._nInactive = 0;
            
            this._setFooterText();
        },
        
        _iniProcess : function() {
            this._oPanel.setVisible(false);
            
            this._oStatus.setText("not registered");
            this._oStatus.setState("None");
            this._oIsBusy.setVisible(false);
            this._oNProcsRun.setText(0);
            this._oNProcsReq.setVisible(false);
            this._oStartButton.setEnabled(false);
            this._oStopButton.setEnabled(false);
            this._oAcceptButton.setVisible(false);
            
            this._setRunningStatus(false);
            this._isRegistered = false;
            
            this._clearModel();
            this._setVisibleChangingProcs(null, null, false);
        },
        
        _setRunningStatus : function(isRunning) {
            this._isRunning = isRunning;
            this._oBuckets.setIsRunning(isRunning);
        },
        
        /* ****************************************************** */
        
        _getChangingProcsObjects : function() {
            return [ this._oNProcsReq, this._oAcceptButton ];
        },
        
        /* ****************************************************** */
        
        _setVisibleChangingProcs : function(runProcs, maxProcs, state) {
            var objects = this._getChangingProcsObjects();
            
            if(objects[0] == null || objects[1] == null) {
                return 0;
            }
            
            if(objects[0].getVisible() == state) {
                return 0;
            }
            
            objects[0].setVisible(state);
            objects[1].setVisible(state);

            objects[0].setMax(1);
            if(maxProcs != null)
                objects[0].setMax(maxProcs);
            
            objects[0].setValue(1);
            if(runProcs != null)
                objects[0].setValue(runProcs);
            
            return 1;
        },
        
        /* ****************************************************** */
        
        _setBusyChangingProcs : function(state) {
            var objects = this._getChangingProcsObjects();
            
            if(objects[0] == null || objects[1] == null) {
                return 0;
            }
            
            if(objects[0].isBusy() == state) {
                return 0;
            }
            
            objects[0].setBusy(state);
            objects[0].setEnabled(!state);
            
            objects[1].setEnabled(!state);
            return 1;
        },

        /* ****************************************************** */

        _getToggleFilter : function(isPlanned, isIdle, isRunning, isFailed) {
            var oFilters = [];
            
            if(isPlanned) {
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, ""));
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, "planned"));
            }
            
            if(isIdle) {
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, "idle"));
            }
            
            if(isRunning) {
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, "running"));
            }
            
            if(isFailed) {
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, "aborted"));
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, "failed"));
                oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.EQ, "error"));
            }
            
            return new sap.ui.model.Filter({ filters: oFilters, and : false });
        },
        
        _applyToggleFilter : function() {
            var isPlanned = this._oTogglePlanned.getPressed();
            var isIdle    = this._oToggleIdle.getPressed();
            var isRunning = this._oToggleRunning.getPressed();
            var isFailed  = this._oToggleFailed.getPressed();
            
            var oFilter = this._getToggleFilter(isPlanned, isIdle, isRunning, isFailed);
            this._table.getBinding("rows").filter(oFilter);
        },
        
        /* ****************************************************** */

        _postAction : function(content) {
            this._success = function(data, that) {
                that._updateMonitor();
            };
            
			doPost(function(data){this._success(data, this);}, function(){this._success(null, this);}, "../daemons?xsl=0", content, this);

/*          $.ajax({
                url: "../daemons?xsl=0",
                type: "POST",
                dataType: "text",
                context: this,
                data: content,
                
                error: function() {
                    this._success(null, this);
                },
                
                success: function(data) {
                    this._success(data, this);
                }
            });
*/
        },
        
        /* ****************************************************** */
        
        _checkUpdateNProcs : function(daemon) {
            
            if(daemon.canChange === false || daemon.maxprocs < 1 || daemon.runprocs < 1) {
                this._setVisibleChangingProcs(null, null, false);
                return;
            }
            
            this._setVisibleChangingProcs(daemon.runprocs, daemon.maxprocs, true);
            
            if(daemon.reqprocs > 0 && daemon.reqprocs != daemon.runprocs) {
                this._setBusyChangingProcs(true);
                return;
            }
            
            this._setBusyChangingProcs(false);
        },
        
        /* ****************************************************** */
        
        _updateMonitor : function() {
            this._mayUpdate = false;
            
            this._getXmlNode = function(data, that) {
                var daemonNode = $("Daemon", data);
                
                if(daemonNode == null || daemonNode.length < 1) {
                    that._iniProcess();
                    that._mayUpdate = true;
                    return;
                }
                
                that._isRegistered = true;
                that._oPanel.setVisible(true);
                
                var daemon = {};
                daemon.pid            = UIGetAttribute(daemonNode, "pid", -1);
                daemon.runprocs    = parseInt(UIGetAttribute(daemonNode, "runprocs", -1));
                daemon.reqprocs    = parseInt(UIGetAttribute(daemonNode, "reqprocs", -1));
                daemon.maxprocs    = parseInt(UIGetAttribute(daemonNode, "maxprocs", -1));
                daemon.name           = UIGetAttribute(daemonNode, "name", "");
                daemon.status         = UIGetAttribute(daemonNode, "status", "");
                daemon.request        = UIGetAttribute(daemonNode, "request", "");
                daemon.type           = UIGetAttribute(daemonNode, "type", "");
                daemon.logfile     = UIGetAttribute(daemonNode, "logfile", "");
                daemon.description = UIGetValue(daemonNode, "");
                
                UISetDaemonProcessStatus(daemon);
                
                var xmlInfoNode = $(daemonNode).find("Cloning");
                if(xmlInfoNode && xmlInfoNode.length > 0) {
                    that._updateInfoTable(xmlInfoNode);
                } else {
                    that._clearModel();
                }
                
                that._oStatus.setText(daemon.status);
                that._oStatus.setState(daemon.statusState);
                that._oIsBusy.setVisible(daemon.isBusy);
                that._oStartButton.setEnabled(daemon.canStart);
                that._oStopButton.setEnabled(daemon.canStop);
                that._oNProcsRun.setText(daemon.runprocs);
                that._setRunningStatus(daemon.status === "running");
                
                if(daemon.maxprocs > 0) {
                    that._oNProcsReq.setMax(daemon.maxprocs);
                }
                
                if(daemon.logfile.length > 0) {
                    that._logFile = daemon.logfile;
                    that._oLogFileButton.setVisible(true);
                } else {
                    that._logFile = null;
                    that._oLogFileButton.setVisible(false);
                }
                
                that._checkUpdateNProcs(daemon);
                
                if(that._nFailed > 0) {
                    that._oStatus.setText(daemon.status + " (with " + that._nFailed + " errors)");
                    that._oStatus.setState("Error");
                }

                that._mayUpdate = true;
            };
            
            $.ajax({
                url: "../daemon/" + this.getProperty("daemon") + "?xsl=0",
                type: "GET",
                dataType: "text",
                cache: false,
                context: this,

                error: function() {
                    this._getXmlNode(null, this);
                },
                
                success: function(data) {
                    var xmlData = $.parseXML(data);
                    this._getXmlNode(xmlData, this);
                }
            });
        },
        
        /* ****************************************************************** */
        
        /* Analyze <Cloning> tag */
        _updateInfoTable : function(xmlInfoNode) {
            
            this._totalCrrRate = parseInt(UIGetAttribute(xmlInfoNode, "percentage", -1));
            var tablesNode = $(xmlInfoNode).find("tables");
            
            var sumPending  = 0;
            var sumRecorded = 0;
            var sumReplayed = 0;
            
            if(tablesNode == null || tablesNode.length < 1) {
                this._clearModel();
                return;
            }
            
            var tables = $(tablesNode).find("table");
            var len = tables.length;
            
            var modelData = [];
            
            this._nRunning = 0;
            this._nFailed = 0;
            this._nPlanned = 0;
            this._nIdle = 0;
            this._nInactive = 0;
            
            /* ******************************************* */
            
            for(var i = 0 ; i < len ; i++) {
                var table = tables[i];
                var data = {};
                
                data.name    = UIGetAttribute(table, "name", "");
                data.refid   = UIGetAttribute(table, "refid", null);
                data.status  = UIGetAttribute(table, "status", "");
                data.logFile = UIGetAttribute(table, "log", null);
                data.showLogFile = (data.logFile !== null) ? true : false;
                data.current = parseInt(UIGetAttribute(table, "current", 0));
                data.total   = parseInt(UIGetAttribute(table, "total", 0));
                data.load    = parseInt(UIGetAttribute(table, "load", 0));
                data.percent = parseInt(UIGetAttribute(table, "percentage", 0));
                data.pending = data.total - data.current;
                data.status  = data.status.toLowerCase();

                if(data.pending < 0) {
                    data.pending = 0;
                }
                
                if(data.percent > 99 && data.pending > 0) {
                    data.percent = 99;
                }
                
                if(data.status == "running")
                {
                    this._nRunning = this._nRunning + 1;
                    data.statusState = "Success";
                }
                else if(data.status == "error" || data.status == "aborted" || data.status == "failed")
                {
                    this._nFailed = this._nFailed + 1;
                    data.statusState = "Error";
                }
                else if(data.status == "inactive")
                {
                    this._nInactive = this._nInactive + 1;
                    data.statusState = "Warning";
                }
                else if(data.status == "idle")
                {
                    this._nIdle = this._nIdle + 1;
                    data.statusState = "Warning";
                }
                else
                {
                    this._nPlanned = this._nPlanned + 1;
                    data.statusState = "None";
                }
                
                modelData.push(data);
                
                sumPending  = sumPending  + data.pending;
                sumReplayed = sumReplayed + data.current;
                sumRecorded = sumRecorded + data.total;
            }
            
            /* ******************************************* */

            modelData.sort(function(a, b) {
                if(a.statusState === "Error" && b.statusState !== "Error") {
                    return -1;
                }
                
                if(a.statusState !== "Error" && b.statusState === "Error") {
                    return 1;
                }
                
                return a.percent - b.percent;
            });
            
            this._sumPending  = sumPending;
            this._sumReplayed = sumReplayed;
            this._sumRecorded = sumRecorded;
            
            this._model.setData({ data : modelData });
            this._setFooterText();
            
            /* ******************************************* */
            
            /* update daemon status */
            if(this._nInactive > 0)
            {
                this._oStatus.setText("inactive");
                this._oStatus.setState("Warning");
            }
            else if(this._nFailed > 0)
            {
                var errorTxt = this._nFailed > 1 ? "failures" : "failure";
                this._oStatus.setText("running (" + this._nFailed + " " + errorTxt + ")");
                this._oStatus.setState("Warning");
            }
            
            if(this._totalCrrRate > 99 && this._sumPending > 0) {
                this._totalCrrRate = 99;
            }
        },
        
        /* ****************************************************** */
        
        _setFooterText : function() {
            var text = "Table Statistics - Planned: " + this._nPlanned + " / Idle: " + this._nIdle + " / Running: " + this._nRunning + " / Failed: " + this._nFailed + " / Inactive: " + this._nInactive;
            this._table.setFooter(text);
        },
        
        /* ****************************************************** */
        
        _getSelectedRowModel : function(oEvent) {
            var path = oEvent.getSource().getBindingContext().getPath();
            return this._model.getProperty(path);
        },
        
        /* ****************************************************** */
        
        _openLogViewer : function(logFilePath) {
            this._oLogViewer = new LogViewer({ 
                                        logFile: logFilePath,
                                        closed : function() {
                                            this._oLogViewer.destroy();
                                        }.bind(this)
                                });
            this._oLogViewer.open();
        },
        
        /* ****************************************************** */
        /* PUBLIC HANDLER FUNCTIONS                                  */
        /* ****************************************************** */
        
        onStart : function(oEvent) {
            MessageToast.show("Starting CRR process '" + this.getProperty("daemon") + "'.");
            var content='<SAPup type="daemon"><Daemon name="' + this.getProperty("daemon") + '" request="START" /></SAPup>';
            this._postAction(content);
        },
        
        onStop : function(oEvent) {
            MessageToast.show("Stopping CRR process '" + this.getProperty("daemon") + "'.");
            this._setVisibleChangingProcs(null, null, false);
            var content='<SAPup type="daemon"><Daemon name="' + this.getProperty("daemon") + '" request="STOP" /></SAPup>';
            this._postAction(content);
        },
        
        onAccept : function(oEvent) {
                        MessageToast.show("Changing number of processes for CRR process '" + this.getProperty("daemon") + "'.");
            this._setBusyChangingProcs(true);
            var nprocs = this._oNProcsReq.getValue();
            var content='<SAPup type="daemon"><Daemon name="' + this.getProperty("daemon") + '" reqprocs="' + nprocs.toString() + '" /></SAPup>';
            this._postAction(content);
        },
        
        onSearch : function(oEvent) {
            this._applyToggleFilter();
        },
        
        onLogFile : function(oEvent) {
            this._openLogViewer("log/" + this._logFile);
        },
        
        onTableLogFile : function(oEvent) {
            var bucket = this._getSelectedRowModel(oEvent);
            if(bucket.logFile !== null) {
                this._openLogViewer("tmp/" + bucket.logFile);
            }
        },
        
        /* ****************************************************** */
        /* ****************************************************** */
        
        renderer : function(oRenderManager, oControl) {
            oControl.doRendering(oRenderManager, oControl);
        }

    });

    return CRRDaemonStatus;
});