sap.ui.define([
	"sap/m/MessageToast",
	"SUMControls/FragmentControl",
	"SUMControls/LogViewer"
], function(MessageToast, FragmentControl, LogViewer) {
	"use strict";

	var CRRProcessTable = FragmentControl.extend("SUMControls.CRRProcessTable", {
		
		metadata: {
			properties: {
				daemon: { type: "String", defaultValue: null }
			}
		},
		
		init : function() {
			/* call super initialization */
			this.initFragmentControls();
			
			this._mayUpdate = true;
			
			this._model = new sap.ui.model.json.JSONModel();
			
			this._table = this.byId("idTableInfo");
			this._table.setModel(this._model);
			
			this._oToggleRunning = this.byId("oToggleRunning");
			this._oToggleFailed = this.byId("oToggleFailed");
			this._oToggleStopped = this.byId("oToggleStopped");
			
			this._clearModel();
			this._applyToggleFilter();
		},
		
		/* ****************************************************** */
		/* PUBLIC HELPER FUNCTIONS								  */
		/* ****************************************************** */
		
		doUpdate : function() {
			if(this._mayUpdate) {
				this._updateTable();
			}
		},
		
		getFragmentName : function() {
			return "SUMFragments/CRRProcessTable";
		},
		
		setDaemon : function(daemon) {
			this.setProperty("daemon", daemon);
			return this;
		},
		
		/* ****************************************************** */
		/* PRIVATE HELPER FUNCTIONS								  */
		/* ****************************************************** */
		
		_clearModel : function() {
			var modelData = {};
			modelData.data = [];
			this._model.setData(modelData);
			
			this._nTotal = 0;
			this._nRunning = 0;
			this._nFailed = 0;
			
			this._setFooterText();
		},
		
		/* ****************************************************** */

		_getToggleFilters : function(isRunning, isFailed, isStopped) {
			
			var oFilters = [];
			
			if(isRunning) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "running"));
			}
			
			if(isFailed) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "aborted"));
			}
			
			if(isStopped) {
				oFilters.push(new sap.ui.model.Filter("status", sap.ui.model.FilterOperator.Contains, "stopped"));
			}
			
			return new sap.ui.model.Filter({ filters: oFilters, and : false });
		},
		
		_applyToggleFilter : function() {
			var isRunning     = this._oToggleRunning.getPressed();
			var isFailed      = this._oToggleFailed.getPressed();
			var isStopped     = this._oToggleStopped.getPressed();
			
			var oFilter = this._getToggleFilters(isRunning, isFailed, isStopped);
			this._table.getBinding("rows").filter(oFilter);
		},
		
		/* ****************************************************** */
		
		_getStatusState : function(status) {
			return (status === "aborted" || status === "error") ? "Error" : (status === "stopped") ? "Warning" : "Success";
		},
		
		/* ****************************************************** */
		
		_getSelectedRowModel : function(oEvent) {
			var path = oEvent.getSource().getBindingContext().getPath();
			return this._model.getProperty(path);
		},
		
		/* ****************************************************** */
		
		_openLogViewer : function(logFile) {
			this._logFile = logFile;
			this._oLogViewer = new LogViewer({ 
										logFile: this._logFile,
										closed : function() {
											this._oLogViewer.destroy();
										}.bind(this)
								});
			this._oLogViewer.open();
		},
		
		/* ****************************************************** */
		
		_getStatusUrl : function() {
			var daemon = this.getProperty("daemon") ? "/daemon/" + this.getProperty("daemon") : "";
			return ".." + daemon + "/status?details=1&xsl=0";
		},
		
		/* ****************************************************** */
		
		_fillTable : function(buckets, isRunning) {
			this._nTotal = 0;
            this._nRunning = 0;
            this._nFailed = 0;
            
            var modelData = {};
			modelData.data = [];
			
			for(var i = 0 ; i < buckets.length ; i++) {
				var bucketNode = buckets[i];
				
				var bucket = {};
				bucket.refid       = UIGetAttribute(bucketNode, "refid", "");
				bucket.logfile     = UIGetAttribute(bucketNode, "log", "");
				bucket.status      = UIGetAttribute(bucketNode, "status", "").toLowerCase();
				bucket.statusState = this._getStatusState(bucket.status);
				
				if(isRunning === 0) {
					bucket.status  = "stopped";
					bucket.statusState = "Warning";
				}
				
				if(bucket.statusState === "Success") {
					this._nRunning = this._nRunning + 1;
				} else {
					this._nFailed = this._nFailed + 1;
				}
				
				this._nTotal = this._nTotal + 1;
				
				modelData.data.push(bucket);
			}
			
			this._model.setData(modelData);
            this._setFooterText();
            this._mayUpdate = true;
		},
		
		/* ****************************************************** */
		
		_loadOldTable : function() {
			$.ajax({
				url: "../doc/analysis/MIGRATE_OPTDMO_REPLAYABAP_DAEMONSTAT.XML",
				type: "GET",
				dataType: "text",
				cache: false,
				context: this,

				error: function() {
					this._clearModel();
		            this._mayUpdate = true;
				},
				
				success: function(data) {
					var xmlData = $.parseXML(data);
					
					var cloning = $("Cloning", xmlData);
					var buckets = $(cloning).find("bucket");
					
					if(buckets == null || cloning == null || buckets.length == 0 || cloning.length == 0) {
						this._clearModel();
			            this._mayUpdate = true;
			            return;
					}
					
					this._fillTable(buckets, 0);
				}
			});
		},
		
		/* ****************************************************** */
		
		_updateTable : function() {
			this._mayUpdate = false;
			
			$.ajax({
				url: this._getStatusUrl(),
				type: "GET",
				dataType: "text",
				cache: false,
				context: this,

				error: function() {
					this._clearModel();
		            this._mayUpdate = true;
				},
				
				success: function(data) {
					var xmlData = $.parseXML(data);
					
					var cloning = $("Cloning", xmlData);
					var buckets = $(cloning).find("bucket");
					
					if(buckets == null || cloning == null || buckets.length == 0 || cloning.length == 0) {
						this._loadOldTable();
			            return;
					}
					
					this._fillTable(buckets, 1);
				}
			});
		},
		
		/* ****************************************************** */
		
		_setFooterText : function() {
			var text = "Process Statistics - Running: " + this._nRunning  + " / Failed: " + this._nFailed + " / Total: " + this._nTotal;
			this._table.setFooter(text);
		},
		
		/* ****************************************************** */
		/* PUBLIC HANDLER FUNCTIONS								  */
		/* ****************************************************** */
		
		handleFilterToggle : function(oEvent) {
			this._applyToggleFilter();
		},
		
		onLog : function(oEvent) {
			var bucket = this._getSelectedRowModel(oEvent);
			this._openLogViewer("tmp/" + bucket.logfile);
		},
		
		/* ****************************************************** */
		/* ****************************************************** */
		
		renderer : function(oRenderManager, oControl) {
			oControl.doRendering(oRenderManager, oControl);
		}

	});

	return CRRProcessTable;
});