---------------------------------------------------------------------------------------------------------------------------------
-- SAP SE
-- Version: 0009 - 17.06.2020
-- This script checks a SAP ASE configuration meeting minimal requirements for a Software Update Manager (SUM) maintenance activity.
-- Send us your feedback, comments and ideas about this script, for example by opening a customer ticket on component BC-DB-SYB, 
--     referring to the SAP Note from which you have downloaded this script as well as this script. 
--
-- Before running this script following parameter need to be adjusted to your environment: 
--     line below line containing '-- CHANGE THIS'
--         @checkMode
--
-- Furthermore, make sure the user executing this script (eg. sapsa) has select permission for table SVERS and starting with 
--     SAP ASE 16.0 SP03 select permission for object CUDF_VERSION.
--
-- To call this script, log on to the SAP ASE database host as operating system user syb<SID> and store this script for example to 
--     the folder /sybase/<SID>. Afterwards issue the command:
--     isql -U<user_name> -P<password> -S<server_name> -w999 -X -i<path_to_script>
--     An expale call could be:
--     UNIX/Linux:
--         isql -Usapsa -P1234abcd -SSID -w999 -X -i/sybase/SID/SYB_miniCheck.sql
--     Windows:
--         isql -Usapsa -P1234abcd -SSID -w999 -X -iE:\sybase\SID\SYB_miniCheck.sql
-- ---------------------------------------------------------------------------------------------------------------------------------

SET NOCOUNT ON
set replication off

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Declarations
-- ---------------------------------------------------------------------------------------------------------------------------------
DECLARE @mySID         CHAR(3)
DECLARE @checkMode     VARCHAR(20)
DECLARE @myStatement   VARCHAR(16384)

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Initializations
-- ---------------------------------------------------------------------------------------------------------------------------------

-- CHANGE THIS TO YOUR DESIRED CHECK MODE  
SELECT @checkMode = 'UPG'  
-- Acceptable values are:
--    'UPG'        Standard SAP Software Update Manager (SUM) maintenance activity
--    'DMO2SYB'    SUM DMO with target SAP ASE, script is to be called on the target SAP ASE database host
--    'DMO2HANA'   SUM DMO system migration with target HANA or system conversion with target S/4 HANA

SELECT @mySID = @@servername 

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Preparations
-- ---------------------------------------------------------------------------------------------------------------------------------
IF @mySID = '0ID'
BEGIN
    print 'Make sure to adjust variable @mySID to your SAP System ID before execution, as well as variable @checkMode. See SAP Note 2529827.'
    select syb_quit()
END

USE @mySID

IF object_id('tempdb..#checkResults') IS NOT NULL
BEGIN
    DROP TABLE #checkResults
END
CREATE TABLE #checkResults
    ( 
        checkStatus         CHAR(7), 
        checkParameter      VARCHAR(65), 
        currentValue        VARCHAR(20), 
        expectedValue       VARCHAR(20),
        SAPnote_Comment     VARCHAR(255)
    )

IF object_id('tempdb..#myVariables') IS NOT NULL
BEGIN
    DROP TABLE #myVariables
END
CREATE TABLE #myVariables
    ( 
        param               VARCHAR(65),
        currentValue        VARCHAR(20)
    )

INSERT INTO #myVariables               
    VALUES ('SID', @mySID)
INSERT INTO #myVariables               
    VALUES ('checkMode', @checkMode)

IF object_id('conv_attr_to_number') IS NOT NULL
BEGIN
    EXEC('DROP FUNCTION conv_attr_to_number')
END

IF object_id('retrieveFreeSpace') IS NOT NULL
BEGIN
    EXEC('DROP PROCEDURE retrieveFreeSpace')
END
GO
set replication off
exec('
CREATE FUNCTION conv_attr_to_number (@c VARCHAR(30))
    RETURNS BIGINT
    AS
    BEGIN
        DECLARE @result INT
        DECLARE @a      BIGINT
        DECLARE @len    BIGINT
        DECLARE @fac    BIGINT
        DECLARE @suffix CHAR(1)
        SELECT @c = rtrim(ltrim(@c))
        SELECT @len = len(@c)
        SELECT @fac = 1
        IF @len > 1
        BEGIN
            SELECT @suffix = substring( @c, @len, 1 )
            IF @suffix = ''B'' OR @suffix = ''%''
            BEGIN
                SELECT @len = @len - 1
                SELECT @c = substring( @c, 1, @len )
            END
        END
        IF @len > 1
        BEGIN
            SELECT @suffix = substring( @c, @len, 1 )
            IF @suffix in (  ''K'', ''M'', ''G'', ''T'' )
            BEGIN
                IF @suffix = ''K''
                    SELECT @fac = 1024
                IF @suffix = ''M''
                    SELECT @fac = 1024 * 1024
                ELSE IF @suffix = ''G''
                    SELECT @fac = 1024 * 1024 * 1024
                ELSE IF @suffix = ''T''
                    SELECT @fac = convert( BIGINT, 1024 * 1024 ) * 1024 * 1024
                SELECT @len = @len - 1
                SELECT @c = substring( @c, 1, @len )
            END
        END
        IF isnumeric( @c ) = 1
            SELECT @a = convert ( BIGINT, @c ) * @fac
        ELSE
            SELECT @a = NULL
        IF @a = 0
            SELECT @a = -1
        RETURN @a
    END')
GO
set replication off
exec('
CREATE PROCEDURE retrieveFreeSpace ( @FS DECIMAL(15) OUTPUT, @AE CHAR(3) OUTPUT )
AS BEGIN
    DECLARE @growth_abs BIGINT
    DECLARE @growth_rel BIGINT
    DECLARE @auto_total BIGINT
    DECLARE @auto_total_MB  INT
    DECLARE @server_is_enabled INT
    DECLARE @PageSize   INT
    DECLARE @UsedSpace  DECIMAL(15)
    DECLARE @TotalSpace DECIMAL(15)
    DECLARE @mySID      CHAR(3)
    
    SELECT @mySID = currentValue 
        FROM #myVariables 
        WHERE param = ''SID''
    SELECT @server_is_enabled = object_info2 
        FROM master.dbo.sysattributes 
        WHERE class = 19 AND object_type = ''XT'' AND object_info1 = 1
    SELECT @growth_abs = dbo.conv_attr_to_number(convert(VARCHAR(30), char_value) )
        FROM master.dbo.sysattributes 
        WHERE class = 19 AND object_type = ''DB'' AND object_cinfo = @mySID  AND object = 1 AND object_info2 = 1 AND object_info1 = 3
    SELECT @growth_rel = dbo.conv_attr_to_number(convert(VARCHAR(30), char_value) )
        FROM master.dbo.sysattributes 
        WHERE class = 19 AND object_type = ''DB'' AND object_cinfo = @mySID  AND object = 1 AND object_info2 = 1 AND object_info1 = 4
    SELECT @auto_total = dbo.conv_attr_to_number(convert(VARCHAR(30), char_value))
        FROM master.dbo.sysattributes 
        WHERE class = 19 AND object_type = ''DB'' AND object_cinfo = @mySID  AND object = 1 AND object_info2 = 1  AND object_info1 = 5
    IF @server_is_enabled = 1 
        AND NOT (  ( @growth_abs IS NULL OR @growth_abs = -1 )  
        AND ( @growth_rel IS NULL OR @growth_rel = -1 ) )
        BEGIN
            SELECT @AE = ''ON''
            IF @auto_total IS NULL OR @auto_total = -1
                SELECT @auto_total_MB = 2147483646 -- we use max signed INT as alias for ''unlimited'' 
            ELSE
                BEGIN
                    SELECT @auto_total = @auto_total / ( 1024 * 1024 )
                    SELECT @auto_total_MB = convert( INT, @auto_total )
                END
        END
    ELSE
        BEGIN
            SELECT @AE = ''OFF''
            SELECT @auto_total_MB = 0
        END
    SELECT @PageSize = low/1024 
        FROM master..spt_values 
        WHERE type = ''E'' AND number = 1 AT ISOLATION 1
    CREATE TABLE #filestats
        (DatabName CHAR(20),
         DataSize VARCHAR(256),
         DataUsed VARCHAR(256))
    INSERT INTO #filestats 
        SELECT DatabName = convert(CHAR(20),db_name(dbid)),
               DataSize = str(sum(cast ( size as BIGINT ) * (@@maxpagesize/1024)) / 1024),
               DataUsed = str(sum((cast ( size as BIGINT ) - curunreservedpgs(dbid, lstart, unreservedpgs)) * (@@maxpagesize/1024)) / 1024)
            FROM master..sysusages
            WHERE dbid =db_id() AND (segmap & 4) = 0
            GROUP BY dbid
    
    SELECT @UsedSpace = sum(cast(DataUsed as DECIMAL(15))) FROM #filestats
    SELECT @TotalSpace = sum(cast(DataSize as DECIMAL(15))) FROM #filestats
    DROP TABLE #filestats
    IF ( @auto_total_MB > @TotalSpace )
        SELECT @TotalSpace = @auto_total_MB
    SELECT @FS = (@TotalSpace - @UsedSpace)
END')
GO
set replication off

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Declarations
-- ---------------------------------------------------------------------------------------------------------------------------------
DECLARE @mySID         CHAR(3)
DECLARE @checkMode     VARCHAR(20)
DECLARE @sRC           VARCHAR(20)
DECLARE @iRC           BIGINT
DECLARE @fRC           FLOAT
DECLARE @TotalSpace    DECIMAL(15)
DECLARE @numpgsmb      FLOAT        /* Number of Pages per Megabyte */
DECLARE @FreeSpace     DECIMAL(15)
DECLARE @autoExtend    CHAR(3)
DECLARE @myStatement   VARCHAR(16384)
DECLARE @numberOfLocks BIGINT

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Initializations
-- ---------------------------------------------------------------------------------------------------------------------------------
SELECT @sRC  = '0'
SELECT @iRC  = 0
SELECT @fRC  = 0
SELECT @mySID = currentValue FROM #myVariables WHERE param = 'SID'
SELECT @checkMode = currentValue FROM #myVariables WHERE param = 'checkMode'
SELECT @numberOfLocks = 5000000

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Checks
-- ---------------------------------------------------------------------------------------------------------------------------------

-- Checks independent from the selected SUM scenario

-- From DB side kernel patch levels cannot be checked, but just put a info message out
INSERT INTO #checkResults                 
           VALUES ('INFO', 'Targeted SAP Kernel release', '   ', '>= 749', 'SAP Note SYB: Add. Info - Software Update Manager 2.0')

-- Checks for the minimum SAP ASE release required for SUM 2.0
IF @@sbssav < '16.0.02.07'
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'SAP ASE release', @@sbssav, '>= 16.0.02.07', 'SAP Note SYB: Add. Info - Software Update Manager 2.0; Upgrade SAP ASE according to SAP Note 2800483.')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_serveroption loopback, ''net password encryption'', true', 'TRUE', 'TRUE', 'SAP Note 2718131')

-- Check existence of saptempdb
SELECT @iRC = COUNT(name) FROM master..sysdatabases WHERE name = 'saptempdb'
IF @iRC > 0
BEGIN    
    INSERT INTO #checkResults                 
           VALUES ('OK', 'database saptempdb exists', 'yes', 'yes', 'SAP Note 1680803')
    
    -- check size of saptempdb   
    SELECT @numpgsmb = (1048576. / v.low) 
           FROM master.dbo.spt_values v
           WHERE v.number = 1 AND v.type = 'E'    AT ISOLATION 1
    SELECT @TotalSpace = sum(u.size) / @numpgsmb
           FROM master.dbo.sysusages u    
           WHERE dbid = db_id() AND u.vdevno >= 0
    IF @TotalSpace >= 8000
        INSERT INTO #checkResults                 
           VALUES ('OK', 'size of database saptempdb', CONVERT( VARCHAR(20), @TotalSpace ), '>= 8000', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
           VALUES ('ERROR', 'size of database saptempdb', CONVERT( VARCHAR(20), @TotalSpace ), '>= 8000', 'SAP Note 1680803; Increase size accordingly.')     
END
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'database saptempdb exists', 'no', 'yes', 'SAP Note 1680803; Create saptempdb accordingly.')

-- Check configuration parameter 'update statistics hashing' <> ON
SELECT @sRC = cu.value2
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'update statistics hashing'
IF @sRC = 'partial'
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''update statistics hashing''', 'partial', 'partial', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''update statistics hashing''', @sRC , 'partial', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'enable bulk inserts'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'enable bulk inserts'
IF @iRC = 0
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''enable bulk inserts''', '0', '0', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''enable bulk inserts''', CONVERT( VARCHAR(20), @iRC ), '0', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'extend implicit conversion'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'extend implicit conversion'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''extend implicit conversion''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''extend implicit conversion''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'enable functionality group'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'enable functionality group'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''enable functionality group''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''enable functionality group''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'enable inline default sharing'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'enable inline default sharing'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''enable inline default sharing''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''enable inline default sharing''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'select for update'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'select for update'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''select for update''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''select for update''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'enable permissive unicode'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'enable permissive unicode'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''enable permissive unicode''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''enable permissive unicode''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'quoted identifier enhancements'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'quoted identifier enhancements'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''quoted identifier enhancements''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''quoted identifier enhancements''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'streamlined dynamic SQL'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'streamlined dynamic SQL'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''streamlined dynamic SQL''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''streamlined dynamic SQL''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check configuration parameter 'suppress js max task message'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'suppress js max task message'
IF @iRC = 1
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''suppress js max task message''', '1', '1', 'SAP Note 1581695')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''suppress js max task message''', CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 1581695; change database configuration parameter accordingly.')

-- Check number of worker processes >= max online engines * 2.5
SELECT @myStatement = 'SELECT @fRC = 2.5 * '
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'max online engines'
SELECT @myStatement = @myStatement + CONVERT( VARCHAR(20), @iRC)
EXEC (@myStatement)

SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'number of worker processes'
IF @iRC < CONVERT(BIGINT,@fRC)
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''number of worker processes''', CONVERT( VARCHAR(20), @iRC ), '>= ' + CONVERT( VARCHAR(20), CONVERT(BIGINT,@fRC)), 'SAP Note 1680803; change database configuration parameter accordingly.' ) 
ELSE
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''number of worker processes''', CONVERT( VARCHAR(20), @iRC), '>= ' + CONVERT( VARCHAR(20), CONVERT(BIGINT,@fRC) ), 'SAP Note 1680803' ) 

-- Check IF sp_serveroption loopback, 'net password encryption', true 
IF @@sbssav < '16.0.00.00'
    BEGIN
    SELECT @iRC = COUNT(*) 
        FROM master.dbo.sysservers 
        WHERE srvname = 'loopback' AND srvstatus & 2 = 2 AND srvnetname = @mySID
    IF @iRC = 1
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_serveroption loopback, ''net password encryption'', true', 'TRUE', 'TRUE', 'SAP Note 2718131')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_serveroption loopback, ''net password encryption'', true', 'FALSE', 'TRUE', 'SAP Note 2718131; change server option accordingly.')
END

-- Check for database option 'unique auto_identity index' set to FALSE in database <SID>
SELECT @iRC = d.status2&64 FROM master.dbo.spt_values v, master.dbo.sysdatabases d WHERE d.dbid = db_id(@mySID) AND v.name = 'unique auto_identity index'
IF 0 <> @iRC
    INSERT INTO #checkResults                 
         VALUES ('ERROR', 'sp_dboption ' +@mySID + ', ''unique auto_identity index''', 'TRUE', 'FALSE', 'SAP Note 1581695; change database option accordingly.')
ELSE
    INSERT INTO #checkResults                 
         VALUES ('OK', 'sp_dboption ' +@mySID + ', ''unique auto_identity index''', 'FALSE', 'FALSE', 'SAP Note 1581695')
    
-- Check for database option 'unique auto_identity index' set to FALSE in database saptools    
SELECT @iRC = d.status2&64 FROM master.dbo.spt_values v, master.dbo.sysdatabases d WHERE d.dbid = db_id('saptools') AND v.name = 'unique auto_identity index'
IF 0 <> @iRC
    INSERT INTO #checkResults                 
         VALUES ('ERROR', 'sp_dboption saptools, ''unique auto_identity index''', 'TRUE', 'FALSE', 'SAP Note 1581695; change database option accordingly.')
ELSE
    INSERT INTO #checkResults                 
         VALUES ('OK', 'sp_dboption saptools, ''unique auto_identity index''', 'FALSE', 'FALSE', 'SAP Note 1581695')

-- Check IF ASE 16. SP03 AND higher release have been upgraded with SAP Host Agent 
IF @@sbssav >= '16.0.03.00'
BEGIN
    -- Check if CUDFs are working
    SELECT @iRC = COUNT(*) 
        FROM sysobjects 
        WHERE name = 'CUDF_VERSION' 
            AND type = 'F' 
            AND uid = user_id('SAPSR3')
    IF @iRC = 0
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'CUDF existance ' + @mySID, CONVERT( VARCHAR(20), @iRC ), '1', 'SAP Note 2800483; Rerun SAP ASE database upgrade with option force.')   
    ELSE
    BEGIN      
        SELECT @myStatement = 'SELECT SAPSR3.CUDF_VERSION()'
        EXEC (@myStatement)
        SELECT @iRC = @@error
        IF @iRC = 0
            INSERT INTO #checkResults                 
               VALUES ('OK', 'CUDF execution ' + @mySID, CONVERT( VARCHAR(20), @iRC ), '0', 'SAP Note 2800483')
        ELSE
            INSERT INTO #checkResults                 
               VALUES ('ERROR', 'CUDF execution ' + @mySID, CONVERT( VARCHAR(20), @iRC ), '0', 'SAP Note 2800483; Rerun SAP ASE database upgrade with option force.')  
    END
    -- Check BALDR setup
    IF EXISTS ( SELECT 1 FROM saptools..sysobjects WHERE name = 'DBH_CONFIG' AND type = 'U' )
    BEGIN
        SELECT @myStatement = 'SELECT @sRC = CONVERT(VARCHAR(20), VALUE) from saptools..DBH_CONFIG WHERE [KEY] = ''BALDR_VERSION'''
        EXEC (@myStatement)
        IF @sRC = '0'
            INSERT INTO #checkResults                 
                   VALUES ('ERROR', 'BALDR is setup for ' + @mySID, CONVERT( VARCHAR(20), 'no' ), 'yes', 'SAP Note 2800483; Rerun SAP ASE database upgrade with option force.') 
        ELSE    
            INSERT INTO #checkResults                 
                       VALUES ('OK', 'BALDR is setup for ' + @mySID, CONVERT( VARCHAR(20), 'yes' ), 'yes', 'Found BALDR Setup from ' + @sRC + '.' ) 
    END
    ELSE
        INSERT INTO #checkResults                 
                   VALUES ('ERROR', 'BALDR is setup for ' + @mySID, CONVERT( VARCHAR(20), 'no' ), 'yes', 'SAP Note 2800483; Rerun SAP ASE database upgrade with option force.') 
END
   
-- Check configuration parameter 'number of user connections'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config 
        AND c.name = 'number of user connections'
IF @iRC >= 200
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''number of user connections''', CONVERT( VARCHAR(20), @iRC ), '>= 200', 'SAP Note 1680803')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''number of user connections''', CONVERT( VARCHAR(20), @iRC ), '>= 200', 'SAP Note 1680803; change database configuration parameter accordingly.')

-- Check configuration parameter 'number of locks'
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config 
        AND c.name = 'number of locks'
IF @iRC >= @numberOfLocks
    INSERT INTO #checkResults                 
           VALUES ('OK', 'sp_configure ''number of locks''', CONVERT( VARCHAR(20), @iRC ), '>= ' + CONVERT( VARCHAR(20), @numberOfLocks ) , 'SAP Note 1680803')
ELSE
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''number of locks''', CONVERT( VARCHAR(20), @iRC ), '>= ' + CONVERT( VARCHAR(20), @numberOfLocks ), 'SAP Note 1680803; change database configuration parameter accordingly.')

-- Check configuration parameter 'number of locks >= (number of user connections + number of worker processes)*20'
SELECT @myStatement = 'SELECT @fRC = 20 * ( '
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'number of user connections'
SELECT @myStatement = @myStatement + CONVERT( VARCHAR(20), @iRC) + ' + '
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'number of worker processes'
SELECT @myStatement = @myStatement + CONVERT( VARCHAR(20), @iRC) + ' )'
EXEC (@myStatement)
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'number of locks'
IF @numberOfLocks <= CONVERT(BIGINT,@fRC)
BEGIN
    IF @iRC < CONVERT(BIGINT,@fRC)
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''number of locks''', CONVERT( VARCHAR(20), @iRC ), '>= ' + CONVERT( VARCHAR(20), CONVERT(BIGINT,@fRC) ), 'SAP Note 1680803; change database configuration parameter accordingly.' ) 
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''number of locks''', CONVERT( VARCHAR(20), @iRC), '>= ' + CONVERT( VARCHAR(20), CONVERT(BIGINT,@fRC) ), 'SAP Note 1680803' ) 
END           

-- Check configuration parameter 'lock hashtable size >= @number of locks@ / 1000000 * 8192  && lock hashtable size >= 8192 , see  https://launchpad.support.sap.com/#/notes/1581695 
SELECT @myStatement = 'SELECT @fRC = '
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'number of locks'
SELECT @myStatement = @myStatement + CONVERT( VARCHAR(20), @iRC) + ' / 1000000 * 8192'
EXEC (@myStatement)
SELECT @iRC = cu.value
    FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
    WHERE c.config = cu.config AND c.name = 'lock hashtable size'
IF @iRC < 8192
    INSERT INTO #checkResults                 
           VALUES ('ERROR', 'sp_configure ''lock hashtable size''', CONVERT( VARCHAR(20), @iRC ), '>= 8192' , 'SAP Note 1680803; change database configuration parameter accordingly.' ) 
ELSE
    BEGIN
        IF @iRC < CONVERT(BIGINT,@fRC)
            INSERT INTO #checkResults                 
                   VALUES ('ERROR', 'sp_configure ''lock hashtable size''', CONVERT( VARCHAR(20), @iRC ), '>= ' + CONVERT( VARCHAR(20), CONVERT(BIGINT,@fRC) ), 'SAP Note 1680803; change database configuration parameter accordingly.' ) 
        ELSE
            INSERT INTO #checkResults                 
                   VALUES ('OK', 'sp_configure ''lock hashtable size''', CONVERT( VARCHAR(20), @iRC), '>= ' + CONVERT( VARCHAR(20), CONVERT(BIGINT,@fRC) ), 'SAP Note 1680803' ) 
    END

-- Check minimum size of named cache saptempdb_cache >= 1GB
IF EXISTS (SELECT 1 FROM master.dbo.syscacheinfo WHERE cache_name = 'saptempdb_cache')
BEGIN
    SELECT @fRC = run_size FROM master.dbo.syscacheinfo WHERE cache_name = 'saptempdb_cache'
    IF @fRC < 1024.0
        INSERT INTO #checkResults                 
           VALUES ('ERROR', 'size of named cache saptempdb_cache', CONVERT( VARCHAR(20), @fRC ), '>= 1024 MB', 'SAP Note 1680803; change database configuration parameter accordingly.' ) 
    ELSE
    INSERT INTO #checkResults                 
           VALUES ('OK', 'size of named cache saptempdb_cache', CONVERT( VARCHAR(20), @fRC ), '>= 1024 MB', 'SAP Note 1680803' ) 
END

-- Checks for a standard SUM maintenance activity
IF @checkMode = 'UPG' 
BEGIN
    print 'Generic SUM tests are executed.'
END

-- Checks for SUM DMO with target SAP ASE
IF @checkMode = 'DMO2SYB' 
BEGIN
    -- Check for empty user database
    SELECT @iRC = COUNT(name) 
        FROM sysobjects 
        WHERE uid = user_id('SAPSR3') 
            AND type <> 'F' 
    IF @iRC = 0
        INSERT INTO #checkResults                 
               VALUES ('OK', 'objects in user database' + @mySID, CONVERT( VARCHAR(20), @iRC ), '0', 'SAP Note 2523137')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'objects in user database ' + @mySID, CONVERT( VARCHAR(20), @iRC ), '0', 'SAP Note 2523137; make sure target user database is empty.')

    -- Check free space in user database
    EXEC retrieveFreeSpace @FS = @FreeSpace OUTPUT, @AE = @autoExtend OUTPUT
    
    IF @autoExtend = 'ON'
        INSERT INTO #checkResults                 
                       VALUES ('WARNING', 'free space in user database ' + @mySID, CONVERT( VARCHAR(20), @FreeSpace ) + ' MB', '>= 50000 MB', 'Auto Extension is active, make sure extension will keep up with growth rate by choosing an adequate free space threshold and extension increment.')
    ELSE
    BEGIN
        IF @FreeSpace >= 50000
            INSERT INTO #checkResults                 
                   VALUES ('OK', 'free space in user database ' + @mySID, CONVERT( VARCHAR(20), @FreeSpace ) + ' MB', '>= 50000 MB', 'SAP Note 1680803')
        ELSE
            INSERT INTO #checkResults                 
                   VALUES ('ERROR', 'free space in user database ' + @mySID, CONVERT( VARCHAR(20), @FreeSpace ) + ' MB', '>= 50000 MB', 'SAP Note 1680803, either increase free space in the database or consider database feature auto extend, but make sure extension will keep up with growth rate by choosing an adequate free space threshold and extension increment.')
    END
END

-- Checks for SUM DMO with target HANA
IF @checkMode = 'DMO2HANA' 
BEGIN
    -- Check configuration parameter 'number of open objects'
    SELECT @iRC = cu.value
        FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
        WHERE c.config = cu.config 
            AND c.name = 'number of open objects'
    IF @iRC >= 150000
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''number of open objects''', CONVERT( VARCHAR(20), @iRC ), '>= 150000', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''number of open objects''', CONVERT( VARCHAR(20), @iRC ), '>= 150000', 'SAP Note 1680803; change database configuration parameter accordingly.')
    
    -- Check configuration parameter 'number of open indexes'
    SELECT @iRC = cu.value
        FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
        WHERE c.config = cu.config 
            AND c.name = 'number of open indexes'
    IF @iRC >= 150000
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''number of open indexes''', CONVERT( VARCHAR(20), @iRC ), '>= 150000', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''number of open indexes''', CONVERT( VARCHAR(20), @iRC ), '>= 150000', 'SAP Note 1680803; change database configuration parameter accordingly.')

    -- Check configuration parameter 'number of open partitions'
    SELECT @iRC = cu.value
        FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
        WHERE c.config = cu.config 
            AND c.name = 'number of open partitions'
    IF @iRC >= 60000
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''number of open partitions''', CONVERT( VARCHAR(20), @iRC ), '>= 60000', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''number of open partitions''', CONVERT( VARCHAR(20), @iRC ), '>= 60000', 'SAP Note 1680803; change database configuration parameter accordingly.')
    
    -- Check configuration parameter 'max online engines'
    SELECT @iRC = cu.value
        FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
        WHERE c.config = cu.config 
            AND c.name = 'max online engines'
    IF @iRC >= 4
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''max online engines''', CONVERT( VARCHAR(20), @iRC ), '>= 4', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''max online engines''', CONVERT( VARCHAR(20), @iRC ), '>= 4', 'SAP Note 1680803; change database configuration parameter accordingly.')

    -- Check configuration parameter 'row lock promotion HWM'
    SELECT @iRC = cu.value
        FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
        WHERE c.config = cu.config 
            AND c.name = 'row lock promotion HWM'
    IF @iRC >= 2147483647
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''row lock promotion HWM''', CONVERT( VARCHAR(20), @iRC ), '>= 2147483647', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''row lock promotion HWM''', CONVERT( VARCHAR(20), @iRC ), '>= 2147483647', 'SAP Note 1680803; change database configuration parameter accordingly.')

    -- Check configuration parameter 'row lock promotion LWM'
    SELECT @iRC = cu.value
        FROM master.dbo.sysconfigures c, master.dbo.syscurconfigs cu
        WHERE c.config = cu.config 
            AND c.name = 'row lock promotion LWM'
    IF @iRC >= 2147483646
        INSERT INTO #checkResults                 
               VALUES ('OK', 'sp_configure ''row lock promotion LWM''', CONVERT( VARCHAR(20), @iRC ), '>= 2147483646', 'SAP Note 1680803')
    ELSE
        INSERT INTO #checkResults                 
               VALUES ('ERROR', 'sp_configure ''row lock promotion LWM''', CONVERT( VARCHAR(20), @iRC ), '>= 2147483646', 'SAP Note 1680803; change database configuration parameter accordingly.')
END

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Summary
-- ---------------------------------------------------------------------------------------------------------------------------------
PRINT 'Check results for SID ''%1!'' with check mode ''%2!'' are: ', @mySID, @checkMode
SELECT * FROM #checkResults ORDER BY checkStatus

-- ---------------------------------------------------------------------------------------------------------------------------------
-- Cleanup
-- ---------------------------------------------------------------------------------------------------------------------------------

IF object_id('tempdb..#myVariables') IS NOT NULL
BEGIN
    DROP TABLE #myVariables
END
IF object_id('tempdb..#checkResults') IS NOT NULL
BEGIN
    DROP TABLE #checkResults
END

IF EXISTS ( SELECT 1 FROM sysobjects WHERE name = 'conv_attr_to_number' AND type = 'SF' AND uid = user_id())
BEGIN
    SELECT @myStatement = 'DROP FUNCTION conv_attr_to_number'
    EXEC(@myStatement)
END

IF EXISTS ( SELECT 1 FROM sysobjects WHERE name = 'retrieveFreeSpace' AND type = 'P' AND uid = user_id())
    SELECT @myStatement = 'DROP PROCEDURE retrieveFreeSpace'
    EXEC(@myStatement)
go