#! /usr/bin/perl
#

=comment

META:
For the sake of brevity, we often do not distinguish between a data structure
and a reference to such in our comments.
So we frequently speak of 'arrays' or 'hashes' where we actually mean array or hash references.


ABSTRACT:
This module is called (with entry point 'readXmlFile()') 
in 'buildpackage.pl'. (i.e. it is part of the 'buildpackage; program.)

It parses an XML File and creates a perl data structure from it.
Parsing and data creation is mainly generic, except for some tags with special semantics
to simulate conditional compiling.


INPUT:
an XML Description used to control MaxDB packaging given by file name 
in '$Vars->{'PACKDEF_MASTER_XML'}'.
This file, typically called 'allpackages.xml', is created during 'allpackages.mac'
from smaller, human readable units with XSLT, and validated against an XML schema.
This way, this parser does not have to perform semantic checking of its input

Platform dependent conditional compiling is simulated in XML by 'constraint' tags
which define logical conditions in the 'constraints' section of the input file.
The attributed 'id's of these tags are referenced in other tags using the 'refid' attribute.
If the referenced condition holds, the referencing tag is parsed, otherwise it is not parsed.

There are also a number of modifier tags, which are used in file descriptions to provide
platform dependent filename extensions (e.g. for executables or libraries) and path prefixes. 
These are currently: 
'shlibExt', 'progExt', 'debugSymbolsExt', 'libDir', 'shellscriptExt', 'batchFileExt',
and 'pyshlibExt'.

an abbreviated example of the input file 'allpackages.xml' (which is created during 'allpackages.mac'
from smaller, human readable units with XSLT, and validated against an XML schema):
(ommissions are marked with '[***]')

-------------------------------------------------------------------------------------------
<?xml version="1.0" encoding="UTF-8"?>

<packDefs xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="allpackages.xsd">

<atomicConstraints>
    
    <!--     several tags in the XML package configuration files in this
             directory are only effective for certain platform/OS
             combinations. These tags have the 'constraint' attribute set which
             references the respective constraint id. 
             An atomic constraint condition is either a match of an environment
             variable against an expected value or a a test of an environment
             variable for definedness. -->

[***]

    <constraint id="win">
      <match>
        <envName>OS</envName>
        <expectedValue>WIN32</expectedValue>
      </match>
    </constraint>
    
    <constraint id="linux">
      <match>
        <envName>OSSPEC</envName>
        <expectedValue>LINUX</expectedValue>
      </match>
    </constraint>
    
    <constraint id="buildBenchMarkKernel">
      <envDefined>CREATE_BENCHMARK_KERNEL</envDefined>
    </constraint>
    
[***]
        
</atomicConstraints>
  
  
<compoundConstraints>
    
    <constraint id="winIa64">
      <and>
        <constraint refid="win"/>
        <constraint refid="bit64"/>
        <constraint refid="machI386"/>
      </and>
    </constraint>
    
    <constraint id="winNotamd64">
      <and>
        <constraint refid="win"/>
        <not>
          <constraint refid="machAmd64"/>
        </not>
      </and>
    </constraint>
    
    <constraint id="mscVersion13or14">
      <or>
        <constraint refid="mscVersion13"/>
        <constraint refid="mscVersion14"/>
      </or>
    </constraint>
    
[***]

</compoundConstraints>
  
  
<packages>

[***]

  <package id="dbana">
      <name>DB Analyzer</name>
      <archive>SDBANA.TGZ</archive>
      <interfaceVersion>0.2</interfaceVersion>
      <softwareVersion>
        <byManifest>
          <manifestSrcPath>manifest/server/manifest</manifestSrcPath>
          <versionKey>release</versionKey>
          <buildKey>fullversion</buildKey>
        </byManifest>
      </softwareVersion>
      <testFile>
        <path>
           <progExt>bin/dbanalyzer</progExt>
        </path>
      </testFile>
      <desc>Database Analyzer: Tool for analyzing database performance</desc>
      <dependencies>
        <dependsOn refid="base" geBuildInfoKey="VERSION"/>
      </dependencies>
      <provides>
        <feature refid="diagnosticTools"/>
      </provides>
    <files>
      <file>
        <path>env/dbanalyzer.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer72.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer73.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer742.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer743.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer75.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer76.cfg</path>
      </file>
      <file>
        <path>env/dbanalyzer77.cfg</path>
      </file>
      <file testFile="yes">
        <path>
          <progExt>bin/dbanalyzer</progExt>
        </path>
      </file>
    </files>
  </package>


  <package id="lcsim">
      <name>LC Simulator</name>
      <archive>LCSIMUL.TGZ</archive>
      <lcPoolSubDir>TEST_COMPONENTS</lcPoolSubDir>
      <interfaceVersion>0.2</interfaceVersion>
      <softwareVersion>
        <byManifest>
          <manifestSrcPath>manifest/server/manifest</manifestSrcPath>
          <versionKey>release</versionKey>
          <buildKey>fullversion</buildKey>
        </byManifest>
      </softwareVersion>
      <dependencies>
        <dependsOn refid="kernel" eqBuildInfoKey="VERSION"/>
      </dependencies>
      <provides>
        <feature refid="testPackages"/>
      </provides>
      <files>
        <file>
            <path><libDir><shlibExt>libmemdbg</shlibExt></libDir></path>
        </file>
        <file testFile="yes">
            <path><libDir><shlibExt>libomssimul</shlibExt></libDir></path>
        </file>
[***]

    </files>
  </package>
[***]

</packages>


<installer>

    <files>
      <file>
        <path>SDBINST.TGZ</path>
      </file>
      <file>
        <path>README</path>
      </file>
      <file>
        <path><progExt>SDBINST</progExt></path>
      </file>
[***]

    </files>
</installer>

</packDefs>

-------------------------------------------------------------------------------------------


OUTPUT:
The output is an exported perl data structure '$packDefs' which e.g. looks like this:
(this dump does not exactly match the abbreviated xml example above, but you will get the idea...)

-------------------------------------------------------------------------------------------
$packDefs: 

    {
      'installer'=>
        [
          
          [
            'SDBINST.TGZ', 
            'README', 
          ], 
          
          [
            'SDBINST.exe', 
            'perl58.dll', 
            'sdbrun.dll', 
            'SDBSETUP.exe', 
            'WXPERL.TGZ', 
            'RESOURCES.TGZ', 
            'wxmsw26_vc_sdb.dll', 
            'Wx.dll', 
            'Grid.dll', 
          ], 
        ],
        
        
      'packages'=>
        [
          
[***]

          {
            'archive'=>'SDBANA.TGZ', 
            'files'=>
              [
                
                {
                  'path'=>'env/dbanalyzer.cfg', 
                }, 
                
[***]
                {
                  'path'=>'bin/dbanalyzer.exe', 
                }, 
              ], 
            'name'=>'DB Analyzer', 
            'dependencies'=>
              [
                
                {
                  'dependsOn'=>'base', 
                  'geBuildInfoKey'=>'VERSION', 
                }, 
              ], 
            'softwareVersion'=>
              {
                'byManifest'=>
                    {
                      'manifestSrcPath'=>'manifest/server/manifest', 
                      'versionKey'=>'release', 
                      'buildKey'=>'fullversion', 
                    }, 
              }, 
            'interfaceVersion'=>'0.2', 
            'provides'=>
              [
                
                {
                  'refid'=>'diagnosticTools', 
                }, 
              ], 
            'desc'=>'Database Analyzer: Tool for analyzing database performance', 
            'id'=>'dbana', 
          }, 
          
        ]
        
        
    }


-------------------------------------------------------------------------------------------



=cut

package BuildPackage::Allpackages;

use Exporter;
use BuildPackage::BuildPackage;
use SDB::Install::DebugUtilities;
use XML::Parser::Expat;

#use strict;

my $packDefs;

@ISA = ('Exporter');

@EXPORT = (
   'readXmlFile',
);

#################################################################################################

my $ATOMIC_CONSTRAINTS_TAGNAME     = 'atomicConstraints';
my $COMPOUND_CONSTRAINTS_TAGNAME   = 'compoundConstraints';
my $TEST_FILE_TAGNAME              = 'testFile';

my $TAG_ID_ATTR                          = 'id';
my $TAG_REFID_ATTR                       = 'refid';
my $TAG_CONSTRAINT_ATTR                  = 'constraint';
my $DEPENDS_ON_TAG_EQUAL_ATTR            = 'eqBuildInfoKey';
my $DEPENDS_ON_TAG_GREATER_OR_EQUAL_ATTR = 'geBuildInfoKey';
my $VALUE_ATTR                           = 'value';


my $PACKAGE_HASH_KEY_ID                          = 'id';

my $DEPENDS_ON_HASH_KEY_PACKAGE_ID               = 'dependsOn';
my $INSTALL_AFTER_HASH_KEY_PACKAGE_ID            = 'installAfter';
my $DEPENDS_ON_HASH_KEY_EQUAL                    = 'eqBuildInfoKey';
my $DEPENDS_ON_HASH_KEY_GREATER_OR_EQUAL         = 'geBuildInfoKey';

#--------------------------------------------------------------------------------------------------

my %tagHandlers = (
    'packDefs' => {
        'start' => undef,
        'end'   => \&packDefsTagEndHandler,
        'text'  => undef
    },

#- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

    $ATOMIC_CONSTRAINTS_TAGNAME => {
        'start' => undef,
        'end'   => undef,
        'text'  => undef
    },

    'constraint' => {
        'start' => \&constraintTagStartHandler,
        'end'   => \&constraintTagEndHandler,
        'text'  => undef
    },

    'match' => {
        'start' => undef,
        'end'   => \&matchTagEndHandler,
        'text'  => undef
    },

    'matchPattern' => {
        'start' => undef,
        'end'   => \&matchPatternTagEndHandler,
        'text'  => undef
    },

    'envName' => {
        'start' => undef,
        'end'   => \&envNameTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'perlName' => {
        'start' => undef,
        'end'   => \&perlNameTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'installerManifestKey' => {
        'start' => undef,
        'end'   => \&installerManifestKeyTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'expectedValue' => {
        'start' => undef,
        'end'   => \&expectedValueTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'expectedPattern' => {
        'start' => undef,
        'end'   => \&expectedValueTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'envDefined' => {
        'start' => undef,
        'end'   => \&envDefinedTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    $COMPOUND_CONSTRAINTS_TAGNAME => {
        'start' => undef,
        'end'   => undef,
        'text'  => undef
    },

    'not' => {
        'start' => undef,
        'end'   => \&notTagEndHandler,
        'text'  => undef
    },

    'and' => {
        'start' => \&andTagStartHandler,
        'end'   => \&andTagEndHandler,
        'text'  => undef
    },

    'or' => {
        'start' => \&orTagStartHandler,
        'end'   => \&orTagEndHandler,
        'text'  => undef
    },

#- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

    'packages' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'SDKprePackagedArchives' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'package' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'name' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'subDir' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'buildSubDir' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'archive' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'noArchive' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'kitSubDir' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'debugPackage' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'interfaceVersion' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'refuseSkip' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'packageSymlinks' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'softwareVersion' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'byManifest' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'byJarManifest' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'jarSrcPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'manifestSrcPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'versionKey' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'buildKey' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'desc' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'distribution' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'dependencies' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'dependsOn' => {
        'start' => undef,
        'end'   => \&dependsOnTagEndHandler,
        'text'  => undef
    },

    'installAfter' => {
        'start' => undef,
        'end'   => \&installAfterTagEndHandler,
        'text'  => undef
    },

    'provides' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'installer' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'isGlobal' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'DEPRECATED' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'isClientPackage' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'suppressSharedFileCheck' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'retainSharedFiles' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'files' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'div' => {
        'start' => undef,
        'end'   => \&divTagEndHandler,
        'text'  => undef
    },

    'flavors' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'flavor' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'additionalInstallerFiles' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'file' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'subTree' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'includeFileNamePattern' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'excludeFileNamePattern' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'includeOnlyWhenExtensionPatternExists' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'forceExtractFileNamePattern' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'permissions' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'script' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'path' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'srcPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'srcPathTemplate' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'envVar' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&envVarSubstitution
    },

    'shlibExt' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&shlibExtSubstitution
    },

    'progExt' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&progExtSubstitution
    },

    'debugSymbolsExt' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&debugSymbolsExtSubstitution
    },

    'libDir' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&libDirSubstitution
    },

    'shellscriptExt' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&shellscriptExtSubstitution
    },

    'batchFileExt' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&batchFileExtSubstitution
    },

    'pyshlibExt' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&pyshlibExtSubstitution
    },

    'systemLibDir' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&systemLibDirSubstitution
    },

    'dataPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler,
        'substitution' => \&dataPathSubstitution
    },

    'pointsTo' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'suppressChecksumGeneration' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'installerGui' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },
    
    'installationVariant' => {
        'start' => undef,
        'end'   => \&hashSubTagEndHandler,
        'text'  => undef
    },

    'manifestRoot' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'virtualManifestRoot' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'componentManifestPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },
    
    'byComponentManifest' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'packagesDir' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'packagesDirPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },
    
    'sdkLibSrcPath' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },
    
    'customInstallerFiles' => {
        'start' => undef,
        'end'   => \&arraySubTagEndHandler,
        'text'  => undef
    },

    'stripBinaries' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'strippedBinarySuffix' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    },

    'noMetadata' => {
        'start' => undef,
        'end'   => \&simpleSubTagEndHandler,
        'text'  => \&cumulativeTextHandler
    }

);

#################################################################################################

=functionComment

This tag text handler appends text encountered in the current tag to
the given text buffer. Heading or trailing whitespace including line breaks
is trimmed.

=cut
sub cumulativeTextHandler {
    my (undef, undef, undef, undef, undef, $textBufferRef, $text) = @_;
    $text =~ s/^(\s|\n|\r)*//;
    $text =~ s/(\s|\n|\r)*$//;
    $$textBufferRef .= $text;
    return 0;
}

#--------------------------------------------------------------------------------------------------
=functionComment

The div tag works only as a subtag of array tags, e.g. <files>.
This must be reflected in the xml schema.

=cut
sub divTagEndHandler {
    my ($expatParser, $type, undef, undef, $currentArray, undef, undef) = @_;
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $lastArray = peek($currArrayStack);
    push(@$lastArray, @$currentArray);
    return 0;
}

#--------------------------------------------------------------------------------------------------

=functionComment

Some tags just contain text or tags which evaluate to text themselves
(i.e. don't give rise to a real data structure like a hash or an array).
We call such tags simple tags.
A simple tag may be contained in either a hash tag, an array tag, or another
simple tag. Since the type (i.e. simple, array, or hash) of the enclosing tag
might not be the same over all occurrences of this tag, we push it onto
the array, put it into the hash, and append it to the text buffer which are
associated with the enclosing tag.
Optionally, if a substitution is defined, the text content of this tag
is modified accordingly.

=cut
sub simpleSubTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    if(defined $tagHandlers{$type}->{'substitution'}) {
        $$textBufferRef = &{$tagHandlers{$type}->{'substitution'}}($expatParser, $$textBufferRef);
    }
    my $txtBufStack = $expatParser->{'buildPackageUserData'}->{'textBufferStack'};
    my $lasttxtBuf = peekRef($txtBufStack);
    $$lasttxtBuf .= $$textBufferRef;
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $lastArray = peek($currArrayStack);
    push(@$lastArray, $$textBufferRef);
    my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
    my $lastHash = peek($currHashStack);
    $lastHash->{$type} = $$textBufferRef;
    return 0;
}

#--------------------------------------------------------------------------------------------------

=functionComment

Some tags model ordered lists. These will be translated to perl arrays.
We call such tags array tags.
An array tag may be contained in either a hash tag, or an array tag.
Since the type (i.e. array, or hash) of the enclosing tag
might not be the same over all occurrences of this tag, we push it onto
the array, and put it into the hash which are
associated with the enclosing tag. 

=cut
sub arraySubTagEndHandler {
    my ($expatParser, $type, undef, undef, $currentArray, undef, undef) = @_;
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $lastArray = peek($currArrayStack);
    push(@$lastArray, $currentArray);
    my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
    my $lastHash = peek($currHashStack);
    $lastHash->{$type} = $currentArray;
    return 0;
}

#--------------------------------------------------------------------------------------------------

=functionComment

Some tags model keyed collections. These will be translated to perl hashes
with the respective sub tag name as keys.
We call such tags hash tags.
A hash tag may be contained in either a hash tag, or an array tag.
Since the type (i.e. array, or hash) of the enclosing tag
might not be the same over all occurrences of this tag, we push it onto
the array, and put it into the hash which are
associated with the enclosing tag.
The attributes (seen as a list of key-value pairs) of a hash tag are
always added to the corresponding perl hash, except the 'constraint'
attribute.
The name of the hash tag currently processed is added to the hash
with key 'type'.

=cut
sub hashSubTagEndHandler {
    my ($expatParser, $type, $attributes, $currentHash, undef, undef, undef) = @_;
    $currentHash->{'type'} = $type;
    my %hash = (%$attributes, %$currentHash);
    delete $hash{'constraint'};
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $lastArray = peek($currArrayStack);
    push(@$lastArray, \%hash);
    my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
    my $lastHash = peek($currHashStack);
    $lastHash->{$type} = \%hash;
    return 0;
}

#--------------------------------------------------------------------------------------------------

## using the s/// 'operator' dynamically, i.e. with source and target
## regexps provided by scalar variables is certainly possible,
## but it is prohibitively time consuming to figure out
## the details. Sorry, Larry, but why didn't you keep it simple ?
##
## For now, we use functions containing hard coded substitiutions
## as parameters to 'textSubstitutionTagEndHandler' instead of
## the regexps themselves.

sub shlibExtSubstitution {
    my ($expatParser, $text) = @_;
    my $constraintsRef = $expatParser->{'buildPackageUserData'}->{'constraints'};
    if($constraintsRef->{'win'}) {
        $text .= '.dll';
    }
    elsif($constraintsRef->{'macosx'}) {
        $text .= '.dylib';
    }
    else {
        $text .= '.so';
    }
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub progExtSubstitution {
    my ($expatParser, $text) = @_;
    my $constraintsRef = $expatParser->{'buildPackageUserData'}->{'constraints'};
    if($constraintsRef->{'win'}) {
        $text .= '.exe';
    }
    else {
        # do nothing.
    }
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub debugSymbolsExtSubstitution {
    my ($expatParser, $text) = @_;
    $text .= '.pdb';
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub libDirSubstitution {
    my ($expatParser, $text) = @_;
    my $constraintsRef = $expatParser->{'buildPackageUserData'}->{'constraints'};
    if($constraintsRef->{'win'}) {
        $text = 'pgm/'.$text;
    }
    elsif ($constraintsRef->{'haveLibDirAsLibLib64SO'}){
        $text = 'lib/lib64/'.$text;
    }
    else {
        $text = 'lib/'.$text;
    }
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub shellscriptExtSubstitution {
    my ($expatParser, $text) = @_;
    my $constraintsRef = $expatParser->{'buildPackageUserData'}->{'constraints'};
    if($constraintsRef->{'win'}) {
        $text .= '.cmd';
    }
    else {
        $text .= '.sh';
    }
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub batchFileExtSubstitution {
    my ($expatParser, $text) = @_;
    my $constraintsRef = $expatParser->{'buildPackageUserData'}->{'constraints'};
    if($constraintsRef->{'win'}) {
        $text .= '.bat';
    }
    else {
        $text .= '.sh';
    }
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub pyshlibExtSubstitution {
    my ($expatParser, $text) = @_;
    my $constraintsRef = $expatParser->{'buildPackageUserData'}->{'constraints'};
    if($constraintsRef->{'win'}) {
        $text .= '.pyd';
    }
    else {
        $text .= '.so';
    }
    return $text;
}

#--------------------------------------------------------------------------------------------------

sub envVarSubstitution {
    my ($expatParser, $text) = @_;
    if(exists($Vars->{'MakeEnv'}->{$text})) {
        $text = $Vars->{'MakeEnv'}->{$text};
    }
    else {
        my $line = $expatParser->current_line();
        my $errmsg = "Environment variable \"".$text."\" not defined at line ".$line.".\n";
        err($errmsg);
    }
    return $text;
}

#################################################################################################

sub constraintTagStartHandler {
    my ($expatParser, $type, $attributes, undef, undef, undef, undef) = @_;
    my $line = $expatParser->current_line();
    if(insideTag($expatParser, $ATOMIC_CONSTRAINTS_TAGNAME)) {
        # Constraint tags either define or reference constraint conditions.
        # In the former case, they have a single attribute "id", in the latter case,
        # their single attribute is "refid":
        if(exists($attributes->{$TAG_ID_ATTR})) {
            my $constraintId = $attributes->{$TAG_ID_ATTR};
            $expatParser->{'buildPackageUserData'}->{'currentScalarStack'} = [];
            $expatParser->{'buildPackageUserData'}->{'operandCountStack'}   = [0];
        }
        else {
            my $errmsg = "Missing \"".$TAG_ID_ATTR."\" attribute in \"".
            $type."\" tag in \"".$expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
            err($errmsg);
        }
    }
    elsif(insideTag($expatParser, $COMPOUND_CONSTRAINTS_TAGNAME)) {
        if(exists($attributes->{$TAG_ID_ATTR})) {
            my $constraintId = $attributes->{$TAG_ID_ATTR};
            $expatParser->{'buildPackageUserData'}->{'currentScalarStack'} = [];
            $expatParser->{'buildPackageUserData'}->{'operandCountStack'}   = [0];
        }
        elsif(exists($attributes->{$TAG_REFID_ATTR})) {
            # we do nothing here.
        }
        else {
            my $errmsg = "Missing either \"".$TAG_ID_ATTR."\" or \"".$TAG_REFID_ATTR.
                         "\" attribute in \"".$type."\" tag in \"".$expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
            err($errmsg);
        }
    }
    else {
        my $errmsg = "Misplaced \"".$type."\": start tag in \"".$expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub constraintTagEndHandler {
    my ($expatParser, $type, $attributes, undef, undef, undef, undef) = @_;
    my $line = $expatParser->current_line();
    if(exists($attributes->{$TAG_ID_ATTR})) {
        if(scalar(@{$expatParser->{'buildPackageUserData'}->{'currentScalarStack'}}) == 1) {
            my $constraintId = $attributes->{$TAG_ID_ATTR};
            $expatParser->{'buildPackageUserData'}->{'constraints'}->{$constraintId} = 
                $expatParser->{'buildPackageUserData'}->{'currentScalarStack'}->[0];
        }
        else {
            my $errmsg = "Unexpected internal parse error while parsing \"".
                      $expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
            err($errmsg);
        }
    }
    elsif(exists($attributes->{$TAG_REFID_ATTR})) {
        my $constraintRefId = $attributes->{$TAG_REFID_ATTR};
        my $constraintValue = $expatParser->{'buildPackageUserData'}->{'constraints'}->{$constraintRefId};
        my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
        push(@$scalarStack , $constraintValue);
        my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
        $cntStack->[scalar(@$cntStack)-1]++;
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub matchTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    my $line = $expatParser->current_line();
    if(scalar(@$scalarStack) >= 2) {
        my $opB = pop(@$scalarStack);
        my $opA = pop(@$scalarStack);
        push(@$scalarStack , $opA eq $opB);
        my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
        $cntStack->[scalar(@$cntStack)-1]++;
    }
    else {
        my $errmsg = "Missing operand(s) in \"".$type."\" tag in \"".
                     $expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub matchPatternTagEndHandler {
    # implements a case insensitive match.
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    my $line = $expatParser->current_line();
    if(scalar(@$scalarStack) >= 2) {
        my $opB = pop(@$scalarStack);
        my $opA = pop(@$scalarStack);
        # regexps are neither data nor code; WTF Larry, WTF?
        my $tt = 0;
        if($opA =~ /$opB/i) {
            $tt = 1;
        }
        push(@$scalarStack , $tt);
        my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
        $cntStack->[scalar(@$cntStack)-1]++;
    }
    else {
        my $errmsg = "Missing operand(s) in \"".$type."\" tag in \"".
                     $expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub envNameTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    my $buf = $$textBufferRef;
    $buf =~ s/^\s*(.*)\s*$/$1/;
    my $estack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    push(@$estack ,($Vars->{'MakeEnv'}->{$buf}));
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub installerManifestKeyTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    my $buf = $$textBufferRef;
    $buf =~ s/^\s*(.*)\s*$/$1/;
    my $estack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    push(@$estack ,($Vars->{'INSTALLER_MANIFEST_OBJ'}->getValue($buf)));
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub perlNameTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    my $buf = $$textBufferRef;
    $buf =~ s/^\s*(.*)\s*$/$1/;
    my $estack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    push(@$estack ,(eval($buf))); ## security: sanitize input before exposing this to end users.
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub expectedValueTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    my $buf = $$textBufferRef;
    my $estack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    push(@$estack ,($buf));
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub expectedPatternTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    my $buf = $$textBufferRef;
    my $estack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    push(@$estack ,($buf));
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub envDefinedTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, $textBufferRef, undef) = @_;
    my $buf = $$textBufferRef;
    $buf =~ s/^\s*(.*)\s*$/$1/;
    my $estack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    push(@$estack ,(exists($Vars->{'MakeEnv'}->{$buf})));
    my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
    $cntStack->[scalar(@$cntStack)-1]++;
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub notTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    my $line = $expatParser->current_line();
    if(scalar(@$scalarStack) >= 1) {
        my $op = pop(@$scalarStack);
        push(@$scalarStack , not $op);
    }
    else {
        my $errmsg = "Missing operand in \"".$type."\" tag in \"".
                     $expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub andTagStartHandler {
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    push(@{$expatParser->{'buildPackageUserData'}->{'operandCountStack'}}, (0));
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub andTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
    my $line = $expatParser->current_line();
    my $opNum = pop(@$cntStack);
    my $result = 1;
    if(scalar(@$scalarStack) >= $opNum) {
        for(my $i = 0; $i < $opNum; $i++) {
            my $operand = pop(@$scalarStack);
            $result &&= $operand;
        }
        push(@$scalarStack , $result);
        $cntStack->[scalar(@$cntStack)-1]++;
    }
    else {
        my $errmsg = "Missing operand(s) in \"".$type."\" tag in \"".
                     $expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub orTagStartHandler {
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    push(@{$expatParser->{'buildPackageUserData'}->{'operandCountStack'}}, (0));
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub orTagEndHandler {
    my ($expatParser, $type, undef, undef, undef, undef, undef) = @_;
    my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
    my $line = $expatParser->current_line();
    my $opNum = pop(@$cntStack);
    my $result = 0;
    if(scalar(@$scalarStack) >= $opNum) {
        for(my $i = 0; $i < $opNum; $i++) {
            my $operand = pop(@$scalarStack);
            $result ||= $operand;
        }
        push(@$scalarStack , $result);
        $cntStack->[scalar(@$cntStack)-1]++;
    }
    else {
        my $errmsg = "Missing operand(s) in \"".$type."\" tag in \"".
                     $expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub packDefsTagEndHandler {
    my ($expatParser, $type, undef, $currentHash, undef, undef, undef) = @_;
    $packDefs = $currentHash;
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub dependsOnTagEndHandler {
    my ($expatParser, $type, $attributes, undef, undef, undef, undef) = @_;
    my $packageId = $attributes->{$TAG_REFID_ATTR};
    my $currentHashRef = { $DEPENDS_ON_HASH_KEY_PACKAGE_ID => $packageId };
    if(exists($attributes->{$DEPENDS_ON_TAG_EQUAL_ATTR})) {
        $currentHashRef->{$DEPENDS_ON_HASH_KEY_EQUAL} = $attributes->{$DEPENDS_ON_TAG_EQUAL_ATTR};
    }
    if(exists($attributes->{$DEPENDS_ON_TAG_GREATER_OR_EQUAL_ATTR})) {
        $currentHashRef->{$DEPENDS_ON_HASH_KEY_GREATER_OR_EQUAL} = $attributes->{$DEPENDS_ON_TAG_GREATER_OR_EQUAL_ATTR};
    }
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $lastArray = peek($currArrayStack);
    push(@$lastArray, $currentHashRef);
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub installAfterTagEndHandler {
    my ($expatParser, $type, $attributes, undef, undef, undef, undef) = @_;
    my $packageId = $attributes->{$TAG_REFID_ATTR};
    my $currentHashRef = { $INSTALL_AFTER_HASH_KEY_PACKAGE_ID => $packageId };
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $lastArray = peek($currArrayStack);
    push(@$lastArray, $currentHashRef);
    return 0;
}

#--------------------------------------------------------------------------------------------------

sub testFilesTagEndHandler {
    my ($expatParser, $type, $attributes, undef, $currentArray, undef, undef) = @_;
    if($currentArray) {
        my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
        my $lastHash = peek($currHashStack);
        $lastHash->{$TEST_FILE_TAGNAME} = @$currentArray[0];
    }
    return 0;
}

#################################################################################################
### EXPORTS

sub readXmlFile{
    my ($filename) = @_;
    my $expatParser = createPackageDescriptionParser();
    # 'buildpackageUserData' is not an expat-specific field, it contains only 
    # data used by this module (Allpackages.pm): 
    $expatParser->{'buildPackageUserData'} = {
        'openTags'              => {},
        'currentScalarStack'    => [],
        'operandCountStack'     => [],
        'constraints'           => {},
        'attributesStack'       => [],
        'textBufferStack'       => [],
        'currentArrayStack'     => [],
        'currentHashStack'      => [],
        'filename'              => $filename
    };
    
    _parseXMLFile($expatParser, $filename);
    
    $expatParser->{'buildPackageUserData'} = undef;
    $expatParser->release();
        #################################
     #printExportStructures(5);
     #die("end of readXmlFile\n");
        #################################
    return $packDefs;
}

### END OF EXPORTS
#################################################################################################
### We are an Exporter:

sub setErrMsgHandler{
    my ($handler) = @_;
    *err = $handler;
}

#--------------------------------------------------------------------------------------------------

sub setMsgHandler{
    my ($handler) = @_;
    *msg = $handler;
}

#################################################################################################

sub createPackageDescriptionParser {
    my $expatParser = new XML::Parser::Expat;
    $expatParser->setHandlers (
            'Start'  => \&handleStart,
            'Char'   => \&handleChar,
            'End'    => \&handleEnd );
    return $expatParser;
}

#--------------------------------------------------------------------------------------------------

sub _parseXMLFile {
    my ($expatParser,$filename) = @_;
    my $rc;
    if($^O=~/mswin/i) {
        $filename =~ s/\//\\/g;
    }
    eval {
        $rc = $expatParser->parsefile($filename);
    };
    if( (!$rc) || $@) {
        msg("Could not parse \"".$filename."\".\n");
        if($@) {
            err("Root cause: ".$@."\n");
        }
    }
}   

#--------------------------------------------------------------------------------------------------

sub handleStart {
    my ($expatParser, $type, %attributes) = @_;
    my $retcode = -1;
    if(!exists($expatParser->{'buildPackageUserData'}->{'openTags'}->{$type})) {
        $expatParser->{'buildPackageUserData'}->{'openTags'}->{$type} = 0;
    }
    $expatParser->{'buildPackageUserData'}->{'openTags'}->{$type}++;
    
    my $attStack = $expatParser->{'buildPackageUserData'}->{'attributesStack'};
    push(@$attStack, \%attributes);
    
    my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
    push(@$currHashStack, {});
    
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    push(@$currArrayStack, []);
    
    my $txtBufStack = $expatParser->{'buildPackageUserData'}->{'textBufferStack'};
    push(@$txtBufStack, "");
    
    $retcode = handleEvent_aux('start', $expatParser, $type, \%attributes, undef, undef, undef, undef);
    return $retcode;
}

#--------------------------------------------------------------------------------------------------

sub handleEnd {
    my ($expatParser, $type) = @_;
    my $retcode = -1;
    
    my $attStack = $expatParser->{'buildPackageUserData'}->{'attributesStack'};
    my $attributes = pop(@$attStack);
    
    my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
    my $currentHash = pop(@$currHashStack);
    
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $currentArray = pop(@$currArrayStack);
    
    my $txtBufStack = $expatParser->{'buildPackageUserData'}->{'textBufferStack'};
    my $textBufferRef = popRef($txtBufStack);
    
    $expatParser->{'buildPackageUserData'}->{'openTags'}->{$type}--;
    $retcode = handleEvent_aux('end', $expatParser, $type, $attributes, $currentHash, $currentArray, $textBufferRef, undef);
    return $retcode;
}

#--------------------------------------------------------------------------------------------------

sub handleChar {
    ($expatParser, $text) = @_;
    my @context = $expatParser->context();
    my $type = $context[$#context];
    my $retcode = -1;
    
    my $attStack = $expatParser->{'buildPackageUserData'}->{'attributesStack'};
    my $attributes = peek($attStack);
    
    my $txtBufStack = $expatParser->{'buildPackageUserData'}->{'textBufferStack'};
    my $textBufferRef = peekRef($txtBufStack);
    
    $retcode = handleEvent_aux('text', $expatParser, $type, $attributes, undef, undef, $textBufferRef, $text);
    return $retcode;
}

#--------------------------------------------------------------------------------------------------

sub handleEvent_aux{
    my $event = shift;
    my ($expatParser, $type, $attributes, $currentHash, $currentArray, $textBufferRef, $text) = @_;
    my $line = $expatParser->current_line();
    my $retcode = -1;
    if(exists($tagHandlers{$type})) {
        my $visibleByConstraint = 1;
        if(exists($attributes->{$TAG_CONSTRAINT_ATTR})) {
            my $constraintName = $attributes->{$TAG_CONSTRAINT_ATTR};
            $visibleByConstraint = $expatParser->{'buildPackageUserData'}->{'constraints'}->{$constraintName};
        }
        if($visibleByConstraint && $tagHandlers{$type}->{$event}) {
            eval {
                $retcode = &{$tagHandlers{$type}->{$event}}($expatParser, $type, $attributes, $currentHash, $currentArray, $textBufferRef, $text);
            };
            if($@) {
                my $errmsg = "Could not call handler for event \"".$event."\" in tag \"".$type.
                             "\" in \"".$expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n".
                             "Root cause: ".$@;
                err($errmsg);
            }
        }
        else {
            $retcode = 0;
        }
    }
    else {
        my $errmsg = "Event \"".$event."\": unrecognized tag \"".$type.
                     "\" in \"".$expatParser->{'buildPackageUserData'}->{'filename'}."\" at line ".$line.".\n";
        err($errmsg);
    }
    return $retcode;
}

#--------------------------------------------------------------------------------------------------

sub insideTag {
    my ($expatParser, $tagname) = @_;
    my $retval = 0;
    if(exists($expatParser->{'buildPackageUserData'}->{'openTags'}->{$tagname})) {
        $retval = $expatParser->{'buildPackageUserData'}->{'openTags'}->{$tagname};
    }
    return $retval;
}

#--------------------------------------------------------------------------------------------------

sub peek {
    my ($stackRef) = @_;
    my $retval = $stackRef->[scalar (@$stackRef) - 1];
    return $retval;
}

#--------------------------------------------------------------------------------------------------

sub peekRef {
    my ($stackRef) = @_;
    my $retval = \($stackRef->[scalar (@$stackRef) - 1]);
    return $retval;
}

#--------------------------------------------------------------------------------------------------

sub popRef {
    my ($stackRef) = @_;
    my $retval = undef;
    if(@$stackRef) {
        $retval = \($stackRef->[scalar (@$stackRef) - 1]);
        pop(@$stackRef);
    }
    return $retval;
}

#################################################################################################
# FOR DIAGNOSTIC OUTPUT:

sub printStacks {
    print('$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$'."\n");
    my ($expatParser, $deep) = @_;
    my $scalarStack = $expatParser->{'buildPackageUserData'}->{'currentScalarStack'};
    my $cntStack = $expatParser->{'buildPackageUserData'}->{'operandCountStack'};
    my $attStack = $expatParser->{'buildPackageUserData'}->{'attributesStack'};
    
    my $openTags = $expatParser->{'buildPackageUserData'}->{'openTags'};
    
    my $constraints = $expatParser->{'buildPackageUserData'}->{'constraints'};

    my $currHashStack = $expatParser->{'buildPackageUserData'}->{'currentHashStack'};
    my $currArrayStack = $expatParser->{'buildPackageUserData'}->{'currentArrayStack'};
    my $txtBufStack = $expatParser->{'buildPackageUserData'}->{'textBufferStack'};
    
    my $line = $expatParser->current_line();
    
    
#    print('%MakeEnv: '."\n");     dumpThings($Vars->{'MakeEnv'}, $deep, 2);             print("\n\n");
    print('$openTags: '."\n");    dumpThings($openTags, $deep, 2);    print("\n\n");
#    print('$textBuffer: '."\n");  dumpThings($textBuffer, $deep, 2);  print("\n\n");
#    print('$scalarStack: '."\n");     dumpThings($scalarStack, $deep, 2);     print("\n\n");
#    print('$cntStack: '."\n");    dumpThings($cntStack, $deep, 2);    print("\n\n");
#    print('$attStack: '."\n");    dumpThings($attStack, $deep, 2);    print("\n\n");
    print('$constraints: '."\n"); dumpThings($constraints, $deep, 2); print("\n\n");
    
    print('$currHashStack: '."\n"); dumpThings($currHashStack, $deep, 2); print("\n\n");
    print('$currArrayStack: '."\n"); dumpThings($currArrayStack, $deep, 2); print("\n\n");
    print('$txtBufStack: '."\n"); dumpThings($txtBufStack, $deep, 2); print("\n\n");
    print('$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$'."\n");
}

#--------------------------------------------------------------------------------------------------

sub printExportStructures {
    my ($deep) = @_;
    print('#################################################################################################'."\n");
#    print('#################################################################################################'."\n");
#    print('@allpackages: '."\n");           dumpThings(\@allpackages, $deep, 2);          print("\n\n");
#    print('#################################################################################################'."\n");
#    print('@alltestpackages: '."\n");       dumpThings(\@alltestpackages, $deep, 2);      print("\n\n");
    print('#################################################################################################'."\n");
    print('$packDefs: '."\n");              dumpThings($packDefs, $deep, 2);              print("\n\n");
    print('#################################################################################################'."\n");
    print('#################################################################################################'."\n");
}

1;
