#! /usr/bin/perl
#

package BuildPackage::Package;


use BuildPackage::Tools;
use SAPDB::Install::BuildInfo;
use SAPDB::Install::MD5Sum;
use BuildPackage::BuildPackage;
use BuildPackage::Worker;
use SDB::Install::SysVars;
use SDB::Install::LoadSysInfoExtension;
use SDB::Install::DebugUtilities;
use SDB::Install::Manifest;
use SDB::Install::System qw (exec_program);
use strict;


our $packagedataname = 'PACKAGEDATA';
our $scriptname = 'script.pm';
our $filelistname = 'files.lst';
our $synthDebugExt = '.debug';


our $sysinfo;



#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------

sub new{
    my $self =  bless({},shift);
    if (@_){
        $self->init (@_);
    }
    return $self;
}

#--------------------------------------------------------------------------------------------------


=functionComment

Generates a package-related checksum from the paths, filenames, and checksums
of the individual files in this package. This is done by dumping into
a tempfile and computing its checksum. This package checksum is passed later 
into the package metadata ('PACKAGEDATA') inside an archived package.
It is used by the installer to determine whether two given package archives do not only
have equal versions but are "from the very same build".
Note that for this purpose it is not necessary to separate the debug files from
the package first. 

=cut
sub GenChecksum{
    my ($self) = @_;
    my $buffer = '';
    foreach my $file (sort @{$self->{'in_files'}}) {
    	if(!$file->{'ignore'}) {
	        if($file->{'type'} eq 'file') {
                if (defined $file->{checksum}){
                    $buffer .= ${$file->{checksum}};
                }
	        }
    	}
    }
    $self->{'checksum'} = MD5Str ($buffer);
    return 1; 
}

#--------------------------------------------------------------------------------------------------

sub init{
    my (
        $self,
        $packdata,
        $p_threads,
        $p_use_pigz,
        $p_pigz,
        $p_Vars,
        $p_componentManifestPath
    ) = @_;
    $self->{'THREADS'} = $p_threads;
    $self->{'USE_PIGZ'} = $p_use_pigz;
    $self->{'PIGZ'} = $p_pigz;
    $self->{'VARS'} = $p_Vars;
    
    foreach my $key (keys %$packdata){
        $self->{$key} = $packdata->{$key};
    }

    if ($^O =~ /mswin/i){
        #
        # force user write bit to avoid dos fs attribute readonly
        # on windows
        #
        if (defined $self->{'umask'}){
            $self->{'umask'} = oct ($self->{'umask'}) & (~0200);
        }
        else{
            $self->{'umask'} = 022;
        }
    }
    else{
        if (defined $self->{'umask'}){
            $self->{'umask'} = oct ($self->{'umask'});
        }
        else{
            $self->{'umask'} = 0222;
        }
    }
    if (defined $packdata->{'dependencies'}){
        foreach my $dependency (@{$packdata->{dependencies}}) {
            my $id = $dependency->{'dependsOn'};    
            
            if (defined $id){
                if (defined $self->{require}){
                    $self->{require} .= ",$id";
                }
                else{
                    $self->{require} = "$id";
                }
            }
            elsif (defined $dependency->{'installAfter'}){
                $id = $dependency->{'installAfter'};
                if (defined $self->{'require_conditional'}){
                    $self->{'require_conditional'} .= ",$id";
                }
                else{
                    $self->{'require_conditional'} = "$id";
                }   
            }
        }   
    }
    if (defined $packdata->{'softwareVersion'}){
        if (defined $packdata->{'softwareVersion'}->{'byManifest'}){
            if(!defined $self->GetVersionInfoByManifest 
                    ($packdata->{'softwareVersion'}->{'byManifest'}->{'manifestSrcPath'},
                     $packdata->{'softwareVersion'}->{'byManifest'}->{'versionKey'},
                     $packdata->{'softwareVersion'}->{'byManifest'}->{'buildKey'})) {
                $self->{'last_error'} = 'could not get manifest version: ' . $self->{'last_error'};
                return undef;
            }
        }
        elsif( (defined $packdata->{'softwareVersion'}->{'byComponentManifest'}) && DecodeBoolTag($packdata->{'softwareVersion'}->{'byComponentManifest'}) ) {
            if(!defined $self->GetVersionInfoByManifest($p_componentManifestPath, undef, undef)) {
                $self->{'last_error'} = 'could not get manifest version: ' . $self->{'last_error'};
                return undef;
            }
        }
    }
    $self->{'in_files'} = $self->{'files'};
    delete $self->{'files'};
    my $rc = $self->ExpandSubTrees();
    if(!defined $rc) {
    	return undef;
    }
    $self->CollectFileStats();
    if ($self->{'THREADS'}){
       if (waitForTasks () > 0){
           $self->{'last_error'} = 'there are md5sum errors.';
           $self->deleteStripDir();
           return undef;
       }
    }
    $self->GenChecksum ();
    $self->SeparateDebugPackage();
#    #########################################################
#    if($self->{'id'} eq 'binaries') {
#        if(exists $self->{'files'}) {
#            my @files_keys = keys %{$self->{'files'}};
#            #dumpThings($self->{'files'}, 3, 2);
#            #dumpThings(\@files_keys, 3, 2);
#        }
#        else {
#            print "files does not exist.";
#        }
#        print "\n";
#        if(exists $self->{'debugFiles'}) {
#            my @debugFiles_keys = keys %{$self->{'debugFiles'}};
#            #dumpThings($self->{'debugFiles'}, 3, 2);
#            #dumpThings(\@debugFiles_keys, 3, 2);
#        }
#        else {
#            print "debugFiles does not exist.";
#        }
#        print "\n";
#    }
#    ############################################################
    if(not $self->{'DEPRECATED'}) {
        my $filecount = scalar(keys %{$self->{'files'}});
        if($filecount <= 0) {
           $self->{'last_error'} = 'refusing to create empty package.';
           $self->deleteStripDir();
           return undef;
        }
        if(exists $self->{'debugPackage'}) {
            my $debugFilecount = scalar(keys %{$self->{'debugFiles'}});
            if($debugFilecount <= 0) {
               $self->{'last_error'} = 'refusing to create empty debug package.';
               $self->deleteStripDir();
               return undef;
            }
        }
    }
    $self->GetPackageSize();
}

#--------------------------------------------------------------------------------------------------

our %manifest_cache;

sub GetVersionInfoByManifest{
    my (
       $self,
       $manifest,
       $versionKey,
       $buildKey
    ) = @_;
    
    my @buffer;
    my $filename = $self->{'VARS'}->{'MAKEOUTPDIR'}.'/'.$manifest;

    my $mf;
    if (exists $manifest_cache{$manifest}){
        $mf = $manifest_cache{$manifest};
    }
    else{
        $mf = new SDB::Install::Manifest ($filename);
        if ($mf->errorState ()){
            $self->{'last_error'} = 'could not open manifest: '. $mf->getErrorString ();
            return undef;
        }
        $manifest_cache{$manifest} = $mf;
    }

    my $versionstr = $mf->getVersion ();
    if (!defined $versionstr){
        $versionstr = $mf->getValue ($versionKey);
    }
    if(!defined $versionstr){
        $self->{'last_error'} = 'could not read in version information from manifest '.$filename.'.';
        return undef;
    }
    my $buildstr = $mf->getBuildstring ();
    if (!defined $buildstr){
        $buildstr = $mf->getValue ($buildKey);
    }
    if(!defined $buildstr){
        $self->{'last_error'} = 'could not read in build information from manifest '.$filename.'.';
        return undef;
    }
    $self->{'softwareVersion'} =  $versionstr;
    $self->{'buildString'} = "$buildstr";
    my $gitHash = $mf->getValue ('git-hash');

   if (defined $gitHash && $gitHash){
	$self->{'gitHash'} = $gitHash;
   }

    return 1;
}

#--------------------------------------------------------------------------------------------------

=functionComment

Separates the regular files in the file list of this package from debug files,
which are packed into a separate archive.
Note that arrays 'debugFiles' and 'files' must be sorted by the 'path' component
of their entries. 

=cut
sub SeparateDebugPackage{
    my ($self) = @_;
    my $sortHash = {};
    foreach my $file (@{$self->{'in_files'}}){
    	if(!$file->{'ignore'}) {
	        if(!exists $sortHash->{$file->{'path'}}) {
	            $sortHash->{$file->{'path'}} = $file;
	        }
    	}
    }
    if(exists $self->{'debugPackage'}) {
        $self->{'debugFiles'} = {};
    }
    $self->{'files'} = {};
    $self->{'testFile'} = undef;
    if(DecodeBoolTag($self->{'noArchive'})) {
        if(not defined $self->{'notArchivedKitFiles'}) {
            my $notArchivedKitFiles = [];
            $self->{'notArchivedKitFiles'} = $notArchivedKitFiles;
        }
     }
    foreach my $fileKey (sort keys %{$sortHash}) {
        my $file = $sortHash->{$fileKey};
        if($file->{'type'} eq 'file') {
            if(!defined $self->{'testFile'} && DecodeBoolTag($file->{'testFile'})) {
                $self->{'testFile'} = $file;
            }
            my $fileIt = 1;
            my $isDebugFile = 0;
            if( DecodeBoolTag($file->{'debugFile'}) || DecodeBoolTag($file->{'debugFiles'}) ) {
                $fileIt = 0;
                $isDebugFile = 1;
            }
            if(exists $self->{'debugPackage'}) {
	            if( $isDebugFile ) {
	                $self->{'debugFiles'}->{$fileKey} = $file;
                }
                elsif(defined $file->{'_syntheticDebugEntry'}) {
                    $self->{'debugFiles'}->{$fileKey.$synthDebugExt} = $file->{'_syntheticDebugEntry'};
                }
            }
            else {
                if(DecodeBoolTag($self->{'noArchive'})) {
                    push @{$self->{'notArchivedKitFiles'}}, $file;
                    $fileIt = 0;
                 }
            }
            if($fileIt) {
                $self->{'files'}->{$fileKey} = $file;
            }
        }
    }
    return;
}

#--------------------------------------------------------------------------------------------------

sub GetPackageSize{
    my ($self) = @_;
    my $size = 0;
    my $dbgsize = 0;
    foreach my $fileKey (keys %{$self->{'files'}}) {
        $size += $self->{'files'}->{$fileKey}->{'statbuf'}->[7];
    }
    $self->{'size'} = $size;
    if (defined $self->{'debugFiles'}){
        foreach my $dbgFileKey (keys %{$self->{'debugFiles'}}) {
            $dbgsize += $self->{'debugFiles'}->{$dbgFileKey}->{'statbuf'}->[7];
        }   
    }
    $self->{'debugSize'} = $dbgsize;    
    return $size;
}

#--------------------------------------------------------------------------------------------------

sub CollectFileStats{
    my ($self) = @_;
    my $passed = 1;
    if(DecodeBoolTag($self->{'stripBinaries'})) {
        $self->{'stripDir'} = $self->{'VARS'}->{'MAKEOUTPDIR'}."/stripped_".$self->{'id'};
        if(!$self->deleteStripDir()) {
            return undef;
        }
        eval {
    		if(!makedir ($self->{'stripDir'}, 0777)){
    			$self->{'last_error'} = "could not create directory '".$self->{'stripDir'}."': ".$!; 
    			return undef;
    		}
        };
        if($@){
            $self->{'last_error'} = "could not create directory '".$self->{'stripDir'}."': ".$@; 
            return undef;
        }
        
    }
    foreach my $entry (@{$self->{'in_files'}}){
        if($entry->{'type'} eq 'file') {
            my $rc = $self->ProcessFileFromList($entry);
            $passed = $passed && $rc;
        }
    }
    if (!$passed){
        $self->deleteStripDir();
        return undef;
    }
    return 1;
}

#--------------------------------------------------------------------------------------------------

sub ExpandSubTrees{
    my ($self) = @_;
    my $passed = 1;
    my @expansions = ();
    foreach my $entry (@{$self->{'in_files'}}) {
        if($entry->{'type'} eq 'subTree') {
            my $expansion = $self->ExpandSubTree($entry);
            if(!defined $expansion) {
            	return undef;
            }
            push(@expansions, @{$expansion});
        }
    }
    push(@{$self->{'in_files'}}, @expansions);
    return 1;
}

#--------------------------------------------------------------------------------------------------

sub ExpandSubTree{
    my (
        $self,
        $entry
    ) = @_;
    my $retval = [];
    my $path = $entry->{'path'};
    my $srcPath = $path;
    if(exists $entry->{'srcPath'}) {
       $srcPath = $entry->{'srcPath'};
    }
#    elsif(DecodeBoolTag($entry->{'isSymlink'})) {
#        $srcFile = $entry->{'pointsTo'};
#    }
    my $rawList = {};
    my $absolutePrefix = $self->{'VARS'}->{'MAKEOUTPDIR'};
    if(exists $self->{'buildSubDir'}) {
        $absolutePrefix = $absolutePrefix.'/'.$self->{'buildSubDir'};
    }
    my $sdir = $absolutePrefix.'/'.$srcPath;
    if(not -d $sdir) {
        print "################ $sdir does not exist.\n";
        $self->{'last_error'} = "$sdir does not exist.";
        return undef;
    }
    my $rc = enumFilesRecursively($absolutePrefix,
                                  $srcPath,
                                  $rawList,
                                  0,
                                  $entry->{'includeFileNamePattern'},
                                  $entry->{'excludeFileNamePattern'},
                                  $entry->{'includeOnlyWhenExtensionPatternExists'},
                                  $self->{'in_files'},
                                  $self,
                                  $self->{'id'} eq 'binaries' ? 0 : undef);
    if(!defined $rc) {
        return undef;
    }
    if(0 == scalar keys %{$rawList}) {
        print STDERR "WARNING: empty directory $srcPath referenced by subTree tag.\n";
    }                           
    foreach my $rawEntry (keys %{$rawList}) {
        # use the subTree entry as template to enable inheritance of attributes:
        my %template = %$entry;
        my $properEntry = \%template;
        $properEntry->{'srcPath'} = $rawEntry;
        $rawEntry =~ s/${srcPath}(\\|\/)?(.*)$/$2/;
        $properEntry->{'path'} = $path ? ($path.'/'.$rawEntry) : $rawEntry;
        if(exists $entry->{forceExtractFileNamePattern} &&
         ($properEntry->{'path'} =~ /$entry->{forceExtractFileNamePattern}/)) {
            $properEntry->{'forceExtract'} = 1;
        }
        $properEntry->{'type'} = 'file';
        push(@$retval, $properEntry);
    }
    return $retval;
}
#--------------------------------------------------------------------------------------------------

sub ProcessFileFromList{
    my (
        $self,
        $entry
    ) = @_;
    my $tgtFile = $entry->{'path'};
    my $srcFile = $entry->{'path'};
    if(defined $entry->{'sdkLibSrcPath'}) {
        $srcFile = $entry->{'sdkLibSrcPath'};
    }
    elsif(defined $entry->{'srcPath'}) {
        $srcFile = $entry->{'srcPath'};
    }
    elsif(DecodeBoolTag($entry->{'isSymlink'})) {
        $srcFile = $entry->{'pointsTo'};
    }
    if(exists $self->{'buildSubDir'}) {
        $srcFile = $self->{'buildSubDir'}.'/'.$srcFile;
    }
    my $mode = $entry->{'permissions'};
    my $is_dir = DecodeBoolTag($entry->{'isDirectory'}) || ($srcFile =~ /\/$/);
    my $fullQualFile;
    if(defined $entry->{'sdkLibSrcPath'}) {
        $fullQualFile = $self->{'VARS'}->{'STAGELIBDIR'}.'/'.$srcFile;
    }
    else {
        $fullQualFile = $self->{'VARS'}->{'MAKEOUTPDIR'}.'/'.$srcFile;
    }
    my @statbuf = stat($fullQualFile);
    if(!@statbuf && !$is_dir){
        $self->{'last_error'} .= "\n file $fullQualFile: $!";
        return undef;
    }
    else {
        # 'getRealPathName' and 'normalizePath' don't seem to fully
        # qualify paths, only some "normalization"
        # ('\'->'/', '//'->'/', '(...)/$'->'(...)$') occurs:
        if($^O =~ /mswin/i){
            $tgtFile = getRealPathName($tgtFile, $self->{'VARS'}->{'MAKEOUTPDIR'});
            $srcFile = getRealPathName($srcFile, $self->{'VARS'}->{'MAKEOUTPDIR'});
        }
        else{
            normalizePath(\$tgtFile);
            normalizePath(\$srcFile);
        }
        if(defined $self->{'subDir'}){
            $entry->{'path'} = $self->{'subDir'}.'/'.$tgtFile;
        }
        ##############################################################################################
        my $haveSyntheticDebugFile = 0;
        if(DecodeBoolTag($self->{'stripBinaries'}) && !$is_dir) {
            my $stripIt = $self->mustStrip($fullQualFile);
            if(not defined $stripIt) {
                return undef;
            }
            if($stripIt) {
                my $target = $self->{'stripDir'}.'/'.$srcFile;
                if(!BuildPackage::Tools::copy($fullQualFile, $target, {'nochown' => 1,'binmode' => 1,'createdir' => 1,'dir_perm' => 0755, 'mode' => 0755})) {
                    $self->{'last_error'} = "Could not copy '$fullQualFile' to '$target' for stripping: $!\n";
                    return undef;
                }
                #######################################
				# objcopy --only-keep-debug libhdbrskernel.so libhdbrskernel.so.debug
				# strip -s libhdbrskernel.so
				# objcopy --add-gnu-debuglink=libhdbrskernel.so.debug libhdbrskernel.so
                ########################################
                my $cfg = {'outLines' => 'yes, please'};
                my $programName = 'objcopy';
                my $rc = exec_program($programName, ['--only-keep-debug', $target, $target.$synthDebugExt], $cfg);
                if((not defined $rc) || $rc != 0) {
                    my $outbuffer = $cfg->{outLines};
                    my $concOutNl = '';
                    foreach my $line (@$outbuffer) {
                        $concOutNl = "$concOutNl$line\n";
                    }
                    my $errmsg = "Could not run program '$programName' on file '$target': $!\n$concOutNl";
                    $self->{'last_error'} = $errmsg;
                    return undef;
                }
                #######################################
                $cfg = {'outLines' => 'yes, please'};
                $programName = 'strip';
                $rc = exec_program($programName, ['-s', $target], $cfg);
                if((not defined $rc) || $rc != 0) {
                    my $outbuffer = $cfg->{outLines};
                    my $concOutNl = '';
                    foreach my $line (@$outbuffer) {
                        $concOutNl = "$concOutNl$line\n";
                    }
                    my $errmsg = "Could not run program '$programName' on file '$target': $!\n$concOutNl";
                    $self->{'last_error'} = $errmsg;
                    return undef;
                }
                ########################################
                $cfg = {'outLines' => 'yes, please'};
                $programName = 'objcopy';
                $rc = exec_program($programName, ['--add-gnu-debuglink='.$target.$synthDebugExt, $target], $cfg);
                if((not defined $rc) || $rc != 0) {
                    my $outbuffer = $cfg->{outLines};
                    my $concOutNl = '';
                    foreach my $line (@$outbuffer) {
                        $concOutNl = "$concOutNl$line\n";
                    }
                    my $errmsg = "Could not run program '$programName' on file '$target': $!\n$concOutNl";
                    $self->{'last_error'} = $errmsg;
                    return undef;
                }
                #######################################
                $haveSyntheticDebugFile = 1;
                $fullQualFile = $target;
            }
        }
        ##############################################################################################
        elsif(defined $self->{'strippedBinarySuffix'}) {
            my $strippedBin = $fullQualFile.$self->{'strippedBinarySuffix'};
            if(-f $strippedBin) {
                $fullQualFile = $strippedBin;
            }
        }
        ##############################################################################################
        $entry->{'srcPath'} = $srcFile;               # these two paths may differ in more than just
        $entry->{'absoluteSrcPath'} = $fullQualFile;  # a prefix, e.g. for a stripped binary.
        if($is_dir){
            if(!defined $self->{'dirs'}){
                $self->{'dirs'} = {};
            }
            $self->{'dirs'}->{$tgtFile}->{'mode'} = oct($mode);
            if(defined $mode){
                $entry->{mode} = oct($mode);
            }
        }
        else {
            if(defined $mode){
                $entry->{mode} = oct($mode);
            }
            else{
                $mode = $statbuf[2] & 07777;
                if (defined $self->{umask}){
                    $mode &= ~($self->{umask});
                }
                $entry->{mode} = $mode;
            }
            
            if(!DecodeBoolTag($self->{'suppressChecksumGeneration'})) {
              my $checksum;
              if ($self->{'THREADS'}){
                  pushTask ([0,$fullQualFile,\$checksum]);
              }
              else{
                  $checksum = MD5Sum ($fullQualFile);
              }
              $entry->{checksum} = \$checksum;
            }
            else {
                $entry->{'checksum'} = \"00000000000000000000000000000000";
            }
            $entry->{'statbuf'} = \@statbuf;
        }
        if($haveSyntheticDebugFile) {
            my %debugEntry = %{$entry};
            $debugEntry{'absoluteSrcPath'} = $entry->{'absoluteSrcPath'}.$synthDebugExt;
            $debugEntry{'debugFile'} = 'true';
            $entry->{'_syntheticDebugEntry'} =  \%debugEntry;
        }
    }
    return 1;
}  
        
#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------


sub asString{
    my ($self) = @_;
    my $string = $self->{name};
    if ($self->{type} eq 'flavor' && $self->{kitSubDir}){
        $string .= ' for ' . $self->{kitSubDir};
    }
    return $string;
}


sub CreatePackage {
    my (
        $self,
        $debug
    ) = @_;
    if(DecodeBoolTag($self->{'noArchive'})) {
        print "not creating archive: 'noArchive' is set for package '$self->{'id'}'.\n";
        if(defined $self->{'debugPackage'}) {
            print "not creating debug archive: 'noArchive' is set for package '$self->{'id'}'.\n";
        }
    }
    else {
        if(defined $debug && !defined $self->{'debugPackage'}){
            return 1;
        }
        my $rc;
        if(defined $debug) {
            $rc = $self->CreateArchiveBRLess($self->{'debugPackage'}->{'archive'}, 1);
        }
        else {
            $rc = $self->CreateArchiveBRLess();
        }
        if(!$rc) {
            $self->{'last_error'} = "...could not create $debug archive: ".$self->{'last_error'};
            return undef;
        }
        elsif (!$self->{'THREADS'}){
            print "...OK, $debug archive created.\n";
        }
        if(not defined $debug) {
            $self->CreatePackage('debug');
        }
    }
    return 1;
}


#--------------------------------------------------------------------------------------------------



sub getPackageData{
    my ($self) = @_;
    my $pdString = 'INTERFACE_VERSION = "' . $self->{'interfaceVersion'} . "\"\n";
    $pdString .= 'ID = "' . $self->{'id'} . "\"\n";
    $pdString .= 'NAME = "' . $self->{'name'} . "\"\n";

    $pdString .= "VERSION = \"".$self->{'softwareVersion'}."\"\n";
    $pdString .= "BUILD_STRING = \"".$self->{'buildString'}."\"\n";

    if (defined $self->{'gitHash'}){
        $pdString .= 'GIT_HASH = "'. $self->{'gitHash'} . "\"\n";
    }

    if (defined $self->{require}){
        $pdString .= 'REQUIRE = "' . $self->{require} . "\"\n";
    }
    
    if (defined $self->{'require_conditional'}){
        $pdString .= 'REQUIRE_CONDITIONAL = "' . $self->{'require_conditional'} . "\"\n";
    }
    
    if (defined $self->{'refuseSkip'}){
        my $val = DecodeBoolTag($self->{'refuseSkip'});
        $pdString .= 'REFUSE_SKIP = "' . $val . "\"\n";
    }
    
    if (defined $self->{'testFile'}){
        my $testfilePath = $self->{'testFile'}->{'path'};
        unless (-f "$self->{'VARS'}->{'MAKEOUTPDIR'}/$testfilePath"){
            $self->{'last_error'} = 'testfile ' . "$testfilePath: $!";
        }
        unless (defined $sysinfo){
            $sysinfo = SAPDB::Install::SysInfo::GetSystemInfo();
        }
        if ($^O =~ /mswin/i){
            $testfilePath = getRealPathName($testfilePath,$self->{'VARS'}->{'MAKEOUTPDIR'});
        }
        else{
            normalizePath(\$testfilePath);
        }   
        $pdString .= 'TEST_FILE = "' .$testfilePath . "\"\n";
        $testfilePath = "$self->{'VARS'}->{'MAKEOUTPDIR'}/$testfilePath";
        if ($^O =~ /mswin/i){
            $testfilePath =~ s/\//\\/g;
        }
        my $rc = GetMagicString($testfilePath);
        if (defined $rc){
            $pdString .= 'TEST_FILE_MAGIC = "' . "$rc\"\n";
        }
        else{
            $self->{'last_error'} = "could not get file magic string of ".$testfilePath;
            return undef;
        }
        foreach my $key (sort keys %$sysinfo){
            my $sysinfokey = $key;
            $sysinfokey =~ s/\s/_/;
            $sysinfokey =~ tr/[a-z]/[A-Z]/;
            $pdString .= "SYSINFO.$sysinfokey = \""  . $sysinfo->{$key} . "\"\n";
        }
    }
    
    if (defined $self->{'debugFiles'} && %{$self->{'debugFiles'}} && !DecodeBoolTag($self->{'debugPackage'}->{'noMetadata'})){
        $pdString .= 'DEBUG_ARCHIVE = "' . $self->{'debugPackage'}->{'archive'} . "\"\n";
    }
    
    if (defined $self->{'distribution'}){
        $pdString .= 'DISTRIBUTION = "'. $self->{'distribution'} . "\"\n";
    }

    $pdString .= 'IS_GLOBAL = "'. DecodeBoolTag($self->{'isGlobal'}) . "\"\n";
    $pdString .= 'IS_CLIENT_PACKAGE = "'. DecodeBoolTag($self->{'isClientPackage'}) . "\"\n";
    $pdString .= 'DESC = "' . $self->{'desc'} . "\"\n";
    $pdString .= 'SIZE = "' . $self->{'size'} . "\"\n";
    $pdString .= 'CHECKSUM = "' . $self->{'checksum'} . "\"\n";
    return \$pdString;
}

#--------------------------------------------------------------------------------------------------

sub GeneratePackageData{
    my ($self,$broot) = @_;
    unless (open (PD,'>'.$broot.'/'.$packagedataname)){
        $self->{'last_error'} = "could not create PACKAGEDATA file '$broot/$packagedataname': ".$!;
        return undef;
    }
    print PD ${$self->getPackageData()};
    close (PD);
}


#--------------------------------------------------------------------------------------------------

sub getDebugPackageData{
    my ($self) = @_;
    my $pdString = 'ID = "' . $self->{'id'} . "\"\n";
    $pdString .= 'NAME = "' . $self->{'name'} . "\"\n";
    $pdString .= 'CHECKSUM = "' . $self->{'checksum'} . "\"\n";
    $pdString .= 'IS_GLOBAL = "'. DecodeBoolTag($self->{'isGlobal'}) . "\"\n";
    $pdString .= 'IS_CLIENT_PACKAGE = "'. DecodeBoolTag($self->{'isClientPackage'}) . "\"\n";
    $pdString .= 'BUILD_STRING = "' . $self->{'buildString'}.
                    "\"\n";
    if (defined $self->{'gitHash'}){
        $pdString .= 'GIT_HASH = "'. $self->{'gitHash'} . "\"\n";
    }
    $pdString .= 'SIZE = "' . $self->{'debugSize'} . "\"\n";
    return \$pdString;
}

#--------------------------------------------------------------------------------------------------

sub GenerateDebugPackageData{
    my ($self, $broot) = @_;
    
    unless (open (PD,'>'.$broot.'/'.$packagedataname)){
        $self->{'last_error'} = 'could not create PACKAGEDATA file: '.$!;
        return undef;
    }
    print PD ${$self->getDebugPackageData()};
    close (PD);
}

#--------------------------------------------------------------------------------------------------


sub getFileList{
    my ($self, $list) = @_;
    if (!defined $list){
        $list = $self->{'files'};
    }
    my $MODE_BIT_LEN = 12;                # no of bits for file permissions
    my $MODE_SHIFT  = 32 - $MODE_BIT_LEN; # no of shifts needed to shift file permissions to 'left' end.
    my $listString = '';
    my $isDir;
    foreach my $fileKey (keys %{$list}){
        my $file = $list->{$fileKey};
        $isDir = DecodeBoolTag($file->{'isDirectory'}) || ($fileKey =~ /\/$/);

        my $mode = $file->{mode};
        $mode <<= $MODE_SHIFT;
        my $flags = 0;
        my $i = 0;      
        my $fileopts = $file;

        foreach my $flag (
            $fileopts->{'forceExtract'},        # unused (as of june2010), a potential file list XML tag/attribute
            $fileopts->{'installTimeTemplate'},
            $fileopts->{'setRootAsOwner'},
            $fileopts->{'specialFile'},
            $fileopts->{'isSymlink'}
        ){
            unless ($flag){
                $i++;
                next;
            }
            $flags |= 1 << $i++; 
        }
        $flags <<= ($MODE_SHIFT - $i);
        $mode |= $flags;
        $fileKey =~ s/\/{2,}/\//g;
        $fileKey =~ s/^\///;
        $listString .= ("\"$fileKey\" ".($isDir ? '' : ${$file->{'checksum'}}).',' .
            sprintf ("%x", $file->{'statbuf'}->[7]) . ','.
            sprintf ("%x", $file->{'statbuf'}->[9]) . ','. 
            sprintf ("%x", $mode)."\n");
    }
    return \$listString;
}


#--------------------------------------------------------------------------------------------------

sub WriteFileList{
    my ($self,$list,$broot) = @_;
    unless (open (LST,">$broot/$filelistname")){
        $self->{'last_error'} = "cannot create file $broot/$filelistname: $!";
        return undef;
    }
    print LST ${$self->getFileList($list)};
    close (LST);
}

#--------------------------------------------------------------------------------------------------

sub CreateArchiveBRLess{
    my ($self, $archive,$isDebug) = @_;

    if ($self->{THREADS}){
        import threads::shared;
    }

    if (!defined $archive){
        $archive = $self->{archive};
    }
    my $kitdirectory = join ($path_separator, $self->{'VARS'}->{'KITDIR'}, $self->{'kitSubDir'});
    if (!-d $kitdirectory){
        if(!makedir ($kitdirectory,0775)) {
            $self->{'last_error'} = "could not create $kitdirectory: " . $!;
            return undef;
        }
    }
    $archive = join ($path_separator, $kitdirectory, $archive);

    my $packageData;
    my $filesLst;
    my $script;
    my $pmask = $self->{umask};
    my @content;
    
    if (defined $self->{THREADS}){
        share (\@content);
    }

    my $files;
    if ($isDebug){
        $packageData = ${$self->getDebugPackageData()};
        $filesLst =  ${$self->getFileList($self->{debugFiles})};
        $files = $self->{debugFiles};
    }
    else{
        $packageData = ${$self->getPackageData()};
        $filesLst = ${$self->getFileList()};
        my $scriptSrcName = $scriptname;
        if(defined $self->{'script'}) {
            $scriptSrcName = $self->{'script'};
        }
        $script = $self->{'VARS'}->{'PACKDEFDIR'} . '/' . $self->{'id'} . '/' . $scriptSrcName;
        if(not -f $script){
            $self->{'last_error'} = "script $script: $!";
            return undef;
        }
        $files = $self->{files};
    }
    my ($file, $isDir);
    foreach my $fileKey (sort keys %{$files}){
        $file = $files->{$fileKey};
        $isDir = DecodeBoolTag($file->{'isDirectory'}) || ($fileKey =~ /\/$/);
        if(DecodeBoolTag($file->{'isSymlink'})) {
            my $link   = $file->{'path'};
            my $target = $file->{'pointsTo'};
            my $srcTarget = $target;
            if(defined $files->{$target}->{'srcPath'}) {
                $srcTarget = $files->{$target}->{'srcPath'};
            }
            if(defined $self->{'buildSubDir'}) {
                $srcTarget = $self->{'buildSubDir'}.'/'.$srcTarget;
            }
            if(not -f $self->{'VARS'}->{'MAKEOUTPDIR'}.'/'.$srcTarget) {
                my $errMsg = "refusing to package dangling symlink:\n".
                             $file->{'path'}.' -> '.$target.".\n".
                             'makeroot: '.$self->{'VARS'}->{'MAKEOUTPDIR'}.'/'.$srcTarget.' does not exist.';
                $self->{'last_error'} = $errMsg;
                return undef;
            }
            my @args;
            if (defined $self->{THREADS}){
                share (\@args);
            }
            @args = (1,$link, $target);
            push @content, \@args;
        }
        elsif (!$isDir) {
            my $from = $file->{'absoluteSrcPath'};
            my $mode = $file->{'mode'};
            my @args;
            if (defined $self->{THREADS}){
                share (\@args);
            }
            my $replaceSymlinksWithTargets = DecodeBoolTag($self->{'packageSymlinks'}) ? 0 : 1;
            if(!$replaceSymlinksWithTargets && $^O !~ /mswin/i && -l $from) {
                if($self->{'id'} eq 'newdbstudioDirector') {
                    # special case with long (>100 byte) paths, handle it differently:
                    # TODO: get rid of the special treatment for this package
                    @args = (0,$from, $fileKey,0, defined $mode ? ($mode) : ());
                }
                else {
                    my $target = readlink ($from);
                    @args = (1,$fileKey,$target);
                }
            }
            else {
                @args = (0,$from, $fileKey,1, defined $mode ? ($mode) : ());
            }
            push @content, \@args;
        }
    }
    if($isDebug && DecodeBoolTag($self->{'debugPackage'}->{'noMetadata'})) {
        $packageData = undef;
        $filesLst = undef;
    }
    if ($self->{THREADS}){
        pushTask ([5,
                   $archive,                                            #  1
                   $packageData,                                        #  2
                   $filesLst,                                           #  3
                   $script,                                             #  4
                   \@content,                                           #  5
                   $pmask,                                              #  6
                   $self->{'PIGZ'},                                     #  7
                   $self->{'USE_PIGZ'},                                 #  8
                   $self->{'VARS'}->{'BP_PIGZ_RSYNCABLE'},              #  9
                   $self->asString()]                                   # 10
        );
    }
    else{
        if (!defined createInstallerPackage (
                   $archive,                                            #  1
                   $packageData,                                        #  2
                   $filesLst,                                           #  3
                   $script,                                             #  4
                   \@content,                                           #  5
                   $pmask,                                              #  6
                   $self->{'PIGZ'},                                     #  7
                   $self->{'USE_PIGZ'},                                 #  8
                   $self->{'VARS'}->{'BP_PIGZ_RSYNCABLE'}               #  9
                )){
            return undef;
        }
    }
    return 1;
}




#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------

sub getKitFiles{
    my ($self) = @_;
    my @result;
    if (defined $self->{'additionalInstallerFiles'}){
        foreach my $file (@{$self->{'additionalInstallerFiles'}}){
            push @result, $file;
        }
    }
    if (defined $self->{'notArchivedKitFiles'}){
        foreach my $file (@{$self->{'notArchivedKitFiles'}}){
            push @result, $file;
        }
    }
    return @result;
}

#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------

sub GetFormattedSize{
    my ($self) = @_;
    my $size = $self->{size} + $self->{debugSize}; 
    $size /= 0x100000;
    return sprintf ("%.3f mb",$size);
}

#--------------------------------------------------------------------------------------------------

sub mustStrip {
    my (
        $self,
        $file
    ) = @_;
    if(-d $file || -l $file) {
        return 0;
    }
    if(   $file =~ /lexicon\/lang/
       || $file =~ /.py$/
       || $file =~ /.h$/
       || $file =~ /.c$/
       || $file =~ /.txt$/
       || $file =~ /.dat$/
       || $file =~ /.xml$/
       || $file =~ /.xsd$/
       || $file =~ /.mf$/
       || $file =~ /.jar$/
       || $file =~ /.zip$/
       || $file =~ /.tgz$/
       || $file =~ /.gif$/
       || $file =~ /.debug$/
       || $file =~ /.stripped$/
       || $file =~ /.debug_hde$/
       || $file =~ /.stripped_hde$/
       || $file =~ /.res$/
       || $file =~ /.SAR$/
       || $file =~ /.png$/) {
        return 0;
    }
    my $cfg = {'outLines' => 'yes, please'};
    my $programName = '/usr/bin/file';
    my $rc = exec_program($programName, [$file], $cfg);
    my $outbuffer = $cfg->{'outLines'};
    my $concOutNl = '';
    my $concOut = '';
    foreach my $line (@$outbuffer) {
        $concOutNl = "$concOutNl$line\n";
        $concOut = "$concOut$line ";
    }
    if((not defined $rc) || $rc != 0) {
        my $errmsg = "Could not run program '$programName' on file '$file': returned '$rc': $!\n$concOutNl";
        $self->{'last_error'} = $errmsg;
        return undef;
    }
    if (!($concOut =~ /not stripped/)) {
        print "Strip symbols: not processing '$file': $concOutNl\n";
        return 0;
    }
    return 1;
}

#--------------------------------------------------------------------------------------------------

sub deleteStripDir {
    my (
        $self
    ) = @_;
    if(DecodeBoolTag($self->{'stripBinaries'})) {
        if($self->{'stripDir'} && -d $self->{'stripDir'}) {
            my $rc;
			eval{
				$rc = deltree($self->{'stripDir'});
			};
			if($@){
				print "cleanup of directory '".$self->{'stripDir'}."' failed: ".$@."\n";
				return undef;
			}
			if(!$rc){
				print "cleanup of directory '".$self->{'stripDir'}."' failed.\n";
				return undef;   
			}
        }
    }
    return 1;
}

#--------------------------------------------------------------------------------------------------

1;
