package LCM::ActionsDetector;

use strict;
use LCM::HostsUtility;
use SDB::Install::Installer;
use LCM::NotApplicableActions;
use LCM::App::ApplicationContext;
use SDB::Install::System qw (isAdmin isSidadmin);
use LCM::PersistencyUtils qw (getAllPendingUpdateInfo);
use LCM::Utils::CommonUtils qw(isSingleDB isSystemReplication);
use SDB::Install::Globals qw($gKeynameEngine
                             $gKeynameInstaller
                             $gKeynameSDA
                             $gKeynameLSS
                             $gProductName
                             $gFlavourPlatform
                             $gFlavourExpress
                             getFlavourProductName);

sub new {
    my ($class, $config, $numberOfSystemHosts, $isWin, $isGui) = @_;
    my $self = {};
    bless $self, $class;
    
    $self->{config} = $config;
    $self->{numberOfSystemHosts} = $numberOfSystemHosts;
    $self->{isWin} = $isWin;
    $self->{isGui} = $isGui;
    $self->{isHostPartOfSystem} = LCM::HostsUtility->IsHostPartOfSystem();
    $self->{pendingInfoResident} = $self->_getPendingInfo(1);
    $self->{pendingInfoNonResident} = $self->_getPendingInfo(0);

    $self->createActions();

    $self->{notApplicableActions} = LCM::NotApplicableActions->new();
    return $self;
}

sub _getPendingInfo {
	my ($self, $isResident) = @_;
	my $installer = new SDB::Install::Installer();
	my $sapSys = $installer->getInstallation($self->{config}->getMsgLst());
	my $pendingInfo = '';
	my $updatePendingInfo = getAllPendingUpdateInfo($sapSys, '  ', '', undef, $self->{config}->getMsgLst(), $isResident);

	if (defined $updatePendingInfo && $updatePendingInfo ne "" && !$self->{isGui}) {
		$pendingInfo = "\n" . $updatePendingInfo;
	}

	return $pendingInfo;
}

sub createActions {
    my ($self, ) = @_;
    my $productName = LCM::App::ApplicationContext::getFlavourProductName();
    if ( $self->{isHostPartOfSystem} ) {
        $self->{actions} = {
            "Uninstall"        => {console=>"uninstall",          gui=>"Uninstall $productName Components",},
            "RenameSystem"     => {console=>"rename_system",      gui=>"Rename the $productName System"},
            "UnregisterSystem" => {console=>"unregister_system",  gui=>"Unregister the $productName System"},
            "UpdateComponents" => {console=>"update_components",  gui=>"Install or Update Additional Components" . $self->{pendingInfoResident}},
            "UpdateSystem"     => {console=>"update",             gui=>"Update the $productName System" . $self->{pendingInfoNonResident}},
        };

        if ( $self->checkFlavour($gFlavourPlatform) ) {
            $self->{actions}->{"ConfigureInternalNetwork"} = {console=>"configure_internal_network", gui=>"Configure Inter-Service Communication"};
            $self->{actions}->{"AddHosts"}                 = {console=>"add_hosts",                  gui=>"Add Hosts to the $productName System"};
            $self->{actions}->{"AddHostRoles"}             = {console=>"add_host_roles",             gui=>"Add Host Roles"};
            $self->{actions}->{"RemoveHostRoles"}          = {console=>"remove_host_roles",          gui=>"Remove Host Roles"};
        }

        if ( ! $self->checkFlavour($gFlavourExpress) ) {
            $self->{actions}->{"ConfigureSLD"}        = {console=>"configure_sld",         gui=>"Configure System Landscape Directory Registration"};
            $self->{actions}->{"UpdateComponentList"} = {console=>"update_component_list", gui=>"Update Component List"};
        }

        if ( !$self->{isGui} ) {
            $self->{actions}->{"UpdateHost"}        = {console=>"update_host",        gui=>"Update the $productName Instance Host integration"};
            $self->{actions}->{"ExtractComponents"} = {console=>"extract_components", gui=>"Extract Components"};
            $self->{actions}->{"CheckInstallation"} = {console=>"check_installation", gui=>"Check $productName Installation"};

            if ( $self->checkFlavour($gFlavourPlatform) ) {
                $self->{actions}->{"PrintComponentList"} = {console=>"print_component_list", gui=>"Print Component List"};
            }
        }
    } else {
        $self->{actions}->{"RegisterRenameSystem"} = {console=>"register_rename_system", gui=>"Register and Rename $productName System"};

        if ( $self->checkFlavour($gFlavourPlatform) ) {
            $self->{actions}->{"AddHosts"} = {console=>"add_hosts", gui=>"Add Hosts to the $productName System"};
        }
	}

    if ( $self->{numberOfSystemHosts} > 1 ) {
        $self->{actions}->{"RemoveHosts"}        = {console=>"remove_hosts",        gui=>"Remove Hosts from the $productName System"};
        $self->{actions}->{"UnregisterInstance"} = {console=>"unregister_instance", gui=>"Unregister the $productName Instance"};
    }
    
    if ( $self->{config}->can('isScopeInstance') && !$self->{isGui} && $self->checkFlavour($gFlavourPlatform) ) {
        if ($self->{config}->isScopeInstance("hdblcm_configure_internal_network")) {
            $self->{actions}->{"ConfigureInternalNetwork"} = {console=>"configure_internal_network", gui=>"Configure Inter-Service Communication"};
        }
        if ($self->{config}->isScopeInstance("hdblcm_rename_system")) {
            delete ( $self->{actions}->{"RegisterRenameSystem"} );
            $self->{actions}->{"RenameSystem"} = {console=>"rename_system", gui=>"Rename the $productName System"};
        }
    }
}

sub checkFlavour {
    require LCM::Installer;
    require File::Spec;

    my ($self, $flavour) = @_;
    my $installer = new LCM::Installer();
    my $sapmnt = File::Spec->catfile($installer->getRuntimeDir(), '..', '..', '..');
    my $scm = $installer->getOwnSystemComponentManager($sapmnt);
    my $systemFlavour = $gFlavourPlatform;

    if(defined($scm)){
        my $serverComponent = $scm->getComponentByKeyName($gKeynameEngine);
        my $serverManifest = defined($serverComponent) ? $serverComponent->getManifest() : undef;
        $systemFlavour = defined($serverManifest) ? $serverManifest->getHANAFlavour() : $gFlavourPlatform;
    }
    return $flavour eq $systemFlavour;
}

sub _getVisibleActions {
    my ($self, ) = @_;
    my @visibleActions = ();

    for my $action(sort keys %{$self->{actions}}) {
        my $sub = $self->can("isVisible$action");
        if ($sub && &$sub($self)) {
            push (@visibleActions, $action);
        } elsif (!$sub) {
            push (@visibleActions, $action);
        }
    }
    return @visibleActions;
}

sub getApplicableActions {
	my ($self, ) = @_;
    my @validActions = ();
    my @validUiActions = ();

	for my $action(sort $self->_getVisibleActions()) {
		my $sub = $self->can("action$action");
		if ($sub && &$sub($self, $action)) {
			push (@validActions, $self->{actions}->{$action}->{console});
			push (@validUiActions, $self->{actions}->{$action}->{gui});
		}
	}

	return (\@validActions, \@validUiActions);
}

sub getNotApplicableActions {
    my ($self, ) = @_;
    my @notApplicableUiActions = ();
    my @notApplicableActions = ();

    for my $action($self->_getVisibleActions()) {
        my $sub = $self->can("action$action");
        if ($sub && !&$sub($self, $action)) {
            push (@notApplicableActions, $self->{actions}->{$action}->{console});
            push (@notApplicableUiActions, $self->{actions}->{$action}->{gui});
        }
    }

    return (\@notApplicableActions, \@notApplicableUiActions);
}

sub getNotApplicableActionsMessage {
    my ($self, ) = @_;
    
    my ($notApplicableActions, $notApplicableUiActions) = $self->getNotApplicableActions();
    
    if (!@$notApplicableActions) {
        return "";
    }
    
    return $self->{notApplicableActions}->getMessage();
}

sub isActionApplicable {
    my ($self, $action) = @_;
    my ($validActions, $validUiActions) = $self->getApplicableActions();

    for my $actionValue( @$validActions ) {
    	if ( $actionValue eq $action ) {
    		return 1;
    	}
    }
    return 0;
}

sub isVisibleConfigureSLD {
    return !$_[0]->{isWin};
}

sub isVisibleConfigureInternalNetwork {
    return !$_[0]->{isWin};
}

sub isVisibleRenameSystem {
    return !$_[0]->{isWin};
}

sub isVisiblePrintComponentList {
    return !$_[0]->{isWin};
}

sub isVisibleUnregisterInstance {
    return !$_[0]->{isWin};
}

sub isVisibleUnregisterSystem {
    return !$_[0]->{isWin};
}

sub isVisibleUpdateSystem {
    return !$_[0]->{isWin};
}

sub isVisibleUpdateComponents {
    return 1;
}

sub isVisibleRegisterRenameSystem {
    return !$_[0]->{isWin};
}

sub isVisibleRemoveHosts {
    return !$_[0]->{isWin};
}

sub isVisibleUpdateHost {
    return ( !$_[0]->{isWin} );
}

sub isVisibleExtractComponents {
    return ( !$_[0]->{isWin} );
}
sub isVisibleAddHostRoles {
    return !$_[0]->{isWin};
}
sub isVisibleCheckInstallation {
    return !$_[0]->{isWin};
}
sub isVisibleRemoveHostRoles {
    return !$_[0]->{isWin};
}

sub actionUninstall {
    my ($self, $action) = @_;

    return 0 if (! $self->checkRootOrSidadmUser($action));
    return 0 if ( $_[0]->{numberOfSystemHosts} <= 0 );

    if (isSidadmin() && ! $self->_hasUninstallableComponents()) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add("root", $action, $self->{actions}->{$action}->{gui});
        return 0;
    }
    return 1;
}

sub actionUpdateComponentList {
    my ($self, $action) = @_;
    if ( ! isAdmin() ) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add( "root", $action, $self->{actions}->{$action}->{gui} );
        return 0;
    }
    return ( $_[0]->{numberOfSystemHosts} > 0 );
}

sub actionConfigureSLD {
    my ($self, $action) = @_;
    if ( ! isAdmin() && ! isSidadmin() ) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add( "root", $action, $self->{actions}->{$action}->{gui} );
        $notApplicableActions->add( "sidadm", $action, $self->{actions}->{$action}->{gui} );
        return 0;
    }
    return ( $_[0]->{numberOfSystemHosts} > 0 );
}

sub actionConfigureInternalNetwork {
    my ($self, $action) = @_;
    return 0 if (! $self->checkRootOrSidadmUser($action));
    if( $_[0]->{isWin} ) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add( "win", $action, $self->{actions}->{$action}->{gui} );
        return 0;
    }

    my $config = $_[0]->{config};
    my $canIsScopeInstance = $config->can('isScopeInstance');
    return 1 if($canIsScopeInstance && $config->isScopeInstance("hdblcm_configure_internal_network"));
    return ($_[0]->{numberOfSystemHosts} > 0);
}

sub actionAddHosts {
    my ($self, $action) = @_;
    if( isAdmin() || (isSidadmin() && $_[0]->{isHostPartOfSystem}) ) {
    	return 1;
    }
    my $notApplicableActions = $self->{notApplicableActions};
    $notApplicableActions->add( "root", $action, $self->{actions}->{$action}->{gui} );
    return 0;
}

sub actionRemoveHosts {
    my ($self, $action) = @_;
    if ($_[0]->{numberOfSystemHosts} > 1 && (isSidadmin() || isAdmin()) ){
    	return 1;
    }
    my $notApplicableActions = $self->{notApplicableActions};
    $notApplicableActions->add( "root", $action, $self->{actions}->{$action}->{gui} );
    return 0;
}

sub actionRenameSystem {
    my ($self, $action) = @_;

    my $notApplicableActions = $self->{notApplicableActions};
    my $reason = undef;
    my $message = $self->{actions}->{$action}->{gui};

    if ( $_[0]->{isWin} ) {
        $reason = "win";
    } elsif ( ! isAdmin() ) {
        $reason = "root";
    }

    if ( defined $reason ) {
        $notApplicableActions->add( $reason, $action, $message );
        return 0;
    }

    my $config = $_[0]->{config};
    my $canIsScopeInstance = $config->can('isScopeInstance');
    return 1 if($canIsScopeInstance && $config->isScopeInstance("hdblcm_rename_system"));
    return ($_[0]->{numberOfSystemHosts} > 0);
}

sub actionUnregisterInstance {
    my ($self, $action) = @_;

    my $notApplicableActions = $self->{notApplicableActions};
    my $reason = undef;
    my $message = $self->{actions}->{$action}->{gui};

    if ( $_[0]->{isWin} ) {
        $reason = "win";
    } elsif ( ! isAdmin() ) {
        $reason = "root";
    }

    if ( defined $reason ) {
        $notApplicableActions->add( $reason, $action, $message );
        return 0;
    }

    return ( $_[0]->{numberOfSystemHosts} > 0 );
}

sub actionUnregisterSystem {
    my ($self, $action) = @_;

    my $notApplicableActions = $self->{notApplicableActions};
    my $reason = undef;
    my $message = $self->{actions}->{$action}->{gui};

    if ( $_[0]->{isWin} ) {
        $reason = "win";
    } elsif ( ! isAdmin() ) {
        $reason = "root";
    }

    if ( defined $reason ) {
        $notApplicableActions->add( $reason, $action, $message );
        return 0;
    }

    return ( $_[0]->{numberOfSystemHosts} > 0 );
}

sub checkRootOrSidadmUser {
    my ($self, $action) = @_;    
    
    if ( ! isAdmin() && ! isSidadmin() ) {
        my $notApplicableActions = $self->{notApplicableActions};
        my $message = $self->{actions}->{$action}->{gui};
        $notApplicableActions->add( "root", $action, $message );
        $notApplicableActions->add( "sidadm", $action, $message );
        return 0;
    }
    return 1;    
}

sub actionUpdateSystem {
    my ($self, $action) = @_;

    return 0 if (! $self->checkRootOrSidadmUser($action));
    return ($self->{numberOfSystemHosts} > 0);
}

sub actionUpdateComponents {
    my ($self, $action) = @_;

    return 0 if (! $self->checkRootOrSidadmUser($action));
    return ( $_[0]->{numberOfSystemHosts} > 0 );
}

sub actionPrintComponentList {
    my ($self, $action) = @_;
    return 0 if (! $self->checkRootOrSidadmUser($action));
    return 0 if ( $_[0]->{isGui} );
    if( $_[0]->{isWin} ) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add( "win", $action, $self->{actions}->{$action}->{gui} );
        return 0;
    }
    return ( $_[0]->{numberOfSystemHosts} > 0 );
}

sub actionRegisterRenameSystem {
    my ($self, $action) = @_;
    if ( ! isAdmin() ) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add( "root", $action, $self->{actions}->{$action}->{gui} );
        return 0;
    }
    return ( $_[0]->{numberOfSystemHosts} == 0 && !$_[0]->{isHostPartOfSystem} );
}

sub actionUpdateHost {
    my ($self, $action) = @_;
    return 1;
}

sub actionExtractComponents {
    my ($self, $action) = @_;
    return ($self->{numberOfSystemHosts} > 0);
}

sub actionAddHostRoles {
    my ($self, $action) = @_;
    return 0 if (! $self->checkRootOrSidadmUser($action));
    return ($self->{numberOfSystemHosts} > 0);
}

sub actionCheckInstallation {
    my ($self, $action) = @_;
    return 0 if (! $self->checkRootOrSidadmUser($action));
    return 0 if ( $_[0]->{isGui} );
    if( $_[0]->{isWin} ) {
        my $notApplicableActions = $self->{notApplicableActions};
        $notApplicableActions->add( "win", $action, $self->{actions}->{$action}->{gui} );
        return 0;
    }
    return 1;
}

sub actionRemoveHostRoles {
    my ($self, $action) = @_;
    return 0 if (! $self->checkRootOrSidadmUser($action));
    return ($self->{numberOfSystemHosts} > 0);
}

sub _hasUninstallableComponents {
    require LCM::Installer;
    require LCM::Component;
    my ($self) = @_;
    
    my $systemComponentManager = LCM::Installer->new()->getOwnSystemComponentManager();
    return 0 if(!defined($systemComponentManager));

	for my $component (@{$systemComponentManager->getAllComponents()}){
		my $componentKeyname = $component->getComponentKeyName();

		next if ($componentKeyname eq $gKeynameInstaller);
		next if($componentKeyname eq $gKeynameEngine);
		next if($componentKeyname eq $gKeynameSDA);
		next if ($componentKeyname eq $gKeynameLSS);
		next if($component->isInternal());

		return 1;
	}
	return 0;
}

1;