package LCM::App::AbstractApplication;

use strict;
use LCM::Installer;
use SAPDB::Install::Hostname;
use LCM::RegenerateCertificates;
use LCM::App::ApplicationContext;
use LCM::ExecutionWarningsObservable;
use SDB::Install::App;
use SDB::Install::Globals qw ($gFlavourPlatform $gKeynameEngine);
use SDB::Install::SysVars qw ($path_separator);
use Browser::Open qw( open_browser);

use parent 'Exporter';

our @EXPORT = qw( getURL openURL );
our $applicationContext;

sub getApplicationContext{
    if (!defined $applicationContext){
        $applicationContext = LCM::App::ApplicationContext::getInstance();
    }
    return $applicationContext;
}

sub isGUIApplication {
	return 0;
}

sub isBatchMode {
	return 0;
}

sub isPassThroughHelp {
    my ($self) = @_;
    return defined($self->{options}->{pass_through_help}) ? 1 : 0;
}

sub setInstconfig {
	my ($self, $configuration) = @_;
	$self->{instconfig} = $configuration;
	getApplicationContext ()->setConfiguration($configuration);
}

sub getInstconfig {
	my ($self) = @_;
	return $self->{instconfig};
}

sub setExecutor {
	my ($self, $executor) = @_;
	$self->{executor} = $executor;
	getApplicationContext ()->setExecutor($executor);
}

sub getExecutor {
	my ($self) = @_;
	return $self->{executor};
}

sub registerExitCodeHandler {
	my ($self) = @_;

	$self->{_exitCodeHandler} = LCM::Executors::ExecutionExitCodeHandler->new();
	LCM::ExecutionWarningsObservable->getInstance()->registerListener($self->{_exitCodeHandler});

	return 1;
}

sub registerAbortSignalHandler {
	my ($self) = @_;

	if (exists $SIG{'INT'}) {
		$self->getMsgLst()->addMessage ("Handler for signal INT");
		$SIG{'INT'} = sub {_abortSignalHandler($self, 'INT');};
	}
}

sub _abortSignalHandler {
	my ($application, $sig) = @_;
	my $task = $application->getExecutor();

	if (!defined $task) {
		SDB::Install::App::signalHandlerDie($sig);
	}

	if (!$task->isa('LCM::Task') || !$task->can('setAborted')) {
		SDB::Install::App::signalHandlerDie($sig);
	}

	$task->setAborted();

	SDB::Install::App::signalHandlerDie($sig);
}

sub handleExecutorMsgLst {
	my ($self) = @_;

	$self->_notifyHandlerOnWarning();
	$self->appendExecutorToAppMsgLst();
	return 1;
}

sub _notifyHandlerOnWarning {
	my ($self) = @_;

	my $executor = $self->getExecutor();
	if (!defined $executor) {
		return undef;
	}

	my $executorMsgLst = $executor->getMsgLst();
	if ($self->_existsWarning($executorMsgLst)) {
		$self->{_exitCodeHandler}->onWarning();
	}

	return 1;
}

sub _existsWarning {
	my ($self, $msgLst) = @_;

	if (! defined $msgLst) {
		return undef;
	}

	my $msgLstStr = $msgLst->getMsgLstString(undef, undef, 1);

	return (${$msgLstStr} =~ /WRN/);
}

sub appendExecutorToAppMsgLst {
	my ($self) = @_;

	my $executor = $self->getExecutor();
	if (!defined $executor) {
		return undef;
	}

	$self->getMsgLst()->appendMsgLst ($executor->getMsgLst());
	$self->getErrMsgLst()->appendMsgLst ($executor->getErrMsgLst());

	return 1;
}

sub clearFeedbackForm {
	my $self = shift();
	my $feedbackFormPath = $self->_getFeedbackFormPath();

	return if (! defined($feedbackFormPath));

	unlink($feedbackFormPath);
}

sub checkFeedbackFormExistance {
	my $self = shift();
	my $feedbackFormPath = $self->_getFeedbackFormPath();
	return defined($feedbackFormPath) && -e $feedbackFormPath;
}

sub getFeedbackMessage{
	my $self = shift();
	my $message = sprintf("%s: %s\n", $self->getFeedbackMessageLabel(), $self->getFeedbackUrl());
	return $message;
}

sub _getFeedbackFormPath{
	my $self = shift();

	my $configuration = $self->getInstconfig();
	return undef if(!defined($configuration));

	my $ownInstance = $configuration->getOwnInstance();
	return undef if(!defined($ownInstance));

	my $globalSidDir = $ownInstance->get_globalSidDir();
	my $feedbackFormFilePath = File::Spec->catfile($globalSidDir, 'hdblcm', 'resources', 'feedback', 'feedback.html');
	return $feedbackFormFilePath;
}

sub getFlavour {
    my ($self, $noPlatformDefault) = @_;
    my $flavour = undef;
    my $installer = new LCM::Installer();
    if($installer->isInstalled()){
        $flavour = $self->_detectFlavourByInternalServerManifest($installer);
    } else {
        $flavour = $self->_detectFlavourByExternalServerManifest();
        # non-resident & no server installation kit
        $flavour = $self->_detectFlavourBySid() unless defined $flavour;
    }
    $flavour = $flavour || $gFlavourPlatform if not $noPlatformDefault;
    return $flavour;
}

sub _detectFlavourByInternalServerManifest {
    my ($self, $installer) = @_;
    my $sharedSidDir = join ( $path_separator, $installer->{runtimedir}, '..', '..' )  ;
    my $trexBinDir = SDB::Install::SAPSystem::get_trexBinDir();
    my $pathTohdbManifest = join($path_separator, $sharedSidDir, $trexBinDir, 'manifest');
    if( -f $pathTohdbManifest ) {
        my $serverManifest = SDB::Install::Manifest->new();
        $serverManifest->read($pathTohdbManifest);
        return $serverManifest->getHANAFlavour();
    }
    return undef;
}

sub _detectFlavourByExternalServerManifest {
    my ($self) = shift();
    my $appContext = LCM::App::ApplicationContext::getInstance();
    my $serverManifest = $appContext->getServerManifest();
    my $flavour = defined($serverManifest) ? $serverManifest->getHANAFlavour() : undef;
    return $flavour;
}

sub _detectFlavourBySid {
    my ($self) = shift();
    my $instconfig = LCM::App::ApplicationContext::getInstance()->getApplication()->getInstconfig();
    $self->{sid} = $instconfig->getSID() if defined($instconfig) && defined($instconfig->getSID());
    return undef unless defined($self->{sid});

    my $systems = SDB::Install::SAPSystem::CollectSAPSystems (undef, 1);
    my $sapSystem = $systems->{$self->{sid}};
    return undef unless defined $sapSystem;

    return $sapSystem->getHANAFlavour();
}

sub getFeedbackMessageLabel {
	return 'You can send feedback to SAP with this form';
}

sub getFeedbackUrl {
	my $self = shift();
	my $feedbackUrl = $self->getURL();
	$feedbackUrl =~ s/index.html/feedback\/feedback.html/;
	return $feedbackUrl;
}

sub getURL {
	my ($application) = shift();

	my $SID = '<SID>';
	my $installer = new LCM::Installer();
	if ( $installer->isInstalled() ) {
		$SID=$installer->getSid();
	} elsif (defined($application)) {
		my $configuration = $application->getInstconfig();
		$SID = defined($configuration) ? $configuration->getValue('SID') : $SID;
	}

	my ( $localCertificateHostname ) = grep{$_ =~ qr/(?<=CN=)/} split(',',LCM::RegenerateCertificates::getCertOwner());
	
	$localCertificateHostname =~ s/CN=//;
	if( $localCertificateHostname eq '' ){
		$localCertificateHostname = hostname();
	}

	return "https://$localCertificateHostname:1129/lmsl/HDBLCM/$SID/index.html";
}

sub openURL {
	my ($url, $forceOpen) = @_;
	my $errorsOccured=0;
	my $installer = new LCM::Installer();
	my $isInstalled = $installer->isInstalled();

	if( !defined $ENV{DISPLAY} || (!$isInstalled && !$forceOpen)){
		$errorsOccured=1;
		return $errorsOccured;
	}
	my $pid=fork;
	if ( $pid == 0 ) { #child process
		my $ok = open_browser($url, 1);
		if($ok != 0){
			$errorsOccured=1;
		}
	} elsif ( defined $pid && $pid != 0 ){ #original process
        sleep 2;
    } else { #error
		print "Error while trying to open browser: $!";
		$errorsOccured=1;
    }
    waitpid($pid,0);
    return $errorsOccured;
}

sub setCLIParser {
    my ($self, $parser) = @_;
    $self->{CLIParser} = $parser;
}

sub getCLIParser {
    my $self = shift;
    return $self->{CLIParser};
}

1;
