#=========================================
# Specifies the custom_input methods for
# the AddHosts and HostsProperties parameters
#=========================================
package LCM::App::AddHostsUtils;
use strict;


use SDB::Install::MsgLst;
use SDB::Install::Tools qw(askConfirmation printTableToArrayOfLines);
use SDB::Install::Globals qw(GetHostRoleProperties GetAllRoles);
use SDB::Install::System qw (nslookup);
use SDB::Install::App::Console;
use SDB::Install::Saphostagent qw(sapadmUserExists);
use LCM::AdditionalHostHelper qw($SP_AUTO_ASSIGN);
use LCM::PersistencyUtils qw(existsHdblcmPendingUpdate);
our @EXPORT = qw ( readInteractiveAddHostsInput readHostsPropertiesForUpdate readHostsProperties readHostsPropertiesForInstall );


sub readInteractiveAddHostsInput {
	my ($console, $config) = @_;
	my $existsPendingUpdate = existsHdblcmPendingUpdate($config->{current_sid});
	if ($config->getUseDefault('AddHosts') && !$config->isMandatory('AddHosts')) {
		$config->{params}->{AddHosts}->{skip} = 1;
		$config->{params}->{HostsProperties}->{skip} = 1;
		return undef;
	}

	my $confirmAddHosts = _confirmAdditionOfHosts($config);
	if ($config->getValue('ContinueUpdate') || !$confirmAddHosts ) {
		$config->{params}->{AddHosts}->{skip} = 1;
		$config->{params}->{HostsProperties}->{skip} = 1;
		return undef;
	}
	my $doNotFailOnLocalHost = _isAddLocalHost($config);
	my $oldContext = $config->setMsgLstContext([new SDB::Install::MsgLst(), new SDB::Install::MsgLst()]);
	my $hostNames = '';

	while (1) {
		$config->resetError();
		print("Enter comma-separated host names to add: ");
		chomp($hostNames = <STDIN>);
		next if (length($hostNames) == 0);
		my @hosts = split(/\s*,\s*/, $hostNames);
		next if(! _areAllHostsReachable($config, \@hosts));

		# Exit loop if all hosts are valid
		last if ($config->handleAddOrRemoveHosts($hostNames, 'AddHosts', $doNotFailOnLocalHost));

		_printFirstError($config->getErrorString());
	}
	$config->{params}->{HostsProperties}->{csv_hostnames} = $hostNames;
	$config->setMsgLstContext($oldContext);

# Hide the AutoAddXS2Roles from the command line, because we need to first collect he HostProperties
	$config->{params}->{AutoAddXS2Roles}->{set_interactive} = 0;
	$config->{params}->{AutoAddXS2Roles}->{init_with_default} = 0;

	return $hostNames;
}

sub _areAllHostsReachable {
	my ($config, $hosts) = @_;
	for my $host (@$hosts) {
		my $nslookup = nslookup($host, $config->getErrMsgLst());
		if (! defined $nslookup->{address}) {
			_printFirstError($config->getErrorString());
			return 0;
		}
	}
	return 1;
}

sub _printFirstError {
	my ($errorString) = @_;
	my @firstError = split("\n", $errorString);
	print($firstError[0] . "\n");
}

sub _isAddLocalHost {
	my ($config) = @_;
	return $config->isa("LCM::Configuration::Hosts::AddHosts::AddLocalAndRemoteHostsConfiguration");
}

sub _isAddRemoteHostsOnly {
	my ($config) = @_;
	return $config->isa("LCM::Configuration::Hosts::AddHosts::AddRemoteHostsConfiguration");
}

# if persistent file hdbaddhost exists & it contains step entry,
# previous add host command is not finished successfully
sub isPendingAddHost {
	my ($instance, $hostname) = @_;
	require SDB::Install::Configuration::AddHost;
	my $addHostConfig = new SDB::Install::Configuration::AddHost();
	return defined $addHostConfig->getAddHostPendingStep($instance, $hostname);
}

sub _confirmAdditionOfHosts {
	my ($config) = @_;
	if (! _isAddLocalHost($config) && ! _isAddRemoteHostsOnly($config)) {
		return askConfirmation("Do you want to add hosts to the system?", "n");
	}
	return 1;
}

# Wrapper for the readHostsProperties function that stores the
# original additional remote hosts objects of the configuration and
# restores them after the new --addhosts value is set by _readHostsProperties
sub readHostsPropertiesForUpdate {
	my ($console, $config) = @_;
	my $additionalRemoteHostsObjects = $config->{_additionalRemoteHosts};
	$config->{_additionalRemoteHosts} = undef;
	_readHostsProperties(@_);
	$config->{_additionalRemoteHosts} = $additionalRemoteHostsObjects;
# Must return undef so that SDB::Install:Console::getParam
# does not add any value for the dummy param in the logs
	return undef;
}


# Wrapper for the readHostsProperties function that stores the
# original remote hosts objects of the configuration and restores them
# after the new --addhosts value is set by _readHostsProperties
sub readHostsPropertiesForInstall {
	my ($console, $config) = @_;
    my $remoteHostsObjects = $config->{_remoteHosts};
    $config->{_remoteHosts} = undef;
    _readHostsProperties(@_);
    $config->{_remoteHosts} = $remoteHostsObjects;
    return undef;
}

sub readHostsProperties {
	my ($console, $config) = @_;
	my $additionalRemoteHostsObjects = $config->{_additionalRemoteHosts};
	$config->{_additionalRemoteHosts} = undef;
	_readHostsProperties(@_);
	$config->{_additionalRemoteHosts} = $additionalRemoteHostsObjects;
# Must return undef so that SDB::Install:Console::getParam
# does not add any value for the dummy param in the logs
	return undef;
}

sub _readHostsProperties {
	require SDB::Install::Configuration::AddHost;

	my ($console, $config) = @_;
	my $hostNames = $config->{params}->{HostsProperties}->{csv_hostnames};
	return if (!$hostNames);

	my $oldContext = $config->setMsgLstContext([new SDB::Install::MsgLst(), new SDB::Install::MsgLst()]);
	my @hostProperties = ();
	my @hostNamesArray = split(/\s*,\s*/, $hostNames);
	my $assignedStoragePartitions = {};

	my $hostRoleDefaultValue = $SDB::Install::Configuration::AnyMultiHostConfig::hostRoleDefaultValue;
	my $validHostRoles = $SDB::Install::Configuration::AnyMultiHostConfig::validHostRoles;

	# collect batch arguments for --add_roles --add_local_roles
	my $addLocalRolesValue = $config->getValue('AddLocalRoles');
	my $addRolesMapValue = $config->getValue('AddRoles');

	my @hostsRoles = ();

	push(@hostsRoles, split(',', $addLocalRolesValue)) if ($addLocalRolesValue);
	if (defined $addRolesMapValue && keys %$addRolesMapValue > 0) {
		foreach (keys %$addRolesMapValue) {
			push(@hostsRoles, split(',', $addRolesMapValue->{$_}));
		}
	}

	for my $host (@hostNamesArray) {
		my $addHostConfig = new SDB::Install::Configuration::AddHost();
		my $params = $addHostConfig->{params};
		my $csvRoles = _readRolesByIndex($config, $host, \@hostsRoles);
		my $rolesForHost = [ split(',', $csvRoles) ];
		my $default_group = 'default';

		push (@hostsRoles, @{$rolesForHost});
		if (defined $validHostRoles->{$csvRoles}) {
			$default_group = $validHostRoles->{$csvRoles}->{defaultGroup};
		}
		$addHostConfig->setDefault('HostFailoverGroup', $default_group);
		my $group = _readSingleProperty($config, $addHostConfig, 'HostFailoverGroup', $params->{HostFailoverGroup}, $host, _buildPropertyValuesStr($csvRoles));
		my $sp = $SP_AUTO_ASSIGN;
		$addHostConfig->setDefault('StoragePartitionNumber', $SP_AUTO_ASSIGN);
		if (_canSpecifyStoragePartitionNumber($csvRoles)) {
			$sp = _readSingleProperty($config, $addHostConfig, 'StoragePartitionNumber', $params->{StoragePartitionNumber}, $host, _buildPropertyValuesStr($csvRoles, $group));
			# while storage partition is taken
			while (defined $assignedStoragePartitions->{$sp}) {
				my $takenBy = $assignedStoragePartitions->{$sp}->{host};
				print("Storage partition $sp is already assigned to host '$takenBy'.\n");
				$sp = _readSingleProperty($config, $addHostConfig, 'StoragePartitionNumber', $params->{StoragePartitionNumber}, $host, _buildPropertyValuesStr($csvRoles, $group));
			}
		}
		my $workergroup;
		my $isWorkerGroupSupported = grep { defined($validHostRoles->{$_}) && $validHostRoles->{$_}->{hasWorkerGroup} } @{$rolesForHost};
		if($isWorkerGroupSupported){
			$workergroup = _readSingleProperty($config, $addHostConfig, 'WorkerGroup', $params->{WorkerGroup}, $host, _buildPropertyValuesStr($csvRoles, $group, $sp));
		}
		$assignedStoragePartitions->{$sp}->{host} = $host if($sp ne $SP_AUTO_ASSIGN);
		push(@hostProperties, _buildPropertyValuesStr($csvRoles, $group, $sp, $workergroup));
	}
	#build full param string
	my @namePropertyPairs = map { $hostNamesArray[$_] . $hostProperties[$_] } (0..$#hostNamesArray);
	my $fullValue = join(',', @namePropertyPairs);

    # Bug 111434 - execution of AddHostsListener skip the root password
    my $isSkipped = $config->isSkipped('RootPassword');
    $config->setValue('AddHosts', $fullValue);
    $config->setSkip('RootPassword', $isSkipped);

	$config->setSkip('HostsProperties', 1);
	$config->setMsgLstContext($oldContext);

	if(!$config->isSkipped('AutoAddXS2Roles') && !$config->hasValue('AutoAddXS2Roles')){
		_readAutoAddXS2Roles($config);
	}

	return 1;
}

sub _readAutoAddXS2Roles {
	my ($config) = @_;
	my $defaultValue = $config->getDefault('AutoAddXS2Roles') ? 'y' : 'n';
	my $interractiveString = $config->{params}->{AutoAddXS2Roles}->{str} . " [${defaultValue}]: ";
	my $input;
	while(1) {
		print $interractiveString;
		chomp($input = <STDIN>);
		$input = length($input) == 0 ? $defaultValue : $input;

		if(!$config->setValue('AutoAddXS2Roles', $input)){
			print($config->getErrorString() . "\n");
			$config->resetError();
			next;
		}
		last;
	}
}

# If one of the valid csvRoles can specify a storage partition
# return True
sub _canSpecifyStoragePartitionNumber {
	my ($csvRoles) = @_;
	my $validHostRoles = $SDB::Install::Configuration::AnyMultiHostConfig::validHostRoles;
	foreach my $role (split(',',$csvRoles)) {
		if (exists $validHostRoles->{$role}->{hasPartition} && $validHostRoles->{$role}->{hasPartition}) {
			return 1;
		}
	}
	return 0;
}

sub _buildPropertyValuesStr {
	my ($role, $group, $sp, $wg) = @_;
	$role =~ s/,/:role=/g;
	my $returnStr = ":role=$role";
	$returnStr .= ":group=$group" if defined($group);
	if (defined $sp && $sp ne $SP_AUTO_ASSIGN) {
		$returnStr .= ":storage_partition=$sp";
	}
	if (defined($wg) && $wg ne '') {
		my @groups = split(',', $wg);
		$returnStr .= ":workergroup=" . join(':workergroup=', @groups);
	}
	return $returnStr;
}

sub _readSingleProperty {
	my ($config, $sourceConfig, $paramId, $param, $hostName, $hostPropertiesString) = @_;
	my $default = $sourceConfig->getDefault($paramId);
	my $str = $param->{str};
	my $input = '';
	my $action = "Enter $str for host '$hostName'";
	$action .= (defined $default) ? " [$default]: " : ': ';

	while(1) {
		my $singlePropertyValue = undef;
		print $action;
		chomp($input = <STDIN>);  # TODO : strip smart-ass comma bugs

		# Repeat loop, user has given empty input when no default is available
		next if ($input eq '' && ! defined($default));
		# User has chosen the default value
		return $default if (length($input) == 0);

		my $formattedString = _getFormattedPropertyString($input, $param->{opt}, sprintf('%s%s', $hostName, $hostPropertiesString));
		if (! $config->handleAddOrRemoveHosts($formattedString, 'AddHosts', 1)) {
			print($config->getErrorString() . "\n");
			$config->resetError();
			next; # User input contains an invalid value, repeat loop
		}
		return $input;
	}
}

sub _readRolesByIndex {
	my ($config, $hostName, $hostsRoles) = @_;
	my $rolesArr = GetAllRoles();
	my $console = new SDB::Install::App::Console();
	my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
	$app->registerAbortSignalHandler();

	my $table = _buildTable();
	my $default = $SDB::Install::Configuration::AnyMultiHostConfig::hostRoleDefaultValue;
	my $defaultIndex;
	$defaultIndex  = _getDefaultRoleIndex() if ! $config->isUpdate();

	my $indexTable = printTableToArrayOfLines($table, ' | ', 1);
	my $action = "Enter comma-separated list of selected indices";
	$action .= (defined $defaultIndex) ? " [$defaultIndex]: " : ": ";

	while (1) {
		my $csvRoles = undef;

		print "\nSelect roles for host '$hostName':\n\n";
		_printTable($indexTable);
		print($action);

		my $input = '';
		chomp($input = <STDIN>);
		my $dummyParam = {type => 'csv', valid_values => $rolesArr};
		$csvRoles = $console->convertFromIndexToValue($dummyParam, $input);
		next if (!defined $csvRoles);

		if ($csvRoles eq '') {
			return $default if !$config->isUpdate();
			print "Please select at least one host role.\n";
			next;
		}

		my $formattedPropertyStr = _getFormattedPropertyString($csvRoles, 'role', $hostName);
		my $checkNewRolesCompatibillity = scalar (@$hostsRoles) > 0;
		if ( $checkNewRolesCompatibillity && ! $config->checkNewRolesCompatibility( [@$hostsRoles, split(',', $csvRoles)] ) ) {
			_printFirstError($config->getErrorString());
			$config->resetError();
			next;
		}

		if ($config->handleAddOrRemoveHosts($formattedPropertyStr, 'AddHosts', 1)) {
			return $csvRoles;
		}
		_printFirstError($config->getErrorString());
		$config->resetError();
	}
}

sub _buildTable {
	my $rolesArr = GetAllRoles();
	my $roleProperties = GetHostRoleProperties();

	my @table = ();
	push @table, ['Index', 'Host Role', 'Description'];

	my $index = 1;
	foreach my $role (@$rolesArr) {
		my $roleStr = $roleProperties->{$role}->{str};
		$roleStr =~ s/\(.*\)?//g;
		push @table, [$index, $role, $roleStr];
		$index++;
	}
	return \@table;
}

sub _getDefaultRoleIndex {
	my $default = $SDB::Install::Configuration::AnyMultiHostConfig::hostRoleDefaultValue;
	return undef if (! defined $default);

	my $rolesArr = GetAllRoles();
	my $index = 1;
	foreach my $role (@$rolesArr) {
		last if ($role eq $default);
		$index++;
	}
	return $index;
}

sub _printTable {
	my ($indexTable) = @_;
	print '  ' . join ("\n  ", @$indexTable) . "\n\n";
}

sub _getFormattedPropertyString {
	my ($csvString, $paramOpt, $hostPropertiesString) = @_;
	my $formattedString = $csvString;
	$formattedString =~ s/,/:$paramOpt=/g;
	return "$hostPropertiesString:$paramOpt=$formattedString";
}

1;
