package LCM::App::BaseApplication;

use strict;
use LCM::TaskHandler;
use SDB::Install::SAPSystem;
use SDB::Install::Globals qw (getFlavourProductName $gProductNameInstaller $gFlavourExpress $gFlavourCockpit $gKeynameEngine $gKeynameCockpitStack);
use SDB::Install::Log;
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::App::Console;
use SDB::Install::App::Console::LCMUnregister;
use LCM::App::ApplicationContext;
use LCM::Executors::ExecutionExitCodeHandler;
use LCM::ExecutionWarningsObservable;
use LCM::PersistencyUtils qw (getAllPendingUpdateInfo);
use LCM::SlAnalytics::SLAMonitor;
use LCM::App::CLIArgumentsParser;

use base qw (LCM::App::AbstractApplication SDB::Install::App::Console);

my $isPrintedHeaderMsg;
my $arePrintedNotApplicableActions;

sub new {
	my $self = shift->SUPER::new();

	$self->{hdblcm} = 1;
	$self->_setProcessFinished(0);
	$self->registerExitCodeHandler();
	$self->registerAbortSignalHandler();
	LCM::App::ApplicationContext::getInstance()->setApplication($self);

	return $self;
}

sub InitApp {

	my ($self) = @_;

	$self->setCommonLogDir('HDBLCM_LOGDIR', 'hdblcm.log');
	$self->setCLIParser(LCM::App::CLIArgumentsParser->new($self));
	$self->SUPER::InitApp();
	return 1;
}

sub start {
	my $app = shift;
	local @ARGV = @_;

	my $rc = undef;
	if (!$app->errorState()) {
		$app->{stackBacktraceMsglst} = new SDB::Install::MsgLst();
		eval {
			my $args = $app->InitCmdLineArgs( \@_, 1 );

			if ( defined $app->{return} ) {
				$app->appendExecutorToAppMsgLst();
				LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
				return $app->{return};
			}

			$rc = ($args) ? $app->init() : undef;
		};
	}

	$app->handleExecutorMsgLst();

	if (!defined $app->{return}) {
		$rc = $app->handleReturnCodes( $@, $rc );
		$app->CleanUp();
	}

	if ( $isWin && !$app->{batch_mode} && !$app->{options}->{noprompt} ) {
		undef $app;
		SDB::Install::App::Console->waitForKeyPress();
	}

	if (defined $app->{_exitCodeHandler} && defined $app->{_exitCodeHandler}->getExitCodeOnWarning()) {
		return $app->{_exitCodeHandler}->getExitCodeOnWarning();
	}
	
	if ( defined $app->{return} ) {
		LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
		return $app->{return};
	}

	# rc = 2  user cancel (ctrl+c)
	# rc = 3  validation error
	if(defined $rc && ($rc == 2 || $rc == 3)){
		$rc = undef;
	}
	return ( defined $rc ) ? 0 : 1;
}

sub getProgramName {
	return 'hdblcm';
}

sub initConfiguration {
	my ($self, $action) = @_;
	my $actionSID = $self->getActionConfiguration()->getValue('Action_SID');
	my $isPendingInstallation = $self->getActionConfiguration()->isPendingInstallation();
	my $configuration = $self->createConfiguration($action);
	if (!defined($configuration)) {
		return 1 if $self->isHelp(); # Help doesn't need a configuration

		$self->getErrMsgLst()->addError("Failed to create configuration for action '$action'.");
		return undef;
	}

	if ( $isPendingInstallation && defined $actionSID ) {
		$configuration->setPendingInstallation(1);
		$configuration->{current_sid} = $actionSID;
		$configuration->{params}->{SID}->{value} = $actionSID;
		my $sapSystems = CollectSAPSystems();
		if (defined  $sapSystems->{$actionSID}){
			$configuration->{params}->{Target}->{value} = $sapSystems->{$actionSID}->get_target();
		}
		if (! $configuration->handlePendingInstall($actionSID)) {
			$configuration->setNoRetry("SID",1);
			$self->getErrMsgLst()->addError($configuration->getErrorString());
			return undef;
		}
	}
	$self->setInstconfig($configuration);
	return 1;
}

sub initLog {
	my ($self, $action) = @_;
	my $configuration	= $self->getInstconfig();
	my $check_only 		= ( defined $configuration ) ? $configuration->isCheckOnly() : 0;
	my $sid        		= ( defined $configuration ) ? $configuration->getBatchValue('SID') : undef;
	if (!$sid) {
		my $installer = new LCM::Installer();
		$sid 		  = $installer->getSid();
	}
	my $logSuffix = $configuration->getCustomLogSuffix();
	my $logName = "hdblcm" . ($logSuffix ? "_$logSuffix" : "");
	if (!$self->defineLog(($self->getProgramName())."_$action", $logName, $check_only, $sid)) {
		$self->getErrMsgLst()->addError("Failed to initialize logs for action '$action'");
		return undef;
	}
	return 1;
}

sub init {
	my ($self) = @_;
    my ($action, $isError) = $self->determineAction();
	# remember action to distinguish validation errors 
	if(defined $action) {
		$self->{action} = $action;
	}

	if ($isError && $self->isStdinError()) {
		return undef;
	}

	if ($isError || (!defined($action) && !$self->isHelp())) {
		return 3;
	}

	if (!$self->initConfiguration($action)) {
		return undef;
	}

	my $executor = $self->createExecutor($action);
	$self->setExecutor($executor);

	if ($self->_shallAddHeaderMessage()) {
		$self->_addHeaderMessage();
	}

	if ($executor->getCLIMetadata()->requiresLog()) {
		return undef if !$self->initLog($action);
	}

	my $taskHandler = $self->createTaskHandler($action);
	$taskHandler->setMsgLstContext( $self->getMsgLstContext() );

	my $returnCode = $taskHandler->process($self->isBatchMode());
	$self->_setProcessFinished($taskHandler->isExecutionStarted());
	return $returnCode;
}

sub isGenericAction {
	my ($self) = @_;
	return $self->isHelp() || $self->isDumpTemplate();
}

sub determineAction {
	my ($self) = @_;
	my $instconfig = $self->createActionConfiguration();
	$self->setInstconfig($instconfig);
	$self->setActionConfiguration($instconfig);
	$self->_setConfigurationProgressHandler($instconfig);

	my $isBatch = $self->isBatchMode() || $self->isGenericAction();
	my $taskHandler = LCM::TaskHandler->new($instconfig, undef, $self->getCLIParser(), $self->GetSysInfo(), $self);
	$taskHandler->setMsgLstContext($self->getMsgLstContext());

	return $isBatch ? $self->determineActionBatch($taskHandler, 1) : $self->determineActionInteractive($taskHandler);
}

sub determineActionBatch {
	my ($self, $taskHandler, $shouldUseDefault) = @_;
	my $configuration = $self->getActionConfiguration();

	if(!$taskHandler->collectInput(1,1)) {
		$self->setUnknownActionMessage( "Unknown action:" . $configuration->getErrorString() );
		return (undef, 1);
	}

	my $action = $configuration->getAction(1);
	if (!defined($action) && $shouldUseDefault) {
		# Try to get default action
		my $lastParameter = $self->_getParamWithHighestOrder($configuration);
		$configuration->setActionBatchValue();
		$configuration->setBatchValueOfRequiredParams($lastParameter);
		$taskHandler->collectInput(1,1);
		$action = $configuration->getAction(1);
	}

	if (!$action) { # in case of missing default value
		$self->setUnknownActionMessage( $self->getSpecifyActionErrorMessage() );
	}

	return ($action, 0);
}

sub determineActionInteractive {
	my ($self, $taskHandler) = @_;
	my $configuration = $self->getActionConfiguration();
# First try to get it from the command line
	my ($action, $isError) = $self->determineActionBatch($taskHandler, 0);
	if ($isError) {
		return (undef, 1); # Unknown action
	} elsif ($action) {
		return ($action, 0); # Successfully fetched the action
	}

# Only if the action was no specified on the command line, try interactively
	$self->_addHeaderMessage();
	$self->_addNotApplicableActionsMessage($configuration);
	if(!$taskHandler->collectInput(0,1)) {
		$self->setUnknownActionMessage( "Unknown action:" . $configuration->getErrorString() );
		return (undef, 1);
	}

	$action = $configuration->getAction(0);
	if (!defined($action) && $configuration->errorState()) {
		$self->setUnknownActionMessage( "Error during collecting input:" . $configuration->getErrorString() );
		return (undef, 1);
	}

	return ($action, 0);
}

sub _setConfigurationProgressHandler {
    my ($self, $instconfig) = @_;
    my $configMsgLst = $instconfig->getMsgLst();
    my $configErrMsgLst = $instconfig->getErrMsgLst();
    
    unless($configMsgLst->getProgressHandler()) {
        $configMsgLst->setProgressHandler($self->getMsgLst()->getProgressHandler());
    }
    
    unless($configErrMsgLst->getProgressHandler()) {
        $configErrMsgLst->setProgressHandler($self->getErrMsgLst()->getProgressHandler());
    }
    return 1;
}

sub createTaskHandler {
    my ($self, $action) = @_;
    my @args = ($self->getInstconfig(), $self->getExecutor(), $self->getCLIParser(), $self->GetSysInfo(), $self->_getConsole());
    return LCM::TaskHandler->new(@args);
}

sub _getParamWithHighestOrder {
	my ($self, $instconfig ) = @_;
	my @params = @{$instconfig->getParamIds()};
	return $params[-1];
	
}

sub setActionConfiguration {
	my ($self, $configuration) = @_;
	$self->{actionInstconfig} = $configuration;
}

sub getActionConfiguration {
	my ($self) = @_;
	return $self->{actionInstconfig};
}

sub printMessageByReturnCode {
	my ( $self, $rc, $systemName ) = @_;
	if (defined $rc && $rc == 3 ) {
		$self->getMsgLst()->addProgressMessage( "\n" . $self->getUnknownActionMessage() . "\n" );
		$self->PrintUsage();
		return;
	}

	if ($self->getAction() eq "print_component_list") {
		return;
	}
	
	$self->SUPER::printMessageByReturnCode( $rc, $systemName );
}

sub getAbortedProgressMessage {
	my ($self) = @_;
	my $executor = $self->getExecutor();
	if (defined($executor)) {
        return $executor->getStatus()->getMessage();
    }
	return $self->SUPER::getAbortedProgressMessage();
}

sub getFailedProgressMessage {
	my ( $self ) = @_;
	my $configuration = $self->getInstconfig();
	if ( defined($configuration) && $configuration->isPendingInstallation() ) {
	    my $flavourProductName = getFlavourProductName($self->getFlavour());
		return "Resume of $flavourProductName Installation failed";
	}

	my $executor = $self->getExecutor();
    if (defined($executor)){
    	return $executor->getStatus()->getMessage();
    }
	return $self->SUPER::getFailedProgressMessage();
}

sub getDoneProgressMessage {
	my ( $self ) = @_;
	my $executor = $self->getExecutor();
	if (defined($executor)) {
		my $finalMessage     = $executor->getStatus()->getMessage($executor->hasWarningMessages()) // '';
		my $finalInfoMessage = $executor->getFinalInfoMessage();
		if(defined($finalInfoMessage)) {
			$finalMessage .= "\n\n$finalInfoMessage";
		}
	    my $warningsMessage = $self->_getWarnings();
	    return defined($warningsMessage) ? "$finalMessage\n\n$warningsMessage" : $finalMessage;
	}
	return $self->SUPER::getDoneProgressMessage();
}

sub _getWarnings {
	my ( $self ) = @_;
	
	my $executor = $self->{executor};
	my $warningsRef = $executor->getAllWarningMessages();
	if (!defined($warningsRef) || !@{$warningsRef}) {
		return undef;
	}

	my $warningMessage = "";
	$warningMessage .= "Note: $_\n" for @{$warningsRef};
	return $warningMessage;
}

sub setUnknownActionMessage {
	$_[0]->{unknownActionMessage} = $_[1];
}

sub getUnknownActionMessage {
	return $_[0]->{unknownActionMessage};
}

sub createActionConfiguration {
	die "Method not implemented. Concrete applications should provide implementation of this method";
}

sub getScope {
    return 'system';
}

#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of batch and help options.
# This subroutine overrides SDB::install:App::GetBatchHelpUsage.
sub GetBatchHelpUsage {
	my ($self) = @_;

	my $superUsage             = $self->SUPER::GetBatchHelpUsage();
	my $paramsWithExtendedHelp = $self->getActionParams();

	if (@$paramsWithExtendedHelp) {
		for my $currParam (@$paramsWithExtendedHelp) {
			if (defined $currParam->{help_switch_usage}
			    && ($currParam->{help_switch_usage} =~ 'pass_through')) {
				push @$superUsage,
				     ['--pass_through_help', undef,
				      'Adds special parameters of subprograms to the help information'];
				last;
			}
		}
	}

	if ( defined $self->{action} or !@$paramsWithExtendedHelp ) {
		return $superUsage;
	}
	my $message = "\nFor extended help, use with the parameter " . $self->getParamOptionsAsString($paramsWithExtendedHelp);
	for my $currUsage (@$superUsage) {
		if ( $currUsage->[0] eq '--help' ) {
			$currUsage->[2] = $currUsage->[2] . $message;
			last;
		}
	}
	return $superUsage;
}

sub getParamOptionsAsString {
	my ( $self, $params ) = @_;
	my @params    = @$params;
	my $message   = "";
	my $lastIndex = $#params;
	
	for my $i (0..$lastIndex) {
		$message .= "'--" . $params[$i]->{opt} . "'";
		if ($i < $lastIndex - 1) {
			$message .= ', ';
		} elsif ($i == $lastIndex - 1) {
			$message .= ' or ';
		}
	}
	return $message;
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options.
# This subroutine overrides the standard help output with the additional
# option 'action'.
sub GetHelpSwitches {
	my ($self, $indent)        = @_;
	my @paramsWithExtendedHelp = @{ $self->getActionParams() };
	if ( !scalar @paramsWithExtendedHelp ) {
		return $self->SUPER::GetHelpSwitches($indent);
	}

	my @extendedSwitches;

	foreach my $param (@paramsWithExtendedHelp) {
		my $line = "--" . $param->{opt};
        if (defined $param->{valid_values}) {
            $line .= '=';
            my $isFirst = 1;
            for my $val (@{$param->{valid_values}}) {
                next if (exists $param->{skip_help_valid_values}->{$val});
                $line .= ($isFirst) ? $val : "|$val";
                $isFirst = 0;
            }
        }
        elsif (defined $self->{action}) {
            $line .= '=' . $self->{action};
        }
		if ( defined $param->{help_switch_usage} ) {
			$line .= ' ' . $param->{help_switch_usage};
		}
		push @extendedSwitches, $line;
	}

	return $self->GetModifiedHelpSwitches(\@extendedSwitches, $indent);
}

sub getActionParams {
	my $self = shift;

	my $instconfig = $self->getInstconfig();
	if ( !defined $instconfig ) {
		return [];
	}
	my @paramsWithExtendedHelp = ();
	my $paramIds               = $instconfig->getParamIds();
	for (@$paramIds) {
		if ( $instconfig->{params}->{$_}->{action_switch} ) {
			push( @paramsWithExtendedHelp, $instconfig->{params}->{$_} );
		}
	}

	return \@paramsWithExtendedHelp;
}

sub getSpecifyActionErrorMessage {
	my $self         = shift;
	my $actionParams = $self->getActionParams();
	if ( !@$actionParams ) {
		return "Specify action parameter";
	}

	return "Specify parameter " . $self->getParamOptionsAsString($actionParams);
}

sub _addHeaderMessage {
    my ( $self) = @_;

    return if ($isPrintedHeaderMsg);

    my $flavour = $self->getFlavour();
    my $configuration = $self->getInstconfig();
    my $productName = getFlavourProductName($flavour);
    my $isInstalled = LCM::Installer->new()->isInstalled();
    my $isInSimpleMode = $configuration->can('isSimpleMode') ? $configuration->isSimpleMode() : undef;
    my $caption = "$gProductNameInstaller - $productName " . $self->getProductVersion($flavour) ;
    my $headerMessage = "\n\n" . $caption . "\n" . ( '*' x length($caption) );
    $headerMessage = $headerMessage."\n\n" if (!$isInstalled && ! $isInSimpleMode);
    my $msglst  = $self->getMsgLst();
    $msglst->addProgressMessage( $headerMessage );
    SDB::Install::App::AnyApp::addStartDateMsg( undef, $msglst, $SAPDB::Install::Config{ProgramName} );
    my $actionHeader = $self->getActionConfiguration()->printActionHeaderMessage();
    $isPrintedHeaderMsg = 1;
}

sub getProductVersion {
    my ($self, $flavour) = @_;
    my $installer = new LCM::Installer();
    my $scm = $installer->getOwnSystemComponentManager();
    my $configuration = $self->getInstconfig();
    my $isCockipt = ($flavour eq $gFlavourCockpit);
    my $componentKeyname = $isCockipt ? $gKeynameCockpitStack : $gKeynameEngine;

    if($installer->isInstalled() && defined($scm)){
        my $installedComponent = $scm->getComponentByKeyName($componentKeyname);
        return $installedComponent->getVersion() if(defined($installedComponent));
    }
    if($configuration->can('getComponentManager')){
        my $componentManager = $configuration->getComponentManager();
        my $detectedComponent = defined($componentManager) ? $componentManager->getComponentByKeyName($componentKeyname) : undef;
        return $detectedComponent->getVersion() if(defined($detectedComponent));
    }
    my $appContext = LCM::App::ApplicationContext::getInstance();
    my $serverManifest = $appContext->getServerManifest();
    return defined($serverManifest) ? $serverManifest->getVersion() : $installer->GetVersion();
}

sub _getNotApplicableActionsMessage {
    my ( $self, $instconfig) = @_;

    if ( $arePrintedNotApplicableActions or $self->{batch_mode} or ref($instconfig) ne "LCM::Configuration::HdbLcmInstalledActionConfiguration" ) {
    	return "";
    }

    my $actionsDetector = $instconfig->{actionsDetector};
    my ($notApplicableActionsMessage) = $actionsDetector->getNotApplicableActionsMessage();
    
    $arePrintedNotApplicableActions = 1;
    return $notApplicableActionsMessage;
}

sub _shallAddHeaderMessage{
	my($self) = @_;
	my $executor = $self->getExecutor();
    return $executor ? $executor->getCLIMetadata()->requiresHeaderMessage() : 0;
}

sub _addNotApplicableActionsMessage{
	my ( $self, $instconfig) = @_;
	my $notApplicableActionsMessage = $self->_getNotApplicableActionsMessage($instconfig);
    my $msglst  = $self->getMsgLst();
    $msglst->addProgressMessage(  $notApplicableActionsMessage );
}

sub isBatchMode {
	return ($_[0]->{batch_mode}) ? 1 : 0;
}

sub getPendingUpdateInfo {
	my ($self, $sapSys) = @_;
	return getAllPendingUpdateInfo($sapSys, "\t", '');
}

sub CleanUp{
	my $self = shift();
	if ($self->hasProcessFinished() && LCM::SlAnalytics::SLAMonitor::hasSLAMonitorFinished() && $self->checkFeedbackFormExistance()){
		print $self->getFeedbackMessage();
	}

	$self->SUPER::CleanUp();
}

sub hasProcessFinished {
	return $_[0]->{_processStarted};
}

sub _setProcessFinished {
	my ($self, $processStarted) = @_;
	$self->{_processStarted} = $processStarted;
}

sub _getConsole {
    my ($self) = @_;
    return $self;
}

sub InitCmdLineArgs {
    my ($self, $arguments) = @_;
    my $cmdLineArguments = $self->SUPER::InitCmdLineArgs($arguments, 1);
    if (!defined($cmdLineArguments)) {
        return undef;
    }

    my $optCtrl = {
        'pass_through_help' => \$self->{options}->{pass_through_help},
        'action=s' => \$self->{options}->{action},
        'uninstall' => \$self->{options}->{uninstall}
    };
    local @ARGV = @$cmdLineArguments;
    $self->getOptions($optCtrl, 1);

    if ($self->isPassThroughHelp()) {
        if (!$self->isHelp() || (!defined $self->{options}->{action} && !defined $self->{options}->{uninstall})) {
            $self->appendErrorMessage ("Wrong usage: --pass_through_help should only be provided along with --help and --action");
            return undef;
        }
    }
    return $cmdLineArguments;
}

1;
