package LCM::App::UpdateSystemUtils;
use strict;
use warnings;

use File::Spec;
use File::Basename;
use SDB::Install::Globals qw ($gKeynameEngine $gSAPDSignerNotDetectedMessage $gUpgradeRestrictionsMessage GetNewerServerAlreadyInstalledMessage);
use SDB::Install::Manifest;
use SDB::Install::System qw(copy_tree deltree isAdmin);
use SDB::Install::MsgLst;
use LCM::Installer;
use LCM::SAPDSigner;
use LCM::FileUtils;
use LCM::ResidentInstallerManifest;
use LCM::Component::Installed;
use LCM::Component::Installable;
use LCM::Manifests::GenericManifest;
use SDB::Install::Globals qw( getFlavourProductName $gProductNameInstaller );
use SDB::Install::SysVars qw($currentPlatformName);
use SDB::Install::User;
use File::stat;

use Exporter;
our @ISA = qw (Exporter);
our @EXPORT = qw(getServerPathFromUserInput
                 updateNonResidentWebResources
                 isPathToServerBinaries
                 isInstalledUi5UpToDate
                 isInstalledAdminHelpToDate
                 validateServerSignature
                 validateResidentInstaller
                 getInstalledServerManifest
                 getServerComponentFromPath
                 isVerificationFailureCritical);

sub getServerPathFromUserInput {
	my ($serverPath, $errorMessageList, $ignoreVersion) = @_;
	$ignoreVersion //= 0;
	my $pathCandidateValidation;
	my $rootPathValidation = isServerPathValid($serverPath, $errorMessageList, $ignoreVersion);

	return $serverPath if($rootPathValidation == 0);
	
	for my $pathCandidate (['server'], ['SAP_HANA_DATABASE', 'server'], ['HDB_SERVER_LINUX_X86_64', 'server'], ['DATA_UNITS', 'HDB_SERVER_LINUX_X86_64', 'server']){
		my $serverPathCandidate = File::Spec->catdir($serverPath, @{$pathCandidate});
		my $msgLst = SDB::Install::MsgLst->new();
		my $retCode = isServerPathValid($serverPathCandidate, $msgLst, $ignoreVersion);
		return $serverPathCandidate if($retCode == 0);

		if ($retCode > 2) {
			$pathCandidateValidation = $retCode;
			$errorMessageList->initWithMsgLst($msgLst);
		}
	}

# retCode > 2 means that a kit was found but it has other problems
	my $detectedKitErrorForPathCandidate = ($pathCandidateValidation > 2);
	my $detectedKitErrorForRootPath      = ($rootPathValidation > 2);
	my $hasDetectedKitErrors             = ($detectedKitErrorForPathCandidate || $detectedKitErrorForRootPath);

	my $subdir = _getSingleSubdirPath($serverPath);
	if (!$hasDetectedKitErrors && defined $subdir && $subdir !~ /^(SAP_HANA_DATABASE|HDB_SERVER_LINUX_X86_64|DATA_UNITS)$/) {
		return (getServerPathFromUserInput($subdir, $errorMessageList));
	}
	if (!$hasDetectedKitErrors) {
		$errorMessageList->initMsgLst();
		$errorMessageList->addError("Unable to detect an installation kit in '$serverPath'");
	}
	return undef;
}

sub _getSingleSubdirPath {
	my ($path) = @_;
	my $contents = listDirectory($path);
	my $subdirPath;
	for my $content (@$contents) {
		next if ($content =~ /^(\.|\.\.)$/);
		my $fullPath = File::Spec->catdir($path, $content);
		if (-d $fullPath && defined $subdirPath) {
			return undef;
		} elsif (! -l $fullPath && -d $fullPath && !defined $subdirPath) {
			$subdirPath = $fullPath;
		}
	}
	return $subdirPath;
}

sub isServerPathValid {
	my ($targetServerPath, $errorMessageList, $ignoreVersion) = @_;
	my $serverPathObject = File::stat::stat($targetServerPath);
	$errorMessageList->initMsgLst();

	if(!$serverPathObject || ! -d $serverPathObject){
		$errorMessageList->addError("The specified directory '$targetServerPath' was not found");
		return 1;
	}
	
	if(isPathToServerBinaries($targetServerPath)){
		$targetServerPath = File::Basename::dirname($targetServerPath);
	}

	if(!_isPathToManifest($targetServerPath)) {
		$errorMessageList->addError("Could not find a sub-directory 'server' containing a 'manifest' file in the specified directory '$targetServerPath'");
		return 2;
	}
	
	if(!_isServerUpdatable($targetServerPath, $errorMessageList, $ignoreVersion)){
		return 3;
	}
	if(!_isHdblcmwebExisting($targetServerPath, $errorMessageList)){
		return 4;
	}
	return 0;
}

sub validateServerSignature {
	my ($serverComponent, $errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $sapdsigner = LCM::SAPDSigner->getInstance();
	if (!defined($sapdsigner)) {
		$errorMessageList->addError($gSAPDSignerNotDetectedMessage);
		return 0;
	}

	if(!$sapdsigner->authenticateComponentSet([$serverComponent])) {
		my $verificationErrors = $sapdsigner->getAuthenticationErrors();
		my $serverPath = $serverComponent->getPath();
		my $genericMessage = "Failed to verify the authenticity and integrity of the installation kit in '$serverPath'.";
		my $errorMessage = @{$verificationErrors} ? $verificationErrors->[0]->getSingleFailedComponentMessage() : $genericMessage;
		$errorMessageList->addError($errorMessage);
		return 0;
	}

	return 1;
}

sub getServerComponentFromPath {
	my ($serverPath, $errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $manifestDir = File::Spec->canonpath(File::Spec->catfile($serverPath, 'server'));
	my $manifestPath = File::Spec->catfile($manifestDir, 'manifest');
	my $manifest = LCM::Manifests::GenericManifest->new($manifestPath);

	if ($manifest->errorState()) {
		$errorMessageList->addError("Invalid manifest file '$manifestPath'");
		return undef;
	}

	my $HDBServer = LCM::Component::Installable->new($manifestDir, $manifest, undef, undef, 1);
	if ($HDBServer->errorState()) {
		$errorMessageList->addError("Failed to create component from the installatin kit");
		return undef;
	}

	return $HDBServer;
}

sub validateResidentInstaller {
	my ($errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $instruntimeDir = LCM::Installer->new()->getRuntimeDir();
	my $manifestFile = File::Spec->catfile($instruntimeDir, 'manifest');
	my $manifest = LCM::ResidentInstallerManifest->new($manifestFile);
	if ($manifest->errorState()) {
		$errorMessageList->addError("Failed to read the manifest of the resident installer", $manifest->getErrMsgLst());
		return undef;
	}

	my $installer = LCM::Component::Installed->new($instruntimeDir, $manifest, undef, undef, 1);
	if ($installer->errorState()) {
		$errorMessageList->addError("Failed to create $gProductNameInstaller component", $installer->getErrMsgLst());
		return undef;
	}

	my $sapdsigner = LCM::SAPDSigner->getInstance();
	return $sapdsigner ? $sapdsigner->authenticateInstaller($installer) : 0;
}

sub isVerificationFailureCritical {
    my ($serverComponent, $errList) = @_;
    my $serverSignature = $serverComponent->getSignature();
# Sidadm execution with privileges escalation is impossible without SIGNATURE.SMF
    if (!isAdmin() && !defined($serverSignature)) {
    	my $currentUsername = SDB::Install::User->new()->getname();
    	$errList->addError("Update system cannot be performed as user '$currentUsername' without SIGNATURE.SMF file in the installation kit.");
    	return 1;
    }
    return 0;
}

sub updateNonResidentWebResources {
	my ($serverPath, $errlst) = @_;
	
	my $exec_dir = LCM::Installer->new()->getRuntimeDir(); # <installation_dir>/<SID>/hdblcm/instruntime	
	my $hdblcm_dir = File::Basename::dirname($exec_dir);
	
	$errlst->initMsgLst();
	
	my $currentNonResidentResourcesDir = File::Spec->catdir($hdblcm_dir, 'resources', 'nonresident');
	if (!deltree($currentNonResidentResourcesDir, $errlst)) {
		return undef;
	}	
	
	my $targetResidentResourcesDir = File::Spec->catdir($serverPath, 'resources', 'resident');
	if (!copy_tree($targetResidentResourcesDir,  File::Spec->catdir($currentNonResidentResourcesDir, 'resident'), $errlst)) {
		return undef;
	}
	
	my $targetNonResidentResourcesDir = File::Spec->catdir($serverPath, 'resources', 'nonresident');
	if (!copy_tree($targetNonResidentResourcesDir, $currentNonResidentResourcesDir, $errlst)) {
		return undef;
	}
	
	my $isUi5Available = 0;
	my $currentResidentResourcesDir = File::Spec->catdir($hdblcm_dir, 'resources', 'resident');
	if(isInstalledUi5UpToDate($targetResidentResourcesDir, $currentResidentResourcesDir) &&  eval {symlink("",""); 1} ){
		my $curentResidentUi5Directory = File::Spec->catdir($currentResidentResourcesDir, 'sapui5libs');
		my $curentNonResidentUi5Directory = File::Spec->catdir($hdblcm_dir, 'resources', 'nonresident', 'resident', 'sapui5libs');
		$isUi5Available = symlink($curentResidentUi5Directory, $curentNonResidentUi5Directory);
	}
	if(!$isUi5Available) {
		require SDB::Install::Archive;
		my $archive = new SDB::Install::Archive ( File::Spec->catfile($hdblcm_dir, 'resources', 'nonresident', 'resident', 'sapui5libs.tgz') );
		my $returnCode = $archive->Extract(File::Spec->catfile($hdblcm_dir, 'resources', 'nonresident', 'resident'));
		
		if(!$returnCode){
			$errlst->initWithMsgLst($archive->getErrMsgLst());
			return undef;
		}
	}
	
	chmod(0755, $currentNonResidentResourcesDir);
	return 1;
}

sub isInstalledUi5UpToDate {
	my ($targetDir, $currentDir) = @_;
	my $targetFile = File::Spec->catfile($targetDir, 'ui5version');
	my $currentFile = File::Spec->catfile($currentDir, 'ui5version');
	my ($targetLines, $currentLines) = (readFile($targetFile), readFile($currentFile)); 

	return 0 if(!defined($targetLines) || !defined($currentLines));
	
	chomp(my $targetVersion = $targetLines->[0]);
	chomp(my $currentVersion = $currentLines->[0]);

	return $targetVersion eq $currentVersion;
}

sub isInstalledAdminHelpToDate {
	my ($targetDir, $currentDir) = @_;
	my $targetFile = File::Spec->catfile($targetDir, 'adminhelpversion');
	my $currentFile = File::Spec->catfile($currentDir, 'adminhelpversion');
	my ($targetLines, $currentLines) = (readFile($targetFile), readFile($currentFile));

	return 0 if(!defined($targetLines) || !defined($currentLines));

	chomp(my $targetVersion = $targetLines->[0]);
	chomp(my $currentVersion = $currentLines->[0]);

	return $targetVersion eq $currentVersion;
}

sub isPathToServerBinaries{
	my ($targetServerPath) = @_;
	my $serverManifestFile = File::Spec->catfile($targetServerPath, 'manifest');
	my $serverManifest = new SDB::Install::Manifest($serverManifestFile);

	return $serverManifest->isServer();
}

sub _isPathToManifest {
	my $serverPath = shift();
	my $manifestCandidatePath = File::Spec->catfile($serverPath, 'server', 'manifest');
	my $manifestPathObject = File::stat::stat($manifestCandidatePath);
	return ($manifestPathObject && -f $manifestPathObject) ? 1 : 0;
}

sub _isHdblcmwebExisting {
	my ($targetServerPath, $errorMessageList) = @_;
	
	my $hdblcmWebPath = File::Spec->catfile($targetServerPath, 'hdblcmweb');
	my $hdblcmWebObject = File::stat::stat($hdblcmWebPath);
	my $hdblcmManifestPath = File::Spec->catfile($targetServerPath, 'instruntime', 'manifest');
	
	my $hdblcmManifest = new SDB::Install::Manifest($hdblcmManifestPath);
	my $isHdblcmManifestOk = defined($hdblcmManifest->read(undef, 1));
	my $isHdblcmExececutableOk = ($hdblcmWebObject && (-f $hdblcmWebObject) && (-x $hdblcmWebObject));
	
	if(!$isHdblcmManifestOk){
		$errorMessageList->addError("Unable to detect hdblcm in the specified location", $hdblcmManifest->getErrMsgLst());
		return undef;
	}
	if(!$isHdblcmExececutableOk){
		$errorMessageList->addError("hdblcmweb executable does not exist or does not have the right permissions");
		return undef;
	}
	
	my $isPlatformSupported = $hdblcmManifest->isPlatformSupported($currentPlatformName);
	if(!$isPlatformSupported){
		$errorMessageList->addError("hdblcmweb does not support current OS platform '$currentPlatformName'");
		return undef;
	}

	return 1;
}

sub _isServerUpdatable {
	my ($targetServerPath, $errorMessageList, $ignoreVersion) = @_;
	
	my $installedServerManifest = getInstalledServerManifest($errorMessageList);
	
	if(! defined $installedServerManifest){
		$errorMessageList->addError('Unable to detect SAP HANA Database installation');
		return (undef, $errorMessageList);
	}
	
	my $installedServerVersion = $installedServerManifest->getVersionObject();
	my $installedVersionString = $installedServerVersion->asString();
    my $installedFlavour = $installedServerManifest->getHANAFlavour();
	my $targetServerManifestPath = File::Spec->catfile($targetServerPath, 'server', 'manifest');
	my $targetServerManifest = new SDB::Install::Manifest($targetServerManifestPath);
	my $isTargetServerManifestOk = defined($targetServerManifest->read(undef, 1));
	
    if(!$isTargetServerManifestOk || !$targetServerManifest->isServer() || $installedFlavour ne $targetServerManifest->getHANAFlavour()){   
        my $flavourProductName = getFlavourProductName($installedFlavour);
        $errorMessageList->addError("Error detecting $flavourProductName installation kit in '$targetServerPath'", $targetServerManifest->getErrMsgLst());
        return undef;
    }
	
	my $targetServerVersion = $targetServerManifest->getVersionObject();
	my $targetVersionString = $targetServerVersion->asString();
	my $isTargetOlderThanSource = $installedServerVersion->isNewerThan($targetServerVersion);
	my $isUpdatePossible = $targetServerManifest->checkSPSUpgradeLimitation($installedServerManifest);

	if($isTargetOlderThanSource && !$ignoreVersion){
		$errorMessageList->addError(GetNewerServerAlreadyInstalledMessage($targetVersionString, $installedVersionString));
		return undef;
	}
	if(! $isUpdatePossible ){
		$errorMessageList->addError($gUpgradeRestrictionsMessage, $targetServerManifest->getErrMsgLst());
		return undef;
	}
	
	return 1;
}

sub getInstalledServerManifest {
	my ($errorMessageList, $sapmnt) = @_;
	require LCM::Component;
	require LCM::Installer;

	my $installer = new LCM::Installer();
	my $systemComponentManager = $installer->getOwnSystemComponentManager($sapmnt);
	return undef if(! defined $systemComponentManager);
	my $installedServerComponent = $systemComponentManager->getComponentByKeyName($gKeynameEngine);
	
	if(!defined $installedServerComponent){
		$errorMessageList->addError('Unable to detect installed SAP HANA Database on the system');
		return undef;
	}
	
	return $installedServerComponent->getManifest();
}

1;