package LCM::Component;

use strict;
use warnings;

use SDB::Install::Configuration::ExternalProgramConfiguration;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw ($gProductNameAccelerator
                              $gProductNameEs
                              $gProductNameStreaming
                              $gProductNameRDSync
                              $gProductNameXS2
                              $gProductNameEngine
                              $gProductNameClient
                              $gProductNameStudio
                              $gProductNameAFL
                              $gProductNameLCA
                              $gProductNameHLM
                              $gProductNameSDA
                              $gKeynameEngine
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameAFL
                              $gKeynameLCA
                              $gKeynameHLM
                              $gKeynameSDA
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameRDSync
                              $gKeynameXS
                              $gKeynameXS2
                              $gKeynameOfficeClient
                              $gKeynameInstaller
                              $gKeynameLMStructure
                              $gSignatureManifestName
                              determineSignedManifestRelativePath);
use Exporter;
use base qw (SDB::Install::Base Exporter LCM::Component::Registrable);
use LCM::HLMConfigurator;
use LCM::Component::Registrable qw($RETURN_CODE_SUCCESS $RETURN_CODE_ERROR);
use LCM::TraceLoggingEnvironment;
use SAPDB::Install::Hostname;
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::NewDBUser;
use SDB::Common::Utils qw(trim);
use LCM::Component::Installable::InstallationKitChecker;
use SDB::Install::Tools qw (getFilenameTimeStamp repeat);
use File::Spec;
use File::Basename;
use LCM::Manifests::SAPSignature;
use SDB::Install::HdbInstallerOutputParser qw (parseHdbInstallerErrorMessages);
use SDB::Install::System qw (getSAPDrive);
use File::stat;

use constant{
    PREPARE_PHASE => "prepare",
    OFFLINE_PHASE => "offline",
    ONLINE_PHASE  => "online",
    SYSTEM_ONLINE_PHASE  => "system_online",
    ALL           => "all",
};


our @EXPORT = qw (
    COMPONENT_NAME_INSTALLER
    COMPONENT_NAME_LM_STRUCTURE
    COMPONENT_NAME_COCKPIT_STACK
    STATUS_INITIAL
    STATUS_FINISHED_SUCCESS
    STATUS_FINISHED_ERROR
    STATUS_FINISHED_WARNING
    $componentKeynameToBatchKey
    parseLogFileLocation
    PREPARE_PHASE
    OFFLINE_PHASE
    ONLINE_PHASE
    SYSTEM_ONLINE_PHASE
    ALL
    $componentBatchKeyToKeycaption
    getComponentKeyCaptionByKeyName
);

sub COMPONENT_NAME_INSTALLER    {'Resident hdblcm'}

sub COMPONENT_NAME_LM_STRUCTURE    {'Component List'}

sub COMPONENT_NAME_COCKPIT_STACK   {'Cockpit stack'}

use constant STATUS_INITIAL     => 0;
use constant STATUS_FINISHED_SUCCESS     => 1;
use constant STATUS_FINISHED_ERROR  => 2;
use constant STATUS_FINISHED_WARNING  => 3;

our $componentKeynameToBatchKey = {
		$gKeynameEngine      => "server",
		$gKeynameClient      => "client",
		$gKeynameStudio      => "studio",
		$gKeynameAFL         => "afl",
		$gKeynameLCA         => "lcapps",
		$gKeynameHLM         => "hlm",
		$gKeynameSDA         => "smartda",
		$gKeynameStreaming   => "streaming",
		$gKeynameEs          => "es",
		$gKeynameAccelerator => "ets",
		$gKeynameRDSync      => "rdsync",
		$gKeynameXS2         => "xs",
};

our $internalComponentKeynameToBatchKey = {
	$gKeynameInstaller   => "hdblcm",
	$gKeynameLMStructure => "lm_structure",
};

our $keynameToComponentKeycaption = {
        $gKeynameEngine       => "SAP HANA Database",
        $gKeynameClient       => "SAP HANA Database Client",
        $gKeynameStudio       => "SAP HANA Database Studio",
        $gKeynameAFL          => "SAP HANA AFL (Misc)",
        $gKeynameLCA          => "SAP HANA LCAPPS",
        $gKeynameHLM          => "SAP HANA Lifecycle Manager",
        $gKeynameSDA          => "SAP HANA Smart Data Access",
        $gKeynameStreaming    => "$gProductNameStreaming",
        $gKeynameEs           => "$gProductNameEs",
        $gKeynameAccelerator  => "$gProductNameAccelerator",
        $gKeynameRDSync       => "$gProductNameRDSync",
        $gKeynameXS2          => "$gProductNameXS2",
};

our $componentBatchKeyToKeycaption = {
        $componentKeynameToBatchKey->{$gKeynameEngine}      => $gProductNameEngine,
        $componentKeynameToBatchKey->{$gKeynameClient}      => $gProductNameClient,
        $componentKeynameToBatchKey->{$gKeynameStudio}      => $gProductNameStudio,
        $componentKeynameToBatchKey->{$gKeynameAFL}         => $gProductNameAFL,
        $componentKeynameToBatchKey->{$gKeynameLCA}         => $gProductNameLCA,
        $componentKeynameToBatchKey->{$gKeynameHLM}         => $gProductNameHLM,
        $componentKeynameToBatchKey->{$gKeynameSDA}         => $gProductNameSDA,
        $componentKeynameToBatchKey->{$gKeynameStreaming}   => $gProductNameStreaming,
        $componentKeynameToBatchKey->{$gKeynameEs}          => $gProductNameEs,
        $componentKeynameToBatchKey->{$gKeynameAccelerator} => $gProductNameAccelerator,
        $componentKeynameToBatchKey->{$gKeynameRDSync}      => $gProductNameRDSync,
        $componentKeynameToBatchKey->{$gKeynameXS2}         => $gProductNameXS2,
};

sub new {
    my ( $class, $manifestDir, $manifest, $componentManager, $instconfig, $isSidAdmUserExecution ) = @_;
    my $self = $class->SUPER::new ();

	$manifestDir = File::Spec->canonpath($manifestDir);

    $self->{manifestDir} = $manifestDir;
    $self->{manifest} = $manifest;
    $self->{componentManager} = $componentManager;
    $self->{instconfig} = $instconfig;
	$self->setSidAdmUserExecution($isSidAdmUserExecution);
    $self->setStatus(STATUS_INITIAL);
    $self->{warnings} = [];

    return $self;
}

sub getAllComponentsBatchKeys {
	my ($isUninstall) = @_;

	my @componentValues = (
		$componentKeynameToBatchKey->{$gKeynameClient},
		$componentKeynameToBatchKey->{$gKeynameEs},
		$componentKeynameToBatchKey->{$gKeynameAccelerator},
		$componentKeynameToBatchKey->{$gKeynameLCA},
		$componentKeynameToBatchKey->{$gKeynameEngine},
		$componentKeynameToBatchKey->{$gKeynameSDA},
		$componentKeynameToBatchKey->{$gKeynameStreaming},
		$componentKeynameToBatchKey->{$gKeynameRDSync},
		$componentKeynameToBatchKey->{$gKeynameXS2},
		$componentKeynameToBatchKey->{$gKeynameStudio},
	);

	push (@componentValues, $componentKeynameToBatchKey->{$gKeynameHLM}) if ($isUninstall);
	push (@componentValues, $componentKeynameToBatchKey->{$gKeynameAFL});

	my $productSpecificAFLs = ['sca', 'sop', 'eml', 'rme', 'rtl', 'trp', 'vch'];

	push (@componentValues, @$productSpecificAFLs);

	return @componentValues;
}

sub addComponentInfosToMsgLst{
    my ($self,$msglst) = @_;
    $msglst->addMessage ("Product: ".$self->getComponentName());
    $msglst->addMessage ("Version: ".$self->{manifest}->getVersion());
    my $gitHash = $self->{manifest}->getValue ('git-hash');
    if (defined $gitHash){
       $msglst->addMessage ("GitHash: " . $gitHash );
    }
}

sub getPath{
    return $_[0]->{manifestDir};
}

sub setPath{
    $_[0]->{manifestDir} = $_[1];
}

sub selectComponent{
    $_[0]->{isSelected} = defined $_[1] ? $_[1] : 1;
}

sub isComponentSelected{
    return $_[0]->{isSelected};
}

sub canSelectComponent{
	return defined $_[0]->{canSelect} ? $_[0]->{canSelect} : $_[0]->isInternal() ? 0 : 1;
}

sub setCanSelectComponent{
	$_[0]->{canSelect} = $_[1];
}

sub canUpgradeComponent{
    return $_[0]->{canUpgrade};
}

sub setCanUpgradeComponent {
    $_[0]->{canUpgrade} = $_[1];
}

sub isInternal{
	return 0;
}

sub getManifest{
    return $_[0]->{manifest};
}

sub setManifest{
    $_[0]->{manifest} = $_[1];
}

sub getVersion{
	return $_[0]->{manifest}->getVersion();
}

sub getVersionObject {
    return $_[0]->getManifest()->getVersionObject();
}

sub getComponentName{
    return $_[0]->getManifest ()->getValue ('keycaption');
}

sub getComponentKeyName{
    return $_[0]->getManifest ()->getValue ('keyname');
}

sub getComponentId{
    return $_[0]->getManifest ()->getCompId ();
}

sub getComponentManager{
    return $_[0]->{componentManager};
}

sub getComponentBatchKey{
	my $self = shift;
	my $key = $self->getManifest()->getComponentKey();
	
	if (defined $key) {
		return $key;
	}

	my $batchKey = $componentKeynameToBatchKey->{$self->getComponentKeyName()};
	if (defined $batchKey){
		return $batchKey;
	}

	return $internalComponentKeynameToBatchKey->{$self->getComponentKeyName()};
}

sub getComponentKeyCaptionByKeyName{
    my $caption = $keynameToComponentKeycaption->{$_[1]};
    if (not defined $caption) {
       $caption = $_[1];
    }
    return $caption;
}

sub getNumberOfExpectedOutputLines{
    return undef;
}

sub getIgnorePersistency{
    return $_[0]->{ignorePersistency};
}

sub setIgnorePersistency{
    $_[0]->{ignorePersistency} = $_[1];
}

sub setProgressHandler{
    $_[0]->{progressHandler} = $_[1];
}

sub setLogLocation{
	my ($self, $logLocation) = @_;
	my $phase = $self->getPhase();

	$self->{"logLocation"} = $logLocation;

	if (defined $phase) {
		if ($phase eq PREPARE_PHASE) {
			$self->setLogLocationForPreparePhase($logLocation);
		} elsif ($phase eq OFFLINE_PHASE) {
			$self->setLogLocationForOfflinePhase($logLocation);
		} elsif ($phase eq ONLINE_PHASE) {
			$self->setLogLocationForOnlinePhase($logLocation);
		} elsif ($phase eq SYSTEM_ONLINE_PHASE) {
            $self->setLogLocationForSystemOnlinePhase($logLocation);
        }
	}
}

sub getLogLocation{
    return $_[0]->{"logLocation"};
}

sub setLogLocationForPreparePhase{
    $_[0]->{"logLocationPreparePhase"} = $_[1];
}

sub getLogLocationForPreparePhase{
    return $_[0]->{"logLocationPreparePhase"};
}

sub setLogLocationForOfflinePhase{
    $_[0]->{"logLocationOfflinePhase"} = $_[1];
}

sub getLogLocationForOfflinePhase{
    return $_[0]->{"logLocationOfflinePhase"};
}

sub setLogLocationForOnlinePhase{
    $_[0]->{"logLocationOnlinePhase"} = $_[1];
}

sub getLogLocationForOnlinePhase{
    return $_[0]->{"logLocationOnlinePhase"};
}

sub setLogLocationForSystemOnlinePhase{
    $_[0]->{"logLocationSystemOnlinePhase"} = $_[1];
}

sub getLogLocationForSystemOnlinePhase{
    return $_[0]->{"logLocationSystemOnlinePhase"};
}

sub setStatus{
    $_[0]->{"status"} = $_[1];
}

sub getStatus{
    return $_[0]->{"status"};
}

sub initProgressHandler{
    my ($self) = @_;
    if (!defined $self->{progressHandler}){
        return 1;
    }
    my $numberOfLines = $self->getNumberOfExpectedOutputLines ();
    if (defined $numberOfLines){
        $self->{progressHandler}->InitProgress ($numberOfLines, 0);
    }
    my $identation = $self->{progressHandler}->getIntendationDepth() if $self->{progressHandler}->can('getIntendationDepth');
    $self->{progressHandler}->setIntendationDepth($identation+1) if $self->{progressHandler}->can('setIntendationDepth');;
}

sub getProgressHandler{
    return $_[0]->{progressHandler};
}


sub getExternalHdbInstallerConfiguration{
    my ($self) = @_;
    my $instconfig = new SDB::Install::Configuration::ExternalProgramConfiguration (
                            $self->getHdbInstallExecutable(),
                            ["--archive_dir=$self->{manifestDir}"]);
    my $msg = $self->getMsgLst ()->addMessage ("Detecting known command line options");
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    my $rc = $instconfig->init();
    if (!defined $rc){
        $self->setErrorMessage ("Cannot detect command line options dynamically",
                                    $instconfig->getErrMsgLst ());
        return undef;
    }
    if (!$rc) {
        return $self->getFallbackHdbInstallerConfiguration();
    }
    return $instconfig;
}

sub getFallbackHdbInstallerConfiguration {
    ...
}


#
# get hdbinst output and parse for top level error messages
# returns a SDB::Install::MsgLst containing the error messages
#

sub getHdbInstallerErrorMessages{
    my ($self, $outputLines) = @_;
    return parseHdbInstallerErrorMessages ($outputLines);
}

sub parseLogFileLocation {
	my ($self, $outputLines) = @_;
	if (!defined $outputLines){
		return undef;
	}
	my $log_location;
	my $pattern = "Log file written to \'(.*?)\'";
	foreach my $line (reverse @$outputLines){
		( $log_location ) = $line =~ /$pattern/i;
		
		if (defined $log_location) {
			$log_location =~ s/\.log/\.msg/g;
			if(-f $log_location){
			     return $log_location;
			}
			$log_location =~ s/\.msg/\.log/g;
			return $log_location;
		}
	}
}


#
# Make sure that logs and traces are not overwritten by several HDB installers.
# Append _<component keyName> to global log file copy and installer trace, which
# are defined in the process environment.
#

sub prepareHdbInstallerEnvironment{
	my($self,$componentKeyName) = @_;
    return LCM::TraceLoggingEnvironment::PrepareComponentHdbInstallerEnvironment($self,$componentKeyName);
}


# Returns the path (e.g. Linux: /hana/shared/<SID>', Windows: '/usr/sap/<SID>')
sub getInstSidPath {
    my ( $self, $instconfig ) = @_;

    my $instPath = undef;
    my $sid      = $instconfig->getValue('SID');

    if($isWin) {
        my $drive = $instconfig->getValue('Drive');
        if (!defined $drive) {
            $drive = getSAPDrive ($self->getErrMsgLst ());
            if (!defined $drive){
                return undef;
            }
        }
        $instPath = File::Spec->catfile($drive, 'usr', 'sap', $sid);
    }
    else {
        my $target = $instconfig->getValue('Target');
        $instPath = File::Spec->catfile($target, $sid);
    }
    return $instPath;
}

#----------------------------------------------------------------------------
#
# Configure HDB components.
#
# $_[0] - self
# $_[1] - instconfig
# $_[2] - ref to array containing hdbcomponents that are to be configured
# the name is used to form the path to the component.
# For example 'hdbclinet' is the name of the hdbclient and it's directory
# 
# $_[3] - shared directory
# $_[4] - SID
# $_[5] - LCM::HDBComponentsConfigurator action
# $_[6] - fail message
# $_[7] - remote hosts list or undef if the system is not distributed
#
sub configureHdbComponents {
    my ($self, $loggerClass, $componentNames, $sharedDir, $sid, $action) = @_;
    my $configurator = $self->getHDBComponentsConfigurator ();
    my $saveCntxt = $self->setMsgLstContext([$loggerClass->getMsgLst ()]);
    my $rc;

    $rc = $configurator->configureLocalHdbComponents ( $componentNames, $action, $sharedDir, $sid );
    $self->setMsgLstContext($saveCntxt);
    if($self->hasError ($rc, $self->getComponentsConfigurationFailedMsg($componentNames, $action))){
     	$self->setMsgLstContext($saveCntxt);
       	return undef;	
     }
    return 1;
}

#----------------------------------------------------------------------------
#
# Configures HLM instance.
#
sub registerHLM {
    my ( $self, $logger, $hlmInstallationDir, $sid, $instanceNumber ) = @_;

    my $configurator = $self->getHLMConfigurator ();
    my $saveCntxt = $self->setMsgLstContext([$logger->getMsgLst ()]);
    my $rc;
    my $failMessage = "Failed to register HANA Lifecycle Manager";
    
    $rc = $configurator->registerLocalHLMInstance ( $hlmInstallationDir, $sid, $instanceNumber );
    $self->setMsgLstContext($saveCntxt);
    $self->hasError ( $rc, $failMessage . ' on the localhost.' );
    
    return 1;
}

#----------------------------------------------------------------------------
#
# Configures HLM instance.
#
sub unregisterHLM {
    my ( $self, $logger, $hlmInstallationDir, $sid, $instanceNumber ) = @_;
    my $configurator = $self->getHLMConfigurator ();
    my $saveCntxt = $self->setMsgLstContext([$logger->getMsgLst ()]);
    my $rc;
    my $failMessage = "Failed to unregister HANA Lifecycle Manager";
    
    $rc = $configurator->unregisterLocalHLMInstance ( $hlmInstallationDir, $sid, $instanceNumber);
    $self->setMsgLstContext($saveCntxt);
    $self->hasError ( $rc, $failMessage . ' on the localhost.' );
    
    return 1;
}

sub getHLMConfigurator {
    # do not set log location to the HLM log file
    my $doNotSetLogLocation = 1;
    return LCM::HLMConfigurator->new ( $_[0], $doNotSetLogLocation, undef );
}

sub getHDBComponentsConfigurator {
    # do not set log location to the hdbclientreg log file
    my $doNotSetLogLocation = 1;
    return LCM::HDBComponentsConfigurator->new ( $_[0], $doNotSetLogLocation );
}

sub getComponentsConfigurationFailedMsg {
    my ($self, $componentNames, $action) = @_;
    my $componentCaptions = join ', ', @{$componentNames};
    my $msgTempalte = "Failed to %s %s on the localhost.";
    ($action) = $action =~ /^action_(.+)$/g;
    $action //= 'configure';

    return sprintf($msgTempalte, $action, $componentCaptions);
}

sub isLSS{
    my ($self) = @_;
    my $manifest = $self->getManifest();
    return 1 if (defined $manifest && $manifest->isLSS());
    return 0;
}

sub isServer{
	my ($self) = @_;
	my $manifest = $self->getManifest();
	return 1 if (defined $manifest && $manifest->isServer());
	return 0;
}

sub isServerPlugin{
	my ($self) = @_;
	my $manifest = $self->getManifest();
	if (defined $manifest && $manifest->isServerPlugin()) {
		return 1;
	}
	my $keyname = $self->getComponentKeyName();
	if ($keyname eq $gKeynameAFL || $keyname eq $gKeynameLCA) {
		return 1;
	} else {
		return 0;
	}
}

sub hasError {
    my ( $self, $rc, $message, $componentHandler ) = @_;
    my $logger = $componentHandler ? $componentHandler : $self;
    if ( !defined $rc ) {        
        $logger->getErrMsgLst()->addMessage($message);
        if($logger->can("getProgressHandler"))
        {
            my $progressHandler = $logger->getProgressHandler();
            if(defined $progressHandler && $progressHandler->can("StepFinished"))
            {
                $progressHandler->StepFinished($message);
            }
        }
        return 1;
    }
    return 0;
}

sub addWarning {
	my ( $self, $warning ) = @_;
    
    push (@{$self->{warnings}}, $warning);
}

sub getWarnings {
	my ($self) = @_;
	return $self->{warnings};
}

sub resetWarnings {
	my ($self) = @_;
	$self->{warnings} = [];
}

sub setPersistedVersion {
	my ( $self, $persistedVerison ) = @_;
	$self->{_persistedVersion} = $persistedVerison;
}

sub getPersistedVersion {
	my ($self) = @_;
	return $self->{_persistedVersion};
}

sub _executeThroughSHAOperation {
	my ($self, $instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs, $action) = @_;

	my $rc = 1;
	my $hostname = hostname();

	my $cmpName = $self->getComponentName();
	my $progress_message = "$action of $cmpName on host '%s'...";
	my $done_message     = "Successful $action of $cmpName on host '%s'.";
	my $error_message    = "$action of $cmpName on host '%s' failed!";

	my @host = ($hostname);
	my $remoteExecution = new SDB::Install::RemoteHostctrlHosts(@host);

	if ($isShowProgressMsgs) {
		$remoteExecution->setOutputHandler ($hostname, $self->getProgressHandler ());
	}

	my $password = $instconfig->getValue('Password');
    if (defined $password) {
		$remoteExecution->useSidadm();
	}

	my $msg = $self->getMsgLst ()->addMessage ("$action of $cmpName through SAP Host Agent Operation.");
	$remoteExecution->resetMsgLstContext ();
	my $saveCntxt = $remoteExecution->setMsgLstContext ([$msg->getSubMsgLst()]);

	my $remoteExecutionReturnCode = $remoteExecution->executeHostctrlParallel($operation,
                                                        $instconfig, # instconfig
                                                        undef, # param IDs
                                                        $passwordKeys, # password IDs
                                                        undef, # remote ignore
                                                        undef, # remote timeout
                                                        $progress_message, # progress_message,
                                                        $done_message, # done_message,
                                                        $error_message, # error_message,
                                                        $optionMap,
                                                        undef, # host option map
                                                        [$hostname], # only on hosts
                                                        undef, # do not fail
														1, # Suppress Console msgs
														);
	if ($self->_hasError($remoteExecution->getExitCode(0))) { # retrieve the actual exit code from the operation
            $instconfig->PushError ("$action of $cmpName through SAP Host Agent Operation failed!", $remoteExecution);
            $rc = undef;
	}

	$self->setLogLocation($self->parseLogFileLocation($remoteExecution->getOutputBuffer()));
	$remoteExecution->setMsgLstContext ($saveCntxt);

	return $rc;
}

sub _hasError {
	my ($self, $returnCode) = @_;
	return !defined($returnCode) || ($returnCode != 0);
}

sub buildSHAOptionsMap {
	my ($self, $instconfig, $optionMap, $args, $isChangeHdbLogsOwnership) = @_;

	if (not defined $optionMap) {
		$optionMap = {};
	}

	for my $arg (@$args) {
		$arg =~ s/^(-{1,2})//;
		my @optionValueArr = split(/=/, $arg);
		my $option = $optionValueArr[0];
		my $value = $optionValueArr[1];

		if (($self->_isParamTypeBoolean($instconfig, $option)) and (not defined $value)) {
				$value = 'on';
		}

		if($option eq 'machine_utilization'){
			$optionMap->{'M_U'} = $value;
			next;
		}

		if($option eq 'skip_hostagent_password'){
			$optionMap->{'S_H_P'} = $value;
			next;
		}
		if($option eq 'skip_modify_sudoers'){
			$optionMap->{'SUD'} = $value;
			next;
		}

		$optionMap->{uc($option)} = $value;
	}

	if ($isChangeHdbLogsOwnership) {
		my $user = new SDB::Install::NewDBUser($instconfig->getValue('SID'));
		my $uid = $user->uid();
		my $gid = $user->gid();
		if (defined $uid && defined $gid) {
			$optionMap->{'INSTLOG_UID'} = $uid;
			$optionMap->{'INSTLOG_GID'} = $gid;
		}
	}

	return $optionMap;
}

sub _isParamTypeBoolean {
    my ($self, $instconfig, $option) = @_;

    for my $paramKey (keys %{$instconfig->{params}}){
        my $opt = $instconfig->{params}->{$paramKey}->{opt};
        my $type = $instconfig->{params}->{$paramKey}->{type};
        next if (!defined($opt) || !defined($type));

        if ((index($opt, $option) != -1) && ($type eq 'boolean')) {
            return 1;
        }
    }
    return 0;
}

sub register{
	return $RETURN_CODE_SUCCESS;
}

sub unregister{
	return $RETURN_CODE_SUCCESS;
}

sub isCompatibleWithScopeInstance {
	return 1;
}

sub isSidAdmUserExecution {
	my ($self) = @_;
	return ($self->{_isSidAdmUserExecution}) ? 1 : 0;
}

sub setSidAdmUserExecution {
	my ($self, $value) = @_;
	$self->{_isSidAdmUserExecution} = $value;
}

sub shallExecuteUpdateHost {
	return 0;
}

sub _getUnsupportedDbFeatures {
    my ( $self ) = @_;
    if(defined $self->{_unsupportedDbFeaturesMap}){
    	return $self->{_unsupportedDbFeaturesMap};
    }
    $self->{_unsupportedDbFeaturesMap} = ();
    my $manifest = $self->getManifest();
    my $unsuportedFeatureValue = $manifest->getValue("unsupported-db-features");
    if(defined $unsuportedFeatureValue){
	    my @unsuportedFeatureList = split( ',',$unsuportedFeatureValue );
	    %{$self->{_unsupportedDbFeaturesMap}} = map { trim($_) => 1 } @unsuportedFeatureList;
    }
    return $self->{_unsupportedDbFeaturesMap};
}

sub isFeatureUnsupported{
	my($self,$feature) = @_;
	my $unsupportedFeatureMap = $self->_getUnsupportedDbFeatures();
	my $isUnsupported = exists($unsupportedFeatureMap->{trim($feature)});
	return $isUnsupported;
}

sub getHostRoles {
	return [];
}

sub createTimestamp {
    return getFilenameTimeStamp();
}

sub isInPendingState {
	return 0;
}


sub forceSelectComponent {
	my ($self) = @_;

	$self->selectComponent();

	my $paramComponents  = $self->getCommandLineComponents();
	my $alreadySpecified = 0;

	if (defined $paramComponents) {
		my $key = $self->getComponentBatchKey();
		for my $currCompName (split (',', lc($paramComponents))) {
			if ($key eq $currCompName) {
				$alreadySpecified = 1;
				last;
			}
		}
	}
	if (!$alreadySpecified) {
		$self->setCanSelectComponent(0);
	}
}


sub getCommandLineComponents {
	local @ARGV = @ARGV; # local copy of ARGV
	my $paramValue;
	Getopt::Long::Configure ('pass_through');
	Getopt::Long::GetOptions ('components=s' => \$paramValue);
	return $paramValue;
}

sub getProgressMsg {
	...
}

sub setPhase {
    $_[0]->{phase} = $_[1];
}

sub getPhase {
    return $_[0]->{phase};
}

sub getDefaultPhase {
	return ONLINE_PHASE;
}

sub supportsPhases {
	return scalar @{$_[0]->getSupportedPhases()} > 0;
}

sub getSupportedPhases {
	my ($self) = @_;
	return $self->getManifest()->getSupportedPhases();
}

sub isSystemRestartRequired {
	my ($self) = @_;
	return $self->getManifest()->isSystemRestartRequired();
}

sub isInstanceRestartRequired {
	my ($self) = @_;
	return $self->getManifest()->isInstanceRestartRequired();
}

sub getServicesToRestart {
    my ($self) = @_;
    return $self->getManifest()->getServicesToRestart();
}

sub isInstanceStopRequired {
    my ($self, $configuration) = @_;
    return 0 if $self->isComponentUpdated($configuration) && !$self->isUpdateToNewerVersion();
    return 0 if !$self->isInstanceRestartRequired();
    return $self->_isStopRequired($configuration);
}

sub isSystemStopRequired {
    my ($self, $configuration) = @_;
    return 0 if $self->isComponentUpdated($configuration) && !$self->isUpdateToNewerVersion();
    return 0 if !$self->isSystemRestartRequired();
    return $self->_isStopRequired($configuration);
}

sub _isStopRequired {
	my ($self, $configuration) = @_;

	if ($self->_canUsePersistency($configuration)) {
		return !$self->isNextPersistedPhase(ONLINE_PHASE, $configuration);
	}

	return 1;
}

sub _canUsePersistency {
	my ($self, $configuration) = @_;

	if ($self->can('hasPersistenceFile') && 
		$self->hasPersistenceFile($configuration) && 
		!$self->getIgnorePersistency() && 
		!$configuration->getIgnore('check_pending_upgrade')) {
		return 1;
	}

	return 0;
}

sub canSetDependencyPhase {
    my( $self ) = @_;
    my $supportedPhasesCount = scalar @{$self->getSupportedPhases()};
    return 1 if $supportedPhasesCount == 0;
    if($supportedPhasesCount == 1 and $self->supportsPhase(ONLINE_PHASE)){
        return 0;
    }
    return 1;
}

sub supportsPhase {
	my($self, $phase) = @_;
	return 0 if (!defined $phase);
	return ( grep {$_ eq $phase} @{$self->getSupportedPhases()} ) ? 1 : 0 ;
}

sub canBeCalledInPhase {
	my($self, $phase, $configuration) = @_;
	my $manifest = $self->getManifest();

	if ($self->_canUsePersistency($configuration)) {
		my $persistency = $self->getPersistenceXMLObject($configuration);
		return 0 if (!defined $persistency);
		my $nextPhase = $persistency->getNextPhaseString();
		return 0 if (!defined $nextPhase);
		return 0 if (!$self->isCurrentEqualOrAfterPersistedPhase($phase, $nextPhase));
	}

	return 0 if $self->isComponentUpdated($configuration) && !$self->isUpdateToNewerVersion();
	return 1 if $self->supportsPhase(ALL);
	return 1 if $self->supportsPhase($phase);
	my $componentPhase = defined $self->getPhase() ? $self->getPhase() : $self->getDefaultPhase();
	return $phase eq $componentPhase;
}

sub isCurrentEqualOrAfterPersistedPhase {
	my ($self, $currentPhase, $persistedNextPhase) = @_;

    return 1 if ($persistedNextPhase eq '<unknown>');

	if ($currentPhase eq PREPARE_PHASE) {
		return ($persistedNextPhase eq PREPARE_PHASE) ? 1 : 0;
	}
	if ($currentPhase eq OFFLINE_PHASE) {
		return (($persistedNextPhase eq PREPARE_PHASE) || ($persistedNextPhase eq OFFLINE_PHASE)) ? 1 : 0;
	}
	if (($currentPhase eq ONLINE_PHASE)) {
		return (($persistedNextPhase eq PREPARE_PHASE) || ($persistedNextPhase eq OFFLINE_PHASE) || ($persistedNextPhase eq ONLINE_PHASE)) ? 1 : 0;
	}
    if (($currentPhase eq SYSTEM_ONLINE_PHASE) || ($currentPhase eq ALL)) {
        return (($persistedNextPhase eq PREPARE_PHASE) || ($persistedNextPhase eq OFFLINE_PHASE) || ($persistedNextPhase eq ONLINE_PHASE) || ($persistedNextPhase eq SYSTEM_ONLINE_PHASE)) ? 1 : 0;
    }
	return 0;
}

sub isNextPersistedPhase {
    my ($self, $phase, $configuration) = @_;
    my $persistency = $self->getPersistenceXMLObject($configuration);
    return 0 if !defined $persistency;
    return $persistency->getNextPhaseString() eq $phase;
}

sub isComponentUpdated{
	my ($self, $configuration) = @_;
    my $componentBatchKey = $self->getComponentBatchKey();
    return 0 if !$componentBatchKey;

    my $persistedSteps = $configuration->getPersistedSteps();
    return 0 if !$persistedSteps;

    my @components = split(",", $persistedSteps);
    foreach my $component (@components){
        return 1 if($componentBatchKey eq $component);
    }
    return 0;
}

sub isInstalled{
	...
}

sub _getSupportedDbFeatures {
    my ( $self ) = @_;
    if(defined $self->{_supportedDbFeaturesMap}){
    	return $self->{_supportedDbFeaturesMap};
    }
    $self->{_supportedDbFeaturesMap} = ();
    my $manifest = $self->getManifest();
    my $supportedFeatureValue = $manifest->getValue("supported-db-features");
    if(defined $supportedFeatureValue){
	    my @supportedFeatureList = split( ',',$supportedFeatureValue );
	    %{$self->{_supportedDbFeaturesMap}} = map { trim($_) => 1 } @supportedFeatureList;
    }
    return $self->{_supportedDbFeaturesMap};
}

sub isFeatureSupported{
	my($self,$feature) = @_;
	my $unsupportedFeatureMap = $self->_getUnsupportedDbFeatures();
	my $isUnSupported = exists($unsupportedFeatureMap->{trim($feature)});
	return 0 if($isUnSupported);
	my $supportedFeatureMap = $self->_getSupportedDbFeatures();
	my $isSupported = exists($supportedFeatureMap->{trim($feature)});
	return $isSupported;
}

sub _calculateSignaturePath {
    my ($self) = @_;
    my $manifestDir = $self->getPath();
    if (!defined $manifestDir) {
        return undef;
    }

    for my $offset (1, 2) { # One or two dirs back
        my $candidate = File::Spec->catfile(repeat(\&dirname, $offset, $manifestDir), $gSignatureManifestName);
        my $statObject = File::stat::stat($candidate);
        return $candidate if ($statObject && -f $statObject && -r $statObject);
    }

    return undef;
}

sub getSignature {
    my ($self, $noCache) = @_;
    if (!exists($self->{signature}) || $noCache) {
        my $signaturePath = $self->_calculateSignaturePath();
        $self->{signature} = $signaturePath ? LCM::Manifests::SAPSignature->new($signaturePath) : undef;
    }
    return $self->{signature};
}

1;

