package LCM::Component::Installable;

use strict;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Version;
use SDB::Install::RemoteHosts;
use SDB::Install::Globals qw ($gProductNameEngine $gProductNameClient determineSignedManifestRelativePath);
use SAPDB::Install::Hostname;
use File::Basename qw (dirname);
use LCM::Component;
use LCM::FileUtils;
use LCM::Persistence::XML;
use LCM::Component::Installable::InstallationKitChecker;

use experimental qw (smartmatch);

our @ISA = qw (LCM::Component);

our $hdbInstallExecutable = 'hdbinst';
our $hdbUpdateExecutable = 'hdbinst';

if ($isWin){
    $hdbInstallExecutable .= '.exe';
    $hdbUpdateExecutable .= '.exe';
}

sub new {
    my $self = shift->SUPER::new (@_);
    my $manifest = $self->{manifest};
    if (!defined $manifest){
        return $self;
    }
    my $classOfComponent;
    my $isSidAdmExecution = $self->isSidAdmUserExecution();
    if ($manifest->isServer()){
        $classOfComponent = ($isSidAdmExecution) ?
        					'LCM::Component::Installable::HDBServer::HDBServerSidAdm' : 
        					'LCM::Component::Installable::HDBServer::HDBServerRoot';
    } elsif ($manifest->isClient()){
        $classOfComponent = ($isSidAdmExecution) ?
        					'LCM::Component::Installable::HDBClient::HDBClientSidAdm' : 
        					'LCM::Component::Installable::HDBClient::HDBClientRoot';
    } elsif ($manifest->isStudio()){
        $classOfComponent = 'LCM::Component::Installable::HDBStudio';
    } elsif ($manifest->isLCAPPS()){
		$classOfComponent = 'LCM::Component::Installable::HDBServerPlugin';
    } elsif ($manifest->isAFL()){
        $classOfComponent = 'LCM::Component::Installable::HDBServerPlugin';
    } elsif ($manifest->isHLM()){
        $classOfComponent = 'LCM::Component::Installable::HLM';
    } elsif ($manifest->isSDA()){
        $classOfComponent = 'LCM::Component::Installable::SDA';
    } elsif ($manifest->isInstaller()){
        $classOfComponent = ($isSidAdmExecution) ?
        					'LCM::Component::Installable::ResidentInstaller::ResidentInstallerSidAdm' : 
        					'LCM::Component::Installable::ResidentInstaller::ResidentInstallerRoot';
    } elsif ($manifest->isLMStructure()){
        $classOfComponent = 'LCM::Component::Installable::LMStructure';
    } elsif ($manifest->isCockpitStack()){
        $classOfComponent = 'LCM::Component::Installable::CockpitStack';
    } elsif ($manifest->isStreaming()){
        $classOfComponent = 'LCM::Component::Installable::Streaming';
    } elsif ($manifest->isES()){
        $classOfComponent = 'LCM::Component::Installable::ES';
    } elsif ($manifest->isAccelerator()){
        $classOfComponent = 'LCM::Component::Installable::Accelerator';
    } elsif ($manifest->isRDSync()){
        $classOfComponent = 'LCM::Component::Installable::RDSync';
    } elsif ($manifest->isXS2()){
        $classOfComponent = 'LCM::Component::Installable::XS2';
    } elsif ($manifest->isXS2Application()){
        $classOfComponent = 'LCM::Component::Installable::XS2Application';
    } elsif ($manifest->isServerPlugin()){
        my $componentKey = $manifest->getComponentKey();
        if(!$componentKey){
	    	$self->setErrorMessage("Component manifest of '" . $self->getComponentName() . "' is missing property 'component-key'.");
	        return $self;
    	}
        $classOfComponent = 'LCM::Component::Installable::HDBServerPlugin';
    } elsif ( $manifest->isReferenceData() ){
    	$classOfComponent = 'LCM::Component::Installable::ReferenceData';
    } elsif ( $manifest->isLSS() ){
        $classOfComponent = 'LCM::Component::Installable::LSS';
    }
    if (!defined $classOfComponent){
        $self->addWarning('Unknown component ' . $self->getComponentName());
        return $self;
    }

    eval("require $classOfComponent;");
    if ($@){
        $self->setErrorMessage ("Cannot load component class '$classOfComponent': $@");
        return $self;
    }
    bless ($self, $classOfComponent);
    return $self;
}

sub isResumable {
	my ( $self, $instconfig ) = @_;
	my $isComponentResumable = $self->getComponentName() ~~ @{$self->getResumableComponentNames()} ? 1 : 0;
	return $isComponentResumable;
}

sub getResumableComponentNames {
	return [$gProductNameEngine, ];	
}

sub hasPersistenceFile {
	my ($self, $configuration) = @_;
	my $sapSystem = $configuration->getSAPSystem();

	return undef if(!defined($sapSystem));

	return -e $self->_getPersistenceFilePath($sapSystem);
}

sub getPersistenceXMLObject {
	my ($self, $configuration) = @_;

	if (defined $self->{_persistenceXML}) {
		return $self->{_persistenceXML};
	}

	my $sapSystem = $configuration->getSAPSystem();

	return undef if(!defined($sapSystem));

	my $filePath = $self->_getPersistenceFilePath($sapSystem);
	return undef if ! defined $filePath;

	my $persistenceXMLObject = LCM::Persistence::XML->new($filePath);

	if ($persistenceXMLObject->errorState()) {
		$configuration->getMsgLst()->addWarning($persistenceXMLObject->getErrorString());
	}

	$self->{_persistenceXML} = $persistenceXMLObject;

	return $persistenceXMLObject;
}

sub getAction {
	return $_[0]->isUpdate() ? 'Update' : 'Install';
}

sub getProgressiveMessage {
	return $_[0]->isUpdate() ? 'Updating' : 'Installing';
}

sub _setMsg {
	$_[0]->{msg} = $_[1];
}

sub _getMsg {
	return $_[0]->{msg}; 
}

sub _setSaveContext {
	$_[0]->{saveContext} = $_[1];
}

sub _getSaveContext {
	return $_[0]->{saveContext}; 
}

sub _buildArgs {
	my ($self, $instconfig) = @_;
	my $isUsingPhases = $instconfig->usePhases();
	my $currentPhase = $isUsingPhases ? $self->getPhase() : '';
	my $isServerComponent = $self->isServer();

	if($isUsingPhases && $self->supportsPhase($currentPhase)){
		return $isServerComponent ? [ "--phase=$currentPhase" ] : [ '-phase', $currentPhase ];
	}
	return [];
}

sub preCheckInstallComponent {
    my ( $self, $instconfig ) = @_;
    $self->setErrorMessage ("preCheckInstallComponent not implemented");
    return undef;
}

sub preCheckUpdateComponent {
    my ( $self, $instconfig ) = @_;
    $self->setErrorMessage ("preCheckUpdateComponent not implemented");
    return undef;
}

sub installComponent {
    $_[0]->setErrorMessage ("installComponent not implemented");
    return undef;
}

sub updateComponent {
    $_[0]->setErrorMessage ("updateComponent not implemented");
    return undef;
}

sub getHdbInstallExecutable{
    my ($self) = @_;
    return File::Spec->catfile($self->getInstallerDir(), $hdbInstallExecutable);
}

sub getHdbUpdateExecutable{
    my ($self) = @_;
    return File::Spec->catfile($self->getInstallerDir(), $hdbUpdateExecutable);
}

sub getInstallAnywareExecutable{
    my ($self) = @_;
    return File::Spec->catfile($self->getPath(), 'setup.bin');
}

sub getInstallActionString{
    return $_[0]->isUpdate () ? 'Updating' : 'Installing';
}

sub isUpdate{
    my ($self) = @_;
    if(defined $self->{_isUpdate}) {
		return $self->{_isUpdate};
    }
    return (defined $self->getInstalledComponent()) ? 1 : 0;
}

sub setIsUpdate{
	my ($self, $isUpdate) = @_;
	$self->{_isUpdate} = $isUpdate;
}

sub isUpdateToNewerVersion {
    my($self) = @_;
    my $installedComponent = $self->getInstalledComponent();

    return 0 if(!$self->isUpdate());
    return 0 if(!defined($installedComponent));

    my $newVersion = $self->getManifest()->getVersionObject();
    my $installedVersion = $installedComponent->getManifest()->getVersionObject();
    return $newVersion->isNewerThan($installedVersion);
}

sub isDowngrade {
    my($self) = @_;
    my $installedComponent = $self->getInstalledComponent();

    return 0 if(!$self->isUpdate());
    return 0 if(!defined($installedComponent));

    my $newVersion = $self->getManifest()->getVersionObject();
    my $installedVersion = $installedComponent->getManifest()->getVersionObject();
    return (!$newVersion->isEqual($installedVersion) && !$newVersion->isNewerThan($installedVersion));
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    my $installedComponent = $self->getInstalledComponent();

    return 1 if (!$stackUpdate);
    return 0 if (!defined($installedComponent));

    my $newVersion = $self->getManifest()->getVersionObject();
    my $installedVersion = $installedComponent->getManifest()->getVersionObject();
    return ($installedVersion->isNewerThan($newVersion)) ? 0 : 1;
}

sub getInstalledComponent{
    my ($self) = @_;
    my $systemComponentManager = $self->{componentManager}->getSystemComponentManager();
    if (defined $systemComponentManager) {
        return $systemComponentManager->getComponentByKeyName($self->getComponentKeyName());
    }
    return undef;
}

sub requireSystemRestart {
	return 0;
}

sub getNotCompatibleWithScopeInstanceMessage {
	return "";
}

sub executeHdbOperation{
	my ($self, $instconfig, $operation, $args, $progress_start_msg, $progress_end_msg) = @_;

    my $msg = $self->getMsgLst()->addProgressMessage( "$progress_start_msg..." );
	my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);

   	my $hdbInstance = $instconfig->getOwnInstance();
	if(!defined $hdbInstance){
   		$self->setErrorMessage ('Unable to get HDB Instance.', $self->getErrMsgLst());
		return undef;
	}
	
   	my $installDir = $hdbInstance->get_globalTrexInstallDir();		
	my $command = File::Spec->catfile($installDir, 'bin', $operation);

	my $stdinLines = $instconfig->getXmlPasswordStream();
	if (defined $stdinLines) {
		push @$args, '--read_password_from_stdin=xml';
	}
	my $executor = new LCM::ProcessExecutor($command,$args, $stdinLines);
    $executor->setOutputHandler($self->getProgressHandler ());
    my $exitCode = $executor->executeProgram();

    $self->getMsgLst ()->addMessage(undef, $executor->getMsgLst());
    $self->{logFileLocation} = $self->LCM::Component::parseLogFileLocation($executor->getOutputLines());

    if (!defined $exitCode || $exitCode){
		$msg->endMessage( undef, "$progress_start_msg failed" );
		$self->setMsgLstContext($saveCntxt);
		
   		$self->setErrorMessage ("$progress_start_msg failed", $executor->getErrMsgLst());
		return undef;
	}
	
	$msg->endMessage( undef, "$progress_end_msg " );
	$self->setMsgLstContext($saveCntxt);

    return 1;
}

sub filterListString {
	my ($self, $listStr, $filter) = @_;

   	my @list = split(/\s*,\s*/, $listStr);
	my @filteredList = grep(/$filter/, @list);

	return @filteredList;
}

sub getSlppLogFileName {
	return undef;
}

# Needed for backwards compatibility with HANA options
# Will be refactored in SPS11 to try and create
# basepath directories only on relevant hosts
sub tryToCreateDirOnLocalHost {
    my ($self, $instconfig, $directory) = @_;

    my $sid    = $instconfig->getSID();
    my $sidAdm = $instconfig->getSysAdminUserName($sid);

    my (undef, undef, $uid, $gid) = getpwnam($sidAdm);
    my $tempErrMsgLst = new SDB::Install::MsgLst();

    if(!(-d $directory)) {
        if (!LCM::FileUtils::createDirectory($directory, 0755, $>, $), 1, $tempErrMsgLst)) {
            my $errorMessage = $tempErrMsgLst->getMsgLstString();
            chomp($$errorMessage);
            $self->getMsgLst()->addWarning($$errorMessage);
            return;
        }
    }
    if (!chown($uid, $gid, $directory)) {
        $self->getMsgLst()->addWarning("Failed to change ownership of directory '$directory'");
        return;
    }
    if (!chmod(0750, $directory)) {
        $self->getMsgLst()->addWarning ("Failed to change mode of directory '$directory'");
        return;
    }
}

sub installThroughSHAOperation {
    my ($self, $instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs) = @_;
    my $action = $self->isUpdate() ? 'Update' : 'Installation';

    my $checker = new LCM::Component::Installable::InstallationKitChecker();
    if($isWin || $checker->isKitOwnedByRoot($self->getInstallerDir(), $self->getFilelistFileName())){
        return $self->_executeThroughSHAOperation($instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs, $action);
    }
    my $remoteExecution = new SDB::Install::RemoteHostctrlHosts(hostname());
    my $cmpName = $self->getComponentName();
    my $timestamp = $self->createTimestamp();

    $remoteExecution->useSidadm() if (defined($instconfig->getValue('Password')));

    $self->getMsgLst ()->addMessage ("Installation kit of $cmpName is not owned by root");
    my $msg = $self->getMsgLst ()->addMessage ("Copying installer of $cmpName as root...");
    $remoteExecution->setMsgLstContext ([$msg->getSubMsgLst()]);
    if($self->_executeAdditionalOperation($instconfig, $remoteExecution, $self->getCopyInstallerOperation(), $timestamp)){
        $instconfig->getMsgLst()->addError ("Copying of ${cmpName}'s installer through SAP Host Agent Operation failed!", $remoteExecution);
        return undef;
    }
    
    my $copiedInstallerDirectory = $self->getCopiedInstallerDirPrefix() . $timestamp;
    my $newInstallationKitPath = File::Spec->catdir($instconfig->getValue('Target'), uc($instconfig->getSID()), $copiedInstallerDirectory);
    $self->updateSHAOptionsMap($optionMap, $newInstallationKitPath);
    my $returnCode = $self->_executeThroughSHAOperation($instconfig, $operation, $optionMap, $passwordKeys, $isShowProgressMsgs, $action);
    
    $msg = $self->getMsgLst ()->addMessage ("Removing $cmpName installer copy...");
    $remoteExecution->setMsgLstContext ([$msg->getSubMsgLst()]);
    if($self->_executeAdditionalOperation($instconfig, $remoteExecution, $self->getDeleteCopiedInstallerOperation(), $timestamp)){
        $instconfig->getMsgLst()->addWarning("Removing $cmpName installer copy through SAP Host Agent Operation failed!", $remoteExecution);
    }
    
    return $returnCode;
}

sub _executeAdditionalOperation {
    my ($self, $instconfig, $remoteExecution, $operation, $timestamp) = @_;
    my $sourceDirectory = $self->getInstallerDir();
    my $signature_dir = determineSignedManifestRelativePath($sourceDirectory);
    my $optionsMap = {
        SAPMNT => $instconfig->getValue('Target'),
        SID => uc($instconfig->getSID()),
        SOURCE_DIR => $sourceDirectory,
        SUFFIX => $timestamp,
        SIGNATURE_DIR => $signature_dir
    };
    my $passwordKeys = ['Password', 'HostagentPassword'];
    return $remoteExecution->executeHostctrlParallel($operation, $instconfig, undef, $passwordKeys, undef, undef, undef, undef, undef, $optionsMap, undef, undef, undef, 1);
}

sub getProgressMsg {
	my ($self) = @_;
	my $componentName = $self->getComponentName();
	my $phase = $self->getPhase();

	if(!defined $phase || !$self->supportsPhase($phase)){
		my $actionPrefix = $self->isUpdate ? 'Updat' : 'Install';    
		return $actionPrefix."ing " . $componentName;
	}

	if($self->isUpdate()){
		if($phase eq PREPARE_PHASE){
			return "Preparing update of $componentName";
		}
		if($phase eq OFFLINE_PHASE){
			return "Updating $componentName";
		}
		if($phase eq ONLINE_PHASE){
			return "Finalizing update of $componentName";
		}
        if($phase eq SYSTEM_ONLINE_PHASE){
            return "Updating content of $componentName";
        }
	} else {
		if($phase eq PREPARE_PHASE){
			return "Preparing installation of $componentName";
		}
		if($phase eq OFFLINE_PHASE){
			return "Installing $componentName";
		}
		if($phase eq ONLINE_PHASE){
			return "Finalizing installation of $componentName";
		}
        if($phase eq SYSTEM_ONLINE_PHASE){
            return "Installing content of $componentName";
        }
	}
}

sub _getPersistenceFilePath {
	my ($self, $sapSystem) = @_;

	my $componentKey = $self->getComponentBatchKey();
	my $fileName = sprintf('pending_%s.xml', $componentKey);
	my $globalSidDirectory = $sapSystem->get_globalSidDir() || $sapSystem->getUsrSapSid(); # Handle the case where /<sapmnt>/<SID> is /usr/sap/<SID>

	my $persistenceFilePath = join($path_separator, $globalSidDirectory, $fileName);
	return $persistenceFilePath if -f $persistenceFilePath;

	my $fileList = listDirectory($globalSidDirectory);
	return undef if ! defined $fileList;
	
	my $filePathList = [ map { File::Spec->catfile($globalSidDirectory, $_) } @{$fileList} ];
	for my $candidateFilePath (@{$filePathList}) {
		next if(!-f $candidateFilePath || lc($candidateFilePath) !~ /pending_$componentKey.xml$/);
		return $candidateFilePath;
	}
	return undef;
}

sub isInstalled{
	my $self = shift;
	return 0;
}

sub requiresSqlSysPasswd{
   return 0;
}

sub isSigned {
    return 0;
}

sub getInstallerDir {
    my ($self) = @_;
    my $manifestDir = $self->getPath();
    return dirname($manifestDir);
}

sub verifyFilesPermissions{
    my ($self, $errorMsgLst) = @_;
    return 1 if (!$self->shouldVerifyFiles());
    my $check = LCM::Component::Installable::InstallationKitChecker->new();
    return $check->isKitWritableForOthers($self->getInstallerDir(), $errorMsgLst);
}

sub checkFilesForPipes {
    my ($self, $errorMsgLst) = @_;
    return 1 if (!$self->shouldVerifyFiles());
    my $check = LCM::Component::Installable::InstallationKitChecker->new();
    return $check->scanForPipes($self->getInstallerDir(), $errorMsgLst);
}

sub shouldVerifyFiles {
    my $self = shift;
    return !$self->isInternal();
}

sub requiresRootPrivilege {
    return 0;
}

1;
