package LCM::Component::Installable::ES;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw ($gDirNameEs
                              $gLogDir
                              $gProductNameEs
                              $gHostRoleEsStandby
                              $gHostRoleEsWorker);
use LCM::ProcessExecutor;
use LCM::Installer;
use LCM::FileUtils;
use File::Basename qw (dirname);
use LCM::ExecutionWarningsObservable;
use strict;

use base 'LCM::Component::Installable';

our $installScript = 'esinst';
our $updateScript = 'esupd';
if ($isWin){
    $installScript .= '.exe';
    $updateScript .= '.exe';
} else {
    $installScript .= '.sh';
    $updateScript .= '.sh';
}

sub installComponent {
    my ( $self, $instconfig) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');

    my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );
	
    my $sapSystem = $self->getSAPSystem ( $instconfig, $msg, $saveCntxt);
    if (not defined $sapSystem) {
		return undef;
    }
	
#	if(!$self->isUpdate()) {
#		$self->tryToCreateDirOnLocalHost($instconfig, $instconfig->getValue('EsDataPath'));
#		$self->tryToCreateDirOnLocalHost($instconfig, $instconfig->getValue('EsLogPath'));
#	}
	
	my $returnCode = $self->_installExtendedStorageComponent($instconfig, $sapSystem);
    $msg->endMessage (undef, $self->_getEndMessage($returnCode));
    $self->setMsgLstContext($saveCntxt);    
    return $returnCode;
}

sub _getEndMessage {
	my ($self, $returnCode) = @_;
	my $message = 'Installation of ' . $self->getComponentName();
	$message .= ($returnCode == 1) ? ' finished' : ' failed';
	return $message;
}

sub _installExtendedStorageComponent {
	my ($self, $instconfig, $sapSystem) = @_;
	my $exer = $self->createProcessExecutor($instconfig, $sapSystem);

    $self->initProgressHandler ();
    $exer->setOutputHandler($self->getProgressHandler ());
    my $exitCode = $exer->executeProgram();

    $self->getMsgLst ()->addMessage(undef, $exer->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($exer->getOutputLines()));

    #check for deploy content error. handle it as warning
    if ((defined $exitCode) && ($exitCode == 2)) {
        my $action = $self->isUpdate() ? 'update' : 'installation';
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning("'$gProductNameEs' $action finished with warning, because of\nFailed deployment of HANA content (part of '$gProductNameEs' archive).\nFor next steps check the '$gProductNameEs' documentation.");
    } elsif (!defined $exitCode || $exitCode){
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($exer->getOutputLines());
        my $actionErr = $self->isUpdate() ?'Update ' : 'Installation ';
        my $errorMessage = "$actionErr of " . $self->getComponentName() . ' failed';
        $self->setErrorMessage ($errorMessage, ($errMsgLst->isEmpty() ? $exer->getErrMsgLst() : $errMsgLst));
        return undef;
    }
 	return 1;
}

sub getSAPSystem {
    my ( $self, $instconfig, $msg, $saveCntxt) = @_;

    my $systems = $instconfig->CollectSAPSystems (undef, 1);
    my $sid = $instconfig->getValue("SID");
    my $sapSystem = $systems->{$sid};
    if (not defined $sapSystem) {
        $self->setErrorMessage ("No SAP system '$sid' found");
        $msg->endMessage( undef, 'Install ' . $self->getComponentName());
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }
	return $sapSystem;
}

sub _buildArgs {
    my ($self, $instconfig, $installDir) = @_;
    my @arguments = ();
    push( @arguments, '-path');
    push( @arguments, $installDir); 
    push( @arguments, '-sid');
    push( @arguments, $instconfig->getSID ()); 
    
    my $instanceNr = undef;
    my $hostname   = undef;
    my $instance = $instconfig->getOwnInstance();

    if (defined $instance) {
    	$instanceNr = $instance->get_nr();
        $hostname   = $instance->get_host();
    }

    $instanceNr = $instconfig->getValue('InstanceNumber') if (!defined $instanceNr);
    $hostname   = $instconfig->getValue('HostName')       if (!defined $hostname);

    if (defined $instanceNr) {
        push( @arguments, '-inst');
        push( @arguments, $instanceNr);
    }

    if (defined $hostname) {
        push( @arguments, '-hdbhost');
        push( @arguments, $hostname);
    }

    my $user = $instconfig->getValue ('SystemUser');
    if (defined $user ) {
        push( @arguments, '-hdbuser');
        push( @arguments, $user);
    }
    
    if(!$self->isUpdate()) {
        my $esDataPath = $instconfig->getValue('EsDataPath');
        push( @arguments, '-data_es');
        push( @arguments, $esDataPath);
        my $esLogPath = $instconfig->getValue('EsLogPath');
        push( @arguments, '-log_es');
        push( @arguments, $esLogPath);     
    }

    push( @arguments, '-log_dir');
    push( @arguments, $gLogDir);
    push @arguments, @{$self->SUPER::_buildArgs($instconfig)};
    return \@arguments;
}

sub createProcessExecutor{
    my ($self, $instconfig, $sapSystem) = @_;
	
    my $installDir = $sapSystem->get_globalSidDir();
    my $user = $sapSystem->getUser();
    my $uid = $user->uid();
    my $gid = $user->gid();

    my $instance = $sapSystem->getNewDBInstances ()->[0];

    my $stdinPassword = [$instconfig->getValue ('SQLSysPasswd')];
    my $command = $self->getESInstaller();
    my $args = $self->_buildArgs($instconfig, $installDir);

    if ($self->isSidAdmUserExecution()) {
    	return new LCM::ProcessExecutor($command, $args, $stdinPassword);
    }

    my $exer = new LCM::ProcessExecutor($command, $args, $stdinPassword, dirname ($command), undef, $uid, $gid);
    my %env = %ENV;
    $exer->setProcessEnvironment ($instance->_getHDBEnv (\%env));
    return $exer;
}

sub preCheckInstallComponent {
    return 1;
}

sub getESInstaller {
    my ($self) = @_;
    my $script = $self->isUpdate() ? $updateScript : $installScript;
    return File::Spec->catfile($self->getInstallerDir(), $script);
}

sub getNumberOfExpectedOutputLines{
    return 75;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    if ($stackUpdate){
        return $self->SUPER::getDefaultSelection ($stackUpdate);
    }
    return 0;
}

sub getComponentName {
	return $gProductNameEs;
}

sub getSlppLogFileName {
	return 'extended_storage.log'
}

sub requireSystemRestart {
    return 1;
}

sub parseLogFileLocation {
    my ($self, $outputLines) = @_;
    if (!defined $outputLines){
        return undef;
    }
    my $log_location;
    my $pattern = "Log file written to [\']*([^\']*)[\']*";
    foreach my $line (reverse @$outputLines){
        if (substr ($line, -1) eq '.') {
           chop $line;
        }
        ( $log_location ) = $line =~ /$pattern/i;
        
        if (defined $log_location) {
            return $log_location;
        }
    }
}

sub updateComponent;
*updateComponent =  \&installComponent;

sub getHostRoles {
    return [$gHostRoleEsWorker, $gHostRoleEsStandby];
}
sub requiresSqlSysPasswd{
    return 1;
}

1;
