package LCM::Component::Installable::HDBServerPlugin;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::System qw(isAdmin);
use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::Configuration::GenericServerPlugin;
use SDB::Install::Globals qw ($gLogDir $gKeynameEngine);
use LCM::Configuration::GenericStackAny qw ($ini_section_plugin);
use LCM::ProcessExecutor;
use LCM::Component qw ( ONLINE_PHASE OFFLINE_PHASE );

use strict;

use base 'LCM::Component::Installable';

sub preCheckInstallComponent {
    return 1;
}

sub preCheckUpdateComponent {
    return 1;
}

sub updateComponent;
*updateComponent =  \&installComponent;

sub installComponent {
    my ( $self, $instconfig, ) = @_;

    my $msg = $self->getMsgLst ()->addProgressMessage ($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
    $self->initProgressHandler ();

    my $rc = 1;
    my $processExecutor = $self->_constructProcessExecutor($instconfig);
    if (!defined $processExecutor){
        return undef;
    }
    my $environment = $self->prepareHdbInstallerEnvironment ();
    $processExecutor->setOutputHandler($self->getProgressHandler ());
    $processExecutor->setProcessEnvironment ($environment);
    my $exitCode = $processExecutor->executeProgram();
    $self->getMsgLst ()->addMessage(undef, $processExecutor->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($processExecutor->getOutputLines()));

    if (!defined $exitCode || $exitCode){
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($processExecutor->getOutputLines());
        my $action = $self->isUpdate ? 'Update' : 'Installation';
        $self->setErrorMessage ($action.' of ' . $self->getComponentName() . ' failed',
            $errMsgLst->isEmpty ? $processExecutor->getErrMsgLst() : $errMsgLst);
        $rc = undef;
    }

    my $action = $self->isUpdate ? 'Update' : 'Install';
    $msg->endMessage (undef, $action. ' ' . $self->getComponentName());
    $self->setMsgLstContext($saveCntxt);
    return $rc;
}

sub _constructProcessExecutor {
	my ($self, $instconfig) = @_;
	my $command = $self->getHdbInstallExecutable();
	my $args = $self->_buildArgs($instconfig);
	if (!defined $args){
		return undef;
	}
	my ($uid, $gid) = (undef, undef);
	my $stdinLines = $instconfig->getXmlPasswordStream();

	push(@$args, ('-b', "--instlog_dir=${gLogDir}"));
	push(@$args, '--read_password_from_stdin=xml') if (defined $stdinLines);

	if (!$isWin && isAdmin()) {
		my $user = new SDB::Install::NewDBUser($instconfig->getSID());
		($uid, $gid) = ($user->uid(), $user->gid());
	}
	return new LCM::ProcessExecutor($command, $args, $stdinLines, undef, undef, $uid, $gid);
}

sub getNumberOfExpectedOutputLines{
    return 30;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;

	if (!$stackUpdate) {
		return 0;
	}

	my $installedComponent = $self->getInstalledComponent();
	if (!$installedComponent) {
		return 0;
	}

	my $newVersion = new SDB::Install::Version(split('\.', $self->getVersion()));
	my $installedVersion = new SDB::Install::Version(split('\.', $installedComponent->getVersion()));
	return ($installedVersion->isNewerThan($newVersion)) ? 0 : 1;
}

sub _buildArgs {
	my ($self, $instconfig) = @_;

    my $pluginConfig = $self->getExternalHdbInstallerConfiguration ();
    return undef if (!defined $pluginConfig);

	my $paramsToSkip =['NoStart'];
	my $args = $instconfig->getCmdLineArgsFromInstconfig($pluginConfig, $paramsToSkip, $ini_section_plugin);
	if(!$self->_shouldStart($instconfig)){
		push(@$args, "--nostart=on");
	}
	push(@$args, "--archive_dir=$self->{manifestDir}");
	push(@$args, '--ignore_unknown_option');
	push @$args, @{$self->SUPER::_buildArgs($instconfig)};
	return $args;
}

sub getConfigParser {
    my ($self) = @_;
    require SDB::Install::ConfigXMLParser;
    my $path = $self->getInstallerDir() . $path_separator . 'packages';
    my $xmlFile = $path . $path_separator . 'InstallParams.xml';
    my $dtdFile = $path . $path_separator . 'installcfg.dtd';
    if (!-f $xmlFile){
        $self->getErrMsgLst()->addError ("Configuration file '$xmlFile' not found");
        return undef;
    }
    if (!-f $dtdFile){
       $self->getMsgLst()->addWarning ("Configuration DTD file '$dtdFile' not found");
       $dtdFile = undef;
    }
    my $parser = new SDB::Install::ConfigXMLParser ();
    eval{
        $parser->parsefile ($xmlFile, $dtdFile);
    };
    if ($@){
       $self->getErrMsgLst()->addError ("Parsing configuration file '$xmlFile' failed: $@");
       return undef;
    }
    return $parser;
}

sub shouldImportContent {
    my ($self) = @_;
    my $configparser = $self->getConfigParser();
    return 0 if(!$configparser);
    return $configparser->getImportContent() =~ /$bool_true_pattern/i;
}

sub requireSystemRestart {
    return 1;
}

sub requiresSqlSysPasswd{
    my ($self) = @_;

    if ($self->shouldImportContent ()){
        return 1;
    }

    my $manifest = $self->getManifest ();
    if ($manifest->isServerPluginOnlineUpdatable ()){
        return 1;
    }
    return 0;
}

sub _shouldStart{
    my ($self, $instconfig) = @_;

    return 0 if $self->_isUpdatingServer();

    my $passThroughPrefix = $instconfig->getPassThroughParamPrefix($ini_section_plugin, 'hdbinst');
    my $passThroughParamIdNoStart = sprintf('%sNoStart', $passThroughPrefix);
    my $isNoStart = $instconfig->getValue($passThroughParamIdNoStart);
    if(!$instconfig->isSkipped($passThroughParamIdNoStart) && ($isNoStart =~ /$bool_true_pattern/)){
        return 0;
    }

    my $isUsingPhases = $instconfig->usePhases();
    if($isUsingPhases) {
	    my $currentPhase = defined($self->getPhase()) ? $self->getPhase() : $self->getDefaultPhase();
	    return 0 if $currentPhase ne ONLINE_PHASE;
    }

    my $keyname = $self->getComponentKeyName();
    my @remainingPluginsForUpdate = grep{$_ ne $keyname} @{$instconfig->{plugins_list}};
    $instconfig->{plugins_list} = \@remainingPluginsForUpdate;
    my $remainingPluginsCount = scalar(@remainingPluginsForUpdate);
    return ($remainingPluginsCount == 0 || $self->shouldImportContent()) ? 1 : 0;
}

sub _isUpdatingServer {
    my ($self) = @_;
    my $componentManager = $self->getComponentManager();
    return 0 if !$componentManager->isHDBServerComponentSelected();
    
    my $serverComponent = $componentManager->getComponentByKeyName($gKeynameEngine);
    return (defined $serverComponent && $serverComponent->isUpdateToNewerVersion());
}

sub getSlppLogFileName {
    my ($self) = @_;
    my $manifest = $self->{manifest};
    my $componentKey = $manifest->getComponentKey();
    return "$componentKey.log"
}

sub getDefaultPhase {
    my ($self) = @_;
    return ONLINE_PHASE if !$self->_isUpdatingServer() && $self->shouldImportContent();
    return OFFLINE_PHASE;
}

sub getFallbackHdbInstallerConfiguration {
    return SDB::Install::Configuration::GenericServerPlugin->new();
}

sub isSigned {
    return 1;
}

1;
