package LCM::Component::Installable::LSS;

use strict;
use SDB::Install::SAPSystemUtilities;
use SDB::Install::SysVars qw( $isWin );
use SDB::Install::Globals qw ( $gProductNameLSS $gLogDir $gSAPLocation);
use File::Basename qw( dirname );
use LCM::ProcessExecutor;
use SDB::Install::MsgLst;
use File::stat;
use LCM::Component qw (OFFLINE_PHASE);

use base 'LCM::Component::Installable';

sub installComponent {
    my ( $self, $instconfig) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );
    my $returnCode = $self->_installLSSComponent($instconfig);

    if ( $returnCode && $instconfig->isUpdate() ) {
        $returnCode = $self->_registerLocalSecureStoreService($instconfig);
    }

    $msg->endMessage(undef, $self->_getEndMessage($returnCode));
    $self->setMsgLstContext($saveCntxt);

    return $returnCode;
}

sub _registerLocalSecureStoreService {
    my ($self, $instconfig) = @_;

    my $sapSys = $instconfig->getSAPSystem();
    return undef if ! defined $sapSys;

    my $instance = $instconfig->getOwnInstance();
    return undef if ! defined $instance;

    $instance->setLSSInstancesInIniFile(1) if ! $self->isUpdate();

    my $rc = 1;
    if (!$instconfig->isOptimizedUpdate()){
        my $userName = $instance->getSidAdmName();
        my $password = $instconfig->getValue('Password');
        my $start_timeout = $instconfig->getTimeout ('start_instance');
        my $stop_timeout = $instconfig->getTimeout ('stop_instance');
        my $noStart = $instconfig->isServerNoStart();
        my $useHttps = $instconfig->isUseHttps();

        my $saveCntxt = $sapSys->setMsgLstContext(
            [new SDB::Install::MsgLst(), new SDB::Install::MsgLst()] );
        my $progressHandler = $self->getMsgLst()->getProgressHandler();
        $sapSys->SetProgressHandler($progressHandler);

        my $util = new SDB::Install::SAPSystemUtilities($instance);
        $rc = $util->restartSystem($sapSys, $instance, $userName, $password, 
            $start_timeout, $stop_timeout, $noStart, $useHttps);
        $self->getMsgLst->appendMsgLst($sapSys->getMsgLst());
        $sapSys->setMsgLstContext( $saveCntxt );
    }

    return $rc;
}

sub updateComponent;
*updateComponent =  \&installComponent;

sub _getEndMessage {
    my ($self, $returnCode) = @_;

    my $message  = ($self->isUpdate() ?'Update' : 'Installation') .' of ' . $self->getComponentName();
    $message .= ($returnCode == 1) ? ' finished' : ' failed';
    return $message;
}

sub getComponentName {
    return $gProductNameLSS;
}

sub _installLSSComponent {
    my ($self, $instconfig) = @_;
    my $executor = $self->createProcessExecutor($instconfig);

    $self->initProgressHandler ();
    $executor->setOutputHandler($self->getProgressHandler());
    my $exitCode = $executor->executeProgram();

    $self->getMsgLst()->addMessage(undef, $executor->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($executor->getOutputLines()));
    if (!defined $exitCode || $exitCode){
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($executor->getOutputLines());
        my $actionErr = $self->isUpdate() ?'Update ' : 'Installation ';
        my $errorMessage = "$actionErr of " . $self->getComponentName() . ' failed';
        $self->setErrorMessage ($errorMessage, ($errMsgLst->isEmpty() ? $executor->getErrMsgLst() : $errMsgLst));
        return undef;
    }
    return 1;
}

sub createProcessExecutor{
    my ($self, $instconfig) = @_;

    my $command = $self->getHdbInstallExecutable();
    my $args = $self->_buildArgs($instconfig);
    return new LCM::ProcessExecutor($command, $args);
}

sub getLSSSharedDir{
    my ($self, $instconfig) = @_;
    my $shared_dir;
    if ($self->isUpdate()){
        my $installedDir = $self->getInstalledComponent()->{manifestDir};
        $shared_dir = dirname($installedDir);
    } else {
        $shared_dir = $instconfig->getValue('LSSInstPath');
    }
    return $shared_dir;
}

sub _buildArgs {
    my ($self, $instconfig) = @_;

    my @arguments = ();
    push( @arguments, '--shared_path');
    push( @arguments, $self->getLSSSharedDir($instconfig));
    push( @arguments, '--sid');
    push( @arguments, $instconfig->getSID());
    if (!$self->isUpdate()){
        push( @arguments, '--userid');
        push( @arguments, $instconfig->getValue('LSSUserID'));
        push( @arguments, '--groupid');
        push( @arguments, $instconfig->getValue('LSSGroupID'));
        push( @arguments, '--home');
        push( @arguments, $instconfig->getValue('LSSUserHomeDir'));
        push( @arguments, '--shell');
        push( @arguments, $instconfig->getValue('LSSUserShell'));
        push( @arguments, '--sapsys_gid');
        push( @arguments, $self->getGID($instconfig));
        push( @arguments, '--system_usage');
        push( @arguments, $self->getSystemUsage($instconfig));
    }
    push( @arguments, '--instlog_dir');
    push( @arguments, $gLogDir);
    push( @arguments, '-b');
    return \@arguments;
}

sub getGID {
    my ($self, $instconfig) = @_;
    return $instconfig->getValue('GID') if ! $instconfig->isUpdate();
    my $sapSystem = $instconfig->getSAPSystem();
    return $sapSystem->getGID();
}

sub getSystemUsage {
    my ($self, $instconfig) = @_;
    return $instconfig->getValue('SystemUsage') if ! $instconfig->isUpdate();
    my $sapSystem = $instconfig->getSAPSystem();
    my $instance = $sapSystem->getNewDBInstances ()->[0];
    return $instance->getSystemUsage();
}

sub parseLogFileLocation {
    my ($self, $outputLines) = @_;
    if (!defined $outputLines){
        return undef;
    }
    my $log_location;
    my $pattern = "Log file written to [\']*([^\']*)[\']*";
    foreach my $line (reverse @$outputLines){
        if (substr ($line, -1) eq '.') {
           chop $line;
        }
        ( $log_location ) = $line =~ /$pattern/i;

        if (defined $log_location) {
            return $log_location;
        }
    }
}

sub _getPersistenceFilePath {
    my ($self, $configuration) = @_;
    my $sid = $configuration->getValue('SID');
    my $keyname = $self->getComponentKeyName();
    $keyname = lc ($keyname);
    return File::Spec->catfile($gSAPLocation, $sid.'.pending_'. $keyname. '.xml');
}

sub hasPersistenceFile {
    my ($self, $configuration) = @_;
    return File::stat::stat($self->_getPersistenceFilePath($configuration));
}

sub getPersistenceXMLObject {
    my ($self, $configuration) = @_;

    if (defined $self->{_persistenceXML}) {
        return $self->{_persistenceXML};
    }

    my $filePath = $self->_getPersistenceFilePath($configuration);
    my $persistenceXMLObject = LCM::Persistence::XML->new($filePath);
    if ($persistenceXMLObject->errorState()) {
        $configuration->getMsgLst()->addWarning($persistenceXMLObject->getErrorString());
    }
    $self->{_persistenceXML} = $persistenceXMLObject;

    return $persistenceXMLObject;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    return ($stackUpdate) ? $self->SUPER::getDefaultSelection ($stackUpdate) : 0;
}

sub getNumberOfExpectedOutputLines{
    return 5;
}

sub getSlppLogFileName {
    return 'lss.log'
}

sub preCheckInstallComponent {
    return 1;
}

sub preCheckUpdateComponent {
    return 1;
}

sub getDefaultPhase {
	return OFFLINE_PHASE;
}

1;