package LCM::Component::Installable::ResidentInstaller::ResidentInstallerRoot;

use strict;
use File::Spec;
use SDB::Install::User;
use SDB::Install::MsgLst;
use experimental qw(smartmatch);
use LCM::App::ApplicationContext;
use SDB::Install::DirectoryWalker;
use LCM::App::UpdateSystemUtils;
use File::Basename qw(basename dirname);
use File::stat;
use LCM::FileUtils qw(MyRealPath writeToFile);
use LCM::ResidentInstallerManifest;
use LCM::Component::Installed;
use SDB::Install::System qw(makedir copy_file);
use SDB::Common::Utils qw(createSysAdminUserName);
use SDB::Install::SysVars qw($isWin);
use LCM::SAPDSigner;
use parent 'LCM::Component::Installable::ResidentInstaller::ResidentInstallerBase';
use SDB::Install::Globals qw($gSignatureManifestName
                             $gCockpitSignatureManifestName
                             $gFlavourPlatform
                             $gFlavourExpress
                             $gFlavourCockpit
                             $gDirNameResidentInstaller
                             $gSAPDSignerNotDetectedMessage
                             getFlavourProductName
                             determineSignedManifestPath);

our $__exeExt = ($^O =~ /mswin/i) ? '.exe' : '';
# 'files/dirs to sign' are exported and used at buildpackage time to create signing directories:
our @files_to_sign = ('hdblcm'.$__exeExt, 'hdblcmgui'.$__exeExt, 'hdblcmweb'.$__exeExt);
our @files_not_to_sign = ($gSignatureManifestName, $gCockpitSignatureManifestName);
our @directories_to_sign = ('instruntime', 'operations.d', 'adapters.d', 'descriptors.d');
our @directories_not_to_sign = ('resources');
my  %FILES_OWNED_BY_ROOT = ('hdblcm' => 1, 'hdblcmgui' =>1 , 'hdblcmweb' => 1, 'filelist.resident' => 1);
my  %EXECUTABLE_FILES = ('hdblcm'.$__exeExt => 1, 'hdblcmgui'.$__exeExt => 1, 'hdblcmweb'.$__exeExt => 1, 'sdbrun'.$__exeExt => 1);
my  $DOCUMENTATION_UI_STRING = 'SAP HANA Documentation';
my  $CONFIGURATIONS_DIR = 'configurations';
my $TEMPLATE_FLAVOUR_JS = <<END;
'use strict';
jQuery.sap.declare('com.sap.lm.sl.hana.SystemFlavour');

com.sap.lm.sl.hana.SystemFlavour = {
	PLATFORM: '%s',
	EXPRESS: '%s',
	COCKPIT: '%s',
	currentFlavour: '%s',
	currentFlavourProductName: '%s',
	getSystemFlavour: function() {
		return this.currentFlavour;
	},
    getFlavourProductName: function() {
        return this.currentFlavourProductName;
    },
};
END

sub executeInstall {
    my ($self, $instconfig) = @_;
    my $cmpName = $self->getComponentName();
    my $msg = $self->getMsgLst()->addProgressMessage("Installing $cmpName...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $oldDepth = $progressHandler->getIntendationDepth();
    $progressHandler->setIntendationDepth($oldDepth + 1);

    my $rc = $self->_installResidentHdblcm($instconfig);
    if ($rc && $self->_shouldValidateInstaller($instconfig)) {
        my $installedComponent = $self->_constructInstalledComponent($instconfig);
        if (!$self->_verifyInstallerAuthenticity($installedComponent)) {
            $installedComponent->uninstallComponent($instconfig);
            $rc = 0;
        }
    }

    $progressHandler->setIntendationDepth($oldDepth);
    $msg->endMessage (undef, "Install $cmpName");
    $self->setMsgLstContext($saveCntxt);
    return $rc;
}

sub executeUpdate {
    my ($self, $instconfig) = @_;
    my $cmpName = $self->getComponentName();
    my $msg = $self->getMsgLst()->addProgressMessage("Updating $cmpName...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $oldDepth = $progressHandler->getIntendationDepth();
    $progressHandler->setIntendationDepth($oldDepth + 1);

    my $rc = $self->_shouldValidateInstaller($instconfig) ? $self->_updateWithVerification($instconfig) : $self->_updateComponent($instconfig);

    $progressHandler->setIntendationDepth($oldDepth);
    $msg->endMessage(undef, "Update $cmpName");
    $self->setMsgLstContext($saveCntxt);
    return $rc;
}

sub _getInstallationKitPath {
	my ($self) = @_;
	my $stackKitPath = $self->getPath() // SDB::Install::Installer->new()->GetRuntimeDir();
    if($isWin) {
        $stackKitPath =~ s/(.*)\\.+\\?/$1/;
        return $stackKitPath;
    }
	$stackKitPath =~ s/(.*)\/.+\/?/$1/;
	return $stackKitPath;
}

sub _installResidentHdblcm {
    my ($self, $instconfig) = @_;
    my $stackKitPath = $self->_getInstallationKitPath();
    my $tgtDir = $self->_getInstallationPath($instconfig);
    $self->{_instconfig} = $instconfig;

    my @copy_list = ();
    foreach my $file (@files_to_sign) {
        push (@copy_list, [File::Spec->catfile($stackKitPath, $file), File::Spec->catfile($tgtDir, $file)]);
    }
    foreach my $file (@files_not_to_sign) {
        if ($file eq $gSignatureManifestName || $file eq $gCockpitSignatureManifestName) {
            # special treatment for signature manifest which may be located in parent dir
            my $sourceSignaturePath = determineSignedManifestPath($stackKitPath, $file);
            if (-f $sourceSignaturePath) {
                push (@copy_list, [$sourceSignaturePath, File::Spec->catfile($tgtDir, $file)]);
            }
        } else {
            push (@copy_list, [File::Spec->catfile($stackKitPath, $file), File::Spec->catfile($tgtDir, $file)]);
        }
    }
    $self->getMsgLst()->addMessage("Copying resident hdblcm kit from '$stackKitPath' to '$tgtDir'");

    my @dirs = (@directories_to_sign, @directories_not_to_sign);
    push @dirs, $CONFIGURATIONS_DIR if -d File::Spec->catdir($stackKitPath, $CONFIGURATIONS_DIR);
    my $rc = $self->_findAndProcessFilesToCopy($stackKitPath, $tgtDir, \@dirs, \@copy_list);
    if(not defined $rc) {
        return undef;
    }

    foreach my $filesToCopy (@copy_list){
    	$rc = $self->copyFile($filesToCopy);
    	if(!$rc){
    		return undef;
    	}
    }

    return undef if(!$self->_extractSapUI5Libraries($tgtDir));
    return undef if(!$self->_extractAdminHelpResources($tgtDir));
    return undef if(!$self->_createFlavourInformationFile($instconfig, $tgtDir));

    # copy signature filelist for resident hdblcm
    my $sourceFilelist = File::Spec->catfile($stackKitPath, 'filelist.resident');
    my $targetFilelist = File::Spec->catfile($tgtDir, 'filelist.resident');
    my @Filelists = ($sourceFilelist,$targetFilelist);
    if (-f $sourceFilelist) {
        $self->copyFile(\@Filelists);
    } else {
        $self->getMsgLst()->addMessage("Cannot copy '$sourceFilelist' to '$targetFilelist' : file not found");
    }
    my $installedSignaturePath = File::Spec->catfile($tgtDir, $gSignatureManifestName);
    if(!$isWin && -e $installedSignaturePath && !chmod(0644, $installedSignaturePath)) { # Bad pattern - if statemnt has side effects
        $self->getErrMsgLst()->addError("Could not change mode of file '$installedSignaturePath': $!");
        return undef;
    }
    my $installedCockpitSignaturePath = File::Spec->catfile($tgtDir, $gCockpitSignatureManifestName);
    if(!$isWin && -e $installedCockpitSignaturePath){
        if(!chmod(0644, $installedCockpitSignaturePath)){
            $self->getErrMsgLst()->addError("Could not change mode of file '$installedCockpitSignaturePath': $!");
            return undef;
        }
    }
    return 1;
}

sub _verifyInstallerAuthenticity {
    my ($self, $installedComponent) = @_;
    my $cmpName = $self->getComponentName();
    my $errorMessage = "Could not verify the authenticity of the $cmpName.";
    my $sapdsigner = LCM::SAPDSigner->getInstance();
    $self->getMsgLst()->addProgressMessage("Verifying authenticity of '$cmpName'...");
    if (!defined($sapdsigner)) {
        $self->getErrMsgLst()->addError("$errorMessage $gSAPDSignerNotDetectedMessage");
        return undef;
    }

    $sapdsigner->resetMsgLstContext(); # Supress log messages
    if (!$sapdsigner->authenticateInstaller($installedComponent)) {
        $self->getErrMsgLst()->addError($errorMessage, $sapdsigner->getErrMsgLst());
        return undef;
    }

    return 1;
}

sub _createFlavourInformationFile {
    my ($self, $instconfig, $installationPath) = @_;
    my $appContext = LCM::App::ApplicationContext::getInstance();
    my $serverManifest = $appContext->getServerManifest();
    my $hanaFlavour = defined($serverManifest) ? $serverManifest->getHANAFlavour() : undef;
    my $hanaFlavourProductName = defined($hanaFlavour) ? getFlavourProductName($hanaFlavour) : undef;
    if (! defined($hanaFlavour) ) {
        my $componentManager = $instconfig->getComponentManager();
        $hanaFlavour = $componentManager->getHANAFlavour() if defined($componentManager);
        $hanaFlavour = $gFlavourPlatform unless defined($hanaFlavour);
        $hanaFlavourProductName = getFlavourProductName($hanaFlavour);
    }
    my $filePath = File::Spec->catfile($installationPath, 'resources', 'resident', 'application', 'SystemFlavour.js');
    my $fileContent = sprintf($TEMPLATE_FLAVOUR_JS, $gFlavourPlatform, $gFlavourExpress, $gFlavourCockpit, $hanaFlavour, $hanaFlavourProductName);
    my ($uid, $gid) = (undef, undef);
    my $errorList = new SDB::Install::MsgLst();
    my $sid = $self->{_instconfig}->getValue('SID');

    if(!$isWin){
        my $sidadmUserName = createSysAdminUserName($sid);
        my $sidadmUser = new SDB::Install::User($sidadmUserName);
        ($uid, $gid) = ($sidadmUser->id(), $sidadmUser->gid());
    }

    if(!writeToFile($filePath, $fileContent, 0644, $uid, $gid, $errorList)){
        $self->getErrMsgLst()->addError("Failed to write file '$filePath' to the file system", $errorList);
        return undef;
    }
    $self->getMsgLst()->addMessage("File '$filePath' was written to the file system.");
    return 1;
}

sub _extractSapUI5Libraries {
	my ($self, $tgtDir) = @_;
	require SDB::Install::Archive;
	my $extractionPath = File::Spec->catdir($tgtDir, 'resources', 'resident');
    my $pathToArchive = File::Spec->catfile($extractionPath, 'sapui5libs.tgz');

	if(-d File::Spec->catdir($extractionPath, 'sapui5libs')){ # resources were not deleted, nothing to do
		$self->getMsgLst()->addMessage("SAPUI5 libraries are up to date");
		$self->getMsgLst()->addMessage("Extraction of file '$pathToArchive' skipped");
		return 1;
	}

    if( ! -e $pathToArchive ) {
        $self->setErrorMessage ("SAPUI5 libraries not found at '$pathToArchive'");
        return undef;
    }
	my $archive = new SDB::Install::Archive ($pathToArchive);

	if (!$isWin){
		my $user = new SDB::Install::NewDBUser($self->{_instconfig}->getValue('SID'));
		$archive->set_owner($user->uid(), $user->gid());
	}

	if (!$archive->Extract($extractionPath)) {
		$self->getErrMsgLst()->addError("Failed to extract SAPUI5 libraries from archive '$pathToArchive' to '$extractionPath'");
		return undef;
	}
	$self->getMsgLst()->addMessage("Extracted SAPUI5 libraries from archive '$pathToArchive' to '$extractionPath'");
	return 1;
}

sub _extractAdminHelpResources {
    require SDB::Install::Archive;

    my ($self, $tgtDir) = @_;
    my $extractionPath = File::Spec->catdir($tgtDir, 'resources', 'resident');
    my $pathToArchive = File::Spec->catfile($extractionPath, 'adminhelp.tgz');

    if(-d File::Spec->catdir($extractionPath, 'adminhelp')){ # resources were not deleted, nothing to do
        $self->getMsgLst()->addMessage("$DOCUMENTATION_UI_STRING resources are up to date");
        $self->getMsgLst()->addMessage("Extraction of file '$pathToArchive' skipped");
        return 1;
    }
    if( ! -e $pathToArchive ) {
        $self->setErrorMessage ("$DOCUMENTATION_UI_STRING resources not found at '$pathToArchive'");
        return undef;
    }

    my $archive = new SDB::Install::Archive ($pathToArchive);

    if (!$isWin){
        my $user = new SDB::Install::NewDBUser($self->{_instconfig}->getValue('SID'));
        $archive->set_owner($user->uid(), $user->gid());
    }
    if (!$archive->Extract($extractionPath)) {
        $self->getErrMsgLst()->addError("Failed to extract $DOCUMENTATION_UI_STRING resources from archive '$pathToArchive' to '$extractionPath'");
        return undef;
    }
    $self->getMsgLst()->addMessage("Extracted $DOCUMENTATION_UI_STRING resources from archive '$pathToArchive' to '$extractionPath'");
    return 1;
}

sub _findAndProcessFilesToCopy {
	my ($self, $stackKitPath, $tgtDir, $directories, $copy_list) = @_;
	
	for my $directory (@{$directories}) {
        my $directoryPath = File::Spec->catfile($tgtDir, $directory);
		$self->makeDir($directoryPath);	
		
	    my $dirWalker = SDB::Install::DirectoryWalker->new(undef, undef, undef, 
	                                                       sub {
	                                                           return $_[6] && basename($_[3]) eq $CONFIGURATIONS_DIR ? 1 : 0; 
	                                                        }, undef, undef,
	                                                       sub {
                                                               my $tgt = File::Spec->catfile($directoryPath, $_[4]);
	                                                           if($_[6] && !defined $self->makeDir($tgt)) {
	                                                           		return undef;
	                                                           }
	                                                           if (!$_[6] && ( basename($_[3]) ne $CONFIGURATIONS_DIR || $_[4] =~ /\.cfg$/i )){
                                                                   push @{$copy_list}, [File::Spec->catfile($_[3], $_[4]), $tgt];
	                                                           }
	                                                           return 1;
	                                                        }, undef, undef, 1, 1, 0);
        my $rc = $dirWalker->findAndProcess(File::Spec->catfile($stackKitPath, $directory));
		if(not defined $rc) {
	        $self->getErrMsgLst()->addError($dirWalker->getErrorString());
	        return $rc;
	    }    
	}
	
    return 1;
}

# In case of installation of the HDBLCM component, the getInstalledComponent will return undef,
# but we still need a way of getting the installed component so we can uninstall it if the verification fails
sub _constructInstalledComponent {
    my ($self, $instconfig) = @_;
    my $instruntimeDir = File::Spec->catfile($self->_getInstallationPath($instconfig), 'instruntime');
    my $manifestPath = File::Spec->catfile($instruntimeDir, 'manifest');
    my $manifest = LCM::ResidentInstallerManifest->new($manifestPath);
    return LCM::Component::Installed->new($instruntimeDir, $manifest);
}

sub _shouldValidateInstaller {
    my ($self, $instconfig) = @_;
    my $shouldVerifySignature = $instconfig->getValue('VerifySignature');
    my $shouldIgnoreVerificationErrors = $instconfig->getIgnore('verify_signature');
    return $shouldVerifySignature && !$shouldIgnoreVerificationErrors;
}

sub _updateComponent {
    my ($self, $instconfig) = @_;
    my $targetPath = $self->getInstSidPath($instconfig);
    my $cmpName = $self->getComponentName();

    if(!$self->_cleanUpOldInstallation($targetPath)) {
        $self->setErrorMessage ('Cleaning up the old installation failed', $self->getErrMsgLst());
        return undef;
    }

    if (!$self->_installResidentHdblcm($instconfig)) {
        $self->setErrorMessage("Installing '$cmpName' failed.", $self->getErrMsgLst());
        return undef;
    }

    return 1;
}

sub _updateWithVerification {
    my ($self, $instconfig) = @_;
    my $cmpName = $self->getComponentName();
    my $installedComponent = $self->getInstalledComponent();
    my $saveCtxt = $installedComponent->setMsgLstContext($self->getMsgLstContext());

    if (!$installedComponent->backupInstallation()) {
        $self->setErrorMessage("Failed to backup old copy of $cmpName", $self->getErrMsgLst());
        return undef;
    }

    return undef if(!$self->_updateComponent($instconfig));

    my $rc = $self->_verifyInstallerAuthenticity($installedComponent);
    if (!$rc) {
        $installedComponent->uninstallComponent($instconfig);
        $installedComponent->restoreBackup();
    } else {
        $installedComponent->removeBackup();
    }

    $installedComponent->setMsgLstContext($saveCtxt);
    return $rc;
}

sub _cleanUpOldInstallation {
    my ($self, $tgtDir) = @_;
    my $compName = $self->getComponentName();
    my $mesg = "Cleaning up old installation of $compName...";
    my $imsg = $self->getMsgLst()->addProgressMessage($mesg);
    
    my $hdblcmInstallationDir = File::Spec->catfile($tgtDir, $gDirNameResidentInstaller);
    my $resourcesDir = File::Spec->catfile($hdblcmInstallationDir, 'resources');
    my $nonresidentDir = File::Spec->catfile($resourcesDir, 'nonresident');
    my $resourcesDirRelativePath = File::Spec->catfile($gDirNameResidentInstaller, 'resources');
    my $doNotDeleteList = [];

    if (-d $nonresidentDir) { # fill list only if resources/nonresident directory exists (i.e. installed hdblcm supports web ui)
        $doNotDeleteList = [$hdblcmInstallationDir, $resourcesDir, $nonresidentDir];
    }

    my $installationKitPath = $self->_getInstallationKitPath();
    my $targetResidentResourcesDir = File::Spec->catfile($installationKitPath, 'resources', 'resident');
    my $residentResourcesDir = File::Spec->catfile($resourcesDir, 'resident');
    my $isUi5UpToDate = isInstalledUi5UpToDate($targetResidentResourcesDir, $residentResourcesDir);
    my $isAdminHelpUpToDate = isInstalledAdminHelpToDate($targetResidentResourcesDir, $residentResourcesDir);

    if($isUi5UpToDate){
        push(@{$doNotDeleteList}, $residentResourcesDir);
        push(@{$doNotDeleteList}, File::Spec->catfile($residentResourcesDir, 'sapui5libs'));
        push(@{$doNotDeleteList}, File::Spec->catfile($residentResourcesDir, 'ui5version'));
    }
    if($isAdminHelpUpToDate){
        push(@{$doNotDeleteList}, $residentResourcesDir);
        push(@{$doNotDeleteList}, File::Spec->catfile($residentResourcesDir, 'adminhelp'));
        push(@{$doNotDeleteList}, File::Spec->catfile($residentResourcesDir, 'adminhelpversion'));
    }

    my $matcherSub = $self->_getCleanUpMatcherSubroutine($resourcesDirRelativePath, $isUi5UpToDate, $isAdminHelpUpToDate);
    my $processSub = $self->_getCleanUpProcessingSubroutine($doNotDeleteList);
    my $dirWalker = SDB::Install::DirectoryWalker->new(undef, undef, undef, $matcherSub, undef, undef, $processSub, undef, undef, 0, 1, 0 );

    $dirWalker->setMsgLstContext([$imsg->getSubMsgLst()]);
    my $rc = $dirWalker->findAndProcess($tgtDir);
    if(not defined $rc) {
        $self->getErrMsgLst()->addError($dirWalker->getErrorString());
        return undef;
    }

    return 1;
}

sub _getCleanUpMatcherSubroutine {
	my ($self, $resourcesPath, $isUi5UpToDate, $isAdminHelpUpToDate) = @_;
    my $nonresidentResourcesPath = File::Spec->catfile($resourcesPath, 'nonresident');
    my $residentUi5Path = File::Spec->catfile($resourcesPath, 'resident', 'sapui5libs');
    my $residentAdminHelpPath = File::Spec->catfile($resourcesPath, 'resident', 'adminhelp');

	return sub {
		my ($rootPath, $relentry) = ($_[3], $_[4]);
		my $isNonresidentSubdir = ($relentry =~ /^\Q$nonresidentResourcesPath\E/);
		my $isResidentUi5Subdir = ($relentry =~ /^\Q$residentUi5Path\E/);
		my $isResidentAdminHelpSubdir = ($relentry =~ /^\Q$residentAdminHelpPath\E/);
		my $matchesResidentIsntallerDirBasename = ($relentry =~ /^$gDirNameResidentInstaller($|\\|\/)/);
		my $matches = $isNonresidentSubdir || !$matchesResidentIsntallerDirBasename || ($isUi5UpToDate && $isResidentUi5Subdir) || ($isAdminHelpUpToDate && $isResidentAdminHelpSubdir); 

		return $matches;
	};
}

sub _getCleanUpProcessingSubroutine {
	my ($self, $doNotDeleteList) = @_;
	return sub {
		my($errorMsgLst, $msgLst, undef, $rootDir, $relentry, $entry, $isDir) = @_;
        my $fullpath = File::Spec->catfile($rootDir, $relentry);

		# Do not delete nonresident web resources
		return 1 if($fullpath ~~ @{$doNotDeleteList});

		my $keyWord = $isDir ? 'directory' : 'file';
		my $deleteResult = $isDir ? rmdir($fullpath) : unlink($fullpath);

		if(! $deleteResult) {
			$errorMsgLst->addError("Cannot delete $keyWord '$fullpath': $!");
			return undef;
		}
		$msgLst->AddMessage(ucfirst($keyWord) . " '$fullpath' deleted") if ($isDir);
		return 1;
	};
}

sub isDirectoryOwnedByRoot{
	my($self,$directory) = @_;
	foreach my $dir_owned_by_root(@directories_to_sign){
		if( index($directory,$dir_owned_by_root) != -1 ){
			return 1;
		 }
	}
	 return 0;
}

sub isFileOwnedByRoot{
	my($self,$file) = @_;
	if( exists($FILES_OWNED_BY_ROOT{basename($file)})){
    	return 1;
    }
    if( $self->isDirectoryOwnedByRoot(MyRealPath($file))){
    	return 1;
    }
    return 0;;
}

sub copyFile{
    my($self,$filesToCopy) = @_;
    if(not -e $filesToCopy->[0]){
        $self->getMsgLst()->AddWarning("Cannot copy file '$filesToCopy->[0]': '$filesToCopy->[0]' is missing.");
        return 1;
    }
    my $fileCfg = {};
    if(!$self->isFileOwnedByRoot($filesToCopy->[0])){
        $fileCfg = $self->_createSidAdmFileCfg();
    }
    my $rc = copy_file(@$filesToCopy, $fileCfg);
    if (!defined $rc) {
        $self->getErrMsgLst()->addError("Cannot copy file '$filesToCopy->[0]' to '$filesToCopy->[1]'", $fileCfg);
        return undef;
    }
    $rc = $self->_changeFilePermissions($filesToCopy->[1]);
    if (!defined $rc) {
        return undef;
    }
    return $rc;
}

sub _createSidAdmFileCfg{
	my($self) = @_;
	my $fileCfg;
	if (!$isWin){
		my $user = new SDB::Install::NewDBUser($self->{_instconfig}->getValue('SID'));
        $fileCfg->{uid} = $user->uid();
        $fileCfg->{gid} = $user->gid();
	}
	return $fileCfg;
}

sub makeDir{
	my($self,$directoryPath) = @_;
	my $isOwnedByRoot = $self->isDirectoryOwnedByRoot($directoryPath);
	my $fileCfg = $isOwnedByRoot ? {} : $self->_createSidAdmFileCfg();

	$fileCfg->{mode} = 0755;
	if(!defined makedir($directoryPath, $fileCfg)) {
		$self->getErrMsgLst()->addError("Could not create directory '$directoryPath'.", $fileCfg);
		return undef;
	}
}

sub _changeFilePermissions{
	my($self, $file) = @_; 

	return 1 if(!-e $file || !defined($file));

	my $fileName = basename($file);
	my $mode = exists($EXECUTABLE_FILES{$fileName}) ? 0755 : 0644;

	if (!chmod ($mode, $file)){
		$self->getErrMsgLst()->addError("Cannot change mode of file '$file': $!");
		return undef;
	}
	return 1;
}

1;
