package LCM::Component::Installable::XS2Application;

use strict;
use File::Basename qw (dirname);
use SDB::Install::NewDBUser;
use SDB::Install::System qw(isAdmin getDirectoryFilenames);
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw($gShortProductNameXS2);
use LCM::App::ApplicationContext;
use SDB::Install::Persistence::XMLGenerator;
use File::Spec;
use Cwd qw (realpath);
use LCM::FileUtils qw(writeToFile);

use parent 'LCM::Component::Installable';

sub installComponent {
    my ($self, $configuration) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext([ $msg->getSubMsgLst() ]);

    $self->initProgressHandler();
    my $returnCode = $self->_deployXsApplication($configuration);

    $msg->endMessage(undef, $self->_getEndMessage($returnCode));
    $self->setMsgLstContext($saveCntxt);
    return $returnCode;
}

sub updateComponent {
    my ($self, $configuration) = @_;
    my $isSameVersionDeployed = $self->_isAlreadyDeployed($configuration);
    if ($isSameVersionDeployed && !$configuration->getIgnore('check_version')) {
        my $cmpName = $self->getComponentName();
        my $version = $self->getVersion();
        $self->getMsgLst()->addMessage("Version '$version' of '$cmpName' is already deployed. Skipping redeployment...");
        return 1;
    }
    return $self->installComponent($configuration);
}

sub requireSystemRestart {
	return 0;
}

sub getSlppLogFileName {
	my ($self) = @_;
	return $self->getManifest()->getComponentKey() . 'log';
}

sub isUpdate {
	my ($self) = @_;
	my $applicationContext = LCM::App::ApplicationContext::getInstance();
	my $configuration = $applicationContext->getConfiguration();

	return 0 if(!defined($configuration));
    my $scm = $configuration->getSystemComponentManager();
    return 0 if !defined($scm);

	my $componentKey = $self->getManifest()->getComponentKey();
	return defined($scm->getComponentByKeyName($componentKey)) ? 1 : 0;
}

sub _isAlreadyDeployed {
    my ($self, $configuration) = @_;
    my $scm = $configuration->getSystemComponentManager();
    my $installedComponent = $scm ? $scm->getComponentByKeyName($self->getComponentKeyName()) : undef;
    return $installedComponent && $installedComponent->getVersionObject()->isEqual($self->getVersionObject());
}

sub _getPersistenceFilePath {
    my ($self, $sapSystem) = @_;
    my $globalSidDirectory = $sapSystem->get_globalSidDir() || $sapSystem->getUsrSapSid(); # Handle the case where /<sapmnt>/<SID> is /usr/sap/<SID>
    return File::Spec->catfile($globalSidDirectory, 'pending_xs_components.xml');
}

sub _createPendingFile {
    my ( $self, $instconfig) = @_;

    my $actionString = $self->isUpdate() ? 'update' : 'install';
    my $componentName = $self->getComponentName();
    my $currentStepString =
      $self->isUpdate() ? 'Update ' : 'Install ' . $componentName;

    my $xmlGenerator = new SDB::Install::Persistence::XMLGenerator();
    $xmlGenerator->setTargetVersionString( $self->getVersion() );
    $xmlGenerator->setComponentNameString($componentName);
    $xmlGenerator->setCurrentStepString($currentStepString);
    $xmlGenerator->setActionString($actionString);
    my $xmlString = $xmlGenerator->generatePersistenceXMLString();

    my $errorList = new SDB::Install::MsgLst();
    my $sid  = $instconfig->getSID();
    my $sapSystem = $instconfig->getSAPSystem();
    my $user = $sapSystem->getUser();
    my $uid  = $user->uid();
    my $gid  = $user->gid();

    my $persistenceFilePath = $self->_getPersistenceFilePath($sapSystem);
    if(!writeToFile($persistenceFilePath, $xmlString, 0644, $uid, $gid, $errorList)){
        $self->getErrMsgLst()->addError("Failed to write file '$persistenceFilePath' to the file system", $errorList);
        return undef;
    }

    $self->getMsgLst()->addMessage("File '$persistenceFilePath' was written to the file system.");
    return $persistenceFilePath;
}

sub _deployXsApplication {
    my ($self, $configuration) = @_;
    my $executor = $self->_createProcessExecutor($configuration);

    $executor->setOutputHandler($self->getProgressHandler());

    my $persistenceFilePath = $self->_createPendingFile($configuration);
    my $exitCode = $executor->executeProgram();
    $self->getMsgLst()->addMessage(undef, $executor->getMsgLst());

    if (!defined $exitCode || $exitCode) {
        my $errMsgLst = $executor->getErrMsgLst();
        my $actionErr    = $self->isUpdate() ? 'Update' : 'Installation';
        my $errorMessage = "$actionErr of " . $self->getComponentName() . ' failed';
        $self->appendErrorMessage($errorMessage, ($errMsgLst->isEmpty() ? undef : $errMsgLst));
        return undef;
    } elsif ( ! $self->isUpdate() ) {
        $self->detectCockpitAppsUrls($configuration, $executor->getOutputLines());
    }
    if(!unlink($persistenceFilePath)) {
        $self->getErrMsgLst()->addError("Cannot delete file $persistenceFilePath: $!");
        return undef;
    }
    return 1;
}

sub detectCockpitAppsUrls {
    my ($self, $instconfig, $outputLines) = @_;
    my $pattern = "Application \"(cockpit-admin-web-app|cockpit-web-app)\" started and available at \"(.*)\"";
    
    return undef if (!defined($outputLines));

    for my $line (reverse(@{$outputLines})){
        $instconfig->{$1} = $2 if($line =~ /$pattern/i);
    }
}

sub _createProcessExecutor {
    my ($self, $configuration) = @_;
    my $sid = $configuration->getSID();
    my $targetDir = $configuration->getValue('Target');
    my $command = File::Spec->catfile($targetDir, $sid, 'xs', 'bin', 'xs');
    my $arguments = $self->_buildXSApplicationArguments($configuration);
    my ($uid, $gid) = (undef, undef);

    if (!$isWin && isAdmin()) {
        my $user = new SDB::Install::NewDBUser($sid);
        ($uid, $gid) = ($user->uid(), $user->gid());
    }
    return new LCM::ProcessExecutor($command, $arguments, undef, dirname($command), undef, $uid, $gid);
}

sub _buildXSApplicationArguments {
    my ($self, $configuration) = @_;
    my @arguments = ('install', $self->getPath(), '-o', 'ALLOW_SC_SAME_VERSION');
    my $shouldUseNoStart = $self->_shouldStartAfterDeployemnt($configuration) ? 0 : 1;
    my $extensionDescriptorsList = $self->_getExtentionDescriptors($configuration) || [];
    my $extensionDescriptorsCSV = join(',', @{$extensionDescriptorsList});
    my $hasExtensionDescriptors = length($extensionDescriptorsCSV) > 0;

    push(@arguments, '--no-start') if($shouldUseNoStart);
    push(@arguments, ('-e', $extensionDescriptorsCSV)) if ($hasExtensionDescriptors);

    return \@arguments;
}

sub _shouldStartAfterDeployemnt {
    my ($self, $configuration) = @_;
    my $ownKeyName = $self->getComponentKeyName();
    my $noStartKeyNamesCSV = $configuration->getValue('XSComponentsNoStart') || '';
    my @selectedKeyNamesNoStart = split(/\s*,\s*/, $noStartKeyNamesCSV);
    my $isSelectedForNoStart = grep { $_ eq $ownKeyName || $_ eq 'all' } @selectedKeyNamesNoStart;
    my $isNoStartParameterActive = !$configuration->isSkipped('XSComponentsNoStart');

    return ($isNoStartParameterActive && $isSelectedForNoStart) ? 0 : 1;
}

sub _getDefaultExtentionDescriptorLocation{
	my ($self, $configuration) = @_;
	my $Xs2ApplicationComponents = $self->getComponentManager()->getXs2ApplicationComponents();
	my @selectedXSAppDirs = ();
	
	foreach(@$Xs2ApplicationComponents){
		if( $_->isComponentSelected()){
			my ($volume,$directories,$file) = File::Spec->splitpath($_->getPath());
			push( @selectedXSAppDirs , $directories) if(! grep(/^$directories$/,@selectedXSAppDirs) );
		}
	}
	return \@selectedXSAppDirs;
}

sub _getExtentionDescriptors{
	my ($self, $configuration) = @_;
	my $extensionDescriptorFilesNames;
	my @extensionDescriptorFiles = ();
	if(!$configuration->isSkipped('XSComponentsCfg') && $configuration->hasValue('XSComponentsCfg')){
	     my $xsComponentsCfgValue = $configuration->getValue('XSComponentsCfg');
	     if ($xsComponentsCfgValue ne "") {
             my $extentionDescriptorDir = realpath($xsComponentsCfgValue);
             $extensionDescriptorFilesNames = $self->_getExntionDescriptorsFromDir($extentionDescriptorDir);
             @extensionDescriptorFiles = map(File::Spec->catfile($extentionDescriptorDir,$_),@$extensionDescriptorFilesNames)	         
	     }
    }
    my $defaultExtentionDescriptorLocation = $self->_getDefaultExtentionDescriptorLocation();
    foreach my $location (@$defaultExtentionDescriptorLocation){
    	my $locationDescriptors = $self->_getExntionDescriptorsFromDir($location);
    	foreach my $descriptor (@$locationDescriptors){
    		push( @extensionDescriptorFiles , File::Spec->catfile($location,$descriptor)) if(! grep(/^$descriptor$/,@$extensionDescriptorFilesNames) );
    	}
    }
    return \@extensionDescriptorFiles
}

sub _getExntionDescriptorsFromDir{
	my ($self, $extentionDescriptorDir) = @_;
	my $filenames = getDirectoryFilenames($extentionDescriptorDir);
	my @extensionDescriptorFiles = grep {$_ =~ /^(.+)\.[Mm][Tt][Aa][Ee][Xx][Tt]$/} @$filenames;
	return \@extensionDescriptorFiles;
}

sub _getEndMessage {
    my ($self, $returnCode) = @_;
    my $action = $self->isUpdate() ? "Update" : "Installation";
    my $suffix = ($returnCode == 1) ? ' finished' : ' failed';

    return sprintf("%s of %s %s", $action, $self->getComponentName(), $suffix);
}

1;