package LCM::Component::Installed;

use strict;
use SDB::Install::ServiceManager;
use SDB::Install::Globals qw($gProductNameEngine);
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::System qw (removeEmptyDirs);
use File::Basename qw (dirname);
use LCM::Component;

our @ISA = qw (LCM::Component);


our $hdbUninstallExecutable = 'hdbuninst';
our $hdbRenameExecutable = 'hdbrename';
our $hdbRegExecutable = 'hdbreg';

if ($isWin){
    $hdbUninstallExecutable .= '.exe';
    $hdbRenameExecutable = undef;
    $hdbRegExecutable = undef;
}

sub new {
    my $self = shift->SUPER::new (@_);
    my $manifest = $self->{manifest};
    if (!defined $manifest){
        return $self;
    }
    my $classOfComponent;
    my $isSidAdmExecution = $self->isSidAdmUserExecution();
    if ($manifest->isServer()){
        $classOfComponent = 'LCM::Component::Installed::HDBServer';
    } elsif ($manifest->isClient()){
        $classOfComponent = 'LCM::Component::Installed::HDBClient';
    } elsif ($manifest->isStudio()){
        $classOfComponent = 'LCM::Component::Installed::HDBStudio';
    } elsif ($manifest->isLCAPPS()){
        $classOfComponent = $isSidAdmExecution ? 'LCM::Component::Installed::HDBServerPlugin::HDBServerPluginSidAdm' :
                                                 'LCM::Component::Installed::HDBServerPlugin::HDBServerPluginRoot';
    } elsif ($manifest->isAFL()){
        $classOfComponent = $isSidAdmExecution ? 'LCM::Component::Installed::HDBServerPlugin::HDBServerPluginSidAdm' :
                                                 'LCM::Component::Installed::HDBServerPlugin::HDBServerPluginRoot';
    } elsif ($manifest->isHLM()){
        $classOfComponent = 'LCM::Component::Installed::HLM';
    } elsif ($manifest->isSDA()){
        $classOfComponent = 'LCM::Component::Installed::SDA';
    } elsif ($manifest->isInstaller()){
        $classOfComponent = 'LCM::Component::Installed::ResidentInstaller';
    } elsif ($manifest->isLMStructure()){
        $classOfComponent = 'LCM::Component::Installed::LMStructure';
    } elsif ($manifest->isCockpitStack()){
        $classOfComponent = 'LCM::Component::Installed::CockpitStack';
    } elsif ($manifest->isStreaming()){
        $classOfComponent = 'LCM::Component::Installed::Streaming';
    } elsif ($manifest->isES()){
        $classOfComponent = 'LCM::Component::Installed::ES';
    } elsif ($manifest->isAccelerator()){
        $classOfComponent = 'LCM::Component::Installed::AcceleratorInstalled';
    } elsif ($manifest->isRDSync()){
        $classOfComponent = 'LCM::Component::Installed::RDSync';
    } elsif ($manifest->isXS2()){
        $classOfComponent = $isSidAdmExecution ? 'LCM::Component::Installed::XS::XS2Sidadm' :
                                                 'LCM::Component::Installed::XS::XS2Root';
    } elsif ($manifest->isXS2Application()) {
        $classOfComponent = 'LCM::Component::Installed::XS2Application';
    } elsif ($manifest->isServerPlugin()){
        my $componentKey = $manifest->getComponentKey();
        if(!$componentKey){
            $self->setErrorMessage("Component manifest of '" . $self->getComponentName() . "' is missing property 'component-key'.");
            return $self;
        }
        $classOfComponent = $isSidAdmExecution ? 'LCM::Component::Installed::HDBServerPlugin::HDBServerPluginSidAdm' :
                                                 'LCM::Component::Installed::HDBServerPlugin::HDBServerPluginRoot';
    } elsif ( $manifest->isReferenceData() ){
        $classOfComponent = 'LCM::Component::Installed::ReferenceData';
    } elsif ( $manifest->isLSS() ){
        $classOfComponent = 'LCM::Component::Installed::LSS';
    }

    if (!defined $classOfComponent){
        $self->setErrorMessage('Unknown component ' . $self->getComponentName());
        return $self;
    }

    eval ("require $classOfComponent;");
    if ($@){
        $self->setErrorMessage ("Cannot load component class '$classOfComponent': $@");
        return $self;
    }
    bless ($self, $classOfComponent);
    return $self;
}

sub uninstallComponent {
    $_[0]->setErrorMessage ("Uninstallation of component '"
                            . $_[0]->getComponentName . "' is not implemented");
    return undef;
}

# get the directory of the hdb* installer tools
sub getHdbInstallerDir{
    my ($self) = @_;
    if (!defined $self->{hdbInstallerDir}){
        $self->{hdbInstallerDir} = $self->{manifestDir};
    }
    return $self->{hdbInstallerDir};
}

sub getHdbUninstallExecutable{
    my ($self) = @_;
    if (!defined $self->{hdbUninstallExecutable}){
        $self->{hdbUninstallExecutable} = $self->getHdbInstallerDir () .
                $path_separator . $hdbUninstallExecutable;
    }
    return $self->{hdbUninstallExecutable};
}

sub getInstallationPath{
    return $_[0]->{manifestDir};
}


sub requireSystemRestart {
	return 0;
}

sub removeComponentDirectory {
    my ($self, $componentDirectory) = @_;
    my $installerDirectory = "$componentDirectory$path_separator" . "installer";

    if(-d $installerDirectory) {
        rmdir($installerDirectory);
    }

    removeEmptyDirs($componentDirectory);
    if(-e $componentDirectory) {
        $self->getMsgLst()->addWarning("Failed to delete directory '$componentDirectory'");
    }
}

sub getComponentRoles {
	return undef;
}

sub checkComponentHostsExist {
	require LCM::Hosts::HostsDetector;

	my ($self,$instconfig) = @_;
	my $hostDetector = new LCM::Hosts::HostsDetector ($instconfig->{installer}, $instconfig->getMsgLstContext());
	my $componentName = $self->getComponentName();
	my $componentRoles = $self->getHostRoles();

	if($hostDetector->hasParsingErrors()){
		$instconfig->PushError("Cannot detect $componentName hosts.");
		return 0;
	}

	return 1 if(! defined $componentRoles);

	my @hosts = @{$hostDetector->getHostsWithRoles($componentRoles)};

	return 1 if(scalar(@hosts) == 0);

	my $error = sprintf("There are existing %s hosts on the system: '%s'. To uninstall %s,you must first remove these hosts or the %s roles from them.", $componentName, join(', ', @hosts), $componentName,$componentName); 
	$instconfig->PushError($error);
	return 0;
}

sub _shouldAddWarningOnFailure {
    my ($self, $configuration) = @_;
    my $systemComponentManager = $configuration->{componentManager}->getSystemComponentManager();
    my $componentName = $self->getComponentName();
    if ($systemComponentManager->isHDBServerComponentSelected() && $componentName ne $gProductNameEngine) {
        return 1;
    }
    return 0;
}

sub _handleUninstallationExitCode {
    my ($self, $exitCode, $configuration, $errMsgLst, $customErrorMessage) = @_;
    return if (defined $exitCode && !$exitCode);

    my $errorMessage = ($customErrorMessage) ? $customErrorMessage
                                             : sprintf('Uninstallation of %s failed', $self->getComponentName());
    if ($self->_shouldAddWarningOnFailure($configuration)) {
        $self->getMsgLst()->addWarning($errorMessage, $errMsgLst); #is executor always used
    } else {
        $self->setErrorMessage($errorMessage, $errMsgLst);
    }
    return;
}

sub getProgressMsg {
	return '';
}

sub isInstalled{
	my $self = shift;
	return 1;
}

sub cleanComponentDaemonIniEntries {
    my ($self, $config) = @_;
    my $manifest = $self->getManifest();
    my $servicesToRestart = $manifest->getServicesToRestart();
    return if (scalar @$servicesToRestart == 0);

    my $serviceManager = SDB::Install::ServiceManager->new();
    $serviceManager->setMsgLstContext($self->getMsgLstContext);

    $serviceManager->cleanServicesEntriesInDaemonIni($config, $servicesToRestart);
}

1;
