package LCM::ComponentManager;

use strict;
use LCM::FileUtils qw(MyRealPath);
use SDB::Install::SysVars qw($path_separator $currentPlatformName);
use base 'SDB::Install::Base';
use LCM::HDBComponentsConfigurator;
use SDB::Install::DebugUtilities qw(dumpThings);
use LCM::Component::Registrable qw($RETURN_CODE_SUCCESS $RETURN_CODE_ERROR);
use LCM::Component::Installable::XS2Application;
use SDB::Install::DependencyParser;
use SDB::Install::Globals qw ($gKeynameEngine $gKeynameXS2);
use LCM::Utils::XsApplicationDependency::InclusiveDependency;
use Cwd qw (realpath);
use File::Basename qw(dirname);
use LCM::App::UpdateSystemUtils qw(getInstalledServerManifest);


sub new {
    my ( $class, $instconfig ) = @_;
    my $self = $class->SUPER::new ();
    $self->{instconfig} = $instconfig;
    $self->{components} = {};
    return $self;
}

sub addComponent{
    my ($self, $component) =  @_;
    my $manifest = $component->getManifest ();

    if (!$manifest->isPlatformSupported($currentPlatformName)){
    	my $componentPlatforms = join(", ",$manifest->getSupportedPlatforms());
        $self->getMsgLst()->addMessage ("Skipping component ". $component->getComponentName() . " ($componentPlatforms)" );
        return 1;
    }
    
    my $keyName = $manifest->getValue ('keyname');
    if (!$keyName){
        $self->appendErrorMessage ("Manifest does not contain 'keyname'");
        return 0;
    }

    if ( defined $self->{instconfig} ) {
	    if ( $self->{instconfig}->can('isScopeInstance') && $self->{instconfig}->isScopeInstance()
	        &&
	         ! $component->isCompatibleWithScopeInstance() ) {
	        return 0;
	    }
    }
	if ( defined $self->{components}->{$keyName} ) {
    	my $foundComponent = $self->{components}->{$keyName};
    	if($self->_shouldReturnIfDuplicateComponents($foundComponent, $component, $keyName)){
    		return 1;
    	}
    }
    
    my $componentBatchKey = $component->getComponentBatchKey();
    if (defined $componentBatchKey){
        $self->{componentsByBatchKey}->{$componentBatchKey} = $component;
    }
    $self->{components}->{$keyName} = $component;
    $self->_setInstallationKitProperty($self->getConfiguration(), $component);
    return 1;
}

# Added because of TIPHANA15IT10-2040
# a 'kit' property is needed in the CollectOtherHostInfos
# method in AnyConfig in case of non-resident hdblcm
sub _setInstallationKitProperty {
    my ($self, $configuration, $component) = @_;
    return if (!$component->isServer);
    $configuration->{kit} = new SDB::Install::Kit(dirname($component->getManifest()->getFileName()));
}

sub addComponentsToLog {
    ...
}

sub _shouldReturnIfDuplicateComponents {
    my ($self, $foundComponent, $newComponent, $keyName) = @_;
    my $addedComponentPath = $foundComponent->getPath();
    my $newComponentPath = $newComponent->getPath ();
    my $newRealPath = (-e $newComponentPath) ? MyRealPath($newComponentPath) : $newComponentPath;
    my $addedRealPath = (-e $addedComponentPath) ? MyRealPath($addedComponentPath) : $addedComponentPath;

    if ($newRealPath ne $addedRealPath) {
        $self->appendErrorMessage ("Component '$keyName' detected under '$newRealPath' was already found under '$addedRealPath'");
    }
    return 1;
}
sub clearDetectedComponents {
	my ($self) =  @_;
	
	$self->{componentsByBatchKey} = {};
	$self->{components} = {};
}

sub hasComponents{
    return keys (%{$_[0]->{components}}) ? 1 : 0;
}

sub getDetectedComponents{
	my ($self) = @_;
	return [ values(%{$self->{components}}) ];
}

sub getExternalDetectedComponents {
	my ($self) = @_;
	my @detectedComponents = @{$self->getDetectedComponents()};
	my @externalDetectedComponents = grep {! $_->isInternal()} @detectedComponents;
	return \@externalDetectedComponents;
}

sub selectAllComponents{
    my ($self, $select) = @_;
    if (!defined $select){
        $select = 1;
    }
    foreach my $component (values %{$self->{components}}){
        $component->selectComponent ($select);
    }
    return 1
}

sub isHDBServerComponentSelected {
    my $serverComponent = $_[0]->{components}->{$gKeynameEngine};
    return (defined $serverComponent && $serverComponent->isComponentSelected());
}

sub isComponentSelected {
    my ($self, $componentKeyname) = @_;

    my $component = $self->{components}->{$componentKeyname};
    return (defined $component && $component->isComponentSelected());
}

sub getComponentByKeyName{
	return $_[0]->{components}->{$_[1]};
}

sub getComponentByBatchKey{
	return $_[0]->{componentsByBatchKey}->{$_[1]};
}

sub getAllComponents{
    my ($self) = @_;
    my $sortedComponentKeynames = $self->getSortedComponentKeynames ();
    my @result;
    foreach my $keyname (@$sortedComponentKeynames){
        if (defined $self->{components}->{$keyname}){
            push @result, $self->{components}->{$keyname};
        }
    }
    return \@result;
}

sub getDependantComponents {
	my ($self, $targetComponent) = @_;
	my $dependantComponents = [];

	for my $component (@{$self->getAllComponents()}){
		if($self->isDependantComponent($targetComponent, $component)){
			push(@{$dependantComponents}, $component);
		}
	}
	return $dependantComponents;
}

sub isDependantComponent {
	my ($self, $targetComponent, $component) = @_;

	if($targetComponent->isa('LCM::Component::Installable::XS2Application')){
		return $self->_isDependantXs2ApplicationComponent($targetComponent, $component);
	}

	my $parser = new SDB::Install::DependencyParser();
	my $targetManifest = $targetComponent->getManifest();
	my $targetKeyname = $targetManifest->getValue('keyname');
	my $targetVendor = $targetManifest->getValue('keyvendor');
	my $manifest = $component->getManifest();
	my $dependencyString = $manifest->getValue('required-components');

	return 0 unless($dependencyString && $parser->parse($dependencyString));

	for my $vendor (keys %{$parser->{parseTree}}) {
		for my $keyname (keys %{$parser->{parseTree}->{$vendor}}) {
			next if( ($targetVendor ne $vendor) || ($targetKeyname ne $keyname) );
			return 1 if($manifest->checkComponentDependency($targetManifest));
		}
	}
	return 0;
}

sub _isDependantXs2ApplicationComponent {
	my ($self, $targetComponent, $component) = @_;
	my $manifest = $targetComponent->getManifest();
	my $name = $targetComponent->getComponentName();
	my $technicalName = $manifest->getComponentKey();
	my $scvVersion = $manifest->getValue('release');
	my $spLelvel = $manifest->getValue('rev-number');
	my $patchLelvel = $manifest->getValue('rev-patchlevel');
	my $dependencyObject = new LCM::Utils::XsApplicationDependency::InclusiveDependency($name, $technicalName, $scvVersion, $spLelvel, $patchLelvel);
	my $targetSet = { $technicalName => $dependencyObject };
	my $dependencies = $component->getManifest()->getValue('dependencies-array') || [];

	for my $dependencyArray (@{$dependencies}){
		for my $dependencyObject (@{$dependencyArray}){
			next if(!$dependencyObject->isa('LCM::Utils::XsApplicationDependency::InclusiveDependency'));
			return 1 if($dependencyObject->isSatisfied($targetSet));
		}
	}
	return 0;
}

sub getSortedComponentKeynames{
	return [sort keys %{$_[0]->{components}}];
}

sub getSelectedComponents{
    my ($self) = @_;
    my @selectedComponents;
    foreach my $component (@{$self->getAllComponents ()}){
        if ($component->isComponentSelected ()){
            push @selectedComponents, $component;
        }
    }
    return \@selectedComponents;
}

sub isExternalComponentSelected {
    my ($self) = @_;

	my $selectedCmps = $self->getSelectedComponents();
	if (! defined $selectedCmps) {
		return 0;
	}

	my @externalSelectedCmps = grep { !($_->isInternal()) } @{$selectedCmps};

	return (scalar (@externalSelectedCmps) > 0) ? 1 : 0;
}

sub getServerPluginComponents{
    my ($self) = @_;
    my @pluginComponents;
    my @pluginComponentsWithDU;

    foreach my $componentKeyname (sort keys %{$self->{components}}){
        my $component = $self->getComponentByKeyName($componentKeyname);
        if($component->isServerPlugin()) {
            if(defined $component->can("shouldImportContent") && $component->shouldImportContent()) {
                push(@pluginComponentsWithDU, $component);
                next;
            }
            push(@pluginComponents, $component); 
        }
    }
    push(@pluginComponents, @pluginComponentsWithDU);
    return \@pluginComponents;
}

sub getSelectedServerPluginComponents {
	my ($self) = @_;
	my $serverPluginComponents = $self->getServerPluginComponents();
	my $selectedServerPluginComponents = [];

	for my $serverPluginComponent (@{$serverPluginComponents}) {
		if ($serverPluginComponent->isComponentSelected()) {
			push @{$selectedServerPluginComponents}, $serverPluginComponent;
		}
	}

	return $selectedServerPluginComponents;
}

sub getReferenceDataComponents{
    my ($self) = @_;
    my @referenceDataComponents;

    for my $componentKeyname (keys %{$self->{components}}){
        my $component = $self->getComponentByKeyName($componentKeyname);
        if ( $component->getManifest()->isReferenceData() ) {
            push(@referenceDataComponents, $component); 
        }
    }
    return \@referenceDataComponents;
}

sub getSelectedReferenceDataComponents {
	my ($self) = @_;
	my @selectedReferenceDataComponents = grep {$_->isComponentSelected()} @{$self->getReferenceDataComponents()};
	return \@selectedReferenceDataComponents;
}

sub existsSelectedServerPluginComponents {
	my ($self) = @_;
	my $selectedServerPluginComponents = $self->getSelectedServerPluginComponents();
	return (scalar @{$selectedServerPluginComponents}) > 0 ? 1 : 0
}

sub getXs2ApplicationComponents {
    my ($self) = @_;
    my @xsApplications;
    foreach my $componentKeyname (sort keys %{$self->{components}}){
        my $component = $self->getComponentByKeyName($componentKeyname);
        if ($component->getManifest()->isXS2Application()){
            push(@xsApplications, $component);
        }
    }
    return \@xsApplications;
}

sub getActionList {
	my ($self, $action) = @_;

	my @action_list;
    my $actionLabel;
    
    for (@{$self->getSelectedComponents()}){
    	if (defined $_->can("getInstallActionString")) {
    		$actionLabel = $_->getInstallActionString ();
    	} else {
            $actionLabel = $action;
        }
        push @action_list, $actionLabel . ' ' . $_->getComponentName ();
    }
    
    return \@action_list;
}

sub isCustomerRelease{
    my ($self) = @_;
    my $hanaServer = $self->getComponentByKeyName ($gKeynameEngine);
    if (!defined $hanaServer){
        return undef;
    }
    return $hanaServer->getManifest ()->isCustomerReleaseBranch();
}

sub isInternalBuild{
    my ($self) = @_;
    my $serverManifest =  $self->getInstallableServerManifest();
    my $installer = new LCM::Installer();
    if(! defined $serverManifest && $installer->isInstalled()){
        my $sapmnt = MyRealPath(File::Spec->catfile($installer->getRuntimeDir(), '..', '..', '..'));
        $serverManifest = getInstalledServerManifest(undef, $sapmnt);
    }
    return undef if(! defined $serverManifest);
    return $serverManifest->isInternalBuild();
}

sub getInstallableServerManifest {
    my ($self) = @_;
    my $appContext = LCM::App::ApplicationContext::getInstance();
    return $appContext->getServerManifest();
}

sub isSidAdmUserExecution {
	my ($self) = @_;
	return ($self->{_isSidAdmUserExecution}) ? 1 : 0;
}

sub setSidAdmUserExecution {
	my ($self, $value) = @_;
	$self->{_isSidAdmUserExecution} = $value;
}

sub isResidentComponentManager {
	return 0;
}

sub existsSelectedComponentInPendingState {
	my ($self, $configuration) = @_;
	my $selectedComponents = $self->getSelectedComponents();
    my $xs2Application = new LCM::Component::Installable::XS2Application(undef, undef, $configuration);
    push @$selectedComponents, $xs2Application;

	for my $selectedComponent (@{$selectedComponents}) {
		if ($selectedComponent->can('hasPersistenceFile') && $selectedComponent->hasPersistenceFile($configuration)) {
			return 1;
		}
	}
	return 0;
}

sub getComponentDirs {
    my ($self, $onlyNotSelected) = @_;

    my $components = $self->getAllComponents();
    my $xs2Components = $self->getXs2ApplicationComponents();
    push @{$components}, @{$xs2Components};
    my $componentDirs = [];
    my $path;
    my %addedXsapplicationDirs = ();

    for my $component (@{$components}) {
        if ($component->isInternal()  || ($onlyNotSelected && !$component->isComponentSelected())) {
            next;
        }
        $path = $component->getPath();
        next if (!defined $path);

        $path = realpath($path);
    # Also handles XSA application components that return
    # path to .zip file instead of their parent directory
        my $isXSAApplication = (-f $path && $path =~ /\.zip$/i) ? 1 : 0;
        $path = dirname($path) if ($isXSAApplication);
        push (@{$componentDirs}, $path) if (! exists($addedXsapplicationDirs{$path}));
        $addedXsapplicationDirs{$path}++ if ($isXSAApplication);
    }
    return $componentDirs;
}

sub getDefaultDetectedComponents {
	my ($self) = @_;
	return $self->{defaultLocation};
}

sub getConfiguration{
	my $self = shift;
	return $self->{instconfig};
}

sub getSignedComponentBatchKeys {
    my ($self) = @_;
    my $allComponents = $self->getAllComponents();
    my @result = ();
    for my $component (@$allComponents) {
        push(@result, $component->getComponentBatchKey()) if ($component->isSigned());
    }
    return \@result;
}

sub _isOnlyHdblcmSelected {
    my ($self) = @_;
    return (scalar (@{$self->getSelectedComponents()}) == 1);
}

1;
