package LCM::ComponentManager::MediumComponentManager;

use strict;
use File::Basename qw (dirname);
use LCM::Mid::DUParser;
use LCM::Component::Installable;
use SDB::Install::SysVars qw($path_separator $isWin $currentPlatformName);
use SDB::Install::DirectoryWalker;
use LCM::Component; # just to import component ids
use LCM::ResidentInstallerManifest;
use LCM::Landscape::LMStructureManifest;
use LCM::Manifests::GenericManifest;
use base qw (LCM::ComponentManager Exporter);
use SDB::Install::DebugUtilities qw(dumpThings);
use LCM::Manifests::XS2ApplicationManifest;
use File::Spec;
use LCM::FileUtils qw(listDirectory);

use Exporter;
use SDB::Install::Globals qw ($gDirNameEs
                              $gDirNameStreaming
                              $gDirNameRDSync
                              $gDirNameXS2
                              $gProductNameInstaller
                              $gDirNameReferenceData
                              $gKeynameEngine
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameAFL
                              $gKeynameLCA
                              $gKeynameHLM
                              $gKeynameSDA
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameRDSync
                              $gKeynameXS
                              $gKeynameXS2
                              $gKeynameOfficeClient
                              $gKeynameLMStructure
                              $gKeynameInstaller
                              $gKeynameCockpitStack
                              $gKeynameLSS
                              $gFlavourPlatform);

our @EXPORT = qw (@componentDirNames);

#es must be before the server in this list !!!
our @componentDirNames = ('packages',
                          $gDirNameEs,
                          $gDirNameStreaming,
                          $gDirNameRDSync,
                          $gDirNameXS2,
                          $gDirNameReferenceData,
                          'asehana',
                          'server',
                          'client',
                          'studio');

my $SUCCESS = 0;
my $ERR_MISSING_MANIFEST = 1;
my $ERR_DUPLICATE_COMPONENTS = 2;
my $ERR_COMPILING_COMPONENT_LIST = 3;
my $ERR_ONLY_SERVER_DETECTED = 4;

sub new {
    my ($class, $instconfig) = @_;
    my $self = $class->SUPER::new ();
    $self->{dvdComponents} = [];
    $self->{rootComponents} = [];
    $self->{directoryComponents} = [];
    $self->{defaultLocation} = [];
    $self->{instconfig} = $instconfig;
    $self->setCanPrintDetectedComponents(1);
    return $self;
}

sub detectComponentsByDVDRoot {
	my ($self,$dvdRootPath) = @_;
    my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
    $dvdRootPath = $app->getAbsPath ($dvdRootPath);
    $dvdRootPath = File::Spec->canonpath($dvdRootPath);

    my $msg = $self->getMsgLst ()->addMessage ("Checking installation medium '$dvdRootPath'");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    my $parser = LCM::Mid::DUParser->new($dvdRootPath);
    eval{
        $parser->parse();
    };
    if ($@){
        $self->setErrorMessage ("Error reading installation medium: $@");
        return 0;
    }
    my $data_units = $parser->getDataUnits();

    my ($manifestDir,$unitPath, $subDir);
    OUTER: for (keys %$data_units){
        $self->getMsgLst ()->addMessage ("Found data unit '$_'");
        $unitPath = $data_units->{$_}->{'path'};
        for $subDir ( @componentDirNames  ) {
            $manifestDir = $unitPath . $path_separator . $subDir;
            if (-d $manifestDir){
                my $isComponentDirs = 0;
                $self->detectComponentByManifestDir($manifestDir, $self->{dvdComponents}, $isComponentDirs);
                next OUTER;
            }
        }
        $self->_detectXS2ApplicationArchives($unitPath, $self->{dvdComponents});
    }

    $self->setMsgLstContext ($saveCntxt);
    $self->_compileComponentList();
    return 1;
}

sub _compileComponentList {
	my ($self) =  @_;

	my $msg = $self->getMsgLst()->addMessage ("Merging detected components");
	my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst(), undef]);

	#reset the components list
	$self->{components} = {};
	my @mergeOrder = ( $self->{directoryComponents}, $self->{rootComponents}, $self->{dvdComponents}, $self->{defaultLocation} );
	my $warningMessage = "";
	$self->{isOnlyServerDetected} = 1;
	
	for my $componentList (@mergeOrder) {
		for my $component (@$componentList) {
			if ( $component->isServer() ) {
				$self->{isServerDetected} = 1;
			} else {
				$self->{isOnlyServerDetected} = 0;
			}
			my $additionSuccessfull = $self->addComponent($component);
		    if (!$additionSuccessfull){
	    	    $self->setErrorMessage ('Cannot add component', $self->getErrMsgLst());
	        	$self->setMsgLstContext ($saveCntxt);
	    	} else {
	    	    my $warning = $self->hasDuplicatedComponentWarning($component);	
	    	    if($warning) {
	    	    	$warningMessage .= $warning;
	    	    }
	    	}
		}
	}

    my $detectedServerComponent = $self->getComponentByKeyName($gKeynameEngine);
    if(defined($detectedServerComponent)){
        my $appContext = LCM::App::ApplicationContext::getInstance();
        $appContext->setServerManifest($detectedServerComponent->getManifest());
    }
    if($warningMessage) {
    	$self->setWarningMessage($warningMessage);
    }
    
    my $rc;
	if ($self->hasComponents()){
        if (!$self->_addResidentInstallerComponent()) {
            $rc = $ERR_COMPILING_COMPONENT_LIST;
        } else {
            $self->_addLMStructureComponent();
            $rc = $SUCCESS;
        }
    } else {
    	if ( ! $self->handleDetectionOfServer() ) {
            $rc = $ERR_ONLY_SERVER_DETECTED;
    	} else {
            $rc = $ERR_COMPILING_COMPONENT_LIST;
    	}
    }
    
    $self->printDetectedComponents();
    return $rc;
}

# Override
sub addComponent{
    my ($self, $component) = (shift(), shift());
    my $componentKey = $component->getComponentBatchKey();
    my $blackListedComponentBatchKeyes = ['xsac_shine', 'xsac_xmla_int', 'xsac_ha_adp'];
    my $isComponentInBlackList = grep { $_ eq $componentKey } @{$blackListedComponentBatchKeyes};

    # Exclude SHINE app from the detected XSA components. See Bug 123462 for details.
    # Exclude XSAC_XMLA_INT and XSAC_HA_ADP. See TIPHANA15IT10-2378 for details.

    if($isComponentInBlackList){
        $self->getMsgLst()->addMessage(sprintf('Skipping component %s', $component->getComponentName()));
        return 1;
    }

    my $isInternalBuild = $self->isInternalBuild();
    if($component->isLSS() && ($self->isSidAdmUserExecution() || (defined $isInternalBuild && !$isInternalBuild))) {
# Installation and update of LSS is still not implemented via sidadm execution
        return 1;
    }

    return $self->SUPER::addComponent($component, @_);
}

sub handleDetectionOfServer {
	my ( $self ) = @_;
	my $rc = 1;
	if ( $self->{isServerDetected} && $self->{isOnlyServerDetected} && $self->isResidentComponentManager() ) {
        $rc = 0;
	}
	return $rc;
	
}

sub printDetectedComponents {
    my ( $self ) = @_;

    return if (!$self->canPrintDetectedComponents);
    if ( ! LCM::App::ApplicationContext::getInstance()->getApplication->isGUIApplication ) {
        $self->getMsgLst()->setProgressHandler(LCM::App::ApplicationContext::getInstance()->getApplication ());
    }
    my $components = $self->getAllComponents();
    my $xsApplications = $self->getXs2ApplicationComponents();
    my $msg;

    if (!$self->isPrintedScanningSwLocationsMsg()) {
        $msg = $self->getMsgLst()->addProgressMessage("Scanning software locations...");
        $self->setPrintedScanningSwLocationsMsg(1);
    }

    my $errorMsg = $self->getErrorString();
    if ( not scalar @{$components}) {
       if(!$errorMsg) {
          $msg->endMessage("No components were detected in the specified software locations.") if defined $msg;
       }
       return;
    }

    if (!$self->isPrintedDetectedComponentsMsg()) {
       $self->getMsgLst()->addProgressMessage("Detected components:");
       $self->setPrintedDetectedComponentsMsg(1);
    }

    for my $component ( @{$components}, @{$xsApplications} ) {
        if ( $component->isInternal() ) {
            next;
        }
        if ($self->isPrintedDetectedComponent($component->getComponentKeyName())) {
            next;
        }

        $self->getMsgLst()->addProgressMessage(" " x 4 . $component->getComponentName() . " (" . $component->getVersion() . ")" . " in " . $component->getPath() );
        $self->setPrintDetectedComponent($component->getComponentKeyName(),1);
    }
    if($errorMsg) {
         $msg->endMessage("Warning:" . $errorMsg) if defined $msg;
    }

    return 1;
}

sub addComponentsToLog {
    my ($self) = @_;
    my $components = $self->getAllComponents();
    my $xsApplications = $self->getXs2ApplicationComponents();

    for my $component ( @{$components}, @{$xsApplications} ) {
        if ( $component->isInternal() ) {
            next;
        }
        $self->getMsgLst()->addMessage($component->getComponentName() . " (" . $component->getVersion() . ")" . " in " . $component->getPath() );
    }
}

sub setWarningMessage() {
	my ($self, $warning) = @_;
	$self->resetError();
    my $warningMessage = "The following components were found in more than one location and will be ignored\n". $warning;
    $self->setErrorMessage ($warningMessage);
}

sub hasDuplicatedComponentWarning() {
	my ($self, $component) = @_;
	my $message = undef;
	my $errorMessage = $self->getErrorString();
	if($errorMessage =~/was already found in directory/) {
		$message = "Component " . $component->getComponentName () . "(" . $component->getVersion () . ") in " . $component->getPath () . ".\n";
		$self->resetError();
	}
	return $message;
	
}

sub _addResidentInstallerComponent {
    my ($self) = @_;
    
    # detect resident installer:
    my $stackKitPath = new SDB::Install::Installer()->GetRuntimeDir();

    if($isWin) {
        $stackKitPath =~ s/(.*)\\.+\\?/$1/;
    } else {
        $stackKitPath =~ s/(.*)\/.+\/?/$1/;
    }

    my $residentInstallerManifestDir = $stackKitPath.$path_separator.'instruntime';
    my $compName = COMPONENT_NAME_INSTALLER;
    my $msg = $self->getMsgLst()->addMessage("Checking directory '$residentInstallerManifestDir' for manifest of '$compName'.");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst(), undef]);
	my $residentInstallerManifest = $self->getResidentInstallerManifestByDir($residentInstallerManifestDir, $saveCntxt);

	if (! defined $residentInstallerManifest) {
		return 0;
	}

    # Installable acts as a factory for its subclasses:
    my $residentInstallerComponent = LCM::Component::Installable->new($residentInstallerManifestDir, $residentInstallerManifest, $self, undef, $self->isSidAdmUserExecution());
    if($residentInstallerComponent->errorState){
        $self->appendErrorMessage("Component '$compName' invalid", $residentInstallerComponent->getErrMsgLst());
        $self->setMsgLstContext($saveCntxt);
        return 0;
    }
    $residentInstallerComponent->addComponentInfosToMsgLst($msg->getSubMsgLst);
    my $additionSuccessfull = $self->addComponent($residentInstallerComponent);
    if(!$additionSuccessfull){
        $self->setErrorMessage("Cannot add component '$compName'.", $self->getErrMsgLst());
        $self->setMsgLstContext($saveCntxt);
        return 0;
    }

    $self->setMsgLstContext($saveCntxt);
}

sub getResidentInstallerManifestByDir {
	my ($self, $residentInstallerManifestDir, $saveCntxt) = @_;

	my $residentInstallerManifestFile = $residentInstallerManifestDir.$path_separator.'manifest';
	my $compName = COMPONENT_NAME_INSTALLER;

	if(! -f $residentInstallerManifestFile) {
		$self->appendErrorMessage("Directory '$residentInstallerManifestDir' does not contain a manifest of '$compName'.");
		$self->setMsgLstContext($saveCntxt) if defined $saveCntxt;
		return undef;
	}

	my $residentInstallerManifest = LCM::ResidentInstallerManifest->new($residentInstallerManifestFile);
	if($residentInstallerManifest->errorState) {
		$self->appendErrorMessage("Invalid manifest of '$compName'.", $residentInstallerManifest->getErrMsgLst());
		$self->setMsgLstContext($saveCntxt) if defined $saveCntxt;
		return undef;
	}

    my $isPlatformSupported = $residentInstallerManifest->isPlatformSupported($currentPlatformName);
    if(!$isPlatformSupported){
        $self->appendErrorMessage("HDBLCM does not support current OS platform '$currentPlatformName'");
        $self->setMsgLstContext($saveCntxt) if defined $saveCntxt;
        return undef;
    }

	return $residentInstallerManifest;
}

sub _addLMStructureComponent{
	my ($self) = @_;
	my $compName = COMPONENT_NAME_LM_STRUCTURE;
	my $manifest = LCM::Landscape::LMStructureManifest->new();
	my $component = LCM::Component::Installable->new(undef, $manifest, $self, undef, $self->isSidAdmUserExecution());
	if ($component->errorState()){
		$self->getMsgLst->addWarning("Cannot create component '$compName'", $component->getErrMsgLst());
        return 0;
	}
	if(!$self->addComponent($component)){
        $self->getMsgLst->addWarning("Cannot add component '$compName'", $self->getErrMsgLst());
        return 0;
    }
    return 1;
}

sub clearDetectedComponents {
	my ($self) =  @_;
	$self->SUPER::clearDetectedComponents();
	
	$self->{directoryComponents} = [];
	$self->{rootComponents} = [];
	$self->{dvdComponents} = [];
	$self->{defaultLocation} = [];
}

sub isComponentDir{
	my $directoryName = $_[0];
	return grep ( /^$directoryName$/, @componentDirNames );
}

# returns 0 : success
#         1 : failed to detect by manifest in a directory
#         2 : duplicate components  
#         3 : _compileComponentList failed
#         4 : _compileComponentList failed, Only server is selected
sub detectComponentsByFsRoot {
    my ($self,$rootPath, $isComponentDirs) = @_;
    my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
    $rootPath = $app->getAbsPath ($rootPath);
   	my $manifestDirectories = $self->_findManifestDirectories($rootPath);
   	return $self->detectComponentsByDirectoryList( $manifestDirectories ,$self->{rootComponents}, $isComponentDirs );
}

sub _findManifestDirectories {
    my ($self,$unitPath) = @_;
    my @manifestDirectories = ();
    my $dirWalker = SDB::Install::DirectoryWalker->new(
        sub {   # actionMatcher callback, see comments in DirectoryWalker.pm for full signature
            my $name = $_[5];
            my $isDir = $_[6];
            my $relativePath = $_[4];
            if( !$isDir && $name eq 'manifest' ){ #visit only manifest files
                my @pathSegments = File::Spec->splitdir($relativePath);
                return ( ( scalar @pathSegments ) > 1 && isComponentDir( $pathSegments[-2] ) );
            }
            if( !$isDir && $name =~ /\.zip$/i ) { #possible XS2 Application archive
                my $candidateArchivePath = File::Spec->catfile($unitPath, $relativePath);
                my $manifest = new LCM::Manifests::XS2ApplicationManifest($candidateArchivePath);
                return $manifest->isXS2Application() ? 1 : 0;
            }
            return 0;
        },
        undef, undef, undef, undef, undef,
        sub {
            my $traversalRoot=$_[3];
            my $relativePath = $_[4];
            my $manifestFilename=$_[5];
            $relativePath =~ s/$manifestFilename//;
            my $manifestDir = $traversalRoot.$path_separator.$relativePath;
            push(@manifestDirectories, $manifestDir);
            return 1;
        },
        undef,undef,
        0,     # no breadth-first traversal, so we do it depth-first.
        1,    # dont collect list
        1     # follow symlinks
    );
    $dirWalker->findAndProcess($unitPath);

    my $fileList = listDirectory($unitPath) || [];
    my $hasXSAppArchive = 0;
    for my $fileName (@{$fileList}){
        my $candidateArchivePath = File::Spec->catfile($unitPath, $fileName);
        my $manifest = new LCM::Manifests::XS2ApplicationManifest($candidateArchivePath);
        if($manifest->isXS2Application()){
            $hasXSAppArchive = 1;
            last;
        }
    }

    # DirectoryWalker does not process root of tree
    if (-f File::Spec->catfile($unitPath, 'manifest') || $hasXSAppArchive) {
        push @manifestDirectories, $unitPath . $path_separator;
    }
    return \@manifestDirectories;
}

sub detectDefaultLocation {
    my ($self,$dirList, $isComponentDirs) = @_;
    return $self->detectComponentsByDirectoryList( $dirList ,$self->{defaultLocation}, $isComponentDirs );
}

# returns 0 : success
# 		  1 : failed to detect by manifest in a directory
#		  2 : duplicate components	
#		  3 : _compileComponentList failed
#         4 : _compileComponentList failed, Only server is selected
sub detectComponentsByDirectoryList {
    my ($self, $dirList, $componentList, $isComponentDirs) = @_;
    my $errorState = $SUCCESS;
    $isComponentDirs = defined($isComponentDirs) ? $isComponentDirs : 1;
    $componentList = defined($componentList) ? $componentList : $self->{directoryComponents};

    $self->resetMsgLstContext();
    my $msg = $self->getMsgLst()->addMessage ("Detecting components from directories");
    my ($messageList, $errorList) = @{$self->setMsgLstContext ([$msg->getSubMsgLst(), undef])};

    if(scalar(@{$dirList}) == 0){
        $self->appendErrorMessage("No components were found in the specified location");
        return $ERR_MISSING_MANIFEST;
    }
    my $app = LCM::App::ApplicationContext::getInstance ()->getApplication ();
    foreach my $dir (@$dirList){
        $dir = $app->getAbsPath ($dir);
        if (! -d $dir && !$dir =~ /\.zip$/i) {
            $errorState = $ERR_MISSING_MANIFEST;
            $self->appendErrorMessage ( "Directory $dir doesn't exist" ) ;
            next;
        }
        $dir = File::Spec->canonpath($dir);

        my $temporaryErrorList = new SDB::Install::MsgLst();
        $self->setMsgLstContext ([undef, $temporaryErrorList]);
        my $successfulDetection = $self->detectComponentByManifestDir ( $dir, $componentList, $isComponentDirs );
        if ( not $successfulDetection ) {
            my @componentSubDirectories = $self->searchSubDirectories ( $dir, $componentList, $isComponentDirs );

            if ( @componentSubDirectories == 1 ) {
               $successfulDetection = 1;
            } elsif ( @componentSubDirectories > 1 ) {
               $self->generateErrorMessage(\@componentSubDirectories, $componentList, $dir, $errorList);
            } else {
               my $detectedXs2ApplicationArchives = $self->_detectXS2ApplicationArchives($dir, $componentList);
               $successfulDetection = $detectedXs2ApplicationArchives > 0 ? 1 : 0;
            }
        }
        if(!$successfulDetection){
            $errorList->appendMsgLst($temporaryErrorList);
            $errorState= $ERR_MISSING_MANIFEST;
        }
    }
    $self->setMsgLstContext ([$messageList, $errorList]);
    return $errorState if $errorState;

    $errorState = $self->_checkDuplicatedComponents ( $componentList );

    return $errorState if $errorState;
    return $self->_compileComponentList();
}

sub generateErrorMessage() {
	my ( $self, $componentSubDirs, $components, $dir, $errorList) = @_;
	my %seen = ();
	foreach my $componentSubDir ( @$componentSubDirs ) {
		$seen{$componentSubDir} = $componentSubDir;
	}
	$errorList->addError("More than one component found in directory $dir.");
	foreach my $component ( @$components ) {
		next if(!exists $seen{$component->getPath()});
		my $message = sprintf("Component %s(%s) was found in %s.", $component->getComponentName(), $component->getVersion(), $component->getPath());
		$errorList->addError($message);
	}
}

sub searchSubDirectories() {
	my ( $self, $parentDir, $componentList, $isComponentDirs ) = @_;
	my @componentDirs = ();
	for my $subDir ( @componentDirNames ) {
		my $manifestDir = $parentDir . $path_separator . $subDir;
		if ( -d $manifestDir ) {
			my $result = $self->detectComponentByManifestDir ( $manifestDir, $componentList, $isComponentDirs );
			if ( $result ) {
				push @componentDirs, $manifestDir;
			}
		}
	}

	return @componentDirs;
}

sub _checkDuplicatedComponents() {
	my ( $self, $componentList ) = @_;

	my %componentNameMap = ();

	$self->resetError ();
	my $result = $SUCCESS;
	foreach my $component ( @$componentList ) {
		my $manifest = $component->getManifest ();
        my $componentName = $component->getComponentName ();
		#take into account components only for our platform
		next if (!$manifest->isPlatformSupported($currentPlatformName));
		
		if ( exists $componentNameMap{$componentName} ) {
			my $comonentFromMap = $componentNameMap{$componentName};
			if ( $component->getPath () eq $comonentFromMap->getPath () ) {
				next;
			}
			my $message =
				"The following components were found in more than one location $componentName ("
				. $component->getVersion () . ") in "
				. $component->getPath () . " and in " . $comonentFromMap->getPath ();
			$result = $ERR_DUPLICATE_COMPONENTS;
			$self->appendErrorMessage ( $message );
			last;
		} else {
			$componentNameMap{$componentName} = $component;
		}
	}

	return $result;
}

sub detectComponentByManifestDir{
    my ($self, $manifestDir, $componentCollection, $isComponentDirs ) = @_;
    my ($manifest, $manifestFile, $component);
    $manifestDir = File::Spec->canonpath($manifestDir);

    $manifestFile = $manifestDir . $path_separator . 'manifest';
    my $msg = $self->getMsgLst ()->addMessage ("Checking component directory $manifestDir");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    if (! -f $manifestFile){
        $self->appendErrorMessage ("No component was found in directory '$manifestDir'");
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    $manifest = new LCM::Manifests::GenericManifest ($manifestFile);
    if ($manifest->errorState){
        $self->appendErrorMessage ('Invalid manifest', $manifest->getErrMsgLst ());
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    if (!$manifest->isKnownComponent()){
        $self->getMsgLst()->addWarning("Unknown commponent was found in directory '$manifestDir'");
        return 1;
    }
    if ( $manifest->isServer() ) {
        my $appContext = LCM::App::ApplicationContext::getInstance();
        $appContext->setServerManifest($manifest);
    } elsif ( $manifest->isHLM() && $isComponentDirs ) {
    	my $SAPNoteNumber = "2068452";
    	$self->appendErrorMessage ("The $gProductNameInstaller component is no longer supported. For more information see SAP Note $SAPNoteNumber.");
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    $component = new LCM::Component::Installable ($manifestDir, $manifest, $self, undef, $self->isSidAdmUserExecution()); # Installable acts as a factory for its subclasses.
    if ($component->errorState){
        $self->appendErrorMessage ("Invalid component", $component->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
	
    if ( defined($component->getWarnings()) && scalar(@{$component->getWarnings()}) ){
        my @warnings = @{$component->getWarnings()};
        push(@{$self->{instconfig}->{warnings}}, @warnings);
        for my $warning (@{$component->getWarnings()}) {
            $self->getMsgLst()->addWarning($warning);
        }
    }

    $component->addComponentInfosToMsgLst ($msg->getSubMsgLst);
    push @$componentCollection, $component;
    $self->setMsgLstContext ($saveCntxt);
    return 1;
}

# Fix for Bug 166247: manifestPath could be a directory or an archive
sub _detectXS2ApplicationArchives {
	my ($self, $manifestPath, $componentCollection) = @_;
	my $numberOfComponentsFound = 0;
	my $msg = $self->getMsgLst ()->addMessage ("Checking component directory $manifestPath");
	my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst(), undef]);

	my $manifestDir;
	my $fileList;
	if($manifestPath =~ /\.zip$/i){
		my (undef,$directory,$fileName) = File::Spec->splitpath($manifestPath);
		$manifestDir = $directory;
		$fileList = [$fileName];
	} else {
		$manifestDir = $manifestPath;
		$fileList = listDirectory($manifestPath);
	}

	return $numberOfComponentsFound if(!defined($fileList));

	for my $fileName (@{$fileList}){
		next if($fileName !~ /\.zip$/i);

		my $candidateArchivePath = File::Spec->catfile($manifestDir, $fileName);
		my $manifest = new LCM::Manifests::XS2ApplicationManifest($candidateArchivePath);

		next if (!$manifest->isXS2Application());

		my $component = new LCM::Component::Installable($candidateArchivePath, $manifest, $self, undef, $self->isSidAdmUserExecution());
		if ($component->errorState()){
			$self->appendErrorMessage ("Invalid component", $component->getErrMsgLst());
			next;
		}
		push(@{$componentCollection}, $component);
		++$numberOfComponentsFound;
	}
	$self->setMsgLstContext ($saveCntxt);
	return $numberOfComponentsFound;
}

sub preCheckComponents {
    my ( $self) = @_;
    my $msg = $self->getMsgLst ()->addProgressMessage ('Checking components...');
    for my $component ( @{$self->getSelectedComponents ()} ) {
    	$component->resetWarnings();
        $component->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if ($component->isUpdate ()){
            if (!$component->preCheckUpdateComponent($self->{instconfig})){
                $component->setStatus(LCM::Component::STATUS_FINISHED_ERROR);
                $self->progressStepFinished (1);
                $self->setErrorMessage (undef, $component->getErrMsgLst());
                $msg->endMessage (undef, 'Checking components');
                return undef;
            }
        } else {
            if (!$component->preCheckInstallComponent($self->{instconfig})){
                $component->setStatus(LCM::Component::STATUS_FINISHED_ERROR);
                $self->progressStepFinished (1);
                $self->setErrorMessage (undef, $component->getErrMsgLst());
                $msg->endMessage (undef, 'Checking components');
                return undef;
            }
        }
        
        my $warnings = $component->getWarnings();
		if (@$warnings) {
			$component->setStatus(LCM::Component::STATUS_FINISHED_WARNING);
		} else {
			$component->setStatus(LCM::Component::STATUS_FINISHED_SUCCESS);	
		}
        $self->progressStepFinished ();
    }
    $msg->endMessage (undef, 'Checking components');
    return 1;
}

sub isStackUpdate {
	my $server = $_[0]->getComponentByKeyName($gKeynameEngine);
	return defined $server ? $server->isUpdate() : 0;
}

sub getSystemComponentManager{
    return $_[0]->{systemComponentManager};
}

sub setSystemComponentManager{
    $_[0]->{systemComponentManager} = $_[1];
}

sub getSortedComponentKeynames{
	my ($self) = @_;
	my @sortedComponentKeynames;
	my @serverPlugins = @{$self->getServerPluginComponents()};
	my @referenceDataComponents = @{$self->getReferenceDataComponents()};
	if ($self->isStackUpdate()) {
		for my $component( @serverPlugins ) {
			push (@sortedComponentKeynames, $component->getComponentKeyName());
		}
		push (@sortedComponentKeynames, $gKeynameLSS, $gKeynameEngine, $gKeynameCockpitStack, $gKeynameInstaller);
	} else {
		push (@sortedComponentKeynames, $gKeynameLSS, $gKeynameEngine, $gKeynameCockpitStack, $gKeynameInstaller);

		for my $component( @serverPlugins ) {
			push (@sortedComponentKeynames, $component->getComponentKeyName());
		}
	}

	my @refetenceDataComponentsKeynames = ();
	push (@refetenceDataComponentsKeynames, $_->getComponentKeyName() ) for ( @referenceDataComponents );

	push (@sortedComponentKeynames,
		$gKeynameClient,
		$gKeynameStudio,
		$gKeynameOfficeClient,
		$gKeynameSDA,
		$gKeynameStreaming,
		$gKeynameEs,
		$gKeynameAccelerator,
		$gKeynameRDSync,
		$gKeynameXS2,
		@refetenceDataComponentsKeynames,
		$gKeynameLMStructure);
	return \@sortedComponentKeynames;
}

sub setCanPrintDetectedComponents {
	my($self, $canPrint) = @_;
	$canPrint = $canPrint // 1;
	$self->{canPrint} = ($canPrint) ? 1 : 0;
}

sub canPrintDetectedComponents {
	my($self) = @_;
	return $self->{canPrint};
}

sub setPrintDetectedComponent {
	my ($self, $componentKeyname, $isPrinted) = @_;
	$isPrinted = $isPrinted // 1;
	$self->{isPrinted}->{$componentKeyname} = ($isPrinted) ? 1 : 0;
}

sub isPrintedDetectedComponent {
	my ($self, $componentKeyname) = @_;
	return $self->{isPrinted}->{$componentKeyname};
}

sub setPrintedDetectedComponentsMsg {
	my ($self, $isPrinted) = @_;
	$isPrinted = $isPrinted // 1;
	$self->{isPrinted}->{_detectedComponentsMsg} = ($isPrinted) ? 1 : 0;
}

sub isPrintedDetectedComponentsMsg {
	my ($self) = @_;
	return $self->{isPrinted}->{_detectedComponentsMsg};
}

sub setPrintedScanningSwLocationsMsg {
	my ($self, $isPrinted) = @_;
	$isPrinted = $isPrinted // 1;
	$self->{isPrinted}->{_scanningSwLocationsMsg} = ($isPrinted) ? 1 : 0;
}

sub isPrintedScanningSwLocationsMsg {
	my ($self) = @_;
	return $self->{isPrinted}->{_scanningSwLocationsMsg};
}

sub isHanaOptionSelected {
    my ($self) = @_;

    if ( !$self->isComponentSelected($gKeynameStreaming) &&
         !$self->isComponentSelected($gKeynameEs) &&
         !$self->isComponentSelected($gKeynameAccelerator) &&
         !$self->isComponentSelected($gKeynameRDSync) &&
         !$self->isComponentSelected($gKeynameXS2) ) {
            return 0;
    }

    return 1;
}

sub isComponentAvailable {
	my($self, $componentKeyname) = @_;

	my $systemManager = $self->{systemComponentManager};
	return 1 if (defined $systemManager && $systemManager->isComponentAvailable($componentKeyname));
	
	my $component = $self->getComponentByKeyName($componentKeyname);
	return 1 if(defined $component && $component->isComponentSelected());
	
	return 0;
}

sub getHANAFlavour {
    my($self) = @_;
    my $flavour = undef;
    my $appContext = LCM::App::ApplicationContext::getInstance();
    my $serverManifest = $appContext->getServerManifest();
    $flavour = $serverManifest->getHANAFlavour() if defined($serverManifest);
    return $flavour if defined ($flavour);
    
    my $scm = $self->getSystemComponentManager();
    return $gFlavourPlatform unless defined $scm;

    $flavour = $scm->getHANAFlavour() if defined($scm);
    return $flavour || $gFlavourPlatform;
}

1;
