package LCM::ComponentManager::SystemComponentManager;

use strict;
use SDB::Install::System qw( readLink );
use SDB::Install::SysVars qw($path_separator $isWin);
use LCM::Component::Installed;
use SDB::Install::Globals qw ($gDirNameAccelerator
                              $gDirNameEs
                              $gDirNameStreaming
                              $gDirNameRDSync
                              $gDirNameXS2
                              $gDirNameCockpitStack
                              $gDirNameResidentInstaller
                              $gKeynameEngine
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameAFL
                              $gKeynameLCA
                              $gKeynameHLM
                              $gKeynameSDA
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameRDSync
                              $gKeynameXS
                              $gKeynameXS2
                              $gShortProductNameXS2
                              $gKeynameOfficeClient
                              $gKeynameLMStructure
                              $gKeynameInstaller
                              $gKeynameLSS);

use SDB::Install::SAPSystem qw(CollectSAPSystems);
use SDB::Common::Utils qw(createXMLParser);
use LCM::Component;
use LCM::Manifests::XS2ApplicationManifest::InstalledXS2ApplicationManifest;
use LCM::ResidentInstallerManifest;
use LCM::Landscape::LMStructureManifest;
use LCM::Manifests::GenericManifest;
use File::Spec 'catfile';
use File::Basename qw (dirname);

use base qw (LCM::ComponentManager);


sub new {
    my ($class) = shift();
    my $self = $class->SUPER::new (@_);
    return $self;
}

sub detectComponentsBySid {
    my ($self, $sid) = @_;
    my $systems = CollectSAPSystems (undef, 1);
    my $sapSystem = $systems->{$sid};
    if (!defined $sapSystem){
        my $config = $self->{instconfig};
# Workaround: See comment in UpdateHostConfigurationFactory
        my $isBeforeRenameWithContinue = defined ($config) ? $config->{isBeforeRenameWithContinue} : undef;
        my $isRenameConfiguration = defined($config) && ($config->isa('LCM::Configuration::RenameSystemConfiguration') || $isBeforeRenameWithContinue);
        my $isRegisterRename = $isRenameConfiguration && $config->can('isRegisterScenario') && $config->isRegisterScenario();
        my $isUpdateAfterRenameConfiguration = defined($config) && $config->isa('LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationInRename');
        my $isContinueRename = $isRenameConfiguration && ($config->{isContinueOption} || $isBeforeRenameWithContinue);
        
        if (!$isRegisterRename && !$isContinueRename && !$isUpdateAfterRenameConfiguration) {
            $self->setErrorMessage ("No SAP system '$sid' found");
        }
        return 0;
    }
    return $self->detectComponentsBySapSystem ($sapSystem)
}

sub detectComponentsBySapmntSid {
    my ($self, $sapmnt, $sid) = @_;
    my $sapSystem = new SDB::Install::SAPSystem ();
    if (!defined $sapSystem->initWithGlobDir ($sid, $sapmnt)){
        $self->setErrorMessage (undef, $sapSystem->getErrMsgLst());
        return undef;
    }
    return $self->detectComponentsBySapSystem ($sapSystem)
}

sub detectComponentsBySapSystem {
    my ($self, $sapSystem) = @_;
    
    my $msglst = $self->getMsgLst ();
    my $msg = $msglst->addMessage ("Looking for components of system " . $sapSystem->get_sid());
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
	
	$self->{sid} = $sapSystem->get_sid ();
    # detect resident installer:
    my $compName = COMPONENT_NAME_INSTALLER;
    my $residentInstallerComponent = _getInstallerComponentBySapSystem($sapSystem, $self->{instconfig},  $self, $self);
    if(not defined $residentInstallerComponent) {
        $msglst->addMessage("Cannot create component class for '$compName'.", $self->getErrMsgLst());
    }
    if(defined $residentInstallerComponent && !$self->addComponent($residentInstallerComponent)){
        $self->setErrorMessage("Cannot add component '$compName'.", $self->getErrMsgLst());
    }
    
	$self->_detectLmStructureComponent($sapSystem);

    my $sapSystemManifest = $sapSystem->getManifest ();
    if (!defined $sapSystemManifest){
        $self->setErrorMessage ('Cannot get manifest', $sapSystem->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return undef;
    }

    if (!$self->detectComponentByManifest
                  ($sapSystem->get_globalTrexInstallDir(), $sapSystemManifest)){
        $self->setMsgLstContext ($saveCntxt);
        return undef;
    }

    #
    # checking for server plugins
    #
    my $hdbInstance = $sapSystem->getNewDBInstances ()->[0];
    my $serverPlugins;
    if (defined $hdbInstance){
        $serverPlugins = $hdbInstance->getPluginInstallations();
    }
    if (defined $serverPlugins && @$serverPlugins){
        foreach my $plugin (@$serverPlugins){
            my $pluginManifest = $plugin->getManifest();
            my $manifestPath = defined($pluginManifest) ? $pluginManifest->getFileName() : '';
            my $manifest = new LCM::Manifests::GenericManifest($manifestPath);
            $self->detectComponentByManifest ($plugin->getProgramPath(), $manifest);
        }
    }
    
    my $refDataManifests = $self->_detectReferenceDataManifests($hdbInstance);
   if (defined $refDataManifests && @$refDataManifests) {
	    foreach my $refDataManifestPath (@$refDataManifests){
	        my $manifest = new LCM::Manifests::GenericManifest($refDataManifestPath);
	        $self->detectComponentByManifest (dirname($manifest), $manifest);
	    }
    }
    
    $self->_detectLSSComponent($sapSystem);
    my $hanamntSid = $isWin ? $sapSystem->getUsrSapSid () : $sapSystem->get_globalSidDir();

    my ($componentDir, $manifestFile, $manifest);
    foreach my $subDir (qw (hdbstudio hdbclient HLM federation),
                        $gDirNameStreaming,
                        'iq',
                        $gDirNameEs,
                        $gDirNameRDSync,
                        $gDirNameAccelerator,
                        $gDirNameXS2,
                        $gDirNameCockpitStack) {
        $componentDir = $hanamntSid . $path_separator . $subDir;
        if (!-d $componentDir){
            next;
        }
        $manifestFile = $componentDir . $path_separator . 'manifest';

        if (!-f $manifestFile){
            next;
        }
        
        $manifest = new LCM::Manifests::GenericManifest($manifestFile);
               
        $self->detectComponentByManifest ($componentDir, $manifest);
    }

    # Check for installed XS2 applications on the system if XSA is detected
    # At this points XSA should be detected if it's present on the system
    if ($self->isComponentAvailable($gKeynameXS2)) {
        $self->_detectInstalledXsComponentsFromRegistry($sapSystem->get_target());
    }

    $self->setMsgLstContext ($saveCntxt);
    return 1;
}

sub _detectInstalledXsComponentsFromRegistry {
    my($self, $targetDir) = @_;
    my $sid = $self->getSid();
    my $registryXmlPath = File::Spec->catfile($targetDir, $sid, 'xs', 'cmp-registry', 'cmp_registry_sap.xml');
    my $xmlTree = undef;

    my $logMessage = $self->getMsgLst()->addMessage("Detecting installed XS applications from registry file: '$registryXmlPath'");
    if(! -f $registryXmlPath) {
        $logMessage->getSubMsgLst()->addMessage("No installed XS applications found");
        return 1;
    }

    eval {
        my $xmlParser = createXMLParser();
        $xmlTree = $xmlParser->parse_file($registryXmlPath);
    };
    if($@) {
        $logMessage->getSubMsgLst()->addWarning("Cannot detect the already installed $gShortProductNameXS2 components");
        $logMessage->getSubMsgLst()->addWarning("Cannot parse component registry file '$registryXmlPath': $@");
    } else {
        my $componentNodes = $xmlTree->getElementsByTagName('component') || [];
        for my $componentNode (@{$componentNodes}){
            my $manifest = LCM::Manifests::XS2ApplicationManifest::InstalledXS2ApplicationManifest->new($componentNode);
            my $xsApp = LCM::Component::Installed->new(undef, $manifest, $self, $self->{instconfig});
            my $xsAppName = $xsApp->getComponentName();
            my $xsAppVersion = $xsApp->getVersion();

            my $appMessage = $logMessage->getSubMsgLst()->addMessage("Detected '$xsAppName' version $xsAppVersion");
            if (!$self->addComponent($xsApp)) {
                $appMessage->getSubMsgLst()->addWarning("Couldn't add the detected XS Application '$xsAppName'");
            }
        }
    }
    return 1;
}

sub _detectReferenceDataManifests {
    my ($self, $hdbinstance) = @_;

    my $result = [];
    my $refDataInstallations;

    my $msglst = $self->getMsgLst ();
    my $msg = $msglst->addMessage ('Detecting Reference Data Installations...');
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);

    my $refInstallDir = $self->_getReferenceDataInstallDir( $hdbinstance );
    if (!defined $refInstallDir) {
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    if (!$refInstallDir) {
        $self->setMsgLstContext($saveCntxt);
        return $result;
    }

    if (!opendir (DH, $refInstallDir)){
        $self->appendErrorMessage ("Cannot open directory '$refInstallDir': $!");
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    my @subdirs = grep {!/^\.{1,2}$/ && -l "$refInstallDir$path_separator$_"} readdir (DH);
    closedir (DH);
    
    if (!@subdirs) {
        $self->setMsgLstContext($saveCntxt);
        return $result;
    }

    foreach my $subdir (@subdirs){
	    my $link = $refInstallDir . $path_separator . $subdir;
	    next if (!-l $link);
	    my $refdata = SDB::Install::System::readLink ( $link );
	    my $parentDir = dirname($refdata);
	    my $manifest = File::Spec->catfile($parentDir, 'manifest');
	    next if(!-f $manifest);
        push @$result, $manifest;
    }
    $self->setMsgLstContext($saveCntxt);
    return $result;
}

sub _getReferenceDataInstallDir {
    my ($self, $hdbinstance) = @_;

    my $globalSidDir = $hdbinstance->get_globalSidDir();
    my $configDir = join ($path_separator, $globalSidDir, 'global', 'hdb', 'custom' ,'config');
    return 0 if(!-f File::Spec->catfile($configDir, 'scriptserver.ini'));
    
    my $layeredCfg = $hdbinstance->getLayeredConfig ();
    if (!defined $layeredCfg){
        $self->appendErrorMessage ('Cannot get layered configuration', $layeredCfg);
        return undef;
    }
    
    $layeredCfg->resetMsgLstContext();
    my $scriptServerIni = $layeredCfg->getIniFile ('scriptserver.ini');
    if (!defined $scriptServerIni){
        $self->appendErrorMessage ('Cannot get scriptserver.ini', $layeredCfg);
        return undef;
    }

    $scriptServerIni->resetMsgLstContext();
    if (!defined $scriptServerIni->readValues ()){
        $self->appendErrorMessage ('Cannot read scriptserver.ini', $scriptServerIni);
        return undef;
    }

    my $referenceDataPath = $scriptServerIni->getValue ('adapter_framework','dq_reference_data_path');
    if(! defined $referenceDataPath || !-d $referenceDataPath){
        return 0;
    }

    return $referenceDataPath;
}

sub detectComponentByManifest{
    my ($self, $installerDir, $manifest) = @_;
    my $component;
    my $msg = $self->getMsgLst ()->addMessage ("Checking component...");
    my $saveCntxt = $self->setMsgLstContext ([$msg->getSubMsgLst(), undef]);
    $component = new LCM::Component::Installed ($installerDir, $manifest, $self, undef, $self->isSidAdmUserExecution()); # Installable acts as a factory for its subclasses.
    if ($component->errorState){
        $self->appendErrorMessage ("Invalid component", $component->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    $component->addComponentInfosToMsgLst ($msg->getSubMsgLst);
    if (!$self->addComponent ($component)){
        $self->setErrorMessage ('Cannot add component', $self->getErrMsgLst());
        $self->setMsgLstContext ($saveCntxt);
        return 0;
    }
    $self->setMsgLstContext ($saveCntxt);
    return 1;
}

sub getSortedComponentKeynames{
	my ($self) = @_;
	my @sortedComponentKeynames;

	push (@sortedComponentKeynames,
		$gKeynameHLM,
		$gKeynameOfficeClient,
		$gKeynameStudio,
		$gKeynameClient );

	foreach my $component(@{$self->getServerPluginComponents()}) {
		push (@sortedComponentKeynames, $component->getComponentKeyName());
	}

    foreach my $component(@{$self->getReferenceDataComponents()}) {
        push (@sortedComponentKeynames, $component->getComponentKeyName());
    }

	push (@sortedComponentKeynames,
		$gKeynameSDA,
		$gKeynameStreaming,
		$gKeynameEs,
		$gKeynameRDSync,
		$gKeynameAccelerator,
		$gKeynameXS2,
		$gKeynameLMStructure,
		$gKeynameLSS,
		$gKeynameEngine );

	return \@sortedComponentKeynames;
}

# function, not a method:
sub _getInstallerComponentBySapSystem {
    my (
        $sapSystem,
        $instconfig,
        $base, # to append msgs to
        $systemComponentManager
    ) = @_;
    my $globalSidDir = $isWin ? $sapSystem->getUsrSapSid() : $sapSystem->get_globalSidDir();
    my $residentInstallerManifestDir = File::Spec->catfile($globalSidDir, $gDirNameResidentInstaller, 'instruntime');
    my $residentInstallerManifestFile = File::Spec->catfile($residentInstallerManifestDir, 'manifest');
    my $compName = COMPONENT_NAME_INSTALLER;
    my $msglst = $base->getMsgLst();
    my $msg = $msglst->addMessage("Checking directory '$residentInstallerManifestDir' for manifest of '$compName'.");
    if(! -f $residentInstallerManifestFile) {
        $msglst->addMessage("Directory '$residentInstallerManifestDir' does not contain a manifest of '$compName'.");
        return undef;
    }
    my $residentInstallerManifest = LCM::ResidentInstallerManifest->new($residentInstallerManifestFile);
    if($residentInstallerManifest->errorState) {
        $base->appendErrorMessage("Invalid manifest of '$compName'.", $residentInstallerManifest->getErrMsgLst());
        return undef;
    }
    # Installed acts as a factory for its subclasses:
    my $residentInstallerComponent = LCM::Component::Installed->new($residentInstallerManifestDir, $residentInstallerManifest, $systemComponentManager, $instconfig, $systemComponentManager->isSidAdmUserExecution());
    if($residentInstallerComponent->errorState){
        $base->appendErrorMessage("Component '$compName' invalid", $residentInstallerComponent->getErrMsgLst());
        return undef;
    }
    $residentInstallerComponent->addComponentInfosToMsgLst($msg->getSubMsgLst);
    return $residentInstallerComponent;
}

sub _detectLmStructureComponent {
	my (
		$self,
        $sapSystem
    ) = @_;
	my $compName = COMPONENT_NAME_LM_STRUCTURE;
	
	#detect lm structure
    my $lmStructurePath = $sapSystem->get_globalSidDir() . $path_separator . "lm_structure";
    if (! -d $lmStructurePath) {
    	return;
    }

    my $manifest = new LCM::Landscape::LMStructureManifest();
    my $lmStructureComponent = LCM::Component::Installed->new(undef, $manifest, $self, $self->{instconfig}, $self->isSidAdmUserExecution());
    if(defined $lmStructureComponent && !$self->addComponent($lmStructureComponent)){
     	$self->setErrorMessage("Cannot add component '$compName'.", $self->getErrMsgLst());
    }
}

sub _detectLSSComponent {
    my ( $self, $sapSystem ) = @_;
    my $lssLink = File::Spec->catfile( $sapSystem->getUsrSapSid(), 'lss' , 'exe' );

    if( !-l $lssLink){
        return 0;
    }
    my $path = readLink($lssLink);
    my $lssInstallerDir = dirname($path);

    if(! defined $lssInstallerDir || !-d $lssInstallerDir){
        return 0;
    }
    my $lssManifest = File::Spec->catfile( $lssInstallerDir, 'manifest' );
    my $manifest = new LCM::Manifests::GenericManifest($lssManifest);

    my $lssComponent = LCM::Component::Installed->new($lssInstallerDir, $manifest, $self, $self->{instconfig}, $self->isSidAdmUserExecution());

    if(defined $lssComponent && !$self->addComponent($lssComponent)){
        my $cmpName = $lssComponent->getComponentName();
        $self->setErrorMessage("Cannot add component '$cmpName'.", $self->getErrMsgLst());
    }
    return 1;
}

sub getSid{
	$_[0]->{sid};
}

sub isComponentAvailable {
	my($self, $componentKeyname) = @_;
	return (defined $self->getComponentByKeyName($componentKeyname));
}

sub getHANAFlavour {
    my ($self) = @_;
    my $systems = CollectSAPSystems (undef, 1);
    my $sapSystem = $systems->{$self->getSid()};

    return undef unless defined $sapSystem;
    return $sapSystem->getHANAFlavour();
}

sub hasIncompatibleToMultiDbComponents{
    my $self = shift;
    my $unsupportComponentNames = $self->getUnsupportedMultiDBComponents();
    return scalar(@$unsupportComponentNames) > 0;
}

sub getUnsupportedMultiDBComponents{
  my $self = shift;
    my $components = $self->getAllComponents();

    my @unsupportedComponents = grep {
        $_->isFeatureUnsupported('multidb') ||
        $_->isFeatureUnsupported('convert_to_multidb')
    } @{$components};

    my @unsupportComponentNames = map { $_->getComponentName() } @unsupportedComponents;
    return \@unsupportComponentNames;
}

1;
