package LCM::Configuration::AddHostRolesConfiguration;
use parent qw ( LCM::Configuration::GenericStackUpdateConfiguration LCM::Configuration::AutoInitializeFamilyServicesConfiguration);
use strict;

use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Configuration::AnyMultiHostConfig;
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker
                              $gHostRoleAcceleratorStandby
                              $gHostRoleEsWorker
                              $gHostRoleEsStandby
                              $gProductNameSystem
                              $gXSParametersConstraint
                              $gXSParametersAddHostOrRoleConstraint
                              $gKeynameXS2
                              getHumanReadableRoleByHostRole
                              GetHostRoleProperties);
use SDB::Install::System qw(isSidadmin $hostname_regex $ipv4_regex);
use SDB::Install::SysVars qw ($isWin);
use LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener;
use LCM::Configuration::ValueChangeListeners::AddHostRoles::XSUserIsolationListener;
use LCM::Configuration::ValueChangeListeners::AddHostRoles::RemoteCredentialsListener;
use LCM::Configuration::ParametersCreator;
use LCM::App::ApplicationContext;
use LCM::HostsParser qw(IsLocalHost);
use LCM::App::AddHostRolesUtils;
use LCM::SummaryTreeBuilders::AddHostRolesSummaryTreeBuilder;
use LCM::Installer;
use LCM::Utils::CommonUtils qw ( getSidadmName getSystemSidadmUid getLocalHostSidadmUid );
use SDB::Install::Configuration::TenantUserListener;
use experimental qw (smartmatch);

                          
sub new {
    my $class = shift();
    my $self = $class->SDB::Install::Configuration::AnyMultiHostConfig::new(@_);
    my $installer = new LCM::Installer();
    $self->{systemComponentManager} = $installer->getOwnSystemComponentManager();
    if (!defined($self->{systemComponentManager})) {
        return 0;
    }
    $self->_defineAddHostRolesParams();
    $self->_addListeners();
    $self->changePasswordStr(lc($installer->getSid()) . "adm");
    return $self;
}

# Override
sub getWarnings {
    my $self = shift();
    return $self->getWarningList(@_);
}

# Override
sub getWarningList {
    my $self = shift();
    my $warningsList = $self->SDB::Install::Configuration::getWarningList(@_);

    foreach my $parameterId (@{$self->getParamIds()}) {
        my $parameter = $self->{params}->{$parameterId};
        my $parameterWarnings = $parameter->{warnings} || [];

        next if($self->isSkipped($parameterId));
        push(@{$warningsList}, @{$parameterWarnings});
    }
    return $warningsList;
}

# Override
sub getSystemComponentManager {
    return $_[0]->{systemComponentManager};
}

# Override
sub handleHostsPropertiesSkipValue { return; }

# Override
sub getIgnoreValues { return undef; }

# Override
sub _handleInitializationOfPendingConfiguration { return 1; }

# Override
sub getRemoteInstallerExecutable{
    my ($self, $actionProgressive, $remoteTool) = @_;
    return $self->getRemoteExecutable($actionProgressive, $remoteTool);
}

sub _defineAddHostRolesParams {
    my ($self) = @_;
    my $order = 0;
    my $section = "AddHostRoles";
    
    my $acceleratorConstraint = "Valid with $gHostRoleAcceleratorWorker/standby only";
    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
    $self->{params} = {
        'Target' => $self->getParamTarget($order++, $section),
        'NoStart' => $self->getParamNoStart($order++, $section, "Do not start hosts with modified roles", "Does not start hosts with modified roles", 0),
        (
            !$isWin
            ? (
                 'RemoteExecution'  => $self->getParamRemoteExecution($order++, $section, 0),
                 'UseHttp'          => $self->getParamUseHttp($order++, $section),
                 'SAPControlUseHttp'=> $self->getParamSAPControlUseHttp($order++, $section),
                 'InstallSSHKey'    => $self->getParamInstallSSHKey($order++, $section),
                 'ImportContentXS2' => $self->getParamImportContentXS2 ($order++, $section, $gXSParametersConstraint),
              )
            : ()
        ),
        'SID' => $self->getParamSID($order++, $section),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $section,undef,$tenantCredentialsConstraint),
        (
            !$isWin
            ? (
                 'AddRoles'         => LCM::Configuration::ParametersCreator::getParamRolesHostmap($self, $order++, $section),
                 'RootUser'         => $self->getParamRootUser($order++, $section),
                 'RootPassword'     => $self->getParamRootPassword ($order++, $section),
              )
            : ()
        ),
        'Password' => $self->getParamPassword($order++, $section),
        'SystemUser'        => $self->getParamSystemUser($order++, $section),
        'SQLSysPasswd'      => $self->getParamSQLSysPasswd($order++, $section, 'passwd'),
        'AcceleratorUser'      => $self->getParamAcceleratorUser($order++, $section, $acceleratorConstraint),
        'AcceleratorPassword'  => $self->getParamAcceleratorPassword($order++, $section, $acceleratorConstraint),
        'OrgManagerUser'     => $self->getParamOrgManagerUser    ($order++, $section, $gXSParametersAddHostOrRoleConstraint),
        'OrgManagerPassword' => $self->getParamOrgManagerPassword($order++, $section, $gXSParametersAddHostOrRoleConstraint),
        (
            !$isWin
            ? (
                 'SkipModifySudoers'   => $self->getParamSkipModifySudoers($order++, $section)
              )
            : ()
        ),
        'TenantUser'     => $self->getParamTenantUser    ($order++, $section,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $section,1,$tenantCredentialsConstraint),
    };
    $self->{params}->{RootUser}->{init_with_default} = 1;
    $self->{params}->{Password}->{str} = sprintf($self->{params}->{Password}->{str_templ},  $self->{sidadmUser});
    $self->setMandatory('AddRoles');
    $self->{params}->{AddRoles}->{custom_input} = \&LCM::App::AddHostRolesUtils::readInteractiveAddHostRolesInput;
    # SystemUser is needed only if adding ets hosts
    $self->setSkip('SystemUser');
    $self->setSkip('SQLSysPasswd');
    $self->setType('OrgManagerPassword', 'passwd');
}

sub setSID {
    my $self = shift;
    my $systemHosts = $self->getSystemHosts();
    $self->{params}->{AddRoles}->{origin_values} = [ @{$systemHosts} ];
    $self->{params}->{AddRoles}->{default_map} = { map { ($_ => '') } @{$systemHosts} };
    return $self->SUPER::setSID(@_);
}

sub InitDefaults {
    my $self = shift();

    return undef if(!$self->SDB::Install::Configuration::AnyConfig::InitDefaults(@_));

    my $isXs2Installed = $self->_isComponentAvailable($gKeynameXS2);
    if(!$isXs2Installed){
        $self->setHidden('ImportContentXS2', 1);
    }
    if(isSidadmin()){
        $self->setValue('RemoteExecution', 'saphostagent');
        $self->setHidden('RemoteExecution', 1);
    }
    my $fillHostRolesListener = new LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener();
    $fillHostRolesListener->onValueChange(undef, $self);

    my $existingHostToValidRolesMap = getPossibleRolesForExistingHosts($self);
    my $canAssignAdditionalRoles = grep { length($existingHostToValidRolesMap->{$_}) > 0 } keys(%{$existingHostToValidRolesMap});
    my $isBatchMode = LCM::App::ApplicationContext->getInstance()->isBatch();
# Don't blow up in batch mdoe because of the cache server role
    if(!$canAssignAdditionalRoles && !$isBatchMode) {
        $self->getErrMsgLst()->addError("There aren't any additional roles which can be assigned");
        return undef;
    }
    return 1;
}

sub CheckParams{
    my ($self, $batchMode) = @_;
    my $rc = $self->SUPER::CheckParams($batchMode);
    return $rc if (!$rc);

    my $addRolesBatchValue = $self->{params}->{AddRoles}->{batchValue};
    my $isGui = $self->{options}->{isGUI};

    if ( (!$isGui) && (!$batchMode && ((!defined $addRolesBatchValue) || ($addRolesBatchValue eq ""))) ){
        $self->{params}->{AddRoles}->{set_interactive} = 1;
    }
    return $rc;
}

sub checkRootUser {
    my ($self, $remoteRootUser) = @_;
    
    if(defined $self->{_remoteHosts}){
        $self->{_remoteHosts}->destroy();
        $self->{_remoteHosts} = undef;
    }
    
    $self->setSkip('RootPassword', 1);
    my $returnCode = $self->SUPER::checkRootUser($remoteRootUser);
    return $returnCode;
}

sub checkPassword {
    my ($self, $password) = @_;
    return 0 if (!$self->initRemoteHosts());

    if(!defined $password){
        $self->appendErrorMessage ("Password is not defined.");
        return 0;
    }elsif(length($password) == 0){
        $self->appendErrorMessage ("Password value cannot be empty.");
        return 0;
    } else {
        require LCM::Configuration::NewDBUser::User;
        my $sidadmUser = new LCM::Configuration::NewDBUser::User(getSidadmName($self));
        $sidadmUser->setMsgLstContext($self->getMsgLstContext());
        my $rc = $sidadmUser->verifyPassword($password, $self);
        if (!defined $rc){
            $self->getMsgLst()->appendMsgLst ($sidadmUser->getErrMsgLst());
            $self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
            return 1;
        }
        # Fix for bug 72608. Prevent doubled 'Unknown user password combination' message. The other message comes from 'LCM::Configuration::NewDBUser::User;verifyPasswordSHA'.
        if($rc != 1 && !isSidadmin($self->getSID())) {
            $self->appendErrorMessage ('Unknown user password combination');
        }
        return $rc;
    }
}

sub _addListeners {
    my ($self) = @_;
    $self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener->new());
    $self->addParameterListener('AddRoles', LCM::Configuration::ValueChangeListeners::AddHostRoles::RemoteCredentialsListener->new());
    $self->addParameterListener('Password', LCM::Configuration::ValueChangeListeners::AddHostRoles::XSUserIsolationListener->new('Password'));
    $self->addParameterListener('AddRoles',$self->getAutoInitFamilyServicesParameterHanlder('AddRoles'));
    $self->addParameterListener('TenantUser',SDB::Install::Configuration::TenantUserListener->new());
}

sub checkAddRoles {
    my ($self, $host, $roles) = @_;
    return 0 if (!$self->SUPER::checkAddRoles($host, $roles));
    if ($roles =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        $self->setSkip('AcceleratorUser', 0);
        $self->setSkip('AcceleratorPassword', 0);
        $self->setSkip('SystemUser', 0);
        $self->setSkip('SQLSysPasswd', 0);
    }
    return 1;
}

# Called by LCM::Configuration::GenericStackUpdateConfiguration::checkAddRoles
sub _checkRolesForHost {
    my ($self, $hostname, $roles) = @_;
    my $validRoles = GetHostRoleProperties();
    my @rolesForCheck = (   $gHostRoleAcceleratorWorker,
                            $gHostRoleAcceleratorStandby,
                            $gHostRoleEsWorker,
                            $gHostRoleEsStandby);
    for my $role ( split(',', $roles) ) {
        if (!exists($validRoles->{$role}) || $validRoles->{$role}->{isColumnStore}) {
            $self->getErrMsgLst()->addError("Invalid host role '$role' in --add_roles entry for host '$hostname'.");
            return 0;
        }
        my $instance = $self->getOwnInstance();
        if (!defined $instance) {
            my $sid = $self->getValue('SID');
            $self->getErrMsgLst()->addError("$gProductNameSystem '$sid' not found.");
            return 0;
        }
        my $existingRoles = defined $instance ? $instance->getHostRolesByIniFile($hostname) : undef;
        if ($role ~~ @rolesForCheck && $role ~~ @$existingRoles) {
            my $humanReadableRole = getHumanReadableRoleByHostRole( $role );

            my $message = "More than one host ($hostname) has been assigned with the role $humanReadableRole. ";
            $message .= "Only one host with this role can be added to a HANA instance.";
            $self->getErrMsgLst()->addError($message);
            return 0;
        }
    }
    return 1;
}

# Called by LCM::Configuration::GenericStackConfiguration::checkHostRoles
sub _isComponentAvailable {
    my ($self, $componentKeyname) = @_;
    return $self->isComponentInstalled($componentKeyname);
}

# Adds a note a message that tells the user
# why not all hosts are available for selection
sub getInvalidHostsForSelectionMessage {
    my ($self, $validHostsToRolesMap) = @_;
    my $instance = $self->getOwnInstance();
    my $allHosts = $instance->get_allhosts();
    my @invalidHostsForSelection = grep {! exists($validHostsToRolesMap->{$_})} @$allHosts;
    if (scalar @invalidHostsForSelection) {
        my $invalidhostsString = join(', ', @invalidHostsForSelection);
        return "Note: There are no available roles that can be assigned to host(s) $invalidhostsString.";
    }
    return undef;
}

sub isAddingAdditionalLocalRoles {
    my ($self) = @_;
    my $addRolesValue = $self->getValue('AddRoles');
    return undef if (! defined($addRolesValue) );
    for my $host ( keys(%{$addRolesValue}) ) {
        return 1 if (IsLocalHost($host));
    }
    return 0;
}

sub isAddingAdditionalRemoteRoles {
    my ($self) = @_;
    my $addRolesValue = $self->getValue('AddRoles');
    return undef if (! defined($addRolesValue) );
    my $numberOfHosts = keys(%{$addRolesValue});
    return 1 if ($numberOfHosts > 1);
    return 1 if ($numberOfHosts == 1 && !$self->isAddingAdditionalLocalRoles());
}

sub isCollectOtherHostInfosRequired {
    my $self = shift();

    return 0 if(!$self->isAddingAdditionalRemoteRoles());
    return $self->SUPER::isCollectOtherHostInfosRequired(@_);
}


sub setRemoteExecution {
    my ($self, $value) = @_;
    
    if(isSidadmin() && $value ne 'saphostagent'){
        my $remoteExecutionOpt = $self->getOpt('RemoteExecution');
        $self->PushError("Invalid value '$value' of option '$remoteExecutionOpt' (only 'saphostagent' allowed when tool is started as system administrator)");
        return undef;
    }
    return $self->SUPER::setRemoteExecution($value);
}

sub getStaticDialogs {
    return [
        'LCM::Gui::Dialogs::ConfigurationSummaryDialog',
        'LCM::Gui::Dialogs::ConfigurationExecutionDialog',
        'LCM::Gui::Dialogs::FinalizeDialog'
    ];
}

sub getConfigureDialogs {
    my ($self, $wizard) = @_;
    require LCM::Gui::Dialogs::AddHostRoles::AddHostRolesPropertiesDialog;
    require LCM::Gui::Dialogs::SystemUserPropertiesDialog;
    require LCM::Gui::Dialogs::AddHostRoles::AcceleratorPropertiesDialog;
    require LCM::Gui::Dialogs::OrgManagerCredentialsDialog;
    return [ new LCM::Gui::Dialogs::AddHostRoles::AddHostRolesPropertiesDialog($wizard),
             new LCM::Gui::Dialogs::SystemUserPropertiesDialog($wizard),
             new LCM::Gui::Dialogs::AddHostRoles::AcceleratorPropertiesDialog($wizard),
             new LCM::Gui::Dialogs::OrgManagerCredentialsDialog($wizard)];
}

sub isSystemInCompatibilityMode {
    my ($self) = @_;
    return $self->SDB::Install::Configuration::AnyConfig::isSystemInCompatibilityMode();
}

sub getAction {
    my ($self) = @_;
    return "add_host_roles";
}

sub createSummaryTree {
    my $self = shift;
    return LCM::SummaryTreeBuilders::AddHostRolesSummaryTreeBuilder::buildSummaryTree($self);
}

sub getProductName {
    my ($self) = @_;
    return "Add Host Roles";
}

sub getSummaryTitle {
    my ($self) = @_;
    return "Add host roles parameters";
}

sub getTimeoutValues{
    return [ qw(start_service stop_service start_instance stop_instance) ];
}

sub isAutoInitializeServicesRequired {
    my $self = shift;
    return $self->SDB::Install::Configuration::AnyConfig::isAutoInitializeServicesRequired();
}

sub isAutoInitializeServicesApplicableParameterCallback{
    my $self = shift;
    return $self->isAddRolesValueApplicableForAutoInitialize(@_);
}

sub checkRequiresTenantUserCredentials {
    my $self = shift;
    return $self->LCM::Configuration::AutoInitializeFamilyServicesConfiguration::checkRequiresTenantUserCredentials(@_);
}

sub checkSID {
    my ($self, $sid) = @_;
    $self->SUPER::checkSID($sid);
    $self->showSystemProperties();
    return 1;
}

1;

