package LCM::Configuration::CertifiedHostConfig;

use strict;

use LCM::RegenerateCertificates;
use SAPDB::Install::Hostname;
use SDB::Install::System  qw ($hostname_regex $ipv4_regex);
use SDB::Install::Globals qw ($gOperationCollectHostInfo);
use LCM::Configuration::ParametersCreator;
our @ISA = qw (SDB::Install::Configuration);


sub new{
    my $self = shift->SUPER::new ();
    return $self;
}

sub getParamCertificatesHostmap {
    my ( $self, $order, $section ) = @_;
    my $paramCertificatesHostmap = LCM::Configuration::ParametersCreator::GetParamCertificatesHostmap( $order );
    
    $paramCertificatesHostmap->{'section'} = $section;
    $paramCertificatesHostmap->{'init_with_default'} = 1;
    $paramCertificatesHostmap->{'set_interactive'} = 1;

    return $paramCertificatesHostmap;
}

sub checkCertificatesHostmap {
    my ( $self, $hostname, $certificateHostname, $overwriteAlreadyDefined ) = @_;
    my $certificatesHostmapParam = $self->{params}->{CertificatesHostmap};
    my $certificatesHostmapValue = $certificatesHostmapParam->{value};
    my $lcHostname               = lc($hostname);
    my %originValues             = map { $_ => 1 } @{ $certificatesHostmapParam->{origin_values} };

    if ( !defined $originValues{$lcHostname} ) {
        $self->AddError("Host name '$hostname' is not part of the system");
        return 0;
    }

    if ( !$overwriteAlreadyDefined && defined $certificatesHostmapValue->{$lcHostname} ) {
        $self->PushError("Host name '$hostname' is already defined");
        return 0;
    }

    if ( $hostname !~ /$hostname_regex|$ipv4_regex/ ) {
        $self->PushError("'$hostname' is no valid host name or ipv4 address");
        return 0;
    }

    return 1;
}

sub setCertificatesHostmap {
    my ( $self, $hostname, $certificateHostname, $overwriteAlreadyDefined ) = @_;

    if ( !$self->checkCertificatesHostmap( $hostname, $certificateHostname, $overwriteAlreadyDefined ) ) {
        return 0;
    }
    if ($certificateHostname eq ""){
        $certificateHostname = undef;
    }    
    $self->{params}->{CertificatesHostmap}->{value}->{ lc($hostname) } = (defined $certificateHostname) ?  $certificateHostname : undef;
    return 1;
}

sub fillCertificatesHostmapDefaultValuesAndLocalData {
    my ($self, $useSHAOperation) = @_;
    $self->{'params'}->{'CertificatesHostmap'}->{'default_map'} = undef;
	$self->fillCertificatesHostmapDataForLocalHost($useSHAOperation);
    my $systemHosts      = $self->getSystemHosts();
	$self->setOriginHosts($systemHosts);
	$self->fillCertificatesHostmapDefaultValues();
    return 1;
}

sub setOriginHosts {
	my ($self, $originHosts) = @_;
	$self->{'params'}->{'CertificatesHostmap'}->{'origin_values'} = $originHosts;
}

# Origin hosts should also be set !
sub fillCertificatesHostmapDefaultValues {
	my ($self, $defaultValue) = @_;

	return if(!exists $self->{params}->{CertificatesHostmap});

	my $collectedValues = $self->{CertificatesHostmap};
	my $param = $self->{params}->{CertificatesHostmap};
	my $hosts = $self->{params}->{CertificatesHostmap}->{origin_values};

	return if(!defined $hosts);

	for my $host (@$hosts){
		next if ( defined $param->{default_map}->{$host} );
		my $value = (defined $defaultValue)? $defaultValue : $host;
		if( $collectedValues->{$host} ){
			$value = $collectedValues->{$host};
		}
		$param->{default_map}->{$host} = $value;
	}
}

sub fillCertificatesHostmapDataForLocalHost {
	my ($self, $useShaOperation) = @_;
	my $localhost = $self->getLocalHanaHost();
	my $localCertificateHostname;

	if ($useShaOperation) {
		$localCertificateHostname = $self->_getCertificateHostnameThroughSHAOperation($localhost, 1);
	} else {
		$localCertificateHostname = LCM::RegenerateCertificates::getCertOwner();
		$localCertificateHostname =~ s/CN=//;
	}

	$self->{CertificatesHostmap}->{$localhost} = $localCertificateHostname;
}

sub _getCertificateHostnameThroughSHAOperation {
	my ($self, $host, $useSidAdm) = @_;

	my $operation = $gOperationCollectHostInfo;
	my $optionMap = {
						'SAPMNT' => $self->getSAPSystem()->get_target(),
						'SID'    => $self->getSID()
					};
	my $progress_message = "Collecting information from host '$host'...";
	my $done_message     = "Information collected from host '$host'.";
	my $error_message    = "Collecting information from host '$host' failed!";
	my @hostArr = ($host);
	my $remoteHostctrlHosts = new SDB::Install::RemoteHostctrlHosts(@hostArr);
	my $msg = $self->getMsgLst ()->addMessage ($progress_message);
	my $saveCntxt = $remoteHostctrlHosts->setMsgLstContext ([$msg->getSubMsgLst()]);
	my $rc = $self->_executeSHAOperation($remoteHostctrlHosts, $host, $operation, $optionMap, $progress_message, $done_message, $error_message, $useSidAdm);

	if (not $rc) {
		$remoteHostctrlHosts->setMsgLstContext ($saveCntxt);
		return undef;
	}

	my $outputs = $remoteHostctrlHosts->getOutputBuffer ();
	my $localCertificateHostname = $self->_getCertificateHostnameFromOutput($outputs, $host);
	$remoteHostctrlHosts->freeOutputBuffer ();
	$remoteHostctrlHosts->setMsgLstContext ($saveCntxt);

	return $localCertificateHostname;
}

sub _executeSHAOperation {
	my ($self, $remoteHostctrlHosts, $host, $operation, $optionMap, $progress_message, $done_message, $error_message, $useSidAdm) = @_;
	
	if (defined $useSidAdm) {
		$remoteHostctrlHosts->useSidadm();
	}

	$remoteHostctrlHosts->resetMsgLstContext ();
	my $remoteExecutionReturnCode = $remoteHostctrlHosts->executeHostctrlParallel($operation,
                                                        $self, # instconfig
                                                        undef, # param IDs
                                                        undef, # password IDs
                                                        undef, # remote ignore
                                                        undef, # remote timeout
                                                        $progress_message, # progress_message,
                                                        $done_message, # done_message,
                                                        $error_message, # error_message,
                                                        $optionMap,
                                                        undef, # host option map
                                                        [$host], # only on hosts
                                                        undef, # do not fail
														1, # Suppress Console msgs
														);
	if ($remoteExecutionReturnCode != 0) {
            $self->PushError ($error_message, $remoteHostctrlHosts);
            return undef;
	}

	return 1;
}

sub _getCertificateHostnameFromOutput {
	my ($self, $outputs, $host) = @_;

	my $success = 1;
	my $parser = new SDB::Install::XMLParser ();

	eval {
		# In some cases CollectHostInfo contains redundant text before xml and parsing fails				
		$parser->parsestring ( substr( $outputs->[0], index( $outputs->[0], '<?xml version="1.0" encoding="UTF-8"?>' ) ) );
	};
	if ($@){
		my $parserErr  = $@;
		my $hostOutput = $outputs->[0];
		my $errlst    = new SDB::Install::MsgLst();

		if (!($parserErr =~ s/^\r\n//)){
			$parserErr =~ s/^\n//
		}

		if (!defined $hostOutput || (length($hostOutput) == 0)) {
			$errlst->addError('CollectHostInfo does not return any information');
		}
		else {
			if (length($hostOutput) > 10000) {
				$hostOutput = substr($hostOutput, 0, 10000) . '...';
			}
			$errlst->addError ("Output of CollectHostInfo: $hostOutput");
		}
		$errlst->addError("Error message from XML parser: $parserErr");
		$self->PushError  ("Cannot parse output of CollectHostInfo of host '$host'", $errlst);
		$success = undef;
	}
	$self->{collectHostInfoData}->{$host} = $parser;

	if (!$success){
		return undef;
	}

	my $localCertificateHostname;
	my $data = $self->{collectHostInfoData}->{$host}; 
	my $certificateHostname = $data->getElementByTagName ('certificateHostname');
	if (defined $certificateHostname && defined $certificateHostname->{content}){
		$localCertificateHostname = $certificateHostname->{content};
	}

	delete $self->{collectHostInfoData};

	return $localCertificateHostname;
}

1;
