package LCM::Configuration::ExtractComponentsConfiguration;

use strict;
use File::Basename;
use LCM::FileUtils;
use LCM::SummaryTreeBuilder;
use LCM::Configuration::ParametersCreator;
use parent qw(LCM::Configuration::ComponentArchivesConfiguration);
use LCM::Configuration::ValueChangeListeners::ExtractComponents::ComponentArchivesDirListener;

sub new {
    my $class = shift();
    my $self = $class->SUPER::new(@_);

    $self->_defineExtractComponentsParams();
    $self->_addListeners();

    return $self;
}

sub CheckParams {
    my ($self, $batchMode) = @_;
    return $self->SUPER::CheckParams($batchMode);
}

sub setComponentArchivesPath {
    my ($self, $pathValue) = @_;

    if (! $self->checkComponentArchivesPath($pathValue)) {
        return 0;
    }
    my $realPath = MyRealPath($pathValue);
    $self->{params}->{ComponentArchivesPath}->{value} = $realPath;
    return 1;
}

sub checkComponentArchivesPath {
    my ($self, $pathValue) = @_;

    if (! -d $pathValue) {
        $self->appendErrorMessage("Invalid directory '$pathValue'.");
        return 0;
    }

    my $realPath = MyRealPath($pathValue);
    if (! -r $realPath) {
        $self->appendErrorMessage("No read permissions for the directory '$realPath'.");
        return 0;
    }
    if (! $self->_containsValidArchives($realPath)) {
        $self->appendErrorMessage("No ZIP, TGZ, TAR or SAR archives were found in the directory '$realPath'");
        return 0;
    }
    return 1;
}

sub getAction {
    return "extract_components";
}

sub createSummaryTree {
    my ($self) = @_;
    return LCM::SummaryTreeBuilder::buildSummaryTree($self);
}

sub getProductName {
    return "Extract Components";
}

sub getSummaryTitle {
    return "Extract Components Parameters";
}

sub getTimeoutValues {
    return [];
}

# Important - changes done to this sub might alse affect LCM::Configuration::Web::ExtractComponentsConfiguration::_addListeners
sub _addListeners {
    my ($self) = @_;
    $self->addParameterListener('ComponentArchivesPath', new LCM::Configuration::ValueChangeListeners::ExtractComponents::ComponentArchivesDirListener());
}

# Important - changes done to this sub might alse affect LCM::Configuration::Web::ExtractComponentsConfiguration::_defineExtractComponentsParams
sub _defineExtractComponentsParams {
    my ($self) = @_;
    my $order = scalar keys %{$self->{params}};
    my $section = "ExtractComponents";

    $self->{params} = {
        'ComponentArchivesPath'   => GetParamComponentArchivesPath($order++, $section),
        'ExtractTempDir'          => GetParamExtractTempDir($order++, $section),
        'OverwriteExtractDir'     => GetParamOverwriteExtractDir($order++, $section),
        'SapcarLocation'          => GetParamSapcarLocation($order++, $section),
        'TarExecutableLocation'   => GetParamTarExecutableLocation($order++, $section),
        'UnrarExecutableLocation' => GetParamUnrarExecutableLocation($order++, $section),
    };
}

sub _containsValidArchives {
    my ($self, $realPath) = @_;
    my $errlst = new SDB::Install::MsgLst();
    my $contents = LCM::FileUtils::listDirectory($realPath, $errlst);

    if (! defined $contents) {
        my $errorMessage = $errlst->getMsgLstString();
        $self->appendErrorMessage($$errorMessage);
        return 0;
    }
    my $hasValidArchives = grep(/\.(zip|tar|tar.gz|tgz|sar|EXE)$/i, @$contents);
    return $hasValidArchives;
}

1;