package LCM::Configuration::GenericStackAny;

use strict;
use File::Basename qw (dirname);
use SDB::Install::Configuration;
use SDB::Install::Configuration::AnyMultiHostConfig qw ($validHostRoles);
use SDB::Install::Configuration::Client;
use SDB::Install::Configuration::DatabaseStudio;
use SDB::Install::Configuration::GenericServerPlugin;
use SDB::Install::Configuration::NewDB;
use SDB::Install::SysVars qw ($isApple $isWin $path_separator);
use SDB::Install::System qw (getSysProgPath isAdmin R_OK W_OK X_OK access_check is_nfsidmap_required);
use SDB::Install::Configuration::Upgrade;
use SDB::Install::Globals qw (	$gSapsysGroupName
								$gProductNameAccelerator
								$gProductNameEs
								$gProductNameRDSync
								$gProductNameStreaming
								$gProductNameXS2
								$gHostRoleAcceleratorStandby
								$gHostRoleAcceleratorWorker
								$gHostRoleEsWorker
								$gHostRoleEsStandby
								$gPlatform
								$gKeynameStreaming
								$gKeynameEs
								$gKeynameAccelerator
								$gKeynameRDSync
								$gKeynameXS2
								GetHostRoleProperties
								getHumanReadableRoleByHostRole);
use Exporter;
use experimental qw (smartmatch);
use LCM::HostsParser qw( ExtractLocalAndRemoteHosts );
use LCM::Configuration::Validators::HanaOptionsPathValidator;
use LCM::Utils::CommonUtils qw(getSidadmName);
use SDB::Common::Utils qw(createXSSpaceSAPUserName createXSSpaceProdUserName getSidcryptName createUsername createGroupname);

our @ISA = qw (SDB::Install::Configuration Exporter);

our $progpath = $isWin || $isApple ? getSysProgPath () : '/usr';

our $ini_section_client   = 'Client';
our $ini_section_general  = 'General';
our $ini_section_plugin   = 'Plugin';
our $ini_section_server   = 'Server';
our $ini_section_studio   = 'Studio';
our $ini_section_hlm 	  = 'LifecycleManager';
our $ini_section_xsa      = 'XS_Advanced';
our $ini_section_addhosts = 'AddHosts';
our $ini_section_refdata  = 'Reference_Data';

our @EXPORT = qw (
    INSTALL_ACTION
	UPDATE_ACTION
	UNINSTALL_ACTION
	ADD_HOSTS              
	REMOVE_HOSTS
	UPDATE_COMPONENTS_ACTION
    $ini_section_client
    $ini_section_general
    $ini_section_plugin
    $ini_section_server
    $ini_section_studio
    $ini_section_hlm
    $ini_section_xsa
    $ini_section_addhosts
    $ini_section_refdata
);

sub INSTALL_ACTION   {"install"};
sub UPDATE_ACTION    {"update"};
sub UNINSTALL_ACTION {"uninstall"};
sub ADD_HOSTS {"add_hosts"};
sub REMOVE_HOSTS {"remove_hosts"};
sub UPDATE_COMPONENTS_ACTION {"update_components"};

sub new{
    my $self = shift->SUPER::new (@_);
    return $self;
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing the parameters and
# pass-through parameters in order to call hdbinst, hdbupd, etc.

sub getCmdLineArgsFromInstconfig {
    my ($self,
        $instconfig,   # e.g. SDB::Install::Configuration::NewDB, etc.
        $skipParamIds, # an array containing paramter IDs to be sipped
        $section,      # e.g. $ini_section_server; without pass-through parameters if undef
        $toolName)     # default: hdbinst
        = @_;

    my (@args, $paramValue, $param, $opt);

    my $prefixID = (defined $section)
                   ? $self->getPassThroughParamPrefix($section, $toolName)
                   : undef;

    foreach my $paramId (keys %{$instconfig->{params}}){
        $param = $instconfig->{params}->{$paramId};
        
        $opt = $param->{opt};

        if (!defined $opt || !exists $param->{type}){
            next;
        }

        if ($param->{type} eq 'passwd' || $param->{type} eq 'initial_passwd'){
            next;
        }

        if (defined $skipParamIds){
            my $found = 0;
            foreach my $skipId (@$skipParamIds){
                if ($skipId eq $paramId){
                    $found = 1;
                    last;
                }
            }
            if ($found){
                next;
            }
        }

        $paramValue = $self->getValue ($paramId);
        if (!defined $paramValue && defined $prefixID) {
            $paramValue = $self->getValue ($prefixID.$paramId);
        }

        if ( defined $paramValue ) {
            if ($param->{type} eq 'boolean') {
                $paramValue = $paramValue ? 'on' : 'off';
            }
            
            if ( $param->{opt} eq 'ignore_not_existing' ) {
                push @args, '--' . $opt; 
            } else {
                push @args, '--' . $opt . '=' . $paramValue;
            }
        }
    }
    my $pathThroughIgnoreArgs = $self->getValue($prefixID.'Ignore');
    my $handlerIgnore         = $self->getOptionIgnore ();
    my $ignoreStr             = undef;

    if (defined $handlerIgnore){
        my $ignoreArgs = $handlerIgnore->getSetKeysValidFor ($instconfig->getOptionIgnore ());
        $ignoreStr     = $handlerIgnore->getOptionStringFromArgList ($ignoreArgs);
    }

    if (defined $pathThroughIgnoreArgs) {

        if (defined($ignoreStr) && length($ignoreStr) > 0 ) {
            $ignoreStr .= ",$pathThroughIgnoreArgs";
        }
        else {
            $ignoreStr = "--ignore=$pathThroughIgnoreArgs";
        }
    }

    if ($ignoreStr){
        push @args, $ignoreStr;
    }

    my $optTimeout = $self->getOptionTimeout ();
    if (defined $optTimeout){
        my $timeoutArgs = $optTimeout->getSetKeysValidFor ($instconfig->getOptionTimeout ());
        my $timeoutStr  = $optTimeout->getOptionStringFromArgList ($timeoutArgs);
        if ($timeoutStr){
            push @args, $timeoutStr;
        }
    }

    if ($self->{options}->{ignore_unknown_option}) {
        push @args, '--ignore_unknown_option';
    }

    if (exists $instconfig->{params}->{SkipHostagentPw}
        && $self->getValue('InstallHostagent')
        && !defined $self->getValue('HostagentPassword')) {

        push @args, $instconfig->getOpt('SkipHostagentPw');
    }

    return  \@args;
}


#-------------------------------------------------------------------------------
# Return a prefix for pass through parameters (e.g. 'hdbinst_server_')

sub getPassThroughParamPrefix {
    my ($self, $section, $toolName) = @_;

    my $prefix = (defined $toolName) ? $toolName : 'hdbinst';
    return $prefix . '_' .  lc($section) . '_';
}

#-------------------------------------------------------------------------------
# Adds pass through parameters of all LCM tools
# if the parameters are not already contained in the parameter hash of hdblcm.
#
# Returns the order number of the last added parameter

sub addPassThroughParameters {

    my ($self,
        $mainParams, # parameter hash of hdblcm
        $order,      # order number for the first additional parameter
        $toolName)   # hdbinst or hdbupd
        = @_;

    my $currOrder = $order;

    $currOrder = $self->addPassThroughComponentParams($mainParams, $currOrder,
                            $toolName eq 'hdbinst' ?
                                new SDB::Install::Configuration::NewDB :
                                new SDB::Install::Configuration::Upgrade,
                            ['CheckOnly', 'Scope', 'ScopeInteractive'],
                            $ini_section_server, $toolName);

    $currOrder = $self->addPassThroughComponentParams($mainParams, $currOrder,
                            new SDB::Install::Configuration::Client,
                            ['SID', 'HostName','PATH'],
                            $ini_section_client);

    $currOrder = $self->addPassThroughComponentParams($mainParams, $currOrder,
                            new SDB::Install::Configuration::DatabaseStudio,
                            ['PATH'], $ini_section_studio);

    $currOrder = $self->addPassThroughComponentParams($mainParams, $currOrder,
                            new SDB::Install::Configuration::GenericServerPlugin,
                            ['SID'], $ini_section_plugin);

    return $currOrder;
}

sub shouldShowPassThroughParams {
	my ($self) = @_;
	return ($self->{options}->{help} && $self->{options}->{pass_through_help});
}

#-------------------------------------------------------------------------------
# Adds parameters of the LCM tool (hdbinst, hdbupd, etc.)
# if the parameters are not already contained in the parameter hash of hdblcm.
#
# Note: This subroutine changes the hash $tool->{params}
#
# Returns the order number of the last added parameter

sub addPassThroughComponentParams {

    my ($self,
        $mainParams,   # parameter hash of hdblcm
        $order,        # order number for the first additional parameter
        $tool,         # instance of the calling tool (e.g. SDB::Install::Configuration::NewDB)
        $skipParamIds, # skips IDs contained in $tool->{params}
        $section,      # e.g. $ini_section_server, $init_section_studio
        $toolName,)     # default: hdbinst
        = @_;

    if (defined $skipParamIds) {
        foreach my $skipId (@$skipParamIds){
            delete $tool->{params}->{$skipId};
        }
    }

    my $showParams = $self->shouldShowPassThroughParams();

    my $currOrder = $order;
    my $prefix    = $self->getPassThroughParamPrefix($section, $toolName);

    foreach my $toolParamID (@{$tool->getParamIds()}) {
        if (!exists $mainParams->{$toolParamID} ||
            !defined $mainParams->{$toolParamID}->{section} ||
            ($mainParams->{$toolParamID}->{section} ne $section)) {

            $currOrder = $self->tryAddPassThroughParam($mainParams,
                                                       $currOrder,
                                                       $tool,
                                                       $toolParamID,
                                                       $section,
                                                       $prefix,
                                                       $showParams);
        }
    }

    my $optHandler = $tool->getOptionIgnore();

    if (defined $optHandler) {

        my $id = $prefix . 'Ignore';

        $mainParams->{$id} = {
            'order'   => $currOrder++,
            'section' => $section,
            'opt'     => $prefix . $optHandler->getOptionName(),
            'opt_arg' => $optHandler->getArgumentSyntax(),
            'type'    => 'csv',
            'value'   => undef,
            'str'     => $optHandler->getLabel(),
            'hidden'  => ($optHandler->isHidden() || !$showParams),
            'set_interactive' => 0,
            'valid_values' => $optHandler->getAllValidKeys()
        }
    }

    return $currOrder;
}


#-------------------------------------------------------------------------------
# Adds the specified pass-through parameter if the parameter exists.
#
# Note: This subroutine changes the hash $tool->{params}
#
# Returns the order number of the last added parameter

sub tryAddPassThroughParam {

    my ($self,
        $mainParams,        # parameter hash of hdblcm
        $order,             # order number for this parameter
        $tool,              # instance of the calling tool (e.g. SDB::Install::Configuration::NewDB)
        $toolParamID,       # e.g. 'ImportContent'
        $section,           # e.g. $ini_section_server
        $passThroughPrefix, # e.g. 'hdbinst_server_'
        $showParams,
       ) = @_;

    my $toolParam = $tool->{params}->{$toolParamID};
    my $currOrder = $order;

    if (defined $toolParam && defined $toolParam->{opt}
        && ($toolParam->{type} ne 'passwd')
        && ($toolParam->{type} ne 'initial_passwd')) {

        $toolParam->{order}      = $currOrder++;
        $toolParam->{section}    = $section;
        $toolParam->{opt}        = $passThroughPrefix . $toolParam->{opt};
        $toolParam->{short_opt}  = undef;
        $toolParam->{hidden}     = $toolParam->{hidden} || !$showParams;
        $toolParam->{mandatory}  = 0;
        $toolParam->{skip}       = 0;
        $toolParam->{constraint} = undef;
        $toolParam->{init_with_default}  = 0;
        $toolParam->{set_interactive}    = 0;
        $toolParam->{may_be_interactive} = 0;

        $mainParams->{$passThroughPrefix . $toolParamID} = $toolParam;
    }
    return $currOrder;
}


sub _validateComponentSpecificPath {
	my ($self, $value) = @_;
	
	if(! -d $value && !isAdmin()){
		$self->appendErrorMessage ("Invalid value '$value'. '$value' must be an existing directory");
		return 0;
	}
	
	if(!LCM::FileUtils::isValidUnquotedAbsolutePath($value)){
		$self->appendErrorMessage ("Invalid value '$value'. '$value' is not a valid absolute path");
		return 0;
	}
	
	if(-e $value && !-d $value){
		$self->appendErrorMessage ("Invalid value '$value'. File '$value' already exists and is not a directory");
		return 0;
	}
	
	return 1;
}

sub checkRole{
    my ($self, $value) = @_;

    if ( ! exists $validHostRoles->{$value} ) {
        my @validHostRoles = keys(%{$validHostRoles});
        $self->PushError ("Invalid value ($value) for host property 'role': only '" . join ("', '", @validHostRoles) . "' are allowed");
        return 0;
    }

	if ( ! $self->isComponentForRoleAvailable( $value ) ) {
		return 0;
	}

	if ( $value eq $gHostRoleAcceleratorStandby && ! $self->canAddAcceleratorStandbyHost() ) {
		return 0;
	}

	if ( $value eq $gHostRoleAcceleratorWorker && ! $self->canAddAcceleratorWorkerHost() ) {
		return 0;
	}
	
	return 1;
}

sub isComponentForRoleAvailable {
    my ($self, $value) = @_;

	my %component = (
		'isStreaming' => {
			'componentKeyname' => $gKeynameStreaming,
			'componentName' => 	$gProductNameStreaming,
		},
		'isExtendedStorage' => {
			'componentKeyname' => $gKeynameEs,
			'componentName' => 	$gProductNameEs,
		},
		'isAccelerator' => {
			'componentKeyname' => $gKeynameAccelerator,
			'componentName' => 	$gProductNameAccelerator,
		},
		'isRDSync' => {
			'componentKeyname' => $gKeynameRDSync,
			'componentName' => 	$gProductNameRDSync,
		},
		'isXS2' => {
			'componentKeyname'   => $gKeynameXS2,
			'componentName' =>  $gProductNameXS2,
		},
	);

	my $isComponentForRoleAvailable = 1;
	for my $key ( keys %component ) {
		if ( $validHostRoles->{$value}->{$key} && ! $self->_isComponentAvailable( $component{$key}{'componentKeyname'} ) ) {
			my $message = "Cannot assign host role '$value'. " . $component{$key}{componentName} . " is neither installed nor selected for installation";
			$self->PushError( $message );
	        $isComponentForRoleAvailable = 0;
	        last;
    	}
	}

    return $isComponentForRoleAvailable;
}

sub _isComponentAvailable {
	my ($self, $componentKeyname) = @_;
	my $scm = $self->getSystemComponentManager();
	if ( ! $scm ) {
        return 0;
    }
	my $componentFound = grep { $_->getComponentKeyName() eq $componentKeyname } @{$scm->getDetectedComponents()};
	return $componentFound;
}

sub canAddAcceleratorWorkerHost {
	my ($self, ) = @_;
	return $self->genericCheckForRolesOnSingleDbSystem( $gHostRoleAcceleratorWorker );
}

sub canAddAcceleratorStandbyHost {
	my ($self, ) = @_;
	return $self->genericCheckForRolesOnSingleDbSystem( $gHostRoleAcceleratorStandby );
}

sub canAddEsWorkerHost {
	my ($self, ) = @_;
	
	if ( ! $self->isSingleTenantSystem() ) {
		return 1;
	}

	return $self->genericCheckForRolesOnSingleDbSystem( $gHostRoleEsWorker );
}

sub genericCheckForRolesOnSingleDbSystem {
	my ($self, $role) = @_;
	
	my @systemHostsWithSpecifiedRole = @ { $self->getSystemHostsWithRole( $role ) };
	if ( @systemHostsWithSpecifiedRole > 0 ) {
		my $humanReadableRole = getHumanReadableRoleByHostRole( $role );
		my $hostsString = join (',', @systemHostsWithSpecifiedRole);
		my $message = "Another host '$hostsString' has already been added with the role $humanReadableRole. ";
        $message .= "Only one host with this role can be added to a HANA instance.";
        $self->PushError( $message );
        return 0;
	}

	@systemHostsWithSpecifiedRole = @ { $self->getAddHostsWithRole( $role ) };
	if ( @systemHostsWithSpecifiedRole > 1 ) {
		my $humanReadableRole = getHumanReadableRoleByHostRole( $role );
		my $hostsString = join (',', @systemHostsWithSpecifiedRole);
		my $message = "More than one host ($hostsString) has been assigned with the role $humanReadableRole. ";
        $message .= "Only one host with this role can be added to a HANA instance.";
        $self->PushError( $message );
        return 0;
	}

	return 1;
}

sub getSystemHostsWithRole {
	my ( $self, $role ) = @_;

	my $sapSystem = $self->_getSAPSystem();
	my $mHostsRoles = $sapSystem->getNewDBInstances()->[0]->getHostRolesInfo();

	my @systemHostsWithRole = ();
	for my $host ( keys %$mHostsRoles ) {
		if ( index($mHostsRoles->{$host}, $role) != -1 ) {
			push @systemHostsWithRole, $host;
		}
	}

	return \@systemHostsWithRole;
}

sub _getSAPSystem {
	my ( $self, ) = @_;
	
	my $sapSystem = $self->getSAPSystem();
	if ( ! defined $sapSystem ) {
		my $sid = $self->getValue('SID');
		my $target = $self->getValue('Target');
		require SDB::Install::SAPSystem;
		$sapSystem = new SDB::Install::SAPSystem();
		$sapSystem->initWithGlobDir( $sid, $target );
	}

	return $sapSystem;	
}

sub getAddHostsWithRole {
	my ( $self, $role ) = @_;
	my $addHosts = $self->getValue('AddHosts');
	if ( ! $addHosts ) {
		return [];
	}

	my ( $localHostString, $remoteHostsString ) = ExtractLocalAndRemoteHosts( $addHosts );
	my @addHostsWithRole = ();
	my $hostsParser = new LCM::HostsParser();

	if ( $localHostString ) {
		$hostsParser->parse($localHostString);
		my $localHostName = $hostsParser->getHosts()->[0];
		if ( $role ~~ @ { $hostsParser->getRoles($localHostName) } ) {
			push ( @addHostsWithRole, $localHostName );
		}
	}

	if ( $remoteHostsString ) {
		$hostsParser->parse($remoteHostsString);
		my $remoteHosts = $hostsParser->getHosts();
		for ( @$remoteHosts ) {
			if ( $role ~~ @ { $hostsParser->getRoles($_) } ) {
				push ( @addHostsWithRole, $_ );
			}
		}
	}

	return \@addHostsWithRole;
}

sub checkNewRolesCompatibility {
	my ($self, $roles) = @_;
	my $hostRoleProperties = GetHostRoleProperties();
	my $rc = 1;

	if ( ! $self->isa( 'LCM::Configuration::GenericStackInstallConfiguration' ) && $self->isSingleTenantSystem() ) {
		if ( $self->_hasMoreThanOneRole( $roles, $gHostRoleAcceleratorWorker ) ) {
			$rc = 0;
		}
		if ( $self->_hasMoreThanOneRole( $roles, $gHostRoleAcceleratorStandby ) ) {
			$rc = 0;
		}
		if ( $self->_hasMoreThanOneRole( $roles, $gHostRoleEsWorker ) ) {
			$rc = 0;
		}
		if ( $self->_hasMoreThanOneRole( $roles, $gHostRoleEsStandby ) ) {
			$rc = 0;
		}
	}

	return $rc;
}

sub _hasMoreThanOneRole {
	my ($self, $roles, $role ) = @_;
	my $rc = 0;
	my $numberOfHosts = grep { $_ eq $role } @$roles;
	if ( $numberOfHosts > 1 ) {
		my $hostRoleProperties = GetHostRoleProperties();
		my $roleString = $hostRoleProperties->{$role}->{str};
		$self->PushError( "There can be only one host with role \"$roleString\"." );
		$rc = 1;
	}
	return $rc;
}

sub isSingleTenantSystem {
    my ($self) = @_;

	my $sapSystem = $self->_getSAPSystem();
	my $instance = $sapSystem->getNewDBInstances()->[0];
    my $dbMode = defined $instance ? $instance->getDbMode() : undef;
	my $isSingleDb = ( defined $dbMode && $dbMode eq 'singledb' );

	return $isSingleDb;
}

sub checkHanaOptionsPathParam {
    my ($self, $param_id, $path) = @_;
    my $validator = new LCM::Configuration::Validators::HanaOptionsPathValidator();
    return $validator->isValid($self, $param_id, $path);
}

sub getFlavourProductName {
    return LCM::App::ApplicationContext::getFlavourProductName();
}

sub getExistingUserId {
    my ($self, $userName) = @_;
    my $user = new SDB::Install::User($userName);
    return $user->id() if($user->exists()); # Local user

    my $remoteUIDs = $self->{remoteUids} || {};
    for my $userId (keys(%{$remoteUIDs})){
        my $userData = $remoteUIDs->{$userId};
        for my $entry (@{$userData}){
            return $userId if($userName eq $entry->[1]); # Remote user
        }
    }
    return undef;
}

sub getExistingGroupId {
    my ($self, $groupName) = @_;
    my $group = new SDB::Install::Group($groupName);
    return $group->id() if($group->exists());

    my $remoteGIDs = $self->{remoteGids} || {};
    for my $groupId (keys(%{$remoteGIDs})){
        my $groupData = $remoteGIDs->{$groupId};
        for my $entry (@{$groupData}){
            return $groupId if($groupName eq $entry->[1]);
        }
    }
    return undef;
}

sub handleExistingOsID {
    my ($self, $existingId, $parameterId) = @_;
    if($self->setValue($parameterId, $existingId)){
        $self->setHidden($parameterId, 1);
        return 1;
    }
    return undef;
}

sub checkReservedOSUserIDs {
    my ($self, $value, $paramID) = @_;
    my $reservedUIDs = $self->getReservedUserIDsProperties();
    if (!$self->checkReservedIDs($reservedUIDs, $value, $paramID, 1)){
        my $username = $reservedUIDs->{$value}->{name};
        $self->getErrMsgLst()->addError("User ID '$value' is already taken for user '$username'");
        return undef;
    }
    return 1;
}

sub checkReservedOSGroupIDs {
    my ($self, $value, $paramID) = @_;
    my $reservedGIDs = $self->getReservedGroupIDsProperties();
    if (!$self->checkReservedIDs($reservedGIDs, $value, $paramID, 0)){
        my $groupname = $reservedGIDs->{$value}->{name};
        $self->getErrMsgLst()->addError("Group ID '$value' is already taken for group '$groupname'");
        return undef;
    }
    return 1;
}

sub checkReservedIDs {
    my ($self, $reservedIDs, $value, $paramID, $isUID) = @_;

    foreach my $reservedID (keys %$reservedIDs){
        my $reservedParamID = $reservedIDs->{$reservedID}->{param};
        if (($reservedParamID eq $paramID && $reservedID != $value && $self->existsReservedID($value, $isUID))
            || ($reservedParamID ne $paramID && $reservedID == $value)) {
            return undef;
        }
    }
    return 1;
}

sub addReservedGroupID {
    my ($self, $paramID, $reservedValue) = @_;
    my $name = createGroupname($paramID, $self->getValue('SID'));
    if (! defined $reservedValue){
        $reservedValue = $self->getValue($paramID) || $self->getDefault($paramID);
    }
    $self->_setReservedID($paramID, $reservedValue, 0, $name);
    $self->deleteUnnesessaryReservedGroupIDs();
}

sub addReservedUserID {
    my ($self, $paramID, $reservedValue) = @_;
    my $name = createUsername($paramID, $self->getValue('SID'));
    if (! defined $reservedValue){
        $reservedValue = $self->getValue($paramID) || $self->getDefault($paramID);
    }
    $self->_setReservedID($paramID, $reservedValue, 1, $name);
    $self->deleteUnnesessaryReservedUserIDs();
}

sub _setReservedID {
    my ($self, $paramID, $reservedID, $isUID, $name) = @_;
    if (! defined $self->{reservedIDs} ){
        $self->{reservedIDs} = {};
    }
    return if ($self->existsReservedID($reservedID, $isUID));
    if($isUID){
        $self->{reservedIDs}->{uids}->{$reservedID}->{param} = $paramID;
        $self->{reservedIDs}->{uids}->{$reservedID}->{name} = $name;
    } else {
        $self->{reservedIDs}->{gids}->{$reservedID}->{param} = $paramID;
        $self->{reservedIDs}->{gids}->{$reservedID}->{name} = $name;
    }
}

sub existsReservedID {
    my ($self, $reservedID, $isUID) = @_;
    if ($isUID){
        return grep( /^$reservedID$/, (keys %{$self->{reservedIDs}->{uids}}) );
    } else {
        return grep( /^$reservedID$/, (keys %{$self->{reservedIDs}->{gids}}) );
    }
}

sub getUpdatedReservedIDs {
    my ($self, $reservedIDs) = @_;
    my ($value, $paramID);
    foreach my $reservedValue (keys %{$reservedIDs}){
        $paramID = $reservedIDs->{$reservedValue}->{param};
        $value = $self->getValue($paramID);
        if ((defined $value && $value != $reservedValue) || $self->isSkipped($paramID)){
            delete ${$reservedIDs}{$value};
            return $reservedIDs;
        }
    }
    return undef;
}

sub deleteUnnesessaryReservedUserIDs {
    my ($self) = @_;
    my $reservedUIDs = $self->getReservedUserIDsProperties();
    my $updatedUIDs = $self->getUpdatedReservedIDs($reservedUIDs);
    $self->setReservedUserIDs($updatedUIDs) if (defined $updatedUIDs);
}

sub deleteUnnesessaryReservedGroupIDs {
    my ($self) = @_;
    my $reservedGIDs = $self->getReservedGroupIDsProperties();
    my $updatedGIDs = $self->getUpdatedReservedIDs($reservedGIDs);
    $self->setReservedGroupIDs($updatedGIDs) if (defined $updatedGIDs);
}

sub getReservedUserIDs {
    my ($self) = @_;
    my @reservedUIDs = ();
    my $reservedUIDsProperties = $self->getReservedUserIDsProperties();
    if (defined $reservedUIDsProperties) {
        @reservedUIDs = (keys %{$reservedUIDsProperties});
    }
    return \@reservedUIDs;
}

sub getReservedGroupIDs {
    my ($self) = @_;
    my @reservedGIDs = ();
    my $reservedGIDsProperties = $self->getReservedGroupIDsProperties();
    if (defined $reservedGIDsProperties) {
        @reservedGIDs = (keys %{$reservedGIDsProperties});
    }
    return \@reservedGIDs;
}

sub getReservedUserIDsProperties {
    my ($self) = @_;
    return $self->{reservedIDs}->{uids};
}

sub getReservedGroupIDsProperties {
    my ($self) = @_;
    return $self->{reservedIDs}->{gids};
}

sub setReservedUserIDs {
    my ($self, $newReservedUIDs) = @_;
    $self->{reservedIDs}->{uids} = $newReservedUIDs;
}

sub setReservedGroupIDs {
    my ($self, $newReservedGIDs) = @_;
    $self->{reservedIDs}->{gids} = $newReservedGIDs;
}

1;
