package LCM::Configuration::GenericStackConfiguration;

use strict;
use parent qw (LCM::Configuration::GenericStackAny
               LCM::Configuration::Streaming::StreamingConfiguration
               LCM::Configuration::ES::ESConfiguration
               LCM::Configuration::AcceleratorConfiguration
               LCM::Configuration::RDSync::RDSyncConfiguration
               LCM::Configuration::LSS::LSSConfiguration
               LCM::Configuration::XS2Configuration
               LCM::Configuration::AddHostsCertifiedHostConfig
               LCM::Configuration::AutoInitializeFamilyServicesConfiguration);
use Exporter;
use Cwd qw (realpath);
use SDB::Install::Configuration;
use SDB::Install::SysVars;
use SDB::Install::Globals qw ($gHostRoleAcceleratorStandby
                              $gHostRoleAcceleratorWorker
                              $gHostRoleEsStandby
                              $gHostRoleEsWorker
                              $gHostRoleRDSync
                              $gHostRoleStandby
                              $gHostRoleStreaming
                              $gHostRoleWorker
                              $gHostRoleXS2Standby
                              $gHostRoleXS2Worker
                              $gOptionOptimizedUpdate
                              $gProductNamePlatform
                              $gProductNameAccelerator
                              $gProductNameClient
                              $gProductNameEngine
                              $gProductNameEs
                              $gProductNameRDSync
                              $gProductNameStreaming
                              $gProductNameXS2
                              $gFlavourPlatform
                              $gFlavourCockpit
                              $gShortProductNameAccelerator
                              $gShortProductNameEs
                              $gShortProductNameRDSync
                              $gShortProductNameStreaming
                              $gShortProductNameXS2
                              $gMandatoryXSComponents
                              $gKeynameLSS
                              $gKeynameXS2
                              $gKeynameEngine
                              $gKeynameInstaller
                              $gKeynameClient
                              $gKeynameStudio
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameRDSync
                              $gKeynameLMStructure
                              $gKeynameCockpitStack
							  $gDirNameCockpitStack
							  $gXSAAppsOnlyDetectedMessage
                              );
use SDB::Install::Configuration::AnyMultiHostConfig qw($validHostRoles);
use SDB::Install::SAPSystem qw (CollectSAPSystems);
use SDB::Install::Version;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Saphostagent qw(sapadmUserExists);
use SDB::Install::System;
use LCM::Installer;
use LCM::Component;
use LCM::ComponentManager::MediumComponentManager qw(@componentDirNames);
use LCM::ComponentManager::SystemComponentManager;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Configuration::NewDB;
use SDB::Install::Configuration::NewServerConfig;
use LCM::Configuration::GenericStackAny; # import of static constants and subroutines
use LCM::Configuration::ValueChangeListeners::Update::SIDListener;
use LCM::Configuration::ValueChangeListeners::RootUserListener;
use LCM::Configuration::ValueChangeListeners::ReservedIDsHandler;
use LCM::Configuration::ValueChangeListeners::Install::InstallHostagentListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedXs2ComponentsListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::AutoAddXS2RolesListener;
use LCM::ComponentManager::ComponentScanner;
use LCM::Configuration::ActionConfiguration;
use LCM::DevelopmentTrace;
use LCM::Utils::CommonUtils;
use LCM::Utils::XsApplicationDependency::InclusiveDependency;
use File::Basename qw(dirname basename);
use LCM::Utils::ComponentActionStringGenerator::InstallUpdate;
use experimental qw (smartmatch);
use LCM::Configuration::ComponentValidation::InstallableDependencyCheck;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::SQLSystemUserPasswordListener;

our $actionMap = {
	'update'  => 'LCM::Configuration::GenericStackUpdateConfiguration',
	'install' => 'LCM::Configuration::GenericStackInstallConfiguration',
};

my $componentDependenciesWarningsCount = 0;

sub new {
	my $self = shift->SUPER::new(@_);
	$self->{params} = {};
	return $self;
}

sub getTimeoutValues{
    return [qw (start_service stop_service start_instance stop_instance)];
}

sub getParamSelectedComponents {
	my ( $order, $section, $isUninstall ) = @_;

	my @component_values = LCM::Component::getAllComponentsBatchKeys($isUninstall);

	@component_values = _filterInternalComponents(\@component_values);
	my @sortedComp    = sort @component_values;
	my $desc          = 'Specifies the components (' . join( ', ', @sortedComp ) . ' and others) to be ';

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
	$desc .= ($isUninstall) ? 'uninstalled' : "installed in combination with $flavourProductName server";

	my $additionalDesc = undef;
	foreach my $currComp (@sortedComp) {
		my $info = undef;
		if ($currComp eq 'es') {
			$info =  "'es'       - $gShortProductNameEs";
		}
		elsif ($currComp eq 'ets') {
			$info =  "'ets'      - $gShortProductNameAccelerator";
		}
		elsif ($currComp eq 'rdsync') {
			$info =  "'rdsync'   - $gShortProductNameRDSync";
		}
		elsif ($currComp eq 'streaming') {
			$info =  "'streaming - $gShortProductNameStreaming";
		}
		elsif ($currComp eq 'xs') {
			$info =  "'xs'       - $gShortProductNameXS2";
		}
		if (defined $info) {
			if (!defined $additionalDesc) {
				$additionalDesc = "Components of additional host roles:";
			}
		$additionalDesc .= "\n  " . $info;
		}
	}
	
	my %param = (
		'order'                       => $order,
		'opt'                         => 'components',
		'opt_arg'                     => 'all|<comp1>[,<comp2>]...',
		'opt_arg_switch'              => '<comp1>[,<comp2>]...',
		'type'                        => 'csv',
		'section'                     => $section,
		'value'                       => undef,
		'init_with_default'           => 1,
		'set_interactive'             => 1,
		'str'                         => 'Components',
		'mandatory'                   => 0,
		'help_with_default'           => 1,
		'interactive_index_selection' => 1,
		'leaveEmptyInConfigDump'      => 1,
		'desc'                        => $desc,
		'additional_desc'             => $additionalDesc,
		'interactive_str'             => 'comma-separated list of the selected indices',
		'console_text'                => '',
        'no_retry'                    => 0,
	);

	my @valid_values = ("all");

	for (@component_values) {
		push( @valid_values, "$_" );
	}

	$param{'valid_values'} = \@valid_values;
	return \%param;
}

sub addDetectedComponentsMsg {
	my($self) = @_;

	my $msg = $self->getMsgLst->addMessage("");
	$self->{detectedComponentsMsg} = $msg;
}

sub addDetectedComponentsToLog {
	my($self) = @_;

	my $mcm = $self->getMediumComponentManager();
	my $msg = $self->{detectedComponentsMsg};
	$msg->setMsgText("Detected components:");
	$msg->resetSubMsgLst();

	my $savedContext = $mcm->setMsgLstContext([ $msg->getSubMsgLst() ]);
	$mcm->addComponentsToLog();
	$mcm->setMsgLstContext($savedContext);
}

sub _filterInternalComponents {
    my ($componentValues) = @_;
    my @filteredComponentValues;
    my $manifest = _getResidentInstallerManifest();
    if (!defined $manifest) {
        return @{$componentValues};
    }
    if (!$manifest->isCustomerReleaseBranch ()) {
        return @{$componentValues};
    }    
        
    foreach my $component (@{$componentValues}) {
        if ($componentKeynameToBatchKey->{$gKeynameStreaming} eq $component) {
            next;
        }
        if ($componentKeynameToBatchKey->{$gKeynameEs} eq $component) {
            next;
        }
        if ($componentKeynameToBatchKey->{$gKeynameAccelerator} eq $component) {
            next;
        }
        if ($componentKeynameToBatchKey->{$gKeynameRDSync} eq $component) {
            next;
        }
        if ($componentKeynameToBatchKey->{$gKeynameXS2} eq $component) {
            next;
        }
        push (@filteredComponentValues, $component);
    }    
    
    return @filteredComponentValues;
}

sub _getResidentInstallerManifest {
    require SDB::Install::Installer;
    my $stackKitPath = new SDB::Install::Installer()->GetRuntimeDir();

    if($isWin) {
        $stackKitPath =~ s/(.*)\\.+\\?/$1/;
    } else {
        $stackKitPath =~ s/(.*)\/.+\/?/$1/;
    }
    my $residentInstallerManifestDir = $stackKitPath.$path_separator.'instruntime';
    my $residentInstallerManifestFile = $residentInstallerManifestDir.$path_separator.'manifest';    
    if(! -f $residentInstallerManifestFile) {
        return undef;
    }
    
    require LCM::ResidentInstallerManifest;
    my $residentInstallerManifest = LCM::ResidentInstallerManifest->new($residentInstallerManifestFile);
    if($residentInstallerManifest->errorState) {
        return undef;
    }
    
    return $residentInstallerManifest;
}

sub getParamCmdLineAction {
	my ($self, $order) = @_;

	my $cmd_line_action = LCM::Configuration::ActionConfiguration::getParamAction($order++);
	$cmd_line_action->{'excludeFromConfigDump'} = 1;
	$cmd_line_action->{'set_interactive'}       = 0;
	$cmd_line_action->{'hidden'}                = 1;
	
	return $cmd_line_action;
}

sub setComponentManager {
    my ($self, $componentManager) = @_;
    $self->{componentManager} = $componentManager;
    $self->{scanner}->setComponentManager($componentManager);
    return 1;
}

sub getComponentManager {
	my ($self) = @_;
	if ( not defined $self->{componentManager} ) {
        $self->{componentManager} = new LCM::ComponentManager::MediumComponentManager($self);
        $self->{componentManager}->setMsgLstContext( [ $self->getMsgLst() ] );
    }
	return $self->{componentManager};
}

sub checkSelectedSystem {
	my ( $self, $selectedSystem ) = @_;

	if ( $selectedSystem eq "install" ) {
		return $self->setAction('install');
	}

	if ( $selectedSystem eq "Exit (do nothing)" ) {
		LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
		exit;
	} 

	if ( $selectedSystem =~ /pending installation/ ) {
		$self->setAction('install');
	} elsif ( not defined $self->isUpdate() ) {
		$self->setAction('update');
	}

	$self->{params}->{SelectedSystem}->{no_retry} = 1;
	my ($sid) = $selectedSystem =~ /^(\w{3})/;

	return $self->setValue( 'SID', $sid );
}
sub checkRootUser {
    my $self = shift;
    return $self->SDB::Install::Configuration::AnyConfig::checkRootUser(@_);
}

sub checkRootPassword {
    my $self = shift;
    return $self->SDB::Install::Configuration::AnyConfig::checkRootPassword(@_);
}

sub getSystemDir{
	my ($self) = @_;
	if ($isWin){
		return join ($path_separator, $self->getValue('Drive'), 'usr', 'sap', $self->getValue('SID'));
	}else{
		return $self->getValue('Target') . $path_separator . $self->getValue('SID');
	}
}


sub checkClientPath {
	my ( $self, $path, $componentName ) = @_;

	if (!defined $componentName){
		$componentName = 'Client';
	}

	my $systemDir = $self->getSystemDir ();

	my $pattern = '^' . quotemeta ($systemDir);

	if ($isWin){
		if ($path =~ /$pattern/i) {
			return 1;
		}
	}
	else{
		if ($path =~ /$pattern/) {
			return 1;
		}
	}
	$self->PushError("$componentName path '$path' is not valid. It has to be located in '$systemDir'");
	return 0;
}

sub checkStudioPath {
	return $_[0]->checkClientPath ($_[1], 'Studio');
}

sub setAutoAddXS2Roles {
	my ($self, $value) = @_;

	return undef if($value !~ /^$bool_false_pattern|$bool_true_pattern$/);

	$value = ($value =~ /^$bool_true_pattern$/) ? 1 : 0;
	$self->{params}->{AutoAddXS2Roles}->{value} = $value;
	return 1;
}

sub initializeSelectedComponents {
	my ($self) = @_;

	$self->handleSelectionOfServerComponent();
	$self->handleSelectionOfInternalComponents();

	if ( !$self->fillValidSelectedComponentValues() ) {
		return undef;
	}

	if ( !$self->fillValidSelectedXs2ApplicationsValues() ) {
		return undef;
	}

	$self->fillSelectedComponentsDefaultValue();
	
	return 1;
}

sub detectInstalledComponents {
	my ( $self, $sid ) = @_;

	if ( not defined $self->{systemComponentManager} ) {
		$self->{systemComponentManager} = new LCM::ComponentManager::SystemComponentManager($self);
		$self->{systemComponentManager}->setMsgLstContext( [ $self->getMsgLst() ] );

		my $componentManager = $self->getComponentManager();
		$componentManager->setSystemComponentManager( $self->{systemComponentManager} );
		$componentManager->setSidAdmUserExecution(1) if(isSidadmin($sid));
	}
	my $systemComponentManager = $self->{systemComponentManager};
	my $rc                     = undef;
	my $sapmnt                 = $self->getValue('Target');
	if ( defined $sapmnt ) {
		$rc = $systemComponentManager->detectComponentsBySapmntSid( $sapmnt, $sid );
	} else {
		$rc = $systemComponentManager->detectComponentsBySid($sid);
	}

	#	do not fail in case of installation scenario
	if ( not defined $rc ) {
		$self->PushError( "Cannot detect installed components", $systemComponentManager->getErrMsgLst() );
		return undef;
	}
	return 1;
}

sub fillValidSelectedXs2ApplicationsValues {
	my ($self) = @_;
	$self->_setDefaultSelectedXs2ApplicationsValues();
	$self->_setSelectedXs2ApplicationsUIandValidValues();
	return 1;
}

sub _setDefaultSelectedXs2ApplicationsValues {
	my ($self) = @_;
	my $xsComponents = $self->getComponentManager()->getXs2ApplicationComponents();
	return if (scalar(@{$xsComponents}) == 0);

	my $cfgValue = $self->getValueFromConfigfile('SelectedXs2Applications');

	my @defaultSelection = ();
	for my $component(@{$xsComponents}) {
		my $componentBatchKey = $component->getComponentBatchKey();
		if ($cfgValue eq 'all') {
			push(@defaultSelection, $componentBatchKey);
			next;
		}
		my $isComponentSAPUI5    = $componentBatchKey =~ /^xsac_ui5_fesv/;
		my $isComponentMandatory = ($componentBatchKey ~~ @{$gMandatoryXSComponents}) ? 1 : 0;
		my $isAlreadyInstalled   = ($component->isUpdate()) ? 1 : 0;
		push(@defaultSelection, $componentBatchKey) if ($isComponentMandatory || $isAlreadyInstalled || $isComponentSAPUI5);
	}
	my $defaultSelectionString = join(',', @defaultSelection) || 'none';
	$self->setDefault('SelectedXs2Applications', $defaultSelectionString);
}

sub _setSelectedXs2ApplicationsUIandValidValues {
	my ($self) = @_;
	my $xsComponents = $self->getComponentManager()->getXs2ApplicationComponents();
	my $xs2ApplicationsParam = $self->{params}->{SelectedXs2Applications};

	if (scalar(@{$xsComponents}) == 0) {
		$xs2ApplicationsParam->{valid_values} = [];
		$xs2ApplicationsParam->{ui_values} = [];
		return;
	}

	$xs2ApplicationsParam->{valid_values} = ['all', 'none'];
	$xs2ApplicationsParam->{ui_values}    = ['All components', 'No components'];

	for my $component (@{$xsComponents}) {
		my $uiString = $self->_getComponentUIString($component);
		push( @{$xs2ApplicationsParam->{ui_values}}, $uiString);
		my $componentBatchKey = $component->getComponentBatchKey();
		push( @{$xs2ApplicationsParam->{valid_values}}, $componentBatchKey);
	}
}

sub _getComponentUIString {
	my ($self, $component) = @_;
	my $componentName = $component->getComponentName();
	my $targetVersion = $component->getVersion();
	my $uiString = sprintf("Install %s version %s", $componentName, $targetVersion);
	my $scm = $self->getSystemComponentManager();
	if($component->isUpdate()){
		my $componentKey = $component->getManifest()->getComponentKey();
		my $installedComponent = $scm->getComponentByKeyName($componentKey);
		my $sourceVersion = $installedComponent->getVersion();
		$uiString = sprintf("Update %s from version %s to version %s", $componentName, $sourceVersion, $targetVersion);
	}
	return $uiString;
}

sub fillValidSelectedComponentValues {
	my ($self)                  = @_;

	my ($orderedKeynames, $componentKeynameToAction) = $self->getComponentKeynameToAction();
	my $selectedComponentsParam = $self->{params}->{SelectedComponents};

	if (!$self->_hasSelectableComponents(keys(%{$componentKeynameToAction}))) {
		#There are no detected components to be selected!
		$selectedComponentsParam->{valid_values} = [];
		$selectedComponentsParam->{ui_values}    = [];
		my $errorMessage = "There are no $gProductNamePlatform components that could be installed or updated.";
		my $componentManager = $self->getComponentManager();
		my $xsComponents = $componentManager->getXs2ApplicationComponents();

		if(scalar(@{$xsComponents}) > 0){
			$errorMessage .= " $gXSAAppsOnlyDetectedMessage";
		}

		$self->getErrMsgLst()->addProgressMessage($errorMessage);
		return undef;
	}

	$selectedComponentsParam->{valid_values} = ["all"];
	$selectedComponentsParam->{ui_values}    = ["All components"];

	if ( !$self->isUpdate() || $self->isBootstrapFromResident()) {
		splice( @{ $selectedComponentsParam->{valid_values} }, 1, 0, "server" );
		splice( @{ $selectedComponentsParam->{ui_values} },    1, 0, "No additional components" );
	}

	for my $componentKeyname ( @{$orderedKeynames} ) {
		my $component = $self->getComponentManager()->getComponentByKeyName($componentKeyname);
		if ( !defined $component || $component->isInternal() || !$component->canSelectComponent() ) {
			next;
		}

        if ( $self->isScopeInstance() && !$component->isCompatibleWithScopeInstance() ) {
            $self->getMsgLst()->addProgressMessage( $component->getNotCompatibleWithScopeInstanceMessage() );
            next;
        }

		my $componentBatchKey = $component->getComponentBatchKey();
		push( @{ $selectedComponentsParam->{valid_values} }, $componentBatchKey );
		push( @{ $selectedComponentsParam->{ui_values} },    $componentKeynameToAction->{$componentKeyname} );
	}

	return 1;
}

sub _hasSelectableComponents {
	my ($self, @componentKeynames) = @_;
	my $componentManager = $self->getComponentManager();
	return 0 if (!defined($componentManager) || scalar(@componentKeynames) == 0);

	for my $componentKeyname (@componentKeynames) {
		my $component = $componentManager->getComponentByKeyName($componentKeyname);

		next if (!defined($component) || $component->isInternal());
		next if (!$component->canSelectComponent() && !$component->isComponentSelected());

		return 1;
	}
	return 0;
}

sub fillSelectedComponentsDefaultValue {
	my ($self)                  = @_;
	my $selectedComponentsParam = $self->{params}->{SelectedComponents};
	my ($orderedKeynames, $componentKeynameToAction) = $self->getComponentKeynameToAction();
	my $stackUpdate             = $self->getComponentManager()->isStackUpdate();
	my $defaultSelection        = "";
	for my $componentKeyname ( @{$orderedKeynames} ) {
		my $component = $self->getComponentManager()->getComponentByKeyName($componentKeyname);
		if ( !defined $component || $component->isInternal() || !$component->canSelectComponent() ) {
			next;
		}
		my $defaultSelected;
		if ( $component->getComponentName() eq $gProductNameClient ) {
			$defaultSelected = 1;
		} else {
			$defaultSelected = $component->getDefaultSelection($self->isUpdate());
		}
		if ( !$defaultSelected ) {
			next;
		}
		my $batchKey = $component->getComponentBatchKey();
		
        if ( grep( /^$batchKey$/, @{ $selectedComponentsParam->{valid_values} } ) ) {			
   			if ( (scalar $defaultSelection) ) {
   				$defaultSelection .= ",";
   			}
   		$defaultSelection .= $batchKey;
   	    }
	}
	my $selectAllComponentsByDefault = 1;
    for my $valid_value (@{$selectedComponentsParam->{valid_values}}) {
    	next if $valid_value eq 'all' || $valid_value eq 'hdblcm';
        
        if ( ! grep( /^$valid_value$/, split(',', $defaultSelection) ) ) {
        	$selectAllComponentsByDefault = 0;
        	last;
        }
    }
    $defaultSelection = 'all' if $selectAllComponentsByDefault;
    $self->setDefault('SelectedComponents', $defaultSelection);
}

sub handleSelectionOfResidentInstaller {
	my ( $self ) = @_;

	my $mcm = $self->getMediumComponentManager();
	if (! defined $mcm) {
		return undef;
	}

	my $serverComponent = $mcm->getComponentByKeyName( $gKeynameEngine );
	my $hdblcmComponent = $mcm->getComponentByKeyName( $gKeynameInstaller );

	if (! defined $hdblcmComponent) {
		return 1;
	}

	my $isServerSelected = ((defined $serverComponent) and ($serverComponent->isComponentSelected()));
	$hdblcmComponent->selectComponent($isServerSelected);

	my $hdblcmCmpStackKitPath;

	if ($isServerSelected) {
		my $serverCmpPath = $serverComponent->getPath();
		$hdblcmCmpStackKitPath = File::Spec->catdir(dirname($serverCmpPath), 'instruntime');
	} else {
		$hdblcmCmpStackKitPath = new SDB::Install::Installer()->GetRuntimeDir();
	}

	my $hdblcmCmpManifest = $mcm->getResidentInstallerManifestByDir($hdblcmCmpStackKitPath);
	if (! defined $hdblcmCmpManifest) {
		$self->appendErrorMessage ("Error in detection of components", $mcm->getErrMsgLst ());
		return 0;
	}

	$hdblcmComponent->setPath($hdblcmCmpStackKitPath);
	$hdblcmComponent->setManifest($hdblcmCmpManifest);

	return 1;
}

sub gatherTargetSetOfComponents {
    my ($self, $selectedOnly) = @_;
    my $targetComponents = [];
    my $mcm = $self->getMediumComponentManager();
    my $scm = $mcm->getSystemComponentManager();
    my @targetComponents = grep { $_->isComponentSelected() } @{$mcm->getDetectedComponents()};

    return [ @targetComponents ] if(! defined($scm));

    for my $installedComponent (@{$scm->getDetectedComponents()}) {
        my $componentKeyname = $installedComponent->getComponentKeyName();
        my $candidateComponent = $mcm->getComponentByKeyName($componentKeyname);

        next if (defined($candidateComponent) && $candidateComponent->isComponentSelected());

        push (@targetComponents, $installedComponent);
    }

    return [ @targetComponents ];
}

sub validateComponent {
	my ($self, $component, $manifests, $selectedOnly) = @_;
	my $validation = LCM::Configuration::ComponentValidation::InstallableDependencyCheck->new();
	$validation->addMessageHandler($self);
	return $validation->checkComponentDependencies($self,$component,$manifests,$selectedOnly);
}

sub _getRequiresComponentMessage {
    my ($self, $component, $keyname, $minVersion, $maxVersion) = @_;

    my $caption = $component->getComponentKeyCaptionByKeyName($keyname);
    my $messageVersionString = $self->getVersionStringMessage($minVersion, $maxVersion);
    return $component->getComponentName() . " requires " . $caption . " with version $messageVersionString.";
}

sub getVersionStringMessage {
    my ($self, $minVersion, $maxVersion) = @_;

    my $minVersionString = $minVersion->getVersionString();
    my $minVersionInclusiveString = $minVersion->{inclusive} ? "inclusive" : "exclusive";
    my $maxVersionString = $maxVersion->getVersionString();
    my $maxVersionInclusiveString = $maxVersion->{inclusive} ? "inclusive" : "exclusive";

    my $messageVersionString = $minVersion->getVersionString();
    if ($minVersionString ne $maxVersionString) {
        $messageVersionString = "between " . $minVersion->getVersionString() . " " . $minVersionInclusiveString .
                  " and " . $maxVersion->getVersionString() . " " . $maxVersionInclusiveString;
    }

	return $messageVersionString;
}

sub validateComponentsDependencies {
    my ($self) = @_;
    my $isPrepareUpdate = $self->isPrepareUpdate();
   
    $self->resetComponentDependencyCounter();
    $self->clearParameterWarnings('SelectedComponents');
    my $validation = LCM::Configuration::ComponentValidation::InstallableDependencyCheck->new();
    $validation->addMessageHandler($self);
	if(!$validation->check($self) && !$isPrepareUpdate){
		return 0;
	}
    if ($isPrepareUpdate && $componentDependenciesWarningsCount > 0) {
        my $suffix = ($componentDependenciesWarningsCount > 1) ? 's' : '';
        my $msg = sprintf('The required component%s must be provided later when executing the prepared update.', $suffix);
        $self->AddWarning($msg);
        $self->addParameterWarning('SelectedComponents', $msg);
    }
    return 1;
}

sub validateDependenciesToXs2Components {
    my ($self) = @_;
    my $targetComponents = $self->gatherTargetSetOfComponents();

    $self->clearParameterWarnings('SelectedXs2Applications');
    $componentDependenciesWarningsCount = 0;

    for my $component (@{$targetComponents}) {
        my $rc = $self->_hasValidXs2ComponentDependencies($component);
        if (!$rc && !$self->isPrepareUpdate()) {
            return 0;
        }
    }

	if ($self->isPrepareUpdate() && $componentDependenciesWarningsCount > 0) {
		my $suffix = ($componentDependenciesWarningsCount > 1) ? 's' : '';
		my $msg = 'The required component' . $suffix . ' must be provided later when executing the prepared update.';
		$self->AddWarning($msg);
		$self->addParameterWarning('SelectedComponents', $msg);
	}

    return 1;
}

sub _hasValidXs2ComponentDependencies {
	my ($self, $component) = @_;
	my $manifest = $component->getManifest();
	my $dependencyString = $manifest->getValue('required-components');
	return 1 if (!$dependencyString);

	require SDB::Install::DependencyParser;
	my $parser = new SDB::Install::DependencyParser();
	if (!defined $parser->parse($dependencyString)) {
		$self->getErrMsgLst()->appendMsgLst ($parser->getErrMsgLst ());
		return 0;
	}

	my $targetSetOfAllComponents = $self->gatherTargetSetOfComponents();
# REFACTORING TOPIC:
# Making XS2 Application keynames lower-case just makes it harder to write code
	my $targetSetOfAllComponentKeynames = [ map {uc($_->getComponentKeyName)} @$targetSetOfAllComponents ];
	my $targetXs2Components = $self->_getTargetXs2Components();

	foreach my $vendor (keys %{$parser->{parseTree}}) {
		foreach my $keyname (keys %{$parser->{parseTree}->{$vendor}}) {
			if (!($keyname ~~ @$targetSetOfAllComponentKeynames)) {
				my $caption = $component->getComponentKeyCaptionByKeyName($keyname);
				my $requiresMessage = $self->_getRequiresComponentMessage($component, $keyname,
					$parser->{parseTree}->{$vendor}->{$keyname}->{minVersion},
					$parser->{parseTree}->{$vendor}->{$keyname}->{maxVersion});

				if (!$self->isPrepareUpdate()) {
					$self->appendErrorMessage("Selected component " . $requiresMessage);
					return 0;
				}

				$componentDependenciesWarningsCount++;
				$self->AddWarning($requiresMessage);
				$self->addParameterWarning('SelectedXs2Applications', $requiresMessage);
			}
			foreach my $xs2ComponentKeyname (keys %$targetXs2Components) {
				next if (lc($keyname) ne $xs2ComponentKeyname);

				my $xs2Component = $targetXs2Components->{$xs2ComponentKeyname};
				my $rel = $xs2Component->getScvVersion();
				my $sp = $xs2Component->getSpLevel();
				my $pl = $xs2Component->getPatchLevel();
				my $minDependency = $parser->getMinVersion($vendor, $keyname);
				my $maxDependency = $parser->getMaxVersion($vendor, $keyname);
				if ((defined $minDependency) && (!$minDependency->isCompatible($rel, $sp, $pl)) ||
					(defined $maxDependency) && (!$maxDependency->isCompatible($rel, $sp, $pl))) {
					my $msg;
					my $caption = $component->getComponentKeyCaptionByKeyName($keyname);
					my $requiresMessage = $self->_getRequiresComponentMessage($component, $keyname,
						$parser->{parseTree}->{$vendor}->{$keyname}->{minVersion},
						$parser->{parseTree}->{$vendor}->{$keyname}->{maxVersion});
					if ($component->isa('LCM::Component::Installed')) {
						$msg = "An already installed component " . $component->getComponentName() . " is not compatible with " .
							$caption . " version '" . $xs2Component->getVersionString() . "'. " . $requiresMessage;
							if($component->getComponentBatchKey() eq 'lcapps'){
								$msg.=" To proceed with the update, follow the instructions in SAP Note 2223318.";
							} elsif ($manifest->isServerPlugin()) {
								$msg.=" Updates for plugins are available on the SAP Support Portal. For withdrawn plugins, follow SAP Note 2293092.";
							}
					} else {
						my $selectedStr = ($component->getComponentKeyName eq $gKeynameInstaller) ? 'Auto selected' : 'Selected';
						$msg = "$selectedStr component " . $component->getComponentName() . " is not compatible with " .
							$caption . " version '" . $xs2Component->getVersionString() . "'. " . $requiresMessage;
					}
					if (!$self->isPrepareUpdate()) {
						$self->appendErrorMessage($msg);
						return 0;
					}

					$componentDependenciesWarningsCount++;
					$self->AddWarning($msg);
					$self->addParameterWarning('SelectedXs2Applications', $msg);
					return 1;
				}
			}
		}
	}
	return 1;
}

sub getDetectedXSAComponentsKeyNameList{
	my $self = shift;
	my $mcm = $self->getComponentManager();
	my $detectedXSAComponents = $mcm->getXs2ApplicationComponents();
	my @detectedXSAComponentsKeynames = map {$_->getComponentKeyName} @$detectedXSAComponents;
	return \@detectedXSAComponentsKeynames;
}

sub validateXs2ComponentDependencies {
	my ($self) = @_;
	my $result = 1;

	if ($self->isServerIgnoreCheckComponentDependencies()) {
		$self->getMsgLst ()->addMessage ("Ignoring XS Advanced component dependency check due to ignore option 'check_component_dependencies'");
		return $result;
	}

	my $mcm = $self->getMediumComponentManager();
	my $targetXs2ComponentsSet = $self->_getTargetXs2Components();
	my $xsComponents = $mcm->getXs2ApplicationComponents();
	my %xsComponentMap = map { $_->getComponentBatchKey() => $_ } @{$xsComponents};

	for my $xs2Component (@{$xsComponents}) {
		next if(! $xs2Component->isComponentSelected());

		my $manifest = $xs2Component->getManifest();
		my $dependencies = $manifest->getValue('dependencies-array');

		# If dependency component is detected - use its name,
		# because values in the SL_MANIFEST.XML might be inconsistent
		for my $dependencyArray (@{$dependencies}){
			for my $dependencyObject (@{$dependencyArray}){
				my $technicalName = lc($dependencyObject->getTechnicalName());
				my $detectedComponent = $xsComponentMap{$technicalName};

				if(defined($detectedComponent)){
					my $name = $detectedComponent->getComponentName() || $dependencyObject->getName();
					$dependencyObject->setName($name);
				}
			}
		}
		# If dependency component is detected - use its name,
		# because values in the SL_MANIFEST.XML might be inconsistent

		my $targetPlatformComponentsSet = $self->_buildPlatformComponentDependencySet();
		my %combinedTargetSet = (%$targetXs2ComponentsSet, %$targetPlatformComponentsSet);
		my ($isSatisfied, $errorMessage) = $manifest->checkComponentDependency(\%combinedTargetSet);
		if(! $isSatisfied){
			$self->AddError($errorMessage);
			$result = 0;
		}
	}
	return $result;
}

sub _buildPlatformComponentDependencySet {
	my ($self) = @_;
	my $targetSet = {};
	my $targetPlatformComponents = $self->gatherTargetSetOfComponents();
	for my $platformComponent (@$targetPlatformComponents) {
		my $manifest = $platformComponent->getManifest();
		my $name = $platformComponent->getComponentName();
		my $componentKey = $platformComponent->getComponentKeyName();
		my $scvVersion = $manifest->getValue('release');
		my $spLelvel    = defined $manifest->getValue('rev-number') ? $manifest->getValue('rev-number') : $manifest->getValue('sp-number');
		my $patchLelvel = defined $manifest->getValue('rev-patchlevel') ? $manifest->getValue('rev-patchlevel') : $manifest->getValue('sp-patchlevel');
		my $dependencyObject = new LCM::Utils::XsApplicationDependency::InclusiveDependency($name, lc($componentKey), $scvVersion, $spLelvel, $patchLelvel);

		$targetSet->{lc($componentKey)} = $dependencyObject;
	}
	return $targetSet;
}


sub _getTargetXs2Components {
	my ($self) = @_;
	my $mcm = $self->getMediumComponentManager();
	my $scm = $self->getSystemComponentManager();
	my $result = {};

	my @installedComponentsKeys = map { $_->getComponentKeyName() } @{$self->getInstalledXsComponents()};
	for my $componentKey (@installedComponentsKeys) {
		my $installedApp = $scm->getComponentByKeyName($componentKey);
		my @existingVersion = split('\.', $installedApp->getVersion());
		my $dependencyObject = new LCM::Utils::XsApplicationDependency::InclusiveDependency($componentKey, $componentKey, @existingVersion);

		$result->{$componentKey} = $dependencyObject;
	}

	for my $xs2Component (@{$mcm->getXs2ApplicationComponents()}) {
		next if(! $xs2Component->isComponentSelected());

		my $manifest = $xs2Component->getManifest();
		my $name = $xs2Component->getComponentName();
		my $technicalName = $manifest->getComponentKey();
		my $scvVersion = $manifest->getValue('release');
		my $spLelvel = $manifest->getValue('rev-number');
		my $patchLelvel = $manifest->getValue('rev-patchlevel');
		my $dependencyObject = new LCM::Utils::XsApplicationDependency::InclusiveDependency($name, $technicalName, $scvVersion, $spLelvel, $patchLelvel);

		$result->{$technicalName} = $dependencyObject;
	}
	return $result;
}

sub validateDependencies {
	my ($self) = @_;
	my $srvIgnoreCheckDependencies = $self->isServerIgnoreCheckComponentDependencies();
	my $rc = 1;

    if ($srvIgnoreCheckDependencies) {
        $self->getMsgLst ()->addMessage ("Ignoring component dependency check due to ignore option 'check_component_dependencies'");
    } else {
        $rc = $self->validateComponentsDependencies();
    }

    return $rc;
}

sub isScopeInstance {
	return 0;
}

sub isServerIgnoreCheckComponentDependencies {
    my ( $self ) = @_;
    my $optionIgnore = $self->getOptionIgnore();
    my $ignoreValues = $self->{options}->{'ignore'}; 

    return 1 if(defined($optionIgnore) && $optionIgnore->getArg('check_component_dependencies'));
    return 0 if (!$ignoreValues);

    my @values = split(',', $ignoreValues);
    for my $value (@values) {
        if ('check_component_dependencies' eq $value ) {
            return 1;
        }
    }
    return 0;
}

sub fillSelectedComponents {
	my ( $self, $selectedComponentsCsv ) = @_;
	if ( $self->isUpdate() and not defined $self->getSID() ) {
		$self->PushError( "Option '" . $self->getOpt('SelectedComponents')
		      . "' cannot be speficied if option '--sid' is not set as well" );
		return undef;
	}
	$self->selectAllComponents(0);

	if( ! $self->_restrictComponents($selectedComponentsCsv) ){
    	return undef;      
	}
    	    
	return 1;
}

sub _restrictComponents {
    my ($self, $selectedComponentsCsv ) = @_;
    if (!defined $selectedComponentsCsv){
        return 1;
    }
    my @ListOfSelectedComponents = split( /\s*,\s*/, $selectedComponentsCsv );
    for my $componentBatchValue ( @ListOfSelectedComponents ) {
        if ("all" eq $componentBatchValue) {
            if ( !$self->_handleAllComponentsSelected(scalar @ListOfSelectedComponents) ) {
                return undef;
            }
            last;
       }
       my $component = $self->getComponentByBatchKey($componentBatchValue);

		if (!defined $component) {
			my $errMsg = "Selected component '$componentBatchValue' was not detected in the specified component locations.";
			if ($self->_isResidentScenarioAndServer($componentBatchValue)) {
				$errMsg = "$gProductNameEngine cannot be updated with the resident hdblcm. Start hdblcm from the installation kit.";
			}
			$self->AddError ($errMsg);
			return undef;
		}

       if ($self->_isResidentScenarioAndServer($componentBatchValue)) {
           next;
       }

       $component->selectComponent(1);
    }
    return 1;	  
}

sub _handleAllComponentsSelected {
    my ($self, $numberOfSelectedComponents) = @_;
    if ( $numberOfSelectedComponents > 1 ) {
        $self->AddError ("Some of the provided values are conflicting with each other");
        return undef;
    }
    $self->selectAllComponents(1);
    return 1;
}

sub _isResidentScenarioAndServer {
	my ($self, $componentBatchValue) = @_;
	return ($self->isa ('LCM::Configuration::GenericStackUpdateResidentConfiguration') 
           && $componentBatchValue eq $componentKeynameToBatchKey->{$gKeynameEngine}) ? 1 : 0;
}

sub getRemoteInstallerExecutable{
    my ($self) = @_;
    return $self->getRemoteInstallerExecutableFromKit ();
}

# executable required for CollectHostinfo
sub getInstallerExe {return 'hdblcm';}

sub _recomputeRequiredParameters {
    my $self = shift;

    $self->_recomputeRequiredCredentials();
    $self->_recomputeRequiredParametersBySectionName();
    $self->_recomputeRequiredRootCredentials();
    $self->_recomputeAddHostAddRolesParam();
    $self->_recomputeCertificatesHostmapParam();
    $self->_recomputeRequiredStreamingParams();
    $self->_recomputeRequiredEsParams();
    $self->_recomputeRequiredAcceleratorParams();
    $self->_recomputeRequiredRDSyncParams();
    $self->_recomputeRequiredXS2Params();
    $self->_recomputeRequiredLSSParams();
}

sub _recomputeRequiredCredentials {
    my $self = shift;
    my $mcm = $self->getMediumComponentManager();
    my $componentKeynamesThatRequirePassword = [ 
        $gKeynameEngine, 
        $gKeynameInstaller, 
        $gKeynameXS2
    ];
    if ($mcm) {
        for my $pluginComponent(@{$mcm->getSelectedServerPluginComponents()}) {
            push (@{$componentKeynamesThatRequirePassword}, $pluginComponent->getComponentKeyName());
        }
    }

    if (isSidadmin($self->getSID())) {
        push (@{$componentKeynamesThatRequirePassword}, [$gKeynameClient, $gKeynameStudio]);
    }
    $self->{params}->{'Password'}->{skip} = ($self->_areAnyOfTheComponentsSelected($componentKeynamesThatRequirePassword) || $self->isStreamingNewInstall()) ? 0 : 1;
    
    my $shallSkipSqlCredentials = $self->_isSqlSysPasswdRequired () ? 0 : 1;
    $self->setSkip('SQLSysPasswd', $shallSkipSqlCredentials);
    if ($self->isUpdate()) {
        $self->setSkip('SystemUser', $shallSkipSqlCredentials);
    }

    if (! $self->isUseSAPHostagent()) {
        my $skipHostagentPassword = $self->_shouldSkipHostagentPassword();
        $self->setSkip('HostagentPassword', $skipHostagentPassword);
    } else {
        if ($self->isSkipped('Password') || $self->isSkipped('HostagentPassword')) {
            $self->setSkip('Password', 0);
        }
    }
}

sub _shouldSkipHostagentPassword {
	...
}


sub _isSqlSysPasswdRequired{
    my ($self) = @_;
    my $componentManager = $self->getComponentManager ();
    my $selectedComponents = $componentManager->getSelectedComponents ();
    if (!defined $selectedComponents){
        return 0;
    }
    foreach my $component (@$selectedComponents){
        if ($component->requiresSqlSysPasswd ()){
            return 1;
        }
    }
    return 0;
}


sub _shouldInstallSHAOnHosts {
	my ($self) = @_;
	return 0 if (!$self->getValue("InstallHostagent"));

	my $isServerSelected = $self->_areAnyOfTheComponentsSelected([$gKeynameEngine]);
	my $instance = $self->getOwnInstance();
	my $additionalHosts = $self->getAdditionalRemoteHosts();
	if ($isServerSelected) {
		my $allSystemHosts = (defined $instance) ? $instance->get_allhosts() : [];
		my $currentHostsHaveSapadm = $self->_sapadmUserExistsOnHosts($allSystemHosts);
		my $additionalHostsHaveSapadm = 1;
		if (defined $additionalHosts){
			$additionalHostsHaveSapadm = $self->_sapadmUserExistsOnHosts($additionalHosts->getHostNames());
		}
		return (!$currentHostsHaveSapadm || !$additionalHostsHaveSapadm);
	} else {
		return 0 if (!defined $additionalHosts);
		return ($self->_sapadmUserExistsOnHosts($additionalHosts->getHostNames())) ? 0 : 1;
	}
}

sub _sapadmUserExistsOnHosts {
	my ($self, $hostNames) = @_;

	return 1 if(scalar(@$hostNames) == 0);

	my $instance = $self->getOwnInstance();
	my $remoteHostsWithSapadm = $self->_getRemoteHostsWithSapadm();
	for my $host (@$hostNames) {
		if (defined $instance && $host eq $instance->get_host()) {
			my $shaUser = $self->getValue('HostagentUserName') || 'sapadm';
			return 0 if (!sapadmUserExists($shaUser));
			next;
		}
		next if (!defined $remoteHostsWithSapadm);
		if (! ($host ~~ @{$remoteHostsWithSapadm})) {
			return 0;
		}
	}
	return 1;
}

sub _getRemoteHostsWithSapadm {
	my ($self) = @_;
	return undef if(!exists($self->{remoteSidadms})); # CollectOtherHostInfos is not executed

	my $remoteSapadmUsers = $self->{remoteSidadms}->{sapadm} || [];
	my @flattenedSapadmEntry = map {@$_} @$remoteSapadmUsers;
# get every third element of the flattened array
	my @remoteHostsWithSapadm = map {$flattenedSapadmEntry[$_]} grep {$_ % 3 == 0} 0..$#flattenedSapadmEntry;
	return \@remoteHostsWithSapadm;
}

sub _recomputeRequiredParametersBySectionName {
	my ($self) = @_;
	my $skipValueBySectionName = {
		'Studio' => !$self->_isComponentSelected( $gKeynameStudio ),
		'Client' => !$self->_isComponentSelected( $gKeynameClient )
	};
	my $paramHash;
	foreach my $paramId ( @{ $self->getParamIds() } ) {
		$paramHash = $self->{params}->{$paramId};
		foreach my $section ( keys %$skipValueBySectionName ) {
			if ( defined $paramHash->{section}
				&& $paramHash->{section} eq $section )
			{
				$paramHash->{skip} = $skipValueBySectionName->{$section};
			}
		}
	}
}

sub _isComponentUpdate {
	my ( $self, $componentKeyname ) = @_;
	my $component = $self->getComponentManager()->getComponentByKeyName($componentKeyname);
	if ( defined $component && $component->isUpdate() ) {
		return 1;
	}
	return 0;
}

sub _isComponentSelected {
	my ( $self, $componentKeyname ) = @_;
	my $component = $self->getComponentManager()->getComponentByKeyName($componentKeyname);
	if ( defined $component && $component->isComponentSelected() ) {
		return 1;
	}
	return 0;
}

sub _recomputeRequiredLSSParams{
    my ($self) = @_;
    my $shouldEnableLSSParameters = $self->_isComponentSelected($gKeynameLSS) && !$self->_isComponentUpdate($gKeynameLSS);
    $self->enableLSSParams($shouldEnableLSSParameters);
}

sub _recomputeRequiredStreamingParams{
    my ($self) = @_;
    $self->enableStreamingParamsIfNeeded();
}

sub _recomputeRequiredEsParams{
    my ($self) = @_;
    my $shouldEnableEsParameters = $self->_isComponentSelected($gKeynameEs);
    $self->enableEsParams($shouldEnableEsParameters);
}

sub _recomputeRequiredAcceleratorParams{
    my ($self) = @_;
    $self->enableAcceleratorParams ($self->_isComponentSelected($gKeynameAccelerator));
}

sub _recomputeRequiredRDSyncParams{
    my ($self) = @_;
    $self->enableRDSyncParams ($self->_isComponentSelected($gKeynameRDSync));
}

sub _recomputeRequiredXS2Params {
    my ($self) = @_;
    my $XS2Component = $self->getComponentByBatchKey('xs');
    my $isUpdatingXS2= defined($XS2Component) && $XS2Component->isUpdate();
    my $paramsShouldBeEnabled = defined($XS2Component) && $XS2Component->isComponentSelected();
    my $isContentDeploymentSkipped = $self->hasValue('ImportContentXS2') && ($self->getValue('ImportContentXS2') =~ /$bool_false_pattern/);

    $self->setType('OrgManagerPassword', 'passwd') if($isUpdatingXS2);
    if($paramsShouldBeEnabled && $XS2Component->isUpdate() && !$isContentDeploymentSkipped){
        $self->setSkip('OrgName', 0);
        $self->setSkip('ProdSpaceName', 1);
        $self->setSkip('OrgManagerUser', 0);
        $self->setSkip('OrgManagerPassword', 0);
    } else {
        $self->enableXS2Params($paramsShouldBeEnabled);
    }
    if($paramsShouldBeEnabled) {
        if ($XS2Component->isUpdate()) {
            $self->setSkip('XsEaDataPath', 1);
            $self->setSkip('XSSpaceIsolation', 1);
            $self->setSkip('XSSpaceUserIdSAP', 1);
            $self->setSkip('XSSpaceUserIdProd', 1);
            $self->setSkip('XSSAPSpaceIsolation', 1);
            $self->setSkip('XSCertPem', 1);
            $self->setSkip('XSCertKey', 1);
            $self->setSkip('XSTrustPem', 1);
        } else {
            $self->setDefault('XsDomainName', $self->getDefaultXsControllerHost());
        }
    }
}

sub _skipParameters {
	my ( $self, $parameters, $skip ) = @_;
	for ( @{$parameters} ) {
		$self->{params}->{$_}->{skip} = $skip;
	}
}

sub _areAnyOfTheComponentsSelected {
    my ($self, $components) = @_;
    my $componentManager = $self->getComponentManager();
    return grep {$_->getComponentKeyName() ~~ @{$components}} @{$componentManager->getSelectedComponents()};
}

sub getComponentByBatchKey {
	my ( $self, $batchKey ) = @_;
	return $self->getComponentManager()->getComponentByBatchKey($batchKey);
}

sub selectAllComponents {
	my ($self, $select) = @_;
	my $componentManager = $self->getComponentManager();

	return if (!defined($componentManager));

	my $setSelected = defined($select) ? $select : 1;
	my ($orderedKeynames, $componentKeynameToAction) = $self->getComponentKeynameToAction();
	my @allComponents = map { $componentManager->getComponentByKeyName($_) } @{$orderedKeynames};
	my @selectableComponents = grep { defined($_) && $_->canSelectComponent() } @allComponents;

	$_->selectComponent($setSelected) for(@selectableComponents);
}

sub getComponentKeynameToAction {
	my ($self) = @_;
	my $generator = new LCM::Utils::ComponentActionStringGenerator::InstallUpdate();

	return $generator->getComponentKeynameToActionMap($self, $self->getComponentManager(), $self->{systemComponentManager});
}

sub _isNewer {
	my ( $version1, $version2 ) = @_;
	my $sdb_version1 = new SDB::Install::Version( split( /\./, $version1 ) );
	my $sdb_version2 = new SDB::Install::Version( split( /\./, $version2 ) );
	return $sdb_version1->isNewerThan($sdb_version2);
}

sub getProductName {
    my $flavourProductName = $_[0]->getFlavourProductName();
	return $flavourProductName . " Components";
}

sub _getScanner {
    my ($self) = @_;
    if ( ! defined $self->{scanner} ) {
        $self->{scanner} = LCM::ComponentManager::ComponentScanner->new($self->getComponentManager(), $self, 1, 0);
    }
    return $self->{scanner};
}

sub addListeners { 
    my ($self) = @_;
    require LCM::Configuration::ValueChangeListeners::ComponentListener;
    $self->addParameterListener( 'ComponentDirs', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->_getScanner(),'ComponentDirs'));
    $self->addParameterListener( 'DvdPath', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->_getScanner(),'DvdPath'));
    $self->addParameterListener( 'ComponentFsRoot', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->_getScanner(),'ComponentFsRoot'));
    $self->addParameterListener( 'RootUser', LCM::Configuration::ValueChangeListeners::RootUserListener->new());
    $self->addParameterListener( 'InstallHostagent', LCM::Configuration::ValueChangeListeners::Install::InstallHostagentListener->new() );
    $self->addParameterListener( 'AutoAddXS2Roles', LCM::Configuration::ValueChangeListeners::InstallUpdate::AutoAddXS2RolesListener->new() );

    return if($self->getAction() eq UNINSTALL_ACTION());

    $self->addParameterListener('SelectedXs2Applications', LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedXs2ComponentsListener->new());
   	$self->addParameterListener('SQLSysPasswd', LCM::Configuration::ValueChangeListeners::InstallUpdate::SQLSystemUserPasswordListener->new());
}

sub addEventHandlers {
    my ($self) = @_;
    $self->addConfigurationChangeEventHandler(LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new());
}

sub isUpdate {
    return 0;
}

sub getMediumComponentManager {
	return $_[0]->{componentManager};
}

sub getSystemComponentManager {
    return $_[0]->{componentManager}->getSystemComponentManager();
}

sub getParamDvdPath {
    my ( $self, $order, $section ) = @_;

    my $flavourProductName = $self->getFlavourProductName();
    my %param = (
            'order'                  => $order,
            'opt'                    => 'component_medium',
            'alias_opts'             => ['dvdpath'],
            'type'                   => 'path',
            'section'                => $section,
            'value'                  => undef,
            'default'                => undef,
            'set_interactive'        => 0,
            'str'                    => "Location of $flavourProductName Installation Medium",
            'leaveEmptyInConfigDump' => 1,
            'mandatory'              => 0
    );

    return \%param;
}

sub getParamComponentDirs {
    my ( $self, $order, $section ) = @_;

    my %param = (
            'order'                  => $order,
            'opt'                    => 'component_dirs',
            'opt_arg'                => '<path1>[,<path2>]...',
            'type'                   => 'string',
            'section'                => $section,
            'value'                  => undef,
            'default'                => undef,
            'set_interactive'        => 0,
            'str'                    => 'Comma separated list of component directories',
            'desc'                   => 'Component directories',
            'leaveEmptyInConfigDump' => 1,
            'mandatory'              => 0
    );

    return \%param;
}

sub getParamComponentFsRoot {
    my ( $self, $order, $section ) = @_;

    my %param = (
            'order'                  => $order,
            'opt'                    => 'component_root',
            'opt_arg'                => '<path>',
            'type'                   => 'path',
            'section'                => $section,
            'value'                  => undef,
            'default'                => undef,
            'set_interactive'        => 0,
            'str'                    => 'Directory root to search for components',
            'desc'                   => 'Directory including sub-directories to search for components',
            'leaveEmptyInConfigDump' => 1,
            'mandatory'              => 0
    );

    return \%param;
}

sub getParamArgIgnore {
    my ( $self, $order, $section ) = @_;

    my %param = (
            'order'             => $order,
            'opt'               => 'ignore',
            'opt_arg'           => '<check1>[,<check2>]...',
            'type'              => 'string',
            'section'           => $section,
            'value'             => undef,
            'str'               => 'Ignore failing prerequisite checks',
            'set_interactive'   => 0,
            'skip'              => 1,            
    );

    return \%param;
}

# overwridden in inheriting classes
sub _recomputeCertificatesHostmapParam {}
sub _recomputeRequiredRootCredentials {}
sub _recomputeAddHostAddRolesParam {}

sub _installedShaVersion {
	my $self     = shift;
	my $filename = '/usr/sap/hostctrl/exe/hostagent.mf';
	my $shaVersion;
	if ( -e $filename ) {
		open( FILE, $filename ) || die "Can't open Hostagent manifest file\n";
		my @manifest = <FILE>;
		$shaVersion = $self->_getShaVersion(@manifest);
		close(FILE);
	}
	return $shaVersion;
}

sub _getShaVersion {
	my ( $self, @manifest ) = @_;
	my $shaVersion;
	for (@manifest) {
		$shaVersion = $self->_getShaRecord($_);
		last if $shaVersion;
	}
	return $shaVersion;
}

sub _getShaRecord {
	my ( $self, $line ) = @_;
	if ( $line =~ /hostagent patch number: (\d+)/ ) {
		return $1;
	} else {
		return undef;
	}
}

sub handleSelectionOfInternalComponents {
	my $self                  = shift;
	
	my $residentInstComponent = $self->getComponentManager()->getComponentByKeyName( $gKeynameInstaller );	
	if ( defined $residentInstComponent) {
		$residentInstComponent->setCanSelectComponent(0);
	}

	my $lmstructure_component = $self->getComponentManager()->getComponentByKeyName( $gKeynameLMStructure );
	if ( defined $lmstructure_component ) {
		$lmstructure_component->selectComponent();
		$lmstructure_component->setCanSelectComponent(0);
	}

	my $cockpitStack_component = $self->getComponentManager()->getComponentByKeyName( $gKeynameCockpitStack );
	if ( defined $cockpitStack_component ) {
		$cockpitStack_component->selectComponent();
		$cockpitStack_component->setCanSelectComponent(0);
	}
}

sub createSummaryTree {
	my $self = shift;
	return LCM::SummaryTreeBuilder::buildSummaryTree( $self, $self->getComponentManager() );
}

sub checkComponentSourcesCompatibility {
	my ( $self, $componentFsRoot ) = @_;
	if ( $self->getValue('ComponentFsRoot') and $self->getValue('DvdPath') ) {
		$self->appendErrorMessage("Simultaneous use of component_medium and component_root is forbidden");
		return 0;
	}
	return 1;
}

sub checkDvdPath {
	my ($self, $dvdPath) = @_;
	my $rc = 1;
	if ( !$self->checkComponentSourcesCompatibility() ) {
		$rc = 0;
	}

	return $rc;
}

sub checkSystemRequirements {
	my ($self, $sysinfo) = @_;

    my $componentManager = $self->getComponentManager();
	my $serverComponent = $componentManager->getComponentByKeyName( $gKeynameEngine );

	if (!defined $serverComponent) {
		return 1;
	}

	require SDB::Install::Configuration::NewDB;
	my $handler = new SDB::Install::Configuration::NewDB();
	$handler->{isWin} = $self->{isWin};
	$handler->{optionIgnore} = $self->getOptionIgnore();
	my $oldcontext = $handler->setMsgLstContext($self->getMsgLstContext());
	my $result = $handler->checkSystemRequirements($sysinfo, $serverComponent->getManifest());
	$handler->setMsgLstContext($oldcontext);

    if(defined $handler->{'warnings'}) {
        if(!defined $self->{'warnings'}) {
        	$self->{'warnings'} = [];
        }
        push @{$self->{'warnings'}}, @{$handler->{'warnings'}};
    }
	if (defined $result) {
		return $result;
	}
	if (defined $handler->errorState()) {
		$self->appendErrorMessage("Checking system requirements failed");
		my $errlst = $handler->getErrMsgLst();
		$self->appendErrorMessage (undef, $errlst);
	}

	return undef;
}

sub checkComponentFsRoot {
    my ( $self, $componentFsRoot ) = @_;

    if ( !$self->checkComponentSourcesCompatibility() ) {
        return 0;
    }

    return 1;
}

#
# Gets refresed instance after install/update was executed.
#
sub getRefreshedInstance {
    my $self    = shift();
    my $systems = CollectSAPSystems ( undef, 1 );
    my $error   = "Failed to detect own instance.";

    unless ( $systems ) {
        $self->PushError ( $error );
        return undef;
    }

    my $system = $systems->{ $self->getValue ( 'SID' ) };
    unless ( $system ) {
        $self->PushError ( $error );
        return undef;
    }
    return $system->getNewDBInstances()->[0];
}

sub getRemoteHostsObjects{
    my ($self) = @_;
    return $self->SDB::Install::Configuration::AnyConfig::getRemoteHostsObjects ();
}

sub _checkIfComponentSupportsMultiDb {
	my ( $self, $component ) = @_;
	my $multiDb = 'multidb';
	
    if ( $component->isFeatureUnsupported($multiDb) ) {
	    $self->getErrMsgLst()->addMessage( $component->getComponentName() . " does not support multitenant database containers");
	    return 0;
    }

    return 1;
}

sub getParamJavaVm{
	my($self , $order, $section, $constraint) = @_;
	my $param = $self->SDB::Install::Configuration::AnyConfig::getParamJavaVm($order, $section, $constraint);
	$param->{'mandatory'} = 0;
    $param->{init_with_default} = 0;
    return $param;
}


sub getUIStringFromValue{
	my ($self,$id,$value) = @_;

	my @ui_values = ();
	for ( (split (",", $value) ) ) {
		push(@ui_values, $self->SUPER::getUIStringFromValue( $id, $_ ) );
	}

	return join(',', @ui_values);
}

sub getLogValue{
	my ($self, $param_id, $value) = @_;
	if (!exists $self->{params}->{$param_id}){
		return undef;
	}
	
	if ($param_id ne 'AddRoles') {
		return $self->SUPER::getLogValue($param_id, $value);
	}

	my $param = $self->{params}->{$param_id};
	my $mapParamValue = $param->{value};
	my $mapKeysCount = keys (%$mapParamValue);
	my $i=0;
	my $logValue="";
	while(my ($mapKey, $mapValue) = each (%$mapParamValue)){
        $logValue .= (defined $mapValue) ? "$mapKey -> " . $self->getCSVRoleDescriptiveNames($mapValue)
                   : "$mapKey -> <not defined>";
		$i++;
		if($i < $mapKeysCount){
			$logValue .= ", ";
		}
	}
	
	return $self->SUPER::getLogValue($param_id, $logValue);
}

sub getCSVRoleDescriptiveNames {
	my ($self, $roles, $delimiter) = @_;
	$delimiter = ',' unless defined $delimiter;
	
	require SDB::Install::Configuration::AnyMultiHostConfig;
	my $validRoles = $SDB::Install::Configuration::AnyMultiHostConfig::validHostRoles;

	my @rolesStr = ();
	for my $role ( split(',', $roles) ) {
		push(@rolesStr, $validRoles->{$role}->{str});	
	}
	return join($delimiter, @rolesStr);
}

sub getValidRolesDescriptiveNames {
	my ($self, $validRoles) = @_;
	
	my @rolesStr = ();
	for my $role (keys %$validRoles) {
		push(@rolesStr, $validRoles->{$role}->{str});	
	}
	return join(',', @rolesStr);
}

sub checkHostRoles{
    my ($self, $host, $roles, $existingHostRoles) = @_;
	if ( ! $roles ) {
		$self->getErrMsgLst()->addError("The roles '$roles' specified for host '$host' should not be empty");
		return 0;
	}

    if ( ! defined $existingHostRoles ) {
        $self->getErrMsgLst()->addError("Host name '$host' is not part of the system");
        return 0;
    }

	my @existingHostRolesList = split(/ /, $existingHostRoles);
	my @hostRoles = split(/,/, $roles);
	my $areNewRolesCompatibleWithExistingRoles = $self->SDB::Install::Configuration::AnyMultiHostConfig::areNewRolesCompatibleWithExistingRoles($host, \@hostRoles, \@existingHostRolesList);
	return 0 if ( ! $areNewRolesCompatibleWithExistingRoles );

	for my $role (@hostRoles) {
		if( $role ~~ @existingHostRolesList ){
			$self->getErrMsgLst()->addError("Host role '$role' already assigned to host '$host'");
			return 0;
		}

		return 0 if ( ! $self->checkRole($role) );

		if ($role eq $gHostRoleWorker || $role eq $gHostRoleStandby) {
			$self->getErrMsgLst()->addError ("Adding worker or standby roles is not allowed");
			return 0;
		}
	}

	return 1;
}

sub _isComponentAvailable{
	my ($self, $componentKeyname) = @_;
	my $componentManager = $self->getComponentManager();
	return $componentManager->isComponentSelected($componentKeyname);
}

sub pers_filename {
	return undef;
}

sub _getListOfSelectedComponets {
	my ($self) = @_;

	my $selectedComponentsValue = $self->getValue('SelectedComponents');

	if ($selectedComponentsValue ne 'all') {
		return $selectedComponentsValue;
	}

	my $mcm = $self->getMediumComponentManager();

	if (!defined $mcm) {
		return $selectedComponentsValue;
	}

	my @notInternalSelectedCmpsBatchKeys = ();
	my $selectedComponents = $mcm->getSelectedComponents();

	for my $component (@{$selectedComponents}) {
		if ($component->isInternal()) {
			next;
		}
		my $componentBatchKey = $component->getComponentBatchKey();
		push @notInternalSelectedCmpsBatchKeys, $componentBatchKey;
	}

	return join(',', @notInternalSelectedCmpsBatchKeys);
}

sub getParamPhase{
	my ( $self, $order, $section ) = @_;

    my %param = (
            'order'                  => $order,
            'opt'                    => 'phase',
            'type'                   => 'string',
            'section'                => $section,
            'value'                  => undef,
            'default'                => undef,
            'valid_values'           => [PREPARE_PHASE,ALL],
            'set_interactive'        => 0,
            'str'                    => "Specify single phase for execution relevant for install/update action",
            'hidden'                 => 1,
            'mandatory'              => 0
    );

    return \%param;	
}

sub usePhases{
	my $self = shift();
	my $isSingleOptimizedPhase = defined $self->getPhase()? 1: 0 ;
	return $self->isOptimizedUpdate() || $isSingleOptimizedPhase;
}

sub getPhase{
	my $self = shift();
    return $self->getValue('Phase');
}

sub isPrepareUpdate {
    return 0;
}

sub isOptimizedUpdate {
	my($self) = @_;
	my $updateMode = $self->getValue('UpdateExecutionMode');
	return (defined $updateMode) && ($updateMode eq $gOptionOptimizedUpdate);
}

sub setStepName{
    my($self,$name) = @_;
    $self->{'step_name'} = $name;
}

sub getStepName{
    my $self= shift;
    return $self->{'step_name'};
}

sub setStep {
	my ($self, $step, $isFirstStep) = @_;

	my $selectedCmpsRealValue = $self->getValue('SelectedComponents');
	my $selectedCmpsList = $self->_getListOfSelectedComponets();
	my $persistenceFile = $self->pers_filename();
	my $existsPersistenceFile = $self->pers_exists($persistenceFile);
	
	$self->{'params'}->{'SelectedComponents'}->{'value'} = $selectedCmpsList;
	my $rc = $self->SUPER::setStep($step, $isFirstStep);
	$self->{'params'}->{'SelectedComponents'}->{'value'} = $selectedCmpsRealValue;

	if (!$existsPersistenceFile) {
		if ($rc) {
			$self->getMsgLst()->addMessage( "Status file '$persistenceFile' created." );
		} else {
			$self->getMsgLst()->addError( "Creation of status file '$persistenceFile' failed." );
		}
	}

	return $rc;
}

sub validatePersistency {
	my ($self, $persistenceData) = @_;
	if(! defined($persistenceData)){
		$persistenceData = $self->pers_load();
	}

	return undef if (!defined $persistenceData);
	return undef if (!$self->getPersistenceManager()->validate($persistenceData));

	if ( $self->isUpdate() ) {
		if ($self->getIgnore ('check_pending_upgrade')) {
			return 1;
		}
		$self->setPersistedSteps($persistenceData->{'pers_update_steps'});
	} else {
        $self->setPersistedSteps($persistenceData->{'pers_install_steps'});
	}
    $self->setDeployedXs2Applications($persistenceData->{'pers_xs2_deployed_applications'});
	$self->setStepName($persistenceData->{'step_name'});
	$self->setSuccessfullyPreparedUpdate($persistenceData->{'pers_successfully_prepared_update'});

	my $paramIdsToSkipDeserialization;

	if ( $self->isUpdate() ) {
		if (isSidadmin($self->{current_sid})) {
			$paramIdsToSkipDeserialization = ['RemoteExecution', 'RootUser'];
		}
		if (!$self->getValue('ContinueUpdate')) {
			my $paramIds = $self->getParamIds();
			push @{$paramIdsToSkipDeserialization}, @{$paramIds};
		} elsif ($self->existsComponentLocationBatchValue()) {
			push @{$paramIdsToSkipDeserialization}, 'ComponentDirs';
		} elsif ($self->existsDefaultComponentLocationOverridingPersistedLocation()) {
			my $modifiedPersistedComponentDirs = $self->getModifiedPersistedComponentDirs();
			if ($modifiedPersistedComponentDirs) {
				$persistenceData->{'ComponentDirs'} = $modifiedPersistedComponentDirs;
			} else {
				push @{$paramIdsToSkipDeserialization}, 'ComponentDirs';
			}
		}
	}

	if (!defined $self->SUPER::validatePersistency($persistenceData, $paramIdsToSkipDeserialization)) {
		return undef;
	}

    if ($self->getValue('ContinueUpdate')) {
        $self->setTimeouts($persistenceData->{'pers_timeouts'});
        $self->setIgnore($persistenceData->{'pers_ignore'});
    }

	return 1;
}

sub setIgnore{
    my($self, $ignoreString) = @_;
    my $ignoreOption = $self->getOptionIgnore();

    if($ignoreOption && $ignoreString){
        return $ignoreOption->parseArgs($ignoreString);
    }
    return 1;
}

sub setTimeouts{
    my($self,$timeouts) = @_;
    return 1 unless $timeouts;

    my $optionTimeout = $self->getOptionTimeout();
    return $optionTimeout->parseArgs ($timeouts);
}

sub pers_load {
	my ($self, $file) = @_;
	my $hash = $self->SUPER::pers_load( $file );
	if ( ! $hash ) {
		return undef;
	}

	my $componentManager = $self->getComponentManager();
	if ( defined $componentManager->getComponentByKeyName( $gKeynameInstaller ) ) {
		$self->{pers_hdblcm_kitversion} = $componentManager->getComponentByKeyName( $gKeynameInstaller )->getVersion();
	}
	$self->_setExecutionList($hash->{'step_execution_list'});
	return $hash;
}

sub pers_keys {
	my ($self) = @_;
	my $pers_keys =  [
			'_kitversion',
			'pers_action',
			'pers_hdblcm_kitversion',
			'pers_hostname',
			'pers_timeouts',
			'pers_ignore',
			'step_id',
			'step_execution_list',
            'pers_xs2_deployed_applications'];

	my @component_values = LCM::Component::getAllComponentsBatchKeys();
	foreach my $compBatchValue (@component_values) {
		push @$pers_keys, 'version_'.$compBatchValue;
	}
	return $pers_keys;
}

sub pers_getstepname {
	my $self = shift();
	return $self->getStepName();
}

sub addToPersistedFinishedSteps{
    my ($self, $step) = @_;
    my $persistedSteps = $self->getPersistedSteps();
    if(!$persistedSteps){
       $persistedSteps = $step;
    }else{
       $persistedSteps .= ",".$step;
    }
    $self->setPersistedSteps($persistedSteps);
}

sub setPersistedSteps{...}
sub getPersistedSteps{...}

sub getDeployedXs2Applications{
    my ($self) = @_;
    return $self->{'pers_xs2_deployed_applications'};
}

sub setDeployedXs2Applications{
    my($self,$xs2apps) = @_;
    if (! defined $xs2apps) {
        delete $self->{'pers_xs2_deployed_applications'};
    } else {
        $self->{'pers_xs2_deployed_applications'} = $xs2apps;
    }
}

sub getPersComponentVersion{
    my ($self, $componentBatchValue) = @_;
    return $self->{'version_'.$componentBatchValue};
}

sub setPersComponentVersion{
    my($self, $componentBatchValue, $componentVersion) = @_;
    $self->{'version_'.$componentBatchValue} = $componentVersion;
}

sub getPersistenceManager {
	return $_[0]->{'_persistenceManager'};
}

sub setPersistenceManager {
	$_[0]->{'_persistenceManager'} = $_[1];
}

# Hdblcm persistence is preloaded in order to obtain metadata for the web UI. This results in duplicated
# log messages (see bug 80621). Therefore we prevent those messages from appearing in the log.
sub _loadHdblcmPersistenceSilently {
	my ($self) = @_;
	my ($msgLst, $errLst) = (new SDB::Install::MsgLst(), new SDB::Install::MsgLst());
	my $saveContext = $self->setMsgLstContext([$msgLst, $errLst]);
	my $hdblcmPersistedData = $self->pers_load();

	$self->setMsgLstContext($saveContext);

	return $hdblcmPersistedData;
}

sub getPersistedValue {
	my ($self, $id) = @_;
	return $self->{'hdblcmPersistedData'}->{$id} if defined $self->{'hdblcmPersistedData'};

	$self->{'hdblcmPersistedData'} = $self->_loadHdblcmPersistenceSilently();
	return ( defined $self->{'hdblcmPersistedData'} ? $self->{'hdblcmPersistedData'}->{$id} : undef );
}

sub getAutoAssignXs2RolesMap {
	...
}

sub existsComponentLocationBatchValue {
	my ($self) = @_;
	my @allLocationParameterIDs = ('DvdPath', 'ComponentFsRoot', 'ComponentDirs');
	my @relevantLocationParameterIDs = grep { exists($self->{params}->{$_}) } @allLocationParameterIDs;
	my @locationParamtersWithBatchValue = grep { defined($self->{params}->{$_}->{batchValue}) } @relevantLocationParameterIDs;

	return scalar(@locationParamtersWithBatchValue) > 0;
}

sub _determineApplicableXs2Role {
	my ($self, $hostString) = @_;

	return '' if($hostString =~ /:role=($gHostRoleXS2Standby|$gHostRoleXS2Worker)/);
	return $gHostRoleXS2Worker if($hostString =~ /:role=$gHostRoleWorker/);
	return $gHostRoleXS2Standby if($hostString =~ /:role=$gHostRoleStandby/);
	return '';
}

sub willExecuteAllComponentsInAllPhases {
	return $_[0]->{_willExecuteAllComponentsInAllPhases};
}

sub setWillExecuteAllComponentsInAllPhases {
	$_[0]->{_willExecuteAllComponentsInAllPhases} = $_[1];
}

sub setSuccessfullyPreparedUpdate {
	$_[0]->{'pers_successfully_prepared_update'} = $_[1];
}

sub isSuccessfullyPreparedUpdate {
	return $_[0]->{'pers_successfully_prepared_update'};
}

sub getPendingConfiguration {
	my ($self) = @_;
	return $self->{_pendingConfiguration};
}

sub _setPendingConfiguration {
	my ($self, $dummyConfing) = @_;
	$self->{_pendingConfiguration} = $dummyConfing;
}

sub isPendingConfiguration {
	my ($self) = @_;
	return $self->{_isPendingConfiguration};
}

sub setIsPendingConfiguration {
	my ($self, $isPendingConfiguration) = @_;
	$self->{_isPendingConfiguration} = $isPendingConfiguration;
}

sub _recomputeXs2ApplicationParameters {
	my ($self) = @_;
	my $shouldSkip = $self->_shouldSkipSelectedXs2Applications();
	$self->setSkip('SelectedXs2Applications', $shouldSkip);
	$self->setSkip('XSComponentsNoStart', $shouldSkip);

	if($self->isSkipped('SelectedXs2Applications')){
		my $mcm = $self->getMediumComponentManager();
		$_->selectComponent(0) for(@{$mcm->getXs2ApplicationComponents()});
	}
}

sub _shouldSkipSelectedXs2Applications {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	my $isXs2Selected = $componentManager->isComponentSelected($gKeynameXS2);
	my $areXs2ApplicationsDetected = scalar(@{$componentManager->getXs2ApplicationComponents()}) > 0;
	my $hasXs2WorkerHost = $self->_isXSWorkerHostAvailable();
	my $isContentDeploymentSkipped = $self->hasValue('ImportContentXS2') && ($self->getValue('ImportContentXS2') =~ /$bool_false_pattern/);

	return 1 if($isContentDeploymentSkipped);
	return ($isXs2Selected && $areXs2ApplicationsDetected && $hasXs2WorkerHost) ? 0 : 1;
}

sub clearHostsParameters {
	my ($self, ) = @_;
	$self->{params}->{AddHosts}->{value} = undef;
	$self->{params}->{CertificatesHostmap}->{value} = undef;
	$self->{_additionalRemoteHosts} = undef;
}

sub clearRestrictedComponentsMessages {
    my ($self, ) = @_;
    $self->{_restrictedComponentsMessages} = ();        
}

sub getRestrictedComponentsMessages {
    my ($self, ) = @_;
    return $self->{_restrictedComponentsMessages};        
}

sub addRestrictedComponentsMessage {
    my ($self, $componentName, $message) = @_;
    my $messages = $self->{_restrictedComponentsMessages};    
    $self->{_restrictedComponentsMessages}->{$componentName} = $message;
}

sub getCannotBeSelectedMessage {
    my ($self) = @_;
    my $consoleMessage = "";    
    my $messages = $self->getRestrictedComponentsMessages();
    foreach my $componentName (keys %{$messages}) {
        my $restrictedComponentMessage = $messages->{$componentName};
        $consoleMessage.="Component $componentName cannot be selected. ".$restrictedComponentMessage;
    }
    return $consoleMessage;
}

sub isOnlyComponentDetected {
	my ($self, $componentId) = @_;
	my $externalDetectedComponents = $self->getMediumComponentManager()->getExternalDetectedComponents();
	return ( scalar @$externalDetectedComponents == 1 && @$externalDetectedComponents[0]->getComponentName() eq $componentId) ? 1 : 0;
}

sub isSAPEnvironment {
    my ($self) = @_;
    if (!defined $self->{SAPEnvironment}) {
        $self->{SAPEnvironment} = LCM::Utils::CommonUtils::isSAPEnvironment() ? 1 : 0 ;
    }
    return $self->{SAPEnvironment};
}

sub setAddHostsFilterRegex {
    my ($self, $filterRegex) = @_;
    $self->{_addHostsFilterRegex} = $filterRegex;
}

sub getAddHostsFilterRegex {
    my ($self) = @_;
    return $self->{_addHostsFilterRegex};
}

sub existsDefaultComponentLocationOverridingPersistedLocation {
	my ($self) = @_;
	return $self->{'_existsDefaultComponentLocationOverridingPersistedLocation'};
}

sub setExistsDefaultComponentLocationOverridingPersistedLocation {
	my ($self, $exists) = @_;
	$self->{'_existsDefaultComponentLocationOverridingPersistedLocation'} = $exists;
}

sub getModifiedPersistedComponentDirs {
	my ($self) = @_;
	return $self->{'_modified_persisted_ComponentDirs'};
}

sub setModifiedPersistedComponentDirs {
	my ($self, $componentDirsCsv) = @_;
	$self->{'_modified_persisted_ComponentDirs'} = $componentDirsCsv;
}

sub getFlavourProductName {
    my ($self) = @_;
    my $componentManager = $self->getComponentManager();
    my $flavour = $componentManager->getHANAFlavour() || $gFlavourPlatform;
    my $flavourProductName = SDB::Install::Globals::getFlavourProductName($flavour);
    return $flavourProductName;
}

sub onComponentValidationMessage{
	my($self,$message) = @_;
	if (!$self->isPrepareUpdate()) {
		$self->appendErrorMessage($message);
		return;
	}
	$componentDependenciesWarningsCount++;
	$self->AddWarning($message);
	$self->addParameterWarning('SelectedComponents', $message);
}
sub resetComponentDependencyCounter{
	my $self = shift;
	$componentDependenciesWarningsCount = 0;
}

sub addExecutionStep{
    my($self,$task) = @_;
    return if (!$task->shouldPersistStatus());
    
    my $step = $task->getId();
    my $status = $task->getStatus()->getState();

    if(!defined $self->{_executionList}){
        $self->{_executionList} = {};
    }
    $self->{_executionList}->{$step} = $status;
    my @taskNames = keys %{$self->{_executionList}};
    my $executionList;
    for my $taskName (@taskNames){
        if(!defined $executionList){
            $executionList = "$taskName:".$self->{_executionList}->{$taskName};
        } else {
            $executionList .= ",$taskName:".$self->{_executionList}->{$taskName};
        }
    }
    $self->{step_execution_list} = $executionList;
}

sub _setExecutionList{
 my($self,$persistedValue) = @_;
 my @executions = split (',', $persistedValue);
 for my $execution(@executions){
  if(!defined  $self->{_pers_executionList}){
    $self->{_pers_executionList} = {};
  }
   my($step,$status) = split(":",$execution);
   $self->{_pers_executionList}->{$step} = $status;
 }
}

sub getResumeExecutionList{
 my ($self) = @_;
 return $self->{_pers_executionList};
}

sub isAutoInitializeServicesApplicableParameterCallback{
    my($self,$paramId,$valKey,$value) = @_;
    if ($paramId eq 'AddHosts') {
        return $self->isAddHostsValueApplicableForAutoInitialize($paramId, $valKey, $value);
    }
    if (($paramId eq 'AddRoles') || ($paramId eq 'AddLocalRoles')) {
        return $self->isAddRolesValueApplicableForAutoInitialize($paramId, $valKey, $value);
    }
    return 0;
}

sub checkRequiresTenantUserCredentials{
    my $self = shift;
    my $isAutoInitializeServicesRequired = !$self->isSkipped('AutoInitializeServices') && $self->SUPER::checkRequiresTenantUserCredentials();
    my $xsaComponent =  $self->getComponentManager()->getComponentByKeyName($gKeynameXS2);
    my $isXSASelectedForInstall = (defined $xsaComponent) ? $xsaComponent->isComponentSelected() && !$xsaComponent->isInstalled() : undef;
    my $isInCompatibilityMode = $self->isSystemInCompatibilityMode();
    return  $isAutoInitializeServicesRequired || ($isInCompatibilityMode && $isXSASelectedForInstall);
}

sub getFamilyComponentsWithAutoInitServices {
	my $self = shift;
	my $result = $self->SUPER::getFamilyComponentsWithAutoInitServices(@_);
	my $mediumComponentManager = $self->getMediumComponentManager();
	return $result if (!defined $mediumComponentManager);

	my @matchedMcmComponents = grep { $_->isFeatureSupported('auto_initialize_services') } @{$mediumComponentManager->getAllComponents};
	push (@$result, @matchedMcmComponents);
	return $result;
}

sub getProductInfo {
	my ($self, $sapSys, $flavour, $isPending) = @_;
	return $self->_getDBProductInfo($sapSys) if ($flavour ne $gFlavourCockpit);
	return $self->getCockpitProductInfo($sapSys, $isPending);
}

sub _getDBProductInfo {
	my ($self, $sapSys) = @_;

	my $info   = $sapSys->GetVersion();

	if (defined $info) {
		if ($sapSys->hasNewDB()) {
			my $hosts = $sapSys->getNewDBInstances()->[0]->getAllHostsWithRoles(1);
			$info .= "\n" . join(', ', @$hosts) if (@$hosts);
			my $pluginInstallations = $sapSys->getNewDBInstances()->[0]->getPluginInstallations();
			my @pluginKeys;
			if (defined $pluginInstallations && @$pluginInstallations){
				foreach my $inst (@$pluginInstallations){
					push @pluginKeys, $inst->getProductKey();
				}
				$info .= "\nplugins: " . join(', ', @pluginKeys) if (@pluginKeys);
			}
		}
	} else {
		$info = 'Version is unknown';
	}
	return $info;
}

sub getCockpitProductInfo {
	my ($self, $sapSys, $isPending) = @_;

	my $productInfo = LCM::Utils::CommonUtils::getCockpitProductVersion($sapSys);
	if (!$isPending) {
		my $incompleteMessage = $self->_getSystemMessageIncompleteCockpit($sapSys);
		$productInfo.= "\n".$incompleteMessage if ($incompleteMessage);
	}

	return $productInfo;
}

sub _getSystemMessageIncompleteCockpit {
	my ($self, $sapSys) = @_;

	my $message = 'System is incomplete due to failed installation or update (resume not possible)';
	my $isIncomplete = 0;

	my $manifestDir = File::Spec->catdir($sapSys->get_globalSidDir(), $gDirNameCockpitStack);
	my $manifestPath = File::Spec->catfile($manifestDir, 'manifest');
	my $manifest = new LCM::Manifests::GenericManifest($manifestPath);
	my $parser = new SDB::Install::DependencyParser();
	my $dependencyString = $manifest->getValue('required-components');
	return '' unless($dependencyString && $parser->parse($dependencyString));

	my $scm = new LCM::ComponentManager::SystemComponentManager($self);
	my $sid = $sapSys->get_sid();
	my $rc = $scm->detectComponentsBySid($sid);
	if ( not defined $rc ) {
		$self->PushError( "Cannot detect installed components", $scm->getErrMsgLst() );
		return undef;
	}

	for my $vendor (keys %{$parser->{parseTree}}) {
		for my $keyname (keys %{$parser->{parseTree}->{$vendor}}) {
			$isIncomplete = 0;
			my $minDependency = $parser->getMinVersion($vendor, $keyname);
			my $maxDependency = $parser->getMaxVersion($vendor, $keyname);
			# If the component is undefined, try lower-casing the name because XS applications' keynames
			# are stored with lowercase but are described with uppercase in the Cockpit stack's manifest
			my $component = $scm->getComponentByKeyName($keyname) || $scm->getComponentByKeyName(lc($keyname));
			if (defined $component) {
				my $compName = $component->getComponentName();
				my $version = $component->getVersion();
				my $manifest = $component->getManifest();
				my $incompatibleComponentMessage = $self->_getIncompatibleComponentMessage($compName, $version, $minDependency, $maxDependency, 
											$manifest->getValue('release'), $manifest->getValue('rev-number'), $manifest->getValue('rev-patchlevel'), $manifest->getValue('rev-changelist'));
				if (defined $incompatibleComponentMessage) {
					$isIncomplete = 1;
					$message .= "\n  ".$incompatibleComponentMessage;
				}
			} else {
				$isIncomplete = 1;
				$message .= "\n  $keyname is not installed";
			}
		}
	}

	return $isIncomplete? $message : '';
}

sub _getIncompatibleComponentMessage {
	my ($self, $compName, $version, $minDependency, $maxDependency, $release, $sp, $pl, $cl) = @_;

	if ((defined $minDependency) && (!$minDependency->isCompatible($release, $sp, $pl, $cl)) ||
		(defined $maxDependency) && (!$maxDependency->isCompatible($release, $sp, $pl, $cl))) {
		my $versionsString = $self->getVersionStringMessage($minDependency, $maxDependency);
		return "$compName is version $version, but should be $versionsString";
	} else {
		return undef;
	}
}

sub getAddLocalRolesString {
    my ($self) = @_;
    my $isAutoAssignEnabled = $self->getValue('AutoAddXS2Roles') && ! $self->isSkipped('AutoAddXS2Roles');
    my $autoAssignedRolesMap = $isAutoAssignEnabled ? $self->getAutoAssignXs2RolesMap() : {};
    my $localHost = $self->getLocalHanaHost();
    my $autoAssignedRole = $autoAssignedRolesMap->{$localHost};

    my $localRolesString = '';
    if(defined $autoAssignedRole && length($autoAssignedRole) > 0){
        $localRolesString = $autoAssignedRole;
    }

    my $addLocalRolesString = $self->getValue('AddLocalRoles');
    if(defined($addLocalRolesString) && length($addLocalRolesString) > 0){
        $localRolesString = length($localRolesString) > 0 ? 
                 join(',', $localRolesString, $addLocalRolesString) : $addLocalRolesString;
    } else {
        my $rolesMap = $self->getValue('AddRoles') || {};
        if(defined $rolesMap->{$localHost} && length($rolesMap->{$localHost}) > 0) {
            $localRolesString = length($localRolesString) > 0 ? 
                 join(',', $localRolesString, $rolesMap->{$localHost}) : $rolesMap->{$localHost};
        }
    }
    return '' if( ! defined($localRolesString) || length($localRolesString) == 0);
    return $localRolesString;
}

sub setPersistedValue {
	my ($self, $id, $param, $persistedValue) = @_;

# We don't need to verify signatures of persisted components during resume
	my $cachedVerifySignatureValue = $self->getValue('VerifySignature');
	if ($id eq 'SelectedComponents' && $self->isPendingConfiguration() && defined $cachedVerifySignatureValue) {
		$self->{params}->{VerifySignature}->{value} = 0;
		my $rc = $self->SUPER::setPersistedValue($id, $param, $persistedValue);
		$self->{params}->{VerifySignature}->{value} = $cachedVerifySignatureValue;
		return $rc;
	}
	return $self->SUPER::setPersistedValue($id, $param, $persistedValue);
}

sub defineLSSParams {
    my ($self) = @_;
    if($self->getComponentManager()->isInternalBuild()){
        $self->SUPER::defineLSSParams();
    }
    return 1;
}

sub canIgnoreVerificationFailure {
    return 1;
}

# This method is overriden to ensure that if someone tries to disable the verification from
# auto configuration file, we won't allow it
sub getValueFromConfigfile {
    my $self = shift();
    my $paramId = $_[0];
    if ($paramId eq 'VerifySignature' && $self->isAutoConfigurationDetected()) {
        return undef if !$self->isAutoConfigInSignature();
    }
    return $self->SUPER::getValueFromConfigfile(@_);
}

# This method is overriden to ensure that if someone tries to disable the verification from
# auto configuration file, we won't allow it
sub getDefaultValueFromConfigfile {
    my $self = shift();
    my $paramId = $_[0];
    if ($paramId eq 'VerifySignature' && $self->isAutoConfigurationDetected()) {
        return undef if !$self->isAutoConfigInSignature();
    }
    return $self->SUPER::getDefaultValueFromConfigfile(@_);
}

sub isAutoConfigurationDetected {
    my ($self) = @_;
    return $self->{isAutoConfigDetected};
}

sub isAutoConfigInSignature {
    my ($self) = @_;
    my $mcm = $self->getComponentManager();
    my $serverComponent = $mcm->getComponentByKeyName($gKeynameEngine);
    my $signature = $serverComponent ? $serverComponent->getSignature() : undef;
    if ($signature) {
        my $configfile = basename($self->{config_file});
        my $signaturePattern = qr/\s+(\w+?\/)?configurations\/$configfile/;
        return $signature->containsEntry($signaturePattern, 1) ? 1 : 0;
    }
    return 0;
}

sub isBootstrapFromResident {
    my ($self) = @_;
    return $self->getValue('BootstrapFromResident');
}

sub isPendingAction { ... };
sub checkAddHostsAvailability { ... };

sub _checkAddHosts {
    my ($self, $value) = @_;

    return 1 if ! defined $value;
    return 0 if ! $self->checkAddHostsAvailability();
    # Remove from the value already added hosts
    my $param = $self->{params}->{'AddHosts'};
    if ($self->isPendingAction()){

        my $instance = $self->getOwnInstance();
        return undef if ! defined $instance;

        my $persValue        = $param->{persFileValue};
        my $msgLstContext    = $self->getMsgLstContext();
        my $addHostsList     = [split(',', $value)];
        my $persAddHostsList = defined $persValue ? [split(',', $persValue)] : undef;
        my $hostDetector     = LCM::Hosts::HostsDetector->new(undef, $msgLstContext, $instance);
        $value = $hostDetector->filterExistingHostsCmd($addHostsList, $persAddHostsList);
        return 1 if ! $value;
    }
    return undef;
}

1;
