package LCM::Configuration::GenericStackUninstallation;

use strict;
use experimental qw(smartmatch);
use parent qw(LCM::Configuration::GenericStackConfiguration SDB::Install::Configuration::AnyConfig);
use SDB::Install::SysVars;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::System qw(isSidadmin isAdmin);
use SDB::Install::Configuration::ServerUninstParams;
use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::Globals qw($gProductNameEngine $gHostRoleAcceleratorWorker $gHostRoleXS2Worker $gHostRoleXS2Standby $gKeynameEngine $gKeynameLMStructure $gXSParametersRemoveHostOrRolesConstraint);
use LCM::Component qw(COMPONENT_NAME_LM_STRUCTURE);
use LCM::Installer;
use LCM::App::ApplicationContext;
use LCM::Configuration::GenericStackAny;
use LCM::SummaryTreeBuilder;
use LCM::Configuration::GenericStackConfiguration;
use LCM::Utils::RemoveRoles qw(IsRemovingRoleFromHost);
use LCM::Utils::ComponentActionStringGenerator::Uninstall;
use LCM::Configuration::ParametersCreator qw(getParamRemoveHostRoles GetParamRemoveHosts);
use LCM::Configuration::ValueChangeListeners::Uninstall::ForceListener;
use LCM::Configuration::ValueChangeListeners::Uninstall::NameserverConnectivityListener;
use LCM::Configuration::ValueChangeListeners::Uninstall::SelectedComponentsListener;
use LCM::Configuration::ValueChangeListeners::Uninstall::RecomputeCredentialsListener;
use SDB::Install::Configuration::TenantUserListener;

my $REMOVE_HOSTS_SECTION = 'RemoveHosts';
my $ACCELERATOR_CONSTRAINT = "Valid only when removing $gHostRoleAcceleratorWorker/standby hosts or roles";

sub new {
    my $self = shift()->SUPER::new (@_);
    my $order = 0;

    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
    $self->{params} = {
        'Target' => { 'hidden' => 1, 'opt' => 'sapmnt', 'set_interactive' => 0, 'order' => $order++, 'type' => 'path'}, # just for internal use
        'SkipHostagentCalls'  => $self->getParamSkipHostagentCalls($order++, $ini_section_server),
        'RemoteExecution'     => $self->getParamRemoteExecution ($order++, $ini_section_server),
        'UseHttp'             => $self->getParamUseHttp         ($order++, $ini_section_server),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $ini_section_server,undef,$tenantCredentialsConstraint),
        'SelectedComponents'  => LCM::Configuration::GenericStackConfiguration::getParamSelectedComponents($order++, $ini_section_general, 1),
        'ForceRemoveHosts'    => $self->getParamForce           ($order++, $ini_section_server),
        'SID'                 => $self->getParamSID($order++, $ini_section_server),
        ($isWin ?   () :
                    ('InstallSSHKey'     => $self->getParamInstallSSHKey($order++, $ini_section_server),
                     'HostagentPassword' => $self->getParamHostagentPassword ($order++, $ini_section_server),
                     'Password'          => $self->getParamPassword( $order++, $ini_section_server ),
                     'RootUser'          => $self->getParamRootUser     ($order++, $ini_section_server),
                     'RootPassword'      => $self->getParamRootPassword ($order++, $ini_section_server),
                    )
        ),
        'AcceleratorUser'     => $self->getParamAcceleratorUser($order++, $ini_section_server, $ACCELERATOR_CONSTRAINT),
        'AcceleratorPassword' => $self->getParamAcceleratorPassword($order++, $ini_section_server, $ACCELERATOR_CONSTRAINT, 'passwd'),
        'SystemUser'          => $self->getParamSystemUser($order++, $ini_section_server),
        'SQLSysPasswd'        => $self->getParamSQLSysPasswd($order++, $ini_section_server, 'passwd', 0),
        'KeepXsUsers'         => $self->getParamKeepXsUsers($order++, $ini_section_general, $gXSParametersRemoveHostOrRolesConstraint),
        ($isWin ?   () :
                    (
                    'SkipModifySudoers'  => $self->getParamSkipModifySudoers($order++, $ini_section_general),
                    'KeepUser'           => $self->getParamKeepUser         ($order++, $REMOVE_HOSTS_SECTION),
                    'KeepUserHomeDir'    => $self->getParamKeepUserHomeDir  ($order++, $REMOVE_HOSTS_SECTION),
                    )
        ),
        'RemoveHosts'         => GetParamRemoveHosts($order++, $ini_section_general),
        'RemoveRoles'         => getParamRemoveHostRoles($self, $order++, $ini_section_general),
        'TenantUser'     => $self->getParamTenantUser    ($order++, $ini_section_server,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $ini_section_server,1,$tenantCredentialsConstraint),
    };

    $self->setHidden('RemoveRoles', 1);
    $self->setHidden('RemoveHosts', 1);
    $self->setSkip('SystemUser', 1);
    $self->setSkip('SQLSysPasswd', 1);
    $self->setSkip('Password', 1);
    $self->setSkip('KeepUser', 1);
    $self->setSkip('KeepUserHomeDir', 1);
    $self->setType('HostagentPassword', 'passwd');
    $self->setMandatory('SelectedComponents', 1);
    $self->setHidden('SID', 1);
    $self->{params}->{SID}->{set_interactive} = 0;
    $self->{params}->{ForceRemoveHosts}->{set_interactive} = 1;
    $self->{params}->{ForceRemoveHosts}->{str} = 'Remove dedicated component hosts and roles';
    $self->{params}->{ForceRemoveHosts}->{console_omit_word_Enter} = 1;

    if (exists $self->{params}->{RootUser}) {
        $self->{params}->{RootUser}->{init_with_default} = 1;
        $self->{params}->{RootUser}->{set_interactive}   = 1;
    }
    $order = $self->addPassThroughComponentParams($self->{params}, $order, new SDB::Install::Configuration::ServerUninstParams, undef, $ini_section_server, 'hdbuninst');
    $self->addListeners();
    return $self;
}

# Override
sub getLocalHanaHost {
    my $self = shift();
    my $hdbInstance = $self->getOwnInstance();
    my $hostName = defined($hdbInstance) ? $hdbInstance->get_host() : undef;

    return defined($hostName) ? $hostName : $self->SUPER::getLocalHanaHost(@_);
}

sub shouldRemoveXsControllerHostRolesSeparately {
	my ($self) = @_;
	my $ownInstance = $self->getOwnInstance();
	my $localHost = $ownInstance->get_host();
	my $xsControllerHost = $ownInstance->getXsControllerHostname();

	return 0 if(!defined($xsControllerHost));
	return 0 if($xsControllerHost eq $localHost);
    return 0 if($self->getNumberOfXsRoles() == 1);
	return IsRemovingRoleFromHost($self, $xsControllerHost, $gHostRoleXS2Worker);
}

sub getNumberOfXsRoles {
    my ($self) = @_;
    my $ownInstance = $self->getOwnInstance();
    return grep {$_ =~ /$gHostRoleXS2Worker|$gHostRoleXS2Standby/} values %{$ownInstance->getHostRolesInfo()};
}

# Override - disable the chance of someone setting value to RemoveHosts from the outside
sub setRemoveHosts {
	return 1;
}

# Override - disable the chance of someone setting value to RemoveRoles from the outside
sub setRemoveRoles {
	return 1;
}

sub checkRootUser {
    my ($self, $remoteRootUser) = @_;

    if(defined $self->{_remoteHosts}){
        $self->{_remoteHosts}->destroy();
        $self->{_remoteHosts}->connect()
    }

    my $returnCode = $self->SUPER::checkRootUser($remoteRootUser);        
    if($returnCode && defined($self->{_remoteHosts}) && !$self->{_remoteHosts}->isHostctrl()){
        my $isAuthenticationOk = $self->{_remoteHosts}->isAuthenticationOk();
        $self->setSkip('RootPassword', $isAuthenticationOk);
        $self->setMandatory('RootPassword', !$isAuthenticationOk);
    }
    
    return $returnCode;
}

sub checkPassword {
    my ($self, $password) = @_;

    if(!defined $password){
        $self->appendErrorMessage ("Password is not defined.");
        return 0;
    }elsif(length($password) == 0){
        $self->appendErrorMessage ("Password name cannot be empty.");
        return 0;
    } else {
        require LCM::Configuration::NewDBUser::User;
        my $sidadmUserName = lc($self->getSID()) . 'adm';
        my $sidadmUser = new LCM::Configuration::NewDBUser::User($sidadmUserName);
        $sidadmUser->setMsgLstContext($self->getMsgLstContext());
        my $rc = $sidadmUser->verifyPassword($password, $self);
        if (!defined $rc){
            $self->getMsgLst()->appendMsgLst ($sidadmUser->getErrMsgLst());
            $self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
            return 1;
        }
# Fix for bug 72608. Prevent doubled 'Unknown user password combination' message. The other message comes from 'LCM::Configuration::NewDBUser::User;verifyPasswordSHA'.
        if($rc != 1 && !isSidadmin($self->getSID())) {
            $self->appendErrorMessage ('Unknown user password combination');
        }
        return $rc;
    }
}

sub checkHostagentPassword {
    my ($self, $value) = @_;

    if (!$value || length ($value) == 0) {
        $self->PushError("$self->{params}->{HostagentPassword}->{str} must not be empty.");
        return 0;
    }

    if (!$self->{isSlave} || !$self->isCheckOnly()) {
        my $userName = $self->getValue('HostagentUserName') || 'sapadm';

        require LCM::Configuration::SapAdmUser;
        my $user = new LCM::Configuration::SapAdmUser($userName);

        if ($user->exists()) {
            my $rc = $user->verifyPassword($value, $self);
            if (!defined $rc){
                $self->getMsgLst()->appendMsgLst ($user->getErrMsgLst());
                $self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
                return 1;
            }
            if (!$rc){
                $self->PushError ($self->{params}->{HostagentPassword}->{str} .
                    ': Unknown user password combination');
                return 0;
            }
        }
    }
    return 1;
}

#
# Due to the multiple inheritance of $self, explicitly point to the "getSystemHosts"
# subruotine implementation.
#
sub getSystemHosts {
    my ($self) = shift;
    return $self->SDB::Install::Configuration::AnyConfig::getSystemHosts(@_);
}

sub InitDefaults{
    my ($self) = @_;
    $self->{installer} = new LCM::Installer();
    my $scm = $self->{installer}->getOwnSystemComponentManager();

    if (! defined($scm)){
        $self->AddError("Undefined SAP System ID");
        return undef;
    }

    my $sid = $scm->getSid();

    if (!$self->setValue ('SID', $sid)) {
        return undef;
    }

    $self->changePasswordStr(lc($sid) . 'adm');
    if(isSidadmin($sid)){
        $self->setValue('RemoteExecution', 'saphostagent');
        $self->{params}->{RemoteExecution}->{hidden} = 1;
        $self->initRemoteHosts();

        for my $parameterId ('InstallSSHKey', 'RootUser', 'RootPassword'){
            $self->setSkip($parameterId, 1);
            $self->{params}->{$parameterId}->{hidden} = 1;
        }
    }

    if(not defined $self->{componentManager}){
        $self->{componentManager} = new LCM::ComponentManager::MediumComponentManager($self);
    }
    $self->{componentManager}->setSystemComponentManager($scm);
    $scm->setMsgLstContext( [ $self->getMsgLst() ] );
    $self->{componentManager}->setMsgLstContext( [ $self->getMsgLst() ] );

    my $consoleText = "Choose components to be uninstalled for system '".$scm->getSid()."':";
    $self->{params}->{SelectedComponents}->{console_text} = $consoleText;

    my $lmstructure_component = $scm->getComponentByKeyName($gKeynameLMStructure);
    if (defined $lmstructure_component) {
        $lmstructure_component->selectComponent();
    }

    $self->initializeSelectedComponents();
    $self->{params}->{RemoveHosts}->{valid_values} = $self->getOwnInstance()->get_allhosts();
    $self->{params}->{RemoveRoles}->{origin_values} = $self->getOwnInstance()->get_allhosts();

    my $applicationContext = LCM::App::ApplicationContext::getInstance();
    if($applicationContext->getMode() eq 'CLI'){
        $self->{params}->{ForceRemoveHosts}->{str} = 'Uninstallation of components with existing hosts/roles is not allowed. Do you want to remove the hosts/roles as well';
    }
    $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SystemUser'});
    $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SQLSysPasswd'});
    $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'TenantUser'});
    $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'SQLTenantUserPassword'});
    return 1;
}

sub initializeSelectedComponents {
    my ($self) = @_;

    if (! $self->fillValidSelectedComponentValues()){
        return undef;
    }

    $self->fillSelectedComponentsDefaultValue();

    return 1;
}

sub fillValidSelectedComponentValues{
    my ($self) = @_;
    my $sid = $self->getValue('SID');
    my ($orderedKeynames, $componentKeynameToAction) = $self->getComponentKeynameToAction();
    my $selectedComponentsParam = $self->{params}->{SelectedComponents};

    $selectedComponentsParam->{valid_values} = isSidadmin($sid) ? [] : [ "all" ];
    $selectedComponentsParam->{ui_values} = isSidadmin($sid) ? [] : [ $self->getAllComponentsUiValue() ];

    for my $componentKeyname (@{$orderedKeynames}){
        my $component = $self->{componentManager}->getSystemComponentManager()->getComponentByKeyName($componentKeyname);
        if ($component->isInternal() or !$component->canSelectComponent()) {
            next;
        }
        my $componentBatchKey = $component->getComponentBatchKey();
        push(@{$selectedComponentsParam->{valid_values}}, $componentBatchKey);
        push(@{$selectedComponentsParam->{ui_values}}, $componentKeynameToAction->{$componentKeyname});
    }

    return 1;
}

sub getAllComponentsUiValue{
    my $self = shift;
    my $serverComponent = $self->{componentManager}->getSystemComponentManager()->getComponentByKeyName($gKeynameEngine);

    return sprintf("%s version %s and all other components", $serverComponent->getComponentName(), $serverComponent->getVersion());
}

sub fillSelectedComponentsDefaultValue {
    my ($self) = @_;

    if (isAdmin()){
        $self->setDefault('SelectedComponents', 'all')
    }
}

sub fillSelectedComponents{
    my ($self, $selectedComponentsCsv) = @_;

    $self->selectAllComponents(0);
    for my $componentBatchValue (split(/\s*,\s*/, $selectedComponentsCsv)) {
        if ("all" eq $componentBatchValue) {
            $self->selectAllComponents(1);
            last;
        }
        my $component = $self->getComponentByBatchKey($componentBatchValue);
        $component->selectComponent(1);
    }
    return 1;
}

sub getComponentByBatchKey {
    my ($self, $batchKey) = @_;
    return $self->getSystemComponentManager()->getComponentByBatchKey($batchKey);
}

sub setRemoteExecution {
    my $self = shift();
    my $result = $self->SUPER::setRemoteExecution(@_);

    if($result && $self->isUseSAPHostagent()){
        $self->initRemoteHosts();
    }
    return $result;
}

sub checkSID{
    my ($self, $sid) = @_;
    my $systems  = $self->getCollectSAPSystems();

    if (defined $systems) {
        my $sapSys = $systems->{$sid};
        if (defined $sapSys) {
            my $instance = $sapSys->getNewDBInstances()->[0];
            if (defined $instance && $instance->exists_remote_host() && !$self->isUseSAPHostagent() ) {
                $self->setSkip('RootUser', 0);
            }
        }
    }
    return 1;
}

sub checkSQLSysPasswd {
    my ($self,
        $value, 
        $msglst_arg,
        $paramId,
        $outSqlConnection,
        $failIfNoConnection,
        $systemUser,
        $service) = @_;
# Fail if no connection was established
    return $self->SUPER::checkSQLSysPasswd(
        $value, $msglst_arg, $paramId, $outSqlConnection, 1, $systemUser, $service);
}

sub checkSelectedComponents {
    my ($self, $selectedComponentsCsv) = @_;

    if (!$selectedComponentsCsv) {
        $self->PushError("There are no components selected for uninstallation. Select at least one component.");
        return 0;
    }

    my $rc = $self->fillSelectedComponents($selectedComponentsCsv);

    return undef if (!$rc);

    my $warnings = $self->getWarningList();
    my $systemComponentManager = $self->getSystemComponentManager();
    splice @$warnings;

    return 0 if(! $self->_checkComponentDependencies($systemComponentManager));

    if ( $systemComponentManager->isHDBServerComponentSelected() ) {
        my $wng = "All data volumes and all log volumes of $gProductNameEngine '" . $self->getSID() . "' will be removed!";
        push @$warnings, $wng;
        return $rc;
    }
    return $rc;
}

sub getIgnoreValues { return ['check_component_dependencies']; }

sub getSystemComponentManager {
    my ($self) = @_;
    return (defined $self->{componentManager})
           ? $self->{componentManager}->getSystemComponentManager()
           : undef;
}

sub getComponentKeynameToAction {
    my ($self) = @_;
    my $generator = new LCM::Utils::ComponentActionStringGenerator::Uninstall();

    return $generator->getComponentKeynameToActionMap($self, undef, $self->getSystemComponentManager());
}

sub selectAllComponents {
    my ($self, $selected) = @_;
    my $componentManager = $self->getSystemComponentManager();
    my $detectedComponents = $componentManager->getAllComponents();

    for my $component (@{$detectedComponents}) {
        next if($component->getComponentKeyName() eq $gKeynameLMStructure);
        $component->selectComponent($selected);
    }
}

sub getAction{
    return UNINSTALL_ACTION;
}

sub getProductName{
    my $flavourProductName = $_[0]->getFlavourProductName();
    return $flavourProductName. " Components";
}

sub CollectOtherHostInfos {
    # Does not collect any information of remote hosts.
    return 1;
}

sub createSummaryTree {
    my $self = shift;
    return LCM::SummaryTreeBuilder::buildSummaryTree($self, $self->{componentManager} );
}

sub getStaticDialogs {
    return [ "LCM::Gui::Dialogs::SummaryDialog", "LCM::Gui::Dialogs::UninstallDialog", "LCM::Gui::Dialogs::FinalizeDialog" ];
}

sub getConfigureDialogs {
    require LCM::Gui::Dialogs::Uninstall::SelectComponentsDialog;
    require LCM::Gui::Dialogs::Uninstall::DatabaseCredentials;
    require LCM::Gui::Dialogs::Uninstall::OSLevelCredentialsDialog;

    my ($self, $wizard) = @_;

    if($isWin){
        $self->AddMessage ( 'No distributed parameters are required. Skipping distributed parameters dialog.' );
    }

    return [
        new LCM::Gui::Dialogs::Uninstall::SelectComponentsDialog($wizard),
        $isWin ? () : (new LCM::Gui::Dialogs::Uninstall::OSLevelCredentialsDialog($wizard)),
        new LCM::Gui::Dialogs::Uninstall::DatabaseCredentials($wizard),
    ];
}

####
# Important - if changed - changes might apply to the Web::GenericStackUninstallation as well
####
sub addListeners {
    my ($self) = @_;
    $self->addParameterListener('SelectedComponents', new LCM::Configuration::ValueChangeListeners::Uninstall::SelectedComponentsListener());
    $self->addParameterListener('SelectedComponents', new LCM::Configuration::ValueChangeListeners::Uninstall::RecomputeCredentialsListener());
    $self->addParameterListener('ForceRemoveHosts', new LCM::Configuration::ValueChangeListeners::Uninstall::ForceListener());
    $self->addParameterListener('ForceRemoveHosts', new LCM::Configuration::ValueChangeListeners::Uninstall::RecomputeCredentialsListener());
    $self->addParameterListener('ForceRemoveHosts', new LCM::Configuration::ValueChangeListeners::Uninstall::NameserverConnectivityListener());
    $self->addParameterListener('RemoveRoles',$self->getAutoInitFamilyServicesParameterHanlder('RemoveRoles'));
    $self->addParameterListener('RemoveHosts',$self->getAutoInitFamilyServicesParameterHanlder('RemoveHosts'));
    $self->addParameterListener('TenantUser',SDB::Install::Configuration::TenantUserListener->new());
}

sub getTimeoutValues {
    return [ 'start_instance', 'start_service' ];
}

sub _shouldIgnoreComponentDependencies {
    my ( $self ) = @_;
    my $optionIgnore = $self->getOptionIgnore();

    return defined($optionIgnore) && $optionIgnore->getArg('check_component_dependencies');
}

sub _checkComponentDependencies {
    my ($self, $componentManager) = @_;
    my @allComponents = @{$componentManager->getAllComponents()};
    my @selectedComponents = @{$componentManager->getSelectedComponents()};

# There is no need to check anything if all components are selected for uninstall
    if(scalar(@allComponents) == scalar(@selectedComponents)){
        return 1;
    }

    my $ignoreDependenciesNotMet = $self->_shouldIgnoreComponentDependencies();
    my $warnings = $self->getWarningList();
    my $returnCode = 1;

    for my $component (@selectedComponents){
        my $dependantComponents = $componentManager->getDependantComponents($component);
        my $hasFailingDependencies = grep { ! $_->isComponentSelected() } @{$dependantComponents};

        next if( ! $hasFailingDependencies );

        if(! $ignoreDependenciesNotMet){
            $self->PushError($self->_getComponentDependeciesErrorMessage($component, $dependantComponents));
            $returnCode = 0;
            next;
        }
        push(@{$warnings}, $self->_getComponentDependeciesWarningMessage($component, $dependantComponents));
    }
    return $returnCode;
}

sub _getComponentDependeciesWarningMessage {
    my ($self, $component, $dependantComponents) = @_;
    my $messageTemplate = "An already installed component%s %s depend%s on component '%s'";
    my $componentSuffix = scalar(@{$dependantComponents}) > 1 ? 's' : '';
    my $dependSuffix = scalar(@{$dependantComponents}) > 1 ? '' : 's';
    my @dependantComponentNames = map { "'" . $_->getComponentName() . "'" } @{$dependantComponents};
    my $dependantComponentsString = join(', ', @dependantComponentNames);
    return sprintf($messageTemplate, $componentSuffix, $dependantComponentsString, $dependSuffix, $component->getComponentName(), );
}

sub _getComponentDependeciesErrorMessage {
    my ($self, $component, $dependantComponents) = @_;
    my $messageTemplate = "Cannot uninstall component '%s'. An already installed component%s %s depend%s on it";
    my $componentSuffix = scalar(@{$dependantComponents}) > 1 ? 's' : '';
    my $dependSuffix = scalar(@{$dependantComponents}) > 1 ? '' : 's';
    my @dependantComponentNames = map { "'" . $_->getComponentName() . "'" } @{$dependantComponents};
    my $dependantComponentsString = join(', ', @dependantComponentNames);
    return sprintf($messageTemplate, $component->getComponentName(), $componentSuffix, $dependantComponentsString, $dependSuffix);
}

sub _recalculateRolesForRemoval {
    my ($self) = @_;
    my $trexInstance = $self->getOwnInstance();
    my $hostRolesInfo = $trexInstance->getHostRolesInfo();
    my $hostRoleRemovalMap = $self->_getHostRoleRemovalMap($hostRolesInfo);
    my %removeRolesMap = ();

    for my $host (keys(%{$hostRolesInfo})){
        my @allRoles = split('\s+', $hostRolesInfo->{$host});
        my $numberOfAllRoles = scalar(@allRoles);
        my $numberOfRolesForRemoval = scalar(@{$hostRoleRemovalMap->{$host}});

        if($numberOfRolesForRemoval > 0 && $numberOfRolesForRemoval < $numberOfAllRoles){
            $removeRolesMap{$host} = join(',', @{$hostRoleRemovalMap->{$host}});
        }
    }

    my $componentManager = $self->getSystemComponentManager();
    my $isServerSelected = $componentManager->isComponentSelected($gKeynameEngine);
    my $shallSkipRemoveRoles = !$isServerSelected && scalar(keys(%removeRolesMap)) > 0 ? 0 : 1;

    $self->setSkip('RemoveRoles', $shallSkipRemoveRoles);
    if (!$shallSkipRemoveRoles) {
        $self->{params}->{RemoveRoles}->{value} = \%removeRolesMap;
        while (my ($host, $roles) = each %removeRolesMap) {
            $self->notifyParameterListeners('RemoveRoles', $roles, $host);
        }
    }
}

sub _recalculateHostsForRemoval {
    my ($self) = @_;
    my $trexInstance = $self->getOwnInstance();
    my $hostRolesInfo = $trexInstance->getHostRolesInfo();
    my $hostRoleRemovalMap = $self->_getHostRoleRemovalMap($hostRolesInfo);
    my @hostsToRemove = ();

    for my $host (keys(%{$hostRolesInfo})){
        my @allRoles = split('\s+', $hostRolesInfo->{$host});
        my $numberOfAllRoles = scalar(@allRoles);
        my $numberOfRolesForRemoval = scalar(@{$hostRoleRemovalMap->{$host}});

        if($numberOfRolesForRemoval == $numberOfAllRoles){
            push(@hostsToRemove, $host);
        }
    }

    my $componentManager = $self->getSystemComponentManager();
    my $isServerSelected = $componentManager->isComponentSelected($gKeynameEngine);
    my $shallSkipRemoveHosts = !$isServerSelected && scalar(@hostsToRemove) > 0 ? 0 : 1;

    $self->setSkip('RemoveHosts', $shallSkipRemoveHosts);
    if (!$shallSkipRemoveHosts) {
        my $csvValue = join(',', @hostsToRemove);
        $self->{params}->{RemoveHosts}->{value} = $csvValue;
        $self->notifyParameterListeners('RemoveHosts', $csvValue);
    }
}

sub _getHostRoleRemovalMap {
    my ($self, $hostRolesInfo) = @_;
    my $componentManager = $self->getSystemComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();
    my %hostRolesForRemovalMap = map { $_ => [] } keys(%{$hostRolesInfo});

    for my $component (@{$selectedComponents}) {
        next if($component->isServer());

        my $componentRoles = $component->getHostRoles();
        for my $host (keys(%{$hostRolesInfo})){
            my @allRoles = split('\s+', $hostRolesInfo->{$host});
            for my $role (@{$componentRoles}){
                if($role ~~ @allRoles){
                    push(@{$hostRolesForRemovalMap{$host}}, $role);
                }
            }
        }
    }
    return \%hostRolesForRemovalMap;
}

sub isAutoInitializeServicesApplicableParameterCallback{
    my($self,$paramId,$valKey,$value) = @_;
    if ($paramId eq 'RemoveHosts') {
        return $self->isRemoveHostsValueApplicableForAutoInitialize($paramId, $valKey, $value);
    }
    if ($paramId eq 'RemoveRoles') {
        return $self->isAddRolesValueApplicableForAutoInitialize($paramId, $valKey, $value);
    }
    return 0;
}

# Add the 'abstract' methods from GenericStackConfiguration in order to avoid unhandled exceptions
sub getPersistedSteps {
    return undef;
}

sub setPersistedSteps {}

sub _shouldSkipHostagentPassword {}

sub getAutoAssignXs2RolesMap {
    return {};
}

1;
