package LCM::Configuration::HdbLcmExternalActionConfiguration;

use strict;
use Cwd qw(realpath);
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::Configuration::GenericStackAny;
use SDB::Install::SysVars;
use LCM::ComponentManager::MediumComponentManager;
use LCM::Installer;
use LCM::Configuration::ParametersCreator;
use LCM::Configuration::ValueChangeListeners::ComponentListener;
use LCM::ComponentManager::ComponentScanner;
use LCM::DevelopmentTrace;
use LCM::Utils::CommonUtils qw(findAutoConfiguration);
use LCM::PersistencyUtils qw ( existsHdblcmPendingUpdate
							   existsHdblcmPendingInstall
							   existsPendingComponentsUpdate
							   getAllPendingInstallInfo
							   getAllPendingUpdateInfo );
use SDB::Install::SAPSystem qw (@STEP_NAMES getPendingInfoFromPersFile);
use LCM::Configuration::GenericStackUpdateConfiguration qw ( getInstalledSystemsValidUIValues);
use LCM::Configuration::ValueChangeListeners::PrepareExecutionEnvironmentListener;
use SDB::Install::Globals qw($gProductNameCockpit $gSAPLocation $gKeynameLSS);
use LCM::Configuration::PersistenceManager;
use LCM::FileUtils;
use experimental qw (smartmatch);

use base qw (LCM::Configuration::ActionConfiguration Exporter);

sub new {
	my ( $class, $options, $configfile, $batchMode ) = @_;

	if(!defined($configfile)) {
		$configfile = findAutoConfiguration();
	}

	require SDB::Install::Configuration::AnyConfig;

	my $self  = $class->SUPER::new( $options, $configfile );
	my $order = 0;

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
	$self->{params} = {
		'BootstrapFromResident'    => LCM::Configuration::ParametersCreator::GetParamBootstrapFromResident($order++, $ini_section_general),
		'UseSimpleUI' => GetParamUseSimpleUI($order++, $ini_section_general),
		'DvdPath' => {
			'order'                  => $order++,
			'opt'                    => 'component_medium',
			'alias_opts'             => ['dvdpath'],
			'type'                   => 'path',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 0,
			'str'                    => "Location of $flavourProductName Installation Medium",
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 0
		},
		'ComponentDirs' => {
			'order'                  => $order++,
			'opt'                    => 'component_dirs',
			'opt_arg'                => '<path1>[,<path2>]...',
			'type'                   => 'string',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 0,
			'str'                    => 'Comma separated list of component directories',
			'desc'                   => 'Component directories',
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 0
		},
		'ComponentFsRoot' => {
			'order'                  => $order++,
			'opt'                    => 'component_root',
			'opt_arg'                => '<path>',
			'type'                   => 'path',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 0,
			'str'                    => 'Directory root to search for components',
			'desc'                   => 'Directory including sub-directories to search for components',
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 0
		},
		 'DefaultLocation' => {
            'order'                  => $order++,
            'opt_arg'                => '<path1>[,<path2>]...',
            'type'                   => 'string',
            'section'                => $ini_section_general,
            'value'                  => undef,
            'default'                => undef,
            'set_interactive'        => 0,
            'str'                    => 'Comma separated list of component directories',
            'desc'                   => 'Default detected components',
            'hidden'                 => 1,
            'leaveEmptyInConfigDump' => 1,
            'mandatory'              => 0
        },
		'Action' => LCM::Configuration::ActionConfiguration::getParamAction( $order++ ),
		'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $ini_section_general),
		(
            !$isWin
            ? (
                 'RemoteExecution' => $self->getParamRemoteExecution($order++, $ini_section_general, 0),
                 'UseHttp'         => $self->getParamUseHttp        ($order++, $ini_section_general),
              )
            : () 
        ),
	};
	$self->{params}->{'Action'}->{set_interactive} = 0;
	
	$self->setValidActions( [ 'install', 'update', 'extract_components'] );

	if ( !$batchMode ) {
		my $action_sid = SDB::Install::Configuration::AnyConfig::getParamSID( $self, $order++, undef );
		$action_sid->{'excludeFromConfigDump'} = 1;
		$action_sid->{'set_interactive'}       = 0;
		$action_sid->{'hidden'}                = 1;
		#fix read from config file
		$action_sid->{'section'}                = $ini_section_server;
		$self->{params}->{'Action_SID'} = $action_sid;

		$self->{params}->{'SelectedSystem'} = $self->getParamSelectedSystem( $order++, undef );
	}
	$self->setHidden('UseSimpleUI', 1);
    $self->addListeners();
	return $self;
}

sub setActionBatchValue {
	my ($self, ) = @_;
	$self->{params}->{'Action'}->{batchValue} = 'install';
}

sub getAction {
	my ($self, $batchMode) = @_;

	my $action = $self->SUPER::getAction();

	if ( (!$batchMode) and ($action eq 'update') ) {
		my $sid = $self->getValue('Action_SID');
		if ( not defined $sid ) {
			return undef;
		}
	}
   
   return $action;
}

sub getSID {
    return $_[0]->getValue('Action_SID')
}

sub checkSelectedSystem {
	my ( $self, $selectedSystem ) = @_;

	if ( $selectedSystem eq "install" ) {
		$self->setValue( 'Action', 'install' );
	} elsif ( $selectedSystem eq "extract_components") {
        $self->setValue('Action', 'extract_components');
    } elsif ( $selectedSystem eq "Exit (do nothing)" ) {
		LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
		exit;
	} else {
		if ( $selectedSystem =~ /pending installation/ ) {
			$self->setValue( 'Action', 'install' );
			$self->setPendingInstallation(1);
			
		} else {
			$self->setValue( 'Action', 'update' );
		}
		my ($sid) = $selectedSystem =~ /^(\w{3})/;
		$self->setValue( 'Action_SID', $sid );
	}
}

sub setPendingServerInstallations {
	my ($self, $pendingServerInstallations) = @_;
    $self->{_pendingServerInstallations} = $pendingServerInstallations;
}

sub isPendingServerInstallation {
	my ($self) = @_;
	my $sid = $self->getSID();
	return undef if ! defined $sid;

	return grep( /^$sid/, split(',', $self->{_pendingServerInstallations}) );
}

sub initDefaultSystems {
	my ($self) = @_;

	my @valid_values = ();
	my @ui_values    = ();
	my $updateValidValues = [];
	my $updateUIValues = [];

	my $curFlavour = LCM::App::ApplicationContext::getInstance()->getApplication()->getFlavour(1);
	my $updConfig = new LCM::Configuration::GenericStackUpdateConfiguration();
	($updateValidValues, $updateUIValues) = $updConfig->getInstalledSystemsValidUIValues( $curFlavour );
	push( @valid_values, @$updateValidValues );
	push( @ui_values, @$updateUIValues );

	my $selectedSystemParam = $self->{params}->{'SelectedSystem'};
	if ( !defined $self->getValue('Action') || $self->getValue('Action') ne 'update' ) {
        my ($pendingInstallations, $pendingServerInstallations) = $self->getHdbLcmPendingInstallations();
        $self->setPendingServerInstallations(join(',', map { $_->get_sid() } @{$pendingServerInstallations}));

        foreach my $currPending (@$pendingInstallations) {
            my $systemFlavour = $currPending->getHANAFlavour();
            if(defined($curFlavour) && defined($systemFlavour)){
                next if($curFlavour ne $systemFlavour);
            }

            my $currSid = $currPending->get_sid();
            my $instconfig = new LCM::Configuration::GenericStackInstallConfiguration();
            $instconfig->{sapSys}                    = $currPending;
            $instconfig->{current_sid}               = $currSid;
            $instconfig->{params}->{SID}->{value}    = $currSid;
            $instconfig->{params}->{Target}->{value} = $currPending->get_target();
            my $info = $instconfig->getProductInfo($currPending, $systemFlavour, 1);
            my $instPendingInfo = getAllPendingInstallInfo($instconfig, 0, $instconfig->getMsgLst());
            $info .= "\n";
            $info .= "$instPendingInfo" if($instPendingInfo);
            my $pendingInfo = $info. "\n";
            push( @valid_values, $currSid . ' (resume pending installation)' );
            push( @ui_values, $pendingInfo. "\n" );
        }

        my $pendingLSSInstSids = $self->getLSSPendingInstallations();
        foreach my $sid(@$pendingLSSInstSids){
            my $instconfig = new LCM::Configuration::GenericStackInstallConfiguration();
            $instconfig->{current_sid} = $sid;
            $instconfig->setValue('SID', $sid);
            my $instPendingInfo = getAllPendingInstallInfo($instconfig, 0, $instconfig->getMsgLst());
            my $info .= "\n";
            $info .= "$instPendingInfo" if($instPendingInfo);
            my $pendingInfo = $info. "\n";
            push( @valid_values, $sid . ' (resume pending installation)' );
            push( @ui_values, $pendingInfo. "\n" );
        }
        push( @valid_values, "install" );
        push( @ui_values, "Install new system" );
    }

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
	if (!$self->isSimpleMode() && $flavourProductName ne $gProductNameCockpit) {
	    my $extractComponentsValue = "extract_components";
	    push(@valid_values, $extractComponentsValue);
    	push(@ui_values, 'Extract components');
	}
	my $exitValue = $self->getDefault('SelectedSystem');
	push( @valid_values, $exitValue );
	push( @ui_values, '' );

	$selectedSystemParam->{'ui_values'}        = \@ui_values;
	$selectedSystemParam->{'valid_values'}     = \@valid_values;
	
	$self->{scanner}->printScannedComponents() if (!$self->isSimpleMode());
}

sub getHdbLcmPendingInstallations {
    my ($self) = @_;
    my @pendingHdbLcmInstallations = ();
    my @pendingServerInstallations = ();

    require SDB::Install::Configuration::NewDB;
    my $newDBConfig = new SDB::Install::Configuration::NewDB();
    my $allSystems = $newDBConfig->CollectSAPSystems();
    foreach my $currSID (sort keys %{$allSystems}) {
        my $currSystem = $allSystems->{$currSID};
        $newDBConfig->{current_sid} = $currSID;
        $newDBConfig->{params}->{Target}->{value} = $currSystem->get_target();
        
        if ( $newDBConfig->pers_exists() && existsHdblcmPendingInstall( $currSID, $currSystem->get_target() )) { # Server pending installation
            push(@pendingServerInstallations, $currSystem);
            push(@pendingHdbLcmInstallations, $currSystem);
            next;
        }

        next if ! existsHdblcmPendingInstall( $currSID, $currSystem->get_target() ) ||
                ! existsPendingComponentsUpdate( $currSID );

        # Component pending installation
        push(@pendingHdbLcmInstallations, $currSystem);
    }

    return (\@pendingHdbLcmInstallations, \@pendingServerInstallations);
}

sub isSimpleMode {
	my ($self) = @_;
	return !$self->isSkipped('UseSimpleUI') && $self->getBatchValue('UseSimpleUI') =~ /$bool_true_pattern/;
}

sub getParamSelectedSystem {
	my ( $self, $order, $section ) = @_;
	my $param = LCM::Configuration::ParametersCreator::GetParamSelectedSystem( $order, $section );
    
    $param->{'f_init_interactive_values'} = sub {
            $self->initDefaultSystems();
        };

    return $param;
}

sub CheckParams {
    my ($self, $batchMode) = @_;
    $self->analyzeMedium();

    return $self->SUPER::CheckParams($batchMode);
}

sub analyzeMedium {
	my ($self, ) = @_;
	require LCM::MediumAnalyzer;
	my $analyzer = new LCM::MediumAnalyzer($self);
	return $analyzer->analyze();
}

sub checkDvdPath {
	my ( $self, $dvdPath ) = @_;
    return $self->checkComponentSourcesCompatibility();
}

sub checkComponentFsRoot {
	my ( $self, $componentFsRoot ) = @_;
    return $self->checkComponentSourcesCompatibility();
}

sub checkComponentSourcesCompatibility {
	my ( $self, $componentFsRoot ) = @_;
	if ( $self->getValue('ComponentFsRoot') and $self->getValue('DvdPath') ) {
		$self->appendErrorMessage("Simultaneous use of component_medium and component_root is forbidden");
		return 0;
	}
	return 1;
}

sub getComponentManager {
	my ($self) = @_;
	if ( not defined $self->{componentManager} ) {
		$self->{componentManager} = new LCM::ComponentManager::MediumComponentManager($self);
	}
	return $self->{componentManager};
}

sub addListeners(){
    my ($self) = @_;
    $self->{scanner} = new  LCM::ComponentManager::ComponentScanner($self->getComponentManager(),$self,0,0);

    $self->addParameterListener( 'ComponentDirs', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->{scanner}, 'ComponentDirs'));
    $self->addParameterListener( 'DvdPath', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->{scanner},'DvdPath'));
    $self->addParameterListener( 'ComponentFsRoot', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->{scanner},'ComponentFsRoot'));
    $self->addParameterListener( 'DefaultLocation', LCM::Configuration::ValueChangeListeners::ComponentListener->new($self->{scanner},'DefaultLocation'));
    $self->addParameterListener( 'Action_SID', LCM::Configuration::ValueChangeListeners::PrepareExecutionEnvironmentListener->new());
}

sub printActionHeaderMessage{
	my($self) = @_;
	$self->{scanner}->printScannedComponents() if (!$self->isSimpleMode());
}

sub getLSSPendingInstallations {
    my ($self) = @_;
    my $componentName = lc($gKeynameLSS);
    my $hdblcmPersFilePattern = '^[A-Z][A-Z0-9][A-Z0-9]\.hdblcm\.install$';
    my $lssPersFilePattern = 'pending_'.$componentName.'\.xml$';
    my $msglst = $self->getMsgLst();
    if (!-d $gSAPLocation){
        return [];
    }
    my $fileList = listDirectory($gSAPLocation);
    my @hdblcmPersistencyFiles = grep {/$hdblcmPersFilePattern/ && -f "$gSAPLocation/$_"} @$fileList;
    my @lssPendingInstallations = grep {/$lssPersFilePattern/ && -f "$gSAPLocation/$_"} @$fileList;
    my $sids = $self->getLSSPendingInstallSids(\@lssPendingInstallations);
    return $sids;
}

sub getLSSPendingInstallSids {
    my ($self, $persistencyFiles ) = @_;
    my @sids;
    if (defined $persistencyFiles){
        foreach my $persFile (@{$persistencyFiles}){
            my $sid = substr $persFile, 0, 3;
            push @sids, $sid;
        }
    }
    return \@sids;
}

sub getParamSkipHostagentCalls;
* getParamSkipHostagentCalls =
        \&SDB::Install::Configuration::AnyConfig::getParamSkipHostagentCalls;

sub getParamRemoteExecution;
* getParamRemoteExecution =
        \&SDB::Install::Configuration::AnyConfig::getParamRemoteExecution;

sub getParamUseHttp;
* getParamUseHttp =
        \&SDB::Install::Configuration::AnyConfig::getParamUseHttp;

1;
