package LCM::Configuration::Hosts::AddHosts::AddHostConfiguration;

use strict;
use SDB::Install::Configuration::NewServerConfig;
use LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration;
use SDB::Install::SysVars qw ($isWin $path_separator);
use SDB::Install::Globals qw ($gHostRoleAcceleratorStandby
                              $gHostRoleAcceleratorWorker
                              $gPlatform);
use SDB::Install::System qw(readLink);

require SDB::Install::Configuration::AddHost;
                               
use base qw( SDB::Install::Configuration::NewServerConfig LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration);

sub new {
    my $self = shift()->SUPER::new (@_);
    my $order = 0;
    my $section = $LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration::ini_section_addhosts;

    $self->{params} = {
        'InstallHostagent' => $self->getParamInstallHostagent ($order++, $section, undef, 1),
        'SID'       => $self->getParamSID      ($order++, $section),
        'HostName'  => $self->getParamHostName ($order++, $section),
        $isWin ? (
            'Drive' => $self->getParamDrive ($order++, $section)
        ) : (),
        'Target' => $self->getParamTarget ($order++, $section),
        'HostRoles'              => $self->getParamHostRoles             ($order++, $section),
        'HostFailoverGroup'      => $self->getParamHostFailoverGroup     ($order++, $section),
        'StoragePartitionNumber' => $self->getParamStoragePartitionNumber($order++, $section),
        'WorkerGroup'            => $self->getParamWorkerGroup           ($order++, $section),
        'Password' => $self->getParamPassword($order++, $section),
        'NoStart' => $self->getParamNoStart ($order++, $section, 'Do not start the instance after installation', 'Does not start the instance after installation', 0),
        ($isWin  ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section)) : ()),
        'HostagentPassword' => $self->getParamHostagentPassword($order++, $section),
        'ImportContentXS2' => $self->getParamImportContentXS2($order++, $section),
        'SystemUser'          => $self->getParamSystemUser         ($order++, $section, 1),
        'SQLSysPasswd'        => $self->getParamSQLSysPasswd       ($order++, $section, 'passwd', 1),
        'AcceleratorUser'     => $self->getParamAcceleratorUser    ($order++, $section),
        'AcceleratorPassword' => $self->getParamAcceleratorPassword($order++, $section),
    };
    $self->setDefault('InstallHostagent', 1);

    return $self;
}

sub getNumberOfExpectedOutputLines {
    return 50;
}

sub checkWorkerGroup {
    my ($self, $workerGroupCSV) = @_;
    my @workerGroups = split(',', $workerGroupCSV);

    for my $group (@workerGroups){
        return undef if(!$self->SUPER::checkWorkerGroup($group));
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Checks the host roles and enables the system user and accelerator user.

sub checkHostRoles {
    my ($self, $roleList) = @_;

	foreach my $currRole (split (',', $roleList)) {
		if (!$self->checkRole($currRole)) {
			return 0;
		}
	}

	my $shouldSkipAcceleratorParameters = 1;
    foreach my $currRole (split (',', $roleList)) {
        if ($currRole =~ /$gHostRoleAcceleratorStandby|$gHostRoleAcceleratorWorker/){
			$shouldSkipAcceleratorParameters = 0;
        	last;
        }
    }
    
    $self->setSkip('SystemUser', $shouldSkipAcceleratorParameters);
	$self->setSkip('SQLSysPasswd', $shouldSkipAcceleratorParameters);
	$self->setSkip('AcceleratorUser', $shouldSkipAcceleratorParameters);
	$self->setSkip('AcceleratorPassword', $shouldSkipAcceleratorParameters);
    return 1;
}


#-------------------------------------------------------------------------------
# Checks if the specified SID and the system administrator user
#
# Parameter: string $wantedSID
#
# Returns int retCode

sub checkSID{
    my ($self, $wantedSID) = @_;

    my $properties = $self->{params}->{SID};

    if (!$self->checkAdditionalSID($wantedSID, $properties->{str})) {
        return 0;
    }

    my $user = new SDB::Install::NewDBUser ($wantedSID);
    my $userName = $user->getSidAdmName ();
    if ($user->exists()){
        $self->AddProgressMessage ("The defined user '$userName' already exists on the system. Neither the password, nor any other attribute of the user will be changed.");
        $self->AddProgressMessage ('Verify that the user is correct.');
    }

    $self->changePasswordStr($userName);
    $self->enableHostagentPasswordIfRequired();
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the path consisting of the given target directory and SID
#
# Parameter: string $targetDirectory
#
# Returns int retCode

sub checkTarget{
    my ($self, $targetDirectory) = @_;
    my $sid = $self->{params}->{SID}->{value};
    my $path = $targetDirectory . $path_separator . $sid;

    if (!-d $path){
        $self->PushError ("Directory '$path' doesn't exist");
        return 0;
    }

	$self->{params}->{Target}->{no_retry} = 1;

	my $globalIni = $self->_getGlobalIni($path);

	if (!defined $globalIni) {
		return undef;
	}

	my $listenInterface = $globalIni->getValue ('communication','listeninterface');
	my $internalNetworkPrefix = $globalIni->getValue ('communication','internal_network');
	my $errMsg1 = "Inter-Service communication is set to 'local'.";
	my $errMsg1Append1 = "Please reconfigure your master node:";
	my $errMsg1Append2 = "Perform 'hdblcm --action=configure_internal_network' on master node.";
	my $errMsg2 = "Inter-Service communication is set to 'internal', but internal network is unknown.";

	if (! $self->validateListenInterface($listenInterface, $internalNetworkPrefix, $errMsg1, $errMsg1Append1, $errMsg1Append2, $errMsg2)) {
		return 0;
	}

    return $self->check_path_access($targetDirectory, 0);
}


sub _getGlobalIni {
	my ($self, $path) = @_;

	if (!defined $path) {
		return undef;
	}

	my $instanceDirLink = join ($path_separator, $path, 'exe',
			$gPlatform, $isWin ? () : ('hdb'));
	my $instanceDirAbsolute = readLink ($instanceDirLink);
	my $configDir = $instanceDirAbsolute . $path_separator . 'config';
	my $hostname = $self->getValue('HostName');
	my $layeredCfg = new SDB::Install::LayeredConfig(undef, $path, $hostname, undef, $configDir);

	if ($layeredCfg->ErrorState ()){
		$self->PushError ('Error initializing layered configuration', $layeredCfg);
		return undef;
	}

	$layeredCfg->resetMsgLstContext();
	my $globalIni = $layeredCfg->getIniFile ('global.ini');

	if (!defined $globalIni){
		$self->PushError ('Cannot get global.ini', $layeredCfg);
		return undef;
	}

	$globalIni->resetMsgLstContext();

	if (!defined $globalIni->readValues ()){
		$self->PushError ('Cannot read global.ini', $globalIni);
		return undef;
	}

	return $globalIni;
}


#-------------------------------------------------------------------------------
# Checks the specified sidadm password
#
# Parameter: string $sidadmPassword
#
# Returns int retCode

sub checkPassword{
    my ($self, $sidadmPassword) = @_;
    if (!$sidadmPassword){
        $self->PushError ("Empty system administrator password not allowed");
        return 0;
    }
    return 1;
}


#===============================================================================
# import subroutines

sub checkStoragePartitionNumber;
* checkStoragePartitionNumber =
        \&SDB::Install::Configuration::AddHost::checkStoragePartitionNumber;

sub getParamHostFailoverGroup;
* getParamHostFailoverGroup =
        \&SDB::Install::Configuration::AddHost::getParamHostFailoverGroup;

sub getParamHostRoles;
* getParamHostRoles = \&SDB::Install::Configuration::AddHost::getParamHostRoles;

sub getParamStoragePartitionNumber;
* getParamStoragePartitionNumber =
        \&SDB::Install::Configuration::AddHost::getParamStoragePartitionNumber;

1;
