package LCM::Configuration::InternalNetworkConfiguration;

use base qw (SDB::Install::Configuration::AnyMultiHostConfig Exporter);

use LCM::Configuration::GenericStackAny;
use Scalar::Util;
use SDB::Install::Installer;
use LCM::InternalNetworkDetector;
use SDB::Install::System qw(isSidadmin);
use strict;

our $INTERNAL_NETWORK_ACTION = "configure_internal_network";
our @EXPORT_OK = qw($INTERNAL_NETWORK_ACTION);

sub new{    
	my $class = shift;
	my $self = $class->SUPER::new (@_);

	my $isGUI = caller() eq 'LCM::Gui::App::LCMGuiAppInstalled';
	if(!$isGUI && $self->_isContinueOption()){
		require LCM::Configuration::InternalNetworkContinueConfiguration;
		return new LCM::Configuration::InternalNetworkContinueConfiguration(@_);
	}

	my $order = 0;
	my $installer = new LCM::Installer ();
	my $sid = $installer->getSid();

	@$self{qw( SID sidAdm )} = ($sid, $self->getSysAdminUserName($sid));

	my $noStartParam = $self->getParamNoStart($order++, $ini_section_general, 'Do not start the instance after reconfiguration', 'Does not start the instance after reconfiguration', 0);
	$noStartParam->{init_with_default} = 1;

	my $remoteExecutionParam = $self->getParamRemoteExecution($order++, $ini_section_general, 0);
	my $useHttpParam         = $self->getParamUseHttp        ($order++, $ini_section_general);

	if(isSidadmin()){
		$remoteExecutionParam->{default} = 'saphostagent' 
	}

	my $scopeParam = $self->getParamScope ($order++, $ini_section_general);
	$self->_initScopeParam($scopeParam);

	my $rootUserParam = $self->getParamRootUser($order++, $ini_section_general);
	$rootUserParam->{init_with_default} = 1;
	my $sidAdmPasswordParam = $self->_getParamSidAdmPassword($order++, $ini_section_general);

	my $rootPasswordParam = $self->getParamRootPassword($order++, $ini_section_general);
	my $listenInterfaceParam = $self->getParamListenInterface($order++, $ini_section_general);
	$listenInterfaceParam->{init_with_default} = 1;
	my $internalNetworkParam = $self->getParamInternalNetwork($order++, $ini_section_general, 1, undef, 1, 0, 1);
	$internalNetworkParam->{init_with_default} = 1;

	$self->{params} = {
		'NoStart'						=> $noStartParam,
		'RemoteExecution' 				=> $remoteExecutionParam,
		'UseHttp'						=> $useHttpParam,
		'Scope' 						=> $scopeParam,
		'RootUser'						=> $rootUserParam,
		'Password'						=> $sidAdmPasswordParam,
		'RootPassword'					=> $rootPasswordParam,
		'ListenInterface'				=> $listenInterfaceParam,
		'InternalNetwork'				=> $internalNetworkParam,
	};

	$self->fillInternalNetworkAndListenInterfaceValidValues();

	return $self;
}

sub getRestartRequiredMessage {
    my ($self) = @_;
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    return "The $flavourProductName System will be restarted";
}

sub setRemoteExecution {
	my ($self, $value) = @_;
	
	if(isSidadmin() && $value ne 'saphostagent'){
		my $remoteExecutionOpt = $self->getOpt('RemoteExecution');
		$self->PushError("Invalid value '$value' of option '$remoteExecutionOpt' (only 'saphostagent' allowed when tool is started as system administrator)");
        return undef;
	}
	
	my $returnCode = $self->SUPER::setRemoteExecution($value);
	
	return undef if(!$returnCode);
	
	my $currentValue = $self->getValue('RemoteExecution');
	my $isDistributedSystem = $self->isDistributedSystem();
	my $isSkipRootCredentials = ($currentValue eq 'saphostagent') || !$isDistributedSystem;
	my $isSkipSHAPassword = (!$isDistributedSystem || $currentValue ne 'saphostagent');
	
	$self->setSkip('RootUser', $isSkipRootCredentials ? 1 : 0);
	$self->setSkip('RootPassword', 1); # root password will be unskipped in AnyConfig's sub tryRemoteKeyAuthorization if needed
	
	return $returnCode;
}

sub _isContinueOption {
	my ($self) = @_;
	my $installer = new SDB::Install::Installer();
	my $sapSys = $installer->getInstallation($self->getMsgLst());
    
    return 0 if (!defined($sapSys) || ! $sapSys->isa('SDB::Install::SAPSystem'));
	return 0 unless ($sapSys->hasNewDB());
	
	my $hdbInstance = $sapSys->getNewDBInstances()->[0];
	my $hostnameDir = $hdbInstance->get_hostNameDir();
	my $persistenceFilePath = File::Spec->catfile($hostnameDir, "hdblcm_configure_internal_network");
	return 0 if (!(-f $persistenceFilePath));

	return (-f File::Spec->catfile($hostnameDir, 'hdbreg'));
}

sub _initScopeParam {
	my ($self, $scopeParam) = @_;
	$scopeParam->{init_with_default} = 1;
	$scopeParam->{skip} = 0;
	$scopeParam->{default} = 'system';
}

sub InitDefaults{
	my $self = shift();
	
	my $returnCode = $self->SUPER::InitDefaults(@_);
	return undef if !$returnCode;
	
	$self->handleInitListenInterfaceParam();
	
	my $internalNetwork = $self->getOwnInstance()->getInternalNetworkPrefix();
	if(length($internalNetwork) > 0){
		$self->setDefault('InternalNetwork', $internalNetwork);
	}
	if($self->isDistributedSystem()){
		$self->restrictListenInterfaceMultiHost();
	}
	
	return 1;
}

sub checkListenInterface {
	my ($self, $newInterface) = @_;
	$self->clearParameterWarnings('ListenInterface');
	my $rc = $self->SUPER::checkListenInterface($newInterface);
	my $instance = $self->getOwnInstance();
	my $currentInternalNetwork = $instance->getInternalNetworkPrefix();
	if ($rc && $newInterface eq 'local' && defined $currentInternalNetwork) {
		$self->addParameterWarning('ListenInterface', $self->getRestartRequiredMessage());
	}
	return $rc;
}

sub checkInternalNetwork {
	my ($self, $newInternalNetwork) = @_;
	$self->clearParameterWarnings('InternalNetwork');
	my $rc = $self->SUPER::checkInternalNetwork($newInternalNetwork);
	if ($rc && $self->mustRestartSystem($newInternalNetwork)) {
		$self->addParameterWarning('InternalNetwork', $self->getRestartRequiredMessage());
	}
	return $rc;
}

sub isNostopAvailable{
#	return $_[0]->getInternalNetworkDetector()->isNostopAvailable();
	return 0;
}

sub getWarnings {
	my ($self) = @_;
	return $self->getWarningList();
}

sub getWarningList {
	my ($self) = @_;
	my $warnings = $self->SUPER::getWarningList();
	if ($self->mustRestartSystem()) {
		push(@$warnings, $self->getRestartRequiredMessage());
	}
	return $warnings;
}

sub mustRestartSystem {
	my ($self, $targetNetworkAddress) = @_;
    my $instance = $self->getOwnInstance();
    my $existingNetPrefix = $instance->getInternalNetworkPrefix();
# In case of the WebUI this check is performed only during parameter
# validation where the actual parameter {value} is not set. That is why
# the optional 'targetNetworkAddress' argument is supported when
# this method is called from the checkInternalNetwork method.
    my $internalNetworkValue = $targetNetworkAddress || $self->getValue('InternalNetwork');
    my $targetNetPrefix = ($internalNetworkValue && $internalNetworkValue ne 'none')
    						? $internalNetworkValue
    						: undef;
    my $isKeepingNoInternalNetwork = (!defined $targetNetPrefix && !defined $existingNetPrefix) ? 1 : 0;
    my $isChangingToNewInternalNetwork = (defined $targetNetPrefix
                                        &&
                                        (!defined $existingNetPrefix ||
                                         ($targetNetPrefix ne $existingNetPrefix))) ? 1 : 0;

    my $isResettingInternalNetwork = (!defined $targetNetPrefix && defined $existingNetPrefix) ? 1 : 0;

    return !$isKeepingNoInternalNetwork && ($isChangingToNewInternalNetwork || $isResettingInternalNetwork);
}

sub setScope {
    my ($self, $value) = @_;

    return undef unless ($self->checkScope($value));

    $self->{params}->{Scope}->{value} = lc($value);
	if ($self->isUseSAPHostagent() && $value eq 'system'){
		return undef if(!$self->initRemoteHosts());
		my $remoteHosts = $self->getRemoteHosts();
        $remoteHosts->useSidadm() if (defined $remoteHosts);
    }
	return 1;
}

sub checkScope {
	my ($self, $newScope) = @_;
	
	my $isUseSHA = $self->isUseSAPHostagent();
	my $isScopeSystem = $newScope eq 'system';
	my $scopeOption   = $self->getOpt('Scope');
	
	if(isSidadmin() && !$isScopeSystem){
		$self->PushError("Invalid value '$newScope' of option '$scopeOption' (only 'system' allowed when tool is started as system administrator)");
        return undef;
	}
	
	if($isUseSHA && !$isScopeSystem){
		$self->PushError("Invalid value '$newScope' of option '$scopeOption' (only 'system' allowed when remote execution method is set to 'saphostagent')");
        return undef;
	}
	
	if ($newScope eq 'instance') {
		$self->setSkip('RootUser', 1);
		$self->setSkip('RootPassword', 1);
		return 1;
	} elsif ($newScope eq 'system') {
		$self->ensureRootCredentialsIfNeeded();
		return 1;
	} else {
		$self->PushError("Invalid value '$newScope' of option '--scope' (only 'instance' or 'system' allowed)");
		return undef;
	}
}

sub ensureRootCredentialsIfNeeded {
       my $self = shift;
       return 1 if($self->isUseSAPHostagent());
       return $self->SUPER::ensureRootCredentialsIfNeeded(@_);
}

sub checkRootUser {
    my ($self, $remoteRootUser) = @_;
	
	if(defined $self->{_remoteHosts}){
		$self->{_remoteHosts}->destroy();
		$self->{_remoteHosts} = undef;
	}
	
	$self->setSkip('RootPassword', 1);
	my $returnCode = $self->SUPER::checkRootUser($remoteRootUser);
	$self->fillInternalNetworkAndListenInterfaceValidValues();
	return $returnCode;
}

sub getSID {
	return $_[0]->{SID};
}

sub getSidAdm(){
	return $_[0]->{sidAdm};
}

sub getRootUser{
	return $_[0]->getValue('RootUser');
}

sub getRootPassword{
	return $_[0]->getValue('RootPassword');
}

sub getListenInterface{
	return $_[0]->getValue('ListenInterface');
}

sub getInternalNetworkAddress {
	return $_[0]->getValue('InternalNetwork');
}

sub getProductName{
    return "Inter-Service Communication Configuration";
}

sub getAction{
	return  $INTERNAL_NETWORK_ACTION;
}

sub getConfigureDialogs {
	my ($self, $wizard) = @_;
	my $dialogs = [];
	
	require LCM::Gui::Dialogs::CredentialsDialog;
	require LCM::Gui::Dialogs::InternalNetworkPropertiesDialog;
	
	if($self->isDistributedSystem()){
		if($self->getValue('Scope') eq 'system' && !$self->isUseSAPHostagent()){
			require LCM::Gui::Dialogs::RootUserDialog;
			push(@$dialogs, new LCM::Gui::Dialogs::RootUserDialog($wizard));
		}
	}
	push(@$dialogs, new LCM::Gui::Dialogs::CredentialsDialog($wizard));
	push(@$dialogs, new LCM::Gui::Dialogs::InternalNetworkPropertiesDialog($wizard));
	
	return $dialogs;
}

sub getStaticDialogs{
	my $self = shift;
	return [	'LCM::Gui::Dialogs::ConfigurationSummaryDialog',
				'LCM::Gui::Dialogs::ConfigurationExecutionDialog', 
				'LCM::Gui::Dialogs::FinalizeDialog'
		   ];
}

sub resetParam {
	my ($self, $paramId) = @_;
	
	my $param = $self->{params}->{$paramId};
	if ($param) {
		$param->{value} = undef;
		$param->{batchValue} = undef;
		$self->{options}->{$param->{opt}} = undef;
	}
	
	my $sub = $self->can ("reset$paramId");
	if ($sub) {
		return &$sub ($self);
	}
}

sub setPassword {
	my ($self, $password) = @_;
	
	return undef if(!$self->checkPassword($password));
	
	$self->{params}->{Password}->{value} = $password;
	
	return 1 if(!isSidadmin() && !$self->isUseSAPHostagent());
	
	my $remoteHostctrlHostsObject = $self->getRemoteHosts();
	my $rc = defined($self->CollectOtherHostInfos());
	$self->fillInternalNetworkAndListenInterfaceValidValues() if ($rc);
	
	return $rc;
}

sub checkPassword {
	my ($self, $password) = @_;
	$self->clearParameterWarnings('Password');

	if(!defined $password){
		$self->appendErrorMessage ("Password is not defined.");
		return 0;
	}elsif(length($password) == 0){
		$self->appendErrorMessage ("Password name cannot be empty.");
		return 0;
	}
	require LCM::Configuration::NewDBUser::User;
	my $sidadmUser = new LCM::Configuration::NewDBUser::User( $self->getSidAdm());
	$sidadmUser->setMsgLstContext($self->getMsgLstContext());
	my $rc = $sidadmUser->verifyPassword($password, $self);
	if (!defined $rc){
		$self->getMsgLst()->appendMsgLst ($sidadmUser->getErrMsgLst());
		my $sidadmUserLastMessage = @{$sidadmUser->getMsgLst()->getMessages()}[-1];
		chomp (my $warningString = ${$sidadmUserLastMessage->getMsgString});
		$self->addParameterWarning('Password', "$warningString");
		$self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
		return 1;
	}
	# Fix for bug 72608. Prevent doubled 'Unknown user password combination' message. The other message comes from 'LCM::Configuration::NewDBUser::User;verifyPasswordSHA'.
	if($rc != 1 && !isSidadmin($self->getSID())) {
		$self->appendErrorMessage ('Unknown user password combination');
	}
	return $rc;

}

sub checkRootPassword {
    my ($self, $password) = @_;
	my $returnCode = $self->SUPER::checkRootPassword($password);
	if($returnCode == 1){
		$self->fillInternalNetworkAndListenInterfaceValidValues();
	}
    return $returnCode;
}

sub createSummaryTree {
	my $self = shift;
	return LCM::SummaryTreeBuilder::buildSummaryTree($self);
}

sub _getParamSidAdmPassword{
	my ($self, $order, $section) = @_;
	return {
	            'order'                     => $order++,
	            'opt'                       => 'password',
	            'opt_arg'                   => '<sid_adm_password>',
	            'type'                      => 'passwd',
	            'section'                   => $section,
	            'value'                     => undef,
	            'default'                   => undef,
	            'set_interactive'           => 1,
	            'str'                       => 'System Administrator Password (' . $self->{sidAdm} . ')',
	            'desc'                      => 'System Administrator Password (' . $self->{sidAdm} . ')',
	            'leaveEmptyInConfigDump'    => 1,
	            'mandatory'                 => 1
	};
}

sub setRequiredParamsValue {
	my ($self) = @_;

	return $self->setBatchValueOfRequiredParams('InternalNetwork');
}

sub _setMandatoryInternalNetworkIfPossible {

    my ($self) = @_;

    my $param = $self->{params}->{InternalNetwork};

    my $hasValidValues = (defined $param->{valid_values} && scalar @{$param->{valid_values}}) ? 1 : 0;
    return 1 if (!$hasValidValues);

    $param->{mandatory} = 1;
    $param->{set_interactive} = 1;
    $self->setSkip('InternalNetwork', 0);
    return 1;
}

1;
