package LCM::Configuration::PersistenceUpdateManager;

use strict;

use experimental qw (smartmatch);

use SDB::Install::SAPSystem qw ( CollectSAPSystems );
use SDB::Install::Globals qw ($gKeynameEngine $gProductNameEngine);

use LCM::PersistencyUtils qw ( 	existsHdbupdPendingUpdateOnly
								existsHdblcmPendingUpdate
								existsHdbupdPendingUpdate );

use base qw ( LCM::Configuration::PersistenceManager );

sub handlePendingUpdate {
	my ($self, $sid) = @_;
	return $self->SUPER::handlePendingAction($sid);
}

sub handleHdbupdPendingUpdate {
	my ($self, $sid) = @_;
	return $self->SUPER::handleHdbPendingAction($sid);
}

sub handleHdblcmPendingUpdate {
	my ($self, $sid) = @_;
	return $self->SUPER::handleHdblcmPendingAction($sid);
}

sub existsHdbPendingAction {
	my ($self, $sid) = @_;
	return existsHdbupdPendingUpdate($sid);
}

sub existsHdblcmPendingAction {
	my ($self, $sid, $target) = @_;
	return existsHdblcmPendingUpdate($sid, $target);
}

sub existsHdbPendingActionOnly {
	my ($self, $sid) = @_;
	return existsHdbupdPendingUpdateOnly($sid);
}

sub getHdbToolName {
	return 'hdbupd';
}

sub _createHdbConfiguration {
	my ($self, $sid) = @_;

	return undef if (!defined $sid);

	my $allSystems = CollectSAPSystems();
	my $sapSystem = $allSystems->{$sid};
	require SDB::Install::Configuration::Upgrade;
	my $updConfig = new SDB::Install::Configuration::Upgrade();

	$updConfig->{optionIgnore} = $self->getConfiguration()->getOptionIgnore();

	$updConfig->{current_sid} = $sid;
	$updConfig->{sapSys}      = $sapSystem;

	return $updConfig;
}

sub validateNonServerComponentVersions {
    my ( $self ) = @_;
    my $config = $self->getConfiguration();
    my $selectedComponents = $config->getComponentManager()->getSelectedComponents();

    for my $component (@{$selectedComponents}){
        next if $component->isServer();
        next if $config->getIgnore ('check_pending_upgrade');
        next if !$component->hasPersistenceFile($config);

        my $persistency = $component->getPersistenceXMLObject($config);
        next unless defined $persistency;
        my $persistedComponentVersion = $persistency->getTargetVersionString();
        my $currentComponentVersion = $component->getVersion();
        my $persistedVersion = new SDB::Install::Version(split('\.', $persistedComponentVersion));
        my $currentVersion   = new SDB::Install::Version(split('\.', $currentComponentVersion));
        my $componentName = $component->getComponentName();

        if ($currentVersion->isNewerThan($persistedVersion)) {
            my $wrnMsg = "Pending update of " . $componentName .
              " to version $persistedComponentVersion will be ignored. Update to version $currentComponentVersion will start from the beginning.";
            $config->getMsgLst()->addWarning($wrnMsg);
            push @{$config->getWarningList()}, $wrnMsg;
            $component->setIgnorePersistency(1);
            $component->setPersistedVersion($persistedComponentVersion);
        } elsif ($persistedVersion->isNewerThan($currentVersion) && !$config->getIgnore('check_version')){
            my $pendingErrorMessage = $self->_getPendingErrorMessage($componentName, $currentComponentVersion, $persistedComponentVersion);
            $config->setErrorMessage($pendingErrorMessage);
            return undef;
        }
    }
    return 1;
}

sub _copyCommonParamsFromHdbupdIntoHdblcmConfig {
	my ($self, $hdbupdConfig, $skipIdsArray) = @_;

	my $hdblcmConfig = $self->getConfiguration();

	foreach my $id (@{$hdblcmConfig->getParamIds ()}){
		if ( grep {$id eq $_} (@{$skipIdsArray}) ) {
			next;
		}

		if (!exists($hdbupdConfig->{params}->{$id})) {
			next;
		}

		my $hdbupdParam = $hdbupdConfig->{params}->{$id};
		my $hdbupdValue = $hdbupdParam->{value};

		if (!defined $hdbupdValue) {
			next;
		}

		my $param = $hdblcmConfig->{params}->{$id};

		if ($param->{type} =~ /map/) {
			$self->_copyMapParam($hdbupdConfig, $id);
		} else {
			$self->_copyNotMapParam($hdbupdConfig, $id);
		}
	}

	return 1;
}

sub _validateDetectedServerKitVersionPerAction {
	my ($self, $persistedVersion, $detectedVersion) = @_;
	my $configuration = $self->getConfiguration();
	my $sDetectedVersion = $detectedVersion->asString();
	my $sPersistedVersion = $persistedVersion->asString();

	if ($persistedVersion->isNewerThan($detectedVersion)) {
		my $pendingErrorMessage = $self->_getPendingErrorMessage($gProductNameEngine, $sDetectedVersion, $sPersistedVersion);
		my $msg = $configuration->setErrorMessage($pendingErrorMessage);
		$msg->getSubMsgLst()->addError("$gProductNameEngine server installation kit '$sPersistedVersion' required.");

		if (!$configuration->getIgnore ('check_pending_upgrade') && !$configuration->getIgnore ('check_version')) {
			return undef;
		}

		$configuration->ResetError ();
		$configuration->getMsgLst()->addMessage ("Ignoring error due to command line switch '--ignore'");
	} elsif ($detectedVersion->isNewerThan($persistedVersion)){
		my $mcm = $configuration->getMediumComponentManager();
		my $detectedServerCmp = $mcm->getComponentByKeyName($gKeynameEngine);
	    if (!$configuration->getIgnore ('check_pending_upgrade')){
    	    my $wrnMsg = "Pending update of " . $detectedServerCmp->getComponentName .
                  " to version $sPersistedVersion will be ignored. Update to version $sDetectedVersion will start from the beginning.";
            $configuration->addWarning($wrnMsg);
            $detectedServerCmp->setIgnorePersistency(1);
            $detectedServerCmp->setPersistedVersion($sPersistedVersion);
	    }
	}
	return 1;
}

sub getParamIdsToSkip {
    my ($self) = @_;

    my @paramIdsToSkip = $self->SUPER::getParamIdsToSkip();
    push @paramIdsToSkip, 'CheckMnt';
    push @paramIdsToSkip, 'Target';
    push @paramIdsToSkip, 'UpdateExecutionMode';
    return @paramIdsToSkip;
}

1;
