#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2013 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

RenameSystemConfiguration

=head1 DESCRIPTION

Configuration class for the 'rename system' scenario.
=cut

#===============================================================================
package LCM::Configuration::RenameSystemConfiguration;

use strict;
use base qw (SDB::Install::Configuration::AnyRename LCM::Configuration::CertifiedHostConfig LCM::Configuration::GenericStackAny Exporter);

use File::Spec;
use SAPDB::Install::Hostname;
use SDB::Install::NewDBUser;
use SDB::Install::SAPProfiles;
use SDB::Install::SAPControl;
use SDB::Install::IniFile;
use SDB::Install::MsgLst;
use LCM::Python::HanaPropertiesHelper;
use LCM::InternalNetworkDetector;
use LCM::Configuration::GenericStackAny;
use SDB::Install::SysVars qw($path_separator $isWin);
use SDB::Install::Tools qw(isMountPoint);
use LCM::ResidentInstaller qw($installedPattern);
use SDB::Install::SAPSystem qw(CollectSAPSystems);
use SDB::Install::Globals qw ($gKeynameEngine $gSapsysGroupIdDefault $gHostRoleRDSync $gXSParametersRenameConstraint $gProductNameSHA 
        $gKeynameEs $gKeynameAccelerator $gFlavourPlatform);
use LCM::Configuration::HdbLcmInstalledActionConfiguration qw(getNumberOfSystemHosts);
use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::Configuration::Rename;
use SDB::Install::Installer;
use SDB::Install::System qw (isAdmin);

use LCM::Utils::CommonUtils qw(isSystemReplication);
use LCM::Configuration::ValueChangeListeners::Rename::DistributedInternalNetworkListener;
use LCM::Configuration::ValueChangeListeners::Rename::TargetSIDListener;
use LCM::Configuration::ValueChangeListeners::Rename::TargetListener;
use LCM::Configuration::ValueChangeListeners::Rename::ShouldAskDBSystemPasswordListener;
use LCM::Configuration::ValueChangeListeners::Rename::CertificatesHostmapListener;

use LCM::Configuration::ParametersCreator;
use LCM::Hosts::HostsDetector;
use SDB::Common::Utils qw(trim);
use LCM::Configuration::Hosts::UpdateHosts::UpdateLocalHostConfiguration qw($UPDATE_LOCAL_HOST_STEP_MSG $UPDATE_REMOTE_HOST_STEP_MSG);
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::SkippedComponentsManager qw($SCENARIO_RENAME);
use LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener;
use LCM::Configuration::ValueChangeListeners::Rename::RemoteExecutionListener;
use LCM::Configuration::ValueChangeListeners::Rename::InstallHostagentListener;
use LCM::Configuration::ValueChangeListeners::Rename::XsSpaceIsolationListener;
use LCM::Configuration::ValueChangeListeners::SHAVersionListener;
use LCM::Configuration::ValueChangeListeners::CheckLocalSystemRequirementsListener;

our $section = 'Rename';
our $SYSTEM_RENAME_ACTION = 'rename_system';
our $RENAME_INTERNAL_NETWORK_INTERFACE =
    '.internal';    # system rename is interested only in .internal network configuration
our @EXPORT_OK = qw($SYSTEM_RENAME_ACTION $RENAME_INTERNAL_NETWORK_INTERFACE);

sub new {
    my $self = shift->SUPER::new ( @_ );
    my $order = 1;

	my $isGUI = caller() eq 'LCM::Gui::App::LCMGuiAppInstalled';
	$self->{isContinueOption} = !$isGUI && $self->_isContinueOption();

    my $flavourProductName = $self->getFlavourProductName();
    $self->{params} = {
        'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $section),
    	'RemoteExecution' => $self->getParamRemoteExecution($order++, $section),
    	(
    	   $self->isRegisterScenario()
    	   ? ('InstallHostagent' => $self->getParamInstallHostagent( $order++, $section, undef, 1 ),
    	      )
    	   : ()
    	),
		'UseHttp'         => $self->getParamUseHttp        ($order++, $section),
        'InitUser'        => $self->getParamInitUser       ($order++, $section, undef, undef, ['hdbrename_rename_init_user']),
        'NoStart' => $self->getParamNoStart($order++, $section,
                        "Do not start $flavourProductName System but start service (sapstartsrv) instead",
                        "Does not start $flavourProductName System but starts service (sapstartsrv) instead",
                        0),
        'Target' => $self->getParamTarget ( $order++, $section ),
        'SID' => $self->getParamSID ( $order++, $section ),
        'SSOCertificate'   => $self->getParamSSOCertificate  ($order++, $section),
        'HostMap' => $self->getParamHostMap ( $order++, $section ),
        'SystemUsage' => $self->getParamSystemUsage($order++, $section),
        (
            $isWin
            ? ()
            : (
                'Scope' => $self->getParamScope ( $order++, $section ),
                'SystemIsOffline' => $self->getParamSystemIsOffline($order++, $section),
                'InstallSSHKey' => $self->getParamInstallSSHKey($order++, $section ),
                'RootUser' => $self->getParamRootUser ( $order++, $section ),
                'RootPassword' => $self->getParamRootPassword ( $order++, $section ),
                'SkipHostagentPw'   => $self->getParamSkipHostagentPw  ($order++, $section),
                'HostagentPassword' => $self->getParamHostagentPassword($order++, $section),
                'ListenInterface' => $self->getParamListenInterface ($order++, $section, 1),
                'InternalNetwork' => $self->getParamInternalNetwork ( $order++, $section, 1, undef, 1, 0, 1 ),
                'CertificatesHostmap' => $self->getParamCertificatesHostmap( $order++, $section ),
            )
        ),        
        'srcPasswd' => $self->getParamSrcPasswd($order++, $section),
        'newSID' => $self->getParamNewSID($order++, $section),
        'InstanceNumber' =>
            $self->getParamInstanceNumber ( $order++, $section, 'Target Instance Number', 0 ),
        'BasePathDataVolumes' =>
            $self->getParamBasePath ( $order++, $section, 'datapath', 'Location of Data Volumes' ),
        'BasePathLogVolumes' =>
            $self->getParamBasePath ( $order++, $section, 'logpath', 'Location of Log Volumes' ),
        'BasePathDataBackup' =>
            $self->getParamBasePath ( $order++, $section, 'databackuppath', 'Location of Data Backups' ),
        'BasePathLogBackup' =>
            $self->getParamBasePath ( $order++, $section, 'logbackuppath', 'Location of Log Backups' ),
        'BasePathCatalogBackup' =>
            $self->getParamBasePath ( $order++, $section, 'catalogbackuppath', 'Location of Catalog Backups' ),
        'EsDataPath' => $self->getParamEsDataPath($order++, $section, 1, 0),
        'EsLogPath' => $self->getParamEsLogPath($order++, $section, 1, 0),
        'AcceleratorDataPath'=> $self->getParamAcceleratorDataPath($order++, $section, 1, 0),
        'AcceleratorLogPath' => $self->getParamAcceleratorLogPath ($order++, $section, 1, 0),
        'BasepathFileDownloadRDSync'=> $self->getParamBasepathFileDownloadRDSync($order++, $section),
        'BasepathFileUploadRDSync'  => $self->getParamBasepathFileUploadRDSync($order++, $section),
        (
            $isWin
            ? (
                'Domain' => $self->getParamDomain($order++, $section),
                'Password' => $self->getParamPassword ( $order++, $section ),
                'XsEaDataPath' => $self->getParamXsEaDataPath($order++, $section, $gXSParametersRenameConstraint),
                )
            : (
                'Password' => $self->getParamPassword ( $order++, $section ),
                'InitUserHomeDir' => $self->getParamInitUserHomeDir ( $order++, $section, undef, undef, ['hdbrename_rename_init_user_home_dir']),
                'HomeDir' => $self->getParamHomeDir ( $order++, $section, '/usr/sap/$newSID/home' ),
                'Shell' => $self->getParamShell ( $order++, $section ),
                'UID' => $self->getParamUID ( $order++, $section ),
                'GID' => $self->getParamGID ( $order++, $section, $gSapsysGroupIdDefault, 0 ),
                'XsEaDataPath' => $self->getParamXsEaDataPath($order++, $section, $gXSParametersRenameConstraint),
                'XSSpaceUserIdSAP' => GetParamXSSpaceUserIdSAP($order++, $section, $gXSParametersRenameConstraint),
                'XSSpaceUserIdProd' => GetParamXSSpaceUserIdProd($order++, $section, $gXSParametersRenameConstraint),
            )
        ),
        'XsDomainName' => $self->getParamXsDomainName($order++, $section, $gXSParametersRenameConstraint),
        'KeepUser' => $self->getParamKeepUser ( $order++, $section, undef, 'Source' ),
        (
            !$isWin
            ? (
                 'KeepUserHomeDir'     => $self->getParamKeepUserHomeDir( $order++, $section, undef, 'Source'),
                 'SkipModifySudoers'   => $self->getParamSkipModifySudoers($order++, $section),
              )
            : ()
        ),
        'KeepXsUsers'     => $self->getParamKeepXsUsers    ($order++, $section, $gXSParametersRenameConstraint),
        'ChangeSystemPasswd' => $self->getParamChangeSystemPasswd($order++, $section, 1),
        'SystemUser'      => $self->getParamSystemUser     ($order++, $section, 1),
        'SrcSQLSysPasswd' => $self->getParamSrcSQLSysPasswd($order++, $section),
        'TrgSQLSysPasswd' => $self->getParamTrgSQLSysPasswd($order++, $section),
        'AutoStart'       => $self->getParamAutoStart      ($order++, $section, 1),
        'TrgSysReplicationCleanUp' => $self->getParamTrgSysReplicationCleanUp ($order++, $section),
    };

	$self->_reconfigureParameters($order);

    return $self;
}

sub _reconfigureParameters {
	my ($self, $order) = @_;
	
    require LCM::Configuration::GenericStackAny;
    my $genericStackAny = new LCM::Configuration::GenericStackAny
                                        ($self->{options}, $self->{configfile});
	
    my $underlyingToolName    = 'hdbrename';
    
	my $skipPassThroughParams = ['CheckOnly', 'ScopeInteractive', 'ShowPending'];
	if( !$self->isRegisterScenario() ){
		push @{$skipPassThroughParams}, 'InstallHostagent';
	}
    require SDB::Install::Configuration::Rename;
    my $underlyingTool  = new SDB::Install::Configuration::Rename();
    my $newOrder = $genericStackAny->addPassThroughComponentParams($self->{params},
                                                                   $order,
                                                                   $underlyingTool,
                                                                   $skipPassThroughParams,
                                                                   $section,
                                                                   $underlyingToolName);
	
    if ($newOrder > $order) {
        $self->{prefixPassThroughParam} =
            $genericStackAny->getPassThroughParamPrefix($section, $underlyingToolName);
    }
	
    $self->{params}->{SID}->{opt} = 'source_sid';
    $self->{params}->{SID}->{str} = 'Source System ID';
	
	my $summaryOrder = 0;	    
    $self->{params}->{Target}->{summaryOrder} = $summaryOrder++;
    $self->{params}->{SID}->{summaryOrder} = $summaryOrder++;
    $self->{params}->{newSID}->{summaryOrder} = $summaryOrder++;
    $self->{params}->{InstanceNumber}->{summaryOrder} = $summaryOrder++;

    $self->setDefault('SID', $self->getSid());
    $self->{params}->{SID}->{set_interactive} = 0;

    $self->{params}->{Target}->{set_interactive} = 0;
    
    $self->{params}->{SystemUser}->{set_interactive} = 0;

    if (exists $self->{params}->{Scope}) {
        $self->setDefault('Scope', 'system');
        $self->{params}->{Scope}->{skip}    = 0;
    }
    
    if (exists $self->{params}->{RootUser}) {
        $self->{params}->{RootUser}->{init_with_default} = 1;
    }
    
    $self->{params}->{InstanceNumber}->{mandatory} = 1;
    
    $self->{params}->{BasePathDataVolumes}->{set_interactive} = 1;
    $self->{params}->{BasePathDataVolumes}->{skip}            = 0;

    $self->{params}->{BasePathLogVolumes}->{set_interactive} = 1;
    $self->{params}->{BasePathLogVolumes}->{skip}            = 0;

    $self->{params}->{BasePathDataBackup}->{set_interactive} = 1;
    $self->{params}->{BasePathDataBackup}->{skip}            = 0;

    $self->{params}->{BasePathLogBackup}->{set_interactive} = 1;
    $self->{params}->{BasePathLogBackup}->{skip}            = 0;

    $self->{params}->{BasePathCatalogBackup}->{set_interactive} = 1;
    $self->{params}->{BasePathCatalogBackup}->{skip}            = 0;

	$self->{params}->{EsDataPath}->{set_interactive} = 1;
    $self->setSkip('EsDataPath', 0);

    $self->{params}->{EsLogPath}->{set_interactive} = 1;
    $self->setSkip('EsLogPath', 0 );

    $self->{params}->{AcceleratorDataPath}->{set_interactive} = 1;
    $self->setSkip('AcceleratorDataPath', 0);

    $self->{params}->{AcceleratorLogPath}->{set_interactive} = 1;
    $self->setSkip('AcceleratorLogPath', 0 );

	$self->{params}->{BasepathFileDownloadRDSync}->{set_interactive} = 1;
    $self->setSkip('BasepathFileDownloadRDSync', 0);

    $self->{params}->{BasepathFileUploadRDSync}->{set_interactive} = 1;
    $self->setSkip('BasepathFileUploadRDSync', 0 );

    $self->setDefault('KeepUser', 0);
    $self->{params}->{KeepUser}->{set_interactive} = 0;
    $self->{params}->{KeepUser}->{skip} = 1;
    $self->{params}->{KeepUser}->{init_with_default} = 0;
    
    $self->{params}->{GID}->{hidden} = 1;
    $self->{params}->{GID}->{silent} = 1;
    
    if ( $self->{params}->{KeepUserHomeDir} ) {
        $self->{params}->{KeepUserHomeDir}->{skip} = 1;
        $self->setDefault('KeepUserHomeDir', 0);
        $self->{params}->{KeepUserHomeDir}->{init_with_default} = 0;
        $self->{params}->{KeepUserHomeDir}->{set_interactive} = 0;
        $self->{params}->{KeepUserHomeDir}->{desc} = $self->{params}->{KeepUserHomeDir}->{str};
    }
	
	if (exists $self->{params}->{Scope}) {
        $self->setDefault('Scope', 'system');
        $self->{params}->{Scope}->{init_with_default} = 1;
        $self->{params}->{Scope}->{skip}    = 0;
    }
	
    $self->{_doNotNeedComponentManagerInSummary} = 1;
	
}

# Override - introduced due to bug 83072
sub showHostMap{}

sub enableMultipleHostParameters {
	my ($self, $useSidadmForHostctrl) = @_;
	return 1 if (! $self->isDistributedSystem());
# Ignore the value of $useSidadmForHostctrl, because Rename scenario is not supported as sidadm
	return $self->SUPER::enableMultipleHostParameters();
}

sub showCertificatesHostmap {
    my ($self) = @_;

    if ($self->isDistributedSystem(1) && $self->isScopeInstance()) {
		$self->getMsgLst()->addProgressMessage( "Note: Cannot retrieve current certificate hostnames for remote hosts, because scope parameter is set to instance." );
		$self->getMsgLst()->addProgressMessage( "      Enter value for 'Certificate Host Name' only if you want to generate certificates for remote hosts.\n" );
    }
}

sub _isContinueOption {
	my ($self) = @_;
	my $installer = new SDB::Install::Installer();
	my $sapSys = $installer->getInstallation($self->getMsgLst());
    
    return 0 if (!defined($sapSys) || ! $sapSys->isa('SDB::Install::SAPSystem'));
	return 0 unless ($sapSys->hasNewDB());
	
	my $hdbInstance = $sapSys->getNewDBInstances()->[0];
	my $hostnameDir = $hdbInstance->get_hostNameDir();
	my $persistenceFilePath = File::Spec->catfile($hostnameDir, "hdblcm_rename_system");
	
	return 0 if (!(-f $persistenceFilePath));
	return (-f File::Spec->catfile($hostnameDir, 'hdbrename'));
}
	
# Override
sub InitDefaults {
    my ( $self, $kit ) = @_;

    return undef if(!$self->SUPER::InitDefaults($kit));

    $self->_addListeners ();
    if (! $self->isDistributedSystem() ) {
        $self->setSkip( "ListenInterface",1 );
        $self->setSkip( "InternalNetwork",1 );
    }

    my $componentManager = $self->getComponentManager();
    my $components = $componentManager->getAllComponents();
    my $operation = $self->isRegisterScenario() ? 'register':'rename';
    my @unsupportedComponents = grep { $_->isFeatureUnsupported($operation) } @{$components};
    my @unsupportComponentNames = map { $_->getComponentName() } @unsupportedComponents;

    my $flavourProductName = $self->getFlavourProductName();
    if(scalar(@unsupportComponentNames) > 0){
        $self->resetError();
        my $componentsString = join(', ', @unsupportComponentNames) . (scalar(@unsupportComponentNames) > 1 ? ' components do' : ' component does');
        $self->setErrorMessage("The $flavourProductName System cannot be renamed, because the installed $componentsString not support this operation");
        return 0;
    }
    if(!$self->isRegisterScenario()){
        my $isSystemReplication = isSystemReplication($self);
        return 0 if (!defined $isSystemReplication);

        if ($isSystemReplication) {
            $self->resetError();
            $self->setErrorMessage("The $flavourProductName System cannot be renamed, because system replication is enabled");
            return 0;
        }
    }
    if( $self->{isContinueOption} && !$self->initPersistenceInfo()){
        return undef;
    }
    if(defined($self->{sapSys}) && !$self->verifyUserConfigFile($self->{sapSys})){
        return undef;
    }
    return 1;
}

sub initPersistenceInfo{
    my ($self)=@_;

    my $hdbInstance = $self->getOwnInstance();
    my $filePath = File::Spec->catfile($hdbInstance->get_hostNameDir(), 'hdblcm_rename_system');
    $self->{persistenceFilePath} = $filePath;

    if(!(-f $filePath)){
        my $flavourProductName = $self->getFlavourProductName();
        $self->getErrMsgLst()->addMessage ("Cannot continue renaming the $flavourProductName System. Persistence file '$filePath' is missing");
        return undef;
    }

    my $parsedData = $self->pers_load($filePath);
    $self->_storeParsedData($parsedData);

    for my $key (keys %{$self->{params}}){
        next if($key eq 'Password');
        $self->{params}->{$key}->{skip} = 1;
    }
    my $user = new SDB::Install::NewDBUser($self->getValue('newSID'));
    if($user->exists() && isAdmin() && $self->isLocalHostContext()){
        $self->setSkip('Password');
    }
    return 1;
}

sub _storeParsedData {
	my ($self, $parsedData) = @_;
	for my $key(keys %{$parsedData}){
		if($key =~ /^params\.(.+)/){
			$self->{params}->{$1}->{value} = $parsedData->{$key};
		} elsif ($key =~ /^map\.([^\.]+?)\.(.+)$/){
			$self->{params}->{$1}->{value}->{$2} = $parsedData->{$key};
		} else {
			$self->{persistedData}->{$key} = $parsedData->{$key};
		}
	}
	if(exists $self->{params}->{HostMap}){
		$self->{params}->{HostMap}->{origin_values} = [keys(%{$self->{params}->{HostMap}->{value}})];
	}
	if(exists $self->{params}->{CertificatesHostmap}){
		$self->{params}->{CertificatesHostmap}->{origin_values} = [keys(%{$self->{params}->{CertificatesHostmap}->{value}})];
	}
}

sub hasInstalledHLM {
    my ($self) = @_;
    my $HLMManifest = $self->getInstaller()->{runtimedir} . '/../../HLM/manifest';    
    return ( -f $HLMManifest );
}

sub _getPersistenceData {
	my ($self, $hdbInstance) = @_;
	my $localHost = $hdbInstance->get_host();
	my $allHosts = $hdbInstance->get_allhosts();
	my $data = {};
	
	$data->{isSIDChanged} 			 = $self->isSIDChanged() ? 'true' : 'false';
	$data->{isAnyCertificateChanged} = $self->isAnyCertificateChanged() ? 'true' : 'false';
	$data->{isInstanceNumberChanged} = $self->isInstanceNumberChanged() ? 'true' : 'false';
	$data->{oldSid} = $self->getValue('SID');
	$data->{newSid} = $self->getValue('SID');
	if( $self->isSIDChanged() ) {
		$data->{newSid} = $self->getValue ( 'newSID' );
    }
	$data->{instanceNumber} = $self->getValue ('InstanceNumber');
	$data->{Target} = $self->getValue ('Target');
	
	foreach my $key (keys %{$self->{params}}) {
		next if($self->{params}->{$key}->{type} =~ /map|passwd/);
		$data->{'params.' . $key} = $self->{params}->{$key}->{value};
	}
	
	if(exists $self->{params}->{HostMap}->{value}){
		foreach my $key (keys %{$self->{params}->{HostMap}->{value}}) {
			$data->{'map.HostMap.' . $key} = $self->{params}->{HostMap}->{value}->{$key};
		}
	}
	if(exists $self->{params}->{CertificatesHostmap}->{value}){
		foreach my $key (keys %{$self->{params}->{CertificatesHostmap}->{value}}) {
			next if($self->{params}->{CertificatesHostmap}->{value}->{$key} eq '');
			$data->{'map.CertificatesHostmap.' . $key} = $self->{params}->{CertificatesHostmap}->{value}->{$key};
		}
	}
	return $data;
}

sub isCertificatesHostMapNeeded {
    my ($self, $instconfig) = @_;
    my $doSkip = $instconfig->{params}->{CertificatesHostmap}->{skip};
    return (!defined $doSkip) || ( (defined $doSkip) && ($doSkip != 1) );
}

sub checknewSID {
    my ($self, $value) = (shift(), shift());
    my $result = $self->SUPER::checknewSID($value, @_);
    my $oldValue = $self->getValue('SID');
    my $user = new SDB::Install::NewDBUser($value);
    my $warningMessage = sprintf('User \'%s\' already exists on the system. Neither the password, nor any other attribute of the user will be changed.', $user->getSidAdmName());

    $self->clearParameterWarnings('newSID');
    $self->setSkip('Password', 0);
    if($user->exists()){
        if ($value ne $oldValue){
            $self->addParameterWarning('newSID', $warningMessage);
        }
        if ($result && isAdmin() && $self->isLocalHostContext()){
            $self->setSkip('Password');
        }
    }
    return $result;
}

sub getWarnings {
    my ($self) = @_;
    return $self->{warnings};
}

sub checkKeepUser {
    my ($self, $value) = @_;
    if($value =~ /$bool_true_pattern/) {
        $self->{params}->{KeepUserHomeDir}->{set_interactive} = 0;
    } else {
        $self->{params}->{KeepUserHomeDir}->{set_interactive} = 1;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the option '--nostart'

sub checkNoStart {

    my ($self, $value) = @_;

    if ($value =~ /$bool_true_pattern/i){

        my $changePasswd = $self->getBatchValue('ChangeSystemPasswd');

        if (defined $changePasswd
           &&
           (($changePasswd eq '') ||  ($changePasswd =~ /$bool_true_pattern/i))){
            $self->PushError ("Parameter '"
                    . $self->getOpt('NoStart')
                    . "' cannot be applied together with '"
                    . $self->getOpt('ChangeSystemPasswd') . "'");
            return 0;
        }
    }
    return 1;
}

# Override
sub tryEnableScopeParams {
    my ( $self ) = @_;   
    my $value = $self->{params}->{Scope}->{value};
    
    if ( defined $value && ($value eq 'system') ) {
        if (!$self->isDistributedSystem()) {
             $self->{params}->{RootUser}->{skip} = 1;
        }
    }
    return 1;
}

# Override
sub checkTrgSQLSysPasswd {
    my ( $self, $value ) = @_;
    
    $self->clearParameterWarnings('TrgSQLSysPasswd');
    
    if ( ! $self->getValue('SrcSQLSysPasswd') ) {
    	return $self->SUPER::checkTrgSQLSysPasswd ( $value );
    }
    
    my $returnCode;
	if ( $self->getValue('SrcSQLSysPasswd') eq $value ) {
		my $systemUser = $self->getValue('SystemUser') ? $self->getValue('SystemUser') : $self->getDefault('SystemUser');
	    my $msglst = new SDB::Install::MsgLst ();
	    $msglst->AddMessage ("Source and Target Database User ($systemUser) passwords are same. Choose another Target password.");
	    $self->PushError ("Target Database User ($systemUser) password is invalid", $msglst);
	    $returnCode = 0;
    } else {
	    $returnCode = $self->SUPER::checkTrgSQLSysPasswd ( $value );
    }
    
    return $returnCode;
}

sub checkSrcSQLSysPasswd {
    my ($self, $value) = @_;

    if($self->isRegisterScenario() && $value ne ''){
        my $dbUser = $self->getValue('SystemUser') || 'SYSTEM';
        my $systemWarning = "Cannot verify source database user ($dbUser) password in advance: no connection available.";
        $self->AddWarning($systemWarning);
        $self->addParameterWarning('SrcSQLSysPasswd', $systemWarning);
        return 1;
    }

    return $self->SUPER::checkSQLSysPasswd($value, undef, 'SrcSQLSysPasswd');
}

#Override
sub doSetBasePathParamState {
    my ($self, $basePathParam, $isBackupParam, $isSuccessful) = @_;
    if($isBackupParam) {
        unless($basePathParam->{value} || $basePathParam->{log_value}) {
            # do not show parameter if value has not been set
            $basePathParam->{hidden} = $isSuccessful ? 0 : 1;
        }
    }
    return 1;
}

#Override
sub getCurrentSID {
    my ($self) = @_;    
    return $self->{_renameExecuted} ? $self->getValue('newSID') : $self->getValue('SID');
}

#----------------------------------------------------------------------------
# In sidmountpoint scenario shared directory is mounted with SID included.
#
sub isSidMountpoint {
    my ( $self, $sharedDir ) = @_;
    my $message;

    $sharedDir = $sharedDir . $path_separator;
    my $currentSidSharedPath =
        $sharedDir . $self->getSid ();    # add detected current SID to the shared path

    $self->AddMessage (
        "Checking if shared directory: $currentSidSharedPath is a SID mountpoint..." );
    my $isMountPoint = isMountPoint ( $currentSidSharedPath );

    if ( not defined $isMountPoint ) {
        $self->PushError (
            "Cannot check if shared directory: $currentSidSharedPath is a SID mountpoint." );
        return undef;
    }

    unless ( $isMountPoint ) {
        $self->AddMessage ( "Shared directory: $currentSidSharedPath is not a SID mountpoint." );
        return 0;
    }

    $self->AddMessage ( "Shared directory: $currentSidSharedPath is a SID mountpoint." );
    return 1;
}

#----------------------------------------------------------------------------
# Read the source SID value from the HDB profile file as in the sidmountpoint
# scenario the detected source SID may not be correct if the user has prepared
# all required manual steps for system rename.
#
sub readSIDFromProfileFile {
    my ( $self, $sharedDirectory, $sid ) = @_;
    return $self->readFromProfileFile($sharedDirectory, $sid, 'SAPSYSTEMNAME');
}

#----------------------------------------------------------------------------
# In sidmountpoint scenario there are two cases:
#
# 1. The user has prepared all required manual steps for system rename. In this case
# the shared directory for the currently existing SID (SID that is to be renamed) is actually
# symlink to the shared directory that contains the target SID in It's path.
# In this case the target SID is actually detected as source by the hdblcm and needs to be corrected.
#
# 2. The user has not prepared the required manual steps for the system rename. In this case the
# shred directory is a SID mountpoint but It is not a symlink that contains the target SID in the path.
# In this case source and target SID are detected to be the same and message is provided for the user
# to check out the documentation.
#
sub setCorrectSidValuesInSidMountpointScenario {
    my ( $self, $sharedDir, $originalyDetectedCurrentSid, $actualSid ) = @_;
	
    my $sourceSid =
        $self->setCorrectSourceSidInSidMountpointScenario ( $originalyDetectedCurrentSid,
        $actualSid );
    return undef unless $sourceSid;

    my $targetSid =
        $self->setCorrectTargetSidInSidMountpointScenario ( $sharedDir,
        $originalyDetectedCurrentSid );
    return undef unless $targetSid;

    my $message = "The HANA system is installed with SID mountpoint.";
    if ( $sourceSid ne $targetSid ) {
	     # In this case the user has already preparead the manual steps required for the rename scenario
	     # the detected current SID is actually the target SID.
        $message .=  " Based on the file system layout, the system id will be changed from $sourceSid to $targetSid";
    } else {
		# In this case share dir is the SID mountpoint but the user still has not prepared the manual steps
		# required for the rename scenario. As current and target SID are equal rename of the SID is not possible.
        $message .= " Check the documentation for the exact procedure to rename such systems.";
    }

    $self->AddWarning ( $message );
    $self->getMsgLst ()->addProgressMessage ( $message );

    # ensure that Instace is available for the newly detected source SID
    unless ( $self->getOwnInstance () ) {
        my $error = "Cannot detect SAP Instance for SID: " . $self->getSid ();
        $self->{'error_state'} = $error;
        $self->getErrMsgLst->addMessage ( $error );
        return undef;
    }
    return 1;
}

sub setCorrectTargetSidInSidMountpointScenario {
    my ( $self, $sharedDir, $originalyDetectedSourceSid ) = @_;
    my $error;
    $sharedDir = $sharedDir . $path_separator;
    my $detectedSidMountpointPath = $sharedDir . $originalyDetectedSourceSid;

    # check how this behaves on windows
    my $resolvedSidSharedPath = File::Spec->canonpath($detectedSidMountpointPath);
    my $targetSid = $resolvedSidSharedPath;
    $targetSid =~ s/$sharedDir//g;    # extract the target SID from the sidmountpoint path

    #
    # hack, use $instconfig->setValue() method to ensure that the new SID is valid,
    # but do not fail if the same SID value has been already set
    #
    $self->{_dontFailOnAlreadyInUseParameter} = 1;

    unless ( $self->setValue ( 'newSID', $targetSid, $self->getMsgLst () ) ) {
        $error = "Cannot set target SID value to $targetSid";
        $self->{'error_state'} = $error;
        $self->getErrMsgLst->addMessage ( $error );
        $self->{_dontFailOnAlreadyInUseParameter} = 0;
        return undef;
    }
    $self->{_dontFailOnAlreadyInUseParameter} = 0;

    # forbidd the user to change the target SID manually
    $self->{params}->{newSID}->{set_interactive} = 0;
    $self->{params}->{newSID}->{skip}            = 1;

    return $targetSid;
}

sub setCorrectSourceSidInSidMountpointScenario {
    my ( $self, $originalyDetectedSid, $actualSid ) = @_;
    my $error;
    #
    # hack, use $instconfig->setValue() method to ensure that the new SID is valid,
    # but do not fail if the same SID value has been already set
    #
    $self->{_dontFailOnAlreadyInUseParameter} = 1;

    unless ( $self->setSID ( $actualSid ) ) {
        $error = "Cannot set current SID value from $originalyDetectedSid to $actualSid";
        $self->{'error_state'} = $error;
        $self->getErrMsgLst->addMessage ( $error );
        $self->{_dontFailOnAlreadyInUseParameter} = 0;
        return undef;
    }
    $self->{_dontFailOnAlreadyInUseParameter} = 0;

    return $actualSid;
}

#----------------------------------------------------------------------------
#
# After sysytem rename own instance points to old sid/instance.
# Explicitly use instance linked to the new sid.
#
sub getOwnInstanceAfterRename {
    my $self               = shift();
    my $flavourProductName = $self->getFlavourProductName();
    my $error              = "Failed to detect own instance after $flavourProductName System rename.";
    my $system             = $self->_getRefreshedSystem();
    return undef unless $system;

    if (!$system->hasNewDB()) {
        $self->PushError ( $error );
        return undef;
    }

    my $instance = $system->getNewDBInstances ()->[0];
    unless ( $instance ) {
        $self->PushError ( $error );
        return undef;
    }
    return $instance;
}

#Override
sub setChangeSystemPasswd {
    my $self = shift;
    my $value = $_[0];
    my $rc = $self->SUPER::setChangeSystemPasswd(@_);
    if($rc) {
        my $shallChangePassword = ($value =~ /$bool_true_pattern/) ? 1 : 0;
        $self->setInteractive('SystemUser', $shallChangePassword);
        $self->setMandatory('TrgSQLSysPasswd', $shallChangePassword);
    }
    return $rc;
}

#Override
sub checkCollectedHostInfo {
    my $self = shift();
    return $self->SUPER::checkCollectedHostInfo(@_) unless $self->{_renameExecuted};
    return 1;
}

# Override
sub checkScope {
    my ( $self, $value ) = @_;    
    if (defined $value && ($value eq 'instance')) {
        $self->{params}->{RootUser}->{skip} = 1;
    } else {
        $self->{params}->{Scope}->{hidden} = 1;
    }
    return 1;
}

# Override
sub checkRootUser {
    my ( $self, $rootUser ) = @_;
    $self->{params}->{Scope}->{hidden} = 1;
    return $self->SUPER::checkRootUser($rootUser);
}

#Override
sub getCurrentSAPSystem {
    my $self    = shift();
    return $self->SUPER::getCurrentSAPSystem() unless $self->{_renameExecuted};
    return $self->_getRefreshedSystem();
}

sub _getRefreshedSystem {
    my $self               = shift();
    my $systems            = CollectSAPSystems ( undef, 1 );
    my $flavourProductName = $self->getFlavourProductName();
    my $error              = "Failed to detect own instance after $flavourProductName System rename.";

    unless ( $systems ) {
        $self->PushError ( $error );
        return undef;
    }

    my $system = $systems->{ $self->getValue ( 'newSID' ) };
    unless ( $system ) {
        $self->PushError ( $error );
        return undef;
    }
    return $system;   
}

sub checkHostMap {
    my ($self, $originHost, $newHost) = @_ ;
    
    if(LCM::HostsParser::IsLocalHost($originHost,$self->getMsgLst())&& !LCM::HostsParser::IsLocalHost($newHost,$self->getMsgLst())){
    	$self->PushError ("The target host name '$newHost' does not belong to the local host");
    	return 0;
    }

    if ( ! $self->SUPER::checkHostMap ( $originHost, $newHost, $self->{dontFailOnAlreadyDefinedHost} ) ) {
    	return undef;
    }

    if ( ! $self->canRenameHost($originHost,$newHost) ) {
    	return undef;
    } 
   
    # very ugly hack, should not be called from here
    # try to detect the internal network    
    $self->fillInternalNetworkAndListenInterfaceValidValues();
    return 1;
}


sub checkListenInterface {
    my $self = shift();

    my $rc = $self->SUPER::checkListenInterface ( @_ );
    return $rc unless $rc;

    # very ugly hack, should not be called from here
    # try to detect external host names
    my $exitCode = $self->fillCertificatesHostmapDefaultValuesAndLocalData();

    unless ( $exitCode ) {
        my $error = 'Failed to detect external host names default values.';        
        $self->getErrMsgLst->addMessage ( $error );
        return undef;
    }

    return $rc;
}

# Override
sub fillCertificatesHostmapDefaultValuesAndLocalData {
	my ($self) = @_;
	my $exitCode = $self->SUPER::fillCertificatesHostmapDefaultValuesAndLocalData();
	
    my $newOriginHostsBasedOnTargetValues = [];
    for my $originHost ( @{$self->{params}->{CertificatesHostmap}->{origin_values}} ){
    	my $targetHostname = $self->{params}->{HostMap}->{value}->{$originHost};
    	if(!defined($targetHostname)){
    		$targetHostname = $originHost;
    	}
    	push(@$newOriginHostsBasedOnTargetValues, $targetHostname);
    	$self->_fixDefaultMapForCertificatesHostmap($originHost, $targetHostname);
    }
    $self->{params}->{CertificatesHostmap}->{origin_values} = $newOriginHostsBasedOnTargetValues;
    return $exitCode;
}

sub _fixDefaultMapForCertificatesHostmap {
	my ($self, $originHost, $targetHost) = @_;
	
	delete($self->{params}->{CertificatesHostmap}->{default_map}->{$originHost});
	my $defaultValue = $self->{CertificatesHostmap}->{$originHost};
	if(!defined($defaultValue) || length($defaultValue) == 0){
		$defaultValue = $targetHost;
		$defaultValue = '' if($self->isScopeInstance());
	}
	$self->{params}->{CertificatesHostmap}->{default_map}->{$targetHost} = $defaultValue;
}


#-------------------------------------------------------------------------------
# Checks validity of the system administrator password used to stop the system

sub checksrcPasswd {

    my ($self, $password) = @_;
    if ( ! $self->SUPER::checksrcPasswd($password) ) {
        return undef;
    }

    if (defined $self->{sapSys}->getTargetSID()) {
    	# skip check in case of mount point
        return 1;
    }

    require SDB::Install::SAPControl;
    my $con = new SDB::Install::SAPControl('localhost',
                                           $self->getDefault('InstanceNumber'),
                                           undef, # user
                                           undef, # password,
                                           $self->isUseHttps());
    if ($con->accessCheck ($self->{sapSys}->getUser()->getSidAdmName(), $password)){
        $self->PushError ('Checking credentials of system administrator failed', $con);
        return 0;
    }
    return 1;
}

# Override
sub handleMasterInitParams {
    my ($self) = @_;

    return undef if ( ! $self->SUPER::handleMasterInitParams() );

    $self->handleMasterInitPathParams('BasePathDataVolumes');
    $self->handleMasterInitPathParams('BasePathLogVolumes');
    $self->handleMasterInitSpecificPathParams('EsDataPath', 'EsLogPath', $gKeynameEs);
    $self->handleMasterInitSpecificPathParams('AcceleratorDataPath', 'AcceleratorLogPath', $gKeynameAccelerator);
    $self->handleMasterInitRDSyncPathParams();

    return 1;
}

sub setBasePathValue {
    my ($self, $path, $param, $isBackupParam) = @_;
    my $rc = $self->SUPER::setBasePathValue($path, $param, $isBackupParam);

    if ($rc){
        my $message = ($path eq $param->{_originPath}) ? undef : 'the path will be set automatically';
# If the path hasn't been changed, hdbrename will ignore the cmd line value, passed to it
        $param->{value} = $path;
        $param->{summaryTitle} = $message;
    }

    return $rc;
}

sub handleMasterInitPathParams {
    my ( $self, $pathParamID ) = @_;
    $self->setSkip($pathParamID, 0);
    $self->{params}->{$pathParamID}->{set_interactive} = 1;
}

sub handleMasterInitSpecificPathParams {
    my ( $self, $dataPathParamID, $logPathParamID, $componentKeyname ) = @_;

    my $systemCmpMgr = $self->getSystemComponentManager();
    if ((defined $systemCmpMgr) and (!$systemCmpMgr->isComponentAvailable($componentKeyname))) {
        $self->setSkip($dataPathParamID, 1);
        $self->setSkip($logPathParamID,  1);
        return 1;
    }

    $self->setSkip($dataPathParamID, 0);
    $self->setSkip($logPathParamID,  0);
    $self->{params}->{$dataPathParamID}->{set_interactive} = 1;
    $self->{params}->{$logPathParamID}->{set_interactive}  = 1;
}

sub handleMasterInitRDSyncPathParams {
    my ($self) = @_;
    my $downloadParamID = 'BasepathFileDownloadRDSync';
    my $uploadParamID   = 'BasepathFileUploadRDSync';
    my $activeRoles     = $self->getOwnInstance()->getActiveHostRoles();
    my $hasActiveRDSync = defined($activeRoles) && defined($activeRoles->{$gHostRoleRDSync});

    if (! $hasActiveRDSync) {
        $self->setSkip($downloadParamID, 1);
        $self->setSkip($uploadParamID,   1);
        return 1;
    }

    $self->setSkip($downloadParamID, 0);
    $self->setSkip($uploadParamID,   0);
    $self->{params}->{$downloadParamID}->{set_interactive} = 1;
    $self->{params}->{$uploadParamID}->{set_interactive}   = 1;
}

sub validateHost {
    my ( $self, $sourceHost, $targetHost ) = @_;
    my $errorList = new SDB::Install::MsgLst ();
    $self->resetMsgLstContext();
    $self->{dontFailOnAlreadyDefinedHost} = 1; 
    unless ( $self->setMapValueItem ('HostMap', $sourceHost, $targetHost) ) {
        $errorList->appendMsgLst($self->getErrMsgLst ());
    }
    return $errorList;
}

sub getInstaller {
    my $self = shift();
    unless ( defined $self->{_installer} ) {
        #
        # use custom rename installer
        #
        $self->{_installer} = LCM::ResidentInstaller->new ();
    }
    return $self->{_installer};
}

sub isDistributedSystem {
    my ( $self, $performRealCheck) = @_;
    
    return 0 if(!$performRealCheck && $self->{isContinueOption});
    # if the system is distributed, but the user does not want execution on remote hosts
    return 0 if(!$performRealCheck && $self->isScopeInstance());
    
    unless ( $self->{params}->{SID}->{value} || $self->getDefault('SID')) {
        $self->PushError (
            "Cannot detect current System Administrator. Cannot check for distributed system setup."
        );
        return undef;
    }
    
    return $self->SUPER::isDistributedSystem ();
}

sub _isParameterChanged {
    my ( $self, $parameterName, $currentValue ) = @_;
    my $newValue = $self->{params}->{$parameterName}->{value};
    return 0 if not defined $newValue;
    return not( $currentValue eq $newValue );
}

sub isSIDChanged {
	if ( $_[0]->{isContinueOption} ) {
		return $_[0]->{persistedData}->{isSIDChanged} eq 'true';
	}

    return $_[0]->_isParameterChanged ( 'newSID', $_[0]->{params}->{SID}->{value} );
}

sub isInstanceNumberChanged {
	if ( $_[0]->{isContinueOption} ) {
		return $_[0]->{persistedData}->{isInstanceNumberChanged} eq 'true';
	}
	
	return $_[0]->_isParameterChanged ( 'InstanceNumber', $_[0]->getOwnInstance ()->get_nr () );
}

#----------------------------------------------------------------------------
#
# Gets origin (original) host name of the localhost.
#
sub getOriginLocalhostName {
    my $self              = shift();
    my $hanaLocalHost = $self->getLocalHanaHost();
    return $hanaLocalHost if defined $hanaLocalHost;
    return lc( hostname() );
}

#Override
sub getLocalHanaHost(){
    my $self              = shift();
    my $instance          = $self->getOwnInstance ();
    my @sortedOriginHosts = sort @{ $instance->get_allhosts () };
    foreach my $currSrcHost ( @sortedOriginHosts ) {
        if ( $instance->isHostLocal ( $currSrcHost ) ) {
            return $currSrcHost;
        }
    }
    return lc(hostname());
}

#----------------------------------------------------------------------------
#
# Gets origin (original) certificate host name of the localhost.
#
sub getOriginLocalhostCertificate {
    my $self              = shift();
    my $instance          = $self->getOwnInstance ();
    my @sortedOriginHosts = sort @{ $instance->get_allhosts () };

    foreach my $currSrcHost ( @sortedOriginHosts ) {
        if ( $instance->isHostLocal ( $currSrcHost ) ) {
            return $self->{params}->{CertificatesHostmap}->{default_map}->{$currSrcHost};
        }
    }
    return undef;
}

sub isLocalhostNameChanged {
    my $self      = shift();
    my %hostNames = %{ $self->{params}->{HostMap}->{value} };
    my $instance  = $self->getOwnInstance ();
    my $localhostFromOriginValues = $self->_getLocalhostFromOldHostnames();
    
    if(defined($localhostFromOriginValues)){
        my $newLocalhostTarget = $hostNames{$localhostFromOriginValues};
        return ( $localhostFromOriginValues ne $newLocalhostTarget );
    }
    return 1;
}

sub _getLocalhostFromOldHostnames {
	my $self      = shift();
    my %hostNames = %{ $self->{params}->{HostMap}->{value} };
    my $instance  = $self->getOwnInstance ();
    foreach my $oldHost ( keys %hostNames ) {
        return $oldHost if ( $instance->isHostLocal($oldHost) );
    }
    return undef;
}

#----------------------------------------------------------------------------
#
# Gets the new host name of the localhost.
#
sub getChangedLocalhostName {
    my $self      = shift();
    my %hostNames = %{ $self->{params}->{HostMap}->{value} };
    my $instance  = $self->getOwnInstance ();
    my $localhostFromOriginValues = $self->_getLocalhostFromOldHostnames();
    if(defined($localhostFromOriginValues)){
        return $hostNames{$localhostFromOriginValues};
    } else{
    	foreach my $oldHost ( keys %hostNames ) {
	        if($instance->isHostLocal ($hostNames{$oldHost}) ){
	        	return $hostNames{$oldHost};
	        }
	    }
    }
	return undef;
}

#----------------------------------------------------------------------------
#
# Gets origin (original) remote hosts.
#
sub getOriginRemoteHosts {
    my $self = shift();

    my $instance = $self->getOwnInstance ();
    my @sortedOriginHosts = sort @{ $instance->get_allhosts () };
    my @originRemoteHosts;
    foreach my $originHost ( @sortedOriginHosts ) {
        if ( $instance->isHostLocal ( $originHost ) ) {
            next;
        }

        push( @originRemoteHosts, $originHost );
    }
    return @originRemoteHosts;
}

sub isDbSystemPasswordChanged {
    return $_[0]->getValue('ChangeSystemPasswd');
}

sub isDataPathChanged {
    my $self            = shift();
    my $currentDataPath = $self->getOwnInstance ()->getDataPath ( 1 );
    my $newDataPath     = $self->{params}->{BasePathDataVolumes}->{value};
    return 1 if ( $currentDataPath ne $newDataPath );
}

sub isLogPathChanged {
    my $self           = shift();
    my $currentLogPath = $self->getOwnInstance ()->getLogPath ( 1 );
    my $newLogPath     = $self->{params}->{BasePathLogVolumes}->{value};
    return 1 if ( $currentLogPath ne $newLogPath );
}

sub isEsDataPathChanged {
    my $self            = shift();
    my $currentEsDataPath = $self->getOwnInstance ()->getEsDataPath ( 1 );
    my $newEsDataPath     = $self->{params}->{EsDataPath}->{value};
    return 1 if ( $currentEsDataPath ne $newEsDataPath );
}

sub isEsLogPathChanged {
    my $self           = shift();
    my $currentEsLogPath = $self->getOwnInstance ()->getEsLogPath ( 1 );
    my $newEsLogPath     = $self->{params}->{EsLogPath}->{value};
    return 1 if ( $currentEsLogPath ne $newEsLogPath );
}

sub isAcceleratorDataPathChanged {
    my $self            = shift();
    my $currentAcceleratorDataPath = $self->getOwnInstance ()->getAcceleratorDataPath ( 1 );
    my $newAcceleratorDataPath     = $self->{params}->{AcceleratorDataPath}->{value};
    return 1 if ( $currentAcceleratorDataPath ne $newAcceleratorDataPath );
}

sub isAcceleratorLogPathChanged {
    my $self           = shift();
    my $currentAcceleratorLogPath = $self->getOwnInstance ()->getAcceleratorLogPath ( 1 );
    my $newAcceleratorLogPath     = $self->{params}->{AcceleratorLogPath}->{value};
    return 1 if ( $currentAcceleratorLogPath ne $newAcceleratorLogPath );
}

sub isBasepathFileDownloadRDSyncChanged {
    my $self = shift();
    my $currentPath = $self->getOwnInstance ()->getRDSyncFileDownloadPath ( 1 );
    my $newPath = $self->{params}->{BasepathFileDownloadRDSync}->{value};

    return $currentPath ne $newPath ? 1 : 0;
}

sub isBasepathFileUploadRDSyncChanged {
    my $self = shift();
    my $currentPath = $self->getOwnInstance ()->getRDSyncFileUploadPath ( 1 );
    my $newPath = $self->{params}->{BasepathFileUploadRDSync}->{value};

    return $currentPath ne $newPath ? 1 : 0;
}

sub isAnyCertificateChanged {
    my ($self) = @_;

	if ( $_[0]->{isContinueOption} ) {
		return $_[0]->{persistedData}->{isAnyCertificateChanged} eq 'true';
	}

    my $paramHostMap = $self->{params}->{CertificatesHostmap};
    if (defined $paramHostMap) {
        foreach my $sourceHost (keys %{$paramHostMap->{value}}) {
            my $targetHost = $paramHostMap->{value}->{$sourceHost};
            if (defined $targetHost && (lc($sourceHost) ne lc($targetHost))) {
                return 1;
            }
        }
    }
    return 0;
}

sub isSystemIdentityChanged {
    my ( $self ) = @_;

    return
           $self->isSIDChanged ()
        || $self->isInstanceNumberChanged ()
        || $self->isLocalhostNameChanged  ();
}

sub isInternalNetworkConfigurationChanged {
    my $self = shift();
    if($self->isListenInterfaceChanged()) {
       return 1; 
    }
    if($self->{params}->{ListenInterface}->{value} eq 'internal'){
       return $self->isInternalNetworkAddressChanged();     
    }
    return 0;
}

sub isListenInterfaceChanged {
    my $self                 = shift();
    my $listenInterface = $self->{params}->{ListenInterface};
    return 0 unless $listenInterface->{valid_values};
    return 0 unless $listenInterface->{value};

    if ( $listenInterface->{value} eq $self->getDefault('ListenInterface') ) {
        return 0;
    }
    return 1;
}

sub isInternalNetworkAddressChanged {
    my $self                 = shift();
    my $internalNetworkParam = $self->{params}->{InternalNetwork};
    return 0 unless $internalNetworkParam->{valid_values};
    return 0 unless $internalNetworkParam->{value};

    if ( $internalNetworkParam->{value} eq $self->getDefault('InternalNetwork') ) {
        return 0;
    }
    return 1;
}

sub shouldRunHdbRename {
    my ( $self ) = @_;
    if($self->isRegisterScenario()) {
        return 1;
    }
    return 1 if $self->isSIDChanged                    ();
    return 1 if $self->isInstanceNumberChanged         ();
    return 1 if $self->isAnyHostChanged                ();
    
    return 0 if($self->{isContinueOption});
    
    return 1 if $self->isDbSystemPasswordChanged       ();
    return 1 if $self->isDataPathChanged               ();
    return 1 if $self->isLogPathChanged                ();
	return 1 if $self->isEsDataPathChanged             ();
	return 1 if $self->isEsLogPathChanged              ();
    return 1 if $self->isAcceleratorDataPathChanged    ();
    return 1 if $self->isAcceleratorLogPathChanged     ();
	return 1 if $self->isBasepathFileDownloadRDSyncChanged();
	return 1 if $self->isBasepathFileUploadRDSyncChanged();
    return 1 if $self->isInternalNetworkConfigurationChanged();

    return 0;
}

sub isRegisterScenario{
	my ($self) = @_;
	my $systemHosts = getNumberOfSystemHosts();
    return ( defined $systemHosts && ($systemHosts <= 0) );
}
#----------------------------------------------------------------------------
#
# Returns list of selected by the user (target SID, target Instance Number, target localhost name)
# or default values.
#
sub getInstanceProperties {
    my $self = shift();

    my $sid =
        ( $self->isSIDChanged () )
        ? $self->{params}->{newSID}->{value}
        : $self->{params}->{SID}->{value};
	
	if($self->{isContinueOption}){
		return ($sid, $self->{params}->{InstanceNumber}->{value}, $self->getChangedLocalhostName());
	}

    my $nr =
        ( $self->isInstanceNumberChanged () )
        ? $self->{params}->{InstanceNumber}->{value}
        : $self->getDefault('InstanceNumber');
    
    my $hostname =
        ( $self->isLocalhostNameChanged () )
        ? $self->getChangedLocalhostName ()
        : $self->{params}->{HostMap}->{own_origin_value};

    return ( $sid, $nr, $hostname );
}

sub getSharedNewSIDPath {
    my $self = shift();
    my $sharedSIDPath =
          $self->{params}->{Target}->{value}
        . $path_separator
        . $self->{params}->{newSID}->{value};    # the new SID shared path
    return $sharedSIDPath;
}

sub getSidAdmHomeDir {
    my $self = shift();
    my $homedir = $self->getValue ( 'HomeDir' );
    unless ( $homedir ) {

        # if the target user already exists the HomeDir property may not be set
        # in such case get the home directory form the OS
        my $user = new SDB::Install::NewDBUser ( $self->getValue ( 'newSID' ) );
        if ( defined $user->getSidAdm () ) {
            $homedir = $user->getSidAdmHome ();
        }
    }
    return $homedir;
}

#----------------------------------------------------------------------------
#
# Listeners that are to be called on property value change.
#
sub _addListeners {
    my ( $self, ) = @_;

    $self->addParameterListener( 'Target', LCM::Configuration::ValueChangeListeners::Rename::TargetListener->new () );
    $self->addParameterListener( 'newSID', LCM::Configuration::ValueChangeListeners::Rename::TargetSIDListener->new () );
    $self->addParameterListener( 'SID', LCM::Configuration::ValueChangeListeners::Rename::ShouldAskDBSystemPasswordListener->new () );
    $self->addParameterListener( 'RootUser', LCM::Configuration::ValueChangeListeners::Rename::DistributedInternalNetworkListener->new () );
    $self->addParameterListener( 'RootUser', LCM::Configuration::ValueChangeListeners::Rename::CertificatesHostmapListener->new () );
    $self->addParameterListener( 'RootPassword', LCM::Configuration::ValueChangeListeners::Rename::DistributedInternalNetworkListener->new () );
    $self->addParameterListener( 'RootPassword', LCM::Configuration::ValueChangeListeners::Rename::CertificatesHostmapListener->new () );
    $self->addParameterListener( 'newSID',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('newSID',0) );
    $self->addParameterListener( 'InstanceNumber', LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('InstanceNumber',0) );
    $self->addParameterListener( 'HostMap',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('HostMap',0));
    $self->addParameterListener( 'HostMap', LCM::Configuration::ValueChangeListeners::Rename::CertificatesHostmapListener->new());
    $self->addParameterListener( 'srcPasswd', LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('srcPasswd',1) );
    $self->addParameterListener( 'BasePathDataVolumes', LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('BasePathDataVolumes',1));
    $self->addParameterListener( 'BasePathLogVolumes',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('BasePathLogVolumes',1));
    $self->addParameterListener( 'EsDataPath',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('EsDataPath',1));
    $self->addParameterListener( 'EsLogPath',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('EsLogPath',1));
    $self->addParameterListener( 'AcceleratorDataPath',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('AcceleratorDataPath',1));
    $self->addParameterListener( 'AcceleratorLogPath',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('AcceleratorLogPath',1));
    $self->addParameterListener( 'BasepathFileDownloadRDSync',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('BasepathFileDownloadRDSync',1));
    $self->addParameterListener( 'BasepathFileUploadRDSync',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('BasepathFileUploadRDSync',1));
    $self->addParameterListener( 'ListenInterface',  LCM::Configuration::ValueChangeListeners::Rename::SystemRestartParamListener->new ('ListenInterface',1));
    $self->addParameterListener( 'RemoteExecution', LCM::Configuration::ValueChangeListeners::Rename::RemoteExecutionListener->new ());
    $self->addParameterListener( 'InstallHostagent', LCM::Configuration::ValueChangeListeners::Rename::InstallHostagentListener->new ());
    $self->addParameterListener( 'InstanceNumber', LCM::Configuration::ValueChangeListeners::Rename::XsSpaceIsolationListener->new ());

    if ($self->isRegisterScenario()) {
        $self->addParameterListener( 'SID', LCM::Configuration::ValueChangeListeners::CheckLocalSystemRequirementsListener->new());
        $self->addParameterListener('RootUser', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
        $self->addParameterListener('RootPassword', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
        $self->addParameterListener('Password', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
        $self->addParameterListener('HostagentPassword', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    }
}

#Override
sub setHostMap {
    my ($self, $originHost, $newHost) = @_;
    my $rc = $self->SUPER::setHostMap($originHost, $newHost, $self->{dontFailOnAlreadyDefinedHost} );
    
    # build remote hosts object for the target hosts 
    if($rc && $self->isDistributedSystem()) {
    	if(defined $self->{_remoteHosts}){
    		$self->{_remoteHosts} -> destroy();
    	}
    	$self->clearRemoteHosts();
    	$self->establishRemoteHostConnections();
    }

    return $rc;
}

sub setSID {
    my ($self, $sourceSID) = @_;

    my $rc = $self->checkSID($sourceSID);
    if (!$rc) {
        return $rc;
    }
    $self->{params}->{SID}->{value} = $sourceSID;
    return 1;
}

sub getSID {
	my ($self, $isGetSIDFromFileSystem) = @_;
	return $isGetSIDFromFileSystem ? $self->getSid() : $self->SUPER::getSID();
}

sub getStaticDialogs {
    return [
        'LCM::Gui::Dialogs::ConfigurationSummaryDialog',
        'LCM::Gui::Dialogs::ConfigurationExecutionDialog',
        'LCM::Gui::Dialogs::FinalizeDialog'
    ];
}

sub getConfigureDialogs {
    my ($self, $wizard) = @_;
    require LCM::Gui::Dialogs::Rename::SystemPropertiesDialog;
    require LCM::Gui::Dialogs::Rename::StoragePropertiesDialog;
    require LCM::Gui::Dialogs::EsStoragePropertiesDialog;
    require LCM::Gui::Dialogs::RDSyncStoragePropertiesDialog;
    require LCM::Gui::Dialogs::Rename::TargetUserDialog;
    require LCM::Gui::Dialogs::Rename::HostsDialog;
    require LCM::Gui::Dialogs::Rename::RenameCertificatesDialog;
    require LCM::Gui::Dialogs::Rename::NetworkAddressDialog;
    require LCM::Gui::Dialogs::Rename::HDBUserPropertiesDialog;
    require LCM::Gui::Dialogs::HostagentPasswordDialog;
    require LCM::Gui::Dialogs::Rename::XSAPropertiesDialog;

    return [
            $self->isDistributedSystem(1) ? new LCM::Gui::Dialogs::Rename::HostsDialog($wizard) : (),
            $self->isDistributedSystem(1) ? new LCM::Gui::Dialogs::Rename::RenameCertificatesDialog($wizard) : (),
            $self->isDistributedSystem() ? new LCM::Gui::Dialogs::Rename::NetworkAddressDialog($wizard) : () ,
            new LCM::Gui::Dialogs::Rename::SystemPropertiesDialog($wizard),
            new LCM::Gui::Dialogs::Rename::StoragePropertiesDialog($wizard),
            $self->isRegisterScenario() ? new LCM::Gui::Dialogs::HostagentPasswordDialog($wizard): (),
            new LCM::Gui::Dialogs::EsStoragePropertiesDialog($wizard),
            new LCM::Gui::Dialogs::RDSyncStoragePropertiesDialog($wizard),
            new LCM::Gui::Dialogs::Rename::TargetUserDialog($wizard),
            new LCM::Gui::Dialogs::Rename::HDBUserPropertiesDialog($wizard),
            new LCM::Gui::Dialogs::Rename::XSAPropertiesDialog($wizard),
    ];
}

sub getHanaSharedSidFromHdblcmRuntimeDir {
    my $self = shift;
    my $result = join ( $path_separator, $self->getInstaller()->{runtimedir}, '..', '..' )  ;
    return $result;
}

sub getHanaSharedFromHdblcmRuntimeDir {
    my $self = shift;
    my $result = join ( $path_separator, $self->getHanaSharedSidFromHdblcmRuntimeDir(), '..' )  ;
    return $result;
}

#----------------------------------------------------------------------------
#
# Gets an array with the names of the logical steps that are to be executed
# for the system rename scenario. Used in progress messages.
#
sub getProgressSteps {
    my ( $self ) = @_;
    my @progressSteps;
    
    my $isContinueAction = $self->{isContinueOption};
    if ( $isContinueAction && ( !$self->shouldRunHdbRename() && !$self->isAnyCertificateChanged() ) ) {
        push( @progressSteps, 'Renaming system' );
        return \@progressSteps;
    }
    my $flavourProductName = $self->getFlavourProductName();
    if($self->shallUnregisterComponents()){
    	push( @progressSteps, "Unregistering $flavourProductName System Components" );
    }
    push( @progressSteps, 'Renaming system' );

    if ( $self->isSystemIdentityChanged () ) {
        push( @progressSteps, 'Registering Instance Service' );
    }

    if ( $self->shouldRunHdbRename () ) {
        push( @progressSteps, 'Updating component list' );
    }

	if($self->shallUpdateLocalHostAfterRename()){
        push( @progressSteps, $UPDATE_LOCAL_HOST_STEP_MSG );
    }
    if($self->shallUpdateRemoteHostsAfterRename()){
        push( @progressSteps, $UPDATE_REMOTE_HOST_STEP_MSG );
    }
    return \@progressSteps;
}

#----------------------------------------------------------------------------
#
# Gets source SID. That is the SID under which the hdblcm is installed.
#
sub getSid {
    return shift()->getInstaller ()->getSid ();
}

sub getAction {
    return $SYSTEM_RENAME_ACTION;
}

sub createSummaryTree {
    return LCM::SummaryTreeBuilder::buildSummaryTree ( $_[0] );
}

sub getSummaryTitle
{
    return 'Rename Parameters';
}


sub setCurrentNetworkListenInterface {
    $_[0]->{_currentNetworkListenInterface} = $_[1];
    return 1;
}

sub getCurrentNetworkListenInterface {
    return shift()->{_currentNetworkListenInterface};
}

sub checkUID{
    my ($self, $value, $msglst) = @_;

    if ($self->getSID() eq $self->getValue('newSID')) {
        delete $self->{remoteUids};    # todo :  correct this implementation
    }
    
    # do not fail on existing remote UID's
    return $self->SUPER::checkUID($value, $msglst, 1);
}

sub isHostagentPasswordInitial {
	my ($self) =@_;
	return 0 unless defined $self->{params}->{'HostagentPassword'}->{type};
	return $self->{params}->{HostagentPassword}->{type} eq 'initial_passwd';
}

#-------------------------------------------------------------------------------
# This method is called from handleInitParams when a previous hdbrename is not
# yet executed at this host due to manually execution ('--scope=instance')
#
# Returns int retCode

sub handleSlaveInitParams {

    my ($self) = @_;

    my $sapSys = $self->{sapSys};
    my $sidDir = (defined $sapSys) ? $sapSys->get_sid () : undef;
    $sidDir    = '<SID>' if not (defined $sidDir);

    $self->setErrorMessage('Previous rename operation is pending, execute: '
                           . join($path_separator,
                                  $self->getValue('Target'),
                                  $sidDir, 'global', 'hdb', 'install', 'bin',
                                  'hdbrename'));
    return undef;
}

sub setRequiredParamsValue {
	my ($self) = @_;
	return $self->setBatchValueOfRequiredParams('TrgSQLSysPasswd');
}

sub getSystemComponentManager {
	my ($self) = @_;
    if(!defined $self->{systemComponentManager}){
		my $installer = new LCM::Installer(undef, $self);
		$installer->setMsgLstContext([$self->getMsgLst()]);
		my $sapmnt = $self->getHanaSharedFromHdblcmRuntimeDir();
	    $self->{systemComponentManager} = $installer->getOwnSystemComponentManager($sapmnt);
    }
    return $self->{systemComponentManager};
}

sub getComponentManager{
    my($self) = @_;
    return $self->getSystemComponentManager();
}

#Override
sub getSAPSystem {
    my $self    = shift();
    return $self->SUPER::getSAPSystem() unless $self->{_renameExecuted};
    return $self->_getRefreshedSystem();
}

sub shallReregisterHLM {
    my ( $self ) = @_;
    return
         $self->hasInstalledHLM()
      && !$isWin
      && ( $self->isSIDChanged()
        || $self->isInstanceNumberChanged()
        || $self->isAnyCertificateChanged() );
}

sub shallReregisterComponents {
    my ( $self ) = @_;
    return $self->isSIDChanged();
}

sub shallUnregisterComponents{
    my($self) = @_;

    return 0 if($self->isRegisterScenario());

    require LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager;
    my $existsRegistrableComponents = 0;
    my $components = $self->getSystemComponentManager()->getAllComponents();
    my $componentRegistrationManager = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager($components,$components,$SCENARIO_RENAME);
    $existsRegistrableComponents = $componentRegistrationManager->existRegistrableComponents();
    return $existsRegistrableComponents && $self->shallReregisterComponents() || $self->shallReregisterHLM();
}

sub shallUnregisterComponentsOnRemoteHosts{
  my($self) = @_;
  require LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager;
  my $existsRegistrableComponents = 0;
  my $components = $self->getSystemComponentManager()->getAllComponents();
  my $componentRegistrationManager = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager($components,$components,$SCENARIO_RENAME);
  $existsRegistrableComponents = $componentRegistrationManager->existRegistrableComponents();
  return $existsRegistrableComponents && $self->shallUnregisterComponents() && !$self->isScopeInstance() && $self->isDistributedSystem() ;  
}

sub skipRegenerationOnLocalhost {
    my ( $self ) = @_;

    if ($self->shouldSkipHostagentCalls()) {
        $self->getMsgLst()->addMessage("Skipping certificate regeneration on localhost because all $gProductNameSHA calls are skipped.");
        return 1;
    }

    my $hostctrlSecDir = File::Spec->catdir('/usr', 'sap', 'hostctrl', 'exe', 'sec');
    my $pseLocation = File::Spec->catfile($hostctrlSecDir, 'SAPSSLS.pse');
    return 0 if (! -f $pseLocation);

    return 0 if ( !$self->{isContinueOption} );

    my $localhost = $self->getLocalHanaHost();
    return 0 if ( exists( $self->{params}->{CertificatesHostmap}->{value}->{$localhost} ) );

    $self->getMsgLst()->addWarning("Skipping certificate regeneration on localhost. No value was provided");
    return 1;
}

sub shallUpdateLocalHostAfterRename{
#	This behaviour changed in order to ensure propperly deployed SHA operations on the local host
    return 1;
}

sub shallUpdateRemoteHostsAfterRename{
   my($self) = @_;
   return !$self->isScopeInstance() && $self->isDistributedSystem(); 
}

sub canRenameHost{
    my($self,$originHost,$newHost) = @_;
    $newHost = trim($newHost);
    if(!defined $newHost || $newHost eq '' || $newHost eq $originHost ){
    	return 1;
    }
    my $hostsDetector = new LCM::Hosts::HostsDetector(new LCM::ResidentInstaller(), $self->getMsgLstContext());

    my $componentManager = $self->getComponentManager();
    my $components = $componentManager->getAllComponents();

    for my $component( @$components ) {
    	if(! $hostsDetector->isComponentHost($originHost,$component)){
            next;
        }
    	if ($component->isFeatureUnsupported('rename-host')) {
        	$self->getErrMsgLst->addMessage ( $component->getComponentName()." does not support rename of a hostname" );
            return 0;
        }
    }
    return 1;
}

sub checkPassword {
    my $self = shift();
    if (defined($self->{_sidAdmUser})) { # Delete any cached _sidAdmUser value (see Bug: 100924)
        delete($self->{_sidAdmUser});
    }
    return $self->SUPER::checkPassword(@_);
}

sub setHostagentPassword {
	my ($self, $value) = @_;

	my $rc = $self->checkHostagentPassword($value);
	if (!$rc) {
		return $rc;
	}

	$self->{params}->{HostagentPassword}->{value} = $value;

	if ($self->isUseSAPHostagent() && $self->isCollectOtherHostInfosRequired()) {
		if (!defined $self->CollectOtherHostInfos()) {
			my $noRetry = ( $self->can('isAuthenticationFailedOnCollectOtherHostInfos') && $self->isAuthenticationFailedOnCollectOtherHostInfos() ) ? 0 : 1;
			$self->setNoRetry('HostagentPassword',$noRetry);
			return undef;
		}
	}

    return 1;
}

sub willInstallOrUpdateSHA {
    my ($self) = @_;
    my $installSHAValue = $self->getValue('InstallHostagent');
    my $isInstallSHA = (!defined $installSHAValue || ($installSHAValue =~ /$bool_true_pattern/)) ? 1 : 0;
    return ($isInstallSHA && $self->isRegisterScenario());
}

sub getProductName {
    my ($self) = @_;
    return ($self->isRegisterScenario()) ? "Register Rename System" : "Rename System";
}

sub getFlavourProductName {
    my ($self) = @_;
    return $self->{flavourProductName} if defined $self->{flavourProductName};
    
    if ($self->isRegisterScenario()){
        my $installer = new LCM::Installer(undef, $self);
        my $sapmnt = $self->getHanaSharedFromHdblcmRuntimeDir();
        my $systemComponentManager = $installer->getOwnSystemComponentManager($sapmnt);
        return LCM::App::ApplicationContext::getFlavourProductName() unless defined $systemComponentManager;

        my $components = $systemComponentManager->getAllComponents();
        my $flavour;
        
        for my $component (@{$components}) {
            if ($component->getComponentKeyName() eq $gKeynameEngine) {
                $flavour = $component->getManifest()->getHANAFlavour() || $gFlavourPlatform;
                last;
            }
        }
        $self->{flavourProductName} = SDB::Install::Globals::getFlavourProductName($flavour);
    } else {
        $self->{flavourProductName} = LCM::App::ApplicationContext::getFlavourProductName();
    }
    return $self->{flavourProductName};
}

1;
