package LCM::Configuration::SLD::RegistrationConfiguration;

use base qw(SDB::Install::Configuration::AnyConfig Exporter);

use strict;
use SDB::Install::Configuration::AnyConfig;
use LCM::Configuration::GenericStackAny;
use LCM::Configuration::ValueChangeListeners::SLD::SLDConnectionListener;
use LCM::SLD::SldregUtility;
use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::SysVars qw ($isWin);
use Scalar::Util;

our $CONFIGURE_SLD_ACTION = "configure_sld";
our @EXPORT_OK = qw($CONFIGURE_SLD_ACTION);

sub new {
	my $class = shift;

	my $self = $class->SUPER::new (@_);
	my $order = 0;

	my $installer = new LCM::Installer ();

	$self->{SID}        = $installer->getSid ();
	$self->{runtimeDir} = $installer->getRuntimeDir ();

	$self->{params} = {
		'SLDHostname' => {
			'order'                  => $order++,
			'opt'                    => 'sld_hostname',
			'opt_arg'                => '<hostname>',
			'type'                   => 'string',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 1,
			'str'                    => "SLD Host Name",
			'desc'                   => "SLD Host Name",
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 1
		},
		'SLDPort' => {
			'order'                  => $order++,
			'opt'                    => 'sld_port',
			'opt_arg'                => '<port>',
			'type'                   => 'string',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 1,
			'str'                    => 'SLD Port',
			'desc'                   => 'SLD Port',
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 1
		},
		'SLDUsername' => {
			'order'                  => $order++,
			'opt'                    => 'sld_username',
			'opt_arg'                => '<sld_username>',
			'type'                   => 'string',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 1,
			'str'                    => 'SLD User Name',
			'desc'                   => 'SLD User Name',
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 1
		},
		'SLDPassword' => {
			'order'                  => $order++,
			'opt'                    => 'sld_password',
			'opt_arg'                => '<sld_password>',
			'type'                   => 'passwd',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => undef,
			'set_interactive'        => 1,
			'str'                    => 'SLD Password',
			'desc'                   => 'SLD Password',
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 1
		},
		'UseHTTPS' => {
			'order'                  => $order++,
			'opt'                    => 'https',
			'opt_arg'                => '<true|false>',
			'type'                   => 'boolean',
			'section'                => $ini_section_general,
			'value'                  => undef,
			'default'                => 0,
			'set_interactive'        => 1,
			'str'                    => 'Use HTTPS',
			'desc'                   => 'Use HTTPS',
			'leaveEmptyInConfigDump' => 1,
			'mandatory'              => 0,
			'init_with_default'      => 1
		},
	};

	$self->_addListeners();

	return $self;
}

sub InitDefaults {
	my ($self) = @_;
	$self->_fillSLDParamsDefaultValues();

	return $self->SUPER::InitDefaults();
}

sub _fillSLDParamsDefaultValues {
	my ($self) = @_;
	my $instance = $self->getOwnInstance();
	if (!$instance) {
		$self->addWarning("Failed to get the configured SLD parameters - the HDB instance is not defined.");
		return;
	}

	my $sldreg = LCM::SLD::SldregUtility->new($self);
	my $storedParams = $sldreg->getStoredSLDProperties();

	for my $paramId (keys %{$storedParams}) {
		my $paramType = $self->getType($paramId);
		my $storedValue = $storedParams->{$paramId};
		next if $paramType =~ /passwd/;

		my $realValue = $paramType eq 'boolean' ? ($storedValue =~ $bool_true_pattern ? 1 : 0) : $storedValue;
		$self->setDefault($paramId, $realValue);
	}
}

sub _addListeners {
	my ($self) = shift();
	my $checkSLDConnectionListener = new LCM::Configuration::ValueChangeListeners::SLD::SLDConnectionListener();
	# the UseHTTPS parameter is expected to be the last validated from the group (SLDHostname, SLDPort, UseHTTPS)
	$self->addParameterListener( 'UseHTTPS', $checkSLDConnectionListener);
}

sub resetParam {
	my ( $self, $paramId ) = @_;

	my $param = $self->{params}->{$paramId};
	if ( $param ) {
		$param->{value}                     = undef;
		$param->{batchValue}                = undef;
		$self->{options}->{ $param->{opt} } = undef;
	}

	my $sub = $self->can ( "reset$paramId" );
	if ( $sub ) {
		return &$sub ( $self );
	}
}

sub checkSLDHostname {
	my ( $self, $hostname ) = @_;
	if ( not defined $hostname ) {
		$self->appendErrorMessage ( "Host name is not defined." );
		return 0;
	}
	if ( length( $hostname ) == 0 ) {
		$self->appendErrorMessage ( "Host name cannot be empty." );
		return 0;
	} elsif ( $hostname =~ /([^.a-zA-Z0-9-])/ ) {
		$self->appendErrorMessage ("Host name contains an invalid symbol; valid symbols are 0-9, a-z, A-Z, '.', '-'." );
		return 0;
	}
	return 1;
}

sub checkSLDPort {
	my ( $self, $port ) = @_;
	if ( not defined $port ) {
		$self->appendErrorMessage ( "Port is not defined." );
		return 0;
	}
	if ( length( $port ) == 0 ) {
		$self->appendErrorMessage ( "Port cannot be empty; enter a valid port number." );
		return 0;
	}
	if ( !Scalar::Util::looks_like_number ( $port ) ) {
		$self->appendErrorMessage ( "Port must be a number; enter a valid port number." );
		return 0;
	}
	if ( $port != 80 && $port != 443 && ( $port < 1024 || $port > 65535 ) ) {
		$self->appendErrorMessage ( "Port is not valid; valid values are: 80;443;1024-65535" );
		return 0;
	}
	return 1;
}

sub checkSLDUsername {
	my ( $self, $username ) = @_;
	if ( not defined $username ) {
		$self->appendErrorMessage ( "User name is not defined." );
		return 0;
	}
	if ( length( $username ) == 0 ) {
		$self->appendErrorMessage ( "User name cannot be empty; enter a valid user name." );
		return 0;
	}
	return 1;
}

sub checkSLDPassword {
	my ( $self, $password ) = @_;
	if ( not defined $password ) {
		$self->appendErrorMessage ( "Password is not defined." );
		return 0;
	}
	if ( length( $password ) == 0 ) {
		$self->appendErrorMessage ( "Password cannot be empty; enter a valid password." );
		return 0;
	}
	return 1;
}

sub getProductName {
	return "SLD Registration Configuration";
}

sub getSID {
	return $_[0]->{SID};
}

sub getRuntimeDir {
	return $_[0]->{runtimeDir};
}

sub getAction {
	return $CONFIGURE_SLD_ACTION;
}

sub getHostname {
	$_[0]->getValue ( 'SLDHostname' );
}

sub getPort {
	$_[0]->getValue ( 'SLDPort' );
}

sub getUsername {
	$_[0]->getValue ( 'SLDUsername' );
}

sub getPassword {
	$_[0]->getValue ( 'SLDPassword' );
}

sub isHTTPS {
	$_[0]->getValue ( 'UseHTTPS' );
}

sub setUseHTTPS {
	my ($self, $value) = @_;
	return undef if(! $self->checkValue('UseHTTPS',$value));

	$self->{params}->{UseHTTPS}->{value} = $value;
	return 1;
}

sub createSummaryTree {
	my $self = shift;
	my $warnings = $self->getWarnings();
	$self->{warnings} = $warnings if(scalar(@{$warnings}));
	return LCM::SummaryTreeBuilder::buildSummaryTree ( $self );
}

sub getStaticDialogs {
	return [
		'LCM::Gui::Dialogs::ConfigurationSummaryDialog',
		'LCM::Gui::Dialogs::ConfigurationExecutionDialog',
		'LCM::Gui::Dialogs::FinalizeDialog'
	];
}

sub getConfigureDialogs {
	my ($self, $wizard) = @_;
	require LCM::Gui::Dialogs::SLD::RegistrationPropertiesDialog;
	return [
				new LCM::Gui::Dialogs::SLD::RegistrationPropertiesDialog($wizard), 
		   ];
}

sub getWarnings {
	return $_[0]->checkSLDConnection();
}

sub checkSLDConnection {
	my ( $self ) = @_;

	my $httpsFlag   = $self->isHTTPS     ();
	my $sldHostname = $self->getHostname ();
	my $sldPort     = $self->getPort     ();
	my $responseCode;
	
	my $cannotConnectWarrningMessage = "Cannot connect to SLD system. Check the SLD host name and port.";
	my $isHttpsWarrningMessage = "Cannot check connection to SLD system.";

	if ( !$httpsFlag ) {
		$responseCode = $self->_checkHTTPConnection ( $sldHostname, $sldPort );
		return [] if ($responseCode < 400);
		$self->getMsgLst()->addWarning($cannotConnectWarrningMessage);
		return [$cannotConnectWarrningMessage];
	} else {
		#	TODO Implement reliable check for HTTPS under Win and Linux
		$self->getMsgLst()->addWarning($isHttpsWarrningMessage);
		return [$isHttpsWarrningMessage];
	}
}

sub _checkHTTPConnection {
	my ( $self, $sldHostname, $sldPort ) = @_;
	my $code = 400;
	
	require Net::HTTP;

	eval {
		my $connection = Net::HTTP->new ( Host => "$sldHostname:$sldPort", Timeout => 10 );
		if($connection){
			$connection->write_request ( GET => '/sld' );
			($code) = $connection->read_response_headers ();
		}
	};
	
	return $code;
}

sub getTimeoutValues {
	return [];
}

1;
