package LCM::Configuration::UnregisterConfiguration;

use base qw(SDB::Install::Configuration::AnyMultiHostConfig);

use SDB::Install::SysVars qw($isWin);
use SDB::Install::Configuration qw($bool_false_pattern $bool_true_pattern);
use SDB::Install::Globals qw($gProductNameXS2 $gXSParametersGenericConstraint);
use LCM::Configuration::GenericStackAny;
use Scalar::Util;
use SDB::Install::Installer;
use strict;

sub new {
    my ($class, $actionName, @args) = @_;
    my $self = $class->SUPER::new (@args);
    my $order = 0;
    my $installer = new LCM::Installer ();
    my $sid       = $installer->getSid();
    $actionName =~ s/unregister_//;
    @$self{qw(SID sidAdm scopeValue)} = ($sid, $self->getSysAdminUserName($sid), $actionName);

    my $skipHostagentCalls = $self->getParamSkipHostagentCalls($order++, $ini_section_general);
    my $remoteExecutionParam = $self->getParamRemoteExecution($order++, $ini_section_general, 0);
    my $useHttpParam         = $self->getParamUseHttp        ($order++, $ini_section_general);
    my $ssoCertParam         = $self->getParamSSOCertificate ($order++, $ini_section_general);
    my $installSSHKeyParam = $self->getParamInstallSSHKey($order++, $ini_section_general);
    my $rootUserParam = $self->getParamRootUser($order++, $ini_section_general);
    my $rootPasswordParam = $self->getParamRootPassword($order++, $ini_section_general);
    my $systemIsOfflineParam = $self->getParamSystemIsOffline($order++, $ini_section_general);
    my $hostagentPasswordParam = $self->getParamHostagentPassword( $order++, $ini_section_general);
    my $sidAdmPasswordParam = $self->_getParamSidAdmPassword($order++, $ini_section_general);
    my $keepUserParam = $self->getParamKeepUser($order++, $ini_section_general, undef, $self->{sidAdm});
    my $keepUserHomeDirParam = $self->getParamKeepUserHomeDir($order++, $ini_section_general, undef, $self->{sidAdm});
    my $keepXsUserParam = $self->getParamKeepXsUsers($order++, $ini_section_general, $gXSParametersGenericConstraint);
    my $skipModifySudoersParam = $self->getParamSkipModifySudoers($order++, $ini_section_general);

    $self->{params} = {
        'SkipHostagentCalls'=> $skipHostagentCalls,
        'RemoteExecution'   => $remoteExecutionParam,
        'UseHttp'           => $useHttpParam,
        'SSOCertificate'    => $ssoCertParam,
        'InstallSSHKey'     => $installSSHKeyParam,
        'RootUser'          => $rootUserParam,
        'RootPassword'      => $rootPasswordParam,
        'HostagentPassword' => $hostagentPasswordParam,
        'SystemIsOffline'   => $systemIsOfflineParam,
        'Password'          => $sidAdmPasswordParam,
        'KeepUser'          => $keepUserParam,
        'KeepUserHomeDir'   => $keepUserHomeDirParam,
        'KeepXsUsers'       => $keepXsUserParam,
        'SkipModifySudoers' => $skipModifySudoersParam,
    };

    $self->_initParameterProperties();
    return $self;
}

sub checkKeepUser {
    my ($self, $value) = @_;
    if($value =~ /$bool_true_pattern/ || $value =~ /$bool_false_pattern/){
        $self->setValue('KeepUserHomeDir', 'true') if ($value =~ /$bool_true_pattern/);
        return 1;
    }
    return 0;
}

sub InitDefaults{
    my $self = shift;

    return undef if(!$self->SUPER::InitDefaults(@_));

    my $installer = new LCM::Installer();
    my $sapSys = $installer->getInstallation($self->getMsgLst());
    my $sapmnt = $sapSys->getTarget();
    $installer->setMsgLstContext([$self->getMsgLst()]);
    my $componentManager = $installer->getOwnSystemComponentManager($sapmnt);
    my $components = $componentManager->getAllComponents();
    my @unsupportedComponents = grep { $_->isFeatureUnsupported('unregister') } @{$components};
    my @unsupportComponentNames = map { $_->getComponentName() } @unsupportedComponents;

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    if(scalar(@unsupportComponentNames) > 0){
        $self->resetError();
        my $componentsString = join(', ', @unsupportComponentNames) . (scalar(@unsupportComponentNames) > 1 ? 'components do' : 'component does');
        $self->setErrorMessage("The $flavourProductName System cannot be unregistered, because the installed $componentsString not support this operation");
        return 0;
    }

    push @{$self->{warnings}},"Data, log and installation directories will not be deleted, because they are required to register the system again later.";

    if(!$self->isDistributedSystem() || $self->getScope() eq 'instance'){
        my @parametersToBeSkipped = ('RootUser', 'RootPassword', 'HostagentPassword', 'RemoteExecution', 'UseHttp', 'InstallSSHKey');
        for my $paramID (@parametersToBeSkipped){
            $self->setSkip($paramID, 1);
        }
    }
    return 1;
}

sub setRemoteExecution {
    my $self = shift;
    my $returnCode = $self->SUPER::setRemoteExecution(@_);
    if($returnCode && ($self->getValue('RemoteExecution') eq 'saphostagent') ){
        $self->setSkip('InstallSSHKey', 1);
        $self->setSkip('RootUser', 1);
        $self->setSkip('RootPassword', 1);
        $self->setSkip('HostagentPassword', 0); # enable
    }
    return $returnCode;
}

sub checkRootUser {
    my ($self, $remoteRootUser) = @_;
    my $remoteHosts = $self->getRemoteHosts();

    if(defined $remoteHosts) {
        $remoteHosts->destroy();
        $remoteHosts->connect();
    }

    my $returnCode = $self->SUPER::checkRootUser($remoteRootUser);
    my $shouldSkipRootPassword = $self->getRemoteHosts()->isAuthenticationOk() ? 1 : 0;

    $self->setSkip('RootPassword', $shouldSkipRootPassword);
    $self->setMandatory('RootPassword', $shouldSkipRootPassword ? 0 : 1);
    return $returnCode;
}

sub checkPassword {
	my ($self, $password) = @_;
	if(!defined $password){
		$self->appendErrorMessage ("Password is not defined.");
		return 0;
	}elsif(length($password) == 0){
		$self->appendErrorMessage ("Password name cannot be empty.");
		return 0;
	} 
	my $sidadmUser = new SDB::Install::User ( $self->{sidAdm} );
	$sidadmUser->setMsgLstContext($self->getMsgLstContext());
	my $rc = $sidadmUser->verifyPassword( $password );
	if (!defined $rc){
		$self->getMsgLst()->appendMsgLst ($sidadmUser->getErrMsgLst());
		$self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
		return 1;
	}
	if($rc != 1) {
		$self->appendErrorMessage ('Unknown user password combination');
	}
	return $rc;
	
}

sub getScope {
	return $_[0]->{scopeValue};
}

sub getConfigureDialogs {
	my ($self, $wizard) = @_;
	require LCM::Gui::Dialogs::UnregisterPropertiesDialog;
	return [ new LCM::Gui::Dialogs::UnregisterPropertiesDialog($self->getScope(), $wizard) ];
}

sub getStaticDialogs{
	return [ 'LCM::Gui::Dialogs::ConfigurationSummaryDialog', 'LCM::Gui::Dialogs::ConfigurationExecutionDialog',  'LCM::Gui::Dialogs::FinalizeDialog' ];
}

sub getWarnings { return $_[0]->getWarningList(); }

sub getProductName{
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    return "Unregister the $flavourProductName " . ucfirst($_[0]->getScope()); }

sub getAction{ return "unregister_" . $_[0]->getScope(); }

sub createSummaryTree {
	my ($self) = @_;
	$self->{params}->{KeepUser}->{init_with_default} = 0;
	$self->{params}->{KeepUserHomeDir}->{init_with_default} = 0;
	my $summaryTree = LCM::SummaryTreeBuilder::buildSummaryTree($_[0]);
	$self->{params}->{KeepUser}->{init_with_default} = 1;
	$self->{params}->{KeepUserHomeDir}->{init_with_default} = 1;
	return $summaryTree;
}

sub getSummaryTitle { return "Unregister " . ucfirst($_[0]->getScope()) . " Parameters"; }

sub _getParamSidAdmPassword{
    my ($self, $order, $section) = @_;
    return {
        'order'                  => $order++,
        'opt'                    => 'password',
        'opt_arg'                => '<sid_adm_password>',
        'type'                   => 'passwd',
        'section'                => $section,
        'value'                  => undef,
        'default'                => undef,
        'set_interactive'        => 1,
        'str'                    => 'System Administrator Password (' . $self->{sidAdm} . ')',
        'desc'                   => 'System Administrator Password (' . $self->{sidAdm} . ')',
        'leaveEmptyInConfigDump' => 1,
        'mandatory'              => 1
    };
}

sub _initParameterProperties {
	my ($self) = @_;
	my $params = $self->{params};
	$params->{RootUser}->{init_with_default} = 1;
	$params->{RootUser}->{set_interactive} = 1;
	$params->{RootUser}->{mandatory} = 1;
	$params->{RootUser}->{skip} = 0;
	$params->{RootPassword}->{skip} = 0;
	$params->{RootPassword}->{mandatory} = 0;
	$params->{KeepUser}->{set_interactive} = 1;
	$params->{KeepUserHomeDir}->{set_interactive} = 1;
	$params->{HostagentPassword}->{type} = 'passwd';
}

1;
