package LCM::Configuration::UpdateSystemBootstrapConfiguration;

use strict;
use parent qw(SDB::Install::Configuration::AnyConfig LCM::Configuration::GenericStackConfiguration);
use LCM::Configuration::GenericStackAny qw($ini_section_general $ini_section_server);
use LCM::Configuration::ParametersCreator qw(GetParamVerifySignature);
use LCM::Configuration::GenericStackUpdateConfiguration;
use LCM::Configuration::ValueChangeListeners::Update::UpdateSystemBootstrapComponentListener;
use LCM::Configuration::ValueChangeListeners::Update::ServerDetectionListener;
use LCM::ComponentManager::ComponentScanner;
use LCM::Configuration::ValueChangeListeners::SignatureVerificationListenerFactory;
use SDB::Install::Globals qw($gKeynameEngine);
use SDB::Install::Manifest;
use SDB::Install::MsgLst;
use SDB::Install::SysVars qw($currentPlatformName);
use LCM::App::UpdateSystemUtils qw(isVerificationFailureCritical);
use LCM::Configuration::PersistenceUpdateManager;
use LCM::SAPDSigner;
use LCM::FileUtils qw(MyRealPath);
use File::Basename;
use File::Spec;

sub new {
    my $class = shift();
    my $self = $class->SUPER::new(@_);

    $self->defineParameters();
    $self->addListeners();
    $self->setPersistenceManager(LCM::Configuration::PersistenceUpdateManager->new($self));

    return $self;
}

sub defineParameters {
    my ($self) = @_;
    my $order = 1;

    $self->{params} = {
        'DvdPath'            => $self->getParamDvdPath( $order++,            $ini_section_general ),
        'ComponentDirs'      => $self->getParamComponentDirs( $order++,      $ini_section_general ),
        'ComponentFsRoot'    => $self->getParamComponentFsRoot( $order++,      $ini_section_general ),
        'VerifySignature'    => GetParamVerifySignature($order++, $ini_section_general),
        'SID'                => $self->getParamSID($order++, $ini_section_server, 1),
    };

    $self->setHidden('SID', 1);
}

sub InitDefaults {
    my $self = shift();
    my $sid = $self->getSID();
    $self->{current_sid} = $sid; # For the sake of persistency_manager->getPersistenceFilename()
    $self->setDefault('SID', $sid, 1);
    $self->setInteractive('SID', 0);
    $self->setDefault('VerifySignature', 1, 1);

# The resident installer must not print the detected components
# The non-resident one will print them after we delegate the execution to it
    $self->getComponentManager()->setCanPrintDetectedComponents(0);
    $self->detectInstalledComponents($sid);

    return 1;
}

sub CheckParams {
    my ($self, $isBatchMode) = @_;
    $self->_handlePersistence();
    return $self->SUPER::CheckParams($isBatchMode);
}

sub _handlePersistence {
    my ($self) = @_;
    if ($self->getIgnore('check_pending_upgrade')) {
        $self->getMsgLst()->addMessage("Ignoring persisted values due to --ignore command line switch.");
        return 1;
    }

    if (!$self->pers_exists()) {
        return 1;
    }

    for my $paramId ('DvdPath', 'ComponentDirs', 'ComponentFsRoot', 'VerifySignature') {
        my $persistedValue = $self->getPersistedValue($paramId);
        if ($persistedValue) {
            $self->setDefault($paramId, $persistedValue, 1);
        }
    }
}

sub pers_filename {
    my ($self) = @_;
    return $self->getPersistenceManager()->getPersistenceFilename();
}

sub checkSID {
    my ($self, $value) = @_;
    my $systemSID = $self->getSID();
    if ($value ne $systemSID) {
        $self->setErrorMessage("Can't use SID '$value' for update of system '$systemSID'. If you want to update another system, use its own dedicated resident hdblcm.");
        return 0;
    }
    return 1;
}

sub getAction {
    return LCM::Configuration::GenericStackAny::UPDATE_ACTION;
}

sub getCustomLogSuffix {
    'resident';
}

sub getIgnoreValues{
    return LCM::Configuration::GenericStackUpdateConfiguration::getIgnoreValues();
}

sub getSID {
    my ($self) = @_;
    if (!$self->{SID}) {
        $self->{SID} = LCM::Installer->new()->getSid();
    }
    return $self->{SID};
}

sub addListeners {
    my ($self) = @_;
    my $scanner = LCM::ComponentManager::ComponentScanner->new($self->getComponentManager(),$self,0,0);

    $self->addParameterListener('ComponentDirs',   LCM::Configuration::ValueChangeListeners::Update::UpdateSystemBootstrapComponentListener->new($scanner, 'ComponentDirs'));
    $self->addParameterListener('DvdPath',         LCM::Configuration::ValueChangeListeners::Update::UpdateSystemBootstrapComponentListener->new($scanner, 'DvdPath'));
    $self->addParameterListener('ComponentFsRoot', LCM::Configuration::ValueChangeListeners::Update::UpdateSystemBootstrapComponentListener->new($scanner, 'ComponentFsRoot'));
    $self->addParameterListener('VerifySignature', LCM::Configuration::ValueChangeListeners::Update::ServerDetectionListener->new());
    $self->addParameterListener('SID',             LCM::Configuration::ValueChangeListeners::SignatureVerificationListenerFactory::getSignatureVerificationListener());
}

sub getExternalExecutable {
    my ($self) = @_;
    my $appContext = LCM::App::ApplicationContext->getInstance();
    my $application = $appContext->getApplication();
    if (!defined($application)) {
        return undef;
    }
    return $application->isGUIApplication() ? 'hdblcmgui' : 'hdblcm';
}

sub isInstallerValid {
    my ($self) = @_;
    my $hdblcmPath = $self->getExternalHdblcmPath();
    return undef if !$hdblcmPath;

    my $instruntimeDir = File::Spec->catfile(dirname($hdblcmPath), 'instruntime');
    my $installerManifestPath = File::Spec->catfile($instruntimeDir, 'manifest');
    my $manifest = SDB::Install::Manifest->new($installerManifestPath);
    if (!$manifest->read(undef, 1)) {
        $self->getErrMsgLst()->addError("Unable to detect manifest in '$instruntimeDir'", $manifest->getErrMsgLst());
        return undef;
    }

    if (!$manifest->isPlatformSupported($currentPlatformName)) {
        $self->getErrMsgLst()->addError("The non-resident installer does not support current OS platform '$currentPlatformName'");
        return undef;
    }

    my $hdblcmExecutable = File::stat::stat($hdblcmPath);
    if (!defined($hdblcmExecutable) || ! -x $hdblcmExecutable) {
        $self->getErrMsgLst()->addError("The external hdblcm ('$hdblcmPath') is not executable");
        return undef;
    }

    return 1;
}

sub getExternalHdblcmPath {
    my ($self) = @_;
    my $executableName = $self->getExternalExecutable();
    if (!$executableName) {
        $self->getErrMsgLst()->addError("Failed to determine the external hdblcm executable.");
        return undef;
    }

    my $hanaServer = $self->getComponentManager()->getComponentByKeyName($gKeynameEngine);
    my $executablePath = $hanaServer ? File::Spec->catfile(dirname($hanaServer->getPath()), $executableName) : undef;
    return $executablePath ? MyRealPath($executablePath) : undef;
}

sub canIgnoreVerificationFailure {
    my ($self, $verbose) = @_;
    $verbose //= 1;
    my $serverComponent = $self->getComponentManager()->getComponentByKeyName($gKeynameEngine);
    my $errorList = $verbose ? $self->getErrMsgLst() : SDB::Install::MsgLst->new();
    return isVerificationFailureCritical($serverComponent, $errorList) ? 0 : 1;
}

sub getStaticDialogs {
    my ($self) = @_;
    require LCM::Gui::Dialogs::UpdateBootstrapParamsDialog;
    require LCM::Gui::Dialogs::BootstrapDelegationDialog;
    require LCM::Gui::Dialogs::ConfigurationExecutionDialog;
    return ['LCM::Gui::Dialogs::UpdateBootstrapParamsDialog',
            'LCM::Gui::Dialogs::BootstrapDelegationDialog',
            'LCM::Gui::Dialogs::ConfigurationExecutionDialog'];
}

sub getConfigureDialogs {
    my ($self) = @_;
    return [];
}

sub getProductName {
    my ($self) = @_;
    my $flavourProductName = $self->getFlavourProductName();
    return "Bootstrap non-resident update of $flavourProductName System and Components";
}

1;