package LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedXs2ComponentsListener;

use strict;
use experimental qw(smartmatch);
use SDB::Install::Version;
use SDB::Install::Globals qw($gShortProductNameXS2);

my $KEY_NAME_ALL = 'all';
my $KEY_NAME_NONE = 'none';
my $ERROR_MESSAGE_SELECTION_CONFLICT = 'Some of the provided values are conflicting with each other';

sub new {
	return bless({}, shift());
}

sub onValueChange {
	my ($self, $value, $configuration) = @_;
	my $selectedComponentKeyNames = [ split(/\s*,\s*/, $value) ];
	my $hasConflictingSelection = $self->_hasConflictingSelection($selectedComponentKeyNames);

	if($hasConflictingSelection){
		$configuration->setErrorMessage($ERROR_MESSAGE_SELECTION_CONFLICT);
		return undef;
	}

	$self->_selectComponentsBasedOnUserInput($selectedComponentKeyNames, $configuration);
	my $rc = $configuration->validateXs2ComponentDependencies() &&
			 $configuration->validateDependenciesToXs2Components() &&
			 $self->_validateXs2ComponentsVersion($configuration);

	return $rc ? 1 : undef;
}

sub _validateXs2ComponentsVersion {
	my ($self, $instconfig) = @_;
	my $mcm = $instconfig->getMediumComponentManager();
	my $scm = $instconfig->getSystemComponentManager();
	my @selectedXs2Components = grep { $_->isComponentSelected() } @{$mcm->getXs2ApplicationComponents()};
	return 1 if !$scm; # In case of installation
	return 1 if scalar(@selectedXs2Components) == 0;

	if ($instconfig->getIgnore('check_version')) {
		$instconfig->getMsgLst()->addMessage("Ignoring version checks of $gShortProductNameXS2 components due to --ignore command line switch");
		return 1;
	}

	for my $xs2app ( @selectedXs2Components ) {
		my $installedApp = $scm->getComponentByKeyName($xs2app->getComponentKeyName());
		next if !$installedApp;

		my $selectedAppVersion = $xs2app->getVersionObject();
		my $installedAppVersion = $installedApp->getVersionObject();

		if ($installedAppVersion->isNewerThan($selectedAppVersion)) {
			my $errorMessage = sprintf("Cannot downgrade '%s' from version %s to version %s",
										$xs2app->getComponentName(),
										$installedApp->getVersion(),
										$xs2app->getVersion());
			$instconfig->setErrorMessage($errorMessage);
			return 0;
		}
	}
	return 1;
}

sub _selectComponentsBasedOnUserInput {
	my ($self, $selectedComponentKeyNames, $configuration) = @_;
	my $componentManager = $configuration->getMediumComponentManager();
	my $allComponents = $componentManager->getXs2ApplicationComponents();

	for my $component (@{$allComponents}){
		my $shouldSelectComponent = $self->_shouldSelectComponent($component, $selectedComponentKeyNames);
		$component->selectComponent($shouldSelectComponent);
	}
}

sub _shouldSelectComponent {
	my ($self, $component, $selectedComponentKeyNames) = @_;
	my $componentKey = $component->getComponentKeyName();
	my $isSelectingAll = grep { $_ eq $KEY_NAME_ALL } @{$selectedComponentKeyNames};
	my $isSelectingComponent = grep { $_ eq $componentKey } @{$selectedComponentKeyNames};

	return ($isSelectingAll || $isSelectingComponent) ? 1 : 0;
}

sub _hasConflictingSelection {
	my ($self, $selectedComponentKeyNames) = @_;
	my $hasSelectedMultipleComponents = scalar(@{$selectedComponentKeyNames}) > 1;
	my $hasSelectedAllComponents = grep { $_ eq $KEY_NAME_ALL } @{$selectedComponentKeyNames};
	my $hasSelectedNoneComponents = grep { $_ eq $KEY_NAME_NONE } @{$selectedComponentKeyNames};
	my $isConflictingSelection = $hasSelectedMultipleComponents && ($hasSelectedAllComponents || $hasSelectedNoneComponents);

	return $isConflictingSelection ? 1 : 0;
}

1;