package LCM::Configuration::ValueChangeListeners::InstallUpdate::SignatureVerificationListenerBase;

use strict;
use warnings;
use Exporter;
use LCM::SAPDSigner;
use SDB::Install::Globals qw($gProductNameSHA $gFailedSignatureVerificationMessage $gSAPDSignerNotDetectedMessage $gFailedVerificationMessage);
use LCM::VerificationError qw(getNamedPipeErrorMessage);
use SDB::Install::MsgLst;
use LCM::Component::Installable::InstallationKitChecker;

sub new {
    return bless({}, shift());
}

sub onValueChange {
    my ($self, $selectedCmpsValue, $instconfig) = @_;
    return 1 if !$instconfig->getValue('VerifySignature');
    $self->resetVerificationErrors();

    if (!$self->checkSelectedComponentsFiles($instconfig)){
        $instconfig->getErrMsgLst()->addError("\nFor security reasons pipe files are not allowed.");
        $instconfig->setNoRetry("SelectedComponents", 1);
        return 0;
    }

    my $filePermissionsVerificationSuccess = $self->verifyComponentFilesPermissions($instconfig);
    return 1 if ($self->verifyComponentAuthenticity($instconfig) && $filePermissionsVerificationSuccess); 
    return 0 if !$instconfig->canIgnoreVerificationFailure();

    if ($instconfig->getIgnore('verify_signature')) {
        $instconfig->getMsgLst()->addMessage("Ignoring signature check errors due to --ignore command line switch");
        return 1;
    }

    return $self->handleVerificationFailure($instconfig);
}

sub verifyComponentAuthenticity {
    my ($self, $instconfig) = @_;
    my $sapdsigner = LCM::SAPDSigner->getInstance();
    if (!defined $sapdsigner) {
        $self->addVerificationError(LCM::VerificationError->new(undef,$gSAPDSignerNotDetectedMessage));
        return 0;
    }
    $self->handleSapdsignerLogging($sapdsigner, $instconfig);

    my $componentManager = $instconfig->getComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();
    my @componentSetToAuthenticate = grep {$_->isSigned()} @$selectedComponents;
    return 1 if (!@componentSetToAuthenticate);
    my @componentNames = map {$_->getComponentName()} @componentSetToAuthenticate;
    $instconfig->getMsgLst()->addMessage("Verifying authenticity of components: " . join (', ', @componentNames));

    my $rc = $sapdsigner->authenticateComponentSet(\@componentSetToAuthenticate);
    $self->addVerificationErrorList($sapdsigner->getAuthenticationErrors());
    return $rc;
}

sub checkSelectedComponentsFiles {
    my ($self, $instconfig) = @_;
    my $componentManager = $instconfig->getComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();
    $instconfig->getMsgLst()->addProgressMessage("Verifying files...");

    my $rc = 1;
    my $errorList = SDB::Install::MsgLst->new();
    foreach my $component (@{$selectedComponents}){
        if (!$component->checkFilesForPipes($errorList)) {
            my $cmpName = $component->getComponentName();
            $instconfig->getErrMsgLst()->addError("$cmpName:", $errorList);
            $rc = 0;
        }
    }
    return $rc;
}

sub handleSapdsignerLogging {
    my ($self, $sapdsigner, $instconfig) = @_;
    $sapdsigner->setMsgLstContext($instconfig->getMsgLstContext());
}

sub handleVerificationFailure {
    my ($self, $instconfig) = @_;
    ...
}

# For GUI and SLP
sub getFailedVerificationMessage {
    my ($self) = @_;
    my $warningMessage;
    my $authenticationErrors = $self->getVerificationErrors();
    my $hasAuthenticityErrors = grep{ $_->isVerificationAuthenticityError()} @{$authenticationErrors};
    my $failuresNumber = scalar(@{$authenticationErrors});
    if ($failuresNumber > 1) {
        my $failedComponents = join("\n", map { $_->getMultipleFailedComponentsMessage() } @{$authenticationErrors});
        $warningMessage  = $hasAuthenticityErrors ? "Authenticity verification" : "Verification";
        $warningMessage .= $warningMessage." of the following components failed:\n\n$failedComponents";
    } elsif( $failuresNumber == 1) {
        $warningMessage = $authenticationErrors->[0]->getSingleFailedComponentMessage();
    }
    $warningMessage .= $hasAuthenticityErrors ? "\n\n$gFailedSignatureVerificationMessage" : "\n\n$gFailedVerificationMessage";
    return $warningMessage;
}

sub getVerificationErrors{
    my ($self,$sapdsigner) = @_;
    return $self->{errors} // [];
}

sub verifyComponentFilesPermissions{
    my($self,$instconfig) = @_;
    my $componentManager = $instconfig->getComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();

    my $hasError = 0;
    foreach my $component(@$selectedComponents){
        my $verificationMsgLst = new SDB::Install::MsgLst();
        if(!$component->verifyFilesPermissions($verificationMsgLst)){
            chomp(my $error = ${$verificationMsgLst->getMsgLstString('')});
            $self->addVerificationError(LCM::VerificationError->new($component->getComponentName(), $error, 0));
            $hasError = 1;
        }
    }
    return !$hasError;
}

sub addVerificationError{
    my ($self,$error) = @_;
    $self->{errors} = [] if (!defined $self->{errors});
    push(@{$self->{errors}}, $error);
}

sub addVerificationErrorList{
    my ($self,$errorList) = @_;
    $self->{errors} = [] if (!defined $self->{errors});
    push(@{$self->{errors}}, @$errorList);
}

sub resetVerificationErrors{
    my $self  = shift;
    $self->{errors} = [];
    my $sapdsigner = LCM::SAPDSigner->getInstance();
    $sapdsigner->clearAuthenticationErrors() if defined $sapdsigner;
}

1;
