package LCM::Configuration::ValueChangeListeners::Uninstall::RecomputeCredentialsListener;

use strict;

use SDB::Install::System qw(isAdmin);
use SDB::Install::Configuration qw($bool_false_pattern);
use SDB::Install::Globals qw($gHostRoleAcceleratorWorker $gHostRoleAcceleratorStandby $gKeynameEngine
                             $gKeynameClient $gKeynameEs $gKeynameXS2 $gKeynameRDSync $gKeynameStreaming);
use LCM::Utils::RemoveHosts qw(IsRemovingRemoteHost IsRemovingHostWithRole);
use LCM::Utils::RemoveRoles qw(IsRemovingLocalRole IsRemovingRemoteRole IsRemovingRole);

sub new { return bless({}, shift()); }

sub onValueChange {
    my ($self, undef, $configuration) = @_;

    my $componentManager = $configuration->getSystemComponentManager();
    if($componentManager->isComponentSelected($gKeynameXS2)){
        $self->{_tenantDB} = $configuration->retrieveXsAdminLoginParam('runtime-database-name');
    }

    my $shallSkipSidadmCredentials = $self->_areSidadmCredentialsNeeded($configuration) ? 0 : 1;
    $configuration->setSkip('Password', $shallSkipSidadmCredentials);

    my $shallSkipASECredentials = $self->_areASECredentialsNeeded($configuration) ? 0 : 1;
    $configuration->setSkip('AcceleratorUser', $shallSkipASECredentials);
    $configuration->setSkip('AcceleratorPassword', $shallSkipASECredentials);

    my $shallSkipDBCredentials = $self->_areDBCredentialsNeeded($configuration) ? 0 : 1;
    $configuration->setSkip('SystemUser', $shallSkipDBCredentials);
    $configuration->setSkip('SQLSysPasswd', $shallSkipDBCredentials);

    my $shallSkipTenantDBCredentials = $self->_areTenantDBCredentialsNeeded($configuration) ? 0 : 1;
    $self->_enableTenantDBCredentials($configuration) if $shallSkipDBCredentials && ! $shallSkipTenantDBCredentials;

    my $shallSkipRootCredentials = $self->_areRootCredentialsNeeded($configuration) ? 0 : 1;
    $configuration->setSkip('RootPassword', 1); # checkRootUser will unskip if needed
    $configuration->setSkip('RootUser', $shallSkipRootCredentials);

    my $shallSkipSHACredentials = $self->_areSHACredentialsNeeded($configuration) ? 0 : 1;
    $configuration->setSkip('HostagentUserName', $shallSkipSHACredentials);
    $configuration->setSkip('HostagentPassword', $shallSkipSHACredentials);

    return 1;
}

sub _isXS2inTenantDB {
    my ($self) = @_;
    return $self->{_tenantDB} ? 1 : 0;
}

sub _areTenantDBCredentialsNeeded {
    my ($self, $configuration, $tenantDB) = @_;
    my $componentManager = $configuration->getSystemComponentManager();

    return 0 if($componentManager->isComponentSelected($gKeynameEngine)); # If the server is selected, no DB credentials are needed
    return 0 if(! $componentManager->isComponentSelected($gKeynameXS2));
    return $self->_isXS2inTenantDB();
}

sub _enableTenantDBCredentials {
    my ($self, $configuration) = @_;
    return if ! $self->{_tenantDB};

    $configuration->setSkip('TenantUser', 0);
    $configuration->setSkip('SQLTenantUserPassword', 0);

}

sub _areDBCredentialsNeeded {
    my ($self, $configuration) = @_;
    my $componentManager = $configuration->getSystemComponentManager();

    return 0 if($componentManager->isComponentSelected($gKeynameEngine)); # If the server is selected, no DB credentials are needed
    return 1 if($componentManager->isComponentSelected($gKeynameEs));
    return 1 if($componentManager->isComponentSelected($gKeynameRDSync));
    return 1 if($componentManager->isComponentSelected($gKeynameStreaming));
    return 1 if($self->_isRemovingASEHost($configuration));
    return 1 if($self->_isRemovingASERole($configuration));
    return 1 if($componentManager->isComponentSelected($gKeynameXS2) && ! $self->_isXS2inTenantDB());
    return 0;
}

sub _areASECredentialsNeeded {
    my ($self, $configuration) = @_;
    return 1 if($self->_isRemovingASEHost($configuration));
    return 1 if($self->_isRemovingASERole($configuration));
    return 0;
}

sub _areRootCredentialsNeeded {
    my ($self, $configuration) = @_;
    my $componentManager = $configuration->getSystemComponentManager();

    return 0 if($configuration->isUseSAPHostagent());
    return 0 if(!$configuration->isDistributedSystem());
    return 1 if(IsRemovingRemoteHost($configuration));
    return 1 if(IsRemovingRemoteRole($configuration));
    return 1 if($componentManager->isComponentSelected($gKeynameClient)); # HDB client must be unregistered from remote hosts
    return 1 if($componentManager->isComponentSelected($gKeynameEngine)); # If the server is selected, remote hosts need to be deleted
    return 0;
}

sub _areSidadmCredentialsNeeded {
    my ($self, $configuration) = @_;

    return 1 if(IsRemovingLocalRole($configuration));
    return 1 if(IsRemovingRemoteRole($configuration));
    return 0 if($self->_areSHACredentialsNeeded($configuration));
    return 1 if(!isAdmin());
    return 0 if(!$configuration->isDistributedSystem());
    return $configuration->isUseSAPHostagent();
}

sub _areSHACredentialsNeeded {
    my ($self, $configuration) = @_;
    my $componentManager = $configuration->getSystemComponentManager();

    return 0 if(!$configuration->isUseSAPHostagent());
    return 0 if(!$configuration->isDistributedSystem());
    return 1 if($componentManager->isComponentSelected($gKeynameEngine));
    return IsRemovingRemoteHost($configuration) ? 1 : 0;
}

sub _isRemovingASERole {
    my ($self, $configuration) = @_;
    return 1 if(IsRemovingRole($configuration, $gHostRoleAcceleratorStandby));
    return 1 if(IsRemovingRole($configuration, $gHostRoleAcceleratorWorker));
    return 0;
}

sub _isRemovingASEHost {
    my ($self, $configuration) = @_;
    return 1 if(IsRemovingHostWithRole($configuration, $gHostRoleAcceleratorStandby));
    return 1 if(IsRemovingHostWithRole($configuration, $gHostRoleAcceleratorWorker));
    return 0;
}

1;
