package LCM::Configuration::ValueChangeListeners::Update::SIDListener;

use File::Spec;
use SDB::Install::SysVars qw ($isWin $path_separator);
use LCM::Configuration::ValueChangeListeners::Update::SIDHelper;
use SDB::Install::System qw(isSidadmin isAdmin);
use SDB::Common::Utils qw(trim);
use LCM::PersistencyUtils qw (existsHdblcmPendingUpdate);

sub new {
	my ($class) = @_;

	my $self = {};
	bless $self, $class;

	return $self;
}

sub onValueChange {
	my ($self, $value, $instconfig) = @_;

# Call to defineLog is needed in order to remain consistent in terms of log directories names in update/installation scenarios
# where the SID is not known in advance. For additional information see Bug 65517
	my $applicationContext = LCM::App::ApplicationContext::getInstance();
	my $action = $instconfig->getAction();
	my $check_only = ( defined $instconfig ) ? $instconfig->isCheckOnly() : 0;
	$applicationContext->getApplication()->defineLog("hdblcm_$action", 'hdblcm', $check_only, $value, 1);

	$instconfig->{current_sid} = $value;

	if ( defined $instconfig->{params}->{Password} ) {
		if (! LCM::Configuration::ValueChangeListeners::Update::SIDHelper::InitConfigUsername($instconfig, $value)) {
			$instconfig->setNoRetry('SID',1);
			return 0;
		}
	}

    my ($target, $srcManifest) = $self->_getTargetAndManifest($value, $instconfig);
    if (defined $target) {
		$instconfig->setDefault('ClientPath', File::Spec->catfile($target, $value, 'hdbclient'));
		$instconfig->setDefault('StudioPath', File::Spec->catfile($target, $value, 'hdbstudio'));
		$instconfig->setDefault('XsEaDataPath', File::Spec->catfile($target, $value, 'xs', 'app_working'));

		if (!$isWin) {
			$instconfig->{params}->{Target}->{value} = $target;
		}
    }

    if (defined $srcManifest) {
        # if start rev < 90, force http communication for SAPControl
        if (new SDB::Install::Version ('1','00','90','00')->isNewerThan ($srcManifest->getVersionObject ())){
            $instconfig->setValue ('SAPControlUseHttp', 1);
        }
        else{
            $instconfig->setValue ('SAPControlUseHttp', 0);
        }
	}

	my $isRootUserExecution = isAdmin();
	my $isSidAdmUserExecution = (isSidadmin($value)) ? 1 : 0;

	if ((not $isRootUserExecution) and (not $isSidAdmUserExecution)) {
		require SDB::Install::NewDBUser;
		my $currentUser = new SDB::Install::User();
		my $username = $currentUser->getname();
		my $action = $instconfig->getAction();
		my $msg = "Can not perform $action on system '$value' with user '$username'. ";

		require SDB::Install::Configuration::NewDB;
		my $getHandler = new SDB::Install::Configuration::NewDB();
		my @installedSIDs = $getHandler->getNewDBInstallations( {}, [] );
		my $systemsCount = scalar @installedSIDs;

		if ($systemsCount > 1) {
			$msg .= "Either select system with system administrator '$username' or start hdblcm as root user.";
		} else {
			$msg .= "Start hdblcm as root user.";
		}

		$instconfig->PushError($msg);
		return undef;
	}

	my $mediumCmpMgr = $instconfig->getComponentManager();
	$mediumCmpMgr->setSidAdmUserExecution($isSidAdmUserExecution) if defined $mediumCmpMgr;
	my $systemCmpMgr = $mediumCmpMgr->getSystemComponentManager();
	$systemCmpMgr->setSidAdmUserExecution($isSidAdmUserExecution) if defined $systemCmpMgr;

	if (! $instconfig->detectInstalledComponents($value)){
		return undef;
	}

	if (! $instconfig->initializeSelectedComponents()) {
		$instconfig->setNoRetry('SID',1);
		return undef;
	}

	if (!existsHdblcmPendingUpdate($value) || !$instconfig->_handleInitializationOfPendingConfiguration($value)) {
		$instconfig->setSkip('ContinueUpdate', 1);
	}
	return 1;
}

sub _getTargetAndManifest {
    my ($self, $value, $instconfig) = @_;

    my $target = undef;
    my $manifest = undef;
    my $sapSys = $instconfig->getCollectSAPSystems()->{$value};

    if (defined $sapSys) {
        $target = $sapSys->get_target();
        my $params = $instconfig->{params};
        my $instance;
        $instance = $sapSys->getNewDBInstances()->[0];
        if (defined $instance && !$instance->exists_remote_host()) {
            $params->{Scope}->{skip} = 1;
        }
        
        if (!defined $target || length(trim($target)) == 0 ) {
            my $globalSidDir = $isWin ? $sapSys->getUsrSapSid() : $sapSys->get_globalSidDir();
            $target = substr ($globalSidDir, 0, -4);
        }

        $manifest = $sapSys->getManifest ();
    }
    
    return ($target, $manifest);
}

1;
