package LCM::Configuration::ValueChangeListeners::Update::SelectedComponentsListener;
use strict;

use LCM::Component;
use SDB::Install::Globals qw ( $gOptionOptimizedUpdate $gOptionStandardUpdate $gKeynameClient $gKeynameEngine $gProductNameClient $gProductNameEngine $gKeynameEngine $gKeynameInstaller $gKeynameLMStructure
                               $gKeynameStreaming $gKeynameEs $gKeynameAccelerator $gKeynameRDSync $gKeynameXS2 $gKeynameLSS);
use SDB::Install::Configuration qw($bool_true_pattern $bool_false_pattern) ;
use base qw ( LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedComponentsListenerBase );
use experimental qw (smartmatch);
use SDB::Install::System;

sub new {
	my $self = shift->SUPER::new(@_);
	return $self;
}

sub onValueChange {
	my ($self, $selectedCmpsValue, $instconfig) = @_;

	if(! $self->validateMultiDBConversionSupport($instconfig,$selectedCmpsValue)) {
		return undef;
	}

	if (! $self->validateMultiDBSupport($instconfig,$selectedCmpsValue)) { # perform this validation first due to bug 100295
		return undef;
	}

	my $isClientInstalled = $instconfig->isComponentInstalled($gKeynameClient);
	my $hideClientPath = $isClientInstalled ? $isClientInstalled :  $instconfig->_isComponentAvailable($gKeynameClient);
	$instconfig->setHidden('ClientPath', $hideClientPath);

	if (! $self->SUPER::onValueChange($selectedCmpsValue, $instconfig)) {
		return undef;
	}

	if (!$self->checkDowngradedComponentsSidadm($instconfig)) {
		return undef;
	}

	if ($self->_isXs2SelectedForInstall($instconfig) && ! $self->_isCryptoProviderValidForXs2($instconfig)) {
		return undef;
	}

	my @selectedComponents = split( /\s*,\s*/, $selectedCmpsValue );

	my $isPendingUpdate = $instconfig->isPendingUpdate();
	my $hasAddHostsPersistedValue = defined $instconfig->getPersistedValue('AddHosts') ? 1 : 0;
	my $hasAddRolesPersistedValue = defined $instconfig->getPersistedValue('AddRoles') ? 1 : 0;
	my $shallEnableAddHosts = $self->_shallEnableAddHosts($instconfig) || $isPendingUpdate && $hasAddHostsPersistedValue;
	my $shallEnableAddRoles = $self->_shallEnableAddHosts($instconfig) || $isPendingUpdate && $hasAddRolesPersistedValue;

	my $applicableOnInstallComponentsWarning = 'It is applicable only when installing components.';
	$instconfig->setSkip('AddHosts', ! $shallEnableAddHosts, $applicableOnInstallComponentsWarning);

	if( defined $instconfig->{params}->{AddRoles} ){
	   $instconfig->setSkip('AddRoles', ! $shallEnableAddRoles, $applicableOnInstallComponentsWarning);
	}

# Note: this line is added in order not to introduce additional listener for the web configuration
	my $shallSkipExistingHosts = $self->_shallEnableAddHosts($instconfig) ? 0 : 1;	
	$instconfig->setSkip('ExistingHosts', $shallSkipExistingHosts);
	$self->_unskipHostnameForClient(\@selectedComponents, $instconfig);
	$self->_handleUpdateExecutionModeDefaultValue($instconfig);
	if (!$instconfig->isSkipped('ReferenceDataPath') && !$instconfig->_updateReferenceDataPathParam()){
        return undef;
    }

	return undef if (!$self->_validatePendingComponents($instconfig));
	return undef if (!$self->_handleSelectionOfResidentInstaller(\@selectedComponents, $instconfig));
	$self->_handleDefaultTenantUserCredentials($instconfig);
	$self->handleSelectionOfLSS($instconfig);
	return 1;
}


sub _validatePendingComponents {
	my ($self, $instconfig) = @_;
	my $mcm = $instconfig->getComponentManager();

	return undef if (!defined $mcm);

	for my $selectedComponent (@{$mcm->getSelectedComponents()}) {
		next if (! $selectedComponent->hasPersistenceFile($instconfig));

		my $persistenceXML = $selectedComponent->getPersistenceXMLObject($instconfig);

		next if (!defined $persistenceXML);

		if ($persistenceXML->existsMandatoryValidationError()) {
			$instconfig->getMsgLst()->addError($persistenceXML->getErrorString());
			$instconfig->getErrMsgLst()->addError("Error validating persistence file '" . $persistenceXML->getFilePath() . "'.");
			$instconfig->getErrMsgLst()->addError("Correct the persistence file and try again.");
			$instconfig->{params}->{SelectedComponents}->{no_retry} = 1;
			return undef;
		}
	}

	my $persistenceManager = $instconfig->getPersistenceManager();

	return undef if (!$persistenceManager->validateDetectedServerKitVersion());

	my $canValidateComponentVersions = $persistenceManager->can('validateNonServerComponentVersions');

	return 1 if (!$canValidateComponentVersions);
	return $persistenceManager->validateNonServerComponentVersions();
}

sub _unskipHostnameForClient{
	my ($self, $selectedComponents, $instconfig) = @_;
	
	my $clientComponent = $instconfig->getComponentManager()->getComponentByKeyName($gKeynameClient);
    if (!defined $clientComponent){
        return 1;
    }
	
	my $clientBatchKey = $clientComponent->getComponentBatchKey();
	my $hostnameParam = $instconfig->{params}->{'HostName'};
	my $systemComponentManager= $instconfig->getSystemComponentManager();
	my $isClientInstalled = $systemComponentManager->isComponentAvailable($gKeynameClient);
	my $isComponentSelected = grep( {$clientBatchKey eq $_} @$selectedComponents);
	
	$hostnameParam->{'skip'} = $isClientInstalled ||  ( ! $isComponentSelected );
}

sub _handleSelectionOfResidentInstaller {
	my ($self, $selectedComponents, $instconfig) = @_;

	my $residentInstallerComponent = $instconfig->getComponentManager()->getComponentByKeyName($gKeynameInstaller);
    if (!defined $residentInstallerComponent){
        return 1;
    }

	my $residentInstallerBatchKey = $residentInstallerComponent->getComponentBatchKey();
	if (grep {$residentInstallerBatchKey eq $_} @$selectedComponents) {
		if (scalar @$selectedComponents > 1) {
			$instconfig->PushError ("\'$residentInstallerBatchKey\' should not be provided with other components !");
			return undef;
		}

		if (not $self->_isResidentInstallerCompatibleWithServer($instconfig)) {
			my $errMsg = $self->_createErrorMsg($instconfig);
			$instconfig->PushError ($errMsg);
			return undef;
		}

		$residentInstallerComponent->selectComponent();

		if(! $self->_checkSignatureFile($instconfig)){
			return undef;
		}

		$self->_unselectLmStructureComponent($instconfig);
		$instconfig->_recomputeRequiredParameters();
	}

	return 1;
}

sub _isResidentInstallerCompatibleWithServer {
	my ($self, $instconfig) = @_;

	my $residentInstallerComponent = $instconfig->{componentManager}->getComponentByKeyName($gKeynameInstaller);
	my $installedServerComponent = $instconfig->{systemComponentManager}->getComponentByKeyName($gKeynameEngine);
	if ($residentInstallerComponent->getManifest()->isCustomerReleaseBranch() 
		and (not $self->_areVersionsEqual($residentInstallerComponent, $installedServerComponent))){
		return undef;
	}

	return 1;
}

sub _areVersionsEqual{
	my ($self, $residentInstallerComponent, $installedServerComponent) = @_;
	my $residentInstallerVersion = $residentInstallerComponent->getManifest()->getVersionObject();
	my $serverVersion = $installedServerComponent->getManifest()->getVersionObject();
	
	return $residentInstallerVersion->isEqual($serverVersion);
}

sub _createErrorMsg {
	my ($self, $instconfig) = @_;

	my $residentInstallerComponent = $instconfig->{componentManager}->getComponentByKeyName($gKeynameInstaller);
	my $installedServerComponent = $instconfig->{systemComponentManager}->getComponentByKeyName($gKeynameEngine);
	my $residentIntstallerName = $residentInstallerComponent->getComponentName();
	my $residentIntstallerVersion = $residentInstallerComponent->getVersion();
	my $serverCmpName = $installedServerComponent->getComponentName();
	my $serverCmpVersion = $installedServerComponent->getVersion();
	my $errMsg = "Versions of detected $residentIntstallerName ($residentIntstallerVersion) " .
		 						"and installed $serverCmpName ($serverCmpVersion) are different !";

	return $errMsg;
}

sub _unselectLmStructureComponent {
	my ($self, $instconfig) = @_;
	my $lmStructureComponent = $instconfig->{componentManager}->getComponentByKeyName( $gKeynameLMStructure );
	
	if ( defined $lmStructureComponent and $lmStructureComponent->isComponentSelected()) {
		$lmStructureComponent->selectComponent(0);
	}
}

sub _shallEnableAddHosts {
	my ($self, $instconfig) = @_;

	return 0 if($instconfig->isScopeInstance());

	my $componentManager = $instconfig->getComponentManager();
	for my $componentKeyname ($gKeynameStreaming, $gKeynameEs, $gKeynameAccelerator, $gKeynameRDSync, $gKeynameXS2) {
		my $component = $componentManager->getComponentByKeyName($componentKeyname);

		return 1 if(defined($component) && $component->isComponentSelected() && ! $component->isUpdate());
	}
	return 0;
}

sub _handleUpdateExecutionModeDefaultValue {
	my ($self, $instconfig) = @_;
	my $componentsSupportingPhasesCount = 0;
	my $mcm = $instconfig->getMediumComponentManager();
	my $selectedComponents = $mcm->getSelectedComponents();

	for my $component (@{$selectedComponents}) {
		if ($component->supportsPhases()) {
			$componentsSupportingPhasesCount++;
		}
	}
	if ($componentsSupportingPhasesCount > 1 || $instconfig->isConvertToMultiDbRequired()) {
		$instconfig->setDefault('UpdateExecutionMode', $gOptionOptimizedUpdate);
	} else {
		$instconfig->setDefault('UpdateExecutionMode', $gOptionStandardUpdate);
	}
}

sub _handleDefaultTenantUserCredentials {
    my ($self,$instconfig) = @_;

    $self->SUPER::_handleDefaultTenantUserCredentials($instconfig);

    if ($instconfig->isConvertToMultiDbRequired()) {
        $self->_hideTenantCredentials($instconfig);
    }
}

sub _hideTenantCredentials {
    my ($self,$instconfig) = @_;

    $instconfig->{params}->{TenantUser}->{set_interactive} = 0;
    $instconfig->{params}->{TenantUser}->{hidden} = 1;
    $instconfig->{params}->{SQLTenantUserPassword}->{set_interactive} = 0;
    $instconfig->{params}->{SQLTenantUserPassword}->{hidden} = 1;
}

sub _isCryptoProviderValidForXs2 {
	my ($self, $configuration) = @_;
	my $instance = $configuration->getOwnInstance();
	my $globalIni = $instance->getGlobalIni();
	my $ssl = $globalIni->existsValue ('communication', 'ssl') ?
		$globalIni->getValue('communication', 'ssl') : 'off';
	my $iscMode = $configuration->getValueOrBatchValue('ISCMode');
# Don't do enything if ISC is set to encrypted before the update
# or --isc_mode is explicitly set to standard
	return 1 if ($iscMode eq 'standard' || $ssl =~ /on|systempki/);
# However if the ISC is off and --isc_mode=ssl check the crypto provider
	my $sslCryptoProvider = $globalIni->getValue('communication', 'sslCryptoProvider');
	if ($sslCryptoProvider eq 'openssl') {
		$configuration->AddError("SAP HANA XS Advanced Runtime cannot be installed on a system using OpenSSL for crypto-related operations. Check SAP Note 2093286 for instructions how to migrate to CommonCryptoLib.");
		return 0;
	}
	return 1;
}

# Override
sub _initISCModeDefaultValue {
	my ($self, $configuration) = @_;
	my $currentValue = $self->_getCurrentISCMode($configuration); 
	my $defaultValue = ($self->_isXs2SelectedForInstall($configuration)) ? 'ssl' : $currentValue;
	$configuration->setDefault('ISCMode', $defaultValue, 1);
}

# This should be called after the SUPER::onValueChange method
# has been called so that the componentManager is properly populated
sub _isXs2SelectedForInstall {
	my ($self, $configuration) = @_;
	my $componentManager = $configuration->getComponentManager();
	my $xsComponent = $componentManager->getComponentByKeyName($gKeynameXS2);
	my $isInstallingXs = (defined($xsComponent) && $xsComponent->isComponentSelected() && !$xsComponent->isUpdate());
	return ($isInstallingXs) ? 1 : 0;
}

sub _getCurrentISCMode {
	my ($self, $configuration) = @_;
	my $ownInstance = $configuration->getOwnInstance();
	my $globalIni = defined($ownInstance) ? $ownInstance->getGlobalIni() : undef;
	my $currentMode;
	if (defined($globalIni) && $globalIni->existsValue ('communication', 'ssl')){
		$currentMode = $globalIni->getValue ('communication', 'ssl');
	}
	else{
		$currentMode = 'off';
	}
	my $isEncrypted = defined($currentMode) && $currentMode ne 'off';

	return $isEncrypted ? 'ssl' : 'standard';
}
sub validateMultiDBSupport {
    my ($self,$instconfig, $selectedComponentsCsv) = @_;

	if ( $instconfig->isSingleTenantSystem() ) {
		return 1;
	}

    my $componentManager = $instconfig->getComponentManager();
    my @allComponents  = @{ $componentManager->getAllComponents() };

    for ( @allComponents ) {
        if ( (index ($selectedComponentsCsv, $_->getComponentBatchKey()) != -1) || $selectedComponentsCsv eq "all" ) {
        	if ( ! $instconfig->_checkIfComponentSupportsMultiDb( $_ ) ) {
        		return 0;
        	}
        }
    }

    return 1;
}

sub validateMultiDBConversionSupport {
    my ($self,$instconfig, $selectedComponentsCsv) = @_;
    my $mcm = $instconfig->getComponentManager();
    my $scm = $instconfig->getSystemComponentManager();
    if ($instconfig->getOwnInstance()->isMultiDb()) {
        return 1;
    }

    my $selectedComponentsValidValues = $instconfig->getValidValues('SelectedComponents');
    my @selectedComponents = ();
    for my $component (@{$mcm->getAllComponents()}) {
        next if !($component->getComponentBatchKey() ~~ @$selectedComponentsValidValues);
        if ((index ($selectedComponentsCsv, $component->getComponentBatchKey()) != -1) || $selectedComponentsCsv eq "all") {
            push(@selectedComponents, $component);
        }
    }
    my @selectedBatchkeys = map { $_->getComponentBatchKey() } @selectedComponents;
    my $server = $instconfig->getComponentManager()->getComponentByKeyName($gKeynameEngine);
    if (!defined $server || !($server->getComponentBatchKey() ~~ @selectedBatchkeys)){
        return 1;
    }

    # At this point we know, that a conversion will happen
    $self->_disableStandardUpdateExecutionMode($instconfig);
    $self->_handleSystemDBSQLPassword($instconfig);

    my @installedComponents = @{ $scm->getAllComponents() };
    for my $component (@installedComponents) {
        if ($component->isFeatureUnsupported('convert_to_multidb')) {
            my $batchKey = $component->getComponentBatchKey();
            if($batchKey ~~ @selectedBatchkeys) {
                my $selectedComponent = $mcm->getComponentByBatchKey($batchKey);
                if (! $selectedComponent->isFeatureUnsupported('convert_to_multidb')) {
                    next;
                }
            }
            # An installed components doesn't support MultiDB conversion, neither is a newer version selected,
            # wchich supports it. In this case the update is impossible.
            if( LCM::App::ApplicationContext::getInstance()->getMode() eq "CLI" ) {
                # If we are in the console application, exit hdblcm
                $instconfig->setNoRetry('SelectedComponents', 1);
            }

            my $errorMsg = "During update the system will be converted to multitenant database containers. Installed component '%s' version %s doesn't support this conversion. You need to provide a newer version, which supports it.";
            $instconfig->setErrorMessage(sprintf($errorMsg, $component->getComponentName(), $component->getManifest()->getVersion()));
            return 0;
        }
    }
    return 1;
}

sub _disableStandardUpdateExecutionMode {
    my ($self,$instconfig) = @_;
    $instconfig->setDefault('UpdateExecutionMode', $gOptionOptimizedUpdate);
}

sub _handleSystemDBSQLPassword{
    my ($self,$instconfig) = @_;
    my $shouldSkip = $self->_shouldSkipSystemDBSQLPassword($instconfig);

    if( $shouldSkip ){
    	$instconfig->setSkip('SystemDBSQLPassword',1);
    }
    if( ! $instconfig->isPrepareUpdate() && !$shouldSkip){
        $instconfig->setSkip('SystemDBSQLPassword',0);
    }
}

sub _shouldSkipSystemDBSQLPassword {
	my ($self, $instconfig) = @_;

	if ($instconfig->getBatchValue('SystemDBUseSingleDBUserPassword')) {
		return 1;
	}
	return 0;
}

sub handleSelectionOfLSS {
    my ($self, $config) = @_;
    if ($config->_isComponentSelected($gKeynameLSS)){
        $config->initLSSUserID($config->getFreeOsUserID(undef, $config->getReservedUserIDs()));
        $config->initLSSGroupID($config->getFreeGroupID(undef, $config->getReservedGroupIDs()));
    }
}

sub checkDowngradedComponentsSidadm {
	my ($self, $instconfig) = @_;
	if (SDB::Install::System::isAdmin()) {
		return 1;
	}
	my $mcm = $instconfig->getComponentManager();

	for my $component (@{$mcm->getSelectedComponents()}) {
		if ($component->requiresRootPrivilege() && $component->isDowngrade()) {
			my $cmpKeyame      = $component->getComponentKeyName();
			my $cmpProductName = $component->getComponentKeyCaptionByKeyName($cmpKeyame);
			$instconfig->getErrMsgLst()->addError("Downgrade of the $cmpProductName can only be done as root.");
			return 0;
		}
	}
	return 1;
}

1;
