package LCM::Configuration::Web::DownloadComponentsConfiguration;

use strict;
use File::Spec;
use File::Basename;
use LCM::Installer;
use LCM::Configuration::ParametersCreator;
use SDB::Common::Utils qw(createXMLParser);
use LCM::FileUtils qw(MyRealPath isEmptyDirectory);
use SDB::Install::SysVars qw($currentPlatformName);
use LCM::Configuration::GenericStackAny qw($ini_section_general);
use parent qw(LCM::Configuration::ComponentArchivesConfiguration);
use SDB::Install::Saphostagent qw(getSHAVersion getSaphostagentPpmsId);
use LCM::Configuration::ValueChangeListeners::DownloadComponents::DownloadOnHANAListener;
use LCM::Configuration::ValueChangeListeners::DownloadComponents::PersistWebServiceURLListener;
use LCM::Configuration::ValueChangeListeners::DownloadComponents::WebServiceConnectionListener;
use LCM::Configuration::ValueChangeListeners::DownloadComponents::AuthorizationProxyConnectionListener;
use LCM::Configuration::ValueChangeListeners::DownloadComponents::NoAuthorizationProxyConnectionListener;
use SDB::Install::Globals qw($gFlavourExpress $gShortProductNameXS2 $gKeynameXS2 $gFlavourCockpit $gKeynameCockpitStack);

# There are different proxy URLS for internal maintenance and test systems
my $DEFAULT_PROXY_URL_SWDC = 'https://apps.support.sap.com/sap/public/bsp/sap/hana_lm/';
my $DEFAULT_PROXY_URL_AMAZON = 'https://d149oh3iywgk04.cloudfront.net/hanaexpress/resources/';
my $CERTIFICATE_FILES_CANDIDATES = [
    '/etc/pki/tls/certs/ca-bundle.crt',   # Fedora/RHEL
    '/etc/ssl/certs/',                    # SUSE
    '/etc/ssl/ca-bundle.pem',             # SUSE
    '/etc/pki/tls/cacert.pem',            # ELEC
    '/etc/ssl/certs/ca-certificates.crt', # Debian/Ubuntu/Gentoo etc.
];

sub new {
    my $class = shift();
    my $self = $class->SUPER::new(@_);
    my $order = 0;

    $self->{params} = {
        'SID'                     => $self->getParamSID($order++, $ini_section_general),
        'CAPath'                  => $self->_getParamCAPath($order++, $ini_section_general),
        'WebServiceProxyURL'      => $self->_getParamWebServiceProxyURL($order++, $ini_section_general),
        'DownloadOnHANA'          => $self->_getParamDownloadOnHANA($order++, $ini_section_general),
        'ProxyHost'               => $self->_getParamProxyHost($order++, $ini_section_general),
        'ProxyPort'               => $self->_getParamProxyPort($order++, $ini_section_general),
        'UseProxyAuthorization'   => $self->_getParamUseProxyAuthorization($order++, $ini_section_general),
        'ProxyUser'               => $self->_getParamProxyUser($order++, $ini_section_general),
        'ProxyPassword'           => $self->_getParamProxyPassword($order++, $ini_section_general),
        'OverwriteExtractDir'     => GetParamOverwriteExtractDir($order++, $ini_section_general, "Delete Existing Files in the Specified Directories"),
        'DownloadDir'             => $self->_getParamDownloadDir($order++, $ini_section_general),
        'ExtractAfterDownload'    => $self->_getParamExtractAfterDownload($order++, $ini_section_general),
        'ComponentArchivesPath'   => GetParamComponentArchivesPath($order++, $ini_section_general),
        'ExtractTempDir'          => GetParamExtractTempDir($order++, $ini_section_general),
        'SapcarLocation'          => GetParamSapcarLocation($order++, $ini_section_general),
        'TarExecutableLocation'   => GetParamTarExecutableLocation($order++, $ini_section_general),
        'UnrarExecutableLocation' => GetParamUnrarExecutableLocation($order++, $ini_section_general),
        'SMPUser'                 => $self->_getParamSMPUser($order++, $ini_section_general),
        'SMPUserPassword'         => $self->_getParamSMPUserPassword($order++, $ini_section_general),
        'InstalledComponents'     => $self->_getParamInstalledComponents($order++, $ini_section_general),
        'OSPlatform'              => $self->_getParamOSPlatform($order++, $ini_section_general),
        'UpdatableComponents'     => $self->_getParamUpdatableComponents($order++, $ini_section_general),
        'InstallableComponents'   => $self->_getParamInstallableComponents($order++, $ini_section_general),
        'SelectedComponents'      => $self->_getParamSelectedComponents($order++, $ini_section_general),
    };
    $self->_addListeners();
    return $self;
}

sub InitDefaults {
    my $self = shift();

    return undef if(!$self->SUPER::InitDefaults(@_));

    my $installer = new LCM::Installer();
    my $systemComponentManager = $installer->getOwnSystemComponentManager();
    my $hdblcmRuntimeDir = $installer->getRuntimeDir();
    my $targetSidDir = MyRealPath(File::Spec->catfile($hdblcmRuntimeDir, '..', '..'));
    my $caPath = $self->_getDefaultCALocation();
    my $httpProxyEnvironment = $ENV{'http_proxy'};
    my ($protocol, undef, $proxyHost, $proxyPort) = ($ENV{'http_proxy'} =~ m!^([a-z]+://)(.*:.*\@)?([.a-zA-Z0-9-]+):?(\d+)?/?$!);

    $self->setValue('CAPath', $caPath) if(defined($caPath));
    $self->setHidden('ComponentArchivesPath', 1);
    $self->setDefault('ProxyPort', $proxyPort);
    $self->setDefault('OverwriteExtractDir', 1);
    $self->setDefault('ProxyHost', sprintf('%s%s', $protocol, $proxyHost));
    $self->setDefault('DownloadDir', File::Spec->catfile($targetSidDir, 'downloads'));
    $self->setDefault('ExtractTempDir', File::Spec->catfile($targetSidDir, 'downloads', 'extracted'));
    $self->setMandatory('ExtractTempDir', 0);
    $self->setValue('SID', $installer->getSid());
    $self->setHidden('SID', 1);
    $self->_initalizeInstalledComponents($systemComponentManager, $targetSidDir);

    for my $parameterId ('SapcarLocation', 'TarExecutableLocation', 'UnrarExecutableLocation'){
        next if(!exists($self->{params}->{$parameterId}));

        my $defaultPath = $self->getDefault($parameterId);
        if($self->SUPER::_checkExecutableLocation($parameterId, $defaultPath, '')){
            $self->setValue($parameterId, $defaultPath);
            $self->setHidden($parameterId, 1);
        } else {
            $self->resetError();
        }
    }

    my $sapSystem = $self->getSAPSystem();
    my $systemManifest = $sapSystem->getManifest();
    my $isHANAExpress = ($systemManifest->getHANAFlavour() eq $gFlavourExpress);
    my $defaultProxyURL = $isHANAExpress ? $DEFAULT_PROXY_URL_AMAZON : $DEFAULT_PROXY_URL_SWDC;
    my $userConfiguration = $sapSystem->getUserConfig() || {};
    my $webServiceUrl = $userConfiguration->{web_service_proxy_url} || $defaultProxyURL;
    my $shallSkipSMPCredentials = $isHANAExpress ? 1 : 0;

    $self->setValue('WebServiceProxyURL', $webServiceUrl);
    $self->setSkip('SMPUser', $shallSkipSMPCredentials);
    $self->setSkip('SMPUserPassword', $shallSkipSMPCredentials);

    return 1;
}

sub checkProxyHost {
    my ($self, $value) = @_;

    if (!defined($value) || length($value) == 0) {
        $self->getErrMsgLst()->addError("Address is empty" );
        return 0;
    }
    if ( $value !~ m!^https?://! ) {
        $self->getErrMsgLst()->addError("No communication protocol specified; valid protocols are 'http' and 'https'");
        return 0;
    }
    if ( $value =~ m!^(http://|https://)?([^.a-zA-Z0-9-])+! ) {
        $self->getErrMsgLst()->addError("Address contains an invalid symbol; valid symbols are 0-9, a-z, A-Z, '.', '-'.");
        return 0;
    }
    return 1;
}

sub checkProxyPort {
    my ($self, $value) = @_;

    if (!defined($value) || length($value) == 0 ) {
        $self->getErrMsgLst()->addError("Port is empty");
        return 0;
    }
    if ( !Scalar::Util::looks_like_number ( $value ) ) {
        $self->getErrMsgLst()->addError("Port is not a number");
        return 0;
    }
    if ($value < 1 || $value > 65535) {
        $self->getErrMsgLst()->addError("Port is not valid; valid values are: 1-65535");
        return 0;
    }
    return 1;
}

sub setExtractTempDir {
    my ($self, $value) = (shift(), shift());

    return 1 if (! $self->getValue('ExtractAfterDownload'));
    if($value eq ''){
        $self->getErrMsgLst()->addError('The required text field is empty');
        return 0;
    }
    return $self->SUPER::setExtractTempDir($value, @_);
}

sub checkExtractTempDir {
    my ($self, $pathValue) = (shift(), shift());
    my $parentDir = File::Spec->canonpath(dirname($pathValue));
    my $downloadDir = File::Spec->canonpath($self->getValue('DownloadDir'));

    return 1 if(!-e $parentDir && $parentDir eq $downloadDir);
    return $self->SUPER::checkExtractTempDir($pathValue, @_)
}

sub setDownloadDir {
    my ($self, $pathValue) = @_;

    return 1 if (! $self->getValue('DownloadOnHANA'));
    return 0 if (! $self->checkDownloadDir($pathValue));

    my $parentDir = MyRealPath(dirname($pathValue));
    my $extractDir = basename($pathValue);
    my $downloadDir = (-d $pathValue) ? MyRealPath($pathValue) : File::Spec->catdir($parentDir, $extractDir);

    $self->{params}->{DownloadDir}->{value} = $downloadDir;
    $self->{params}->{ComponentArchivesPath}->{value} = $downloadDir;

    return 1;
}

sub checkDownloadDir {
    my ($self, $pathValue) = @_;

    if($pathValue eq ''){
        $self->getErrMsgLst()->addError('The required text field is empty');
        return 0;
    }

    my $parentDir = dirname($pathValue);
    my $realPath = (-e $pathValue) ? MyRealPath($pathValue) : $pathValue;
    my $realParentDir = (-e $parentDir) ? MyRealPath($parentDir) : $parentDir;

    if(!File::Spec->file_name_is_absolute($pathValue)){
        $self->getErrMsgLst()->addError("Path '$pathValue' must be absolute");
        return 0;
    }
    if (-e $realPath && !-d $realPath){
        $self->getErrMsgLst()->addError("The path '$pathValue' must point to a directory");
        return 0;
    }
    if (!-d $realParentDir){
        $self->getErrMsgLst()->addError("Parent directory '$parentDir' does not exist");
        return 0;
    }
    if (!-d $realPath && ! -w $realParentDir){
        $self->getErrMsgLst()->addError("Cannot create directory '$pathValue'. Insufficient privileges");
        return 0;
    }
    return 1;
}

sub getProductName {
    return "Download Components From the SAP Support Portal";
}

sub getAction {
    return 'download_components';
}

sub getTimeoutValues {
    return [];
}

# Override
sub _checkExecutableLocation {
    my ($self, $paramId, $pathValue, $executable) = @_;

    $self->clearParameterWarnings($paramId);
    if (! -x $pathValue || ! -f $pathValue) {
        $self->addParameterWarning($paramId, "The path '$pathValue' does not point to an executable.");
    }
    return 1;
}

sub _addListeners {
    my ($self) = @_;

    $self->addParameterListener('DownloadOnHANA', new LCM::Configuration::ValueChangeListeners::DownloadComponents::DownloadOnHANAListener());
    $self->addParameterListener('DownloadOnHANA', new LCM::Configuration::ValueChangeListeners::DownloadComponents::WebServiceConnectionListener());
    $self->addParameterListener('WebServiceProxyURL', new LCM::Configuration::ValueChangeListeners::DownloadComponents::PersistWebServiceURLListener());
    $self->addParameterListener('ProxyPassword', new LCM::Configuration::ValueChangeListeners::DownloadComponents::AuthorizationProxyConnectionListener());
    $self->addParameterListener('UseProxyAuthorization', new LCM::Configuration::ValueChangeListeners::DownloadComponents::NoAuthorizationProxyConnectionListener());
}

sub _getDefaultCALocation {
    my ($self) = @_;

    for my $candidatePath (@{$CERTIFICATE_FILES_CANDIDATES}){
        my $path = (-l $candidatePath) ? readlink($candidatePath) : $candidatePath;

        return $path if(-e $path && -r $path);
    }
    return undef;
}

sub _initalizeInstalledComponents {
    my ($self, $systemComponentManager, $targetSidDir) = @_;
    my $systemFlavour = $systemComponentManager->getHANAFlavour();
    my $isCockiptInstallation = ($systemFlavour eq $gFlavourCockpit);
    my $components = $isCockiptInstallation ? [] : [ @{$systemComponentManager->getAllComponents()}, @{$systemComponentManager->getXs2ApplicationComponents()} ];
    my $shaVersionString = $self->_getSapHostAgentVersionString();
    my @result = ();

    push(@result, $shaVersionString) if(defined($shaVersionString));

    if($isCockiptInstallation){
        my ($release, $revision, $patchLevel) = ('2.0', '0', '0');
        my $cockpitStack = $systemComponentManager->getComponentByKeyName($gKeynameCockpitStack);

        if(defined($cockpitStack)){
            my $manifest = $cockpitStack->getManifest();
            $release = $manifest->getValue('release') || '2.0';
            $revision = $manifest->getValue('rev-number') || '0';
            $patchLevel = $manifest->getValue('rev-patchlevel') || '0';
        }

        my $fullVersion = join('.', $release, $revision, $patchLevel);
        push(@result, "73555000100200005745:fullversion=$fullVersion:revision=$revision:patch=$patchLevel");
    }

    for my $component (@{$components}){
        next if($component->isInternal());

        my $ppmsId = $component->getComponentId();
        my $manifest = $component->getManifest();
        my $release = $manifest->getValue('release') || '0';
        my $revision = int($manifest->getValue('rev-number'));
        my $patchLevel = int($manifest->getValue('rev-patchlevel')) || 0;
        my $fullVersion = join('.', $release, $revision, $patchLevel);
        push(@result, "$ppmsId:fullversion=$fullVersion:revision=$revision:patch=$patchLevel");
    }

    $self->setValue('InstalledComponents', join(',', @result));
}

sub _getSapHostAgentVersionString {
    my ($self) = @_;
    my $fullVersion = getSHAVersion();

    return undef if(!defined($fullVersion));

    my ($release, $revision, $patch) = split('\.', $fullVersion);
    my $ppmsId = getSaphostagentPpmsId();
    return sprintf('%s:fullversion=%s:revision=%s:patch=%s', $ppmsId, $fullVersion, $patch, $patch); # SWDC web service asks for the SAP Host Agent patch level, not SP level
}

sub _getParamWebServiceProxyURL {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'web_service_proxy_url',
        'type'              => 'string',
        'section'           => $section,
        'default'           => undef,
        'value'             => undef,
        'set_interactive'   => 0,
        'str'               => 'Web Service Proxy URL',
        'desc'              => 'Web Service Proxy URL',
        'mandatory'         => 1,
        'init_with_default' => 1
    };
}

sub _getParamInstalledComponents {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'installed_components',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'str'               => 'Installed Components',
        'mandatory'         => 0,
    };
}

sub _getParamOSPlatform {
    my ($self, $order, $section) = @_;
    return {
        'order'   => $order,
        'opt'     => 'os_platform',
        'type'    => 'string',
        'section' => $section,
        'value'   => lc($currentPlatformName),
        'str'     => 'OS Platform',
    };
}

sub _getParamSMPUser {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'smp_user',
        'opt_arg'         => '<name>',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'str'             => 'SAP Support Portal User Name',
        'desc'            => 'SAP Support Portal User Name',
        'set_interactive' => 1,
        'mandatory'       => 1,
    };
}

sub _getParamSMPUserPassword {
    my ($self, $order, $section) = @_;

    return {
        'order'     => $order,
        'opt'       => 'smp_user_password',
        'type'      => 'passwd',
        'opt_arg'   => '<password>',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'SAP Support Portal Password',
        'desc'      => 'SAP Support Portal Password',
        'log_value' => sub {'***';},
        'mandatory' => 1,
    };
}

sub _getParamDownloadOnHANA {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'download_on_hana',
        'opt_arg'           => '<true|false>',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 1,
        'set_interactive'   => 1,
        'str'               => 'Download Archives on the SAP HANA Host',
        'desc'              => 'Download Archives on the SAP HANA Host',
        'mandatory'         => 0,
        'init_with_default' => 1
    };
}

sub _getParamUpdatableComponents {
    my ($self, $order, $section) = @_;

    return {
        'order'   => $order,
        'opt'     => 'updatable_components',
        'type'    => 'string',
        'section' => $section,
        'value'   => undef,
        'str'     => 'Updatable Components',
    };
}

sub _getParamInstallableComponents {
    my ($self, $order, $section) = @_;

    return {
        'order'   => $order,
        'opt'     => 'installable_components',
        'type'    => 'string',
        'section' => $section,
        'value'   => undef,
        'str'     => 'Installable Components',
    };
}

sub _getParamSelectedComponents {
    my ($self, $order, $section) = @_;

    return {
        'order'   => $order,
        'opt'     => 'selected_components',
        'type'    => 'string',
        'section' => $section,
        'value'   => undef,
        'str'     => 'Selected Components',
    };
}

sub _getParamDownloadDir {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'download_dir',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Download Directory",
        'desc'              => "Download Directory",
        'mandatory'         => 0,
        'init_with_default' => 1,
    };
}

sub _getParamExtractAfterDownload {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'extract_after_download',
        'opt_arg'           => '<true|false>',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 1,
        'set_interactive'   => 1,
        'str'               => 'Extract Archives After Download',
        'desc'              => 'Extract Archives After Download',
        'mandatory'         => 0,
        'init_with_default' => 1
    };
}

sub _getParamCAPath {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'ca_path',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Certificate Authority Path",
        'desc'              => "Certificate Authority Path",
        'mandatory'         => 1,
        'init_with_default' => 1,
    };
}

sub _getParamProxyHost {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'proxy_host',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Proxy Host",
        'desc'              => "Proxy Host",
        'mandatory'         => 1,
        'init_with_default' => 1,
    };
}

sub _getParamProxyPort {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'proxy_port',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Proxy Port",
        'desc'              => "Proxy Port",
        'mandatory'         => 1,
        'init_with_default' => 1,
    };
}

sub _getParamUseProxyAuthorization {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'use_proxy_authorization',
        'opt_arg'           => '<true|false>',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 0,
        'set_interactive'   => 1,
        'str'               => 'Use Proxy Authorization',
        'desc'              => 'Use Proxy Authorization',
        'mandatory'         => 0,
        'init_with_default' => 1
    };
}

sub _getParamProxyUser {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'proxy_user',
        'opt_arg'         => '<name>',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'str'             => 'Proxy User',
        'desc'            => 'Proxy User',
        'set_interactive' => 1,
    };
}

sub _getParamProxyPassword {
    my ($self, $order, $section) = @_;

    return {
        'order'     => $order,
        'opt'       => 'proxy_user_password',
        'type'      => 'passwd',
        'opt_arg'   => '<password>',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'Proxy User Password',
        'desc'      => 'Proxy User Password',
        'log_value' => sub {'***';},
    };
}

1;