package LCM::Configuration::Web::GenericStackUpdateConfiguration;

use strict;
use LCM::Installer;
use LCM::Configuration::ParametersCreator qw(GetParamExistingHosts GetParamIgnore);
use LCM::Slpp::Update::Listeners::ComponentLocationsListener;
use LCM::Slpp::Update::MediumComponentManager;
use LCM::Slpp::Update::Listeners::IgnoreListener;
use LCM::Utils::ComponentActionStringGenerator::WebUpdate;
use SDB::Install::Configuration qw($bool_false_pattern $bool_true_pattern );
use SDB::Install::SysVars qw ( $path_separator );
use LCM::Configuration::GenericStackUpdateConfiguration qw($unableToContinueUpdateMsg);
use SDB::Install::SAPSystem qw (CollectSAPSystems @UPGRADE_STEP_NAMES);
use LCM::PersistencyUtils qw(existsHdblcmPendingUpdate existsHdbupdPendingUpdate);
use LCM::Component;
use LCM::Persistence::XMLDetector;

use base qw (LCM::Configuration::GenericStackUpdateConfiguration);

sub new{
	my $class = shift;
	my $self = $class->SUPER::new(@_);

	$self->setValue('RemoteExecution', 'saphostagent');
	# Hide SID parameter - system is preselected in case of a SLPP update
	$self->{params}->{SID}->{hidden} = 1;

	$self->_skipUnnededParameters();
	$self->_addSlpSpecificParameters();
# This is done in order to create propper ParameterAdapter for the InternalNetwork parameter
	$self->{params}->{InternalNetwork}->{valid_values} = [];
	$self->{params}->{ComponentDirs}->{str} = 'Component directory';

	return $self;
}

sub setSelectedXs2Applications {
	my ($self, $value) = @_;

	# Handle the case where there aren't any selected XSA Applications
	if (!$value) {
		$self->resetParam('SelectedXs2Applications');
		return 1;
	}

	my $mcm = $self->getComponentManager();
	my $xsComponents = $mcm->getXs2ApplicationComponents();
	my @validValues = map { $_->getComponentBatchKey() } @{$xsComponents};

	return undef if(!$self->checkSelectedXs2Applications($value));

	$value = ($value ne 'all') ? $value : join(',', @validValues);
	$self->{params}->{SelectedXs2Applications}->{value} = $value;
}

# 1. Ignore key 'check_signature_file' is not relevant - signature checks can be ignored
# only when the tool is running with root privileges, therefore we disable this key
# 2. Ignore key 'verify_signature' is irrelevant because this ignore is interactive in the Web UI
sub getIgnoreValues {
	my $self = shift();
	my $ignoreValues = $self->SUPER::getIgnoreValues(@_);
	my @filteredIgnoreValues = grep { $_ ne 'check_signature_file' && $_ ne 'verify_signature' } @{$ignoreValues};

	return \@filteredIgnoreValues;
}

sub setAutoAddXS2Roles {
	my ($self, $value) = @_;
	my $result = $self->SUPER::setAutoAddXS2Roles($value);

	if($result && $value =~ /$bool_true_pattern/){
		my $hostToRoleMap = $self->getAutoAssignXs2RolesMap();
		my @hostsWithAdditionalRoles = grep { length($hostToRoleMap->{$_}) > 0 } keys(%{$hostToRoleMap});
		my $resultString = join(',', map { sprintf('Host: %s Role: %s', $_, $hostToRoleMap->{$_}) } @hostsWithAdditionalRoles);
		$self->{params}->{AutoAddXS2Roles}->{desc} = $resultString;
	}

	return $result;
}

sub _skipUnnededParameters{
	my $self = shift();
	for my $paramID ('SelectedSystem', 'RemoteExecution', 'InstallSSHKey', 'RootUser', 'RootPassword', 'hdbinst_plugin_SystemUser', 'hdbupd_server_PrepareUpgrade', 'CannotResumePendingUpdate', 'XSComponentsNoStart'){
		if(exists($self->{params}->{$paramID})){
			$self->setSkip($paramID, 1);
			$self->{params}->{$paramID}->{hidden} = 1;
		}
	}
}

sub _addSlpSpecificParameters {
	my ($self) = @_;

	$self->{params}->{DetectedComponents} = $self->_getParamDetectedComponents();
	$self->{params}->{ExistingHosts} = $self->_getParamExistingHosts();
	$self->{params}->{SignedComponents} = $self->_getParamSignedComponents();
    $self->{params}->{CannotResumePendingUpdate} = $self->_getParamCannotResumePendingUpdate();
	$self->{params}->{ResumePendingUpdate} = $self->_getParamResumePendingUpdate();
	$self->{params}->{DeprecatedPlugins} = $self->_getParamDeprecatedPlugins();

	if(defined($self->getIgnoreValues()) && scalar(@{$self->getIgnoreValues()})){
		$self->{params}->{Ignore} = $self->_getParamIgnore();
		$self->addParameterListener( 'Ignore', LCM::Slpp::Update::Listeners::IgnoreListener->new() );
	}
}

sub setSID {
	my ($self, $value) = @_;

	if(!$self->SUPER::setSID($value)){
		return undef;
	}

	$self->_initParamExistingHosts();

# Do not use the listener, because it can result in problems with the batch execution (Bug: 77813)
    my $systemHosts = $self->getSystemHosts();
    $self->{params}->{AddRoles}->{origin_values} = [ @{$systemHosts} ];
    $self->{params}->{AddRoles}->{default_map} = { map { ($_ => '') } @{$systemHosts} };

# Explicitly set Scope to system and disable Scope parameter
# so this way remotehosts object will be propperly initialized
# and noone will change the value to instance
	$self->setUseSidadmUserForRemoteHostctrlHosts(1);
	$self->setValue('Scope', 'system');
	$self->{params}->{Scope}->{skip} = 1;
	$self->{params}->{Scope}->{hidden} = 1;

	my $instance = $self->getOwnInstance();
	if (!defined($instance) || !$instance->exists_remote_host()) {
# Skip the HostagentPassword on single host systems (it's unskipped in AnyConfig's sub setScope)
		$self->setSkip('HostagentPassword', 1);
	}

    $self->_updateReferenceDataPathParam();
    if ( defined $self->getValue('ReferenceDataPath') ) {
        $self->setHidden('ReferenceDataPath');
    }
    
    return 1;
}

sub checkAddRoles {
	my ($self, $hostname, $roles) = @_;

# Enable the possibility to unset the already set roles,
# because in the webUI this parameter is interractive
	return 1 if(length($roles) == 0);
	return $self->SUPER::checkAddRoles($hostname, $roles);
}

sub getComponentManager {
	my ($self) = @_;
	if ( not defined $self->{componentManager} ) {
        $self->{componentManager} = new LCM::Slpp::Update::MediumComponentManager($self);
        $self->{componentManager}->setMsgLstContext( [ $self->getMsgLst() ] );
        my $isResident = LCM::Installer->new()->isInstalled();
        $self->{componentManager}->setResidentComponentManager($isResident);
        $self->{componentManager}->setSidAdmUserExecution(1);
    }
	return $self->{componentManager};
}

sub detectInstalledComponents {
	my ( $self, $sid ) = @_;
    if(!$self->{areInstalledComponentsDeteced}){
    	$self->{areInstalledComponentsDeteced} = 1;
    	return $self->SUPER::detectInstalledComponents($sid);
    }
	return 1;
}

sub initializeDetectedComponents{
	my ($self) = @_;
	my $componentManager = $self->{componentManager};
	my $detectedComponents = [ @{$componentManager->getAllComponents()}, @{$componentManager->getXs2ApplicationComponents()} ];
	my $detectedComponentsParam = $self->{params}->{DetectedComponents};
	my $signedComponentsParam = $self->{params}->{SignedComponents};
	my @signedComponentsBatchKeys = ();

	$detectedComponentsParam->{ui_values} = [];
	$detectedComponentsParam->{valid_values} = [];

	for my $component (@{$detectedComponents}) {
		next if($component->isInternal() || !$component->canSelectComponent());

		my $componentName = $component->getComponentName();
		my $availableVersion = $component->getVersion();
		my $location = $component->getPath();
		my $componentBatchKey = $component->getComponentBatchKey();

		push(@signedComponentsBatchKeys, $componentBatchKey) if $component->isSigned();
		push(@{$detectedComponentsParam->{valid_values}}, $componentBatchKey);
		push(@{$detectedComponentsParam->{ui_values}}, "Name: $componentName Version: $availableVersion Location: $location");
	}
	$signedComponentsParam->{value} = $signedComponentsParam->{default} = join(',', @signedComponentsBatchKeys);
	return 1;
}

sub _getComponentUIString {
	my ($self, $component) = @_;
	my $componentName = $component->getComponentName();
	my $targetVersion = $component->getVersion();
	my $location = $component->getPath();
	my $uiString = sprintf("Action: Install Name: %s VersionInfo: version %s Location: %s", $componentName, $targetVersion, $location);
	if($component->isUpdate()){;
		my $componentKey = $component->getManifest()->getComponentKey();
		my $installedComponent = $self->getSystemComponentManager()->getComponentByKeyName($componentKey);
		my $sourceVersion = $installedComponent->getVersion();
		$uiString = sprintf("Action: Update Name: %s VersionInfo: from version %s to version %s Location: %s", $componentName, $sourceVersion, $targetVersion, $location);
	}
	return $uiString;
}

sub fillSelectedComponentsDefaultValue {
	my $self = shift();
	my $defaultSelection = "";

	$self->SUPER::fillSelectedComponentsDefaultValue(@_);
	if( $self->getDefault('SelectedComponents') eq 'all' ) {
		my $validValues = $self->getValidValues('SelectedComponents');

	    for my $valid_value (@$validValues) {
	    	next if( $valid_value eq 'all' || $valid_value eq 'hdblcm' );
   			$defaultSelection .= "," if ( length $defaultSelection );
   			$defaultSelection .= $valid_value;
	    }
		$self->setDefault('SelectedComponents', $defaultSelection);
	}
}

sub getComponentKeynameToAction {
	my ($self) = @_;
	my $generator = new LCM::Utils::ComponentActionStringGenerator::WebUpdate();

	return $generator->getComponentKeynameToActionMap($self, $self->{componentManager}, $self->{systemComponentManager});
}

sub initDefaultSystems { }

sub addListeners {
	my ($self) = @_;
	$self->SUPER::addListeners();
	$self->addParameterListener( 'DvdPath', LCM::Slpp::Update::Listeners::ComponentLocationsListener->new() );
	$self->addParameterListener( 'ComponentDirs', LCM::Slpp::Update::Listeners::ComponentLocationsListener->new() );
	$self->addParameterListener( 'ComponentFsRoot', LCM::Slpp::Update::Listeners::ComponentLocationsListener->new() );
	$self->addParameterListener( 'XSSpaceIsolation', sub{
		$self->clearParameterWarnings('XSSpaceIsolation');
		for my $paramId ('XSSpaceUserIdSAP', 'XSSpaceUserIdProd'){
			next if($self->isSkipped($paramId)); # Parameter $paramId is not needed
			next if(!$self->isHidden($paramId)); # Parameter $paramId will be prompted interactively

			my $paramWarnings = $self->{params}->{$paramId}->{warnings} || [];
			$self->addParameterWarning('XSSpaceIsolation', $_) for(@{$paramWarnings});
		}
		return 1;
	});
}

sub _getParamDetectedComponents {
	my ($self) = @_;
	my $order = scalar(keys(%{$self->{params}}));
	my $section = $self->{params}->{ComponentDirs}->{section};
	return {
		'order'             => $order++,
		'opt'               => 'detected_components',
		'type'              => 'csv',
		'section'           => $section,
		'value'             => undef,
		'init_with_default' => 0,
		'set_interactive'   => 0,
		'str'               => 'Detected Components',
		'mandatory'         => 0,
		'valid_values'      => [],
		'ui_values'         => [],
	};
}

sub _getParamSignedComponents {
    my ($self) = @_;
    return {
        'order'				=> scalar(keys(%{$self->{params}})) + 1,
        'section'			=> $self->{params}->{SelectedComponents}->{section},
        'type'				=> 'string',
        'value'				=> undef,
        'init_with_default'	=> 0,
        'set_interactive'	=> 0,
        'str'				=> 'Signed Components',
        'mandatory'			=> 0,
    };
}

sub _getParamExistingHosts {
	my ($self) = @_;
	my $order = scalar(keys(%{$self->{params}}));
	my $section = $self->{params}->{AddHosts}->{section};

	return GetParamExistingHosts($order++, $section);
}

sub _initParamExistingHosts {
	my ($self) = @_;
	my $existingHostsParameter = $self->{params}->{ExistingHosts};
	my $instance = $self->getOwnInstance(1);
	my $hostRolesInfo = $instance->getHostRolesInfo();
	my @existingHostsWithRoles = map { "${_}:role=" . join(':role=', split('\s+', $hostRolesInfo->{$_})) } @{$self->getSystemHosts()};

	$existingHostsParameter->{value} = join(',', @existingHostsWithRoles);
}

sub shouldShowPassThroughParams { return 1; }

sub setIgnore {
	my ($self, $value) = @_;
	my $optionIgnore = $self->getOptionIgnore();

	return 0 if(!defined($optionIgnore));
	if(!$optionIgnore->parseArgs($value)){
		$self->appendErrorMessage(undef, $optionIgnore->getErrMsgLst());
		return 0;
	}
	$self->{params}->{Ignore}->{value} = $value;
	return 1;
}

sub _getParamIgnore {
	my ($self) = @_;
	my $order = scalar(keys(%{$self->{params}})) + 1;
	my $section = $self->{params}->{ComponentDirs}->{section};
	my $ignoreValues = $self->getIgnoreValues();
	return GetParamIgnore($order++, $section, $ignoreValues);
}

sub getOptionTimeout{
    my ($self) = @_;

    require LCM::Configuration::Web::OptionTimeout;
    if (!exists $self->{optionTimeouts}){
        $self->{optionTimeouts} = new LCM::Configuration::Web::OptionTimeout($self->getTimeoutValues());
    }
	return $self->{optionTimeouts};
}

sub fillValidSelectedComponentValues {
	my ($self) = @_;
	$self->SUPER::fillValidSelectedComponentValues();
	$self->{params}->{SelectedComponents}->{desc} = $self->getCannotBeSelectedMessages();
	$self->{params}->{DeprecatedPlugins}->{desc} = "";
	$self->_addDeprecatedPlugins();
	return 1;
}

sub getCannotBeSelectedMessages {
    my ($self) = @_;    
    my $consoleMessage = "";    
    my $messages = $self->getRestrictedComponentsMessages();
    foreach my $componentName (keys %{$messages}) {
        my $restrictedComponentMessage = $messages->{$componentName};
        $consoleMessage.="Message: Component $componentName cannot be selected. $restrictedComponentMessage";
    }
    return $consoleMessage;    
}

sub _initPendingConfiguration {
    my ($self, $sid) = @_;
    my $application = LCM::App::ApplicationContext::getInstance()->getApplication();
    my $serverPath = $application->getServerPath();
    my $mcm = $self->getComponentManager();

    $self->setValue("ComponentFsRoot", $serverPath) if ($serverPath);

	if (!$serverPath) {
		my $hasHdblcmPendingUpdateFile = existsHdblcmPendingUpdate($sid);
		my $hdblcmPersistedData = $hasHdblcmPendingUpdateFile ? $self->_loadHdblcmPersistenceSilently() : undef;

		if (defined $hdblcmPersistedData) {
			my $componentDirs = $hdblcmPersistedData->{'ComponentDirs'};
			if (defined $componentDirs) {
				$self->setValue("ComponentDirs", $componentDirs);
			}
		}
	} 
	if (defined $mcm) {
        my $componentDirs = $mcm->getComponentDirs(0);
        $self->{'params'}->{'ComponentDirs'}->{'value'} = join(',', @{$componentDirs}) if(scalar(@{$componentDirs}) > 0);
    }
    my $rc = $self->SUPER::_initPendingConfiguration($sid);

    if (! $rc) {
        $self->setSkip('CannotResumePendingUpdate', 0);
        $self->setValue('CannotResumePendingUpdate', 1);
        my $pendingConfig = $self->getPendingConfiguration();
        my $errorMessage = $pendingConfig->getErrorString();
        $self->{'params'}->{'CannotResumePendingUpdate'}->{'desc'} = $unableToContinueUpdateMsg.' '.$errorMessage;
    }
    return $rc;
}

sub _getParamCannotResumePendingUpdate {
    my ($self) = @_;
    my $order = scalar(keys(%{$self->{params}})) + 1; 
    my $section = $self->{params}->{ComponentDirs}->{section};
    return {
        'order'             => $order++,
        'opt'               => 'cannot_resume_pending_update',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'str'               => '',
        'mandatory'         => 0,
        'skip'              => 1,
    };
}

sub setResumePendingUpdate {
	my ($self, $value) = @_;

	if($value !~ /^$bool_false_pattern|$bool_true_pattern$/){
		$self->PushError("Invalid value '$value' of parameter ResumePendingUpdate (only 'true' and 'false' are allowed)");
	}

	my $parameter = $self->{params}->{ResumePendingUpdate};
	if ($self->getIgnore ('check_pending_upgrade')) {
		$self->getMsgLst()->addMessage("Ignoring pending update due to --ignore command line switch and starting a new update");
		$self->setSkip('ResumePendingUpdate', 1);
		$self->setSkip('ContinueUpdate', 1);
		return 1;
	}

	$parameter->{value} = 1;
	return 1;
}

sub _getParamResumePendingUpdate {
	my ($self) = @_;
	my $order = scalar(keys(%{$self->{params}})) + 1; 
	my $section = $self->{params}->{ComponentDirs}->{section};
	return {
		'order'             => $order++,
		'opt'               => 'resume_pending_update',
		'type'              => 'boolean',
		'section'           => $section,
		'value'             => undef,
		'init_with_default' => 0,
		'set_interactive'   => 0,
		'str'               => '',
		'mandatory'         => 1,
		'skip'              => 1,
	};
}

sub _initParamResumePendingUpdate {
	my ($self, $sid) = @_;
	$self->{current_sid} = $sid;
	my $resumeParam = $self->{params}->{ResumePendingUpdate};
	my $hasHdblcmPendingUpdateFile = existsHdblcmPendingUpdate($sid);
	my $hasHdbupdPendingUpdateFile = existsHdbupdPendingUpdate($sid);
	my $hdblcmPersistedData = $hasHdblcmPendingUpdateFile ? $self->_loadHdblcmPersistenceSilently() : undef;
	my $hdbupdPersistedData = $hasHdbupdPendingUpdateFile ? $self->_createHdbupdConfiguration()->pers_load() : undef;

	if($hasHdblcmPendingUpdateFile || $hasHdbupdPendingUpdateFile){
		$resumeParam->{desc} = $self->_getPendingUpdateString($hdblcmPersistedData, $hdbupdPersistedData);
		$resumeParam->{recommendation} = $self->_getPendingComponentDetails($hdblcmPersistedData, $hdbupdPersistedData, $sid);
		$resumeParam->{skip} = 0;
	}
}

sub _getPendingComponentDetails {
	my ($self, $hdblcmPersistedData, $hdbupdPersistedData, $sid) = @_;
	my $detailStrings = [];

	if(defined($hdblcmPersistedData)){
		push(@{$detailStrings}, $self->_getHdblcmDetails($hdblcmPersistedData));
	}
	if(defined($hdbupdPersistedData)){
		push(@{$detailStrings}, $self->_getHdbupdDetails($hdbupdPersistedData));
	}
	push(@{$detailStrings}, @{$self->_getPendingDetailsFromPersistenceXmlFiles($sid)});

	return join('|', @{$detailStrings});
}

sub _getHdblcmDetails {
	my ($self, $persistedData) = @_;
	my $detailsString = "Pending Update Details,Pending since,%s,Current Step,%s,Installation Kit Version,%s";
	my $dateString = $self->pers_date_string();
	my $stepName = $persistedData->{step_name};
	my $currentServerVersion = $self->_getCurrentServerVersion();

	return sprintf($detailsString, $dateString, $stepName, $currentServerVersion);
}

sub _getHdbupdDetails {
	my ($self, $persistedData) = @_;
	my $detailsString = "Pending Update of SAP HANA Database Details,Pending since,%s,Current Step,%s,Target Version,%s";
	my $hdbupdConfiguration = $self->_createHdbupdConfiguration();
	my $dateString = $hdbupdConfiguration->pers_date_string();
	my $stepName = $UPGRADE_STEP_NAMES[$persistedData->{step}];
	my $targetVersion = $persistedData->{_kitversion};

	return sprintf($detailsString, $dateString, $stepName, $targetVersion);
}

sub _getPendingDetailsFromPersistenceXmlFiles {
	my ($self, $sid) = @_;
	my $result = [];
	my $detailsStringTemplate = "Pending %s of %s Details,Pending since,%s,Current Step,%s,Target Version,%s";
	my $persistenceFilesSet = LCM::Persistence::XMLDetector::detect($sid, $self->getMsgLst());

	for my $componentKey (keys(%{$persistenceFilesSet})) {
		my $persistenceXmlObject = $persistenceFilesSet->{$componentKey};
		if (!defined $persistenceXmlObject) {
			next;
		}
		my $componentName = $persistenceXmlObject->getComponentNameString();
		my $dateString = $persistenceXmlObject->getCreationDateString();
		my $stepName = $persistenceXmlObject->getCurrentStepString();
		my $targetVersion = $persistenceXmlObject->getTargetVersionString();
		my $action = ucfirst($persistenceXmlObject->getActionString());

		push(@{$result}, sprintf($detailsStringTemplate, $action, $componentName, $dateString, $stepName, $targetVersion));
	}
	return $result;
}

sub _createHdbupdConfiguration {
	require SDB::Install::Configuration::Upgrade;

	my ($self) = @_;
	my $sid = $self->getSID();
	my $allSystems = CollectSAPSystems();
	my $sapSystem = $allSystems->{$sid};
	my $updConfig = new SDB::Install::Configuration::Upgrade();

	$updConfig->{current_sid} = $sid;
	$updConfig->{sapSys}      = $sapSystem;

	return $updConfig;
}

sub validatePersistency {
	my ($self, $persistedData) = @_;

	if(defined($persistedData)){
		delete($persistedData->{RemoteExecution});
		delete($persistedData->{RootUser});
	}

	return $self->SUPER::validatePersistency($persistedData);
}

sub _getPendingUpdateString {
	my ($self, $hdblcmPersistedData, $hdbupdPersistedData) = @_;
	my $currentServerVersion = $self->_getCurrentServerVersion();
	my $originServerVersion = $hdbupdPersistedData->{_kitversion} || $currentServerVersion;
	my $dateString = $self->pers_date_string();
	my $localHost = $self->getOwnInstance()->get_host();
	my $origintHost = $hdblcmPersistedData->{pers_hostname} || $localHost;
	my $sameHost = ($localHost eq $origintHost) ? 'true' : 'false';

	return sprintf("Host:'%s' SameHost:'%s' OriginServer:'%s' CurrentServer:'%s'",
		$origintHost, $sameHost, $originServerVersion, $currentServerVersion,
	);
}

sub setContinueUpdate {
	my ($self, $value) = @_;

	return 1 if ($self->getIgnore ('check_pending_upgrade'));

	my $result = $self->SUPER::setContinueUpdate($value);
	if($result && $value =~ /^$bool_false_pattern$/) {
		$self->setSkip('ResumePendingUpdate', 1);
	}
	if($result && $value =~ /^$bool_true_pattern$/) {
		return undef if(! $self->_loadParameterValuesFromPersistence());
	}
	return $result;
}

sub _loadParameterValuesFromPersistence {
	my ($self) = @_;

	for my $parameterId (@{$self->getParamIds()}){
		my $parameter = $self->{params}->{$parameterId};
		my $persistedValue = $parameter->{persFileValue};
		my $optionString = $parameter->{opt};

		next if (!defined($persistedValue));
		next if ($parameter->{hidden});
		next if ($self->setPersistedValue($parameterId, $parameter, $persistedValue));

		$self->setErrorMessage ("Checking persisted value for parameter '$optionString' failed.");
		return undef;
	}
	return 1;
}

sub _getParamDeprecatedPlugins {
	my ($self) = @_;
	my $order = scalar(keys(%{$self->{params}})) + 1;
	my $section = $self->{params}->{ComponentDirs}->{section};
	return {
		'order'             => $order++,
		'opt'               => 'deprecated_plugins',
		'type'              => 'csv',
		'section'           => $section,
		'value'             => undef,
		'init_with_default' => 0,
		'set_interactive'   => 0,
		'str'               => '',
		'mandatory'         => 0,
		'skip'              => 1,
	};
}

sub _addDeprecatedPlugins {
	my ($self) = @_;
	my $componentKeynameToAction     = $self->getComponentKeynameToAction();
	my $deprecatedPluginsParam = $self->{params}->{DeprecatedPlugins};
	$deprecatedPluginsParam->{valid_values} = [];
	$deprecatedPluginsParam->{ui_values} = [];
	my $deprecatedComponents = $self->getDeprecatedComponentsForUninstall();
	for my $deprecatedComponent (@$deprecatedComponents) {
		my $componentBatchKey = $deprecatedComponent->getComponentBatchKey();
		my $componentKeyname = $deprecatedComponent->getComponentKeyName();
		push( @{ $deprecatedPluginsParam->{valid_values} }, $componentBatchKey );
		push( @{ $deprecatedPluginsParam->{ui_values} }, $componentKeynameToAction->{$componentKeyname} );
	}
	$self->_addDeprecatedPluginsWarnings();
}

sub _addDeprecatedPluginWarning {
	my ($self, $warning) = @_;
	$self->SUPER::_addDeprecatedPluginWarning($warning);
	my $messages = $self->{params}->{DeprecatedPlugins}->{desc};
	$self->{params}->{DeprecatedPlugins}->{desc} = $messages. "Message: $warning";
	$self->{params}->{DeprecatedPlugins}->{skip} = 0;
}

1;
