package LCM::Configuration::XS2Configuration;

use strict;
use LCM::Installer;
use LCM::ProcessExecutor;
use SDB::Install::MsgLst;
use SDB::Install::User;
use SDB::Install::NewDBUser;
use File::Basename qw(dirname);
use experimental qw(smartmatch);
use SDB::Install::System qw(isAdmin nslookup $hostname_regex);
use SDB::Install::SysVars qw($isWin);
use LCM::Configuration::ParametersCreator;
use SDB::Common::Utils qw(createXSSpaceSAPUserName createXSSpaceProdUserName createXMLParser checkOsUserID);
use LCM::Utils::CommonUtils qw(getSidadmName);
use SDB::Install::Globals qw ($gXSParametersConstraint $gShortProductNameXS2 $gHostRoleXS2Worker $gFlavourPlatform $gKeynameXS2);
use parent 'LCM::Configuration::GenericStackAny';

use constant ROUTING_MODE_PORTS => 'ports';
use constant ROUTING_MODE_HOSTNAMES => 'hostnames';

sub InitDefaults {
    my $self = shift();
    $self->SUPER::InitDefaults(@_);
    $self->setDefault('XsDomainName', $self->getDefaultXsControllerHost());
    return 1;
}

sub defineXS2Params {
	my ($self) = @_;
	my $order = scalar keys %{ $self->{params} };
	my $section = 'XS_Advanced';
	my $XS2Params = {
		'XsEaDataPath'            => $self->getParamXsEaDataPath($order++, $section, $gXSParametersConstraint),
		'XsMasterPassword'        => $self->getParamXsMasterPassword($order++, $section, $gXSParametersConstraint),
		'OrgName'                 => $self->getParamOrgName($order++, $section, $gXSParametersConstraint),
		'OrgManagerUser'          => $self->getParamOrgManagerUser($order++, $section, $gXSParametersConstraint),
		'OrgManagerPassword'      => $self->getParamOrgManagerPassword($order++, $section, $gXSParametersConstraint,'OrgManagerUser'),
		'ProdSpaceName'           => $self->getParamProdSpaceName($order++, $section, $gXSParametersConstraint),
		'RoutingMode'             => $self->getParamRoutingMode($order++, $section, $gXSParametersConstraint),
		'XsDomainName'            => $self->getParamXsDomainName($order++, $section, $gXSParametersConstraint),
		'XSSAPSpaceIsolation'     => GetParamXSSAPSpaceIsolation($order++, $section, $gXSParametersConstraint),
		'XSSpaceIsolation'        => GetParamXSSpaceIsolation($order++, $section, $gXSParametersConstraint),
		'XSSpaceUserIdSAP'        => GetParamXSSpaceUserIdSAP($order++, $section, $gXSParametersConstraint),
		'XSSpaceUserIdProd'       => GetParamXSSpaceUserIdProd($order++, $section, $gXSParametersConstraint),
		'SelectedXs2Applications' => GetParamSelectedXs2Applications($order++, $section, $gXSParametersConstraint),
		'XSComponentsNoStart'     => GetParamXSComponentsNoStart($order++, $section, $gXSParametersConstraint),
		'XSComponentsCfg'         => GetParamXSComponentsCfg($order++, $section, $gXSParametersConstraint),
		'XSCertPem'               => GetParamXSCertPem($order++, $section, $gXSParametersConstraint),
        'XSCertKey'               => GetParamXSCertKey($order++, $section, $gXSParametersConstraint),
        'XSTrustPem'              => GetParamXSTrustPem($order++, $section, $gXSParametersConstraint),
	};
	map { $self->{params}->{$_} = $XS2Params->{$_} } keys %$XS2Params;
	return 1;
}

sub checkSelectedXs2Applications {
    my ($self, $selectedXsApplicationsCsv) = @_;
    my $componentManager = ($self->can('getComponentManager')) ? $self->getComponentManager() : undef;

    return 1 if $selectedXsApplicationsCsv eq 'all';
    return 1 if !defined($componentManager);
    return 1 if $componentManager->getHANAFlavour() eq $gFlavourPlatform;

    my $mandatorySelection = [];
    my $xsApplications =  (defined $componentManager) ? $componentManager->getXs2ApplicationComponents() : [];
    for my $xsApplication (@{$xsApplications}) {
        next if ! $xsApplication->isUpdate();

        if ( index ($selectedXsApplicationsCsv, $xsApplication->getComponentBatchKey()) == -1 ) {
            push @$mandatorySelection, $xsApplication->getComponentName();
        }
    }

    if ( scalar(@{$mandatorySelection}) != 0 ) {
        my $mandatorySelectionList = join (',', @$mandatorySelection );
        $self->PushError("Some XS Advanced components are mandatory for update. The following components must be selected: $mandatorySelectionList\n");
        return 0;
    }

    return 1;
}

sub checkXsEaDataPath {
    return $_[0]->checkHanaOptionsPathParam('XsEaDataPath', $_[1]);
}

sub checkXSSpaceUserIdSAP {
	my ($self, $value) = @_;
	my $userName = createXSSpaceSAPUserName($self->getSID());

	return undef if(!checkOsUserID($self, 'XSSpaceUserIdSAP', $userName, $value, 0)); # Cannot be in group 'sapsys'
    return $self->checkReservedOSUserIDs($value, 'XSSpaceUserIdSAP');
}

sub checkXSSpaceUserIdProd {
	my ($self, $value) = @_;
	my $userName = createXSSpaceProdUserName($self->getSID());

	return undef if(!checkOsUserID($self, 'XSSpaceUserIdProd', $userName, $value, 0)); # Cannot be in group 'sapsys'
    return $self->checkReservedOSUserIDs($value, 'XSSpaceUserIdProd');
}

sub checkXSComponentsCfg {
	my ($self, $value) = @_;
	return 1 if ($value eq "");
	if(not -d $value){
		$self->PushError("Directory '$value' is not valid.");
		return undef;
	}
	return 1;
}

sub checkXSCertPem {
    my ($self, $value) = @_;
    if(!-f $value){
        $self->PushError("XS Advanced Certificate file '$value' doesn't exist.");
        return undef;
    }
    return 1;
}

sub checkXSCertKey {
    my ($self, $value) = @_;
    if(!-f $value){
        $self->PushError("XS Advanced Certificate Key file '$value' doesn't exist.");
        return undef;
    }
    return 1;
}

sub checkXSTrustPem {
    my ($self, $value) = @_;
    if(!-f $value){
        $self->PushError("XS Advanced Trust Certificate file '$value' doesn't exist.");
        return undef;
    }
    return 1;
}

sub enableXS2Params{
	my ($self, $on) = @_;
	$self->setSkip('OrgManagerUser', !$on);
	$self->setSkip('OrgManagerPassword', !$on);
	$self->setSkip('OrgName', !$on);
	$self->setSkip('ProdSpaceName', !$on);
	$self->setSkip('XSSpaceIsolation', !$on);
	$self->setSkip('XSSAPSpaceIsolation', !$on);
	$self->setSkip('RoutingMode', !$on);
	$self->setSkip('XsDomainName', !$on);
	$self->setSkip('XSSpaceUserIdSAP', !$on);
	$self->setSkip('XSSpaceUserIdProd', !$on);
	$self->setSkip('XsEaDataPath', !$on);
	$self->setSkip('XSCertPem', !$on);
	$self->setSkip('XSCertKey', !$on);
	$self->setSkip('XSTrustPem', !$on);

	return 1;
}

sub getParamXsMasterPassword {
	my ($self, $order, $section, $constraint) = @_;
	return {
		'order' => $order,
		'opt' => 'xs_master_password',
		'type' => 'initial_passwd',
		'section' => $section,
		'value' => undef,
		'str' => 'XS Advanced Runtime Master Password',
		'desc' => 'XS Advanced Runtime Master Password',
		'skip' => 1,
		'mandatory' => 0,
		'set_interactive' => 0,
		'hidden' => 1,
		'constraint' => $constraint,
	};
}

sub getParamOrgName {
	my ($self, $order, $section, $constraint) = @_;
	return {
		'order' => $order,
		'opt' => 'org_name',
		'type' => 'string',
		'section' => $section,
		'value' => undef,
		'default' => 'orgname',
		'str' => 'Organization Name For Space "SAP"',
		'desc' => 'Organization Name For Space "SAP"',
		'skip' => 1,
		'mandatory' => 1,
		'init_with_default' => 1,
		'set_interactive' => 1,
		'constraint' => $constraint,
	};
}

sub getParamProdSpaceName {
	my ($self, $order, $section, $constraint) = @_;
	return {
		'order' => $order,
		'opt' => 'prod_space_name',
		'type' => 'string',
		'section' => $section,
		'value' => undef,
		'default' => 'PROD',
		'str' => 'Customer Space Name',
		'desc' => 'Customer Space Name',
		'skip' => 1,
		'mandatory' => 1,
		'init_with_default' => 1,
		'set_interactive' => 1,
		'constraint' => $constraint,
	};
}

sub getParamRoutingMode {
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order' => $order,
        'opt' => 'xs_routing_mode',
        'type' => 'string',
        'section' => $section,
        'value' => undef,
        'default' => 'ports',
        'valid_values' => [ 'ports', 'hostnames' ],
        'ui_values' => [ "Application URLs are based on ports", "Application URLs are based on hostnames" ],
        'interactive_index_selection' => 1,
        'console_text' => "Select Routing Mode:\n",
        'str' => 'Routing Mode',
        'desc' => 'Routing Mode',
        'skip' => 1,
        'mandatory' => 1,
        'init_with_default' => 1,
        'set_interactive' => 1,
        'constraint' => $constraint,
    };
}

sub _checkSqlUserPassword{
    my ($self, $sqlUserPassword, $paramProperties) = @_;
    my $msglst = new SDB::Install::MsgLst ();
    my $rc;
    if (defined $self->getOwnInstance()) {
        $rc = $self->complySqlPasswordPolicy ($sqlUserPassword, 1, $msglst, 'OrgManagerPassword');
    } else {
        $rc = $self->complyDefaultSqlPasswordPolicy ($sqlUserPassword, $msglst, 'OrgManagerPassword');
    }
    if (!$rc){
        $self->PushError ("$paramProperties->{str} is invalid", $msglst);
    }
    return $rc;
}

sub checkOrgName{
    my ($self, $orgName) = @_;
    my $paramStr = $self->{params}->{OrgName}->{str};
    if ($orgName !~ /^[[:alnum:][:punct:][:print:]]+$/){
        $self->PushError ("Parameter $paramStr must contain only printable characters");
        return 0;
    }
    return 1;
}

sub checkOrgManagerPassword {
    my ($self, $orgManagerPassword) = @_;
    my $ownInstance = $self->getOwnInstance();
    my $xsControllerHost = defined($ownInstance) ? $ownInstance->getXsControllerHostname() : undef;

    if($self->getType('OrgManagerPassword') =~ /initial/){
        return $self->_checkSqlUserPassword ($orgManagerPassword, $self->{params}->{OrgManagerPassword});
    }
    return $self->_checkExistingOrgManagerPassword($orgManagerPassword);
}

sub checkProdSpaceName{
    my ($self, $prodSpaceName) = @_;
    if ($prodSpaceName eq 'SAP'){
    my $paramStr = $self->getString ('ProdSpaceName');
        $self->appendErrorMessage ("$paramStr '$prodSpaceName' is not allowed.");
        return 0;
    }
    return 1;
}

sub checkXSComponentsNoStart {
    my ($self, $value) = @_;
    my $selectedComponentKeyNames = [ split(/\s*,\s*/, $value) ];
    my $hasSelectedMultipleComponents = scalar(@{$selectedComponentKeyNames}) > 1;
    my $hasSelectedAllComponents = grep { $_ eq 'all' } @{$selectedComponentKeyNames};
    my $hasSelectedNoneComponents = grep { $_ eq 'none' } @{$selectedComponentKeyNames};
    my $isConflictingSelection = $hasSelectedMultipleComponents && ($hasSelectedAllComponents || $hasSelectedNoneComponents);

    if($isConflictingSelection){
        $self->AddError('Some of the provided values are conflicting with each other');
        return undef;
    }
    return 1;
}

sub setXSComponentsNoStart {
    my ($self, $value) = @_;

    return undef if(!$self->checkXSComponentsNoStart($value));

    my $componentManager = $self->getMediumComponentManager();
    my $allXsComponents = $componentManager->getXs2ApplicationComponents();
    my @selectedXsComponents = grep { $_->isComponentSelected() } @{$allXsComponents};
    my @validComponentKeys = map { $_->getComponentKeyName() } @selectedXsComponents;
    my @selectedKeysNoStart = split(/\s*,\s*/, $value);
    my @validSelectedKeysNoStart = grep { $_ ~~ @validComponentKeys || $_ =~ /^(all|none)$/  } @selectedKeysNoStart;
    my $noStartValue = join(',', @validSelectedKeysNoStart) || 'none';

    $self->{params}->{XSComponentsNoStart}->{value} = $noStartValue;
}

sub getInstalledXsComponents {
    my ($self) = @_;
    my $scm = $self->getSystemComponentManager();
    return defined($scm) ? $scm->getXs2ApplicationComponents() : [];
}

sub _checkExistingOrgManagerPassword {
    my ($self, $passwordValue) = @_;
    my $componentManager = ($self->can('getComponentManager')) ? $self->getComponentManager() : undef;
    my $xsApplications =  (defined $componentManager) ? $componentManager->getXs2ApplicationComponents() : [];
    my $hasDetectedXsApps = scalar(@{$xsApplications});
    $self->clearParameterWarnings('OrgManagerPassword');

    my $rc = $self->_checkPasswordViaDBLogin($passwordValue);

    if($rc && $hasDetectedXsApps){
        $self->fillValidSelectedXs2ApplicationsValues();
    }

    return $rc;
}

sub _checkPasswordViaDBLogin {
    my ($self, $passwordValue) = @_;
    my $orgManagerUser  = $self->getValue('OrgManagerUser');
    my $xsRuntimeDbName = $self->retrieveXsAdminLoginParam('runtime-database-name');
    my $service = $xsRuntimeDbName ? 'IndexServer' : 'SystemServer';
    return $self->checkDatabaseUserPassword($orgManagerUser, $passwordValue, 'OrgManagerPassword', undef, undef, $service);
}

sub getApiUrl {
    my ($self) = @_;
    my $storedAPIUrl = $self->_retrieveXsAdminLoginApi();

    return $storedAPIUrl if(defined($storedAPIUrl));

    my $ownInstance = $self->getOwnInstance(1);
    my $hostNames = $ownInstance->get_allhosts();
    my $instanceNumber = $ownInstance->get_nr();

    for my $hostName (@{$hostNames}){
        if($ownInstance->hasXS2Controller($hostName)){
            return sprintf('https://%s:3%02d30', $hostName, $instanceNumber);
        }
    }
    return undef;
}


sub retrieveXsAdminLoginParam {
    my ($self, $adminLoginParam, $noCache) = @_;
    if (!exists($self->{$adminLoginParam}) || $noCache) {
        my $executor    = $self->_createXsAdminLoginProcessExecutor(["--$adminLoginParam"], undef);
        my $exitCode    = $executor->executeProgram();
        my $outputLines = defined($exitCode) && ($exitCode == 0) ? $executor->getOutputLines() : [];
# We expect that the output of xs-admin-login is a single-line string. It is the same if we read the first or the last entry
# as they are the same line. Here we read the last line because when the tool is started via the SAPHOSTAGENT and the environment
# contains a variable with multi-line value, bash errors appear in the beginning and the actual result is in the end
        $self->{$adminLoginParam} = $outputLines->[-1] ? $outputLines->[-1] : undef;
    }
    return $self->{$adminLoginParam};
}

sub _retrieveXsAdminLoginApi {
    my ($self) = @_;
    my $apiUrl = $self->retrieveXsAdminLoginParam('api');

    if (!$apiUrl || $apiUrl !~ /^https?:\/\//) {
        $self->getMsgLst()->AddMessage("Retrieval of $gShortProductNameXS2 API URL via 'xs-admin-login' was not successful");
        $self->getMsgLst()->AddMessage("Using fallback detection mechanism based on 'daemon.ini' files...");
        return undef;
    }
    return $apiUrl;
}

sub _createXsAdminLoginProcessExecutor {
    my ($self, $arguments, $stdin) = @_;
    my $sid = $self->getSID();
    my $targetDirectory = $self->getValue('Target');
    my $command = File::Spec->catfile($targetDirectory, $sid, 'xs', 'bin', 'xs-admin-login');
    my $installer = new LCM::Installer();
    my $runtimeDirectory = $installer->getRuntimeDir();
    my $wrapperCommand = File::Spec->catfile(dirname($runtimeDirectory), 'hdblcm');
    my $wrappedArguments = [ '--main', 'LCM::EnvironmentWrapper::main', $sid, $command, @{$arguments} ]; # LCM::EnvironmentWrapper will switch to sidadm if root

    return new LCM::ProcessExecutor($wrapperCommand, $wrappedArguments, $stdin, dirname($command));
}

sub setRoutingMode {
    my ($self, $value) = @_;

    my $domainNameParam = $self->{params}->{XsDomainName};

    if ($value eq ROUTING_MODE_HOSTNAMES) {
        $self->setMandatory('XsDomainName', 1);
        $self->setDefault('XsDomainName', undef, 0);
    } elsif ($value eq ROUTING_MODE_PORTS) {
        # The default value is set in the InitDefaults method
        $self->setMandatory('XsDomainName', 0);
    } else {
        $self->PushError("Invalid value '$value' of RoutingMode parameter.");
        return undef;
    }

    $self->{params}->{RoutingMode}->{value} = $value;
    return 1;
}

sub getDefaultXsControllerHost {
    my ($self) = @_;

    return $self->_getLocalHostname(1) if $self->_isLocalHostXsWorker(); # Use FQDN if possible

    my $isAutoAssignEnabled = $self->getValue('AutoAddXS2Roles') && ! $self->isSkipped('AutoAddXS2Roles');
    if ($isAutoAssignEnabled) {
        my $autoAssignedRolesMap = $isAutoAssignEnabled ? $self->getAutoAssignXs2RolesMap() : {};
        return (keys %$autoAssignedRolesMap)[0] if (length($autoAssignedRolesMap));
    }

    my $addRolesMapValue = $self->getValue('AddRoles');
    if (defined $addRolesMapValue && keys %$addRolesMapValue > 0) {
        foreach (keys %$addRolesMapValue) {
            return $_ if (index($addRolesMapValue->{$_}, $gHostRoleXS2Worker) != -1);
        }
    }

    my @addHostsWithRoleXsWorker = @ { $self->getAddHostsWithRole( $gHostRoleXS2Worker ) };
    return $addHostsWithRoleXsWorker[0] if (scalar @addHostsWithRoleXsWorker > 0);
    return "";
}

sub _getLocalHostname {
    my ($self, $useFQDN) = @_;
    my $instance = $self->getOwnInstance();
    my $hostName = defined($instance) ? $instance->get_host() : $self->getValue('HostName');
    my $fqdnHostName = undef;

    eval {
        if($useFQDN){
            require Net::Domain;
            $fqdnHostName = Net::Domain::domainname();
        }
    };
    return $useFQDN && (index($fqdnHostName, sprintf('%s.', $hostName)) == 0) ? $fqdnHostName : $hostName;
}

sub _isLocalHostXsWorker {
    my ($self) = @_;
    my $localRoles = $self->getValue('AddLocalRoles');
    if ((defined $localRoles) &&  (index($localRoles, $gHostRoleXS2Worker) != -1)) {
        return 1;
    }
    my $localHostname = $self->_getLocalHostname();
    my $isAutoAssignEnabled = $self->getValue('AutoAddXS2Roles') && ! $self->isSkipped('AutoAddXS2Roles');
    if ($isAutoAssignEnabled) {
        my $autoAssignedRolesMap = $isAutoAssignEnabled ? $self->getAutoAssignXs2RolesMap() : {};
        if (defined $autoAssignedRolesMap->{$localHostname}) {
            return 1;
        }
    }

    my $addRolesMapValue = $self->getValue('AddRoles');
    if (defined $addRolesMapValue && keys %$addRolesMapValue > 0) {
        return 1 if (index($addRolesMapValue->{$localHostname}, $gHostRoleXS2Worker) != -1);
    }

    return 0;
}

sub _getAllXsWorkerHosts {
    my ($self) = @_;
    my $xsWorkerHosts = [];

    if ($self->_isLocalHostXsWorker()) {
        push (@{$xsWorkerHosts}, $self->_getLocalHostname());
    }

    my $isAutoAssignEnabled = $self->getValue('AutoAddXS2Roles') && ! $self->isSkipped('AutoAddXS2Roles');
    if ($isAutoAssignEnabled) {
        my $autoAssignedRolesMap = $isAutoAssignEnabled ? $self->getAutoAssignXs2RolesMap() : {};
        if (length($autoAssignedRolesMap)) {
            push (@{$xsWorkerHosts}, keys %$autoAssignedRolesMap);
        }
    }

    my $addRolesMapValue = $self->getValue('AddRoles');
    if (defined $addRolesMapValue && keys %$addRolesMapValue > 0) {
        foreach (keys %$addRolesMapValue) {
            if (index($addRolesMapValue->{$_}, $gHostRoleXS2Worker) != -1) {
                my @hostRoles = split ('=', $_);
                push (@{$xsWorkerHosts}, $hostRoles[0]);
            }
        }
    }

    my @addHostsWithRoleXsWorker = @ { $self->getAddHostsWithRole( $gHostRoleXS2Worker ) };
    if (scalar @addHostsWithRoleXsWorker > 0) {
        push (@{$xsWorkerHosts}, @addHostsWithRoleXsWorker);
    }

    return $xsWorkerHosts;
}

sub checkXsDomainName {
    my ($self, $value) = @_;

    if (!$value){
        return 1;
    }

    if ($value !~ /$hostname_regex/){
        $self->PushError ("\'$value' is no valid host name.");
        return 0;
    }

    if (!$self->_isHostAccessible($value)) {
        $self->PushError ("Host name $value is not accessible.");
        return 0;
    }

    if ($self->{params}->{RoutingMode}->{value} eq 'hostnames') {
        my $subdomain = "subdomaintest.".$value;
        if (!$self->_isHostAccessible($subdomain)) {
            $self->PushError ("XSA Domain Name needs to allow wildcard subdomains. Please configure your DNS accordingly (see SAP Note 2245631).");
            return 0;
        }
    }
    
    my $address = $self->_getIpAddress($value);
    
    my $xsWorkerHosts = $self->_getAllXsWorkerHosts();
    for my $host (@{$xsWorkerHosts}){
        if ($self->_isSameIpAddress($host, $address)) {
            $self->setXsControllerHostname($host);
            last;
        }
    }

    return 1;
}

sub _isSameIpAddress {
    my ($self, $host, $address) = @_;

    my $hostAddress = $self->_getIpAddress($host);
    return ($address eq $hostAddress);
}

sub _isHostAccessible {
    my ($self, $hostname) = @_;

    my $errlst   = new SDB::Install::MsgLst();
    my $nslookup = nslookup($hostname, $errlst);

    if (!defined $nslookup || !defined $nslookup->{address}) {
        return 0;
    }

    return 1;
}

sub _getIpAddress {
    my ($self, $hostname) = @_;
    my $errlst   = new SDB::Install::MsgLst();
    my $nslookup = nslookup($hostname, $errlst);

    if (!defined $nslookup || !defined $nslookup->{address}) {
        return undef;
    }

    return $nslookup->{address};
}

sub setXsControllerHostname {
    my ($self, $hostname) = @_;
    $self->{_xsControllerHostname} = $hostname;
}

sub getXsControllerHostname {
    my ($self) = @_;
    return $self->{_xsControllerHostname};
}

1;