package LCM::Executors::BaseExecutor;

use strict;

use base qw (SDB::Install::Base Exporter);
use LCM::TraceLoggingEnvironment;
use LCM::SlAnalytics::SLAMonitor;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager;
use SDB::Install::System;
use SDB::Install::NewDBUser;

our $gActionScope = 'ACTIONSCOPE';
our @EXPORT = qw ( $gActionScope );
                
use constant ACTION_STATUS_DONE => 1;
use constant ACTION_STATUS_DONE_WITH_WARNING => 2;
use constant ACTION_STATUS_FAILED => 3;

sub new {
	my ( $class, $options ) = @_;
	my %self = defined $options ? %{$options} : ();
	my $selfRef = \%self;
	bless( $selfRef, $class );

	return $selfRef;
}

sub buildSHAOptionsMap {
    my ($self, $configuration, $parameterToOptionMapping) = @_;
    my $optionsMap = {};

    for my $parameterId (@{$configuration->getParamIds()}) {
        next if (!exists($parameterToOptionMapping->{$parameterId}));
        next if (!$configuration->hasValue($parameterId));

        my $option = $parameterToOptionMapping->{$parameterId};
        my $value = $configuration->getValue($parameterId);

        if($configuration->getType($parameterId) =~ /boolean/){ # Resolve boolean values to true and false
            $value = $value ? 'on' : 'off';
        }
        $optionsMap->{$option} = $value;
    }

    my $sid = $configuration->getValue('SID');
    if ((defined $sid) and (SDB::Install::System::isSidadmin($sid))) {
        my $user = new SDB::Install::NewDBUser($sid);
        my ($uid, $gid) = ($user->uid(), $user->gid());

        if (defined $uid && defined $gid) {
            $optionsMap->{'INSTLOG_UID'} = $uid;
            $optionsMap->{'INSTLOG_GID'} = $gid;
        }
    }

    return $optionsMap;
}

sub requiresInput {
	return defined $_[0]->{requiresInput} ? $_[0]->{requiresInput} : 1;
}

sub requiresSummary {
	return defined $_[0]->{requiresSummary} ? $_[0]->{requiresSummary} : 1;
}

sub requiresConfirmation {
	return defined $_[0]->{requiresConfirmation} ? $_[0]->{requiresConfirmation} : 1;
}

sub requiresLog {
	return defined $_[0]->{requiresLog} ? $_[0]->{requiresLog} : 1;
}

sub requiresHeaderMsg {
	return defined $_[0]->{requiresHeaderMsg} ? $_[0]->{requiresHeaderMsg} : 1;
}

sub getMessageBeforeConfirm {
	return undef;
}

sub setActionScope {
	$_[0]->{actionScope} = $_[1];
}

sub getActionScope {
	my ( $self ) = @_;
	return $self->{actionScope} if defined $self->{actionScope};
	
    my $instconfig = $self->{instconfig};
    if(!defined $instconfig){
    	$instconfig = $self->{config};
    }
    return 'system' if !defined $instconfig;
    my $scope = $instconfig->getValue('Scope');
    
    return 'system' if ! defined $scope;
    return $scope;
}

sub getFinalMessage {
	my ( $self, $message ) = @_;
	my $scope = $self->getActionScope();
	$message =~ s/$gActionScope/$scope/g;
	return $message;
}

sub setActionStatusDone {
	$_[0]->{actionStatus} = ACTION_STATUS_DONE;
}

sub hasActionStatusDone {
	return $_[0]->{actionStatus} == ACTION_STATUS_DONE;
}

sub setActionStatusDoneWithWarnings {
	$_[0]->{actionStatus} = ACTION_STATUS_DONE_WITH_WARNING;
}

sub hasActionStatusDoneWithWarnings {
	return $_[0]->{actionStatus} == ACTION_STATUS_DONE_WITH_WARNING;
}

sub setActionStatusFailed {
	$_[0]->{actionStatus} = ACTION_STATUS_FAILED;
}

sub hasActionStatusFailed {
	return $_[0]->{actionStatus} == ACTION_STATUS_FAILED;
}

sub getActionDone {
	return undef;
}

sub getActionFailed {
	return undef;
}

sub getActionAborted {
	return undef;
}

sub getWarnings {
	return $_[0]->{warnings};
}

sub setWarnings {
	my ( $self, $warnings ) = @_;
	$_[0]->{warnings} = $warnings;
}

sub getExecutionSteps {
	return $_[0]->{steps};
}

sub addExecutionStep {
	my ( $self, $executionStep ) = @_;
	if ( $self->{steps} == 0 ) {
		$self->{steps} = [];
	}
	push( @{$self->{steps}}, $executionStep );

}

sub addExecutionSteps {
	my ( $self, $executionStep ) = @_;
	if ( $self->{steps} == 0 ) {
		$self->{steps} = [];
	}
	push( @{$self->{steps}}, @$executionStep );

}

sub prepareHdbInstallerEnvironment {
	my ( $self, $command ) = @_;
	return LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($command);
}

sub startSLAnalyticsMonitor{
    my($self) = @_;
    my $instconfig = $self->{instconfig};
    if(!defined $instconfig){
    	$instconfig = $self->{config};
    }
    LCM::SlAnalytics::SLAMonitor::start($instconfig);
}

sub stopSLAnalyticsMonitor{
    my($self, $rc, $executionStatusMessage) = @_;
    if($rc){
    	LCM::SlAnalytics::SLAMonitor::stop(undef, undef);
    } else {
        LCM::SlAnalytics::SLAMonitor::stop(1, $executionStatusMessage);
    }
}

#Default implementation for executors which don't provide this data
sub getNumberOfExpectedOutputLines{
	return 30;
}

sub getSkippedComponentsForRegistration{
    my ($self, $isRemote, $componentManager, $scenario) = @_;
    my $installedComponents = $componentManager->getAllComponents();
    my $componentsRegistrationMgr;
    if ($isRemote){
        $componentsRegistrationMgr = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager($self->{components}, $installedComponents, $scenario); 
    } else{
        $componentsRegistrationMgr = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager($self->{components}, $installedComponents, $scenario);
    }
    $componentsRegistrationMgr->detectComponents();
    return $componentsRegistrationMgr->getSkippedComponents();
}

1;
