package LCM::Gui::Dialogs::AddHosts::AdditionalHostsPropertiesDialog;

use strict;
use base qw (SAPDB::Install::Wizard::Dialog::Base);
use Wx qw ( wxVERTICAL wxGROW wxALL wxEXPAND );
use LCM::Gui::Dialogs::HostsPanel;
use LCM::Gui::Dialogs::UiComponentsFactory;
use LCM::Configuration::HdbLcmInstalledActionConfiguration qw ( IsHostPartOfSystem);
use LCM::Gui::Dialogs::AddHosts::AddHostsPanel;
use LCM::Gui::DialogUtilities;
use SDB::Install::Globals qw ($gHostRoleXS2Standby $gHostRoleXS2Worker);

sub new {
	my ($self) = shift->SUPER::new(@_);

	$self->{id}   = 'HostsProperties';
	$self->{name} = "Enter Additional Hosts Properties";

	return $self;
}

sub OnListItemsUpdated {
	my ($self, ) = @_;
	$self->{isAddHostParamaterSet} = 0;
}

sub init {
	my ($self, ) = @_;

	if ( ! $self->{dialogInitialized} ) {
		my $app        = Wx::wxTheApp;
	    my $instconfig = $app->{instconfig};  
		$instconfig->setBatchValueOfRequiredParams('AutoAddXS2Roles');
	
		my $outer_sizer = new Wx::BoxSizer(wxVERTICAL);
		$self->SetSizer($outer_sizer);
	
		my $sizer = new Wx::BoxSizer(wxVERTICAL);
		$self->{sizer} = $sizer;
		$outer_sizer->Add( $sizer, 1, wxALL | wxEXPAND, 20 );
	
		my $is_gtk1 = Wx::wxGTK1 ();
	    my $title = CreateTitle($self, 'Add Hosts', $is_gtk1);
	    $sizer->Add($title, 0);
	    $sizer->AddSpacer(20);
	
		my $hostsPanel = LCM::Gui::Dialogs::AddHosts::AddHostsPanel->new($self);
		my $componentsSizer = $hostsPanel->addNewHostPanel( wxVERTICAL, 0, wxEXPAND, 40, wxEXPAND | wxGROW );
		$sizer->Add( $componentsSizer, 1, wxEXPAND );
	    $self->{addHostPanel} = $hostsPanel;
	
		my $shouldShow = $instconfig->isSpecialComponentInstalled($gHostRoleXS2Worker) ? 1 : 0;
		$hostsPanel->getWindowControls()->{AutoAddXS2RolesLabel}->Show($shouldShow);
		$hostsPanel->getWindowControls()->{AutoAddXS2Roles}->Show($shouldShow);
	}

	if( ! IsHostPartOfSystem() ) {
        $self->_addLocalHostToListCtrl();
    }
    
    $self->_addHostsFromCmdLineToListCtrl();
	$self->SetAutoLayout(1);
	$self->{dialogInitialized} = 1;
	return {};
}

sub _addLocalHostToListCtrl {
    my ($self, ) = @_;
    
    my $addHostConfig = $self->_getAddHostConfig();
    my $localHost = $addHostConfig->{params}->{HostName};
       
    if( ! defined $localHost ) {
        return;
    }
    
    my $hostsList = $self->{Controls}->{'HostsList'};
    my $numberOfItems = $hostsList->GetItemCount();
    if($numberOfItems != 0) {
    	$self->_removeAllHostsFromHostsList();
    }
    
    my $role = $addHostConfig->{params}->{HostRoles};
    my @roles = split(",", $role->{value});
	my $roleValue = $roles[0];
    
    my $group = $addHostConfig->{params}->{HostFailoverGroup};
    my $workergroup = $addHostConfig->{params}->{WorkerGroup};
    my $storagePartition = $addHostConfig->{params}->{StoragePartitionNumber};
    
    my $hostObj = {
            host => $localHost->{value},
            role => $roleValue,
            group => $group->{value},
            workergroup => $workergroup->{value},
            storage_partition => $storagePartition->{value}
        };
    my $hostsPanel = LCM::Gui::Dialogs::AddHosts::AddHostsPanel->new($self);
    $hostsPanel->addHost( new LCM::AdditionalHostHelper($hostObj), 1 );
}

sub _addHostsFromCmdLineToListCtrl {
    my ($self, ) = @_;
    
    my $app        = Wx::wxTheApp;
    my $instconfig = $app->{instconfig};

    my $addHostsValue = $instconfig->getValueOrBatchValue('AddHosts');
	return if !$addHostsValue;

    require SDB::Install::Configuration::AddHostsParser;
    my $parser = new SDB::Install::Configuration::AddHostsParser();
    $parser->setMsgLstContext([$app->getMsgLst()]);

    if ( ! defined $parser->parse ( $addHostsValue ) ) {
        if (IsHostPartOfSystem()) {
            $app->ShowErrorMsg ("Error parsing command line option " . $instconfig->getOpt('AddHosts'), $parser->getErrMsgLst());
        }
        # Error message already showed in LCM::Gui::Dialogs::AddHosts::LocalHostPropertiesDialog
        return;
    }

    for my $host ( sort @{$parser->getHosts()} ) {
        next if(! $host);

        my @roles = @ { $parser->getRoles($host) };
        my $csvRoles = join(",", @roles);
        my $workerGroups = $parser->getValue($host, 'workergroup') || [];
        my $workerGroupsString = scalar(@{$workerGroups}) > 0 ? join(',', @{$workerGroups}) : undef;
        my $hostObj = {
            host => $host,
            role => $csvRoles,
            workergroup => $workerGroupsString,
            group => $parser->getValue($host, 'group'),
            storage_partition => $parser->getValue($host, 'storage_partition')
        };
        my $hostsPanel = LCM::Gui::Dialogs::AddHosts::AddHostsPanel->new($self);
        $hostsPanel->addHost( new LCM::AdditionalHostHelper($hostObj) );
    }   
}

sub _removeAllHostsFromHostsList {
	my ($self, ) = @_;
	my $hostsList = $self->{Controls}->{'HostsList'};
	$hostsList->DeleteAllItems();
}

sub Process {
    my ( $self, @args ) = @_;
    my $app = Wx::wxTheApp();
    my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();

    $instconfig->resetError();
    $self->_resetParams();

    if ( ! $self->_validateAddHosts() ) {
        return $self->errorState();
    }
    if ( ! $instconfig->isUseSAPHostagent() and ! $self->_validateRootCredentials() ) {
        return $self->errorState();
    }
    if ( ! $self->_validateHostagentPassword()) {
        return $self->errorState();
    }
    if(!$instconfig->isSkipped('AutoAddXS2Roles')){
        my $addHostPanel = $self->{addHostPanel};
        my $value = $addHostPanel->getWindowControls()->{AutoAddXS2Roles}->IsChecked() ? 1 : 0;
        if(!$instconfig->setValue('AutoAddXS2Roles', $value)){
            return $self->errorState();
        }
    }
    if ( ! $self->_validateSidadmPassword() ) {
        return $self->errorState();
    }
    $self->GetWizard()->RefreshSkippableDialogs();
    return {};
}

sub _resetParams {
	my ($self, ) = @_;
	my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();
    $instconfig->{params}->{AddHosts}->{value} = undef;
    $instconfig->{_additionalRemoteHosts} = undef;
}

sub _validateAddHosts {
	my ($self, ) = @_;
    my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();
    
    my $hostsList = $self->{Controls}->{'HostsList'};
    my $numberOfItems = $hostsList->GetItemCount();

    if ( $numberOfItems == 0 ) {
        $instconfig->AddError("There must be at least one host to proceed!");
        $hostsList->SetFocus();
        return undef;
    }
    
    my $addHosts = $self->_getAddHostsFromHostsListControl($hostsList);
    return 1 if($addHosts eq '');

    if ( ! $instconfig->setValue('AddHosts', $addHosts) ) {
        $hostsList->SetFocus();
        return undef;
    }
    
    return 1;
}

sub _validateRootCredentials {
	my ($self, ) = @_;
	return ( ! $self->_validateRootUser() or ! $self->_validateRootPassword() ) ? undef : 1;
}

sub _validateHostagentPassword {
    my ($self, ) = @_;
    my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();

    if ( $instconfig->isSkipped('HostagentPassword') ) {
    	return 1;
    }

    my $hostagentPasswordCtrl = $self->{Controls}->{'HostagentPassword'};
    my $hostagentPassword = $hostagentPasswordCtrl->GetValue();

    if ( ! defined $hostagentPassword or $hostagentPassword eq '' ) {
    	$instconfig->AddError("SAP Host Agent Password is empty");
    	$hostagentPasswordCtrl->SetFocus();
        return undef;
    }

    if ( ! $instconfig->setValue('HostagentPassword', $hostagentPassword) ) {
    	$hostagentPasswordCtrl->SetFocus();
        return undef;
    }
    
    return 1;
}

sub _validateRootUser {
    my ($self, ) = @_;
    my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();

    if ( $instconfig->isSkipped('RootUser') ) {
    	return 1;
    }

    my $rootUserCtrl = $self->{Controls}->{'RootUser'};
    my $rootUser = $rootUserCtrl->GetValue();

    if ( ! defined $rootUser or $rootUser eq '' ) {
    	$instconfig->AddError("Root user is a mandatory field!");
    	$rootUserCtrl->SetFocus();
        return undef;
    }

    if ( ! $instconfig->setValue('RootUser', $rootUser) ) {
    	$rootUserCtrl->SetFocus();
        return undef;
    }
    
    return 1;
}

sub _validateRootPassword {
    my ($self, ) = @_;
    my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();

    if ( $instconfig->isSkipped('RootPassword') ) {
    	return 1;
    }

    my $rootPassCtrl = $self->{Controls}->{'RootPassword'};
    my $rootPass = $rootPassCtrl->GetValue();

    if ( ! $instconfig->setValue('RootPassword', $rootPass) ) {
        $rootPassCtrl->SetFocus();
        return undef;
    }

    return 1;
}

sub _validateSidadmPassword {
    my ($self) = @_;
    my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();

    if (!IsHostPartOfSystem()) {
        # Password parameter was set on the previous dialog but with respect to the
        # parameters order (see AddHostsBaseConfiguration) the value is being set once
        # more in order to correctly detect some configurations (e.g. in the XSUserIsolationListener)
        my $password = $instconfig->getValue('Password');
        $instconfig->setValue('Password', $password);
        return 1;
    }

    my $sidadmPasswordCtrl = $self->{Controls}->{'Password'};
    my $sidadmPassword = $sidadmPasswordCtrl->GetValue();

    if (!defined($sidadmPassword) || $sidadmPassword eq '') {
        $instconfig->AddError("System Administrator Password is empty");
        $sidadmPasswordCtrl->SetFocus();
        return undef;
    }
    if (! $instconfig->setValue( 'Password', $sidadmPassword )) {
        $sidadmPasswordCtrl->SetFocus();
        return undef;
    }
    return 1;
}

sub _getAddHostConfig {
	return LCM::Gui::DialogUtilities::GetAppConfig()->{localHostConfiguration};
}

sub errorState {
    my ( $self ) = @_;
    my $app = Wx::wxTheApp;
    my $instconfig = $app->{instconfig};
    $app->ShowErrorMsg ('Configuration error', $instconfig->getErrMsgLst());
    return undef;   
}

sub _getAddHostsFromHostsListControl {
	my ( $self, $hostsList ) = @_;
	
	my $numberOfHosts = $hostsList->GetItemCount();
	my $addHostsCommand = "";

	my %roles = ();
	for my $index (0 .. ($numberOfHosts-1)) {
		my $item = LCM::Gui::Dialogs::HostsPanel::GetListItem($self, $index);
		my ($host, $role) = ($item->{host}, $item->{role});
		my @hostRoles = split(",", $role);
		for ( @hostRoles ) {
			if ( exists $roles{$host} ) {   
				push ( @ { $roles{$host} }, $_ );
			} else {
				$roles{$host} = [$_];
			}
		}
	}
	
	for my $index (0 .. ($numberOfHosts-1)) {
		$addHostsCommand .= "," if ($index > 0);
		my $item = LCM::Gui::Dialogs::HostsPanel::GetListItem($self, $index);
		$self->_addHost($addHostsCommand, $item->{host});
		foreach my $role (@{$roles{$item->{host}}}){
			$self->_addRole($addHostsCommand, $role);
		}
		$self->_addGroup($addHostsCommand, $item->{group});
		$self->_addWorkerGroup($addHostsCommand, $item->{workergroup});
		$self->_addSP($addHostsCommand, $item->{storage_partition});
	}
	return $addHostsCommand;
}

sub _handleAutoAddCheckBox {
	my ($self) = @_;
	my $addHostPanel = $self->{addHostPanel};
	my $addHostString = $addHostPanel->buildAddHostsOptionValue();
	my $instconfig = LCM::Gui::DialogUtilities::GetAppConfig();
	my $isAddingXsRole = ($addHostString =~ /:role=($gHostRoleXS2Standby|$gHostRoleXS2Worker)/);
	my $isXs2Installed = $instconfig->isSpecialComponentInstalled($gHostRoleXS2Worker);
	my $shouldDisable = $isAddingXsRole || ! $isXs2Installed;
	my $value = $instconfig->hasValue('AutoAddXS2Roles') ? $instconfig->getValue('AutoAddXS2Roles') : 1;
	my $uiControl = $addHostPanel->getWindowControls()->{AutoAddXS2Roles};

	$uiControl->Enable($shouldDisable ? 0 : 1);
	$uiControl->SetValue($shouldDisable ? 0 : $value);
}

sub _addHost {
	my ( $self, $addHostsCommand, $host ) = @_;
    $_[1] .= $host;
}

sub _addRole {
    my ( $self, $addHostsCommand, $role ) = @_;
    $_[1] .= ":role=$role";
}

sub _addGroup {
    my ( $self, $addHostsCommand, $group ) = @_;
    if($group ne '' and $group ne 'default') {
        $_[1] .= ":group=$group";
    } 
}

sub _addWorkerGroup {
    my ( $self, $addHostsCommand, $workergroup ) = @_;
    if(defined $workergroup && $workergroup) {
        my @workerGroups = split(',', $workergroup);
        $_[1] .= ":workergroup=" . join(':workergroup=', @workerGroups);
    }
}

sub _addSP {
    my ( $self, $addHostsCommand, $storage_partition ) = @_;
       
    if(defined $storage_partition) {
        $_[1] .= ":storage_partition=$storage_partition";
    }
}

1;
