#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2014 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

HDBUserPropertiesDialog

=head1 DESCRIPTION

HDB user parameters for the system rename scenario.
=cut

#===============================================================================
package LCM::Gui::Dialogs::Rename::HDBUserPropertiesDialog;

use strict;
our @ISA = qw (SDB::Install::Gui::Dialogs::GenericParameter);
use Wx qw(wxALIGN_CENTER_VERTICAL wxEXPAND wxTE_PASSWORD);
use Wx::Event qw (EVT_CHECKBOX);

my $ChangeSystemPasswd_chbId = 'ChangeSystemPasswd_chb';
my $SystemUser_fldId = 'SystemUser_fld';
my $SrcSQLSysPasswd_fldId = 'SrcSQLSysPasswd_fld';
my $TrgSQLSysPasswd_fldId = 'TrgSQLSysPasswd_fld';
my $configrmTrgSQLSysPasswd_fldId = 'configrmTrgSQLSysPasswd_fld';

sub new {
	my ($self) = shift()->SUPER::new(@_);

	my $instconfig = Wx::wxTheApp->{instconfig};
	$instconfig->setChangeSystemPasswd(0);
	my $instance = $instconfig->getOwnInstance();
	my $userString = "Database User";
	if(defined $instance && $instance->isMultiDb()){
		$userString =~ s/(?<!System\s)Database/System Database/g;
	}

	$self->{id}         = 'HDBUserPropertiesDialog';
	$self->{label_text} = "Enter $userString Properties";
	$self->{name}       = "Define $userString";


	return $self;
}

sub init {
	my $self = shift();
	my $returnValue = $self->SUPER::init(@_);
# Remove the auto_next property, because it creates a UI issue
	delete($returnValue->{auto_next});
	return $returnValue;
}

sub getParamKeys {
    return [];
}

#Override
sub AddAdditionalParams {
	my ( $self, $instconfig ) = @_;

	return 1 if($self->{$ChangeSystemPasswd_chbId});

	$self->_addChangeSystemPasswdControl($instconfig, $ChangeSystemPasswd_chbId);
	$self->_addSystemUserControl($instconfig, 'SystemUser', $SystemUser_fldId);
	$self->_addSystemPasswdControl($instconfig->{params}->{SrcSQLSysPasswd}, $SrcSQLSysPasswd_fldId);
	$self->_addSystemPasswdControl($instconfig->{params}->{TrgSQLSysPasswd}, $TrgSQLSysPasswd_fldId);

	my %confirmTrgSQLSysPasswdParam;
	$confirmTrgSQLSysPasswdParam{str} = 'Confirm ' . $instconfig->{params}->{TrgSQLSysPasswd}->{str};
	$confirmTrgSQLSysPasswdParam{desc} = 'Confirm ' . $instconfig->{params}->{TrgSQLSysPasswd}->{desc};

	$self->_addSystemPasswdControl ( \%confirmTrgSQLSysPasswdParam, $configrmTrgSQLSysPasswd_fldId );
	EVT_CHECKBOX($self, $self->{'ChangeSystemPasswd_chb'}->GetId(), \&OnChangeSystemPasswd);

	return 1;
}

sub Process {
	my ( $self, $config ) = @_;

	my $isEnabled = $self->{$ChangeSystemPasswd_chbId}->IsChecked();
	if ( ! $isEnabled ) {
		return {};
	}

	my $rc = $self->SUPER::Process ( $config );
	my $instconfig = Wx::wxTheApp()->{instconfig};
	$instconfig->resetMsgLstContext ();

	$instconfig->{params}->{SrcSQLSysPasswd}->{value} = undef;
	$instconfig->setDefault('SrcSQLSysPasswd', undef);

	my $systemUserValue = $self->{$SystemUser_fldId}->GetValue();
	if ( ! $systemUserValue ) {
		my $error = 'Missing mandatory parameter: ' . $instconfig->getString('SystemUser');
		$instconfig->getErrMsgLst()->AddError($error);
		return $self->_handleConfigurationError($instconfig);
	}
	if ( ! $instconfig->setValue('SystemUser', $systemUserValue)) {
		return $self->_handleConfigurationError($instconfig);
	}

	my $sourcePasswordValue = $self->{$SrcSQLSysPasswd_fldId}->GetValue();
	if ( ! $sourcePasswordValue ) {
		my $error = 'Missing mandatory parameter: ' . $instconfig->getString('SrcSQLSysPasswd');
		$instconfig->getErrMsgLst()->AddError($error);
		return $self->_handleConfigurationError($instconfig);
	}

	if ( ! $instconfig->setValue('SrcSQLSysPasswd', $sourcePasswordValue)) {
		return $self->_handleConfigurationError($instconfig);
	}
	$instconfig->resetError(); # setting the SrcSQLSysPasswd can add unwanted error messages

	$instconfig->{params}->{TrgSQLSysPasswd}->{value} = undef;
	$instconfig->setDefault('TrgSQLSysPasswd', undef);

	my $targetPasswordValue = $self->{$TrgSQLSysPasswd_fldId}->GetValue();
	if ( ! $targetPasswordValue ) {
		my $error = 'Missing mandatory parameter: ' . $instconfig->getString('TrgSQLSysPasswd');
		$instconfig->getErrMsgLst()->AddError($error);
		return $self->_handleConfigurationError($instconfig);
	}

	if ( $targetPasswordValue ne $self->{$configrmTrgSQLSysPasswd_fldId}->GetValue() ) {
		my $error = "Password confirmation doesn't match.";
		$instconfig->getErrMsgLst()->AddError($error);
		return $self->_handleConfigurationError($instconfig);
	}

	if ( ! $instconfig->setValue('TrgSQLSysPasswd', $targetPasswordValue) ) {
		return $self->_handleConfigurationError($instconfig);
	}

	return $rc;
}

sub _handleConfigurationError {
	my ($self, $configuration) = @_;
	Wx::wxTheApp()->ShowErrorMsg ('Configuration error', $configuration->getErrMsgLst());
	return undef;
	
}

sub OnChangeSystemPasswd {
    my ($self) = @_;
    my $isEnabled = $self->{$ChangeSystemPasswd_chbId}->IsChecked();

    $self->{$SystemUser_fldId}->Enable($isEnabled);
    $self->{$SystemUser_fldId . 'Label'}->SetRequired($isEnabled);

    $self->{$SrcSQLSysPasswd_fldId}->Enable($isEnabled);
    $self->{$SrcSQLSysPasswd_fldId . 'Label'}->SetRequired($isEnabled);

    $self->{$TrgSQLSysPasswd_fldId}->Enable($isEnabled);
    $self->{$TrgSQLSysPasswd_fldId . 'Label'}->SetRequired($isEnabled);

    $self->{$configrmTrgSQLSysPasswd_fldId}->Enable($isEnabled);
    $self->{$configrmTrgSQLSysPasswd_fldId . 'Label'}->SetRequired($isEnabled);

    Wx::wxTheApp->{instconfig}->setChangeSystemPasswd($isEnabled ? 1 : 0);
    $self->{'grid_sizer'}->Layout();
    return 1;
}

sub _addChangeSystemPasswdControl {
	my ($self, $instconfig, $id) = @_;

	my $changeSystemPasswdParam = $instconfig->{params}->{'ChangeSystemPasswd'};
	my $changeSystemPasswdLabel = new Wx::StaticText ( $self, -1, '' );
	$changeSystemPasswdLabel->SetToolTip ( $changeSystemPasswdParam->{desc} );

	my $changeSystemPasswdControl =
		new Wx::CheckBox ( $self, -1, $changeSystemPasswdParam->{'str'}, [ -1, -1 ], [ -1, -1 ] );
	$changeSystemPasswdControl->SetToolTip ( $changeSystemPasswdParam->{desc} );
	$self->{$id} = $changeSystemPasswdControl;

	$self->{'grid_sizer'}->Add ( $changeSystemPasswdLabel,   0, wxALIGN_CENTER_VERTICAL );
	$self->{'grid_sizer'}->Add ( $changeSystemPasswdControl, 1, wxEXPAND );

	return 1;
}

sub _addSystemPasswdControl {
	my ( $self, $param, $id ) = @_;

	my $label = SDB::Install::Gui::Control::Label->new ( $self, -1, $param->{str});
	$label->SetToolTip ( $param->{desc} );
	$self->{$id . 'Label'} = $label;

	my $control = new Wx::TextCtrl ( $self, -1, '', [ -1, -1 ], [ -1, -1 ], wxTE_PASSWORD );
    $control->SetToolTip ( $param->{desc} );
	$control->Enable ( $self->_isDbSystemPasswordChanged () );
	$self->{$id} = $control;

	$self->{'grid_sizer'}->Add ( $label,   0, wxALIGN_CENTER_VERTICAL );
	$self->{'grid_sizer'}->Add ( $control, 1, wxEXPAND );

	return 1;
}

sub _addSystemUserControl {
	my ( $self, $instconfig, $paramId, $controlId) = @_;
	my $param = $instconfig->{params}->{$paramId};
	my $label = SDB::Install::Gui::Control::Label->new ( $self, -1, $param->{str});

	$label->SetToolTip ( $param->{desc} );
	$self->{$controlId . 'Label'} = $label;

	my $value = $param->{value} || $instconfig->getDefault($paramId) || 'SYSTEM';
	my $control = new Wx::TextCtrl ( $self, -1, $value, [ -1, -1 ], [ -1, -1 ]);
	$control->SetToolTip ( $param->{desc} );
	$control->Enable ( $self->_isDbSystemPasswordChanged () );
	$self->{$controlId} = $control;

	$self->{'grid_sizer'}->Add ( $label,   0, wxALIGN_CENTER_VERTICAL );
	$self->{'grid_sizer'}->Add ( $control, 1, wxEXPAND );

	return 1;
}

sub _isDbSystemPasswordChanged() {
	return Wx::wxTheApp->{instconfig}->isDbSystemPasswordChanged ();
}

1;
