#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2014 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

SystemPropertiesDialog

=head1 DESCRIPTION

Parameters for the system rename scenario.
=cut

#===============================================================================
package LCM::Gui::Dialogs::Rename::SystemPropertiesDialog;

use strict;
our @ISA = qw (SDB::Install::Gui::Dialogs::GenericParameter);
use Wx qw (wxALIGN_CENTER_VERTICAL wxEXPAND);
use SDB::Install::NewDBUser;

my $localhostFld = 'LocalhostFld';
my $localhostCertificateFld = 'LocalhostCertificateFld';

sub new {
	my ( $self ) = shift->SUPER::new ( @_ );

	$self->{id}         = 'SystemPropertiesDialog';
	$self->{label_text} = "System Properties Configuration";
	$self->{name}       = 'Define System Properties';
	
	my $instconfig = Wx::wxTheApp->{instconfig};
    $instconfig->setChangeSystemPasswd(0);

	return $self;
}

sub getParamKeys {
	my ( $self ) = @_;
	my @paramKeys;
	push( @paramKeys, 'srcPasswd' );
	push( @paramKeys, 'newSID' );
	push( @paramKeys, 'InstanceNumber' );
	push( @paramKeys, 'SystemUsage' );

	return \@paramKeys;
}

#Override
sub AddAdditionalParams {
	my ( $self, $instconfig ) = @_;
	$self->_addLocalhostNameControl ($instconfig) unless $instconfig->isDistributedSystem(1);
	$self->_addLocalhostCertificateControl ($instconfig) if $self->isCertificatesNeeded($instconfig);
	return 1;
}

sub Process {
	my ( $self, $config ) = @_;
    my $rc = $self->SUPER::Process ( $config );
    my $app = Wx::wxTheApp();
    my $instconfig = $app->{instconfig};
    $self->_resetPathParameters($instconfig);
	$instconfig->resetMsgLstContext ();
	unless($instconfig->isDistributedSystem(1)) {
		my $origin_value = $instconfig->{params}->{HostMap}->{own_origin_value};
		unless(defined $origin_value) {
			$origin_value = $instconfig->getLocalHanaHost ();
		}
		my $errorList = $instconfig->validateHost ( $origin_value, $self->{$localhostFld}->GetValue () );
		if ( $errorList && !$errorList->isEmpty () ) {
			$app->ShowErrorMsg ( 'Configuration error', $errorList );
			return undef;
		}
		if(!$instconfig->checkAllMapValues('HostMap')){
			$app->ShowErrorMsg('Configuration error', $instconfig->getErrMsgLst());
			return undef;
		}
	}

	my $xsSpaceIsolation = LCM::Configuration::ValueChangeListeners::Rename::XsSpaceIsolation->new ();
	$xsSpaceIsolation->handleOsUsers($instconfig);

	if($self->isCertificatesNeeded($instconfig)) {
		#single host only
		#initializes the default certificates for the new hostname
		$instconfig->fillCertificatesHostmapDefaultValuesAndLocalData();
        $instconfig->resetMsgLstContext();
        my $originLocalhostName;
        my @hosts = $instconfig->getOriginRemoteHosts();
        if(scalar(@hosts) > 0 ){
            $originLocalhostName = $hosts[0];
        }
        if(!defined $originLocalhostName){
            $originLocalhostName = $instconfig->getLocalHanaHost();         
        }
	    my $targetHost = $instconfig->{params}->{HostMap}->{value}->{$originLocalhostName};
        
        unless(defined $targetHost){
        	my @originValues = $instconfig->{params}->{CertificatesHostmap}->{origin_values};
		    if(scalar @originValues == 1){
			    $originLocalhostName = $originValues[0];
		    }
	    $targetHost = $instconfig->{params}->{CertificatesHostmap}->{default_map}->{$originLocalhostName};
        }
        my $exitCode = $instconfig->setCertificatesHostmap($targetHost, $self->{$localhostCertificateFld}->GetValue (), 1);
        unless($exitCode) {
            $app->ShowErrorMsg ( 'Configuration error', $instconfig->getErrMsgLst() );
            return undef;
        }	    
	}

    my $wizard = $self->GetWizard ();
    if (!$instconfig->isSIDChanged () && $self->isOsUserAlreadyExists($instconfig) && $instconfig->hasValue('Password')) {
		$instconfig->AddMessage ( 'Target SID is not set. Skipping OS user properties dialog.' );
		$wizard->SkipDialog ( 'TargetUserDialog' );    # skip the OS user properties dialog	
	} else {
	    $wizard->SkipDialog ( 'TargetUserDialog', 0 ); # do not skip
	}
	$wizard->RefreshSkippableDialogs();

	return $rc;
}

sub _resetPathParameters {
    my ($self, $config) = @_;
    $config->resetParam('BasePathDataVolumes');
    $config->resetParam('BasePathLogVolumes');
    $config->resetParam('BasePathDataBackup');
    $config->resetParam('BasePathLogBackup');
    $config->resetParam('BasePathCatalogBackup');
    $config->resetParam('EsDataPath');
    $config->resetParam('EsLogPath');
    $config->resetParam('AcceleratorDataPath');
    $config->resetParam('AcceleratorLogPath');
    $config->resetParam('BasepathFileDownloadRDSync');
    $config->resetParam('BasepathFileUploadRDSync');
}

sub _addLocalhostNameControl {
    my ($self, $instconfig) = @_;
    my $localhostLabelStr = 'Target Host Name';
    my $localhostLabel = new SDB::Install::Gui::Control::Label ( $self, -1, $localhostLabelStr, 1 );
    $localhostLabel->SetToolTip ( $localhostLabelStr );
    $self->{'grid_sizer'}->Add ( $localhostLabel, 0, wxALIGN_CENTER_VERTICAL );

    my $control = new Wx::TextCtrl ( $self, -1, $instconfig->getOriginLocalhostName (), [ -1, -1 ], [ -1, -1 ]);
    $control->SetToolTip ( $localhostLabelStr );
    $self->{$localhostFld} = $control;
    $self->{'grid_sizer'}->Add ( $control, 1, wxEXPAND );
    
    return 1;    
}

sub _addLocalhostCertificateControl {
    my ($self, $instconfig) = @_;
    
    #try to detect localhost certificate
    $instconfig->fillCertificatesHostmapDefaultValuesAndLocalData();
    
    my $localhostLabelStr = 'Target Certificate Host Name';
    my $localhostLabel = new SDB::Install::Gui::Control::Label ( $self, -1, $localhostLabelStr, 1 );
    $localhostLabel->SetToolTip ( $localhostLabelStr );
    $self->{'grid_sizer'}->Add ( $localhostLabel, 0, wxALIGN_CENTER_VERTICAL );
    my $hostMap = $instconfig->{params}->{HostMap}; 
    my $defaultTarget = $instconfig->getOriginLocalhostName ();

    my $defCertificate = $instconfig->getOriginLocalhostCertificate ();
    if (not defined $defCertificate and defined $instconfig->{CertificatesHostmap}){
    	my @certificateValues = values %{$instconfig->{CertificatesHostmap}};
    	if(scalar @certificateValues == 1){
    		$defCertificate = $certificateValues[0];
    	}
        
    }
    my $control = new Wx::TextCtrl ( $self, -1, $defCertificate, [ -1, -1 ], [ -1, -1 ]);
    $control->SetToolTip ( $localhostLabelStr );
    $self->{$localhostCertificateFld} = $control;
    $self->{'grid_sizer'}->Add ( $control, 1, wxEXPAND );
    
    return 1;    
}

sub isCertificatesNeeded {
    my ($self, $instconfig) = @_;
    return !$instconfig->isDistributedSystem(1) && $instconfig->isCertificatesHostMapNeeded();
}

sub isOsUserAlreadyExists {
    my ($self, $instconfig) = @_;
    my $osUser = new SDB::Install::NewDBUser ($instconfig->{params}->{newSID}->{value});
    return $osUser->exists();
}

1;
